/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    dlog.c
*   @brief   
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "dlog.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

#if DLOG_ENABLE

/* status variables for uint32_t data log array0 and array1 */
uint32_t dlog_u32_idx0;    //data log index, from 0 to DLOG_U32_SIZE0
uint32_t dlog_u32_status0; //dlog status, 0-stop, 1-start
uint32_t dlog_u32_prescaler0;  //dlog prescaler, 4 - one data is logged for every 4 data
uint32_t dlog_u32_skip0;       //internal used variable
uint32_t dlog_u32_size0;       //size, it's replaced by MACRO DLOG_U32_SIZE0

/* status variables for uint32_t data log array2 and array3 */
uint32_t dlog_u32_idx2;
uint32_t dlog_u32_status2;
uint32_t dlog_u32_prescaler2;
uint32_t dlog_u32_skip2;
uint32_t dlog_u32_size2;

/* status variables for uint16_t data log array0 and array1 */
uint32_t dlog_u16_idx0;
uint32_t dlog_u16_status0;
uint32_t dlog_u16_prescaler0;
uint32_t dlog_u16_skip0;
uint32_t dlog_u16_size0;

/* status variables for uint16_t data log array2 and array3 */
uint32_t dlog_u16_idx2;
uint32_t dlog_u16_status2;
uint32_t dlog_u16_prescaler2;
uint32_t dlog_u16_skip2;
uint32_t dlog_u16_size2;

/* status variables for float data log array0 and array1 */
uint32_t dlog_f32_idx0;
uint32_t dlog_f32_status0;
uint32_t dlog_f32_prescaler0;
uint32_t dlog_f32_skip0;
uint32_t dlog_f32_size0;

/* status variables for float data log array2 and array3 */
uint32_t dlog_f32_idx2;
uint32_t dlog_f32_status2;
uint32_t dlog_f32_prescaler2;
uint32_t dlog_f32_skip2;
uint32_t dlog_f32_size2;

DLOG_2CH_F dlog_2ch_f_cfg0;
DLOG_2CH_F dlog_2ch_f_cfg1;

DLOG_2CH_U32 dlog_2ch_u32_cfg0;
DLOG_2CH_U32 dlog_2ch_u32_cfg1;

DLOG_2CH_U16 dlog_2ch_u16_cfg0;
DLOG_2CH_U16 dlog_2ch_u16_cfg1;

__attribute__ ((section (".testbss"))) uint32_t dlog_u32_array0[DLOG_U32_SIZE0];
__attribute__ ((section (".testbss"))) uint32_t dlog_u32_array1[DLOG_U32_SIZE0];
__attribute__ ((section (".testbss"))) uint32_t dlog_u32_array2[DLOG_U32_SIZE2];
__attribute__ ((section (".testbss"))) uint32_t dlog_u32_array3[DLOG_U32_SIZE2];

__attribute__ ((section (".testbss"))) uint16_t dlog_u16_array0[DLOG_U16_SIZE0];
__attribute__ ((section (".testbss"))) uint16_t dlog_u16_array1[DLOG_U16_SIZE0];
__attribute__ ((section (".testbss"))) uint16_t dlog_u16_array2[DLOG_U16_SIZE2];
__attribute__ ((section (".testbss"))) uint16_t dlog_u16_array3[DLOG_U16_SIZE2];

__attribute__ ((section (".testbss"))) float32_t dlog_f32_array0[DLOG_F32_SIZE0];
__attribute__ ((section (".testbss"))) float32_t dlog_f32_array1[DLOG_F32_SIZE2];
__attribute__ ((section (".testbss"))) float32_t dlog_f32_array2[DLOG_F32_SIZE0];
__attribute__ ((section (".testbss"))) float32_t dlog_f32_array3[DLOG_F32_SIZE2];

#endif

/* ========================================================================== */
/*                         External Function Prototypes                       */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

/*****************************************************************************
 * The following groups of functions are for logging float32_t data into
 * dlog_f32_array0, dlog_f32_array1
 *****************************************************************************/
/* start data logging into f32_array0/1 */
void dlog_2ch_f32_start0(void)
{
	dlog_f32_status0 = 1;
}

/* stop data logging into f32_array0/1 */
void dlog_2ch_f32_stop0(void)
{
	dlog_f32_status0 = 1;
}

/*
 * It logs data once.
 * Data log stops when idx reaches dlog size.
 * Call this function in the loop you want to log data.
 * Pass the data into this function as input parameters.
 * */
void dlog_2ch_f32_run0_once(float32_t input0, float32_t input1)
{
	if(dlog_f32_status0)
	{
		if(dlog_f32_idx0 < DLOG_F32_SIZE0)
		{
			dlog_f32_array0[dlog_f32_idx0] = input0;
			dlog_f32_array1[dlog_f32_idx0] = input1;
			dlog_f32_idx0++;
		}
		else
			dlog_f32_status0 = 0;
	}
}

/*
 * It logs data continuously.
 * When idx reaches dlog size, it starts over from 0 again.
 * Call this function in the loop you want to log data.
 * Pass the data into this function as input parameters.
 * */
void dlog_2ch_f32_run0_cont(float32_t input0, float32_t input1)
{
	if(dlog_f32_status0)
	{
		if(dlog_f32_idx0 < DLOG_F32_SIZE0)
		{
			dlog_f32_array0[dlog_f32_idx0] = input0;
			dlog_f32_array1[dlog_f32_idx0] = input1;
			dlog_f32_idx0++;
		}
		else
			dlog_f32_idx0 = 0;
	}
}


/*****************************************************************************
 * The following groups of functions are for logging uint32_t data into
 * dlog_u32_array0, dlog_u32_array1
 *****************************************************************************/
/* start data logging into u32_array0/1 */
void dlog_2ch_u32_start0(void)
{
	dlog_u32_status0 = 1;
}

/* stop data logging into u32_array0/1 */
void dlog_2ch_u32_stop0(void)
{
	dlog_u32_status0 = 1;
}

/*
 * It logs data once.
 * Data log stops when idx reaches dlog size.
 * Call this function in the loop you want to log data.
 * Pass the data into this function as input parameters.
 * */
void dlog_2ch_u32_run0_once(uint32_t input0, uint32_t input1)
{
	if(dlog_u32_status0)
	{
		if(dlog_u32_idx0 < DLOG_U32_SIZE0)
		{
			dlog_u32_array0[dlog_u32_idx0] = input0;
			dlog_u32_array1[dlog_u32_idx0] = input1;
			dlog_u32_idx0++;
		}
		else
			dlog_u32_status0 = 0;
	}
}

/*
 * It logs data continuously.
 * When idx reaches dlog size, it starts over from 0 again.
 * Call this function in the loop you want to log data.
 * Pass the data into this function as input parameters.
 * */
void dlog_2ch_u32_run0_cont(uint32_t input0, uint32_t input1)
{
	if(dlog_u32_status0)
	{
		if(dlog_u32_idx0 < DLOG_U32_SIZE0)
		{
			dlog_u32_array0[dlog_u32_idx0] = input0;
			dlog_u32_array1[dlog_u32_idx0] = input1;
			dlog_u32_idx0++;
		}
		else
			dlog_u32_idx0 = 0;
	}
}


/*****************************************************************************
 * The following groups of functions are for logging uint16_t data into
 * dlog_u16_array0, dlog_u16_array1
 *****************************************************************************/
/* start data logging into u16_array0/1 */
void dlog_2ch_u16_start0(void)
{
	dlog_u32_status0 = 1;
}

/* stop data logging into u16_array0/1 */
void dlog_2ch_u16_stop0(void)
{
	dlog_u32_status0 = 1;
}

/*
 * It logs data once.
 * Data log stops when idx reaches dlog size.
 * Call this function in the loop you want to log data.
 * Pass the data into this function as input parameters.
 * */
void dlog_2ch_u16_run0_once(uint16_t input0, uint16_t input1)
{
	if(dlog_u16_status0)
	{
		if(dlog_u16_idx0 < DLOG_U16_SIZE0)
		{
			dlog_u16_array0[dlog_u16_idx0] = input0;
			dlog_u16_array1[dlog_u16_idx0] = input1;
			dlog_u16_idx0++;
		}
		else
			dlog_u16_status0 = 0;
	}
}

/*
 * It logs data continuously.
 * When idx reaches dlog size, it starts over from 0 again.
 * Call this function in the loop you want to log data.
 * Pass the data into this function as input parameters.
 * */
void dlog_2ch_u16_run0_cont(uint16_t input0, uint16_t input1)
{
	if(dlog_u16_status0)
	{
		if(dlog_u16_idx0 < DLOG_U16_SIZE0)
		{
			dlog_u16_array0[dlog_u16_idx0] = input0;
			dlog_u16_array1[dlog_u16_idx0] = input1;
			dlog_u16_idx0++;
		}
		else
			dlog_u16_idx0 = 0;
	}
}

/*****************************************************************************
 * The following groups of functions are for logging float32_t data into
 * dlog_f32_array2, dlog_f32_array3
 *****************************************************************************/
/* start data logging into f32_array2/3 */
void dlog_2ch_f32_start2(void)
{
	dlog_f32_status2 = 1;
}

/* stop data logging into f32_array2/3 */
void dlog_2ch_f32_stop2(void)
{
	dlog_f32_status2 = 1;
}

/*
 * It logs data once.
 * Data log stops when idx reaches dlog size.
 * Call this function in the loop you want to log data.
 * Pass the data into this function as input parameters.
 * */
void dlog_2ch_f32_run2_once(float32_t input2, float32_t input3)
{
	if(dlog_f32_status2)
	{
		if(dlog_f32_idx2 < DLOG_F32_SIZE2)
		{
			dlog_f32_array2[dlog_f32_idx2] = input2;
			dlog_f32_array3[dlog_f32_idx2] = input3;
			dlog_f32_idx2++;
		}
		else
			dlog_f32_status2 = 0;
	}
}

/*
 * It logs data continuously.
 * When idx reaches dlog size, it starts over from 0 again.
 * Call this function in the loop you want to log data.
 * Pass the data into this function as input parameters.
 * */
void dlog_2ch_f32_run2_cont(float32_t input2, float32_t input3)
{
	if(dlog_f32_status2)
	{
		if(dlog_f32_idx2 < DLOG_F32_SIZE2)
		{
			dlog_f32_array2[dlog_f32_idx2] = input2;
			dlog_f32_array3[dlog_f32_idx2] = input3;
			dlog_f32_idx2++;
		}
		else
			dlog_f32_idx0 = 0;
	}
}


/*****************************************************************************
 * The following groups of functions are for logging uint32_t data into
 * dlog_u32_array2, dlog_u32_array3
 *****************************************************************************/
/* start data logging into u32_array2/3 */
void dlog_2ch_u32_start2(void)
{
	dlog_u32_status2 = 1;
}

/* stop data logging into u32_array2/3 */
void dlog_2ch_u32_stop2(void)
{
	dlog_u32_status2 = 1;
}

/*
 * It logs data once.
 * Data log stops when idx reaches dlog size.
 * Call this function in the loop you want to log data.
 * Pass the data into this function as input parameters.
 * */
void dlog_2ch_u32_run2_once(uint32_t input2, uint32_t input3)
{
	if(dlog_u32_status2)
	{
		if(dlog_u32_idx2 < DLOG_U32_SIZE2)
		{
			dlog_u32_array2[dlog_u32_idx2] = input2;
			dlog_u32_array3[dlog_u32_idx2] = input3;
			dlog_u32_idx2++;
		}
		else
			dlog_u32_status2 = 0;
	}
}

/*
 * It logs data continuously.
 * When idx reaches dlog size, it starts over from 0 again.
 * Call this function in the loop you want to log data.
 * Pass the data into this function as input parameters.
 * */
void dlog_2ch_u32_run2_cont(uint32_t input2, uint32_t input3)
{
	if(dlog_u32_status2)
	{
		if(dlog_u32_idx2 < DLOG_U32_SIZE2)
		{
			dlog_u32_array2[dlog_u32_idx2] = input2;
			dlog_u32_array3[dlog_u32_idx2] = input3;
			dlog_u32_idx2++;
		}
		else
			dlog_u32_idx2 = 0;
	}
}


/*****************************************************************************
 * The following groups of functions are for logging uint16_t data into
 * dlog_u16_array2, dlog_u16_array3
 *****************************************************************************/
/* start data logging into u16_array0/1 */
void dlog_2ch_u16_start2(void)
{
	dlog_u32_status2 = 1;
}

/* stop data logging into u16_array0/1 */
void dlog_2ch_u16_stop2(void)
{
	dlog_u32_status2 = 1;
}

/*
 * It logs data once.
 * Data log stops when idx reaches dlog size.
 * Call this function in the loop you want to log data.
 * Pass the data into this function as input parameters.
 * */
void dlog_2ch_u16_run2_once(uint16_t input2, uint16_t input3)
{
	if(dlog_u16_status2)
	{
		if(dlog_u16_idx2 < DLOG_U16_SIZE2)
		{
			dlog_u16_array2[dlog_u16_idx2] = input2;
			dlog_u16_array3[dlog_u16_idx2] = input3;
			dlog_u16_idx2++;
		}
		else
			dlog_u16_status2 = 0;
	}
}

/*
 * It logs data continuously.
 * When idx reaches dlog size, it starts over from 0 again.
 * Call this function in the loop you want to log data.
 * Pass the data into this function as input parameters.
 * */
void dlog_2ch_u16_run2_cont(uint16_t input2, uint16_t input3)
{
	if(dlog_u16_status2)
	{
		if(dlog_u16_idx2 < DLOG_U16_SIZE2)
		{
			dlog_u16_array2[dlog_u16_idx2] = input2;
			dlog_u16_array3[dlog_u16_idx2] = input3;
			dlog_u16_idx2++;
		}
		else
			dlog_u16_idx2 = 0;
	}
}

/**
 * \brief   Structure Init Function for 2 Channel float data log
 *
 * \param   v   dlog structure
 * \param   in_ptr1   pointer to input data1
 * \param   in_ptr2   pointer to input data2
 * \param   out_ptr1   pointer to output data1
 * \param   out_ptr2   pointer to output data2
 * \param   prev_value   previous data value
 * \param   trig_value   when input data1 is higher than trig_value, dlog is started
 * \param   nPrescalar   when nPrescaler is 4, one data is logged every 4 loops
 * \param   nSize   number of data in one data array
 *
 * \return  None
 */
void DLOG_2CH_F_init(DLOG_2CH_F *v,
		float32_t *in_ptr1, float32_t *in_ptr2,
		float32_t *out_ptr1, float32_t *out_ptr2,
		float32_t prev_value, float32_t trig_value,
		uint32_t nPrescalar, uint32_t nSize)
{
	v->input_ptr1=in_ptr1;
	v->input_ptr2=in_ptr2;

	v->output_ptr1=out_ptr1;
	v->output_ptr2=out_ptr1;

	v->prev_value=prev_value;
	v->trig_value=trig_value;
	v->status=0;
	v->pre_scalar=nPrescalar;
	v->skip_count=0;
	v->size=nSize;
	v->count=0;
}

/**
 * \brief   startTrigger / start / stop datalog
 *
 * \param   v   dlog structure
 * \param   nStatus = 0, stop dlog;
 *                    1, dlog waiting for trigger;
 *                    2, dlog start without trigger;
 *
 * \return  None
 */
/*
 * nStatus = 0, stop dlog;
 * nStatus = 1, dlog waiting for trigger;
 * nStatus = 2, dlog start without trigger;
 * */
void DLOG_2CH_F_start(DLOG_2CH_F *v, uint32_t nStatus)
{
	v->status=nStatus;
}

//*********** Function Definition ********//
void DLOG_2CH_F_FUNC(DLOG_2CH_F *v)
{
	if(v->status == 1)
	{
		if(*v->input_ptr1>v->trig_value && v->prev_value<v->trig_value)
			v->status = 2;
	}
	else if (v->status == 2)
	{
		v->skip_count++;

		if(v->skip_count==v->pre_scalar)
		{
			v->skip_count=0;
			v->output_ptr1[v->count]=*v->input_ptr1;
			v->output_ptr2[v->count]=*v->input_ptr2;
			v->count++;

			if(v->count==v->size)
			{
				v->count=0;
				v->status=1;
			}
		}
	}

	v->prev_value=*v->input_ptr1;
}

#ifdef __cplusplus
}
#endif

