/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 * history
 * 20240627,WenGuangYong,add ADC_setVREF() for 1.2CS     
 * 20240913,WenGuangYong,remov init config(already done in asysctl) in  ADC_setVREF(),avoid user recall it with same param,
 * 20241210,Zhaolei, ADC_setVREF ASysCtl_setAdcSpare bug fix
 */

#include "gs32_version.h"
#include "adc.h"

/*
 * @brief The address where the offset calibration is stored in OTP.
 *
 * Can be passed in the trigger parameter to the following functions:
 * 			ADC_setVREF(),
 * 			ADC_setOffsetTrimAll(),
 * 			ADC_setOffsetTrim().
 */
#define ADC_OFFSET_TRIM_OTP                     0x7016CU

#define ADC_VOLTAGE_REF_REG_OFFSET              8U

/*
 * @brief The following macro calculates the INL trim location
 * in OTP memory required to calibrate the ADC linearity.
 */
#define ADC_getINLTrimOTPLoc(offset) ((uint32_t *)(0x70160U + (0x4U * offset)))

/* TI-OTP key value expected to be programmed in trimmed device */
#define TI_OTP_DEV_KEY                          (0x5A5AU)

/* Macro to read the key value programmed in the device */
#define TI_OTP_DEV_PRG_KEY                      (HWREGH(0x7026EUL))

#if IS_GS32F00xx(0x12)

/* ADC_setVREF */
GS32_DRIVER_INIT_FUNC_T void
ADC_setVREF(uint32_t base, ADC_ReferenceMode refMode, ADC_ReferenceVoltage refVoltage)
{
    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Configure the reference mode (internal or external). */
    if(refMode == ADC_REFERENCE_EXTERNAL)
    {
        HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) &=
            ~(ASYSCTL_ANAREFCTL_ANAREFSEL);
        HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) |=
			ASYSCTL_ANAREFCTL_ANAREF2P5SEL;

        if(base == ADCA_BASE)
        {
        	/* bit21:24 VCM, bit10:13 VREFHI */
        	ASysCtl_setAdcSpare(0, 0x06621757);
        }
        else if(base == ADCB_BASE)
        {
        	ASysCtl_setAdcSpare(1, 0x06621757);
        }
        else if(base == ADCC_BASE)
        {
        	ASysCtl_setAdcSpare(2, 0x06621757);
        }
        else
        {

		}
    }
    else  /* internal ref mode */
    {
        HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) |=
            ASYSCTL_ANAREFCTL_ANAREFSEL;

        /* Select the internal reference voltage (3.3V or 2.5V). */
        if(refVoltage == ADC_REFERENCE_3_3V)
        {
            HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) &=
                ~(ASYSCTL_ANAREFCTL_ANAREF2P5SEL);

            if(base == ADCA_BASE)
            {
            	/*bit21:24 VCM, bit10:13 VREFHI*/
            	ASysCtl_setAdcSpare(0, 0x1662D717);
            }
            else if(base == ADCB_BASE)
            {
            	HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_SPARE) |= 1<<14;
            	ASysCtl_setAdcSpare(1, 0x1662D717);
            }
            else if(base == ADCC_BASE)
            {
            	HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_SPARE) |= 1<<14;
            	ASysCtl_setAdcSpare(2, 0x1662D717);
            }
            else
            {}
        }
        else  /* internal 2.5V reference */
        {
            HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) |=
                ASYSCTL_ANAREFCTL_ANAREF2P5SEL;

            if(base == ADCA_BASE)
            {
            	/*bit21:24 VCM, bit10:13 VREFHI*/
            	ASysCtl_setAdcSpare(0, 0x1AE1DF57);
            }
            else if(base == ADCB_BASE)
            {
            	HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_SPARE) |= 1<<14;
            	ASysCtl_setAdcSpare(1, 0x1AE1DF57);
            }
            else if(base == ADCC_BASE)
            {
            	HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_SPARE) |= 1<<14;
            	ASysCtl_setAdcSpare(2, 0x1AE1DF57);
            }
            else
            {}
        }

    }

    /* configure all trim values */
    AsysCtl_trimAdc(base, refMode, refVoltage);
}

#elif IS_GS32F3xx(0x22)

/* ADC_setVREF */
GS32_DRIVER_INIT_FUNC_T void
ADC_setVREF(uint32_t base, ADC_ReferenceMode refMode, ADC_ReferenceVoltage refVoltage)
{
    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Configure the reference mode (internal or external). */
    if(refMode == ADC_REFERENCE_EXTERNAL)
    {

       	switch (base)
		{
			case ADCA_BASE:
				HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) &=~(ASYSCTL_ANAREFCTL_ANAREFASEL);
				HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) |= (ASYSCTL_ANAREFCTL_ANAREFA2P5SEL);
				break;
			case ADCB_BASE:
				HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) &=~(ASYSCTL_ANAREFCTL_ANAREFBSEL);
				HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) |= (ASYSCTL_ANAREFCTL_ANAREFB2P5SEL);
				break;
			case ADCC_BASE:
				HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) &=~(ASYSCTL_ANAREFCTL_ANAREFCSEL);
				HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) |= (ASYSCTL_ANAREFCTL_ANAREFC2P5SEL);
				break;
			case ADCD_BASE:
				HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) &=~(ASYSCTL_ANAREFCTL_ANAREFDSEL);
				HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) |= (ASYSCTL_ANAREFCTL_ANAREFD2P5SEL);
				break;
		}

    }
    else  /* internal ref mode */
    {
    	switch (base)
    	{
    		case ADCA_BASE:
    			HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) |= ASYSCTL_ANAREFCTL_ANAREFASEL;break;
    		case ADCB_BASE:
    			HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) |= ASYSCTL_ANAREFCTL_ANAREFBSEL;break;
    		case ADCC_BASE:
    			HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) |= ASYSCTL_ANAREFCTL_ANAREFCSEL;break;
    		case ADCD_BASE:
    			HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) |= ASYSCTL_ANAREFCTL_ANAREFDSEL;break;
    	}
        /* Select the internal reference voltage (3.3V or 2.5V). */
        if(refVoltage == ADC_REFERENCE_3_3V)  //internal 1.65V reference
        {
           	switch (base)
			{
				case ADCA_BASE:
					HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) &=~(ASYSCTL_ANAREFCTL_ANAREFA2P5SEL);
					ASysCtl_setAdcaSpare(0x1AE1DC17);
					/* TODO internal 3.3  ADCA TOP SPARE */
					break;
				case ADCB_BASE:
					HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) &=~(ASYSCTL_ANAREFCTL_ANAREFB2P5SEL);
					ASysCtl_setAdcbSpare(0x1AE1DC17);
					/* TODO internal 3.3  ADCB TOP SPARE */
					break;
				case ADCC_BASE:
					HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) &=~(ASYSCTL_ANAREFCTL_ANAREFC2P5SEL);
					ASysCtl_setAdccSpare(0x1AE1DC17);
					/* TODO internal 3.3  ADCC TOP SPARE */
					break;
				case ADCD_BASE:
					HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) &=~(ASYSCTL_ANAREFCTL_ANAREFD2P5SEL);
					ASysCtl_setAdcdSpare(0x1AE1DC17);
					/* TODO internal 3.3  ADCD TOP SPARE */
					break;
			}
        }
        else  //internal 2.5V reference
        {
           	switch (base)
			{
				case ADCA_BASE:
					HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) |= (ASYSCTL_ANAREFCTL_ANAREFA2P5SEL);
					ASysCtl_setAdcaSpare(0x1AE1DC57);
					break;
				case ADCB_BASE:
					HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) |= (ASYSCTL_ANAREFCTL_ANAREFB2P5SEL);
					ASysCtl_setAdcbSpare(0x1AE1DC57);
					break;
				case ADCC_BASE:
					HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) |= (ASYSCTL_ANAREFCTL_ANAREFC2P5SEL);
					ASysCtl_setAdccSpare(0x1AE1DC57);
					break;
				case ADCD_BASE:
					HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) |= (ASYSCTL_ANAREFCTL_ANAREFD2P5SEL);
					ASysCtl_setAdcdSpare(0x1AE1DC57);
					break;
			}
        }

    }

    /* configure all trim values */
    AsysCtl_trimAdc(base, refMode, refVoltage);
}

#elif IS_GS32F00xx(0x30)

/* ADC_setVREF */
GS32_DRIVER_INIT_FUNC_T void
ADC_setVREF(uint32_t base, ADC_ReferenceMode refMode, ADC_ReferenceVoltage refVoltage)
{
    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Configure the reference mode (internal or external). */
    if(refMode == ADC_REFERENCE_EXTERNAL)
    {
    	/* Analog reference mode select an external reference mode*/
        HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) &=
            ~(ANA_SYSCTRL_PARA_ANAREFCTL_ANAREFSEL);
        HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) |=
        		ANA_SYSCTRL_PARA_ANAREFCTL_ANAREF2P5SEL;

        HWREG(0x400D6408) = 0x1EE39C57U;//exterior 3.3V VCM=1.65 and VRIFHI=3.3
        HWREG(0x400D6410) = 0x1EE00057U;

    }
    else  /* internal ref mode */
    {
    	/* Analog reference 2.5V source select an Internal 1.65V reference mode */
        HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) |=
        		ANA_SYSCTRL_PARA_ANAREFCTL_ANAREFSEL;

        /* Select the internal reference voltage (3.3V or 2.5V). */
        if(refVoltage == ADC_REFERENCE_3_3V)
        {
            HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) &=
                ~(ANA_SYSCTRL_PARA_ANAREFCTL_ANAREF2P5SEL);

			HWREG(0x400D6408) = 0x1EE3DC17U;//interior 3.3V VCM=2.475 and VRIFHI=1.65
			HWREG(0x400D6410) = 0x1EE00017U;//interior 3.3V VCM=2.475 and VRIFHI=1.65

        }
        else  /* internal 2.5V reference */
        {
        	/* Analog reference 2.5V source select an Internal 2.5V reference mode */
            HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) |=
            		ANA_SYSCTRL_PARA_ANAREFCTL_ANAREF2P5SEL;

            HWREG(0x400D6408) = 0x1EE3DC57U;//interior or exterior 2.5V VCM=1.25 and VRIFHI=2.5
            HWREG(0x400D6410) = 0x1EE00057U;//interior or exterior 2.5V VCM=1.25 and VRIFHI=2.5
        }

    }



    /* ADC calibration */
    AsysCtl_trimAdc(base,refMode,refVoltage);

}
#endif

/*
 * @brief ADC_setINLTrim
 *
 * Set correction.
 */
GS32_DRIVER_INIT_FUNC_T void
ADC_setINLTrim(uint32_t base)
{
	ASSERT(ADC_isBaseValid(base));

	HWREG(base + ADC_O_INLTRIM1 ) = ADC_TRIM_VALUE;
}

/*
 * @brief ADC_setPPBTripLimits
 *
 * Set PPB modules's Limit.
 */
GS32_DRIVER_ADC_FUNC_T void
ADC_setPPBTripLimits(uint32_t base, ADC_PPBNumber ppbNumber,
                     int32_t tripHiLimit, int32_t tripLoLimit)
{
    uint32_t ppbHiOffset;
    uint32_t ppbLoOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));
    ASSERT((tripHiLimit <= 65535) && (tripHiLimit >= -65536));
    ASSERT((tripLoLimit <= 65535) && (tripLoLimit >= -65536));

    /* Get the offset to the appropriate trip limit registers. */
    ppbHiOffset = (ADC_PPBxTRIPHI_STEP * (uint32_t)ppbNumber) +
                  ADC_O_PPB1TRIPHI;
    ppbLoOffset = (ADC_PPBxTRIPLO_STEP * (uint32_t)ppbNumber) +
                  ADC_O_PPB1TRIPLO;

    /* Set the trip high limit. */
    HWREG(base + ppbHiOffset) =
        (HWREG(base + ppbHiOffset) & ~ADC_PPBTRIP_MASK) |
        ((uint32_t)tripHiLimit & ADC_PPBTRIP_MASK);

    /* Set the trip low limit. */
    HWREG(base + ppbLoOffset) =
        (HWREG(base + ppbLoOffset) & ~ADC_PPBTRIP_MASK) |
        ((uint32_t)tripLoLimit & ADC_PPBTRIP_MASK);

}

/*
 * @brief ADC_configureRepeater
 *
 * configure Repeater module.
 */
GS32_DRIVER_ADC_FUNC_T void
ADC_configureRepeater(uint32_t base, uint16_t repInstance,
                      ADC_RepeaterConfig *config)
{
    uint32_t regOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));
    ASSERT(repInstance <= 2U);
    ASSERT(config->repCount <= 127U);

    regOffset = base + (repInstance * (ADC_REPxCTL_STEP));

    /* Configure repeater mode, trigger and syncin source. */
    HWREG(regOffset + ADC_O_REP1CTL) = (HWREG(regOffset + ADC_O_REP1CTL) &
                     ~((uint32_t)REPCTL_MASK))  | (((uint32_t)config->repMode) |
                      ((uint32_t)config->repTrigger << ADC_REP1CTL_TRIGGER_S) |
                      ((uint32_t)config->repSyncin << ADC_REP1CTL_SYNCINSEL_S));


    /* Configure repeater count. */
    HWREGH(regOffset + ADC_O_REP1N) = (HWREGH(regOffset + ADC_O_REP1N) &
                                      ~(ADC_REP1N_NSEL_M)) | config->repCount;


    /* Configure repeater phase. */
    HWREGH(regOffset + ADC_O_REP1PHASE) = (HWREGH(regOffset + ADC_O_REP1PHASE) &
                                          ~(ADC_REP1PHASE_PHASE_M)) |
                                          config->repPhase;

    /* Configure repeater spread. */
    HWREGH(regOffset + ADC_O_REP1SPREAD) =
                                      (HWREGH(regOffset + ADC_O_REP1SPREAD) &
                                      ~(ADC_REP1SPREAD_SPREAD_M)) |
                                      config->repSpread;

}
