/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
*  commit history
*  20240324, Jason, add external trim array for 2.0ES 3.3V VREF
*  20240423, ZhaoLei, Compatible with 3.0
*  20240530, WenGuangYong, Compatible with 3.0
*  20240531, WenGuangYong, add macro ADC_VERSION==0002,0003 on ADC_FORCE_SOC16-19\
*            and ADC_SOC_NUMBER16-19,plus SOC20 on ADC0003
*  20240603,WenGuangYong, 3.0 adc support mean group
*  20240704,WenGuangYong,fix ADC_forceSOC()
*  20240707,WenGuangYong,fix ADC_forceSOC() and ADC_forceMultipleSOC() to support SOC16/17/18/19
*  20240725,WenGuangYong,fix ADC_setInterruptSOCTrigger() on 32bit write
*                         add ADC_configOSDetectMode for supporting 2.2
*  20240813,WenGuangYong, update NSEL Width from 0x3ff To 0xFFFF
*  20240909,WenGuangYong, add read Tsensor temperure api 
*  2024/09/02,WenGuangYong,fix soc20 mask
*  20240922,WenGuangYong,add macro to restrict soc option,1.2 not available on soc 16-19
*/

#ifndef DEVICE_DRIVERLIB_ADC_H
#define DEVICE_DRIVERLIB_ADC_H

/*
 * @brief If building with a C++ compiler, make all of the definitions
 * 		  in this header have a C binding.
 */
#ifdef __cplusplus
extern "C"
{
#endif

#include <stdbool.h>

#include "gs32_version.h"
#include "inc/hw_adc.h"
#include "inc/hw_types.h"
#include "inc/hw_memmap.h"
#include "sysctl.h"

/*
 * @brief Useful defines used within the driver functions.
 * @note Not intended for use by application code.
 */
#define ADC_NUM_INTERRUPTS          4U

#define ADC_SOCxCTL_MASK			0xF7FF81FFU

#define ADC_SOCxCTL_OFFSET_BASE     ADC_O_SOC0CTL
#define ADC_RESULTx_OFFSET_BASE     ADC_O_RESULT0
#define ADC_INTSELxNy_OFFSET_BASE   ADC_O_INTSEL1N2
#define ADC_PPBxRESULT_OFFSET_BASE  ADC_O_PPB1RESULT

#define ADC_PPBxCONFIG_STEP         (ADC_O_PPB2CONFIG - ADC_O_PPB1CONFIG)
#define ADC_PPBxTRIPHI_STEP         (ADC_O_PPB2TRIPHI - ADC_O_PPB1TRIPHI)
#define ADC_PPBxTRIPLO_STEP         (ADC_O_PPB2TRIPLO - ADC_O_PPB1TRIPLO)
#define ADC_PPBxSTAMP_STEP          (ADC_O_PPB2STAMP - ADC_O_PPB1STAMP)
#define ADC_PPBxOFFCAL_STEP         (ADC_O_PPB2OFFCAL - ADC_O_PPB1OFFCAL)
#define ADC_PPBxOFFREF_STEP         (ADC_O_PPB2OFFREF - ADC_O_PPB1OFFREF)

#if (ADC_VERSION == 0002) || (ADC_VERSION == 0003)
/*
 * @brief Define to mask out the bits in the REPxCTL register
 * 		  that aren't associated with repeater configuration.
 */
#define REPCTL_MASK                 (ADC_REP1CTL_MODE                        |\
                                     ADC_REP1CTL_TRIGGER_M                   |\
                                     ADC_REP1CTL_SYNCINSEL_M)

#define ADC_PPBxPSUM_OFFSET_BASE     ADC_O_PPB1PSUM
#define ADC_PPBxPCOUNT_OFFSET_BASE   ADC_O_PPBP1COUNT
#define ADC_PPBxPMAX_OFFSET_BASE     ADC_O_PPB1PMAX
#define ADC_PPBxPMIN_OFFSET_BASE     ADC_O_PPB1PMIN
#define ADC_PPBxPMAXI_OFFSET_BASE    ADC_O_PPB1PMAXI
#define ADC_PPBxPMINI_OFFSET_BASE    ADC_O_PPB1PMINI

#define ADC_PPBxLIMIT_OFFSET_BASE   ADC_O_PPB1LIMIT
#define ADC_PPBxSUM_OFFSET_BASE     ADC_O_PPB1SUM
#define ADC_PPBxCOUNT_OFFSET_BASE   ADC_O_PPB1COUNT
#define ADC_PPBxMAX_OFFSET_BASE     ADC_O_PPB1MAX
#define ADC_PPBxMIN_OFFSET_BASE     ADC_O_PPB1MIN
#define ADC_PPBxMAXI_OFFSET_BASE    ADC_O_PPB1MAXI
#define ADC_PPBxMINI_OFFSET_BASE    ADC_O_PPB1MINI

#define ADC_PPBxLIMIT_STEP          (ADC_O_PPB2LIMIT - ADC_O_PPB1LIMIT)
#define ADC_PPBxCONFIG2_STEP        (ADC_O_PPB2CONFIG2 - ADC_O_PPB1CONFIG2)
#define ADC_REPxCTL_STEP            (ADC_O_REP2CTL - ADC_O_REP1CTL)
#define ADC_REPxN_STEP              (ADC_O_REP2N - ADC_O_REP1N)
#define ADC_REPxPHASE_STEP          (ADC_O_REP2PHASE - ADC_O_REP1PHASE)
#define ADC_REPxSPREAD_STEP         (ADC_O_REP2SPREAD - ADC_O_REP1SPREAD)

#define ADC_PPBTRIP_MASK            (uint32_t)ADC_PPB1TRIPHI_LIMITHI_M
#define ADC_PPBxTRIPLO2_STEP        (ADC_O_PPB2TRIPLO2 - ADC_O_PPB1TRIPLO2)

#define ADC_INT_REF_TSSLOPE         (*(int16_t *)((uintptr_t)0x720C0))
#define ADC_INT_REF_TSOFFSET        (*(int16_t *)((uintptr_t)0x720C1))
#define ADC_EXT_REF_TSSLOPE         (*(int16_t *)((uintptr_t)0x720BE))
#define ADC_EXT_REF_TSOFFSET        (*(int16_t *)((uintptr_t)0x720BF))

#elif ADC_VERSION == 0001
#define ADC_PPBTRIP_MASK            ((uint32_t)ADC_PPB1TRIPHI_LIMITHI_M      |\
                                     (uint32_t)ADC_PPB1TRIPHI_HSIGN)
#define ADC_INT_REF_TSSLOPE         (*(int16_t *)((uintptr_t)0x701CA))
#define ADC_INT_REF_TSOFFSET        (*(int16_t *)((uintptr_t)0x701CB))
#define ADC_EXT_REF_TSSLOPE         (*(int16_t *)((uintptr_t)0x701C8))
#define ADC_EXT_REF_TSOFFSET        (*(int16_t *)((uintptr_t)0x701C9))

#endif  //1.2EC

/*
 * @brief PPB EVENT
 *
 * Can be passed in the trigger parameter to the following functions:
 * 		ADC_enablePPBEvent(),
 * 		ADC_disablePPBEvent(),
 * 		ADC_enablePPBEventInterrupt(),
 * 		ADC_disablePPBEventInterrupt(),
 * 		ADC_clearPPBEventStatus().
 *
 * They also make up the enumerated bit field returned by
 * 		ADC_getPPBEventStatus().
 */
#define ADC_EVT_TRIPHI              0x0001U //!< Trip High Event
#define ADC_EVT_TRIPLO              0x0002U //!< Trip Low Event
#define ADC_EVT_ZERO                0x0004U //!< Zero Crossing Event

/*
 * @brief The number of SOC to force trigger.
 *
 * @remarks
 * - Can be passed in the trigger parameter to the following functions:
 * 	 ADC_forceMultipleSOC().
 */
#define ADC_FORCE_SOC0              0x0001U 	/*SW trigger ADC SOC 0   */
#define ADC_FORCE_SOC1              0x0002U 	/*SW trigger ADC SOC 1   */
#define ADC_FORCE_SOC2              0x0004U 	/*SW trigger ADC SOC 2   */
#define ADC_FORCE_SOC3              0x0008U 	/*SW trigger ADC SOC 3   */
#define ADC_FORCE_SOC4              0x0010U 	/*SW trigger ADC SOC 4   */
#define ADC_FORCE_SOC5              0x0020U 	/*SW trigger ADC SOC 5   */
#define ADC_FORCE_SOC6              0x0040U 	/*SW trigger ADC SOC 6   */
#define ADC_FORCE_SOC7              0x0080U 	/*SW trigger ADC SOC 7   */
#define ADC_FORCE_SOC8              0x0100U 	/*SW trigger ADC SOC 8   */
#define ADC_FORCE_SOC9              0x0200U 	/*SW trigger ADC SOC 9   */
#define ADC_FORCE_SOC10             0x0400U 	/*SW trigger ADC SOC 10  */
#define ADC_FORCE_SOC11             0x0800U 	/*SW trigger ADC SOC 11  */
#define ADC_FORCE_SOC12             0x1000U 	/*SW trigger ADC SOC 12  */
#define ADC_FORCE_SOC13             0x2000U 	/*SW trigger ADC SOC 13  */
#define ADC_FORCE_SOC14             0x4000U 	/*SW trigger ADC SOC 14  */
#define ADC_FORCE_SOC15             0x8000U 	/*SW trigger ADC SOC 15  */
#if(IS_GS32F00xx(0x30))
#define ADC_FORCE_SOC16          	0x00010000U /* SW trigger ADC SOC 16 */
#define ADC_FORCE_SOC17          	0x00020000U /* SW trigger ADC SOC 17 */
#define ADC_FORCE_SOC18          	0x00040000U /* SW trigger ADC SOC 18 */
#define ADC_FORCE_SOC19          	0x00080000U /* SW trigger ADC SOC 19 */
#define ADC_FORCE_SOC20          	0x00100000U /* SW trigger ADC SOC 20 */
#endif

#if (ADC_VERSION == 0002) || (ADC_VERSION == 0003)
/*
 * @brief Defines to specify ADC Trigger Repeater block status
 */
#define ADC_REPEATER_MODULE_BUSY      ADC_REP1CTL_MODULEBUSY
#define ADC_REPEATER_PHASE_OVERFLOW   ADC_REP1CTL_PHASEOVF
#define ADC_REPEATER_TRIGGER_OVERFLOW ADC_REP1CTL_TRIGGEROVF
#define ADC_REPEATER_OVERFLOW	(ADC_REPEATER_PHASE_OVERFLOW |\
								 ADC_REPEATER_TRIGGER_OVERFLOW)

/*
 * @brief Defines for the bit fields in the REPxCTL register
 */
#define ADC_REP1CTL_ACTIVEMODE_S      1U
#define ADC_REP1CTL_MODULEBUSY_S      3U

/*
 * @brief Define to mask out the bits in the REPSTATUS register
 * that aren't associated with trigger repeater block status.
 */
#define ADC_REPSTATUS_MASK          (ADC_REP1CTL_MODULEBUSY                  |\
                                     ADC_REP1CTL_PHASEOVF                    |\
                                     ADC_REP1CTL_TRIGGEROVF)

/*
 * @brief Defines to specify ADC Safety Checker result status
 */
#define ADC_SAFECHECK_RESULT1_READY ADC_CHECKSTATUS_RES1READY
#define ADC_SAFECHECK_RESULT2_READY ADC_CHECKSTATUS_RES2READY
#define ADC_SAFECHECK_RESULT_OOT    ADC_CHECKSTATUS_OOT

/*
 * @brief
 * Define to mask out the bits in the CHECKSTATUS register that aren't
 * associated with safety checker result status.
 */
#define ADC_SAFECHECK_STATUS_MASK   (ADC_SAFECHECK_RESULT1_READY             |\
                                     ADC_SAFECHECK_RESULT2_READY             |\
                                     ADC_SAFECHECK_RESULT_OOT)
#endif //1.2EC

/*
 * @brief
 * Values that can be passed to ADC_setPrescaler() as the clkPrescale parameter.
 */
typedef enum
{
	ADC_CLK_DIV_1_0 = 0U,			/* ADCCLK = (input clock) / 1.0 */
    ADC_CLK_DIV_2_0 = 2U,			/* ADCCLK = (input clock) / 2.0 */
    ADC_CLK_DIV_3_0 = 4U,			/* ADCCLK = (input clock) / 3.0 */
    ADC_CLK_DIV_4_0 = 6U,			/* ADCCLK = (input clock) / 4.0 */
    ADC_CLK_DIV_5_0 = 8U,			/* ADCCLK = (input clock) / 5.0 */
    ADC_CLK_DIV_6_0 = 10U,			/* ADCCLK = (input clock) / 6.0 */
    ADC_CLK_DIV_7_0 = 12U,			/* ADCCLK = (input clock) / 7.0 */
	ADC_CLK_DIV_8_0 = 14U,			/* ADCCLK = (input clock) / 8.0 */
} ADC_ClkPrescale;

/*
 * @brief The number of the trigger source.
 *
 * Can be passed in the trigger parameter to the following functions:
 * 		ADC_setupSOC(),
 * 		ADC_setBurstModeConfig(),
 * 		ADC_triggerRepeaterSelect(),
 */
#if (IS_GS32F00xx(0x12)||IS_GS32F3xx(0x22))
typedef enum
{
	ADC_TRIGGER_SW_ONLY     = 0U,	/* Software only            */
	ADC_TRIGGER_CPU1_TINT0  = 1U,   /* CPU1 Timer 0, TINT0      */
	ADC_TRIGGER_CPU1_TINT1  = 2U,   /* CPU1 Timer 1, TINT1      */
	ADC_TRIGGER_CPU1_TINT2  = 3U,   /* CPU1 Timer 2, TINT2      */
	ADC_TRIGGER_GPIO        = 4U,   /* GPIO, Input X-Bar INPUT5 */
	ADC_TRIGGER_EPWM1_SOCA  = 5U,   /* ePWM1, ADCSOCA           */
	ADC_TRIGGER_EPWM1_SOCB  = 6U,   /* ePWM1, ADCSOCB           */
	ADC_TRIGGER_EPWM2_SOCA  = 7U,   /* ePWM2, ADCSOCA           */
	ADC_TRIGGER_EPWM2_SOCB  = 8U,   /* ePWM2, ADCSOCB           */
	ADC_TRIGGER_EPWM3_SOCA  = 9U,   /* ePWM3, ADCSOCA           */
	ADC_TRIGGER_EPWM3_SOCB  = 10U,  /* ePWM3, ADCSOCB           */
	ADC_TRIGGER_EPWM4_SOCA  = 11U,  /* ePWM4, ADCSOCA           */
	ADC_TRIGGER_EPWM4_SOCB  = 12U,  /* ePWM4, ADCSOCB           */
	ADC_TRIGGER_EPWM5_SOCA  = 13U,  /* ePWM5, ADCSOCA           */
	ADC_TRIGGER_EPWM5_SOCB  = 14U,  /* ePWM5, ADCSOCB           */
	ADC_TRIGGER_EPWM6_SOCA  = 15U,  /* ePWM6, ADCSOCA           */
	ADC_TRIGGER_EPWM6_SOCB  = 16U,  /* ePWM6, ADCSOCB           */
	ADC_TRIGGER_EPWM7_SOCA  = 17U,  /* ePWM7, ADCSOCA           */
	ADC_TRIGGER_EPWM7_SOCB  = 18U,  /* ePWM7, ADCSOCB           */
	ADC_TRIGGER_EPWM8_SOCA  = 19U,  /* ePWM8, ADCSOCA           */
	ADC_TRIGGER_EPWM8_SOCB  = 20U,  /* ePWM8, ADCSOCB           */
	ADC_TRIGGER_EPWM9_SOCA  = 21U,  /* ePWM9, ADCSOCA           */
	ADC_TRIGGER_EPWM9_SOCB  = 22U,  /* ePWM9, ADCSOCB           */
	ADC_TRIGGER_EPWM10_SOCA = 23U,  /* ePWM10, ADCSOCA          */
	ADC_TRIGGER_EPWM10_SOCB = 24U,  /* ePWM10, ADCSOCB          */
	ADC_TRIGGER_EPWM11_SOCA = 25U,  /* ePWM11, ADCSOCA          */
	ADC_TRIGGER_EPWM11_SOCB = 26U,  /* ePWM11, ADCSOCB          */
	ADC_TRIGGER_EPWM12_SOCA = 27U,  /* ePWM12, ADCSOCA          */
    ADC_TRIGGER_EPWM12_SOCB = 28U,  /* ePWM12, ADCSOCB          */
	ADC_TRIGGER_REPEATER1	= 40U,	/* REP1TRIG                 */
	ADC_TRIGGER_REPEATER2	= 41U,	/* REP2TRIG                 */
	ADC_TRIGGER_CPU1_TINT3  = 42U,  /* CPU1 Timer 3, TINT3      */
#if IS_GS32F3xx(0x22)
	ADC_TRIGGER_CPU1_TINT4  = 46U,  /* CPU1 Timer 4, TINT4      */
	ADC_TRIGGER_CPU1_TINT5  = 47U,  /* CPU1 Timer 5, TINT5      */
	ADC_TRIGGER_EPWM13_SOCA = 88U,  /* ePWM13, ADCSOCA          */
	ADC_TRIGGER_EPWM13_SOCB = 89U,  /* ePWM13, ADCSOCB          */
	ADC_TRIGGER_EPWM14_SOCA = 90U,  /* ePWM14, ADCSOCA          */
	ADC_TRIGGER_EPWM14_SOCB = 91U,  /* ePWM14, ADCSOCB          */
	ADC_TRIGGER_EPWM15_SOCA = 92U,  /* ePWM15, ADCSOCA          */
	ADC_TRIGGER_EPWM15_SOCB = 93U,  /* ePWM15, ADCSOCB          */
	ADC_TRIGGER_EPWM16_SOCA = 94U,  /* ePWM16, ADCSOCA          */
	ADC_TRIGGER_EPWM16_SOCB = 95U,  /* ePWM16, ADCSOCB          */
	ADC_TRIGGER_EPWM17_SOCA = 96U,  /* ePWM17, ADCSOCA          */
    ADC_TRIGGER_EPWM17_SOCB = 97U,  /* ePWM17, ADCSOCB          */
	ADC_TRIGGER_EPWM18_SOCA = 98U,  /* ePWM18, ADCSOCA          */
    ADC_TRIGGER_EPWM18_SOCB = 99U,  /* ePWM18, ADCSOCB          */
#endif

} ADC_Trigger;
#elif (IS_GS32F00xx(0x30))
typedef enum
{

	ADC_TRIGGER_SW_ONLY     = 0U,	/* Software only            */
	ADC_TRIGGER_CPU1_TINT0  = 1U,   /* CPU1 Timer 0, TINT0      */
	ADC_TRIGGER_CPU1_TINT1  = 2U,   /* CPU1 Timer 1, TINT1      */
	ADC_TRIGGER_GPIO        = 4U,   /* GPIO, Input X-Bar INPUT5 */
	ADC_TRIGGER_EPWM1_SOCA  = 5U,   /* ePWM1, ADCSOCA           */
	ADC_TRIGGER_EPWM1_SOCB  = 6U,   /* ePWM1, ADCSOCB           */
	ADC_TRIGGER_EPWM2_SOCA  = 7U,   /* ePWM2, ADCSOCA           */
	ADC_TRIGGER_EPWM2_SOCB  = 8U,   /* ePWM2, ADCSOCB           */
	ADC_TRIGGER_EPWM3_SOCA  = 9U,   /* ePWM3, ADCSOCA           */
	ADC_TRIGGER_EPWM3_SOCB  = 10U,  /* ePWM3, ADCSOCB           */
	ADC_TRIGGER_EPWM4_SOCA  = 11U,  /* ePWM4, ADCSOCA           */
	ADC_TRIGGER_EPWM4_SOCB  = 12U,  /* ePWM4, ADCSOCB           */
	ADC_TRIGGER_EPWM5_SOCA  = 13U,  /* ePWM5, ADCSOCA           */
	ADC_TRIGGER_EPWM5_SOCB  = 14U,  /* ePWM5, ADCSOCB           */
	ADC_TRIGGER_EPWM6_SOCA  = 15U,  /* ePWM6, ADCSOCA           */
	ADC_TRIGGER_EPWM6_SOCB  = 16U,  /* ePWM6, ADCSOCB           */
	ADC_TRIGGER_EPWM7_SOCA  = 17U,  /* ePWM7, ADCSOCA           */
	ADC_TRIGGER_EPWM7_SOCB  = 18U,  /* ePWM7, ADCSOCB           */
	ADC_TRIGGER_EPWM8_SOCA  = 19U,  /* ePWM8, ADCSOCA           */
	ADC_TRIGGER_EPWM8_SOCB  = 20U,  /* ePWM8, ADCSOCB           */
	ADC_TRIGGER_REPEATER1	= 40U,	/* REP1TRIG                 */
	ADC_TRIGGER_REPEATER2	= 41U,	/* REP2TRIG                 */
	ADC_TRIGGER_CPU1_TINT3  = 42U,  /* CPU1 Timer 3, TINT2n     */
	ADC_TRIGGER_DACB_TIMER	= 44U,	/* DACB Timer				*/
	ADC_TRIGGER_ECAP1		= 80U,	/* eCAP1					*/
	ADC_TRIGGER_ECAP2		= 81U,	/* eCAP2					*/
	ADC_TRIGGER_ECAP3		= 82U,	/* eCAP3					*/

} ADC_Trigger;
#endif

/*
 * @brief The number of The input on which the signal to be converted is located.
 *
 * Can be passed in the trigger parameter to the following functions:
 * 		ADC_setupSOC().
 */
typedef enum
{
    ADC_CH_ADCIN0  = 0U,          /* ADCIN0 is converted  */
    ADC_CH_ADCIN1  = 1U,          /* ADCIN1 is converted  */
    ADC_CH_ADCIN2  = 2U,          /* ADCIN2 is converted  */
    ADC_CH_ADCIN3  = 3U,          /* ADCIN3 is converted  */
    ADC_CH_ADCIN4  = 4U,          /* ADCIN4 is converted  */
    ADC_CH_ADCIN5  = 5U,          /* ADCIN5 is converted  */
    ADC_CH_ADCIN6  = 6U,          /* ADCIN6 is converted  */
    ADC_CH_ADCIN7  = 7U,          /* ADCIN7 is converted  */
    ADC_CH_ADCIN8  = 8U,          /* ADCIN8 is converted  */
    ADC_CH_ADCIN9  = 9U,          /* ADCIN9 is converted  */
    ADC_CH_ADCIN10 = 10U,         /* ADCIN10 is converted */
    ADC_CH_ADCIN11 = 11U,         /* ADCIN11 is converted */
    ADC_CH_ADCIN12 = 12U,         /* ADCIN12 is converted */
    ADC_CH_ADCIN13 = 13U,         /* ADCIN13 is converted */
    ADC_CH_ADCIN14 = 14U,         /* ADCIN14 is converted */
    ADC_CH_ADCIN15 = 15U,         /* ADCIN15 is converted */
#if (IS_GS32F00xx(0x30))
    ADC_CH_ADCIN16 = 16U,         /* ADCIN16 is converted */
    ADC_CH_ADCIN17 = 17U,         /* ADCIN17 is converted */
    ADC_CH_ADCIN18 = 18U,         /* ADCIN18 is converted */
    ADC_CH_ADCIN19 = 19U,         /* ADCIN19 is converted */
    ADC_CH_ADCIN20 = 20U,         /* ADCIN20 is converted */
#endif
    ADC_CH_ADCIN_MAX ,
} ADC_Channel;

/*
 * @brief The number of ADC Interrupt Pulse Position.
 *
 * Can be passed in the trigger parameter to the following functions:
 * 		ADC_setInterruptPulseMode().
 */
typedef enum
{
    /* Occurs at the end of the acquisition window */
    ADC_PULSE_END_OF_ACQ_WIN = 0x00U,
    /* Occurs at the end of the conversion */
    ADC_PULSE_END_OF_CONV    = 0x04U
} ADC_PulseMode;

/*
 * @brief Number of ADC Interrupt.
 *
 * Can be passed in the trigger parameter to the following functions:
 */
typedef enum
{
    ADC_INT_NUMBER1 = 0U,        /* ADCINT1 Interrupt */
    ADC_INT_NUMBER2 = 1U,        /* ADCINT2 Interrupt */
    ADC_INT_NUMBER3 = 2U,        /* ADCINT3 Interrupt */
    ADC_INT_NUMBER4 = 3U         /* ADCINT4 Interrupt */
} ADC_IntNumber;

/*
 * @brief The number of PPB module.
 *
 * Can be passed in the trigger parameter to the following functions:
 */

typedef enum
{
    ADC_PPB_NUMBER1 = 0U,        /* Post-processing block 1 */
    ADC_PPB_NUMBER2 = 1U,        /* Post-processing block 2 */
    ADC_PPB_NUMBER3 = 2U,        /* Post-processing block 3 */
    ADC_PPB_NUMBER4 = 3U         /* Post-processing block 4 */
} ADC_PPBNumber;

/*
 * @brief the number of the SOC or EOC.
 *
 * Can be passed in the socNumber parameter to the following functions:
 * 		ADC_setupSOC(),
 * 		ADC_setInterruptSOCTrigger(),
 * 		ADC_forceSOC(),
 * 		ADC_readResult(),
 * 		ADC_setupPPB(),
 * 		ADC_enableMeanGroup(),
 * 		ADC_disableMeanGroup(),
 * 		ADC_getEPPBResult(),
 * 		ADC_setEPPBScale(),
 * 		ADC_setEPPBOffset(),
 *
 */
typedef enum
{
    ADC_SOC_NUMBER0  = 0U,        /* SOC/EOC number 0  */
    ADC_SOC_NUMBER1  = 1U,        /* SOC/EOC number 1  */
    ADC_SOC_NUMBER2  = 2U,        /* SOC/EOC number 2  */
    ADC_SOC_NUMBER3  = 3U,        /* SOC/EOC number 3  */
    ADC_SOC_NUMBER4  = 4U,        /* SOC/EOC number 4  */
    ADC_SOC_NUMBER5  = 5U,        /* SOC/EOC number 5  */
    ADC_SOC_NUMBER6  = 6U,        /* SOC/EOC number 6  */
    ADC_SOC_NUMBER7  = 7U,        /* SOC/EOC number 7  */
    ADC_SOC_NUMBER8  = 8U,        /* SOC/EOC number 8  */
    ADC_SOC_NUMBER9  = 9U,        /* SOC/EOC number 9  */
    ADC_SOC_NUMBER10 = 10U,       /* SOC/EOC number 10 */
    ADC_SOC_NUMBER11 = 11U,       /* SOC/EOC number 11 */
    ADC_SOC_NUMBER12 = 12U,       /* SOC/EOC number 12 */
    ADC_SOC_NUMBER13 = 13U,       /* SOC/EOC number 13 */
    ADC_SOC_NUMBER14 = 14U,       /* SOC/EOC number 14 */
    ADC_SOC_NUMBER15 = 15U,       /* SOC/EOC number 15 */
#if (IS_GS32F00xx(0x30))
    ADC_SOC_NUMBER16 = 16U,       /* SOC number 16 */
    ADC_SOC_NUMBER17 = 17U,       /* SOC number 17 */
    ADC_SOC_NUMBER18 = 18U,       /* SOC number 18 */
    ADC_SOC_NUMBER19 = 19U,       /* SOC number 19 */
	ADC_SOC_NUMBER20 = 20U,		  /* SOC number 20 */
#endif
	ADC_SOC_NUMBER_MAX ,
} ADC_SOCNumber;

/*
 * @brief The interrupt trigger source of SOC.
 *
 * Can be passed in the socNumber parameter to the following functions:
 * 		ADC_setInterruptSOCTrigger();
 */
typedef enum
{
    ADC_INT_SOC_TRIGGER_NONE    = 0U,   /* No ADCINT will trigger the SOC */
    ADC_INT_SOC_TRIGGER_ADCINT1 = 1U,   /* ADCINT1 will trigger the SOC   */
    ADC_INT_SOC_TRIGGER_ADCINT2 = 2U    /* ADCINT2 will trigger the SOC   */
} ADC_IntSOCTrigger;

/*
 * @brief SOC Priority.
 *
 * Can be passed in the trigger parameter to the following functions:
 * 		ADC_setSOCPriority()
 */
typedef enum
{
    ADC_PRI_ALL_ROUND_ROBIN = 0U,    /* Round robin mode is used for all       */
    ADC_PRI_SOC0_HIPRI      = 1U,    /* SOC 0 hi pri, others in round robin    */
    ADC_PRI_THRU_SOC1_HIPRI = 2U,    /* SOC 0-1 hi pri, others in round robin  */
    ADC_PRI_THRU_SOC2_HIPRI = 3U,    /* SOC 0-2 hi pri, others in round robin  */
    ADC_PRI_THRU_SOC3_HIPRI = 4U,    /* SOC 0-3 hi pri, others in round robin  */
    ADC_PRI_THRU_SOC4_HIPRI = 5U,    /* SOC 0-4 hi pri, others in round robin  */
    ADC_PRI_THRU_SOC5_HIPRI = 6U,    /* SOC 0-5 hi pri, others in round robin  */
    ADC_PRI_THRU_SOC6_HIPRI = 7U,    /* SOC 0-6 hi pri, others in round robin  */
    ADC_PRI_THRU_SOC7_HIPRI = 8U,    /* SOC 0-7 hi pri, others in round robin  */
    ADC_PRI_THRU_SOC8_HIPRI = 9U,    /* SOC 0-8 hi pri, others in round robin  */
    ADC_PRI_THRU_SOC9_HIPRI = 10U,   /* SOC 0-9 hi pri, others in round robin  */
    ADC_PRI_THRU_SOC10_HIPRI = 11U,  /* SOC 0-10 hi pri, others in round robin */
    ADC_PRI_THRU_SOC11_HIPRI = 12U,  /* SOC 0-11 hi pri, others in round robin */
    ADC_PRI_THRU_SOC12_HIPRI = 13U,  /* SOC 0-12 hi pri, others in round robin */
    ADC_PRI_THRU_SOC13_HIPRI = 14U,  /* SOC 0-13 hi pri, others in round robin */
    ADC_PRI_THRU_SOC14_HIPRI = 15U,  /* SOC 0-14 hi pri, SOC15 in round robin  */
    ADC_PRI_ALL_HIPRI = 16U          /* All priorities based on SOC number     */
} ADC_PriorityMode;

/*
 * @brief Reference mode.
 *
 * Can be passed in the trigger parameter to the following functions:
 * 		ADC_getTemperatureC(),
 * 		ADC_getTemperatureK(),
 * 		ADC_setVREF(),
 * 		ADC_setOffsetTrimAll().
 */
typedef enum
{
    ADC_REFERENCE_INTERNAL = 0U,
    ADC_REFERENCE_EXTERNAL = 1U
} ADC_ReferenceMode;

/*
 * @brief  Reference voltage.
 *
 * Can be passed in the trigger parameter to the following functions:
 * 		ADC_setVREF(),
 * 		ADC_setOffsetTrimAll().
 */
typedef enum
{
    ADC_REFERENCE_3_3V = 0U,
	ADC_REFERENCE_2_5V = 1U,
	ADC_REFERENCE_3_0V = 2U
} ADC_ReferenceVoltage;

/*
 * @brief ADC Opens and Shorts Detect Configuration.
 *
 * Can be passed in the trigger parameter to the following functions:
 * 		ADC_configOSDetectMode().
 */
typedef enum
{
	/* Open/Shorts detection cir-cuit(O/S DC) is disabled */
    ADC_OSDETECT_MODE_DISABLED            = 0x0U,
    /* O/S DC is enabled at zero scale */
    ADC_OSDETECT_MODE_VSSA                = 0x1U,
    /* O/S DC is enabled at full scale */
    ADC_OSDETECT_MODE_VDDA                = 0x2U,
	/* O/S DC is enabled at 5/12 scale */
    ADC_OSDETECT_MODE_5BY12_VDDA          = 0x3U,
    /* O/S DC is enabled at 7/12 scale */
    ADC_OSDETECT_MODE_7BY12_VDDA          = 0x4U,
    /* O/S DC is enabled at 5K pulldown to VSSA */
    ADC_OSDETECT_MODE_5K_PULLDOWN_TO_VSSA = 0x5U,
    /* O/S DC is enabled at 5K pullup to VDDA */
    ADC_OSDETECT_MODE_5K_PULLUP_TO_VDDA   = 0x6U,
    /* O/S DC is enabled at 7K pulldown to VSSA */
    ADC_OSDETECT_MODE_7K_PULLDOWN_TO_VSSA = 0x7U
/* Providing a different specific load to detect circuit status,
 * different resistor values an provide different detection
 * sensitivities and load conditions in different electrical environments.
 */
} ADC_OSDetectMode;

#if (ADC_VERSION == 0002) || (ADC_VERSION == 0003)
/*
 * @brief  Repeater sync event.
 *
 * Can be passed in the socNumber parameter to the following functions:
 * 		ADC_selectPPBSyncInput(),
 * 		ADC_triggerRepeaterSyncIn().
 *
 */
typedef enum
{
    ADC_SYNCIN_DISABLE          = 0x00U,  /* ADC Syncin is disabled         */
    ADC_SYNCIN_EPWM1SYNCOUT     = 0x01U,  /* ADC Syncin is EPWM1SYNCOUT     */
    ADC_SYNCIN_EPWM2SYNCOUT     = 0x02U,  /* ADC Syncin is EPWM2SYNCOUT     */
    ADC_SYNCIN_EPWM3SYNCOUT     = 0x03U,  /* ADC Syncin is EPWM3SYNCOUT     */
    ADC_SYNCIN_EPWM4SYNCOUT     = 0x04U,  /* ADC Syncin is EPWM4SYNCOUT     */
    ADC_SYNCIN_EPWM5SYNCOUT     = 0x05U,  /* ADC Syncin is EPWM5SYNCOUT     */
    ADC_SYNCIN_EPWM6SYNCOUT     = 0x06U,  /* ADC Syncin is EPWM6SYNCOUT     */
    ADC_SYNCIN_EPWM7SYNCOUT     = 0x07U,  /* ADC Syncin is EPWM7SYNCOUT     */
    ADC_SYNCIN_EPWM8SYNCOUT     = 0x08U,  /* ADC Syncin is EPWM8SYNCOUT     */
    ADC_SYNCIN_EPWM9SYNCOUT     = 0x09U,  /* ADC Syncin is EPWM9SYNCOUT     */
    ADC_SYNCIN_EPWM10SYNCOUT    = 0x0AU,  /* ADC Syncin is EPWM10SYNCOUT    */
    ADC_SYNCIN_EPWM11SYNCOUT    = 0x0BU,  /* ADC Syncin is EPWM11SYNCOUT    */
    ADC_SYNCIN_EPWM12SYNCOUT    = 0x0CU,  /* ADC Syncin is EPWM12SYNCOUT    */
    ADC_SYNCIN_ECAP1YNCOUT      = 0x13U,  /* ADC Syncin is ECAP1YNCOUT      */
    ADC_SYNCIN_ECAP2SYNCOUT     = 0x14U,  /* ADC Syncin is ECAP2SYNCOUT     */
    ADC_SYNCIN_ECAP3SYNCOUT     = 0x15U,  /* ADC Syncin is ECAP3SYNCOUT     */
    ADC_SYNCIN_ECAP4SYNCOUT     = 0x16U,  /* ADC Syncin is ECAP4SYNCOUT     */
    ADC_SYNCIN_ECAP5SYNCOUT     = 0x17U,  /* ADC Syncin is ECAP5SYNCOUT     */
    ADC_SYNCIN_ECAP6SYNCOUT     = 0x18U,  /* ADC Syncin is ECAP6SYNCOUT     */
    ADC_SYNCIN_ECAP7SYNCOUT     = 0x19U,  /* ADC Syncin is ECAP7SYNCOUT     */
    ADC_SYNCIN_INPUTXBAROUTPUT5 = 0x1AU,  /* ADC Syncin is INPUTXBAROUTPUT5 */
    ADC_SYNCIN_INPUTXBAROUTPUT6 = 0x1BU,  /* ADC Syncin is INPUTXBAROUTPUT6 */
} ADC_SyncInput;

//PPB
/*
 * @brief OSINT mode.
 *
 * Can be passed in the socNumber parameter to the following functions:
 * 		ADC_selectPPBOSINTSource().
 */
typedef enum
{
    ADC_PPB_OS_INT_1 = 0x0U,          /* PCount generates PPB intterupt		 */
    ADC_PPB_OS_INT_2 = 0x1U,          /* PCount/Sync generates PPB intterupt */
} ADC_PPBIntSrcSelect;

/*
 * @brief EOC trigger source.
 */
typedef enum
{
    ADC_INT_TRIGGER_EOC0   = 0U,        /* SOC/EOC0  */
    ADC_INT_TRIGGER_EOC1   = 1U,        /* SOC/EOC1  */
    ADC_INT_TRIGGER_EOC2   = 2U,        /* SOC/EOC2  */
    ADC_INT_TRIGGER_EOC3   = 3U,        /* SOC/EOC3  */
    ADC_INT_TRIGGER_EOC4   = 4U,        /* SOC/EOC4  */
    ADC_INT_TRIGGER_EOC5   = 5U,        /* SOC/EOC5  */
    ADC_INT_TRIGGER_EOC6   = 6U,        /* SOC/EOC6  */
    ADC_INT_TRIGGER_EOC7   = 7U,        /* SOC/EOC7  */
    ADC_INT_TRIGGER_EOC8   = 8U,        /* SOC/EOC8  */
    ADC_INT_TRIGGER_EOC9   = 9U,        /* SOC/EOC9  */
    ADC_INT_TRIGGER_EOC10  = 10U,       /* SOC/EOC10 */
    ADC_INT_TRIGGER_EOC11  = 11U,       /* SOC/EOC11 */
    ADC_INT_TRIGGER_EOC12  = 12U,       /* SOC/EOC12 */
    ADC_INT_TRIGGER_EOC13  = 13U,       /* SOC/EOC13 */
    ADC_INT_TRIGGER_EOC14  = 14U,       /* SOC/EOC14 */
    ADC_INT_TRIGGER_EOC15  = 15U,       /* SOC/EOC15 */
    ADC_INT_TRIGGER_OSINT1 = 16U,       /* OSINT1    */
    ADC_INT_TRIGGER_OSINT2 = 17U,       /* OSINT2    */
    ADC_INT_TRIGGER_OSINT3 = 18U,       /* OSINT3    */
    ADC_INT_TRIGGER_OSINT4 = 19U,       /* OSINT4    */
	ADC_INT_TRIGGER_EOC16  = 20U,		/* SOC/EOC0  */
	ADC_INT_TRIGGER_EOC17  = 21U,		/* SOC/EOC0  */
	ADC_INT_TRIGGER_EOC18  = 22U,		/* SOC/EOC0  */
	ADC_INT_TRIGGER_EOC19  = 23U,		/* SOC/EOC0  */
	ADC_INT_TRIGGER_EOC20  = 24U		/* SOC/EOC0  */
} ADC_IntTrigger;

//*****************************************************************************
//
//! Values that can be passed to ADC_selectPPBCompareSource() as the @e compSrc
//! parameter.
//
//*****************************************************************************
typedef enum
{
    ADC_PPB_COMPSOURCE_RESULT = 0x0U,   /* PPB compare source is ADCRESULT */
    ADC_PPB_COMPSOURCE_PSUM   = 0x1U,   /* PPB compare source is PSUM      */
    ADC_PPB_COMPSOURCE_SUM    = 0x2U    /* PPB compare source is SUM       */
} ADC_PPBCompSource;

/*
 * @brief Number of the trigger repeater mode.
 *
 * Can be passed in the repInstance parameter to the following functions:
 * 		ADC_triggerRepeaterMode(),
 * 		ADC_forceRepeaterTrigger(),
 * 		ADC_triggerRepeaterActiveMode(),
 * 		ADC_triggerRepeaterModuleBusy(),
 * 		ADC_triggerRepeaterSelect(),
 * 		ADC_triggerRepeaterSyncIn(),
 * 		ADC_forceRepeaterTriggerSync(),
 * 		ADC_triggerRepeaterCount(),
 * 		ADC_triggerRepeaterPhase(),
 * 		ADC_triggerRepeaterSpread(),
 * 		ADC_configureRepeater()
 *
 */
typedef enum
{
    ADC_REPINST1 = 0x0U,                /* Select ADC repeater instance 1 */
    ADC_REPINST2 = 0x1U                 /* Select ADC repeater instance 2 */
} ADC_RepInstance;

/*
 * @brief Repeater Mode
 */
typedef enum
{
    ADC_REPMODE_OVERSAMPLING   = 0x0U,  /* ADC repeater mode is oversampling  */
    ADC_REPMODE_UNDERSAMPLING  = 0x1U   /* ADC repeater mode is undersampling */
} ADC_RepMode;

/*
 * @brief The structure used to initialize Rpeater
 */
typedef struct
{
    ADC_RepMode repMode;     /* Repeater Mode                                 */
    ADC_Trigger repTrigger;  /* Repeater Trigger                              */
    ADC_SyncInput repSyncin; /* Repeater Syncin                               */
    uint16_t repCount;       /* Repeater trigger count                        */
    uint16_t repPhase;       /* Repeater trigger phase delay in sysclk cycles */
    uint16_t repSpread;      /* Repeater trigger spread in sysclk cycles      */
} ADC_RepeaterConfig;

#endif //1.2EC

typedef enum
{
	ADC_PPBx_INT_CLR_TRIPLO = 0x1U,
	ADC_PPBx_INT_CLR_TRIPHI = 0x2U,
	ADC_PPBx_INT_CLR_ZERO	= 0x4U
}ADC_PPBInterrupClear;

typedef enum 
{
    ADC_MEAN_GROUP0 = 0x0U,
    ADC_MEAN_GROUP1 = 0x1U
}ADC_MEAN_GROUP;

typedef enum 
{
    ADC_INT_PULSE_MODE = 0x0000U,
    ADC_INT_LEVEL_MODE = 0x0001U
}ADC_PPB_PulseMode;

//*****************************************************************************
//
//! @internal
//!
//!
//! @param base specifies the ADC module base address.
//!
//! This function determines if a ADC module base address is valid.
//!
//! @return Returns @b true if the base address is valid and @b false
//! otherwise.
//
//*****************************************************************************
/*
 * @brief
 */
#ifdef DEBUG
__STATIC_INLINE bool
ADC_isBaseValid(uint32_t base)
{
    return(
           (base == ADCA_BASE)
#if defined (ADCB_BASE)
        || (base == ADCB_BASE)
#endif
        || (base == ADCC_BASE)
#if defined(ADCD_BASE)
        || (base == ADCD_BASE)
#endif
          );
}

__STATIC_INLINE bool
ADC_isResultBaseValid(uint32_t resultBase)
{
    return(
           (resultBase == ADCARESULT_BASE)
#if defined(ADCB_BASE)
        || (resultBase == ADCBRESULT_BASE)
#endif
        || (resultBase == ADCCRESULT_BASE)
#if defined(ADCD_BASE)
        || (resultBase == ADCDRESULT_BASE)
#endif
          );
}
#endif

/*
 * @brief Configures the ADC module Pre-scaler.
 *
 * @details
 * This function configures the ADC module's ADCCLK.
 *
 * @param[in]	base 			The base address of the ADC module.
 * @param[in]	clkPrescale	The ADC clock prescaler.
 *
 * @remarks
 * - The clkPrescale parameter specifies the value by which the input clock
 * 	 is dicided to make the ADCCLK.
 * - The clkPrescale value can be specified with any of the following enum values:
 * 	 ADC_CLK_DIV_1_0,ADC_CLK_DIV_2_0,ADC_CLK_DIV_3_0, ...,
 * 	 ADC_CLK_DIV_6_0,ADC_CLK_DIV_7_0,ADC_CLK_DIV_8_0
 */
__STATIC_INLINE void
ADC_setPrescaler(uint32_t base, ADC_ClkPrescale clkPrescale)
{
	/* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Set the configuration of the ADC module prescaler. */
    HWREGH(base + ADC_O_CTL2) = (HWREGH(base + ADC_O_CTL2) &
                                 ~ADC_CTL2_PRESCALE_M) | (uint16_t)clkPrescale;
}

/*
 * @brief Powers up the analog-to-digital converter core.
 *
 * @param[in]	base	The base address of the ADC module.
 *
 * @remarks
 * - Allow at least a 500us delay before sampling after calling this API.
 *   If you enable multiple ADCs, you can delay after they all have begun
 *   powering up.
 *
 */
__STATIC_INLINE void
ADC_enableConverter(uint32_t base)
{
	/* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Set the bit that powers up the analog circuitry. */
    HWREGH(base + ADC_O_CTL1) |= ADC_CTL1_ADCPWDNZ;
}

/*
 * @brief Powers down the analog-to-digital converter core.
 *
 * @param[in]	base	The base address of the ADC module.
 *
 */
__STATIC_INLINE void
ADC_disableConverter(uint32_t base)
{
	/* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Clear the bit that powers down the analog circuitry. */
    HWREGH(base + ADC_O_CTL1) &= (~ADC_CTL1_ADCPWDNZ);
}

/*
 * @brief Configures a start-of-conversion (SOC) in the ADC.
 *
 * @details
 * This function configures the a start-of-conversion (SOC) in the ADC module.
 *
 * @param[in]	base			The base address of the ADC module.
 * @param[in]	socNumber		The number of the start-of-conversion.
 * @param[in]	trigger			The Source that triggers the SOC.
 * @param[in]	channel			The number associated with the input signal.
 * @param[in]	sampleWindow	The acquisition window duration.
 *
 * @remarks
 * - The socNumber number is a value ADC_SOC_NUMBERX where X is a number
 * 	 from 0 to 15 specifying which SOC is to be configured on the ADC module
 * 	 specified by base.
 *
 * - The trigger specifies the event that causes the SOC such as software, a
 * 	 timer interrupt, an ePWM event, or an ADC interrupt.It should be a value
 * 	 in the format of ADC_TRIGGER_XXXX where XXXX is the event such as
 * 	 ADC_TRIGGER_SW_ONLY,ADC_TRIGGER_CPU1_TINT0,ADC_TRIGGER_GPIO,
 * 	 ADC_TRIGGER_EPWM1_SOCA, and so on.
 *
 * - The channel parameter specifies the channel to be converted. In single-ended
 * 	  mode this is a single pin given by ADC_CH_ADCINx where x is the number
 * 	  identifying the pin between 0 and 15 inclusive.
 *
 * - The sampleWindow parameter is the acquisition window duration in SYSCLK
 * 	 cycles. It should be a value between 1 and 512 cycles inclusive. The
 * 	 selected duration must be at least as long as one ADCCLK cycle. Also, the
 * 	 datasheet will specify a minimum window duration requirement in nanoseconds.
 */
__STATIC_INLINE void
ADC_setupSOC(uint32_t base, ADC_SOCNumber socNumber, ADC_Trigger trigger,
             ADC_Channel channel, uint32_t sampleWindow)
{
	/* Slave register address */
    uint32_t ctlRegAddr;
    uint32_t ctlRegResult;

	/* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));
    ASSERT((sampleWindow >= 1U) && (sampleWindow <= 512U));

    /* Calculate address for the SOCx(x=0~15) control register. */
    ctlRegAddr = base + ADC_SOCxCTL_OFFSET_BASE + ((uint32_t)socNumber * 4U);

#if(IS_GS32F00xx(0x30))
    if( socNumber >=  ADC_SOC_NUMBER16)
    {
    	/* Enable to support SOC16 - SOC19. */
        HWREGH(base + ADC_O_SOCPRICTL2) |= ADC_SOCPRICTL2_SOC_MODE;

        /* Calculate address for the SOCx(x=16~19) control register. */
    	ctlRegAddr = base + ADC_O_SOC16CTL + (((uint32_t)socNumber % 16) * 4U);
    }
#endif
    ctlRegResult = (((uint32_t)channel << ADC_SOC0CTL_CHSEL_S)|
    				((uint32_t)trigger << ADC_SOC0CTL_TRIGSEL_S)|
					(sampleWindow - 1U));
    /* Set the configuration of the specified SOC. */
    HWREG(ctlRegAddr) = (HWREG(ctlRegAddr) & (~ADC_SOCxCTL_MASK)) |  ctlRegResult;
#if(IS_GS32F00xx(0x30))
    HWREG(ctlRegAddr) |= ADC_SOC0CTL_RST_DIS_M;
#endif

}

#if(IS_GS32F00xx(0x30))
__STATIC_INLINE void ADC_socDisableRst(uint32_t base,ADC_SOCNumber socNumber)
{
	/* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

	/* Slave register address */
    uint32_t ctlRegAddr;

    /* Calculate address for the SOCx(x=0~15) control register. */
    ctlRegAddr = base + ADC_SOCxCTL_OFFSET_BASE + ((uint32_t)socNumber * 4U);

    if( socNumber >=  ADC_SOC_NUMBER16)
    {
    	/* Enable to support SOC16 - SOC19. */
        HWREGH(base + ADC_O_SOCPRICTL2) |= ADC_SOCPRICTL2_SOC_MODE;

        /* Calculate address for the SOCx(x=16~19) control register. */
    	ctlRegAddr = base + ADC_O_SOC16CTL + (((uint32_t)socNumber % 16) * 4U);
    }

    HWREG(ctlRegAddr) = HWREG(ctlRegAddr) | ADC_SOC0CTL_RST_DIS_M;

}

__STATIC_INLINE void ADC_socEnableRst(uint32_t base,ADC_SOCNumber socNumber)
{
	/* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

	/* Slave register address */
    uint32_t ctlRegAddr;

    /* Calculate address for the SOCx(x=0~15) control register. */
    ctlRegAddr = base + ADC_SOCxCTL_OFFSET_BASE + ((uint32_t)socNumber * 4U);

    if( socNumber >=  ADC_SOC_NUMBER16)
    {
    	/* Enable to support SOC16 - SOC19. */
        HWREGH(base + ADC_O_SOCPRICTL2) |= ADC_SOCPRICTL2_SOC_MODE;

        /* Calculate address for the SOCx(x=16~19) control register. */
    	ctlRegAddr = base + ADC_O_SOC16CTL + (((uint32_t)socNumber % 16) * 4U);
    }

    HWREG(ctlRegAddr) = HWREG(ctlRegAddr) & (~ADC_SOC0CTL_RST_DIS_M);

}
#endif

/*
 * @brief Configures the interrupt SOC trigger of an SOC.
 *
 * @details
 * This functionality is useful for creating continuous conversions.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	socNumber	The number of the start-of-conversion.
 * @param[in]	trigger		The interrupt source that will cause the SOC.
 *
 */
__STATIC_INLINE void
ADC_setInterruptSOCTrigger(uint32_t base, ADC_SOCNumber socNumber,
                           ADC_IntSOCTrigger trigger)
{
    uint16_t shiftVal;

	/* Slave register address */
    ASSERT(ADC_isBaseValid(base));

    /* Each SOC has a 2-bit field in this register. */
    shiftVal = (uint16_t)(socNumber % 16U) << 1U;

#if(IS_GS32F00xx(0x30))
    if( socNumber >=  ADC_SOC_NUMBER16)
    {

        HWREGH(base + ADC_O_INTSOCSEL3) = (HWREGH(base + ADC_O_INTSOCSEL3) &
                                    ~((uint32_t)ADC_INTSOCSEL3_SOC16_M << shiftVal)) |
                                    ((uint32_t)trigger << shiftVal);

        return;
    }
#endif

    /* Not that we're treating ADCINTSOCSEL1 and ADCINTSOCSEL2
       as one 32-bit register here.*/
	HWREG(base + ADC_O_INTSOCSEL1) = (HWREG(base + ADC_O_INTSOCSEL1) &
							   ~((uint32_t)ADC_INTSOCSEL1_SOC0_M << shiftVal)) |
							   ((uint32_t)trigger << shiftVal);

}

/*
 * @brief Configures the trigger repeater mode select.
 *
 * @param[in]	base			The base address of the ADC module.
 * @param[in]	repInstance		RepInstance is the repeater instance.
 * @param[in]	mode			The repeater mode.
 *
 * @remarks
 * - The repInstance parameter specifies the Repeater to set up.The value of
 * the repInstance parameter can refer to the enum ADC_RepInstance:
 * 		ADC_REPINST1
 * 		ADC_REPINST2
 *
 * - The premise of using a Repeater is to use a repeater as a trigger source.
 *
 */
static inline void
ADC_triggerRepeaterMode(uint32_t base, uint32_t repInstance, ADC_RepMode mode)
{
    uint32_t regOffset;

    /* @brief Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Calculate address for the Repeater control register. */
    regOffset = base + (repInstance * (ADC_REPxCTL_STEP));

    /* Set the specified repeater trigger source to modify. */
    HWREG(regOffset + ADC_O_REP1CTL) = (HWREG(regOffset + ADC_O_REP1CTL) &
                                       ~(ADC_REP1CTL_MODE)) | mode;

}

/*
 * @brief Configures the trigger source of the trigger repeater.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	repInstance	RepInstance is the repeater instance.
 * @param[in]	trigger		The Source that triggers the SOC.
 *
 */
static inline void
ADC_triggerRepeaterSelect(uint32_t base, uint16_t repInstance,
                          ADC_Trigger trigger)
{
    uint32_t regOffset;

    /* @brief Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    regOffset = base + (repInstance * (ADC_REPxCTL_STEP));

    /* Set the specified repeater trigger source to modify. */
    HWREG(regOffset + ADC_O_REP1CTL) = (HWREG(regOffset + ADC_O_REP1CTL) &
                        ~((uint32_t)ADC_REP1CTL_TRIGGER_M)) |
                        ((uint32_t)trigger << ADC_REP1CTL_TRIGGER_S);

}

/*
 * @brief Configures the trigger repeater syncin source.
 *
 * @details
 * This function configures desired ADC trigger repeater sync event that
 * corresponds to syncInput to reset all counters and any pending repeated
 * triggers are discarded.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	repInstance	RepInstance is the repeater instance.
 * @param[in]	syncInput	The desired sync event to reset all counters.
 *
 */
static inline void
ADC_triggerRepeaterSyncIn(uint32_t base, uint16_t repInstance,
                          ADC_SyncInput syncInput)
{
    uint32_t regOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    regOffset = base + (repInstance * (ADC_REPxCTL_STEP));

    /* Set the specified trigger sync input. */
    HWREG(regOffset + ADC_O_REP1CTL) = (HWREG(regOffset + ADC_O_REP1CTL) &
                        ~((uint32_t)ADC_REP1CTL_SYNCINSEL_M)) |
                        ((uint32_t)syncInput << ADC_REP1CTL_SYNCINSEL_S);

}

/*
 * @brief Clear Repeater error status
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	repInstance	RepInstance is the repeater instance.
 * @param[in]	flag		Flag bit to be cleared,
 *
 * 	@remarks
 * 	- The flag parameter following macro definitions can be used:
 * 	  ADC_REPEATER_PHASE_OVERFLOW,
 * 	  ADC_REPEATER_TRIGGER_OVERFLOW.
 *
 * @ref ADC_getRepeaterStatus
 *
 */
__STATIC_INLINE void
ADC_clearRepterOverflow(uint32_t base,uint16_t repInstance,
						uint16_t flag)
{
	uint32_t regOffset;

	ASSERT(ADC_isBaseValid(base));

	regOffset = base + (repInstance * (ADC_REPxCTL_STEP));

	HWREG(regOffset + ADC_O_REP1CTL) |= flag & ADC_REPEATER_OVERFLOW;
}

/*
 * @brief Configures the trigger repeater count.
 *
 * @details
 * This function indicates the number of repeated triggers passed into
 *
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	repInstance	RepInstance is the repeater instance.
 * @param[in]	repCount	The number of triggers.
 *
 * @remarks
 * - In oversampling mode, the repCount parameter is the number of desired
 *   repeated triggers to be generated.It should be a value between 0 to 127
 *   where (repCount + 1) triggers will be generated. For example, when repCount
 *   = 2, 3 triggers will be generated on receiving corresponding REPxCTL.TRIGSEL.
 * - In unversampling mode, the repCount parameter is the number of desired
 *   triggers to be supressed. It should be a value between 0 to 127 where 1 out
 *   of (repCount + 1) triggers received will be passed through.For Example, when
 *   repCount = 2, 1 out of 3 triggers will be generated.
 *
 * @ref ADC_getRepeaterResidualCount
 */
static inline void
ADC_triggerRepeaterCount(uint32_t base, uint16_t repInstance,
                         uint16_t repCount)
{
    uint32_t regOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));
    ASSERT(repCount <= 0xFFFFU);

    regOffset = base + (repInstance * (ADC_REPxN_STEP));

    /* Configure repeater count. */
    HWREG(regOffset + ADC_O_REP1N) = (HWREG(regOffset + ADC_O_REP1N) &
                                      ~(ADC_REP1N_NSEL_M)) | repCount;
}

/*
 * @brief Configures the trigger repeater phase.
 *
 * @details
 * This function configures the phase delay that corresponds to repPhase
 * by defining the number of sysclk to delay the selected trigger.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	repInstance	RepInstance is the repeater instance.
 * @param[in]	repPhase	Number of the repeater initial trigger delay periods.
 *
 * @remarks
 * - The repPhase parameter should be a value between 0 and 65535 inclusive.
 * For example, passing a 2 to the offset parameter will delay the trigger by
 * 2 sysclks and passing 0 will pass through the trigger without delay.
 *
 * @ref ADC_getRepeaterResidualPhase
 *
 */
static inline void
ADC_triggerRepeaterPhase(uint32_t base, uint16_t repInstance,
                          uint16_t repPhase)
{
    uint32_t regOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    regOffset = base + (repInstance * (ADC_REPxPHASE_STEP));

    /* Configure repeater phase. */
    HWREG(regOffset + ADC_O_REP1PHASE) = (HWREG(regOffset + ADC_O_REP1PHASE) &
                                          ~(ADC_REP1PHASE_PHASE_M)) | repPhase;

}

/*
 * @brief Configures the trigger repeater spread.
 *
 * @details
 * - This function configures the spread time by setting repSpread
 * to number of sysclk desired between triggers.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	repInstance	RepInstance is the repeater instance.
 * @param[in]	repSpread	repSpread is the desired trigger spread in sysclk cycle.
 *
 * @remarks
 * - In oversampling mode,the repSpread parameter is the minimum number of
 *   sysclks to wait before creating the next repeated trigger to the ADC.
 *   It should be a value between 0 and 65535 inclusive. For example, passing
 *   a 2 to the offset parameter will create at least 2 sysclk time between
 *   repeated triggers.
 *
 *   @ref ADC_getRepeaterResidualSpread
 *
 */
static inline void
ADC_triggerRepeaterSpread(uint32_t base, uint16_t repInstance,
                          uint16_t repSpread)
{
    uint32_t regOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    regOffset = base + (repInstance * (ADC_REPxSPREAD_STEP));

    /* Configure repeater spread. */
    EALLOW;
    HWREG(regOffset + ADC_O_REP1SPREAD) =
                                      (HWREG(regOffset + ADC_O_REP1SPREAD) &
                                      ~(ADC_REP1SPREAD_SPREAD_M)) | repSpread;

}

/*
 * @brief Forces software sync for the trigger repeater block.
 *
 * @details
 * This function forces the software sync for the trigger repeater block.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	repInstance	RepInstance is the repeater instance.
 *
 * @remarks
 * - The repInstance is the repeater instance to be configured. Valid values
 *   for this parameter can be referred from the enum ADC_RepInstance.
 *
 */
static inline void
ADC_forceRepeaterTriggerSync(uint32_t base, uint16_t repInstance)
{
    uint32_t regOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    regOffset = base + (repInstance * (ADC_REPxCTL_STEP));

    /* Force software sync for the trigger repeater block. */
    HWREG(regOffset + ADC_O_REP1CTL) |= ADC_REP1CTL_SWSYNC;
}

/*
 * @brief Get the trigger repeater active mode status.
 *
 * @details
 * This function returns the status of the ADC trigger repeater mode for
 * the selected repeater instance.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	repInstance	RepInstance is the repeater instance.
 *
 * @remarks
 * - The repInstance is the repeater instance to be configured. Valid values
 * for this parameter can be referred from the enum ADC_RepInstance.
 *
 * @return Returns the status of the trigger repeater active mode.
 * 		   - 1 : oversampling
 * 		   - 0 : undersampling
 */
static inline bool
ADC_triggerRepeaterActiveMode(uint32_t base, uint32_t repInstance)
{
    uint32_t regOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    regOffset = base + (repInstance * (ADC_REPxCTL_STEP));

    /* get the specified repeater trigger active mode status. */
    return(HWREG(regOffset + ADC_O_REP1CTL) &
                (1U << ADC_REP1CTL_ACTIVEMODE_S));
}

/*
 * @brief Get the trigger repeater module busy indication.
 *
 * @details
 * This function returns the status of the ADC trigger repeater module for
 * the selected repeater instance.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	repInstance	RepInstance is the repeater instance.
 *
 * @return Returns the staus of the trigger repeater module busy.
 * 		   - 0 : Repeater is idle and can accept a new repeated trigger in
 * 		   		 oversampling mode.
 * 		   - 1 : Repeater still has repeated triggers remaining (NCOUNT>0)
 * 		   		 or associated SOCs are still pending (SOCBUSY is 1).
 *
 */
static inline bool
ADC_triggerRepeaterModuleBusy(uint32_t base, uint32_t repInstance)
{
    uint32_t regOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    regOffset = base + (repInstance * (ADC_REPxCTL_STEP));

    /* get the specified repeater trigger active mode status. */
    return(HWREG(regOffset + ADC_O_REP1CTL) &
                (1U << ADC_REP1CTL_MODULEBUSY_S));
}

/*
 * @brief Gets the current status for repeater block.
 *
 * @details
 * This function returns the current status for the repeater block.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	repInstance	RepInstance is the repeater instance.
 *
 * @return Returns the current event status,enumerated as a bit field of
 * 			ADC_REPEATER_MODULE_BUSY,
 * 			ADC_REPEATER_PHASE_OVERFLOW,
 * 			ADC_REPEATER_TRIGGER_OVERFLOW.
 *
 */
static inline uint16_t
ADC_getRepeaterStatus(uint32_t base, uint16_t repInstance)
{
    uint32_t regOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    regOffset = base + (repInstance * (ADC_REPxCTL_STEP));

    /* Return the status of repeater. */
    return(HWREGH(regOffset + ADC_O_REP1CTL) & ADC_REPSTATUS_MASK);
}

/*
 * @brief Get Repeater trigger count.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	repInstance	RepInstance is the repeater instance.
 *
 * @return The number of Repeater trigger count.
 */
__STATIC_INLINE uint16_t
ADC_getRepeaterResidualCount(uint32_t base,uint16_t repInstance)
{
	uint32_t regOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    regOffset = base + ADC_O_REP1N + (repInstance * (ADC_REPxCTL_STEP));

    return((HWREG(regOffset)&ADC_REP1N_NCOUNT_M)>>ADC_REP1N_NCOUNT_S);
}

/*
 * @brief Get Repeater trigger phase count.
 *
 * @param[in]	baseThe 	base address of the ADC module.
 * @param[in]	repInstance	RepInstance is the repeater instance.
 *
 * @return The number of Repeater trigger phase count.
 */
__STATIC_INLINE uint16_t
ADC_getRepeaterResidualPhase(uint32_t base,uint16_t repInstance)
{
	uint32_t regOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    regOffset = base + ADC_O_REP1PHASE + (repInstance * (ADC_REPxCTL_STEP));

    return((HWREG(regOffset)&ADC_REP1N_NCOUNT_M)>>ADC_REP1N_NCOUNT_S);
}

/*
 * @brief Get Repeater trigger spread count.
 *
 * @param[in]base] The base address of the ADC module.
 * @param[in]repInstance] RepInstance is the repeater instance.
 *
 * @return The number of Repeater trigger spread count.
 */
__STATIC_INLINE uint16_t
ADC_getRepeaterResidualSpread(uint32_t base,uint16_t repInstance)
{
	uint32_t regOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    regOffset = base + ADC_O_REP1SPREAD + (repInstance * (ADC_REPxCTL_STEP));

    return((HWREG(regOffset)&ADC_REP1N_NCOUNT_M)>>ADC_REP1N_NCOUNT_S);
}

/*
 * @brief Sets the priority mode of the SOCs.
 *
 * @details
 * This function sets the priority mode of the SOCs.
 *
 * @param[in]base] The base address of the ADC module.
 * @param[in]priMode] The priority mode of the SOCs.
 *
 * @remarks
 * There are three main modes that can be passed in the priMode parameter.
 * - All SOCs are in round-robin mode.
 * - All priorities are in high priority mode.
 * - A range of SOCs are assigned high priority,with all others in round robin mode.
 *	 High priority mode means that an SOC with high priority will interrupt the
 *	 round robin wheel and insert itself as the next conversion.Passing in the value
 *	 ADC_PRI_SOC0_HIPRI will make SOC0 highest priority,ADC_PRI_THRU_SOC1_HIPRI will
 *	 put SOC0 and SOC 1 in high priority, and so on up to ADC_PRI_THRU_SOC14_HIPRI
 *	 where SOCs 0 through 14 are in high priority.
 *
 */
__STATIC_INLINE void
ADC_setSOCPriority(uint32_t base, ADC_PriorityMode priMode)
{
	/* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    HWREGH(base + ADC_O_SOCPRICTL) = (HWREGH(base + ADC_O_SOCPRICTL) &
                                      ~ADC_SOCPRICTL_SOCPRIORITY_M) |
                                     (uint16_t)priMode;
}

/*
 * @brief Set SOC burst mode.
 *
 * @details
 * This function configures the burst trigger and burstSize of an ADC module.
 * Burst mode allows a single trigger to walk through the round-robin SOCs one
 * or more at a time.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	trigger		The Source that triggers the SOC.
 * @param[in]	burstSize	The number of the SOC burst size.
 *
 * @remarks
 * -  When burst mode is enabled, the trigger selected by the ADC_setupSOC()
 * API will no longer have an effect on the SOCs in round-robin mode. Instead,
 * the source specified through the @e trigger parameter will cause a burst of
 * burstSize conversions to occur.
 * - The trigger parameter takes the same values as the ADC_setupSOC() API
 * - The burstSize parameter should be a value between 1 and 16 inclusive.
 *
 */
__STATIC_INLINE void
ADC_setBurstModeConfig(uint32_t base, ADC_Trigger trigger, uint16_t burstSize)
{
    uint16_t regValue;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));
    ASSERT(((uint16_t)trigger & ~((uint16_t)0x1FU)) == 0U);
    ASSERT((burstSize >= 1U) && (burstSize <= 16U));

    /* Write the burst mode configuration to the register. */
    regValue = (uint16_t)trigger | ((burstSize - 1U) <<
                                    ADC_BURSTCTL_BURSTSIZE_S);

    HWREGH(base + ADC_O_BURSTCTL) = (HWREGH(base + ADC_O_BURSTCTL) &
                                     ~((uint16_t)ADC_BURSTCTL_BURSTTRIGSEL_M |
                                       ADC_BURSTCTL_BURSTSIZE_M)) | regValue;

}

/*
 * @brief Enables SOC burst mode.
 *
 * @param[in]	base	The base address of the ADC module.
 *
 */
__STATIC_INLINE void
ADC_enableBurstMode(uint32_t base)
{
	/* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Enable burst mode. */
    HWREGH(base + ADC_O_BURSTCTL) |= ADC_BURSTCTL_BURSTEN;
}

/*
 * @brief Disables SOC burst mode.
 *
 * @param[in]	base	The base address of the ADC module.
 *
 */
__STATIC_INLINE void
ADC_disableBurstMode(uint32_t base)
{
	/* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Disable burst mode. */
    HWREGH(base + ADC_O_BURSTCTL) &= ~ADC_BURSTCTL_BURSTEN;
}

/*
 * @brief Sets the timing of the end-of-conversion pulse.
 *
 * @details
 * This function configures the end-of-conversion (EOC) pulse generated by ADC.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	pulseMode	The generation mode of the EOC pulse.
 *
 * @remarks
 * - This pulse will be generated either at the end of the acquisition window
 * plus a number of SYSCLK cycles configured by ADC_setInterruptCycleOffset()
 * or at the end of the voltage conversion, one cycle prior to the ADC result
 * latching into it's result register.
 *
 */
__STATIC_INLINE void
ADC_setInterruptPulseMode(uint32_t base, ADC_PulseMode pulseMode)
{
	/* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Set the position of the pulse. */
    HWREGH(base + ADC_O_CTL1) = (HWREGH(base + ADC_O_CTL1) &
                                 ~ADC_CTL1_INTPULSEPOS) | (uint16_t)pulseMode;
}

/*
 * @brief Sets the timing of early interrupt generation.
 *
 * @details
 * This function configures cycle offset between the negative edge of a sample
 * pulse and an early interrupt pulse being generated.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	cycleOffset	The cycles from an SOC falling edge to an early
 * 					   		interrupt pulse.
 *
 * @remarks
 * - This number of cycles is specified with the cycleOffset parameter.
 *
 */
__STATIC_INLINE void
ADC_setInterruptCycleOffset(uint32_t base, uint16_t cycleOffset)
{
	/* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Set the position of the pulse. */
    HWREGH(base + ADC_O_INTCYCLE) = cycleOffset;
}

/*
 * @brief Sets the source EOC for an analog-to-digital converter interrupt.
 *
 * @details
 * Sets the source EOC for an analog-to-digital converter interrupt.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	adcIntNum	Interrupt number within the ADC wrapper.
 * @param[in]	intTrigger	The number of the start-of-conversion.
 * 					  		Refer ADC_IntTrigger enum for valid values for this input
 *
 */
__STATIC_INLINE void
ADC_setInterruptSource(uint32_t base, ADC_IntNumber adcIntNum,
                       uint16_t intTrigger)
{
    uint32_t intRegAddr;
    uint16_t shiftVal;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));
    ASSERT(intTrigger < 16U);

    /*
     * Each INTSEL register manages two interrupts.
     * If the interrupt number is even,
     * we'll be accessing the upper byte and will need to shift.
     */
    intRegAddr = base + ADC_INTSELxNy_OFFSET_BASE + ((uint32_t)adcIntNum >> 1)*2;
    shiftVal = ((uint16_t)adcIntNum & 0x1U) << 3U;

    /* Set the specified ADC interrupt source. */
    HWREGH(intRegAddr) =
        (HWREGH(intRegAddr) & ~(ADC_INTSEL1N2_INT1SEL_M << shiftVal)) |
        ((uint16_t)intTrigger << shiftVal);
}

/*
 * @brief Enables continuous mode for an ADC interrupt.
 *
 * @details
 * This function enables continuous mode for the ADC interrupt passed into
 * adcIntNum.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	adcIntNum	Interrupt number within the ADC wrapper.
 *
 * @remarks
 * - This means that pulses will be generated for the specified
 * ADC interrupt whenever an EOC pulse is generated irrespective of whether
 * or not the flag bit is set.
 *
 */
__STATIC_INLINE void
ADC_enableContinuousMode(uint32_t base, ADC_IntNumber adcIntNum)
{
    uint32_t intRegAddr;
    uint16_t shiftVal;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /*
     * Each INTSEL register manages two interrupts. If the interrupt number is
     * even, we'll be accessing the upper byte and will need to shift.
     */
    intRegAddr = base + ADC_INTSELxNy_OFFSET_BASE + ((uint32_t)adcIntNum >> 1)*2;
    shiftVal = ((uint16_t)adcIntNum & 0x1U) << 3U;

    /* Enable continuous mode for the specified ADC interrupt. */
    HWREGH(intRegAddr) |= ADC_INTSEL1N2_INT1CONT << shiftVal;
}

/*
 * @brief Disables continuous mode for an ADC interrupt.
 *
 * @details
 * This function disables continuous mode for the ADC interrupt passed into
 * adcIntNum.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	adcIntNum	Interrupt number within the ADC wrapper.
 *
 * @remarks
 * - This means that pulses will be generated for the specified ADC interrupt
 * whenever an EOC pulse is generated irrespective of whether or not the flag
 * bit is set.
 *
 */
__STATIC_INLINE void
ADC_disableContinuousMode(uint32_t base, ADC_IntNumber adcIntNum)
{
    uint32_t intRegAddr;
    uint16_t shiftVal;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /*
     * Each INTSEL register manages two interrupts. If the interrupt number is
     * even, we'll be accessing the upper byte and will need to shift.
     */
    intRegAddr = base + ADC_INTSELxNy_OFFSET_BASE + ((uint32_t)adcIntNum >> 1)*2;
    shiftVal = ((uint16_t)adcIntNum & 0x1U) << 3U;

    /* Disable continuous mode for the specified ADC interrupt. */
    HWREGH(intRegAddr) &= ~(ADC_INTSEL1N2_INT1CONT << shiftVal);
}

/*
 * @brief Gets the current ADC interrupt status.
 *
 * @details
 * This function returns the interrupt status for the ADC converter.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	adcIntNum	Interrupt number within the ADC wrapper.
 *
 * @return Return the corresponding interrupt flag status
 */
__STATIC_INLINE bool
ADC_getInterruptStatus(uint32_t base, ADC_IntNumber adcIntNum)
{
	/* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Get the specified ADC interrupt status. */
    return((HWREGH(base + ADC_O_INTFLG) & (1U << (uint16_t)adcIntNum)) != 0U);
}

/*
 * @brief Clears ADC interrupt sources.
 *
 * @details
 * This function clears the specified ADC interrupt sources so that they no
 * longer assert.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	adcIntNum	Interrupt number within the ADC wrapper.
 *
 * @remarks
 * - If not in continuous mode, this function must be called
 * before any further interrupt pulses may occur.
 *
 */
__STATIC_INLINE void
ADC_clearInterruptStatus(uint32_t base, ADC_IntNumber adcIntNum)
{
	/* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Clear the specified interrupt. */
    HWREGH(base + ADC_O_INTFLGCLR) = (uint16_t)1U << (uint16_t)adcIntNum;
}

/*
 * @brief Gets the current ADC interrupt overflow status.
 *
 * @details
 * This function returns the interrupt overflow status for the
 * analog-to-digital converter. An overflow condition is generated
 * irrespective of the continuous mode.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	adcIntNum	Interrupt number within the ADC wrapper.
 *
 * @return Return the corresponding interrupt flag status
 *
 */
__STATIC_INLINE bool
ADC_getInterruptOverflowStatus(uint32_t base, ADC_IntNumber adcIntNum)
{
	/* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Get the specified ADC interrupt status. */
    return((HWREGH(base + ADC_O_INTOVF) & (1U << (uint16_t)adcIntNum)) != 0U);
}

/*
 * @brief Clears ADC interrupt overflow sources.
 *
 * @details
 * This function clears the specified ADC interrupt overflow sources so that
 * they no longer assert. If software tries to clear the overflow in the same
 * cycle that hardware tries to set the overflow, then hardware has priority.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	adcIntNum	Interrupt number within the ADC wrapper.
 *
 */
__STATIC_INLINE void
ADC_clearInterruptOverflowStatus(uint32_t base, ADC_IntNumber adcIntNum)
{
	/* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Clear the specified interrupt overflow bit. */
    HWREGH(base + ADC_O_INTOVFCLR) = (uint16_t)1U << (uint16_t)adcIntNum;
}

/*
 * @brief Enables an ADC interrupt source.
 *
 * @details
 * This function enables the indicated ADC interrupt source.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	adcIntNum	Interrupt number within the ADC wrapper.
 *
 * @remarks
 * Only the sources that are enabled can be reflected to the processor
 * interrupt. Disabled sources have no effect on the processor.
 *
 *
 */
__STATIC_INLINE void
ADC_enableInterrupt(uint32_t base, ADC_IntNumber adcIntNum)
{
    uint32_t intRegAddr;
    uint16_t shiftVal;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /*
     * Each INTSEL register manages two interrupts. If the interrupt number is
     * even, we'll be accessing the upper byte and will need to shift.
     */
    intRegAddr = base + ADC_INTSELxNy_OFFSET_BASE + ((uint32_t)adcIntNum >> 1)*2;
    shiftVal = ((uint16_t)adcIntNum & 0x1U) << 3U;

    /* Enable the specified ADC interrupt. */
    HWREGH(intRegAddr) |= ADC_INTSEL1N2_INT1E << shiftVal;
}

/*
 * @brief Disables an ADC interrupt source.
 *
 * @details
 * This function disables an ADC interrupt source.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	adcIntNum	Interrupt number within the ADC wrapper.
 *
 * @remarks
 * - Only the sources that are enabled can be reflected to the processor
 * interrupt. Disabled sources have no effect on the processor.
 *
 */
__STATIC_INLINE void
ADC_disableInterrupt(uint32_t base, ADC_IntNumber adcIntNum)
{
    uint32_t intRegAddr;
    uint16_t shiftVal;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /*
     * Each INTSEL register manages two interrupts. If the interrupt number is
     * even, we'll be accessing the upper byte and will need to shift.
     */
    intRegAddr = base + ADC_INTSELxNy_OFFSET_BASE + ((uint32_t)adcIntNum >> 1)*2;
    shiftVal = ((uint16_t)adcIntNum & 0x1U) << 3U;

    /* Disable the specified ADC interrupt. */
    HWREGH(intRegAddr) &= ~(ADC_INTSEL1N2_INT1E << shiftVal);
}

/*
 * @brief Forces a SOC flag to a 1 in the analog-to-digital converter.
 *
 * @details
 * This function forces the SOC flag associated with the SOC specified by
 * socNumber.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	socNumber	The number of the start-of-conversion.
 *
 * @remarks
 * - This initiates a conversion once that SOC is given priority.
 * - This software trigger can be used whether or not the SOC has been
 * 	 configured to accept some other specific trigger.
 *
 */
__STATIC_INLINE void
ADC_forceSOC(uint32_t base, ADC_SOCNumber socNumber)
{
	uint16_t shiftVal;
	uint32_t frcRegAddr;

	/* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));
    frcRegAddr = ADC_O_SOCFRC1;

#if(IS_GS32F00xx(0x30))
    if (socNumber  >= ADC_SOC_NUMBER16 )
    {
        frcRegAddr = ADC_O_SOCFRC2;
    }
#endif

    shiftVal = (uint16_t)socNumber % 16;

    /* Write to the register that will force a 1 to the corresponding SOC flag */
    HWREGH(base + frcRegAddr) = ((uint16_t)1U << shiftVal);
}

/*
 * @brief Forces multiple SOC flags to 1 in the analog-to-digital converter.
 *
 * @details
 * This function to forces multiple SOC flags to 1 in the analog-to-digital converter.
 *
 * @param[in]	base	The base address of the ADC module.
 * @param[in]	socMask	The SOCs to be forced through software
 *
 * @remarks
 * - This initiates a conversion once the desired SOCs are given priority.
 * - This software trigger can be used whether or not the SOC has been
 * 	 configured to accept some other specific trigger.
 * - Valid values for socMask parameter can be any of the individual
 * 	 ADC_FORCE_SOCx values or any of their OR'd combination to trigger multiple
 * 	 SOCs.
 *
 */
__STATIC_INLINE void
ADC_forceMultipleSOC(uint32_t base, uint32_t socMask)
{
	uint16_t shiftVal;

	/* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

	HWREGH(base + ADC_O_SOCFRC1) = socMask & 0xffff;
#if(IS_GS32F00xx(0x30))
    if( socMask >> 16 )
    {
    	socMask >>= 16;
    	HWREGH(base + ADC_O_SOCFRC2) = socMask & 0x1F;
    }
#endif
}

/*
 * @brief Forces software trigger to ADC trigger repeater block.
 *
 * @details
 * This function forces the selected ADC repeater block.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	repInstance	RepInstance is the repeater instance.
 *
 * @remarks
 * - Force a trigger to repeater input regardless of the value of TRIGGER.
 *
 */
static inline void
ADC_forceRepeaterTrigger(uint32_t base, uint16_t repInstance)
{
  uint32_t regOffset;

  /* Check the arguments. */
  ASSERT(ADC_isBaseValid(base));

  regOffset = base + (repInstance * (ADC_REPxCTL_STEP));

  /* Triggers the selected repeater instance. */
  HWREGH(regOffset + ADC_O_REP1FRC) |= ADC_REP1FRC_SWFRC;

}

/*
 * @brief Configures a post-processing block (PPB) in the ADC.
 *
 * @details
 * This function associates a post-processing block with a SOC.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 * @param[in]	socNumber	The number of the start-of-conversion.
 *
 * @remarks
 * - You can have more that one PPB associated with the same SOC, but a
 * PPB can only be configured to correspond to one SOC at a time. Also note
 * that when you have multiple PPBs for the same SOC, the calibration offset
 * that actually gets applied will be that of the PPB with the highest number.
 * Since SOC0 is the default for all PPBs, look out for unintentional
 * overwriting of a lower numbered PPB's offset.
 */
__STATIC_INLINE void
ADC_setupPPB(uint32_t base, ADC_PPBNumber ppbNumber, ADC_SOCNumber socNumber)
{
    uint32_t ppbOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Get the offset to the appropriate PPB configuration register.*/
    ppbOffset = (ADC_PPBxCONFIG_STEP * (uint32_t)ppbNumber) + ADC_O_PPB1CONFIG;

    /* Write the configuration to the register. */
    HWREGH(base + ppbOffset) = (HWREGH(base + ppbOffset) &
                                ~ADC_PPB1CONFIG_CONFIG_M)
								|((uint16_t)(socNumber%16U) & ADC_PPB1CONFIG_CONFIG_M)
#if (IS_GS32F00xx(0x30))
							   /* ADC_SOC_NUMBER16 set exp to support PPB */
                                |((uint16_t)(socNumber >= ADC_SOC_NUMBER16 ? 1: 0) << 7)
#endif
                               ;
}

/*
 * @brief Sets the post processing block offset correction.
 *
 * @details
 * This function sets the PPB offset correction value.
 * This value can be used to digitally remove any system-level offset inherent
 * in the ADCIN circuit before it is stored in the appropriate result register.
 * The offset parameter is subtracted from the ADC output and is a signed value
 * from -512 to 511 inclusive.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 * @param[in]	offset		The 10-bit signed value subtracted from ADC the output.
 *
 */
__STATIC_INLINE void
ADC_setPPBCalibrationOffset(uint32_t base, ADC_PPBNumber ppbNumber,
                            int16_t offset)
{
    uint32_t ppbOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Get the offset to the appropriate offset register. */
    ppbOffset = (ADC_PPBxOFFCAL_STEP * (uint32_t)ppbNumber) + ADC_O_PPB1OFFCAL;

    /* Write the offset amount. */
    HWREGH(base + ppbOffset) = (HWREGH(base + ppbOffset) &
                                ~ADC_PPB1OFFCAL_OFFCAL_M) |
                               ((uint16_t)offset & ADC_PPB1OFFCAL_OFFCAL_M);
}

/*
 * @brief Sets the post processing block reference offset.
 *
 * @details
 * This function sets the PPB reference offset value.
 * This can be used to either calculate the feedback error
 * or convert a unipolar signal to bipolar by subtracting a reference value.
 * The result will be stored in the appropriate PPB result register which
 * can be read using ADC_readPPBResult().
 *
 * @param[in]base] The base address of the ADC module.
 * @param[in]ppbNumber] The number of the post-processing block.
 * @param[in]offset] The 12-bit signed value subtracted from ADC the output.
 *
 */
__STATIC_INLINE void
ADC_setPPBReferenceOffset(uint32_t base, ADC_PPBNumber ppbNumber,
                          uint16_t offset)
{
    uint32_t ppbOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));
    ASSERT(offset <= 0xFFF);

    /* Get the offset to the appropriate offset register. */
    ppbOffset = (ADC_PPBxOFFREF_STEP * (uint32_t)ppbNumber) + ADC_O_PPB1OFFREF;

    /* Write the offset amount. */
    HWREGH(base + ppbOffset) = offset;
}

/*
 * @brief Enables absolute value capability in the PPB.
 *
 * @details
 * This function enables the absolute value functionality in the post-
 * processing block specified by the ppbNumber parameter.
 * When enabled,absolute value calculation would be done on
 * the ADC Result associated with the selected SOC.
 * In other words, the PPB result will calculated as
 * shown below: (ADCPPBxRESULT = abs(ADCRESULTx - ADCPPBxOFFREF))
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 *
 */
static inline void
ADC_enablePPBAbsoluteValue(uint32_t base, ADC_PPBNumber ppbNumber)
{
    uint32_t ppbOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Get the offset to the appropriate PPB configuration register. */
    ppbOffset = (ADC_PPBxCONFIG_STEP * (uint32_t)ppbNumber) + ADC_O_PPB1CONFIG;

    /* Enable PPB absolute value. */
    HWREGH(base + ppbOffset) |= ADC_PPB1CONFIG_ABSEN;
}

/*
 * @brief Disables absolute value capability in the PPB.
 *
 * @details
 * This function disables the absolute value functionality in the post-
 * processing block specified by the ppbNumber parameter.
 * When disabled,absolute value calculation would be done on
 * the ADC Result associated with the selected SOC.
 * In other words, the PPB result will calculated as
 * shown below: (ADCPPBxRESULT = ADCRESULTx - ADCPPBxOFFREF)
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 *
 */
static inline void
ADC_disablePPBAbsoluteValue(uint32_t base, ADC_PPBNumber ppbNumber)
{
    uint32_t ppbOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Check the arguments. */
    ppbOffset = (ADC_PPBxCONFIG_STEP * (uint32_t)ppbNumber) + ADC_O_PPB1CONFIG;

    /* Check the arguments. */
    HWREGH(base + ppbOffset) &= ~ADC_PPB1CONFIG_ABSEN;

}

/*
 * @brief Enables two's complement capability in the PPB.
 *
 * @details
 * This function enables two's complement in the post-processing block
 * specified by the ppbNumber parameter.
 * When enabled, a two's complement will be performed on the output of
 * the offset subtraction before it is stored in the appropriate PPB
 * result register.
 * In other words, the PPB result will be the reference offset value minus
 * the the ADC result value (ADCPPBxRESULT = ADCSOCxOFFREF - ADCRESULTx).
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 *
 */
__STATIC_INLINE void
ADC_enablePPBTwosComplement(uint32_t base, ADC_PPBNumber ppbNumber)
{
    uint32_t ppbOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Get the offset to the appropriate PPB configuration register. */
    ppbOffset = (ADC_PPBxCONFIG_STEP * (uint32_t)ppbNumber) + ADC_O_PPB1CONFIG;

    /* Enable PPB two's complement. */
    HWREGH(base + ppbOffset) |= ADC_PPB1CONFIG_TWOSCOMPEN;
}

/*
 * @brief Disables two's complement capability in the PPB.
 *
 * @details
 * This function disables two's complement in the post-processing block
 * specified by the ppbNumber parameter.
 * When enabled, a two's complement will be performed on the output of
 * the offset subtraction before it is stored in the appropriate PPB
 * result register.
 * In other words, the PPB result will be the reference offset value minus
 * the the ADC result value (ADCPPBxRESULT = ADCSOCxOFFREF - ADCRESULTx).
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 */
__STATIC_INLINE void
ADC_disablePPBTwosComplement(uint32_t base, ADC_PPBNumber ppbNumber)
{
    uint32_t ppbOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Get the offset to the appropriate PPB configuration register. */
    ppbOffset = (ADC_PPBxCONFIG_STEP * (uint32_t)ppbNumber) + ADC_O_PPB1CONFIG;

    /* Disable PPB two's complement. */
    HWREGH(base + ppbOffset) &= ~ADC_PPB1CONFIG_TWOSCOMPEN;
}

/*
 * @brief Enables individual ADC PPB event sources.
 *
 * @details
 * This function enables the indicated ADC PPB event sources.
 * This will allow the specified events to propagate through
 * the X-BAR to a pin or to an ePWM module.
 *
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 * @param[in]	evtFlags	A bit mask of the event sources to be enabled.
 *
 * @remarks
 * - The evtFlags parameter can be any of the ADC_EVT_TRIPHI,
 * 											  ADC_EVT_TRIPLO,
 * 											  ADC_EVT_ZERO values.
 *
 */
__STATIC_INLINE void
ADC_enablePPBEvent(uint32_t base, ADC_PPBNumber ppbNumber, uint16_t evtFlags)
{
	/* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));
    ASSERT((evtFlags & ~0x7U) == 0U);

    /* Enable the specified event. */
    HWREGH(base + ADC_O_EVTSEL) |= evtFlags << ((uint16_t)ppbNumber * 4U);
}

/*
 * @brief Disables individual ADC PPB event sources.
 *
 * @details
 * This function disables the indicated ADC PPB event sources.
 * This will allow the specified events to propagate through
 * the X-BAR to a pin or to an ePWM module.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 * @param[in]	evtFlags	A bit mask of the event sources to be enabled.
 *
 * @remarks
 * - The evtFlags parameter can be any of the ADC_EVT_TRIPHI,
 * 											  ADC_EVT_TRIPLO,
 * 											  ADC_EVT_ZERO values.
 *
 */
__STATIC_INLINE void
ADC_disablePPBEvent(uint32_t base, ADC_PPBNumber ppbNumber, uint16_t evtFlags)
{
	/* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));
    ASSERT((evtFlags & ~0x7U) == 0U);

    /* Disable the specified event. */
    HWREGH(base + ADC_O_EVTSEL) &= ~(evtFlags << ((uint16_t)ppbNumber * 4U));
}

/*
 * @brief Enables individual ADC PPB event interrupt sources.
 *
 * @details
 * This function enables the indicated ADC PPB interrupt sources.
 * Only the sources that are enabled can be reflected to the processor
 * interrupt.
 * Disabled sources have no effect on the processor.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 * @param[in]	intFlags	A bit mask of the interrupt source to be disabled.
 *
 * @remarks
 * - The intFlags parameter can be any of the ADC_EVT_TRIPHI,
 * 											  ADC_EVT_TRIPLO,
 * 											  ADC_EVT_ZERO values.
 */
__STATIC_INLINE void
ADC_enablePPBEventInterrupt(uint32_t base, ADC_PPBNumber ppbNumber,
                            uint16_t intFlags)
{
    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));
    ASSERT((intFlags & ~0x7U) == 0U);

    /* Enable the specified event interrupts. */
    HWREGH(base + ADC_O_EVTINTSEL) |= intFlags << ((uint16_t)ppbNumber * 4U);
}

/*
 * @brief Disables individual ADC PPB event interrupt sources.
 *
 * @details
 * This function enables the indicated ADC PPB interrupt sources.
 * Only the sources that are enabled can be reflected to the processor
 * interrupt.
 * Disabled sources have no effect on the processor.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 * @param[in]	intFlags	A bit mask of the interrupt source to be disabled.
 *
 * @remarks
 * - The intFlags parameter can be any of the ADC_EVT_TRIPHI,
 * 											  ADC_EVT_TRIPLO,
 * 											  ADC_EVT_ZERO values.
 *
 */
__STATIC_INLINE void
ADC_disablePPBEventInterrupt(uint32_t base, ADC_PPBNumber ppbNumber,
                             uint16_t intFlags)
{
    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));
    ASSERT((intFlags & ~0x7U) == 0U);

    /* Disable the specified event interrupts. */
    HWREGH(base + ADC_O_EVTINTSEL) &= ~(intFlags <<
                                        ((uint16_t)ppbNumber * 4U));
}

/*
 * @brief Select PPB interrupt pulse mode.
 *
 * @details
 * This function select PPB ADC interrupt pulse mode:Pulse mode or Level mode.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 * @param[in]	mode		The mode of ADC interrupt pulse.
 *
 */
__STATIC_INLINE void
ADC_setPPBINTMode(uint32_t base, ADC_PPBNumber ppbNumber,
				  ADC_PPB_PulseMode mode)
{
    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    HWREG(base + (uint32_t)ADC_O_PPB_INT_MODE) = mode;
}

/*
 * @brief Clear PPB interrupt for level mode.
 *
 * @detail
 * This function to clear PPB interrupt flag.
 *
 * @param[in]	base] The base address of the ADC module.
 * @param[in]	ppbNumber] The number of the post-processing block.
 * @param[in]	intFlags] The number of PPB interrupt flag.
 *
 * @remarks
 * - If the ppb interrupt is a pulse interrupt, the hardware clears it
 *   automatically
 *
 */
__STATIC_INLINE void
ADC_clearPPBINT(uint32_t base, ADC_PPBNumber ppbNumber,
				ADC_PPBInterrupClear intFlags)
{
    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));
    ASSERT((intFlags & ~0x7U) == 0U);

    HWREGH(base + ADC_O_PPB_INT_CLR) |= (intFlags << (ppbNumber * 3));
}

/*
 * @brief Get PPB Interrupt FLag. Only for test use.
 *
 * @details
 * This function to get the specific PPB interrupt flag.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 * @param[in]	intFlags	The number of PPB interrupt flag.
 *
 * @return Return the specific of PPB interrupt flag status.
 *
 */
__STATIC_INLINE bool
ADC_getPPBINTFlag(uint32_t base, ADC_PPBNumber ppbNumber,
				  ADC_PPBInterrupClear intFlags)
{
    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));
    ASSERT((intFlags & ~0x7U) == 0U);

    return(HWREG(base + ADC_O_PPB_INT_FLAG) & (intFlags << (ppbNumber * 3)));
}

/*
 * @brief Gets the current ADC event status.
 *
 * @details
 * This function returns the event status for the analog-to-digital converter.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 *
 * @return Return the PPBEvent status
 *
 */
__STATIC_INLINE uint16_t
ADC_getPPBEventStatus(uint32_t base, ADC_PPBNumber ppbNumber)
{
	/* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Get the event status for the specified post-processing block. */
    return((HWREGH(base + ADC_O_EVTSTAT) >> ((uint16_t)ppbNumber * 4U)) & 0x7U);
}

/*
 * @brief Clears ADC event flags.
 *
 * @details
 * This function clears the indicated ADC PPB event flags.
 * After an event occurs this function must be called to allow
 * additional events to be produced.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 * @param[in]	evtFlags	A bit mask of the event source to be cleared.
 *
 * @remarks
 * - The evtFlags parameter can be any of the ADC_EVT_TRIPHI,
 * 											ADC_EVT_TRIPLO,
 * 											ADC_EVT_ZERO values.
 *
 */
__STATIC_INLINE void
ADC_clearPPBEventStatus(uint32_t base, ADC_PPBNumber ppbNumber,
                        uint16_t evtFlags)
{
    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));
    ASSERT((evtFlags & ~0x7U) == 0U);

    /* Clear the specified event interrupts. */
    HWREGH(base + ADC_O_EVTCLR) |= evtFlags << ((uint16_t)ppbNumber * 4U);
}

/*
 * @brief Enables cycle-by-cycle clear of ADC PPB event flags.
 *
 * @details
 * This function enables the automatic cycle-by-cycle clear of
 * ADC PPB event flags.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 *
 * @remarks
 *  - When enabled, the desired PPB event flags are automatically cleared
 * on the next PPBxRESULT load, unless a set condition is also occurring
 * at the same time, in which case the set takes precedence.
 *
 *
 */
__STATIC_INLINE void
ADC_enablePPBEventCBCClear(uint32_t base, ADC_PPBNumber ppbNumber)
{
    uint32_t ppbOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Get the offset to the appropriate PPB configuration register. */
    ppbOffset = (ADC_PPBxCONFIG_STEP * (uint32_t)ppbNumber) + ADC_O_PPB1CONFIG;

    /* Set automatic cycle-by-cycle flag clear bit */
    HWREGH(base + ppbOffset) |= ADC_PPB1CONFIG_CBCEN;
}

/*
 * @brief Disables cycle-by-cycle clear of ADC PPB event flags.
 *
 * @details
 * This function disables the cycle-by-cycle clear of ADC PPB event flags.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 *
 * @remarks
 * - When disabled, the desired PPB event flags are to be cleared explicitly in
 * 	 software inorder to generate next set of interrupts/events.
 *
 */
__STATIC_INLINE void
ADC_disablePPBEventCBCClear(uint32_t base, ADC_PPBNumber ppbNumber)
{
    uint32_t ppbOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Get the offset to the appropriate PPB configuration register. */
    ppbOffset = (ADC_PPBxCONFIG_STEP * (uint32_t)ppbNumber) + ADC_O_PPB1CONFIG;

    /* Clear automatic cycle-by-cycle flag clear bit */
    HWREGH(base + ppbOffset) &= ~ADC_PPB1CONFIG_CBCEN;
}

/*
 * @brief Reads sample delay time stamp from a PPB.
 *
 * @details
 * This function returns the sample delay time stamp.
 * This delay is the number of system clock cycles between the SOC being
 * triggered and when it began converting.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 *
 * @return Return the delay time stamp.
 */
__STATIC_INLINE uint16_t
ADC_getPPBDelayTimeStamp(uint32_t base, ADC_PPBNumber ppbNumber)
{
    uint32_t ppbOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Get the offset to the appropriate delay. */
    ppbOffset = (ADC_PPBxSTAMP_STEP * (uint32_t)ppbNumber) + ADC_O_PPB1STAMP;

    /* Return the delay time stamp. */
    return(HWREGH(base + ppbOffset) & ADC_PPB2STAMP_DLYSTAMP_M);
}

/*
 * @brief Configures PPB count limit.
 *
 * @remarks
 * This function configures the PPB oversampling count limit which defines the
 * number of ADC conversions to accumulate before partial sum is automatically
 * loaded to the sum registes.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 * @param[in]	limit		The number of sample count limit.
 *
 */
static inline void
ADC_setPPBCountLimit(uint32_t base, ADC_PPBNumber ppbNumber, uint16_t limit)
{
    uint32_t ppbOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));
    ASSERT(limit <= ADC_PPB1LIMIT_LIMIT_M);

    /* Get the offset to the appropriate PPB configuration register. */
    ppbOffset = (ADC_PPBxLIMIT_STEP * (uint32_t)ppbNumber) + ADC_O_PPB1LIMIT;

    /*  Enable PPB two's complement.*/
    HWREGH(base + ppbOffset) =
                (HWREGH(base + ppbOffset) & ~(ADC_PPB1LIMIT_LIMIT_M)) |
                (limit << ADC_PPB1LIMIT_LIMIT_S);

}

/*
 * @brief Returns the PPB count limit.
 *
 * @details
 * This function returns the PPB oversampling count limit which defines the
 * number of ADC conversions to accumulate before partial sum is automatically
 * loaded to the sum registes.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 *
 * @return Returns the PPB count limit.
 */
static inline uint16_t
ADC_getPPBCountLimit(uint32_t base, ADC_PPBNumber ppbNumber)
{
    uint16_t limit;
    uint32_t ppbOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Get the offset to the appropriate PPB configuration register. */
    ppbOffset = (ADC_PPBxLIMIT_STEP * (uint32_t)ppbNumber) + ADC_O_PPB1LIMIT;

    return((HWREGH(base + ppbOffset) & ADC_PPB1LIMIT_LIMIT_M));
}

/*
 * @brief Enables the extended low limit in the PPB.
 *
 * @details
 * This function enables the low limit for PPB by allowing the PPB low
 * limit to be set by ADCPPBxTRIPLO2 register.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 *
 */
static inline void
ADC_enablePPBExtendedLowLimit(uint32_t base, ADC_PPBNumber ppbNumber)
{
    uint32_t ppbLoOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Get the offset to the appropriate trip limit registers. */
    ppbLoOffset = (ADC_PPBxTRIPLO_STEP * (uint32_t)ppbNumber) +
                  ADC_O_PPB1TRIPLO;

    /* Enable PPB extended low limit. */
    HWREGH(base + ppbLoOffset) |= ADC_PPB1TRIPLO_LIMITLO2EN;
}

/*
 * @brief Disables the extended low limit in the PPB.
 *
 * @details
 * This function disables the low limit for PPB by allowing the PPB low
 * limit to be set by ADCPPBxTRIPLO register.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 *
 */
static inline void
ADC_disablePPBExtendedLowLimit(uint32_t base, ADC_PPBNumber ppbNumber)
{
    uint32_t ppbOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Get the offset to the appropriate PPB configuration register. */
    ppbOffset = (ADC_PPBxCONFIG_STEP * (uint32_t)ppbNumber) + ADC_O_PPB1TRIPLO2;

    /* Disable PPB extended low limit. */
    HWREGH(base + ppbOffset) &= ~ADC_PPB1TRIPLO_LIMITLO2EN;
}

/*
 * @breif Sets the windowed trip low limits for a PPB.
 *
 * @details
 * This function is used when an external lower limit is used.
 * When comparing to an ADCPPBxRESULT register, the upper bits will be ignored.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 * @param[in]	tripLoLimit	The number of the trip high low limit.
 *
 * @ref ADC_enablePPBExtendedLowLimit
 *
 */
__STATIC_INLINE void
ADC_setPPBTripExtendLowLimits(uint32_t base,ADC_PPBNumber ppbNumber,
							  uint32_t tripLoLimit)
{
	uint32_t ppbOffset;

	/* Check the arguments. */
	ASSERT(ADC_isBaseValid(base));
	ASSERT(tripLoLimit <= 0xFFFFFF);

	ppbOffset = (ADC_PPBxTRIPLO2_STEP * ppbNumber) + ADC_O_PPB1TRIPLO2;

	/* Set Extend trip low limits*/
	HWREG(base + ppbOffset) = tripLoLimit;

}

/*
 * @brief Configures PPB compare source.
 *
 * @details
 * This function configures the desired source to be used for zero crossing
 * detect logic and threshold compare.
 * For valid values of compSrc refer to enum ADC_PPBCompSource.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 * @param[in]	compSrc		The desired source to be used for zero crossing detect
 * 				   			logic and threshold compare.
 *
 */
static inline void
ADC_selectPPBCompareSource(uint32_t base, ADC_PPBNumber ppbNumber,
                           uint16_t compSrc)
{
    uint32_t ppbOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));
    ASSERT(compSrc <= 2U);

    /* Get the offset to the appropriate PPB configuration register. */
    ppbOffset = (ADC_PPBxCONFIG2_STEP * (uint32_t)ppbNumber) +
                    ADC_O_PPB1CONFIG2;

    /* Select PPB compare source. */
    HWREGH(base + ppbOffset) =
                 (HWREGH(base + ppbOffset) & ~(ADC_PPB1CONFIG2_COMPSEL_M)) |
                 (compSrc << ADC_PPB1CONFIG2_COMPSEL_S);
}

/*
 * @brief Configures PPB shift value.
 *
 * @details
 * This function configured the shift value reuired to right shift the PPB
 * PSUM before loading into the final PPB SUM.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 * @param[in]	shiftVal	the number of bits to right shift PSUM before loading
 * 							to final PPB SUM.
 *
 */
static inline void
ADC_setPPBShiftValue(uint32_t base, ADC_PPBNumber ppbNumber, uint16_t shiftVal)
{
    uint32_t ppbOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));
    ASSERT(shiftVal <= 10U);

    /* Get the offset to the appropriate PPB configuration register. */
    ppbOffset = (ADC_PPBxCONFIG2_STEP * (uint32_t)ppbNumber) +
                    ADC_O_PPB1CONFIG2;

    /* Configure shift value for the PPB. */
    HWREGH(base + ppbOffset) =
                 (HWREGH(base + ppbOffset) & ~(ADC_PPB1CONFIG2_SHIFT_M)) |
                 (shiftVal << ADC_PPB1CONFIG2_SHIFT_S);
}

/*
 * @brief Configures PPB sync input.
 *
 * @details
 * This function configures desired sync event to transfer partial registers
 * to final registers and reset the partial registers. For valid values of
 * syncInput refer to enum ADC_SyncInput.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 * @param[in]	syncInput	The desired sync event to transfer partial registers
 * 					 		to final registers and reset the partial registers.
 *
 */
static inline void
ADC_selectPPBSyncInput(uint32_t base, ADC_PPBNumber ppbNumber,
                       uint16_t syncInput)
{
    uint32_t ppbOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));
    ASSERT(syncInput <= 0x1DU);

    /* Get the offset to the appropriate PPB configuration register. */
    ppbOffset = (ADC_PPBxCONFIG2_STEP * (uint32_t)ppbNumber) +
                    ADC_O_PPB1CONFIG2;

    /* Select sync input for the PPB. */
    HWREGH(base + ppbOffset) =
                 (HWREGH(base + ppbOffset) & ~(ADC_PPB1CONFIG2_SYNCINSEL_M)) |
                 (syncInput << ADC_PPB1CONFIG2_SYNCINSEL_S);
}

/*
 * @brief Forces PPB software sync.
 *
 * @details
 * This function forces the software sync for the desired PPB.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 *
 */
static inline void
ADC_forcePPBSync(uint32_t base, ADC_PPBNumber ppbNumber)
{
    uint32_t ppbOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Get the offset to the appropriate PPB configuration register. */
    ppbOffset = (ADC_PPBxCONFIG2_STEP * (uint32_t)ppbNumber) +
                    ADC_O_PPB1CONFIG2;

    /* Force software sync for the PPB. */
    HWREGH(base + ppbOffset) |= ADC_PPB1CONFIG2_SWSYNC;
}

/*
 * @brief Configures PPB interrupt source.
 *
 * @details
 * This function sets which oversampling event is the source of a PPB intterupt.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 * @param[in]	osIntSrc	selects PPB interrupt source.
 *
 * @remark
 * - For valid values of osIntSrc refer to enum ADC_PPBIntSrcSelect.
 */
static inline void
ADC_selectPPBOSINTSource(uint32_t base, ADC_PPBNumber ppbNumber,
                         uint16_t osIntSrc)
{
    uint32_t ppbOffset;

    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));
    ASSERT(osIntSrc <= 1U);

    /* Get the offset to the appropriate PPB configuration register. */
    ppbOffset = (ADC_PPBxCONFIG2_STEP * (uint32_t)ppbNumber) +
                    ADC_O_PPB1CONFIG2;

    /* Select PPB OSINT source. */
    HWREGH(base + ppbOffset) =
                 (HWREGH(base + ppbOffset) & ~(ADC_PPB1CONFIG2_OSINTSEL)) |
                 (osIntSrc << 12U);
}

/*
 * @brief Set the ePPB scale factor.
 *
 * @details
 * This function set the scale factor of ePPB.
 *
 * @param[in]	resultBase	The base address of the ADC results.
 * @param[in]	socNumber	The number of the start-of-conversion.
 * @param[in]	scale		The scale factor to be applied.
 *
 */
#if !IS_GS32F3xx(0x12)
__STATIC_INLINE void
ADC_setEPPBScale(uint32_t resultBase, ADC_SOCNumber socNumber,float32_t scale)
{
    /* Check the arguments. */
    ASSERT(ADC_isResultBaseValid(resultBase));

	uint32_t scale_uint = *((uint32_t *)(&scale));

	HWREG(resultBase + (uint32_t)ADC_E_SCALE0 + (uint32_t)(socNumber*4) ) = scale_uint;
}
#endif
/*
 * @brief Set the ePPB offset factor.
 *
 * @detals
 * This function set the offset factor of ePPB.
 *
 * @param[in]	resultBase	The base address of the ADC results.
 * @param[in]	socNumber	The number of the start-of-conversion.
 * @param[in]	offset		The offset factor to be applied.
 *
 */
#if !IS_GS32F3xx(0x12)
__STATIC_INLINE void
ADC_setEPPBOffset(uint32_t resultBase, ADC_SOCNumber socNumber,float32_t offset)
{
    /* Check the arguments. */
    ASSERT(ADC_isResultBaseValid(resultBase));

	uint32_t offset_uint = *((uint32_t *)(&offset));

	HWREG(resultBase + (uint32_t)ADC_E_OFFSET0 + (uint32_t)(socNumber*4) ) = offset_uint;
}
#endif
/*
 * @brief Enable Mean group for ADC SOCx results.
 *
 * @details
 * This function enable ADCx SOCy results to be averaged with GROUPz.
 *
 * @param[in]	resultBase	The base address of the ADC results.
 * @param[in]	socNumber	The number of the start-of-conversion.
 * @param[in]	group		The group to be averaged.
 *
 */
__STATIC_INLINE void
ADC_enableMeanGroup(uint32_t resultBase, ADC_SOCNumber socNumber,
					ADC_MEAN_GROUP group)
{
    ASSERT(ADC_isResultBaseValid(resultBase));

#if (ADC_VERSION == 0002) || (ADC_VERSION == 0003)
    if( socNumber <= ADC_SOC_NUMBER15 )
    {
        HWREG(resultBase + ADC_O_MEANEN) |= ((uint32_t)0x1 << (socNumber*2 + group));
    }
#elif(IS_GS32F00xx(0x30))
    else
    {
        socNumber = (ADC_SOCNumber)((uint32_t)socNumber - 16);

        HWREG(resultBase + ADC_O_MEANEN2) |= ((uint32_t)0x1 << socNumber*2 + group);
    }
#endif

}

/*
 * @brief Disable Mean group for ADC SOCx results.
 *
 * @details
 * This function disable ADCx SOCy results to be averaged with GROUPz.
 *
 * @param[in]	resultBase	The base address of the ADC results.
 * @param[in]	socNumber	The number of the start-of-conversion.
 * @param[in]	group		The group to be averaged.
 *
 */
__STATIC_INLINE void
ADC_disableMeanGroup(uint32_t resultBase, ADC_SOCNumber socNumber,
					 ADC_MEAN_GROUP group)
{
    ASSERT(ADC_isResultBaseValid(resultBase));


    if( socNumber <= ADC_SOC_NUMBER15 )
    {
        uint32_t shift = socNumber*2 + group;

        HWREG(resultBase + ADC_O_MEANEN) &= ~((uint32_t)0x1 << (socNumber*2 + group));
    }
#if(IS_GS32F00xx(0x30))
    else
    {
        socNumber = (ADC_SOCNumber)((uint32_t)socNumber - 16);

        HWREG(resultBase + ADC_O_MEANEN2) &= ~((uint32_t)0x1 << (socNumber*2 + group));
    }
#endif
}

/*
 * @brief Reads the conversion result.
 *
 * This function returns the conversion result that corresponds to the base
 * address passed into resultBase and the SOC passed into socNumber.
 *
 * @param[in]resultBase] The base address of the ADC Result module.
 * @param[in]ppbNumber] The number of the post-processing block.
 *
 * @return Returns the conversion result.
 *
 * @note Take care that you are using a base address for the result registers
 * 		 (ADCxRESULT_BASE) and not a base address for the control registers.
 */
__STATIC_INLINE uint16_t
ADC_readResult(uint32_t resultBase, ADC_SOCNumber socNumber)
{
    /* Check the arguments. */
    ASSERT(ADC_isResultBaseValid(resultBase));

    /* Return the ADC result for the selected SOC. */
#if(IS_GS32F00xx(0x30))
    if(socNumber >= ADC_SOC_NUMBER16)
    {
        return(HWREGH(resultBase + (uint32_t)ADC_O_RESULT16 +
                      ((uint32_t)socNumber%16)*2U));
    }
#endif
	return(HWREGH(resultBase + (uint32_t)ADC_RESULTx_OFFSET_BASE +
                  (uint32_t)socNumber*2U));

}

/*
 * @brief Reads the processed conversion result from the PPB.
 *
 * @details
 * This function returns the processed conversion result that corresponds to
 * the base address passed into resultBase and the PPB passed into  ppbNumber.
 *
 * @param[in]	resultBase	The base address of the ADC Result module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 *
 * @return Returns the signed 32-bit conversion result.
 *
 * @remarks
 * - Take care that you are using a base address for the result registers
 * 		 (ADCxRESULT_BASE) and not a base address for the control registers.
 */
__STATIC_INLINE int32_t
ADC_readPPBResult(uint32_t resultBase, ADC_PPBNumber ppbNumber)
{
	/* Check the arguments. */
    ASSERT(ADC_isResultBaseValid(resultBase));

    /*Return the result of selected PPB. */
    return((int32_t)HWREG(resultBase + (uint32_t)ADC_PPBxRESULT_OFFSET_BASE +
           ((uint32_t)ppbNumber * 4UL)));
}

/*
 * @brief Reads the oversampled partial sum from the PPB.
 *
 * @details
 * This function returns the oversampled partial sum of PPB results that
 * corresponds to the base address passed into base and the PPB passed
 * into ppbNumber.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 *
 * @return Return the Partial sum value.
 *
 */
static inline int32_t
ADC_readPPBPSum(uint32_t base, ADC_PPBNumber ppbNumber)
{
	/* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Returns the partial sum result of selected PPB. */
    return((int32_t)HWREG(base + (uint32_t)ADC_PPBxPSUM_OFFSET_BASE +
           ((uint32_t)ppbNumber * 52UL)));
}

/*
 * @brief Reads the over
 *
 * @details
 * This function returns the oversampled partial count that corresponds to
 * the base address passed into base and the PPB passed into ppbNumber.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 *
 * @return Return the oversampled partial count from the PPB.
 *
 */
static inline uint16_t
ADC_readPPBPCount(uint32_t base, ADC_PPBNumber ppbNumber)
{
    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Returns the partial count of the selected PPB. */
    return(HWREG(base + (uint32_t)ADC_PPBxPCOUNT_OFFSET_BASE +
            ((uint32_t)ppbNumber * 52UL)));
}

/*
 * @brief Reads the processed conversion result's partial
 * 		  maximum value from the PPB.
 *
 * @details
 * This function returns the oversampled partial maximum that corresponds to
 * the base address passed into base and the PPB passed into ppbNumber.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 *
 * @return Returns the oversampled partial maximum value.
 *
 */
static inline int32_t
ADC_readPPBPMax(uint32_t base, ADC_PPBNumber ppbNumber)
{
    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Return the partial maximum value of selected PPB. */
    return((int32_t)HWREG(base + (uint32_t)ADC_PPBxPMAX_OFFSET_BASE +
           ((uint32_t)ppbNumber * 52UL)));
}

/*
 * @brief Reads the processed conversion result's
 * 		  partial minimum value from the PPB.
 *
 * @details
 * This function returns the oversampled partial minimum that corresponds to
 * the base address passed into base and the PPB passed into ppbNumber.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 *
 * @return Returns the oversampled partial minimum value.
 *
 */
static inline int32_t
ADC_readPPBPMin(uint32_t base, ADC_PPBNumber ppbNumber)
{
    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Return the partial minimum value of selected PPB. */
    return((int32_t)HWREG(base + (uint32_t)ADC_PPBxPMIN_OFFSET_BASE +
           ((uint32_t)ppbNumber * 52UL)));
}

/*
 * @brief Reads the index of the result with partial maximum value from the PPB.
 *
 * @details
 * This function returns the index of the oversampled partial maximum value
 * that corresponds to the base address passed into base and the PPB passed
 * into ppbNumber.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 *
 * @return Returns the index of the oversampled partial maximum value.
 *
 */
static inline uint16_t
ADC_readPPBPMaxIndex(uint32_t base, ADC_PPBNumber ppbNumber)
{
    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Returns the index of the partial maximum value of selected PPB. */
    return((uint16_t)HWREGH(base + (uint32_t)ADC_PPBxPMAXI_OFFSET_BASE +
           ((uint32_t)ppbNumber * 52UL)));
}

/*
 * @brief Reads the index of the result with partial minimum value from the PPB.
 *
 * @details
 * This function returns the index of the oversampled partial minimum value
 * that corresponds to the base address passed into base and the PPB passed
 * into ppbNumber.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 *
 * @return Returns the index of the oversampled partial minimum value.
 *
 */
static inline uint16_t
ADC_readPPBPMinIndex(uint32_t base, ADC_PPBNumber ppbNumber)
{
    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Returns the index of the partial minimum value of selected PPB. */
    return((uint16_t)HWREGH(base + (uint32_t)ADC_PPBxPMINI_OFFSET_BASE +
           ((uint32_t)ppbNumber * 52UL)));
}

/*
 * @brief Reads the oversampled final sum from the PPB.
 *
 * @details
 * This function returns the processed sum of results that corresponds to
 * the base address passed into resultBase and the PPB passed into ppbNumber.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 *
 * @return Returns the oversampled final sum value.
 *
 */
static inline int32_t
ADC_readPPBSum(uint32_t resultBase, ADC_PPBNumber ppbNumber)
{
    /* Check the arguments. */
    ASSERT(ADC_isResultBaseValid(resultBase));

    /* Return the result of selected PPB. */
    return((int32_t)HWREG(resultBase + (uint32_t)ADC_PPBxSUM_OFFSET_BASE +
           ((uint32_t)ppbNumber * 8UL)));
}

/*
 * @brief Reads the oversampled final count from the PPB.
 *
 * @details
 * This function returns the oversampled final count that
 * corresponds to the base address passed into resultBase and the
 * Take care that you are using a base address for the result registers
 * (ADCxRESULT_BASE) and not a base address for the control registers.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 *
 * @return Returns the oversampled final count value.
 */
static inline uint32_t
ADC_readPPBCount(uint32_t resultBase, ADC_PPBNumber ppbNumber)
{
    /* Check the arguments. */
    ASSERT(ADC_isResultBaseValid(resultBase));

    /* Return the final count of selected PPB. */
    return((uint32_t)HWREGH(resultBase + (uint32_t)ADC_PPBxCOUNT_OFFSET_BASE +
           ((uint32_t)ppbNumber * 8UL)));
}

/*
 * @brief Reads the processed conversion result's maximum value from the PPB.
 *
 * @details
 * This function returns the oversampled final maximum that corresponds to
 * the base address passed into @e resultBase and the PPB passed into ppbNumber.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 *
 * @return Returns the oversampled final maximum value.
 */
static inline int32_t
ADC_readPPBMax(uint32_t resultBase, ADC_PPBNumber ppbNumber)
{
    /* Check the arguments. */
    ASSERT(ADC_isResultBaseValid(resultBase));

    /* Return the final maximum value of selected PPB. */
    return((int32_t)HWREG(resultBase + (uint32_t)ADC_PPBxMAX_OFFSET_BASE +
           ((uint32_t)ppbNumber * 16UL)));
}

/*
 * @brief Reads the processed conversion result's minimum value from the PPB.
 *
 * @details
 * This function returns the processed conversion result that corresponds to
 * the base address passed into resultBase and the PPB passed into ppbNumber.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 *
 * @return Returns the oversampled final maximum value.
 */
static inline int32_t
ADC_readPPBMin(uint32_t resultBase, ADC_PPBNumber ppbNumber)
{
    /* Check the arguments. */
    ASSERT(ADC_isResultBaseValid(resultBase));

    /* Return the final minimum value of selected PPB. */
    return((int32_t)HWREG(resultBase + (uint32_t)ADC_PPBxMIN_OFFSET_BASE +
           ((uint32_t)ppbNumber * 16UL)));
}

/*
 * @brief Reads the index of the result with maximum value from the PPB.
 *
 * @details
 * This function returns the index of the processed conversion's maximum value
 * that corresponds to the base address passed into resultBase and the PPB
 * passed into ppbNumber.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 *
 * @return Returns the index of the result with maximum value.
 *
 * @remarks
 * - Take care that you are using a base address for the result registers
 * 	 (ADCxRESULT_BASE) and not a base address for the control registers.
 */
static inline uint16_t
ADC_readPPBMaxIndex(uint32_t resultBase, ADC_PPBNumber ppbNumber)
{
    /* Check the arguments. */
    ASSERT(ADC_isResultBaseValid(resultBase));

    /* Returns the index of the final maximum value of selected PPB. */
    return(HWREG(resultBase + (uint32_t)ADC_PPBxMAXI_OFFSET_BASE +
           ((uint32_t)ppbNumber * 16UL)));
}

/*
 * @brief Reads the index of the result with minimum value from the PPB.
 *
 * @details
 * This function returns the index of the processed conversion's minimum value
 * that corresponds to the base address passed into resultBase and the PPB
 * passed into ppbNumber.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 *
 * @return Returns the index of the result with final minimum value.
 *
 * @remarks
 * - Take care that you are using a base address for the result registers
 * 	 (ADCxRESULT_BASE) and not a base address for the control registers.
 */
static inline uint16_t
ADC_readPPBMinIndex(uint32_t resultBase, ADC_PPBNumber ppbNumber)
{
    /* Check the arguments. */
    ASSERT(ADC_isResultBaseValid(resultBase));

    /* Returns the index of the final minimum value of the selected PPB. */
    return(HWREG(resultBase + (uint32_t)ADC_PPBxMINI_OFFSET_BASE +
            ((uint32_t)ppbNumber * 16UL)));
}

/*
 * @brief Read the result of ePPB.
 *
 * @details
 * This function returns the ePPB result.
 *
 * @param[in]	resultBase	The base address of the ADC Result module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 *
 * @return Return The ePPB result.
 *
 */
#if !IS_GS32F3xx(0x12)
__STATIC_INLINE float32_t
ADC_readEPPBResult(uint32_t resultBase, ADC_SOCNumber socNumber)
{
	uint32_t *result_addr = (uint32_t *)( resultBase +\
					(uint32_t)ADC_E_RESULT0 + (uint32_t)socNumber*4);

	float32_t result = *((float32_t *)result_addr);

	return result;
}
#endif
/*
 * @brief Determines whether the ADC is busy or not.
 *
 * @details
 * This function allows the caller to determine whether or not the ADC is
 * busy and can sample another channel.
 *
 * @param[in]	base	The base address of the ADC module.
 *
 * @return Returns true if the ADC is sampling or false if all
 * 		   samples are complete.
 */
__STATIC_INLINE bool
ADC_isBusy(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(ADC_isBaseValid(base));

    /* Determine if the ADC is busy. */
    return((HWREGH(base + ADC_O_CTL1) & ADC_CTL1_ADCBSY) != 0U);
}

/*
 * @brief Configures Open/Shorts Detection Circuit Mode.
 *
 * @details
 * This function configures the open/shorts detection circuit mode of the ADC.
 *
 * @param[in]	base	The base address of the ADC module.
 * @param[in]	modeVal	The desired open/shorts detection circuit mode.
 *
 */
#if IS_GS32F00xx(0x12)
__STATIC_INLINE void
ADC_configOSDetectMode(uint32_t base, ADC_OSDetectMode modeVal)
{
    /* Configure open/shorts detection circuit mode. */
    /* external */
	switch (base)
    {
    	case ADCA_BASE:ASysCtl_setAdcSpare(0, 0x0E6254D7);break;
    	case ADCB_BASE:ASysCtl_setAdcSpare(1, 0x0E6254D7);break;
    	case ADCC_BASE:ASysCtl_setAdcSpare(2, 0x0E6254D7);break;
		default:
			break;
    }

    if(HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL))
    {
    	switch (base)
        {
        	case ADCA_BASE:ASysCtl_setAdcSpare(0, 0x1662D497);break;
        	case ADCB_BASE:ASysCtl_setAdcSpare(1, 0x1662D497);break;
        	case ADCC_BASE:ASysCtl_setAdcSpare(2, 0x1662D497);break;
    		default:
    			break;
        }

    	if(HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) & ASYSCTL_ANAREFCTL_ANAREF2P5SEL)
    	{
    		/* for internal 2.5V */
        	switch (base)
            {
            	case ADCA_BASE:ASysCtl_setAdcSpare(0, 0x1AE1DCD7);break;
            	case ADCB_BASE:ASysCtl_setAdcSpare(1, 0x1AE1DCD7);break;
            	case ADCC_BASE:ASysCtl_setAdcSpare(2, 0x1AE1DCD7);break;
        		default:
        			break;
            }
    	}
    }


    HWREGH(base + ADC_O_OSDETECT) = ((HWREGH(base + ADC_O_OSDETECT) &
                                     (~ADC_OSDETECT_DETECTCFG_M)) |
                                     (uint16_t)modeVal);
}
#elif IS_GS32F3xx(0x22)

__STATIC_INLINE void
ADC_configOSDetectMode(uint32_t base, ADC_OSDetectMode modeVal)
{
    /* Configure open/shorts detection circuit mode. */
    /* external */
	switch (base)
    {
//    	case ADCA_BASE:ASysCtl_setAdcSpare(0, 0x0E6254D7);break;
//    	case ADCB_BASE:ASysCtl_setAdcSpare(1, 0x0E6254D7);break;
//    	case ADCC_BASE:ASysCtl_setAdcSpare(2, 0x0E6254D7);break;
		default:
			break;
    }

    if(HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL))//for internal
    {
		/* for internal 3.3V */
    	switch (base)
        {
        	case ADCA_BASE:
        		ASysCtl_setAdcSpare(0, 0x1AE1DC97);break;
        	case ADCB_BASE:
        		ASysCtl_setAdcSpare(1, 0x1AE1DC97);break;
        	case ADCC_BASE:
        		ASysCtl_setAdcSpare(2, 0x1AE1DC97);break;
        	case ADCD_BASE:
        		ASysCtl_setAdcSpare(3, 0x1AE1DC97);break;
    		default:
    			break;
        }
//    	ASysCtl_setAdcSpare(0, 0x1AE1DC97);
//    	ASysCtl_setAdcSpare(1, 0x1AE1DC97);
//    	ASysCtl_setAdcSpare(2, 0x1AE1DC97);
//    	ASysCtl_setAdcSpare(3, 0x1AE1DC97);
//    	ASysCtl_setAdcSpare(0, 0x1AE1DCD7);
//    	ASysCtl_setAdcSpare(1, 0x1AE1DCD7);
//    	ASysCtl_setAdcSpare(2, 0x1AE1DCD7);
//    	ASysCtl_setAdcSpare(3, 0x1AE1DCD7);

    	if(HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) & ASYSCTL_ANAREFCTL_ANAREFA2P5SEL)
    	{
    		/* for internal 2.5V */
        	switch (base)
            {
				case ADCA_BASE:ASysCtl_setAdcSpare(0, 0x1AE1DCD7);break;
        		default:
        			break;
            }
    	}

    	if(HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) & ASYSCTL_ANAREFCTL_ANAREFB2P5SEL)
    	{
    		/* for internal 2.5V */
        	switch (base)
            {
				case ADCB_BASE:ASysCtl_setAdcSpare(1, 0x1AE1DCD7);break;
        		default:
        			break;
            }
    	}

    	if(HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) & ASYSCTL_ANAREFCTL_ANAREFC2P5SEL)
    	{
    		/* for internal 2.5V */
        	switch (base)
            {
            	case ADCC_BASE:ASysCtl_setAdcSpare(2, 0x1AE1DCD7);break;
        		default:
        			break;
            }
    	}

    	if(HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) & ASYSCTL_ANAREFCTL_ANAREFD2P5SEL)
    	{
    		/* for internal 2.5V */
        	switch (base)
            {
            	case ADCD_BASE:ASysCtl_setAdcSpare(3, 0x1AE1DCD7);break;
        		default:
        			break;
            }
    	}


    }


    HWREGH(base + ADC_O_OSDETECT) = ((HWREGH(base + ADC_O_OSDETECT) &
                                     (~ADC_OSDETECT_DETECTCFG_M)) |
                                     (uint16_t)modeVal);

}

#endif

/*
 * @brief Converts temperature from sensor reading to degrees C.
 *
 * @details
 * This function converts temperature from temp sensor reading to degrees C.
 * Temp sensor values in production test are derived with 2.5V reference.
 * The vref argument in the function is used to scale the temp sensor
 * reading accordingly if temp sensor value is read at a different VREF
 * setting.
 *
 * @param[in]	tempResult	The raw ADC C conversion result from the temp sensor.
 * @param[in]	refMode		The reference voltage being used (for example 3.3 for 3.3V).
 * @param[in]	vref		The reference mode being used (ADC_REFERENCE_INTERNAL
 * 											  or ADC_REFERENCE_EXTERNAL).
 *
 * @return Returns the temperature sensor reading converted to degrees C.
 *
 */
__STATIC_INLINE float
ADC_getTemperatureC(uint16_t tempResult, float32_t vref)
{
    //return ((double)-545.0993290108169 * (double)(((double)tempResult/4095)*vref) + (double)391.33699528225316);
	float t_sensor_volt;

	t_sensor_volt = (float)tempResult * vref / 4095.0f;

	return ((-151.151f * t_sensor_volt * t_sensor_volt) - (370.9f * t_sensor_volt) + 351.698f);
}

/*
 * @brief Converts temperature from sensor reading to degrees K.
 *
 * @details
 * This function converts temperature from temp sensor reading to degrees K.
 * Temp sensor values in production test are derived with 2.5V reference.
 * The vref argument in the function is used to scale the temp sensor
 * reading accordingly if temp sensor value is read at a different VREF
 * setting.
 *
 * @param[in]	tempResult	The raw ADC C conversion result from the temp sensor.
 * @param[in]	refMode		The reference voltage being used (for example 3.3 for 3.3V).
 * @param[in]	vref		The reference mode being used (ADC_REFERENCE_INTERNAL
 * 											  or ADC_REFERENCE_EXTERNAL).
 *
 * @return Returns the temperature sensor reading converted to degrees K.
 */
__STATIC_INLINE int16_t
ADC_getTemperatureK(uint16_t tempResult, ADC_ReferenceMode refMode,
                        float32_t vref)
{
    float32_t temp;

    /* Read temp sensor slope and offset locations from OTP and convert */
    temp = (float32_t)tempResult * (vref / 2.5F);
    if(refMode == ADC_REFERENCE_INTERNAL)
    {
        return((int16_t)(((((int32_t)temp - ADC_INT_REF_TSOFFSET) * 4096) /
                         ADC_INT_REF_TSSLOPE) + 273));
    }
    else
    {
        return((int16_t)(((((int32_t)temp - ADC_EXT_REF_TSOFFSET) * 4096) /
                         ADC_EXT_REF_TSSLOPE) + 273));
    }
}

/*
 * @brief Configures the trigger repeater.
 *
 * @details
 * This function configures ADC trigger repeater for repeater mode, trigger
 * and sync along with count, phase delay and spread of triggers.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	repInstance	RepInstance is the repeater instance.
 * @param[in]	config		The structure for Repeater initialization.
 *
 */
extern void
ADC_configureRepeater(uint32_t base, uint16_t repInstance,
                      ADC_RepeaterConfig *config);

/*
 * @brief Configures the ADC module's reference mode and offset trim.
 *
 * @details
 * This function configures the ADC module's reference mode and loads the
 * corresponding offset trims.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	refMode		The reference mode being used (ADC_REFERENCE_INTERNAL
 * 							or ADC_REFERENCE_EXTERNAL).
 * @param[in]	refVoltage	The reference voltage being used (ADC_REFERENCE_2_5V
 * 					  		or ADC_REFERENCE_3_3V). This is ignored when the
 * 					  		reference mode is external.
 *
 * @remarks
 * - In this device, the bandgaps are common for all the ADC instances,
 * 	 hence common Vref configuration needs to be done for all the ADCs.This
 * 	 API configures same Vref configuration for all the supported ADCs in the
 * 	 device.
 * - When the refMode parameter is ADC_REFERENCE_EXTERNAL, the value of the
 * 	 refVoltage parameter has no effect on the operation of the ADC.
 *
 */
extern void
ADC_setVREF(uint32_t base, ADC_ReferenceMode refMode,
            ADC_ReferenceVoltage refVoltage);

/*
 * @brief Configures the offset trim for all the ADC instances in a device.
 *
 * @details
 * This function loads the offset trims of all the ADC instances available in
 * a device.
 *
 * @param[in]	refMode] 	The reference mode being used
 * @param[in]	refVoltage	The reference voltage being used. This is ignored
 * 					  		when the reference mode is external.
 *
 * @remark
 * - When the refMode parameter is ADC_REFERENCE_EXTERNAL, the value
 * 	 of the refVoltage parameter has no effect on the operation of the ADC.
 */
extern void
ADC_setOffsetTrimAll(ADC_ReferenceMode refMode,
                ADC_ReferenceVoltage refVoltage);

/*
 * @brief Configures the offset trim for the desired ADC instance.
 *
 * @details
 * This function loads the offset trims for the desired ADC instance.
 *
 * @param[in]	base	The base address of the ADC module.
 *
 */
extern void
ADC_setOffsetTrim(uint32_t base);

/*
 * @brief Configures the INL trim for the desired ADC instance
 *
 * @details
 * This function loads the INL trims for the desired ADC instance.
 *
 * @param[in]	base	The base address of the ADC module.
 *
 */
extern void
ADC_setINLTrim(uint32_t base);

/*
 * @brief Sets the windowed trip limits for a PPB.
 *
 * @details
 * This function sets the windowed trip limits for a PPB.
 * These values set the digital comparator so that when one of
 * the values is exceeded,either a high or low trip event will occur.
 *
 * @param[in]	base		The base address of the ADC module.
 * @param[in]	ppbNumber	The number of the post-processing block.
 * @param[in]	tripHiLimit	The value is the digital comparator trip high limit.
 * @param[in]	tripLoLimit	The value is the digital comparator trip low limit.
 */
extern void
ADC_setPPBTripLimits(uint32_t base, ADC_PPBNumber ppbNumber,
                     int32_t tripHiLimit, int32_t tripLoLimit);

/* Mark the end of the C bindings section for C++ compilers. */
#ifdef __cplusplus
}
#endif

#endif // DRIVERLIB_ADC_H
