/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 * commit history
 * 20240313, LYF, verify for HAL_CAN Send/Receive
 * 20240316, LYF, verify for HAL_CAN Bus error/Bus off/Fifo depth MailBox/Can Nums
 * 20240322, LYF, translate source files to C source, ASCII text. 
 * 20240324, Jason, add "CAN_configRxMsg" function.
 * 20240706, LYF, add feat: can and canfd timing setting APIs
                  CAN_setBitRate and CAN_setBitTiming;      
                  CAN_setDataBitRate and CAN_setDataBitTiming; 
 * 20240709, Hezhiyuan, Change ASSERT(HAL_CAN_isBaseValid(base)) to  ASSERT(CAN_isBaseValid(base));
 * 20240716, Danne, rewrite functions according to new macros.                   
 * 20240811, LYF, add two APIs to check T/Rbuf empty.
 * 20240907, LYF, fix code error in CAN_clearIntrStatus
 * 20241006, Jason, re-create most API functions and some structure defines.
 * 20241025, Jihong, recode the CAN-CTRL driver.
 * 20241217, Jihong, modify comments.
 */

#ifndef __CAN_H__
#define __CAN_H__

#ifdef __cplusplus
extern "C"
{
#endif

#include <stdbool.h>
#include <string.h>

#include "gs32_version.h"
#include "inc/hw_memmap.h"
#include "inc/hw_can.h"
#include "inc/hw_types.h"
#include "sysctl.h"

/**
 * @brief Base address array of CAN-CTRL IP instance.
 *
 */
#if (IS_GS32F00xx(0x12))

#define CAN_CTRL_NUMBER_OF_INSTANCES	2U

#define CAN_BASE_ADDR_ARRAY		{	\
		CANA_BASE,					\
		CANB_BASE,					\
}

#elif (IS_GS32F3xx(0x22))

#define CAN_CTRL_NUMBER_OF_INSTANCES	4U

#define CAN_BASE_ADDR_ARRAY		{	\
		CANA_BASE,					\
		CANB_BASE,					\
		CANC_BASE,					\
		CAND_BASE,					\
}

#else

#define CAN_CTRL_NUMBER_OF_INSTANCES	0U

#define CAN_BASE_ADDR_ARRAY		{}

#endif

/**
 * @brief Check if the base address is valid or not.
 *
 */
#define CAN_CHECK_BASE_VALID					1U

/**
 * @brief The maximum value of SSP value for transmission delay compensation.
 *
 */
#define CAN_TDC_TDCO_MAX						(0x7FU)

/**
 * @brief Maximum value of normal phase for segment 1 of CAN-CTRL instance
 * in CAN/CANFD mode
 *
 */
#define CAN_NBTP_NTSEG1_MAX						(63U)

/**
 * @brief Maximum value of normal phase for segment 2 of CAN-CTRL instance in CAN mode.
 *
 */
#define CAN_NBTP_NTSEG2_MAX_CAN_MODE			(7U)

/**
 * @brief Maximum value of normal phase for segment 2 of
 * CAN-CTRL instance in CANFD mode.
 *
 */
#define CAN_NBTP_NTSEG2_MAX_CANFD_MODE			(31U)

/**
 * @brief  Maximum value of normal phase for sync jump width of CAN-CTRL instance
 * in CAN/CANFD mode.
 *
 */
#define CAN_NBTP_NSJW_MAX						(15U)

/**
 * @brief Maximum value of normal phase for prescaler of CAN-CTRL instance in
 * CANFD mode.
 *
 */
#define CAN_NBTP_NBRP_MAX						(0xFFU)

/**
 * @brief Maximum value of data phase for segment 1 of CAN-CTRL instance in CANFD mode.
 *
 */
#define CAN_DBTP_DTSEG1_MAX						(15U)

/**
 * @brief Maximum value of data phase for segment 1 of CAN-CTRL instance in CANFD mode.
 *
 */
#define CAN_DBTP_DTSEG2_MAX						(7U)

/**
 * @brief Maximum value of data phase for segment 1 of CAN-CTRL instance in CANFD mode.
 *
 */
#define CAN_DBTP_DSJW_MAX						(7U)

/**
 * @brief Maximum value of data phase for segment 1 of CAN-CTRL instance in CANFD mode
 *
 */
#define CAN_DBTP_DBRP_MAX						(0xFFU)

/**
 * @brief Number of acceptance filter in CAN-CTRL.
 *
 */
#define NUMBER_OF_ACF							(16U)

#define CAN_ALL_INT_EN_MASK			(CAN_RTIE_RIE_M		|	\
									CAN_RTIE_ROIE_M		|	\
									CAN_RTIE_RFIE_M		|	\
									CAN_RTIE_RAFIE_M	|	\
									CAN_RTIE_TPIE_M		|	\
									CAN_RTIE_TSIE_M		|	\
									CAN_RTIE_EIE_M		|	\
									CAN_ERRINT_BEIE_M	|	\
									CAN_ERRINT_ALIE_M	|	\
									CAN_ERRINT_EPIE_M		\
									)

#define CAN_INT_STATUS_MASK			(CAN_RTIF_RIF_M		|	\
									CAN_RTIF_ROIF_M		|	\
									CAN_RTIF_RFIF_M		|	\
									CAN_RTIF_RAFIF_M	|	\
									CAN_RTIF_TPIF_M		|	\
									CAN_RTIF_TSIF_M		|	\
									CAN_RTIF_EIF_M		|	\
									CAN_RTIF_AIF_M		|	\
									CAN_ERRINT_BEIF_M	|	\
									CAN_ERRINT_ALIF_M	|	\
									CAN_ERRINT_EPIF_M		\
									)

/**
 * @brief Maximum payload supported by CAN-FD protocol in bytes.
 *
 */
#define CAN_MAX_PAYLOAD_BYTES					(64U)

/**
 * @brief CAN loopback mode.
 *
 * @param CAN_LPBK_MODE_INTERNAL CAN use internal loopback mode.
 * @param CAN_LPBK_MODE_EXTERNAL CAN use external loopback mode.
 */
typedef enum {
	CAN_LPBK_MODE_INTERNAL		= 0U,
	CAN_LPBK_MODE_EXTERNAL		= 1U,
} CAN_LpbkMode_t;

/**
 * @brief CAN-CTRL receive message ID filter types.
 *
 * @param CAN_IDTYPE_BOTH_STD_EXT acceptance filter accepts both standard
 * or extended frames.
 * @param CAN_IDTYPE_ONLY_STD acceptance filter accepts only standard frames.
 * @param CAN_IDTYPE_ONLY_EXT acceptance filter accepts only extended frames.
 */
typedef enum {
	CAN_IDTYPE_BOTH_STD_EXT		= 0x00,
	CAN_IDTYPE_ONLY_STD			= 0x01,
	CAN_IDTYPE_ONLY_EXT			= 0x02,
} CAN_IDFilterType_t;

/**
 * @brief Select Transmission buffer mode.
 *
 */
typedef enum {
	CAN_PTB_PRIORITY			= 0x00,		/* Transmission Primary buffer. */
	CAN_STB_SECONDARY			= 0x01		/* Transmission Secondary buffer. */
} CAN_TxBufMode_t;

/**
 * @brief Select Transmission buffer mode in secondary buffer mode.
 *
 */
typedef enum {
	CAN_TX_MODE_SINGLE_MSG		= 0x01U,	/* Only transfer one message. */
	CAN_TX_MODE_ALL_MSG			= 0x02U,	/* Transfer all messages. */
} CAN_SendMsgMode_t;

/**
 * @brief CAN-CTRL transmission buffer mode.
 *
 */
typedef enum {
	CAN_TS_MODE_FIFO			= 0x00, /* In FIFO mode frames are transmitted
										in the order in that they are written into the
										STB. */
	CAN_TS_MODE_PRIORITY		= 0x01,	/* In priority decision mode the frame with the
										highest
										priority in the STB is automatically transmitted
										first.
										The ID of a frame is used for the priority
										decision. */
} CAN_TxBufferMode_t;

/**
 * @brief Configuration of CAN-CTRL timestamp mode.
 *
 * @param CAN_DIS_TIMESTAMP Disable timestamp generation.
 * @param CAN_TIMESTAMP_SOF Timestamp generation at start of frame.
 * @param CAN_TIMESTAMP_EOF Timestamp generation at end of frame.
 */
typedef enum {
	CAN_DIS_TIMESTAMP			= 0x00U,
	CAN_TIMESTAMP_SOF			= 0x01U,
	CAN_TIMESTAMP_EOF			= 0x02U,
} CAN_TimeStampMode_t;

/**
 * @brief CAN-CTRL CAN bus error state.
 *
 */
typedef enum {
	CAN_STATUS_OK				= (0U << 0),
	CAN_STATUS_EWARN			= (1U << 1),	/* One of the error counters RECNT
												or TECNT is equal or bigger than EWL. */
	CAN_STATUS_EPASS			= (1U << 2),	/* Error Passive mode active. */
	CAN_STATUS_BUS_OFF			= (1U << 3),	/* Bus off state active. */
} CAN_BusErrState_t;

/**
 * @brief CAN-CTRL Tx message buffer status.
 *
 */
typedef enum {
	CAN_TXBUFFER_EMPTY			= 0x00,			/* Message buffer is empty. */
	CAN_TXBUFFER_LESS_THAN_HALF	= 0x01,			/* Message buffer is less than or
												equal to half full. */
	CAN_TXBUFFER_MORE_THAN_HALF	= 0x02,			/* Message buffer is more than half full. */
	CAN_TXBUFFER_FULL			= 0x03,			/* Message buffer is full. */
	CAN_TXBUFFER_ERR			= 0x04,			/* Message buffer error. */
} CAN_TxMsgBufStatus_t;

/**
 * @brief CAN-CTRL Rx message buffer status.
 *
 */
typedef enum {
	CAN_RXBUFFER_EMPTY					= 0x00,	/* Message buffer is empty. */
	CAN_RXBUFFER_LESS_THAN_ALMOST_FULL	= 0x01,	/* Message buffer is more than empty
												full and less than almost full. */
	CAN_RXBUFFER_MORE_THAN_ALMOST_FULL	= 0x02,	/* Message buffer is more than half
												full. */
	CAN_RXBUFFER_FULL					= 0x03,	/* Message buffer is full. */
	CAN_RXBUFFER_ERR					= 0x04,	/* Message buffer error. */
} CAN_RxMsgBufStatus_t;

/**
 * @brief Data Length Code.
 *
 * 0-8= CAN + CAN FD: transmit frame has 0-8 data bytes.
 * 9-15= CAN: transmit frame has 8 data bytes.
 * 9-15=CAN FD: transmit frame has 12/16/20/24/32/48/64 data bytes.
 */
typedef enum {
	CAN_DATA_LENGTH_0		= 0U,
	CAN_DATA_LENGTH_1		= 1U,
	CAN_DATA_LENGTH_2		= 2U,
	CAN_DATA_LENGTH_3		= 3U,
	CAN_DATA_LENGTH_4		= 4U,
	CAN_DATA_LENGTH_5		= 5U,
	CAN_DATA_LENGTH_6		= 6U,
	CAN_DATA_LENGTH_7		= 7U,
	CAN_DATA_LENGTH_8		= 8U,
	CAN_DATA_LENGTH_12		= 9U,
	CAN_DATA_LENGTH_16		= 10U,
	CAN_DATA_LENGTH_20		= 11U,
	CAN_DATA_LENGTH_24		= 12U,
	CAN_DATA_LENGTH_32		= 13U,
	CAN_DATA_LENGTH_48		= 14U,
	CAN_DATA_LENGTH_64		= 15U,
} CAN_MsgDataLength_t;

/**
 * @brief CAN-CTRL Operation Mode.
 *
 * @param CAN_OPERATION_MODE_NORMAL				The CAN-CTRL instance is not resetting.
 * @param CAN_OPERATION_MODE_SW_INIT			Reset CAN-CTRL instance.
 */
typedef enum {
	CAN_OPERATION_MODE_NORMAL	= 0U,
	CAN_OPERATION_MODE_SW_INIT	= 1U
} CAN_OperationMode_t;

/**
 * @brief CANFD frame format mode.
 *
 */
typedef enum {
	CAN_FD_ISO_11898_1			= 0U,			/* CANFD mode use ISO11898.1
												frame format. */
	CAN_FD_BOSCH_STANDARD		= 1U,			/* CANFD mode use BOSCH CANFD
												frame format. */
} CAN_FDISOMode_t;

/**
 * @brief Structure for MCAN initialization parameters.
 *
 * @param fdMode true is enable CANFD mode, false is disable CANFD mode.
 * @param brsEnable true is enable Bit Rate Switch in CANFD mode, false is
 * disable Bit Rate Switch.
 * @param listenOnlyEnable true is enable Listen Only mode, false is disable
 * Listen Only mode.
 * @param fdFrame CANFD frame type(@CAN_FDISOMode_t).
 * @param timestamp Set timestamp mode or disable timestamp.
 */
typedef struct {
	bool fdMode;
	bool listenOnlyEnable;
	CAN_FDISOMode_t fdFrame;
	CAN_TxBufferMode_t txbmode;
	CAN_TimeStampMode_t timestamp;
} CAN_InitParams_t;

/**
 * @brief CAN-CTRL Bit Rate Parameters.
 *
 * @param nomRatePrescalar Nominal Baud Rate Pre-scaler(Range:[0x0-0xFF]).
 * @param nomTimeSeg1 Nominal Time segment before sample point(Range:[0x0-0xFF]).
 * @param nomTimeSeg2 Nominal Time segment after sample point(Range:[0x0-0x7F]).
 * @param nomSynchJumpWidth Nominal (Re)Synchronization Jump Width(Range:[0x0-0x7F]).
 *
 * @param dataRatePrescalar Data Baud Rate Pre-scaler(Range:[0x0-0xFF]).
 * @param dataTimeSeg1 Data Time segment before sample point(Range:[0x0-0x1F]).
 * @param dataTimeSeg2 Data Time segment after sample point(Range:[0x0-0xF]).
 * @param dataSynchJumpWidth Data (Re)Synchronization Jump Width(Range:[0x0-0xF]).
 */
typedef struct {
	volatile uint8_t nomRatePrescalar;
	volatile uint8_t nomTimeSeg1;
	volatile uint8_t nomTimeSeg2;
	volatile uint8_t nomSynchJumpWidth;
	volatile uint8_t dataRatePrescalar;
	volatile uint8_t dataTimeSeg1;
	volatile uint8_t dataTimeSeg2;
	volatile uint8_t dataSynchJumpWidth;
} CAN_BitTimingParams_t;

/**
 * @brief CAN-CTRL IP instance ID filter configuration paramters.
 *
 * @param acode	1 - ACC bit value to compare with ID bit of the received message.
 * 				0 - ACC bit value to compare with ID bit of the received message.
 * @param amsk	1 - acceptance check for these bits of receive identifier disabled.
 * 				0 - acceptance check for these bits of receive identifier enable.
 * @param filerFrameType ID filter type(@CAN_IDFilterType_t).
 */
typedef struct {
	uint32_t acode;
	uint32_t amsk;
	CAN_IDFilterType_t filerFrameType;
} CAN_FilterElement_t;

/**
 * @brief CAN-CTRL instance receives message buffer structure.
 *
 * @param id CAN message identifier(Standard frame ID is 11 bit(10:0),
 * Extended frame ID is 29 bit(28:0)).
 * @param esi Error State Indicator.
 * This is a read-only status bit for RBUF and is not available in TBUF.
 * The protocol machine automatically embeds the correct value of ESI
 * into transmitted frames.
 * ESI is only included in CAN FD frames and does not exist in CAN 2.0 frames.
 * 0 - CAN node is error active
 * 1 - CAN node is error passive
 * ESI in RBUF is always low for CAN 2.0 frames.
 * The error state for transmission is shown with bit EPASS in register ERRINT.
 * @param dlc Data length code(0-64, @CAN_MsgDataLength_t).
 * @param brs Bit Rate Switch.
 * 0 - nominal / slow bit rate for the complete frame.
 * 1 - switch to data / fast bit rate for the data payload and the CRC.
 * Only CAN FD frames can switch the bitrate. Therefore BRS is forced to 0 if FDF =0.
 * @param fdf CAN FD frame.
 * 0 - CAN 2.0 frame (up to 8 bytes payload).
 * 1 - CAN FD frame (up to 64 bytes payload).
 * @param rtr Remote Transmission Request.
 * 0 - data frame.
 * 1 - remote frame.
 * Only CAN 2.0 frames can be remote frames. There is no remote frame for CAN FD.
 * Therefore RTR is forced to 0 if FDF =1 in TBUF and RBUF.
 * If a CAN FD frame is received with bit RRS=1, then this is ignored,
 * a data payload is expected for reception instead and RTR in RBUF is overridden
 * but the CRC of the frame is calculated with RRS=1.
 * @param ide Extended Frame Format.
 * 0 - Standard Format: ID(10:0)
 * 1 - Extended Format: ID(28:0)
 * @param tx Status bit TX in RBUF is set to 1 if the loop back mode (Chapter 3.9.10.4) is
 * activated and the core has received its own transmitted frame.
 * This can be useful if LBME=1 and
 * other nodes in the network do also transmissions.
 * @param koer KOER in RBUF has the same meaning as the bits KOER in register EALCAP.
 * KOER in RBUF becomes meaningful if RBALL=1.
 * @param cycletime The time-stamp CYCLE_TIME will be stored in RBUF only in TTCAN mode.
 * This is the cycle time at the SOF of this frame.
 * The cycle time of a reference message is always 0.
 * @param data[64] CAN message data buffer.
 * @param timestamp CAN message timestamp.
 * The Reception Time Stamps (RTS) for CiA 603 time-stamping are stored for each received
 * message at the end of the RBUF address range. Therefore in contrast to TTS,
 * RTS is related to the actual selected RBUF slot.
 */
typedef struct CAN_RxMessage {
	uint32_t id			: 29;
	uint32_t resved_0	: 2;
	uint32_t esi		: 1;
	uint32_t dlc		: 4;
	uint32_t brs		: 1;
	uint32_t fdf		: 1;
	uint32_t rtr		: 1;
	uint32_t ide		: 1;
	uint32_t resved_1	: 4;
	uint32_t tx			: 1;
	uint32_t koer		: 3;
	uint32_t cycletime	: 16;
	uint8_t data[CAN_MAX_PAYLOAD_BYTES];
	uint64_t timestamp;
} CAN_RxMessage_t;

/**
 * @brief CAN-CTRL instance transmission message buffer structure.
 *
 * @param id CAN message identifier(Standard frame ID is 11 bit(10:0),
 * Extended frame ID is 29 bit(28:0)).
 * @param ttsen Transmit Time-Stamp Enable.
 * In contrast to RTS, which is stored for every received frame, TTS is stored only for
 * the last transmitted frame if TTSEN=1. TTS is not related to the actual selected
 * TBUF slot.
 * @param dlc Data length code(0-64, @CAN_MsgDataLength_t).
 * @param brs Bit Rate Switch.
 * 0 - nominal / slow bit rate for the complete frame.
 * 1 - switch to data / fast bit rate for the data payload and the CRC.
 * Only CAN FD frames can switch the bitrate. Therefore BRS is forced to 0 if FDF =0.
 * @param fdf CAN FD frame.
 * 0 - CAN 2.0 frame (up to 8 bytes payload).
 * 1 - CAN FD frame (up to 64 bytes payload).
 * @param rtr Remote Transmission Request.
 * 0 - data frame.
 * 1 - remote frame.
 * Only CAN 2.0 frames can be remote frames. There is no remote frame for CAN FD.
 * Therefore RTR is forced to 0 if FDF =1 in TBUF and RBUF.
 * If a CAN FD frame is received with bit RRS=1, then this is ignored,
 * a data payload is expected for reception instead and RTR in RBUF is overridden
 * but the CRC of the frame is calculated with RRS=1.
 * @param ide Extended Frame Format.
 * 0 - Standard Format: ID(10:0)
 * 1 - Extended Format: ID(28:0)
 * @param data[64] CAN message data buffer.
 */
typedef struct CAN_TxMessage {
	uint32_t id			: 29;
	uint32_t resved_0	: 2;
	uint32_t ttsen		: 1;
	uint32_t dlc		: 4;
	uint32_t brs		: 1;
	uint32_t fdf		: 1;
	uint32_t rtr		: 1;
	uint32_t ide		: 1;
	uint32_t resved_1	: 24;
	uint8_t data[CAN_MAX_PAYLOAD_BYTES];
} CAN_TxMessage_t;

static const uint32_t CAN_BaseAddressArray[] = CAN_BASE_ADDR_ARRAY;
static const uint8_t CAN_DataLengthArray[16] = {0, 1, 2, 3, 4, 5, 6, 7, \
												8, 12, 16, 20, 24, 32, 48, 64};

/**
 * @brief Check if the CAN-CTRL IP instance is valid.
 *
 * @param base The base address of CAN-CTRL IP.
 * @return true The base address of CAN-CTRL controller is valid.
 * @return false The base address of CAN-CTRL controller is invalid.
 */
static inline bool
CAN_isBaseValid(uint32_t base)
{
#if CAN_CHECK_BASE_VALID

	for (uint8_t i = 0; i < CAN_CTRL_NUMBER_OF_INSTANCES; i++) {
		if (CAN_BaseAddressArray[i] == base)
			return true;
	}

	return false;

#else

	return true;

#endif
}

static inline bool
CAN_isInReset(uint32_t base)
{
	if (!CAN_isBaseValid(base))
		return false;

	if(HWREG(base + CAN_O_CFG_STAT) & CAN_CFG_STAT_RESET_M)
		return true;
	else
		return false;
}

static inline bool
CAN_isFDOpEnable(uint32_t base)
{
	if (!CAN_isBaseValid(base))
		return false;

	for (uint8_t i = 0; i < CAN_CTRL_NUMBER_OF_INSTANCES; i++) {
		if (CAN_BaseAddressArray[i] == base)
			return SysCtl_getCanFdEnable((E_CANX_PORT)i);
	}

	return false;
}

static inline bool
CAN_setFDMode(uint32_t base, bool enable)
{
	if (!CAN_isBaseValid(base))
		return false;

	for (uint8_t i = 0; i < CAN_CTRL_NUMBER_OF_INSTANCES; i++) {
		if (CAN_BaseAddressArray[i] == base) {
			SysCtl_setCanFdEnable((E_CANX_PORT)i, enable);
			return true;
		}
	}

	return false;
}

/**
 * @brief Configuration the CAN-CTRL transmit delay Compensation value in CANFD mode.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param TDCValue Transmission delay Compensation.
 */
static inline void
CAN_setTDCValue(uint32_t base, uint32_t TDCValue)
{
	HWREG(base + CAN_O_TDC) &= ~CAN_TDC_SSPOFF_M;

	HWREG(base + CAN_O_TDC) |= ((TDCValue << CAN_TDC_SSPOFF_S) & CAN_TDC_SSPOFF_M);
}

/**
 * @brief Enbale/Disable Transmission Delay Compensation.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param enable true is enable TDC, false is disable TDC.
 */
static inline void
CAN_enableTDC(uint32_t base, bool enable)
{
	if (enable)
		HWREG(base + CAN_O_TDC) |= CAN_TDC_TDCEN_M;
	else
		HWREG(base + CAN_O_TDC) &= ~CAN_TDC_TDCEN_M;
}

static inline bool
CAN_setFdTDC(uint32_t base, uint32_t TDCValue, bool enable)
{
	if (!CAN_isBaseValid(base))
		return false;

	CAN_enableTDC(base, enable);

	CAN_setTDCValue(base, TDCValue);

	return true;
}

/**
 * @brief Enable/Disable the CAN-CTRL listen only mode.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param enable true is enable listen only mode, false is disable listen only mode.
 */
static inline void
CAN_setListenOnlyMode(uint32_t base, bool enable)
{
	uint32_t reg_val = HWREG(base + CAN_O_TCMD);

	reg_val &= ~(CAN_TCMD_TSONE_M | CAN_TCMD_TSALL_M |
				CAN_TCMD_TPE_M | CAN_TCMD_TPA_M |
				CAN_TCMD_TSA_M | CAN_TCTRL_TSNEXT_M |
				CAN_RCTRL_RBALL_M | CAN_RCTRL_RREL_M |
				CAN_CFG_STAT_BUSOFF_M);

	if (enable)
		reg_val |= CAN_TCMD_LOM_M;
	else
		reg_val &= ~CAN_TCMD_LOM_M;

	HWREG(base + CAN_O_TCMD) = reg_val;
}

/**
 * @brief Configuration the CAN-CTRL transmit buffer mode.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param mode Transmit buffer secondary operation mode(@CAN_TxBufferMode_t).
 */
static inline void
CAN_setTransmitBufferMode(uint32_t base, CAN_TxBufferMode_t mode)
{
	uint32_t reg_val = HWREG(base + CAN_O_TCTRL);

	reg_val &= ~(CAN_TCMD_TSONE_M | CAN_TCMD_TSALL_M |
				CAN_TCMD_TPE_M | CAN_TCMD_TPA_M |
				CAN_TCMD_TSA_M | CAN_TCTRL_TSNEXT_M |
				CAN_RCTRL_RBALL_M | CAN_RCTRL_RREL_M |
				CAN_CFG_STAT_BUSOFF_M);

	if (mode == CAN_TS_MODE_FIFO)
		reg_val &= ~CAN_TCTRL_TSMODE_M;
	else if (mode == CAN_TS_MODE_PRIORITY)
		reg_val |= CAN_TCTRL_TSMODE_M;

	HWREG(base + CAN_O_TCTRL) = reg_val;
}

/**
 * @brief Configuration the CAN-CTRL timestamp mode.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param mode Transmit buffer secondary operation mode(@CAN_TimeStampMode_t).
 */
static inline void
CAN_setTimeStampMode(uint32_t base, CAN_TimeStampMode_t mode)
{
	if (mode == CAN_DIS_TIMESTAMP)
		HWREG(base + CAN_O_TIMECFG) &= ~CAN_TIMECFG_TIMEEN_M;
	else if (mode == CAN_TIMESTAMP_EOF) {
		HWREG(base + CAN_O_TIMECFG) |= CAN_TIMECFG_TIMEEN_M;
		HWREG(base + CAN_O_TIMECFG) |= CAN_TIMECFG_TIMEPOS_M;
	} else if (mode == CAN_TIMESTAMP_SOF) {
		HWREG(base + CAN_O_TIMECFG) |= CAN_TIMECFG_TIMEEN_M;
		HWREG(base + CAN_O_TIMECFG) &= ~CAN_TIMECFG_TIMEPOS_M;
	}
}

static inline bool
CAN_configMsgFilterEnable(uint32_t base, uint32_t filtNum)
{
	if (!CAN_isBaseValid(base))
		return false;

	if (filtNum >= NUMBER_OF_ACF)
		return false;

	HWREG(base + CAN_O_ACF_EN_0) |= ((1U << filtNum) << CAN_ACF_EN_0_AE_0_S);

	return true;
}

static inline bool
CAN_configMsgFilterDisable(uint32_t base, uint32_t filtNum)
{
	if (!CAN_isBaseValid(base))
		return false;

	if (filtNum >= NUMBER_OF_ACF)
		return false;

	HWREG(base + CAN_O_ACF_EN_0) &= ~((1U << filtNum) << CAN_ACF_EN_0_AE_0_S);

	return true;
}

/**
 * @brief Set the self acknowledge.
 * This mode is only enabled in external loopback mode.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param enable true is enable, false is disable.
 */
static inline void
setSelfACKnowledge(uint32_t base, bool enable)
{
	uint32_t reg_val = HWREG(base + CAN_O_RCTRL);

	if (enable)
		reg_val |= CAN_RCTRL_SACK_M;
	else
		reg_val &= ~CAN_RCTRL_SACK_M;

	HWREG(base + CAN_O_RCTRL) = reg_val;
}

static inline bool
CAN_getArbitrationLostCapture(uint32_t base, uint8_t *alc)
{
	if (!CAN_isBaseValid(base))
		return false;

	*alc = (uint8_t)((HWREG(base + CAN_O_EALCAP) & CAN_EALCAP_ALC_M) >> CAN_EALCAP_ALC_S);

	return true;
}

static inline CAN_RxMsgBufStatus_t
CAN_getRxBufState(uint32_t base)
{
	if (!CAN_isBaseValid(base))
		return CAN_RXBUFFER_ERR;

	return (CAN_RxMsgBufStatus_t)((HWREG(base + CAN_O_RCTRL) & CAN_RCTRL_RSTAT_M) >> CAN_RCTRL_RSTAT_S);
}

static inline CAN_TxMsgBufStatus_t
CAN_getTxSecondaryBufState(uint32_t base)
{
	if (!CAN_isBaseValid(base))
		return CAN_TXBUFFER_ERR;

	return (CAN_TxMsgBufStatus_t)((HWREG(base + CAN_O_TCTRL) & CAN_TCTRL_TSSTAT_M) >> CAN_TCTRL_TSSTAT_S);
}

static inline int
CAN_getTxPrimaryBufState(uint32_t base)
{
	if (!CAN_isBaseValid(base))
		return -1;

	return !!(HWREG(base + CAN_O_TCMD) & CAN_TCMD_TPE_M);
}

static inline bool
CAN_getErrorCount(uint32_t base, uint32_t *rxCount, uint32_t *txCount)
{
	if (!CAN_isBaseValid(base))
		return false;

	if (rxCount == NULL || txCount == NULL)
		return false;

	*txCount = (HWREG(base + CAN_O_TECNT) & CAN_TECNT_TECNT_M) >>
								CAN_TECNT_TECNT_S;

	*rxCount = (HWREG(base + CAN_O_RECNT) & CAN_RECNT_RECNT_M) >>
								CAN_RECNT_RECNT_S;

	return true;
}

static inline bool
CAN_setErrWarnLevel(uint32_t base, uint8_t level)
{
	if (!CAN_isBaseValid(base))
		return false;

	if (level > 15)
		return false;

	HWREG(base + CAN_O_LIMIT) |= ((level & (CAN_LIMIT_EWL_M >> CAN_LIMIT_EWL_S)) >>
									CAN_LIMIT_EWL_S);

	return true;
}

static inline int
CAN_getErrWarnLevel(uint32_t base)
{
	if (!CAN_isBaseValid(base))
		return -1;

	return ((HWREG(base + CAN_O_LIMIT) & CAN_LIMIT_EWL_M) >> CAN_LIMIT_EWL_S);
}

static inline bool
CAN_setRxBufAlmostFullLevel(uint32_t base, uint8_t level)
{
	if (!CAN_isBaseValid(base))
		return false;

	if (level > 15)
		return false;

		HWREG(base + CAN_O_LIMIT) &= ~CAN_LIMIT_AFWL_M;
		HWREG(base + CAN_O_LIMIT) |= ((level & (CAN_LIMIT_AFWL_M >> CAN_LIMIT_AFWL_S)) <<
										CAN_LIMIT_AFWL_S);

	return true;
}

static inline int
CAN_getRxBufAlmostFullLevel(uint32_t base)
{
	if (!CAN_isBaseValid(base))
		return -1;

	return ((HWREG(base + CAN_O_LIMIT) & CAN_LIMIT_AFWL_M) >> CAN_LIMIT_AFWL_S);
}

static inline bool
CAN_enableInterrupt(uint32_t base, uint32_t intFlag)
{
	if (!CAN_isBaseValid(base))
		return false;

	HWREG(base + CAN_O_RTIE) |= (intFlag & CAN_ALL_INT_EN_MASK);

	return true;
}

static inline bool
CAN_disableInterrupt(uint32_t base, uint32_t intFlag)
{
	if (!CAN_isBaseValid(base))
		return false;

	HWREG(base + CAN_O_RTIE) &= ~(intFlag & CAN_ALL_INT_EN_MASK);

	return true;
}

static inline uint32_t
CAN_getIntrStatus(uint32_t base)
{
	if (!CAN_isBaseValid(base))
		return 0;

	return (HWREG(base + CAN_O_RTIF) & CAN_INT_STATUS_MASK);
}

/**
 * @brief Set CAN-CTRL operation mode.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param mode The operation mode of CAN-CTRL(@CAN_OperationMode_t:Reset/Normal).
 * @return true is CAN-CTRL is initialized successfully.
 * @return false is CAN-CTRL is failed to initialize.
 */
bool CAN_setOpMode(uint32_t base, CAN_OperationMode_t mode);

/**
 * @brief CAN-CTRL instance start running.
 *
 * @param base The base address of CAN-CTRL IP.
 * @return true is CAN-CTRL is started successfully.
 * @return false is CAN-CTRL is failed to start.
 */
bool CAN_startModule(uint32_t base);

/**
 * @brief CAN-CTRL instance reset to initialization state.
 *
 * @param base The base address of CAN-CTRL IP.
 * @return true is CAN-CTRL is reset successfully.
 * @return false is CAN-CTRL is failed to reset.
 */
bool CAN_initModule(uint32_t base);

/**
 * @brief Check if the CAN_CTRL is in resetting.
 *
 * @param base The base address of CAN-CTRL IP.
 * @return true The CAN-CTRL Controller is resetting.
 * @return false The CAN-CTRL Controller is not resetting.
 */
bool CAN_isInReset(uint32_t base);

/**
 * @brief Enable/Disable CAN-CTRL IP CANFD mode.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param enable true is enable CANFD mode, false is disable CANFD mode.
 * @return true CAN-CTRL IP CANFD mode is enabled.
 * @return false CAN-CTRL IP CANFD mode is disabled.
 */
bool CAN_setFDMode(uint32_t base, bool enable);

/*
 * @brief Select CAN FD mode
 * @param[in] base, CAN base address
 * @param[in] mode, CAN_FD_ISO_11898_1 or CAN_FD_BOSCH_STANDARD.
 * @return true CAN-CTRL IP FD is enabled.
 * @return false CAN-CTRL IP FD is disabled.
 */
bool CAN_SetCanFdIsoMode(uint32_t base, CAN_FDISOMode_t mode);

/**
 * @brief Enable/Disable CAN-CTRL IP FD TDC.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param enable true is enable FD TDC, false is disable FD TDC.
 * @param TDCValue TDC value(Range:[0x0-0xFF]).
 * Suggest setting the value of TDC equal to t_seg1 + 2.
 * @return true CAN-CTRL IP FD TDC is enabled.
 * @return false CAN-CTRL IP FD TDC is disabled.
 */
bool CAN_setFdTDC(uint32_t base, uint32_t TDCValue, bool enable);

/**
 * @brief Check if the CAN-CTRL IP CANFD mode is enabled.
 *
 * @param base The base address of CAN-CTRL IP.
 * @return true CAN-CTRL IP CANFD mode is enabled.
 * @return false CAN-CTRL IP CANFD mode is disabled.
 */
bool CAN_isFDOpEnable(uint32_t base);

/**
 * @brief CAN-Ctrl instance initialize.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param initParams Initialize CAN-CTRL instance parameters.
 * @return true is CAN-CTRL is initialized successfully.
 * @return false is CAN-CTRL is failed to initialize.
 */
bool CAN_initConfig(uint32_t base, const CAN_InitParams_t *initParams);

/**
 * @brief Set the CAN-CTRL CAN arbitration phase bit rate.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param prescaler Nominal Baud Rate Pre-scaler(Range:[0x0-0xFF]).
 * The value of parameter @prescaler should be equal to the actual value minus 1.
 * @param tSeg1 Nominal Time segment before sample point(Range:[0x0-0xFF]).
 * The value of parameter @tSeg1 should be equal to the actual value minus 2.
 * @param tSeg2 Nominal Time segment after sample point(Range:[0x0-0x7F]).
 * The value of parameter @tSeg2 should be equal to the actual value minus 1.
 * @param sjw Nominal (Re)Synchronization Jump Width(Range:[0x0-0x7F]).
 * The value of parameter @sjw should be equal to the actual value minus 1.
 * @return true Set the bit rate is successfully.
 * @return false Set the bit rate is failed.
 */
bool CAN_setBitTiming(uint32_t base, uint16_t prescaler,
					uint16_t tSeg1, uint16_t tSeg2, uint16_t sjw);

/**
 * @brief Set the CAN-CTRL CANFD data phase bit rate.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param prescaler Data phase Baud Rate Pre-scaler(Range:[0x0-0xFF]).
 * The value of parameter @prescaler should be equal to the actual value minus 1.
 * @param tSeg1 Data phase Time segment before sample point(Range:[0x0-0xFF]).
 * The value of parameter @tSeg1 should be equal to the actual value minus 2.
 * @param tSeg2 Data phase Time segment after sample point(Range:[0x0-0x7F]).
 * The value of parameter @tSeg2 should be equal to the actual value minus 1.
 * @param sjw Data phase (Re)Synchronization Jump Width(Range:[0x0-0x7F]).
 * The value of parameter @sjw should be equal to the actual value minus 1.
 * @return true Set the bit rate is successfully.
 * @return false Set the bit rate is failed.
 */
bool CAN_setDataBitTiming(uint32_t base, uint16_t prescaler,
						uint16_t tSeg1, uint16_t tSeg2, uint16_t sjw);

/**
 * @brief Set the CAN-CTRL CAN bit rate in arbitration phase.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param clockFreq CAN-CTRL instance main clock frequency(Hz).
 * @param bitRate Targets bit rate(bps).
 * @param bitTime Targets bit time(bittime = frequency/(prescalar * (tSeg1 + tSeg2))).
 * @return true Set the bit rate is successfully.
 * @return false Set the bit rate is failed.
 */
bool CAN_setBitRate(uint32_t base, uint32_t clockFreq,
					uint32_t bitRate, uint16_t bitTime);

/**
 * @brief Set the CAN-CTRL CAN bit rate in data phase.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param clockFreq CAN-CTRL instance main clock frequency(Hz).
 * @param bitRate Targets bit rate(bps).
 * @param bitTime Targets bit time(bittime = frequency/(prescalar * (tSeg1 + tSeg2))).
 * @return true Set the bit rate is successfully.
 * @return false Set the bit rate is failed.
 */
bool CAN_setDataBitRate(uint32_t base, uint32_t clockFreq,
						uint32_t bitRate, uint16_t bitTime);

/**
 * @brief Configuration the CAN-CTRL bit rate in arbitration phase.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param clockFreq CAN-CTRL instance main clock frequency(Hz).
 * @param bitRate Targets bit rate(bps).
 * @param sample_point CAN2.0 sample point(S_seg1 / (S_seg1 + S_seg2)).
 * @return true Configuration the bit rate is successfully.
 * @return false Configuration the bit rate is failed.
 */
bool CAN_setBitRateSamplePoint(uint32_t base, uint32_t clockFreq,
							uint32_t bitrate, float32 sample_point);

/**
 * @brief Configuration the CAN-CTRL bit rate in data phase.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param clockFreq CAN-CTRL instance main clock frequency(Hz).
 * @param bitrate Targets bit rate(bps).
 * @param sample_point CANFD sample point(F_seg1 / (F_seg1 + F_seg2)).
 * @return true Configuration the bit rate is successfully.
 * @return false Configuration the bit rate is failed.
 */
bool CAN_setDataBitRateSamplePoint(uint32_t base, uint32_t clockFreq,
								uint32_t bitrate, float32 sample_point);

/**
 * @brief Configuration the CAN-CTRL receive message filter.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param filtNum The filter number.
 * @param elem The filter element(@CAN_FilterElement_t).
 * @return true Configuration the filter is successfully.
 * @return false Configuration the filter is failed.
 */
bool CAN_configMsgFilter(uint32_t base, uint32_t filtNum,
						const CAN_FilterElement_t *elem);

/**
 * @brief Enable the CAN ID filter.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param filtNum The filter number.
 * @return true Enable the filter is successfully.
 * @return false Enable the filter is failed.
 */
bool CAN_configMsgFilterEnable(uint32_t base, uint32_t filtNum);

/**
 * @brief Disable the CAN ID filter.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param filtNum The filter number.
 * @return true Disable the filter is successfully.
 * @return false Disable the filter is failed.
 */
bool CAN_configMsgFilterDisable(uint32_t base, uint32_t filtNum);

/**
 * @brief Add a CAN ID filter.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param acode The filter acceptance code.
 * @param mask The filter acceptance mask.
 * @param type The filter type(standard or extended, @CAN_IDFilterType_t).
 * @return true Add a filter is successfully.
 * @return false Add a filter is failed.
 */
bool CAN_addMsgFilter(uint32_t base, uint32_t acode,
					uint32_t mask, CAN_IDFilterType_t type);

/**
 * @brief Enable/Disable the loopback mode.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param lpbkMode Choose between internal or external loop modes(@CAN_LpbkMode_t).
 * @param enable true is enable, false is disable.
 * @return true Enable/Disable the loopback mode is successfully.
 * @return false Enable/Disable the loopback mode is failed.
 */
bool CAN_lpbkModeEnable(uint32_t base, CAN_LpbkMode_t lpbkMode, bool enable);

/**
 * @brief Enable/Disable the CAN-CTRL auto retransmission.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param enable true is enable auto retransmission, false is disable auto retransmission.
 */
bool CAN_setAutoRetransmission(uint32_t base, bool enable);

/**
 * @brief Get the number of errors during reception and transmission for the CAN-CTRL.
 * 
 * @param base The base address of CAN-CTRL IP.
 * @param rxCount The pointer to the reveive error counter.
 * @param txCount The pointer to the transmit error counter.
 * @return true Get the error counter successfully.
 * @return false Get the error counter failed.
 */
bool CAN_getErrorCount(uint32_t base, uint32_t *rxCount, uint32_t *txCount);

/**
 * @brief Get the CAN bus error status.
 *
 * @param base The base address of CAN-CTRL IP.
 * @return int The CAN bus error status(@CAN_IDFilterType_t).
 * If return value is 0, the CAN bus is in normal state,
 * otherwise the CAN bus is in error state or input parameter is invalid.
 */
int CAN_getErrStatus(uint32_t base);

/**
 * @brief Clear the CAN bus off state.
 * @note Suggest using it only during debugging.
 *
 * @param base The base address of CAN-CTRL IP.
 * @return true clear the bus off state successfully.
 * @return false clear the bus off state failed.
 */
bool CAN_clearBusOff(uint32_t base);

/**
 * @brief Obtain the bit position of arbitration failure.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param alc Point to the memory space for saving ALC.
 * @return true Obtain the bit position of arbitration failure is successfully.
 * @return false Obtain the bit position of arbitration failure is failed.
 * */
bool CAN_getArbitrationLostCapture(uint32_t base, uint8_t *alc);

/**
 * @brief Get the status of the receive buffer.
 *
 * @param base The base address of CAN-CTRL IP.
 * @return CAN_TxBufMode_t CAN-CTRL receive buffer status(@CAN_TxBufMode_t).
 */
CAN_RxMsgBufStatus_t CAN_getRxBufState(uint32_t base);

/**
 * @brief Get the status of the secondary buffer.
 *
 * @param base The base address of CAN-CTRL IP.
 * @return CAN_TxMsgBufStatus_t CAN-CTRL secondary buffer status(@CAN_TxMsgBufStatus_t).
 */
CAN_TxMsgBufStatus_t CAN_getTxSecondaryBufState(uint32_t base);

/**
 * @brief Get the status of the primary buffer.
 *
 * @param base The base address of CAN-CTRL IP.
 * @return If the return value is non-zero, the Primary buffer is full.
 * If the return value is 0, the Primary buffer is empty.
 * If the return value is 1, the Primary buffer is full.
 */
int CAN_getTxPrimaryBufState(uint32_t base);

/**
 * @brief Write a CAN message to the secondary buffer.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param pTxFrame The pointer to the CAN message.
 * @return true Write successfully.
 * @return false Write failed.
 */
bool CAN_WriteTxMsgToSTB(uint32_t base, CAN_TxMessage_t *pTxFrame);

/**
 * @brief Start transmission once message in Tx buffer.
 * Only uesd in secondary buffer mode.
 *
 * @param base The base address of CAN-CTRL IP.
 * @return true Start transmission successfully.
 * @return false Start transmission failed.
 */
bool CAN_startTransmissionOnce(uint32_t base);

/**
 * @brief Start transmission all message in Tx buffer.
 * Only uesd in secondary buffer mode.
 * 
 * @param base The base address of CAN-CTRL IP.
 * @return true Start transmission successfully.
 * @return false Start transmission failed.
 */
bool CAN_startTransmissionAll(uint32_t base);

/**
 * @brief Start transmission high priority message in primary buffer.
 *
 * @param base The base address of CAN-CTRL IP.
 * @return true Start transmission successfully.
 * @return false Start transmission failed.
 */
bool CAN_startTransmissionHighPrimaryMsg(uint32_t base);

/**
 * @brief Get the data length of the message.
 *
 * @param size The data length of the message.
 * @return int If the return value is less than zero, the length retrieval fails
 * otherwise, the length is returned.
 */
int CAN_getMessageLength(CAN_MsgDataLength_t size);

/**
 * @brief Transmit a CAN message.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param pTxFrame The pointer to the CAN message.
 * @return true Successfully transmitted CAN message.
 * @return false Transmission of CAN message failed.
 */
bool CAN_transmitMsg(uint32_t base, CAN_TxMessage_t *pTxFrame);

/**
 * @brief Transmit a CAN message in primary buffer.
 * The Primary buff is used to transmit a CAN message with the highest priority.
 * This message priority is different from the arbitration priority defined in the CAN
 * protocol.It is an absolute priority, and the messages stored in this buffer must be
 * sent out with the highest priority.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param pTxFrame The pointer to the CAN message.
 * @return true Successfully transmitted CAN message.
 * @return false Transmission of CAN message failed.
 */
bool CAN_transmitHighPriorityMsg(uint32_t base, CAN_TxMessage_t *pTxFrame);

/**
 * @brief Aborts a transmission from STB which has been requested but not started yet.
 * For a TSONE transmission, only one frame is aborted while for a TSALL
 * Transmission, all frames are aborted.
 *
 * @param base The base address of CAN-CTRL IP.
 * @return true Abort successfully.
 * @return false Abort failed.
 */
bool CAN_transmitSecondaryAbort(uint32_t base);

/**
 * @brief Aborts a transmission from Primary buffer.
 *
 * @param base The base address of CAN-CTRL IP.
 * @return true Abort successfully.
 * @return false Abort failed.
 */
bool CAN_transmitPrimaryAbort(uint32_t base);

/**
 * @brief Release current Rx buffer.
 *
 * @param base The base address of CAN-CTRL IP.
 * @return true Release successfully.
 * @return false Release failed.
 */
bool CAN_releaseRxBuf(uint32_t base);

/**
 * @brief Receive a CAN message.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param pRxFrame The pointer to the CAN message.
 * @return true Received CAN message successfully.
 * @return false Received CAN message failed.
 */
bool CAN_receiveMsg(uint32_t base, CAN_RxMessage_t *pRxFrame);

/**
 * @brief Set error warning level.
 *
 * Programmable Error Warning Limit = (EWL+1)*8. Possible Limit values: 8, 16, 64, 128.
 * The value of EWL controls EIF.EWL needs to be transferred using CDC from host
 * to CAN clock domain. During transfer EWL register bits are write-locked for
 * the host for a few clocks until CDC is complete.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param level The error warning level (the level valid range is 0~15).
 * @return true Set error level successfully.
 * @return false Set error level failed.
 */
bool CAN_setErrWarnLevel(uint32_t base, uint8_t level);

/**
 * @brief Obtain the error warning level.
 *
 * @param base The base address of CAN-CTRL IP.
 * @return int The error warning level. If the return value is less than 0,
 * it indicates an error.
 */
int CAN_getErrWarnLevel(uint32_t base);

/**
 * @brief Set the Rx buffer almost full level.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param level The almost full level (the level valid range is 0~15).
 * @return true Set almost full level successfully.
 * @return false Set almost full level failed.
 */
bool CAN_setRxBufAlmostFullLevel(uint32_t base, uint8_t level);

/**
 * @brief Obtain the Rx buffer almost full level.
 *
 * @param base The base address of CAN-CTRL IP.
 * @return int The almost full level. If the return value is less than 0,
 * it indicates an error.
 */
int CAN_getRxBufAlmostFullLevel(uint32_t base);

/**
 * @brief Enable CAN transfer interrupt. This API enables interrupts during transmission.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param intFlag The interrupt flag.
 * @CAN_RTIE_RIE_M Receive Interrupt Enable.
 * @CAN_RTIE_ROIE_M RB Overrun Interrupt Enable.
 * @CAN_RTIE_RFIE_M RB Full Interrupt Enable.
 * @CAN_RTIE_RAFIE_M RB Almost Full Interrupt Enable.
 * @CAN_RTIE_TPIE_M Transmission Primary Interrupt Enable.
 * @CAN_RTIE_TSIE_M Transmission Secondary Interrupt Enable
 * @CAN_RTIE_EIE_M Error Interrupt Enable.
 * @CAN_ERRINT_ALIE_M Arbitration Lost Interrupt Enable.
 * @CAN_ERRINT_BEIE_M Bus Error Interrupt Enable.
 * @CAN_ERRINT_EPIE_M Error Passive Interrupt Enable.
 *
 * @return true Enable interrupt successfully.
 * @return false Enable interrupt failed.
 */
bool CAN_enableInterrupt(uint32_t base, uint32_t intFlag);

/**
 * @brief Disable CAN transfer interrupt.
 *
 * @param baseAddr The base address of CAN-CTRL IP.
 * @param intFlag The interrupt flag.
 * @CAN_RTIE_RIE_M Receive Interrupt Disable.
 * @CAN_RTIE_ROIE_M RB Overrun Interrupt Disable.
 * @CAN_RTIE_RFIE_M RB Full Interrupt Disable.
 * @CAN_RTIE_RAFIE_M RB Almost Full Interrupt Disable.
 * @CAN_RTIE_TPIE_M Transmission Primary Interrupt Disable.
 * @CAN_RTIE_TSIE_M Transmission Secondary Interrupt Disable
 * @CAN_RTIE_EIE_M Error Interrupt Disable.
 * @CAN_ERRINT_ALIE_M Arbitration Lost Interrupt Disable.
 * @CAN_ERRINT_BEIE_M Bus Error Interrupt Disable.
 * @CAN_ERRINT_EPIE_M Error Passive Interrupt Disable.
 *
 * @return true Disable interrupt successfully.
 * @return false Disable interrupt failed.
 */
bool CAN_disableInterrupt(uint32_t base, uint32_t intFlag);

/**
 * @brief Get the interrupt status.
 *
 * @param base The base address of CAN-CTRL IP.
 * @return uint32_t The interrupt status.
 * @CAN_RTIF_RIF_M Data or a remote frame has been received and is
 * available in the receive buffer.
 * @CAN_RTIF_ROIF_M At least one received message has been overwritten in the RB.
 * @CAN_RTIF_RFIF_M All RBs are full. If no RB will be released until
 * the next valid message
 * is received,the oldest message will be lost.
 * @CAN_RTIF_RAFIF_M RB Almost Full Interrupt Flag.
 * @CAN_RTIF_TPIF_M The requested transmission of the PTB has been successfully comp
 * @CAN_RTIF_TSIF_M The requested transmission of the STB has been successfully completed.
 * @CAN_RTIF_EIF_M The border of the error warning limit has been crossed
 * in either direction, or the BUSOFF bit has been changed in either direction.
 * @CAN_RTIF_AIF_M After setting TPA or TSA the appropriated message(s) have been aborted.
 * It is recommended to not set both TPA and TSA simultaneously because both source AIF.
 * @CAN_ERRINT_BEIF_M EPIF will be activated if the error status changes from error
 * active to error passive or vice versa and if this interrupt is enabled.
 * @CAN_ERRINT_ALIF_M Arbitration Lost Interrupt Flag.
 * @CAN_ERRINT_EPIF_M Bus Error Interrupt Flag.
 * 
 */
uint32_t CAN_getIntrStatus(uint32_t base);

/**
 * @brief Get the interrupt status.
 *
 * @param base The base address of CAN-CTRL IP.
 * @return uint32_t The interrupt status.
 * @CAN_RTIF_RIF_M Data or a remote frame has been received and is available
 * in the receive buffer.
 * @CAN_RTIF_ROIF_M At least one received message has been overwritten in the RB.
 * @CAN_RTIF_RFIF_M All RBs are full. If no RB will be released until
 * the next valid message
 * is received,the oldest message will be lost.
 * @CAN_RTIF_RAFIF_M RB Almost Full Interrupt Flag.
 * @CAN_RTIF_TPIF_M The requested transmission of the PTB has been successfully comp
 * @CAN_RTIF_TSIF_M The requested transmission of the STB has been successfully completed.
 * @CAN_RTIF_EIF_M The border of the error warning limit has been crossed
 * in either direction, or the BUSOFF bit has been changed in either direction.
 * @CAN_RTIF_AIF_M After setting TPA or TSA the appropriated message(s) have been aborted.
 * It is recommended to not set both TPA and TSA simultaneously because both source AIF.
 * @CAN_ERRINT_BEIF_M EPIF will be activated if the error status changes from error
 * active to error passive or vice versa and if this interrupt is enabled.
 * @CAN_ERRINT_ALIF_M Arbitration Lost Interrupt Flag.
 * @CAN_ERRINT_EPIF_M Bus Error Interrupt Flag.
 */
uint32_t CAN_clearIntrStatus(uint32_t base, uint32_t intFlag);

/**
 * @brief Select the next secondary buffer.
 *
 * @param base The base address of CAN-CTRL IP.
 */
void CAN_selectNextSecondaryBuf(uint32_t base);

/**
 * @brief Start transmission message in Tx buffer.
 * Only uesd in secondary buffer mode.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param mode Select the transmission mode(@CAN_SendMsgMode_t).
 */
void CAN_startTransmission(uint32_t base, CAN_SendMsgMode_t mode);

/**
 * @brief Start transmission high priority message.
 *
 * @param base The base address of CAN-CTRL IP.
 */
void CAN_startTransmissionPrimaryMsg(uint32_t base);

#ifdef __cplusplus
}
#endif

#endif
