
//###########################################################################
//
// FILE:   clb.h
//
//
//###########################################################################
//
//
//
// $
//###########################################################################

#ifndef CLB_H
#define CLB_H

//*****************************************************************************
//
// If building with a C++ compiler, make all of the definitions in this header
// have a C binding.
//
//*****************************************************************************
#ifdef __cplusplus
extern "C"
{
#endif

//*****************************************************************************
//
//! \addtogroup clb_api CLB
//! @{
//
//*****************************************************************************

#include <stdint.h>
#include <stdbool.h>
#include "gs32_version.h"
#include "inc/hw_clb.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "cpu.h"
#include "debug.h"

#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)

//*****************************************************************************
//
// Address offsets from LOGICCFG to LOGICCTL and DATAEXCH register memory maps
//
//*****************************************************************************
#define CLB_LOGICCTL                0x0100U
#define CLB_DATAEXCH                0x0200U

//*****************************************************************************
//
// Address offsets for CLB-internal memory space
//
//*****************************************************************************
#define CLB_ADDR_COUNTER_0_LOAD     0xA4U
#define CLB_ADDR_COUNTER_1_LOAD     0xA8U
#define CLB_ADDR_COUNTER_2_LOAD     0xACU

#define CLB_ADDR_COUNTER_0_MATCH1   0x10U
#define CLB_ADDR_COUNTER_1_MATCH1   0x18U
#define CLB_ADDR_COUNTER_2_MATCH1   0x20U

#define CLB_ADDR_COUNTER_0_MATCH2   0x14U
#define CLB_ADDR_COUNTER_1_MATCH2   0x1CU
#define CLB_ADDR_COUNTER_2_MATCH2   0x24U

#define CLB_ADDR_HLC_R0             0x160U
#define CLB_ADDR_HLC_R1             0x164U
#define CLB_ADDR_HLC_R2             0x168U
#define CLB_ADDR_HLC_R3             0x16cU

#define CLB_ADDR_HLC_BASE           0x20U
#define CLB_NUM_HLC_INSTR           31U

//*****************************************************************************
//
// PUSH/PULL FIFO size (32-bit registers)
//
//*****************************************************************************
#define CLB_FIFO_SIZE               4U

//*****************************************************************************
//
// Key to enable writes to the CLB registers
//
//*****************************************************************************
#define CLB_LOCK_KEY                0x5A5AU

//*****************************************************************************
//
// Shift and masks needed by the API for Input selection
//
//*****************************************************************************
#define CLB_LCL_MUX_SEL_MISC_INPUT_SEL_M          0x20U
#define CLB_LCL_MUX_SEL_MISC_INPUT_SEL_S          28U
#define CLB_LCL_MUX_SEL_MISC_INPUT_SEL_BITM       (uint32_t)1U

//*****************************************************************************
//
//! Values that can be passed to control the CLB output enable signal. It can
//! be passed to CLB_setOutputMask() as the \e outputMask parameter.
//
//*****************************************************************************
#define CLB_OUTPUT_00 0x00000001U //!< Mask for CLB OUTPUT ENABLE/DISABLE 0
#define CLB_OUTPUT_01 0x00000002U //!< Mask for CLB OUTPUT ENABLE/DISABLE 1
#define CLB_OUTPUT_02 0x00000004U //!< Mask for CLB OUTPUT ENABLE/DISABLE 2
#define CLB_OUTPUT_03 0x00000008U //!< Mask for CLB OUTPUT ENABLE/DISABLE 3
#define CLB_OUTPUT_04 0x00000010U //!< Mask for CLB OUTPUT ENABLE/DISABLE 4
#define CLB_OUTPUT_05 0x00000020U //!< Mask for CLB OUTPUT ENABLE/DISABLE 5
#define CLB_OUTPUT_06 0x00000040U //!< Mask for CLB OUTPUT ENABLE/DISABLE 6
#define CLB_OUTPUT_07 0x00000080U //!< Mask for CLB OUTPUT ENABLE/DISABLE 7
#define CLB_OUTPUT_08 0x00000100U //!< Mask for CLB OUTPUT ENABLE/DISABLE 8
#define CLB_OUTPUT_09 0x00000200U //!< Mask for CLB OUTPUT ENABLE/DISABLE 9
#define CLB_OUTPUT_10 0x00000400U //!< Mask for CLB OUTPUT ENABLE/DISABLE 10
#define CLB_OUTPUT_11 0x00000800U //!< Mask for CLB OUTPUT ENABLE/DISABLE 11
#define CLB_OUTPUT_12 0x00001000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 12
#define CLB_OUTPUT_13 0x00002000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 13
#define CLB_OUTPUT_14 0x00004000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 14
#define CLB_OUTPUT_15 0x00008000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 15
#define CLB_OUTPUT_16 0x00010000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 16
#define CLB_OUTPUT_17 0x00020000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 17
#define CLB_OUTPUT_18 0x00040000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 18
#define CLB_OUTPUT_19 0x00080000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 19
#define CLB_OUTPUT_20 0x00100000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 20
#define CLB_OUTPUT_21 0x00200000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 21
#define CLB_OUTPUT_22 0x00400000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 22
#define CLB_OUTPUT_23 0x00800000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 23
#define CLB_OUTPUT_24 0x01000000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 24
#define CLB_OUTPUT_25 0x02000000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 25
#define CLB_OUTPUT_26 0x04000000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 26
#define CLB_OUTPUT_27 0x08000000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 27
#define CLB_OUTPUT_28 0x10000000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 28
#define CLB_OUTPUT_29 0x20000000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 29
#define CLB_OUTPUT_30 0x40000000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 30
#define CLB_OUTPUT_31 0x80000000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 31

//*****************************************************************************
//
//! Values that can be passed to select CLB input signal
//
//*****************************************************************************
typedef enum
{
    CLB_IN0 = 0,                    //!< Input 0
    CLB_IN1 = 1,                    //!< Input 1
    CLB_IN2 = 2,                    //!< Input 2
    CLB_IN3 = 3,                    //!< Input 3
    CLB_IN4 = 4,                    //!< Input 4
    CLB_IN5 = 5,                    //!< Input 5
    CLB_IN6 = 6,                    //!< Input 6
    CLB_IN7 = 7                     //!< Input 7
} CLB_Inputs;

//*****************************************************************************
//
//! Values that can be passed to select CLB input signal
//
//*****************************************************************************


//*****************************************************************************
//
//! Values that can be passed to select CLB output signal. It can be passed to
//! CLB_configOutputLUT() as the \e outID parameter.
//
//*****************************************************************************
typedef enum
{
    CLB_OUT0 = 0,                   //!< Output 0
    CLB_OUT1 = 1,                   //!< Output 1
    CLB_OUT2 = 2,                   //!< Output 2
    CLB_OUT3 = 3,                   //!< Output 3
    CLB_OUT4 = 4,                   //!< Output 4
    CLB_OUT5 = 5,                   //!< Output 5
    CLB_OUT6 = 6,                   //!< Output 6
    CLB_OUT7 = 7                    //!< Output 7
} CLB_Outputs;

//*****************************************************************************
//
//! Values that can be passed to select CLB AOC signal. It can be passed to
//! CLB_configAOC() as the \e aocID parameter.
//! AOC is the Asynchronous Output Conditioning block.
//
//*****************************************************************************
typedef enum
{
    CLB_AOC0 = 0,                   //!< AOC 0
    CLB_AOC1 = 1,                   //!< AOC 1
    CLB_AOC2 = 2,                   //!< AOC 2
    CLB_AOC3 = 3,                   //!< AOC 3
    CLB_AOC4 = 4,                   //!< AOC 4
    CLB_AOC5 = 5,                   //!< AOC 5
    CLB_AOC6 = 6,                   //!< AOC 6
    CLB_AOC7 = 7                    //!< AOC 7
} CLB_AOCs;

//*****************************************************************************
//
//! Values that can be passed to set/clear CLB SW release signals. It can
//! be passed to CLB_writeSWReleaseControl() as the \e inID parameter.
//
//*****************************************************************************
typedef enum
{
    CLB_SW_RLS_CTRL0 = 0,           //!< SW RLS CTRL 0
    CLB_SW_RLS_CTRL1 = 1,           //!< SW RLS CTRL 1
    CLB_SW_RLS_CTRL2 = 2,           //!< SW RLS CTRL 2
    CLB_SW_RLS_CTRL3 = 3,           //!< SW RLS CTRL 3
    CLB_SW_RLS_CTRL4 = 4,           //!< SW RLS CTRL 4
    CLB_SW_RLS_CTRL5 = 5,           //!< SW RLS CTRL 5
    CLB_SW_RLS_CTRL6 = 6,           //!< SW RLS CTRL 6
    CLB_SW_RLS_CTRL7 = 7            //!< SW RLS CTRL 7
} CLB_SWReleaseCtrl;

//*****************************************************************************
//
//! Values that can be passed to set/clear CLB SW release signals. It can
//! be passed to CLB_writeSWGateControl() as the \e inID parameter.
//
//*****************************************************************************
typedef enum
{
    CLB_SW_GATE_CTRL0 = 0,           //!< SW GATE CTRL 0
    CLB_SW_GATE_CTRL1 = 1,           //!< SW GATE CTRL 1
    CLB_SW_GATE_CTRL2 = 2,           //!< SW GATE CTRL 2
    CLB_SW_GATE_CTRL3 = 3,           //!< SW GATE CTRL 3
    CLB_SW_GATE_CTRL4 = 4,           //!< SW GATE CTRL 4
    CLB_SW_GATE_CTRL5 = 5,           //!< SW GATE CTRL 5
    CLB_SW_GATE_CTRL6 = 6,           //!< SW GATE CTRL 6
    CLB_SW_GATE_CTRL7 = 7            //!< SW GATE CTRL 7
} CLB_SWGateCtrl;

//*****************************************************************************
//
//! Values that can be passed to select CLB counter. It can be passed to
//! CLB_configCounterLoadMatch() as the \e counterID parameter.
//
//*****************************************************************************
typedef enum
{
    CLB_CTR0 = 0,                   //!< Counter 0
    CLB_CTR1 = 1,                   //!< Counter 1
    CLB_CTR2 = 2                    //!< Counter 2
} CLB_Counters;

//*****************************************************************************
//
//! Values that can be passed to CLB_getRegister() as the \e registerID
//! parameter.
//
//*****************************************************************************
typedef enum
{
    CLB_REG_HLC_R0 = CLB_O_DBG_R0,  //!< HLC R0 register
    CLB_REG_HLC_R1 = CLB_O_DBG_R1,  //!< HLC R1 register
    CLB_REG_HLC_R2 = CLB_O_DBG_R2,  //!< HLC R2 register
    CLB_REG_HLC_R3 = CLB_O_DBG_R3,  //!< HLC R3 register
    CLB_REG_CTR_C0 = CLB_O_DBG_C0,  //!< Counter 0 register
    CLB_REG_CTR_C1 = CLB_O_DBG_C1,  //!< Counter 1 register
    CLB_REG_CTR_C2 = CLB_O_DBG_C2   //!< Counter 2 register
} CLB_Register;

//*****************************************************************************
//
//! Values that can be passed to CLB_selectInputFilter() as the \e filterType
//! parameter.
//
//*****************************************************************************
typedef enum
{
    CLB_FILTER_NONE         = 0,    //!< No filtering
    CLB_FILTER_RISING_EDGE  = 1,    //!< Rising edge detect
    CLB_FILTER_FALLING_EDGE = 2,    //!< Falling edge detect
    CLB_FILTER_ANY_EDGE     = 3     //!< Any edge detect
} CLB_FilterType;

//*****************************************************************************
//
//! Values that can be passed to CLB_configGPInputMux() as the \e gpMuxCfg
//! parameter.
//
//*****************************************************************************
typedef enum
{
    CLB_GP_IN_MUX_EXTERNAL  = 0,    //!< Use external input path
    CLB_GP_IN_MUX_GP_REG    = 1     //!< Use CLB_GP_REG bit value as input
} CLB_GPInputMux;

//*****************************************************************************
//
//! Values that can be passed to CLB_configLocalInputMux() as the
//! \e localMuxCfg parameter.
//
//*****************************************************************************
typedef enum
{
    CLB_LOCAL_IN_MUX_GLOBAL_IN          = 0,   //!< Global input mux selection
    CLB_LOCAL_IN_MUX_EPWM_DCAEVT1       = 1,   //!< EPWMx DCAEVT1
    CLB_LOCAL_IN_MUX_EPWM_DCAEVT2       = 2,   //!< EPWMx DCAEVT2
    CLB_LOCAL_IN_MUX_EPWM_DCBEVT1       = 3,   //!< EPWMx DCBEVT1
    CLB_LOCAL_IN_MUX_EPWM_DCBEVT2       = 4,   //!< EPWMx DCBEVT2
    CLB_LOCAL_IN_MUX_EPWM_DCAH          = 5,   //!< EPWMx DCAH
    CLB_LOCAL_IN_MUX_EPWM_DCAL          = 6,   //!< EPWMx DCAL
    CLB_LOCAL_IN_MUX_EPWM_DCBH          = 7,   //!< EPWMx DCBH
    CLB_LOCAL_IN_MUX_EPWM_DCBL          = 8,   //!< EPWMx DCBL
    CLB_LOCAL_IN_MUX_EPWM_OST           = 9,   //!< EPWMx OST
    CLB_LOCAL_IN_MUX_EPWM_CBC           = 10,  //!< EPWMx CBC
    CLB_LOCAL_IN_MUX_ECAP_ECAPIN        = 11,  //!< ECAPx ECAPIN
    CLB_LOCAL_IN_MUX_ECAP_ECAP_OUT      = 12,  //!< ECAPx ECAP_OUT
    CLB_LOCAL_IN_MUX_ECAP_ECAP_OUT_EN   = 13,  //!< ECAPx ECAP_OUT_EN
    CLB_LOCAL_IN_MUX_ECAP_CEVT1         = 14,  //!< ECAPx CEVT1
    CLB_LOCAL_IN_MUX_ECAP_CEVT2         = 15,  //!< ECAPx CEVT2
    CLB_LOCAL_IN_MUX_ECAP_CEVT3         = 16,  //!< ECAPx CEVT3
    CLB_LOCAL_IN_MUX_ECAP_CEVT4         = 17,  //!< ECAPx CEVT4
    CLB_LOCAL_IN_MUX_EQEP_EQEPA         = 18,  //!< EQEPx EQEPA
    CLB_LOCAL_IN_MUX_EQEP_EQEPB         = 19,  //!< EQEPx EQEPB
    CLB_LOCAL_IN_MUX_EQEP_EQEPI         = 20,  //!< EQEPx EQEPI
    CLB_LOCAL_IN_MUX_EQEP_EQEPS         = 21,  //!< EQEPx EQEPS
    CLB_LOCAL_IN_MUX_CPU1_TBCLKSYNC     = 22,  //!< CPU1.TBCLKSYNC
    CLB_LOCAL_IN_MUX_CPU1_HALT          = 24,  //!< CPU1.HALT
    CLB_LOCAL_IN_MUX_SPICLK             = 26,  //!< SPI Clock
    CLB_LOCAL_IN_MUX_SPIPICO_PERIPHERAL = 27,  //!< SPIPICO Peripheral Input
    CLB_LOCAL_IN_MUX_SPIPTE             = 28,  //!< SPI PTE
    CLB_LOCAL_IN_MUX_SCI_TX             = 29,  //!< SCI TX
    CLB_LOCAL_IN_MUX_CLB_PSCLK          = 31,  //!< CLB prescaled clock
} CLB_LocalInputMux;

//*****************************************************************************
//
//! Values that can be passed to CLB_configGlobalInputMux() as the
//! \e globalMuxCfg parameter.
//
//*****************************************************************************
typedef enum
{
    CLB_GLOBAL_IN_MUX_EPWM1A            = 0,   //!< EPWM1A
    CLB_GLOBAL_IN_MUX_EPWM1A_OE         = 1,   //!< EPWM1A trip output
    CLB_GLOBAL_IN_MUX_EPWM1B            = 2,   //!< EPWM1B
    CLB_GLOBAL_IN_MUX_EPWM1B_OE         = 3,   //!< EPWM1B trip output
    CLB_GLOBAL_IN_MUX_EPWM1_CTR_ZERO    = 4,   //!< EPWM1 TBCTR = Zero
    CLB_GLOBAL_IN_MUX_EPWM1_CTR_PRD     = 5,   //!< EPWM1 TBCTR = TBPRD
    CLB_GLOBAL_IN_MUX_EPWM1_CTRDIR      = 6,   //!< EPWM1 CTRDIR
    CLB_GLOBAL_IN_MUX_EPWM1_TBCLK       = 7,   //!< EPWM1 TBCLK
    CLB_GLOBAL_IN_MUX_EPWM1_CTR_CMPA    = 8,   //!< EPWM1 TBCTR = CMPA
    CLB_GLOBAL_IN_MUX_EPWM1_CTR_CMPB    = 9,   //!< EPWM1 TBCTR = CMPB
    CLB_GLOBAL_IN_MUX_EPWM1_CTR_CMPC    = 10,  //!< EPWM1 TBCTR = CMPC
    CLB_GLOBAL_IN_MUX_EPWM1_CTR_CMPD    = 11,  //!< EPWM1 TBCTR = CMPD
    CLB_GLOBAL_IN_MUX_EPWM1A_AQ         = 12,  //!< EPWM1A AQ submodule output
    CLB_GLOBAL_IN_MUX_EPWM1B_AQ         = 13,  //!< EPWM1B AQ submodule output
    CLB_GLOBAL_IN_MUX_EPWM1A_DB         = 14,  //!< EPWM1A DB submodule output
    CLB_GLOBAL_IN_MUX_EPWM1B_DB         = 15,  //!< EPWM1B DB submodule output

    CLB_GLOBAL_IN_MUX_EPWM2A            = 16,  //!< EPWM2A
    CLB_GLOBAL_IN_MUX_EPWM2A_OE         = 17,  //!< EPWM2A trip output
    CLB_GLOBAL_IN_MUX_EPWM2B            = 18,  //!< EPWM2B
    CLB_GLOBAL_IN_MUX_EPWM2B_OE         = 19,  //!< EPWM2B trip output
    CLB_GLOBAL_IN_MUX_EPWM2_CTR_ZERO    = 20,  //!< EPWM2 TBCTR = Zero
    CLB_GLOBAL_IN_MUX_EPWM2_CTR_PRD     = 21,  //!< EPWM2 TBCTR = TBPRD
    CLB_GLOBAL_IN_MUX_EPWM2_CTRDIR      = 22,  //!< EPWM2 CTRDIR
    CLB_GLOBAL_IN_MUX_EPWM2_TBCLK       = 23,  //!< EPWM2 TBCLK
    CLB_GLOBAL_IN_MUX_EPWM2_CTR_CMPA    = 24,  //!< EPWM2 TBCTR = CMPA
    CLB_GLOBAL_IN_MUX_EPWM2_CTR_CMPB    = 25,  //!< EPWM2 TBCTR = CMPB
    CLB_GLOBAL_IN_MUX_EPWM2_CTR_CMPC    = 26,  //!< EPWM2 TBCTR = CMPC
    CLB_GLOBAL_IN_MUX_EPWM2_CTR_CMPD    = 27,  //!< EPWM2 TBCTR = CMPD
    CLB_GLOBAL_IN_MUX_EPWM2A_AQ         = 28,  //!< EPWM2A AQ submodule output
    CLB_GLOBAL_IN_MUX_EPWM2B_AQ         = 29,  //!< EPWM2B AQ submodule output
    CLB_GLOBAL_IN_MUX_EPWM2A_DB         = 30,  //!< EPWM2A DB submodule output
    CLB_GLOBAL_IN_MUX_EPWM2B_DB         = 31,  //!< EPWM2B DB submodule output

    CLB_GLOBAL_IN_MUX_EPWM3A            = 32,  //!< EPWM3A
    CLB_GLOBAL_IN_MUX_EPWM3A_OE         = 33,  //!< EPWM3A trip output
    CLB_GLOBAL_IN_MUX_EPWM3B            = 34,  //!< EPWM3B
    CLB_GLOBAL_IN_MUX_EPWM3B_OE         = 35,  //!< EPWM3B trip output
    CLB_GLOBAL_IN_MUX_EPWM3_CTR_ZERO    = 36,  //!< EPWM3 TBCTR = Zero
    CLB_GLOBAL_IN_MUX_EPWM3_CTR_PRD     = 37,  //!< EPWM3 TBCTR = TBPRD
    CLB_GLOBAL_IN_MUX_EPWM3_CTRDIR      = 38,  //!< EPWM3 CTRDIR
    CLB_GLOBAL_IN_MUX_EPWM3_TBCLK       = 39,  //!< EPWM3 TBCLK
    CLB_GLOBAL_IN_MUX_EPWM3_CTR_CMPA    = 40,  //!< EPWM3 TBCTR = CMPA
    CLB_GLOBAL_IN_MUX_EPWM3_CTR_CMPB    = 41,  //!< EPWM3 TBCTR = CMPB
    CLB_GLOBAL_IN_MUX_EPWM3_CTR_CMPC    = 42,  //!< EPWM3 TBCTR = CMPC
    CLB_GLOBAL_IN_MUX_EPWM3_CTR_CMPD    = 43,  //!< EPWM3 TBCTR = CMPD
    CLB_GLOBAL_IN_MUX_EPWM3A_AQ         = 44,  //!< EPWM3A AQ submodule output
    CLB_GLOBAL_IN_MUX_EPWM3B_AQ         = 45,  //!< EPWM3B AQ submodule output
    CLB_GLOBAL_IN_MUX_EPWM3A_DB         = 46,  //!< EPWM3A DB submodule output
    CLB_GLOBAL_IN_MUX_EPWM3B_DB         = 47,  //!< EPWM3B DB submodule output

    CLB_GLOBAL_IN_MUX_EPWM4A            = 48,  //!< EPWM4A
    CLB_GLOBAL_IN_MUX_EPWM4A_OE         = 49,  //!< EPWM4A trip output
    CLB_GLOBAL_IN_MUX_EPWM4B            = 50,  //!< EPWM4B
    CLB_GLOBAL_IN_MUX_EPWM4B_OE         = 51,  //!< EPWM4B trip output
    CLB_GLOBAL_IN_MUX_EPWM4_CTR_ZERO    = 52,  //!< EPWM4 TBCTR = Zero
    CLB_GLOBAL_IN_MUX_EPWM4_CTR_PRD     = 53,  //!< EPWM4 TBCTR = TBPRD
    CLB_GLOBAL_IN_MUX_EPWM4_CTRDIR      = 54,  //!< EPWM4 CTRDIR
    CLB_GLOBAL_IN_MUX_EPWM4_TBCLK       = 55,  //!< EPWM4 TBCLK
    CLB_GLOBAL_IN_MUX_EPWM4_CTR_CMPA    = 56,  //!< EPWM4 TBCTR = CMPA
    CLB_GLOBAL_IN_MUX_EPWM4_CTR_CMPB    = 57,  //!< EPWM4 TBCTR = CMPB
    CLB_GLOBAL_IN_MUX_EPWM4_CTR_CMPC    = 58,  //!< EPWM4 TBCTR = CMPC
    CLB_GLOBAL_IN_MUX_EPWM4_CTR_CMPD    = 59,  //!< EPWM4 TBCTR = CMPD
    CLB_GLOBAL_IN_MUX_EPWM4A_AQ         = 60,  //!< EPWM4A AQ submodule output
    CLB_GLOBAL_IN_MUX_EPWM4B_AQ         = 61,  //!< EPWM4B AQ submodule output
    CLB_GLOBAL_IN_MUX_EPWM4A_DB         = 62,  //!< EPWM4A DB submodule output
    CLB_GLOBAL_IN_MUX_EPWM4B_DB         = 63,  //!< EPWM4B DB submodule output

    CLB_GLOBAL_IN_MUX_EPWM5A            = 64,  //!< EPWM5A
    CLB_GLOBAL_IN_MUX_EPWM5A_OE         = 65,  //!< EPWM5A trip output
    CLB_GLOBAL_IN_MUX_EPWM5B            = 66,  //!< EPWM5B
    CLB_GLOBAL_IN_MUX_EPWM5B_OE         = 67,  //!< EPWM5B trip output
    CLB_GLOBAL_IN_MUX_EPWM6A            = 68,  //!< EPWM5A
    CLB_GLOBAL_IN_MUX_EPWM6A_OE         = 69,  //!< EPWM5A trip output
    CLB_GLOBAL_IN_MUX_EPWM6B            = 70,  //!< EPWM6B
    CLB_GLOBAL_IN_MUX_EPWM6B_OE         = 71,  //!< EPWM6B trip output
    CLB_GLOBAL_IN_MUX_EPWM7A            = 72,  //!< EPWM7A
    CLB_GLOBAL_IN_MUX_EPWM7A_OE         = 73,  //!< EPWM7A trip output
    CLB_GLOBAL_IN_MUX_EPWM7B            = 74,  //!< EPWM7B
    CLB_GLOBAL_IN_MUX_EPWM7B_OE         = 75,  //!< EPWM7B trip output
    CLB_GLOBAL_IN_MUX_EPWM8A            = 76,  //!< EPWM8A
    CLB_GLOBAL_IN_MUX_EPWM8A_OE         = 77,  //!< EPWM8A trip output
    CLB_GLOBAL_IN_MUX_EPWM8B            = 78,  //!< EPWM8B
    CLB_GLOBAL_IN_MUX_EPWM8B_OE         = 79,  //!< EPWM8B trip output
    CLB_GLOBAL_IN_MUX_EPWM9A            = 80,  //!< EPWM9A
    CLB_GLOBAL_IN_MUX_EPWM9A_OE         = 81,  //!< EPWM9A trip output
    CLB_GLOBAL_IN_MUX_EPWM9B            = 82,  //!< EPWM9B
    CLB_GLOBAL_IN_MUX_EPWM9B_OE         = 83,  //!< EPWM9B trip output
    CLB_GLOBAL_IN_MUX_EPWM10A            = 84,  //!< EPWM10A
    CLB_GLOBAL_IN_MUX_EPWM10A_OE         = 85,  //!< EPWM10A trip output
    CLB_GLOBAL_IN_MUX_EPWM10B            = 86,  //!< EPWM10B
    CLB_GLOBAL_IN_MUX_EPWM10B_OE         = 87,  //!< EPWM10B trip output
    CLB_GLOBAL_IN_MUX_EPWM11A            = 88,  //!< EPWM11A
    CLB_GLOBAL_IN_MUX_EPWM11A_OE         = 89,  //!< EPWM11A trip output
    CLB_GLOBAL_IN_MUX_EPWM11B            = 90,  //!< EPWM11B
    CLB_GLOBAL_IN_MUX_EPWM11B_OE         = 91,  //!< EPWM11B trip output
    CLB_GLOBAL_IN_MUX_EPWM12A            = 92,  //!< EPWM12A
    CLB_GLOBAL_IN_MUX_EPWM12A_OE         = 93,  //!< EPWM12A trip output
    CLB_GLOBAL_IN_MUX_EPWM12B            = 94,  //!< EPWM12B
    CLB_GLOBAL_IN_MUX_EPWM12B_OE         = 95,  //!< EPWM12B trip output

    CLB_GLOBAL_IN_MUX_EPWM1_DACA1EVT    = 96,  //!< EPWM1
	CLB_GLOBAL_IN_MUX_EPWM1_DACA2EVT    = 97,  //!< EPWM1
	CLB_GLOBAL_IN_MUX_EPWM1_DACB1EVT    = 98,  //!< EPWM1
	CLB_GLOBAL_IN_MUX_EPWM1_DACB2EVT    = 99,  //!< EPWM1
	CLB_GLOBAL_IN_MUX_EPWM1_DCAH        = 100,  //!< EPWM1
	CLB_GLOBAL_IN_MUX_EPWM1_DCAI        = 101,  //!< EPWM1
	CLB_GLOBAL_IN_MUX_EPWM1_DCBH        = 102,  //!< EPWM1
	CLB_GLOBAL_IN_MUX_EPWM1_DCBI        = 103,  //!< EPWM1
	CLB_GLOBAL_IN_MUX_EPWM1_OST         = 104,  //!< EPWM1
	CLB_GLOBAL_IN_MUX_EPWM1_CBC         = 105,  //!< EPWM1
	CLB_GLOBAL_IN_MUX_EPWM1_OST_CBC     = 106,  //!< EPWM1
	CLB_GLOBAL_IN_MUX_EPWM1_CTR_CMP1A   = 107,  //!< EPWM1
	CLB_GLOBAL_IN_MUX_EPWM1_CTR_CMP1B   = 108,  //!< EPWM1
	CLB_GLOBAL_IN_MUX_EPWM1_ZERO        = 109,  //!< EPWM1

    CLB_GLOBAL_IN_MUX_EPWM2_DACA1EVT    = 110,  //!< EPWM2
	CLB_GLOBAL_IN_MUX_EPWM2_DACA2EVT    = 111,  //!< EPWM2
	CLB_GLOBAL_IN_MUX_EPWM2_DACB1EVT    = 112,  //!< EPWM2
	CLB_GLOBAL_IN_MUX_EPWM2_DACB2EVT    = 113,  //!< EPWM2
	CLB_GLOBAL_IN_MUX_EPWM2_DCAH        = 114,  //!< EPWM2
	CLB_GLOBAL_IN_MUX_EPWM2_DCAI        = 115,  //!< EPWM2
	CLB_GLOBAL_IN_MUX_EPWM2_DCBH        = 116,  //!< EPWM2
	CLB_GLOBAL_IN_MUX_EPWM2_DCBI        = 117,  //!< EPWM2
	CLB_GLOBAL_IN_MUX_EPWM2_OST         = 118,  //!< EPWM2
	CLB_GLOBAL_IN_MUX_EPWM2_CBC         = 119,  //!< EPWM2
	CLB_GLOBAL_IN_MUX_EPWM2_OST_CBC     = 120,  //!< EPWM2
	CLB_GLOBAL_IN_MUX_EPWM2_CTR_CMP1A   = 121,  //!< EPWM2
	CLB_GLOBAL_IN_MUX_EPWM2_CTR_CMP1B   = 122,  //!< EPWM2
	CLB_GLOBAL_IN_MUX_EPWM2_ZERO        = 123,  //!< EPWM2

    CLB_GLOBAL_IN_MUX_EPWM3_DACA1EVT    = 124,  //!< EPWM3
	CLB_GLOBAL_IN_MUX_EPWM3_DACA2EVT    = 125,  //!< EPWM3
	CLB_GLOBAL_IN_MUX_EPWM3_DACB1EVT    = 126,  //!< EPWM3
	CLB_GLOBAL_IN_MUX_EPWM3_DACB2EVT    = 127,  //!< EPWM3
	CLB_GLOBAL_IN_MUX_EPWM3_DCAH        = 128,  //!< EPWM3
	CLB_GLOBAL_IN_MUX_EPWM3_DCAI        = 129,  //!< EPWM3
	CLB_GLOBAL_IN_MUX_EPWM3_DCBH        = 130,  //!< EPWM3
	CLB_GLOBAL_IN_MUX_EPWM3_DCBI        = 131,  //!< EPWM3
	CLB_GLOBAL_IN_MUX_EPWM3_OST         = 132,  //!< EPWM3
	CLB_GLOBAL_IN_MUX_EPWM3_CBC         = 133,  //!< EPWM3
	CLB_GLOBAL_IN_MUX_EPWM3_OST_CBC     = 134,  //!< EPWM3
	CLB_GLOBAL_IN_MUX_EPWM3_CTR_CMP1A   = 135,  //!< EPWM3
	CLB_GLOBAL_IN_MUX_EPWM3_CTR_CMP1B   = 136,  //!< EPWM3
	CLB_GLOBAL_IN_MUX_EPWM3_ZERO        = 137,  //!< EPWM3

    CLB_GLOBAL_IN_MUX_EPWM4_DACA1EVT    = 138,  //!< EPWM4
	CLB_GLOBAL_IN_MUX_EPWM4_DACA2EVT    = 139,  //!< EPWM4
	CLB_GLOBAL_IN_MUX_EPWM4_DACB1EVT    = 140,  //!< EPWM4
	CLB_GLOBAL_IN_MUX_EPWM4_DACB2EVT    = 141,  //!< EPWM4
	CLB_GLOBAL_IN_MUX_EPWM4_DCAH        = 142,  //!< EPWM4
	CLB_GLOBAL_IN_MUX_EPWM4_DCAI        = 143,  //!< EPWM4
	CLB_GLOBAL_IN_MUX_EPWM4_DCBH        = 144,  //!< EPWM4
	CLB_GLOBAL_IN_MUX_EPWM4_DCBI        = 145,  //!< EPWM4
	CLB_GLOBAL_IN_MUX_EPWM4_OST         = 146,  //!< EPWM4
	CLB_GLOBAL_IN_MUX_EPWM4_CBC         = 147,  //!< EPWM4
	CLB_GLOBAL_IN_MUX_EPWM4_OST_CBC     = 148,  //!< EPWM4
	CLB_GLOBAL_IN_MUX_EPWM4_CTR_CMP1A   = 149,  //!< EPWM4
	CLB_GLOBAL_IN_MUX_EPWM4_CTR_CMP1B   = 150,  //!< EPWM4
	CLB_GLOBAL_IN_MUX_EPWM4_ZERO        = 151,  //!< EPWM4

//    CLB_GLOBAL_IN_MUX_FSIRXA_DATA_PACKET_RCVD  = 112,  //!< FSIRXA Data Packet Received
//    CLB_GLOBAL_IN_MUX_FSIRXA_ERROR_PACKET_RCVD = 113,  //!< FSIRXA Error Packet Received
//    CLB_GLOBAL_IN_MUX_FSIRXA_PING_PACKET_RCVD  = 114,  //!< FSIRXA PING Packet Received
//    CLB_GLOBAL_IN_MUX_FSIRXA_OUTPUT_FRAME_DONE = 115,  //!< FSIRXA Output Frame Done
//    CLB_GLOBAL_IN_MUX_FSIRXA_PACKET_TAG0       = 116,  //!< FSIRXA Packet TAG0
//    CLB_GLOBAL_IN_MUX_FSIRXA_PACKET_TAG1       = 117,  //!< FSIRXA Packet TAG1
//    CLB_GLOBAL_IN_MUX_FSIRXA_PACKET_TAG2       = 118,  //!< FSIRXA Packet TAG2
//    CLB_GLOBAL_IN_MUX_FSIRXA_PACKET_TAG3       = 119,  //!< FSIRXA Packet TAG3
//
//    CLB_GLOBAL_IN_MUX_SPI1_SPICLK               = 120,  //!< SPI1 SPICLK OUT
//    CLB_GLOBAL_IN_MUX_SPI1_SPIPOCI_CONTROLLER   = 121,  //!< SPI1 SPIPOCI Controller IN
//    CLB_GLOBAL_IN_MUX_SPI1_SPIPTE               = 122,  //!< SPI1 SPIPTE OUT
//    CLB_GLOBAL_IN_MUX_SPI2_SPICLK               = 123,  //!< SPI2 SPICLK OUT
//    CLB_GLOBAL_IN_MUX_SPI2_SPIPOCI_CONTROLLER   = 124,  //!< SPI2 SPIPOCI Controller IN
//    CLB_GLOBAL_IN_MUX_SPI2_SPIPTE               = 125,  //!< SPI2 SPIPTE OUT


    CLB_GLOBAL_IN_MUX_XBAE_CLB0_OUT     =152,  //!< plu_xbar
    CLB_GLOBAL_IN_MUX_XBAE_CLB1_OUT     =153,  //!< plu_xbar
    CLB_GLOBAL_IN_MUX_XBAE_CLB2_OUT     =154,  //!< plu_xbar
    CLB_GLOBAL_IN_MUX_XBAE_CLB3_OUT     =155,  //!< plu_xbar
    CLB_GLOBAL_IN_MUX_XBAE_CLB4_OUT     =156,  //!< plu_xbar
    CLB_GLOBAL_IN_MUX_XBAE_CLB5_OUT     =157,  //!< plu_xbar
    CLB_GLOBAL_IN_MUX_XBAE_CLB6_OUT     =158,  //!< plu_xbar
    CLB_GLOBAL_IN_MUX_XBAE_CLB7_OUT     =159,  //!< plu_xbar

	CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL1_OUT   = 160,
	CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL2_OUT   = 161,
	CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL3_OUT   = 162,
	CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL4_OUT   = 163,
	CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL5_OUT   = 164,
	CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL6_OUT   = 165,
	CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL7_OUT   = 166,
	CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL8_OUT   = 167,
	CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL9_OUT   = 168,
	CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL10_OUT  = 169,
	CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL11_OUT  = 170,
	CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL12_OUT  = 171,
	CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL13_OUT  = 172,
	CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL14_OUT  = 173,
	CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL15_OUT  = 174,
	CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL16_OUT  = 175,

	CLB_GLOBAL_IN_MUX_CLB1_OUT0                 = 176,
	CLB_GLOBAL_IN_MUX_CLB1_OUT1                 = 177,
	CLB_GLOBAL_IN_MUX_CLB1_OUT2                 = 178,
	CLB_GLOBAL_IN_MUX_CLB1_OUT3                 = 179,
	CLB_GLOBAL_IN_MUX_CLB1_OUT4                 = 180,
	CLB_GLOBAL_IN_MUX_CLB1_OUT5                 = 181,
	CLB_GLOBAL_IN_MUX_CLB1_OUT6                 = 182,
	CLB_GLOBAL_IN_MUX_CLB1_OUT7                 = 183,

	CLB_GLOBAL_IN_MUX_CLB2_OUT0                 = 188,
	CLB_GLOBAL_IN_MUX_CLB2_OUT1                 = 189,
	CLB_GLOBAL_IN_MUX_CLB2_OUT2                 = 190,
	CLB_GLOBAL_IN_MUX_CLB2_OUT3                 = 191,
	CLB_GLOBAL_IN_MUX_CLB2_OUT4                 = 192,
	CLB_GLOBAL_IN_MUX_CLB2_OUT5                 = 193,
	CLB_GLOBAL_IN_MUX_CLB2_OUT6                 = 194,
	CLB_GLOBAL_IN_MUX_CLB2_OUT7                 = 195,

	CLB_GLOBAL_IN_MUX_CLB3_OUT0                 = 200,
	CLB_GLOBAL_IN_MUX_CLB3_OUT1                 = 201,
	CLB_GLOBAL_IN_MUX_CLB3_OUT2                 = 202,
	CLB_GLOBAL_IN_MUX_CLB3_OUT3                 = 203,
	CLB_GLOBAL_IN_MUX_CLB3_OUT4                 = 204,
	CLB_GLOBAL_IN_MUX_CLB3_OUT5                 = 205,
	CLB_GLOBAL_IN_MUX_CLB3_OUT6                 = 206,
	CLB_GLOBAL_IN_MUX_CLB3_OUT7                 = 207,

	CLB_GLOBAL_IN_MUX_CLB4_OUT0                 = 212,
	CLB_GLOBAL_IN_MUX_CLB4_OUT1                 = 213,
	CLB_GLOBAL_IN_MUX_CLB4_OUT2                 = 214,
	CLB_GLOBAL_IN_MUX_CLB4_OUT3                 = 215,
	CLB_GLOBAL_IN_MUX_CLB4_OUT4                 = 216,
	CLB_GLOBAL_IN_MUX_CLB4_OUT5                 = 217,
	CLB_GLOBAL_IN_MUX_CLB4_OUT6                 = 218,
	CLB_GLOBAL_IN_MUX_CLB4_OUT7                 = 219,

	CLB_GLOBAL_IN_MUX_ECAP1_IN                    = 224,
	CLB_GLOBAL_IN_MUX_ECAP1_OUT                   = 225,
	CLB_GLOBAL_IN_MUX_ECAP1_OEN                   = 226,
	CLB_GLOBAL_IN_MUX_ECAP1_CEVT1                 = 227,
	CLB_GLOBAL_IN_MUX_ECAP1_CEVT2                 = 228,
	CLB_GLOBAL_IN_MUX_ECAP1_CEVT3                 = 229,
	CLB_GLOBAL_IN_MUX_ECAP1_CEVT4                 = 230,
	CLB_GLOBAL_IN_MUX_ECAP2_IN                    = 231,
	CLB_GLOBAL_IN_MUX_ECAP2_OUT                   = 232,
	CLB_GLOBAL_IN_MUX_ECAP2_OEN                   = 233,
	CLB_GLOBAL_IN_MUX_ECAP2_CEVT1                 = 234,
	CLB_GLOBAL_IN_MUX_ECAP2_CEVT2                 = 235,
	CLB_GLOBAL_IN_MUX_ECAP2_CEVT3                 = 236,
	CLB_GLOBAL_IN_MUX_ECAP2_CEVT4                 = 237,
	CLB_GLOBAL_IN_MUX_ECAP3_IN                    = 238,
	CLB_GLOBAL_IN_MUX_ECAP3_OUT                   = 239,
	CLB_GLOBAL_IN_MUX_ECAP3_OEN                   = 240,
	CLB_GLOBAL_IN_MUX_ECAP3_CEVT1                 = 241,
	CLB_GLOBAL_IN_MUX_ECAP3_CEVT2                 = 242,
	CLB_GLOBAL_IN_MUX_ECAP3_CEVT3                 = 243,
	CLB_GLOBAL_IN_MUX_ECAP3_CEVT4                 = 244,
	CLB_GLOBAL_IN_MUX_ECAP4_IN                    = 245,
	CLB_GLOBAL_IN_MUX_ECAP4_OUT                   = 246,
	CLB_GLOBAL_IN_MUX_ECAP4_OEN                   = 247,
	CLB_GLOBAL_IN_MUX_ECAP4_CEVT1                 = 248,
	CLB_GLOBAL_IN_MUX_ECAP4_CEVT2                 = 249,
	CLB_GLOBAL_IN_MUX_ECAP4_CEVT3                 = 250,
	CLB_GLOBAL_IN_MUX_ECAP4_CEVT4                 = 251,
	CLB_GLOBAL_IN_MUX_TBCLKSYN_TO_CLB             = 252,
	CLB_GLOBAL_IN_MUX_CPU_HATL                    = 253,
	CLB_GLOBAL_IN_MUX_DSA0_DBG_STOPTIME           = 254,
	CLB_GLOBAL_IN_MUX_DSA1_DBG_STOPTIME           = 255,

	CLB_GLOBAL_IN_MUX_ERAD_EBC_EVENT0             = 256,
	CLB_GLOBAL_IN_MUX_ERAD_EBC_EVENT1             = 257,
	CLB_GLOBAL_IN_MUX_ERAD_EBC_EVENT2             = 258,
	CLB_GLOBAL_IN_MUX_ERAD_EBC_EVENT3             = 259,
	CLB_GLOBAL_IN_MUX_ERAD_EBC_EVENT4             = 260,
	CLB_GLOBAL_IN_MUX_ERAD_EBC_EVENT5             = 261,
	CLB_GLOBAL_IN_MUX_ERAD_SEC_EVENT0             = 262,
	CLB_GLOBAL_IN_MUX_ERAD_SEC_EVENT1             = 263,
	CLB_GLOBAL_IN_MUX_ERAD_SEC_EVENT2             = 264,
	CLB_GLOBAL_IN_MUX_ERAD_SEC_EVENT3             = 265,

	CLB_GLOBAL_IN_MUX_SPI1_SIMO_IN                = 266,
	CLB_GLOBAL_IN_MUX_SPI1_SIMO_OUT               = 267,
	CLB_GLOBAL_IN_MUX_SPI1_SIMO_OEN               = 268,
	CLB_GLOBAL_IN_MUX_SPI1_SOMI_IN                = 269,
	CLB_GLOBAL_IN_MUX_SPI1_SOMI_OUT               = 270,
	CLB_GLOBAL_IN_MUX_SPI1_SOMI_OEN               = 271,
	CLB_GLOBAL_IN_MUX_SPI1_SLK_IN                 = 272,
	CLB_GLOBAL_IN_MUX_SPI1_SLK_OUT                = 273,
	CLB_GLOBAL_IN_MUX_SPI2_SIMO_IN                = 274,
	CLB_GLOBAL_IN_MUX_SPI2_SIMO_OUT               = 275,
	CLB_GLOBAL_IN_MUX_SPI2_SIMO_OEN               = 276,
	CLB_GLOBAL_IN_MUX_SPI2_SOMI_IN                = 277,
	CLB_GLOBAL_IN_MUX_SPI2_SOMI_OUT               = 278,
	CLB_GLOBAL_IN_MUX_SPI2_SOMI_OEN               = 279,
	CLB_GLOBAL_IN_MUX_SPI2_SLK_IN                 = 280,
	CLB_GLOBAL_IN_MUX_SPI2_SLK_OUT                = 281,

#if IS_GS32F3xx(0x22)
	CLB_GLOBAL_IN_MUX_SPI3_SIMO_IN                = 282,
	CLB_GLOBAL_IN_MUX_SPI3_SIMO_OUT               = 283,
	CLB_GLOBAL_IN_MUX_SPI3_SIMO_OEN               = 284,
	CLB_GLOBAL_IN_MUX_SPI3_SOMI_IN                = 285,
	CLB_GLOBAL_IN_MUX_SPI3_SOMI_OUT               = 286,
	CLB_GLOBAL_IN_MUX_SPI3_SOMI_OEN               = 287,
	CLB_GLOBAL_IN_MUX_SPI3_SLK_IN                 = 288,
	CLB_GLOBAL_IN_MUX_SPI3_SLK_OUT                = 289,
	CLB_GLOBAL_IN_MUX_SPI4_SIMO_IN                = 290,
	CLB_GLOBAL_IN_MUX_SPI4_SIMO_OUT               = 291,
	CLB_GLOBAL_IN_MUX_SPI4_SIMO_OEN               = 292,
	CLB_GLOBAL_IN_MUX_SPI4_SOMI_IN                = 293,
	CLB_GLOBAL_IN_MUX_SPI4_SOMI_OUT               = 294,
	CLB_GLOBAL_IN_MUX_SPI4_SOMI_OEN               = 295,
	CLB_GLOBAL_IN_MUX_SPI4_SLK_IN                 = 296,
	CLB_GLOBAL_IN_MUX_SPI4_SLK_OUT                = 297,

    CLB_GLOBAL_IN_MUX_EQEP3_A                     =306,
	CLB_GLOBAL_IN_MUX_EQEP3_B                     =307,
	CLB_GLOBAL_IN_MUX_EQEP3_I                     =308,
	CLB_GLOBAL_IN_MUX_EQEP3_S                     =309,

	CLB_GLOBAL_IN_MUX_EPWM13_XSYNCI               =326,
	CLB_GLOBAL_IN_MUX_EPWM14_XSYNCI               =327,
	CLB_GLOBAL_IN_MUX_EPWM15_XSYNCI               =328,
	CLB_GLOBAL_IN_MUX_EPWM16_XSYNCI               =329,
	CLB_GLOBAL_IN_MUX_EPWM17_XSYNCI               =330,
	CLB_GLOBAL_IN_MUX_EPWM18_XSYNCI               =331,
    CLB_GLOBAL_IN_MUX_EPWM13A                     =332,  //!< EPWM13A
    CLB_GLOBAL_IN_MUX_EPWM13A_OE                  =333,  //!< EPWM13A trip output
    CLB_GLOBAL_IN_MUX_EPWM13B                     =334,  //!< EPWM13B
    CLB_GLOBAL_IN_MUX_EPWM13B_OE                  =335,  //!< EPWM13B trip output
    CLB_GLOBAL_IN_MUX_EPWM14A                     =336,  //!< EPWM14A
    CLB_GLOBAL_IN_MUX_EPWM14A_OE                  =337,  //!< EPWM14A trip output
    CLB_GLOBAL_IN_MUX_EPWM14B                     =338,  //!< EPWM14B
    CLB_GLOBAL_IN_MUX_EPWM14B_OE                  =339,  //!< EPWM14B trip output
    CLB_GLOBAL_IN_MUX_EPWM15A                     =340,  //!< EPWM15A
    CLB_GLOBAL_IN_MUX_EPWM15A_OE                  =341,  //!< EPWM15A trip output
    CLB_GLOBAL_IN_MUX_EPWM15B                     =342,  //!< EPWM15B
    CLB_GLOBAL_IN_MUX_EPWM15B_OE                  =343,  //!< EPWM15B trip output
    CLB_GLOBAL_IN_MUX_EPWM16A                     =344,  //!< EPWM16A
    CLB_GLOBAL_IN_MUX_EPWM16A_OE                  =345,  //!< EPWM16A trip output
    CLB_GLOBAL_IN_MUX_EPWM16B                     =346,  //!< EPWM16B
    CLB_GLOBAL_IN_MUX_EPWM16B_OE                  =347,  //!< EPWM16B trip output
    CLB_GLOBAL_IN_MUX_EPWM17A                     =348,  //!< EPWM17A
    CLB_GLOBAL_IN_MUX_EPWM17A_OE                  =349,  //!< EPWM17A trip output
    CLB_GLOBAL_IN_MUX_EPWM17B                     =350,  //!< EPWM17B
    CLB_GLOBAL_IN_MUX_EPWM17B_OE                  =351,  //!< EPWM17B trip output
    CLB_GLOBAL_IN_MUX_EPWM18A                     =352,  //!< EPWM18A
    CLB_GLOBAL_IN_MUX_EPWM18A_OE                  =353,  //!< EPWM18A trip output
    CLB_GLOBAL_IN_MUX_EPWM18B                     =354,  //!< EPWM18B
    CLB_GLOBAL_IN_MUX_EPWM18B_OE                  =355,  //!< EPWM18B trip output
    CLB_GLOBAL_IN_MUX_EQEP4_A                     =356, //< eqep
	CLB_GLOBAL_IN_MUX_EQEP4_B                     =357,
	CLB_GLOBAL_IN_MUX_EQEP4_I                     =358,
	CLB_GLOBAL_IN_MUX_EQEP4_S                     =359,
    CLB_GLOBAL_IN_MUX_EQEP5_A                     =360,
	CLB_GLOBAL_IN_MUX_EQEP5_B                     =361,
	CLB_GLOBAL_IN_MUX_EQEP5_I                     =362,
	CLB_GLOBAL_IN_MUX_EQEP5_S                     =363,
    CLB_GLOBAL_IN_MUX_EQEP6_A                     =364,
	CLB_GLOBAL_IN_MUX_EQEP6_B                     =365,
	CLB_GLOBAL_IN_MUX_EQEP6_I                     =366,
	CLB_GLOBAL_IN_MUX_EQEP6_S                     =367,
    CLB_GLOBAL_IN_MUX_CMSPI_SIMO_IN              =368, //< SPI
	CLB_GLOBAL_IN_MUX_CMSPI_SIMO_OUT             =369,
	CLB_GLOBAL_IN_MUX_CMSPI_SIMO_OEN             =370,
	CLB_GLOBAL_IN_MUX_CMSPI_SOMI_IN              =371,
	CLB_GLOBAL_IN_MUX_CMSPI_SOMI_OUT             =372,
	CLB_GLOBAL_IN_MUX_CMSPI_SOMI_OEN             =373,
	CLB_GLOBAL_IN_MUX_CMSPI_SLK_IN               =374,
	CLB_GLOBAL_IN_MUX_CMSPI_SLK_OUT              =375,
	CLB_GLOBAL_IN_MUX_CMUART_SOUT                =376,
	CLB_GLOBAL_IN_MUX_CPU_HALT                   =377,
#endif

	CLB_GLOBAL_IN_MUX_EQEP1_A                     =298, //< eqep
	CLB_GLOBAL_IN_MUX_EQEP1_B                     =299,
	CLB_GLOBAL_IN_MUX_EQEP1_I                     =300,
	CLB_GLOBAL_IN_MUX_EQEP1_S                     =301,
	CLB_GLOBAL_IN_MUX_EQEP2_A                     =302,
	CLB_GLOBAL_IN_MUX_EQEP2_B                     =303,
	CLB_GLOBAL_IN_MUX_EQEP2_I                     =304,
	CLB_GLOBAL_IN_MUX_EQEP2_S                     =305,

	CLB_GLOBAL_IN_MUX_UART0_SOUT                  =310, //< UART
    CLB_GLOBAL_IN_MUX_UART1_SOUT                  =311, //< UART
    CLB_GLOBAL_IN_MUX_UART2_SOUT                  =312, //< UART
    CLB_GLOBAL_IN_MUX_UART3_SOUT                  =313, //< UART

    CLB_GLOBAL_IN_MUX_EPWM1_XSYNCI                =314, //< epwm
	CLB_GLOBAL_IN_MUX_EPWM2_XSYNCI                =315,
	CLB_GLOBAL_IN_MUX_EPWM3_XSYNCI                =316,
	CLB_GLOBAL_IN_MUX_EPWM4_XSYNCI                =317,
    CLB_GLOBAL_IN_MUX_EPWM5_XSYNCI                =318,
	CLB_GLOBAL_IN_MUX_EPWM6_XSYNCI                =319,
	CLB_GLOBAL_IN_MUX_EPWM7_XSYNCI                =320,
	CLB_GLOBAL_IN_MUX_EPWM8_XSYNCI                =321,
    CLB_GLOBAL_IN_MUX_EPWM9_XSYNCI                =322,
	CLB_GLOBAL_IN_MUX_EPWM10_XSYNCI               =323,
	CLB_GLOBAL_IN_MUX_EPWM11_XSYNCI               =324,
	CLB_GLOBAL_IN_MUX_EPWM12_XSYNCI               =325,





} CLB_GlobalInputMux;

//*****************************************************************************
//
// Prototypes for the APIs.
//
//*****************************************************************************
//
//!
//! Checks the CLB base address.
//!
//! \param base is the base address of a CLB tile's logic config register.
//!
//! This function determines if a CLB base address is valid.
//!
//! \return Returns \b true if the base address is valid and \b false
//! otherwise.
//
//*****************************************************************************
#ifdef DEBUG
static inline bool CLB_isBaseValid(uint32_t base)
{
    return(
           (base == CLB1_BASE) ||
           (base == CLB1_BASE)
//           (base == CLB2_BASE) ||
//           (base == CLB3_BASE)
          );
}

//*****************************************************************************
//
//!
//! Checks the CLB internal memory address.
//!
//! \param base is the base address of a CLB tile's logic config register.
//!
//! This function determines if a CLB base address is valid.
//!
//! \return Returns \b true if the address is valid and \b false otherwise.
//
//*****************************************************************************
static inline bool CLB_isAddressValid(uint32_t address)
{
    return(address <= (CLB_ADDR_HLC_BASE + CLB_NUM_HLC_INSTR));
}
#endif



//*****************************************************************************
//
//! Clear global enable.
//!
//! \param base is the base address of a CLB tile's logic config register.
//!
//! This function disables the CLB via global enable register.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_disableCLB(uint32_t base)
{
    ASSERT(CLB_isBaseValid(base));

    EALLOW;
    HWREG(base + CLB_LOGICCTL + CLB_O_LOAD_EN) &= ~0x2;
    EDIS;
}


//*****************************************************************************
//
//! Configures the general purpose SW release control value.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param inID is the specified CLB SW Release input signal.
//! \param val is the value of the SW RLS control.
//!
//! This function configures the general purpose SW release control value.
//! The \e inID parameter can have one enumeration value from CLB_SWReleaseCtrl.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_writeSWReleaseControl(uint32_t base,
                                            CLB_SWReleaseCtrl inID, Bool val)
{
    ASSERT(CLB_isBaseValid(base));


    HWREG(base + CLB_LOGICCTL + CLB_O_GP_REG) =
            (HWREG(base + CLB_LOGICCTL + CLB_O_GP_REG) &
                ~(0x1000000U << inID)) |
                    (((uint32_t)val) << (24U + inID));
}


//*****************************************************************************
//
//! Configures the general purpose SW gate control value.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param inID is the specified CLB SW Release input signal.
//! \param val is the value of the SW GATE control.
//!
//! This function configures the general purpose SW release control value.
//! The \e inID parameter can have one enumeration value from CLB_SWGateCtrl.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_writeSWGateControl(uint32_t base,
                                            CLB_SWGateCtrl inID, Bool val)
{
    ASSERT(CLB_isBaseValid(base));

    HWREG(base + CLB_LOGICCTL + CLB_O_GP_REG) =
            (HWREG(base + CLB_LOGICCTL + CLB_O_GP_REG) &
                ~(0x10000U << inID)) |
                    (((uint32_t)val) << (16U + inID));

}


//*****************************************************************************
//
//! Configures Counter TAP Selects.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param miscCtrl is the value to represent counter tap selects.
//!        Generated by tool as \b TILEx_CFG_TAP_SELL.
//!
//! This function configures the counter tap selects.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_configCounterTapSelects(uint32_t base, uint32_t tapSel)
{
    ASSERT(CLB_isBaseValid(base));

    EALLOW;
    HWREG(base + CLB_O_COUNT_MATCH_TAP_SEL) = tapSel;
    EDIS;
}

//*****************************************************************************
//
//! Configures AOC (Asynchronous Output Conditioning) functions.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param aocID is the specified CLB tile AOC signal.
//! \param aocCfg is the value for the AOC signal function and input
//!        signal selections. Generated by tool as \b TILEx_OUTPUT_COND_CTR_n
//!        where n is the output number.
//!
//! This function configures the input signals and equations of the aoc LUT
//! corresponding to the /e aocID parameter.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_configAOC(uint32_t base, CLB_AOCs aocID,
                                     uint32_t aocCfg)
{
    ASSERT(CLB_isBaseValid(base));

    EALLOW;
    HWREG(base + CLB_O_OUTPUT_COND_CTRL_0 + (0X4U * aocID)) = aocCfg;
    EDIS;
}



//*****************************************************************************
//
//! Write value to address.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param address is the address of CLB internal memory.
//! \param value is the value to write to specified address.
//!
//! This function writes the specified value to CLB internal memory.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_writeInterface(uint32_t base, uint32_t address,
                                      uint32_t value)
{
    ASSERT(CLB_isBaseValid(base));
    ASSERT(CLB_isAddressValid(address));

    EALLOW;
//    HWREG(base + CLB_LOGICCTL + CLB_O_LOAD_ADDR) = address;
    HWREG(base + CLB_LOGICCTL + address) =  value;

    EDIS;
}

//*****************************************************************************
//
//! Select input filter type.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param inID is the specified CLB tile input signal.
//! \param filterType is the selected type of filter applied to the input.
//!
//! This function configures the filter selection for the specified input.
//! The \e inID parameter can have one enumeration value from CLB_Inputs.
//! The \e filterType parameter can have one enumeration value from
//! CLB_FilterType.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_selectInputFilter(uint32_t base, CLB_Inputs inID,
                                         CLB_FilterType filterType)
{
    uint32_t shiftVal;

    ASSERT(CLB_isBaseValid(base));

    //
    // Each input has a 2-bit field in this register so need to calculate
    // shift amount accordingly.
    //
    shiftVal = (uint32_t)inID * 0X2U ;

    HWREG(base + CLB_LOGICCTL + CLB_O_INPUT_FILTER ) =
        (HWREG(base + CLB_LOGICCTL + CLB_O_INPUT_FILTER) |
        (uint32_t)filterType << shiftVal);
}

//*****************************************************************************
//
//! Select input Delayed sealing wave
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param inID is the specified CLB tile input signal.
//! \param delayType is Select the latency type.
//! \param delay is delay time .
//!
//! This function configures the filter selection for the specified input.
//! The \e inID parameter can have one enumeration value from CLB_Inputs.
//! The \e filterType parameter can have one enumeration value from
//! CLB_FilterType.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_selectInputDelay(uint32_t base, CLB_Inputs inID,uint32_t Faildelay,uint32_t risdelay)
{
    uint32_t shiftVal;

    ASSERT(CLB_isBaseValid(base));

    //
    // Each input has a 2-bit field in this register so need to calculate
    // shift amount accordingly.
    //


    HWREG(base + CLB_LOGICCTL + CLB_O_CLB_INPUT_FALLING_DELAY+inID*8) =
    		Faildelay;

	HWREG(base + CLB_LOGICCTL + CLB_O_CLB_INPUT_RISING_DELAY+inID*8) =
			risdelay;

}



//*****************************************************************************
//
//! Configures the general purpose input mux.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param inID is the specified CLB tile input signal.
//! \param gpMuxCfg is the mux selection for the general purpose input mux.
//!
//! This function configures the general purpose input mux. The \e gpMuxCfg
//! parameter can select either the use of an external input signal
//! (\b CLB_GP_IN_MUX_EXTERNAL) or the use of the corresponding CLB_GP_REG bit
//! as an input (\b CLB_GP_IN_MUX_GP_REG).
//! The \e inID parameter can have one enumeration value from CLB_Inputs.
//!
//! \sa CLB_setGPREG() to write to the CLB_GP_REG.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_configGPInputMux(uint32_t base, CLB_Inputs inID,
                                        CLB_GPInputMux gpMuxCfg)
{
    ASSERT(CLB_isBaseValid(base));

    HWREG(base + CLB_LOGICCTL + CLB_O_GPIN_MUX_SEL_0) =
        (HWREG(base + CLB_LOGICCTL + CLB_O_GPIN_MUX_SEL_0) &
         ~(CLB_IN_MUX_SEL_0_SEL_GP_IN_0 << inID)) | (gpMuxCfg << inID);
}

//*****************************************************************************
//
//! Sets the CLB_GP_REG register value.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param gpRegVal is the value to be written to CLB_GP_REG.
//!
//! This function writes to the CLB_GP_REG register. When the general purpose
//! input mux is configured to use CLB_GP_REG, each bit in \e gpRegVal
//! corresponds to an input signal (bit 0 to Input 0, bit 1 to Input 1, and
//! so on).
//!
//! \sa CLB_configGPInputMux() to select the CLB_GP_REG as the source for
//! an input signal.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_setGPREG(uint32_t base, uint32_t gpRegVal)
{
    ASSERT(CLB_isBaseValid(base));

   HWREG(base + CLB_LOGICCTL + CLB_O_GP_REG) =  (HWREG(base + CLB_LOGICCTL + CLB_O_GP_REG) & !0xFF )|gpRegVal;
}

//*****************************************************************************
//
//! Gets the CLB_GP_REG register value.
//!
//! \param base is the base address of a CLB tile's logic config register.
//!
//! This function writes to the CLB_GP_REG register. When the general purpose
//! input mux is configured to use CLB_GP_REG, each bit in \e gpRegVal
//! corresponds to an input signal (bit 0 to Input 0, bit 1 to Input 1, and
//! so on).
//!
//! \sa CLB_configGPInputMux() to select the CLB_GP_REG as the source for
//! an input signal.
//!
//! \return CLB_GP_REG value.
//
//*****************************************************************************
static inline uint32_t CLB_getGPREG(uint32_t base)
{
    ASSERT(CLB_isBaseValid(base));

    return(HWREG(base + CLB_LOGICCTL + CLB_O_GP_REG));
}


//*****************************************************************************
//
//! Configures the global input mux.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param inID is the specified CLB tile input signal.
//! \param globalMuxCfg is the mux selection for the global input mux.
//!
//! This function configures the global input mux for the specified input
//! signal.
//! The \e inID parameter can have one enumeration value from CLB_Inputs.
//! The \e globalMuxCfg parameter can have one enumeration value from
//! CLB_GlobalInputMux.
//!
//! \note The global input mux options' peripheral sources depend on which
//! instance of the CLB (\e base) you are using. For example, for CLB1 the
//! EPWM signal selections come from EPWM1 but for CLB2 they come from EPWM2.
//! See your technical reference manual for details.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_configGlobalInputMux(uint32_t base, CLB_Inputs inID,
                                            CLB_GlobalInputMux globalMuxCfg)
{
    uint32_t shiftVal;

    ASSERT(CLB_isBaseValid(base));

    //
    // Each local input has a 5-bit field in this register so need to calculate
    // shift amount accordingly.
    //


    if(inID < CLB_IN3)
    {
    	shiftVal = (uint32_t)inID * 9;
        HWREG(base + CLB_LOGICCTL + CLB_O_IN_MUX_SEL_0) =
            (HWREG(base + CLB_LOGICCTL + CLB_O_IN_MUX_SEL_0) &
             ~((uint32_t)CLB_LCL_MUX_SEL_1_LCL_MUX_SEL_IN_0_M << shiftVal)) |
            ((uint32_t)globalMuxCfg << shiftVal);
    }
    else if(inID < CLB_IN6)
    {
        //
        // Calculating shift amount for inputs > input3
        //
        shiftVal = (inID-3U) * 9;
        HWREG(base + CLB_LOGICCTL + 0X4U) =
            (HWREG(base + CLB_LOGICCTL + 0X4) &
             ~((uint32_t)CLB_LCL_MUX_SEL_1_LCL_MUX_SEL_IN_0_M << shiftVal)) |
            ((uint32_t)globalMuxCfg << shiftVal);
    }
    else
        {
            //
            // Calculating shift amount for inputs > input3
            //
            shiftVal =(inID-6U) * 9;
            HWREG(base + CLB_LOGICCTL + 0X8) =
                (HWREG(base + CLB_LOGICCTL + 0X8) &
                 ~((uint32_t)CLB_LCL_MUX_SEL_1_LCL_MUX_SEL_IN_0_M << shiftVal)) |
                ((uint32_t)globalMuxCfg << shiftVal);
        }
}

//*****************************************************************************
//
//! Controls the output enable.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param outputMask is a mask of the outputs to be enabled.
//! \param enable is a switch to decide if the CLB outputs need to be enabled
//!  or not.
//!
//! This function is used to enable and disable CLB outputs by writing a mask
//! to CLB_OUT_EN. Each bit corresponds to a CLB output. When a bit is 1, the
//! corresponding output is enabled; when a bit is 0, the output is disabled.
//!
//! The \e outputMask parameter takes a logical OR of any of the CLB_OUTPUT_0x
//! values that correspond to the CLB OUTPUT ENABLE for the respective outputs.
//! The \e enable parameter can have one of the values from:
//! false: Disable the respective CLB outputs
//! true: Enable the respective CLB outputs
//!
//! \note Note that the 8 CLB outputs are replicated to create more output
//!  paths. See your technical reference manual for more details.
//!  If no further modifications are expected, then it is advised to set the
//!  block writes bit of the MISC_ACCESS_CTRL Register. This will prevent
//!  accidental writes.
//!
//! \return None.
//
//*****************************************************************************
//*****************************************************************************
//
//! Set global enable.
//!
//! \param base is the base address of a CLB tile's logic config register.
//!
//! This function enables the CLB via global enable register.
//!
//! \return None.
static inline void CLB_enableCLB(uint32_t base)
{
    ASSERT(CLB_isBaseValid(base));


    HWREG(base + CLB_LOGICCTL + CLB_O_LOAD_EN) |= 0x2;


}

//*****************************************************************************
//
//! Reads the interrupt tag register.
//!
//! \param base is the base address of a CLB tile's logic config register.
//!
//! \return Returns the value in the interrupt tag register which is a 6-bit
//!         constant set by the HLC.
//
//*****************************************************************************
static inline uint32_t CLB_getInterruptTag(uint32_t base)
{
    ASSERT(CLB_isBaseValid(base));

    return(HWREG(base + CLB_LOGICCTL + CLB_O_HIL_INTR));
}

//*****************************************************************************
//
//!  interrupt enable.
static inline void CLB_InterruptEnable(uint32_t base,uint32_t event)
{

	HWREG(base  + CLB_LOGICCTL+CLB_O_HIL_INTR_MASK ) &= (0xff ^(0x1<< event));
}

//*****************************************************************************
//
//! Clears the interrupt tag register.
//!
//! \param base is the base address of a CLB tile's logic config register.
//!
//! This function clears the interrupt tag register, setting it to 0.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_clearInterruptTag(uint32_t base)
{
    ASSERT(CLB_isBaseValid(base));

    HWREG(base + CLB_LOGICCTL + CLB_O_HIL_INTR_RAW) = 0xFFU;
}

//*****************************************************************************
//
//! Selects LUT4 inputs.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param lut4In0 is the value for LUT4 input signal 0. Generated by tool as
//!        \b TILEx_CFG_LUT4_IN0.
//! \param lut4In1 is the value for LUT4 input signal 1. Generated by tool as
//!        \b TILEx_CFG_LUT4_IN1.
//! \param lut4In2 is the value for LUT4 input signal 2. Generated by tool as
//!        \b TILEx_CFG_LUT4_IN2.
//! \param lut4In3 is the value for LUT4 input signal 3. Generated by tool as
//!        \b TILEx_CFG_LUT4_IN3.
//!
//! This function configures the LUT4 block's input signals.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_selectLUT4Inputs(uint32_t base, uint32_t lut4In0,
                                        uint32_t lut4In1, uint32_t lut4In2,
                                        uint32_t lut4In3)
{
    ASSERT(CLB_isBaseValid(base));

    EALLOW;
    HWREG(base + CLB_O_LUT4_IN0) = lut4In0;
    HWREG(base + CLB_O_LUT4_IN1) = lut4In1;
    HWREG(base + CLB_O_LUT4_IN2) = lut4In2;
    HWREG(base + CLB_O_LUT4_IN3) = lut4In3;
    EDIS;
}

//*****************************************************************************
//
//! Configures LUT4 functions.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param lut4Fn10 is the equation value for LUT4 blocks 0 and 1. Generated by
//!        tool as \b TILEx_CFG_LUT4_FN10.
//! \param lut4Fn2 is the equation value for LUT4 block2. Generated by tool as
//!        \b TILEx_CFG_LUT4_FN2.
//!
//! This function configures the LUT4 block's equations.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_configLUT4Function(uint32_t base, uint32_t lut4Fn10,
                                          uint32_t lut4Fn2)
{
    ASSERT(CLB_isBaseValid(base));

    EALLOW;
    HWREG(base + CLB_O_LUT4_FN1_0) = lut4Fn10;
    HWREG(base + CLB_O_LUT4_FN2) = lut4Fn2;
    EDIS;
}

//*****************************************************************************
//
//! Selects FSM inputs.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param external0 is the value for FSM external 0 input. Generated by tool
//!        as \b TILEx_CFG_FSM_EXT_IN0.
//! \param external1 is the value for FSM external 1 input. Generated by tool
//!        as \b TILEx_CFG_FSM_EXT_IN1.
//! \param extra0 is the value for FSM extra 0 input. Generated by tool
//!        as \b TILEx_CFG_FSM_EXTRA_IN0.
//! \param extra1 is the value for FSM extra 1 input. Generated by tool
//!        as \b TILEx_CFG_FSM_EXTRA_IN1.
//!
//! This function configures the FSM block's external inputs and extra external
//! inputs.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_selectFSMInputs(uint32_t base, uint32_t external0,
                                       uint32_t external1, uint32_t extra0,
                                       uint32_t extra1)
{
    ASSERT(CLB_isBaseValid(base));

    EALLOW;
    HWREG(base + CLB_O_FSM_EXTERNAL_IN0) = external0;
    HWREG(base + CLB_O_FSM_EXTERNAL_IN1) = external1;
    HWREG(base + CLB_O_FSM_EXTRA_IN0) = extra0;
    HWREG(base + CLB_O_FSM_EXTRA_IN1) = extra1;
    EDIS;
}

//*****************************************************************************
//
//! Configures FSM LUT function.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param fsmLutFn10 is the value for FSM 0 & FSM 1 LUT function. Generated
//!        by tool as \b TILEx_CFG_FSM_LUT_FN10.
//! \param fsmLutFn2 is the value for FSM 2 LUT function. Generated by tool as
//!        \b TILEx_CFG_FSM_LUT_FN2.
//!
//! This function configures the FSM block's LUT equations.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_configFSMLUTFunction(uint32_t base, uint32_t fsmLutFn10,
                                            uint32_t fsmLutFn2)
{
    ASSERT(CLB_isBaseValid(base));

    EALLOW;
    HWREG(base + CLB_O_FSM_LUT_FN1_0) = fsmLutFn10;
    HWREG(base + CLB_O_FSM_LUT_FN2) = fsmLutFn2;
    EDIS;
}

//*****************************************************************************
//
//! Configures FSM next state.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param nextState0 is the value for FSM 0's next state. Generated by tool as
//!        \b TILEx_CFG_FSM_NEXT_STATE_0.
//! \param nextState1 is the value for FSM 1's next state. Generated by tool as
//!        \b TILEx_CFG_FSM_NEXT_STATE_1.
//! \param nextState2 is the value for FSM 2's next state. Generated by tool as
//!        \b TILEx_CFG_FSM_NEXT_STATE_2.
//!
//! This function configures the FSM's next state equation.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_configFSMNextState(uint32_t base, uint32_t nextState0,
                                          uint32_t nextState1,
                                          uint32_t nextState2)
{
    ASSERT(CLB_isBaseValid(base));

    EALLOW;
    HWREG(base + CLB_O_FSM_NEXT_STATE_0) = nextState0;
    HWREG(base + CLB_O_FSM_NEXT_STATE_1) = nextState1;
    HWREG(base + CLB_O_FSM_NEXT_STATE_2) = nextState2;
    EDIS;
}

static inline void CLB_FSM_MODE(uint32_t base, uint32_t fsm_mode)
{


    EALLOW;
    HWREG(base + CLB_LOGICCTL + CLB_O_FSM_MODE) =fsm_mode;

    EDIS;
}

//*****************************************************************************
//
//! Selects Counter inputs.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param reset is the value for counter's reset inputs. Generated by tool as
//!        \b TILEx_CFG_COUNTER_RESET.
//! \param event is the value for counter's event inputs. Generated by tool as
//!        \b TILEx_CFG_COUNTER_EVENT.
//! \param mode0 is the value for counter's mode 0 inputs. Generated by tool as
//!        \b TILEx_CFG_COUNTER_MODE_0.
//! \param mode1 is the value for counter's mode 1 inputs. Generated by tool as
//!        \b TILEx_CFG_COUNTER_MODE_1.
//!
//! This function selects the input signals to the counter block.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_selectCounterInputs(uint32_t base, uint32_t reset,
                                           uint32_t event, uint32_t mode0,
                                           uint32_t mode1)
{
    ASSERT(CLB_isBaseValid(base));

    EALLOW;
    HWREG(base + CLB_O_COUNT_RESET) = reset;
    HWREG(base + CLB_O_COUNT_EVENT) = event;
    HWREG(base + CLB_O_COUNT_MODE_0) = mode0;
    HWREG(base + CLB_O_COUNT_MODE_1) = mode1;
    EDIS;
}


static inline uint32_t GetCurrentCount(uint32_t base,uint32_t counter)
{
	uint32_t value;
	value =  counter*4;
	return(HWREG(base + CLB_LOGICCTL + CLB_O_DBG_C0 + value));
}
//*****************************************************************************
//
//! Configures Counter and FSM modes.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param miscCtrl is the value to represent counter and FSM modes.
//!        Generated by tool as \b TILEx_CFG_MISC_CONTROL.
//!
//! This function configures the counter mode, particularly add/shift, load
//! modes. The function also configures whether the FSM should use state inputs
//! or an extra external input.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_configMiscCtrlModes(uint32_t base, uint32_t miscCtrl)
{
    ASSERT(CLB_isBaseValid(base));

    EALLOW;
    HWREG(base + CLB_O_MISC_CONTROL) = miscCtrl;
    EDIS;
}

//*****************************************************************************
//
//! Configures Output LUT functions.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param outID is the specified CLB tile output signal.
//! \param outputCfg is the value for the output LUT signal function and input
//!        signal selections. Generated by tool as \b TILEx_CFG_OUTLUT_n where
//!        n is the output number.
//!
//! This function configures the input signals and equations of the output LUT
//! corresponding to the /e outID parameter.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_configOutputLUT(uint32_t base, CLB_Outputs outID,
                                       uint32_t outputCfg)
{
    ASSERT(CLB_isBaseValid(base));

    EALLOW;
    HWREG(base + CLB_O_OUTPUT_LUT_0 + (0X4U * outID)) = outputCfg;
    EDIS;
}

//*****************************************************************************
//
//! Configures HLC event selection.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param event_num is the event id
//! \param eventSel is the value for HLC event selection. Generated by tool as
//!        \b TILEx_HLC_EVENT_SEL.
//!
//! This function configures the event selection for the High Level Controller.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_configHLCEventSelect(uint32_t base,uint32_t event_num, uint32_t eventSel)
{
    ASSERT(CLB_isBaseValid(base));

    EALLOW;
    uint32_t offset;
    if(event_num<4)
    {
    	offset=5*event_num;
    	HWREG(base + CLB_O_HLC_EVENT_SEL1) |= eventSel<<offset;
    }
    else
    {
    	offset=5*(event_num-4);
    HWREG(base + CLB_O_HLC_EVENT_SEL2) |= eventSel<<offset;
    }
    EDIS;
}
//*****************************************************************************
//
//! Configures counter source of load in count
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param cnt_num is the counter id
//! \param loadSel is the source of count which select from r0~r3
//!
//!
//! This function configures the event selection for the High Level Controller.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_configCntLoadSource(uint32_t base,CLB_Counters cnt_num, uint32_t loadSel)
{
    ASSERT(CLB_isBaseValid(base));

    EALLOW;
    uint32_t offset;
    offset=24+cnt_num*2;
    HWREG(base + CLB_O_HLC_EVENT_SEL2) |= loadSel<<offset;


    EDIS;
}
//*****************************************************************************
//
//! Program HLC instruction.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param instructionNum is the index into the HLC instruction memory. For
//!        example, a value of 0 corresponds to instruction 0 of event 0,
//!        a value of 1 corresponds to instruction 1 of event 0, and so on up
//!        to a value of 31 which corresponds to instruction 7 of event 3.
//! \param instruction is the instruction to be programmed. Generated by tool
//!        as \b TILEx_HLCINSTR_n where n is the instruction number.
//!
//! This function configures the CLB internal memory corresponding to the
//! specified HLC instruction number with the given instruction.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_programHLCInstruction(uint32_t base,
                                             uint32_t instructionNum,
                                             uint32_t instruction)
{
    ASSERT(CLB_isBaseValid(base));
    ASSERT(instructionNum < 32);

    CLB_writeInterface(base, CLB_ADDR_HLC_BASE + instructionNum, instruction);
}

//*****************************************************************************
//
//! Set HLC registers.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param r0Init is the value to write to HLC register R0. Generated by tool
//!        as \b TILEx_HLC_R0_INIT.
//! \param r1Init is the value to write to HLC register R1. Generated by tool
//!        as \b TILEx_HLC_R1_INIT.
//! \param r2Init is the value to write to HLC register R2. Generated by tool
//!        as \b TILEx_HLC_R2_INIT.
//! \param r3Init is the value to write to HLC register R3. Generated by tool
//!        as \b TILEx_HLC_R3_INIT.
//!
//! This function configures the CLB internal memory corresponding to the HLC
//! registers R0-R3 with the specified values.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_setHLCRegisters(uint32_t base, uint32_t r0Init,
                                       uint32_t r1Init, uint32_t r2Init,
                                       uint32_t r3Init)
{
    ASSERT(CLB_isBaseValid(base));
    CLB_writeInterface(base, CLB_O_DBG_R0, r0Init);
    CLB_writeInterface(base, CLB_O_DBG_R1, r1Init);
    CLB_writeInterface(base, CLB_O_DBG_R2, r2Init);
    CLB_writeInterface(base, CLB_O_DBG_R3, r3Init);
}
//*****************************************************************************
//
//! Enable HLC event.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param HLC_CTRnum is the value to Control which CNT
//! \param match_num is the value to  Control which match.Ex:match1 match2
//!
//! The function of this function is to enable the loading event of the
//! corresponding CNT match
//! \return None.
//
//*****************************************************************************
static inline  void CLB_enableLoadValue(uint32_t base,CLB_Counters HLC_CTRnum,uint32_t match_num)
{

	HWREG(base + CLB_O_CTRL_0 +HLC_CTRnum*4)= match_num;
}

//*****************************************************************************
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param load_value0 is the value to load  count0 CNT
//! \param load_value1 is the value to load  count1 CNT
//!
//! This function is used to set the load value of CNT,
//! which is only valid when enableloadCNT is enabled
//! \return None.
//
//*****************************************************************************
static inline  void CLB_LoadValue(uint32_t base,uint32_t load_value0,uint32_t load_value1,uint32_t load_value2)
{
	HWREG(base +CLB_O_LOAD_VALUE_0) =  load_value0;
	HWREG(base +CLB_O_LOAD_VALUE_1) =  load_value1;
	HWREG(base +CLB_O_LOAD_VALUE_2) =  load_value2;
}

static inline  void CLB_writeDbgRx(uint32_t base ,uint32_t RX ,uint32_t match_num)
{
	if(RX < 4)
	{
		CLB_writeInterface(base, CLB_O_DBG_R0+RX*4, match_num);
	}
	else
	{
		HWREG(base +CLB_O_DBG_R4 + (RX-4)*4) =  match_num;
	}
}
//*****************************************************************************
//
//! Get HLC or counter register values.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param registerID is the internal register from which to read. Can be
//!        either an HLC register (\b CLB_REG_HLC_Rn) or a counter value
//!        (\b CLB_REG_CTR_Cn).
//!
//! \return Returns the value in the specified HLC register or counter.
//
//*****************************************************************************
static inline uint32_t CLB_getRegister(uint32_t base, CLB_Register registerID)
{
    ASSERT(CLB_isBaseValid(base));

    return(HWREG(base + CLB_LOGICCTL + (uint32_t)registerID));
}

//*****************************************************************************
//
//! Get output status.
//!
//! \param base is the base address of a CLB tile's logic config register.
//!
//! \return Returns the output status of various components within the CLB tile
//!         such as a counter match or LUT output. Use the \b CLB_DBG_OUT_*
//!         masks from <tt>hw_clb.h</tt> to decode the bits.
//
//*****************************************************************************
static inline uint32_t CLB_getOutputStatus(uint32_t base)
{
    ASSERT(CLB_isBaseValid(base));

    return(HWREG(base + CLB_LOGICCTL + CLB_O_DBG_OUT));
}
//*****************************************************************************
//
//! CLB_writeSWReleaseControl.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param EventID is the trigger event which you want to occur
//! \This function is used to trigger the HLC event that you want to produce
//
//*****************************************************************************
static inline void CLB_SW_EVENT_TRIG(uint32_t base,uint32_t EventID)
{
    ASSERT(CLB_isBaseValid(base));


    HWREG(base + CLB_LOGICCTL + CLB_O_SW_EVENT_TRIG) =
            (HWREG(base + CLB_LOGICCTL + CLB_O_SW_EVENT_TRIG) |
                    ( 1<<  EventID));
}

static inline void CLB_LEVEL_23_ORDER(uint32_t base,CLB_Outputs CLB_OUT)
{

	 HWREG(base + CLB_LOGICCTL + CLB_O_LEVEL_23_ORDER) =
	 	 	 (HWREG(base + CLB_LOGICCTL + CLB_O_LEVEL_23_ORDER) &
	 		         ~(0x1<<CLB_OUT)) | (1<<CLB_OUT);
}


static inline void CLB_LEVEL_RLS_TZ_SEL(uint32_t base,CLB_Outputs CLB_OUT)
{
	  ASSERT(CLB_isBaseValid(base));


	    HWREG(base + CLB_LOGICCTL + CLB_O_RLS_TZ_SEL) =
	    		(HWREG(base + CLB_LOGICCTL + CLB_O_RLS_TZ_SEL) &
		 		         ~(0x1<<CLB_OUT)) | (1<<CLB_OUT);
}

static inline void CLB_LEVEL_RLS_TZ_TYPE(uint32_t base,CLB_Outputs CLB_OUT,uint32_t type )
{

	  ASSERT(CLB_isBaseValid(base));
	  int saval= CLB_OUT * 0x4;

	    HWREG(base + CLB_LOGICCTL + CLB_O_RLS_TZ_TYPE) =
	    	(HWREG(base + CLB_LOGICCTL + CLB_O_RLS_TZ_TYPE) &
	   		 		         ~(0x7<<saval)) | (type<<saval);


}
static inline void CLB_LEVEL_RLS_TZ_CPU1(uint32_t base,uint32_t enable )
{

	HWREG(base + CLB_LOGICCTL + CLB_O_RLS_TZ_CPU1) =
	        (HWREG(base + CLB_LOGICCTL + CLB_O_RLS_TZ_CPU1) &
	         ~(0xff)) | (enable);
}
static inline void CLB_LEVEL_RLS_TZ_CPU2(uint32_t base,uint32_t enable )
{

	HWREG(base + CLB_LOGICCTL + CLB_O_RLS_TZ_CPU2) |=
			 (HWREG(base + CLB_LOGICCTL + CLB_O_RLS_TZ_CPU2) &
				         ~(0xff)) | (enable);
}


static inline void CLB_Connect_SPICLK(uint32_t spi_base,XBAR_OutputMuxConfig CLB_OUTCLK_sel,XBAR_OutputMuxConfig CLB_OUTCS_sel )
{
 uint32_t clkmuxsel=(CLB_OUTCLK_sel>>2)/2;
 uint32_t csmuxsel=(CLB_OUTCS_sel>>2)/2;
 uint32_t base=CLBOUTPUTXBAR_BASE+0x100;

#if IS_GS32F00xx(0x12)
	if(spi_base == SPIA_BASE)
	{
	    XBAR_enableOutputMux(base ,XBAR_OUTPUT11,clkmuxsel);
	   	XBAR_enableOutputMux(base ,XBAR_OUTPUT9,csmuxsel);
		HWREG(SYSCTL_BASE +0xf18) |= 0x5;
	}
	else {
		
		XBAR_enableOutputMux(base ,XBAR_OUTPUT12,clkmuxsel);
		XBAR_enableOutputMux(base ,XBAR_OUTPUT10,csmuxsel);
		HWREG(SYSCTL_BASE +0xf18) |= 0xA;
	}
#endif
#if IS_GS32F3xx(0x22)
switch (spi_base) {
	case SPIA_BASE:
	    XBAR_setOutputMuxConfig(CLBOUTPUTXBAR_BASE,XBAR_OUTPUT1,CLB_OUTCLK_sel );
	    XBAR_enableOutputMux(CLBOUTPUTXBAR_BASE ,XBAR_OUTPUT1,clkmuxsel);
	    XBAR_setOutputMuxConfig(CLBOUTPUTXBAR_BASE,XBAR_OUTPUT2,CLB_OUTCS_sel );
	   	XBAR_enableOutputMux(CLBOUTPUTXBAR_BASE ,XBAR_OUTPUT2,csmuxsel);
		HWREG(SYSCTL_BASE +0xf18) |= 0x3;

		break;
	case SPIB_BASE:
	    XBAR_setOutputMuxConfig(CLBOUTPUTXBAR_BASE,XBAR_OUTPUT3,CLB_OUTCLK_sel );
	    XBAR_enableOutputMux(CLBOUTPUTXBAR_BASE ,XBAR_OUTPUT3,clkmuxsel);
	    XBAR_setOutputMuxConfig(CLBOUTPUTXBAR_BASE,XBAR_OUTPUT4,CLB_OUTCS_sel );
	   	XBAR_enableOutputMux(CLBOUTPUTXBAR_BASE ,XBAR_OUTPUT4,csmuxsel);
		HWREG(SYSCTL_BASE +0xf18) |= 0xC;

		break;
	case SPIC_BASE:
	    XBAR_setOutputMuxConfig(CLBOUTPUTXBAR_BASE,XBAR_OUTPUT5,CLB_OUTCLK_sel );
	    XBAR_enableOutputMux(CLBOUTPUTXBAR_BASE ,XBAR_OUTPUT5,clkmuxsel);
	    XBAR_setOutputMuxConfig(CLBOUTPUTXBAR_BASE,XBAR_OUTPUT6,CLB_OUTCS_sel );
	   	XBAR_enableOutputMux(CLBOUTPUTXBAR_BASE ,XBAR_OUTPUT6,csmuxsel);
		HWREG(SYSCTL_BASE +0xf18) |= 0x30;

		break;
	case SPID_BASE:
	    XBAR_setOutputMuxConfig(CLBOUTPUTXBAR_BASE,XBAR_OUTPUT7,CLB_OUTCLK_sel );
	    XBAR_enableOutputMux(CLBOUTPUTXBAR_BASE ,XBAR_OUTPUT7,clkmuxsel);
	    XBAR_setOutputMuxConfig(CLBOUTPUTXBAR_BASE,XBAR_OUTPUT8,CLB_OUTCS_sel );
	   	XBAR_enableOutputMux(CLBOUTPUTXBAR_BASE ,XBAR_OUTPUT8,csmuxsel);
		HWREG(SYSCTL_BASE +0xf18) |= 0xC0;

		break;

}
#endif
}
//*****************************************************************************
//
//! Configures Counter load and match.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param counterID is the specified counter unit.
//! \param load is the value for counter's load mode. Generated by tool as
//!        \b TILEx_COUNTER_n_LOAD_VAL where n is the counter number.
//! \param match1 is the value for counter's match 1. Generated by tool as
//!        \b TILEx_COUNTER_n_MATCH1_VAL where n is the counter number.
//! \param match2 is the value for counter's match 2. Generated by tool as
//!        \b TILEx_COUNTER_n_MATCH2_VAL where n is the counter number.
//!
//! This function configures the CLB internal memory corresponding to the
//! counter block's load and match values.
//!
//! \return None.
//
//*****************************************************************************
extern void CLB_configCounterLoadMatch(uint32_t base, CLB_Counters counterID,
                                       uint32_t load, uint32_t match1,
                                       uint32_t match2);



#endif

#if IS_GS32F00xx(0x30)
//*****************************************************************************
//
// Address offsets from LOGICCFG to LOGICCTL and DATAEXCH register memory maps
//
//*****************************************************************************
#define CLB_INPUTCCTL               0x0100U
#define CLB_LOGICCTL                0x0200U
#define CLB_DATAEXCH                0x0300U

//*****************************************************************************
//
// Address offsets for CLB-internal memory space
//
//*****************************************************************************
#define CLB_ADDR_COUNTER_0_LOAD     0x0U
#define CLB_ADDR_COUNTER_1_LOAD     0x1U
#define CLB_ADDR_COUNTER_2_LOAD     0x2U

#define CLB_ADDR_COUNTER_0_MATCH1   0x4U
#define CLB_ADDR_COUNTER_1_MATCH1   0x5U
#define CLB_ADDR_COUNTER_2_MATCH1   0x6U

#define CLB_ADDR_COUNTER_0_MATCH2   0x8U
#define CLB_ADDR_COUNTER_1_MATCH2   0x9U
#define CLB_ADDR_COUNTER_2_MATCH2   0xAU

#define CLB_ADDR_HLC_R0             0xCU
#define CLB_ADDR_HLC_R1             0xDU
#define CLB_ADDR_HLC_R2             0xEU
#define CLB_ADDR_HLC_R3             0xFU

#define CLB_ADDR_HLC_BASE           0x20U
#define CLB_NUM_HLC_INSTR           31U

//*****************************************************************************
//
// PUSH/PULL FIFO size (32-bit registers)
//
//*****************************************************************************
#define CLB_FIFO_SIZE               4U

//*****************************************************************************
//
// Key to enable writes to the CLB registers
//
//*****************************************************************************
#define CLB_LOCK_KEY                0x5A5AU

//*****************************************************************************
//
// Shift and masks needed by the API for Input selection
//
//*****************************************************************************
#define CLB_LCL_MUX_SEL_MISC_INPUT_SEL_M          0x20U
#define CLB_LCL_MUX_SEL_MISC_INPUT_SEL_S          28U
#define CLB_LCL_MUX_SEL_MISC_INPUT_SEL_BITM       (uint32_t)1U

//*****************************************************************************
//
//! Values that can be passed to control the CLB output enable signal. It can
//! be passed to CLB_setOutputMask() as the \e outputMask parameter.
//
//*****************************************************************************
#define CLB_OUTPUT_00 0x00000001U //!< Mask for CLB OUTPUT ENABLE/DISABLE 0
#define CLB_OUTPUT_01 0x00000002U //!< Mask for CLB OUTPUT ENABLE/DISABLE 1
#define CLB_OUTPUT_02 0x00000004U //!< Mask for CLB OUTPUT ENABLE/DISABLE 2
#define CLB_OUTPUT_03 0x00000008U //!< Mask for CLB OUTPUT ENABLE/DISABLE 3
#define CLB_OUTPUT_04 0x00000010U //!< Mask for CLB OUTPUT ENABLE/DISABLE 4
#define CLB_OUTPUT_05 0x00000020U //!< Mask for CLB OUTPUT ENABLE/DISABLE 5
#define CLB_OUTPUT_06 0x00000040U //!< Mask for CLB OUTPUT ENABLE/DISABLE 6
#define CLB_OUTPUT_07 0x00000080U //!< Mask for CLB OUTPUT ENABLE/DISABLE 7
#define CLB_OUTPUT_08 0x00000100U //!< Mask for CLB OUTPUT ENABLE/DISABLE 8
#define CLB_OUTPUT_09 0x00000200U //!< Mask for CLB OUTPUT ENABLE/DISABLE 9
#define CLB_OUTPUT_10 0x00000400U //!< Mask for CLB OUTPUT ENABLE/DISABLE 10
#define CLB_OUTPUT_11 0x00000800U //!< Mask for CLB OUTPUT ENABLE/DISABLE 11
#define CLB_OUTPUT_12 0x00001000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 12
#define CLB_OUTPUT_13 0x00002000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 13
#define CLB_OUTPUT_14 0x00004000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 14
#define CLB_OUTPUT_15 0x00008000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 15
#define CLB_OUTPUT_16 0x00010000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 16
#define CLB_OUTPUT_17 0x00020000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 17
#define CLB_OUTPUT_18 0x00040000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 18
#define CLB_OUTPUT_19 0x00080000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 19
#define CLB_OUTPUT_20 0x00100000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 20
#define CLB_OUTPUT_21 0x00200000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 21
#define CLB_OUTPUT_22 0x00400000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 22
#define CLB_OUTPUT_23 0x00800000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 23
#define CLB_OUTPUT_24 0x01000000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 24
#define CLB_OUTPUT_25 0x02000000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 25
#define CLB_OUTPUT_26 0x04000000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 26
#define CLB_OUTPUT_27 0x08000000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 27
#define CLB_OUTPUT_28 0x10000000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 28
#define CLB_OUTPUT_29 0x20000000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 29
#define CLB_OUTPUT_30 0x40000000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 30
#define CLB_OUTPUT_31 0x80000000U //!< Mask for CLB OUTPUT ENABLE/DISABLE 31

//*****************************************************************************
//
//! Values that can be passed to select CLB input signal
//
//*****************************************************************************
typedef enum
{
    CLB_IN0 = 0,                    //!< Input 0
    CLB_IN1 = 1,                    //!< Input 1
    CLB_IN2 = 2,                    //!< Input 2
    CLB_IN3 = 3,                    //!< Input 3
    CLB_IN4 = 4,                    //!< Input 4
    CLB_IN5 = 5,                    //!< Input 5
    CLB_IN6 = 6,                    //!< Input 6
    CLB_IN7 = 7                     //!< Input 7
} CLB_Inputs;

//*****************************************************************************
//
//! Values that can be passed to select CLB output signal. It can be passed to
//! CLB_configOutputLUT() as the \e outID parameter.
//
//*****************************************************************************
typedef enum
{
    CLB_OUT0 = 0,                   //!< Output 0
    CLB_OUT1 = 1,                   //!< Output 1
    CLB_OUT2 = 2,                   //!< Output 2
    CLB_OUT3 = 3,                   //!< Output 3
    CLB_OUT4 = 4,                   //!< Output 4
    CLB_OUT5 = 5,                   //!< Output 5
    CLB_OUT6 = 6,                   //!< Output 6
    CLB_OUT7 = 7                    //!< Output 7
} CLB_Outputs;

//*****************************************************************************
//
//! Values that can be passed to select CLB AOC signal. It can be passed to
//! CLB_configAOC() as the \e aocID parameter.
//! AOC is the Asynchronous Output Conditioning block.
//
//*****************************************************************************
typedef enum
{
    CLB_AOC0 = 0,                   //!< AOC 0
    CLB_AOC1 = 1,                   //!< AOC 1
    CLB_AOC2 = 2,                   //!< AOC 2
    CLB_AOC3 = 3,                   //!< AOC 3
    CLB_AOC4 = 4,                   //!< AOC 4
    CLB_AOC5 = 5,                   //!< AOC 5
    CLB_AOC6 = 6,                   //!< AOC 6
    CLB_AOC7 = 7                    //!< AOC 7
} CLB_AOCs;

//*****************************************************************************
//
//! Values that can be passed to set/clear CLB SW release signals. It can
//! be passed to CLB_writeSWReleaseControl() as the \e inID parameter.
//
//*****************************************************************************
typedef enum
{
    CLB_SW_RLS_CTRL0 = 0,           //!< SW RLS CTRL 0
    CLB_SW_RLS_CTRL1 = 1,           //!< SW RLS CTRL 1
    CLB_SW_RLS_CTRL2 = 2,           //!< SW RLS CTRL 2
    CLB_SW_RLS_CTRL3 = 3,           //!< SW RLS CTRL 3
    CLB_SW_RLS_CTRL4 = 4,           //!< SW RLS CTRL 4
    CLB_SW_RLS_CTRL5 = 5,           //!< SW RLS CTRL 5
    CLB_SW_RLS_CTRL6 = 6,           //!< SW RLS CTRL 6
    CLB_SW_RLS_CTRL7 = 7            //!< SW RLS CTRL 7
} CLB_SWReleaseCtrl;

//*****************************************************************************
//
//! Values that can be passed to set/clear CLB SW release signals. It can
//! be passed to CLB_writeSWGateControl() as the \e inID parameter.
//
//*****************************************************************************
typedef enum
{
    CLB_SW_GATE_CTRL0 = 0,           //!< SW GATE CTRL 0
    CLB_SW_GATE_CTRL1 = 1,           //!< SW GATE CTRL 1
    CLB_SW_GATE_CTRL2 = 2,           //!< SW GATE CTRL 2
    CLB_SW_GATE_CTRL3 = 3,           //!< SW GATE CTRL 3
    CLB_SW_GATE_CTRL4 = 4,           //!< SW GATE CTRL 4
    CLB_SW_GATE_CTRL5 = 5,           //!< SW GATE CTRL 5
    CLB_SW_GATE_CTRL6 = 6,           //!< SW GATE CTRL 6
    CLB_SW_GATE_CTRL7 = 7            //!< SW GATE CTRL 7
} CLB_SWGateCtrl;

//*****************************************************************************
//
//! Values that can be passed to select CLB counter. It can be passed to
//! CLB_configCounterLoadMatch() as the \e counterID parameter.
//
//*****************************************************************************
typedef enum
{
    CLB_CTR0 = 0,                   //!< Counter 0
    CLB_CTR1 = 1,                   //!< Counter 1
    CLB_CTR2 = 2                    //!< Counter 2
} CLB_Counters;

//*****************************************************************************
//
//! Values that can be passed to CLB_getRegister() as the \e registerID
//! parameter.
//
//*****************************************************************************
typedef enum
{
    CLB_REG_HLC_R0 = CLB_O_DBG_R0,  //!< HLC R0 register
    CLB_REG_HLC_R1 = CLB_O_DBG_R1,  //!< HLC R1 register
    CLB_REG_HLC_R2 = CLB_O_DBG_R2,  //!< HLC R2 register
    CLB_REG_HLC_R3 = CLB_O_DBG_R3,  //!< HLC R3 register
    CLB_REG_CTR_C0 = CLB_O_DBG_C0,  //!< Counter 0 register
    CLB_REG_CTR_C1 = CLB_O_DBG_C1,  //!< Counter 1 register
    CLB_REG_CTR_C2 = CLB_O_DBG_C2   //!< Counter 2 register
} CLB_Register;

//*****************************************************************************
//
//! Values that can be passed to CLB_selectInputFilter() as the \e filterType
//! parameter.
//
//*****************************************************************************
typedef enum
{
    CLB_FILTER_NONE         = 0,    //!< No filtering
    CLB_FILTER_RISING_EDGE  = 1,    //!< Rising edge detect
    CLB_FILTER_FALLING_EDGE = 2,    //!< Falling edge detect
    CLB_FILTER_ANY_EDGE     = 3     //!< Any edge detect
} CLB_FilterType;

//*****************************************************************************
//
//! Values that can be passed to CLB_configGPInputMux() as the \e gpMuxCfg
//! parameter.
//
//*****************************************************************************
typedef enum
{
    CLB_GP_IN_MUX_EXTERNAL  = 0,    //!< Use external input path
    CLB_GP_IN_MUX_GP_REG    = 1     //!< Use CLB_GP_REG bit value as input
} CLB_GPInputMux;




//*****************************************************************************
//
//! Values that can be passed to CLB_configGlobalInputMux() as the
//! \e globalMuxCfg parameter.
//
//*****************************************************************************
typedef enum
{
	    CLB_GLOBAL_IN_MUX_EPWM1A            = 0,   //!< EPWM1A
	    CLB_GLOBAL_IN_MUX_EPWM1A_OE         = 1,   //!< EPWM1A trip output
	    CLB_GLOBAL_IN_MUX_EPWM1B            = 2,   //!< EPWM1B
	    CLB_GLOBAL_IN_MUX_EPWM1B_OE         = 3,   //!< EPWM1B trip output
	    CLB_GLOBAL_IN_MUX_EPWM1_CTR_ZERO    = 4,   //!< EPWM1 TBCTR = Zero
	    CLB_GLOBAL_IN_MUX_EPWM1_CTR_PRD     = 5,   //!< EPWM1 TBCTR = TBPRD
	    CLB_GLOBAL_IN_MUX_EPWM1_CTRDIR      = 6,   //!< EPWM1 CTRDIR
	    CLB_GLOBAL_IN_MUX_EPWM1_TBCLK       = 7,   //!< EPWM1 TBCLK
	    CLB_GLOBAL_IN_MUX_EPWM1_CTR_CMPA    = 8,   //!< EPWM1 TBCTR = CMPA
	    CLB_GLOBAL_IN_MUX_EPWM1_CTR_CMPB    = 9,   //!< EPWM1 TBCTR = CMPB
	    CLB_GLOBAL_IN_MUX_EPWM1_CTR_CMPC    = 10,  //!< EPWM1 TBCTR = CMPC
	    CLB_GLOBAL_IN_MUX_EPWM1_CTR_CMPD    = 11,  //!< EPWM1 TBCTR = CMPD
	    CLB_GLOBAL_IN_MUX_EPWM1A_AQ         = 12,  //!< EPWM1A AQ submodule output
	    CLB_GLOBAL_IN_MUX_EPWM1B_AQ         = 13,  //!< EPWM1B AQ submodule output
	    CLB_GLOBAL_IN_MUX_EPWM1A_DB         = 14,  //!< EPWM1A DB submodule output
	    CLB_GLOBAL_IN_MUX_EPWM1B_DB         = 15,  //!< EPWM1B DB submodule output

	    CLB_GLOBAL_IN_MUX_EPWM2A            = 16,  //!< EPWM2A
	    CLB_GLOBAL_IN_MUX_EPWM2A_OE         = 17,  //!< EPWM2A trip output
	    CLB_GLOBAL_IN_MUX_EPWM2B            = 18,  //!< EPWM2B
	    CLB_GLOBAL_IN_MUX_EPWM2B_OE         = 19,  //!< EPWM2B trip output
	    CLB_GLOBAL_IN_MUX_EPWM2_CTR_ZERO    = 20,  //!< EPWM2 TBCTR = Zero
	    CLB_GLOBAL_IN_MUX_EPWM2_CTR_PRD     = 21,  //!< EPWM2 TBCTR = TBPRD
	    CLB_GLOBAL_IN_MUX_EPWM2_CTRDIR      = 22,  //!< EPWM2 CTRDIR
	    CLB_GLOBAL_IN_MUX_EPWM2_TBCLK       = 23,  //!< EPWM2 TBCLK
	    CLB_GLOBAL_IN_MUX_EPWM2_CTR_CMPA    = 24,  //!< EPWM2 TBCTR = CMPA
	    CLB_GLOBAL_IN_MUX_EPWM2_CTR_CMPB    = 25,  //!< EPWM2 TBCTR = CMPB
	    CLB_GLOBAL_IN_MUX_EPWM2_CTR_CMPC    = 26,  //!< EPWM2 TBCTR = CMPC
	    CLB_GLOBAL_IN_MUX_EPWM2_CTR_CMPD    = 27,  //!< EPWM2 TBCTR = CMPD
	    CLB_GLOBAL_IN_MUX_EPWM2A_AQ         = 28,  //!< EPWM2A AQ submodule output
	    CLB_GLOBAL_IN_MUX_EPWM2B_AQ         = 29,  //!< EPWM2B AQ submodule output
	    CLB_GLOBAL_IN_MUX_EPWM2A_DB         = 30,  //!< EPWM2A DB submodule output
	    CLB_GLOBAL_IN_MUX_EPWM2B_DB         = 31,  //!< EPWM2B DB submodule output

	    CLB_GLOBAL_IN_MUX_EPWM3A            = 32,  //!< EPWM3A
	    CLB_GLOBAL_IN_MUX_EPWM3A_OE         = 33,  //!< EPWM3A trip output
	    CLB_GLOBAL_IN_MUX_EPWM3B            = 34,  //!< EPWM3B
	    CLB_GLOBAL_IN_MUX_EPWM3B_OE         = 35,  //!< EPWM3B trip output
	    CLB_GLOBAL_IN_MUX_EPWM3_CTR_ZERO    = 36,  //!< EPWM3 TBCTR = Zero
	    CLB_GLOBAL_IN_MUX_EPWM3_CTR_PRD     = 37,  //!< EPWM3 TBCTR = TBPRD
	    CLB_GLOBAL_IN_MUX_EPWM3_CTRDIR      = 38,  //!< EPWM3 CTRDIR
	    CLB_GLOBAL_IN_MUX_EPWM3_TBCLK       = 39,  //!< EPWM3 TBCLK
	    CLB_GLOBAL_IN_MUX_EPWM3_CTR_CMPA    = 40,  //!< EPWM3 TBCTR = CMPA
	    CLB_GLOBAL_IN_MUX_EPWM3_CTR_CMPB    = 41,  //!< EPWM3 TBCTR = CMPB
	    CLB_GLOBAL_IN_MUX_EPWM3_CTR_CMPC    = 42,  //!< EPWM3 TBCTR = CMPC
	    CLB_GLOBAL_IN_MUX_EPWM3_CTR_CMPD    = 43,  //!< EPWM3 TBCTR = CMPD
	    CLB_GLOBAL_IN_MUX_EPWM3A_AQ         = 44,  //!< EPWM3A AQ submodule output
	    CLB_GLOBAL_IN_MUX_EPWM3B_AQ         = 45,  //!< EPWM3B AQ submodule output
	    CLB_GLOBAL_IN_MUX_EPWM3A_DB         = 46,  //!< EPWM3A DB submodule output
	    CLB_GLOBAL_IN_MUX_EPWM3B_DB         = 47,  //!< EPWM3B DB submodule output

	    CLB_GLOBAL_IN_MUX_EPWM4A            = 48,  //!< EPWM4A
	    CLB_GLOBAL_IN_MUX_EPWM4A_OE         = 49,  //!< EPWM4A trip output
	    CLB_GLOBAL_IN_MUX_EPWM4B            = 50,  //!< EPWM4B
	    CLB_GLOBAL_IN_MUX_EPWM4B_OE         = 51,  //!< EPWM4B trip output
	    CLB_GLOBAL_IN_MUX_EPWM4_CTR_ZERO    = 52,  //!< EPWM4 TBCTR = Zero
	    CLB_GLOBAL_IN_MUX_EPWM4_CTR_PRD     = 53,  //!< EPWM4 TBCTR = TBPRD
	    CLB_GLOBAL_IN_MUX_EPWM4_CTRDIR      = 54,  //!< EPWM4 CTRDIR
	    CLB_GLOBAL_IN_MUX_EPWM4_TBCLK       = 55,  //!< EPWM4 TBCLK
	    CLB_GLOBAL_IN_MUX_EPWM4_CTR_CMPA    = 56,  //!< EPWM4 TBCTR = CMPA
	    CLB_GLOBAL_IN_MUX_EPWM4_CTR_CMPB    = 57,  //!< EPWM4 TBCTR = CMPB
	    CLB_GLOBAL_IN_MUX_EPWM4_CTR_CMPC    = 58,  //!< EPWM4 TBCTR = CMPC
	    CLB_GLOBAL_IN_MUX_EPWM4_CTR_CMPD    = 59,  //!< EPWM4 TBCTR = CMPD
	    CLB_GLOBAL_IN_MUX_EPWM4A_AQ         = 60,  //!< EPWM4A AQ submodule output
	    CLB_GLOBAL_IN_MUX_EPWM4B_AQ         = 61,  //!< EPWM4B AQ submodule output
	    CLB_GLOBAL_IN_MUX_EPWM4A_DB         = 62,  //!< EPWM4A DB submodule output
	    CLB_GLOBAL_IN_MUX_EPWM4B_DB         = 63,  //!< EPWM4B DB submodule output

	    CLB_GLOBAL_IN_MUX_EPWM5A            = 64,  //!< EPWM5A
	    CLB_GLOBAL_IN_MUX_EPWM5A_OE         = 65,  //!< EPWM5A trip output
	    CLB_GLOBAL_IN_MUX_EPWM5B            = 66,  //!< EPWM5B
	    CLB_GLOBAL_IN_MUX_EPWM5B_OE         = 67,  //!< EPWM5B trip output
	    CLB_GLOBAL_IN_MUX_EPWM6A            = 68,  //!< EPWM5A
	    CLB_GLOBAL_IN_MUX_EPWM6A_OE         = 69,  //!< EPWM5A trip output
	    CLB_GLOBAL_IN_MUX_EPWM6B            = 70,  //!< EPWM6B
	    CLB_GLOBAL_IN_MUX_EPWM6B_OE         = 71,  //!< EPWM6B trip output
	    CLB_GLOBAL_IN_MUX_EPWM7A            = 72,  //!< EPWM7A
	    CLB_GLOBAL_IN_MUX_EPWM7A_OE         = 73,  //!< EPWM7A trip output
	    CLB_GLOBAL_IN_MUX_EPWM7B            = 74,  //!< EPWM7B
	    CLB_GLOBAL_IN_MUX_EPWM7B_OE         = 75,  //!< EPWM7B trip output
	    CLB_GLOBAL_IN_MUX_EPWM8A            = 76,  //!< EPWM8A
	    CLB_GLOBAL_IN_MUX_EPWM8A_OE         = 77,  //!< EPWM8A trip output
	    CLB_GLOBAL_IN_MUX_EPWM8B            = 78,  //!< EPWM8B
	    CLB_GLOBAL_IN_MUX_EPWM8B_OE         = 79 ,  //!< EPWM8B trip output
//	    CLB_GLOBAL_IN_MUX_EPWM9A            = 80,  //!< EPWM9A
//	    CLB_GLOBAL_IN_MUX_EPWM9A_OE         = 81,  //!< EPWM9A trip output
//	    CLB_GLOBAL_IN_MUX_EPWM9B            = 82,  //!< EPWM9B
//	    CLB_GLOBAL_IN_MUX_EPWM9B_OE         = 83,  //!< EPWM9B trip output
//	    CLB_GLOBAL_IN_MUX_EPWM10A            = 84,  //!< EPWM10A
//	    CLB_GLOBAL_IN_MUX_EPWM10A_OE         = 85,  //!< EPWM10A trip output
//	    CLB_GLOBAL_IN_MUX_EPWM10B            = 86,  //!< EPWM10B
//	    CLB_GLOBAL_IN_MUX_EPWM10B_OE         = 87,  //!< EPWM10B trip output
//	    CLB_GLOBAL_IN_MUX_EPWM11A            = 88,  //!< EPWM11A
//	    CLB_GLOBAL_IN_MUX_EPWM11A_OE         = 89,  //!< EPWM11A trip output
//	    CLB_GLOBAL_IN_MUX_EPWM11B            = 90,  //!< EPWM11B
//	    CLB_GLOBAL_IN_MUX_EPWM11B_OE         = 91,  //!< EPWM11B trip output
//	    CLB_GLOBAL_IN_MUX_EPWM12A            = 92,  //!< EPWM12A
//	    CLB_GLOBAL_IN_MUX_EPWM12A_OE         = 93,  //!< EPWM12A trip output
//	    CLB_GLOBAL_IN_MUX_EPWM12B            = 94,  //!< EPWM12B
//	    CLB_GLOBAL_IN_MUX_EPWM12B_OE         = 95,  //!< EPWM12B trip output

	    CLB_GLOBAL_IN_MUX_EPWM1_DACA1EVT    = 96,  //!< EPWM1
		CLB_GLOBAL_IN_MUX_EPWM1_DACA2EVT    = 97,  //!< EPWM1
		CLB_GLOBAL_IN_MUX_EPWM1_DACB1EVT    = 98,  //!< EPWM1
		CLB_GLOBAL_IN_MUX_EPWM1_DACB2EVT    = 99,  //!< EPWM1
		CLB_GLOBAL_IN_MUX_EPWM1_DCAH        = 100,  //!< EPWM1
		CLB_GLOBAL_IN_MUX_EPWM1_DCAI        = 101,  //!< EPWM1
		CLB_GLOBAL_IN_MUX_EPWM1_DCBH        = 102,  //!< EPWM1
		CLB_GLOBAL_IN_MUX_EPWM1_DCBI        = 103,  //!< EPWM1
		CLB_GLOBAL_IN_MUX_EPWM1_OST         = 104,  //!< EPWM1
		CLB_GLOBAL_IN_MUX_EPWM1_CBC         = 105,  //!< EPWM1
		CLB_GLOBAL_IN_MUX_EPWM1_OST_CBC     = 106,  //!< EPWM1
		CLB_GLOBAL_IN_MUX_EPWM1_CTR_CMP1A   = 107,  //!< EPWM1
		CLB_GLOBAL_IN_MUX_EPWM1_CTR_CMP1B   = 108,  //!< EPWM1
		CLB_GLOBAL_IN_MUX_EPWM1_ZERO         = 109,  //!< EPWM1

	    CLB_GLOBAL_IN_MUX_EPWM2_DACA1EVT    = 110,  //!< EPWM2
		CLB_GLOBAL_IN_MUX_EPWM2_DACA2EVT    = 111,  //!< EPWM2
		CLB_GLOBAL_IN_MUX_EPWM2_DACB1EVT    = 112,  //!< EPWM2
		CLB_GLOBAL_IN_MUX_EPWM2_DACB2EVT    = 113,  //!< EPWM2
		CLB_GLOBAL_IN_MUX_EPWM2_DCAH        = 114,  //!< EPWM2
		CLB_GLOBAL_IN_MUX_EPWM2_DCAI        = 115,  //!< EPWM2
		CLB_GLOBAL_IN_MUX_EPWM2_DCBH        = 116,  //!< EPWM2
		CLB_GLOBAL_IN_MUX_EPWM2_DCBI        = 117,  //!< EPWM2
		CLB_GLOBAL_IN_MUX_EPWM2_OST         = 118,  //!< EPWM2
		CLB_GLOBAL_IN_MUX_EPWM2_CBC         = 119,  //!< EPWM2
		CLB_GLOBAL_IN_MUX_EPWM2_OST_CBC     = 120,  //!< EPWM2
		CLB_GLOBAL_IN_MUX_EPWM2_CTR_CMP1A   = 121,  //!< EPWM2
		CLB_GLOBAL_IN_MUX_EPWM2_CTR_CMP1B   = 122,  //!< EPWM2
		CLB_GLOBAL_IN_MUX_EPWM2_ZERO         = 123,  //!< EPWM2

	    CLB_GLOBAL_IN_MUX_EPWM3_DACA1EVT    = 124,  //!< EPWM3
		CLB_GLOBAL_IN_MUX_EPWM3_DACA2EVT    = 125,  //!< EPWM3
		CLB_GLOBAL_IN_MUX_EPWM3_DACB1EVT    = 126,  //!< EPWM3
		CLB_GLOBAL_IN_MUX_EPWM3_DACB2EVT    = 127,  //!< EPWM3
		CLB_GLOBAL_IN_MUX_EPWM3_DCAH        = 128,  //!< EPWM3
		CLB_GLOBAL_IN_MUX_EPWM3_DCAI        = 129,  //!< EPWM3
		CLB_GLOBAL_IN_MUX_EPWM3_DCBH        = 130,  //!< EPWM3
		CLB_GLOBAL_IN_MUX_EPWM3_DCBI        = 131,  //!< EPWM3
		CLB_GLOBAL_IN_MUX_EPWM3_OST         = 132,  //!< EPWM3
		CLB_GLOBAL_IN_MUX_EPWM3_CBC         = 133,  //!< EPWM3
		CLB_GLOBAL_IN_MUX_EPWM3_OST_CBC     = 134,  //!< EPWM3
		CLB_GLOBAL_IN_MUX_EPWM3_CTR_CMP1A   = 135,  //!< EPWM3
		CLB_GLOBAL_IN_MUX_EPWM3_CTR_CMP1B   = 136,  //!< EPWM3
		CLB_GLOBAL_IN_MUX_EPWM3_ZERO         = 137,  //!< EPWM3

	    CLB_GLOBAL_IN_MUX_EPWM4_DACA1EVT    = 138,  //!< EPWM4
		CLB_GLOBAL_IN_MUX_EPWM4_DACA2EVT    = 139,  //!< EPWM4
		CLB_GLOBAL_IN_MUX_EPWM4_DACB1EVT    = 140,  //!< EPWM4
		CLB_GLOBAL_IN_MUX_EPWM4_DACB2EVT    = 141,  //!< EPWM4
		CLB_GLOBAL_IN_MUX_EPWM4_DCAH        = 142,  //!< EPWM4
		CLB_GLOBAL_IN_MUX_EPWM4_DCAI        = 143,  //!< EPWM4
		CLB_GLOBAL_IN_MUX_EPWM4_DCBH        = 144,  //!< EPWM4
		CLB_GLOBAL_IN_MUX_EPWM4_DCBI        = 145,  //!< EPWM4
		CLB_GLOBAL_IN_MUX_EPWM4_OST         = 146,  //!< EPWM4
		CLB_GLOBAL_IN_MUX_EPWM4_CBC         = 147,  //!< EPWM4
		CLB_GLOBAL_IN_MUX_EPWM4_OST_CBC     = 148,  //!< EPWM4
		CLB_GLOBAL_IN_MUX_EPWM4_CTR_CMP1A   = 149,  //!< EPWM4
		CLB_GLOBAL_IN_MUX_EPWM4_CTR_CMP1B   = 150,  //!< EPWM4
		CLB_GLOBAL_IN_MUX_EPWM4_ZERO         = 151,  //!< EPWM4

//	    CLB_GLOBAL_IN_MUX_EPWM5_CTR_PRD     = 53,  //!< EPWM5 TBCTR = TBPRD
//	    CLB_GLOBAL_IN_MUX_EPWM5_CTRDIR      = 54,  //!< EPWM5 CTRDIR
//	    CLB_GLOBAL_IN_MUX_EPWM5_TBCLK       = 55,  //!< EPWM5 TBCLK
//	    CLB_GLOBAL_IN_MUX_EPWM5_CTR_CMPA    = 56,  //!< EPWM5 TBCTR = CMPA
//	    CLB_GLOBAL_IN_MUX_EPWM5_CTR_CMPB    = 57,  //!< EPWM5 TBCTR = CMPB
//	    CLB_GLOBAL_IN_MUX_EPWM5_CTR_CMPC    = 58,  //!< EPWM5 TBCTR = CMPC
//	    CLB_GLOBAL_IN_MUX_EPWM5_CTR_CMPD    = 59,  //!< EPWM5 TBCTR = CMPD
//	    CLB_GLOBAL_IN_MUX_EPWM5A_AQ         = 60,  //!< EPWM5A AQ submodule output
//	    CLB_GLOBAL_IN_MUX_EPWM5B_AQ         = 61,  //!< EPWM5B AQ submodule output
//	    CLB_GLOBAL_IN_MUX_EPWM5A_DB         = 62,  //!< EPWM5A DB submodule output
//	    CLB_GLOBAL_IN_MUX_EPWM5B_DB         = 63,  //!< EPWM5B DB submodule output
//
//
//	    CLB_GLOBAL_IN_MUX_FSIRXA_DATA_PACKET_RCVD  = 112,  //!< FSIRXA Data Packet Received
//	    CLB_GLOBAL_IN_MUX_FSIRXA_ERROR_PACKET_RCVD = 113,  //!< FSIRXA Error Packet Received
//	    CLB_GLOBAL_IN_MUX_FSIRXA_PING_PACKET_RCVD  = 114,  //!< FSIRXA PING Packet Received
//	    CLB_GLOBAL_IN_MUX_FSIRXA_OUTPUT_FRAME_DONE = 115,  //!< FSIRXA Output Frame Done
//	    CLB_GLOBAL_IN_MUX_FSIRXA_PACKET_TAG0       = 116,  //!< FSIRXA Packet TAG0
//	    CLB_GLOBAL_IN_MUX_FSIRXA_PACKET_TAG1       = 117,  //!< FSIRXA Packet TAG1
//	    CLB_GLOBAL_IN_MUX_FSIRXA_PACKET_TAG2       = 118,  //!< FSIRXA Packet TAG2
//	    CLB_GLOBAL_IN_MUX_FSIRXA_PACKET_TAG3       = 119,  //!< FSIRXA Packet TAG3
//
//	    CLB_GLOBAL_IN_MUX_SPI1_SPICLK               = 120,  //!< SPI1 SPICLK OUT
//	    CLB_GLOBAL_IN_MUX_SPI1_SPIPOCI_CONTROLLER   = 121,  //!< SPI1 SPIPOCI Controller IN
//	    CLB_GLOBAL_IN_MUX_SPI1_SPIPTE               = 122,  //!< SPI1 SPIPTE OUT
//	    CLB_GLOBAL_IN_MUX_SPI2_SPICLK               = 123,  //!< SPI2 SPICLK OUT
//	    CLB_GLOBAL_IN_MUX_SPI2_SPIPOCI_CONTROLLER   = 124,  //!< SPI2 SPIPOCI Controller IN
//	    CLB_GLOBAL_IN_MUX_SPI2_SPIPTE               = 125,  //!< SPI2 SPIPTE OUT
//

	    CLB_GLOBAL_IN_MUX_XBAE_CLB_OUT     =152,  //!< plu_xbar
	    CLB_GLOBAL_IN_MUX_XBAE_CLB1_OUT     =153,  //!< plu_xbar
	    CLB_GLOBAL_IN_MUX_XBAE_CLB2_OUT     =154,  //!< plu_xbar
	    CLB_GLOBAL_IN_MUX_XBAE_CLB3_OUT     =155,  //!< plu_xbar
	    CLB_GLOBAL_IN_MUX_XBAE_CLB4_OUT     =156,  //!< plu_xbar
	    CLB_GLOBAL_IN_MUX_XBAE_CLB5_OUT     =157,  //!< plu_xbar
	    CLB_GLOBAL_IN_MUX_XBAE_CLB6_OUT     =158,  //!< plu_xbar
	    CLB_GLOBAL_IN_MUX_XBAE_CLB7_OUT     =159,  //!< plu_xbar

		CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL1_OUT   = 160, //plu input xbar
		CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL2_OUT   = 161,
		CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL3_OUT   = 162,
		CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL4_OUT   = 163,
		CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL5_OUT   = 164,
		CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL6_OUT   = 165,
		CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL7_OUT   = 166,
		CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL8_OUT   = 167,
		CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL9_OUT   = 168,
		CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL10_OUT  = 169,
		CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL11_OUT  = 170,
		CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL12_OUT  = 171,
		CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL13_OUT  = 172,
		CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL14_OUT  = 173,
		CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL15_OUT  = 174,
		CLB_GLOBAL_IN_MUX_XBAE_CLB_INPUT_SEL16_OUT  = 175,

		CLB_GLOBAL_IN_MUX_CLB1_OUT0                 = 176,
		CLB_GLOBAL_IN_MUX_CLB1_OUT1                 = 177,
		CLB_GLOBAL_IN_MUX_CLB1_OUT2                 = 178,
		CLB_GLOBAL_IN_MUX_CLB1_OUT3                 = 179,
		CLB_GLOBAL_IN_MUX_CLB1_OUT4                 = 180,
		CLB_GLOBAL_IN_MUX_CLB1_OUT5                 = 181,
		CLB_GLOBAL_IN_MUX_CLB1_OUT6                 = 182,
		CLB_GLOBAL_IN_MUX_CLB1_OUT7                 = 183,
		CLB_GLOBAL_IN_MUX_CLB1_PRESCALE_CLK         = 184,

		CLB_GLOBAL_IN_MUX_CLB2_OUT0                 = 188,
		CLB_GLOBAL_IN_MUX_CLB2_OUT1                 = 189,
		CLB_GLOBAL_IN_MUX_CLB2_OUT2                 = 190,
		CLB_GLOBAL_IN_MUX_CLB2_OUT3                 = 191,
		CLB_GLOBAL_IN_MUX_CLB2_OUT4                 = 192,
		CLB_GLOBAL_IN_MUX_CLB2_OUT5                 = 193,
		CLB_GLOBAL_IN_MUX_CLB2_OUT6                 = 194,
		CLB_GLOBAL_IN_MUX_CLB2_OUT7                 = 195,
		CLB_GLOBAL_IN_MUX_CLB2_PRESCALE_CLK         = 196,



		CLB_GLOBAL_IN_MUX_CLB3_OUT0                 = 200,
		CLB_GLOBAL_IN_MUX_CLB3_OUT1                 = 201,
		CLB_GLOBAL_IN_MUX_CLB3_OUT2                 = 202,
		CLB_GLOBAL_IN_MUX_CLB3_OUT3                 = 203,
		CLB_GLOBAL_IN_MUX_CLB3_OUT4                 = 204,
		CLB_GLOBAL_IN_MUX_CLB3_OUT5                 = 205,
		CLB_GLOBAL_IN_MUX_CLB3_OUT6                 = 206,
		CLB_GLOBAL_IN_MUX_CLB3_OUT7                 = 207,

		CLB_GLOBAL_IN_MUX_CLB4_OUT0                 = 212,
		CLB_GLOBAL_IN_MUX_CLB4_OUT1                 = 213,
		CLB_GLOBAL_IN_MUX_CLB4_OUT2                 = 214,
		CLB_GLOBAL_IN_MUX_CLB4_OUT3                 = 215,
		CLB_GLOBAL_IN_MUX_CLB4_OUT4                 = 216,
		CLB_GLOBAL_IN_MUX_CLB4_OUT5                 = 217,
		CLB_GLOBAL_IN_MUX_CLB4_OUT6                 = 218,
		CLB_GLOBAL_IN_MUX_CLB4_OUT7                 = 219,

		CLB_GLOBAL_IN_MUX_ECAP1_IN                    = 224,
		CLB_GLOBAL_IN_MUX_ECAP1_OUT                   = 225,
		CLB_GLOBAL_IN_MUX_ECAP1_OEN                   = 226,
		CLB_GLOBAL_IN_MUX_ECAP1_CEVT1                 = 227,
		CLB_GLOBAL_IN_MUX_ECAP1_CEVT2                 = 228,
		CLB_GLOBAL_IN_MUX_ECAP1_CEVT3                 = 229,
		CLB_GLOBAL_IN_MUX_ECAP1_CEVT4                 = 230,
		CLB_GLOBAL_IN_MUX_ECAP2_IN                    = 231,
		CLB_GLOBAL_IN_MUX_ECAP2_OUT                   = 232,
		CLB_GLOBAL_IN_MUX_ECAP2_OEN                   = 233,
		CLB_GLOBAL_IN_MUX_ECAP2_CEVT1                 = 234,
		CLB_GLOBAL_IN_MUX_ECAP2_CEVT2                 = 235,
		CLB_GLOBAL_IN_MUX_ECAP2_CEVT3                 = 236,
		CLB_GLOBAL_IN_MUX_ECAP2_CEVT4                 = 237,
		CLB_GLOBAL_IN_MUX_ECAP3_IN                    = 238,
		CLB_GLOBAL_IN_MUX_ECAP3_OUT                   = 239,
		CLB_GLOBAL_IN_MUX_ECAP3_OEN                   = 240,
		CLB_GLOBAL_IN_MUX_ECAP3_CEVT1                 = 241,
		CLB_GLOBAL_IN_MUX_ECAP3_CEVT2                 = 242,
		CLB_GLOBAL_IN_MUX_ECAP3_CEVT3                 = 243,
		CLB_GLOBAL_IN_MUX_ECAP3_CEVT4                 = 244,
//		CLB_GLOBAL_IN_MUX_ECAP4_IN                    = 245,
//		CLB_GLOBAL_IN_MUX_ECAP4_OUT                   = 246,
//		CLB_GLOBAL_IN_MUX_ECAP4_OEN                   = 247,
//		CLB_GLOBAL_IN_MUX_ECAP4_CEVT1                 = 248,
//		CLB_GLOBAL_IN_MUX_ECAP4_CEVT2                 = 249,
//		CLB_GLOBAL_IN_MUX_ECAP4_CEVT3                 = 250,
//		CLB_GLOBAL_IN_MUX_ECAP4_CEVT4                 = 251,
		CLB_GLOBAL_IN_MUX_TBCLKSYN_TO_CLB             = 252,
		CLB_GLOBAL_IN_MUX_CPU_HATL                    = 253,
		CLB_GLOBAL_IN_MUX_DSA0_DBG_STOPTIME           = 254,
		CLB_GLOBAL_IN_MUX_DSA1_DBG_STOPTIME           = 255,

		CLB_GLOBAL_IN_MUX_SPI1_SIMO_IN               = 266,
		CLB_GLOBAL_IN_MUX_SPI1_SIMO_OUT              = 267,
		CLB_GLOBAL_IN_MUX_SPI1_SIMO_OEN              = 268,
		CLB_GLOBAL_IN_MUX_SPI1_SOMI_IN               = 269,
		CLB_GLOBAL_IN_MUX_SPI1_SOMI_OUT              = 270,
		CLB_GLOBAL_IN_MUX_SPI1_SOMI_OEN              = 271,
		CLB_GLOBAL_IN_MUX_SPI1_SLK_IN                = 272,
		CLB_GLOBAL_IN_MUX_SPI1_SLK_OUT               = 273,
		CLB_GLOBAL_IN_MUX_SPI2_SIMO_IN               = 274,
		CLB_GLOBAL_IN_MUX_SPI2_SIMO_OUT              = 275,
		CLB_GLOBAL_IN_MUX_SPI2_SIMO_OEN              = 276,
		CLB_GLOBAL_IN_MUX_SPI2_SOMI_IN               = 277,
		CLB_GLOBAL_IN_MUX_SPI2_SOMI_OUT              = 278,
		CLB_GLOBAL_IN_MUX_SPI2_SOMI_OEN              = 279,
		CLB_GLOBAL_IN_MUX_SPI2_SLK_IN                = 280,
		CLB_GLOBAL_IN_MUX_SPI2_SLK_OUT               = 281,
//		CLB_GLOBAL_IN_MUX_SPI3_SIMO_IN               = 282,
//		CLB_GLOBAL_IN_MUX_SPI3_SIMO_OUT              = 283,
//		CLB_GLOBAL_IN_MUX_SPI3_SIMO_OEN              = 284,
//		CLB_GLOBAL_IN_MUX_SPI3_SOMI_IN               = 285,
//		CLB_GLOBAL_IN_MUX_SPI3_SOMI_OUT              = 286,
//		CLB_GLOBAL_IN_MUX_SPI3_SOMI_OEN              = 287,
//		CLB_GLOBAL_IN_MUX_SPI3_SLK_IN                = 288,
//		CLB_GLOBAL_IN_MUX_SPI3_SLK_OUT               = 289,
//		CLB_GLOBAL_IN_MUX_SPI4_SIMO_IN               = 290,
//		CLB_GLOBAL_IN_MUX_SPI4_SIMO_OUT              = 291,
//		CLB_GLOBAL_IN_MUX_SPI4_SIMO_OEN              = 292,
//		CLB_GLOBAL_IN_MUX_SPI4_SOMI_IN               = 293,
//		CLB_GLOBAL_IN_MUX_SPI4_SOMI_OUT              = 294,
//		CLB_GLOBAL_IN_MUX_SPI4_SOMI_OEN              = 295,
//		CLB_GLOBAL_IN_MUX_SPI4_SLK_IN                = 296,
//		CLB_GLOBAL_IN_MUX_SPI4_SLK_OUT               = 297,
		CLB_GLOBAL_IN_MUX_EQEP1_A                    =298, //< eqep
		CLB_GLOBAL_IN_MUX_EQEP1_B                    =299,
		CLB_GLOBAL_IN_MUX_EQEP1_I                    =300,
		CLB_GLOBAL_IN_MUX_EQEP1_S                    =301,
		CLB_GLOBAL_IN_MUX_EQEP2_A                    =302,
		CLB_GLOBAL_IN_MUX_EQEP2_B                    =303,
		CLB_GLOBAL_IN_MUX_EQEP2_I                   = 304,
		CLB_GLOBAL_IN_MUX_EQEP2_S                    =305,
	    CLB_GLOBAL_IN_MUX_EQEP3_A                    =306,
		CLB_GLOBAL_IN_MUX_EQEP3_B                    =307,
		CLB_GLOBAL_IN_MUX_EQEP3_I                   = 308,
		CLB_GLOBAL_IN_MUX_EQEP3_S                    =309,

		CLB_GLOBAL_IN_MUX_UART0_SOUT                 =310, //< UART
	    CLB_GLOBAL_IN_MUX_UART1_SOUT                 =311, //< UART
	    CLB_GLOBAL_IN_MUX_UART2_SOUT                 =312, //< UART
	    CLB_GLOBAL_IN_MUX_UART3_SOUT                 =313, //< UART

	    CLB_GLOBAL_IN_MUX_EPWM1_XSYNCI               =314, //<
		CLB_GLOBAL_IN_MUX_EPWM2_XSYNCI               =315, //<
		CLB_GLOBAL_IN_MUX_EPWM3_XSYNCI               =316, //<
		CLB_GLOBAL_IN_MUX_EPWM4_XSYNCI               =317, //<
		CLB_GLOBAL_IN_MUX_EPWM5_XSYNCI               =318, //<
		CLB_GLOBAL_IN_MUX_EPWM6_XSYNCI               =319, //<
		CLB_GLOBAL_IN_MUX_EPWM7_XSYNCI               =320, //<
		CLB_GLOBAL_IN_MUX_EPWM8_XSYNCI               =321, //<


} CLB_GlobalInputMux;

//*****************************************************************************
//
// Prototypes for the APIs.
//
//*****************************************************************************
//
//!
//! Checks the CLB base address.
//!
//! \param base is the base address of a CLB tile's logic config register.
//!
//! This function determines if a CLB base address is valid.
//!
//! \return Returns \b true if the base address is valid and \b false
//! otherwise.
//
//*****************************************************************************
#ifdef DEBUG
static inline bool CLB_isBaseValid(uint32_t base)
{
    return(
           (base == CLB1_BASE) ||
           (base == CLB2_BASE)
#ifdef CLB3_BASE
           || (base == CLB3_BASE)
#endif

#ifdef CLB4_BASE
           || (base == CLB4_BASE)
#endif
          );
}

//*****************************************************************************
//
//!
//! Checks the CLB internal memory address.
//!
//! \param base is the base address of a CLB tile's logic config register.
//!
//! This function determines if a CLB base address is valid.
//!
//! \return Returns \b true if the address is valid and \b false otherwise.
//
//*****************************************************************************
static inline bool CLB_isAddressValid(uint32_t address)
{
    return(address <= (CLB_ADDR_HLC_BASE + CLB_NUM_HLC_INSTR));
}
#endif

//*****************************************************************************
//
//! Set global enable.
//!
//! \param base is the base address of a CLB tile's logic config register.
//!
//! This function enables the CLB via global enable register.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_enableCLB(uint32_t base)
{
    ASSERT(CLB_isBaseValid(base));


    HWREGH(base + CLB_LOGICCTL + CLB_O_LOAD_EN) |= CLB_LOAD_EN_GLOBAL_EN;

}

//*****************************************************************************
//
//! Clear global enable.
//!
//! \param base is the base address of a CLB tile's logic config register.
//!
//! This function disables the CLB via global enable register.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_disableCLB(uint32_t base)
{
    ASSERT(CLB_isBaseValid(base));


    HWREGH(base + CLB_LOGICCTL + CLB_O_LOAD_EN) &= ~CLB_LOAD_EN_GLOBAL_EN;

}

//*****************************************************************************
//
//! Enable HLC NMI.
//!
//! \param base is the base address of a CLB tile's logic config register.
//!
//! This function enables the CLB HLC NMI.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_enableNMI(uint32_t base)
{
    ASSERT(CLB_isBaseValid(base));


    HWREGH(base + CLB_LOGICCTL + CLB_O_LOAD_EN) |= CLB_LOAD_EN_NMI_EN;

}

//*****************************************************************************
//
//! Disable HLC NMI.
//!
//! \param base is the base address of a CLB tile's logic config register.
//!
//! This function disables the CLB HLC NMI.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_disableNMI(uint32_t base)
{
    ASSERT(CLB_isBaseValid(base));


    HWREGH(base + CLB_LOGICCTL + CLB_O_LOAD_EN) &= ~CLB_LOAD_EN_NMI_EN;

}

//*****************************************************************************
//
//! Configure Clock Prescalar.
//!
//! \param base is the base address of a CLB tile's logic config register.
//!
//! This function enables and configures the CLB Clock Precalar.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_configureClockPrescalar(uint32_t base, uint16_t prescl)
{
    ASSERT(CLB_isBaseValid(base));

    EALLOW;
    HWREG(base + CLB_LOGICCTL + CLB_O_PRESCALE_CTRL) =
        (HWREG(base + CLB_LOGICCTL + CLB_O_PRESCALE_CTRL) &
         ~(CLB_PRESCALE_CTRL_PRESCALE_M)) |
        ((uint32_t)prescl << CLB_PRESCALE_CTRL_PRESCALE_S);
    HWREG(base + CLB_LOGICCTL + CLB_O_PRESCALE_CTRL) |= CLB_PRESCALE_CTRL_CLKEN;
    EDIS;
}

//*****************************************************************************
//
//! Configures the general purpose SW release control value.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param inID is the specified CLB SW Release input signal.
//! \param val is the value of the SW RLS control.
//!
//! This function configures the general purpose SW release control value.
//! The \e inID parameter can have one enumeration value from CLB_SWReleaseCtrl.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_writeSWReleaseControl(uint32_t base,
                                            CLB_SWReleaseCtrl inID, uint32_t val)
{
    ASSERT(CLB_isBaseValid(base));


    HWREG(base + CLB_LOGICCTL + CLB_O_GP_REG) =
              (HWREG(base + CLB_LOGICCTL + CLB_O_GP_REG) &
                  ~(0x1000000U << inID)) |
                      (((uint32_t)val) << (24U + inID));
}


//*****************************************************************************
//
//! Configures the general purpose SW gate control value.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param inID is the specified CLB SW Release input signal.
//! \param val is the value of the SW GATE control.
//!
//! This function configures the general purpose SW release control value.
//! The \e inID parameter can have one enumeration value from CLB_SWGateCtrl.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_writeSWGateControl(uint32_t base,
                                            CLB_SWGateCtrl inID, uint32_t val)
{
    ASSERT(CLB_isBaseValid(base));

    HWREG(base + CLB_LOGICCTL + CLB_O_GP_REG) =
            (HWREG(base + CLB_LOGICCTL + CLB_O_GP_REG) &
                ~(0x10000U << inID)) |
                    (((uint32_t)val) << (16U + inID));

}


//*****************************************************************************
//
//! Configures Counter TAP Selects.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param miscCtrl is the value to represent counter tap selects.
//!        Generated by tool as \b TILEx_CFG_TAP_SELL.
//!
//! This function configures the counter tap selects.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_configCounterTapSelects(uint32_t base, uint32_t tapSel)
{
    ASSERT(CLB_isBaseValid(base));


    HWREG(base + CLB_O_COUNT_MATCH_TAP_SEL) = tapSel;

}

//*****************************************************************************
//
//! Configures AOC (Asynchronous Output Conditioning) functions.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param aocID is the specified CLB tile AOC signal.
//! \param aocCfg is the value for the AOC signal function and input
//!        signal selections. Generated by tool as \b TILEx_OUTPUT_COND_CTR_n
//!        where n is the output number.
//!
//! This function configures the input signals and equations of the aoc LUT
//! corresponding to the /e aocID parameter.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_configAOC(uint32_t base, CLB_AOCs aocID,
                                     uint32_t aocCfg)
{
    ASSERT(CLB_isBaseValid(base));


    HWREG(base + CLB_O_OUTPUT_COND_CTRL_0 + (sizeof(uint32_t) * aocID)) = aocCfg;

}

//*****************************************************************************
//
//! Enable CLB lock.
//!
//! \param base is the base address of a CLB tile's logic config register.
//!
//! This function enables the lock bit of the lock register. The lock can only
//! be set once and can only be cleared by a device reset.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_enableLock(uint32_t base)
{
    ASSERT(CLB_isBaseValid(base));

    //
    // Setting the lock bit requires key 0x5A5A to be written at the same time
    //

    HWREG(base + CLB_LOGICCTL + CLB_O_LOCK) =
        (uint32_t)CLB_LOCK_LOCK | ((uint32_t)CLB_LOCK_KEY << CLB_LOCK_KEY_S);

}

//*****************************************************************************
//
//! Write value to address.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param address is the address of CLB internal memory.
//! \param value is the value to write to specified address.
//!
//! This function writes the specified value to CLB internal memory.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_writeInterface(uint32_t base, uint32_t address,
                                      uint32_t value)
{
    ASSERT(CLB_isBaseValid(base));
    ASSERT(CLB_isAddressValid(address));


    HWREG(base + CLB_LOGICCTL + CLB_O_LOAD_ADDR) = address;
    HWREG(base + CLB_LOGICCTL + CLB_O_LOAD_DATA) =  value;
    HWREG(base + CLB_LOGICCTL + CLB_O_LOAD_EN) |= CLB_LOAD_EN_LOAD_EN;

}

//*****************************************************************************
//
//! Select input Delayed sealing wave
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param inID is the specified CLB tile input signal.
//! \param delayType is Select the latency type.
//! \param delay is delay time .
//!
//! This function configures the filter selection for the specified input.
//! The \e inID parameter can have one enumeration value from CLB_Inputs.
//! The \e filterType parameter can have one enumeration value from
//! CLB_FilterType.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_selectInputDelay(uint32_t base, CLB_Inputs inID,uint32_t delay)
{
    uint16_t shiftVal;

    ASSERT(CLB_isBaseValid(base));

    //
    // Each input has a 2-bit field in this register so need to calculate
    // shift amount accordingly.
    //


    HWREGH(base + CLB_INPUTCCTL + CLB_O_CLB_INPUT_FALLING_FILTER+inID*8) =
    		delay;

	HWREGH(base + CLB_INPUTCCTL + CLB_O_CLB_INPUT_RISING_DELAY+inID*8) =
	    		delay;

}



//*****************************************************************************
//
//! Select input filter type.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param inID is the specified CLB tile input signal.
//! \param filterType is the selected type of filter applied to the input.
//!
//! This function configures the filter selection for the specified input.
//! The \e inID parameter can have one enumeration value from CLB_Inputs.
//! The \e filterType parameter can have one enumeration value from
//! CLB_FilterType.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_selectInputFilter(uint32_t base, CLB_Inputs inID,
                                         CLB_FilterType filterType)
{
    uint16_t shiftVal;

    ASSERT(CLB_isBaseValid(base));

    //
    // Each input has a 2-bit field in this register so need to calculate
    // shift amount accordingly.
    //
    shiftVal = (uint16_t)inID << 1;

    HWREGH(base + CLB_LOGICCTL + CLB_O_INPUT_FILTER) =
        (HWREGH(base + CLB_LOGICCTL + CLB_O_INPUT_FILTER) &
         ~(CLB_INPUT_FILTER_FIN0_M << shiftVal)) |
        ((uint16_t)filterType << shiftVal);
}



//*****************************************************************************
//
//! Configures the general purpose input mux.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param inID is the specified CLB tile input signal.
//! \param gpMuxCfg is the mux selection for the general purpose input mux.
//!
//! This function configures the general purpose input mux. The \e gpMuxCfg
//! parameter can select either the use of an external input signal
//! (\b CLB_GP_IN_MUX_EXTERNAL) or the use of the corresponding CLB_GP_REG bit
//! as an input (\b CLB_GP_IN_MUX_GP_REG).
//! The \e inID parameter can have one enumeration value from CLB_Inputs.
//!
//! \sa CLB_setGPREG() to write to the CLB_GP_REG.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_configGPInputMux(uint32_t base, CLB_Inputs inID,
                                        CLB_GPInputMux gpMuxCfg)
{
    ASSERT(CLB_isBaseValid(base));

    HWREGH(base + CLB_LOGICCTL + CLB_O_IN_MUX_SEL_0) =
        (HWREGH(base + CLB_LOGICCTL + CLB_O_IN_MUX_SEL_0) &
         ~(CLB_IN_MUX_SEL_0_SEL_GP_IN_0 << inID)) | (gpMuxCfg << inID);
}

//*****************************************************************************
//
//! Sets the CLB_GP_REG register value.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param gpRegVal is the value to be written to CLB_GP_REG.
//!
//! This function writes to the CLB_GP_REG register. When the general purpose
//! input mux is configured to use CLB_GP_REG, each bit in \e gpRegVal
//! corresponds to an input signal (bit 0 to Input 0, bit 1 to Input 1, and
//! so on).
//!
//! \sa CLB_configGPInputMux() to select the CLB_GP_REG as the source for
//! an input signal.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_setGPREG(uint32_t base, uint32_t gpRegVal)
{
    ASSERT(CLB_isBaseValid(base));

    HWREG(base + CLB_LOGICCTL + CLB_O_GP_REG) =  (HWREG(base + CLB_LOGICCTL + CLB_O_GP_REG) & !0xFF )|gpRegVal;
}


static inline void CLB_LEVEL_RLS_TZ_SEL(uint32_t base,CLB_Outputs CLB_OUT)
{
	HWREG(base + CLB_INPUTCCTL + CLB_O_RLS_TZ_SEL)|=1<<CLB_OUT;
	HWREG(base + CLB_INPUTCCTL + CLB_O_LEVEL_23_ORDER)|=1<<CLB_OUT;
}

static inline void CLB_LEVEL_RLS_TZ_TYPE(uint32_t base,CLB_Outputs CLB_OUT,uint32_t type )
{
	int saval= CLB_OUT * 0x2;
	HWREG(base + CLB_INPUTCCTL + CLB_O_RLS_TZ_TYPE)|=type<<saval;
}

static inline void CLB_LEVEL_RLS_TZ_CORE0(uint32_t base,CLB_Outputs CLB_OUT,uint32_t enable )
{
	HWREG(base + CLB_INPUTCCTL +CLB_O_RLS_TZ_CPU1)|=enable<<CLB_OUT;
}
static inline void CLB_LEVEL_RLS_TZ_CORE1(uint32_t base,CLB_Outputs CLB_OUT,uint32_t enable )
{

	HWREG(base + CLB_INPUTCCTL +CLB_O_RLS_TZ_CPU2)|=enable<<CLB_OUT;
}

//*****************************************************************************
//
//! Gets the CLB_GP_REG register value.
//!
//! \param base is the base address of a CLB tile's logic config register.
//!
//! This function writes to the CLB_GP_REG register. When the general purpose
//! input mux is configured to use CLB_GP_REG, each bit in \e gpRegVal
//! corresponds to an input signal (bit 0 to Input 0, bit 1 to Input 1, and
//! so on).
//!
//! \sa CLB_configGPInputMux() to select the CLB_GP_REG as the source for
//! an input signal.
//!
//! \return CLB_GP_REG value.
//
//*****************************************************************************
static inline uint32_t CLB_getGPREG(uint32_t base)
{
    ASSERT(CLB_isBaseValid(base));

    return(HWREG(base + CLB_LOGICCTL + CLB_O_GP_REG));
}


//*****************************************************************************
//
//! Configures the global input mux.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param inID is the specified CLB tile input signal.
//! \param globalMuxCfg is the mux selection for the global input mux.
//!
//! This function configures the global input mux for the specified input
//! signal.
//! The \e inID parameter can have one enumeration value from CLB_Inputs.
//! The \e globalMuxCfg parameter can have one enumeration value from
//! CLB_GlobalInputMux.
//!
//! \note The global input mux options' peripheral sources depend on which
//! instance of the CLB (\e base) you are using. For example, for CLB1 the
//! EPWM signal selections come from EPWM1 but for CLB2 they come from EPWM2.
//! See your technical reference manual for details.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_configGlobalInputMux(uint32_t base, CLB_Inputs inID,
                                            CLB_GlobalInputMux globalMuxCfg)
{
    uint16_t shiftVal;

    ASSERT(CLB_isBaseValid(base));

    //
    // Each local input has a 5-bit field in this register so need to calculate
    // shift amount accordingly.
    //


    if(inID < CLB_IN3)
    {
    	shiftVal = (uint16_t)inID * 9;
        HWREG(base + CLB_INPUTCCTL + CLB_O_IN0_SEL) =
            (HWREG(base +CLB_INPUTCCTL + CLB_O_IN0_SEL)  |
            ((uint32_t)globalMuxCfg << shiftVal));
    }
    else if(inID < CLB_IN6)
    {
        //
        // Calculating shift amount for inputs > input3
        //
        shiftVal = (inID-3U) * 9;
        HWREG(base + CLB_INPUTCCTL + CLB_O_IN1_SEL) =
            (HWREG(base + CLB_INPUTCCTL + CLB_O_IN1_SEL) |
            ((uint32_t)globalMuxCfg << shiftVal));
    }
    else
        {
            //
            // Calculating shift amount for inputs > input3
            //
            shiftVal =(inID-6U) * 9;
            HWREG(base + CLB_INPUTCCTL + CLB_O_IN2_SEL) =
                (HWREG(base + CLB_INPUTCCTL +CLB_O_IN2_SEL)  |
                ((uint32_t)globalMuxCfg << shiftVal));
        }
}

//*****************************************************************************
//
//! Controls the output enable.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param outputMask is a mask of the outputs to be enabled.
//! \param enable is a switch to decide if the CLB outputs need to be enabled
//!  or not.
//!
//! This function is used to enable and disable CLB outputs by writing a mask
//! to CLB_OUT_EN. Each bit corresponds to a CLB output. When a bit is 1, the
//! corresponding output is enabled; when a bit is 0, the output is disabled.
//!
//! The \e outputMask parameter takes a logical OR of any of the CLB_OUTPUT_0x
//! values that correspond to the CLB OUTPUT ENABLE for the respective outputs.
//! The \e enable parameter can have one of the values from:
//! false: Disable the respective CLB outputs
//! true: Enable the respective CLB outputs
//!
//! \note Note that the 8 CLB outputs are replicated to create more output
//!  paths. See your technical reference manual for more details.
//!  If no further modifications are expected, then it is advised to set the
//!  block writes bit of the MISC_ACCESS_CTRL Register. This will prevent
//!  accidental writes.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_setOutputMask(uint32_t base, uint32_t outputMask ,
		                                            uint32_t enable)
{
    ASSERT(CLB_isBaseValid(base));

    if(enable == 0x1)
     {
        HWREG(base + CLB_LOGICCTL + CLB_O_OUT_EN) |= outputMask;
     }
    else
     {
        HWREG(base + CLB_LOGICCTL + CLB_O_OUT_EN) &= ~outputMask;
     }
}

//*****************************************************************************
//
//! Reads the interrupt tag register.
//!
//! \param base is the base address of a CLB tile's logic config register.
//!
//! \return Returns the value in the interrupt tag register which is a 6-bit
//!         constant set by the HLC.
//
//*****************************************************************************
static inline uint16_t CLB_getInterruptTag(uint32_t base)
{
    ASSERT(CLB_isBaseValid(base));

    return(HWREGH(base + CLB_LOGICCTL + CLB_O_INTR_TAG_REG));
}

//*****************************************************************************
//
//! Clears the interrupt tag register.
//!
//! \param base is the base address of a CLB tile's logic config register.
//!
//! This function clears the interrupt tag register, setting it to 0.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_clearInterruptTag(uint32_t base)
{
    ASSERT(CLB_isBaseValid(base));

    HWREGH(base + CLB_LOGICCTL + CLB_O_INTR_TAG_REG) = 0U;
}

//*****************************************************************************
//
//! Selects LUT4 inputs.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param lut4In0 is the value for LUT4 input signal 0. Generated by tool as
//!        \b TILEx_CFG_LUT4_IN0.
//! \param lut4In1 is the value for LUT4 input signal 1. Generated by tool as
//!        \b TILEx_CFG_LUT4_IN1.
//! \param lut4In2 is the value for LUT4 input signal 2. Generated by tool as
//!        \b TILEx_CFG_LUT4_IN2.
//! \param lut4In3 is the value for LUT4 input signal 3. Generated by tool as
//!        \b TILEx_CFG_LUT4_IN3.
//!
//! This function configures the LUT4 block's input signals.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_selectLUT4Inputs(uint32_t base, uint32_t lut4In0,
                                        uint32_t lut4In1, uint32_t lut4In2,
                                        uint32_t lut4In3)
{
    ASSERT(CLB_isBaseValid(base));


    HWREG(base + CLB_O_LUT4_IN0) = lut4In0;
    HWREG(base + CLB_O_LUT4_IN1) = lut4In1;
    HWREG(base + CLB_O_LUT4_IN2) = lut4In2;
    HWREG(base + CLB_O_LUT4_IN3) = lut4In3;

}

//*****************************************************************************
//
//! Configures LUT4 functions.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param lut4Fn10 is the equation value for LUT4 blocks 0 and 1. Generated by
//!        tool as \b TILEx_CFG_LUT4_FN10.
//! \param lut4Fn2 is the equation value for LUT4 block2. Generated by tool as
//!        \b TILEx_CFG_LUT4_FN2.
//!
//! This function configures the LUT4 block's equations.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_configLUT4Function(uint32_t base, uint32_t lut4Fn10,
                                          uint32_t lut4Fn2)
{
    ASSERT(CLB_isBaseValid(base));


    HWREG(base + CLB_O_LUT4_FN1_0) = lut4Fn10;
    HWREG(base + CLB_O_LUT4_FN2) = lut4Fn2;

}

//*****************************************************************************
//
//! Selects FSM inputs.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param external0 is the value for FSM external 0 input. Generated by tool
//!        as \b TILEx_CFG_FSM_EXT_IN0.
//! \param external1 is the value for FSM external 1 input. Generated by tool
//!        as \b TILEx_CFG_FSM_EXT_IN1.
//! \param extra0 is the value for FSM extra 0 input. Generated by tool
//!        as \b TILEx_CFG_FSM_EXTRA_IN0.
//! \param extra1 is the value for FSM extra 1 input. Generated by tool
//!        as \b TILEx_CFG_FSM_EXTRA_IN1.
//!
//! This function configures the FSM block's external inputs and extra external
//! inputs.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_selectFSMInputs(uint32_t base, uint32_t external0,
                                       uint32_t external1, uint32_t extra0,
                                       uint32_t extra1)
{
    ASSERT(CLB_isBaseValid(base));


    HWREG(base + CLB_O_FSM_EXTERNAL_IN0) = external0;
    HWREG(base + CLB_O_FSM_EXTERNAL_IN1) = external1;
    HWREG(base + CLB_O_FSM_EXTRA_IN0) = extra0;
    HWREG(base + CLB_O_FSM_EXTRA_IN1) = extra1;

}

//*****************************************************************************
//
//! Configures FSM LUT function.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param fsmLutFn10 is the value for FSM 0 & FSM 1 LUT function. Generated
//!        by tool as \b TILEx_CFG_FSM_LUT_FN10.
//! \param fsmLutFn2 is the value for FSM 2 LUT function. Generated by tool as
//!        \b TILEx_CFG_FSM_LUT_FN2.
//!
//! This function configures the FSM block's LUT equations.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_configFSMLUTFunction(uint32_t base, uint32_t fsmLutFn10,
                                            uint32_t fsmLutFn2)
{
    ASSERT(CLB_isBaseValid(base));


    HWREG(base + CLB_O_FSM_LUT_FN1_0) = fsmLutFn10;
    HWREG(base + CLB_O_FSM_LUT_FN2) = fsmLutFn2;

}

//*****************************************************************************
//
//! Configures FSM next state.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param nextState0 is the value for FSM 0's next state. Generated by tool as
//!        \b TILEx_CFG_FSM_NEXT_STATE_0.
//! \param nextState1 is the value for FSM 1's next state. Generated by tool as
//!        \b TILEx_CFG_FSM_NEXT_STATE_1.
//! \param nextState2 is the value for FSM 2's next state. Generated by tool as
//!        \b TILEx_CFG_FSM_NEXT_STATE_2.
//!
//! This function configures the FSM's next state equation.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_configFSMNextState(uint32_t base, uint32_t nextState0,
                                          uint32_t nextState1,
                                          uint32_t nextState2)
{
    ASSERT(CLB_isBaseValid(base));


    HWREG(base + CLB_O_FSM_NEXT_STATE_0) = nextState0;
    HWREG(base + CLB_O_FSM_NEXT_STATE_1) = nextState1;
    HWREG(base + CLB_O_FSM_NEXT_STATE_2) = nextState2;

}

//*****************************************************************************
//
//! Selects Counter inputs.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param reset is the value for counter's reset inputs. Generated by tool as
//!        \b TILEx_CFG_COUNTER_RESET.
//! \param event is the value for counter's event inputs. Generated by tool as
//!        \b TILEx_CFG_COUNTER_EVENT.
//! \param mode0 is the value for counter's mode 0 inputs. Generated by tool as
//!        \b TILEx_CFG_COUNTER_MODE_0.
//! \param mode1 is the value for counter's mode 1 inputs. Generated by tool as
//!        \b TILEx_CFG_COUNTER_MODE_1.
//!
//! This function selects the input signals to the counter block.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_selectCounterInputs(uint32_t base, uint32_t reset,
                                           uint32_t event, uint32_t mode0,
                                           uint32_t mode1)
{
    ASSERT(CLB_isBaseValid(base));


    HWREG(base + CLB_O_COUNT_RESET) = reset;
    HWREG(base + CLB_O_COUNT_EVENT) = event;
    HWREG(base + CLB_O_COUNT_MODE_0) = mode0;
    HWREG(base + CLB_O_COUNT_MODE_1) = mode1;

}

//*****************************************************************************
//
//! Configures Counter and FSM modes.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param miscCtrl is the value to represent counter and FSM modes.
//!        Generated by tool as \b TILEx_CFG_MISC_CONTROL.
//!
//! This function configures the counter mode, particularly add/shift, load
//! modes. The function also configures whether the FSM should use state inputs
//! or an extra external input.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_configMiscCtrlModes(uint32_t base, uint32_t miscCtrl)
{
    ASSERT(CLB_isBaseValid(base));


    HWREG(base + CLB_O_MISC_CONTROL) = miscCtrl;

}

//*****************************************************************************
//
//! Configures Output LUT functions.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param outID is the specified CLB tile output signal.
//! \param outputCfg is the value for the output LUT signal function and input
//!        signal selections. Generated by tool as \b TILEx_CFG_OUTLUT_n where
//!        n is the output number.
//!
//! This function configures the input signals and equations of the output LUT
//! corresponding to the /e outID parameter.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_configOutputLUT(uint32_t base, CLB_Outputs outID,
                                       uint32_t outputCfg)
{
    ASSERT(CLB_isBaseValid(base));


    HWREG(base + CLB_O_OUTPUT_LUT_0 + (sizeof(uint32_t) * outID)) = outputCfg;

}

//*****************************************************************************
//
//! Configures HLC event selection.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param eventSel is the value for HLC event selection. Generated by tool as
//!        \b TILEx_HLC_EVENT_SEL.
//!
//! This function configures the event selection for the High Level Controller.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_configHLCEventSelect(uint32_t base, uint32_t eventSel)
{
    ASSERT(CLB_isBaseValid(base));


    HWREG(base + CLB_O_HLC_EVENT_SEL) = eventSel;

}

//*****************************************************************************
//
//! Program HLC instruction.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param instructionNum is the index into the HLC instruction memory. For
//!        example, a value of 0 corresponds to instruction 0 of event 0,
//!        a value of 1 corresponds to instruction 1 of event 0, and so on up
//!        to a value of 31 which corresponds to instruction 7 of event 3.
//! \param instruction is the instruction to be programmed. Generated by tool
//!        as \b TILEx_HLCINSTR_n where n is the instruction number.
//!
//! This function configures the CLB internal memory corresponding to the
//! specified HLC instruction number with the given instruction.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_programHLCInstruction(uint32_t base,
                                             uint32_t instructionNum,
                                             uint32_t instruction)
{
    ASSERT(CLB_isBaseValid(base));
    ASSERT(instructionNum < 32);

    CLB_writeInterface(base, CLB_ADDR_HLC_BASE + instructionNum, instruction);
}

//*****************************************************************************
//
//! Set HLC registers.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param r0Init is the value to write to HLC register R0. Generated by tool
//!        as \b TILEx_HLC_R0_INIT.
//! \param r1Init is the value to write to HLC register R1. Generated by tool
//!        as \b TILEx_HLC_R1_INIT.
//! \param r2Init is the value to write to HLC register R2. Generated by tool
//!        as \b TILEx_HLC_R2_INIT.
//! \param r3Init is the value to write to HLC register R3. Generated by tool
//!        as \b TILEx_HLC_R3_INIT.
//!
//! This function configures the CLB internal memory corresponding to the HLC
//! registers R0-R3 with the specified values.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_setHLCRegisters(uint32_t base, uint32_t r0Init,
                                       uint32_t r1Init, uint32_t r2Init,
                                       uint32_t r3Init)
{
    ASSERT(CLB_isBaseValid(base));

    CLB_writeInterface(base, CLB_ADDR_HLC_R0, r0Init);
    CLB_writeInterface(base, CLB_ADDR_HLC_R1, r1Init);
    CLB_writeInterface(base, CLB_ADDR_HLC_R2, r2Init);
    CLB_writeInterface(base, CLB_ADDR_HLC_R3, r3Init);
}

//*****************************************************************************
//
//! Get HLC or counter register values.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param registerID is the internal register from which to read. Can be
//!        either an HLC register (\b CLB_REG_HLC_Rn) or a counter value
//!        (\b CLB_REG_CTR_Cn).
//!
//! \return Returns the value in the specified HLC register or counter.
//
//*****************************************************************************
static inline uint32_t CLB_getRegister(uint32_t base, CLB_Register registerID)
{
    ASSERT(CLB_isBaseValid(base));

    return(HWREG(base + CLB_LOGICCTL + (uint32_t)registerID));
}

//*****************************************************************************
//
//! Get output status.
//!
//! \param base is the base address of a CLB tile's logic config register.
//!
//! \return Returns the output status of various components within the CLB tile
//!         such as a counter match or LUT output. Use the \b CLB_DBG_OUT_*
//!         masks from <tt>hw_clb.h</tt> to decode the bits.
//
//*****************************************************************************
static inline uint32_t CLB_getOutputStatus(uint32_t base)
{
    ASSERT(CLB_isBaseValid(base));

    return(HWREG(base + CLB_LOGICCTL + CLB_O_DBG_OUT));
}


//*****************************************************************************
//
//! Disable CLB Output Mask Updates.
//!
//! \param base is the base address of a CLB tile's logic config register.
//!
//! This function disables the CLB Output Mask updates
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_disableOutputMaskUpdates(uint32_t base)
{
    ASSERT(CLB_isBaseValid(base));


    HWREGH(base + CLB_O_MISC_ACCESS_CTRL) |= CLB_MISC_ACCESS_CTRL_BLKEN;

}

//*****************************************************************************
//
//! Enable CLB Output Mask Updates.
//!
//! \param base is the base address of a CLB tile's logic config register.
//!
//! This function enables the CLB Output Mask updates
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_enableOutputMaskUpdates(uint32_t base)
{
    ASSERT(CLB_isBaseValid(base));


    HWREGH(base + CLB_O_MISC_ACCESS_CTRL) &= ~CLB_MISC_ACCESS_CTRL_BLKEN;

}


//*****************************************************************************
//
//! Disable SPI RX Buffer Access.
//!
//! \param base is the base address of a CLB tile's logic config register.
//!
//! This function disables the CLB SPI RX Buffer access
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_disableSPIBufferAccess(uint32_t base)
{
    ASSERT(CLB_isBaseValid(base));


    HWREGH(base + CLB_O_MISC_ACCESS_CTRL) &= ~CLB_MISC_ACCESS_CTRL_SPIEN;

}

//*****************************************************************************
//
//! Enable SPI RX Buffer Access.
//!
//! \param base is the base address of a CLB tile's logic config register.
//!
//! This function enables the CLB SPI RX Buffer access
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_enableSPIBufferAccess(uint32_t base)
{
    ASSERT(CLB_isBaseValid(base));


    HWREGH(base + CLB_O_MISC_ACCESS_CTRL) |= CLB_MISC_ACCESS_CTRL_SPIEN;

}

//*****************************************************************************
//
//! Configures SPI RX Buffer Load Signal event selection.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param eventSel is the value for HLC event selection. Generated by tool as
//!        \b TILEx_SPI_BUF_EVENT_SEL.
//!
//! This function configures the event selection for the SPI RX Buffer.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_configSPIBufferLoadSignal(uint32_t base, uint16_t eventSel)
{
    ASSERT(CLB_isBaseValid(base));


    HWREGH(base + CLB_O_SPI_DATA_CTRL_HI) =
        (HWREGH(base + CLB_O_SPI_DATA_CTRL_HI) &
         ~((uint16_t)CLB_SPI_DATA_CTRL_HI_STRB_M << CLB_SPI_DATA_CTRL_HI_STRB_S)) |
        ((uint16_t)eventSel << CLB_SPI_DATA_CTRL_HI_STRB_S);

}

//*****************************************************************************
//
//! Configures SPI Export HLC R0 Shift value.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param shiftVal is the value for SPI export HLC R0 bit range selection.
//!
//! This function configures the SPI Export HLC R0 Shift value.
//!
//! \return None.
//
//*****************************************************************************
static inline void CLB_configSPIBufferShift(uint32_t base, uint16_t shiftVal)
{
    ASSERT(CLB_isBaseValid(base));


    HWREGH(base + CLB_O_SPI_DATA_CTRL_HI) =
        (HWREGH(base + CLB_O_SPI_DATA_CTRL_HI) &
         ~((uint16_t)CLB_SPI_DATA_CTRL_HI_SHIFT_M << CLB_SPI_DATA_CTRL_HI_SHIFT_S)) |
        ((uint16_t)shiftVal << CLB_SPI_DATA_CTRL_HI_SHIFT_S);

}

//*****************************************************************************
//
//! Configures Counter load and match.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param counterID is the specified counter unit.
//! \param load is the value for counter's load mode. Generated by tool as
//!        \b TILEx_COUNTER_n_LOAD_VAL where n is the counter number.
//! \param match1 is the value for counter's match 1. Generated by tool as
//!        \b TILEx_COUNTER_n_MATCH1_VAL where n is the counter number.
//! \param match2 is the value for counter's match 2. Generated by tool as
//!        \b TILEx_COUNTER_n_MATCH2_VAL where n is the counter number.
//!
//! This function configures the CLB internal memory corresponding to the
//! counter block's load and match values.
//!
//! \return None.
//
//*****************************************************************************
extern void CLB_configCounterLoadMatch(uint32_t base, CLB_Counters counterID,
                                       uint32_t load, uint32_t match1,
                                       uint32_t match2);

//*****************************************************************************
//
//! Clear FIFO registers.
//!
//! \param base is the base address of a CLB tile's logic config register.
//!
//! This function clears the PUSH/PULL FIFOs as well as its pointers.
//!
//! \return None.
//
//*****************************************************************************
extern void CLB_clearFIFOs(uint32_t base);

//*****************************************************************************
//
//! Configure the FIFO registers.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param pullData[] is a pointer to an array of bytes which needs to be
//! written into the FIFO. The 0th FIFO data is in the 0th index.
//!
//! This function writes to the PULL FIFO. This also clears the FIFOs and
//! its pointer using the CLB_clearFIFOs() API prior to writing to
//! the FIFO.
//!
//! \return None.
//
//*****************************************************************************
extern void CLB_writeFIFOs(uint32_t base, const uint32_t pullData[]);

//*****************************************************************************
//
//! Read FIFO registers.
//!
//! \param base is the base address of a CLB tile's logic config register.
//! \param pushData[] is a pointer to an array of bytes which needs to be
//! read from the FIFO.
//!
//! This function reads from the PUSH FIFO. The 0th FIFO data would be in
//! the 0th index.
//!
//! \return None.
//
//*****************************************************************************
extern void CLB_readFIFOs(uint32_t base , uint32_t pushData[]);

//*****************************************************************************
//
// Close the Doxygen group.
//! @}
//
//*****************************************************************************

//*****************************************************************************
//
// Mark the end of the C bindings section for C++ compilers.
//
//*****************************************************************************


#endif

#ifdef __cplusplus
}
#endif

#endif  // CLB_H
