/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    dac.h
*   @brief   
*
*/

#ifndef DEVICE_DRIVERLIB_DAC_H
#define DEVICE_DRIVERLIB_DAC_H

/*
 * If building with a C++ compiler, make all of the definitions in this header
 * have a C binding.
 */
#ifdef __cplusplus
extern "C"
{
#endif

#include <stdbool.h>
#include <stdint.h>
#include "inc/hw_dac.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "cpu.h"
#include "debug.h"

#if defined(DACA_BASE) || defined(DACB_BASE)
/*
 * @brief A 8-bit register mask
 */
#define DAC_REG_BYTE_MASK       (0xFFU)   //!< Register Byte Mask

/*
 * @brief
 * The following are defines for the reg parameter of the
 * DAC_lockRegister() and DAC_isRegisterLocked() functions.
 */
#define DAC_LOCK_CONTROL        (0x1U)    //!< Lock the control register
#define DAC_LOCK_SHADOW         (0x2U)    //!< Lock the shadow value register
#define DAC_LOCK_OUTPUT         (0x4U)    //!< Lock the output enable register

/*
 * @brief
 * Values that can be passed to DAC_setReferenceVoltage() as the \e source
 * parameter.
 */
typedef enum
{
    DAC_REF_VDAC        = 0,  //!< VDAC reference voltage
    DAC_REF_INTERNAL    = 1   //!< ADC VREFHI reference voltage
}DAC_ReferenceVoltage;

/*
 * @brief Values that can be passed to DAC_setGainMode() as the \e mode parameter.
 */
typedef enum
{
    DAC_GAIN_ONE = 0,     //!< Gain set to 1
    DAC_GAIN_TWO = 2      //!< Gain set to 2
}DAC_GainMode;

/*
 * @brief Values that can be passed to DAC_setLoadMode() as the \e mode parameter.
 */
typedef enum
{
    DAC_LOAD_SYSCLK  = 0,     //!< Load on next SYSCLK
    DAC_LOAD_PWMSYNC = 4      //!< Load on next PWMSYNC specified by SYNCSEL
}DAC_LoadMode;

/*
 * @brief Values that can be passed to DAC_setAnaClkFreq() as the \e mode parameter.
 */
typedef enum
{
    DAC_AnaClk_500K  = 0x0,     //!< Analog clock set to 500KHz
    DAC_AnaClk_1M    = 0x1,     //!< Analog clock set to 1MHz
    DAC_AnaClk_2M    = 0x2,     //!< Analog clock set to 2MHz
    DAC_AnaClk_4M    = 0x3      //!< Analog clock set to 4MHz
}DAC_AnalogClkFreq;

/*
 * @brief Checks DAC base address.
 *
 * @details
 * This function determines if an DAC module base address is valid.
 *
 * @param[in] base Specifies the DAC module base address.
 *
 * @return
 * Returns true if the base address is valid and false otherwise.
 */
#ifdef DEBUG
__STATIC_INLINE bool
DAC_isBaseValid(uint32_t base)
{
    return(

#ifdef DACA_BASE
           (base == DACA_BASE) ||
#endif
           (base == DACB_BASE)
          );
}
#endif

/*
 * @brief Sets the DAC Reference Voltage.
 *
 * @details
 * This function sets the DAC reference voltage.
 *
 * param[in] base is the DAC module base address
 * param[in] source is the selected reference voltage
 *
 * @remarks
 * The source parameter can have one of two values:
 * - DAC_REF_VDAC       - The VDAC reference voltage
 * - DAC_REF_ADC_VREFHI - The ADC VREFHI reference voltage
 *
 */
__STATIC_INLINE void
DAC_setReferenceVoltage(uint32_t base, DAC_ReferenceVoltage source)
{
    /* Check the arguments. */
    ASSERT(DAC_isBaseValid(base));

    /* Set the reference  voltage */
    EALLOW;

    HWREGH(base + DAC_O_CTL) = ((HWREGH(base + DAC_O_CTL) &
                                ~DAC_CTL_DACREFSEL)) | (uint16_t)source;

    EDIS;
}

/*
 * @brief Sets the DAC Gain Mode
 *
 * @details
 * This function sets the DAC gain mode for the buffered output.
 *
 * @param[in] base is the DAC module base address
 * @param[in] mode is the selected gain mode
 *
 * @remarks
 * - The mode parameter can have one of two values:
 * 	 - DAC_GAIN_ONE  - Gain is set to 1
 * 	 - DAC_GAIN_TWO  - Gain is set to 2
 *
 * @note
 * - This value is only used when DAC_REF_ADC_VREFHI is set using
 * 	 DAC_setReferenceVoltage() and internal ADC reference mode is selected.
 *
 */
__STATIC_INLINE void
DAC_setGainMode(uint32_t base, DAC_GainMode mode)
{
    /* Check the arguments. */
    ASSERT(DAC_isBaseValid(base));

    /* Set the gain mode */
    EALLOW;

    HWREGH(base + DAC_O_CTL) = ((HWREGH(base + DAC_O_CTL) &
                                ~DAC_CTL_MODE)) | (uint16_t)mode;

    EDIS;
}

/*
 * @brief Sets the DAC Load Mode
 *
 * @details
 * This function sets the DAC load mode.
 *
 * @param[in] base is the DAC module base address
 * @param[in] mode is the selected load mode
 *
 * @remarks
 * - The mode parameter can have one of two values:
 * 	 - DAC_LOAD_SYSCLK   - Load on next SYSCLK
 * 	 - DAC_LOAD_PWMSYNC  - Load on next PWMSYNC specified by SYNCSEL
 *
 */
__STATIC_INLINE void
DAC_setLoadMode(uint32_t base, DAC_LoadMode mode)
{
    /* Check the arguments. */
    ASSERT(DAC_isBaseValid(base));

    /* Set the load mode */
    EALLOW;

    HWREGH(base + DAC_O_CTL) = ((HWREGH(base + DAC_O_CTL) &
                                ~DAC_CTL_LOADMODE)) | (uint16_t)mode;

    EDIS;
}

/*
 * @brief Sets the DAC PWMSYNC Signal
 *
 * @details
 * This function sets the DAC PWMSYNC signal.
 *
 * @param[in] base is the DAC module base address
 * @param[in] signal is the selected PWM signal
 *
 * @remarks
 * - The signal parameter must be set to a number that represents the PWM
 * 	 signal that will be set. For instance, passing 2 into signal will
 * 	 select PWM sync signal 2.
 *
 */
__STATIC_INLINE void
DAC_setPWMSyncSignal(uint32_t base, uint16_t signal)
{
    /* Check the arguments. */
    ASSERT(DAC_isBaseValid(base));
    ASSERT((signal > 0U) && (signal < 17U));

    /* Set the PWM sync signal */
    EALLOW;

    HWREGH(base + DAC_O_CTL) = (HWREGH(base + DAC_O_CTL) &
                                ~DAC_CTL_SYNCSEL_M) |
                               ((uint16_t)(signal - 1U) << DAC_CTL_SYNCSEL_S);

    EDIS;
}

/*
 * @brief Get the DAC Active Output Value
 *
 * @details
 * This function gets the DAC active output value.
 *
 * @param[in] base is the DAC module base address
 *
 * @return Returns the DAC active output value.
 *
 */
__STATIC_INLINE uint16_t
DAC_getActiveValue(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(DAC_isBaseValid(base));

    /* Get the active value */
    return(HWREGH(base + DAC_O_VALA) & DAC_VALA_DACVALA_M);
}

/*
 * @brief Set the DAC Shadow Output Value.
 *
 * @details
 * This function sets the DAC shadow output value.
 *
 * @param[in] base is the DAC module base address
 * @param[in] value is the 12-bit code to be loaded into the active value register
 *
 * @remarks
 *
 *
 */
__STATIC_INLINE void
DAC_setShadowValue(uint32_t base, uint16_t value)
{
    /* Check the arguments. */
    ASSERT(DAC_isBaseValid(base));
    ASSERT(value <= DAC_VALS_DACVALS_M);

    /* Set the shadow value */
    HWREGH(base + DAC_O_VALS) = (HWREGH(base + DAC_O_VALS) &
                                 ~DAC_VALS_DACVALS_M) |
                                (uint16_t)(value & DAC_VALS_DACVALS_M);
}

/*
 * @brief Get the DAC Shadow Output Value
 *
 * @details
 * This function gets the DAC shadow output value.
 *
 * @param[in] base is the DAC module base address
 *
 * @return Returns the DAC shadow output value.
 *
 *
 */
__STATIC_INLINE uint16_t
DAC_getShadowValue(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(DAC_isBaseValid(base));

    /* Get the shadow value */
    return(HWREGH(base + DAC_O_VALS) & DAC_VALS_DACVALS_M);
}

/*
 * @brief Enable the DAC Output
 *
 * @details
 * This function enables the DAC output.
 *
 * @param[in] base is the DAC module base address
 *
 * @note
 * A delay is required after enabling the DAC. Further details
 * regarding the exact delay time length can be found in the device datasheet.
 *
 */
__STATIC_INLINE void
DAC_enableOutput(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(DAC_isBaseValid(base));

    /* Enable the output */
    EALLOW;

    HWREGH(base + DAC_O_OUTEN) |= DAC_OUTEN_DACOUTEN;

    EDIS;
}

/*
 * @brief Disable the DAC Output
 *
 * @details
 * This function disables the DAC output.
 *
 * @param[in] base is the DAC module base address
 *
 */
__STATIC_INLINE void
DAC_disableOutput(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(DAC_isBaseValid(base));

    /* Disable the output */
    EALLOW;

    HWREGH(base + DAC_O_OUTEN) &= ~DAC_OUTEN_DACOUTEN;

    EDIS;
}

/*
 * @brief Set DAC Offset Trim
 *
 * @details
 * This function Set DAC Offset Trim.
 * The offset value should be asigned number in the range of -128 to 127.
 *
 * @param[in] base is the DAC module base address
 * @param[in] offset is the DAC Offset Trim
 */
__STATIC_INLINE void
DAC_setOffsetTrim(uint32_t base, int16_t offset)
{
    //
    // Check the arguments.
    //
    ASSERT(DAC_isBaseValid(base));
    ASSERT((offset > -129) && (offset < 128));

    //
    // Set the offset trim value
    //
    EALLOW;

    HWREGH(base + DAC_O_TRIM) = (HWREGH(base + DAC_O_TRIM) &
                                 ~(0xFF)) | (int16_t)offset;

    EDIS;
}

/*
 * @brief Enable the DAC
 *
 * @details
 * This function enables the DAC.
 *
 * @param[in] base is the DAC module base address
 *
 */
__STATIC_INLINE void
DAC_enable(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(DAC_isBaseValid(base));

    /* Enable the output */
    EALLOW;

    HWREGH(base + DAC_O_EN) |= DAC_ENABLE;

    EDIS;
}

/*
 * @brief Disable the DAC
 *
 * @details
 * This function disables the DAC.
 *
 * @param[in] base is the DAC module base address
 *
 */
__STATIC_INLINE void
DAC_disable(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(DAC_isBaseValid(base));

    /* Disable the output */
    EALLOW;

    HWREGH(base + DAC_O_EN) &= ~DAC_ENABLE;

    EDIS;
}

/*
 * @brief Lock the DAC
 *
 * @details
 * This function lock the DACCTL,DACVALS and DACCOUTEN register
 *
 * @param[in] base is the DAC module base address
 *
 */
__STATIC_INLINE void
DAC_lockRegister(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(DAC_isBaseValid(base));

    /* lock the register */
    HWREG(base + DAC_O_LOCK) = (HWREGH(base + DAC_O_LOCK)) | 1;
}

/*
 * @brief Unlock the DAC
 *
 * @details
 * This function lock the DACCTL,DACVALS and DACCOUTEN register
 *
 * @param[in] base is the DAC module base address
 *
 */
__STATIC_INLINE void
DAC_unlockRegister(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(DAC_isBaseValid(base));

    /* lock the register */
    HWREG(base + DAC_O_LOCK) = DAC_REGS_LOCKVALUE;
}

/*
 * @brief Check whether the DAC is locked
 *
 * @details
 * This function  Check whether the DAC is locked
 *
 * @param[in] base is the DAC module base address
 *
 * @return Return true if the DAC locked and false otherwise
 *
 *
 */
__STATIC_INLINE bool
DAC_isRegisterLocked(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(DAC_isBaseValid(base));

    /* check the register */
    return (HWREG(base + DAC_O_LOCK) != DAC_REGS_LOCKVALUE);
}

/*
 * @brief Configure the Analog Clock Frequency
 *
 * @details
 * This function configures the DAC analog clock frequency.
 *
 * @param[in] base is the DAC module base address
 * @param[in] clkFreq is the selected clock frequency
 *
 * @remarks
 * - Obtained by the system clock frequency division and the actual frequency
 * 	 division coefficient is clkFreq * 2.
 *
 */
__STATIC_INLINE void
DAC_setAnaClkFreq(uint32_t base, uint16_t clkFreq)
{
    /* Check the arguments. */
    ASSERT(DAC_isBaseValid(base));

    /* Set the clock frequency */
    EALLOW;

    HWREG(base + DAC_O_ANACLKCFG) = (HWREG(base + DAC_O_ANACLKCFG) &
    								(~DAC_ANACLKCDG_ANACLK_DIV_M)) |
									(clkFreq & DAC_ANACLKCDG_ANACLK_DIV_M);

    EDIS;
}

/*
 * @brief Set the delay time between the generated ANACLK and the ANACLK
 * 		  seen by the analog part.
 *
 * @details
 * ANACLK and the ANACLK seen by the analog part, configure value is the actual
 * delay value.
 *
 * @param[in] base is the DAC module base address
 * @param[in] delay is the number of SYSCLK cycles delay between the generated
 *
 */
extern void
DAC_setPhaseDelay(uint32_t base, uint16_t delay);


/*
 * @brief Enable the DAC Functional safety
 *
 * @details
 * This function enables the DAC Functional safety.
 *
 * @param[in] base is the DAC module base address
 *
 */
__STATIC_INLINE void
DAC_Safety_enable(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(DAC_isBaseValid(base));

    /* Functional safety register */
    HWREG(base + DAC_O_SAFETY_EN) = (HWREGH(base + DAC_O_SAFETY_EN)) | 1;
}


/*
 * @brief Disable the DAC Functional safety
 *
 * @details
 * This function disables the DAC Functional safety.
 *
 * @param[in] base is the DAC module base address
 *
 */
__STATIC_INLINE void
DAC_Safety_disable(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(DAC_isBaseValid(base));

    /* Functional safety register */
    HWREG(base + DAC_O_SAFETY_EN) &= ~(1);
}
/*
 * Mark the end of the C bindings section for C++ compilers.
 */
#endif
#ifdef __cplusplus
}
#endif

#endif /* DEVICE_DRIVERLIB_CRC_H_ */
