/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef __cplusplus
extern "C"{
#endif


#include <stdio.h>
#include "gs32_version.h"
#include "dma.h"
#include "sysctl.h"

#if IS_GS32F00xx(0x12, 0x30)
GS32_DRIVER_DMA_FUNC_T void DMA_configAddresses(uint32_t base, uint32_t destAddr, uint32_t srcAddr)
{

    ASSERT(DMA_isBaseValid(base));
    
    /* Set up SOURCE address. */
    HWREG(base + DMA_O_SAR) = (uint32_t)srcAddr;
    
    /* Set up DESTINATION address. */
    HWREG(base + DMA_O_DAR) = (uint32_t)destAddr;
}

GS32_DRIVER_DMA_FUNC_T void DMA_configBurst(uint32_t base, uint16_t size, int16_t srcStep,
                     int16_t destStep)
{
    
    /* Check the arguments. */
    ASSERT(DMA_isBaseValid(base));
    ASSERT((size >= 1U) && (size <= 32U));

    /* Set up BURST registers. */
    HWREG(base + DMA_O_CTL) = (HWREG(base + DMA_O_CTL) & (~(DMA_CTL_DEST_MSIZE | DMA_CTL_SRC_MSIZE)) | (size << 11) | (size << 14));
}

GS32_DRIVER_DMA_FUNC_T void DMA_configTransfer(uint32_t base, uint32_t transferSize, int16_t srcStep,
                        int16_t destStep)
{
    uint32_t burst_size;

    ASSERT(DMA_isBaseValid(base));
    
    /* Set up TRANSFER registers. */
    burst_size =  ((HWREG(base + DMA_O_CTL) & DMA_CTL_SRC_MSIZE) >> 14);

    ASSERT((transferSize * burst_size) <= 0xFFFU);

    DMAREGH(base, DMA_O_CTL) = (DMAREGH(base, DMA_O_CTL) & (~(DMA_CTL_BLOCK_TS >> 32)) | (transferSize*burst_size));
}

GS32_DRIVER_DMA_FUNC_T void DMA_configWrap(uint32_t base, uint32_t srcWrapSize, int16_t srcStep,
                    uint32_t destWrapSize, int16_t destStep)
{
    ASSERT(FALSE);
}

GS32_DRIVER_DMA_FUNC_T void DMA_configMode(uint32_t base, DMAMUX_TrigId_Type trigger, uint32_t config)
{
    uint8_t data_size;
    uint32_t channel = 0U;
    uint32_t dmamuxBase = 0U;

    ASSERT(DMA_isBaseValid(base));

    channel = DMA_convertChnBase2ChnNum(base) ;
    dmamuxBase = DMA_convertChnbase2DmamuxBase(base);

    
    /* Set up trigger selection in the CMA/CLA trigger source selection
    registers. These are considered part of system control. */
    
    DMAMUX_setDmaMuxReqId(dmamuxBase, channel, trigger);

    /* CONTINUOUS MODE */
    if (config & DMA_CFG_CONTINUOUS_ENABLE)
        HWREG(base + DMA_O_CFG) |= (DMA_CFG_RELOAD_SRC | DMA_CFG_RELOAD_DST);
    else
        HWREG(base + DMA_O_CFG) &= (~(DMA_CFG_RELOAD_SRC | DMA_CFG_RELOAD_DST));

    /* DATA SIZE */
    if (config & DMA_CFG_SIZE_8BIT)
        data_size = 0;
    if (config & DMA_CFG_SIZE_16BIT)
        data_size = 1;
    if (config & DMA_CFG_SIZE_32BIT)
        data_size = 2;
    if (config & DMA_CFG_SIZE_64BIT)
        data_size = 3;
    if (config & DMA_CFG_SIZE_128BIT)
        data_size = 4;
    if (config & DMA_CFG_SIZE_256BIT)
        data_size = 5;
    HWREG(base + DMA_O_CTL) = (HWREG(base + DMA_O_CTL) & (~(DMA_CTL_SRC_TR_WIDTH | DMA_CTL_DST_TR_WIDTH)) | (data_size << 1) | (data_size << 4));
}

GS32_DRIVER_DMA_FUNC_T void DMA_configChannel(uint32_t base, const DMA_ConfigParams *transfParams)
{
	uint32_t high = 0;
	uint32_t low = 0;
    uint32_t dmamuxBase = 0U;
    uint32_t channel = 0U;
    DMA_HandshakingSelect srcHkSelect;
    DMA_HandshakingSelect destHkSelect;
    DMA_HK_HARDWARE_INF   srcHardInf; 
    DMA_HK_HARDWARE_INF   destHardInf;
    
    ASSERT(DMA_isBaseValid(base));
#if !(DMAMUX_RGCR_P2P_SUPPORT)
    
    /* Not support DMAMUX P2P trigger case,*/
    ASSERT((transfParams->dmaSrcReqId < DMAMUX_ReqId_max) ||
            (transfParams->ttfc == DMA_TT_FC_2_P2M_DMAC) ||
            (transfParams->ttfc == DMA_TT_FC_4_P2M_P) ||
            (transfParams->ttfc == DMA_TT_FC_1_M2P_DMAC) ||
            (transfParams->ttfc == DMA_TT_FC_5_P2P_SP));
#endif 

    
    /* Covert the channel base to the channel number and the dmamux base address.. */
    channel = DMA_convertChnBase2ChnNum(base);    
    dmamuxBase = DMA_convertChnbase2DmamuxBase(base);
    
    /* Forced distribution of the handshake signals by channel. */
    switch (transfParams->ttfc)
    {
        case DMA_TT_FC_2_P2M_DMAC:
        case DMA_TT_FC_4_P2M_P:
            srcHkSelect  = DMA_HKS_HARDWARE;
            destHkSelect = DMA_HKS_SOFTWARE;
            break;

        case DMA_TT_FC_1_M2P_DMAC:
        case DMA_TT_FC_6_M2P_P:
            srcHkSelect  = DMA_HKS_SOFTWARE;
            destHkSelect = DMA_HKS_HARDWARE;
            break;

        case DMA_TT_FC_3_P2P_DMAC:
        case DMA_TT_FC_5_P2P_SP:
        case DMA_TT_FC_7_P2P_DP:
            srcHkSelect  = DMA_HKS_HARDWARE;
            destHkSelect = DMA_HKS_HARDWARE;

            break;

        default:
            srcHkSelect  = DMA_HKS_SOFTWARE;
            destHkSelect = DMA_HKS_SOFTWARE;
            break;
    }


    
    /* DMA_HKS_HARD_INF_0 used as source handshaking signal of channel0
    DMA_HKS_HARD_INF_1 used as dest handshaking signal of channel0
    DMA_HKS_HARD_INF_2 used as source handshaking signal of channel1
    DMA_HKS_HARD_INF_3 used as dest handshaking signal of channel1
    ..............................................................
    ..............................................................
    DMA_HKS_HARD_INF_14 used as source handshaking signal of channel7
    DMA_HKS_HARD_INF_15 used as dest handshaking signal of channel7 */
    srcHardInf   = 2* channel;
    destHardInf  = 2* channel + 1U;


    
    /* Configure DMA Channel */
    DMA_configAddresses(base, transfParams->destAddr, transfParams->srcAddr);
/* DMA channel key configuration - channel CTRL register */

    /* Src Master */
	low += (DMA_AHB_MASTER_1<<25);
    /* Dest master */
	low += (DMA_AHB_MASTER_1<<23);
    /* TransType 0=m2m, 1=m2p, 2=p2m, 3=p2p, 4=p2m_fc_p, 5=p2p_fc_sp, ...... */
	low += (transfParams->ttfc<<20);
    /* Src Burst Size, 1/4/8/.../256cnt */
	low += (transfParams->srcBtl<<14);
    /* Dst Burst Size, 1/4/8/.../256cnt  */
	low += (transfParams->destBtl<<11);
    /* SrcInc, 0=increment src addr */
	low += (transfParams->srcAddrDirect<<9);
    /* DstInc, 2=unchanged dst addr */
	low += (transfParams->destAddrDirect<<7);
    /* Src_Tr_Width, 8/16/32/.../256bit, 2=32bit */
	low += (transfParams->srcTrWidthBytes<<4);
    /* Dst_Tr_Width, 8/16/32/.../256bit, 2=32bit */
	low += (transfParams->destTrWidthBytes<<1);
    /* Interrupt enable, it's used to mask all interrupts related to this channel */
	low += (transfParams->enableInterrupt<<0);

    /* SRC_GATHER_EN */
    if(transfParams->srcGatherInterval && transfParams->srcGatherCount)
    {
        DMA_configSrcGather(base, transfParams->srcGatherInterval, transfParams->srcGatherCount);
        low += BIT(17);
    }

    /* DST_SCATTER_EN */
    if(transfParams->destScatterInterval && transfParams->destScatterCount)
    {
        DMA_configDstScatter(base, transfParams->destScatterInterval, transfParams->destScatterCount);
        low += BIT(18);
    }
#if (DMAC_VERSION == 0x30)
    /* bit47, DONE */
	high = (1<<15);
    /* Bit32~Bit46, Block_Ts max is 32767  */
	high += (transfParams->blockTS & 0x00007fff);
#else
    /* bit44, DONE */
	high = (1<<12);
    /* Bit32~x, Block_Ts max is 4095  */
	high += (transfParams->blockTS & 0x00000fff);
#endif

    DMA_configChannelCTL(base, (uint64_t)high << 32 | low);

#if (DMAC_VERSION == 0x30)
    if (transfParams->enableBlkOfst)
    {
        low  = BIT(16);
        low += (transfParams->blockOffset & 0xFFFF);
        high = 0;
        DMA_configChannelBLKOFST(base, (uint64_t)high << 32 | low);
    }
    if (transfParams->enableBlkCnt)
    {
        low = BIT(16);
        low += (transfParams->blockCnt & 0xFFFF);
        high = 0;
        DMA_configChannelBLKCNT(base, (uint64_t)high << 32 | low);
    }
#endif

    low = (transfParams->chPriority<<5);
    /* Bit_11, HS_SEL_DST select handshaking source */
	low += (srcHkSelect<<11);
    /* Bit_10, HS_SEL_SRC select handshaking source */
	low += (destHkSelect<<10);
    /* Bit_30, RELOAD_SRC Automatic Source Reload */
	low += (transfParams->reloadSrc<<30);
    /* Bit_31, RELOAD_DST Automatic Destination Reload */
	low += (transfParams->reloadDst<<31);
    /* Bit_39, SRC_PER select handshaking source */
	high = (srcHardInf<<7);
    /* Bit_43, DST_PER select handshaking source */
	high += (destHardInf<<11);
	DMA_configChannelCFG(base, (uint64_t)high << 32 | low);

	if (transfParams->ttfc == DMA_TT_FC_2_P2M_DMAC || transfParams->ttfc == DMA_TT_FC_3_P2P_DMAC ||
        transfParams->ttfc == DMA_TT_FC_4_P2M_P || transfParams->ttfc == DMA_TT_FC_5_P2P_SP || transfParams->ttfc == DMA_TT_FC_7_P2P_DP)
		DMAMUX_configDmaMux(dmamuxBase, srcHardInf, transfParams->dmaSrcReqId);
	if (transfParams->ttfc == DMA_TT_FC_1_M2P_DMAC || transfParams->ttfc == DMA_TT_FC_3_P2P_DMAC ||
        transfParams->ttfc == DMA_TT_FC_5_P2P_SP || transfParams->ttfc == DMA_TT_FC_6_M2P_P || transfParams->ttfc == DMA_TT_FC_7_P2P_DP)
		DMAMUX_configDmaMux(dmamuxBase, destHardInf, transfParams->dmaDstReqId);
}


GS32_DRIVER_DMA_FUNC_T void DMA_DeConfChannel(uint32_t base, const DMA_ConfigParams *transfParams)
{
    uint32_t dmamuxBase = 0U;
    uint32_t channel = 0U;
    DMA_HK_HARDWARE_INF   srcHardInf; 
    DMA_HK_HARDWARE_INF   destHardInf;

    ASSERT(DMA_isBaseValid(base));

    channel = DMA_convertChnBase2ChnNum(base);    
    dmamuxBase = DMA_convertChnbase2DmamuxBase(base);
    
    /* DMA_HKS_HARD_INF_0 used as source handshaking signal of channel0
    DMA_HKS_HARD_INF_1 used as dest handshaking signal of channel0
    DMA_HKS_HARD_INF_2 used as source handshaking signal of channel1
    DMA_HKS_HARD_INF_3 used as dest handshaking signal of channel1
    ..............................................................
    ..............................................................
    DMA_HKS_HARD_INF_14 used as source handshaking signal of channel7
    DMA_HKS_HARD_INF_15 used as dest handshaking signal of channel7 */

    srcHardInf   = 2* channel;
    destHardInf  = 2* channel + 1U;


    if (transfParams->ttfc == DMA_TT_FC_2_P2M_DMAC || transfParams->ttfc == DMA_TT_FC_3_P2P_DMAC ||
        transfParams->ttfc == DMA_TT_FC_4_P2M_P || transfParams->ttfc == DMA_TT_FC_5_P2P_SP || transfParams->ttfc == DMA_TT_FC_7_P2P_DP)
        DMAMUX_deConfDmaMux(dmamuxBase, srcHardInf, transfParams->dmaSrcReqId);
    if (transfParams->ttfc == DMA_TT_FC_1_M2P_DMAC || transfParams->ttfc == DMA_TT_FC_3_P2P_DMAC ||
        transfParams->ttfc == DMA_TT_FC_5_P2P_SP || transfParams->ttfc == DMA_TT_FC_6_M2P_P || transfParams->ttfc == DMA_TT_FC_7_P2P_DP)
        DMAMUX_deConfDmaMux(dmamuxBase, destHardInf, transfParams->dmaDstReqId);
}

GS32_DRIVER_DMA_FUNC_T void DmaChan_config(uint32_t base, const DmaCh_Parameters *chParams)
{
    uint32_t ttfc = chParams->ctl.ctl_l.bit.TT_FC;
    uint32_t srcHkSelect = chParams->cfg.cfg_l.bit.HS_SEL_SRC;
    uint32_t destHkSelect = chParams->cfg.cfg_l.bit.HS_SEL_DST;
    uint32_t srcHardInf = chParams->cfg.cfg_h.bit.SRC_PER;
    uint32_t destHardInf = chParams->cfg.cfg_h.bit.DEST_PER;
    uint32_t srcReqId = chParams->srcCcr.bit.dmareq_id;
    uint32_t destReqId = chParams->destCcr.bit.dmareq_id;
    uint32_t dmamuxBase = 0U;

    ASSERT(DMA_isBaseValid(base));

    /* Covert the channel base to the dmamux base address. */
    dmamuxBase = DMA_convertChnbase2DmamuxBase(base);

    /* config ctl register */
    DMA_Ch_setCtl(base, (DmaCh_Ctl *)&(chParams->ctl));

    /* config cfg register */
    DMA_Ch_setCfg(base, (DmaCh_Cfg *)&(chParams->cfg));

    if (srcHkSelect == DMA_HKS_HARDWARE)
    {
        /* source is Peripheral */
        if (ttfc == DMA_TT_FC_2_P2M_DMAC || ttfc == DMA_TT_FC_3_P2P_DMAC || ttfc == DMA_TT_FC_4_P2M_P ||
            ttfc == DMA_TT_FC_5_P2P_SP || ttfc == DMA_TT_FC_7_P2P_DP)
        {
            /* config dmamux ccr */
            DMAMUX_setDmaMuxCcr(dmamuxBase, srcHardInf, chParams->srcCcr);

            /* config dmamux rgcr */
            if (srcReqId >= 0 && srcReqId <= 7)
                DMAMUX_setDmaMuxRgcr(dmamuxBase, srcReqId, chParams->srcRgcr);
        }
    }

    if (destHkSelect == DMA_HKS_HARDWARE)
    {
        /* dest is Peripheral */
        if (ttfc == DMA_TT_FC_1_M2P_DMAC || ttfc == DMA_TT_FC_3_P2P_DMAC || ttfc == DMA_TT_FC_5_P2P_SP ||
            ttfc == DMA_TT_FC_6_M2P_P || ttfc == DMA_TT_FC_7_P2P_DP)
        {
            /* config dmamux ccr */
            DMAMUX_setDmaMuxCcr(dmamuxBase, destHardInf, chParams->destCcr);

            /* config dmamux rgcr */
            if (destReqId >= 0 && destReqId <= 7)
                DMAMUX_setDmaMuxRgcr(dmamuxBase, destReqId, chParams->destRgcr);
        }
    }
}

#ifdef __cplusplus
}
#endif

#endif
