/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    dmamux.h
*   @brief   
*
*/

#ifndef _DMAMUX_H_
#define _DMAMUX_H_

#ifdef __cplusplus
extern "C"
{
#endif

#include "inc/hw_types.h"
#include "inc/hw_memmap.h"
#include "inc/hw_dmamux.h"
#include "regs/regs_dmamux.h"
#include "debug.h"
#include "stdbool.h"
#include "sysctl.h"

#if (DMAMUX_VERSION == 0x11)

#define  DMAMUX_REG_OFFSET      (0x0450U)
#define  DMAMUX_CCR_OFFSET      (0x000U)
#define  DMAMUX_CCR_NUMS        (16U)
#define  DMAMUX_RGCR_OFFSET     (DMAMUX_CCR_OFFSET + DMAMUX_CCRx(DMAMUX_CCR_NUMS) + 0x010U)
#define  DMAMUX_RGCR_NUMS       (8U)
#define  DMAMUX_INT_OFFSET      (DMAMUX_RGCR_OFFSET + DMAMUX_RGCRx(DMAMUX_RGCR_NUMS))
//*****************************************************************************
//!
//! Only one shared DMAMUX module accessible by both ARM and RISCV.
//!
//*****************************************************************************
#define  DMA_MUX_BASE       (SYSCTL_DIG_BASE + DMAMUX_REG_OFFSET)

#elif (DMAMUX_VERSION == 0x20)

#define  DMAMUX_ARM_REG_OFFSET  (0x0800U)
#define  DMAMUX_DSP_REG_OFFSET  (0x0400U)
#define  DMAMUX_CCR_OFFSET      (0x000U)
#define  DMAMUX_CCR_NUMS        (64U)
#define  DMAMUX_RGCR_OFFSET     (DMAMUX_CCR_OFFSET + DMAMUX_CCRx(DMAMUX_CCR_NUMS))
#define  DMAMUX_RGCR_NUMS       (8U)
#define  DMAMUX_INT_OFFSET      (DMAMUX_RGCR_OFFSET + DMAMUX_RGCRx(DMAMUX_RGCR_NUMS))
//*******************************************************************************
/**
 * @brief Each ARM/RISCV domain on the chip has an independent DMA module, and ARM/RISCV 
 * can access each other's DMA modules. The DMA module within the ARM domain is named ARM_DMA 
 * in the header file,  the two RISCV domains can access each other's DSP_DMA modules. However, 
 * the DMAMUX module is absolutely bound to the DMA module, and they cannot be mixed. Please 
 * follow the rules below to use the corresponding DMAMUX module.
 * ARM domain 's DMA Module:
 * ARM_DMA_BASE        use ------------>ARM_DMA_MUX_BASE
 * ARM_DMA_CH0_BASE    use ------------>ARM_DMA_MUX_BASE
 * ARM_DMA_CH1_BASE    use ------------>ARM_DMA_MUX_BASE
 * ARM_DMA_CH2_BASE    use ------------>ARM_DMA_MUX_BASE
 * ARM_DMA_CH3_BASE    use ------------>ARM_DMA_MUX_BASE
 * ARM_DMA_CH4_BASE    use ------------>ARM_DMA_MUX_BASE
 * ARM_DMA_CH5_BASE    use ------------>ARM_DMA_MUX_BASE
 * ARM_DMA_CH6_BASE    use ------------>ARM_DMA_MUX_BASE
 * ARM_DMA_CH7_BASE    use------------->ARM-DMA_MUX_BASE
 * 
 * DSP domain 's DMA Module:
 * DSP_DMA_BASE        use------------>DSP_DMA_MUX_BASE
 * DSP_DMA_CH0_BASE    use------------>DSP_DMA_MUX_BASE
 * DSP_DMA_CH1_BASE    use------------>DSP_DMA_MUX_BASE
 * DSP_DMA_CH2_BASE    use------------>DSP_DMA_MUX_BASE
 * DSP_DMA_CH3_BASE    use------------>DSP_DMA_MUX_BASE
 * DSP_DMA_CH4_BASE    use------------>DSP_DMA_MUX_BASE
 * DSP_DMA_CH5_BASE    use------------>DSP_DMA_MUX_BASE
 * DSP_DMA_CH6_BASE    use------------>DSP_DMA_MUX_BASE
 * DSP_DMA_CH7_BASE    use------------>DSP_DMA_MUX_BASE
 */
#define  ARM_DMA_MUX_BASE       (SYSCTL_ARM_BASE + DMAMUX_ARM_REG_OFFSET)
#define  DSP_DMA_MUX_BASE       (SYSCTL_DSP_BASE + DMAMUX_DSP_REG_OFFSET)

#else /* #if (DMAMUX_VERSION == 0x12 || DMAMUX_VERSION == 0x21 || DMAMUX_VERSION == 0x22 || DMAMUX_VERSION == 0X23) */

#define  DMAMUX_CCR_OFFSET      (0x000U)                                               /* offset 0x00 */
#define  DMAMUX_CCR_NUMS        (16U)
#define  DMAMUX_RGCR_OFFSET     (DMAMUX_CCR_OFFSET + DMAMUX_CCRx(DMAMUX_CCR_NUMS))      /* offset 0x40 */
#define  DMAMUX_RGCR_NUMS       (8U)
#define  DMAMUX_INT_OFFSET      (DMAMUX_RGCR_OFFSET + DMAMUX_RGCRx(DMAMUX_RGCR_NUMS) + 0x010U)  /* offset 0x70 */

#endif  /* #if (DMAMUX_VERSION == 0x11) */

/**
 * @brief The following are values that can be passed to DMAMUX_setDmamuxCcr()
 * and DMAMUX_setDmaReqId() as the DMA request number. These trigger ID and Request 
 * ID are not same in different devices.
 */
#if (DMAMUX_VERSION == 0x11)
typedef enum
{
    DMAMUX_ReqId_dma_gen_out0 = 0,
    DMAMUX_ReqId_dma_gen_out1 = 1,
    DMAMUX_ReqId_dma_gen_out2 = 2,
    DMAMUX_ReqId_dma_gen_out3 = 3,
    DMAMUX_ReqId_dma_gen_out4 = 4,
    DMAMUX_ReqId_dma_gen_out5 = 5,
    DMAMUX_ReqId_dma_gen_out6 = 6,
    DMAMUX_ReqId_dma_gen_out7 = 7,
    DMAMUX_ReqId_ecap1 = 8,
    DMAMUX_ReqId_ecap2 = 9,
    DMAMUX_ReqId_ecap3 = 10,
    DMAMUX_ReqId_ecap4 = 11,
    DMAMUX_ReqId_ecap5 = 12,
    DMAMUX_ReqId_ecap6 = 13,
    DMAMUX_ReqId_ecap7 = 14,
    DMAMUX_ReqId_qspi_rx = 16,
    DMAMUX_ReqId_qspi_tx = 17,
    DMAMUX_ReqId_spi0_rx = 19,
    DMAMUX_ReqId_spi0_tx = 20,
    DMAMUX_ReqId_spi1_rx = 21,
    DMAMUX_ReqId_spi1_tx = 22,
    DMAMUX_ReqId_lc1 = 33,
    DMAMUX_ReqId_lc2 = 34,
    DMAMUX_ReqId_lc3 = 35,
    DMAMUX_ReqId_plu1_lc1 = DMAMUX_ReqId_lc1,
    DMAMUX_ReqId_plu1_lc2 = DMAMUX_ReqId_lc2,
    DMAMUX_ReqId_plu1_lc3 = DMAMUX_ReqId_lc3,
    DMAMUX_ReqId_plu2_lc1 = 36,
    DMAMUX_ReqId_plu2_lc2 = 37,
    DMAMUX_ReqId_plu2_lc3 = 38,    
    DMAMUX_ReqId_i2ca_rx = 46,
    DMAMUX_ReqId_i2ca_tx = 47,
    DMAMUX_ReqId_i2cb_rx = 48,
    DMAMUX_ReqId_i2cb_tx = 49,
    DMAMUX_ReqId_i2cc_rx = 50,
    DMAMUX_ReqId_i2cc_tx = 51,
    DMAMUX_ReqId_scia_rx = 53,
    DMAMUX_ReqId_scia_tx = 54,
    DMAMUX_ReqId_scib_rx = 55,
    DMAMUX_ReqId_scib_tx = 56,
    DMAMUX_ReqId_scic_rx = 57,
    DMAMUX_ReqId_scic_tx = 58,
    DMAMUX_ReqId_scid_rx = 59,
    DMAMUX_ReqId_scid_tx = 60,
    DMAMUX_ReqId_max = 64,/* max req id, do not use */

    /* trigID */
    DMAMUX_ReqId_Trig_ADCA_1int = DMAMUX_ReqId_max + 0, /* ADCA_1int */
    DMAMUX_ReqId_Trig_ADCA_2int = DMAMUX_ReqId_max + 1, /* ADCA_2int */
    DMAMUX_ReqId_Trig_ADCA_3int = DMAMUX_ReqId_max + 2, /* ADCA_3int */
    DMAMUX_ReqId_Trig_ADCA_4int = DMAMUX_ReqId_max + 3,
    DMAMUX_ReqId_Trig_ADCA_evt_int = DMAMUX_ReqId_max + 4,
    DMAMUX_ReqId_Trig_ADCB_1int = DMAMUX_ReqId_max + 5,
    DMAMUX_ReqId_Trig_ADCB_2int = DMAMUX_ReqId_max + 6,
    DMAMUX_ReqId_Trig_ADCB_3int = DMAMUX_ReqId_max + 7,
    DMAMUX_ReqId_Trig_ADCB_4int = DMAMUX_ReqId_max + 8,
    DMAMUX_ReqId_Trig_ADCB_evt_int = DMAMUX_ReqId_max + 9,
    DMAMUX_ReqId_Trig_ADCC_1int = DMAMUX_ReqId_max + 10,
    DMAMUX_ReqId_Trig_ADCC_2int = DMAMUX_ReqId_max + 11,
    DMAMUX_ReqId_Trig_ADCC_3int = DMAMUX_ReqId_max + 12,
    DMAMUX_ReqId_Trig_ADCC_4int = DMAMUX_ReqId_max + 13,
    DMAMUX_ReqId_Trig_ADCC_evt_int = DMAMUX_ReqId_max + 14,
    DMAMUX_ReqId_Trig_cpu_1xint = DMAMUX_ReqId_max + 21,
    DMAMUX_ReqId_Trig_cpu_2xint = DMAMUX_ReqId_max + 22,
    DMAMUX_ReqId_Trig_cpu_3xint = DMAMUX_ReqId_max + 23,
    DMAMUX_ReqId_Trig_cpu_4xint = DMAMUX_ReqId_max + 24,
    DMAMUX_ReqId_Trig_cpu_5xint = DMAMUX_ReqId_max + 25,
    DMAMUX_ReqId_Trig_cpu_6xint = DMAMUX_ReqId_max + 26,
    DMAMUX_ReqId_Trig_cpu_7xint = DMAMUX_ReqId_max + 27,
    DMAMUX_ReqId_Trig_cpu_8xint = DMAMUX_ReqId_max + 28,
    DMAMUX_ReqId_Trig_cpu_9xint = DMAMUX_ReqId_max + 29,
    DMAMUX_ReqId_Trig_cpu_10xint = DMAMUX_ReqId_max + 30,
    DMAMUX_ReqId_Trig_cpu_11xint = DMAMUX_ReqId_max + 31,
    DMAMUX_ReqId_Trig_cpu_12xint = DMAMUX_ReqId_max + 32,
    DMAMUX_ReqId_Trig_cpu_13xint = DMAMUX_ReqId_max + 33,
    DMAMUX_ReqId_Trig_cpu_14xint = DMAMUX_ReqId_max + 34,
    DMAMUX_ReqId_Trig_cpu_15xint = DMAMUX_ReqId_max + 35,
    DMAMUX_ReqId_Trig_cpu_16xint = DMAMUX_ReqId_max + 36,
    DMAMUX_ReqId_Trig_EPWM1xSOCA = DMAMUX_ReqId_max + 37,
    DMAMUX_ReqId_Trig_EPWM1xSOCB = DMAMUX_ReqId_max + 38,
    DMAMUX_ReqId_Trig_EPWM2xSOCA = DMAMUX_ReqId_max + 39,
    DMAMUX_ReqId_Trig_EPWM2xSOCB = DMAMUX_ReqId_max + 40,
    DMAMUX_ReqId_Trig_EPWM3xSOCA = DMAMUX_ReqId_max + 41,
    DMAMUX_ReqId_Trig_EPWM3xSOCB = DMAMUX_ReqId_max + 42,
    DMAMUX_ReqId_Trig_EPWM4xSOCA = DMAMUX_ReqId_max + 43,
    DMAMUX_ReqId_Trig_EPWM4xSOCB = DMAMUX_ReqId_max + 44,
    DMAMUX_ReqId_Trig_EPWM5xSOCA = DMAMUX_ReqId_max + 45,
    DMAMUX_ReqId_Trig_EPWM5xSOCB = DMAMUX_ReqId_max + 46,
    DMAMUX_ReqId_Trig_EPWM6xSOCA = DMAMUX_ReqId_max + 47,
    DMAMUX_ReqId_Trig_EPWM6xSOCB = DMAMUX_ReqId_max + 48,
    DMAMUX_ReqId_Trig_EPWM7xSOCA = DMAMUX_ReqId_max + 49,
    DMAMUX_ReqId_Trig_EPWM7xSOCB = DMAMUX_ReqId_max + 50,
    DMAMUX_ReqId_Trig_EPWM8xSOCA = DMAMUX_ReqId_max + 51,
    DMAMUX_ReqId_Trig_EPWM8xSOCB = DMAMUX_ReqId_max + 52,
    DMAMUX_ReqId_Trig_EPWM9xSOCA = DMAMUX_ReqId_max + 53,
    DMAMUX_ReqId_Trig_EPWM9xSOCB = DMAMUX_ReqId_max + 54,
    DMAMUX_ReqId_Trig_EPWM10xSOCA = DMAMUX_ReqId_max + 55,
    DMAMUX_ReqId_Trig_EPWM10xSOCB = DMAMUX_ReqId_max + 56,
    DMAMUX_ReqId_Trig_EPWM11xSOCA = DMAMUX_ReqId_max + 57,
    DMAMUX_ReqId_Trig_EPWM11xSOCB = DMAMUX_ReqId_max + 58,
    DMAMUX_ReqId_Trig_EPWM12xSOCA = DMAMUX_ReqId_max + 59,
    DMAMUX_ReqId_Trig_EPWM12xSOCB = DMAMUX_ReqId_max + 60,
    DMAMUX_ReqId_Trig_ap_timer0 = DMAMUX_ReqId_max + 69,
    DMAMUX_ReqId_Trig_ap_timer1 = DMAMUX_ReqId_max + 70,
    DMAMUX_ReqId_Trig_ap_timer2 = DMAMUX_ReqId_max + 71,
    DMAMUX_ReqId_Trig_ap_timer3 = DMAMUX_ReqId_max + 72,
    DMAMUX_ReqId_Trig_can0_if2 = DMAMUX_ReqId_max + 73,
    DMAMUX_ReqId_Trig_can0_if3 = DMAMUX_ReqId_max + 74,
    DMAMUX_ReqId_Trig_can0_if4 = DMAMUX_ReqId_max + 75,
    DMAMUX_ReqId_Trig_can0_if5 = DMAMUX_ReqId_max + 76,
    DMAMUX_ReqId_Trig_can1_if2 = DMAMUX_ReqId_max + 77,
    DMAMUX_ReqId_Trig_can1_if3 = DMAMUX_ReqId_max + 78,
    DMAMUX_ReqId_Trig_can1_if4 = DMAMUX_ReqId_max + 79,
    DMAMUX_ReqId_Trig_can1_if5 = DMAMUX_ReqId_max + 80,
    DMAMUX_ReqId_Trig_dmamux_evt0 = DMAMUX_ReqId_max + 81,
    DMAMUX_ReqId_Trig_dmamux_evt1 = DMAMUX_ReqId_max + 82,
    DMAMUX_ReqId_Trig_dmamux_evt2 = DMAMUX_ReqId_max + 83,
    DMAMUX_ReqId_Trig_dmamux_evt3 = DMAMUX_ReqId_max + 84,
    DMAMUX_ReqId_Trig_dmamux_evt4 = DMAMUX_ReqId_max + 85,
    DMAMUX_ReqId_Trig_dmamux_evt5 = DMAMUX_ReqId_max + 86,
    DMAMUX_ReqId_Trig_dmamux_evt6 = DMAMUX_ReqId_max + 87,
    DMAMUX_ReqId_Trig_dmamux_evt7 = DMAMUX_ReqId_max + 88,
} DMAMUX_ReqId_Type;


typedef enum
{
    DMAMUX_TrigId_ADCA_1int = 0,
    DMAMUX_TrigId_ADCA_2int = 1,
    DMAMUX_TrigId_ADCA_3int = 2,
    DMAMUX_TrigId_ADCA_4int = 3,
    DMAMUX_TrigId_ADCA_evt_int = 4,
    DMAMUX_TrigId_ADCB_1int = 5,
    DMAMUX_TrigId_ADCB_2int = 6,
    DMAMUX_TrigId_ADCB_3int = 7,
    DMAMUX_TrigId_ADCB_4int = 8,
    DMAMUX_TrigId_ADCB_evt_int = 9,
    DMAMUX_TrigId_ADCC_1int = 10,
    DMAMUX_TrigId_ADCC_2int = 11,
    DMAMUX_TrigId_ADCC_3int = 12,
    DMAMUX_TrigId_ADCC_4int = 13,
    DMAMUX_TrigId_ADCC_evt_int = 14,
    DMAMUX_TrigId_cpu_1xint = 21,
    DMAMUX_TrigId_cpu_2xint = 22,
    DMAMUX_TrigId_cpu_3xint = 23,
    DMAMUX_TrigId_cpu_4xint = 24,
    DMAMUX_TrigId_cpu_5xint = 25,
    DMAMUX_TrigId_cpu_6xint = 26,
    DMAMUX_TrigId_cpu_7xint = 27,
    DMAMUX_TrigId_cpu_8xint = 28,
    DMAMUX_TrigId_cpu_9xint = 29,
    DMAMUX_TrigId_cpu_10xint = 30,
    DMAMUX_TrigId_cpu_11xint = 31,
    DMAMUX_TrigId_cpu_12xint = 32,
    DMAMUX_TrigId_cpu_13xint = 33,
    DMAMUX_TrigId_cpu_14xint = 34,
    DMAMUX_TrigId_cpu_15xint = 35,
    DMAMUX_TrigId_cpu_16xint = 36,
    DMAMUX_TrigId_EPWM1xSOCA = 37,
    DMAMUX_TrigId_EPWM1xSOCB = 38,
    DMAMUX_TrigId_EPWM2xSOCA = 39,
    DMAMUX_TrigId_EPWM2xSOCB = 40,
    DMAMUX_TrigId_EPWM3xSOCA = 41,
    DMAMUX_TrigId_EPWM3xSOCB = 42,
    DMAMUX_TrigId_EPWM4xSOCA = 43,
    DMAMUX_TrigId_EPWM4xSOCB = 44,
    DMAMUX_TrigId_EPWM5xSOCA = 45,
    DMAMUX_TrigId_EPWM5xSOCB = 46,
    DMAMUX_TrigId_EPWM6xSOCA = 47,
    DMAMUX_TrigId_EPWM6xSOCB = 48,
    DMAMUX_TrigId_EPWM7xSOCA = 49,
    DMAMUX_TrigId_EPWM7xSOCB = 50,
    DMAMUX_TrigId_EPWM8xSOCA = 51,
    DMAMUX_TrigId_EPWM8xSOCB = 52,
    DMAMUX_TrigId_EPWM9xSOCA = 53,
    DMAMUX_TrigId_EPWM9xSOCB = 54,
    DMAMUX_TrigId_EPWM10xSOCA = 55,
    DMAMUX_TrigId_EPWM10xSOCB = 56,
    DMAMUX_TrigId_EPWM11xSOCA = 57,
    DMAMUX_TrigId_EPWM11xSOCB = 58,
    DMAMUX_TrigId_EPWM12xSOCA = 59,
    DMAMUX_TrigId_EPWM12xSOCB = 60,
    DMAMUX_TrigId_ap_timer0 = 69,
    DMAMUX_TrigId_ap_timer1 = 70,
    DMAMUX_TrigId_ap_timer2 = 71,
    DMAMUX_TrigId_ap_timer3 = 72,
    DMAMUX_TrigId_can0_if2 = 73,
    DMAMUX_TrigId_can0_if3 = 74,
    DMAMUX_TrigId_can0_if4 = 75,
    DMAMUX_TrigId_can0_if5 = 76,
    DMAMUX_TrigId_can1_if2 = 77,
    DMAMUX_TrigId_can1_if3 = 78,
    DMAMUX_TrigId_can1_if4 = 79,
    DMAMUX_TrigId_can1_if5 = 80,
    DMAMUX_TrigId_dmamux_evt0 = 81,
    DMAMUX_TrigId_dmamux_evt1 = 82,
    DMAMUX_TrigId_dmamux_evt2 = 83,
    DMAMUX_TrigId_dmamux_evt3 = 84,
    DMAMUX_TrigId_dmamux_evt4 = 85,
    DMAMUX_TrigId_dmamux_evt5 = 86,
    DMAMUX_TrigId_dmamux_evt6 = 87,
    DMAMUX_TrigId_dmamux_evt7 = 88,
} DMAMUX_TrigId_Type;

#elif (DMAMUX_VERSION == 0x12)
typedef enum
{
    DMAMUX_ReqId_dma_gen_out0 = 1,
    DMAMUX_ReqId_dma_gen_out1 = 2,
    DMAMUX_ReqId_dma_gen_out2 = 3,
    DMAMUX_ReqId_dma_gen_out3 = 4,
    DMAMUX_ReqId_dma_gen_out4 = 5,
    DMAMUX_ReqId_dma_gen_out5 = 6,
    DMAMUX_ReqId_dma_gen_out6 = 7,
    DMAMUX_ReqId_dma_gen_out7 = 8,

    DMAMUX_ReqId_qspi_rx = 10,
    DMAMUX_ReqId_qspi_tx = 11,
    
    DMAMUX_ReqId_spi0_rx = 13,
    DMAMUX_ReqId_spi0_tx = 14,
    DMAMUX_ReqId_spi1_rx = 15,
    DMAMUX_ReqId_spi1_tx = 16,

    DMAMUX_ReqId_plu1_lc1 = 18,
    DMAMUX_ReqId_plu1_lc2 = 19,
    DMAMUX_ReqId_plu1_lc3 = 20,
    DMAMUX_ReqId_plu2_lc1 = 21,
    DMAMUX_ReqId_plu2_lc2 = 22,
    DMAMUX_ReqId_plu2_lc3 = 23,
    DMAMUX_ReqId_plu3_lc1 = 24,
    DMAMUX_ReqId_plu3_lc2 = 25,
    DMAMUX_ReqId_plu3_lc3 = 26,
    DMAMUX_ReqId_plu4_lc1 = 27,
    DMAMUX_ReqId_plu4_lc2 = 28,
    DMAMUX_ReqId_plu4_lc3 = 29,

    DMAMUX_ReqId_i2ca_rx = 33,
    DMAMUX_ReqId_i2ca_tx = 34,
    DMAMUX_ReqId_i2cb_rx = 35,
    DMAMUX_ReqId_i2cb_tx = 36,

    DMAMUX_ReqId_pmbus_rx = 38,
    DMAMUX_ReqId_pmbus_tx = 39,

    DMAMUX_ReqId_scia_rx = 41,
    DMAMUX_ReqId_scia_tx = 42,
    DMAMUX_ReqId_scib_rx = 43,
    DMAMUX_ReqId_scib_tx = 44,

    DMAMUX_ReqId_lin1_rx = 49,
    DMAMUX_ReqId_lin1_tx = 50,
    DMAMUX_ReqId_lin2_rx = 51,
    DMAMUX_ReqId_lin2_tx = 52,

    DMAMUX_ReqId_sd1_flt1 = 54,
    DMAMUX_ReqId_sd1_flt2 = 55,
    DMAMUX_ReqId_sd1_flt3 = 56,
    DMAMUX_ReqId_sd1_flt4 = 57,
    DMAMUX_ReqId_sd2_flt1 = 58,
    DMAMUX_ReqId_sd2_flt2 = 59,
    DMAMUX_ReqId_sd2_flt3 = 60,
    DMAMUX_ReqId_sd2_flt4 = 61,

    DMAMUX_ReqId_max = 64,

    DMAMUX_ReqId_Trig_ADCA_1int = DMAMUX_ReqId_max + 1,
    DMAMUX_ReqId_Trig_ADCA_2int = DMAMUX_ReqId_max + 2,
    DMAMUX_ReqId_Trig_ADCA_3int = DMAMUX_ReqId_max + 3,
    DMAMUX_ReqId_Trig_ADCA_4int = DMAMUX_ReqId_max + 4,
    DMAMUX_ReqId_Trig_ADCA_evt_int = DMAMUX_ReqId_max + 5,
    DMAMUX_ReqId_Trig_ADCB_1int = DMAMUX_ReqId_max + 6,
    DMAMUX_ReqId_Trig_ADCB_2int = DMAMUX_ReqId_max + 7,
    DMAMUX_ReqId_Trig_ADCB_3int = DMAMUX_ReqId_max + 8,
    DMAMUX_ReqId_Trig_ADCB_4int = DMAMUX_ReqId_max + 9,
    DMAMUX_ReqId_Trig_ADCB_evt_int = DMAMUX_ReqId_max + 10,
    DMAMUX_ReqId_Trig_ADCC_1int = DMAMUX_ReqId_max + 11,
    DMAMUX_ReqId_Trig_ADCC_2int = DMAMUX_ReqId_max + 12,
    DMAMUX_ReqId_Trig_ADCC_3int = DMAMUX_ReqId_max + 13,
    DMAMUX_ReqId_Trig_ADCC_4int = DMAMUX_ReqId_max + 14,
    DMAMUX_ReqId_Trig_ADCC_evt_int = DMAMUX_ReqId_max + 15,
    
    DMAMUX_ReqId_Trig_cpu_1xint = DMAMUX_ReqId_max + 17,
    DMAMUX_ReqId_Trig_cpu_2xint = DMAMUX_ReqId_max + 18,
    DMAMUX_ReqId_Trig_cpu_3xint = DMAMUX_ReqId_max + 19,
    DMAMUX_ReqId_Trig_cpu_4xint = DMAMUX_ReqId_max + 20,
    DMAMUX_ReqId_Trig_cpu_5xint = DMAMUX_ReqId_max + 21,
    DMAMUX_ReqId_Trig_cpu_6xint = DMAMUX_ReqId_max + 22,
    DMAMUX_ReqId_Trig_cpu_7xint = DMAMUX_ReqId_max + 23,
    DMAMUX_ReqId_Trig_cpu_8xint = DMAMUX_ReqId_max + 24,
    DMAMUX_ReqId_Trig_cpu_9xint = DMAMUX_ReqId_max + 25,
    DMAMUX_ReqId_Trig_cpu_10xint = DMAMUX_ReqId_max + 26,
    DMAMUX_ReqId_Trig_cpu_11xint = DMAMUX_ReqId_max + 27,
    DMAMUX_ReqId_Trig_cpu_12xint = DMAMUX_ReqId_max + 28,
    DMAMUX_ReqId_Trig_cpu_13xint = DMAMUX_ReqId_max + 29,
    DMAMUX_ReqId_Trig_cpu_14xint = DMAMUX_ReqId_max + 30,
    DMAMUX_ReqId_Trig_cpu_15xint = DMAMUX_ReqId_max + 31,
    DMAMUX_ReqId_Trig_cpu_16xint = DMAMUX_ReqId_max + 32,
    
    DMAMUX_ReqId_Trig_EPWM1xSOCA = DMAMUX_ReqId_max + 34,
    DMAMUX_ReqId_Trig_EPWM1xSOCB = DMAMUX_ReqId_max + 35,
    DMAMUX_ReqId_Trig_EPWM2xSOCA = DMAMUX_ReqId_max + 36,
    DMAMUX_ReqId_Trig_EPWM2xSOCB = DMAMUX_ReqId_max + 37,
    DMAMUX_ReqId_Trig_EPWM3xSOCA = DMAMUX_ReqId_max + 38,
    DMAMUX_ReqId_Trig_EPWM3xSOCB = DMAMUX_ReqId_max + 39,
    DMAMUX_ReqId_Trig_EPWM4xSOCA = DMAMUX_ReqId_max + 40,
    DMAMUX_ReqId_Trig_EPWM4xSOCB = DMAMUX_ReqId_max + 41,
    DMAMUX_ReqId_Trig_EPWM5xSOCA = DMAMUX_ReqId_max + 42,
    DMAMUX_ReqId_Trig_EPWM5xSOCB = DMAMUX_ReqId_max + 43,
    DMAMUX_ReqId_Trig_EPWM6xSOCA = DMAMUX_ReqId_max + 44,
    DMAMUX_ReqId_Trig_EPWM6xSOCB = DMAMUX_ReqId_max + 45,
    DMAMUX_ReqId_Trig_EPWM7xSOCA = DMAMUX_ReqId_max + 46,
    DMAMUX_ReqId_Trig_EPWM7xSOCB = DMAMUX_ReqId_max + 47,
    DMAMUX_ReqId_Trig_EPWM8xSOCA = DMAMUX_ReqId_max + 48,
    DMAMUX_ReqId_Trig_EPWM8xSOCB = DMAMUX_ReqId_max + 49,
    DMAMUX_ReqId_Trig_EPWM9xSOCA = DMAMUX_ReqId_max + 50,
    DMAMUX_ReqId_Trig_EPWM9xSOCB = DMAMUX_ReqId_max + 51,
    DMAMUX_ReqId_Trig_EPWM10xSOCA = DMAMUX_ReqId_max + 52,
    DMAMUX_ReqId_Trig_EPWM10xSOCB = DMAMUX_ReqId_max + 53,
    DMAMUX_ReqId_Trig_EPWM11xSOCA = DMAMUX_ReqId_max + 54,
    DMAMUX_ReqId_Trig_EPWM11xSOCB = DMAMUX_ReqId_max + 55,
    DMAMUX_ReqId_Trig_EPWM12xSOCA = DMAMUX_ReqId_max + 56,
    DMAMUX_ReqId_Trig_EPWM12xSOCB = DMAMUX_ReqId_max + 57,
    
    DMAMUX_ReqId_Trig_ap_timer0 = DMAMUX_ReqId_max + 59,
    DMAMUX_ReqId_Trig_ap_timer1 = DMAMUX_ReqId_max + 60,
    DMAMUX_ReqId_Trig_ap_timer2 = DMAMUX_ReqId_max + 61,
    DMAMUX_ReqId_Trig_ap_timer3 = DMAMUX_ReqId_max + 62,

    DMAMUX_ReqId_Trig_ecap1 = DMAMUX_ReqId_max + 64,
    DMAMUX_ReqId_Trig_ecap2 = DMAMUX_ReqId_max + 65,
    DMAMUX_ReqId_Trig_ecap3 = DMAMUX_ReqId_max + 66,
    DMAMUX_ReqId_Trig_ecap4 = DMAMUX_ReqId_max + 67,
    DMAMUX_ReqId_Trig_ecap5 = DMAMUX_ReqId_max + 68,
    DMAMUX_ReqId_Trig_ecap6 = DMAMUX_ReqId_max + 69,
    DMAMUX_ReqId_Trig_ecap7 = DMAMUX_ReqId_max + 70,
    
    DMAMUX_ReqId_Trig_can0_if2 = DMAMUX_ReqId_max + 72,
    DMAMUX_ReqId_Trig_can0_if3 = DMAMUX_ReqId_max + 73,
    DMAMUX_ReqId_Trig_can0_if4 = DMAMUX_ReqId_max + 74,
    DMAMUX_ReqId_Trig_can0_if5 = DMAMUX_ReqId_max + 75,
    DMAMUX_ReqId_Trig_can1_if2 = DMAMUX_ReqId_max + 76,
    DMAMUX_ReqId_Trig_can1_if3 = DMAMUX_ReqId_max + 77,
    DMAMUX_ReqId_Trig_can1_if4 = DMAMUX_ReqId_max + 78,
    DMAMUX_ReqId_Trig_can1_if5 = DMAMUX_ReqId_max + 79,
    
    DMAMUX_ReqId_Trig_dmamux_evt0 = DMAMUX_ReqId_max + 81,
    DMAMUX_ReqId_Trig_dmamux_evt1 = DMAMUX_ReqId_max + 82,
    DMAMUX_ReqId_Trig_dmamux_evt2 = DMAMUX_ReqId_max + 83,
    DMAMUX_ReqId_Trig_dmamux_evt3 = DMAMUX_ReqId_max + 84,
    DMAMUX_ReqId_Trig_dmamux_evt4 = DMAMUX_ReqId_max + 85,
    DMAMUX_ReqId_Trig_dmamux_evt5 = DMAMUX_ReqId_max + 86,
    DMAMUX_ReqId_Trig_dmamux_evt6 = DMAMUX_ReqId_max + 87,
    DMAMUX_ReqId_Trig_dmamux_evt7 = DMAMUX_ReqId_max + 88,

    DMAMUX_ReqId_Trig_epg_trig = DMAMUX_ReqId_max + 92,
    DMAMUX_ReqId_Trig_soft_trig = DMAMUX_ReqId_max + 94,    
} DMAMUX_ReqId_Type;
    
    
typedef enum
{
    DMAMUX_Trig_ADCA_1int = 1,
    DMAMUX_Trig_ADCA_2int = 2,
    DMAMUX_Trig_ADCA_3int = 3,
    DMAMUX_Trig_ADCA_4int = 4,
    DMAMUX_Trig_ADCA_evt_int = 5,
    DMAMUX_Trig_ADCB_1int = 6,
    DMAMUX_Trig_ADCB_2int = 7,
    DMAMUX_Trig_ADCB_3int = 8,
    DMAMUX_Trig_ADCB_4int = 9,
    DMAMUX_Trig_ADCB_evt_int = 10,
    DMAMUX_Trig_ADCC_1int = 11,
    DMAMUX_Trig_ADCC_2int = 12,
    DMAMUX_Trig_ADCC_3int = 13,
    DMAMUX_Trig_ADCC_4int = 14,
    DMAMUX_Trig_ADCC_evt_int = 15,

    DMAMUX_Trig_cpu_1xint = 17,
    DMAMUX_Trig_cpu_2xint = 18,
    DMAMUX_Trig_cpu_3xint = 19,
    DMAMUX_Trig_cpu_4xint = 20,
    DMAMUX_Trig_cpu_5xint = 21,
    DMAMUX_Trig_cpu_6xint = 22,
    DMAMUX_Trig_cpu_7xint = 23,
    DMAMUX_Trig_cpu_8xint = 24,
    DMAMUX_Trig_cpu_9xint = 25,
    DMAMUX_Trig_cpu_10xint = 26,
    DMAMUX_Trig_cpu_11xint = 27,
    DMAMUX_Trig_cpu_12xint = 28,
    DMAMUX_Trig_cpu_13xint = 29,
    DMAMUX_Trig_cpu_14xint = 30,
    DMAMUX_Trig_cpu_15xint = 31,
    DMAMUX_Trig_cpu_16xint = 32,
    
    DMAMUX_Trig_EPWM1xSOCA = 34,
    DMAMUX_Trig_EPWM1xSOCB = 35,
    DMAMUX_Trig_EPWM2xSOCA = 36,
    DMAMUX_Trig_EPWM2xSOCB = 37,
    DMAMUX_Trig_EPWM3xSOCA = 38,
    DMAMUX_Trig_EPWM3xSOCB = 39,
    DMAMUX_Trig_EPWM4xSOCA = 40,
    DMAMUX_Trig_EPWM4xSOCB = 41,
    DMAMUX_Trig_EPWM5xSOCA = 42,
    DMAMUX_Trig_EPWM5xSOCB = 43,
    DMAMUX_Trig_EPWM6xSOCA = 44,
    DMAMUX_Trig_EPWM6xSOCB = 45,
    DMAMUX_Trig_EPWM7xSOCA = 46,
    DMAMUX_Trig_EPWM7xSOCB = 47,
    DMAMUX_Trig_EPWM8xSOCA = 48,
    DMAMUX_Trig_EPWM8xSOCB = 49,
    DMAMUX_Trig_EPWM9xSOCA = 50,
    DMAMUX_Trig_EPWM9xSOCB = 51,
    DMAMUX_Trig_EPWM10xSOCA = 52,
    DMAMUX_Trig_EPWM10xSOCB = 53,
    DMAMUX_Trig_EPWM11xSOCA = 54,
    DMAMUX_Trig_EPWM11xSOCB = 55,
    DMAMUX_Trig_EPWM12xSOCA = 56,
    DMAMUX_Trig_EPWM12xSOCB = 57,
    
    DMAMUX_Trig_ap_timer0 = 59,
    DMAMUX_Trig_ap_timer1 = 60,
    DMAMUX_Trig_ap_timer2 = 61,
    DMAMUX_Trig_ap_timer3 = 62,

    DMAMUX_Trig_ecap1 = 64,
    DMAMUX_Trig_ecap2 = 65,
    DMAMUX_Trig_ecap3 = 66,
    DMAMUX_Trig_ecap4 = 67,
    DMAMUX_Trig_ecap5 = 68,
    DMAMUX_Trig_ecap6 = 69,
    DMAMUX_Trig_ecap7 = 70,

    
    DMAMUX_Trig_can0_if2 = 72,
    DMAMUX_Trig_can0_if3 = 73,
    DMAMUX_Trig_can0_if4 = 74,
    DMAMUX_Trig_can0_if5 = 75,
    DMAMUX_Trig_can1_if2 = 76,
    DMAMUX_Trig_can1_if3 = 77,
    DMAMUX_Trig_can1_if4 = 78,
    DMAMUX_Trig_can1_if5 = 79,
    
    DMAMUX_Trig_dmamux_evt0 = 81,
    DMAMUX_Trig_dmamux_evt1 = 82,
    DMAMUX_Trig_dmamux_evt2 = 83,
    DMAMUX_Trig_dmamux_evt3 = 84,
    DMAMUX_Trig_dmamux_evt4 = 85,
    DMAMUX_Trig_dmamux_evt5 = 86,
    DMAMUX_Trig_dmamux_evt6 = 87,
    DMAMUX_Trig_dmamux_evt7 = 88,
    DMAMUX_Trig_epg_trig    = 92,
    DMAMUX_Trig_soft_trig   = 94,
} DMAMUX_TrigId_Type;

#elif (DMAMUX_VERSION == 0x30)
typedef enum
{
    DMAMUX_ReqId_dma_gen_out0 = 1,
    DMAMUX_ReqId_dma_gen_out1 = 2,
    DMAMUX_ReqId_dma_gen_out2 = 3,
    DMAMUX_ReqId_dma_gen_out3 = 4,
    DMAMUX_ReqId_dma_gen_out4 = 5,
    DMAMUX_ReqId_dma_gen_out5 = 6,
    DMAMUX_ReqId_dma_gen_out6 = 7,
    DMAMUX_ReqId_dma_gen_out7 = 8,
    DMAMUX_ReqId_spi0_rx = 10,
    DMAMUX_ReqId_spi0_tx = 11,
    DMAMUX_ReqId_spi1_rx = 12,
    DMAMUX_ReqId_spi1_tx = 13,

    DMAMUX_ReqId_i2ca_rx = 46,
    DMAMUX_ReqId_i2ca_tx = 47,
    DMAMUX_ReqId_i2cb_rx = 48,
    DMAMUX_ReqId_i2cb_tx = 49,

    DMAMUX_ReqId_pmbus_rx = 52,
    DMAMUX_ReqId_pmbus_tx = 53,

    DMAMUX_ReqId_scia_rx = 57,
    DMAMUX_ReqId_scia_tx = 58,
    DMAMUX_ReqId_scib_rx = 59,
    DMAMUX_ReqId_scib_tx = 60,
    DMAMUX_ReqId_scic_rx = 61,
    DMAMUX_ReqId_scic_tx = 62,

    DMAMUX_ReqId_lin1_rx = 70,
    DMAMUX_ReqId_lin1_tx = 71,
    DMAMUX_ReqId_lin2_rx = 72,
    DMAMUX_ReqId_lin2_tx = 73,

    DMAMUX_ReqId_sd1_flt1 = 75,
    DMAMUX_ReqId_sd1_flt2 = 76,
    DMAMUX_ReqId_sd1_flt3 = 77,
    DMAMUX_ReqId_sd1_flt4 = 78,
    DMAMUX_ReqId_sd2_flt1 = 79,
    DMAMUX_ReqId_sd2_flt2 = 80,
    DMAMUX_ReqId_sd2_flt3 = 81,
    DMAMUX_ReqId_sd2_flt4 = 82,

    DMAMUX_ReqId_max = 96, /* max req id, do not use */

    DMAMUX_ReqId_Trig_ADCA_1int = DMAMUX_ReqId_max + 1,
    DMAMUX_ReqId_Trig_ADCA_2int = DMAMUX_ReqId_max + 2,
    DMAMUX_ReqId_Trig_ADCA_3int = DMAMUX_ReqId_max + 3,
    DMAMUX_ReqId_Trig_ADCA_4int = DMAMUX_ReqId_max + 4,
    DMAMUX_ReqId_Trig_ADCA_evt_int = DMAMUX_ReqId_max + 5,
    DMAMUX_ReqId_Trig_ADCC_1int = DMAMUX_ReqId_max + 6,
    DMAMUX_ReqId_Trig_ADCC_2int = DMAMUX_ReqId_max + 7,
    DMAMUX_ReqId_Trig_ADCC_3int = DMAMUX_ReqId_max + 8,
    DMAMUX_ReqId_Trig_ADCC_4int = DMAMUX_ReqId_max + 9,
    DMAMUX_ReqId_Trig_ADCC_evt_int = DMAMUX_ReqId_max + 10,
    
    DMAMUX_ReqId_Trig_cpu_1xint = DMAMUX_ReqId_max + 22,
    DMAMUX_ReqId_Trig_cpu_2xint = DMAMUX_ReqId_max + 23,
    DMAMUX_ReqId_Trig_cpu_3xint = DMAMUX_ReqId_max + 24,
    DMAMUX_ReqId_Trig_cpu_4xint = DMAMUX_ReqId_max + 25,
    DMAMUX_ReqId_Trig_cpu_5xint = DMAMUX_ReqId_max + 26,
    DMAMUX_ReqId_Trig_cpu_6xint = DMAMUX_ReqId_max + 27,
    DMAMUX_ReqId_Trig_cpu_7xint = DMAMUX_ReqId_max + 28,
    DMAMUX_ReqId_Trig_cpu_8xint = DMAMUX_ReqId_max + 29,
    DMAMUX_ReqId_Trig_cpu_9xint = DMAMUX_ReqId_max + 30,
    DMAMUX_ReqId_Trig_cpu_10xint = DMAMUX_ReqId_max + 31,
    DMAMUX_ReqId_Trig_cpu_11xint = DMAMUX_ReqId_max + 32,
    DMAMUX_ReqId_Trig_cpu_12xint = DMAMUX_ReqId_max + 33,
    DMAMUX_ReqId_Trig_cpu_13xint = DMAMUX_ReqId_max + 34,
    DMAMUX_ReqId_Trig_cpu_14xint = DMAMUX_ReqId_max + 35,
    DMAMUX_ReqId_Trig_cpu_15xint = DMAMUX_ReqId_max + 36,
    DMAMUX_ReqId_Trig_cpu_16xint = DMAMUX_ReqId_max + 37,
    
    DMAMUX_ReqId_Trig_EPWM1xSOCA = DMAMUX_ReqId_max + 39,
    DMAMUX_ReqId_Trig_EPWM1xSOCB = DMAMUX_ReqId_max + 40,
    DMAMUX_ReqId_Trig_EPWM2xSOCA = DMAMUX_ReqId_max + 41,
    DMAMUX_ReqId_Trig_EPWM2xSOCB = DMAMUX_ReqId_max + 42,
    DMAMUX_ReqId_Trig_EPWM3xSOCA = DMAMUX_ReqId_max + 43,
    DMAMUX_ReqId_Trig_EPWM3xSOCB = DMAMUX_ReqId_max + 44,
    DMAMUX_ReqId_Trig_EPWM4xSOCA = DMAMUX_ReqId_max + 45,
    DMAMUX_ReqId_Trig_EPWM4xSOCB = DMAMUX_ReqId_max + 46,
    DMAMUX_ReqId_Trig_EPWM5xSOCA = DMAMUX_ReqId_max + 47,
    DMAMUX_ReqId_Trig_EPWM5xSOCB = DMAMUX_ReqId_max + 48,
    DMAMUX_ReqId_Trig_EPWM6xSOCA = DMAMUX_ReqId_max + 49,
    DMAMUX_ReqId_Trig_EPWM6xSOCB = DMAMUX_ReqId_max + 50,
    DMAMUX_ReqId_Trig_EPWM7xSOCA = DMAMUX_ReqId_max + 51,
    DMAMUX_ReqId_Trig_EPWM7xSOCB = DMAMUX_ReqId_max + 52,
    DMAMUX_ReqId_Trig_EPWM8xSOCA = DMAMUX_ReqId_max + 53,
    DMAMUX_ReqId_Trig_EPWM8xSOCB = DMAMUX_ReqId_max + 54,
    
    DMAMUX_ReqId_Trig_ap_timer0 = DMAMUX_ReqId_max + 76,
    DMAMUX_ReqId_Trig_ap_timer1 = DMAMUX_ReqId_max + 77,
    DMAMUX_ReqId_Trig_ap_timer2 = DMAMUX_ReqId_max + 78,
    DMAMUX_ReqId_Trig_ap_timer3 = DMAMUX_ReqId_max + 79,

    DMAMUX_ReqId_Trig_ecap1 = DMAMUX_ReqId_max + 83,
    DMAMUX_ReqId_Trig_ecap2 = DMAMUX_ReqId_max + 84,
    DMAMUX_ReqId_Trig_ecap3 = DMAMUX_ReqId_max + 85,
    
    DMAMUX_ReqId_Trig_can0_if2 = DMAMUX_ReqId_max + 91,
    DMAMUX_ReqId_Trig_can0_if3 = DMAMUX_ReqId_max + 92,
    DMAMUX_ReqId_Trig_can0_if4 = DMAMUX_ReqId_max + 93,
    DMAMUX_ReqId_Trig_can0_if5 = DMAMUX_ReqId_max + 94,
    DMAMUX_ReqId_Trig_can1_if2 = DMAMUX_ReqId_max + 95,
    DMAMUX_ReqId_Trig_can1_if3 = DMAMUX_ReqId_max + 96,
    DMAMUX_ReqId_Trig_can1_if4 = DMAMUX_ReqId_max + 97,
    DMAMUX_ReqId_Trig_can1_if5 = DMAMUX_ReqId_max + 98,

    DMAMUX_ReqId_Trig_epg_trig = DMAMUX_ReqId_max + 108,
    DMAMUX_ReqId_Trig_soft_trig = DMAMUX_ReqId_max + 110,    
    
    DMAMUX_ReqId_Trig_dmamux_evt0 = DMAMUX_ReqId_max + 112,
    DMAMUX_ReqId_Trig_dmamux_evt1 = DMAMUX_ReqId_max + 113,
    DMAMUX_ReqId_Trig_dmamux_evt2 = DMAMUX_ReqId_max + 114,
    DMAMUX_ReqId_Trig_dmamux_evt3 = DMAMUX_ReqId_max + 115,
    DMAMUX_ReqId_Trig_dmamux_evt4 = DMAMUX_ReqId_max + 116,
    DMAMUX_ReqId_Trig_dmamux_evt5 = DMAMUX_ReqId_max + 117,
    DMAMUX_ReqId_Trig_dmamux_evt6 = DMAMUX_ReqId_max + 118,
    DMAMUX_ReqId_Trig_dmamux_evt7 = DMAMUX_ReqId_max + 119,

} DMAMUX_ReqId_Type;
    
    
typedef enum
{
    DMAMUX_Trig_ADCA_1int = 1,
    DMAMUX_Trig_ADCA_2int = 2,
    DMAMUX_Trig_ADCA_3int = 3,
    DMAMUX_Trig_ADCA_4int = 4,
    DMAMUX_Trig_ADCA_evt_int = 5,
    DMAMUX_Trig_ADCB_1int = 6,
    DMAMUX_Trig_ADCB_2int = 7,
    DMAMUX_Trig_ADCB_3int = 8,
    DMAMUX_Trig_ADCB_4int = 9,
    DMAMUX_Trig_ADCB_evt_int = 10,

    DMAMUX_Trig_cpu_1xint = 22,
    DMAMUX_Trig_cpu_2xint = 23,
    DMAMUX_Trig_cpu_3xint = 24,
    DMAMUX_Trig_cpu_4xint = 25,
    DMAMUX_Trig_cpu_5xint = 26,
    DMAMUX_Trig_cpu_6xint = 27,
    DMAMUX_Trig_cpu_7xint = 28,
    DMAMUX_Trig_cpu_8xint = 29,
    DMAMUX_Trig_cpu_9xint = 30,
    DMAMUX_Trig_cpu_10xint = 31,
    DMAMUX_Trig_cpu_11xint = 32,
    DMAMUX_Trig_cpu_12xint = 33,
    DMAMUX_Trig_cpu_13xint = 34,
    DMAMUX_Trig_cpu_14xint = 35,
    DMAMUX_Trig_cpu_15xint = 36,
    DMAMUX_Trig_cpu_16xint = 37,
    
    DMAMUX_Trig_EPWM1xSOCA = 39,
    DMAMUX_Trig_EPWM1xSOCB = 40,
    DMAMUX_Trig_EPWM2xSOCA = 41,
    DMAMUX_Trig_EPWM2xSOCB = 42,
    DMAMUX_Trig_EPWM3xSOCA = 43,
    DMAMUX_Trig_EPWM3xSOCB = 44,
    DMAMUX_Trig_EPWM4xSOCA = 45,
    DMAMUX_Trig_EPWM4xSOCB = 46,
    DMAMUX_Trig_EPWM5xSOCA = 47,
    DMAMUX_Trig_EPWM5xSOCB = 48,
    DMAMUX_Trig_EPWM6xSOCA = 49,
    DMAMUX_Trig_EPWM6xSOCB = 50,
    DMAMUX_Trig_EPWM7xSOCA = 51,
    DMAMUX_Trig_EPWM7xSOCB = 52,
    DMAMUX_Trig_EPWM8xSOCA = 53,
    DMAMUX_Trig_EPWM8xSOCB = 54,

    DMAMUX_Trig_ap_timer0 = 76,
    DMAMUX_Trig_ap_timer1 = 77,
    DMAMUX_Trig_ap_timer2 = 78,
    DMAMUX_Trig_ap_timer3 = 79,

    DMAMUX_Trig_ecap1 = 83,
    DMAMUX_Trig_ecap2 = 84,
    DMAMUX_Trig_ecap3 = 85,
    
    DMAMUX_Trig_can0_if2 = 91,
    DMAMUX_Trig_can0_if3 = 92,
    DMAMUX_Trig_can0_if4 = 93,
    DMAMUX_Trig_can0_if5 = 94,
    DMAMUX_Trig_can1_if2 = 95,
    DMAMUX_Trig_can1_if3 = 96,
    DMAMUX_Trig_can1_if4 = 97,
    DMAMUX_Trig_can1_if5 = 98,

    DMAMUX_Trig_epg_trig    = 108,
    DMAMUX_Trig_soft_trig   = 110,     
    
    DMAMUX_Trig_dmamux_evt0 = 112,
    DMAMUX_Trig_dmamux_evt1 = 113,
    DMAMUX_Trig_dmamux_evt2 = 114,
    DMAMUX_Trig_dmamux_evt3 = 115,
    DMAMUX_Trig_dmamux_evt4 = 116,
    DMAMUX_Trig_dmamux_evt5 = 117,
    DMAMUX_Trig_dmamux_evt6 = 118,
    DMAMUX_Trig_dmamux_evt7 = 119,
  
} DMAMUX_TrigId_Type;        

#elif (DMAMUX_VERSION == 0x20)
typedef enum
{
    DMAMUX_ReqId_dma_gen_out0 = 0,
    DMAMUX_ReqId_dma_gen_out1 = 1,
    DMAMUX_ReqId_dma_gen_out2 = 2,
    DMAMUX_ReqId_dma_gen_out3 = 3,
    DMAMUX_ReqId_dma_gen_out4 = 4,
    DMAMUX_ReqId_dma_gen_out5 = 5,
    DMAMUX_ReqId_dma_gen_out6 = 6,
    DMAMUX_ReqId_dma_gen_out7 = 7,
    DMAMUX_ReqId_ecap1 = 8,
    DMAMUX_ReqId_ecap2 = 9,
    DMAMUX_ReqId_ecap3 = 10,
    DMAMUX_ReqId_ecap4 = 11,
    DMAMUX_ReqId_ecap5 = 12,
    DMAMUX_ReqId_ecap6 = 13,
    DMAMUX_ReqId_ecap7 = 14,

    DMAMUX_ReqId_spi0_rx = 15,
    DMAMUX_ReqId_spi0_tx = 16,
    DMAMUX_ReqId_spi1_rx = 17,
    DMAMUX_ReqId_spi1_tx = 18,
    DMAMUX_ReqId_spi2_rx = 19,
    DMAMUX_ReqId_spi2_tx = 20,
    DMAMUX_ReqId_spi3_rx = 21,
    DMAMUX_ReqId_spi3_tx = 22,

    DMAMUX_ReqId_Plu1_lc1 = 23,
    DMAMUX_ReqId_Plu1_lc2 = 24,
    DMAMUX_ReqId_Plu1_lc3 = 25,
    DMAMUX_ReqId_Plu2_lc1 = 26,
    DMAMUX_ReqId_Plu2_lc2 = 27,
    DMAMUX_ReqId_Plu2_lc3 = 28,

    DMAMUX_ReqId_i2c0_rx = 29,
    DMAMUX_ReqId_i2c0_tx = 30,
    DMAMUX_ReqId_i2c1_rx = 31,
    DMAMUX_ReqId_i2c1_tx = 32,
    DMAMUX_ReqId_i2c2_rx = 33,
    DMAMUX_ReqId_i2c2_tx = 34,

    DMAMUX_ReqId_scia_rx = 35,
    DMAMUX_ReqId_scia_tx = 36,
    DMAMUX_ReqId_scib_rx = 37,
    DMAMUX_ReqId_scib_tx = 38,
    DMAMUX_ReqId_scic_rx = 39,
    DMAMUX_ReqId_scic_tx = 40,
    DMAMUX_ReqId_scid_rx = 41,
    DMAMUX_ReqId_scid_tx = 42,

    DMAMUX_ReqId_lin1_rx = 43,
    DMAMUX_ReqId_lin1_tx = 44,
    DMAMUX_ReqId_lin2_rx = 45,
    DMAMUX_ReqId_lin2_tx = 46,

    DMAMUX_ReqId_fft_rx = 47,
    DMAMUX_ReqId_fft_tx = 48,

    DMAMUX_ReqId_max = 64, //max req id, do not use

    //trigID
    DMAMUX_ReqId_Trig_ADCA_1int    = DMAMUX_ReqId_max + 0, //!< ADCA_1int
    DMAMUX_ReqId_Trig_ADCA_2int    = DMAMUX_ReqId_max + 1, //!< ADCA_2int
    DMAMUX_ReqId_Trig_ADCA_3int    = DMAMUX_ReqId_max + 2, //!< ADCA_3int
    DMAMUX_ReqId_Trig_ADCA_4int    = DMAMUX_ReqId_max + 3,
    DMAMUX_ReqId_Trig_ADCA_evt_int = DMAMUX_ReqId_max + 4,
    DMAMUX_ReqId_Trig_ADCB_1int    = DMAMUX_ReqId_max + 5,
    DMAMUX_ReqId_Trig_ADCB_2int    = DMAMUX_ReqId_max + 6,
    DMAMUX_ReqId_Trig_ADCB_3int    = DMAMUX_ReqId_max + 7,
    DMAMUX_ReqId_Trig_ADCB_4int    = DMAMUX_ReqId_max + 8,
    DMAMUX_ReqId_Trig_ADCB_evt_int = DMAMUX_ReqId_max + 9,
    DMAMUX_ReqId_Trig_ADCC_1int    = DMAMUX_ReqId_max + 10,
    DMAMUX_ReqId_Trig_ADCC_2int    = DMAMUX_ReqId_max + 11,
    DMAMUX_ReqId_Trig_ADCC_3int    = DMAMUX_ReqId_max + 12,
    DMAMUX_ReqId_Trig_ADCC_4int    = DMAMUX_ReqId_max + 13,
    DMAMUX_ReqId_Trig_ADCC_evt_int = DMAMUX_ReqId_max + 14,
    DMAMUX_ReqId_Trig_ADCD_1int    = DMAMUX_ReqId_max + 15,
    DMAMUX_ReqId_Trig_ADCD_2int    = DMAMUX_ReqId_max + 16,
    DMAMUX_ReqId_Trig_ADCD_3int    = DMAMUX_ReqId_max + 17,
    DMAMUX_ReqId_Trig_ADCD_4int    = DMAMUX_ReqId_max + 18,
    DMAMUX_ReqId_Trig_ADCD_evt_int = DMAMUX_ReqId_max + 19,


    DMAMUX_ReqId_Trig_cpu_1xint  = DMAMUX_ReqId_max + 20,
    DMAMUX_ReqId_Trig_cpu_2xint  = DMAMUX_ReqId_max + 21,
    DMAMUX_ReqId_Trig_cpu_3xint  = DMAMUX_ReqId_max + 22,
    DMAMUX_ReqId_Trig_cpu_4xint  = DMAMUX_ReqId_max + 23,
    DMAMUX_ReqId_Trig_cpu_5xint  = DMAMUX_ReqId_max + 24,
    DMAMUX_ReqId_Trig_cpu_6xint  = DMAMUX_ReqId_max + 25,
    DMAMUX_ReqId_Trig_cpu_7xint  = DMAMUX_ReqId_max + 26,
    DMAMUX_ReqId_Trig_cpu_8xint  = DMAMUX_ReqId_max + 27,
    DMAMUX_ReqId_Trig_cpu_9xint  = DMAMUX_ReqId_max + 28,
    DMAMUX_ReqId_Trig_cpu_10xint = DMAMUX_ReqId_max + 29,
    DMAMUX_ReqId_Trig_cpu_11xint = DMAMUX_ReqId_max + 30,
    DMAMUX_ReqId_Trig_cpu_12xint = DMAMUX_ReqId_max + 31,
    DMAMUX_ReqId_Trig_cpu_13xint = DMAMUX_ReqId_max + 32,
    DMAMUX_ReqId_Trig_cpu_14xint = DMAMUX_ReqId_max + 33,
    DMAMUX_ReqId_Trig_cpu_15xint = DMAMUX_ReqId_max + 34,
    DMAMUX_ReqId_Trig_cpu_16xint = DMAMUX_ReqId_max + 35,

    DMAMUX_ReqId_Trig_EPWM1xSOCA  = DMAMUX_ReqId_max + 36,
    DMAMUX_ReqId_Trig_EPWM1xSOCB  = DMAMUX_ReqId_max + 37,
    DMAMUX_ReqId_Trig_EPWM2xSOCA  = DMAMUX_ReqId_max + 38,
    DMAMUX_ReqId_Trig_EPWM2xSOCB  = DMAMUX_ReqId_max + 39,
    DMAMUX_ReqId_Trig_EPWM3xSOCA  = DMAMUX_ReqId_max + 40,
    DMAMUX_ReqId_Trig_EPWM3xSOCB  = DMAMUX_ReqId_max + 41,
    DMAMUX_ReqId_Trig_EPWM4xSOCA  = DMAMUX_ReqId_max + 42,
    DMAMUX_ReqId_Trig_EPWM4xSOCB  = DMAMUX_ReqId_max + 43,
    DMAMUX_ReqId_Trig_EPWM5xSOCA  = DMAMUX_ReqId_max + 44,
    DMAMUX_ReqId_Trig_EPWM5xSOCB  = DMAMUX_ReqId_max + 45,
    DMAMUX_ReqId_Trig_EPWM6xSOCA  = DMAMUX_ReqId_max + 46,
    DMAMUX_ReqId_Trig_EPWM6xSOCB  = DMAMUX_ReqId_max + 47,
    DMAMUX_ReqId_Trig_EPWM7xSOCA  = DMAMUX_ReqId_max + 48,
    DMAMUX_ReqId_Trig_EPWM7xSOCB  = DMAMUX_ReqId_max + 49,
    DMAMUX_ReqId_Trig_EPWM8xSOCA  = DMAMUX_ReqId_max + 50,
    DMAMUX_ReqId_Trig_EPWM8xSOCB  = DMAMUX_ReqId_max + 51,
    DMAMUX_ReqId_Trig_EPWM9xSOCA  = DMAMUX_ReqId_max + 52,
    DMAMUX_ReqId_Trig_EPWM9xSOCB  = DMAMUX_ReqId_max + 53,
    DMAMUX_ReqId_Trig_EPWM10xSOCA = DMAMUX_ReqId_max + 54,
    DMAMUX_ReqId_Trig_EPWM10xSOCB = DMAMUX_ReqId_max + 55,
    DMAMUX_ReqId_Trig_EPWM11xSOCA = DMAMUX_ReqId_max + 56,
    DMAMUX_ReqId_Trig_EPWM11xSOCB = DMAMUX_ReqId_max + 57,
    DMAMUX_ReqId_Trig_EPWM12xSOCA = DMAMUX_ReqId_max + 58,
    DMAMUX_ReqId_Trig_EPWM12xSOCB = DMAMUX_ReqId_max + 59,
    DMAMUX_ReqId_Trig_EPWM13xSOCA = DMAMUX_ReqId_max + 60,
    DMAMUX_ReqId_Trig_EPWM13xSOCB = DMAMUX_ReqId_max + 61,
    DMAMUX_ReqId_Trig_EPWM14xSOCA = DMAMUX_ReqId_max + 62,
    DMAMUX_ReqId_Trig_EPWM14xSOCB = DMAMUX_ReqId_max + 63,
    DMAMUX_ReqId_Trig_EPWM15xSOCA = DMAMUX_ReqId_max + 64,
    DMAMUX_ReqId_Trig_EPWM15xSOCB = DMAMUX_ReqId_max + 65,
    DMAMUX_ReqId_Trig_EPWM16xSOCA = DMAMUX_ReqId_max + 66,
    DMAMUX_ReqId_Trig_EPWM16xSOCB = DMAMUX_ReqId_max + 67,

    DMAMUX_ReqId_Trig_ap_timer0 = DMAMUX_ReqId_max + 68,
    DMAMUX_ReqId_Trig_ap_timer1 = DMAMUX_ReqId_max + 69,
    DMAMUX_ReqId_Trig_ap_timer2 = DMAMUX_ReqId_max + 70,
    DMAMUX_ReqId_Trig_ap_timer3 = DMAMUX_ReqId_max + 71,
    DMAMUX_ReqId_Trig_ap_timer4 = DMAMUX_ReqId_max + 72,
    DMAMUX_ReqId_Trig_ap_timer5 = DMAMUX_ReqId_max + 73,

    DMAMUX_ReqId_Trig_can0_if2 = DMAMUX_ReqId_max + 74,
    DMAMUX_ReqId_Trig_can0_if3 = DMAMUX_ReqId_max + 75,
    DMAMUX_ReqId_Trig_can0_if4 = DMAMUX_ReqId_max + 76,
    DMAMUX_ReqId_Trig_can0_if5 = DMAMUX_ReqId_max + 77,
    DMAMUX_ReqId_Trig_can1_if2 = DMAMUX_ReqId_max + 78,
    DMAMUX_ReqId_Trig_can1_if3 = DMAMUX_ReqId_max + 79,
    DMAMUX_ReqId_Trig_can1_if4 = DMAMUX_ReqId_max + 80,
    DMAMUX_ReqId_Trig_can1_if5 = DMAMUX_ReqId_max + 81,
    DMAMUX_ReqId_Trig_can2_if2 = DMAMUX_ReqId_max + 82,
    DMAMUX_ReqId_Trig_can2_if3 = DMAMUX_ReqId_max + 83,
    DMAMUX_ReqId_Trig_can2_if4 = DMAMUX_ReqId_max + 84,
    DMAMUX_ReqId_Trig_can2_if5 = DMAMUX_ReqId_max + 85,
    DMAMUX_ReqId_Trig_can3_if2 = DMAMUX_ReqId_max + 86,
    DMAMUX_ReqId_Trig_can3_if3 = DMAMUX_ReqId_max + 87,
    DMAMUX_ReqId_Trig_can3_if4 = DMAMUX_ReqId_max + 88,
    DMAMUX_ReqId_Trig_can3_if5 = DMAMUX_ReqId_max + 89,

    DMAMUX_ReqId_Trig_dmamux_evt0  = DMAMUX_ReqId_max + 90,
    DMAMUX_ReqId_Trig_dmamux_evt1  = DMAMUX_ReqId_max + 91,
    DMAMUX_ReqId_Trig_dmamux_evt2  = DMAMUX_ReqId_max + 92,
    DMAMUX_ReqId_Trig_dmamux_evt3  = DMAMUX_ReqId_max + 93,
    DMAMUX_ReqId_Trig_dmamux_evt4  = DMAMUX_ReqId_max + 94,
    DMAMUX_ReqId_Trig_dmamux_evt5  = DMAMUX_ReqId_max + 95,
    DMAMUX_ReqId_Trig_dmamux_evt6  = DMAMUX_ReqId_max + 96,
    DMAMUX_ReqId_Trig_dmamux_evt7  = DMAMUX_ReqId_max + 97,
    DMAMUX_ReqId_Trig_dmamux_evt8  = DMAMUX_ReqId_max + 98,
    DMAMUX_ReqId_Trig_dmamux_evt9  = DMAMUX_ReqId_max + 99,
    DMAMUX_ReqId_Trig_dmamux_evt10 = DMAMUX_ReqId_max + 100,
    DMAMUX_ReqId_Trig_dmamux_evt11 = DMAMUX_ReqId_max + 101,
    DMAMUX_ReqId_Trig_dmamux_evt12 = DMAMUX_ReqId_max + 102,
    DMAMUX_ReqId_Trig_dmamux_evt13 = DMAMUX_ReqId_max + 103,
    DMAMUX_ReqId_Trig_dmamux_evt14 = DMAMUX_ReqId_max + 104,
    DMAMUX_ReqId_Trig_dmamux_evt15 = DMAMUX_ReqId_max + 105,
} DMAMUX_ReqId_Type;


typedef enum
{
    DMAMUX_TrigId_ADCA_1int    = 0, //!< ADCA_1int
    DMAMUX_TrigId_ADCA_2int    = 1, //!< ADCA_2int
    DMAMUX_TrigId_ADCA_3int    = 2, //!< ADCA_3int
    DMAMUX_TrigId_ADCA_4int    = 3,
    DMAMUX_TrigId_ADCA_evt_int = 4,
    DMAMUX_TrigId_ADCB_1int    = 5,
    DMAMUX_TrigId_ADCB_2int    = 6,
    DMAMUX_TrigId_ADCB_3int    = 7,
    DMAMUX_TrigId_ADCB_4int    = 8,
    DMAMUX_TrigId_ADCB_evt_int = 9,
    DMAMUX_TrigId_ADCC_1int    = 10,
    DMAMUX_TrigId_ADCC_2int    = 11,
    DMAMUX_TrigId_ADCC_3int    = 12,
    DMAMUX_TrigId_ADCC_4int    = 13,
    DMAMUX_TrigId_ADCC_evt_int = 14,
    DMAMUX_TrigId_ADCD_1int    = 15,
    DMAMUX_TrigId_ADCD_2int    = 16,
    DMAMUX_TrigId_ADCD_3int    = 17,
    DMAMUX_TrigId_ADCD_4int    = 18,
    DMAMUX_TrigId_ADCD_evt_int = 19,


    DMAMUX_TrigId_cpu_1xint  = 20,
    DMAMUX_TrigId_cpu_2xint  = 21,
    DMAMUX_TrigId_cpu_3xint  = 22,
    DMAMUX_TrigId_cpu_4xint  = 23,
    DMAMUX_TrigId_cpu_5xint  = 24,
    DMAMUX_TrigId_cpu_6xint  = 25,
    DMAMUX_TrigId_cpu_7xint  = 26,
    DMAMUX_TrigId_cpu_8xint  = 27,
    DMAMUX_TrigId_cpu_9xint  = 28,
    DMAMUX_TrigId_cpu_10xint = 29,
    DMAMUX_TrigId_cpu_11xint = 30,
    DMAMUX_TrigId_cpu_12xint = 31,
    DMAMUX_TrigId_cpu_13xint = 32,
    DMAMUX_TrigId_cpu_14xint = 33,
    DMAMUX_TrigId_cpu_15xint = 34,
    DMAMUX_TrigId_cpu_16xint = 35,

    DMAMUX_TrigId_EPWM1xSOCA  = 36,
    DMAMUX_TrigId_EPWM1xSOCB  = 37,
    DMAMUX_TrigId_EPWM2xSOCA  = 38,
    DMAMUX_TrigId_EPWM2xSOCB  = 39,
    DMAMUX_TrigId_EPWM3xSOCA  = 40,
    DMAMUX_TrigId_EPWM3xSOCB  = 41,
    DMAMUX_TrigId_EPWM4xSOCA  = 42,
    DMAMUX_TrigId_EPWM4xSOCB  = 43,
    DMAMUX_TrigId_EPWM5xSOCA  = 44,
    DMAMUX_TrigId_EPWM5xSOCB  = 45,
    DMAMUX_TrigId_EPWM6xSOCA  = 46,
    DMAMUX_TrigId_EPWM6xSOCB  = 47,
    DMAMUX_TrigId_EPWM7xSOCA  = 48,
    DMAMUX_TrigId_EPWM7xSOCB  = 49,
    DMAMUX_TrigId_EPWM8xSOCA  = 50,
    DMAMUX_TrigId_EPWM8xSOCB  = 51,
    DMAMUX_TrigId_EPWM9xSOCA  = 52,
    DMAMUX_TrigId_EPWM9xSOCB  = 53,
    DMAMUX_TrigId_EPWM10xSOCA = 54,
    DMAMUX_TrigId_EPWM10xSOCB = 55,
    DMAMUX_TrigId_EPWM11xSOCA = 56,
    DMAMUX_TrigId_EPWM11xSOCB = 57,
    DMAMUX_TrigId_EPWM12xSOCA = 58,
    DMAMUX_TrigId_EPWM12xSOCB = 59,
    DMAMUX_TrigId_EPWM13xSOCA = 60,
    DMAMUX_TrigId_EPWM13xSOCB = 61,
    DMAMUX_TrigId_EPWM14xSOCA = 62,
    DMAMUX_TrigId_EPWM14xSOCB = 63,
    DMAMUX_TrigId_EPWM15xSOCA = 64,
    DMAMUX_TrigId_EPWM15xSOCB = 65,
    DMAMUX_TrigId_EPWM16xSOCA = 66,
    DMAMUX_TrigId_EPWM16xSOCB = 67,

    DMAMUX_TrigId_ap_timer0 = 68,
    DMAMUX_TrigId_ap_timer1 = 69,
    DMAMUX_TrigId_ap_timer2 = 70,
    DMAMUX_TrigId_ap_timer3 = 71,
    DMAMUX_TrigId_ap_timer4 = 72,
    DMAMUX_TrigId_ap_timer5 = 73,

    DMAMUX_TrigId_can0_if2 = 74,
    DMAMUX_TrigId_can0_if3 = 75,
    DMAMUX_TrigId_can0_if4 = 76,
    DMAMUX_TrigId_can0_if5 = 77,
    DMAMUX_TrigId_can1_if2 = 78,
    DMAMUX_TrigId_can1_if3 = 79,
    DMAMUX_TrigId_can1_if4 = 80,
    DMAMUX_TrigId_can1_if5 = 81,
    DMAMUX_TrigId_can2_if2 = 82,
    DMAMUX_TrigId_can2_if3 = 83,
    DMAMUX_TrigId_can2_if4 = 84,
    DMAMUX_TrigId_can2_if5 = 85,
    DMAMUX_TrigId_can3_if2 = 86,
    DMAMUX_TrigId_can3_if3 = 87,
    DMAMUX_TrigId_can3_if4 = 88,
    DMAMUX_TrigId_can3_if5 = 89,

    DMAMUX_TrigId_dmamux_evt0  = 90,
    DMAMUX_TrigId_dmamux_evt1  = 91,
    DMAMUX_TrigId_dmamux_evt2  = 92,
    DMAMUX_TrigId_dmamux_evt3  = 93,
    DMAMUX_TrigId_dmamux_evt4  = 94,
    DMAMUX_TrigId_dmamux_evt5  = 95,
    DMAMUX_TrigId_dmamux_evt6  = 96,
    DMAMUX_TrigId_dmamux_evt7  = 97,
    DMAMUX_TrigId_dmamux_evt8  = 98,
    DMAMUX_TrigId_dmamux_evt9  = 99,
    DMAMUX_TrigId_dmamux_evt10 = 100,
    DMAMUX_TrigId_dmamux_evt11 = 101,
    DMAMUX_TrigId_dmamux_evt12 = 102,
    DMAMUX_TrigId_dmamux_evt13 = 103,
    DMAMUX_TrigId_dmamux_evt14 = 104,
    DMAMUX_TrigId_dmamux_evt15 = 105,
} DMAMUX_TrigId_Type;

#elif (DMAMUX_VERSION == 0x22)
typedef enum
{
    DMAMUX_ReqId_dma_gen_out0 = 1,
    DMAMUX_ReqId_dma_gen_out1 = 2,
    DMAMUX_ReqId_dma_gen_out2 = 3,
    DMAMUX_ReqId_dma_gen_out3 = 4,
    DMAMUX_ReqId_dma_gen_out4 = 5,
    DMAMUX_ReqId_dma_gen_out5 = 6,
    DMAMUX_ReqId_dma_gen_out6 = 7,
    DMAMUX_ReqId_dma_gen_out7 = 8,

    DMAMUX_ReqId_spi0_rx = 10,
    DMAMUX_ReqId_spi0_tx = 11,
    DMAMUX_ReqId_spi1_rx = 12,
    DMAMUX_ReqId_spi1_tx = 13,
    DMAMUX_ReqId_spi2_rx = 14,
    DMAMUX_ReqId_spi2_tx = 15,
    DMAMUX_ReqId_spi3_rx = 16,
    DMAMUX_ReqId_spi3_tx = 17,
    DMAMUX_ReqId_cmspi0_rx = 18,
    DMAMUX_ReqId_cmspi0_tx = 19,

    DMAMUX_ReqId_plu1_lc1 = 21,
    DMAMUX_ReqId_plu1_lc2 = 22,
    DMAMUX_ReqId_plu1_lc3 = 23,
    DMAMUX_ReqId_plu2_lc1 = 24,
    DMAMUX_ReqId_plu2_lc2 = 25,
    DMAMUX_ReqId_plu2_lc3 = 26,
    DMAMUX_ReqId_plu3_lc1 = 27,
    DMAMUX_ReqId_plu3_lc2 = 28,
    DMAMUX_ReqId_plu3_lc3 = 29,
    DMAMUX_ReqId_plu4_lc1 = 30,
    DMAMUX_ReqId_plu4_lc2 = 31,
    DMAMUX_ReqId_plu4_lc3 = 32,
    DMAMUX_ReqId_plu5_lc1 = 33,
    DMAMUX_ReqId_plu5_lc2 = 34,
    DMAMUX_ReqId_plu5_lc3 = 35,
    DMAMUX_ReqId_plu6_lc1 = 36,
    DMAMUX_ReqId_plu6_lc2 = 37,
    DMAMUX_ReqId_plu6_lc3 = 38,
    DMAMUX_ReqId_plu7_lc1 = 39,
    DMAMUX_ReqId_plu7_lc2 = 40,
    DMAMUX_ReqId_plu7_lc3 = 41,
    DMAMUX_ReqId_plu8_lc1 = 42,
    DMAMUX_ReqId_plu8_lc2 = 43,
    DMAMUX_ReqId_plu8_lc3 = 44,

    DMAMUX_ReqId_i2ca_rx = 46,
    DMAMUX_ReqId_i2ca_tx = 47,
    DMAMUX_ReqId_i2cb_rx = 48,
    DMAMUX_ReqId_i2cb_tx = 49,
    DMAMUX_ReqId_cmi2c_rx = 50,
    DMAMUX_ReqId_cmi2c_tx = 51,
    
    DMAMUX_ReqId_pmbus_rx = 52,
    DMAMUX_ReqId_pmbus_tx = 53,

    DMAMUX_ReqId_scia_rx = 57,
    DMAMUX_ReqId_scia_tx = 58,
    DMAMUX_ReqId_scib_rx = 59,
    DMAMUX_ReqId_scib_tx = 60,
    DMAMUX_ReqId_scic_rx = 61,
    DMAMUX_ReqId_scic_tx = 62,
    DMAMUX_ReqId_scid_rx = 63,
    DMAMUX_ReqId_scid_tx = 64,
    DMAMUX_ReqId_scie_rx = 65,
    DMAMUX_ReqId_scie_tx = 66,
    DMAMUX_ReqId_scif_rx = 67,
    DMAMUX_ReqId_scif_tx = 68,

    DMAMUX_ReqId_lin1_rx = 70,
    DMAMUX_ReqId_lin1_tx = 71,
    DMAMUX_ReqId_lin2_rx = 72,
    DMAMUX_ReqId_lin2_tx = 73,

    DMAMUX_ReqId_sd1_flt1 = 75,
    DMAMUX_ReqId_sd1_flt2 = 76,
    DMAMUX_ReqId_sd1_flt3 = 77,
    DMAMUX_ReqId_sd1_flt4 = 78,
    DMAMUX_ReqId_sd2_flt1 = 79,
    DMAMUX_ReqId_sd2_flt2 = 80,
    DMAMUX_ReqId_sd2_flt3 = 81,
    DMAMUX_ReqId_sd2_flt4 = 82,
    DMAMUX_ReqId_sd3_flt1 = 83,
    DMAMUX_ReqId_sd3_flt2 = 84,
    DMAMUX_ReqId_sd3_flt3 = 85,
    DMAMUX_ReqId_sd3_flt4 = 86,
    DMAMUX_ReqId_sd4_flt1 = 87,
    DMAMUX_ReqId_sd4_flt2 = 88,
    DMAMUX_ReqId_sd4_flt3 = 89,
    DMAMUX_ReqId_sd4_flt4 = 90,    

    DMAMUX_ReqId_max = 96, //max req id, do not use

    //trigID
  
    DMAMUX_ReqId_Trig_ADCA_1int = DMAMUX_ReqId_max + 1,
    DMAMUX_ReqId_Trig_ADCA_2int = DMAMUX_ReqId_max + 2,
    DMAMUX_ReqId_Trig_ADCA_3int = DMAMUX_ReqId_max + 3,
    DMAMUX_ReqId_Trig_ADCA_4int = DMAMUX_ReqId_max + 4,
    DMAMUX_ReqId_Trig_ADCA_evt_int = DMAMUX_ReqId_max + 5,
    DMAMUX_ReqId_Trig_ADCB_1int = DMAMUX_ReqId_max + 6,
    DMAMUX_ReqId_Trig_ADCB_2int = DMAMUX_ReqId_max + 7,
    DMAMUX_ReqId_Trig_ADCB_3int = DMAMUX_ReqId_max + 8,
    DMAMUX_ReqId_Trig_ADCB_4int = DMAMUX_ReqId_max + 9,
    DMAMUX_ReqId_Trig_ADCB_evt_int = DMAMUX_ReqId_max + 10,
    DMAMUX_ReqId_Trig_ADCC_1int = DMAMUX_ReqId_max + 11,
    DMAMUX_ReqId_Trig_ADCC_2int = DMAMUX_ReqId_max + 12,
    DMAMUX_ReqId_Trig_ADCC_3int = DMAMUX_ReqId_max + 13,
    DMAMUX_ReqId_Trig_ADCC_4int = DMAMUX_ReqId_max + 14,
    DMAMUX_ReqId_Trig_ADCC_evt_int = DMAMUX_ReqId_max + 15,
    DMAMUX_ReqId_Trig_ADCD_1int = DMAMUX_ReqId_max + 16,
    DMAMUX_ReqId_Trig_ADCD_2int = DMAMUX_ReqId_max + 17,
    DMAMUX_ReqId_Trig_ADCD_3int = DMAMUX_ReqId_max + 18,
    DMAMUX_ReqId_Trig_ADCD_4int = DMAMUX_ReqId_max + 19,
    DMAMUX_ReqId_Trig_ADCD_evt_int = DMAMUX_ReqId_max + 20,
        
    DMAMUX_ReqId_Trig_cpu_1xint = DMAMUX_ReqId_max + 22,
    DMAMUX_ReqId_Trig_cpu_2xint = DMAMUX_ReqId_max + 23,
    DMAMUX_ReqId_Trig_cpu_3xint = DMAMUX_ReqId_max + 24,
    DMAMUX_ReqId_Trig_cpu_4xint = DMAMUX_ReqId_max + 25,
    DMAMUX_ReqId_Trig_cpu_5xint = DMAMUX_ReqId_max + 26,
    DMAMUX_ReqId_Trig_cpu_6xint = DMAMUX_ReqId_max + 27,
    DMAMUX_ReqId_Trig_cpu_7xint = DMAMUX_ReqId_max + 28,
    DMAMUX_ReqId_Trig_cpu_8xint = DMAMUX_ReqId_max + 29,
    DMAMUX_ReqId_Trig_cpu_9xint = DMAMUX_ReqId_max + 30,
    DMAMUX_ReqId_Trig_cpu_10xint = DMAMUX_ReqId_max + 31,
    DMAMUX_ReqId_Trig_cpu_11xint = DMAMUX_ReqId_max + 32,
    DMAMUX_ReqId_Trig_cpu_12xint = DMAMUX_ReqId_max + 33,
    DMAMUX_ReqId_Trig_cpu_13xint = DMAMUX_ReqId_max + 34,
    DMAMUX_ReqId_Trig_cpu_14xint = DMAMUX_ReqId_max + 35,
    DMAMUX_ReqId_Trig_cpu_15xint = DMAMUX_ReqId_max + 36,
    DMAMUX_ReqId_Trig_cpu_16xint = DMAMUX_ReqId_max + 37,
    
    DMAMUX_ReqId_Trig_EPWM1xSOCA = DMAMUX_ReqId_max + 39,
    DMAMUX_ReqId_Trig_EPWM1xSOCB = DMAMUX_ReqId_max + 40,
    DMAMUX_ReqId_Trig_EPWM2xSOCA = DMAMUX_ReqId_max + 41,
    DMAMUX_ReqId_Trig_EPWM2xSOCB = DMAMUX_ReqId_max + 42,
    DMAMUX_ReqId_Trig_EPWM3xSOCA = DMAMUX_ReqId_max + 43,
    DMAMUX_ReqId_Trig_EPWM3xSOCB = DMAMUX_ReqId_max + 44,
    DMAMUX_ReqId_Trig_EPWM4xSOCA = DMAMUX_ReqId_max + 45,
    DMAMUX_ReqId_Trig_EPWM4xSOCB = DMAMUX_ReqId_max + 46,
    DMAMUX_ReqId_Trig_EPWM5xSOCA = DMAMUX_ReqId_max + 47,
    DMAMUX_ReqId_Trig_EPWM5xSOCB = DMAMUX_ReqId_max + 48,
    DMAMUX_ReqId_Trig_EPWM6xSOCA = DMAMUX_ReqId_max + 49,
    DMAMUX_ReqId_Trig_EPWM6xSOCB = DMAMUX_ReqId_max + 50,
    DMAMUX_ReqId_Trig_EPWM7xSOCA = DMAMUX_ReqId_max + 51,
    DMAMUX_ReqId_Trig_EPWM7xSOCB = DMAMUX_ReqId_max + 52,
    DMAMUX_ReqId_Trig_EPWM8xSOCA = DMAMUX_ReqId_max + 53,
    DMAMUX_ReqId_Trig_EPWM8xSOCB = DMAMUX_ReqId_max + 54,
    DMAMUX_ReqId_Trig_EPWM9xSOCA = DMAMUX_ReqId_max + 55,
    DMAMUX_ReqId_Trig_EPWM9xSOCB = DMAMUX_ReqId_max + 56,
    DMAMUX_ReqId_Trig_EPWM10xSOCA = DMAMUX_ReqId_max + 57,
    DMAMUX_ReqId_Trig_EPWM10xSOCB = DMAMUX_ReqId_max + 58,
    DMAMUX_ReqId_Trig_EPWM11xSOCA = DMAMUX_ReqId_max + 59,
    DMAMUX_ReqId_Trig_EPWM11xSOCB = DMAMUX_ReqId_max + 60,
    DMAMUX_ReqId_Trig_EPWM12xSOCA = DMAMUX_ReqId_max + 61,
    DMAMUX_ReqId_Trig_EPWM12xSOCB = DMAMUX_ReqId_max + 62,
    DMAMUX_ReqId_Trig_EPWM13xSOCA = DMAMUX_ReqId_max + 63,
    DMAMUX_ReqId_Trig_EPWM13xSOCB = DMAMUX_ReqId_max + 64,
    DMAMUX_ReqId_Trig_EPWM14xSOCA = DMAMUX_ReqId_max + 65,
    DMAMUX_ReqId_Trig_EPWM14xSOCB = DMAMUX_ReqId_max + 66,
    DMAMUX_ReqId_Trig_EPWM15xSOCA = DMAMUX_ReqId_max + 67,
    DMAMUX_ReqId_Trig_EPWM15xSOCB = DMAMUX_ReqId_max + 68,
    DMAMUX_ReqId_Trig_EPWM16xSOCA = DMAMUX_ReqId_max + 69,
    DMAMUX_ReqId_Trig_EPWM16xSOCB = DMAMUX_ReqId_max + 70,
    DMAMUX_ReqId_Trig_EPWM17xSOCA = DMAMUX_ReqId_max + 71,
    DMAMUX_ReqId_Trig_EPWM17xSOCB = DMAMUX_ReqId_max + 72,
    DMAMUX_ReqId_Trig_EPWM18xSOCA = DMAMUX_ReqId_max + 73,
    DMAMUX_ReqId_Trig_EPWM18xSOCB = DMAMUX_ReqId_max + 74,

    DMAMUX_ReqId_Trig_ap_timer0 = DMAMUX_ReqId_max + 76,
    DMAMUX_ReqId_Trig_ap_timer1 = DMAMUX_ReqId_max + 77,
    DMAMUX_ReqId_Trig_ap_timer2 = DMAMUX_ReqId_max + 78,
    DMAMUX_ReqId_Trig_ap_timer3 = DMAMUX_ReqId_max + 79,
    DMAMUX_ReqId_Trig_ap_timer4 = DMAMUX_ReqId_max + 80,
    DMAMUX_ReqId_Trig_ap_timer5 = DMAMUX_ReqId_max + 81,

    DMAMUX_ReqId_Trig_ecap1 = DMAMUX_ReqId_max + 83,
    DMAMUX_ReqId_Trig_ecap2 = DMAMUX_ReqId_max + 84,
    DMAMUX_ReqId_Trig_ecap3 = DMAMUX_ReqId_max + 85,
    DMAMUX_ReqId_Trig_ecap4 = DMAMUX_ReqId_max + 86,
    DMAMUX_ReqId_Trig_ecap5 = DMAMUX_ReqId_max + 87,
    DMAMUX_ReqId_Trig_ecap6 = DMAMUX_ReqId_max + 88,
    DMAMUX_ReqId_Trig_ecap7 = DMAMUX_ReqId_max + 89,
    
    DMAMUX_ReqId_Trig_can0_if2 = DMAMUX_ReqId_max + 91,
    DMAMUX_ReqId_Trig_can0_if3 = DMAMUX_ReqId_max + 92,
    DMAMUX_ReqId_Trig_can0_if4 = DMAMUX_ReqId_max + 93,
    DMAMUX_ReqId_Trig_can0_if5 = DMAMUX_ReqId_max + 94,
    DMAMUX_ReqId_Trig_can1_if2 = DMAMUX_ReqId_max + 95,
    DMAMUX_ReqId_Trig_can1_if3 = DMAMUX_ReqId_max + 96,
    DMAMUX_ReqId_Trig_can1_if4 = DMAMUX_ReqId_max + 97,
    DMAMUX_ReqId_Trig_can1_if5 = DMAMUX_ReqId_max + 98,
    DMAMUX_ReqId_Trig_can2_if2 = DMAMUX_ReqId_max + 99,
    DMAMUX_ReqId_Trig_can2_if3 = DMAMUX_ReqId_max + 100,
    DMAMUX_ReqId_Trig_can2_if4 = DMAMUX_ReqId_max + 101,
    DMAMUX_ReqId_Trig_can2_if5 = DMAMUX_ReqId_max + 102,
    DMAMUX_ReqId_Trig_can3_if2 = DMAMUX_ReqId_max + 103,
    DMAMUX_ReqId_Trig_can3_if3 = DMAMUX_ReqId_max + 104,
    DMAMUX_ReqId_Trig_can3_if4 = DMAMUX_ReqId_max + 105,
    DMAMUX_ReqId_Trig_can3_if5 = DMAMUX_ReqId_max + 106,

    DMAMUX_ReqId_Trig_epg_trig = DMAMUX_ReqId_max + 108,
    DMAMUX_ReqId_Trig_soft_trig = DMAMUX_ReqId_max + 110,  
    
    DMAMUX_ReqId_Trig_dmamux_evt0 = DMAMUX_ReqId_max + 112,
    DMAMUX_ReqId_Trig_dmamux_evt1 = DMAMUX_ReqId_max + 113,
    DMAMUX_ReqId_Trig_dmamux_evt2 = DMAMUX_ReqId_max + 114,
    DMAMUX_ReqId_Trig_dmamux_evt3 = DMAMUX_ReqId_max + 115,
    DMAMUX_ReqId_Trig_dmamux_evt4 = DMAMUX_ReqId_max + 116,
    DMAMUX_ReqId_Trig_dmamux_evt5 = DMAMUX_ReqId_max + 117,
    DMAMUX_ReqId_Trig_dmamux_evt6 = DMAMUX_ReqId_max + 118,
    DMAMUX_ReqId_Trig_dmamux_evt7 = DMAMUX_ReqId_max + 119,  
} DMAMUX_ReqId_Type;
    
    
typedef enum
{
    DMAMUX_Trig_ADCA_1int = 1,
    DMAMUX_Trig_ADCA_2int = 2,
    DMAMUX_Trig_ADCA_3int = 3,
    DMAMUX_Trig_ADCA_4int = 4,
    DMAMUX_Trig_ADCA_evt_int = 5,
    DMAMUX_Trig_ADCB_1int = 6,
    DMAMUX_Trig_ADCB_2int = 7,
    DMAMUX_Trig_ADCB_3int = 8,
    DMAMUX_Trig_ADCB_4int = 9,
    DMAMUX_Trig_ADCB_evt_int = 10,
    DMAMUX_Trig_ADCC_1int = 11,
    DMAMUX_Trig_ADCC_2int = 12,
    DMAMUX_Trig_ADCC_3int = 13,
    DMAMUX_Trig_ADCC_4int = 14,
    DMAMUX_Trig_ADCC_evt_int = 15,
    DMAMUX_Trig_ADCD_1int = 16,
    DMAMUX_Trig_ADCD_2int = 17,
    DMAMUX_Trig_ADCD_3int = 18,
    DMAMUX_Trig_ADCD_4int = 19,
    DMAMUX_Trig_ADCD_evt_int = 20,

    DMAMUX_Trig_cpu_1xint = 22,
    DMAMUX_Trig_cpu_2xint = 23,
    DMAMUX_Trig_cpu_3xint = 24,
    DMAMUX_Trig_cpu_4xint = 25,
    DMAMUX_Trig_cpu_5xint = 26,
    DMAMUX_Trig_cpu_6xint = 27,
    DMAMUX_Trig_cpu_7xint = 28,
    DMAMUX_Trig_cpu_8xint = 29,
    DMAMUX_Trig_cpu_9xint = 30,
    DMAMUX_Trig_cpu_10xint = 31,
    DMAMUX_Trig_cpu_11xint = 32,
    DMAMUX_Trig_cpu_12xint = 33,
    DMAMUX_Trig_cpu_13xint = 34,
    DMAMUX_Trig_cpu_14xint = 35,
    DMAMUX_Trig_cpu_15xint = 36,
    DMAMUX_Trig_cpu_16xint = 37,
    
    DMAMUX_Trig_EPWM1xSOCA = 39,
    DMAMUX_Trig_EPWM1xSOCB = 40,
    DMAMUX_Trig_EPWM2xSOCA = 41,
    DMAMUX_Trig_EPWM2xSOCB = 42,
    DMAMUX_Trig_EPWM3xSOCA = 43,
    DMAMUX_Trig_EPWM3xSOCB = 44,
    DMAMUX_Trig_EPWM4xSOCA = 45,
    DMAMUX_Trig_EPWM4xSOCB = 46,
    DMAMUX_Trig_EPWM5xSOCA = 47,
    DMAMUX_Trig_EPWM5xSOCB = 48,
    DMAMUX_Trig_EPWM6xSOCA = 49,
    DMAMUX_Trig_EPWM6xSOCB = 50,
    DMAMUX_Trig_EPWM7xSOCA = 51,
    DMAMUX_Trig_EPWM7xSOCB = 52,
    DMAMUX_Trig_EPWM8xSOCA = 53,
    DMAMUX_Trig_EPWM8xSOCB = 54,
    DMAMUX_Trig_EPWM9xSOCA = 55,
    DMAMUX_Trig_EPWM9xSOCB = 56,
    DMAMUX_Trig_EPWM10xSOCA = 57,
    DMAMUX_Trig_EPWM10xSOCB = 58,
    DMAMUX_Trig_EPWM11xSOCA = 59,
    DMAMUX_Trig_EPWM11xSOCB = 60,
    DMAMUX_Trig_EPWM12xSOCA = 61,
    DMAMUX_Trig_EPWM12xSOCB = 62,
    DMAMUX_Trig_EPWM13xSOCA = 63,
    DMAMUX_Trig_EPWM13xSOCB = 64,
    DMAMUX_Trig_EPWM14xSOCA = 65,
    DMAMUX_Trig_EPWM14xSOCB = 66,
    DMAMUX_Trig_EPWM15xSOCA = 67,
    DMAMUX_Trig_EPWM15xSOCB = 68,
    DMAMUX_Trig_EPWM16xSOCA = 69,
    DMAMUX_Trig_EPWM16xSOCB = 70,
    DMAMUX_Trig_EPWM17xSOCA = 71,
    DMAMUX_Trig_EPWM17xSOCB = 72,
    DMAMUX_Trig_EPWM18xSOCA = 73,
    DMAMUX_Trig_EPWM18xSOCB = 74,
    
    DMAMUX_Trig_ap_timer0 = 76,
    DMAMUX_Trig_ap_timer1 = 77,
    DMAMUX_Trig_ap_timer2 = 78,
    DMAMUX_Trig_ap_timer3 = 79,
    DMAMUX_Trig_ap_timer4 = 80,
    DMAMUX_Trig_ap_timer5 = 81,

    DMAMUX_Trig_ecap1 = 83,
    DMAMUX_Trig_ecap2 = 84,
    DMAMUX_Trig_ecap3 = 85,
    DMAMUX_Trig_ecap4 = 86,
    DMAMUX_Trig_ecap5 = 87,
    DMAMUX_Trig_ecap6 = 88,
    DMAMUX_Trig_ecap7 = 89,

    DMAMUX_Trig_can0_if2 = 91,
    DMAMUX_Trig_can0_if3 = 92,
    DMAMUX_Trig_can0_if4 = 93,
    DMAMUX_Trig_can0_if5 = 94,
    DMAMUX_Trig_can1_if2 = 95,
    DMAMUX_Trig_can1_if3 = 96,
    DMAMUX_Trig_can1_if4 = 97,
    DMAMUX_Trig_can1_if5 = 98,
    DMAMUX_Trig_can2_if2 = 99,
    DMAMUX_Trig_can2_if3 = 100,
    DMAMUX_Trig_can2_if4 = 101,
    DMAMUX_Trig_can2_if5 = 102,
    DMAMUX_Trig_can3_if2 = 103,
    DMAMUX_Trig_can3_if3 = 104,
    DMAMUX_Trig_can3_if4 = 105,
    DMAMUX_Trig_can3_if5 = 106,

    DMAMUX_Trig_epg_trig    = 108,
    DMAMUX_Trig_soft_trig   = 110,   
    
    DMAMUX_Trig_dmamux_evt0 = 112,
    DMAMUX_Trig_dmamux_evt1 = 113,
    DMAMUX_Trig_dmamux_evt2 = 114,
    DMAMUX_Trig_dmamux_evt3 = 115,
    DMAMUX_Trig_dmamux_evt4 = 116,
    DMAMUX_Trig_dmamux_evt5 = 117,
    DMAMUX_Trig_dmamux_evt6 = 118,
    DMAMUX_Trig_dmamux_evt7 = 119,
    
} DMAMUX_TrigId_Type;        

#endif /* #if (DMAMUX_VERSION ==0x11) */


/**
 * @brief Defines the edge polarity of the selected synchronization input
 * @param DMAMUX_POL_NONE No event (no synchronization, no detection)
 * @param DMAMUX_POL_RISING Rising edge
 * @param DMAMUX_POL_FALLING Falling edge
 * @param DMAMUX_POL_RISING_AND_FALLING Rising and falling edges
 */
typedef enum
{
    DMAMUX_POL_NONE                    = 0x0,
    DMAMUX_POL_RISING                  = 0x1,
    DMAMUX_POL_FALLING                 = 0x2,
    DMAMUX_POL_RISING_AND_FALLING      = 0x3,
} DMAMUX_Pol_Polarity;

/**
 * @brief Synchronization enable
 * @param DMAMUX_SE_DISABLE  Synchronization disabled
 * @param DMAMUX_SE_ENABLE   Synchronization enabled
 */
typedef enum
{
    DMAMUX_SE_DISABLE                   = 0x0,
    DMAMUX_SE_ENABLE                    = 0x1,
} DMAMUX_Se_Enable;

/**
 * @brief Event generation enable
 * @param DMAMUX_EGE_DISABLE Event generation disabled
 * @param DMAMUX_EGE_ENABLE Event generation enabled
 */
typedef enum
{
    DMAMUX_EGE_DISABLE                   = 0x0,
    DMAMUX_EGE_ENABLE                    = 0x1,
} DMAMUX_Ege_Enable;

/**
 * @brief Synchronization overrun interrupt enable
 * @param DMAMUX_SOIE_DISABLE Interrupt disabled
 * @param DMAMUX_SOIE_ENABLE  Interrupt enabled
 */
typedef enum
{
    DMAMUX_SOIE_DISABLE                   = 0x0,
    DMAMUX_SOIE_ENABLE                    = 0x1,
} DMAMUX_Soie_Enable;

/**
 * @brief DMA request generator channel x enable
 * @param DMAMUX_GE_DISABLE DMA request generator channel x disabled
 * @param DMAMUX_GE_ENABLE  DMA request generator channel x enabled
 */
typedef enum
{
    DMAMUX_GE_DISABLE                   = 0x0,
    DMAMUX_GE_ENABLE                    = 0x1,
} DMAMUX_Ge_Enable;

/**
 * @brief Trigger overrun interrupt enable
 * @param DMAMUX_OIE_DISABLE Interrupt on a trigger overrun event occurrence is disabled
 * @param DMAMUX_OIE_ENABLE Interrupt on a trigger overrun event occurrence is enabled
 */
typedef enum
{
    DMAMUX_OIE_DISABLE                   = 0x0,
    DMAMUX_OIE_ENABLE                    = 0x1,
} DMAMUX_Oie_Enable;


#if ((DMAMUX_VERSION ==  0x11) || (DMAMUX_VERSION ==  0x30))


/**
 * @brief Checks a DMAMUX base address.
 * @param base is the base address of the DMAMUX controller.
 * @return true if the base address is valid
 *         false otherwise
 */

__STATIC_INLINE uint8_t
DMAMUX_isBaseValid(uint32_t base)
{
    return (base == DMA_MUX_BASE);
}

#elif ((DMAMUX_VERSION ==  0x12) || (DMAMUX_VERSION ==  0x20))

/**
 * @brief Checks a DMAMUX base address.
 * @param base is the base address of the DMAMUX controller.
 * @return true if the base address is valid
 *         false otherwise
 */
__STATIC_INLINE uint8_t
DMAMUX_isBaseValid(uint32_t base)
{
    return (      
            (base == DMA1_MUX_BASE)
            || (base == DMA2_MUX_BASE)
           );
}

#else

/**
 * @brief Checks a DMAMUX base address.
 * @param base is the base address of the DMAMUX controller.
 * @return true if the base address is valid
 *         false otherwise
 */
__STATIC_INLINE uint8_t
DMAMUX_isBaseValid(uint32_t base)
{
    return (             
            (base == DMA1_MUX_BASE)
            || (base == DMA2_MUX_BASE)
#if (DMAMUX_VERSION != 0x22)
            || (base == ARM_DMA_MUX_BASE) 
#endif               
           );
}
#endif 

/**
 * @brief Set DMA MUX CCRx config by selected for special DMA channel n
 * @param base is the base address of the DMAMUX controller.
 * @param nDmaChan is the DMA channel number
 * @param reg is the DMA MUX CCRx whole configuration
 * @return None
 */
__STATIC_INLINE void DMAMUX_setDmaMuxCcr(uint32_t base, uint32_t nDmaChan, union DMAMUX_CCR_REG reg)
{

    ASSERT(DMAMUX_isBaseValid(base));
    ASSERT(nDmaChan < DMAMUX_CCR_NUMS);    

    
    /* Set the is the DMA MUX CCRx whole configuration */
    HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(nDmaChan)) = reg.all;

}

/**
 * @brief Get the DMA MUX CCRx config by selected for specified DMA channel n.
 * @param base is the base address of the DMAMUX controller.
 * @param nDmaChan is the DMA channel number
 * @param DmaReqId is DMA request id
 * @return DmaReqId is DMA request id
 */
__STATIC_INLINE union DMAMUX_CCR_REG DMAMUX_getDmaMuxCcr(uint32_t base, uint32_t nDmaChan)
{
    union DMAMUX_CCR_REG reg;
      
    ASSERT(DMAMUX_isBaseValid(base));
    ASSERT(nDmaChan < DMAMUX_CCR_NUMS);

    /* Get the is the DMA MUX CCRx whole configuration */
    reg.all = HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(nDmaChan));

    return reg;
}

/**
 * @brief Set DMA MUX sync Id selected for DMA channel n
 * @param base is the base address of the DMAMUX controller
 * @param nDmaChan is the DMA channel number
 * @param syncId is DMA Synchronization identification
 * @return None
 */
__STATIC_INLINE void DMAMUX_setDmaMuxSyncId(uint32_t base, uint32_t nDmaChan, uint32_t syncId)
{

    ASSERT(DMAMUX_isBaseValid(base));
    ASSERT(nDmaChan < DMAMUX_CCR_NUMS);
    ASSERT(syncId <= (DMAMUX_CCRx_SYNCID_M >>  DMAMUX_CCRx_SYNCID_S));
    
    /* Set the SYNC ID for the DMAMUX_CCRx register */
    HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(nDmaChan)) = \
            (HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(nDmaChan)) & (~DMAMUX_CCRx_SYNCID_M)) | \
            ((syncId  << DMAMUX_CCRx_SYNCID_S) & DMAMUX_CCRx_SYNCID_M);

}


/**
 * @brief Get the DMA sync ID for the specified DMA channel n.
 * @param base is the base address of the DMAMUX controller.
 * @param nDmaChan is the DMA channel number
 * @return Returns the DMA Synchronization identification corresponding to the DMA channel.
 */
__STATIC_INLINE uint32_t DMAMUX_getDmaMuxSyncId(uint32_t base, uint32_t nDmaChan)
{
    uint32_t sync_id = 0U;

    ASSERT(DMAMUX_isBaseValid(base));
    ASSERT(nDmaChan < DMAMUX_CCR_NUMS);

    /* Get the SYNC ID from the DMAMUX_CCRx register */
    sync_id = HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(nDmaChan)) & DMAMUX_CCRx_SYNCID_M;
    sync_id >>= DMAMUX_CCRx_SYNCID_S;

    return sync_id;
}


/**
 * @brief Set the numbers of DMA requests selected for DMA channel n to forward to the DMA controller.
 * @param base is the base address of the DMAMUX controller.
 * @param nDmaChan is the DMA channel number.
 * @param DmaReqId is DMA request id
 * @return None.
 */
__STATIC_INLINE void DMAMUX_setDmaMuxNumReqs(uint32_t base, uint32_t nDmaChan, uint32_t numReqs)
{
    ASSERT(DMAMUX_isBaseValid(base));
    ASSERT(nDmaChan < DMAMUX_CCR_NUMS);
    ASSERT(numReqs <=(DMAMUX_CCRx_NBREQ_M >> DMAMUX_CCRx_NBREQ_S));    

    /* Set the Number of DMA Requests to the DMAMUX_CCRx register */
    HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(nDmaChan)) = \
            (HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(nDmaChan)) & (~DMAMUX_CCRx_NBREQ_M)) | \
            ((numReqs << DMAMUX_CCRx_NBREQ_S) & DMAMUX_CCRx_NBREQ_M);
}

/**
 * @brief Get the numbers of DMA requests selected for DMA channel n to forward to the DMA controller.
 * @param base is the base address of the DMAMUX controller.
 * @param nDmaChan is the DMA channel number
 * @param DmaReqId is DMA request id
 * @return Returns the numbers of DMA requests corresponding to the DMA channel to forward to the DMA controller.
 */
__STATIC_INLINE uint32_t DMAMUX_getDmaMuxNumReqs(uint32_t base, uint32_t nDmaChan)
{
    uint32_t numReqs = 0U;
    
    ASSERT(DMAMUX_isBaseValid(base));
    ASSERT(nDmaChan < DMAMUX_CCR_NUMS);

    /* Set the Number of DMA Requests to the DMAMUX_CCRx register */
    numReqs = HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(nDmaChan)) & DMAMUX_CCRx_NBREQ_M;
    numReqs >>= DMAMUX_CCRx_NBREQ_S;

    return numReqs;
}

/**
 * @brief Set Synchronization polarity type.
 * @param base is the base address of the DMAMUX controller.
 * @param nDmaChan is the DMA channel number
 * @return None.
 */
__STATIC_INLINE void
DMAMUX_setDmaMuxSyncPolarity(uint32_t base, uint32_t nDmaChan, DMAMUX_Pol_Polarity spol)
{
    ASSERT(DMAMUX_isBaseValid(base));
    ASSERT(nDmaChan < DMAMUX_CCR_NUMS);
    ASSERT(spol <= (DMAMUX_CCRx_SPOL_M >> DMAMUX_CCRx_SPOL_S));

    /* Set the SPOL for the DMAMUX_CCRx register */
    HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(nDmaChan)) = \
            (HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(nDmaChan)) & (~DMAMUX_CCRx_SPOL_M))  | \
            ((spol << DMAMUX_CCRx_SPOL_S) & DMAMUX_CCRx_SPOL_M);
}

/**
 * @brief Enable Synchronization.
 * @param base is the base address of the DMAMUX controller.
 * @param nDmaChan is the DMA channel number.
 * @return None.
 */
__STATIC_INLINE void
DMAMUX_enableDmaMuxSync(uint32_t base, uint32_t nDmaChan)
{
    ASSERT(DMAMUX_isBaseValid(base));
    ASSERT(nDmaChan < DMAMUX_CCR_NUMS);

    /* Eanble the SE bit field in the DMAMUX_CCRx register orresponding to the DMA channel. */
    HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(nDmaChan)) |= DMAMUX_CCRx_SE;    
}

/**
 * @brief Disable Synchronization.
 * @param base is the base address of the DMAMUX controller.
 * @param nDmaChan is the DMA channel number
 * @return None.
 */
__STATIC_INLINE void
DMAMUX_disableDmaMuxSync(uint32_t base, uint32_t nDmaChan)
{
    ASSERT(DMAMUX_isBaseValid(base));
    ASSERT(nDmaChan < DMAMUX_CCR_NUMS);

    /* Eanble the SE bit field in the DMAMUX_CCRx register orresponding to the DMA channel. */
    HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(nDmaChan)) &= ~(DMAMUX_CCRx_SE);
}

/**
 * @brief Enable Event generation
 * @param base is the base address of the DMAMUX controller
 * @param nDmaChan is the DMA channel number
 * @return None
 */
__STATIC_INLINE void
DMAMUX_enableDmaMuxEventGen(uint32_t base, uint32_t nDmaChan)
{

    ASSERT(DMAMUX_isBaseValid(base));
    ASSERT(nDmaChan < DMAMUX_CCR_NUMS);

    /* Eanble the SE bit field in the DMAMUX_CCRx register orresponding to the DMA channel. */
    HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(nDmaChan)) |= DMAMUX_CCRx_EGE;
}

/**
 * @brief Disable Event generation.
 * @param base is the base address of the DMAMUX controller.
 * @param nDmaChan is the DMA channel number
 * @return None.
 */
__STATIC_INLINE void
DMAMUX_disableDmaMuxEventGen(uint32_t base, uint32_t nDmaChan)
{

    ASSERT(DMAMUX_isBaseValid(base));
    ASSERT(nDmaChan < DMAMUX_CCR_NUMS);
    
    /* Eanble the SE bit field in the DMAMUX_CCRx register orresponding to the DMA channel. */
    HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(nDmaChan)) &= ~(DMAMUX_CCRx_EGE);
}

/**
 * @brief Set DMA MUX request Id selected for DMA channel n
 * @param base is the base address of the DMAMUX controller.
 * @param nDmaChan is the DMA channel number
 * @param DmaReqId is DMA request id
 * @return None.
 */
__STATIC_INLINE void 
DMAMUX_setDmaMuxReqId(uint32_t base, uint32_t nDmaChan, DMAMUX_ReqId_Type DmaReqId)
{
    ASSERT(DMAMUX_isBaseValid(base));
    ASSERT(nDmaChan < DMAMUX_CCR_NUMS);
    
    /* Set the DMAReqId for the DMAMUX_CCRx register */
    HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(nDmaChan)) = \
            (HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(nDmaChan)) & (~DMAMUX_CCRx_REQID_M)) | \
            ((DmaReqId << DMAMUX_CCRx_REQID_S) & DMAMUX_CCRx_REQID_M);

}

/**
 * @brief Get the DMA request ID for the specified DMA channel n.
 * @param base is the base address of the DMAMUX controller.
 * @param nDmaChan is the DMA channel number
 * @param DmaReqId is DMA request id
 * @return Returns the DMA request ID corresponding to the DMA channel.
 */
__STATIC_INLINE DMAMUX_ReqId_Type 
DMAMUX_getDmaMuxReqId(uint32_t base, uint32_t nDmaChan)
{
    DMAMUX_ReqId_Type reqid;

    ASSERT(DMAMUX_isBaseValid(base));
    ASSERT(nDmaChan < DMAMUX_CCR_NUMS);

    reqid = (DMAMUX_ReqId_Type)((HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(nDmaChan)) & DMAMUX_CCRx_REQID_M) >> DMAMUX_CCRx_REQID_S);

    return reqid;
}

/**
 * @brief Set DMA MUX RGCRx config by selected for the DMA Request Generator channel. n
 * @param base is the base address of the DMAMUX controller.
 * @param nDmaReqGen is the DMA Request Generator channel
 * @param reg is the DMA MUX RGCRx whole config corresponding to the DMA Request Generator channel.
 * @return None.
 */
__STATIC_INLINE void 
DMAMUX_setDmaMuxRgcr(uint32_t base, uint32_t nDmaReqGen, union DMAMUX_RGCR_REG reg)
{       
    ASSERT(DMAMUX_isBaseValid(base));
    ASSERT(nDmaReqGen < DMAMUX_RGCR_NUMS);

    HWREG(base + DMAMUX_RGCR_OFFSET + DMAMUX_RGCRx(nDmaReqGen)) = reg.all;
}

/**
 * @brief Get the DMA MUX RGCRx config by selected for the DMA Request Generator channel n.
 * @param base is the base address of the DMAMUX controller.
 * @param nDmaReqGen is the DMA Request Generator channel.
 * @return DMAMUX_RGCR_REG the DMA MUX RGCRx whole configuration to the DMA Request Generator channel.
 */
__STATIC_INLINE union DMAMUX_RGCR_REG 
DMAMUX_getDmaMuxRgcr(uint32_t base, uint32_t nDmaReqGen)
{
    union DMAMUX_RGCR_REG reg;
    
    ASSERT(DMAMUX_isBaseValid(base));
    ASSERT(nDmaReqGen < DMAMUX_RGCR_NUMS);

    reg.all = HWREG(base + DMAMUX_RGCR_OFFSET + DMAMUX_RGCRx(nDmaReqGen));

    return reg;
}

/**
 * @brief Set the numbers of DMA requests selected for DMA channel n to be generated after a trigger event.
 * @param base is the base address of the DMAMUX controller.
 * @param nDmaReqGen is the DMA Request Generator channel
 * @param numreqsGen is the numbers of DMA requests corresponding to the DMA channel to be generated after a trigger event.
 * @return None.
 */
__STATIC_INLINE void 
DMAMUX_setDmaMuxGenNumReqs(uint32_t base, uint32_t nDmaReqGen, uint32_t numreqsGen)
{  
    uint32_t regOffset = 0U;

    ASSERT(DMAMUX_isBaseValid(base));
    ASSERT(nDmaReqGen < DMAMUX_RGCR_NUMS);
    ASSERT(numreqsGen <= (DMAMUX_RGCRx_GNBREQ_M >> DMAMUX_RGCRx_GNBREQ_S));    

    regOffset = DMAMUX_RGCRx(nDmaReqGen);
    
    /* Set the Number of DMA Requests to the DMAMUX_CCRx register */
    HWREG(base + DMAMUX_RGCR_OFFSET + regOffset) = \
          (HWREG(base + DMAMUX_RGCR_OFFSET + regOffset) & (~DMAMUX_RGCRx_GNBREQ_M)) | \
          ((numreqsGen << DMAMUX_RGCRx_GNBREQ_S) & DMAMUX_RGCRx_GNBREQ_M);

}

/**
 * @brief Get the numbers of DMA requests selected for DMA channel n to be generated after a trigger event.
 * @param base is the base address of the DMAMUX controller.
 * @param nDmaReqGen is the DMA Request Generator channel
 * @return Returns the numbers of DMA requests corresponding to the DMA channel to be generated after a trigger event.
 */
__STATIC_INLINE uint32_t 
DMAMUX_getDmaMuxGenNumReqs(uint32_t base, uint32_t nDmaReqGen)
{
    uint32_t numreqsGen = 0U;
    
    ASSERT(DMAMUX_isBaseValid(base));
    ASSERT(nDmaReqGen < DMAMUX_RGCR_NUMS);
    
    /* Set the Number of DMA Requests to the DMAMUX_CCRx register */
    numreqsGen = HWREG(base + DMAMUX_RGCR_OFFSET + DMAMUX_RGCRx(nDmaReqGen)) & DMAMUX_RGCRx_GNBREQ_M;
    numreqsGen >>= DMAMUX_RGCRx_GNBREQ_S;

    return numreqsGen;
}

/**
 * @brief Set the DMA request generator trigger polarity type.
 * @param base is the base address of the DMAMUX controller.
 * @param nDmaReqGen is the DMA Request Generator channel
 * @param gpol is the DMA request generator trigger polarity type.
 * @return None.
 */
__STATIC_INLINE void
DMAMXU_setDmaMuxGenTrigPolarity(uint32_t base, uint32_t nDmaReqGen, DMAMUX_Pol_Polarity gpol)
{
    uint32_t regOffset = 0U;

    ASSERT(DMAMUX_isBaseValid(base));
    ASSERT(nDmaReqGen < DMAMUX_RGCR_NUMS);

    regOffset = DMAMUX_RGCRx(nDmaReqGen);
    
    /* Set the SPOL for the DMAMUX_CCRx register */
    HWREG(base + DMAMUX_RGCR_OFFSET +regOffset) = \
        (HWREG(base + DMAMUX_RGCR_OFFSET + regOffset) & (~DMAMUX_RGCRx_GPOL_M)) | \
        ((gpol << DMAMUX_RGCRx_GPOL_S) & DMAMUX_RGCRx_GPOL_M);
}

/**
 * @brief Enable the DMA request generator channel n
 * @param base is the base address of the DMAMUX controller.
 * @param nDmaReqGen is the DMA Request Generator channel
 * @return None.
 */
__STATIC_INLINE void
DMAMUX_enableDmaMuxReqGenChn(uint32_t base, uint32_t nDmaReqGen)
{

    ASSERT(DMAMUX_isBaseValid(base));
    ASSERT(nDmaReqGen < DMAMUX_RGCR_NUMS);

    /* Eanble the GE bit field in the DMAMUX_CCRx register orresponding to the DMA channel. */
    HWREG(base + DMAMUX_RGCR_OFFSET + DMAMUX_CCRx(nDmaReqGen)) |= DMAMUX_RGCRx_GE;
}

/**
 * @brief Disable the DMA request generator channel 
 * @param base is the base address of the DMAMUX controller.
 * @param nDmaReqGen is the DMA Request Generator channel
 * @return None.
 */
__STATIC_INLINE void
DMAMUX_disableDmaMuxReqGenChn(uint32_t base, uint32_t nDmaReqGen)
{
    ASSERT(DMAMUX_isBaseValid(base));
    ASSERT(nDmaReqGen < DMAMUX_RGCR_NUMS);

    /* Eanble the GE bit field in the DMAMUX_CCRx register orresponding to the DMA channel. */
    HWREG(base + DMAMUX_RGCR_OFFSET + DMAMUX_RGCRx(nDmaReqGen)) &= ~(DMAMUX_RGCRx_GE);
}

/**
 * @brief Set the DMA trigger source Id for DMA channel p and enable GE
 * @param base is the base address of the DMAMUX controller.
 * @param nDmaReqGen is the DMA Request Generator channel
 * @param DmaTrigId is DMA trigger source ID
 * @return None.
 */
__STATIC_INLINE void 
DMAMUX_setDmaMuxTrigId(uint32_t base, uint32_t nDmaReqGen, DMAMUX_TrigId_Type DmaTrigId)
{
    uint32_t regOffset = 0U;
   
    ASSERT(DMAMUX_isBaseValid(base));
    ASSERT(nDmaReqGen < DMAMUX_RGCR_NUMS);
    ASSERT(DmaTrigId <= (DMAMUX_RGCRx_TRGID_M >> DMAMUX_RGCRx_TRGID_S));

    regOffset = DMAMUX_RGCRx(nDmaReqGen);
    
    /* Set the DMA trigger source Id for DMA channel p */
    HWREG(base + DMAMUX_RGCR_OFFSET + regOffset) = \
        (HWREG(base + DMAMUX_RGCR_OFFSET + regOffset) & (~DMAMUX_RGCRx_TRGID_M)) | \
        ((DmaTrigId << DMAMUX_RGCRx_TRGID_S) & DMAMUX_RGCRx_TRGID_M);
}

/**
 * @brief Get the DMA trigger source ID for the specified DMA Request Generator channel p.
 * @param base is the base address of the DMAMUX controller.
 * @param nDmaReqGen is the DMA Request Generator channel
 * @return Returns the DMA trigger source ID corresponding to the DMA Request Generator channel.
 */
__STATIC_INLINE DMAMUX_TrigId_Type 
DMAMUX_getDmaMuxTridId(uint32_t base, uint32_t nDmaReqGen)
{
    DMAMUX_TrigId_Type trigid;
  
    ASSERT(DMAMUX_isBaseValid(base));
    ASSERT(nDmaReqGen < DMAMUX_RGCR_NUMS);

    trigid = (DMAMUX_TrigId_Type)(HWREG(base + DMAMUX_RGCR_OFFSET + DMAMUX_RGCRx(nDmaReqGen)) & DMAMUX_RGCRx_TRGID_M);

    return trigid;
}

/**
 * @brief config the DMA mux for DMA channel n
 * @param base is the base address of the DMAMUX controller.
 * @param nDmaChan is the DMA channel number
 * @param DmaReqId is the DMA request
 * @return None.
 */
__STATIC_INLINE void 
DMAMUX_configDmaMux(uint32_t base, uint32_t nDmaChan, DMAMUX_ReqId_Type DmaReqId)
{
    uint8_t i       = 0U;
    uint32_t ccr    = 0U;
    uint32_t rgcr   = 0U;
    uint32_t regOffset = 0U;

    ASSERT(DMAMUX_isBaseValid(base));
    ASSERT(nDmaChan < DMAMUX_CCR_NUMS);

    /* Get the DMA MUX CCRx config by selected for specified DMA channel n. */
    ccr = HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(nDmaChan));

    if (DmaReqId < DMAMUX_ReqId_max)
    {

        HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(nDmaChan)) = ((ccr & (~DMAMUX_CCRx_REQID_M)) | DmaReqId);
    }
    else
    { 
#if DMAMUX_RGCR_P2P_SUPPORT    
        
        /* If the ID is greater than or equal to DMAMUX_ReqId_max, DMA requests need to be generated 
        using a request generator, select an available generator */
        for (i = 0; i < DMAMUX_RGCR_NUMS; i++)
        {
            regOffset = DMAMUX_RGCRx(i);
            rgcr = HWREG(base + DMAMUX_RGCR_OFFSET + regOffset);

            if ((rgcr & (1 << 16)) == 0)
            {
                /* Set the DMAReq_Id for the DMAMUX_CCR[n] register to the generator ID, i.e., i */
                HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(nDmaChan)) = (ccr & (~DMAMUX_CCRx_REQID_M)) | \
                        (i + DMAMUX_ReqId_dma_gen_out0);

                /* Set the DMAMUX_RGCR[n] register, SIG_ID = DmaReqId - DMAMUX_ReqId_max; GE = 1, enable; GPOL = 3, 
                rising edge-triggered */
                HWREG(base + DMAMUX_RGCR_OFFSET + regOffset) =
                    (rgcr & (~(DMAMUX_RGCRx_GPOL_M | DMAMUX_RGCRx_TRGID_M))) | \
                    (DmaReqId - DMAMUX_ReqId_max) | DMAMUX_RGCRx_GE | \
                    (DMAMUX_POL_RISING << DMAMUX_RGCRx_GPOL_S);
                
                break;
            }
        }
#else
        /*
         RGCR0 used for channel0 by force, when nDmaChan is 0 or 1
         RGCR1 used for channel1 by force, when nDmaChan is 2 or 3
         .........................................................
         RGCR7 used for cahngel7 by force, when nDmaChan is 14 or 15 */
        regOffset = DMAMUX_RGCRx(nDmaChan/2U);
        rgcr = HWREG(base + DMAMUX_RGCR_OFFSET + regOffset);
        
        /* Set the DMAReq_Id for the DMAMUX_CCR[n] register to the generator ID, i.e., nDmaChan/2 */
        HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(nDmaChan)) = (ccr & (~DMAMUX_CCRx_REQID_M)) | \
                (nDmaChan/2 + DMAMUX_ReqId_dma_gen_out0);

        /* Set the DMAMUX_RGCR[n] register, SIG_ID = DmaReqId - DMAMUX_ReqId_max; GE = 1, enable; GPOL = 3, 
        rising edge-triggered */
        HWREG(base + DMAMUX_RGCR_OFFSET + regOffset) =
            (rgcr & (~(DMAMUX_RGCRx_GPOL_M | DMAMUX_RGCRx_TRGID_M))) | \
            (DmaReqId - DMAMUX_ReqId_max) | DMAMUX_RGCRx_GE | \
            (DMAMUX_POL_RISING << DMAMUX_RGCRx_GPOL_S);

#endif
    }
}

/**
 * @brief De-config the DMA mux for DMA channel n
 * @param base is the base address of the DMAMUX controller.
 * @param nDmaChan is the DMA hardware handkshake INF ID.
 * @param DmaReqId is the DMA request
 * @return None.
 */
__STATIC_INLINE void 
DMAMUX_deConfDmaMux(uint32_t base, uint32_t nDmaChan, DMAMUX_ReqId_Type DmaReqId)
{
    uint8_t i = 0;
    uint32_t ccr = 0U; 
    uint32_t rgcr = 0U;

    ASSERT(DMAMUX_isBaseValid(base));
    ASSERT(nDmaChan < DMAMUX_CCR_NUMS);

    if (DmaReqId < DMAMUX_ReqId_max)
    {
        /* If the ID is less than DMAMUX_ReqId_max, it can reset CCR register as zero directly. */
        HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(nDmaChan)) = 0U;
    }
    else
    {
        /* Get the DMA MUX CCRx config by selected for specified DMA channel n. */
        ccr = HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(nDmaChan));

        /* Get the correct RGCRx index need to be cleared. */
        i = (ccr & DMAMUX_CCRx_REQID_M) >> DMAMUX_CCRx_REQID_S;

        /* delete the offset gen_out0. */
        i -= DMAMUX_ReqId_dma_gen_out0;
        
        HWREG(base + DMAMUX_RGCR_OFFSET + DMAMUX_RGCRx(i)) = 0U;
    }
}




#if (DMAMUX_VERSION == 0x11)
/**
 * @brief Clear the DMA MUX overrun intterupt status
 * @param base is the base address of the DMAMUX controller.
 * @return None.
 */
__STATIC_INLINE void 
DMAMUX_clearDmaMuxSofInt(uint32_t base)
{
    ASSERT(DMAMUX_isBaseValid(base));

    HWREG(base + DMAMUX_INT_OFFSET + DMAMUX_O_SOF_INT_RAW) = DMAMUX_SOF_INT_RAW_M;
}

/**
 * @brief Get the DMA MUX overrun intterupt status
 * @param base is the base address of the DMAMUX controller.
 * @return Returns the DMA MUX overrun intterupt status
 */

__STATIC_INLINE uint32_t 
DMAMUX_getDmaMuxSofInt(uint32_t base)
{
    ASSERT(DMAMUX_isBaseValid(base));

    return HWREG(base + DMAMUX_INT_OFFSET + DMAMUX_O_SOF_INT) & DMAMUX_SOF_INT_M;
}

/**
 * @brief Set the DMA MUX overrun intterupt mask to enable/disable interrupt.
 * @param base is the base address of the DMAMUX controller.
 * @param mask is the mask value to set mask field in CC register.
 * @return None.
 */
__STATIC_INLINE void 
DMAMUX_setDmaMuxSofIntMask(uint32_t base, uint32_t mask)
{
    uint8_t i = 0;
    uint32_t ccr = 0; 

    ASSERT(DMAMUX_isBaseValid(base));

    /* The current IP lacks a MASK register. The pertinent interrupt enable fields are situated 
    within the SOIE field in each CCRx group of registers. Therefore, it is essential to 
    traverse and implement the masking functionality in the program. */
    for (i = 0; i < DMAMUX_CCR_NUMS; i++)
    {
        ccr = HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(i));
        if ((BIT(i) & mask) != 0U)
        {
            ccr |= DMAMUX_CCRx_SOIE;
        }
        else
        {
            ccr &= ~DMAMUX_CCRx_SOIE;
        }
        HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(i)) = ccr;       
    }
}

/**
 * @brief Get the DMA MUX overrun intterupt mask
 * @param base is the base address of the DMAMUX controller.
 * @return Returns the value of mask register.
 */
__STATIC_INLINE uint32_t 
DMAMUX_getDmaMuxSofIntMask(uint32_t base)
{
    uint8_t i = 0;
    uint32_t ccr = 0;
    uint32_t mask = 0U;

    ASSERT(DMAMUX_isBaseValid(base));

    /* The current IP lacks a MASK register. The pertinent interrupt enable fields are situated 
    within the SOIE field in each CCRx group of registers. Therefore, it is essential to 
    traverse and implement the masking functionality in the program.*/
    for (i = 0; i < DMAMUX_CCR_NUMS; i++)
    {
        ccr = HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(i));
        if ((ccr & DMAMUX_CCRx_SOIE) != 0U)
        {
            mask |= BIT(i);
        }   
    }

    return mask;
}

/**
 * @brief Clear the DMA MUX request generator interrupt status
 * @param base is the base address of the DMAMUX controller.
 * @return None.
 */
__STATIC_INLINE void 
DMAMUX_clearDmaMuxOfInt(uint32_t base)
{
    ASSERT(DMAMUX_isBaseValid(base));

    HWREG(base + DMAMUX_INT_OFFSET + DMAMUX_O_OF_INT_RAW) = DMAMUX_OF_INT_RAW_M;
}

/**
 * @brief Get the DMA MUX orequest generator interrupt status
 * @param base is the base address of the DMAMUX controller.
 * @return Returns the DMA MUX request generator interrupt status
 */
__STATIC_INLINE uint32_t 
DMAMUX_getDmaMuxOfInt(uint32_t base)
{
    ASSERT(DMAMUX_isBaseValid(base));

    return HWREG(base + DMAMUX_INT_OFFSET + DMAMUX_O_OF_INT) & DMAMUX_OF_INT_M;    
}

/**
 * @brief Set the DMA MUX request generator interrupt MASK
 * @param base is the base address of the DMAMUX controller.
 * @param mask is the mask value to set mask register.
 * @return None
 */
__STATIC_INLINE void 
DMAMUX_setDmaMuxOfIntMask(uint32_t base, uint32_t mask)
{
    uint8_t i = 0;
    uint32_t rgcr = 0; 

    ASSERT(DMAMUX_isBaseValid(base));

    /* The current IP lacks a MASK register. The pertinent interrupt enable fields are situated 
    within the SOIE field in each CCRx group of registers. Therefore, it is essential to 
    traverse and implement the masking functionality in the program. */
    for (i = 0; i < DMAMUX_RGCR_NUMS; i++)
    {
        rgcr = HWREG(base + DMAMUX_RGCR_OFFSET + DMAMUX_RGCRx(i));
        if ((BIT(i) & mask) != 0U)
        {
            rgcr |= DMAMUX_RGCRx_OIE;
        }
        else
        {
            rgcr &= ~DMAMUX_RGCRx_OIE;
        }
        HWREG(base + DMAMUX_RGCR_OFFSET + DMAMUX_RGCRx(i)) = rgcr;       
    }
}

/**
 * @brief base Get the DMA MUX request generator interrupt MASK
 * @param base is the base address of the DMAMUX controller.
 * @return Returns the value of mask register.
 */
__STATIC_INLINE uint32_t 
DMAMUX_getDmaMuxOfIntMask(uint32_t base)
{
    uint8_t i = 0;
    uint32_t rgcr = 0;
    uint32_t mask = 0U;

    ASSERT(DMAMUX_isBaseValid(base));

    /* The current IP lacks a MASK register. The pertinent interrupt enable fields are situated 
    within the SOIE field in each CCRx group of registers. Therefore, it is essential to 
    traverse and implement the masking functionality in the program. */
    for (i = 0; i < DMAMUX_RGCR_NUMS; i++)
    {
        rgcr = HWREG(base + DMAMUX_RGCR_OFFSET + DMAMUX_RGCRx(i));
        if ((rgcr & DMAMUX_RGCRx_OIE) != 0U)
        {
            mask |= BIT(i);
        }   
    }

    return mask;

}

/**
 * @brief Reset the DMA mux module.
 * @param base is the base address of the DMAMUX controller.
 * @return None.
 */
__STATIC_INLINE void 
DMAMUX_resetDmaMux(uint32_t base)
{
    uint8_t i = 0;
    uint32_t ccr = 0U; 
    uint32_t rgcr = 0U;

    ASSERT(DMAMUX_isBaseValid(base));

    for (i= 0; i < DMAMUX_CCR_NUMS; i++)
    {
        HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(i)) = 0U;
    }

    for (i= 0; i < DMAMUX_RGCR_NUMS; i++)
    {
        
        HWREG(base + DMAMUX_RGCR_OFFSET + DMAMUX_RGCRx(i)) = 0U;

    }

    DMAMUX_clearDmaMuxSofInt(base);
    DMAMUX_clearDmaMuxOfInt(base);
}


#elif (DMAMUX_VERSION == 0x20)

/**
 * @brief Clear the DMA MUX  overrun intterup raw status
 * @param base is the base address of the DMAMUX controller.
 * @return None.
 */
__STATIC_INLINE void 
DMAMUX_clearDmaMuxSofIntRaw(uint32_t base)
{
    HWREG(base + DMAMUX_INT_OFFSET+ DMAMUX_O_SOF_INT_RAW) = 0xFFFFFFFF;
}

/**
 * @brief Get the DMA MUX overrun intterupt Extention raw status
 * @param base is the base address of the DMAMUX controller.
 * @return Returns the DMA MUX overrun intterupt Extention raw status
 */
__STATIC_INLINE uint32_t 
DMAMUX_getDmaMuxSofIntRaw(uint32_t base)
{
    return HWREG(base + DMAMUX_INT_OFFSET + DMAMUX_O_SOF_INT_RAW);
}

/**
 * @brief Clear the DMA MUX overrun intterupt raw Extention status
 * @param base is the base address of the DMAMUX controller.
 * @return None
 */
__STATIC_INLINE void 
DMAMUX_clearDmaMuxSofIntRawExt(uint32_t base)
{
    HWREG(base + DMAMUX_INT_OFFSET + DMAMUX_O_SOF_INT_RAW_EXT) = 0xFFFFFFFF;
}

/**
 * @brief Get the DMA MUX overrun intterupt Extention raw status
 * @param base is the base address of the DMAMUX controller.
 * @return Returns the DMA MUX overrun intterupt Extention raw status
 */
__STATIC_INLINE uint32_t 
DMAMUX_getDmaMuxSofIntRawExt(uint32_t base)
{
    return HWREG(base + DMAMUX_INT_OFFSET + DMAMUX_O_SOF_INT_RAW_EXT);
}

/**
 * @brief Get the DMA MUX overrun intterupt status
 * @param base is the base address of the DMAMUX controller.
 * @return Returns the DMA MUX overrun intterupt status
 */
__STATIC_INLINE uint32_t 
DMAMUX_getDmaMuxSofInt(uint32_t base)
{
    return HWREG(base + DMAMUX_INT_OFFSET + DMAMUX_O_SOF_INT);
}

/**
 * @brief Get the DMA MUX overrun intterupt Extention status
 * @param base is the base address of the DMAMUX controller.
 * @return Returns the DMA MUX overrun intterupt Extention status
 */
__STATIC_INLINE uint32_t 
DMAMUX_getDmaMuxSofIntExt(uint32_t base)
{
    return HWREG(base + DMAMUX_INT_OFFSET + DMAMUX_O_SOF_INT_EXT);
}

/**
 * @brief Clear the DMA MUX request generator interrupt raw status
 * @param base is the base address of the DMAMUX controller
 * @return None
 */
__STATIC_INLINE void 
DMAMUX_clearDmaMuxOfIntRaw(uint32_t base)
{
    HWREG(base + DMAMUX_INT_OFFSET + DMAMUX_O_OF_INT_RAW) = 0xff;
}

/**
 * @brief Get the DMA MUX request generator interrupt raw status
 * @param base is the base address of the DMAMUX controller.
 * @return Returns the DMA MUX request generator interrupt raw status
 */
__STATIC_INLINE uint32_t 
DMAMUX_getDmaMuxOfIntRaw(uint32_t base)
{
    return (HWREG(base + DMAMUX_INT_OFFSET + DMAMUX_O_OF_INT_RAW) >> 0) & 0xff;
}

/**
 * @brief Get the DMA MUX request generator interrupt status
 * @param base is the base address of the DMAMUX controller.
 * @return Returns the DMA MUX request generator interrupt status
 */
__STATIC_INLINE uint32_t 
DMAMUX_getDmaMuxOfInt(uint32_t base)
{
    return (HWREG(base + DMAMUX_INT_OFFSET + DMAMUX_O_OF_INT) >> 0) & 0xff;
}

/**
 * @brief Reset the DMA mux module.
 * @param base is the base address of the DMAMUX controller.
 * @return None.
 */
__STATIC_INLINE void 
DMAMUX_resetDmaMux(uint32_t base)
{
    uint8_t i = 0;
    uint32_t ccr = 0U; 
    uint32_t rgcr = 0U;

    ASSERT(DMAMUX_isBaseValid(base));

    for (i= 0; i < DMAMUX_CCR_NUMS; i++)
    {
        HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(i)) = 0U;
    }

    for (i= 0; i < DMAMUX_RGCR_NUMS; i++)
    {
        
        HWREG(base + DMAMUX_RGCR_OFFSET + DMAMUX_RGCRx(i)) = 0U;

    }

    DMAMUX_clearDmaMuxSofIntRaw(base);
    DMAMUX_clearDmaMuxSofIntRawExt(base);
    DMAMUX_clearDmaMuxOfIntRaw(base);
}

#else /* #if (DMAMUX_VERSION == 0x12 || DMAMUX_VERSION == 0x21 || DMAMUX_VERSION == 0x0022) */

/**
 * @brief Clear the DMA MUX overrun intterupt status
 * @param base is the base address of the DMAMUX controller.
 * @return None.
 */
__STATIC_INLINE void 
DMAMUX_clearDmaMuxSofInt(uint32_t base)
{
    ASSERT(DMAMUX_isBaseValid(base));

    /* Clear the DMA MUX overrun intterupt status */
    HWREG(base + DMAMUX_INT_OFFSET + DMAMUX_O_SOF_INT_RAW) = DMAMUX_SOF_INT_RAW_M;
}

/**
 * @brief Get the DMA MUX overrun intterupt status
 * @param base is the base address of the DMAMUX controller.
 * @return Returns the DMA MUX overrun intterupt status
 */
__STATIC_INLINE uint32_t 
DMAMUX_getDmaMuxSofInt(uint32_t base)
{
    ASSERT(DMAMUX_isBaseValid(base));

    /* Get the DMA MUX overrun intterupt status */
    return HWREG(base + DMAMUX_INT_OFFSET + DMAMUX_O_SOF_INT) & DMAMUX_SOF_INT_M;
}

/**
 * @brief Set the DMA MUX overrun intterupt mask
 * @param base is the base address of the DMAMUX controller.
 * @param mask is the mask value to set mask register.
 * @return None.
 */
__STATIC_INLINE void 
DMAMUX_setDmaMuxSofIntMask(uint32_t base, uint32_t mask)
{
    ASSERT(DMAMUX_isBaseValid(base));

    /* Set the DMA MUX overrun intterupt setting. */
    HWREG(base + DMAMUX_INT_OFFSET + DMAMUX_O_SOF_INT_MASK) = mask & DMAMUX_SOF_INT_MASK_M;
}

/**
 * @brief Get the DMA MUX overrun intterupt mask
 * @param base is the base address of the DMAMUX controller.
 * @return the value of mask register
 */
__STATIC_INLINE uint32_t 
DMAMUX_getDmaMuxSofIntMask(uint32_t base)
{
    ASSERT(DMAMUX_isBaseValid(base));

    /* Get the DMA MUX overrun intterupt setting. */
    return HWREG(base + DMAMUX_INT_OFFSET + DMAMUX_O_SOF_INT_MASK) & DMAMUX_SOF_INT_MASK_M;
}

/**
 * @brief Clear the DMA MUX request generator interrupt status
 * @param base is the base address of the DMAMUX controller.
 * @return None.
 */
__STATIC_INLINE void 
DMAMUX_clearDmaMuxOfInt(uint32_t base)
{
    ASSERT(DMAMUX_isBaseValid(base));

    /* Clear the DMA MUX orequest generator interrupt status */
    HWREG(base + DMAMUX_INT_OFFSET + DMAMUX_O_OF_INT_RAW) = DMAMUX_OF_INT_RAW_M;
}

/**
 * @brief Get the DMA MUX request generator interrupt status
 * @param base is the base address of the DMAMUX controller.
 * @return Returns the DMA MUX request generator interrupt status
 */
__STATIC_INLINE uint32_t 
DMAMUX_getDmaMuxOfInt(uint32_t base)
{
    ASSERT(DMAMUX_isBaseValid(base));

    /* Get the DMA MUX orequest generator interrupt status */
    return HWREG(base + DMAMUX_INT_OFFSET + DMAMUX_O_OF_INT) & DMAMUX_OF_INT_M;
}

/**
 * @brief Set the DMA MUX request generator interrupt MASK
 * @param base is the base address of the DMAMUX controller.
 * @param mask is the mask value to set mask register.
 * @return None.
 */
__STATIC_INLINE void 
DMAMUX_setDmaMuxOfIntMask(uint32_t base, uint32_t mask)
{
    ASSERT(DMAMUX_isBaseValid(base));

    /* Set the DMA MUX request generator interrupt setting. */
    HWREG(base + DMAMUX_INT_OFFSET + DMAMUX_O_OF_INT_MASK) = mask & DMAMUX_OF_INT_MASK_M;
}

/**
 * @brief Get the DMA MUX request generator interrupt MASK
 * @param base is the base address of the DMAMUX controller.
 * @return Returns the value of mask register.
 */
__STATIC_INLINE uint32_t 
DMAMUX_getDmaMuxOfIntMask(uint32_t base)
{
    ASSERT(DMAMUX_isBaseValid(base));

    /* Get the DMA MUX request generator interrupt setting. */
    return HWREG(base + DMAMUX_INT_OFFSET + DMAMUX_O_OF_INT_MASK) & DMAMUX_OF_INT_MASK_M;
}

/**
 * @brief Reset the DMA mux module.
 * @param base is the base address of the DMAMUX controller.
 * @return None.
 */
__STATIC_INLINE void 
DMAMUX_resetDmaMux(uint32_t base)
{
    uint8_t i = 0;
    uint32_t ccr = 0U; 
    uint32_t rgcr = 0U;

    ASSERT(DMAMUX_isBaseValid(base));

    for (i= 0; i < DMAMUX_CCR_NUMS; i++)
    {
        HWREG(base + DMAMUX_CCR_OFFSET + DMAMUX_CCRx(i)) = 0U;
    }

    for (i= 0; i < DMAMUX_RGCR_NUMS; i++)
    {
        HWREG(base + DMAMUX_RGCR_OFFSET + DMAMUX_RGCRx(i)) = 0U;
    }

    DMAMUX_clearDmaMuxSofInt(base);
    DMAMUX_clearDmaMuxOfInt(base);
}

#endif /* #if (DMAMUX_VERSION == 0x11) */

#ifdef __cplusplus
}
#endif

#endif // _DMAMUX_H_
