/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    epwm_db_et.h
*   @brief   
*
*   commit history
*   20240804, Jason, initial creation
*/

#ifndef DEVICE_DRIVERLIB_EPWM_DB_ET_H_
#define DEVICE_DRIVERLIB_EPWM_DB_ET_H_

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "inc/hw_types.h"

#include "gs32_version.h"
#include "inc/hw_memmap.h"

#include "inc/hw_epwm.h"
#include "inc/hw_sysctl_ahb.h"
#include "epwm_define.h"
#include "debug.h"




//
// Dead Band Module related APIs
//

/**
 *  @brief Sets Dead Band signal output swap mode.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] output is the ePWM Dead Band output.
 *  @param [in] enableSwapMode is the output swap mode.
 *  @remarks
 *  This function sets up the output signal swap mode. For example if the
 *  output variable is set to EPWM_DB_OUTPUT_A and enableSwapMode is true, then
 *  the ePWM A output gets its signal from the ePWM B signal path. Valid values
 *  for the input variables are:
 *   - output
 *       - EPWM_DB_OUTPUT_A   - ePWM output A
 *       - EPWM_DB_OUTPUT_B   - ePWM output B
 *   - enableSwapMode
 *       - true     - the output is swapped
 *       - false    - the output and the signal path are the same.
*/
__STATIC_INLINE void
EPWM_setDeadBandOutputSwapMode(uint32_t base, EPWM_DeadBandOutput output,
                               bool enableSwapMode)
{
    uint16_t mask;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    mask = (uint16_t)1U << ((uint16_t)output + EPWM_DBCTL_OUTSWAP_S);

    if(enableSwapMode)
    {
        //
        // Set the appropriate outswap bit to swap output
        //
        HWREGH(base + EPWM_O_DBCTL) = (HWREGH(base + EPWM_O_DBCTL) | mask);
    }
    else
    {
        //
        // Clear the appropriate outswap bit to disable output swap
        //
        HWREGH(base + EPWM_O_DBCTL) = (HWREGH(base + EPWM_O_DBCTL) & ~mask);
    }
}

/**
 *  @brief Sets Dead Band signal output mode.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] delayMode is the Dead Band delay type.
 *  @param [in] enableDelayMode is the dead band delay mode.
 *  @remarks
 *  This function sets up the dead band delay mode. The delayMode variable
 *  determines if the applied delay is Rising Edge or Falling Edge. The
 *  enableDelayMode determines if a dead band delay should be applied.
 *  Valid values for the variables are:
 *   - delayMode
 *       - EPWM_DB_RED   - Rising Edge delay
 *       - EPWM_DB_FED   - Falling Edge delay
 *   - enableDelayMode
 *       - true     - Falling edge or Rising edge delay is applied.
 *       - false    - Dead Band delay is bypassed.
*/
__STATIC_INLINE void
EPWM_setDeadBandDelayMode(uint32_t base, EPWM_DeadBandDelayMode delayMode,
                          bool enableDelayMode)
{
    uint16_t mask;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    mask = (uint16_t)1U << ((uint16_t)delayMode + EPWM_DBCTL_OUT_MODE_S);

    if(enableDelayMode)
    {
         //
         // Set the appropriate outmode bit to enable Dead Band delay
         //
         HWREGH(base + EPWM_O_DBCTL) = (HWREGH(base + EPWM_O_DBCTL) | mask);
    }
    else
    {
        //
        // Clear the appropriate outswap bit to disable output swap
        //
        HWREGH(base + EPWM_O_DBCTL) = (HWREGH(base + EPWM_O_DBCTL) & ~ mask);
    }
}

/**
 *  @brief Sets Dead Band delay polarity.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] delayMode is the Dead Band delay type.
 *  @param [in] polarity is the polarity of the delayed signal.
 *  @remarks
 *  This function sets up the polarity as determined by the variable polarity
 *  of the Falling Edge or Rising Edge delay depending on the value of
 *  delayMode. Valid values for the variables are:
 *    - delayMode
 *        - EPWM_DB_RED   - Rising Edge delay
 *        - EPWM_DB_FED   - Falling Edge delay
 *    - polarity
 *        - EPWM_DB_POLARITY_ACTIVE_HIGH   - polarity is not inverted.
 *        - EPWM_DB_POLARITY_ACTIVE_LOW    - polarity is inverted.
*/
__STATIC_INLINE void
EPWM_setDeadBandDelayPolarity(uint32_t base,
                              EPWM_DeadBandDelayMode delayMode,
                              EPWM_DeadBandPolarity polarity)
{
    uint16_t shift;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    shift = (((uint16_t)delayMode ^ 0x1U) + EPWM_DBCTL_POLSEL_S);

    //
    // Set the appropriate polsel bits for dead band polarity
    //
    HWREGH(base + EPWM_O_DBCTL) =
                       ((HWREGH(base + EPWM_O_DBCTL) & ~ (1U << shift)) |
                        ((uint16_t)polarity << shift));
}

/**
 *  @brief Sets Rising Edge Dead Band delay input.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] input is the input signal to the dead band.
 *  @remarks
 *  This function sets up the rising Edge delay input signal.
 *  Valid values for input are:
 *      - EPWM_DB_INPUT_EPWMA   - Input signal is ePWMA( Valid for both Falling
 *                                   Edge and Rising Edge)
 *      - EPWM_DB_INPUT_EPWMB   - Input signal is ePWMB( Valid for both Falling
 *                                   Edge and Rising Edge)
*/
__STATIC_INLINE void
EPWM_setRisingEdgeDeadBandDelayInput(uint32_t base, uint16_t input)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT((input == EPWM_DB_INPUT_EPWMA) ||
           (input == EPWM_DB_INPUT_EPWMB));

    //
    // Set the Rising Edge Delay input
    //
    HWREGH(base + EPWM_O_DBCTL) =
            ((HWREGH(base + EPWM_O_DBCTL) & ~(1U << (EPWM_DBCTL_IN_MODE_S))) |
             (input << EPWM_DBCTL_IN_MODE_S));
}

/**
 *  @brief Sets Dead Band delay input.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] input is the input signal to the dead band.
 *  @remarks
 *  This function sets up the rising Edge delay input signal.
 *  Valid values for input are:
 *    - EPWM_DB_INPUT_EPWMA   - Input signal is ePWMA(Valid for both Falling
 *                                 Edge and Rising Edge)
 *    - EPWM_DB_INPUT_EPWMB   - Input signal is ePWMB(Valid for both Falling
 *                                 Edge and Rising Edge)
 *    - EPWM_DB_INPUT_DB_RED  - Input signal is the output of Rising
 *                                 Edge delay.
 *                                (Valid only for Falling Edge delay)
*/
__STATIC_INLINE void
EPWM_setFallingEdgeDeadBandDelayInput(uint32_t base, uint16_t input)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT((input == EPWM_DB_INPUT_EPWMA) ||
           (input == EPWM_DB_INPUT_EPWMB) ||
           (input == EPWM_DB_INPUT_DB_RED));

    if(input == EPWM_DB_INPUT_DB_RED)
    {
        //
        // Set the Falling Edge Delay input
        //
        HWREGH(base + EPWM_O_DBCTL) |= EPWM_DBCTL_DEDB_MODE;
    }
    else
    {
        //
        // Set the Falling Edge Delay input
        //
        HWREGH(base + EPWM_O_DBCTL) &= ~EPWM_DBCTL_DEDB_MODE;

        //
        // Set the Rising Edge Delay input
        //
        HWREGH(base + EPWM_O_DBCTL) =
        ((HWREGH(base + EPWM_O_DBCTL) & ~(1U << (EPWM_DBCTL_IN_MODE_S + 1U))) |
         (input << (EPWM_DBCTL_IN_MODE_S + 1U)));
    }
}

/**
 *  @brief Set the Dead Band control shadow load mode.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] loadMode is the shadow to active load mode.
 *  @remarks
 *  This function enables and sets the Dead Band control register shadow
 *  load mode.
 *  Valid values for the \e loadMode parameter are:
 *      - EPWM_DB_LOAD_ON_CNTR_ZERO         - load when counter equals zero.
 *      - EPWM_DB_LOAD_ON_CNTR_PERIOD       - load when counter equals period.
 *      - EPWM_DB_LOAD_ON_CNTR_ZERO_PERIOD  - load when counter equals zero or
 *                                             period.
 *      - EPWM_DB_LOAD_FREEZE                - Freeze shadow to active load.
*/
__STATIC_INLINE void
EPWM_setDeadBandControlShadowLoadMode(uint32_t base,
                                      EPWM_DeadBandControlLoadMode loadMode)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Enable the shadow mode and setup the load event
    //
    HWREGH(base + EPWM_O_DBCTL2) =
          ((HWREGH(base + EPWM_O_DBCTL2) & ~EPWM_DBCTL2_LOADDBCTLMODE_M) |
           (EPWM_DBCTL2_SHDWDBCTLMODE | (uint16_t)loadMode));
}

/**
 *  @brief Disable Dead Band control shadow load mode.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function disables the Dead Band control register shadow
 *  load mode.
*/
__STATIC_INLINE void
EPWM_disableDeadBandControlShadowLoadMode(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Disable the shadow load mode. Only immediate load mode only.
    //
    HWREGH(base + EPWM_O_DBCTL2) =
                 (HWREGH(base + EPWM_O_DBCTL2) & ~EPWM_DBCTL2_SHDWDBCTLMODE);
}

/**
 *  @brief Set the RED (Rising Edge Delay) shadow load mode.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] loadMode is the shadow to active load event.
 *  @remarks
 *  This function sets the Rising Edge Delay register shadow load mode.
 *  Valid values for the \e loadMode parameter are:
 *      - EPWM_RED_LOAD_ON_CNTR_ZERO        - load when counter equals zero.
 *      - EPWM_RED_LOAD_ON_CNTR_PERIOD      - load when counter equals period.
 *      - EPWM_RED_LOAD_ON_CNTR_ZERO_PERIOD - load when counter equals zero or
 *                                            period.
 *      - EPWM_RED_LOAD_FREEZE               - Freeze shadow to active load.
*/
__STATIC_INLINE void
EPWM_setRisingEdgeDelayCountShadowLoadMode(uint32_t base,
                                         EPWM_RisingEdgeDelayLoadMode loadMode)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Enable the shadow mode. Set-up the load mode
    //
    HWREGH(base + EPWM_O_DBCTL) =
               ((HWREGH(base + EPWM_O_DBCTL) & ~EPWM_DBCTL_LOADREDMODE_M) |
                ((uint16_t)EPWM_DBCTL_SHDWDBREDMODE |
                ((uint16_t)loadMode << EPWM_DBCTL_LOADREDMODE_S)));

}

/**
 *  @brief Disable the RED (Rising Edge Delay) shadow load mode.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function disables the Rising Edge Delay register shadow load mode.
*/
__STATIC_INLINE void
EPWM_disableRisingEdgeDelayCountShadowLoadMode(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));


    //
    // Disable the shadow mode.
    //
    HWREGH(base + EPWM_O_DBCTL) =
                   (HWREGH(base + EPWM_O_DBCTL) & ~EPWM_DBCTL_SHDWDBREDMODE);

}

/**
 *  @brief Set the FED (Falling Edge Delay) shadow load mode.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] loadMode is the shadow to active load event.
 *  @remarks
 *  This function enables and sets the Falling Edge Delay register shadow load
 *  mode. Valid values for the \e loadMode parameters are:
 *      - EPWM_FED_LOAD_ON_CNTR_ZERO        - load when counter equals zero.
 *      - EPWM_FED_LOAD_ON_CNTR_PERIOD      - load when counter equals period.
 *      - EPWM_FED_LOAD_ON_CNTR_ZERO_PERIOD - load when counter equals zero or
 *                                             period.
 *      - EPWM_FED_LOAD_FREEZE               - Freeze shadow to active load.
*/
__STATIC_INLINE void
EPWM_setFallingEdgeDelayCountShadowLoadMode(uint32_t base,
                                        EPWM_FallingEdgeDelayLoadMode loadMode)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Enable the shadow mode. Setup the load mode.
    //
    HWREGH(base + EPWM_O_DBCTL) =
            ((HWREGH(base + EPWM_O_DBCTL) & ~EPWM_DBCTL_LOADFEDMODE_M) |
                (EPWM_DBCTL_SHDWDBFEDMODE |
                ((uint16_t)loadMode << EPWM_DBCTL_LOADFEDMODE_S)));

}

/**
 *  @brief Disables the FED (Falling Edge Delay) shadow load mode.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *   This function disables the Falling Edge Delay register shadow load mode.
*/
__STATIC_INLINE void
EPWM_disableFallingEdgeDelayCountShadowLoadMode(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Disable the shadow mode.
    //
    HWREGH(base + EPWM_O_DBCTL) =
              (HWREGH(base + EPWM_O_DBCTL) & ~EPWM_DBCTL_SHDWDBFEDMODE);
}

/**
 *  @brief Sets Dead Band Counter clock rate.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] clockMode is the Dead Band counter clock mode.
 *  @remarks
 *  This function sets up the Dead Band counter clock rate with respect to
 *  TBCLK (ePWM time base counter).
 *  Valid values for clockMode are:
 *    - EPWM_DB_COUNTER_CLOCK_FULL_CYCLE  -Dead band counter runs at TBCLK
 *                                            (ePWM Time Base Counter) rate.
 *    - EPWM_DB_COUNTER_CLOCK_HALF_CYCLE  -Dead band counter runs at 2*TBCLK
 *                                          (twice ePWM Time Base Counter)rate.
*/
__STATIC_INLINE void
EPWM_setDeadBandCounterClock(uint32_t base,
                             EPWM_DeadBandClockMode clockMode)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set the DB clock mode
    //
    HWREGH(base + EPWM_O_DBCTL) =
                ((HWREGH(base + EPWM_O_DBCTL) & ~EPWM_DBCTL_HALFCYCLE) |
                 ((uint16_t)clockMode << 15U));
}

/**
 *  @brief Set ePWM RED count
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] redCount is the RED(Rising Edge Delay) count.
 *  @remarks
 *  This function sets the RED (Rising Edge Delay) count value.
 *  The value of redCount should be less than 0x4000U.
*/
__STATIC_INLINE void
EPWM_setRisingEdgeDelayCount(uint32_t base, uint16_t redCount)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT(redCount < 0x4000U);

    //
    // Set the RED (Rising Edge Delay) count
    //
    HWREGH(base + EPWM_O_DBRED) = redCount;
}

/**
 *  @brief Set ePWM FED count
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] fedCount is the FED(Falling Edge Delay) count.
 *  @remarks
 *  This function sets the FED (Falling Edge Delay) count value.
 *  The value of fedCount should be less than 0x4000U.
*/
__STATIC_INLINE void
EPWM_setFallingEdgeDelayCount(uint32_t base, uint16_t fedCount)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT(fedCount < 0x4000U);

    //
    // Set the RED (Rising Edge Delay) count
    //
    HWREGH(base + EPWM_O_DBFED) = fedCount;
}

//
// @brief Chopper module related APIs
//
/**
 *  @brief Enable chopper mode
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function enables ePWM chopper module.
*/
__STATIC_INLINE void
EPWM_enableChopper(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set CHPEN bit. Enable Chopper
    //
    HWREGH(base + EPWM_O_PCCTL) |= EPWM_PCCTL_CHPEN;
}

/**
 *  @brief Enable chopper mode
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function enables ePWM chopper module.
*/
__STATIC_INLINE void
EPWM_disableChopper(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Clear CHPEN bit. Disable Chopper
    //
    HWREGH(base + EPWM_O_PCCTL) &= ~EPWM_PCCTL_CHPEN;
}

/**
 *  @brief Set chopper duty cycle.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] dutyCycleCount is the chopping clock duty cycle count.
 *  @remarks
 *  This function sets the chopping clock duty cycle. The value of
 *  dutyCycleCount should be less than 7. The dutyCycleCount value is converted
 *  to the actual chopper duty cycle value base on the following equation:
 *    chopper duty cycle = (dutyCycleCount + 1) / 8
*/
__STATIC_INLINE void
EPWM_setChopperDutyCycle(uint32_t base, uint16_t dutyCycleCount)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT(dutyCycleCount < 7U);

    //
    // Set the chopper duty cycle
    //
    HWREGH(base + EPWM_O_PCCTL) =
               ((HWREGH(base + EPWM_O_PCCTL) & ~EPWM_PCCTL_CHPDUTY_M) |
                (dutyCycleCount << EPWM_PCCTL_CHPDUTY_S));
}

/**
 *  @brief Set chopper clock frequency scaler.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] freqDiv is the chopping clock frequency divider.
 *  @remarks
 *  This function sets the scaler for the chopping clock frequency. The value
 *  of freqDiv should be less than 8. The chopping clock frequency is altered
 *  based on the following equation.
 *    chopper clock frequency = SYSCLKOUT / ( 1 + freqDiv)
*/
__STATIC_INLINE void
EPWM_setChopperFreq(uint32_t base, uint16_t freqDiv)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT(freqDiv < 8U);

    //
    // Set the chopper clock
    //
    HWREGH(base + EPWM_O_PCCTL) =
                    ((HWREGH(base + EPWM_O_PCCTL) &
                      ~(uint16_t)EPWM_PCCTL_CHPFREQ_M) |
                       (freqDiv << EPWM_PCCTL_CHPFREQ_S));
}

/**
 *  @brief Set chopper clock frequency scaler.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] firstPulseWidth is the width of the first pulse.
 *  @remarks
 *  This function sets the first pulse width of chopper output waveform. The
 *  value of firstPulseWidth should be less than 0x10. The value of the first
 *  pulse width in seconds is given using the following equation:
 *      first pulse width = 1 / (((firstPulseWidth + 1) * SYSCLKOUT)/8)
*/
__STATIC_INLINE void
EPWM_setChopperFirstPulseWidth(uint32_t base, uint16_t firstPulseWidth)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT(firstPulseWidth < 16U);

    //
    // Set the chopper clock
    //
    HWREGH(base + EPWM_O_PCCTL) =
              ((HWREGH(base + EPWM_O_PCCTL) &
               ~(uint16_t)EPWM_PCCTL_OSHTWTH_M) |
               (firstPulseWidth << EPWM_PCCTL_OSHTWTH_S));
}


//
// Event Trigger related APIs
//
/**
 *  @brief Enable ePWM interrupt.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] firstPulseWidth is the width of the first pulse.
 *  @remarks
 *  This function enables the ePWM interrupt.
*/
__STATIC_INLINE void
EPWM_enableInterrupt(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Enable ePWM interrupt
    //
    HWREGH(base + EPWM_O_ETSEL) |= EPWM_ETSEL_INTEN;
}

/**
 *  @brief disable ePWM interrupt.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] firstPulseWidth is the width of the first pulse.
 *  @remarks
 *  This function disables the ePWM interrupt.
*/
__STATIC_INLINE void
EPWM_disableInterrupt(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Disable ePWM interrupt
    //
    HWREGH(base + EPWM_O_ETSEL) &= ~EPWM_ETSEL_INTEN;
}

/**
 *  @brief Sets the ePWM interrupt source.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] interruptSource is the ePWM interrupt source.
 *  @remarks
 *  This function sets the ePWM interrupt source.
 *  Valid values for interruptSource are:
 *    - EPWM_INT_TBCTR_DISABLED       - Time-base counter is disabled
 *    - EPWM_INT_TBCTR_ZERO           - Time-base counter equal to zero
 *    - EPWM_INT_TBCTR_PERIOD         - Time-base counter equal to period
 *    - EPWM_INT_TBCTR_ZERO_OR_PERIOD - Time-base counter equal to zero or
 *                                      period
 *    - EPWM_INT_TBCTR_ETINTMIX 		- Time-base counter equal to zero or
 *                                      period
 *    - EPWM_INT_TBCTR_U_CMPx         - Where x is A,B,C or D
 *                                      Time-base counter equal to CMPA, CMPB,
 *                                      CMPC or CMPD (depending the value of x)
 *                                      when the timer is incrementing
 *    - EPWM_INT_TBCTR_D_CMPx         - Where x is A,B,C or D
 *                                      Time-base counter equal to CMPA, CMPB,
 *                                      CMPC or CMPD (depending the value of x)
 *                                      when the timer is decrementing
*/
__STATIC_INLINE void
EPWM_setInterruptSource(uint32_t base, uint16_t interruptSource)
{
    uint16_t intSource;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT(((interruptSource > 0U) && (interruptSource < 9U)) ||
           (interruptSource == 10U) || (interruptSource == 12U) ||
           (interruptSource == 14U));

    if((interruptSource == EPWM_INT_TBCTR_U_CMPC) ||
       (interruptSource == EPWM_INT_TBCTR_U_CMPD) ||
       (interruptSource == EPWM_INT_TBCTR_D_CMPC) ||
       (interruptSource == EPWM_INT_TBCTR_D_CMPD))
    {
          //
          // Shift the interrupt source by 1
          //
          intSource = interruptSource >> 1U;

          //
          // Enable events based on comp C or comp D
          //
          HWREGH(base + EPWM_O_ETSEL) |= EPWM_ETSEL_INTSELCMP;
    }
    else if((interruptSource == EPWM_INT_TBCTR_U_CMPA) ||
            (interruptSource == EPWM_INT_TBCTR_U_CMPB) ||
            (interruptSource == EPWM_INT_TBCTR_D_CMPA) ||
            (interruptSource == EPWM_INT_TBCTR_D_CMPB))
    {
        intSource = interruptSource;

        //
        // Enable events based on comp A or comp B
        //
        HWREGH(base + EPWM_O_ETSEL) &= ~EPWM_ETSEL_INTSELCMP;
    }
    else
    {
        intSource = interruptSource;
    }

    //
    // Set the interrupt source
    //
    HWREGH(base + EPWM_O_ETSEL) =
            ((HWREGH(base + EPWM_O_ETSEL) & ~EPWM_ETSEL_INTSEL_M) | intSource);
}

/**
 *  @brief Sets the ePWM interrupt event counts.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] eventCount is the event count for interrupt scale
 *  @remarks
 *  This function sets the interrupt event count that determines the number of
 *  events that have to occur before an interrupt is issued.
 *  Maximum value for eventCount is 15.
*/
__STATIC_INLINE void
EPWM_setInterruptEventCount(uint32_t base, uint16_t eventCount)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT(eventCount < 16U);

    //
    // Enable advanced feature of interrupt every up to 15 events
    //
    HWREGH(base + EPWM_O_ETPS) |= EPWM_ETPS_INTPSSEL;
    HWREGH(base + EPWM_O_ETINTPS) =
         ((HWREGH(base + EPWM_O_ETINTPS) & ~EPWM_ETINTPS_INTPRD2_M) |
           eventCount);
}

/**
 *  @brief Return the interrupt status.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function returns the ePWM interrupt status.
 *  This function doesn't return the Trip Zone status.
 *
 *  @return Returns true if ePWM interrupt was generated.
 *          Returns false if no interrupt was generated
*/
__STATIC_INLINE bool
EPWM_getEventTriggerInterruptStatus(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Return INT bit of ETFLG register
    //
    return(((HWREGH(base + EPWM_O_ETFLG) & 0x1U) == 0x1U) ? true : false);
}

/**
 *  @brief Clear interrupt flag.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function clears the ePWM interrupt flag.
*/
__STATIC_INLINE void
EPWM_clearEventTriggerInterruptFlag(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Clear INT bit of ETCLR register
    //
    HWREGH(base + EPWM_O_ETCLR) |= EPWM_ETCLR_INT;
}

/**
 *  @brief Enable Pre-interrupt count load.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function enables the ePWM interrupt counter to be pre-interrupt loaded
 *  with a count value.
 * This is valid only for advanced/expanded interrupt mode
*/
__STATIC_INLINE void
EPWM_enableInterruptEventCountInit(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Enable interrupt event count initializing/loading
    //
    HWREGH(base + EPWM_O_ETCNTINITCTL) |= EPWM_ETCNTINITCTL_INTINITEN;
}

/**
 *  @brief Disable interrupt count load.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function disables the ePWM interrupt counter from being loaded with
 *  pre-interrupt count value.
*/
__STATIC_INLINE void
EPWM_disableInterruptEventCountInit(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Disable interrupt event count initializing/loading
    //
    HWREGH(base + EPWM_O_ETCNTINITCTL) &= ~EPWM_ETCNTINITCTL_INTINITEN;
}

/**
 *  @brief Force a software pre interrupt event counter load.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function forces the ePWM interrupt counter to be loaded with the
 *  contents set by EPWM_setPreInterruptEventCount().
 *
 *  make sure the EPWM_enablePreInterruptEventCountLoad() function is
 *        is called before invoking this function.
*/
__STATIC_INLINE void
EPWM_forceInterruptEventCountInit(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Load the Interrupt Event counter value
    //
    HWREGH(base + EPWM_O_ETCNTINITCTL) |= EPWM_ETCNTINITCTL_INTINITFRC;
}

/**
 *  @brief Set interrupt count.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] eventCount is the ePWM interrupt count value.
 *  @remarks
 *  This function sets the ePWM interrupt count. eventCount is the value of the
 *  pre-interrupt value that is to be loaded. The maximum value of eventCount
 *  is 15.
*/
__STATIC_INLINE void
EPWM_setInterruptEventCountInitValue(uint32_t base, uint16_t eventCount)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT(eventCount < 16U);

    //
    // Set the Pre-interrupt event count
    //
    HWREGH(base + EPWM_O_ETCNTINIT) =
         ((HWREGH(base + EPWM_O_ETCNTINIT) & ~EPWM_ETCNTINIT_INTINIT_M) |
          (uint16_t)(eventCount & 0xFU));
}

/**
 *  @brief Get the interrupt count.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function returns the ePWM interrupt event count.
 *
 *  @return The interrupt event counts that have occurred.
*/
__STATIC_INLINE uint16_t
EPWM_getInterruptEventCount(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Return the interrupt event count
    //
    return(((HWREGH(base + EPWM_O_ETINTPS) & EPWM_ETINTPS_INTCNT2_M) >>
             EPWM_ETINTPS_INTCNT2_S));
}

/**
 *  @brief Force ePWM interrupt.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function forces an ePWM interrupt.
*/
__STATIC_INLINE void
EPWM_forceEventTriggerInterrupt(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set INT bit of ETFRC register
    //
    HWREGH(base + EPWM_O_ETFRC) |= EPWM_ETFRC_INT;
}

#if (EPWM_VERSION == 30)
/**
 *  @brief Sets the mixed event trigger interrupt source.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] etmixSignal is the module for which trigger needs to be enabled.
 *  @param [in] interruptSource is the Mixed ET interrupt source.
 *  @remarks
 *  This function sets the ePWM interrupt source.
 *  Valid values for etmixSignal are:
 *    - EPWM_ETMIX_INTERRUPT            - Mixed ET Interrupt signal
 *    - EPWM_ETMIX_SOCA                 - Mixed ET SOCA signal
 *    - EPWM_ETMIX_SOCB                 - Mixed ET SOCB signal
 *  Valid values for interruptSource are:
 *    - EPWM_ETMIX_TBCTR_ZERO           - Time-base counter equal to zero
 *    - EPWM_ETMIX_TBCTR_PERIOD         - Time-base counter equal to period
 *    - EPWM_ETMIX_TBCTR_U_CMPx         - Where x is A,B,C or D
 *                                      Time-base counter equal to CMPA, CMPB,
 *                                      CMPC or CMPD (depending the value of x)
 *                                      when the timer is incrementing
 *    - EPWM_ETMIX_TBCTR_D_CMPx         - Where x is A,B,C or D
 *                                      Time-base counter equal to CMPA, CMPB,
 *                                      CMPC or CMPD (depending the value of x)
 *                                      when the timer is decrementing
 *    - EPWM_ETMIX_DCAEVT1              - DCAEVT1 interrupt signal to trigger
 *                                      mixed ET interrupt
*/
static inline void
EPWM_setMixEvtTriggerSource(uint32_t base, EPWM_ETMixSignalSelect etmixSignal,
                                EPWM_ETMixTriggerSource interruptSource)
{
    uint32_t registerOffset;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    registerOffset = base + EPWM_O_ETINTMIXEN + (uint32_t)etmixSignal;

    //
    // Set the ETMIX source
    //
    HWREGH(registerOffset) |= (uint32_t)interruptSource;
}

/**
 *  @brief Sets the mixed event trigger interrupt source.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] etmixSignal is the module for which trigger needs to be enabled.
 *  @param [in] interruptSource is the Mixed ET interrupt source.
 *  @remarks
 *  This function sets the ePWM interrupt source.
 *  Valid values for etmixSignal are:
 *    - EPWM_ETMIX_INTERRUPT            - Mixed ET Interrupt signal
 *    - EPWM_ETMIX_SOCA                 - Mixed ET SOCA signal
 *    - EPWM_ETMIX_SOCB                 - Mixed ET SOCB signal
 *  Valid values for interruptSource are:
 *    - EPWM_ETMIX_TBCTR_ZERO           - Time-base counter equal to zero
 *    - EPWM_ETMIX_TBCTR_PERIOD         - Time-base counter equal to period
 *    - EPWM_ETMIX_TBCTR_U_CMPx         - Where x is A,B,C or D
 *                                      Time-base counter equal to CMPA, CMPB,
 *                                      CMPC or CMPD (depending the value of x)
 *                                      when the timer is incrementing
 *    - EPWM_ETMIX_TBCTR_D_CMPx         - Where x is A,B,C or D
 *                                      Time-base counter equal to CMPA, CMPB,
 *                                      CMPC or CMPD (depending the value of x)
 *                                      when the timer is decrementing
 *    - EPWM_ETMIX_DCAEVT1              - DCAEVT1 interrupt signal to trigger
 *                                      mixed ET interrupt
*/
static inline void
EPWM_clearMixEvtTriggerSource(uint32_t base, EPWM_ETMixSignalSelect etmixSignal,
                                EPWM_ETMixTriggerSource interruptSource)
{
    uint32_t registerOffset;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    registerOffset = base + EPWM_O_ETINTMIXEN + (uint32_t)etmixSignal;

    //
    // Set the ETMIX source
    //
    HWREGH(registerOffset) &= (~(uint32_t)interruptSource);
}
#endif

//
// ADC SOC configuration related APIs
//
/**
 *  @brief Enable ADC SOC event.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] adcSOCType is the ADC SOC type.
 *  @remarks
 *  This function enables the ePWM module to trigger an ADC SOC event.
 *  Valid values for adcSOCType are:
 *    - EPWM_SOC_A  - SOC A
 *    - EPWM_SOC_B  - SOC B
*/
__STATIC_INLINE void
EPWM_enableADCTrigger(uint32_t base, EPWM_ADCStartOfConversionType adcSOCType)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Enable an SOC
    //
    if(adcSOCType == EPWM_SOC_A)
    {
         HWREGH(base + EPWM_O_ETSEL) |= EPWM_ETSEL_SOCAEN;
    }
    else
    {
         HWREGH(base + EPWM_O_ETSEL) |= EPWM_ETSEL_SOCBEN;
    }
}

/**
 *  @brief Disable ADC SOC event.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] adcSOCType is the ADC SOC type.
 *  @remarks
 *  This function enables the ePWM module to trigger an ADC SOC event.
 *  Valid values for adcSOCType are:
 *    - EPWM_SOC_A  - SOC A
 *    - EPWM_SOC_B  - SOC B
*/
__STATIC_INLINE void
EPWM_disableADCTrigger(uint32_t base, EPWM_ADCStartOfConversionType adcSOCType)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Disable an SOC
    //
    if(adcSOCType == EPWM_SOC_A)
    {
         HWREGH(base + EPWM_O_ETSEL) &= ~EPWM_ETSEL_SOCAEN;
    }
    else
    {
         HWREGH(base + EPWM_O_ETSEL) &= ~EPWM_ETSEL_SOCBEN;
    }
}

/**
 *  @brief Sets the ePWM SOC source.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] adcSOCType is the ADC SOC type.
 *  @param [in] socSource is the SOC source.
 *  @remarks
 *  This function sets the ePWM ADC SOC source.
 *  Valid values for socSource are:
 *   - adcSOCType
 *      - EPWM_SOC_A  - SOC A
 *      - EPWM_SOC_B  - SOC B
 *   - socSource
 *      - EPWM_SOC_DCxEVT1              - Event is based on DCxEVT1
 *      - EPWM_SOC_TBCTR_ZERO           - Time-base counter equal to zero
 *      - EPWM_SOC_TBCTR_PERIOD         - Time-base counter equal to period
 *      - EPWM_SOC_TBCTR_ZERO_OR_PERIOD - Time-base counter equal to zero or
 *                                        period
 *      - EPWM_SOC_TBCTR_U_CMPx         - Where x is A,B,C or D
 *                                        Time-base counter equal to CMPA, CMPB,
 *                                        CMPC or CMPD(depending the value of x)
 *                                        when the timer is incrementing
 *      - EPWM_SOC_TBCTR_D_CMPx         - Where x is A,B,C or D
 *                                        Time-base counter equal to CMPA, CMPB,
 *                                        CMPC or CMPD(depending the value of x)
 *                                        when the timer is decrementing
*/
__STATIC_INLINE void
EPWM_setADCTriggerSource(uint32_t base,
                         EPWM_ADCStartOfConversionType adcSOCType,
                         EPWM_ADCStartOfConversionSource socSource)
{
    uint16_t source;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    if((socSource == EPWM_SOC_TBCTR_U_CMPC) ||
       (socSource == EPWM_SOC_TBCTR_U_CMPD) ||
       (socSource == EPWM_SOC_TBCTR_D_CMPC) ||
       (socSource == EPWM_SOC_TBCTR_D_CMPD))
    {
        source = (uint16_t)socSource >> 1U;
    }
    else
    {
        source = (uint16_t)socSource;
    }

    if(adcSOCType == EPWM_SOC_A)
    {
        //
        // Set the SOC source
        //
        HWREGH(base + EPWM_O_ETSEL) =
                ((HWREGH(base + EPWM_O_ETSEL) & ~EPWM_ETSEL_SOCASEL_M) |
                 (source << EPWM_ETSEL_SOCASEL_S));

        //
        // Enable the comparator selection
        //
        if((socSource == EPWM_SOC_TBCTR_U_CMPA) ||
           (socSource == EPWM_SOC_TBCTR_U_CMPB) ||
           (socSource == EPWM_SOC_TBCTR_D_CMPA) ||
           (socSource == EPWM_SOC_TBCTR_D_CMPB))
        {
            //
            // Enable events based on comp A or comp B
            //
            HWREGH(base + EPWM_O_ETSEL) &= ~EPWM_ETSEL_SOCASELCMP;
        }
        else if((socSource == EPWM_SOC_TBCTR_U_CMPC) ||
                (socSource == EPWM_SOC_TBCTR_U_CMPD) ||
                (socSource == EPWM_SOC_TBCTR_D_CMPC) ||
                (socSource == EPWM_SOC_TBCTR_D_CMPD))
        {
            //
            // Enable events based on comp C or comp D
            //
            HWREGH(base + EPWM_O_ETSEL) |= EPWM_ETSEL_SOCASELCMP;
        }
        else
        {
            //
            // No action required for the other socSource options
            //
        }
    }
    else
    {
        //
        // Enable the comparator selection
        //
        HWREGH(base + EPWM_O_ETSEL) =
                ((HWREGH(base + EPWM_O_ETSEL) & ~EPWM_ETSEL_SOCBSEL_M) |
                 (source << EPWM_ETSEL_SOCBSEL_S));

        //
        // Enable the comparator selection
        //
        if((socSource == EPWM_SOC_TBCTR_U_CMPA) ||
           (socSource == EPWM_SOC_TBCTR_U_CMPB) ||
           (socSource == EPWM_SOC_TBCTR_D_CMPA) ||
           (socSource == EPWM_SOC_TBCTR_D_CMPB))
        {
            //
            // Enable events based on comp A or comp B
            //
            HWREGH(base + EPWM_O_ETSEL) &= ~EPWM_ETSEL_SOCBSELCMP;
        }
        else if((socSource == EPWM_SOC_TBCTR_U_CMPC) ||
                (socSource == EPWM_SOC_TBCTR_U_CMPD) ||
                (socSource == EPWM_SOC_TBCTR_D_CMPC) ||
                (socSource == EPWM_SOC_TBCTR_D_CMPD))
        {
            //
            // Enable events based on comp C or comp D
            //
            HWREGH(base + EPWM_O_ETSEL) |= EPWM_ETSEL_SOCBSELCMP;
        }
        else
        {
            //
            // No action required for the other socSource options
            //
        }
    }
}

/**
 *  @brief Sets the ePWM SOC event counts.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] adcSOCType is the ADC SOC type.
 *  @param [in] preScaleCount is the event count number.
 *  @remarks
 *  This function sets the SOC event count that determines the number of
 *  events that have to occur before an SOC is issued.
 *   Valid values for the parameters are:
 *    - adcSOCType
 *        - EPWM_SOC_A  - SOC A
 *        - EPWM_SOC_B  - SOC B
 *    - preScaleCount
 *         - [1 - 15]  -  Generate SOC pulse every preScaleCount
 *                        up to 15 events.
 *
 *   A preScaleCount value of 0 disables the prescale.
*/
__STATIC_INLINE void
EPWM_setADCTriggerEventPrescale(uint32_t base,
                                EPWM_ADCStartOfConversionType adcSOCType,
                                uint16_t preScaleCount)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT(preScaleCount < 16U);

    //
    // Enable advanced feature of SOC every up to 15 events
    //
    HWREGH(base + EPWM_O_ETPS) |= EPWM_ETPS_SOCPSSEL;
    if(adcSOCType == EPWM_SOC_A)
    {
        //
        // Set the count for SOC A
        //
        HWREGH(base + EPWM_O_ETSOCPS) =
           ((HWREGH(base + EPWM_O_ETSOCPS) & ~EPWM_ETSOCPS_SOCAPRD2_M) |
             preScaleCount);
    }
    else
    {
        //
        // Set the count for SOC B
        //
        HWREGH(base + EPWM_O_ETSOCPS) =
             ((HWREGH(base + EPWM_O_ETSOCPS) & ~EPWM_ETSOCPS_SOCBPRD2_M) |
              (preScaleCount << EPWM_ETSOCPS_SOCBPRD2_S));
    }
}

/**
 *  @brief Return the SOC event status.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] adcSOCType is the ADC SOC type.
 *  @remarks
 *  This function returns the ePWM SOC status.
 *  Valid values for adcSOCType are:
 *    - EPWM_SOC_A  - SOC A
 *    - EPWM_SOC_B  - SOC B
 *
 *  @return Returns true if the selected adcSOCType SOC was generated.
 *          Returns false if the selected adcSOCType SOC was not generated.
*/
__STATIC_INLINE bool
EPWM_getADCTriggerFlagStatus(uint32_t base,
                             EPWM_ADCStartOfConversionType adcSOCType)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Return the SOC A/ B status
    //
    return((((HWREGH(base + EPWM_O_ETFLG) >>
              ((uint16_t)adcSOCType + 2U)) & 0x1U) == 0x1U) ? true : false);
}

/**
 *  @brief Clear SOC flag.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] adcSOCType is the ADC SOC type.
 *  @remarks
 *  This function returns the ePWM SOC status.
 *  Valid values for adcSOCType are:
 *    - EPWM_SOC_A  - SOC A
 *    - EPWM_SOC_B  - SOC B
*/
__STATIC_INLINE void
EPWM_clearADCTriggerFlag(uint32_t base,
                         EPWM_ADCStartOfConversionType adcSOCType)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Clear SOC A/B bit of ETCLR register
    //
    HWREGH(base + EPWM_O_ETCLR) |= 1U << ((uint16_t)adcSOCType + 2U);
}

/**
 *  @brief Enable Pre-SOC event count load.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] adcSOCType is the ADC SOC type.
 *  @remarks
 *  This function returns the ePWM SOC status.
 *  Valid values for adcSOCType are:
 *    - EPWM_SOC_A  - SOC A
 *    - EPWM_SOC_B  - SOC B
 *
 *    This is valid only for advanced/expanded SOC mode
*/
__STATIC_INLINE void
EPWM_enableADCTriggerEventCountInit(uint32_t base,
                                    EPWM_ADCStartOfConversionType adcSOCType)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Enable SOC event count initializing/loading
    //
    HWREGH(base + EPWM_O_ETCNTINITCTL) |= 1U << ((uint16_t)adcSOCType + 14U);
}

/**
 *  @brief Disable Pre-SOC event count load.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] adcSOCType is the ADC SOC type.
 *  @remarks
 *  This function returns the ePWM SOC status.
 *  Valid values for adcSOCType are:
 *    - EPWM_SOC_A  - SOC A
 *    - EPWM_SOC_B  - SOC B
 *
 *    This is valid only for advanced/expanded SOC mode
*/
__STATIC_INLINE void
EPWM_disableADCTriggerEventCountInit(uint32_t base,
                                     EPWM_ADCStartOfConversionType adcSOCType)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Disable SOC event count initializing/loading
    //
    HWREGH(base + EPWM_O_ETCNTINITCTL) &=
                                  ~(1U << ((uint16_t)adcSOCType + 14U));
}

/**
 *  @brief Force a software pre SOC event counter load.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] adcSOCType is the ADC SOC type.
 *  @remarks
 *  This function forces the ePWM SOC counter to be loaded with the
 *  contents set by EPWM_setPreADCStartOfConversionEventCount().
 *
 *  make sure the EPWM_enableADCTriggerEventCountInit()
 *        function is called before invoking this function.
*/
__STATIC_INLINE void
EPWM_forceADCTriggerEventCountInit(uint32_t base,
                                   EPWM_ADCStartOfConversionType adcSOCType)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Load the Interrupt Event counter value
    //
    HWREGH(base + EPWM_O_ETCNTINITCTL) |= 1U << ((uint16_t)adcSOCType + 11U);
}

/**
 *  @brief Set ADC Trigger count values.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] adcSOCType is the ADC SOC type.
 *  @param [in] eventCount is the ePWM interrupt count value.
 *  @remarks
 *  This function sets the ePWM ADC Trigger count values.
 *  Valid values for adcSOCType are:
 *    - EPWM_SOC_A  - SOC A
 *    - EPWM_SOC_B  - SOC B
 *  The eventCount has a maximum value of 15.
*/
__STATIC_INLINE void
EPWM_setADCTriggerEventCountInitValue(uint32_t base,
                                      EPWM_ADCStartOfConversionType adcSOCType,
                                      uint16_t eventCount)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT(eventCount < 16U);

    //
    // Set the ADC Trigger event count
    //
    if(adcSOCType == EPWM_SOC_A)
    {
        HWREGH(base + EPWM_O_ETCNTINIT) =
            ((HWREGH(base + EPWM_O_ETCNTINIT) & ~EPWM_ETCNTINIT_SOCAINIT_M) |
             (uint16_t)(eventCount << EPWM_ETCNTINIT_SOCAINIT_S));
    }
    else
    {
        HWREGH(base + EPWM_O_ETCNTINIT) =
             ((HWREGH(base + EPWM_O_ETCNTINIT) & ~EPWM_ETCNTINIT_SOCBINIT_M) |
              (eventCount << EPWM_ETCNTINIT_SOCBINIT_S));
    }
}

/**
 *  @brief Get the SOC event count.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] adcSOCType is the ADC SOC type.
 *  @remarks
 *  This function sets the ePWM ADC Trigger count values.
 *  Valid values for adcSOCType are:
 *    - EPWM_SOC_A  - SOC A
 *    - EPWM_SOC_B  - SOC B
 *
 *  \return The SOC event counts that have occurred.
*/
__STATIC_INLINE uint16_t
EPWM_getADCTriggerEventCount(uint32_t base,
                             EPWM_ADCStartOfConversionType adcSOCType)
{
    uint16_t eventCount;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Return the SOC event count
    //
    if(adcSOCType == EPWM_SOC_A)
    {
        eventCount = (HWREGH(base + EPWM_O_ETSOCPS) >>
                                         EPWM_ETSOCPS_SOCACNT2_S) & 0xFU;
    }
    else
    {
        eventCount = (HWREGH(base + EPWM_O_ETSOCPS) >>
                                        EPWM_ETSOCPS_SOCBCNT2_S) & 0xFU;
    }
    return(eventCount);
}

/**
 *  @brief Force SOC event.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] adcSOCType is the ADC SOC type.
 *  @remarks
 *  This function sets the ePWM ADC Trigger count values.
 *  Valid values for adcSOCType are:
 *    - EPWM_SOC_A  - SOC A
 *    - EPWM_SOC_B  - SOC B
*/
__STATIC_INLINE void
EPWM_forceADCTrigger(uint32_t base, EPWM_ADCStartOfConversionType adcSOCType)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set SOC A/B bit of ETFRC register
    //
    HWREGH(base + EPWM_O_ETFRC) |= 1U << ((uint16_t)adcSOCType + 2U);
}



#ifdef __cplusplus
}
#endif


#endif /* DEVICE_DRIVERLIB_EPWM_DB_ET_H_ */
