/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    epwm_mimdb.h
*   @brief   
*
*   commit history
*   20240722, lihao, Separate the MINDB function from the epwm.h..
*/
#ifndef DEVICE_DRIVERLIB_EPWM_MINDB_H_
#define DEVICE_DRIVERLIB_EPWM_MINDB_H_

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include <stdbool.h>
#include <stdint.h>
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "inc/hw_epwm.h"
#include "inc/hw_sysctl_ahb.h"
#include "cpu.h"
#include "debug.h"

#if (EPWM_VERSION  == 30)
/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
//*****************************************************************************
//
// Address offsets from EPWM_BASE to MINDB register memory maps
//
//*****************************************************************************
//#define EPWM_O_MINDB               0x0680U


//
// Minimum Dead Band
//
//*****************************************************************************
//
//! Values that can be passed to EPWM_invertMinimumDeadBandSignal(),
//! EPWM_selectMinimumDeadBandAndOrLogic(),
//! EPWM_selectMinimumDeadBandBlockingSignal(), EPWM_selectXbarInput(),
//! EPWM_selectMinimumDeadBandReferenceSignal(), EPWM_getMinDeadBandDelay(),
//! EPWM_setMinDeadBandDelay(), EPWM_enableIllegalComboLogic() and
//! EPWM_setLutDecX() as the \e block parameter.
//
//*****************************************************************************
//! Minimum Dead Band Block A
#define EPWM_MINDB_BLOCK_A    0x0U
//! Minimum Dead Band Block B
#define EPWM_MINDB_BLOCK_B    0x1U

//*****************************************************************************
//
//! Values that can be passed to EPWM_invertMinimumDeadBandSignal() as the
//! \e invert parameter.
//
//*****************************************************************************
//! Minimum Dead Band don't invert reference signal
#define EPWM_MINDB_REF_SIG          0x0U
//! Minimum Dead invert reference signal
#define EPWM_MINDB_REF_SIG_INVERT   0x1U

//*****************************************************************************
//
//! Values that can be passed to EPWM_selectMinimumDeadBandAndOrLogic() as
//! the \e logic parameter.
//
//*****************************************************************************
//! Minimum Dead Band Invert and perform logical AND
#define EPWM_MINDB_POLSEL_INVERT_LOGICAL_AND    0x0U
//! Minimum Dead Band Block perform logical OR
#define EPWM_MINDB_POLSEL_LOGICAL_OR            0x1U

//*****************************************************************************
//
//! Values that can be passed to EPWM_selectMinimumDeadBandBlockingSignal()
//! as the \e blockingSignal parameter.
//
//*****************************************************************************
//! Minimum Dead Band keep blocking signal
#define EPWM_MINDB_BLOCKING_SIGNAL_BYPASS       0x0U
//! Minimum Dead Band swap blocking signal
#define EPWM_MINDB_BLOCKING_SIGNAL_SWAPBLOCK    0x1U

//*****************************************************************************
//
//! Values that can be passed to EPWM_selectMinimumDeadBandReferenceSignal()
//! as the \e referenceSignal parameter.
//
//*****************************************************************************
//! Minimum Dead Band Reference signal
#define EPWM_MINDB_SEL_DEPWM                0x0U
//! Minimum Dead Band Reference Signal PWM Output Xbar 1
#define EPWM_MINDB_SEL_OUTXBAR_OUT1     0x1U
//! Minimum Dead Band Reference Signal PWM Output Xbar 2
#define EPWM_MINDB_SEL_OUTXBAR_OUT2     0x2U
//! Minimum Dead Band Reference Signal PWM Output Xbar 3
#define EPWM_MINDB_SEL_OUTXBAR_OUT3     0x3U
//! Minimum Dead Band Reference Signal PWM Output Xbar 4
#define EPWM_MINDB_SEL_OUTXBAR_OUT4     0x4U
//! Minimum Dead Band Reference Signal PWM Output Xbar 5
#define EPWM_MINDB_SEL_OUTXBAR_OUT5     0x5U
//! Minimum Dead Band Reference Signal PWM Output Xbar 6
#define EPWM_MINDB_SEL_OUTXBAR_OUT6     0x6U
//! Minimum Dead Band Reference Signal PWM Output Xbar 7
#define EPWM_MINDB_SEL_OUTXBAR_OUT7     0x7U
//! Minimum Dead Band Reference Signal PWM Output Xbar 8
#define EPWM_MINDB_SEL_OUTXBAR_OUT8     0x8U
//! Minimum Dead Band Reference Signal PWM Output Xbar 9
#define EPWM_MINDB_SEL_OUTXBAR_OUT9     0x9U
//! Minimum Dead Band Reference Signal PWM Output Xbar 10
#define EPWM_MINDB_SEL_OUTXBAR_OUT10    0xAU
//! Minimum Dead Band Reference Signal PWM Output Xbar 11
#define EPWM_MINDB_SEL_OUTXBAR_OUT11    0xBU
//! Minimum Dead Band Reference Signal PWM Output Xbar 12
#define EPWM_MINDB_SEL_OUTXBAR_OUT12    0xCU
//! Minimum Dead Band Reference Signal PWM Output Xbar 13
#define EPWM_MINDB_SEL_OUTXBAR_OUT13    0xDU
//! Minimum Dead Band Reference Signal PWM Output Xbar 14
#define EPWM_MINDB_SEL_OUTXBAR_OUT14    0xEU
//! Minimum Dead Band Reference Signal PWM Output Xbar 15
#define EPWM_MINDB_SEL_OUTXBAR_OUT15    0xFU

//*****************************************************************************
//
//! Values that can be passed to EPWM_selectXbarInput() as the \e xbarInput
//! parameter.
//
//*****************************************************************************
//! ICL Xbar Out 1
#define EPWM_MINDB_ICL_XBAR_OUT1     0x0U
//! ICL Xbar Out 2
#define EPWM_MINDB_ICL_XBAR_OUT2     0x1U
//! ICL Xbar Out 3
#define EPWM_MINDB_ICL_XBAR_OUT3     0x2U
//! ICL Xbar Out 4
#define EPWM_MINDB_ICL_XBAR_OUT4     0x3U
//! ICL Xbar Out 5
#define EPWM_MINDB_ICL_XBAR_OUT5     0x4U
//! ICL Xbar Out 6
#define EPWM_MINDB_ICL_XBAR_OUT6     0x5U
//! ICL Xbar Out 7
#define EPWM_MINDB_ICL_XBAR_OUT7     0x6U
//! ICL Xbar Out 8
#define EPWM_MINDB_ICL_XBAR_OUT8     0x7U
//! ICL Xbar Out 9
#define EPWM_MINDB_ICL_XBAR_OUT9     0x8U
//! ICL Xbar Out 10
#define EPWM_MINDB_ICL_XBAR_OUT10     0x9U
//! ICL Xbar Out 11
#define EPWM_MINDB_ICL_XBAR_OUT11    0xAU
//! ICL Xbar Out 12
#define EPWM_MINDB_ICL_XBAR_OUT12    0xBU
//! ICL Xbar Out 13
#define EPWM_MINDB_ICL_XBAR_OUT13    0xCU
//! ICL Xbar Out 14
#define EPWM_MINDB_ICL_XBAR_OUT14    0xDU
//! ICL Xbar Out 15
#define EPWM_MINDB_ICL_XBAR_OUT15    0xEU
//! ICL Xbar Out 16
#define EPWM_MINDB_ICL_XBAR_OUT16    0xFU

//*****************************************************************************
//
//! Values that can be passed to EPWM_setLutDecX() as the \e decx parameter
//
//*****************************************************************************
//! LUT output PWMA/B value on decoding 0
#define EPWM_MINDB_ICL_LUT_DEC0     0
//! LUT output PWMA/B value on decoding 1
#define EPWM_MINDB_ICL_LUT_DEC1     1
//! LUT output PWMA/B value on decoding 2
#define EPWM_MINDB_ICL_LUT_DEC2     2
//! LUT output PWMA/B value on decoding 3
#define EPWM_MINDB_ICL_LUT_DEC3     3
//! LUT output PWMA/B value on decoding 4
#define EPWM_MINDB_ICL_LUT_DEC4     4
//! LUT output PWMA/B value on decoding 5
#define EPWM_MINDB_ICL_LUT_DEC5     5
//! LUT output PWMA/B value on decoding 6
#define EPWM_MINDB_ICL_LUT_DEC6     6
//! LUT output PWMA/B value on decoding 7
#define EPWM_MINDB_ICL_LUT_DEC7     7


//*****************************************************************************
//
// Prototypes for the API.
//
//*****************************************************************************

//
// Minimum Dead Band Module related APIs
//
/**
 *  @brief Enable Minimum DeadBand module
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] block is the Minimum DeadBand Block to be enabled.
 *  @remarks
 *  This function enables the Minimum DeadBand module.
 *  Valid values for the input variables are:
 *   - block
 *       - EPWM_MINDB_BLOCK_A   - ePWM output A
 *       - EPWM_MINDB_BLOCK_B   - ePWM output B
*/
static inline void
EPWM_enableMinimumDeadBand(uint32_t base, uint32_t block)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_MINDB + EPWM_O_MINDBCFG;

    //
    // Set the appropriate block to enable minimum dead band logic.
    //
    if(block == EPWM_MINDB_BLOCK_A)
    {
        HWREG(registerOffset) |= EPWM_MINDBCFG_ENABLEA;
    }
    else
    {
        HWREG(registerOffset) |= EPWM_MINDBCFG_ENABLEB;
    }
}

/**
 *  @brief Disable Minimum DeadBand module
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] block is the Minimum DeadBand Block to be disabled.
 *  @remarks
 *  This function enables the Minimum DeadBand module.
 *  Valid values for the input variables are:
 *   - block
 *       - EPWM_MINDBCFG_ENABLEA   - ePWM output A
 *       - EPWM_MINDBCFG_ENABLEB   - ePWM output B
*/
static inline void
EPWM_disableMinimumDeadBand(uint32_t base, uint32_t block)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_MINDB;

    //
    // Set the appropriate block to disable minimum dead band logic.
    //
    if(block == EPWM_MINDB_BLOCK_A)
    {
        HWREG(registerOffset) &= ~EPWM_MINDBCFG_ENABLEA;
    }
    else
    {
        HWREG(registerOffset) &= ~EPWM_MINDBCFG_ENABLEB;
    }
}

/**
 *  @brief Invert the Minimum DeadBand Reference Signal
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] block is the Minimum DeadBand block to be configured.
 *  @param [in] invert specifies whether reference signal is to be inverted or not.
 *  @remarks
 *  This function configures the Minimum DeadBand module to invert reference
 *  signal which is used in the Minimum DeadBand module.
 *  Valid values for the input variables are:
 *   - block
 *       - EPWM_MINDB_BLOCK_A    - ePWM output A
 *       - EPWM_MINDB_BLOCK_B    - ePWM output B
 *   - invert
 *       - EPWM_MINDB_REF_SIG            - No inversion
 *       - EPWM_MINDB_REF_SIG_INVERT     - Invert the signal
*/
static inline void
EPWM_invertMinimumDeadBandSignal(uint32_t base, uint32_t block,
                                 uint32_t invert)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_MINDB + EPWM_O_MINDBCFG;

    //
    // Set the appropriate output bit to enable minimum dead band logic.
    //
    if(block == EPWM_MINDB_BLOCK_A)
    {
        HWREGH(registerOffset) =
                (HWREGH(registerOffset) & ~EPWM_MINDBCFG_INVERTA) |
                (invert << 2U);
    }
    else
    {
        HWREG(registerOffset) =
                (HWREG(registerOffset) & ~EPWM_MINDBCFG_INVERTB) |
                (invert << 18U);
    }
}

/**
 *  @brief Select signal for AND OR logic of Minimum DeadBand Module
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] block is the Minimum DeadBand block to be configured.
 *  @param [in] logic specifies how the signal will be combined.
 *  @remarks
 *  This function configures how the signal will be combined with the
 *  PWM output. The signal can be inverted and ANDed with PWM output or it can
 *  be ORed with the PWM output.
 *  Valid values for the input variables are:
 *   - block
 *       - EPWM_MINDB_BLOCK_A    - ePWM output A
 *       - EPWM_MINDB_BLOCK_B    - ePWM output B
 *   - logic
 *       - EPWM_MINDB_POLSEL_INVERT_LOGICAL_AND   - Invert and Logical AND
 *       - EPWM_MINDB_POLSEL_LOGICAL_OR           - Logical OR
*/
static inline void
EPWM_selectMinimumDeadBandAndOrLogic(uint32_t base, uint32_t block,
                                     uint32_t logic)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_MINDB + EPWM_O_MINDBCFG;

    if(block == EPWM_MINDB_BLOCK_A)
    {
        //
        // Configure logic operation on EPWMA
        //
        HWREGH(registerOffset) =
                (HWREGH(registerOffset) & ~EPWM_MINDBCFG_POLSELA) |
                ((uint32_t)logic << 8U);
    }
    else
    {
        //
        // Configure logic operation on EPWMB
        //
        HWREG(registerOffset) =
                (HWREG(registerOffset) & ~EPWM_MINDBCFG_POLSELB) |
                ((uint32_t)logic << 24U);
    }
}

/**
 *  @brief Select Minimum DeadBand Blocking Signal
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] block is the Minimum DeadBand block to be configured.
 *  @param [in] blockingSignal is the blocking signal to be used for a block.
 *  @remarks
 *  This function configures the blocking signal for Minimum DeadBand module.
 *  Either of the Block A or Block B signal can be selected as blocking signal.
 *  Valid values for the input variables are:
 *   - block
 *       - EPWM_MINDB_BLOCK_A    - ePWM output A
 *       - EPWM_MINDB_BLOCK_B    - ePWM output B
 *   - blockSrc
 *       - EPWM_MINDB_BLOCKING_SIGNAL_BYPASS    - Block Source same
 *       - EPWM_MINDB_BLOCKING_SIGNAL_SWAPBLOCK - Block Source different
*/
static inline void
EPWM_selectMinimumDeadBandBlockingSignal(uint32_t base, uint32_t block,
                                         uint32_t blockingSignal)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_MINDB + EPWM_O_MINDBCFG;

    if(block == EPWM_MINDB_BLOCK_A)
    {
        //
        // Configure block signal source on EPWMA
        //
        HWREGH(registerOffset) =
                (HWREGH(registerOffset) & ~EPWM_MINDBCFG_SELBLOCKA) |
                ((uint32_t)blockingSignal << 3U);
    }
    else
    {
        //
        // Configure block signal source on EPWMB
        //
        HWREG(registerOffset) =
                (HWREG(registerOffset) & ~EPWM_MINDBCFG_SELBLOCKB) |
                ((uint32_t)blockingSignal << 19U);
    }
}

/**
 *  @brief Select Minimum DeadBand Reference Signal
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] block is the Minimum DeadBand block to be configured.
 *  @param [in] referenceSignal specifies the reference signal to be selected.
 *  @remarks
 *  This function selects the reference signal for Minimum DeadBand module.
 *  Valid values for the input variables are:
 *   - block
 *       - EPWM_MINDB_BLOCK_A    - ePWM output A
 *       - EPWM_MINDB_BLOCK_B    - ePWM output B
 *   - reference
 *       - EPWM_MINDB_DEPWM             - Minimum Dead Band DE reference
 *       - EPWM_MINDB_SEL_OUTXBAR_OUTy  - Output y from PWM Output XBAR
 *                                        ('y' can be from 1 to 15)
 *                                        ex. \b EPWM_MINDB_SEL_OUTXBAR_OUT1
*/
static inline void
EPWM_selectMinimumDeadBandReferenceSignal(uint32_t base, uint32_t block,
                                          uint32_t referenceSignal)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_MINDB + EPWM_O_MINDBCFG;

    if(block == EPWM_MINDB_BLOCK_A)
    {
        //
        // Select reference source for minimum dead band on EPWMA
        //
        HWREGH(registerOffset) =
                (HWREGH(registerOffset) & ~EPWM_MINDBCFG_SELA_M) |
                ((uint32_t)referenceSignal << EPWM_MINDBCFG_SELA_S);
    }
    else
    {
        //
        // Select reference source for minimum dead band on EPWMB
        //
        HWREG(registerOffset) =
                (HWREG(registerOffset) & ~EPWM_MINDBCFG_SELB_M) |
                ((uint32_t)referenceSignal << EPWM_MINDBCFG_SELB_S);
    }
}

/**
 *  @brief Get Minimum DeadBand Delay Value
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] block is the Minimum DeadBand block whose value is needed.
 *  @remarks
 *  This function returns the delay value for the Minimum DeadBand module.
 *  Valid values for the input variables are:
 *   - block
 *       - EPWM_MINDB_BLOCK_A    - ePWM output A
 *       - EPWM_MINDB_BLOCK_B    - ePWM output B
 *
 *  @return Delay value of the Minimum DeadBand module.
*/
static inline uint32_t
EPWM_getMinDeadBandDelay(uint32_t base, uint32_t block)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t retval;
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_MINDB + EPWM_O_MINDBDLY;

    if(block == EPWM_MINDB_BLOCK_A)
    {
        //
        // Return dead band delay value of ePWMA
        //
        retval = ((uint32_t)HWREGH(registerOffset) &
                  EPWM_MINDBDLY_DELAYA_M);
    }
    else
    {
        //
        // Return dead band delay value of ePWMB
        //
        retval = (HWREG(registerOffset) &
                  EPWM_MINDBDLY_DELAYB_M);
    }

    return(retval);
}

/**
 *  @brief Set the Minimum DeadBand Delay Value
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] block is the Minimum DeadBand block to be configured.
 *  @param [in] delay is the delay value to be programmed for the Minimum DeadBand.
 *  @remarks
 *  This function configures the Minimum DeadBand delay value for the
 *  specified block.
 *  Valid values for the input variables are:
 *   - block
 *       - EPWM_MINDB_BLOCK_A    - ePWM output A
 *       - EPWM_MINDB_BLOCK_B    - ePWM output B
 *   - delay: Minimum dead band delay on ePWM in terms of SYSCLK cycles
*/
static inline void
EPWM_setMinDeadBandDelay(uint32_t base, uint32_t block, uint32_t delay)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_MINDB + EPWM_O_MINDBDLY;

    if(block == EPWM_MINDB_BLOCK_A)
    {
        //
        // Dead band delay for ePWMA
        //
        HWREGH(registerOffset) =
                (HWREGH(registerOffset) & ~EPWM_MINDBDLY_DELAYA_M) |
                ((uint32_t)delay << EPWM_MINDBDLY_DELAYA_S);
    }
    else
    {
        //
        // Dead band delay for ePWMB
        //
        HWREG(registerOffset) =
                (HWREG(registerOffset) & ~EPWM_MINDBDLY_DELAYB_M) |
                ((uint32_t)delay << EPWM_MINDBDLY_DELAYB_S);
    }
}

//
// Illegal Combo Logic
//

/**
 *  @brief Enable Illegal Combo Logic
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] block is the Illegal Combo Logic block to be enabled.
 *  @remarks
 *  This function enables the Illegal Combo Logic block.
 *  Valid values for the input variables are:
 *   - block
 *       - EPWM_MINDB_BLOCK_A    - ePWM output A
 *       - EPWM_MINDB_BLOCK_B    - ePWM output B
*/
static inline void
EPWM_enableIllegalComboLogic(uint32_t base, uint32_t block)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_MINDB;

    if(block == EPWM_MINDB_BLOCK_A)
    {
        //
        // Enable LUTA
        //
        HWREGH(registerOffset + EPWM_O_LUTCTLA) =
                (HWREGH(registerOffset + EPWM_O_LUTCTLA) &
                 ~EPWM_LUTCTLA_BYPASS);
    }
    else
    {
        //
        // Enable LUTB
        //
        HWREGH(registerOffset + EPWM_O_LUTCTLB) =
                (HWREGH(registerOffset + EPWM_O_LUTCTLB) &
                 ~EPWM_LUTCTLB_BYPASS);
    }
}

/**
 *  @brief Disable Illegal Combo Logic
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] block is the Illegal Combo Logic block to be disabled.
 *  @remarks
 *  This function enables the Illegal Combo Logic block.
 *  Valid values for the input variables are:
 *   - block
 *       - EPWM_MINDB_BLOCK_A    - ePWM output A
 *       - EPWM_MINDB_BLOCK_B    - ePWM output B
*/
static inline void
EPWM_disableIllegalComboLogic(uint32_t base, uint32_t block)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_MINDB;

    if(block == EPWM_MINDB_BLOCK_A)
    {
        //
        // Bypass LUTA
        //
        HWREGH(registerOffset + EPWM_O_LUTCTLA) =
                (HWREGH(registerOffset + EPWM_O_LUTCTLA) |
                 EPWM_LUTCTLA_BYPASS);
    }
    else
    {
        //
        // Bypass LUTB
        //
        HWREGH(registerOffset + EPWM_O_LUTCTLB) =
                (HWREGH(registerOffset + EPWM_O_LUTCTLB) |
                 EPWM_LUTCTLB_BYPASS);
    }
}

/**
 *  @brief Select XBAR input for Illegal Combo Logic
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] block is the Illegal Combo Logic block to be configured.
 *  @param [in] xbarInput is the Xbar signal to feed into the Illegal Combo Logic.
 *  @remarks
 *  This function selects which Xbar signal feeds into the Illegal Combo Logic.
 *  Valid values for the input variables are:
 *   - block
 *       - EPWM_MINDB_BLOCK_A    - ePWM output A
 *       - EPWM_MINDB_BLOCK_B    - ePWM output B
 *   - xbarInput
 *       - EPWM_MINDB_ICL_XBAR_OUTy - Output y from ICL XBAR
 *                                     ('y' can be from 0 to 15)
 *                                     ex. \b EPWM_MINDB_ICL_XBAR_OUT1
*/
static inline void
EPWM_selectXbarInput(uint32_t base, uint32_t block, uint32_t xbarInput)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_MINDB;

    if(block == EPWM_MINDB_BLOCK_A)
    {
       //
       // Select one of the 16 outputs of ICL XBAR for ePWMA
       //
       HWREGH(registerOffset + EPWM_O_LUTCTLA) =
         (HWREGH(registerOffset + EPWM_O_LUTCTLA) & ~EPWM_LUTCTLA_SELXBAR_M) |
         ((uint32_t)xbarInput << EPWM_LUTCTLA_SELXBAR_S);
    }
    else
    {
       //
       // Select one of the 16 outputs of ICL XBAR for ePWMB
       //
       HWREGH(registerOffset + EPWM_O_LUTCTLB) =
         (HWREGH(registerOffset + EPWM_O_LUTCTLB) & ~EPWM_LUTCTLB_SELXBAR_M) |
         ((uint32_t)xbarInput << EPWM_LUTCTLB_SELXBAR_S);
    }
}

/**
 *  @brief Force Decx values
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] block is the Illegal Combo Logic block to be configured.
 *  @param [in] decx is the register to be programmed.
 *  @param [in] force is the value to be programmed in the register.
 *  @remarks
 *  This function programs value in the Decx register fields of the Illegal
 *  Combo Logic module.
 *  Valid values for the input variables are:
 *   - block
 *       - EPWM_MINDB_BLOCK_A    - ePWM output A
 *       - EPWM_MINDB_BLOCK_B    - ePWM output B
 *   - decx
 *       - EPWM_MINDB_ICL_LUT_DECx - 'x' can be from 0 to 7
 *   - force
 *       - value that can be forced can be either 0 or 1
*/
static inline void
EPWM_setLutDecX(uint32_t base, uint32_t block, uint32_t decx, uint32_t force)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_MINDB;

    if(block == EPWM_MINDB_BLOCK_A)
    {
        //
        // Forces either 0 or 1 in the LUTDECx bit of LUTCTLA register
        //
        HWREGH(registerOffset + EPWM_O_LUTCTLA + 2U) =
                ((HWREGH(registerOffset + EPWM_O_LUTCTLA + 2U) & ~(1U << decx)) |
                 (force << decx));
    }
    else
    {
        //
        // Forces either 0 or 1 in the LUTDECx bit of LUTCTLB register
        //
        HWREGH(registerOffset + EPWM_O_LUTCTLB + 2U) =
                ((HWREGH(registerOffset + EPWM_O_LUTCTLB + 2U) & ~(1U << decx)) |
                 (force << decx));
    }
}

#endif

//*****************************************************************************
//
// Close the Doxygen group.
//! @}
//
//*****************************************************************************


#ifdef __cplusplus
}
#endif


#endif /* DEVICE_DRIVERLIB_EPWM_MINDB_H_ */
