/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    epwm_tb_sync.h
*   @brief   
*
*   commit history
*   20240804, Jason, initial creation
*/

#ifndef DEVICE_DRIVERLIB_EPWM_TB_SYNC_H_
#define DEVICE_DRIVERLIB_EPWM_TB_SYNC_H_

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "inc/hw_types.h"
#include "gs32_version.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "inc/hw_epwm.h"
#include "inc/hw_sysctl_ahb.h"
#include "epwm_define.h"
#include "debug.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
//*****************************************************************************
//
// Defines for the API.
//
//*****************************************************************************


//
// Time Base Sub Module related APIs
//

/**
 *  @brief Set the time base count
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] count is the time base count value.
 *  @remarks
 *
 *  This function sets the 16 bit counter value of the time base counter.
*/
__STATIC_INLINE void
EPWM_setTimeBaseCounter(uint32_t base, uint16_t count)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Write to TBCTR register
    //
    HWREGH(base + EPWM_O_TBCTR) = count;
}

/**
 *  @brief Set count mode after phase shift sync
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] mode is the count mode.
 *  @remarks
 *  This function sets the time base count to count up or down after a new
 *  phase value set by the EPWM_setPhaseShift(). The count direction is
 *  determined by the variable mode. Valid inputs for mode are:
 *   - EPWM_COUNT_MODE_UP_AFTER_SYNC      - Count up after sync
 *   - EPWM_COUNT_MODE_DOWN_AFTER_SYNC    - Count down after sync
*/
__STATIC_INLINE void
EPWM_setCountModeAfterSync(uint32_t base, EPWM_SyncCountMode mode)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    if(mode == EPWM_COUNT_MODE_UP_AFTER_SYNC)
    {
        //
        // Set PHSDIR bit
        //
        HWREGH(base + EPWM_O_TBCTL) |= EPWM_TBCTL_PHSDIR;
    }
    else
    {
        //
        // Clear PHSDIR bit
        //
        HWREGH(base + EPWM_O_TBCTL) &= ~EPWM_TBCTL_PHSDIR;
    }
}

/**
 *  @brief Set the time base clock and the high speed time base clock count pre-scaler
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] prescaler is the time base count pre scale value.
 *  @param [in] highSpeedPrescaler is the high speed time base count pre scale
 *         value.
 *  @remarks
 *  This function sets the pre scaler(divider)value for the time base clock
 *  counter and the high speed time base clock counter.
 *  Valid values for pre-scaler and highSpeedPrescaler are EPWM_CLOCK_DIVIDER_X,
 *  where X is 1,2,4,8,16, 32,64 or 128.
 *  The actual numerical values for these macros represent values 0,1...7.
 *  The equation for the output clock is:
 *    TBCLK = EPWMCLK/(highSpeedPrescaler * pre-scaler)
 *
 *   Note: EPWMCLK is a scaled version of SYSCLK. At reset EPWMCLK is half
 *           SYSCLK.
*/
__STATIC_INLINE void
EPWM_setClockPrescaler(uint32_t base, EPWM_ClockDivider prescaler,
                       EPWM_HSClockDivider highSpeedPrescaler)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Write to CLKDIV and HSPCLKDIV bit
    //
    HWREGH(base + EPWM_O_TBCTL) =
                ((HWREGH(base + EPWM_O_TBCTL)  &
                ~(EPWM_TBCTL_CLKDIV_M | EPWM_TBCTL_HSPCLKDIV_M)) |
                (((uint16_t)prescaler << EPWM_TBCTL_CLKDIV_S) |
                ((uint16_t)highSpeedPrescaler << EPWM_TBCTL_HSPCLKDIV_S)));
}

/**
 *  @brief Force a software sync pulse
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function causes a single software initiated sync pulse. Make sure the
 *  appropriate mode is selected using EPWM_setupSyncOutputMode() before using
 *  this function.
*/
__STATIC_INLINE void
EPWM_forceSyncPulse(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set SWFSYNC bit
    //
    HWREGH(base + EPWM_O_TBCTL) |= EPWM_TBCTL_SWFSYNC;
}

/**
 *  @brief Enables sync-out pulse source.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] source is the sync-out pulse source.
 *  @remarks
 *  This function enables the sync-out pulse source.
 *  Below valid values for param \b source can be OR'd together to enable
 *  multiple sync-out sources:
 *  - EPWM_SYNC_OUT_PULSE_ON_SOFTWARE - sync-out pulse is generated by
 *                                      software when EPWM_forceSyncPulse()
 *                                      function is called or by EPWMxSYNCI
 *                                      signal.
 *  - EPWM_SYNC_OUT_PULSE_ON_CNTR_ZERO - sync-out pulse is generated when
 *                                       time base counter equals zero.
 *  - EPWM_SYNC_OUT_PULSE_ON_CNTR_COMPARE_B - sync-out pulse is generated when
 *                                            time base counter equals compare
 *                                            B value.
 *  - EPWM_SYNC_OUT_PULSE_ON_CNTR_COMPARE_C - sync-out pulse is generated when
 *                                            time base counter equals compare
 *                                            C value.
 *  - EPWM_SYNC_OUT_PULSE_ON_CNTR_COMPARE_D - sync-out pulse is generated when
 *                                            time base counter equals compare
 *                                            D value.
 *  - EPWM_SYNC_OUT_PULSE_ON_DCA_EVT1_SYNC - sync-out pulse is generated by DCA
 *                                           event1 sync signal
 *  - EPWM_SYNC_OUT_PULSE_ON_DCB_EVT1_SYNC - sync-out pulse is generated by DCB
 *                                           event1 sync signal
 *  - EPWM_SYNC_OUT_PULSE_ON_ALL  - sync-out pulse is generated by all
 *                                  the above sources
*/
__STATIC_INLINE void
EPWM_enableSyncOutPulseSource(uint32_t base, uint16_t source)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT(source <= EPWM_SYNC_OUT_SOURCE_M);

    //
    // Enable selected EPWM Sync-Out Sources.
    //
    HWREGH(base + EPWM_O_SYNCOUTEN) = HWREGH(base + EPWM_O_SYNCOUTEN) |
                                      (uint16_t)source;
}

/**
 *  @brief Disables sync-out pulse source.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] source is the sync-out pulse source.
 *  @remarks
 *  This function disables the sync-out pulse source.
 *  Below valid values for param \b source can be OR'd together to disable
 *  multiple sync-out sources:
 *   - EPWM_SYNC_OUT_PULSE_ON_SOFTWARE - disables software as sync-out source
 *
 *   - EPWM_SYNC_OUT_PULSE_ON_CNTR_ZERO - disables counter equal to zero event
 *                                        as sync-out source
 *   - EPWM_SYNC_OUT_PULSE_ON_CNTR_COMPARE_B - disables counter equal to cmpb
 *                                             event as sync-out source
 *   - EPWM_SYNC_OUT_PULSE_ON_CNTR_COMPARE_C - disables counter equal to cmpc
 *                                             event as sync-out source
 *   - EPWM_SYNC_OUT_PULSE_ON_CNTR_COMPARE_D - disables counter equal to cmpd
 *                                             event as sync-out source
 *   - EPWM_SYNC_OUT_PULSE_ON_DCA_EVT1_SYNC - disables DCA event1 sync signal as
 *                                            sync-out source
 *   - EPWM_SYNC_OUT_PULSE_ON_DCB_EVT1_SYNC - disables DCB event1 sync signal as
 *                                            sync-out source
 *   - EPWM_SYNC_OUT_PULSE_ON_ALL - disables all the above sync-out sources
*/
__STATIC_INLINE void
EPWM_disableSyncOutPulseSource(uint32_t base, uint16_t source)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT(source <= EPWM_SYNC_OUT_SOURCE_M);

    //
    // Disable EPWM Sync-Out Sources.
    //
    HWREGH(base + EPWM_O_SYNCOUTEN) = HWREGH(base + EPWM_O_SYNCOUTEN) &
                                      ~((uint16_t)source);
}

/**
 *  @brief Set up the one-shot sync-out trigger source.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] trigger is the one-shot sync-out signal trigger source.
 *  @remarks
 *  This function sets the one-shot sync-out trigger source.
 *  Valid values for param \b trigger are:
 *   - EPWM_OSHT_SYNC_OUT_TRIG_SYNC - Trigger for one-shot sync-out signal is
 *                                     one-shot sync event.
 *   - EPWM_OSHT_SYNC_OUT_TRIG_RELOAD - Trigger for one-shot sync-out signal is
 *                                      one-shot reload event.
*/
__STATIC_INLINE void
EPWM_setOneShotSyncOutTrigger(uint32_t base,
                              EPWM_OneShotSyncOutTrigger trigger)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set source for One-Shot Sync-Out Pulse.
    //
    HWREGH(base + EPWM_O_TBCTL3) =
            (HWREGH(base + EPWM_O_TBCTL3) & ~(EPWM_TBCTL3_OSSFRCEN)) |
            (uint16_t)trigger;
}

/**
 *  @brief Set up the source for sync-in pulse.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] source is the sync-in pulse source.
 *  @remarks
 *  This function set the sync out pulse mode.
 *  Valid values for mode are:
 *   - EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM1-16 - sync-in pulse source can be
 *                                               any of the EPWM1-16 sync-out
 *                                               signal
 *   - EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_ECAP1-7 - sync-in pulse source can be
 *                                              selected as any of the ECAP1-16
 *                                              sync-out signal
 *   - EPWM_SYNC_IN_PULSE_SRC_INPUTXBAR_OUT5-6 - sync-in pulse source can be
 *                                               selected as any of the Input
 *                                               xbar out5-6 signal
 *   - EPWM_SYNC_IN_PULSE_SRC_ETHERCAT_SYNC0-1 - sync-in pulse source can be
 *                                               selected as any of the Ethercat
 *                                               sync0-1 signal
 *   - EPWM_SYNC_IN_PULSE_SRC_DISABLE - sync-in pulse is disabled for the
 *                                      EPWM module
*/
static inline void
EPWM_setSyncInPulseSource(uint32_t base, EPWM_SyncInPulseSource source)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set EPWM Sync-In Source Mode.
    //
    HWREGH(base + EPWM_O_SYNCINSEL) =
            (HWREGH(base + EPWM_O_SYNCINSEL) & (~EPWM_SYNCINSEL_SEL_M)) |
            ((uint16_t)source & EPWM_SYNCINSEL_SEL_M);
}

/**
 *  @brief Set PWM period load mode.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] loadMode is the PWM period load mode.
 *  @remarks
 *  This function sets the load mode for the PWM period. If loadMode is set to
 *  EPWM_PERIOD_SHADOW_LOAD, a write or read to the TBPRD (PWM Period count
 *  register) accesses the shadow register. If loadMode is set to
 *  EPWM_PERIOD_DIRECT_LOAD, a write or read to the TBPRD register accesses the
 *  register directly.
*/
__STATIC_INLINE void
EPWM_setPeriodLoadMode(uint32_t base, EPWM_PeriodLoadMode loadMode)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    if(loadMode == EPWM_PERIOD_SHADOW_LOAD)
    {
        //
        // Clear PRDLD
        //
        HWREGH(base + EPWM_O_TBCTL) &= ~EPWM_TBCTL_PRDLD;
    }
    else
    {
        //
        // Set PRDLD
        //
        HWREGH(base + EPWM_O_TBCTL) |= EPWM_TBCTL_PRDLD;
    }
}

/**
 *  @brief Enable phase shift load
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function enables loading of phase shift when the appropriate sync
 *  event occurs.
*/
__STATIC_INLINE void
EPWM_enablePhaseShiftLoad(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set PHSEN bit
    //
    HWREGH(base + EPWM_O_TBCTL) |= EPWM_TBCTL_PHSEN;
}

/**
 *  @brief Disable phase shift load
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function disables loading of phase shift.
*/
__STATIC_INLINE void
EPWM_disablePhaseShiftLoad(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Clear PHSEN bit
    //
    HWREGH(base + EPWM_O_TBCTL) &= ~EPWM_TBCTL_PHSEN;
}

/**
 *  @brief Set time base counter mode
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function sets up the time base counter mode.
 *  Valid values for counterMode are:
 *   - EPWM_COUNTER_MODE_UP          - Up - count mode.
 *   - EPWM_COUNTER_MODE_DOWN        - Down - count mode.
 *   - EPWM_COUNTER_MODE_UP_DOWN     - Up - down - count mode.
 *   - EPWM_COUNTER_MODE_STOP_FREEZE - Stop - Freeze counter.
*/
__STATIC_INLINE void
EPWM_setTimeBaseCounterMode(uint32_t base, EPWM_TimeBaseCountMode counterMode)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Write to CTRMODE bit
    //
    HWREGH(base + EPWM_O_TBCTL) =
            ((HWREGH(base + EPWM_O_TBCTL) & ~(EPWM_TBCTL_CTRMODE_M)) |
             ((uint16_t)counterMode));
}

/**
 *  @brief Set shadow to active period load on sync mode
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] shadowLoadMode is the shadow to active load mode.
 *  @remarks
 *  This function sets up the shadow to active Period register load mode with
 *  respect to a sync event. Valid values for shadowLoadMode are:
 *   - EPWM_SHADOW_LOAD_MODE_COUNTER_ZERO - shadow to active load occurs when
 *                                          time base counter reaches 0.
 *   - EPWM_SHADOW_LOAD_MODE_COUNTER_SYNC - shadow to active load occurs when
 *                                          time base counter reaches 0 and a
 *                                          SYNC occurs.
 *   - EPWM_SHADOW_LOAD_MODE_SYNC         - shadow to active load occurs only
 *                                          when a SYNC occurs.
*/
__STATIC_INLINE void
EPWM_selectPeriodLoadEvent(uint32_t base,
                           EPWM_PeriodShadowLoadMode shadowLoadMode)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Write to PRDLDSYNC bit
    //
    HWREGH(base + EPWM_O_TBCTL2) =
                ((HWREGH(base + EPWM_O_TBCTL2) & ~(EPWM_TBCTL2_PRDLDSYNC_M)) |
                 ((uint16_t)shadowLoadMode << EPWM_TBCTL2_PRDLDSYNC_S));
}

/**
 *  @brief Enable one shot sync mode
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function enables one shot sync mode.
*/
__STATIC_INLINE void
EPWM_enableOneShotSync(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set OSHTSYNCMODE bit
    //
    HWREGH(base + EPWM_O_TBCTL2) |= EPWM_TBCTL2_OSHTSYNCMODE;
}

/**
 *  @brief Disable one shot sync mode
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 * This function disables one shot sync mode.
*/
__STATIC_INLINE void
EPWM_disableOneShotSync(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Clear OSHTSYNCMODE bit
    //
    HWREGH(base + EPWM_O_TBCTL2) &= ~EPWM_TBCTL2_OSHTSYNCMODE;
}

/**
 *  @brief Start one shot sync mode
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 * This function propagates a one shot sync pulse.
*/
__STATIC_INLINE void
EPWM_startOneShotSync(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set OSHTSYNC bit
    //
    HWREGH(base + EPWM_O_TBCTL2) |= EPWM_TBCTL2_OSHTSYNC;
}

/**
 *  @brief Returns time base counter value.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function returns the current value of the time base counter.
 *
 *  @return returns time base counter value
*/
__STATIC_INLINE uint16_t
EPWM_getTimeBaseCounterValue(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Returns TBCTR value
    //
    return(HWREGH(base + EPWM_O_TBCTR));
}

/**
 *  @brief Return time base counter maximum status.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function returns the status of the time base max counter.
 *
 *  @return Returns true if the counter has reached 0xFFFF.
 *          Returns false if the counter hasn't reached 0xFFFF.
*/
__STATIC_INLINE bool
EPWM_getTimeBaseCounterOverflowStatus(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Return true if CTRMAX bit is set, false otherwise
    //
    return(((HWREGH(base + EPWM_O_TBSTS) & EPWM_TBSTS_CTRMAX) ==
            EPWM_TBSTS_CTRMAX) ? true : false);
}

/**
 *  @brief Clear max time base counter event.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function clears the max time base counter latch event. The latch event
 *  occurs when the time base counter reaches its maximum value of 0xFFFF.
*/
__STATIC_INLINE void
EPWM_clearTimeBaseCounterOverflowEvent(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set CTRMAX bit
    //
    HWREGH(base + EPWM_O_TBSTS) = EPWM_TBSTS_CTRMAX;
}

/**
 *  @brief Return external sync signal status.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function returns the external sync signal status.
 *
 *  @return Returns true if if an external sync signal event
 *          Returns false if there is no event.
*/
__STATIC_INLINE bool
EPWM_getSyncStatus(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Return true if SYNCI bit is set, false otherwise
    //
    return(((HWREGH(base + EPWM_O_TBSTS) & EPWM_TBSTS_SYNCI) ==
            EPWM_TBSTS_SYNCI) ? true : false);
}

/**
 *  @brief Clear external sync signal event.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function clears the external sync signal latch event.
*/
__STATIC_INLINE void
EPWM_clearSyncEvent(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set SYNCI bit
    //
    HWREGH(base + EPWM_O_TBSTS) = EPWM_TBSTS_SYNCI;
}

/**
 *  @brief Return time base counter direction.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *
 *  This function returns the direction of the time base counter.
 *
 *  @return returns EPWM_TIME_BASE_STATUS_COUNT_UP if the counter is counting
 *                  up or EPWM_TIME_BASE_STATUS_COUNT_DOWN if the counter is
 *                  counting down.
*/
__STATIC_INLINE uint16_t
EPWM_getTimeBaseCounterDirection(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Return CTRDIR bit
    //
    return(HWREGH(base + EPWM_O_TBSTS) & EPWM_TBSTS_CTRDIR);
}

/**
 *  @brief Sets the phase shift offset counter value.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] phaseCount is the phase shift count value.
 *  @remarks
 *  This function sets the 16 bit time-base counter phase of the ePWM relative
 *  to the time-base that is supplying the synchronization input signal. Call
 *  the EPWM_enablePhaseShiftLoad() function to enable loading of the
 *  phaseCount phase shift value when a sync event occurs.
*/
__STATIC_INLINE void
EPWM_setPhaseShift(uint32_t base, uint16_t phaseCount)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Write to TBPHS bit
    //
    HWREG(base + EPWM_O_TBPHS) =
                   ((HWREG(base + EPWM_O_TBPHS) &
                    ~((uint32_t)EPWM_TBPHS_TBPHS_M)) |
                    ((uint32_t)phaseCount << EPWM_TBPHS_TBPHS_S));
}

/**
 *  @brief Sets the PWM period count.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] periodCount is period count value.
 *  @remarks
 *  This function sets the period of the PWM count. The value of periodCount is
 *  the value written to the register. User should map the desired period or
 *  frequency of the waveform into the correct periodCount.
 *  Invoke the function EPWM_selectPeriodLoadEvent() with the appropriate
 *  parameter to set the load mode of the Period count. periodCount has a
 *  maximum valid value of 0xFFFF
*/
__STATIC_INLINE void
EPWM_setTimeBasePeriod(uint32_t base, uint16_t periodCount)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Write to TBPRD bit
    //
    HWREGH(base + EPWM_O_TBPRD) = periodCount;
}

/**
 *  @brief Gets the PWM period count.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] periodCount is period count value.
 *  @remarks
 *
 *  This function gets the period of the PWM count.
 *
 *  @return The period count value.
*/
__STATIC_INLINE uint16_t
EPWM_getTimeBasePeriod(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Read from TBPRD bit
    //
    return(HWREGH(base + EPWM_O_TBPRD));
}






/**
 *  @brief Set the output mode of the synchronization signal event
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] Channel .
 *  @param [in] Mode.
 *
 *  @return void .
*/
static inline void
EPWM_setControlSyncEventOutputMode_Extend(uint32_t base, EPWM_SyncEventOutputChannel_Extend Channel,EPWM_SyncEventOutputMode_Extend Mode)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    if( Mode == EPWM_Event_OutputMode_Software)
    {
        HWREG(base + EPWM_O_EPWM_TBCTR_CHG) |=   (Mode<<(Channel-1));

    }else {

        HWREG(base + EPWM_O_EPWM_TBCTR_CHG) &=   (~(!Mode<<(Channel-1)));
	}


}



/**
 *  @brief Set the Load Mode of the synchronization signal event
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] Channel .
 *  @param [in] Mode.
 *
 *  @return void .
*/
static inline void
EPWM_setControlSyncEventLoadMode_Extend(uint32_t base, EPWM_SyncEventOutputChannel_Extend Channel,EPWM_SyncEventLoadMode_Extend Mode)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    if( Mode == EPWM_Event_LoadMode_ZERO_PERIOD )
    {
        HWREG(base + EPWM_O_EPWM_TBCTR_CHG)  =   (HWREG(base + EPWM_O_EPWM_TBCTR_CHG) & (~0x00c0)) | (Mode<<(Channel+1));

    }else {

        HWREG(base + EPWM_O_EPWM_TBCTR_CHG)  =   (HWREG(base + EPWM_O_EPWM_TBCTR_CHG) & (~0x000c)) & (~(!Mode<<(Channel+1)));
	}


}

/**
 *  @brief Select the polarity of the event output
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] Channel .
 *  @param [in] Polarity.
 *  This configuration is only valid when the output mode of
 *  the synchronization signal is determined by software.
 *  @return void .
*/
static inline void
EPWM_setControlSyncEventOutputPolarity_Extend(uint32_t base, EPWM_SyncEventOutputChannel_Extend Channel,EPWM_SyncEventOutputPolarity_Extend Polarity)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    if( Polarity == EPWM_Event_OutputPolarity_HIGH )
    {
        HWREG(base + EPWM_O_EPWM_TBCTR_CHG) |=  (Polarity<<Channel);

    }else
    {
        HWREG(base + EPWM_O_EPWM_TBCTR_CHG) &=  (~(!Polarity<<Channel));
    }


}


#ifdef __cplusplus
}
#endif


#endif /* DEVICE_DRIVERLIB_EPWM_TB_SYNC_H_ */
