/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    epwm_tz_dc.h
*   @brief   
*
*   commit history
*   20240804, Jason, initial creation
*/

#ifndef DEVICE_DRIVERLIB_EPWM_TZ_DC_H_
#define DEVICE_DRIVERLIB_EPWM_TZ_DC_H_

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include <stdbool.h>
#include <stdint.h>
#include "gs32_version.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "inc/hw_epwm.h"
#include "inc/hw_sysctl_ahb.h"
#include "epwm_define.h"
#include "debug.h"
#include "sysctl.h"


//
// Trip Zone module related APIs
//

/**
 * @brief Enables Trip Zone signal.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  tzSignal is the Trip Zone signal.
 * @remarks
 *  This function enables the Trip Zone signals specified by tzSignal as a
 *  source for the Trip Zone module.
 *  Valid values for tzSignal are:
 *    - EPWM_TZ_SIGNAL_CBC1       - TZ1 Cycle By Cycle
 *    - EPWM_TZ_SIGNAL_CBC2       - TZ2 Cycle By Cycle
 *    - EPWM_TZ_SIGNAL_CBC3       - TZ3 Cycle By Cycle
 *    - EPWM_TZ_SIGNAL_CBC4       - TZ4 Cycle By Cycle
 *    - EPWM_TZ_SIGNAL_CBC5       - TZ5 Cycle By Cycle
 *    - EPWM_TZ_SIGNAL_CBC6       - TZ6 Cycle By Cycle
 *    - EPWM_TZ_SIGNAL_DCAEVT2    - DCAEVT2 Cycle By Cycle
 *    - EPWM_TZ_SIGNAL_DCBEVT2    - DCBEVT2 Cycle By Cycle
 *    - EPWM_TZ_SIGNAL_OSHT1      - One-shot TZ1
 *    - EPWM_TZ_SIGNAL_OSHT2      - One-shot TZ2
 *    - EPWM_TZ_SIGNAL_OSHT3      - One-shot TZ3
 *    - EPWM_TZ_SIGNAL_OSHT4      - One-shot TZ4
 *    - EPWM_TZ_SIGNAL_OSHT5      - One-shot TZ5
 *    - EPWM_TZ_SIGNAL_OSHT6      - One-shot TZ6
 *    - EPWM_TZ_SIGNAL_DCAEVT1    - One-shot DCAEVT1
 *    - EPWM_TZ_SIGNAL_DCBEVT1    - One-shot DCBEVT1
 *
 *   A logical OR of the valid values can be passed as the tzSignal
 *            parameter.
 */
__STATIC_INLINE void
EPWM_enableTripZoneSignals(uint32_t base, uint16_t tzSignal)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set the trip zone bits
    //
    EALLOW;
    HWREGH(base + EPWM_O_TZSEL) |= tzSignal;
    EDIS;
}

/**
 * @brief Disables Trip Zone signal.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  tzSignal is the Trip Zone signal.
 * @remarks
 *  This function enables the Trip Zone signals specified by tzSignal as a
 *  source for the Trip Zone module.
 *  Valid values for tzSignal are:
 *    - EPWM_TZ_SIGNAL_CBC1       - TZ1 Cycle By Cycle
 *    - EPWM_TZ_SIGNAL_CBC2       - TZ2 Cycle By Cycle
 *    - EPWM_TZ_SIGNAL_CBC3       - TZ3 Cycle By Cycle
 *    - EPWM_TZ_SIGNAL_CBC4       - TZ4 Cycle By Cycle
 *    - EPWM_TZ_SIGNAL_CBC5       - TZ5 Cycle By Cycle
 *    - EPWM_TZ_SIGNAL_CBC6       - TZ6 Cycle By Cycle
 *    - EPWM_TZ_SIGNAL_DCAEVT2    - DCAEVT2 Cycle By Cycle
 *    - EPWM_TZ_SIGNAL_DCBEVT2    - DCBEVT2 Cycle By Cycle
 *    - EPWM_TZ_SIGNAL_OSHT1      - One-shot TZ1
 *    - EPWM_TZ_SIGNAL_OSHT2      - One-shot TZ2
 *    - EPWM_TZ_SIGNAL_OSHT3      - One-shot TZ3
 *    - EPWM_TZ_SIGNAL_OSHT4      - One-shot TZ4
 *    - EPWM_TZ_SIGNAL_OSHT5      - One-shot TZ5
 *    - EPWM_TZ_SIGNAL_OSHT6      - One-shot TZ6
 *    - EPWM_TZ_SIGNAL_DCAEVT1    - One-shot DCAEVT1
 *    - EPWM_TZ_SIGNAL_DCBEVT1    - One-shot DCBEVT1
 *
 *   A logical OR of the valid values can be passed as the tzSignal
 *            parameter.
 */
__STATIC_INLINE void
EPWM_disableTripZoneSignals(uint32_t base, uint16_t tzSignal)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Clear the trip zone bits
    //
    EALLOW;
    HWREGH(base + EPWM_O_TZSEL) &= ~tzSignal;
    EDIS;
}
#if (EPWM_VERSION == 30)
//*****************************************************************************
//
//! Enables Trip Zone 2 signal (TZSEL2).
//!
//! \param base is the base address of the EPWM module.
//! \param tzSignal is the Trip Zone signal.
//!
//! This function enables the Trip Zone signals specified by tzSignal as a
//! source for the Trip Zone module.
//! Valid values for tzSignal are:
//!   - EPWM_TZ_SIGNAL_CAPEVT_OST    - One-shot CAPEVT
//!   - EPWM_TZ_SIGNAL_CAPEVT_CBC    - Cycle By Cycle CAPEVT
//!
//! \b note:  A logical OR of the valid values can be passed as the tzSignal
//!           parameter.
//!
//! \return None.
//
//*****************************************************************************
static inline void
EPWM_enableTripZone2Signals(uint32_t base, uint16_t tzSignal)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set the trip zone bits
    //
    EALLOW;
    HWREGH(base + EPWM_O_TZSEL2) |= tzSignal;
    EDIS;
}

//*****************************************************************************
//
//! Disables Trip Zone 2 signal (TZSEL2).
//!
//! \param base is the base address of the EPWM module.
//! \param tzSignal is the Trip Zone signal.
//!
//! This function disables the Trip Zone signals specified by tzSignal as a
//! source for the Trip Zone module.
//! Valid values for tzSignal are:
//!   - EPWM_TZ_SIGNAL_CAPEVT_OST    - One-shot CAPEVT
//!   - EPWM_TZ_SIGNAL_CAPEVT_CBC    - Cycle By Cycle CAPEVT
//!
//! \b note:  A logical OR of the valid values can be passed as the tzSignal
//!           parameter.
//!
//! \return None.
//
//*****************************************************************************
static inline void
EPWM_disableTripZone2Signals(uint32_t base, uint16_t tzSignal)
{
    //
    // Clear the trip zone bits
    //
    EALLOW;
    HWREGH(base + EPWM_O_TZSEL2) &= ~tzSignal;
    EDIS;
}
#endif
/**
 * @brief Set Digital compare conditions that cause Trip Zone event.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  dcType is the Digital compare output type.
 * @param [in]  dcEvent is the Digital Compare output event.
 * @remarks
 *  This function sets up the Digital Compare output Trip Zone event sources.
 *  The dcType variable specifies the event source to be whether Digital
 *  Compare output A or Digital Compare output B. The dcEvent parameter
 *  specifies the event that causes Trip Zone.
 *  Valid values for the parameters are:
 *   - dcType
 *       - EPWM_TZ_DC_OUTPUT_A1     - Digital Compare output 1 A
 *       - EPWM_TZ_DC_OUTPUT_A2     - Digital Compare output 2 A
 *       - EPWM_TZ_DC_OUTPUT_B1     - Digital Compare output 1 B
 *       - EPWM_TZ_DC_OUTPUT_B2     - Digital Compare output 2 B
 *   - dcEvent
 *       - EPWM_TZ_EVENT_DC_DISABLED         - Event Trigger is disabled
 *       - EPWM_TZ_EVENT_DCXH_LOW            - Trigger event when DCxH low
 *       - EPWM_TZ_EVENT_DCXH_HIGH           - Trigger event when DCxH high
 *       - EPWM_TZ_EVENT_DCXL_LOW            - Trigger event when DCxL low
 *       - EPWM_TZ_EVENT_DCXL_HIGH           - Trigger event when DCxL high
 *       - EPWM_TZ_EVENT_DCXL_HIGH_DCXH_LOW  - Trigger event when DCxL high
 *                                             DCxH low
 *
 *    x in DCxH/DCxL represents DCAH/DCAL or DCBH/DCBL
 */
__STATIC_INLINE void
EPWM_setTripZoneDigitalCompareEventCondition(uint32_t base,
                                 EPWM_TripZoneDigitalCompareOutput dcType,
                                 EPWM_TripZoneDigitalCompareOutputEvent dcEvent)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set Digital Compare Events conditions that cause a Digital Compare trip
    //
    EALLOW;
    HWREGH(base + EPWM_O_TZDCSEL) =
           ((HWREGH(base + EPWM_O_TZDCSEL) & ~(0x7U << (uint16_t)dcType)) |
            ((uint16_t)dcEvent << (uint16_t)dcType));
    EDIS;
}

/**
 * @brief Enable advanced Trip Zone event Action.
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 *  This function enables the advanced actions of the Trip Zone events. The
 *  advanced features combine the trip zone events with the direction of the
 *  counter.
 */
__STATIC_INLINE void
EPWM_enableTripZoneAdvAction(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Enable Advanced feature. Set ETZE bit
    //
    EALLOW;
    HWREGH(base + EPWM_O_TZCTL2) |= EPWM_TZCTL2_ETZE;
    EDIS;
}

/**
 * @brief Disable advanced Trip Zone event Action.
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 * This function disables the advanced actions of the Trip Zone events.
 */
__STATIC_INLINE void
EPWM_disableTripZoneAdvAction(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Disable Advanced feature. clear ETZE bit
    //
    EALLOW;
    HWREGH(base + EPWM_O_TZCTL2) &= ~EPWM_TZCTL2_ETZE;
    EDIS;
}

/**
 * @brief Set Trip Zone Action.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  tzEvent is the Trip Zone event type.
 * @param [in]  tzAction is the Trip zone Action.
 * @remarks
 *  This function sets the Trip Zone Action to be taken when a Trip Zone event
 *  occurs.
 *  Valid values for the parameters are:
 *   - tzEvent
 *       - EPWM_TZ_ACTION_EVENT_DCBEVT2  - DCBEVT2 (Digital Compare B event 2)
 *       - EPWM_TZ_ACTION_EVENT_DCBEVT1  - DCBEVT1 (Digital Compare B event 1)
 *       - EPWM_TZ_ACTION_EVENT_DCAEVT2  - DCAEVT2 (Digital Compare A event 2)
 *       - EPWM_TZ_ACTION_EVENT_DCAEVT1  - DCAEVT1 (Digital Compare A event 1)
 *       - EPWM_TZ_ACTION_EVENT_TZB      - TZ1 - TZ6, DCBEVT2, DCBEVT1
 *       - EPWM_TZ_ACTION_EVENT_TZA      - TZ1 - TZ6, DCAEVT2, DCAEVT1
 *   - tzAction
 *       - EPWM_TZ_ACTION_HIGH_Z         - high impedance output
 *       - EPWM_TZ_ACTION_HIGH           - high output
 *       - EPWM_TZ_ACTION_LOW            - low low
 *       - EPWM_TZ_ACTION_DISABLE        - disable action
 *
 *   Disable the advanced Trip Zone event using
 *        EPWM_disableTripZoneAdvAction()  before calling this function.
 *   This function operates on both ePWMA and ePWMB depending on the
 *        tzEvent parameter.
 */
__STATIC_INLINE void
EPWM_setTripZoneAction(uint32_t base, EPWM_TripZoneEvent tzEvent,
                       EPWM_TripZoneAction tzAction)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set the Action for Trip Zone events
    //
    EALLOW;
    HWREGH(base + EPWM_O_TZCTL) =
         ((HWREGH(base + EPWM_O_TZCTL) & ~(0x3U << (uint16_t)tzEvent)) |
          ((uint16_t)tzAction << (uint16_t)tzEvent)) ;
    EDIS;
}

/**
 * @brief Set Advanced Trip Zone Action.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  tzEvent is the Trip Zone event type.
 * @param [in]  tzAction is the Trip zone Action.
 * @remarks
 *  This function sets the Advanced Trip Zone Action to be taken when an
 *  advanced Trip Zone event occurs.
 *
 *  Valid values for the parameters are:
 *   - tzAdvEvent
 *       - EPWM_TZ_ADV_ACTION_EVENT_TZB_D  - TZ1 - TZ6, DCBEVT2, DCBEVT1 while
 *                                              counting down
 *       - EPWM_TZ_ADV_ACTION_EVENT_TZB_U  - TZ1 - TZ6, DCBEVT2, DCBEVT1 while
 *                                              counting up
 *       - EPWM_TZ_ADV_ACTION_EVENT_TZA_D  - TZ1 - TZ6, DCAEVT2, DCAEVT1 while
 *                                              counting down
 *       - EPWM_TZ_ADV_ACTION_EVENT_TZA_U  - TZ1 - TZ6, DCAEVT2, DCAEVT1 while
 *                                              counting up
 *   - tzAdvAction
 *       - EPWM_TZ_ADV_ACTION_HIGH_Z    - high impedance output
 *       - EPWM_TZ_ADV_ACTION_HIGH      - high voltage state
 *       - EPWM_TZ_ADV_ACTION_LOW       - low voltage state
 *       - EPWM_TZ_ADV_ACTION_TOGGLE    - Toggle output
 *       - EPWM_TZ_ADV_ACTION_DISABLE   - disable action
 *
 *   This function enables the advanced Trip Zone event.
 *
 *   This function operates on both ePWMA and ePWMB depending on the
 *        tzAdvEvent  parameter.
 *   Advanced Trip Zone events take into consideration the direction of
 *        the counter in addition to Trip Zone events.
 */
__STATIC_INLINE void
EPWM_setTripZoneAdvAction(uint32_t base, EPWM_TripZoneAdvancedEvent tzAdvEvent,
                          EPWM_TripZoneAdvancedAction tzAdvAction)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set the Advanced Action for Trip Zone events
    //
    EALLOW;
    HWREGH(base + EPWM_O_TZCTL2) =
       ((HWREGH(base + EPWM_O_TZCTL2) & ~(0x7U << (uint16_t)tzAdvEvent)) |
        ((uint16_t)tzAdvAction << (uint16_t)tzAdvEvent));

    HWREGH(base + EPWM_O_TZCTL2) |= EPWM_TZCTL2_ETZE;
    EDIS;
}

/**
 * @brief Set Advanced Digital Compare Trip Zone Action on ePWMA.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  tzAdvDCEvent is the Digital Compare Trip Zone event type.
 * @param [in]  tzAdvDCAction is the Digital Compare Trip zone Action.
 * @remarks
 *  This function sets the Digital Compare (DC) Advanced Trip Zone Action to be
 *  taken on ePWMA when an advanced Digital Compare Trip Zone A event occurs.
 *  Valid values for the parameters are:
 *   - tzAdvDCEvent
 *       - EPWM_TZ_ADV_ACTION_EVENT_DCxEVT2_D  - Digital Compare event A2 while
 *                                                  counting down
 *       - EPWM_TZ_ADV_ACTION_EVENT_DCxEVT2_U  - Digital Compare event A2 while
 *                                                  counting up
 *       - EPWM_TZ_ADV_ACTION_EVENT_DCxEVT1_D  - Digital Compare event A1 while
 *                                                  counting down
 *       - EPWM_TZ_ADV_ACTION_EVENT_DCxEVT1_U  - Digital Compare event A1 while
 *                                                  counting up
 *   - tzAdvDCAction
 *       - EPWM_TZ_ADV_ACTION_HIGH_Z    - high impedance output
 *       - EPWM_TZ_ADV_ACTION_HIGH      - high voltage state
 *       - EPWM_TZ_ADV_ACTION_LOW       - low voltage state
 *       - EPWM_TZ_ADV_ACTION_TOGGLE    - Toggle output
 *       - EPWM_TZ_ADV_ACTION_DISABLE   - disable action
 *
 *   This function enables the advanced Trip Zone event.
 *
 *   Advanced Trip Zone events take into consideration the direction of
 *        the counter in addition to Digital Compare Trip Zone events.
 */
__STATIC_INLINE void
EPWM_setTripZoneAdvDigitalCompareActionA(uint32_t base,
                              EPWM_TripZoneAdvDigitalCompareEvent tzAdvDCEvent,
                              EPWM_TripZoneAdvancedAction tzAdvDCAction)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set the Advanced Action for Trip Zone events
    //
    EALLOW;
    HWREGH(base + EPWM_O_TZCTLDCA) =
      ((HWREGH(base + EPWM_O_TZCTLDCA) & ~(0x7U << (uint16_t)tzAdvDCEvent)) |
       ((uint16_t)tzAdvDCAction << (uint16_t)tzAdvDCEvent));

    HWREGH(base + EPWM_O_TZCTL2) |= EPWM_TZCTL2_ETZE;
    EDIS;
}

/**
 * @brief Set Advanced Digital Compare Trip Zone Action on ePWMB.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  tzAdvDCEvent is the Digital Compare Trip Zone event type.
 * @param [in]  tzAdvDCAction is the Digital Compare Trip zone Action.
 * @remarks
 *  This function sets the Digital Compare (DC) Advanced Trip Zone Action to be
 *  taken on ePWMB when an advanced Digital Compare Trip Zone B event occurs.
 *  Valid values for the parameters are:
 *   - tzAdvDCEvent
 *       - EPWM_TZ_ADV_ACTION_EVENT_DCxEVT2_D  - Digital Compare event B2 while
 *                                                  counting down
 *       - EPWM_TZ_ADV_ACTION_EVENT_DCxEVT2_U  - Digital Compare event B2 while
 *                                                  counting up
 *       - EPWM_TZ_ADV_ACTION_EVENT_DCxEVT1_D  - Digital Compare event B1 while
 *                                                  counting down
 *       - EPWM_TZ_ADV_ACTION_EVENT_DCxEVT1_U  - Digital Compare event B1 while
 *                                                  counting up
 *   - tzAdvDCAction
 *       - EPWM_TZ_ADV_ACTION_HIGH_Z    - high impedance output
 *       - EPWM_TZ_ADV_ACTION_HIGH      - high voltage state
 *       - EPWM_TZ_ADV_ACTION_LOW       - low voltage state
 *       - EPWM_TZ_ADV_ACTION_TOGGLE    - Toggle output
 *       - EPWM_TZ_ADV_ACTION_DISABLE   - disable action
 *
 *   This function enables the advanced Trip Zone event.
 *
 *   Advanced Trip Zone events take into consideration the direction of
 *        the counter in addition to Digital Compare Trip Zone events.
 */
__STATIC_INLINE void EPWM_setTripZoneAdvDigitalCompareActionB(uint32_t base,
                              EPWM_TripZoneAdvDigitalCompareEvent tzAdvDCEvent,
                              EPWM_TripZoneAdvancedAction tzAdvDCAction)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set the Advanced Action for Trip Zone events
    //
    EALLOW;
    HWREGH(base + EPWM_O_TZCTLDCB) =
     ((HWREGH(base + EPWM_O_TZCTLDCB) & ~(0x7U << (uint16_t)tzAdvDCEvent)) |
      ((uint16_t)tzAdvDCAction << (uint16_t)tzAdvDCEvent));

    HWREGH(base + EPWM_O_TZCTL2) |= EPWM_TZCTL2_ETZE;
    EDIS;
}

/**
 * @brief Enable Trip Zone interrupts.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  tzInterrupt is the Trip Zone interrupt.
 * @remarks
 *  This function enables the Trip Zone interrupts.
 *  Valid values for tzInterrupt are:
 *    - EPWM_TZ_INTERRUPT_CBC     - Trip Zones Cycle By Cycle interrupt
 *    - EPWM_TZ_INTERRUPT_OST     - Trip Zones One Shot interrupt
 *    - EPWM_TZ_INTERRUPT_DCAEVT1 - Digital Compare A Event 1 interrupt
 *    - EPWM_TZ_INTERRUPT_DCAEVT2 - Digital Compare A Event 2 interrupt
 *    - EPWM_TZ_INTERRUPT_DCBEVT1 - Digital Compare B Event 1 interrupt
 *    - EPWM_TZ_INTERRUPT_DCBEVT2 - Digital Compare B Event 2 interrupt
 *
 *  \b note:  A logical OR of the valid values can be passed as the tzInterrupt
 *            parameter.
 */
__STATIC_INLINE void
EPWM_enableTripZoneInterrupt(uint32_t base, uint16_t tzInterrupt)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT((tzInterrupt > 0U) && (tzInterrupt < 0x80U));

    //
    // Enable Trip zone interrupts
    //
    EALLOW;
    HWREGH(base + EPWM_O_TZEINT) |= tzInterrupt;
    EDIS;
}



/**
 * @brief Disable Trip Zone interrupts.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  tzInterrupt is the Trip Zone interrupt.
 * @remarks
 *  This function enables the Trip Zone interrupts.
 *  Valid values for tzInterrupt are:
 *    - EPWM_TZ_INTERRUPT_CBC     - Trip Zones Cycle By Cycle interrupt
 *    - EPWM_TZ_INTERRUPT_OST     - Trip Zones One Shot interrupt
 *    - EPWM_TZ_INTERRUPT_DCAEVT1 - Digital Compare A Event 1 interrupt
 *    - EPWM_TZ_INTERRUPT_DCAEVT2 - Digital Compare A Event 2 interrupt
 *    - EPWM_TZ_INTERRUPT_DCBEVT1 - Digital Compare B Event 1 interrupt
 *    - EPWM_TZ_INTERRUPT_DCBEVT2 - Digital Compare B Event 2 interrupt
 *
 *  A logical OR of the valid values can be passed as the tzInterrupt
 *            parameter.
 */
__STATIC_INLINE void
EPWM_disableTripZoneInterrupt(uint32_t base, uint16_t tzInterrupt)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT((tzInterrupt > 0U) && (tzInterrupt < 0x80U));

    //
    // Disable Trip zone interrupts
    //
    EALLOW;
    HWREGH(base + EPWM_O_TZEINT) &= ~tzInterrupt;
    EDIS;
}

/**
 * @brief Gets the Trip Zone status flag
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 *  This function returns the Trip Zone status flag.
 *
 * @return The function returns the following or the bitwise OR value
 *          of the following values.
 *          - EPWM_TZ_INTERRUPT    - Trip Zone interrupt was generated
 *                                   due to the following TZ events.
 *          - EPWM_TZ_FLAG_CBC     - Trip Zones Cycle By Cycle event status flag
 *          - EPWM_TZ_FLAG_OST     - Trip Zones One Shot event status flag
 *          - EPWM_TZ_FLAG_DCAEVT1 - Digital Compare A Event 1 status flag
 *          - EPWM_TZ_FLAG_DCAEVT2 - Digital Compare A Event 2 status flag
 *          - EPWM_TZ_FLAG_DCBEVT1 - Digital Compare B Event 1 status flag
 *          - EPWM_TZ_FLAG_DCBEVT2 - Digital Compare B Event 2 status flag
 *          - EPWM_TZ_FLAG_CAPEVT  - Trip Zones Capture Event flag
 */
__STATIC_INLINE uint16_t
EPWM_getTripZoneFlagStatus(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Return the Trip zone flag status
    //
#if (EPWM_VERSION == 30)
    return(HWREGH(base + EPWM_O_TZFLG) & 0xFFU);
#else
    return(HWREGH(base + EPWM_O_TZFLG) & 0x7FU);
#endif

}

/**
 * @brief Gets the Trip Zone Cycle by Cycle flag status
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 *  This function returns the specific Cycle by Cycle Trip Zone flag
 *  status.
 *
 * @return The function returns the following values.
 *            - EPWM_TZ_CBC_FLAG_1     - CBC 1 status flag
 *            - EPWM_TZ_CBC_FLAG_2     - CBC 2 status flag
 *            - EPWM_TZ_CBC_FLAG_3     - CBC 3 status flag
 *            - EPWM_TZ_CBC_FLAG_4     - CBC 4 status flag
 *            - EPWM_TZ_CBC_FLAG_5     - CBC 5 status flag
 *            - EPWM_TZ_CBC_FLAG_6     - CBC 6 status flag
 *            - EPWM_TZ_CBC_FLAG_DCAEVT2  - CBC status flag for Digital compare
 *                                                 event A2
 *            - EPWM_TZ_CBC_FLAG_DCBEVT2  - CBC status flag for Digital compare
 *                                                 event B2
 *            - EPWM_TZ_CBC_FLAG_CAPEVT - CBC status flag for
 *                                                 capture event
 */
__STATIC_INLINE uint16_t
EPWM_getCycleByCycleTripZoneFlagStatus(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Return the Cycle By Cycle Trip zone flag status
    //
#if (EPWM_VERSION == 30)
    return(HWREGH(base + EPWM_O_TZCBCFLG) & 0x1FFU);
#else
    return(HWREGH(base + EPWM_O_TZCBCFLG) & 0xFFU);
#endif
}

/**
 * @brief Gets the Trip Zone One Shot flag status
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 *  This function returns the specific One Shot Trip Zone flag status.
 *
 * @return The function returns the bitwise OR of the following flags.
 *            - EPWM_TZ_OST_FLAG_OST1     - OST status flag for OST1
 *            - EPWM_TZ_OST_FLAG_OST2     - OST status flag for OST2
 *            - EPWM_TZ_OST_FLAG_OST3     - OST status flag for OST3
 *            - EPWM_TZ_OST_FLAG_OST4     - OST status flag for OST4
 *            - EPWM_TZ_OST_FLAG_OST5     - OST status flag for OST5
 *            - EPWM_TZ_OST_FLAG_OST6     - OST status flag for OST6
 *            - EPWM_TZ_OST_FLAG_DCAEVT1  - OST status flag for Digital
 *                                                  compare event A1
 *            - EPWM_TZ_OST_FLAG_DCBEVT1  - OST status flag for Digital
 *                                                  compare event B1
 *            - EPWM_TZ_OST_FLAG_CAPEVT   - OST status flag for
 *                                                  capture event
 */
__STATIC_INLINE uint16_t
EPWM_getOneShotTripZoneFlagStatus(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Return the One Shot Trip zone flag status
    //
#if (EPWM_VERSION == 30)
    return(HWREGH(base + EPWM_O_TZOSTFLG) & 0x1FFU);
#else
    return(HWREGH(base + EPWM_O_TZOSTFLG) & 0xFFU);
#endif
}

/**
 * @brief Set the Trip Zone CBC pulse clear event.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  clearEvent is the CBC trip zone clear event.
 * @remarks
 *  This function set the event which automatically clears the
 *  CBC (Cycle by Cycle) latch.
 *  Valid values for clearEvent are:
 *    - EPWM_TZ_CBC_PULSE_CLR_CNTR_ZERO         - Clear CBC pulse when counter
 *                                                   equals zero
 *    - EPWM_TZ_CBC_PULSE_CLR_CNTR_PERIOD       - Clear CBC pulse when counter
 *                                                   equals period
 *    - EPWM_TZ_CBC_PULSE_CLR_CNTR_ZERO_PERIOD  - Clear CBC pulse when counter
 *                                                   equals zero or period
 */
__STATIC_INLINE void
EPWM_selectCycleByCycleTripZoneClearEvent(uint32_t base,
                                 EPWM_CycleByCycleTripZoneClearMode clearEvent)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set the Cycle by Cycle Trip Latch mode
    //
    EALLOW;
    HWREGH(base + EPWM_O_TZCLR) =
                 ((HWREGH(base + EPWM_O_TZCLR) & ~EPWM_TZCLR_CBCPULSE_M) |
                  ((uint16_t)clearEvent << EPWM_TZCLR_CBCPULSE_S));
    EDIS;
}

/**
 * @brief Clear Trip Zone flag
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  tzFlags is the Trip Zone flags.
 * @remarks
 *  This function clears the Trip Zone flags
 *  Valid values for tzFlags are:
 *    - EPWM_TZ_INTERRUPT    - Global Trip Zone interrupt flag
 *    - EPWM_TZ_FLAG_CBC     - Trip Zones Cycle By Cycle flag
 *    - EPWM_TZ_FLAG_OST     - Trip Zones One Shot flag
 *    - EPWM_TZ_FLAG_DCAEVT1 - Digital Compare A Event 1 flag
 *    - EPWM_TZ_FLAG_DCAEVT2 - Digital Compare A Event 2 flag
 *    - EPWM_TZ_FLAG_DCBEVT1 - Digital Compare B Event 1 flag
 *    - EPWM_TZ_FLAG_DCBEVT2 - Digital Compare B Event 2 flag
 *    - EPWM_TZ_FLAG_CAPEVT  - Trip Zones Capture Event flag
 *
 *  A bitwise OR of the valid values can be passed as the tzFlags
 *  parameter.
 */
__STATIC_INLINE void
EPWM_clearTripZoneFlag(uint32_t base, uint16_t tzFlags)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT(tzFlags < 0x80U);

    //
    // Clear Trip zone event flag
    //
    EALLOW;
    HWREGH(base + EPWM_O_TZCLR) |= tzFlags;
    EDIS;

}

/**
 * @brief Clear the Trip Zone Cycle by Cycle flag.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  tzCBCFlags is the CBC flag to be cleared.
 * @remarks
 *  This function clears the specific Cycle by Cycle Trip Zone flag.
 *  The following are valid values for tzCBCFlags.
 *    - EPWM_TZ_CBC_FLAG_1     - CBC 1 flag
 *    - EPWM_TZ_CBC_FLAG_2     - CBC 2 flag
 *    - EPWM_TZ_CBC_FLAG_3     - CBC 3 flag
 *    - EPWM_TZ_CBC_FLAG_4     - CBC 4 flag
 *    - EPWM_TZ_CBC_FLAG_5     - CBC 5 flag
 *    - EPWM_TZ_CBC_FLAG_6     - CBC 6 flag
 *    - EPWM_TZ_CBC_FLAG_DCAEVT2  - CBC flag Digital compare
 *                                          event A2
 *    - EPWM_TZ_CBC_FLAG_DCBEVT2  - CBC flag Digital compare
 *                                          event B2
 *    - EPWM_TZ_CBC_FLAG_CAPEVT   - CBC flag capture event
 */
__STATIC_INLINE void
EPWM_clearCycleByCycleTripZoneFlag(uint32_t base, uint16_t tzCBCFlags)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT(tzCBCFlags < 0x800U);

    //
    // Clear the Cycle By Cycle Trip zone flag
    //
    EALLOW;
    HWREGH(base + EPWM_O_TZCBCCLR) |= tzCBCFlags;
    EDIS;
}

/**
 * @brief Clear the Trip Zone One Shot flag.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  tzOSTFlags is the OST flags to be cleared.
 * @remarks
 *  This function clears the specific One Shot (OST) Trip Zone flag.
 *  The following are valid values for tzOSTFlags.
 *   - EPWM_TZ_OST_FLAG_OST1      - OST flag for OST1
 *   - EPWM_TZ_OST_FLAG_OST2      - OST flag for OST2
 *   - EPWM_TZ_OST_FLAG_OST3      - OST flag for OST3
 *   - EPWM_TZ_OST_FLAG_OST4      - OST flag for OST4
 *   - EPWM_TZ_OST_FLAG_OST5      - OST flag for OST5
 *   - EPWM_TZ_OST_FLAG_OST6      - OST flag for OST6
 *   - EPWM_TZ_OST_FLAG_DCAEVT1   - OST flag for Digital compare event A1
 *   - EPWM_TZ_OST_FLAG_DCBEVT1   - OST flag for Digital compare event B1
 *   - EPWM_TZ_OST_FLAG_CAPEVT    - OST flag capture event
 */
__STATIC_INLINE void
EPWM_clearOneShotTripZoneFlag(uint32_t base, uint16_t tzOSTFlags)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT(tzOSTFlags < 0x800U);

    //
    // Clear the Cycle By Cycle Trip zone flag
    //
    EALLOW;
    HWREGH(base + EPWM_O_TZOSTCLR) |= tzOSTFlags;
    EDIS;
}

/**
 * @brief Force Trip Zone events.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  tzForceEvent is the forced Trip Zone event.
 * @remarks
 *  This function forces a Trip Zone event.
 *  Valid values for tzForceEvent are:
 *    - EPWM_TZ_FORCE_EVENT_CBC     - Force Trip Zones Cycle By Cycle event
 *    - EPWM_TZ_FORCE_EVENT_OST     - Force Trip Zones One Shot Event
 *    - EPWM_TZ_FORCE_EVENT_DCAEVT1 - Force Digital Compare A Event 1
 *    - EPWM_TZ_FORCE_EVENT_DCAEVT2 - Force Digital Compare A Event 2
 *    - EPWM_TZ_FORCE_EVENT_DCBEVT1 - Force Digital Compare B Event 1
 *    - EPWM_TZ_FORCE_EVENT_DCBEVT2 - Force Digital Compare B Event 2
 */
__STATIC_INLINE void
EPWM_forceTripZoneEvent(uint32_t base, uint16_t tzForceEvent)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT((tzForceEvent & 0xFF81U)== 0U);

    //
    // Force a Trip Zone event
    //
    EALLOW;
    HWREGH(base + EPWM_O_TZFRC) |= tzForceEvent;
    EDIS;
}

#if (EPWM_VERSION == 30)
/**
 * @brief Select TRIPOUT signals for the ePWM module.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  tzTripOut is the trip zone event to be selected.
 * @remarks
 *  Valid values for \e tzTripOut are:
 *    - EPWM_TZ_TRIPOUT_CBC     - CBC as TRIPOUT source
 *    - EPWM_TZ_TRIPOUT_OST     - OST as TRIPOUT source
 *    - EPWM_TZ_TRIPOUT_TZ[1-6] - TZx as TRIPOUT source
 *    - EPWM_TZ_TRIPOUT_DCAEVT1 - DCAEVT1 as TRIPOUT source
 *    - EPWM_TZ_TRIPOUT_DCAEVT2 - DCAEVT2 as TRIPOUT source
 *    - EPWM_TZ_TRIPOUT_DCBEVT1 - DCBEVT1 as TRIPOUT source
 *    - EPWM_TZ_TRIPOUT_DCBEVT2 - DCBEVT2 as TRIPOUT source
 *    - EPWM_TZ_TRIPOUT_CAPEVT  - Capture Event as TRIPOUT source
 */
static inline void
EPWM_enableTripOutSource(uint32_t base, EPWM_selectTripOutSource tzTripOut)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Configures the TRIPOUT source
    //
    EALLOW;
    HWREGH(base + EPWM_O_TZTRIPOUTSEL) |= (uint32_t)tzTripOut;
    EDIS;
}

/**
 * @brief Disables TRIPOUT signals for the ePWM module.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  tzTripOut is the trip zone event to be disabled.
 * @remarks
 *  Valid values for \e tzTripOut are:
 *    - EPWM_TZ_TRIPOUT_CBC     - CBC as TRIPOUT source
 *    - EPWM_TZ_TRIPOUT_OST     - OST as TRIPOUT source
 *    - EPWM_TZ_TRIPOUT_TZ[1-6] - TZx as TRIPOUT source
 *    - EPWM_TZ_TRIPOUT_DCAEVT1 - DCAEVT1 as TRIPOUT source
 *    - EPWM_TZ_TRIPOUT_DCAEVT2 - DCAEVT2 as TRIPOUT source
 *    - EPWM_TZ_TRIPOUT_DCBEVT1 - DCBEVT1 as TRIPOUT source
 *    - EPWM_TZ_TRIPOUT_DCBEVT2 - DCBEVT2 as TRIPOUT source
 *    - EPWM_TZ_TRIPOUT_CAPEVT  - Capture Event as TRIPOUT source
 */
static inline void
EPWM_disableTripOutSource(uint32_t base, EPWM_selectTripOutSource tzTripOut)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Disables the TRIPOUT source
    //
    EALLOW;
    HWREGH(base + EPWM_O_TZTRIPOUTSEL) &= (~(uint32_t)tzTripOut);
    EDIS;
}
#endif


//
// Digital Compare module related APIs
//

/**
 * @brief Set the DC trip input.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  tripSource is the tripSource.
 * @param [in]  dcType is the Digital Compare type.
 * @remarks
 *  This function sets the trip input to the Digital Compare (DC). For a given
 *  dcType the function sets the tripSource to be the input to the DC.
 *  Valid values for the parameter are:
 *   - tripSource
 *      - EPWM_DC_TRIP_TRIPINx - Trip x,where x ranges from 1 to 15 excluding 13
 *      - EPWM_DC_TRIP_COMBINATION - selects all the Trip signals whose input
 *                                   is enabled by the following function
 *                               EPWM_enableDigitalCompareTripCombinationInput()
 *   - dcType
 *      - EPWM_DC_TYPE_DCAH  - Digital Compare A High
 *      - EPWM_DC_TYPE_DCAL  - Digital Compare A Low
 *      - EPWM_DC_TYPE_DCBH  - Digital Compare B High
 *      - EPWM_DC_TYPE_DCBL  - Digital Compare B Low
 */
__STATIC_INLINE void
EPWM_selectDigitalCompareTripInput(uint32_t base,
                                   EPWM_DigitalCompareTripInput tripSource,
                                   EPWM_DigitalCompareType dcType)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set the DC trip input
    //
    EALLOW;
    HWREGH(base + EPWM_O_DCTRIPSEL) =
     ((HWREGH(base + EPWM_O_DCTRIPSEL) & ~(0xFU << ((uint16_t)(dcType/2) << 2U))) |
      ((uint16_t)tripSource << ((uint16_t)(dcType/2) << 2U)));
    EDIS;
}

//
// DCFILT
//

/**
 * @brief Enable DC filter blanking window.
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 * This function enables the DC filter blanking window.
 */
__STATIC_INLINE void
EPWM_enableDigitalCompareBlankingWindow(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Enable DC filter blanking window
    //
    EALLOW;
    HWREGH(base + EPWM_O_DCFCTL) |= EPWM_DCFCTL_BLANKE;
    EDIS;
}

/**
 * @brief Disable DC filter blanking window.
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 * This function disables the DC filter blanking window.
 */
__STATIC_INLINE void
EPWM_disableDigitalCompareBlankingWindow(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Disable DC filter blanking window
    //
    EALLOW;
    HWREGH(base + EPWM_O_DCFCTL) &= ~EPWM_DCFCTL_BLANKE;
    EDIS;
}

/**
 * @brief Enable Digital Compare Window inverse mode.
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 *  This function enables the Digital Compare Window inverse mode. This will
 *  invert the blanking window.
 */
__STATIC_INLINE void
EPWM_enableDigitalCompareWindowInverseMode(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Enable DC window inverse mode.
    //
    EALLOW;
    HWREGH(base + EPWM_O_DCFCTL) |= EPWM_DCFCTL_BLANKINV;
    EDIS;
}

/**
 * @brief Disable Digital Compare Window inverse mode.
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 *  This function disables the Digital Compare Window inverse mode.
 */
__STATIC_INLINE void
EPWM_disableDigitalCompareWindowInverseMode(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Disable DC window inverse mode.
    //
    EALLOW;
    HWREGH(base + EPWM_O_DCFCTL) &= ~EPWM_DCFCTL_BLANKINV;
    EDIS;
}

/**
 * @brief Set the Digital Compare filter blanking pulse.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  blankingPulse is Pulse that starts blanking window.
 * @remarks
 *  This function sets the input pulse that starts the Digital Compare blanking
 *  window.
 *  Valid values for blankingPulse are:
 *    - EPWM_DC_WINDOW_START_TBCTR_PERIOD - Time base counter equals period
 *    - EPWM_DC_WINDOW_START_TBCTR_ZERO   - Time base counter equals zero
 *    - EPWM_DC_WINDOW_START_TBCTR_ZERO_PERIOD - Time base counter equals zero
 *                                               or period.
 */
__STATIC_INLINE void
EPWM_setDigitalCompareBlankingEvent(uint32_t base,
                                EPWM_DigitalCompareBlankingPulse blankingPulse)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set DC blanking event
    //
    EALLOW;
    HWREGH(base + EPWM_O_DCFCTL) =
            ((HWREGH(base + EPWM_O_DCFCTL) & ~EPWM_DCFCTL_PULSESEL_M) |
             ((uint16_t)((uint32_t)blankingPulse << EPWM_DCFCTL_PULSESEL_S)));
    EDIS;
}

/**
 * @brief Set up the Digital Compare filter input.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  filterInput is Digital Compare signal source.
 * @remarks
 *  This function sets the signal input source that will be filtered by the
 *  Digital Compare module.
 *  Valid values for filterInput are:
 *    - EPWM_DC_WINDOW_SOURCE_DCAEVT1  - DC filter signal source is DCAEVT1
 *    - EPWM_DC_WINDOW_SOURCE_DCAEVT2  - DC filter signal source is DCAEVT2
 *    - EPWM_DC_WINDOW_SOURCE_DCBEVT1  - DC filter signal source is DCBEVT1
 *    - EPWM_DC_WINDOW_SOURCE_DCBEVT2  - DC filter signal source is DCBEVT2
 */
__STATIC_INLINE void
EPWM_setDigitalCompareFilterInput(uint32_t base,
                                  EPWM_DigitalCompareFilterInput filterInput)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set the signal source that will be filtered
    //
    EALLOW;
    HWREGH(base + EPWM_O_DCFCTL) =
                  ((HWREGH(base + EPWM_O_DCFCTL) & ~EPWM_DCFCTL_SRCSEL_M) |
                   ((uint16_t)filterInput));
    EDIS;
}

//
// DC Edge Filter
//

/**
 * @brief Enable Digital Compare Edge Filter.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  filterInput is Digital Compare signal source.
 * @remarks
 *  This function enables the Digital Compare Edge filter to generate event
 *  after configured number of edges.
 */
__STATIC_INLINE void
EPWM_enableDigitalCompareEdgeFilter(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Enable DC Edge Filter
    //
    EALLOW;
    HWREGH(base + EPWM_O_DCFCTL) |= EPWM_DCFCTL_EDGEFILTSEL;
    EDIS;
}

/**
 * @brief Disable Digital Compare Edge Filter.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  filterInput is Digital Compare signal source.
 * @remarks
 * This function disables the Digital Compare Edge filter.
 */
__STATIC_INLINE void
EPWM_disableDigitalCompareEdgeFilter(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Disable DC Edge Filter
    //
    EALLOW;
    HWREGH(base + EPWM_O_DCFCTL) &= ~EPWM_DCFCTL_EDGEFILTSEL;
    EDIS;
}

/**
 * @brief Set the Digital Compare Edge Filter Mode
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  edgeMode is Digital Compare Edge filter mode.
 * @remarks
 *  This function sets the Digital Compare Event filter mode. Valid values
 *  for edgeMode are:
 *    - EPWM_DC_EDGEFILT_MODE_RISING  - DC edge filter mode is rising edge
 *    - EPWM_DC_EDGEFILT_MODE_FALLING - DC edge filter mode is falling edge
 *    - EPWM_DC_EDGEFILT_MODE_BOTH    - DC edge filter mode is both edges
 */
__STATIC_INLINE void
EPWM_setDigitalCompareEdgeFilterMode(uint32_t base,
                                     EPWM_DigitalCompareEdgeFilterMode edgeMode)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set DC Edge filter mode
    //
    EALLOW;
    HWREGH(base + EPWM_O_DCFCTL) =
        (HWREGH(base + EPWM_O_DCFCTL) & ~EPWM_DCFCTL_EDGEMODE_M) |
         ((uint16_t)edgeMode << EPWM_DCFCTL_EDGEMODE_S);
    EDIS;
}

/**
 * @brief Set the Digital Compare Edge Filter Edge Count.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  edgeCount is Digital Compare event filter count
 * @remarks
 *  This function sets the Digital Compare Event filter Edge Count to generate
 *  events. Valid values for edgeCount can be:
 *   - EPWM_DC_EDGEFILT_EDGECNT_0 - No edge is required to generate event
 *   - EPWM_DC_EDGEFILT_EDGECNT_1 - 1 edge is required for event generation
 *   - EPWM_DC_EDGEFILT_EDGECNT_2 - 2 edges are required for event generation
 *   - EPWM_DC_EDGEFILT_EDGECNT_3 - 3 edges are required for event generation
 *   - EPWM_DC_EDGEFILT_EDGECNT_4 - 4 edges are required for event generation
 *   - EPWM_DC_EDGEFILT_EDGECNT_5 - 5 edges are required for event generation
 *   - EPWM_DC_EDGEFILT_EDGECNT_6 - 6 edges are required for event generation
 *   - EPWM_DC_EDGEFILT_EDGECNT_7 - 7 edges are required for event generation
 */
__STATIC_INLINE void
EPWM_setDigitalCompareEdgeFilterEdgeCount(uint32_t base, uint16_t edgeCount)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set DC Edge filter edge count
    //
    EALLOW;
    HWREGH(base + EPWM_O_DCFCTL) = (HWREGH(base + EPWM_O_DCFCTL) &
                                    ~EPWM_DCFCTL_EDGECOUNT_M) |
                                   (edgeCount << EPWM_DCFCTL_EDGECOUNT_S);
    EDIS;
}

/**
 * @brief Returns the Digital Compare Edge Filter Edge Count.
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 *  This function returns the configured Digital Compare Edge filter edge
 *  count required to generate events. It can return values from 0-7.
 *
 *  @return Returns the configured DigitalCompare Edge filter edge count.
 */
__STATIC_INLINE uint16_t
EPWM_getDigitalCompareEdgeFilterEdgeCount(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Return configured DC edge filter edge count
    //
    return((HWREGH(base + EPWM_O_DCFCTL) & EPWM_DCFCTL_EDGECOUNT_M) >>
           EPWM_DCFCTL_EDGECOUNT_S);
}

/**
 * @brief Returns the Digital Compare Edge filter captured edge count status.
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 *  This function returns the count of edges captured by Digital Compare Edge
 *  filter. It can return values from 0-7.
 *
 *  @return Returns the count of captured edges
 */
__STATIC_INLINE uint16_t
EPWM_getDigitalCompareEdgeFilterEdgeStatus(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Return captured edge count by DC Edge filter
    //
    return((HWREGH(base + EPWM_O_DCFCTL) & EPWM_DCFCTL_EDGESTATUS_M) >>
           EPWM_DCFCTL_EDGESTATUS_S);
}

/**
 * @brief Set up the Digital Compare filter window offset
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  windowOffsetCount is blanking window offset length.
 * @remarks
 *  This function sets the offset between window start pulse and blanking
 *  window in TBCLK count.
 *  The function take a 16bit count value for the offset value.
 */
__STATIC_INLINE void
EPWM_setDigitalCompareWindowOffset(uint32_t base, uint16_t windowOffsetCount)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set the blanking window offset in TBCLK counts
    //
    HWREGH(base + EPWM_O_DCFOFFSET) = windowOffsetCount;
}

/**
 * @brief Set up the Digital Compare filter window length
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  windowLengthCount is blanking window length.
 * @remarks
 *  This function sets up the Digital Compare filter blanking window length in
 *  TBCLK count.The function takes a 16bit count value for the window length.
 */
__STATIC_INLINE void
EPWM_setDigitalCompareWindowLength(uint32_t base, uint16_t windowLengthCount)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set the blanking window length in TBCLK counts
    //
    HWREGH(base + EPWM_O_DCFWINDOW) = windowLengthCount;
}

/**
 * @brief Return DC filter blanking window offset count.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  windowLengthCount is blanking window length.
 * @remarks
 *  This function returns DC filter blanking window offset count.
 */
__STATIC_INLINE uint16_t
EPWM_getDigitalCompareBlankingWindowOffsetCount(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Return the Blanking Window Offset count
    //
    return(HWREGH(base + EPWM_O_DCFOFFSETCNT));
}

/**
 * @brief Return DC filter blanking window length count.
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 *  This function returns DC filter blanking window length count.
 */
__STATIC_INLINE uint16_t
EPWM_getDigitalCompareBlankingWindowLengthCount(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Return the Blanking Window Length count
    //
    return(HWREGH(base + EPWM_O_DCFWINDOWCNT));
}

/**
 * @brief Set up the Digital Compare Event source.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  dcModule is the Digital Compare module.
 * @param [in]  dcEvent is the Digital Compare Event number.
 * @param [in]  dcEventSource is the - Digital Compare Event source.
 * @remarks
 *  This function sets up the Digital Compare module Event sources.
 *  The following are valid values for the parameters.
 *   - dcModule
 *       - EPWM_DC_MODULE_A  - Digital Compare Module A
 *       - EPWM_DC_MODULE_B  - Digital Compare Module B
 *   - dcEvent
 *       - EPWM_DC_EVENT_1   - Digital Compare Event number 1
 *       - EPWM_DC_EVENT_2   - Digital Compare Event number 2
 *   - dcEventSource
 *       - EPWM_DC_EVENT_SOURCE_FILT_SIGNAL  - signal source is filtered
 *             \note The signal source for this option is DCxEVTy, where the
 *                   value of x is dependent on dcModule and the value of y is
 *                   dependent on dcEvent. Possible signal sources are DCAEVT1,
 *                   DCBEVT1, DCAEVT2 or DCBEVT2 depending on the value of both
 *                   dcModule and dcEvent.
 *       - EPWM_DC_EVENT_SOURCE_ORIG_SIGNAL  - signal source is unfiltered
 *                    The signal source for this option is DCxEVTy.
 */
__STATIC_INLINE void
EPWM_setDigitalCompareEventSource(uint32_t base,
                                  EPWM_DigitalCompareModule dcModule,
                                  EPWM_DigitalCompareEvent dcEvent,
                                  EPWM_DigitalCompareEventSource dcEventSource)
{
    uint32_t registerOffset;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    registerOffset = EPWM_O_DCACTL + (uint32_t)dcModule;

    //
    // Set the DC event 1 source source
    //
    EALLOW;
    if(dcEvent == EPWM_DC_EVENT_1)
    {
        HWREGH(base + registerOffset) =
                ((HWREGH(base + registerOffset) & ~EPWM_DCACTL_EVT1SRCSEL) |
                 (uint16_t)dcEventSource);
    }
    else
    {
        HWREGH(base + registerOffset) =
                 ((HWREGH(base + registerOffset) & ~EPWM_DCACTL_EVT2SRCSEL) |
                  ((uint16_t)dcEventSource << 8U));
    }
    EDIS;
}

/**
 * @brief Set up the Digital Compare input sync mode.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  dcModule is the Digital Compare module.
 * @param [in]  dcEvent is the Digital Compare Event number.
 * @param [in]  syncMode is the Digital Compare Event sync mode.
 * @remarks
 *  This function sets up the Digital Compare module Event sources.
 *  The following are valid values for the parameters.
 *   - dcModule
 *       - EPWM_DC_MODULE_A  - Digital Compare Module A
 *       - EPWM_DC_MODULE_B  - Digital Compare Module B
 *   - dcEvent
 *       - EPWM_DC_EVENT_1   - Digital Compare Event number 1
 *       - EPWM_DC_EVENT_2   - Digital Compare Event number 2
 *   - syncMode
 *       - EPWM_DC_EVENT_INPUT_SYNCED      - DC input signal is synced with
 *                                           TBCLK
 *       - EPWM_DC_EVENT_INPUT_NOT SYNCED  - DC input signal is not synced with
 *                                           TBCLK
 */
__STATIC_INLINE void
EPWM_setDigitalCompareEventSyncMode(uint32_t base,
                                    EPWM_DigitalCompareModule dcModule,
                                    EPWM_DigitalCompareEvent dcEvent,
                                    EPWM_DigitalCompareSyncMode syncMode)
{
    uint32_t registerOffset;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    registerOffset = EPWM_O_DCACTL + (uint32_t)dcModule;

    //
    // Set the DC event sync mode
    //
    EALLOW;
    if(dcEvent == EPWM_DC_EVENT_1)
    {
        HWREGH(base + registerOffset) =
           ((HWREGH(base + registerOffset) & ~EPWM_DCACTL_EVT1FRCSYNCSEL) |
            ((uint16_t)syncMode << 1U));
    }
    else
    {
        HWREGH(base + registerOffset) =
            ((HWREGH(base + registerOffset) & ~EPWM_DCACTL_EVT2FRCSYNCSEL) |
             ((uint16_t)syncMode << 9U));
    }
    EDIS;
}

/**
 * @brief Enable Digital Compare to generate Start of Conversion.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  dcModule is the Digital Compare module.
 * @remarks
 *  This function enables the Digital Compare Event 1 to generate Start of
 *  Conversion.
 *  The following are valid values for the \e dcModule parameter.
 *      - EPWM_DC_MODULE_A  - Digital Compare Module A
 *      - EPWM_DC_MODULE_B  - Digital Compare Module B
 */
__STATIC_INLINE void
EPWM_enableDigitalCompareADCTrigger(uint32_t base,
                                    EPWM_DigitalCompareModule dcModule)
{
    uint32_t registerOffset;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    registerOffset = EPWM_O_DCACTL + (uint32_t)dcModule;

    //
    // Enable Digital Compare start of conversion generation
    //
    EALLOW;
    HWREGH(base + registerOffset) =
                       (HWREGH(base + registerOffset) | EPWM_DCACTL_EVT1SOCE);
    EDIS;
}

/**
 * @brief Disable Digital Compare from generating Start of Conversion.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  dcModule is the Digital Compare module.
 * @remarks
 *  This function disables the Digital Compare Event 1 from generating Start of
 *  Conversion.
 *  The following are valid values for the \e dcModule parameter.
 *      - EPWM_DC_MODULE_A  - Digital Compare Module A
 *      - EPWM_DC_MODULE_B  - Digital Compare Module B
 */
__STATIC_INLINE void
EPWM_disableDigitalCompareADCTrigger(uint32_t base,
                                     EPWM_DigitalCompareModule dcModule)
{
    uint32_t registerOffset;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    registerOffset = EPWM_O_DCACTL + (uint32_t)dcModule;

    //
    // Disable Digital Compare start of conversion generation
    //
    EALLOW;
    HWREGH(base + registerOffset) =
                      (HWREGH(base + registerOffset) & ~EPWM_DCACTL_EVT1SOCE);
    EDIS;
}

/**
 * @brief Enable Digital Compare to generate sync out pulse.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  dcModule is the Digital Compare module.
 * @remarks
 *  This function enables the Digital Compare Event 1 to generate sync out
 *  pulse
 *  The following are valid values for the \e dcModule parameter.
 *      - EPWM_DC_MODULE_A  - Digital Compare Module A
 *      - EPWM_DC_MODULE_B  - Digital Compare Module B
 */
__STATIC_INLINE void
EPWM_enableDigitalCompareSyncEvent(uint32_t base,
                                   EPWM_DigitalCompareModule dcModule)
{
    uint32_t registerOffset;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    registerOffset = EPWM_O_DCACTL + (uint32_t)dcModule;

    //
    // Enable Digital Compare sync out pulse generation
    //
    EALLOW;
    HWREGH(base + registerOffset) =
                     (HWREGH(base + registerOffset) | EPWM_DCACTL_EVT1SYNCE);
    EDIS;
}

/**
 * @brief Disable Digital Compare from generating Start of Conversion.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  dcModule is the Digital Compare module.
 * @remarks
 *  This function disables the Digital Compare Event 1 from generating synch
 *  out pulse.
 *  The following are valid values for the \e dcModule parameters.
 *      - EPWM_DC_MODULE_A  - Digital Compare Module A
 *      - EPWM_DC_MODULE_B  - Digital Compare Module B
 */
__STATIC_INLINE void
EPWM_disableDigitalCompareSyncEvent(uint32_t base,
                                    EPWM_DigitalCompareModule dcModule)
{
    uint32_t registerOffset;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    registerOffset = EPWM_O_DCACTL + (uint32_t)dcModule;

    //
    // Disable Digital Compare sync out pulse generation
    //
    EALLOW;
    HWREGH(base + registerOffset) =
                      (HWREGH(base + registerOffset) & ~EPWM_DCACTL_EVT1SYNCE);
    EDIS;
}

/**
 * @brief Set up the Digital Compare CBC latch mode.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  dcModule is the Digital Compare module.
 * @param [in]  dcEvent is the Digital Compare Event number.
 * @param [in]  latchMode is the Digital Compare CBC latch mode.
 * @remarks
 *  This function sets up the Digital Compare CBC latch mode.
 *  The following are valid values for the parameters.
 *   - dcModule
 *       - EPWM_DC_MODULE_A  - Digital Compare Module A
 *       - EPWM_DC_MODULE_B  - Digital Compare Module B
 *   - dcEvent
 *       - EPWM_DC_EVENT_1   - Digital Compare Event number 1
 *       - EPWM_DC_EVENT_2   - Digital Compare Event number 2
 *   - latchMode
 *       - EPWM_DC_CBC_LATCH_DISABLED - DC cycle-by-cycle(CBC) latch is disabled
 *       - EPWM_DC_CBC_LATCH_ENABLED  - DC cycle-by-cycle(CBC) latch is enabled
 */
__STATIC_INLINE void
EPWM_setDigitalCompareCBCLatchMode(uint32_t base,
                                   EPWM_DigitalCompareModule dcModule,
                                   EPWM_DigitalCompareEvent dcEvent,
                                   EPWM_DigitalCompareCBCLatchMode latchMode)
{
    uint32_t registerOffset;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    registerOffset = EPWM_O_DCACTL + (uint32_t)dcModule;

    //
    // Set the DC CBC Latch Mode
    //
    EALLOW;
    if(dcEvent == EPWM_DC_EVENT_1)
    {
        HWREGH(base + registerOffset) =
           ((HWREGH(base + registerOffset) & ~EPWM_DCACTL_EVT1LATSEL) |
            ((uint16_t)latchMode << 4U));
    }
    else
    {
        HWREGH(base + registerOffset) =
            ((HWREGH(base + registerOffset) & ~EPWM_DCACTL_EVT2LATSEL) |
             ((uint16_t)latchMode << 12U));
    }
    EDIS;
}

/**
 * @brief Sets the Digital Compare CBC latch clear event.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  dcModule is the Digital Compare module.
 * @param [in]  dcEvent is the Digital Compare Event number.
 * @param [in]  clearEvent is the Digital Compare CBC latch clear event.
 * @remarks
 *  This function sets the Digital Compare CBC latch clear event.
 *  The following are valid values for the parameters.
 *   - dcModule
 *       - EPWM_DC_MODULE_A  - Digital Compare Module A
 *       - EPWM_DC_MODULE_B  - Digital Compare Module B
 *   - dcEvent
 *       - EPWM_DC_EVENT_1   - Digital Compare Event number 1
 *       - EPWM_DC_EVENT_2   - Digital Compare Event number 2
 *   - clearEvent
 *       - EPWM_DC_CBC_LATCH_CLR_CNTR_ZERO - DC CBC latch is cleared when
 *                                           counter is zero
 *       - EPWM_DC_CBC_LATCH_CLR_ON_CNTR_PERIOD - DC CBC latch is cleared when
 *                                                counter is equal to period
 *       - EPWM_DC_CBC_LATCH_CLR_ON_CNTR_ZERO_PERIOD - DC CBC latch is cleared
 *                                                     when either counter is
 *                                                     zero or equal to period
 */
__STATIC_INLINE void
EPWM_selectDigitalCompareCBCLatchClearEvent(uint32_t base,
                              EPWM_DigitalCompareModule dcModule,
                              EPWM_DigitalCompareEvent dcEvent,
                              EPWM_DigitalCompareCBCLatchClearEvent clearEvent)
{
    uint32_t registerOffset;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    registerOffset = EPWM_O_DCACTL + (uint32_t)dcModule;

    //
    // Set the DC CBC Latch Clear Event
    //
    EALLOW;
    if(dcEvent == EPWM_DC_EVENT_1)
    {
        HWREGH(base + registerOffset) =
           ((HWREGH(base + registerOffset) & ~EPWM_DCACTL_EVT1LATCLRSEL_M) |
            ((uint16_t)clearEvent << EPWM_DCACTL_EVT1LATCLRSEL_S));
    }
    else
    {
        HWREGH(base + registerOffset) =
            ((HWREGH(base + registerOffset) & ~EPWM_DCACTL_EVT2LATCLRSEL_M) |
             ((uint16_t)clearEvent << EPWM_DCACTL_EVT2LATCLRSEL_S));
    }
    EDIS;
}

/**
 * @brief Gets the Digital Compare CBC latch status
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  dcModule is the Digital Compare module.
 * @param [in]  dcEvent is the Digital Compare Event number.
 * @remarks
 *  This function returns the Digital Compare module cycle-by-cycle(CBC) latch
 *  status.
 *  The following are valid values for the parameters.
 *   - dcModule
 *       - EPWM_DC_MODULE_A  - Digital Compare Module A
 *       - EPWM_DC_MODULE_B  - Digital Compare Module B
 *   - dcEvent
 *       - EPWM_DC_EVENT_1   - Digital Compare Event number 1
 *       - EPWM_DC_EVENT_2   - Digital Compare Event number 2
 *
 * @return Returns Digital Compare CBC latch status.
 *   true  - CBC latch is enabled
 *   false - CBC latch is disabled
 */
__STATIC_INLINE bool
EPWM_getDigitalCompareCBCLatchStatus(uint32_t base,
                                     EPWM_DigitalCompareModule dcModule,
                                     EPWM_DigitalCompareEvent dcEvent)
{
    uint32_t registerOffset;
    uint16_t status;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    registerOffset = EPWM_O_DCACTL + (uint32_t)dcModule;

    //
    // Get DC CBC Latch Clear Event
    //
    if(dcEvent == EPWM_DC_EVENT_1)
    {
        status = HWREGH(base + registerOffset) & EPWM_DCACTL_EVT1LAT;
    }
    else
    {
        status = HWREGH(base + registerOffset) & EPWM_DCACTL_EVT2LAT;
    }
    return(status != 0U);
}

//
// DC capture mode
//

/**
 * @brief Enables the Time Base Counter Capture controller.
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 *  This function enables the time Base Counter Capture.
 */
__STATIC_INLINE void
EPWM_enableDigitalCompareCounterCapture(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Enable Time base counter capture
    //
    EALLOW;
    HWREGH(base + EPWM_O_DCCAPCTL) |= EPWM_DCCAPCTL_CAPE;
    EDIS;
}

/**
 * @brief Disables the Time Base Counter Capture controller.
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 *  This function disable the time Base Counter Capture.
 */
__STATIC_INLINE void
EPWM_disableDigitalCompareCounterCapture(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Disable Time base counter capture
    //
    EALLOW;
    HWREGH(base + EPWM_O_DCCAPCTL) &= ~EPWM_DCCAPCTL_CAPE;
    EDIS;
}

/**
 * @brief Set the Time Base Counter Capture mode.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  enableShadowMode is the shadow read mode flag.
 * @remarks
 *  This function sets the mode the Time Base Counter value is read from. If
 *  enableShadowMode is true, CPU reads of the DCCAP register will return the
 *  shadow register contents.If enableShadowMode is false, CPU reads of the
 *  DCCAP register will return the active register contents.
 */
__STATIC_INLINE void
EPWM_setDigitalCompareCounterShadowMode(uint32_t base, bool enableShadowMode)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    EALLOW;
    if(enableShadowMode)
    {
        //
        // Enable DC counter shadow mode
        //
        HWREGH(base + EPWM_O_DCCAPCTL) &= ~EPWM_DCCAPCTL_SHDWMODE;
    }
    else
    {
       //
       // Disable DC counter shadow mode
       //
       HWREGH(base + EPWM_O_DCCAPCTL) |= EPWM_DCCAPCTL_SHDWMODE;
    }
    EDIS;
}

/**
 * @brief Return the DC Capture event status.
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 *  This function returns the DC capture event status.
 *
 * @return Returns true if a DC capture event has occurs.
 *          Returns false if no DC Capture event has occurred.
 */
__STATIC_INLINE bool
EPWM_getDigitalCompareCaptureStatus(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Return the DC compare status
    //
    return((HWREGH(base + EPWM_O_DCCAPCTL) & EPWM_DCCAPCTL_CAPSTS) ==
                                                        EPWM_DCCAPCTL_CAPSTS);
}

/**
 * @brief Clears DC capture latched status flag
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 * This function is used to clear the CAPSTS (set) condition.
 */
static inline void
EPWM_clearDigitalCompareCaptureStatusFlag(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Clear digital compare capture status flag
    //
    EALLOW;
    HWREGH(base + EPWM_O_DCCAPCTL) |= EPWM_DCCAPCTL_CAPCLR;
    EDIS;
}

/**
 * @brief Configures DC capture operating mode
 * @param [in]  base is the base address of the EPWM module.
 * @param [in] disableClearMode is the clear mode bit.
 * @remarks
 *  This function is used to configure the DC capture operating mode. If
 *  \e disableClearMode is false, the TBCNT value is captured in active register
 *  on occurance of DCEVTFILT event. The trip events are ignored until next
 *  PRD or ZRO event re-triggers the capture mechanism.
 *  If \e disableClearMode is true, the TBCNT value is captured, CAPSTS flag is
 *  set and further trips are ignored until CAPSTS bit is cleared.
 */
static inline void
EPWM_configureDigitalCompareCounterCaptureMode(uint32_t base,
                                               bool disableClearMode)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    EALLOW;
    if(disableClearMode)
    {
        //
        // Disable DC counter auto-clear on PULSESEL event
        //
        HWREGH(base + EPWM_O_DCCAPCTL) |= EPWM_DCCAPCTL_CAPMODE;
    }
    else
    {
       //
       // Enable DC counter clear on PULSESEL events
       //
       HWREGH(base + EPWM_O_DCCAPCTL) &= ~EPWM_DCCAPCTL_CAPMODE;
    }
    EDIS;
}

/**
 * @brief Return the DC Time Base Counter capture value.
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 *  This function returns the DC Time Base Counter capture value. The value
 *  read is determined by the mode as set in the
 *  EPWM_setTimeBaseCounterReadMode() function.
 *
 * @return Returns the DC Time Base Counter Capture count value.
 */
__STATIC_INLINE uint16_t
EPWM_getDigitalCompareCaptureCount(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Return the DC Time Base Counter Capture count value
    //
    return(HWREGH(base + EPWM_O_DCCAP));
}

/**
 * @brief Enable DC TRIP combinational input.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  tripInput is the Trip number.
 * @param [in]  dcType is the Digital Compare module.
 * @remarks
 *  This function enables the specified Trip input.
 *  Valid values for the parameters are:
 *   - tripInput
 *       - EPWM_DC_COMBINATIONAL_TRIPINx, where x is 1,2,...12,14,15
 *   - dcType
 *       - EPWM_DC_TYPE_DCAH  - Digital Compare A High
 *       - EPWM_DC_TYPE_DCAL  - Digital Compare A Low
 *       - EPWM_DC_TYPE_DCBH  - Digital Compare B High
 *       - EPWM_DC_TYPE_DCBL  - Digital Compare B Low
 */
__STATIC_INLINE void
EPWM_enableDigitalCompareTripCombinationInput(uint32_t base,
                                              uint16_t tripInput,
                                              EPWM_DigitalCompareType dcType)
{
    uint32_t registerOffset;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Get the DCAHTRIPSEL, DCALTRIPSEL, DCBHTRIPSEL, DCBLTRIPSEL register
    // offset with respect to DCAHTRIPSEL
    //
    registerOffset = EPWM_O_DCAHTRIPSEL + (uint32_t)dcType;

    //
    // Set the DC trip input
    //
    EALLOW;
    HWREGH(base + registerOffset) =
                           (HWREGH(base + registerOffset) | tripInput);

    //
    // Enable the combination input
    //
    HWREGH(base + EPWM_O_DCTRIPSEL) =
      (HWREGH(base + EPWM_O_DCTRIPSEL) | (0xFU << ((uint16_t)(dcType/2)<< 2U)));
    EDIS;
}

/**
 * @brief Disable DC TRIP combinational input.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  tripInput is the Trip number.
 * @param [in]  dcType is the Digital Compare module.
 * @remarks
 *  This function disables the specified Trip input.
 *  Valid values for the parameters are:
 *   - tripInput
 *       - EPWM_DC_COMBINATIONAL_TRIPINx, where x is 1,2,...12,14,15
 *   - dcType
 *       - EPWM_DC_TYPE_DCAH  - Digital Compare A High
 *       - EPWM_DC_TYPE_DCAL  - Digital Compare A Low
 *       - EPWM_DC_TYPE_DCBH  - Digital Compare B High
 *       - EPWM_DC_TYPE_DCBL  - Digital Compare B Low
 */
__STATIC_INLINE void
EPWM_disableDigitalCompareTripCombinationInput(uint32_t base,
                                               uint16_t tripInput,
                                               EPWM_DigitalCompareType dcType)
{
    uint32_t registerOffset;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Get the DCAHTRIPSEL, DCALTRIPSEL, DCBHTRIPSEL, DCBLTRIPSEL register
    // offset with respect to DCAHTRIPSEL
    //
    registerOffset = EPWM_O_DCAHTRIPSEL + (uint32_t)dcType;

    //
    // Set the DC trip input
    //
    EALLOW;
    HWREGH(base + registerOffset) =
                           (HWREGH(base + registerOffset) & ~tripInput);
    EDIS;
}

#if EPWM_VERSION == 30
//
// Event capture mode
//

/**
 * @brief Enables the Capture event .
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 *  This function enables the CAPIN.sync on which the edge detection logic
 *  is performed.
 */
static inline void
EPWM_enableCaptureInEvent(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Enables CAPIN.sync signal
    //
    EALLOW;
    HWREGH(base + EPWM_O_CAPCTL) |= EPWM_CAPCTL_SRCSEL;
    EDIS;
}

/**
 * @brief Disables the Capture event.
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 *  This function disables the CAPIN.sync on which the edge detection logic
 *  is performed.
 */
static inline void
EPWM_disableCaptureInEvent(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Disables CAPIN.sync signal
    //
    EALLOW;
    HWREGH(base + EPWM_O_CAPCTL) &= ~EPWM_CAPCTL_SRCSEL;
    EDIS;
}

/**
 * @brief Polarity selection for capture gate input.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  polSel is the polarity to be selected for CAPGATE.
 * @remarks
 *  This function selects the input polarity for capture gate.
 *  Valid values for the \e polSel are:
 *       - EPWM_CAPGATE_INPUT_ALWAYS_ON  - always on
 *       - EPWM_CAPGATE_INPUT_ALWAYS_OFF  - always off
 *       - EPWM_CAPGATE_INPUT_SYNC  - CAPGATE.sync
 *       - EPWM_CAPGATE_INPUT_SYNC_INVERT  - CAPGATE.sync inverted
 */
static inline void
EPWM_configCaptureGateInputPolarity(uint32_t base,
                                    EPWM_selectCaptureGateInputPolarity polSel)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Configures polarity for CAPGATE
    //
    EALLOW;
    HWREGH(base + EPWM_O_CAPCTL) =
             ((HWREGH(base + EPWM_O_CAPCTL) & ~EPWM_CAPCTL_CAPGATEPOL_M) |
              ((uint32_t)polSel << EPWM_CAPCTL_CAPGATEPOL_S));
    EDIS;
}

/**
 * @brief Polarity selection for capture input.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  polSel is the polarity to be selected for CAPIN.
 * @remarks
 *  This function selects the input polarity for capture.
 *  Valid values for the \e polSel are:
 *       - EPWM_CAPTURE_INPUT_CAPIN_SYNC         - not inverted
 *       - EPWM_CAPTURE_INPUT_CAPIN_SYNC_INVERT  - inverted
 */
static inline void
EPWM_invertCaptureInputPolarity(uint32_t base,
                                EPWM_selectCaptureInputPolarity polSel)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Configures polarity for Capture Input
    //
    EALLOW;
    HWREGH(base + EPWM_O_CAPCTL) =
             ((HWREGH(base + EPWM_O_CAPCTL) & ~EPWM_CAPCTL_CAPINPOL) |
              ((uint32_t)polSel << 3U));
    EDIS;
}

/**
 * @brief Enables independent pulse selection for Blanking and Capture Logic
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 * This function enables pulse selection determined by the CAPMIXSEL register.
 */
static inline void
EPWM_enableIndependentPulseLogic(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Configures polarity for Capture Input
    //
    EALLOW;
    HWREGH(base + EPWM_O_CAPCTL) |= EPWM_CAPCTL_PULSECTL;
    EDIS;
}

/**
 * @brief Disables independent pulse selection for Blanking and Capture Logic.
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 *  This function disables pulse selection determined by the CAPMIXSEL register.
 *  The pulse selection is determined by DCFCTL[PULSESEL] bits.
 */
static inline void
EPWM_disableIndependentPulseLogic(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Configures polarity for Capture Input
    //
    EALLOW;
    HWREGH(base + EPWM_O_CAPCTL) &= ~EPWM_CAPCTL_PULSECTL;
    EDIS;
}

/**
 * @brief Capture event force load.
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 *  This function forces a load to occur on DCCAP.
 */
static inline void
EPWM_forceCaptureEventLoad(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Force a Capture Event Load
    //
    EALLOW;
    HWREGH(base + EPWM_O_CAPCTL) |= EPWM_CAPCTL_FRCLOAD;
    EDIS;
}

/**
 * @brief Set the capture trip input.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  tripSource is the tripSource.
 * @param [in]  dcType is the Digital Compare type.
 * @remarks
 *  This function sets the trip input to the Digital Compare (DC). For a given
 *  dcType the function sets the tripSource to be the input to the DC.
 *  Valid values for the parameter are:
 *   - tripSource
 *     - EPWM_DC_TRIP_TRIPINx - Trip x,where x ranges from 1 to 15 excluding 13
 *     - EPWM_DC_TRIP_COMBINATION - selects all the Trip signals whose input
 *                                  is enabled by the following function
 *                                  EPWM_enableCaptureTripCombinationInput()
 *   - dcType
 *       - EPWM_CAPTURE_GATE
 *       - EPWM_CAPTURE_INPUT
 */
static inline void
EPWM_selectCaptureTripInput(uint32_t base,
                            EPWM_DigitalCompareTripInput tripSource,
                            EPWM_CaptureInputType dcType)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set the Capture trip input
    //
    EALLOW;
    HWREGH(base + EPWM_O_CAPTRIPSEL) =
     ((HWREGH(base + EPWM_O_CAPTRIPSEL) & ~(0xFU << ((uint16_t)dcType << 2U))) |
      ((uint16_t)tripSource << ((uint16_t)dcType << 2U)));
    EDIS;
}

/**
 * @brief Enable Capture TRIP combinational input.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  tripInput is the Trip number.
 * @param [in]  dcType is the Digital Compare module.
 * @remarks
 *  This function enables the specified Trip input.
 *  Valid values for the parameters are:
 *   - tripInput
 *       - EPWM_DC_COMBINATIONAL_TRIPINx, where x is 1,2,...12,14,15
 *   - dcType
 *       - EPWM_CAPTURE_GATE
 *       - EPWM_CAPTURE_INPUT
 */
static inline void
EPWM_enableCaptureTripCombinationInput(uint32_t base,
                                            uint16_t tripInput,
                                            EPWM_CaptureInputType dcType)
{
    uint32_t registerOffset;
    uint8_t  offset = dcType?1U:2U;
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    registerOffset = EPWM_O_CAPGATETRIPSEL + (offset - (uint32_t)dcType);

    //
    // Set the capture trip input
    //
    EALLOW;
    HWREGH(base + registerOffset) =
                           (HWREGH(base + registerOffset) | tripInput);

    //
    // Enable the combination input
    //
    HWREGH(base + EPWM_O_CAPTRIPSEL) =
     (HWREGH(base + EPWM_O_CAPTRIPSEL) | (0xFU << ((uint16_t)dcType << 2U)));
    EDIS;
}

/**
 * @brief Disable Capture TRIP combinational input.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  tripInput is the Trip number.
 * @param [in]  dcType is the Digital Compare module.
 * @remarks
 *  This function disables the specified Trip input.
 *  Valid values for the parameters are:
 *   - tripInput
 *       - EPWM_DC_COMBINATIONAL_TRIPINx, where x is 1,2,...12,14,15
 *   - dcType
 *       - EPWM_CAPTURE_GATE
 *       - EPWM_CAPTURE_INPUT
 */
static inline void
EPWM_disableCaptureTripCombinationInput(uint32_t base,
                                         uint16_t tripInput,
                                         EPWM_CaptureInputType dcType)
{
    uint32_t registerOffset;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    registerOffset = EPWM_O_CAPGATETRIPSEL + (1U - (uint32_t)dcType);

    //
    // Set the capture trip input
    //
    EALLOW;
    HWREGH(base + registerOffset) =
                            (HWREGH(base + registerOffset) & ~tripInput);
    EDIS;
}

#endif

//
// Valley switching
//

/**
 * @brief Enable valley capture mode.
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 *  This function enables Valley Capture mode.
 */
__STATIC_INLINE void
EPWM_enableValleyCapture(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set VCAPE bit
    //
    EALLOW;
    HWREGH(base + EPWM_O_VCAPCTL) |= EPWM_VCAPCTL_VCAPE;
    EDIS;
}

/**
 * @brief Disable valley capture mode
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 *  This function disables Valley Capture mode.
 */
__STATIC_INLINE void
EPWM_disableValleyCapture(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Clear VCAPE bit
    //
    EALLOW;
    HWREGH(base + EPWM_O_VCAPCTL) &= ~EPWM_VCAPCTL_VCAPE;
    EDIS;
}

/**
 * @brief Start valley capture mode.
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 * This function starts Valley Capture sequence.
 *
 * Make sure you invoke EPWM_setValleyTriggerSource with the trigger
 *     variable set to EPWM_VALLEY_TRIGGER_EVENT_SOFTWARE before calling this
 *     function.
 */
__STATIC_INLINE void
EPWM_startValleyCapture(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set VCAPSTART bit
    //
    EALLOW;
    HWREGH(base + EPWM_O_VCAPCTL) |= EPWM_VCAPCTL_VCAPSTART;
    EDIS;
}

/**
 * @brief Set valley capture trigger.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  trigger is the Valley counter trigger.
 * @remarks
 *  This function sets the trigger value that initiates Valley Capture sequence
 *
 *  Set the number of Trigger source events for starting and stopping the
 *     valley capture using EPWM_setValleyTriggerEdgeCounts().
 */
__STATIC_INLINE void
EPWM_setValleyTriggerSource(uint32_t base, EPWM_ValleyTriggerSource trigger)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Write to TRIGSEL bits
    //
    EALLOW;
    HWREGH(base + EPWM_O_VCAPCTL) =
             ((HWREGH(base + EPWM_O_VCAPCTL) & ~EPWM_VCAPCTL_TRIGSEL_M) |
              ((uint16_t)trigger << 2U));
    EDIS;
}

/**
 * @brief Set valley capture trigger source count.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  startCount
 * @param [in]  stopCount
 * @remarks
 *  This function sets the number of trigger events required to start and stop
 *  the valley capture count.
 *  Maximum values for both startCount and stopCount is 15 corresponding to the
 *  15th edge of the trigger event.
 *
 *  Note:
 *     A startCount value of 0 prevents starting the valley counter.
 *     A stopCount value of 0 prevents the valley counter from stopping.
 */
__STATIC_INLINE void
EPWM_setValleyTriggerEdgeCounts(uint32_t base, uint16_t startCount,
                                uint16_t stopCount)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT((startCount < 16U) && (stopCount < 16U));

    //
    // Write to STARTEDGE and STOPEDGE bits
    //
    EALLOW;
    HWREGH(base + EPWM_O_VCNTCFG) =
                    ((HWREGH(base + EPWM_O_VCNTCFG) &
                      ~(EPWM_VCNTCFG_STARTEDGE_M | EPWM_VCNTCFG_STOPEDGE_M)) |
                      (startCount | (stopCount << 8U)));
    EDIS;
}

/**
 * @brief Enable valley switching delay.
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 * This function enables Valley switching delay.
 */
__STATIC_INLINE void
EPWM_enableValleyHWDelay(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set EDGEFILTDLYSEL bit
    //
    EALLOW;
    HWREGH(base + EPWM_O_VCAPCTL) |= EPWM_VCAPCTL_EDGEFILTDLYSEL;
    EDIS;
}

/**
 * @brief Disable valley switching delay.
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 * This function disables Valley switching delay.
 */
__STATIC_INLINE void
EPWM_disableValleyHWDelay(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Clear EDGEFILTDLYSEL bit
    //
    EALLOW;
    HWREGH(base + EPWM_O_VCAPCTL) &= ~EPWM_VCAPCTL_EDGEFILTDLYSEL;
    EDIS;
}

/**
 * @brief Set Valley delay values.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  delayOffsetValue is the software defined delay offset value.
 * @remarks
 * This function sets the Valley delay value.
 */
__STATIC_INLINE void
EPWM_setValleySWDelayValue(uint32_t base, uint16_t delayOffsetValue)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Write to SWVDELVAL bits
    //
    HWREGH(base + EPWM_O_SWVDELVAL) = delayOffsetValue;
}

/**
 * @brief Set Valley delay mode.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  delayMode is the Valley delay mode.
 * @remarks
 * This function sets the Valley delay mode values.
 */
__STATIC_INLINE void
EPWM_setValleyDelayDivider(uint32_t base, EPWM_ValleyDelayMode delayMode)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Write to VDELAYDIV bits
    //
    EALLOW;
    HWREGH(base + EPWM_O_VCAPCTL) =
         ((HWREGH(base + EPWM_O_VCAPCTL) & ~EPWM_VCAPCTL_VDELAYDIV_M) |
          ((uint16_t)delayMode << 7U));
    EDIS;
}

/**
 * @brief Get the valley edge status bit.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  edge is the start or stop edge.
 * @remarks
 *  This function returns the status of the start or stop valley status
 *  depending on the value of edge.
 *  If a start or stop edge has occurred, the function returns true, if not it
 *  returns false.
 *
 * @return Returns true if the specified edge has occurred,
 *          Returns false if the specified edge has not occurred.
 */
__STATIC_INLINE bool
EPWM_getValleyEdgeStatus(uint32_t base, EPWM_ValleyCounterEdge edge)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    if(edge == EPWM_VALLEY_COUNT_START_EDGE)
    {
        //
        // Returns STARTEDGESTS status
        //
        return(((HWREGH(base + EPWM_O_VCNTCFG) & EPWM_VCNTCFG_STARTEDGESTS) ==
                   EPWM_VCNTCFG_STARTEDGESTS ) ? true : false);
    }
    else
    {
        //
        // Returns STOPEDGESTS status
        //
        return(((HWREGH(base + EPWM_O_VCNTCFG) & EPWM_VCNTCFG_STOPEDGESTS) ==
                EPWM_VCNTCFG_STOPEDGESTS) ? true : false);
    }
}

/**
 * @brief Get the Valley Counter value.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  edge is the start or stop edge.
 * @remarks
 *  This function returns the valley time base count value which is captured
 *  upon occurrence of the stop edge condition selected by
 *  EPWM_setValleyTriggerSource() and by the stopCount variable of the
 *  EPWM_setValleyTriggerEdgeCounts() function.
 *
 * @return Returns the valley base time count.
 */
__STATIC_INLINE uint16_t
EPWM_getValleyCount(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Read VCNTVAL register
    //
    return(HWREGH(base + EPWM_O_VCNTVAL));
}

/**
 * @brief Get the Valley delay value.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  edge is the start or stop edge.
 * @remarks
 *  This function returns the hardware valley delay count.
 *
 * @return Returns the valley delay count.
 */
__STATIC_INLINE uint16_t
EPWM_getValleyHWDelay(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Read HWVDELVAL  register
    //
    return(HWREGH(base + EPWM_O_HWVDELVAL));
}

/**
 * @brief enable PllUnlock Signals to DC trip15
 * @remarks
 * This function enable PllUnlock Signals to DC trip15.
 */
__STATIC_INLINE void
EPWM_enablePllUnlock_Signals(void)
{
	uint16_t offset = 0;
#if(EPWM_VERSION == 30)
	offset = 3;
#elif(EPWM_VERSION == 22)
	offset = 7;
#elif(EPWM_VERSION == 12)
	offset = 5;
#endif
    //
    // enable PllUnlock Signals
    //
    HWREGH(SYSCTL_BASE + SYSCTL_O_PWM_SYSFAULT_MASK) = HWREGH(SYSCTL_BASE + SYSCTL_O_PWM_SYSFAULT_MASK)& (~(1<<offset));
}

/**
 * @brief disable PllUnlock Signals to DC trip15
 * @remarks
 * This function enable PllUnlock Signals to DC trip15.
 */
__STATIC_INLINE void
EPWM_disablePllUnlock_Signals(void)
{
	uint16_t offset = 0;
#if(EPWM_VERSION == 30)
	offset = 3;
#elif(EPWM_VERSION == 22)
	offset = 7;
#elif(EPWM_VERSION == 12)
	offset = 5;
#endif
    //
    // disable PllUnlock Signals
    //
    HWREGH(SYSCTL_BASE + SYSCTL_O_PWM_SYSFAULT_MASK) = HWREGH(SYSCTL_BASE + SYSCTL_O_PWM_SYSFAULT_MASK)| (1<<offset);
}

/**
 * @brief enable the extended DC trip signals
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  dcType is to Select Trip signal-triggered DCAL/H,DCBL/H.
 * @param [in]  mode :	ExtendDCInput_1        = 1, //!< TZ7~9,Trip16~28
 *	 					ExtendDCInput_2        = 2, //!< Trip29~31,Combination_input,Combination_input_new1~3.
 * @remarks
 * !!! If this function is used, the basic Trip signal will no longer be available for the selected DCXX to use.
 * This function enable the extended DC trip signals
 */
__STATIC_INLINE void
EPWM_enableDigitalCompareTripInput_Extend(uint32_t base,EPWM_DigitalCompareType dcType,DCInput_Extend mode)
{
    //
    // enable the extended DC trip signals
    //
    HWREGH(base + EPWM_O_DCTRIPSELEXP) = ( (HWREGH(base + EPWM_O_DCTRIPSELEXP))& ~(0x2 << dcType) )| (mode<< dcType) ;
}

/**
 * @brief disable the extended DC trip signals
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  dcType is to Select Trip signal-triggered DCAL/H,DCBL/H.
 * @remarks
 * This function disable the extended DC trip signals
 */
__STATIC_INLINE void
EPWM_disableDigitalCompareTripInput_Extend(uint32_t base,EPWM_DigitalCompareType dcType)
{
    //
    // disable the extended DC trip signals
    //
    HWREGH(base + EPWM_O_DCTRIPSELEXP) = (HWREGH(base + EPWM_O_DCTRIPSELEXP))& ~(0x2 << dcType);
}
#ifdef __cplusplus
}
#endif


#endif /* DEVICE_DRIVERLIB_EPWM_TZ_DC_H_ */
