/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    eqep.c
*   @brief   
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "eqep.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

/**
 * \brief   Configures eQEP module position-compare unit.
 * 
 *  \param base is the base address of the eQEP module.
 *  \param config is the configuration for the eQEP module
 *  position-compare unit.  See below for a description of this parameter.
 *  \param compareValue is the value to which the position count value is
 *  compared for a position-compare event.
 *  \param cycles is the width of the pulse that can be generated on a
 *  position-compare event.  It is in units of 4 SYSCLKOUT cycles.
 * 
 *  This function configures the operation of the eQEP module position-compare
 *  unit.  The \e config parameter provides the configuration of the
 *  position-compare unit and is the logical OR of several values:
 * 
 *  - \b EQEP_COMPARE_NO_SYNC_OUT, \b EQEP_COMPARE_IDX_SYNC_OUT, or
 *    \b EQEP_COMPARE_STROBE_SYNC_OUT specify if there is a sync output pulse
 *    and which pin should be used.
 *  - \b EQEP_COMPARE_NO_SHADOW, \b EQEP_COMPARE_LOAD_ON_ZERO, or
 *    \b EQEP_COMPARE_LOAD_ON_MATCH specify if a shadow is enabled and when
 *    should the load should occur--QPOSCNT = 0 or QPOSCNT = QPOSCOMP.
 * 
 *  The \e cycles is used to select the width of the sync output pulse. The
 *  width of the resulting pulse will be \e cycles * 4 * the width of a
 *  SYSCLKOUT cycle. The maximum width is 4096 * 4 * SYSCLKOUT cycles.
 * 
 *  \note You can set the sync pulse width independently using the
 *  EQEP_setComparePulseWidth() function.
 * 
 *  \retval None.
*/
GS32_DRIVER_INIT_FUNC_T void
EQEP_setCompareConfig(uint32_t base, uint16_t config, uint32_t compareValue,
                      uint16_t cycles)
{
    uint16_t regValue;

    //
    // Check the arguments.
    //
    ASSERT(EQEP_isBaseValid(base));
    ASSERT(cycles <= (EQEP_QPOSCTL_PCSPW_M + 2U));

    //
    // Set the compare match value
    //
    HWREG(base + EQEP_O_QPOSCMP) = compareValue;

    //
    // Set the shadow register settings and pulse width.
    //
    regValue = (config & (uint16_t)(EQEP_QPOSCTL_PCSHDW |
                  EQEP_QPOSCTL_PCLOAD)) | (cycles - 1U);

    HWREGH(base + EQEP_O_QPOSCTL) = (HWREGH(base + EQEP_O_QPOSCTL) &
                                     ~(EQEP_QPOSCTL_PCSPW_M |
                                       EQEP_QPOSCTL_PCLOAD |
                                       EQEP_QPOSCTL_PCSHDW)) | regValue;

    //
    // Set position compare sync-output mode.
    //
    regValue = config & (uint16_t)(EQEP_QDECCTL_SOEN | EQEP_QDECCTL_SPSEL);

    HWREGH(base + EQEP_O_QDECCTL) = (HWREGH(base + EQEP_O_QDECCTL) &
                                     ~(EQEP_QDECCTL_SOEN |
                                       EQEP_QDECCTL_SPSEL)) | regValue;
}

/**
 * \brief   Sets the polarity of the eQEP module's input signals.
 * 
 *  \param base is the base address of the eQEP module.
 *  \param invertQEPA is the flag to negate the QEPA input.
 *  \param invertQEPB is the flag to negate the QEPA input.
 *  \param invertIndex is the flag to negate the index input.
 *  \param invertStrobe is the flag to negate the strobe input.
 * 
 *  This function configures the polarity of the inputs to the eQEP module. To
 *  negate the polarity of any of the input signals, pass \b true into its
 *  corresponding parameter in this function. Pass \b false to leave it as-is.
 * 
 *  \retval None.
*/
GS32_DRIVER_INIT_FUNC_T void
EQEP_setInputPolarity(uint32_t base, Bool invertQEPA, Bool invertQEPB,
                      Bool invertIndex, Bool invertStrobe)
{
    //
    // Check the arguments.
    //
    ASSERT(EQEP_isBaseValid(base));

    //
    // Configure QEPA signal
    //
    if(invertQEPA)
    {
        HWREGH(base + EQEP_O_QDECCTL) |= EQEP_QDECCTL_QAP;
    }
    else
    {
        HWREGH(base + EQEP_O_QDECCTL) &= ~EQEP_QDECCTL_QAP;
    }

    //
    // Configure QEPB signal
    //
    if(invertQEPB)
    {
        HWREGH(base + EQEP_O_QDECCTL) |= EQEP_QDECCTL_QBP;
    }
    else
    {
        HWREGH(base + EQEP_O_QDECCTL) &= ~EQEP_QDECCTL_QBP;
    }

    //
    // Configure index signal
    //
    if(invertIndex)
    {
        HWREGH(base + EQEP_O_QDECCTL) |= EQEP_QDECCTL_QIP;
    }
    else
    {
        HWREGH(base + EQEP_O_QDECCTL) &= ~EQEP_QDECCTL_QIP;
    }

    //
    // Configure strobe signal
    //
    if(invertStrobe)
    {
        HWREGH(base + EQEP_O_QDECCTL) |= EQEP_QDECCTL_QSP;
    }
    else
    {
        HWREGH(base + EQEP_O_QDECCTL) &= ~EQEP_QDECCTL_QSP;
    }
}

/**
 * \brief   Clear EQEP registers
 *
 * \param   none
 *
 * \retval  None
 */
GS32_DRIVER_INIT_FUNC_T void EQEP_DeInit(){
		HWREG(EQEP1_BASE + EQEP_O_QPOSCNT) &= 0U;
		HWREG(EQEP1_BASE + EQEP_O_QPOSINIT) &= 0U;
		HWREG(EQEP1_BASE + EQEP_O_QPOSMAX) &= 0U;
		HWREG(EQEP1_BASE + EQEP_O_QPOSCMP) &= 0U;
		HWREG(EQEP1_BASE + EQEP_O_QPOSILAT) &= 0U;
		HWREG(EQEP1_BASE + EQEP_O_QPOSSLAT) &= 0U;
		HWREG(EQEP1_BASE + EQEP_O_QPOSLAT) &= 0U;
		HWREG(EQEP1_BASE + EQEP_O_QUTMR) &= 0U;
		HWREG(EQEP1_BASE + EQEP_O_QUPRD) &= 0U;
	
		HWREGH(EQEP1_BASE + EQEP_O_QWDTMR) &= 0U;
		HWREGH(EQEP1_BASE + EQEP_O_QWDPRD) &= 0U;
		HWREGH(EQEP1_BASE + EQEP_O_QDECCTL) &= 0U;
		//clear position counter
		HWREGH(EQEP1_BASE + EQEP_O_QEPCTL) |= 0x80;
		HWREGH(EQEP1_BASE + EQEP_O_QEPCTL) &= 0U;
		HWREGH(EQEP1_BASE + EQEP_O_QCAPCTL) &= 0U;
		HWREGH(EQEP1_BASE + EQEP_O_QPOSCTL) &= 0U;
		HWREGH(EQEP1_BASE + EQEP_O_QEINT) &= 0U;
		HWREGH(EQEP1_BASE + EQEP_O_QFLG) &= 0U;
		HWREGH(EQEP1_BASE + EQEP_O_QCLR) |= 0x1FFFU;
		HWREGH(EQEP1_BASE + EQEP_O_QCLR) &= 0U;
		HWREGH(EQEP1_BASE + EQEP_O_QFRC) &= 0U;
		HWREGH(EQEP1_BASE + EQEP_O_QEPSTS) |= 0b10001110U;//0x8E
		HWREGH(EQEP1_BASE + EQEP_O_QEPSTS) &= 0U;
		HWREGH(EQEP1_BASE + EQEP_O_QCTMR) &= 0U;
		HWREGH(EQEP1_BASE + EQEP_O_QCPRD) &= 0U;
		HWREGH(EQEP1_BASE + EQEP_O_QCTMRLAT) &= 0U;
		HWREGH(EQEP1_BASE + EQEP_O_QCPRDLAT) &= 0U;
		HWREG(EQEP1_BASE + EQEP_O_REV) &= 0U;
		HWREG(EQEP1_BASE + EQEP_O_QEPSTROBESEL) &= 0U;
		HWREG(EQEP1_BASE + EQEP_O_QMACTRL) &= 0U;
		HWREG(EQEP1_BASE + EQEP_O_QEPSRCSEL) &= 0U;
		
}

#ifdef __cplusplus
}
#endif

