/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    flash.c
*   @brief   
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif

#include "gs32_version.h"
#include "inc/hw_types.h"
/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "device.h"
#include "flash.h"

#if IS_GS32F00xx(0x30)
#include "core_feature_cache.h"
#endif

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

/* None */
/**
 * \brief   "program data." 
 *
 * \param   base is the base address of the flash registers.
 * \param   data is data to program.
 * \param   length is data length to program.
 *
 * \retval  None
 */
void Flash_programData(uint32_t base, uint8_t *data, uint8_t length)
{
    uint8_t i;
    uint8_t num = length / 4;
    uint8_t write_bit = (length % 4) * 8;

    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    ASSERT(length > 0 && length <= 16);

    for (i = 0; i < num; i++)
    {
        HWREG(base + FLASH_O_DATA0 + i * 4) = *((uint32_t *)data + i);
    }

    if (write_bit != 0)
				HWREG(base + FLASH_O_DATA0 + num * 4) = (*((uint32_t *)data + num)) | (0xFFFFFFFF - (1 << write_bit) + 1);//write remain byte and other bit set to 1

    for (i = num + 1; i < 4; i++)
    {
        HWREG(base + FLASH_O_DATA0 + i * 4) = 0xFFFFFFFF;
    }
}

/**
 * \brief   "program trim data." 
 *
 * \param   base is the base address of the flash registers.
 * \param   data is trim data to program.
 * \param   length is trim data length to program.
 *
 * \retval  None
 */
void Flash_programTrimData(uint32_t base, uint32_t *trim_data, uint8_t length)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    ASSERT(length <= 5);

    if (length >= 1)
        HWREG(base + FLASH_O_DATA0) = *trim_data;
    if (length >= 2)
        HWREG(base + FLASH_O_DATA1) = *(trim_data + 1);
    if (length >= 3)
        HWREG(base + FLASH_O_DATA2) = *(trim_data + 2);
    if (length >= 4)
        HWREG(base + FLASH_O_DATA3) = *(trim_data + 3);
    if (length >= 5)
        HWREG(base + FLASH_O_DATA4) = *(trim_data + 4);
}

/**
 * \brief   "get flash status." 
 *
 * \param   base is the base address of the flash registers.
 * \param   status as bellow:
 *          FLASH_STATUS_WRITE_ERROR
 *          FLASH_STATUS_DISCHARGED_STATUS
 *          FLASH_STATUS_WRITE_COMP_STATUS
 *          FLASH_STATUS_SEC_DETECTED
 *          FLASH_STATUS_DED_DETECTED
 *          FLASH_STATUS_SMW_ERR_STATUS
 *          FLASH_STATUS_SMW_LATEST_STATUS
 *          FLASH_STATUS_SMW_LOOP_STATUS
 * \retval  NONE
 */
void Flash_clearStatus(uint32_t base, uint32_t status)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    if (status & FLASH_STATUS_WRITE_ERROR)
        HWREG(base + FLASH_O_STATUS) |= FLASH_STATUS_WRITE_ERROR;
    if (status & FLASH_STATUS_DISCHARGED_STATUS)
        HWREG(base + FLASH_O_STATUS) |= FLASH_STATUS_DISCHARGED_STATUS;
    if (status & FLASH_STATUS_WRITE_COMP_STATUS)
        HWREG(base + FLASH_O_STATUS) |= FLASH_STATUS_WRITE_COMP_STATUS;
    if (status & FLASH_STATUS_SEC_DETECTED)
        HWREG(base + FLASH_O_STATUS) |= FLASH_STATUS_SEC_DETECTED;
    if (status & FLASH_STATUS_DED_DETECTED)
        HWREG(base + FLASH_O_STATUS) |= FLASH_STATUS_DED_DETECTED;
    if (status & FLASH_STATUS_SMW_ERR_STATUS)
        HWREG(base + FLASH_O_STATUS) |= FLASH_STATUS_SMW_ERR_STATUS;
    if (status & FLASH_STATUS_SMW_LATEST_STATUS)
        HWREG(base + FLASH_O_STATUS) |= FLASH_STATUS_SMW_LATEST_STATUS;
    if (status & FLASH_STATUS_SMW_LOOP_STATUS)
        HWREG(base + FLASH_O_STATUS) |= FLASH_STATUS_SMW_LOOP_STATUS;
}

/*
*  3.4.2 Manual power and auto invalidate mode
*      This section describes enable and disable sequences when in the manual power and auto
*      invalidate mode.
*  To enable the cache:
*      1. Set operation mode by setting the CCR.SET_MAN_POW bit to 1 and the
*          CCR.SET_MAN_INV bit to 0.
*          (Set register CCR to 0x08.)
*      2. Request power by setting the CCR.POW_REQ bit to 1.
*          (Set CCR to 0x0C.)
*      3. Wait until the SR.POW_STAT bits are equal to 1 which indicates that the power up has
*          completed.
*      4. Enable the cache by setting the CCR.EN bit to 1.
*          (Set register CCR to 0x0D.)
*
*  To disable the cache:
*      1. Set the CCR.EN bit to 0.
*          (Set register CCR to 0xC,)
*      2. Optionally power down the SRAMs by setting the CCR.POW_REQ bit to 0.
*          (Set register CCR to 0x18.)
*/
#if defined(FLASH_INTERFACE_ICACHE_BASE) && defined(FLASH_INTERFACE_DCACHE_BASE)
// TODO: Critical Section!
RAMFUNC_T void FlashSetupCache(uint32_t iCacheEn, uint32_t dCacheEn, uint32_t preFetchEn)
{
	if (iCacheEn && (HWREG(FLASH_INTERFACE_ICACHE_BASE + 4) & 0x03) != FLASH_CACHE_SR_ENABLED) {
		HWREG(FLASH_INTERFACE_ICACHE_BASE) = (1<<6) | (preFetchEn ? (1<<5) : 0) | 0x08;
		HWREG(FLASH_INTERFACE_ICACHE_BASE) = (1<<6) | (preFetchEn ? (1<<5) : 0) | 0x0C;
		while((HWREG(FLASH_INTERFACE_ICACHE_BASE + 4) & 0x10) == 0);
		HWREG(FLASH_INTERFACE_ICACHE_BASE) = (1<<6) | (preFetchEn ? (1<<5) : 0) | 0x0D;
		while ((HWREG(FLASH_INTERFACE_ICACHE_BASE + 4) & 0x03) != FLASH_CACHE_SR_ENABLED);
	} else if (!iCacheEn && (HWREG(FLASH_INTERFACE_ICACHE_BASE + 4) & 0x03) != FLASH_CACHE_SR_DISABLED) {
		HWREG(FLASH_INTERFACE_ICACHE_BASE) = (1<<6) | (preFetchEn ? (1<<5) : 0) | 0x0C;
		HWREG(FLASH_INTERFACE_ICACHE_BASE) = (1<<6) | (preFetchEn ? (1<<5) : 0) | 0x08;
	}

	if (dCacheEn && (HWREG(FLASH_INTERFACE_DCACHE_BASE + 4) & 0x03) != FLASH_CACHE_SR_ENABLED) {
		HWREG(FLASH_INTERFACE_DCACHE_BASE) = (1<<6) | (preFetchEn ? (1<<5) : 0) | 0x08;
		HWREG(FLASH_INTERFACE_DCACHE_BASE) = (1<<6) | (preFetchEn ? (1<<5) : 0) | 0x0C;
		while((HWREG(FLASH_INTERFACE_DCACHE_BASE + 4) & 0x10) == 0);
		HWREG(FLASH_INTERFACE_DCACHE_BASE) = (1<<6) | (preFetchEn ? (1<<5) : 0) | 0x0D;
		while ((HWREG(FLASH_INTERFACE_DCACHE_BASE + 4) & 0x03) != FLASH_CACHE_SR_ENABLED);
	} else if (!dCacheEn && (HWREG(FLASH_INTERFACE_DCACHE_BASE + 4) & 0x03) != FLASH_CACHE_SR_DISABLED) {
		HWREG(FLASH_INTERFACE_DCACHE_BASE) = (1<<6) | (preFetchEn ? (1<<5) : 0) | 0x0C;
		HWREG(FLASH_INTERFACE_DCACHE_BASE) = (1<<6) | (preFetchEn ? (1<<5) : 0) | 0x08;
	}
}
#endif

RAMFUNC_T void FlashInvlaidCache(void)
{
#if IS_GS32F00xx(0x12)
	FlashSetupCache(0, 0, 1);
	FlashSetupCache(1, 1, 1);
#elif IS_GS32F00xx(0x30) || IS_GS32F3xx(0x22)
#if defined(__DCACHE_PRESENT) && (__DCACHE_PRESENT == 1)
	if (DCachePresent()) { // Check whether dcache real present or not
		MFlushDCache();
		MInvalDCache();
	}
#endif
#else
#error "Use the GS32F00xx or GS32F3xx macro to define the specific chip model."
#endif
}

/*
 * @brief     read user option byte
 * @param[in] offset, 0~6
 * @return    one user option byte: byte0~byte6
 */
uint32_t Fapi_getUserOptByte(uint32_t offset)
{
	return HWREGB(OPTION_BITS_BASE+offset);
}

#if IS_GS32F3xx()
int32_t ReadInfoFlash(uint32_t StartAddr, uint8_t *buf, uint32_t bytes)
{
    uint32_t len;

    if (StartAddr < FLASH_INFO_BASE || StartAddr + bytes > FLASH_INFO_BASE + FLASH_INFO_SIZE) {
        //printf("outof info flash address range.\r\n");
        return 0;
    }

    __disable_irq();

    HWREG(FLASH_INTERFACE_EFC_BASE + FLASH_O_ADDRESS) = 0x80000;   //switch to info flash

    uint8_t *info = (uint8_t *)(StartAddr - FLASH_INFO_BASE + FLASH_BASE);    //map to FLASH_BASE

    for (len= 0; len <bytes; len += 1) {
        buf[len] = info[len];
    }

    HWREG(FLASH_INTERFACE_EFC_BASE + FLASH_O_ADDRESS) = 0;   //switch to main flash

    __enable_irq();

    return len;
}
#endif

#ifdef __cplusplus
}
#endif
