/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    hal_can.h
*   @brief   
*
*/
/*
 * commit history
 * 20240313, LYF, verify for HAL_CAN Send/Receive
 * 20240316, LYF, verify for HAL_CAN Bus error/Bus off/Fifo depth MailBox/Can Nums
 * 20240320, LYF, bug fix: TX frame halting for TPE is ongoing. 
                  buf fix: HAL_CAN_setBitRate Calculate the baud rate to match industry sampling point requirements.
                  add: HAL_CAN_frameGet, HAL_CAN_frameSet, HAL_CAN_findRxMsgID, HAL_CAN_findTxMsgID
 * 20240322, LYF, fix: code error for CANC and CAND port, 
                  translate source files to C source, ASCII text.
 * 20240706, LYF, add feat: canfd timing setting APIs
                  HAL_CAN_setDataBitRate and HAL_CAN_setDataBitTiming;
 * 20240716, Danne, rewrite functions according to new macros.                    
 * 20240814, LYF, disable all interrupts by defaults in func HAL_CAN_setDefConfigs.
 * 20240929, Jason, remove CANFD support, support each MB as Tx or Rx.
 * 20241006, Jason, re-create some API functions and some structure defines.
 */

#ifndef _HAL_CAN_H_
#define _HAL_CAN_H_

#ifdef __cplusplus
extern "C"
{
#endif

/*============================================================================*/
/*                                         INCLUDE FILES
/*============================================================================*/

//#include <can.h>
#include "string.h"
#include "inc/hw_hal_can.h"
#include "regs/regs_hal_can.h"
#include "regs/regs_can.h"
#include "debug.h"

#include "inc/hw_memmap.h"
#include "inc/hw_can.h"
#include "can.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* Total number of MailBox is 32 max or 64 max */
#if HAL_CAN_ENABLE_64_MB
#define HAL_TOTAL_MB_NUMS   USER_DEFINED_MB_NUM
#else
#define HAL_TOTAL_MB_NUMS   (32)
#endif

/* Total number of HW filter */
#define ACF_NUMBER          (16)

/* define HAL_CAN_ENABLE_FD=1 in project settings if need support CANFD*/
//#define HAL_CAN_ENABLE_FD (0)

#if HAL_CAN_ENABLE_FD
#define CAN_PAYLOAD_MAX_LENGTH (64U)
#else
#define CAN_PAYLOAD_MAX_LENGTH (8U)
#endif

/* max read tries in one call of HAL_CAN_updateRxMB */
#define HAL_CAN_MAX_READ_TIMES    16

/* TxFIFO and RxFIFO depth */
#define HAL_CAN_FIFO_SLOTS          (16U)


/*
 * These are the flags used by the flags parameter when calling CAN_setupMessageObject().
 */
/* This indicates that transmit interrupts should be enabled, or are enabled. */
#define CAN_MSG_OBJ_TX_INT_ENABLE          HAL_CAN_IF1MCTL_TXIE
/* This indicates that receive interrupts should be enabled, or are enabled. */
#define CAN_MSG_OBJ_RX_INT_ENABLE          HAL_CAN_IF1MCTL_RXIE
/*
 * This indicates that a message object will use or is using filtering
 * based on the object's message identifier.
 */
#define CAN_MSG_OBJ_USE_ID_FILTER          (0x00000001U)
#define HAL_CAN_MSG_OBJ_USE_ID_FILTER      (0x00000001U)
/*
 * This indicates that a message object will use or is using filtering
 * based on the direction of the transfer.
 */
#define CAN_MSG_OBJ_USE_DIR_FILTER         HAL_CAN_IF1MSK_MDIR
#define HAL_CAN_MSG_OBJ_USE_DIR_FILTER     (0x00000004U)
/*
 * This indicates that a message object will use or is using message
 * identifier filtering based on the extended identifier.
 */
#define CAN_MSG_OBJ_USE_EXT_FILTER         HAL_CAN_IF1MSK_MXTD
#define HAL_CAN_MSG_OBJ_USE_EXT_FILTER     HAL_CAN_IF1MSK_MXTD
/* This indicates that a message object has no flags set. */
#define CAN_MSG_OBJ_NO_FLAGS               (0x00000000U)
#define HAL_CAN_MSG_OBJ_NO_FLAGS           (0x00000000U)
/* CANFD support flags */
#define HAL_CAN_MSG_ENABLE_FDF                 (0x00010000U)
#define HAL_CAN_MSG_ENABLE_BRS                 (0x00020000U)


#define CAN_REG32(base, offset) (*((volatile uint32_t *)((base) + (offset))))

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

typedef enum
{
    HAL_CAN_A,
	HAL_CAN_B,
	HAL_CAN_C,
	HAL_CAN_D
} HAL_CAN_IndexType;


/******************************************************************************
 * This data type is used to for MailBox pMB->ID.bit.xtd
 *****************************************************************************/
typedef enum
{
    //! Set the message ID to standard.
    HAL_CAN_MSG_ID_STD = 0,
	CAN_MSG_ID_STD = 0,

    //! Set the message ID to extended.
    HAL_CAN_MSG_ID_EXT = 1,
	CAN_MSG_ID_EXT = 1
} HAL_CAN_MsgIdType;

/*****************************************************************************
 * This data type is used in CAN_setupMessageObject
 ******************************************************************************/
typedef enum
{
    //! Set the message ID frame to standard.
    CAN_MSG_FRAME_STD,

    //! Set the message ID frame to extended.
    CAN_MSG_FRAME_EXT
} CAN_MsgFrameType;

/*****************************************************************************
 * This data type is used to for MailBox pMB->ID.bit.Dir
 *****************************************************************************/
typedef enum
{
    //! Receive message.
    HAL_CAN_MSG_TYPE_RECEIVE = 0,
	CAN_MSG_TYPE_RECEIVE = 0,
    
    //! Transmit message.
    HAL_CAN_MSG_TYPE_TRANSMIT = 1,
	CAN_MSG_TYPE_TRANSMIT = 1
}
HAL_CAN_MsgType;

/******************************************************************************
 * This definition is used for the type of message object that will
 * be used in HAL_CAN_setupMessageObject() API.
 ******************************************************************************/
typedef enum
{
    //! Transmit message object.
    HAL_CAN_MSG_OBJ_TYPE_TX = 0,
	CAN_MSG_OBJ_TYPE_TX = 0,

    //! Transmit remote request message object
    HAL_CAN_MSG_OBJ_TYPE_TX_REMOTE = 1,
	CAN_MSG_OBJ_TYPE_TX_REMOTE = 1,

    //! Receive message object.
    HAL_CAN_MSG_OBJ_TYPE_RX = 2,
	CAN_MSG_OBJ_TYPE_RX = 2,

    //! Remote frame receive remote, with auto-transmit message object.
    HAL_CAN_MSG_OBJ_TYPE_RXTX_REMOTE = 3,
	CAN_MSG_OBJ_TYPE_RXTX_REMOTE = 3
} HAL_CAN_MsgObjType;

/*
 * This is used to set ID filter type to configure the CAN HW Filter_0~15
 */
typedef enum
{
	CAN_IDTYPE_STD_OR_EXT    = 0b00,  //AIDEE=0, AIDE=0, accept both std ID and ext ID
    CAN_IDTYPE_STD           = 0b10,  //AIDEE=1, AIDE=0, accept std ID only
    CAN_IDTYPE_EXT           = 0b11   //AIDEE=1, AIDE=1, accept ext ID only
}CAN_IDType;

typedef enum
{
    //! CAN Status Inactived.
    CAN_INACTIVED = 0U,
    //! CAN Status Actived.
    CAN_ACTIVED = 1U
}CAN_StatusActived;

typedef enum
{
    //! CAN set to no clearn.
    CAN_NO_CLEAR = 0U,
    //! CAN set to Clear Action.
    CAN_CLEAR = 1U
}CAN_ActionClear;

/* It's the CAN TxFIFO or RxFIFO status, TCTRL.bit.TSSTAT or RCTRL.bit.RSTAT */
typedef enum
{
    CAN_FIFO_EMPTY       = 0x00,  /* FIFO empty */
    CAN_FIFO_BELOW_HALF  = 0x01,  /* TxFIFO less than or equal to half-full */
	                              /* RxFIFO >empty and <AFWL */
	CAN_FIFO_ABOVE_HALF  = 0x02,  /* more than half-full */
	                              /* RxFIFO >=AFWL */
	CAN_FIFO_FULL        = 0x03   /* FIFO full */
}CAN_FIFO_STATUS;


/******************************************************************************
 * This definition is used to determine the clock source that will
 * be set up via a call to the HAL_CAN_selectClockSource() API.
 ******************************************************************************/
typedef enum
{
    //! Peripheral System Clock Source
    HAL_CAN_CLOCK_SOURCE_SYS    = 0x0,

    //! External Oscillator Clock Source
    HAL_CAN_CLOCK_SOURCE_XTAL   = 0x1,

    //! Auxiliary Clock Input Source
    HAL_CAN_CLOCK_SOURCE_AUX    = 0x2
} HAL_CAN_ClockSource;


typedef enum
{
    //! HAL_CAN status disabled
    HAL_CAN_DISABLE = 0U,

	//! HAL_CAN status enabled
    HAL_CAN_ENABLE = 1U
}HAL_CAN_StatusEnable;

typedef enum
{
    //! HAL_CAN Status Inactived.
    HAL_CAN_INACTIVED = 0U,
    //! HAL_CAN Status Actived.
    HAL_CAN_ACTIVED = 1U
}HAL_CAN_StatusActived;

typedef enum
{
    //! HAL_CAN Status Invalid.
    HAL_CAN_INVALID = 0U,
    //! HAL_CAN Status Valid.
    HAL_CAN_VALID = 1U
}HAL_CAN_StatusValid;

/* mailbox number 1~64 */
typedef enum{
	can_mailbox_1,
	can_mailbox_2,
	can_mailbox_3,
	can_mailbox_4,
	can_mailbox_5,
	can_mailbox_6,
	can_mailbox_7,
	can_mailbox_8,
	can_mailbox_9,
	can_mailbox_10,
	can_mailbox_11,
	can_mailbox_12,
	can_mailbox_13,
	can_mailbox_14,
	can_mailbox_15,
	can_mailbox_16,
	can_mailbox_17,
	can_mailbox_18,
	can_mailbox_19,
	can_mailbox_20,
	can_mailbox_21,
	can_mailbox_22,
	can_mailbox_23,
	can_mailbox_24,
	can_mailbox_25,
	can_mailbox_26,
	can_mailbox_27,
	can_mailbox_28,
	can_mailbox_29,
	can_mailbox_30,
	can_mailbox_31,
	can_mailbox_32,
#if HAL_CAN_ENABLE_64_MB
	can_mailbox_33,
	can_mailbox_34,
	can_mailbox_35,
	can_mailbox_36,
	can_mailbox_37,
	can_mailbox_38,
	can_mailbox_39,
	can_mailbox_40,
	can_mailbox_41,
	can_mailbox_42,
	can_mailbox_43,
	can_mailbox_44,
	can_mailbox_45,
	can_mailbox_46,
	can_mailbox_47,
	can_mailbox_48,
	can_mailbox_49,
	can_mailbox_50,
	can_mailbox_51,
	can_mailbox_52,
	can_mailbox_53,
	can_mailbox_54,
	can_mailbox_55,
	can_mailbox_56,
	can_mailbox_57,
	can_mailbox_58,
	can_mailbox_59,
	can_mailbox_60,
	can_mailbox_61,
	can_mailbox_62,
	can_mailbox_63,
	can_mailbox_64,
#endif
}CAN_MAILBOX_ID;

//!
//! HAL_CAN mailbox RAM Objects, Tx or Rx;
//!
typedef struct{
    __IO union HAL_CAN_MSGx_MSK_REG      Mask; //IF_MSK
    __IO union HAL_CAN_MSGx_ID_REG       ID;   //IF_ARB
    __IO union HAL_CAN_MSGx_CTRL_REG     Ctrl; //IF_MCTL
#if defined(CAN_PAYLOAD_MAX_LENGTH) && (CAN_PAYLOAD_MAX_LENGTH <= 8U)
    __IO union HAL_CAN_MSGx_DATA_REG     Data[CAN_PAYLOAD_MAX_LENGTH/4];
#else
    __IO union HAL_CAN_MSGx_DATA_REG     Data[16];
#endif

    /* filter number =(1~15) used by this MB, only Rx MB need this value, 0 means no filter linked to this MB */
    uint32_t       filterNum;
    /* Counter for the numbs of RX Msg lost by Mailbox*/
    uint32_t       lostFrameCnt;
    uint32_t       rxFrameCnt;
    uint32_t       txLaunchFrameCnt;
    uint32_t       txDoneFrameCnt;
} HAL_CAN_MAIL_BOX;

//!
//! HAL_CAN Receive Data control Objects;
//!
typedef struct {
    __IO union HAL_CAN_IFxCMD_REG            CMD;
    __IO union HAL_CAN_IFxMSK_REG            MSK;
    __IO union HAL_CAN_IFxARB_REG            ARB;
    __IO union HAL_CAN_IFxMCTL_REG           MCTL;
    __IO union HAL_CAN_IFxDATA_REG           DATA;
    __IO union HAL_CAN_IFxDATB_REG           DATB;
}HAL_CAN_RW_IF;

//!
//! HAL_CAN Transmit Data control Objects;
//!
typedef struct {
    __IO union HAL_CAN_IF3OBS_REG            OBS;
    __IO union HAL_CAN_IFxMSK_REG            MSK;
    __IO union HAL_CAN_IFxARB_REG            ARB;
    __IO union HAL_CAN_IFxMCTL_REG           MCTL;
    __IO union HAL_CAN_IFxDATA_REG           DATA;
    __IO union HAL_CAN_IFxDATB_REG           DATB;
    __IO union HAL_CAN_IF3UPD_REG            UPD;
}HAL_CAN_RO_IF;

//!
//! HAL_CAN Transmit Data control Objects;
//!
typedef uint32_t (*HAL_CAN_UserFunc_type)(uint32_t base);

//!
//! HAL_CAN Control Domain Register Objects;
//!
typedef struct {
    __IO union HAL_CAN_CTRL_REG              CAN_CTL;
    __IO union HAL_CAN_ES_REG                CAN_ES;
    __IO union HAL_CAN_ERRC_REG              CAN_ERRC;
    __IO union CAN_STIME_REG                 CAN_BTR; // use HW register directly for BTR (CAN & CANFD Control TIming)
    __IO union CAN_FTIME_REG                 CAN_DTR; // use HW register directly for DTR (CANFD DATA Timing only)
    __IO union HAL_CAN_INT_REG               CAN_INT;
    __IO union HAL_CAN_GLB_INT_EN_REG        CAN_GLB_INT_EN;
    __IO union HAL_CAN_GLB_INT_FLG_REG       CAN_GLB_INT_FLG;
    __IO union HAL_CAN_GLB_INT_CLR_REG       CAN_GLB_INT_CLR;
    __IO union HAL_CAN_ABOTR_REG             CAN_ABOTR;
    
    __IO union HAL_CAN_TXRQ_X_REG            CAN_TXRQ_X;
    __IO union HAL_CAN_TXRQ_21_REG           CAN_TXRQ_21;

    __IO union HAL_CAN_NDAT_X_REG            CAN_NDAT_X;
    __IO union HAL_CAN_NDAT_21_REG           CAN_NDAT_21;

    __IO union HAL_CAN_IPEN_X_REG            CAN_IPEN_X;
    __IO union HAL_CAN_IPEN_21_REG           CAN_IPEN_21;

    __IO union HAL_CAN_MVAL_X_REG            CAN_MVAL_X;
    __IO union HAL_CAN_MVAL_21_REG           CAN_MVAL_21;

    __IO union HAL_CAN_IP_MUX21_REG          CAN_IP_MUX21;
    HAL_CAN_RW_IF                       CAN_IF1;  // Read/Write Interface for MailBox
    HAL_CAN_RW_IF                       CAN_IF2;  // Read/Write Interface for MailBox
    HAL_CAN_RO_IF                       CAN_IF3;  // Auto Readonly Interface for MailBox

    __IO uint32_t             CAN_TXRQ_32_63;
    __IO uint32_t             CAN_NDAT_32_63;
    __IO uint32_t             CAN_IPEN_32_63;
    __IO uint32_t             CAN_MVAL_32_63;
    __IO uint32_t             HAL_CAN_MB_Status[3]; //each bit represents one MB
    HAL_CAN_UserFunc_type     HAL_CAN_UserFunc_RX_ISR;
    HAL_CAN_UserFunc_type     HAL_CAN_UserFunc_TX_ISR;
    HAL_CAN_UserFunc_type     HAL_CAN_UserFunc_Error_ISR;
    HAL_CAN_UserFunc_type     HAL_CAN_UserFunc_BusOff_ISR;
    CAN_MSGx                  *pCanTxBuf;
    CAN_MSGx                  *pCanRxBuf;
    HAL_CAN_MAIL_BOX          *pMailBox;
    uint32_t                  RxDiscardMsgNum;
    uint32_t                  TxDiscardMsgNum;
}HAL_CAN_REGS;

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

extern HAL_CAN_REGS HAL_CAN_R[HAL_CAN_NUMS];
extern HAL_CAN_MAIL_BOX HAL_CAN_MB[HAL_CAN_NUMS][HAL_TOTAL_MB_NUMS];

/* ========================================================================== */
/*                            External Functions                              */
/* ========================================================================== */
extern int32 __gs_ctz(uint32 src);

/* ========================================================================== */
/*                    Prototypes of static inline API functions               */
/* ========================================================================== */
//*****************************************************************************
//
//! \internal
//!
//! Checks a HAL_CAN base address.
//!
//! \param base is the base address of the HAL_CAN controller.
//!
//! This function determines if a HAL_CAN controller base address is valid.
//!
//! \return Returns \b true if the base address is valid and \b false
//! otherwise.
//
//*****************************************************************************
static inline bool
HAL_CAN_isBaseValid(uint32_t base)
{
    return(
           (base == CANA_BASE)
           || (base == CANB_BASE)
#ifdef CANC_BASE
		   || (base == CANC_BASE)
#endif
#ifdef CAND_BASE
		   || (base == CAND_BASE)
#endif
		);
}

//******************************************************
//!
//! Get Error Counter Register RECNT and TECNT 
//!
//*******************************************************
static inline uint32_t
HAL_CAN_getTxErrorCounter(uint32_t base)
{
	ASSERT(HAL_CAN_isBaseValid(base));

	return (HWREG(base + CAN_O_TECNT) >> CAN_TECNT_TECNT_S);
}

static inline uint32_t
HAL_CAN_getRxErrorCounter(uint32_t base)
{
	ASSERT(HAL_CAN_isBaseValid(base));

    return (HWREG(base + CAN_O_RECNT) >> CAN_RECNT_RECNT_S);
}

/*
 * @brief       get the specified MailBox status bit
 * @param [in]  base, CAN base address
 * @param [in]  objID, MB number, 0~63, meaning mailbox_1~64
 * @return      1, if the specified MailBox status bit is 1;
 *              0, if the specified MailBox status bit is 0;
 */
static inline Bool
HAL_CAN_getMailBoxStatus(uint32_t base, uint32_t objID)
{
    ASSERT(HAL_CAN_isBaseValid(base));
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];

	if(objID < 32)
	{
		if(pHalCanRegs->HAL_CAN_MB_Status[0] & (1<<objID))
			return 1;
		else
			return 0;
	}
	else if(objID < 64)
	{
		if(pHalCanRegs->HAL_CAN_MB_Status[1] & (1<<(objID-32)))
			return 1;
		else
			return 0;
	}
	else
		return 0;
}

/*
 * @brief    get 32 MailBoxes status, 32bit, mailbox_1~32
 */
static inline uint32_t
HAL_CAN_getAllMailBoxStatus(uint32_t base)
{
    ASSERT(HAL_CAN_isBaseValid(base));
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];

	return (pHalCanRegs->HAL_CAN_MB_Status[0]);
}

/*
 * @brief    set the specified MailBox status bit
 */
static inline void
HAL_CAN_setMailBoxStatus(HAL_CAN_REGS *pHalCanRegs, uint32_t objID)
{
	if(objID < 32)
   	    pHalCanRegs->HAL_CAN_MB_Status[0] |= (1<<objID);
	else if(objID < 64)
		pHalCanRegs->HAL_CAN_MB_Status[1] |= (1<<(objID-32));
}

/*
 * @brief    clear the specified MailBox status bit
 */
static inline void
HAL_CAN_clearMailBoxStatus(HAL_CAN_REGS *pHalCanRegs, uint32_t objID)
{
	if(objID < 32)
   	   	pHalCanRegs->HAL_CAN_MB_Status[0] &= (~(1<<objID));
	else if(objID < 64)
		pHalCanRegs->HAL_CAN_MB_Status[1] &= (~(1<<(objID-32)));
}

/*
 * @brief   clear TxOK and RxOK bit each time poll the CAN status register
 */
static inline void
HAL_CAN_clearTxOKRxOK(uint32_t base)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];
   	pHalCanRegs->CAN_ES.all &= (~(HAL_CAN_ES_RXOK | HAL_CAN_ES_TXOK));
}

/*
 * @brief   get the CAN ID from specified Message Buffer
 */
static inline uint32_t
HAL_CAN_getMsgIDFromBuf(CAN_MSGx *pMsgBuf)
{
	uint32_t msgID;
    if (pMsgBuf->CTRL.bit.IDE == 0U)
    {
        msgID = pMsgBuf->ID.bit.ID & 0x7FF; //Standard Format: ID(10:0)
    }
    else
    {
        msgID = pMsgBuf->ID.bit.ID & 0x1FFFFFFF;         //Extended Format: ID(28:0)
    }
    return msgID;
}

/*
 * @brief    copy a CAN message from CAN HW buffer to CAN_MSGx buffer
 */
static inline void
HAL_CAN_readRxBuf(uint32_t base, CAN_MSGx* pMsgBuf)
{
    CAN_MSGx* pCanRbuf = NULL;
    uint32_t  idx = 0U;
    uint32_t  dlc = 8U;
    uint32_t  nSize;

    /* Check the arguments. */
    ASSERT(HAL_CAN_isBaseValid(base));
    ASSERT(pMsgBuf != NULL);

    /* Point to rx buf domain reg base. */
    pCanRbuf = PTR_TO_CAN_RXBUF(base);

    /* Get the RX frame data from registers to RAM. */
    pMsgBuf->ID.all      = pCanRbuf->ID.all;
    pMsgBuf->CTRL.all    = pCanRbuf->CTRL.all;

#if HAL_CAN_ENABLE_FD
    if(pMsgBuf->CTRL.bit.FDF)
    {
    	nSize = CAN_getMessageLength(pMsgBuf->CTRL.bit.DLC);
    	for(idx=0; idx<nSize/4; idx++)
    		pMsgBuf->Data[idx].all = pCanRbuf->Data[idx].all;
    }
    else
    {
    	pMsgBuf->Data[0].all = pCanRbuf->Data[0].all;
    	pMsgBuf->Data[1].all = pCanRbuf->Data[1].all;
    }
#else
    dlc = 8;

    /* Update Msg Data. */
    for (idx = 0; idx < dlc/4; idx++)
    {
    	pMsgBuf->Data[idx].all = pCanRbuf->Data[idx].all;
    }
#endif

    pMsgBuf->RTS[0].all  = pCanRbuf->RTS[0].all;
    pMsgBuf->RTS[1].all  = pCanRbuf->RTS[1].all;
}

/*
 * @brief Get the first pending status MailBox by counting trailing zeros in the MB status variable.
 *        Each bit of the MB status represents one mailbox
 *        The instruction __ctz is used to count the trailing zero of a 32bit integer.
 *
 * @para[in]  pHalCanRegs, pointer to a HalCan struct
 *
 * @return    objID  the MailBox ID with status bit set.
 *            return value is 0~31 for mailbox_1~32
 *            return value is 32 if all mailboxes' status bits are 0.
 *            return value is 32~63 for mailbox_33~64
 *            return value is 64 if all mailboxes' status bits are 0.
 *            If all status bits are 0, objID will be 32 (total 32 mailboxes) or 64 (total 64 mailboxes).
 */
static inline uint32_t
HAL_CAN_getObjIDbyMBStatus(HAL_CAN_REGS *pHalCanRegs)
{
	uint32_t nStatus;
	uint32_t objID;
	nStatus = pHalCanRegs->HAL_CAN_MB_Status[0];
	objID = __gs_ctz(nStatus);

#if HAL_CAN_ENABLE_64_MB
	if(nStatus == 0)
	{
		nStatus = pHalCanRegs->HAL_CAN_MB_Status[1];
		objID = __gs_ctz(nStatus) + 32;
	}
#endif

    return objID;
}

/*
 * @brief Get the first MailBox with Rx new data pending
 * @para[in]  pHalCanRegs, pointer to a HalCan struct
 * @return    objID  the MailBox ID with status bit set, value 0~31
 *                   if no MB received new data, returns value 32 (or 64 if there are total 64 mailboxes).
 */
static inline uint32_t
HAL_CAN_getRxObjIDbyMBStatus(HAL_CAN_REGS *pHalCanRegs)
{
	uint32_t nStatus;
	uint32_t objID;
	HAL_CAN_MAIL_BOX *pMB = pHalCanRegs->pMailBox;

	nStatus = pHalCanRegs->HAL_CAN_MB_Status[0];

	for(objID=0; objID<HAL_TOTAL_MB_NUMS; objID++)
	{
		if((nStatus & (1<<objID)) && (pMB->ID.bit.Dir == CAN_MSG_TYPE_RECEIVE))
			break;
	}
    return objID;
}

/*
 * @brief     Get the first MailBox with pending interrupt status by counting trailing zeros in the IPEN_21 register.
 * @para[in]  base, CAN base address
 * @return    int_id  the first MailBox ID with interrupt pending;
 *            return value is 0 if no mailbox has interrupt pending;
 *            return value is 1~32 for mailbox_1~32;
 *            return value is 33~64 for mailbox_33~64
 */
static inline uint32_t
HAL_CAN_getINT0ID(uint32_t base)
{
	uint32_t int_id;
	uint32_t nStatus;
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];
	nStatus = pHalCanRegs->CAN_IPEN_21.all;
	if(nStatus)
		int_id = __gs_ctz(nStatus)+1; /* count the trailing zeros of MB status */
	else
	{
#if HAL_CAN_ENABLE_64_MB
		nStatus = pHalCanRegs->CAN_IPEN_32_63;
		if(nStatus)
			int_id = __gs_ctz(nStatus) + 32 + 1;
		else
			int_id = 0;
#else
		int_id = 0;
#endif
	}

    return int_id;
}

/*
 * @brief     Get the virtual TXRQ_21 register
 * @para[in]  base, CAN base address
 * @return    TXRQ_21 value
 */
static inline uint32_t
HAL_CAN_getTXRQ21(uint32_t base)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];

    return pHalCanRegs->CAN_TXRQ_21.all;
}

/*
 * @brief     Clear the specified bit in virtual TXRQ_21 register
 * @para[in]  base, CAN base address
 * @para[in]  objID, 0~31 (0~63 if total 64 MB)
 */
static inline void
HAL_CAN_clearTXRQ21(uint32_t base, uint32_t objID)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];

#if HAL_CAN_ENABLE_64_MB
	if(objID < 32)
		pHalCanRegs->CAN_TXRQ_21.all &= (~(1<<objID));
	else if(objID < 64)
		pHalCanRegs->CAN_TXRQ_32_63 &= (~(1<<(objID-32)));
#else
	pHalCanRegs->CAN_TXRQ_21.all &= (~(1<<objID));
#endif
}

/*
 * @brief     Set the specified bit in virtual TXRQ_21 register
 * @para[in]  base, CAN base address
 * @para[in]  objID, 0~31 (0~63 if total 64 MB)
 */
static inline void
HAL_CAN_setTXRQ21(uint32_t base, uint32_t objID)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];
#if HAL_CAN_ENABLE_64_MB
	if(objID < 32)
		pHalCanRegs->CAN_TXRQ_21.all |= (1<<objID);
	else if(objID < 64)
		pHalCanRegs->CAN_TXRQ_32_63 |= (1<<(objID-32));
#else
	pHalCanRegs->CAN_TXRQ_21.all |= (1<<objID);
#endif
}

/*
 * @brief     Get the virtual NDAT_21 register
 * @para[in]  base, CAN base address
 * @return    NDAT_21 value
 */
static inline uint32_t
HAL_CAN_getNDAT21(uint32_t base)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];

    return pHalCanRegs->CAN_NDAT_21.all;
}

/*
 * @brief     Clear the specified bit in virtual NDAT_21 register
 * @para[in]  base, CAN base address
 * @para[in]  objID, 0~31 (0~63 if total 64 MB)
 */
static inline void
HAL_CAN_clearNDAT21(uint32_t base, uint32_t objID)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];
#if HAL_CAN_ENABLE_64_MB
	if(objID < 32)
		pHalCanRegs->CAN_NDAT_21.all &= (~(1<<objID));
	else if(objID < 64)
		pHalCanRegs->CAN_NDAT_32_63 &= (~(1<<(objID-32)));
#else
	pHalCanRegs->CAN_NDAT_21.all &= (~(1<<objID));
#endif
}

/*
 * @brief     Set the specified bit in virtual NDAT_21 register
 * @para[in]  base, CAN base address
 * @para[in]  objID, 0~31 (0~63 if total 64 MB)
 */
static inline void
HAL_CAN_setNDAT21(uint32_t base, uint32_t objID)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];
#if HAL_CAN_ENABLE_64_MB
	if(objID < 32)
		pHalCanRegs->CAN_NDAT_21.all |= (1<<objID);
	else if(objID < 64)
		pHalCanRegs->CAN_NDAT_32_63 |= (1<<(objID-32));
#else
	pHalCanRegs->CAN_NDAT_21.all |= (1<<objID);
#endif
}

/*
 * @brief     Get the virtual IPEN_21 register
 * @para[in]  base, CAN base address
 * @return    IPEN_21 value
 */
static inline uint32_t
HAL_CAN_getIPEN21(uint32_t base)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];

    return pHalCanRegs->CAN_IPEN_21.all;
}

/*
 * @brief     Clear the specified bit in virtual IPEN21 register
 * @para[in]  base, CAN base address
 * @para[in]  objID, 0~31 (0~63 if total 64 MB)
 */
static inline void
HAL_CAN_clearIPEN21(uint32_t base, uint32_t objID)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];

#if HAL_CAN_ENABLE_64_MB
	if(objID < 32)
        pHalCanRegs->CAN_IPEN_21.all &= (~(1<<objID));
	else if(objID < 64)
		pHalCanRegs->CAN_IPEN_32_63 &= (~(1<<(objID-32)));
#else
	pHalCanRegs->CAN_IPEN_21.all &= (~(1<<objID));
#endif
}

/*
 * @brief     Set the specified bit in virtual IPEN21 register
 * @para[in]  base, CAN base address
 * @para[in]  objID, 0~31 (0~63 if total 64 MB)
 */
static inline void
HAL_CAN_setIPEN21(uint32_t base, uint32_t objID)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];

#if HAL_CAN_ENABLE_64_MB
	if(objID < 32)
        pHalCanRegs->CAN_IPEN_21.all |= (1<<objID);
	else if(objID < 64)
		pHalCanRegs->CAN_IPEN_32_63 |= (1<<(objID-32));
#else
	pHalCanRegs->CAN_IPEN_21.all |= (1<<objID);
#endif
}

#if HAL_CAN_ENABLE_64_MB
/*
 * @brief     Get the virtual TXRQ_32_63 register
 * @para[in]  base, CAN base address
 * @return    TXRQ_32_63 value
 */
static inline uint32_t
HAL_CAN_getTXRQ_32_63(uint32_t base)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];

    return pHalCanRegs->CAN_TXRQ_32_63;
}

/*
 * @brief     Get the virtual NDAT_32_63 register
 * @para[in]  base, CAN base address
 * @return    NDAT_32_63 value
 */
static inline uint32_t
HAL_CAN_getNDAT_32_63(uint32_t base)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];

    return pHalCanRegs->CAN_NDAT_32_63;
}

/*
 * @brief     Get the virtual IPEN_32_63 register
 * @para[in]  base, CAN base address
 * @return    IPEN_32_63 value
 */
static inline uint32_t
HAL_CAN_getIPEN_32_63(uint32_t base)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];

    return pHalCanRegs->CAN_IPEN_32_63;
}

#endif

/*
 * @brief  After reading the data in the specified mailbox(with objID)
 *         call this function to release that mailbox for new data.
 *         The same operation is performed in HAL_CAN_readMessage
 *         though it doesn't call this function.
 *
 * @para[in]  base, CAN base address
 * @para[in]  objID, the ID of the mailbox, 0~31
 *
 * @return    none
 */
static inline void
HAL_CAN_releaseMB(uint32_t base, uint32_t objID)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];

	pHalCanRegs->pMailBox[objID].Ctrl.bit.NewDat = FALSE;  //clear NewDat bit
	pHalCanRegs->pMailBox[objID].Ctrl.bit.IntPnd = FALSE;  //clear IntPnd bit
	if(objID < 32)
	{
		pHalCanRegs->CAN_NDAT_21.all &= (~(1<<objID));
		pHalCanRegs->CAN_IPEN_21.all &= (~(1<<objID));
		pHalCanRegs->HAL_CAN_MB_Status[0] &= (~(1<<objID));
	}
#if HAL_CAN_ENABLE_64_MB
	else if(objID < 64)
	{
		pHalCanRegs->CAN_NDAT_32_63 &= (~(1<<(objID-32)));
		pHalCanRegs->CAN_IPEN_32_63 &= (~(1<<(objID-32)));
		pHalCanRegs->HAL_CAN_MB_Status[1] &= (~(1<<(objID-32)));
	}
#endif
}

/*
 * @brief     Get the pointer to the MB of specified objID
 * @para[in]  base, CAN base address
 * @para[in]  objID, 0~31 (or 0~63 if there are total 64 mailboxes)
 */
static inline HAL_CAN_MAIL_BOX *HAL_CAN_getMBbyObjID(uint32_t base, uint32_t objID)
{
	uint32_t can_indx = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_indx];
	return ((pHalCanRegs->pMailBox) + objID);
}

/*
 * @brief    Copy CAN Frame from MailBox to TBUF (CAN transmit buffer registers)
 * @para[in] pMB, pointer to the virtual mailbox from which data is copied.
 */
static inline void
HAL_CAN_writeMBtoTBUF(uint32_t base, HAL_CAN_MAIL_BOX *pMB)
{
    CAN_MSGx* pTBUF = PTR_TO_CAN_TXBUF(base);  // Point to tx buf domain reg base.
    uint32_t nSize;
    union CAN_MSGx_CTRL_REG nCtrl;

    /* Set the TX frame data from RAM to registers. */
    pTBUF->ID.bit.ID      = pMB->ID.bit.ID;

    /* Update Msg Data. */
    pTBUF->Data[0].all = pMB->Data[0].all;
    pTBUF->Data[1].all = pMB->Data[1].all;

    /*
     * Update control field.
     * Must clear the CTRL register to avoid previous Tx Msg set the FDF bit.
     */
    pTBUF->CTRL.all = 0;
    nCtrl.all = 0;

    nCtrl.bit.IDE = pMB->ID.bit.xtd;
    nCtrl.bit.DLC = pMB->Ctrl.bit.DLC;

//    pTBUF->CTRL.bit.DLC = pMB->Ctrl.bit.DLC;
//    pTBUF->CTRL.bit.IDE = pMB->ID.bit.xtd;

    /* Additional data copy for CANFD, more data and FDF+BRS bit. */
#if HAL_CAN_ENABLE_FD
    if(pMB->Ctrl.bit.FDF)
    {
    	uint32_t nDLC = pMB->Ctrl.bit.DLC;
    	nSize = CAN_getMessageLength(nDLC);
    	for(uint32_t i=2; i<nSize/4; i++)
    		pTBUF->Data[i].all = pMB->Data[i].all;
    	nCtrl.bit.BRS = pMB->Ctrl.bit.BRS;
    	nCtrl.bit.FDF = pMB->Ctrl.bit.FDF;
    }
#endif
    pTBUF->CTRL.all = nCtrl.all;
}

/*
 * @brief  set interrupt pending bit for specified MB
 *         same as setIPEN21
 *         CAN_IPEN_21 each bit is for each mailbox_1~32
 *         CAN_IPEN_32_63 each bit is for each mailbox_33~64
 * @para[in]  pHalCanRegs, pointer to HAL_CAN regs structure
 * @para[in]  objID, 0~31 (0~63 if total 64MB)
 */
static inline void
HAL_CAN_setIntPnd(HAL_CAN_REGS *pHalCanRegs, uint32_t objID)
{
	HAL_CAN_MAIL_BOX *pMB = pHalCanRegs->pMailBox + objID;
	pMB->Ctrl.bit.IntPnd = 1;

#if HAL_CAN_ENABLE_64_MB
	if(objID < 32)
        pHalCanRegs->CAN_IPEN_21.all |= (1<<objID);
	else if(objID < 64)
		pHalCanRegs->CAN_IPEN_32_63 |= (1<<(objID-32));
#else
	pHalCanRegs->CAN_IPEN_21.all |= (1<<objID);
#endif
}

/*
 * @brief     clear interrupt pending bit for specified MB same as clearIPEN21
 * @para[in]  pHalCanRegs, pointer to HAL_CAN regs structure
 * @para[in]  objID, 0~31 (0~63 if total 64MB)
 */
static inline void
HAL_CAN_clearIntPnd(HAL_CAN_REGS *pHalCanRegs, uint32_t objID)
{
	HAL_CAN_MAIL_BOX *pMB = pHalCanRegs->pMailBox + objID;
	pMB->Ctrl.bit.IntPnd = 0;

#if HAL_CAN_ENABLE_64_MB
	if(objID < 32)
        pHalCanRegs->CAN_IPEN_21.all &= (~(1<<objID));
	else if(objID < 64)
		pHalCanRegs->CAN_IPEN_32_63 &= (~(1<<(objID-32)));
#else
	pHalCanRegs->CAN_IPEN_21.all &= (~(1<<objID));
#endif
}

/*
 * @brief  return the RxFIFO status.
 * @ret    0 = empty
 *         1 = >empty and <AFWL
 *         2 = >=AFWL
 *         3 = Full
 */
static inline uint32_t
CAN_getRxFifoStatus(uint32_t base)
{
	ASSERT(HAL_CAN_isBaseValid(base));
	return ((HWREG(base + CAN_O_RCTRL) & CAN_RCTRL_RSTAT_M) >> CAN_RCTRL_RSTAT_S);
}

/*
 * @brief  release the current RBUF
 * @details  if current RBUF is read by CPU, this function is called to place next received message into RBUF
 */
static inline void
CAN_releaseRB(uint32_t baseAddr)
{
    HWREG(baseAddr + CAN_O_RCTRL) |= CAN_RCTRL_RREL_M;
}

/*
 * @brief  return the STB TxFIFO status.
 * @ret    0 = empty
 *         1 = less or equal to half-full
 *         2 = more than half-full
 *         3 = full
 */
static inline uint32_t
CAN_getTxFifoStatus(uint32_t base)
{
	ASSERT(HAL_CAN_isBaseValid(base));
	return ((HWREG(base + CAN_O_TCTRL) & CAN_TCTRL_TSSTAT_M) >> CAN_TCTRL_TSSTAT_S);
}

/*
 * @brief select Secondary Transmit Buffer (STB)
 * @para[in] base, CAN base address
 */
static inline void
CAN_selectSTB(uint32_t base)
{
    HWREG(base + CAN_O_TCMD) |= CAN_TCMD_TBSEL_M;
}

/*
 * @brief     mark the current Tx slot as filled and switch to next slot
 * @para[in]  baseAddr, can base address
 * @details   After all frame bytes are written to the TBUF registers, the host controller has to set
 *            TSNEXT to signal that this slot has been filled. Then the CAN-CTRL core connects the TBUF
 *            registers to the next slot. Once a slot is marked as filled a transmission can be started
 *            using TSONE or TSALL.
 */
static inline void
CAN_switchNextSTBSlot(uint32_t baseAddr)
{
//    CAN_REG32(baseAddr, CAN_O_TCTRL) |= CAN_TCTRL_TSNEXT_M;
	uint32_t reg_val = HWREG(baseAddr + CAN_O_TCTRL);

	reg_val &= ~(CAN_TCMD_TSONE_M | CAN_TCMD_TSALL_M |
				CAN_TCMD_TPE_M | CAN_TCMD_TPA_M |
				CAN_TCMD_TSA_M | CAN_TCTRL_TSNEXT_M |
				CAN_RCTRL_RBALL_M | CAN_RCTRL_RREL_M |
				CAN_CFG_STAT_BUSOFF_M);

	reg_val |= CAN_TCTRL_TSNEXT_M;

	HWREG(baseAddr + CAN_O_TCTRL) = reg_val;
}

/*
 * @brief, transmit all filled slots in Tx FIFO
 * @para[in] baseAddr, can base address
 */
static inline void
CAN_setTransmitAll(uint32_t baseAddr)
{
//    CAN_REG32(baseAddr, CAN_O_TCMD) |= CAN_TCMD_TSALL_M;
	uint32_t reg_val;

	reg_val &= ~(CAN_TCMD_TSONE_M | CAN_TCMD_TSALL_M |
				CAN_TCMD_TPE_M | CAN_TCMD_TPA_M |
				CAN_TCMD_TSA_M | CAN_TCTRL_TSNEXT_M |
				CAN_RCTRL_RBALL_M | CAN_RCTRL_RREL_M |
				CAN_CFG_STAT_BUSOFF_M);

	reg_val |= CAN_TCMD_TSALL_M;

	HWREG(baseAddr + CAN_O_TCMD) = reg_val;
}

/*
 * @brief  disable the specified ID filter
 */
static inline void
CAN_enableMsgFilter(uint32_t base, uint32_t filtNum)
{
	ASSERT(HAL_CAN_isBaseValid(base));
    ASSERT(filtNum < ACF_NUMBER);

    CAN_REG32(base, CAN_O_ACFCTRL) |= (1 << (filtNum + CAN_ACF_EN_0_AE_0_S));
}

/*
 * @brief  disable the specified ID filter
 */
static inline void
CAN_disableMsgFilter(uint32_t base, uint32_t filtNum)
{
	ASSERT(HAL_CAN_isBaseValid(base));
    ASSERT(filtNum < ACF_NUMBER);

    CAN_REG32(base, CAN_O_ACFCTRL) &= (~(1 << (filtNum + CAN_ACF_EN_0_AE_0_S)));
}

/*
 * @brief  disable all ID filter
 */
static inline void
CAN_disableAllMsgFilter(uint32_t base)
{
	ASSERT(HAL_CAN_isBaseValid(base));

    CAN_REG32(base, CAN_O_ACFCTRL) &= (~(0xFFFF << CAN_ACF_EN_0_AE_0_S));
}

/*
 * @brief  get the status of all ID filters,
 *         bit0~bit15 indicates filter0~15 enabled(=1) or disabled(=0)
 */
static inline uint32_t
CAN_getAllMsgFilterStatus(uint32_t base)
{
	ASSERT(HAL_CAN_isBaseValid(base));

    return (CAN_REG32(base, CAN_O_ACF_EN_0)>>CAN_ACF_EN_0_AE_0_S);
}

/*
 * @brief    Need counter the trailing zeros in status register, so declare the built-in CTZ function here.
 */
__EXTERN_INLINE_MANUAL int32 __CAN_ctz(uint32 src){  return __gs_ctz(src); }

/*
 * @brief  get the first available filter from 1 to 15
 * @ret    filter number, 1~15;
 * @details   the filter_enable bits are processed like this
 *            filter_enable=0001 => FFFF0001 => 0000FFFE => __ctz(FFFE)=1  => return available filter number = 1;
 *            filter_enable=0003 => FFFF0003 => 0000FFFC => __ctz(FFFC)=2  => return available filter number = 2;
 *            ......
 *            filter_enable=7FFF => FFFF7FFF => 00008000 => __ctz(8000)=15 => return available filter number = 15;
 *            filter_enable=FFFF => FFFFFFFF => 00000000 => __ctz(0000)=32 => return available filter number > 15;
 *            filter_enable=FFEF => FFFFFFEF => 00000010 => __ctz(0010)=4  => return available filter number = 4;
 */
static inline uint32_t
CAN_getAvailableMsgFilterNum(uint32_t base)
{
	ASSERT(HAL_CAN_isBaseValid(base));

    /*
     * get the filter status, each bit stands for one filter.
     * bitN==1 means filter N is already used by certain Rx mailbox;
     * bitN==0 means filter N is available for new Rx mailbox.
     */
    uint32_t filt_enable = CAN_getAllMsgFilterStatus(base);
    /*
     * set bit0, which is for filter 0, meaning it's occupied by certain purpose
     * set bit16~31, because filter 16~31 doesn't exist.
     * Available filters are filter number 1~15
     */
    filt_enable = (~(filt_enable|0xFFFF0001));

    return __CAN_ctz(filt_enable);
}


/* ========================================================================== */
/*                         Global Functions Declare                           */
/* ========================================================================== */

void CAN_setMsgFilter(uint32_t baseAddr, uint32_t filter_num, uint32_t can_id, uint32_t id_mask, CAN_IDType id_type);

void HAL_CAN_setupMessageObject(uint32_t base,
		               uint32_t objID, uint32_t msgID,
					   CAN_MsgFrameType idType, HAL_CAN_MsgObjType msgType,
                       uint32_t msgIDMask, uint32_t flags, uint16_t msgLen);
bool HAL_CAN_setupMsgFilter(uint32_t base, uint32_t objID, uint32_t id_type);
void HAL_CAN_disableMsgFilter(uint32_t base, uint32_t objID);

//Call this API in CAN ISR or in periodical task
uint32_t HAL_CAN_updateRegsAndMB(uint32_t can_base);

/* CAN send message object, input data is 16bit*8 array, each element is used as uint8_t */
bool HAL_CAN_sendMessage(uint32_t base, uint32_t objID, uint16_t msgLen, const uint16_t *msgData);
/* CAN send message object, input data is 8bit*8 array */
bool HAL_CAN_sendMessage_8bit(uint32_t base, uint32_t objID, uint16_t msgLen, const uint8_t *msgData);
/* CAN send message object, input data is 16bit*4 array */
bool HAL_CAN_sendMessage_16bit(uint32_t base, uint32_t objID, uint16_t msgLen, const uint16_t *msgData);
/* CAN send message object, input data is 32bit*2 array */
bool HAL_CAN_sendMessage_32bit(uint32_t base, uint32_t objID, uint16_t msgLen, const uint32_t *msgData);

/* Read data from CAN mail box into specified buffer */
bool HAL_CAN_readMessage(uint32_t base, uint32_t objID, uint16_t *msgData);
bool HAL_CAN_readMessage_8bit(uint32_t base, uint32_t objID, uint8_t *msgData);
bool HAL_CAN_readMessageWithID(uint32_t base,
                           uint32_t objID,
                           CAN_MsgFrameType *frameType,
                           uint32_t *msgID,
                           uint16_t *msgData);
bool HAL_CAN_checkRxMessageId(uint32_t base, uint32_t can_id);

/* CAN send message is composed of two actions: load mailbox and send mailbox. */
/* Load data into mail box */
bool HAL_CAN_loadMailBox(uint32_t base, uint32_t objID, uint32_t msgID, uint16_t *pMsgData,uint32_t nLength);
bool HAL_CAN_loadMailBoxDataOnly(uint32_t base, uint32_t objID, const uint16_t *pMsgData,uint32_t nLength);
bool HAL_CAN_loadMailBoxDataOnly_8bit(uint32_t base, uint32_t objID, const uint8_t *pMsgData,uint32_t nLength);
bool HAL_CAN_loadMailBoxDataOnly_16bit(uint32_t base, uint32_t objID, const uint16_t *pMsgData,uint32_t nLength);
bool HAL_CAN_loadMailBoxDataOnly_32bit(uint32_t base, uint32_t objID, const uint32_t *pMsgData,uint32_t nLength);
/* Send out the specified mail box */
bool HAL_CAN_sendMailBox(uint32_t base, uint32_t objID);

/*
 * @brief Go through all 32 mailbox to match the specified CAN_ID (msgID)
 *
 * @para[in]  pHalCanRegs, pointer to a HalCan structure.
 * @para[in]  msgID, the 11bit or 28bit ID to be accepted by a mailbox
 *
 * @return    objID, the first MailBox ID which matches the specified CAN_ID
 */
uint32_t HAL_CAN_lookupObjIDbyMsgID(HAL_CAN_REGS *pHalCanRegs, uint32_t msgID);

/* Initialize some variables for HAL_CAN */
void HAL_CAN_initCfg(uint32_t base);

void HAL_CAN_setBitRate(uint32_t base, uint32_t clockFreq, uint32_t bitRate,
               uint16_t bitTime);
void HAL_CAN_setDataBitRate(uint32_t base, uint32_t clockFreq, uint32_t bitRate,
		       uint16_t bitTime);


void HAL_CANA_IRQHandler(void);
void HAL_CANB_IRQHandler(void);
#if (HAL_CAN_NUMS >= 3U)
void HAL_CANC_IRQHandler(void);
#endif
#if (HAL_CAN_NUMS >= 4U)
void HAL_CAND_IRQHandler(void);
#endif

#ifdef __cplusplus
}
#endif

#endif /* _HAL_CAN_H_ */
