/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifndef __DRIVERLIB_IIC_V30_H_
#define __DRIVERLIB_IIC_V30_H_

#ifdef __cplusplus
extern "C"{
#endif

#include <stdbool.h>

#include "gs32_version.h"
#include "inc/hw_i2c.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "debug.h"

#if IS_GS32F00xx(0x30) || IS_GS32F3xx(0x23)


#ifndef DOXYGEN_PDF_IGNORE

/**
 * @brief Controller-transmitter mode
 *
 */
#define I2C_CONTROLLER_SEND_MODE    0x0600U

/**
 * @brief Controller-receiver mode
 *
 */
#define I2C_CONTROLLER_RECEIVE_MODE 0x0400U

/**
 * @brief Target-transmitter mode
 *
 */
#define I2C_TARGET_SEND_MODE        0x0200U

/**
 * @brief Target-receiver mode
 *
 */
#define I2C_TARGET_RECEIVE_MODE     0x0000U

/**
 * @brief Only applies to Controller mode
 *
 */
#define I2C_REPEAT_MODE         0x0080U

/**
 * @brief Enable start byte mode
 *
 */
#define I2C_START_BYTE_MODE     0x0010U

/**
 * @brief Enable free data (no addr) format
 *
 */
#define I2C_FREE_DATA_FORMAT    0x0008U

/**
 * @brief Arbitration-lost interrupt
 *
 */
#define I2C_INT_ARB_LOST        0x00001U

/**
 * @brief NACK interrupt
 *
 */
#define I2C_INT_NO_ACK          0x00002U

/**
 * @brief Register-access-ready interrupt
 *
 */
#define I2C_INT_REG_ACCESS_RDY  0x00004U

/**
 * @brief Receive-data-ready interrupt
 *
 */
#define I2C_INT_RX_DATA_RDY     0x00008U

/**
 * @brief Transmit-data-ready interrupt
 *
 */
#define I2C_INT_TX_DATA_RDY     0x00010U

/**
 * @brief Stop condition detected
 *
 */
#define I2C_INT_STOP_CONDITION  0x00020U

/**
 * @brief Addressed as target interrupt
 *
 */
#define I2C_INT_ADDR_TARGET     0x00200U

/**
 * @brief RX FIFO level interrupt
 *
 */
#define I2C_INT_RXFF            0x10000U

/**
 * @brief TX FIFO level interrupt
 *
 */
#define I2C_INT_TXFF            0x20000U


#define I2C_MASTER_SEND_MODE              I2C_CONTROLLER_SEND_MODE
#define I2C_MASTER_RECEIVE_MODE           I2C_CONTROLLER_RECEIVE_MODE
#define I2C_SLAVE_SEND_MODE               I2C_TARGET_SEND_MODE
#define I2C_SLAVE_RECEIVE_MODE            I2C_TARGET_RECEIVE_MODE
#define I2C_INT_ADDR_SLAVE                I2C_INT_ADDR_TARGET
#define I2C_STS_ADDR_SLAVE                I2C_STS_ADDR_TARGET
#define I2C_STS_SLAVE_DIR                 I2C_STS_TARGET_DIR
#define I2C_INTSRC_ADDR_SLAVE             I2C_INTSRC_ADDR_TARGET

#define I2C_initMaster                    I2C_initController
#define I2C_setSlaveAddress               I2C_setTargetAddress
#define I2C_setOwnSlaveAddress            I2C_setOwnAddress



#define I2C_STR_INTMASK     ((uint16_t)I2C_INT_ARB_LOST |                      \
                             (uint16_t)I2C_INT_NO_ACK |                        \
                             (uint16_t)I2C_INT_REG_ACCESS_RDY |                \
                             (uint16_t)I2C_INT_RX_DATA_RDY |                   \
                             (uint16_t)I2C_INT_TX_DATA_RDY |                   \
                             (uint16_t)I2C_INT_STOP_CONDITION |                \
                             (uint16_t)I2C_INT_ADDR_TARGET)


/**
 * @brief Arbitration-lost
 *
 */
#define I2C_STS_ARB_LOST        0x0001U

/**
 * @brief No-acknowledgment (NACK)
 *
 */
#define I2C_STS_NO_ACK          0x0002U

/**
 * @brief Register-access-ready (ARDY)
 *
 */
#define I2C_STS_REG_ACCESS_RDY  0x0004U

/**
 * @brief Receive-data-ready
 *
 */
#define I2C_STS_RX_DATA_RDY     0x0008U

/**
 * @brief Transmit-data-ready
 *
 */
#define I2C_STS_TX_DATA_RDY     0x0010U

/**
 * @brief Stop condition detected
 *
 */
#define I2C_STS_STOP_CONDITION  0x0020U

/**
 * @brief Byte transmit complete
 *
 */
#define I2C_STS_BYTE_SENT       0x0040U

/**
 * @brief Address of all zeros detected
 *
 */
#define I2C_STS_ADDR_ZERO       0x0100U

/**
 * @brief Addressed as target
 *
 */
#define I2C_STS_ADDR_TARGET     0x0200U

/**
 * @brief Transmit shift register empty
 *
 */
#define I2C_STS_TX_EMPTY        0x0400U

/**
 * @brief Receive shift register full
 *
 */
#define I2C_STS_RX_FULL         0x0800U

/**
 * @brief Bus busy, wait for STOP or reset
 *
 */
#define I2C_STS_BUS_BUSY        0x1000U

/**
 * @brief NACK was sent
 *
 */
#define I2C_STS_NACK_SENT       0x2000U

/**
 * @brief Addressed as target transmitter
 *
 */
#define I2C_STS_TARGET_DIR      0x4000U

#endif

/**
 * @brief  I2C interrupts to be returned by I2C_getInterruptSource().
 *
 * @param I2C_INTSRC_NONE           No interrupt pending.
 * @param I2C_INTSRC_ARB_LOST       Arbitration-lost interrupt.
 * @param I2C_INTSRC_NO_ACK         NACK interrupt.
 * @param I2C_INTSRC_REG_ACCESS_RDY Register-access-ready interrupt.
 * @param I2C_INTSRC_RX_DATA_RDY    Receive-data-ready interrupt.
 * @param I2C_INTSRC_TX_DATA_RDY    Transmit-data-ready interrupt.
 * @param I2C_INTSRC_STOP_CONDITION Stop condition detected.
 * @param I2C_INTSRC_ADDR_TARGET    Addressed as target interrupt.
 *
 */
typedef enum
{
    I2C_INTSRC_NONE,
    I2C_INTSRC_ARB_LOST,
    I2C_INTSRC_NO_ACK,
    I2C_INTSRC_REG_ACCESS_RDY,
    I2C_INTSRC_RX_DATA_RDY,
    I2C_INTSRC_TX_DATA_RDY,
    I2C_INTSRC_STOP_CONDITION,
    I2C_INTSRC_ADDR_TARGET
} I2C_InterruptSource;

/**
 * @brief Values that can be passed to I2C_setFIFOInterruptLevel() as the \e txLevel
 * 		  parameter, returned by I2C_getFIFOInterruptLevel() in the \e txLevel
 * 		  parameter, and returned by I2C_getTxFIFOStatus().
 *
 * @param I2C_FIFO_TXEMPTY Transmit FIFO empty
 * @param I2C_FIFO_TX0 Transmit FIFO empty
 * @param I2C_FIFO_TX1 Transmit FIFO 1/16 full.
 * @param I2C_FIFO_TX2 Transmit FIFO 2/16 full.
 * @param I2C_FIFO_TX3 Transmit FIFO 3/16 full.
 * @param I2C_FIFO_TX4 Transmit FIFO 4/16 full.
 * @param I2C_FIFO_TX5 Transmit FIFO 5/16 full.
 * @param I2C_FIFO_TX6 Transmit FIFO 6/16 full.
 * @param I2C_FIFO_TX7 Transmit FIFO 7/16 full.
 * @param I2C_FIFO_TX8 Transmit FIFO 8/16 full.
 * @param I2C_FIFO_TX9 Transmit FIFO 9/16 full.
 * @param I2C_FIFO_TX10 Transmit FIFO 10/16 full.
 * @param I2C_FIFO_TX11 Transmit FIFO 11/16 full.
 * @param I2C_FIFO_TX12 Transmit FIFO 12/16 full.
 * @param I2C_FIFO_TX13 Transmit FIFO 13/16 full.
 * @param I2C_FIFO_TX14 Transmit FIFO 14/16 full.
 * @param I2C_FIFO_TX15 Transmit FIFO 15/16 full.
 * @param I2C_FIFO_TX16 Transmit FIFO full.
 * @param I2C_FIFO_TXFULL Transmit FIFO full.
 */
typedef enum
{
    I2C_FIFO_TXEMPTY    = 0x0000U,
    I2C_FIFO_TX0        = 0x0000U,
    I2C_FIFO_TX1        = 0x0001U,
    I2C_FIFO_TX2        = 0x0002U,
    I2C_FIFO_TX3        = 0x0003U,
    I2C_FIFO_TX4        = 0x0004U,
    I2C_FIFO_TX5        = 0x0005U,
    I2C_FIFO_TX6        = 0x0006U,
    I2C_FIFO_TX7        = 0x0007U,
    I2C_FIFO_TX8        = 0x0008U,
    I2C_FIFO_TX9        = 0x0009U,
    I2C_FIFO_TX10       = 0x000AU,
    I2C_FIFO_TX11       = 0x000BU,
    I2C_FIFO_TX12       = 0x000CU,
    I2C_FIFO_TX13       = 0x000DU,
    I2C_FIFO_TX14       = 0x000EU,
    I2C_FIFO_TX15       = 0x000FU,
    I2C_FIFO_TX16       = 0x0010U,
    I2C_FIFO_TXFULL     = 0x0010U
} I2C_TxFIFOLevel;

/**
 * @brief Values that can be passed to I2C_setFIFOInterruptLevel() as the \e rxLevel
 * 		  parameter, returned by I2C_getFIFOInterruptLevel() in the \e rxLevel
 * 		  parameter, and returned by I2C_getRxFIFOStatus().
 *
 * @param I2C_FIFO_RXEMPTY Receive FIFO empty
 * @param I2C_FIFO_RX1 Receive FIFO 1/16 full
 * @param I2C_FIFO_RX2 Receive FIFO 2/16 full.
 * @param I2C_FIFO_RX3 Receive FIFO 3/16 full.
 * @param I2C_FIFO_RX4 Receive FIFO 4/16 full.
 * @param I2C_FIFO_RX5 Receive FIFO 5/16 full.
 * @param I2C_FIFO_RX6 Receive FIFO 6/16 full.
 * @param I2C_FIFO_RX7 Receive FIFO 7/16 full.
 * @param I2C_FIFO_RX8 Receive FIFO 8/16 full.
 * @param I2C_FIFO_RX9 Receive FIFO 9/16 full.
 * @param I2C_FIFO_RX10 Receive FIFO 10/16 full.
 * @param I2C_FIFO_RX11 Receive FIFO 11/16 full.
 * @param I2C_FIFO_RX12 Receive FIFO 12/16 full.
 * @param I2C_FIFO_RX13 Receive FIFO 13/16 full.
 * @param I2C_FIFO_RX14 Receive FIFO 14/16 full.
 * @param I2C_FIFO_RX15 Receive FIFO 15/16 full.
 * @param I2C_FIFO_RX16 Receive FIFO full.
 * @param I2C_FIFO_RXFULL Receive FIFO full.
 */
typedef enum
{
    I2C_FIFO_RXEMPTY    = 0x0000U,
    I2C_FIFO_RX0        = 0x0000U,
    I2C_FIFO_RX1        = 0x0001U,
    I2C_FIFO_RX2        = 0x0002U,
    I2C_FIFO_RX3        = 0x0003U,
    I2C_FIFO_RX4        = 0x0004U,
    I2C_FIFO_RX5        = 0x0005U,
    I2C_FIFO_RX6        = 0x0006U,
    I2C_FIFO_RX7        = 0x0007U,
    I2C_FIFO_RX8        = 0x0008U,
    I2C_FIFO_RX9        = 0x0009U,
    I2C_FIFO_RX10       = 0x000AU,
    I2C_FIFO_RX11       = 0x000BU,
    I2C_FIFO_RX12       = 0x000CU,
    I2C_FIFO_RX13       = 0x000DU,
    I2C_FIFO_RX14       = 0x000EU,
    I2C_FIFO_RX15       = 0x000FU,
    I2C_FIFO_RX16       = 0x0010U,
    I2C_FIFO_RXFULL     = 0x0010U
} I2C_RxFIFOLevel;

/**
 * @brief Values that can be passed to I2C_setBitCount() as the \e size parameter.
 *
 * @param I2C_BITCOUNT_1 1 bit per data byte
 * @param I2C_BITCOUNT_2 2 bits per data byte
 * @param I2C_BITCOUNT_3 3 bits per data byte
 * @param I2C_BITCOUNT_4 4 bits per data byte
 * @param I2C_BITCOUNT_5 5 bits per data byte
 * @param I2C_BITCOUNT_6 6 bits per data byte
 * @param I2C_BITCOUNT_7 7 bits per data byte
 * @param I2C_BITCOUNT_8 8 bits per data byte
 *
 */
typedef enum
{
    I2C_BITCOUNT_1  = 1U,
    I2C_BITCOUNT_2  = 2U,
    I2C_BITCOUNT_3  = 3U,
    I2C_BITCOUNT_4  = 4U,
    I2C_BITCOUNT_5  = 5U,
    I2C_BITCOUNT_6  = 6U,
    I2C_BITCOUNT_7  = 7U,
    I2C_BITCOUNT_8  = 0U
} I2C_BitCount;

/**
 * @brief Values that can be passed to I2C_setAddressMode() as the \e mode parameter.
 *
 * @param I2C_ADDR_MODE_7BITS   7-bit address
 * @param I2C_ADDR_MODE_10BITS  10-bit address
 *
 */
typedef enum
{
    I2C_ADDR_MODE_7BITS        = 0x0000U,
    I2C_ADDR_MODE_10BITS       = 0x0100U
} I2C_AddressMode;

/**
 * @brief Values that can be passed to I2C_setExtendedMode() as the \e mode parameter.
 *
 * @param I2C_EADDR_MODE_BC Backward compatibility
 * @param I2C_EADDR_MODE_FWD Forward compatibility
 *
 */
typedef enum
{
    I2C_EADDR_MODE_BC  = 1U,   //!<  Backward compatibility
    I2C_EADDR_MODE_FWD = 2U    //!<  Forward compatibility
} I2C_ExtendedMode;

/**
 * @brief Values that can be passed to I2C_setEmulationMode() as the \e mode parameter.
 *
 * @param I2C_EMULATION_STOP_SCL_LOW If SCL is low, keep it low. If high, stop when it goes low again.
 * @param I2C_EMULATION_FREE_RUN Continue I2C operation regardless
 *
 */
typedef enum
{
    I2C_EMULATION_STOP_SCL_LOW = 0x0000U,
    I2C_EMULATION_FREE_RUN     = 0x4000U
} I2C_EmulationMode;

/**
 * @brief Values that can be passed to I2C_initController() as the \e dutyCycle parameter.
 *
 * @param I2C_DUTYCYCLE_33 Clock duty cycle is 33%
 * @param I2C_DUTYCYCLE_50 Clock duty cycle is 50%
 *
 */
typedef enum
{
    I2C_DUTYCYCLE_33,
    I2C_DUTYCYCLE_50
} I2C_DutyCycle;

/**
 * @brief Checks an I2C base address.
 *
 * @param base is the base address of the I2C instance used.
 *
 * @return Returns \b true if the base address is valid and \b false
 * otherwise.
 *
 */
#ifdef DEBUG
static inline bool
I2C_isBaseValid(uint32_t base)
{
    return(
           (base == I2CA_BASE)
          );
}
#endif

/**
 * @brief Enables the I2C module.
 *
 * @param base is the base address of the I2C instance used.
 *
 * @return None.
 *
 */
static inline void
I2C_enableModule(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));

    HWREGH(base + I2C_O_MDR) |= I2C_MDR_IRS;
}

/**
 * @brief Disables the I2C module.
 *
 * @param base is the base address of the I2C instance used.
 *
 * @return None.
 *
 */
static inline void
I2C_disableModule(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));

    HWREGH(base + I2C_O_MDR) &= ~(I2C_MDR_IRS);
}

/**
 * @brief Enables the transmit and receive FIFOs.
 *
 * @param base is the base address of the I2C instance used.
 *
 * @return None.
 *
 */
static inline void
I2C_enableFIFO(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));

    HWREGH(base + I2C_O_FFTX) |= I2C_FFTX_I2CFFEN | I2C_FFTX_TXFFRST;
    HWREGH(base + I2C_O_FFRX) |= I2C_FFRX_RXFFRST;
}

/**
 * @brief Disables the transmit and receive FIFOs.
 *
 * @param base is the base address of the I2C instance used.
 *
 * @return None.
 *
 */
static inline void
I2C_disableFIFO(uint32_t base)
{

    ASSERT(I2C_isBaseValid(base));

    HWREGH(base + I2C_O_FFTX) &= ~(I2C_FFTX_I2CFFEN | I2C_FFTX_TXFFRST);
    HWREGH(base + I2C_O_FFRX) &= ~I2C_FFRX_RXFFRST;
}

/**
 * @brief Sets the FIFO level at which interrupts are generated.
 *
 * @param base is the base address of the I2C instance used.
 * @param txLevel is the transmit FIFO interrupt level, specified as
 * \b I2C_FIFO_TX0, \b I2C_FIFO_TX1, \b I2C_FIFO_TX2, . . . or
 * \b I2C_FIFO_TX16.
 * @param rxLevel is the receive FIFO interrupt level, specified as
 * \b I2C_FIFO_RX0, \b I2C_FIFO_RX1, \b I2C_FIFO_RX2, . . . or
 * \b I2C_FIFO_RX16.
 *
 * @return None.
 *
 */
static inline void
I2C_setFIFOInterruptLevel(uint32_t base, I2C_TxFIFOLevel txLevel,
                          I2C_RxFIFOLevel rxLevel)
{
    ASSERT(I2C_isBaseValid(base));

    HWREGH(base + I2C_O_FFTX) = (HWREGH(base + I2C_O_FFTX) &
                                 (~I2C_FFTX_TXFFIL_M)) | (uint16_t)txLevel;
    HWREGH(base + I2C_O_FFRX) = (HWREGH(base + I2C_O_FFRX) &
                                 (~I2C_FFRX_RXFFIL_M)) | (uint16_t)rxLevel;
}

/**
 * @brief Gets the FIFO level at which interrupts are generated.
 *
 * @param base is the base address of the I2C instance used.
 * @param txLevel is a pointer to storage for the transmit FIFO level,
 * returned as one of \b I2C_FIFO_TX0, \b I2C_FIFO_TX1,
 * \b I2C_FIFO_TX2, . . . or \b I2C_FIFO_TX16.
 * @param rxLevel is a pointer to storage for the receive FIFO level,
 * returned as one of \b I2C_FIFO_RX0, \b I2C_FIFO_RX1,
 * \b I2C_FIFO_RX2, . . . or \b I2C_FIFO_RX16.
 *
 * @return None.
 *
 */
static inline void
I2C_getFIFOInterruptLevel(uint32_t base, I2C_TxFIFOLevel *txLevel,
                          I2C_RxFIFOLevel *rxLevel)
{

    ASSERT(I2C_isBaseValid(base));

    *txLevel = (I2C_TxFIFOLevel)(HWREGH(base + I2C_O_FFTX) &
                                 I2C_FFTX_TXFFIL_M);
    *rxLevel = (I2C_RxFIFOLevel)(HWREGH(base + I2C_O_FFRX) &
                                 I2C_FFRX_RXFFIL_M);
}

/**
 * @brief  Get the transmit FIFO status
 *
 * @param base is the base address of the I2C instance used.
 *
 * @return Returns the current number of words in the transmit FIFO specified
 * as one of the following:
 * \b I2C_FIFO_TX0, \b I2C_FIFO_TX1, \b I2C_FIFO_TX2, \b I2C_FIFO_TX3,
 * ..., or \b I2C_FIFO_TX16
 *
 */
static inline I2C_TxFIFOLevel
I2C_getTxFIFOStatus(uint32_t base)
{
    uint16_t level;

    ASSERT(I2C_isBaseValid(base));

    level = ((HWREGH(base + I2C_O_FFTX) & I2C_FFTX_TXFFST_M) >>
              I2C_FFTX_TXFFST_S);

    return((I2C_TxFIFOLevel)level);
}

/**
 * @brief  Get the receive FIFO status.
 *
 * @param base is the base address of the I2C instance used.
 *
 * @return Returns the current number of words in the receive FIFO specified
 * as one of the following:
 * \b I2C_FIFO_RX0, \b I2C_FIFO_RX1, \b I2C_FIFO_RX2, \b I2C_FIFO_RX3,
 * ..., or \b I2C_FIFO_RX16
 *
 */
static inline I2C_RxFIFOLevel
I2C_getRxFIFOStatus(uint32_t base)
{
    uint16_t level;

    ASSERT(I2C_isBaseValid(base));

    level = ((HWREGH(base + I2C_O_FFRX) & I2C_FFRX_RXFFST_M) >>
              I2C_FFRX_RXFFST_S);

    return((I2C_RxFIFOLevel)level);
}

/**
 * @brief  Reads I2C Module clock prescaler value.
 *
 * @param base is the base address of the I2C instance used.
 *
 * @return Returns the I2C prescaler(I2CPSC) cast as an uint16_t.
 *
 */
static inline uint16_t
I2C_getPreScaler(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));

    return(HWREGH(base + I2C_O_PSC));
}

/**
 * @brief  Sets the address that the I2C Controller places on the bus.
 *
 * @param base is the base address of the I2C instance used.
 * @param targetAddr 7-bit or 10-bit target address
 *
 * @return None.
 *
 */
static inline void
I2C_setTargetAddress(uint32_t base, uint16_t targetAddr)
{
    ASSERT(I2C_isBaseValid(base));
    ASSERT(targetAddr <= I2C_TAR_TAR_M);

    HWREGH(base + I2C_O_TAR) = targetAddr;
}

/**
 * @brief  Sets the own address for this I2C module.
 *
 * @param base is the base address of the I2C Target module.
 * @param  Addr is the 7-bit or 10-bit address
 *
 * @return None.
 *
 */
static inline void
I2C_setOwnAddress(uint32_t base, uint16_t Addr)
{
    ASSERT(I2C_isBaseValid(base));
    ASSERT(Addr <= I2C_OAR_OAR_M);

    HWREGH(base + I2C_O_OAR) = Addr;
}

/**
 * @brief  Sets the address that the I2C Controller places on the bus.
 *
 * @param base is the base address of the I2C instance used.
 *
 * @return Returns \b true if the I2C bus is busy; otherwise, returns
 * \b false.
 *
 */
static inline boolean
I2C_isBusBusy(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));

    return((HWREGH(base + I2C_O_STR) & I2C_STR_BB) == I2C_STR_BB);
}

/**
 * @brief  Gets the current I2C module status.
 *
 * @param base is the base address of the I2C instance used.
 *
 * @return The current module status, enumerated as a bit field of
 * \b I2C_STS_ARB_LOST - Arbitration-lost
 * \b I2C_STS_NO_ACK - No-acknowledgment (NACK)
 * \b I2C_STS_REG_ACCESS_RDY - Register-access-ready (ARDY)
 * \b I2C_STS_RX_DATA_RDY - Receive-data-ready
 * \b I2C_STS_TX_DATA_RDY - Transmit-data-ready
 * \b I2C_STS_STOP_CONDITION - Stop condition detected
 * \b I2C_STS_BYTE_SENT - Byte transmit complete
 * \b I2C_STS_ADDR_ZERO - Address of all zeros detected
 * \b I2C_STS_ADDR_TARGET - Addressed as Target
 * \b I2C_STS_TX_EMPTY - Transmit shift register empty
 * \b I2C_STS_RX_FULL - Receive shift register full
 * \b I2C_STS_BUS_BUSY - Bus busy, wait for STOP or reset
 * \b I2C_STS_NACK_SENT - NACK was sent
 * \b I2C_STS_TARGET_DIR- Addressed as Target transmitter
 *
 */
static inline uint16_t
I2C_getStatus(uint32_t base)
{

    ASSERT(I2C_isBaseValid(base));

    return(HWREGH(base + I2C_O_STR));
}

/**
 * @brief  Clears I2C status flags.
 *
 * @param base is the base address of the I2C instance used.
 * @param stsFlags is a bit mask of the status flags to be cleared.
 *
 * This function clears the specified I2C status flags. The \e stsFlags
 * parameter is the logical OR of the following values:
 * - \b I2C_STS_ARB_LOST
 * - \b I2C_STS_NO_ACK
 * - \b I2C_STS_REG_ACCESS_RDY
 * - \b I2C_STS_RX_DATA_RDY
 * - \b I2C_STS_STOP_CONDITION
 * - \b I2C_STS_BYTE_SENT
 * - \b I2C_STS_NACK_SENT
 * - \b I2C_STS_TARGET_DIR
 *
 * \note Note that some of the status flags returned by I2C_getStatus() cannot
 * be cleared by this function. Some may only be cleared by hardware or a
 * reset of the I2C module.
 *
 * @return None.
 *
 */
static inline void
I2C_clearStatus(uint32_t base, uint16_t stsFlags)
{
    ASSERT(I2C_isBaseValid(base));

    HWREGH(base + I2C_O_STR) = stsFlags;
}

/**
 * @brief  Controls the state of the I2C module.
 *
 * @param base is the base address of the I2C instance used.
 * @param config is the command to be issued to the I2C  module.
 *
 * This function is used to control the state of the controller and target send and
 * receive operations. The \e config is a logical OR of the following options.
 *
 * One of the following four options:
 * - \b I2C_CONTROLLER_SEND_MODE - Controller-transmitter mode
 * - \b I2C_CONTROLLER_RECEIVE_MODE - Controller-receiver mode
 * - \b I2C_TARGET_SEND_MODE - Target-transmitter mode
 * - \b I2C_TARGET_RECEIVE_MODE - Target-receiver mode
 *
 *  Any of the following:
 * - \b I2C_REPEAT_MODE - Sends data until stop bit is set, ignores data count
 * - \b I2C_START_BYTE_MODE - Use start byte mode
 * - \b I2C_FREE_DATA_FORMAT - Use free data format, transfers have no address
 *
 * @return None.
 *
 */
static inline void
I2C_setConfig(uint32_t base, uint16_t config)
{
    ASSERT(I2C_isBaseValid(base));

    HWREGH(base + I2C_O_MDR) = (HWREGH(base + I2C_O_MDR) &
                                ~(I2C_MDR_CNT | I2C_MDR_TRX | I2C_MDR_RM |
                                  I2C_MDR_STB | I2C_MDR_FDF)) | config;
}

/**
 * @brief  Sets the data byte bit count the I2C module.
 *
 * @param base is the base address of the I2C instance used.
 * @param size is the number of bits per data byte.
 *
 * The \e size parameter is a value I2C_BITCOUNT_x where x is the number of
 * bits per data byte.  The default and maximum size is 8 bits.
 *
 * @return None.
 *
 */
static inline void
I2C_setBitCount(uint32_t base, I2C_BitCount size)
{
    ASSERT(I2C_isBaseValid(base));

    HWREGH(base + I2C_O_MDR) = (HWREGH(base + I2C_O_MDR) & ~I2C_MDR_BC_M) |
                               (uint16_t)size;
}

//*****************************************************************************
//
//! Issues an I2C START condition.
//!
//! @param base is the base address of the I2C instance used.
//!
//! This function causes the I2C module to generate a start condition. This
//! function is only valid when the I2C module specified by the \b base
//! parameter is a controller.
//!
//! @return None.
//
//*****************************************************************************

/**
 * @brief  Issues an I2C START condition.
 *
 * @param base is the base address of the I2C instance used.
 *
 * @return None.
 *
 */
static inline void
I2C_sendStartCondition(uint32_t base)
{

    ASSERT(I2C_isBaseValid(base));

    HWREGH(base + I2C_O_MDR) |= I2C_MDR_STT;
}

/**
 * @brief  Issues an I2C STOP condition.
 *
 * @param base is the base address of the I2C instance used.
 *
 * @return None.
 *
 */
static inline void
I2C_sendStopCondition(uint32_t base)
{

    ASSERT(I2C_isBaseValid(base));

    HWREGH(base + I2C_O_MDR) |= I2C_MDR_STP;
}


/**
 * @brief  Issues a no-acknowledge (NACK) bit.
 *
 * @param base is the base address of the I2C instance used.
 *
 * @return None.
 *
 */
static inline void
I2C_sendNACK(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));

    HWREGH(base + I2C_O_MDR) |= I2C_MDR_NACKMOD;
}

/**
 * @brief  Receives a byte that has been sent to the I2C.
 *
 * @param base is the base address of the I2C instance used.
 *
 * @return  Returns the byte received from by the I2C cast as an uint16_t.
 *
 */
static inline uint16_t
I2C_getData(uint32_t base)
{

    ASSERT(I2C_isBaseValid(base));

    return(HWREGH(base + I2C_O_DRR));
}

/**
 * @brief Transmits a byte from the I2C.
 *
 * @param base is the base address of the I2C instance used.
 * @param data is the data to be transmitted from the I2C Controller.
 *
 * @return  None.
 *
 */
static inline void
I2C_putData(uint32_t base, uint16_t data)
{
    ASSERT(I2C_isBaseValid(base));

    HWREGH(base + I2C_O_DXR) = data;
}

/**
 * @brief Get stop condition status.
 *
 * @param base is the base address of the I2C instance used.
 *
 * @return Returns \b true if the STP bit has been set by the device to
 * generate a stop condition when the internal data counter of the I2C module
 * has reached 0. Returns \b false when the STP bit is zero. This bit is
 * automatically cleared after the stop condition has been generated.
 *
 */
static inline boolean
I2C_getStopConditionStatus(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));

    return((HWREGH(base + I2C_O_MDR) & I2C_MDR_STP) != 0U);
}

/**
 * @brief Set number of bytes to be to transfer or receive when repeat mode is off.
 *
 * @param base is the base address of the I2C instance used.
 * @param count is the value to be put in the I2C data count register.
 *
 * @return None.
 *
 */
static inline void
I2C_setDataCount(uint32_t base, uint16_t count)
{
    ASSERT(I2C_isBaseValid(base));

    HWREGH(base + I2C_O_CNT) = count;
}

/**
 * @brief Sets the addressing mode to either 7-bit or 10-bit.
 *
 * @param base is the base address of the I2C instance used.
 * @param mode is the address mode, 7-bit or 10-bit.
 *
 * @return None.
 *
 */
static inline void
I2C_setAddressMode(uint32_t base, I2C_AddressMode mode)
{
    ASSERT(I2C_isBaseValid(base));

    HWREGH(base + I2C_O_MDR) = (HWREGH(base + I2C_O_MDR) & ~I2C_MDR_XA) |
                               (uint16_t)mode;
}

/**
 * @brief Sets I2C emulation mode.
 *
 * @param base is the base address of the I2C instance used.
 * @param mode is the emulation mode.
 *
 * @return None.
 *
 */
static inline void
I2C_setEmulationMode(uint32_t base, I2C_EmulationMode mode)
{
    ASSERT(I2C_isBaseValid(base));

    HWREGH(base + I2C_O_MDR) = (HWREGH(base + I2C_O_MDR) & ~I2C_MDR_FREE) |
                               (uint16_t)mode;
}

/**
 * @brief Enables I2C loopback mode.
 *
 * @param base is the base address of the I2C instance used.
 *
 * @return None.
 *
 */
static inline void
I2C_enableLoopback(uint32_t base)
{

    ASSERT(I2C_isBaseValid(base));

    HWREGH(base + I2C_O_MDR) |= I2C_MDR_DLB;
}

/**
 * @brief Disables I2C loopback mode.
 *
 * @param base is the base address of the I2C instance used.
 *
 * @return None.
 *
 */
static inline void
I2C_disableLoopback(uint32_t base)
{

    ASSERT(I2C_isBaseValid(base));

    HWREGH(base + I2C_O_MDR) &= ~I2C_MDR_DLB;
}

/**
 * @brief Returns the current I2C interrupt source.
 *
 * @param base is the base address of the I2C instance used.
 *
 * @return This function returns the event that generated an I2C basic (non-FIFO)
 * interrupt. The possible sources are the following:
 * - \b I2C_INTSRC_NONE
 * - \b I2C_INTSRC_ARB_LOST
 * - \b I2C_INTSRC_NO_ACK
 * - \b I2C_INTSRC_REG_ACCESS_RDY
 * - \b I2C_INTSRC_RX_DATA_RDY
 * - \b I2C_INTSRC_TX_DATA_RDY
 * - \b I2C_INTSRC_STOP_CONDITION
 * - \b I2C_INTSRC_ADDR_TARGET
 *
 */
static inline I2C_InterruptSource
I2C_getInterruptSource(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));

    return((I2C_InterruptSource)(HWREGH(base + I2C_O_ISRC) &
                                 I2C_ISRC_INTCODE_M));
}

/**
 * @brief Sets the compatibility mode to support I2C Extended Mode.
 *
 * @param base is the base address of the I2C instance used.
 * @param mode is the compatibility modes.
 *
 * @return None.
 *
 */
static inline void
I2C_setExtendedMode(uint32_t base, I2C_ExtendedMode mode)
{

    ASSERT(I2C_isBaseValid(base));

    HWREGH(base + I2C_O_EMDR) =
   (HWREGH(base + I2C_O_EMDR) & ~(I2C_EMDR_BC | I2C_EMDR_FCM)) |
   (uint16_t)mode;
}

/**
 * @brief Initializes the I2C Controller.
 *
 * @param base is the base address of the I2C instance used.
 * @param sysclkHz is the rate of the clock supplied to the I2C module
 * (SYSCLK) in Hz.
 * @param bitRate is the rate of the controller clock signal, SCL.
 * @param dutyCycle is duty cycle of the SCL signal.
 *
 * This function initializes operation of the I2C Controller by configuring the
 * bus speed for the controller. Note that the I2C module \b must be put into
 * reset before calling this function. You can do this with the function
 * I2C_disableModule().
 *
 * A programmable prescaler in the I2C module divides down the input clock
 * (rate specified by \e sysclkHz) to produce the module clock (calculated to
 * be around 10 MHz in this function). That clock is then divided down further
 * to configure the SCL signal to run at the rate specified by \e bitRate. The
 * \e dutyCycle parameter determines the percentage of time high and time low
 * on the clock signal. The valid values are \b I2C_DUTYCYCLE_33 for 33% and
 * \b I2C_DUTYCYCLE_50 for 50%.
 *
 * The peripheral clock is the system clock.  This value is returned by
 * SysCtl_getClock(), or it can be explicitly hard coded if it is
 * constant and known (to save the code/execution overhead of a call to
 * SysCtl_getClock()).
 *
 * @return None.
 *
 */
extern void
I2C_initController(uint32_t base, uint32_t sysclkHz, uint32_t bitRate,
               I2C_DutyCycle dutyCycle);

/**
 * @brief Enables I2C interrupt sources.
 *
 * @param base is the base address of the I2C instance used.
 * @param intFlags is the bit mask of the interrupt sources to be enabled.
 *
 * This function enables the indicated I2C Controller interrupt sources.  Only the
 * sources that are enabled can be reflected to the processor interrupt.
 * Disabled sources have no effect on the processor.
 *
 * The \e intFlags parameter is the logical OR of any of the following:
 *
 * - \b I2C_INT_ARB_LOST - Arbitration-lost interrupt
 * - \b I2C_INT_NO_ACK - No-acknowledgment (NACK) interrupt
 * - \b I2C_INT_REG_ACCESS_RDY - Register-access-ready interrupt
 * - \b I2C_INT_RX_DATA_RDY - Receive-data-ready interrupt
 * - \b I2C_INT_TX_DATA_RDY - Transmit-data-ready interrupt
 * - \b I2C_INT_STOP_CONDITION - Stop condition detected
 * - \b I2C_INT_ADDR_TARGET - Addressed as target interrupt
 * - \b I2C_INT_RXFF - RX FIFO level interrupt
 * - \b I2C_INT_TXFF - TX FIFO level interrupt
 *
 * \note \b I2C_INT_RXFF and \b I2C_INT_TXFF are associated with the I2C FIFO
 * interrupt vector. All others are associated with the I2C basic interrupt.
 *
 * @return None.
 *
 */
extern void
I2C_enableInterrupt(uint32_t base, uint32_t intFlags);

/**
 * @brief disables the interrupt source of the I2C controller
 *
 * @param base is the base address of the I2C instance used.
 * @param intFlags interrupt flag bit mask
 *
 * @return  None
 */
extern void
I2C_disableInterrupt(uint32_t base, uint32_t intFlags);

/**
 * @brief Obtain the interrupt status of the I2C instance.
 *
 * @param base is the base address of the I2C instance used.
 *
 * @return The combined interrupt status bit
 */
extern uint32_t
I2C_getInterruptStatus(uint32_t base);

/**
 * @brief clear Interrupt Status.
 *
 * @param base is the base address of the I2C instance used.
 * @param intFlags interrupt flag bit mask
 *
 * @return  None
 */
extern void
I2C_clearInterruptStatus(uint32_t base, uint32_t intFlags);

/**
 * @brief Configure the working frequency of the I2C module.
 *
 * @param base is the base address of the I2C instance used.
 * @param sysclkHz is System clock frequency
 *
 * @return  None
 */
extern void
I2C_configureModuleFrequency(uint32_t base, uint32_t sysclkHz);


#endif

#ifdef __cplusplus
}
#endif

#endif /* DRIVERLIB_IIC_V30_H_ */
