/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    Interrupt.c
*   @brief   
*   @details
*
*/
/*
 * commit history
 * 20240308, LYF, chane INT vector number name for chip 2.0.
 * 2024/03/13, Zhao Lei, add 3.0 irq trigger table, change eQEP to be edge trigger IRQ.
 * 2024/04/19, Zhao Lei, change 2.0 MU to be edge trigger IRQ.
 * 2024/05/11, Zhaolei, INT_MUx/INT_MUx_GEN -> INT_MUx_GENERAL.
 * 2024/05/13, Zhaolei, update 3.0 interrupt.
 * 2024/05/14, Zhaolei, 2.2 INT_MUx_GEN -> INT_MUx_GENERAL.
 * 2024/05/18, Zhaolei, change 3.0 jtag irq trigger type to be ECLIC_POSTIVE_EDGE_TRIGGER
 * 2024/05/20, Zhaolei, change 3.0 CLB/ENC irq trigger type to be ECLIC_POSTIVE_EDGE_TRIGGER
 * 2024/07/26, WenGuangYong, remove arm on 2.2CS
 * 2024/12/04, Zhaolei, add macro "USING_VECTOR_INTERRUPT" to select default interrupt mode
 * 2024/12/23, Zhaolei, add Interrupt_enableDualcoreShare(), clear comment
 */



#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "gs32_version.h"
#include "interrupt.h"
#include "inc/hw_ints.h"
#include <nmsis_core.h>
#include "debug.h"

/* None */

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
typedef struct __DSP_IRQ_ATTR {
	uint8_t shv;
	ECLIC_TRIGGER_Type trig_mode;
	uint8_t lvl;
	uint8_t priority;
}DSP_IRQ_ATTR;

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */
const ECLIC_TRIGGER_Type dspIRQTriggerType[] = {
	[Reserved0_IRQn   ]   = ECLIC_LEVEL_TRIGGER,              /* Internal reserved         */
	[Reserved1_IRQn   ]   = ECLIC_LEVEL_TRIGGER,              /* Internal reserved         */
	[Reserved2_IRQn   ]   = ECLIC_LEVEL_TRIGGER,              /* Internal reserved         */
	[SysTimerSW_IRQn  ]   = ECLIC_LEVEL_TRIGGER,              /* System Timer SW interrupt */
	[Reserved3_IRQn   ]   = ECLIC_LEVEL_TRIGGER,              /* Internal reserved         */
	[Reserved4_IRQn   ]   = ECLIC_LEVEL_TRIGGER,              /* Internal reserved         */
	[Reserved5_IRQn   ]   = ECLIC_LEVEL_TRIGGER,              /* Internal reserved         */
	[SysTimer_IRQn    ]   = ECLIC_POSTIVE_EDGE_TRIGGER,       /* System Timer Interrupt    */
	[Reserved6_IRQn   ]   = ECLIC_LEVEL_TRIGGER,              /* Internal reserved         */
	[Reserved7_IRQn   ]   = ECLIC_LEVEL_TRIGGER,              /* Internal reserved         */
	[Reserved8_IRQn   ]   = ECLIC_LEVEL_TRIGGER,              /* Internal reserved         */
	[Reserved9_IRQn   ]   = ECLIC_LEVEL_TRIGGER,              /* Internal reserved         */
	[Reserved10_IRQn  ]   = ECLIC_LEVEL_TRIGGER,              /* Internal reserved         */
	[Reserved11_IRQn  ]   = ECLIC_LEVEL_TRIGGER,              /* Internal reserved         */
	[Reserved12_IRQn  ]   = ECLIC_LEVEL_TRIGGER,              /* Internal reserved         */
	[Reserved13_IRQn  ]   = ECLIC_LEVEL_TRIGGER,              /* Internal reserved         */
	[InterCore_IRQn   ]   = ECLIC_LEVEL_TRIGGER,              /* CIDU Inter Core Interrupt */
	[Reserved15_IRQn  ]   = ECLIC_LEVEL_TRIGGER,              /* Internal reserved         */
	[Reserved16_IRQn  ]   = ECLIC_LEVEL_TRIGGER,              /* Internal reserved         */

#if IS_GS32F00xx(0x12)
	[INT_SLV_RST_FINISH   ] = ECLIC_LEVEL_TRIGGER,
	[INT_DSP_M_FAIL       ] = ECLIC_LEVEL_TRIGGER,
	[INT_STC              ] = ECLIC_LEVEL_TRIGGER,     //memory self-test completed
	[INT_EPG              ] = ECLIC_LEVEL_TRIGGER,
	[INT_RTOSINT1         ] = ECLIC_LEVEL_TRIGGER,
	[INT_FPU_IO           ] = ECLIC_LEVEL_TRIGGER,
	[INT_FPU_DZ           ] = ECLIC_LEVEL_TRIGGER,
	[INT_FPU_ID           ] = ECLIC_LEVEL_TRIGGER,
	[INT_FPU_IX           ] = ECLIC_LEVEL_TRIGGER,
	[INT_FPU_UNDERFLOW    ] = ECLIC_LEVEL_TRIGGER,
	[INT_FPU_OVERFLOW     ] = ECLIC_LEVEL_TRIGGER,
	[INT_AES              ] = ECLIC_LEVEL_TRIGGER,
	[INT_WINDOW_FFT       ] = ECLIC_LEVEL_TRIGGER,
	[INT_FFT              ] = ECLIC_LEVEL_TRIGGER,
	[INT_CRC              ] = ECLIC_LEVEL_TRIGGER,
	[INT_DBUS             ] = ECLIC_LEVEL_TRIGGER,
	[INT_IBUS             ] = ECLIC_LEVEL_TRIGGER,
	[INT_EFC              ] = ECLIC_LEVEL_TRIGGER,
	[INT_EFC_DONE         ] = ECLIC_LEVEL_TRIGGER,
	[INT_XINT16           ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_XINT15           ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_XINT14           ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_XINT13           ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_XINT12           ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_XINT11           ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_XINT10           ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_XINT9            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_XINT8            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_XINT7            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_XINT6            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_XINT5            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_XINT4            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_XINT3            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_ADCC4            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_ADCC3            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_ADCC2            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_ADCC_EVT         ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_ADCB4            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_ADCB3            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_ADCB2            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_ADCB_EVT         ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_ADCA4            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_ADCA3            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_ADCA2            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_ADCA_EVT         ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_PMBUSA_ALERT     ] = ECLIC_LEVEL_TRIGGER,
	[INT_PMBUSA           ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_GPIO             ] = ECLIC_LEVEL_TRIGGER,
	[INT_QSPI             ] = ECLIC_LEVEL_TRIGGER,
	[INT_CANB             ] = ECLIC_LEVEL_TRIGGER,
	[INT_CANA             ] = ECLIC_LEVEL_TRIGGER,
	[INT_SCIB             ] = ECLIC_LEVEL_TRIGGER,
	[INT_SCIA             ] = ECLIC_LEVEL_TRIGGER,
	[INT_LINB             ] = ECLIC_LEVEL_TRIGGER,
	[INT_LINA             ] = ECLIC_LEVEL_TRIGGER,
	[INT_USER12           ] = ECLIC_LEVEL_TRIGGER,
	[INT_USER11           ] = ECLIC_LEVEL_TRIGGER,
	[INT_USER10           ] = ECLIC_LEVEL_TRIGGER,
	[INT_USER9            ] = ECLIC_LEVEL_TRIGGER,
	[INT_USER8            ] = ECLIC_LEVEL_TRIGGER,
	[INT_USER7            ] = ECLIC_LEVEL_TRIGGER,
	[INT_USER6            ] = ECLIC_LEVEL_TRIGGER,
	[INT_USER5            ] = ECLIC_LEVEL_TRIGGER,
	[INT_USER4            ] = ECLIC_LEVEL_TRIGGER,
	[INT_USER3            ] = ECLIC_LEVEL_TRIGGER,
	[INT_USER2            ] = ECLIC_LEVEL_TRIGGER,
	[INT_USER1            ] = ECLIC_LEVEL_TRIGGER,
	[INT_I2C1             ] = ECLIC_LEVEL_TRIGGER,
	[INT_I2C0             ] = ECLIC_LEVEL_TRIGGER,
	[INT_DMA1_CH8         ] = ECLIC_LEVEL_TRIGGER,
	[INT_DMA1_CH7         ] = ECLIC_LEVEL_TRIGGER,
	[INT_DMA1_CH6         ] = ECLIC_LEVEL_TRIGGER,
	[INT_DMA1_CH5         ] = ECLIC_LEVEL_TRIGGER,
	[INT_DMA1_CH4         ] = ECLIC_LEVEL_TRIGGER,
	[INT_DMA1_CH3         ] = ECLIC_LEVEL_TRIGGER,
	[INT_DMA1_CH2         ] = ECLIC_LEVEL_TRIGGER,
	[INT_DMA1_CH1         ] = ECLIC_LEVEL_TRIGGER,
	[INT_DMA2_CH8         ] = ECLIC_LEVEL_TRIGGER,
	[INT_DMA2_CH7         ] = ECLIC_LEVEL_TRIGGER,
	[INT_DMA2_CH6         ] = ECLIC_LEVEL_TRIGGER,
	[INT_DMA2_CH5         ] = ECLIC_LEVEL_TRIGGER,
	[INT_DMA2_CH4         ] = ECLIC_LEVEL_TRIGGER,
	[INT_DMA2_CH3         ] = ECLIC_LEVEL_TRIGGER,
	[INT_DMA2_CH2         ] = ECLIC_LEVEL_TRIGGER,
	[INT_DMA2_CH1         ] = ECLIC_LEVEL_TRIGGER,
	[INT_SDFM2DR4         ] = ECLIC_LEVEL_TRIGGER,
	[INT_SDFM2DR3         ] = ECLIC_LEVEL_TRIGGER,
	[INT_SDFM2DR2         ] = ECLIC_LEVEL_TRIGGER,
	[INT_SDFM2DR1         ] = ECLIC_LEVEL_TRIGGER,
	[INT_DMA2_MUX         ] = ECLIC_LEVEL_TRIGGER,
	[INT_DMA1_MUX         ] = ECLIC_LEVEL_TRIGGER,
	[INT_TIMER3           ] = ECLIC_LEVEL_TRIGGER,
	[INT_TIMER2           ] = ECLIC_LEVEL_TRIGGER,
	[INT_TIMER1           ] = ECLIC_LEVEL_TRIGGER,
	[INT_SPIA             ] = ECLIC_LEVEL_TRIGGER,
	[INT_SPIB             ] = ECLIC_LEVEL_TRIGGER,
	[INT_SDFM1DR4         ] = ECLIC_LEVEL_TRIGGER,
	[INT_SDFM1DR3         ] = ECLIC_LEVEL_TRIGGER,
	[INT_SDFM1DR2         ] = ECLIC_LEVEL_TRIGGER,
	[INT_SDFM1DR1         ] = ECLIC_LEVEL_TRIGGER,
	[INT_SDFM2            ] = ECLIC_LEVEL_TRIGGER,
	[INT_SDFM1            ] = ECLIC_LEVEL_TRIGGER,
	[INT_CLB4             ] = ECLIC_LEVEL_TRIGGER,
	[INT_CLB3             ] = ECLIC_LEVEL_TRIGGER,
	[INT_CLB2             ] = ECLIC_LEVEL_TRIGGER,
	[INT_CLB1             ] = ECLIC_LEVEL_TRIGGER,
	[INT_EQEP2            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_EQEP1            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_ECAP7_2          ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_ECAP6_2          ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_ECAP3_2          ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_ECAP7            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_ECAP6            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_ECAP5            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_ECAP4            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_ECAP3            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_ECAP2            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_ECAP1            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_EPWM12           ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_EPWM11           ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_EPWM10           ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_EPWM9            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_EPWM8            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_EPWM7            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_EPWM6            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_EPWM5            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_EPWM4            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_EPWM3            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_EPWM2            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_EPWM1            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_EPWM12_TZ        ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_EPWM11_TZ        ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_EPWM10_TZ        ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_EPWM9_TZ         ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_EPWM8_TZ         ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_EPWM7_TZ         ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_EPWM6_TZ         ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_EPWM5_TZ         ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_EPWM4_TZ         ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_EPWM3_TZ         ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_EPWM2_TZ         ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_EPWM1_TZ         ] = ECLIC_POSTIVE_EDGE_TRIGGER,

	[INT_DSP_WWDG         ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_SYS_ERR          ] = ECLIC_LEVEL_TRIGGER,
	[INT_DSP_WDG          ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_TIMER0           ] = ECLIC_LEVEL_TRIGGER,
	[INT_WAKE             ] = ECLIC_LEVEL_TRIGGER,
	[INT_XINT2            ] = ECLIC_LEVEL_TRIGGER,
	[INT_XINT1            ] = ECLIC_LEVEL_TRIGGER,
	[INT_ADCC1            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_ADCB1            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
	[INT_ADCA1            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
#elif IS_GS32F00xx(0x30)
    [INT_JTAG_AUTH        ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_JTAG_INFO        ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_EPG              ] = ECLIC_LEVEL_TRIGGER,
    [INT_RTOSINT1         ] = ECLIC_LEVEL_TRIGGER,
    [INT_AES              ] = ECLIC_LEVEL_TRIGGER,
    [INT_CRC              ] = ECLIC_LEVEL_TRIGGER,
    [INT_EFC_DONE         ] = ECLIC_LEVEL_TRIGGER,
    [INT_EFC_SEC          ] = ECLIC_LEVEL_TRIGGER,
    [INT_XINT16           ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_XINT15           ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_XINT14           ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_XINT13           ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_XINT12           ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_XINT11           ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_XINT10           ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_XINT9            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_XINT8            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_XINT7            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_XINT6            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_XINT5            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_XINT4            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_XINT3            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_ADCC4            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_ADCC3            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_ADCC2            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_ADCC_EVT         ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_ADCA4            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_ADCA3            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_ADCA2            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_ADCA_EVT         ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_PMBUSA_ALERT     ] = ECLIC_LEVEL_TRIGGER,
    [INT_PMBUSA           ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_I2CB_FIFO        ] = ECLIC_LEVEL_TRIGGER,
    [INT_I2CA_FIFO        ] = ECLIC_LEVEL_TRIGGER,
	[INT_GPIO             ] = ECLIC_LEVEL_TRIGGER,
    [INT_SCIC_RX          ] = ECLIC_LEVEL_TRIGGER,
    [INT_SCIC_TX          ] = ECLIC_LEVEL_TRIGGER,
    [INT_SCIB_RX          ] = ECLIC_LEVEL_TRIGGER,
    [INT_SCIB_TX          ] = ECLIC_LEVEL_TRIGGER,
    [INT_SCIA_RX          ] = ECLIC_LEVEL_TRIGGER,
    [INT_SCIA_TX          ] = ECLIC_LEVEL_TRIGGER,
    [INT_LINB             ] = ECLIC_LEVEL_TRIGGER,
    [INT_LINA             ] = ECLIC_LEVEL_TRIGGER,
    [INT_USER12           ] = ECLIC_LEVEL_TRIGGER,
    [INT_USER11           ] = ECLIC_LEVEL_TRIGGER,
    [INT_USER10           ] = ECLIC_LEVEL_TRIGGER,
    [INT_USER9            ] = ECLIC_LEVEL_TRIGGER,
    [INT_USER8            ] = ECLIC_LEVEL_TRIGGER,
    [INT_USER7            ] = ECLIC_LEVEL_TRIGGER,
    [INT_USER6            ] = ECLIC_LEVEL_TRIGGER,
    [INT_USER5            ] = ECLIC_LEVEL_TRIGGER,
    [INT_USER4            ] = ECLIC_LEVEL_TRIGGER,
    [INT_USER3            ] = ECLIC_LEVEL_TRIGGER,
    [INT_USER2            ] = ECLIC_LEVEL_TRIGGER,
    [INT_USER1            ] = ECLIC_LEVEL_TRIGGER,
    [INT_I2CB             ] = ECLIC_LEVEL_TRIGGER,
    [INT_I2CA             ] = ECLIC_LEVEL_TRIGGER,
    [INT_DMA1_CH6         ] = ECLIC_LEVEL_TRIGGER,
    [INT_DMA1_CH5         ] = ECLIC_LEVEL_TRIGGER,
    [INT_DMA1_CH4         ] = ECLIC_LEVEL_TRIGGER,
    [INT_DMA1_CH3         ] = ECLIC_LEVEL_TRIGGER,
    [INT_DMA1_CH2         ] = ECLIC_LEVEL_TRIGGER,
    [INT_DMA1_CH1         ] = ECLIC_LEVEL_TRIGGER,
    [INT_SDFM2DR4         ] = ECLIC_LEVEL_TRIGGER,
    [INT_SDFM2DR3         ] = ECLIC_LEVEL_TRIGGER,
    [INT_SDFM2DR2         ] = ECLIC_LEVEL_TRIGGER,
    [INT_SDFM2DR1         ] = ECLIC_LEVEL_TRIGGER,
    [INT_DMAMUX1_OVERFLOW ] = ECLIC_LEVEL_TRIGGER,
    [INT_TIMER1           ] = ECLIC_LEVEL_TRIGGER,
    [INT_SPIB_TX          ] = ECLIC_LEVEL_TRIGGER,
    [INT_SPIB_RX          ] = ECLIC_LEVEL_TRIGGER,
    [INT_SPIA_TX          ] = ECLIC_LEVEL_TRIGGER,
    [INT_SPIA_RX          ] = ECLIC_LEVEL_TRIGGER,
    [INT_SDFM1DR4         ] = ECLIC_LEVEL_TRIGGER,
    [INT_SDFM1DR3         ] = ECLIC_LEVEL_TRIGGER,
    [INT_SDFM1DR2         ] = ECLIC_LEVEL_TRIGGER,
    [INT_SDFM1DR1         ] = ECLIC_LEVEL_TRIGGER,
    [INT_SDFM2            ] = ECLIC_LEVEL_TRIGGER,
    [INT_SDFM1            ] = ECLIC_LEVEL_TRIGGER,
    [INT_ENC2             ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_ENC1             ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_CLB2             ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_CLB1             ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_EQEP2            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_EQEP1            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_ECAP3_2          ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_EQEP3            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_ECAP3            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_ECAP2            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_ECAP1            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_CANB1            ] = ECLIC_LEVEL_TRIGGER,
    [INT_CANB0            ] = ECLIC_LEVEL_TRIGGER,
    [INT_CANA1            ] = ECLIC_LEVEL_TRIGGER,
    [INT_CANA0            ] = ECLIC_LEVEL_TRIGGER,
    [INT_EPWM8            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_EPWM7            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_EPWM6            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_EPWM5            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_EPWM4            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_EPWM3            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_EPWM2            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_EPWM1            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_EPWM8_TZ         ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_EPWM7_TZ         ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_EPWM6_TZ         ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_EPWM5_TZ         ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_EPWM4_TZ         ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_EPWM3_TZ         ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_EPWM2_TZ         ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_EPWM1_TZ         ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_DSP_WWDG         ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_SYS_ERR          ] = ECLIC_LEVEL_TRIGGER,
    [INT_DSP_WDG          ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_TIMER0           ] = ECLIC_LEVEL_TRIGGER,
    [INT_WAKE             ] = ECLIC_LEVEL_TRIGGER,
    [INT_XINT2            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_XINT1            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_ADCC1            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
    [INT_ADCA1            ] = ECLIC_POSTIVE_EDGE_TRIGGER,
#elif IS_GS32F3xx(0x22)
	[INT_XINT16      ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //XINT16
	[INT_XINT15      ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //XINT15
	[INT_XINT14      ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //XINT14
	[INT_XINT13      ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //XINT13
	[INT_XINT12      ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //XINT12
	[INT_XINT11      ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //XINT11
	[INT_XINT10      ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //XINT10
	[INT_XINT9       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //XINT9
	[INT_XINT8       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //XINT8
	[INT_XINT7       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //XINT7
	[INT_XINT6       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //XINT6
	[INT_DMA2_CH8    ]    = ECLIC_LEVEL_TRIGGER,      //CPU2_DMA_CH8
	[INT_DMA2_CH7    ]    = ECLIC_LEVEL_TRIGGER,      //CPU2_DMA_CH7
	[INT_DMA2_CH6    ]    = ECLIC_LEVEL_TRIGGER,      //CPU2_DMA_CH6
	[INT_DMA2_CH5    ]    = ECLIC_LEVEL_TRIGGER,      //CPU2_DMA_CH5
	[INT_DMA2_CH4    ]    = ECLIC_LEVEL_TRIGGER,      //CPU2_DMA_CH4
	[INT_DMA2_CH3    ]    = ECLIC_LEVEL_TRIGGER,      //CPU2_DMA_CH3
	[INT_DMA2_CH2    ]    = ECLIC_LEVEL_TRIGGER,      //CPU2_DMA_CH2
	[INT_DMA2_CH1    ]    = ECLIC_LEVEL_TRIGGER,      //CPU2_DMA_CH1
	[INT_DCACHE      ]    = ECLIC_LEVEL_TRIGGER,      //dcache
	[INT_ICACHE      ]    = ECLIC_LEVEL_TRIGGER,      //icache
	[INT_EFC_DONE    ]    = ECLIC_LEVEL_TRIGGER,      //efc_done_intr
	[INT_EFC         ]    = ECLIC_LEVEL_TRIGGER,      //efc_intr
	[INT_CPU2_PC_ERR ]    = ECLIC_LEVEL_TRIGGER,      //CPU2 PC ERR
	[INT_CPU1_PC_ERR ]    = ECLIC_LEVEL_TRIGGER,      //CPU1 PC ERR
	[INT_DMA2_MUX    ]    = ECLIC_LEVEL_TRIGGER,      //CPU2 DMA MUX
	[INT_DMA1_MUX    ]    = ECLIC_LEVEL_TRIGGER,      //CPU1 DMA MUX
	[INT_EPG         ]    = ECLIC_LEVEL_TRIGGER,      //EPG
	[INT_CRC         ]    = ECLIC_LEVEL_TRIGGER,      //CRC
	[INT_FPIOC       ]    = ECLIC_LEVEL_TRIGGER,      //FPIOC
	[INT_FPDZC       ]    = ECLIC_LEVEL_TRIGGER,      //FPDZC
	[INT_FPIDC       ]    = ECLIC_LEVEL_TRIGGER,      //FPIDC
	[INT_FPIXC       ]    = ECLIC_LEVEL_TRIGGER,      //FPIXC
	[INT_FPUFC       ]    = ECLIC_LEVEL_TRIGGER,      //FPUFC
	[INT_FPOFC       ]    = ECLIC_LEVEL_TRIGGER,      //FPOFC
	[INT_XINT5       ]    = ECLIC_LEVEL_TRIGGER,      //XINT5 interrupt
	[INT_XINT4       ]    = ECLIC_LEVEL_TRIGGER,      //XINT4 interrupt
	[INT_XINT3       ]    = ECLIC_LEVEL_TRIGGER,      //XINT3 interrupt
	[INT_SDFM4_DR4   ]    = ECLIC_LEVEL_TRIGGER,      //SDFM4 DR4
	[INT_SDFM4_DR3   ]    = ECLIC_LEVEL_TRIGGER,      //SDFM4 DR3
	[INT_SDFM4_DR2   ]    = ECLIC_LEVEL_TRIGGER,      //SDFM4 DR2
	[INT_SDFM4_DR1   ]    = ECLIC_LEVEL_TRIGGER,      //SDFM4 DR1
	[INT_USER11_12   ]    = ECLIC_LEVEL_TRIGGER,
	[INT_USER9_10    ]    = ECLIC_LEVEL_TRIGGER,
	[INT_USER7_8     ]    = ECLIC_LEVEL_TRIGGER,
	[INT_USER5_6     ]    = ECLIC_LEVEL_TRIGGER,
	[INT_USER3_4     ]    = ECLIC_LEVEL_TRIGGER,
	[INT_USER1_2     ]    = ECLIC_LEVEL_TRIGGER,
	[INT_GPIO        ]    = ECLIC_LEVEL_TRIGGER,      //GPIO Combine Interrupt
	[INT_ADCD4       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //ADCD4 interrupt
	[INT_ADCD3       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //ADCD3 interrupt
	[INT_ADCD2       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //ADCD2 interrupt
	[INT_ADCD_EVT    ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //ADCD_EVT interrupt
	[INT_ADCC4       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //ADCC4 interrupt
	[INT_ADCC3       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //ADCC3 interrupt
	[INT_ADCC2       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //ADCC2 interrupt
	[INT_ADCC_EVT    ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //ADCC_EVT interrupt
	[INT_ADCB4       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //ADCB4 interrupt
	[INT_ADCB3       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //ADCB3 interrupt
	[INT_ADCB2       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //ADCB2 interrupt
	[INT_ADCB_EVT    ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //ADCB_EVT interrupt
	[INT_ADCA4       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //ADCA4 interrupt
	[INT_ADCA3       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //ADCA3 interrupt
	[INT_ADCA2       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //ADCA2 interrupt
	[INT_ADCA_EVT    ]    = ECLIC_POSTIVE_EDGE_TRIGGER,      //ADCA_EVT interrupt
	[INT_PMBUSA_ALERT]    = ECLIC_LEVEL_TRIGGER,      //PMBUSA alert
	[INT_PMBUSA      ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //PMBUSA interrupt
	[INT_CAND        ]    = ECLIC_LEVEL_TRIGGER,     //CAND interrupt
	[INT_CANC        ]    = ECLIC_LEVEL_TRIGGER,     //CANC interrupt
	[INT_EPWM18      ]    = ECLIC_LEVEL_TRIGGER,     //EPWM18 interrupt
	[INT_EPWM17      ]    = ECLIC_LEVEL_TRIGGER,     //EPWM17 interrupt
	[INT_CANB        ]    = ECLIC_LEVEL_TRIGGER,     //CANB interrupt
	[INT_CANA        ]    = ECLIC_LEVEL_TRIGGER,     //CANA interrupt
	[INT_SCIB        ]    = ECLIC_LEVEL_TRIGGER,     //SCI_B interrupt
	[INT_SCIA        ]    = ECLIC_LEVEL_TRIGGER,     //SCI_A interrupt
	[INT_CLB8        ]    = ECLIC_LEVEL_TRIGGER,     //CLB8 interrupt
	[INT_CLB7        ]    = ECLIC_LEVEL_TRIGGER,     //CLB7 interrupt
	[INT_CLB6        ]    = ECLIC_LEVEL_TRIGGER,     //CLB6 interrupt
	[INT_CLB5        ]    = ECLIC_LEVEL_TRIGGER,     //CLB5 interrupt
	[INT_SDFM4       ]    = ECLIC_LEVEL_TRIGGER,     //SDFM4 interrupt
	[INT_SDFM3       ]    = ECLIC_LEVEL_TRIGGER,     //SDFM3 intterupt
	[INT_EWPM18_TZ   ]    = ECLIC_LEVEL_TRIGGER,     //EWPM18_TZ
	[INT_EPWM17_TZ   ]    = ECLIC_LEVEL_TRIGGER,     //EPWM17_TZ
	[INT_SCID        ]    = ECLIC_LEVEL_TRIGGER,     //SCI_D interrupt
	[INT_SCIC        ]    = ECLIC_LEVEL_TRIGGER,     //SCI_C interrupt
	[INT_I2CB        ]    = ECLIC_LEVEL_TRIGGER,     //I2CB interrupt
	[INT_I2CA        ]    = ECLIC_LEVEL_TRIGGER,     //I2CA interrupt
	[INT_SDFM3DR4    ]    = ECLIC_LEVEL_TRIGGER,     //SDFM3DR4 interrupt
	[INT_SDFM3DR3    ]    = ECLIC_LEVEL_TRIGGER,     //SDFM3DR3 interrupt
	[INT_SDFM3DR2    ]    = ECLIC_LEVEL_TRIGGER,     //SDFM3DR2 interrupt
	[INT_SDFM3DR1    ]    = ECLIC_LEVEL_TRIGGER,     //SDFM3DR1 interrupt
	[INT_HRCAP2      ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //HRCAP2
	[INT_HRCAP1      ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //HRCAP1
	[INT_EQEP6       ]    = ECLIC_LEVEL_TRIGGER,     //EQEP6
	[INT_EQEP5       ]    = ECLIC_LEVEL_TRIGGER,     //EQEP5
	[INT_DMA1_CH8    ]    = ECLIC_LEVEL_TRIGGER,     //CPU1_CH8 interrupt
	[INT_DMA1_CH7    ]    = ECLIC_LEVEL_TRIGGER,     //CPU1_CH7 interrupt
	[INT_DMA1_CH6    ]    = ECLIC_LEVEL_TRIGGER,     //CPU1_CH6 interrupt
	[INT_DMA1_CH5    ]    = ECLIC_LEVEL_TRIGGER,     //CPU1_CH5 interrupt
	[INT_DMA1_CH4    ]    = ECLIC_LEVEL_TRIGGER,     //CPU1_CH4 interrupt
	[INT_DMA1_CH3    ]    = ECLIC_LEVEL_TRIGGER,     //CPU1_CH3 interrupt
	[INT_DMA1_CH2    ]    = ECLIC_LEVEL_TRIGGER,     //CPU1_CH2 interrupt
	[INT_DMA1_CH1    ]    = ECLIC_LEVEL_TRIGGER,     //CPU1_CH1 interrupt
	[INT_SDFM2DR4    ]    = ECLIC_LEVEL_TRIGGER,     //SDFM2DR4 interrupt
	[INT_SDFM2DR3    ]    = ECLIC_LEVEL_TRIGGER,     //SDFM2DR3 interrupt
	[INT_SDFM2DR2    ]    = ECLIC_LEVEL_TRIGGER,     //SDFM2DR2 interrupt
	[INT_SDFM2DR1    ]    = ECLIC_LEVEL_TRIGGER,     //SDFM2DR1 interrupt
	[INT_TIMER5      ]    = ECLIC_LEVEL_TRIGGER,     //TIMER5 interrupt
	[INT_TIMER4      ]    = ECLIC_LEVEL_TRIGGER,     //TIMER4 interrupt
	[INT_TIMER3      ]    = ECLIC_LEVEL_TRIGGER,     //TIMER3 interrupt
	[INT_TIMER2      ]    = ECLIC_LEVEL_TRIGGER,     //TIMER2 interrupt
	[INT_TIMER1      ]    = ECLIC_LEVEL_TRIGGER,     //TIMER1 interrupt
	[INT_SPID        ]    = ECLIC_LEVEL_TRIGGER,     //SPID interrupt
	[INT_SPIC        ]    = ECLIC_LEVEL_TRIGGER,     //SPIC interrupt
	[INT_SPIB        ]    = ECLIC_LEVEL_TRIGGER,     //SPIB interrupt
	[INT_SPIA        ]    = ECLIC_LEVEL_TRIGGER,     //SPIA interrupt
	[INT_SDFM1DR4    ]    = ECLIC_LEVEL_TRIGGER,     //SDFM1DR4 interrupt
	[INT_SDFM1DR3    ]    = ECLIC_LEVEL_TRIGGER,     //SDFM1DR3 interrupt
	[INT_SDFM1DR2    ]    = ECLIC_LEVEL_TRIGGER,     //SDFM1DR2 interrupt
	[INT_SDFM1DR1    ]    = ECLIC_LEVEL_TRIGGER,     //SDFM1DR1 interrupt
	[INT_SD2         ]    = ECLIC_LEVEL_TRIGGER,     //SD2 interrupt
	[INT_SD1         ]    = ECLIC_LEVEL_TRIGGER,     //SD1 interrupt
	[INT_CLB4        ]    = ECLIC_LEVEL_TRIGGER,     //CLB4 interrupt
	[INT_CLB3        ]    = ECLIC_LEVEL_TRIGGER,     //CLB3 interrupt
	[INT_CLB2        ]    = ECLIC_LEVEL_TRIGGER,     //CLB2 interrupt
	[INT_CLB1        ]    = ECLIC_LEVEL_TRIGGER,     //CLB1 interrupt
	[INT_EQEP4       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EQEP4 interrupt
	[INT_EQEP3       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EQEP3 interrupt
	[INT_EQEP2       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EQEP2 interrupt
	[INT_EQEP1       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EQEP1 interrupt
	[INT_EXMC        ]    = ECLIC_LEVEL_TRIGGER,     //EXMC
	[INT_SCIF    	 ]    = ECLIC_LEVEL_TRIGGER,     //SCIF
	[INT_SCIE   	 ]    = ECLIC_LEVEL_TRIGGER,     //SCIE
	[INT_I2CC	     ]    = ECLIC_LEVEL_TRIGGER,     //CM-I2C
	[INT_SPIE        ]    = ECLIC_LEVEL_TRIGGER,     //SPIE interrupt
	[INT_LINB        ]    = ECLIC_LEVEL_TRIGGER,     //LINB
	[INT_LINA        ]    = ECLIC_LEVEL_TRIGGER,     //LINA
	[INT_ECAP7       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //ECAP7 interrupt
	[INT_ECAP6       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //ECAP6 interrupt
	[INT_ECAP5       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //ECAP5 interrupt
	[INT_ECAP4       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //ECAP4 interrupt
	[INT_ECAP3       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //ECAP3 interrupt
	[INT_ECAP2       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //ECAP2 interrupt
	[INT_ECAP1       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //ECAP1 interrupt
	[INT_EPWM16      ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM16 interrupt
	[INT_EPWM15      ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM15 interrupt
	[INT_EPWM14      ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM14 interrupt
	[INT_EPWM13      ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM13 interrupt
	[INT_EPWM12      ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM12 interrupt
	[INT_EPWM11      ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM11 interrupt
	[INT_EPWM10      ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM10 interrupt
	[INT_EPWM9       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM9 interrupt
	[INT_EPWM8       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM8 interrupt
	[INT_EPWM7       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM7 interrupt
	[INT_EPWM6       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM6 interrupt
	[INT_EPWM5       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM5 interrupt
	[INT_EPWM4       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM4 interrupt
	[INT_EPWM3       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM3 interrupt
	[INT_EPWM2       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM2 interrupt
	[INT_EPWM1       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM1 interrupt
	[INT_EPWM16_TZ   ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM16_TZ interrupt
	[INT_EPWM15_TZ   ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM15_TZ interrupt
	[INT_EPWM14_TZ   ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM14_TZ interrupt
	[INT_EPWM13_TZ   ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM13_TZ interrupt
	[INT_EPWM12_TZ   ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM12_TZ interrupt
	[INT_EPWM11_TZ   ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM11_TZ interrupt
	[INT_EPWM10_TZ   ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM10_TZ interrupt
	[INT_EPWM9_TZ    ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM9_TZ interrupt
	[INT_EPWM8_TZ    ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM8_TZ interrupt
	[INT_EPWM7_TZ    ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM7_TZ interrupt
	[INT_EPWM6_TZ    ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM6_TZ interrupt
	[INT_EPWM5_TZ    ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM5_TZ interrupt
	[INT_EPWM4_TZ    ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM4_TZ interrupt
	[INT_EPWM3_TZ    ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM3_TZ interrupt
	[INT_EPWM2_TZ    ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM2_TZ interrupt
	[INT_EPWM1_TZ    ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //EPWM1_TZ interrupt
	[INT_SYS_ERR     ]    = ECLIC_LEVEL_TRIGGER,     //SYS_ERR interrupt
	[INT_CPU2_WDG    ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //cpu2 watchdog interrupt
	[INT_CPU1_WDG    ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //cpu1 watchdog interrupt
	[INT_TIMER0      ]    = ECLIC_LEVEL_TRIGGER,     //TIMER0 interrupt
	[INT_ADCD1       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //ADCD1 interrupt
	[INT_XINT2       ]    = ECLIC_LEVEL_TRIGGER,     //XINT2 interrupt
	[INT_XINT1       ]    = ECLIC_LEVEL_TRIGGER,     //XINT1 interrupt
	[INT_ADCC1       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //ADCC1 interrupt
	[INT_ADCB1       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //ADCB1 interrupt
	[INT_ADCA1       ]    = ECLIC_POSTIVE_EDGE_TRIGGER,     //ADCA1 interrupt
#endif
};

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

/**
  * @brief   Enable Interrupt
  * @details Enables a device specific interrupt in the ECLIC interrupt controller.
  * @param   [in]      IRQn  Device specific interrupt number.
  * @note    IRQn must not be negative.
 */
void Interrupt_enable(IRQn_Type IRQn)
{
    ECLIC_EnableIRQ(IRQn);
}


/**
  * @brief   Disable Interrupt
  * @details Disables a device specific interrupt in the NVIC interrupt controller.
  * @param   [in]      IRQn  Device specific interrupt number.
  * @note    IRQn must not be negative.
 */
void Interrupt_disable(IRQn_Type IRQn)
{
    ECLIC_DisableIRQ(IRQn);
}

/**
  * @brief  Set the priority of an interrupt.
  * @param  [in] IRQn: External interrupt number.
  *         This parameter can be an enumerator of IRQn_Type enumeration
  * @param  [in] PreemptPriority: The pre-emption priority for the IRQn channel.
  *         This parameter can be a value between 0 and 7
  *         A big priority value indicates a higher priority
  * @param  [in] SubPriority: the subpriority level for the IRQ channel.
  *         This parameter is unused
  *         A big priority value indicates a higher priority.
  * @retval None
  */
void Interrupt_SetPriority(IRQn_Type IRQn, uint32_t PreemptPriority, uint32_t SubPriority)
{
  ECLIC_SetLevelIRQ(IRQn, PreemptPriority); //need confirm it's meaning
  ECLIC_SetPriorityIRQ(IRQn, SubPriority);
}


/**
  * @brief   Set Priority Grouping
  * @details Sets the priority grouping field using the required unlock sequence.
  *          Only values from 0..7 are used.
  * @param   [in]      PriorityGroup  Priority grouping field.
  * @retval  None
 */
void Interrupt_SetPriorityGrouping(uint32_t PriorityGroup)
{
    //this is an empty function
}


/**
  * @brief  Initiate a system reset request to reset the MCU.
  * @retval None
  */
void CPU_SystemReset(void)
{
	//this is an empty function
}

/**
  * @brief   Get Priority Grouping
  * @details Reads the priority grouping field from the NVIC Interrupt Controller.
  * @return  0
 */
uint32_t Interrupt_GetPriorityGrouping(void)
{
    return 0;
}

/**
  * @brief   Register an interrupt
  * @details register an ISR for an IRQ number
  * @param   [in] IRQn: External interrupt number.
  *          This parameter can be an enumerator of IRQn_Type enumeration
  * @param   [in] handler: interrupt service function
  * @return  none
 */

void Interrupt_register(IRQn_Type IRQn, IrqHandlerType handler)
{
#if IS_GS32F3xx(0x22)
    if (IRQn >= SOC_EXTERNAL_MAP_TO_ECLIC_IRQn_OFFSET) {
        uint32_t hartId = __get_hart_id();
        CIDU_BroadcastExtInterrupt(IRQn_MAP_TO_EXT_ID(IRQn), CIDU_RECEIVE_INTERRUPT_EN(hartId));
        CIDU_ResetFirstClaimMode(IRQn_MAP_TO_EXT_ID(IRQn));
    }
#endif

	/* set interrupt vector mode */
#if USING_VECTOR_INTERRUPT != 0
	ECLIC_SetShvIRQ(IRQn, ECLIC_VECTOR_INTERRUPT);
#else
	ECLIC_SetShvIRQ(IRQn, ECLIC_NON_VECTOR_INTERRUPT);
#endif
	/* set interrupt trigger mode and polarity */
	ECLIC_SetTrigIRQ(IRQn, dspIRQTriggerType[IRQn]);

	/* set interrupt level */
	//ECLIC_SetLevelIRQ(IRQn, lvl);

	/* set interrupt priority */
	//ECLIC_SetPriorityIRQ(IRQn, priority);

	if (handler != NULL) {
		/* set interrupt handler entry to vector table */
		ECLIC_SetVector(IRQn, (rv_csr_t)handler);
	}
}

/**
  * @brief  Unregister an interrupt
  * @param  [in] IRQn: External interrupt number.
  *         This parameter can be an enumerator of IRQn_Type enumeration
  * @retval None
  */
void Interrupt_unregister(IRQn_Type IRQn)
{
	ECLIC_DisableIRQ(IRQn);
	ECLIC_SetVector(IRQn, (rv_csr_t)NULL);
}

/**
  * @brief  Initiate a system reset request to reset the MCU.
  * @param  [in] vec_addr: vector table address
  * @retval None
  */
void Interrupt_reload_vectable(uint32_t vec_addr)
{
    //this is an empty function
}

/**
  * @brief  Initiate a system reset request to reset the MCU.
  * @param  [in] vec_addr: vector table address
  * @retval None
  */
void Interrupt_SetVTOR(uint32_t vec_addr)
{
    //this is an empty function
}

/**
  * @brief   set interrupt mode
  * @details set vector/non-vector interrupt mode for a given IRQ nunber
  * @param   [in] IRQn: External interrupt number.
  *          This parameter can be an enumerator of IRQn_Type enumeration
  * @param   [in] mode: The interrupt mode,
  *          This parameter can be ECLIC_NON_VECTOR_INTERRUPT or ECLIC_VECTOR_INTERRUPT
  * @return  none
 */
void Interrupt_setMode(IRQn_Type IRQn, uint32_t mode)
{
    ECLIC_SetShvIRQ(IRQn, mode);
}

/**
  * @brief   set interrupt mode
  * @details set vector/non-vector interrupt mode for a given IRQ nunber
  * @param   [in] IRQn: External interrupt number.
  *          This parameter can be an enumerator of IRQn_Type enumeration
  * @return  none
 */
void Interrupt_enableDualcoreShare(IRQn_Type IRQn)
{
#if IS_GS32F3xx(0x22)
    if (IRQn >= SOC_EXTERNAL_MAP_TO_ECLIC_IRQn_OFFSET) {
        CIDU_BroadcastExtInterrupt(IRQn_MAP_TO_EXT_ID(IRQn), CIDU_RECEIVE_INTERRUPT_EN(1) | CIDU_RECEIVE_INTERRUPT_EN(0));
        CIDU_ResetFirstClaimMode(IRQn_MAP_TO_EXT_ID(IRQn));
    }
#endif
}

/**
  * @brief  Initiate interrupt module
  * @retval None
  */
void Interrupt_initModule(void)
{
    //this is an empty function
}

/**
  * @brief  Initiate vector table
  * @retval None
  */
void Interrupt_initVectorTable(void)
{
	//this is an empty function
}

#ifdef __cplusplus
}
#endif

