/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifndef DEVICE_DRIVERLIB_MCAN_H_
#define DEVICE_DRIVERLIB_MCAN_H_

#ifdef __cplusplus
extern "C"{
#endif

#include <stdbool.h>
#include <string.h>

#include "inc/hw_types.h"
#include "inc/hw_mcan.h"
#include "inc/hw_memmap.h"

#define MCAN_MRAM_ADDR(base)		((base) - 0x1200U)
#define MCAN_SSREG_ADDR(base)		((base) - 0x200U)
#define MCAN_ERRSRPT_ADDR(base)		((base) + 0x200U)

/**
 * @brief MCAN RX buffer data phase length and message size.
 *
 */
#define M_CAN_RX_MAX_DATA_LENGTH	64U
#define M_CAN_RX_MAX_MSG_LENGTH		(M_CAN_RX_MAX_DATA_LENGTH + 8U)

/**
 * @brief MCAN TX buffer data phase length and message size.
 *
 */
#define M_CAN_TX_MAX_DATA_LENGTH	64U
#define M_CAN_TX_MAX_MSG_LENGTH		(M_CAN_RX_MAX_DATA_LENGTH + 8U)

/**
 * @brief MCAN TX event message size.
 *
 */
#define M_CAN_TX_EVENT_FIFO_SIZE	8U

/**
 * @brief Number of MCAN instance.
 *
 */
#define M_CAN_NUMBER_OF_INSTANCES	2U

/**
 * @brief Base address array of MCAN instance.
 *
 */
#define MCAN_MRAM_BASE(x)		((x) - 0x1200U)

#define M_CAN_BASE_ADDR_ARRAY		{	\
		CANA_BASE,						\
		CANB_BASE,						\
}

/**
 * @brief Standard ID MASK.
 *
 */
#define M_CAN_STANDARD_ID_S		18U
#define M_CAN_STANDARD_ID_M		0x1ffc0000U
#define M_CAN_STANDARD_ID_R(x)	(((uint32_t)(x) & M_CAN_STANDARD_ID_M) >>		\
												  M_CAN_STANDARD_ID_S)
#define M_CAN_STANDARD_ID_W(x)	(((uint32_t)(x) & (M_CAN_STANDARD_ID_M >>		\
												   M_CAN_STANDARD_ID_S)) <<		\
												   M_CAN_STANDARD_ID_S)

/**
 * @brief Extended ID MASK.
 *
 */
#define M_CAN_EXTENDED_ID_S		0U
#define M_CAN_EXTENDED_ID_M		0x1fffffffU
#define M_CAN_EXTENDED_ID_R(x)	((uint32_t)(x) & M_CAN_EXTENDED_ID_M)
#define M_CAN_EXTENDED_ID_W(x)	((uint32_t)(x) & M_CAN_EXTENDED_ID_M)

static const uint32_t MCAN_BaseAddressArray[] = M_CAN_BASE_ADDR_ARRAY;
static const uint8_t MCAN_DataPhaseSizeArray[8] = {8, 12, 16, 20, 24, 32, 48, 64};
static const uint8_t MCAN_DataLengthArray[16] = {0, 1, 2, 3, 4, 5, 6, 7, 8, 12, 16, 20, 24, 32, 48, 64};
static uint32_t MCAN_TX_COMPLETED_STATUS_REG[M_CAN_NUMBER_OF_INSTANCES] = {0};

/**
 * @brief MCAN controller operation mode.
 *
 */
typedef enum {
	MCAN_OPERATION_NORMAL_MODE	= 0U,			/* MCAN normal mode. */
	MCAN_OPERATION_SW_INIT_MODE	= 1U,			/* MCAN initialization mode. */
} MCAN_OperationMode_t;

/**
 * @brief MCAN frame format in CANFD mode.
 *
 */
typedef enum {
	MCAN_FD_ISO_11898_1			= 0U,			/* CANFD mode use ISO11898.1
												frame format. */
	MCAN_FD_BOSCH_STANDARD		= 1U,			/* CANFD mode use BOSCH CANFD
												frame format. */
} MCAN_FDISOMode_t;

/**
 * @brief Wide Message Marker mode.
 *
 */
typedef enum {
	MCAN_WMM_8BIT_MODE			= 0U,			/* Wide message markers use 8 bit mode. */
	MCAN_WMM_16BIT_MODE			= 1U,			/* Wide message markers use 16 bit mode.*/
} MCAN_WideMessageMarker_t;

/**
 * @brief MCAN loopback mode.
 *
 */
typedef enum {
	MCAN_LPBK_MODE_INTERNAL		= 0U,			/* MCAN use internal loopback mode. */
	MCAN_LPBK_MODE_EXTERNAL		= 1U,			/* MCAN use external loopback mode. */
} MCAN_LpbkMode_t;

/**
 * @brief MCAN listen only mode.
 * For MON mode, It meets the ISO standard and only send recessive bits on the CAN BUS.
 * For ASM mode, It only receives valid data and send acknowledge bit.
 */
typedef enum {
	MCAN_LISTEN_ONLY_MODE_DISABLE	= 0U,		/* Disable MCAN listen only mode. */
	MCAN_LISTEN_ONLY_MON_MODE		= 1U,		/* MCAN use Bus Monitoring Mode. */
} MCAN_ListenOnlyMode_t;

/**
 * @brief Transmitter Delay Compensation in CANFD mode.
 *
 * @param tdco indicates that offset value defining the distance between the measured
 * delay from m_can_tx to m_can_rx and the secondary sample point. Valid values are 0
 * to 127 mtq.
 * @param tdcf indicates that defines the minimum value for the SSP position, dominant
 * edges on m_can_rx that would result in an earlier SSP position are ignored for
 * transmitter delay measurement. The feature is enabled when TDCF is configured to a
 * value greater than TDCO. Valid values are 0 to 127 mtq.
 */
typedef struct MCAN_TDCConfig {
	uint8_t tdco;
	uint8_t tdcf;
} MCAN_TDCConfig_t;

/**
 * @brief Data phase and normal phase bitrate in CANFD and CAN mode.
 *
 * @param nomRatePrescalar indicates that Nominal Bit Rate Prescaler.
 * (Valid values are 0 to 511).
 * @param nomTimeSeg1 indicates that Nominal Time segment before sample point.
 * (Valid values are 1 to 255).
 * @param nomTimeSeg2 indicates that Nominal Time segment after sample point.
 * (Valid values are 1 to 127).
 * @param nomSynchJumpWidth indicates that Nominal (Re)Synchronization Jump Width.
 * (Valid values are 0 to 127).
 * @param dataRatePrescalar indicates that Data Bit Rate Prescaler.
 * (Valid values are 0 to 31).
 * @param dataTimeSeg1 indicates that Data time segment before sample point.
 * (Valid values are 0 to 31).
 * @param dataTimeSeg2 indicates that Data time segment after sample point.
 * (Valid values are 1 to 15).
 * @param dataSynchJumpWidth indicates that Data (Re)Synchronization Jump Width.
 * (Valid values are 0 to 15).
 */
typedef struct
{
	uint16_t nomRatePrescalar;
	uint8_t nomTimeSeg1;
	uint8_t nomTimeSeg2;
	uint8_t nomSynchJumpWidth;
	uint8_t dataRatePrescalar;
	uint8_t dataTimeSeg1;
	uint8_t dataTimeSeg2;
	uint8_t dataSynchJumpWidth;
} MCAN_BitTimingParams_t;

/**
 * @brief Global Filter Configuration options.
 *
 */
typedef enum {
	MCAN_NON_MATCH_ACCEPT_IN_FIFO0	= 0U,	/* Accept Non-matching Frames To Rx FIFO0. */
	MCAN_NON_MATCH_ACCEPT_IN_FIFO1	= 1U,	/* Accept Non-matching Frames To Rx FIFO1. */
	MCAN_NON_MATCH_REJECT_ACCEPT	= 2U,	/* Reject Accept Non-matching Frames. */
} MCAN_NonMatching_t;

/**
 * @brief Global Filter Configuration parameters.
 *
 * @param anfs indicates whether accept Non-matching Frames Standard.
 * @param anfe indicates whether accept Non-matching Frames Extended.
 * @param rrfs indicates whether filter remote frames with 11-bit standard IDs.
 * (false : Filter remote frames with 11-bit standard IDs).
 * (true : Reject all remote frames with 11-bit standard IDs).
 * @param rrfe indicates whether filter remote frames with 29-bit extended IDs.
 * (false : Filter remote frames with 29-bit extended IDs).
 * (true : Reject all remote frames with 29-bit extended IDs).
 */
typedef struct MCAN_GlobalFilter {
	MCAN_NonMatching_t anfs;
	MCAN_NonMatching_t anfe;
	bool rrfs;
	bool rrfe;
} MCAN_GlobalFiltConfig_t;

/**
 * @brief Select timeout operation mode.
 *
 */
typedef enum {
	MCAN_TIMEOUT_SELECT_CONT			= 0U,	/* Continuous operation. */
	MCAN_TIMEOUT_SELECT_TX_EVENT_FIFO	= 1U,	/* Timeout controlled by Tx Event FIFO. */
	MCAN_TIMEOUT_SELECT_RX_FIFO0		= 2U,	/* Timeout controlled by Rx FIFO 0. */
	MCAN_TIMEOUT_SELECT_RX_FIFO1		= 3U,	/* Timeout controlled by Rx FIFO 1. */
} MCAN_TimeOutSelect_t;

/**
 * @brief MCAN timestamp clock source.
 *
 */
typedef enum {
	MCAN_INTERNAL_TIMESTAMP				= 0U,	/* MCAN timestamp use internal
												clock source. */
	MCAN_EXTERNAL_TIMESTAMP				= 1U,	/* MCAN timestamp use external
												clock source. */
} MCAN_TimestampClk_t;

/**
 * @brief Timestamp counter trigger event parameters.
 *
 */
typedef enum {
	MCAN_TSCNTVAL_ALWAYS0			= 0U,		/* Timestamp counter value
												always 0x0000. */
	MCAN_TSCNTVAL_INC_TCP			= 1U,		/* Timestamp counter value incremented
												according to TCP. */
	MCAN_EXTERNAL_TSCNTVAL			= 2U,		/* External timestamp counter
												value used. */
} MCAN_TimestampSource_t;

/**
 * @brief MCAN standard ID configuration parameters.
 *
 * @param lss List size of standard ID filter(The number of standard ID filter is 0-128).
 * @param flssa Base address of standard ID filter list.
 */
typedef struct MCAN_StdIDFilterConfig {
	uint8_t lss;
	uint16_t flssa;
} MCAN_StdIDFilterConfig_t;

/**
 * @brief MCAN extended ID filter configuration parameters.
 *
 * @param lse List size of extended ID filter(The number of extended ID filter is 0-64).
 * @param flesa Base address of extended ID filter list.
 * @param mask Mask of extended ID filter.
 * For acceptance filtering of extended frames the Extended ID AND Mask is ANDed
 * with the Message ID of a received frame. Intended for masking of 29-bit IDs in SAE
 * J1939. With the reset value of all bits set to one the mask is not active.
 */
typedef struct MCAN_ExtIDFilterConfig {
	uint8_t lse;
	uint16_t flesa;
	uint32_t mask;
} MCAN_ExtIDFilterConfig_t;

/**
 * @brief Extended ID filter element function configuration options.
 *
 * @param MCAN_EXTIDF_ELE_DISABLE Disable filter element.
 * @param MCAN_EXTIDF_ELE_STORE_IN_FO0_OF_MATCH_ID Store in Rx FIFO 0 if filter matches.
 * @param MCAN_EXTIDF_ELE_STORE_IN_FO1_OF_MATCH_ID Store in Rx FIFO 1 if filter matches.
 * @param MCAN_EXTIDF_ELE_REJECT_MSG_OF_MATCH_ID Reject ID if filter matches, not
 * intended to be used with Sync messages.
 * @param MCAN_EXTIDF_ELE_PRIO_NO_STORE_OF_MATCH_ID Set priority if filter matches,
 * not intended to be used with Sync messages, no storage.
 * @param MCAN_EXTIDF_ELE_PRIO_STORE_IN_FO0_OF_MATCH_ID Set priority and store in FIFO 0
 * if filter matches.
 * @param MCAN_EXTIDF_ELE_PRIO_STORE_IN_FO1_OF_MATCH_ID Set priority and store
 * in FIFO 1 if filter matches.
 * @param MCAN_EXTIDF_ELE_STORE_IN_RXB_OR_DMSG Store into Rx Buffer or as debug message,
 * configuration of EFT[1:0] ignored.
 */
typedef enum {
	MCAN_EXTIDF_ELE_DISABLE							= 0U,
	MCAN_EXTIDF_ELE_STORE_IN_FO0_OF_MATCH_ID		= 1U,
	MCAN_EXTIDF_ELE_STORE_IN_FO1_OF_MATCH_ID		= 2U,
	MCAN_EXTIDF_ELE_REJECT_MSG_OF_MATCH_ID			= 3U,
	MCAN_EXTIDF_ELE_PRIO_NO_STORE_OF_MATCH_ID		= 4U,
	MCAN_EXTIDF_ELE_PRIO_STORE_IN_FO0_OF_MATCH_ID	= 5U,
	MCAN_EXTIDF_ELE_PRIO_STORE_IN_FO1_OF_MATCH_ID	= 6U,
	MCAN_EXTIDF_ELE_STORE_IN_RXB_OR_DMSG			= 7U,
} MCAN_ExtIDElemntConfig_t;

/**
 * @brief Only evaluated when CCCR.UTSU = 1. When this bit is set and a matching message
 * is received, a pulse with the duration of one m_can_hclk period is generated at output
 * m_can_tsrx to signalthe reception of a Sync message to the Timestamping Unit (TSU)
 * connected to the M_CAN.
 *
 * @param MCAN_EXTIDF_SYNC_MSG_DISABLE Timestamping for the matching Sync
 * message disabled.
 * @param MCAN_EXTIDF_SYNC_MSG_ENABLE Timestamping for the matching Sync message enabled.
 */
typedef enum {
	MCAN_EXTIDF_SYNC_MSG_DISABLE					= 0U,
	MCAN_EXTIDF_SYNC_MSG_ENABLE						= 1U,
} MCAN_ExtIDFilterSyncMsg_t;

/**
 * @brief Extended ID filter type options.
 *
 * @param MCAN_EXTIDF_RANGE_FROM_EFID1_TO_EFID2 Range filter from EFID1 to EFID2
 * (EFID2 >= EFID1).
 * @param MCAN_EXTIDF_DUAL_ID_FILTER Dual ID filter for EFID1 or EFID2.
 * @param MCAN_EXTIDF_CLASSIC_ID_FILTER Classic filter: EFID1 = filter, EFID2 = mask(bit = 1 check).
 * @param MCAN_EXTIDF_RANGE_FROM_EFID1_TO_EFID2_NON_XIDAM Range filter from EFID1 to EFID2
 * (EFID2 >= EFID1), XIDAM mask not applied.
 */
typedef enum {
	MCAN_EXTIDF_RANGE_FROM_EFID1_TO_EFID2			= 0U,
	MCAN_EXTIDF_DUAL_ID_FILTER						= 1U,
	MCAN_EXTIDF_CLASSIC_ID_FILTER					= 2U,
	MCAN_EXTIDF_RANGE_FROM_EFID1_TO_EFID2_NON_XIDAM	= 3U,
} MCAN_ExtIDFileterType_t;

/**
 * @brief MCAN extended ID filter element format.
 *
 * @param efid1 Extended Filter ID 1.
 * @param efec Extended ID Filter Element Configuration.
 * (options:@MCAN_ExtIDElemntConfig_t).
 * @param efid2 Extended Filter ID 2.
 * @param esync Extended ID Sync Message. (options:@MCAN_ExtIDFilterSyncMsg_t).
 * @param eft Extended ID Filter Type. (options:@MCAN_ExtIDFileterType_t).
 */
typedef struct MCAN_ExtMsgIDFilterElement {
	uint32_t efid1	: 29;
	uint32_t efec	: 3;
	uint32_t efid2	: 29;
	uint32_t esync	: 1;
	uint32_t eft	: 2;
} MCAN_ExtMsgIDFilterElement_t;

/**
 * @brief Standard ID filter element function configuration options.
 *
 * @param MCAN_STDIDF_ELE_DISABLE Disable filter element.
 * @param MCAN_STDIDF_ELE_STORE_IN_FO0_OF_MATCH_ID Store in Rx FIFO 0 if filter matches.
 * @param MCAN_STDIDF_ELE_STORE_IN_FO1_OF_MATCH_ID Store in Rx FIFO 1 if filter matches.
 * @param MCAN_STDIDF_ELE_REJECT_MSG_OF_MATCH_ID Reject ID if filter matches, not
 * intended to be used with Sync messages.
 * @param MCAN_STDIDF_ELE_PRIO_NO_STORE_OF_MATCH_ID Set priority if filter matches,
 * not intended to be used with Sync messages, no storage.
 * @param MCAN_STDIDF_ELE_PRIO_STORE_IN_FO0_OF_MATCH_ID Set priority and store in FIFO 0
 * if filter matches.
 * @param MCAN_STDIDF_ELE_PRIO_STORE_IN_FO1_OF_MATCH_ID Set priority and store in FIFO 1
 * if filter matches.
 * @param MCAN_STDIDF_ELE_STORE_IN_RXB_OR_DMSG Store into Rx Buffer or as debug message,
 * configuration of SFT[1:0] ignored.
 */
typedef enum {
	MCAN_STDIDF_ELE_DISABLE							= 0U,
	MCAN_STDIDF_ELE_STORE_IN_FO0_OF_MATCH_ID		= 1U,
	MCAN_STDIDF_ELE_STORE_IN_FO1_OF_MATCH_ID		= 2U,
	MCAN_STDIDF_ELE_REJECT_MSG_OF_MATCH_ID			= 3U,
	MCAN_STDIDF_ELE_PRIO_NO_STORE_OF_MATCH_ID		= 4U,
	MCAN_STDIDF_ELE_PRIO_STORE_IN_FO0_OF_MATCH_ID	= 5U,
	MCAN_STDIDF_ELE_PRIO_STORE_IN_FO1_OF_MATCH_ID	= 6U,
	MCAN_STDIDF_ELE_STORE_IN_RXB_OR_DMSG			= 7U,
} MCAN_StdIDElemntConfig_t;

/**
 * @brief Standard Sync Message.
 *
 * @param MCAN_STDIDF_SYNC_MSG_DISABLE Timestamping for the matching Sync
 * message disabled.
 * @param MCAN_STDIDF_SYNC_MSG_ENABLE Timestamping for the matching Sync message enabled.
 */
typedef enum {
	MCAN_STDIDF_SYNC_MSG_DISABLE					= 0,
	MCAN_STDIDF_SYNC_MSG_ENABLE						= 1,
} MCAN_StdIDFilterSyncMsg_t;

/**
 * @brief Standard ID filter type options.
 *
 * @param MCAN_STDIDF_RANGE_FROM_SFID1_TO_SFID2 Range filter from SFID1 to SFID2
 * (SFID2 >= SFID1).
 * @param MCAN_STDIDF_DUAL_ID_FILTER Dual ID filter for SFID1 or SFID2.
 * @param MCAN_STDIDF_CLASSIC_ID_FILTER Classic filter: SFID1 = filter, SFID2 = mask.
 * @param MCAN_STDIDF_FILTER_DISABLE Filter element disabled.
 */
typedef enum {
	MCAN_STDIDF_RANGE_FROM_SFID1_TO_SFID2			= 0U,
	MCAN_STDIDF_DUAL_ID_FILTER						= 1U,
	MCAN_STDIDF_CLASSIC_ID_FILTER					= 2U,
	MCAN_STDIDF_FILTER_DISABLE						= 3U,
} MCAN_StdIDFileterType_t;

/**
 * @brief MCAN standard ID filter element format.
 *
 * @param sfid2 Standard Filter ID 2.
 * This bit field has a different meaning depending on the configuration of SFEC:
 * (1) SFEC = 001...110 Second ID of standard ID filter element.
 * (2) SFEC = 111 Filter for Rx Buffers or for debug messages.
 * SFID2[10:9] decides whether the received message is stored into an
 * Rx Buffer or treated as
 * message A, B, or C of the debug message sequence.
 * 00= Store message into an Rx Buffer.
 * 01= Debug Message A.
 * 10= Debug Message B.
 * 11= Debug Message C.
 * SFID2[8:6] is used to control the filter event pins m_can_fe[2:0]
 * at the Extension Interface. A one at the respective bit position enables
 * generation of a pulse at the related filter event pin with the duration of one
 * m_can_hclk period in case the filter matches.
 * SFID2[5:0] defines the offset to the Rx Buffer Start Address RXBC.RBSA for storage of
 * a matching message.
 *
 * @param res Reserved.
 * @param ssync Standard ID Sync Message. (options:@MCAN_StdIDFilterSyncMsg_t).
 * @param sfid1 Standard Filter ID 1.
 * @param sfec Standard ID Filter Element Configuration. (options :
 * @MCAN_StdIDElemntConfig_t).
 * @param sft Standard ID Filter Type. (options:@MCAN_StdIDFileterType_t).
 */
typedef struct MCAN_StdMsgIDFilterElement {
	uint32_t sfid2	: 11;
	uint32_t res	: 4;
	uint32_t ssync	: 1;
	uint32_t sfid1	: 11;
	uint32_t sfec	: 3;
	uint32_t sft	: 2;
} MCAN_StdMsgIDFilterElement_t;

/**
 * @brief MCAN data phase data format size (8,12,16,20,24,32,48,64).
 *
 */
typedef enum {
	MCAN_ELEM_SIZE_8BYTES = 0U,
	MCAN_ELEM_SIZE_12BYTES = 1U,
	MCAN_ELEM_SIZE_16BYTES = 2U,
	MCAN_ELEM_SIZE_20BYTES = 3U,
	MCAN_ELEM_SIZE_24BYTES = 4U,
	MCAN_ELEM_SIZE_32BYTES = 5U,
	MCAN_ELEM_SIZE_48BYTES = 6U,
	MCAN_ELEM_SIZE_64BYTES = 7U
} MCAN_ElemSize_t;

/**
 * @brief Data Length Code.
 * 0-8= CAN + CAN FD: transmit frame has 0-8 data bytes.
 * 9-15= CAN: transmit frame has 8 data bytes.
 * 9-15=CAN FD: transmit frame has 12/16/20/24/32/48/64 data bytes.
 *
 */
typedef enum {
	MCAN_DATA_LENGTH_0					= 0U,
	MCAN_DATA_LENGTH_1					= 1U,
	MCAN_DATA_LENGTH_2					= 2U,
	MCAN_DATA_LENGTH_3					= 3U,
	MCAN_DATA_LENGTH_4					= 4U,
	MCAN_DATA_LENGTH_5					= 5U,
	MCAN_DATA_LENGTH_6					= 6U,
	MCAN_DATA_LENGTH_7					= 7U,
	MCAN_DATA_LENGTH_8					= 8U,
	MCAN_DATA_LENGTH_12					= 9U,
	MCAN_DATA_LENGTH_16					= 10U,
	MCAN_DATA_LENGTH_20					= 11U,
	MCAN_DATA_LENGTH_24					= 12U,
	MCAN_DATA_LENGTH_32					= 13U,
	MCAN_DATA_LENGTH_48					= 14U,
	MCAN_DATA_LENGTH_64					= 15U,
} MCAN_MsgDataLength_t;

/**
 * @brief Rx FIFO can be operated in blocking or in overwrite mode.
 *
 */
typedef enum {
	MCAN_RXFIFO_OP_MODE_BLOCKING		= 0,	/* FIFO blocking mode. */
	MCAN_RXFIFO_OP_MODE_OVERWRITE		= 1,	/* FIFO overwrite mode. */
} MCAN_RxFifoOperationMode_t;

/**
 * @brief Tx FIFO/Queue Mode options.
 *
 * @param MCAN_TXBUF_OP_IN_FIFO_MODE The Tx buffer as the Tx FIFO mode.
 * @param MCAN_TXBUF_OP_IN_QUEUE_MODE The Tx buffer as the Tx Queue mode.
 */
typedef enum {
	MCAN_TXBUF_OP_IN_FIFO_MODE			= 0,
	MCAN_TXBUF_OP_IN_QUEUE_MODE			= 1,
} MCAN_TxBufferOpMode_t;

/**
 * @brief MCAN Rx FIFO number options.
 *
 */
typedef enum MCAN_RxFIFONum {
	MCAN_RX_FIFO_NUM_0 = 0U,	/* MCAN Rx FIFO 0. */
	MCAN_RX_FIFO_NUM_1 = 1U		/* MCAN Rx FIFO 1. */
} MCAN_RxFIFONum_t;

typedef struct MCAN_RxFIFOStatus {
	MCAN_RxFIFONum_t num;
	union {
		struct {
			uint32_t fillLvl	: 7;
			uint32_t reserved0	: 1;
			uint32_t getIdx		: 7;
			uint32_t reserved1	: 2;
			uint32_t putIdx		: 6;
			uint32_t reserved2	: 2;
			uint32_t fifoFull	: 7;
			uint32_t msgLost	: 7;
			uint32_t reserved3	: 6;
		};
		uint32_t rxfifo_stat;
	};
} MCAN_RxFIFOStatus_t;

/**
 * @brief Tx FIFO/Queue Status.
 *
 * @param freeLvl Tx FIFO Free Level.
 * @param getIdx Tx FIFO/Queue Get Index.
 * @param putIdx Tx FIFO/Queue Put Index.
 * @param fifoFull Tx FIFO/Queue Full.
 */
typedef struct MCAN_TxFIFOStatus {
	union {
		struct {
			uint32_t freeLvl	: 6;
			uint32_t reserved0	: 2;
			uint32_t getIdx		: 5;
			uint32_t reserved1	: 3;
			uint32_t putIdx		: 5;
			uint32_t fifoFull	: 1;
			uint32_t reserved2	: 10;
		};
		uint32_t txfifo_stat;
	};
} MCAN_TxFIFOStatus_t;

/**
 * @brief MCAN Memory initialize paramters.
 *
 * @param lss List size of standard ID filter(The number of standard ID filter is 0-128).
 * @param lse List size of extended ID filter(The number of extended ID filter is 0-64).
 * @param flssa Base address of standard ID filter list.
 * @param flesa Base address of extended ID filter list.
 * @param txStartAddr Tx Buffers Start Address.
 * @param rxBufStartAddr Rx Buffer Start Address.
 * Configures the start address of the Rx Buffers section in the
 * Message RAM (32-bit word address). Also used to reference debug messages A,B,C.
 * @param txEventFIFOStartAddr Event FIFO Start Address.
 * @param rxFIFO0startAddr Rx FIFO 0 Start Address.
 * @param rxFIFO1startAddr Rx FIFO 1 Start Address.
 * @param txBufNum Number of Dedicated Transmit Buffers.(options:1-32).
 * @param txBufMode Tx FIFO/Queue Mode.(options:@MCAN_TxBufferOpMode_t).
 * @param rxFIFO0OpMode FIFO 0 Operation Mode.(options:@MCAN_RxFifoOperationMode_t).
 * @param rxFIFO1OpMode FIFO 1 Operation Mode.(options:@MCAN_RxFifoOperationMode_t).
 * @param txEventFIFOSize Event FIFO Size.(options: 0~32).
 * @param rxFIFO0size Rx FIFO 0 Size.(options:1~64, 0 or more than 64
 * watermark interrupt disable).
 * @param rxFIFO1size Rx FIFO 1 Size.(options:1~64, 0 or more than
 * 64 watermark interrupt disable).
 * @param txFIFOSize Transmit FIFO/Queue Size.(options:1-32).
 * @param txEventFIFOWaterMark Event FIFO Watermark.(options: 0~32).
 * @param rxFIFO0waterMark Rx FIFO 0 Watermark(options:1~64, more than
 * 64 are interpreted as 64).
 * @param rxFIFO1waterMark Rx FIFO 1 Watermark(options:1~64, more than
 * 64 are interpreted as 64).
 * @param txBufElemSize MCAN data phase data format size (8,12,16,20,24,32,48,64).
 * @param rxBufElemSize MCAN data phase data format size (8,12,16,20,24,32,48,64).
 * @param rxFIFO0ElemSize MCAN data phase data format size (8,12,16,20,24,32,48,64).
 * @param rxFIFO1ElemSize MCAN data phase data format size (8,12,16,20,24,32,48,64).
 */
typedef struct MCAN_MsgRAMConfigParams {
	uint8_t lss;
	uint8_t lse;
	uint16_t flssa;
	uint16_t flesa;
	uint16_t txStartAddr;
	uint16_t rxBufStartAddr;
	uint16_t txEventFIFOStartAddr;
	uint16_t rxFIFO0startAddr;
	uint16_t rxFIFO1startAddr;
	uint8_t txBufNum;
	MCAN_TxBufferOpMode_t txBufMode;
	MCAN_RxFifoOperationMode_t rxFIFO0OpMode;
	MCAN_RxFifoOperationMode_t rxFIFO1OpMode;
	uint8_t txEventFIFOSize;
	uint8_t rxFIFO0size;
	uint8_t rxFIFO1size;
	uint8_t txFIFOSize;
	uint8_t txEventFIFOWaterMark;
	uint8_t rxFIFO0waterMark;
	uint8_t rxFIFO1waterMark;
	MCAN_ElemSize_t txBufElemSize;
	MCAN_ElemSize_t rxBufElemSize;
	MCAN_ElemSize_t rxFIFO0ElemSize;
	MCAN_ElemSize_t rxFIFO1ElemSize;
} MCAN_MsgRAMConfigParams_t;

/**
 * @brief The MCAN initialization parameters.
 *
 * @param fdMode Enable/Disable CANFD mode(true is enable, else disable).
 * @param fdFormat Use standard CANFD format or Bosch CANFD format.
 * @param brsEnable Enable/Disable bitrate switch in mode(true is enable, else disable).
 * @param txpEnable indicates whether enable/disable the Transmit Pause. If enabled, the
 * M_CAN pauses for two CAN bit times before starting the next transmission after itself
 * has successfully transmitted a frame.
 * @param efbi indicates whether enable/disable the Edge Filtering During Bus Integration.
 * @param pxhddisable indicates that enable/disable the Protocol Exception
 * Handling Disable.
 * @param darEnable indicates whether enable/disable the Automatic Retransmission.
 * @param wkupReqEnable
 * @param autoWkupEnable
 * @param emulationEnable
 * @param tdcEnable Enable/Disable Transmitter Delay Compensation(true is enable,
 * else disable).
 * @param wdcPreload indicates that start value of the Message RAM Watchdog Counter.
 * With the reset value of 00 the counter is disabled.
 * @param wmMarker indicates the use of 8/16 bit wide message markers.
 * @param tdcConfig TDC initialization parameters.
 */
typedef struct MCAN_InitParams {
	bool fdMode;
	bool brsEnable;
	bool txpEnable;
	bool efbi;
	bool pxhddisable;
	bool darEnable;
	bool wkupReqEnable;
	bool autoWkupEnable;
	bool emulationEnable;
	bool tdcEnable;
	uint8_t wdcPreload;
	MCAN_FDISOMode_t fdFormat;
	MCAN_WideMessageMarker_t wmMarker;
	MCAN_TDCConfig_t tdcConfig;
} MCAN_InitParams_t;

/**
 * @brief The MCAN configuration parameters.
 *
 * @param monEnable Enable/Disable Bus Monitoring Mode.
 * @param asmEnable Enable/Disable Restricted Operation Mode.
 * The Restricted Operation Mode must not be combined with the
 * Loop Back Mode (internalor external).
 * @param timeoutCntEnable Enable/Disable Timeout Counter.
 * @param tsPrescalar Timestamp Counter Prescaler.
 * Configures the timestamp and timeout counters time unit in multiples of CAN bit times
 * [1-16]. The actual interpretation by the hardware of this value is such that one more
 * than the value programmed here is used.
 * @param tsClock MCAN timestamp clock source(@MCAN_TimestampClk_t).
 * @param tsSelect Timestamp source selection(@MCAN_TimestampSource_t).
 * @param timeoutSelect Select timeout operation mode(@MCAN_TimeOutSelect_t).
 * @param timeoutPreload Timeout Period.
 * @param filterConfig Global Filter Configuration parameters(@MCAN_GlobalFiltConfig_t).
 */
typedef struct MCAN_ConfigParams {
	bool monEnable;
	bool asmEnable;
	bool timeoutCntEnable;
	uint8_t tsPrescalar;
	MCAN_TimestampClk_t tsClock;
	MCAN_TimestampSource_t tsSelect;
	MCAN_TimeOutSelect_t timeoutSelect;
	uint16_t timeoutPreload;
	MCAN_GlobalFiltConfig_t filterConfig;
} MCAN_ConfigParams_t;

typedef enum MCAN_MemType {
	MCAN_MEM_TYPE_BUF = 0U,		/* MCAN Msg RAM buffers. */
	MCAN_MEM_TYPE_FIFO = 1U		/* MCAN Msg RAM FIFO/Queue. */
} MCAN_MemType_t;

/**
 * @brief MCAN received message storage space.
 * When no TSU is used (CCCR.UTSU = '0'), User should read utcu structure.
 * Otherwise, user should read tcu structure.
 *
 * @param id ID number of the receive message
 * @param rtr Remote Transmission Request.
 * @param xtd Extended Identifier.
 * @param esi Error State Indicator.
 * (0 : ESI bit in CAN FD format depends only on error passive flag).
 * (1 : ESI bit in CAN FD format transmitted recessive).
 * @param rxts Rx Timestamp.
 * @param dlc Data Length Code.(options:@MCAN_MsgDataLength_t).
 * 0-8= CAN + CAN FD: transmit frame has 0-8 data bytes.
 * 9-15= CAN: transmit frame has 8 data bytes.
 * 9-15=CAN FD: transmit frame has 12/16/20/24/32/48/64 data bytes.
 * @param brs Bit Rate Switch.
 * @param fdf FD Format. If set, the frame is CANFD format.
 * @param fidx Index of matching Rx acceptance filter element.
 * @param anmf Accepted Non-matching Frame.
 * @param data CAN data message storage area.
 * @param rxtsp Rx Timestamp Pointer.
 * @param tsc Timestamp Captured.
 * @param res[n] Reserved for future use.
 * @param msgsize Array mapping Rx buffer storage space.
 */
typedef union MCAN_RxMessage {
	struct {
		union {
			struct {
				uint32_t res0	: 18;
				uint32_t id_std	: 11;
				uint32_t res1	: 1;
				uint32_t res2	: 1;
				uint32_t res3	: 1;
			};

			struct {
				uint32_t id_ext	: 29;
				uint32_t res4	: 1;
				uint32_t res5	: 1;
				uint32_t res6	: 1;
			};

			struct {
				uint32_t id		: 29;
				uint32_t rtr	: 1;
				uint32_t xtd	: 1;
				uint32_t esi	: 1;
			};
		};

		union {
			/* Not using TCU mode or CCCR.WMM = '0'. */
			uint16_t rxts;
			/* Using TCU mode or CCCR.WMM = '1'. */
			struct {
				uint16_t rxtsp	: 4;
				uint16_t tsc	: 1;
				uint16_t res7	: 11;
			};
		};

		uint16_t dlc	: 4;
		uint16_t brs	: 1;
		uint16_t fdf	: 1;
		uint16_t res8	: 2;
		uint16_t fidx	: 7;
		uint16_t anmf	: 1;

		uint8_t data[M_CAN_RX_MAX_DATA_LENGTH];
	};

	volatile uint8_t msgsize[M_CAN_RX_MAX_MSG_LENGTH];
} MCAN_RxMessage_t;

/**
 * @brief MCAN Tx event message storage space.
 *
 * @param id ID number of the sent message.
 * @param rtr Remote Transmission Request.
 * @param xtd Extended Identifier.
 * @param esi Error State Indicator. 
 * (0 : ESI bit in CAN FD format depends only on error passive flag).
 * (1 : ESI bit in CAN FD format transmitted recessive).
 * @param mm Message Marker low 8 bit.(CCCR.WMM = '0').
 * @param mm1 Message Marker high 8 bit.(CCCR.WMM = '1').
 * @param dlc Data Length Code.(options:@MCAN_MsgDataLength_t).
 * 0-8= CAN + CAN FD: transmit frame has 0-8 data bytes.
 * 9-15= CAN: transmit frame has 8 data bytes.
 * 9-15=CAN FD: transmit frame has 12/16/20/24/32/48/64 data bytes.
 * @param brs Bit Rate Switch.
 * @param fdf FD Format. If set, the frame is CANFD format.
 * @param tsce Time Stamp Capture Enable for TSU.
 * @param efc Event FIFO Control(store/do not store Tx event).
 * @param data CAN data message storage area.
 * @param msgsize Array mapping Tx event buffer storage space.
 */
typedef union MCAN_TxMessage {
	struct {
		struct {
			union {
				struct {
					uint32_t res0	: 18;
					uint32_t id_std	: 11;
					uint32_t res1	: 1;
					uint32_t res2	: 1;
					uint32_t res3	: 1;
				};

				struct {
					uint32_t id_ext	: 29;
					uint32_t res4	: 1;
					uint32_t res5	: 1;
					uint32_t res6	: 1;
				};

				struct {
					uint32_t id		: 29;
					uint32_t rtr	: 1;
					uint32_t xtd	: 1;
					uint32_t esi	: 1;
				};
			};
		};
		uint32_t res7			: 8;
		uint32_t mm1			: 8;
		uint32_t dlc			: 4;
		uint32_t brs			: 1;
		uint32_t fdf			: 1;
		uint32_t tsce			: 1;
		uint32_t efc			: 1;
		uint32_t mm				: 8;
		uint8_t data[M_CAN_TX_MAX_DATA_LENGTH];
	};

	volatile uint8_t msgsize[M_CAN_TX_MAX_MSG_LENGTH];
} MCAN_TxMessage_t;

/**
 * @brief MCAN Tx event message storage space.
 * When no TSU is used (CCCR.UTSU = '0'), User should read utcu structure.
 * Otherwise, user should read tcu structure.
 *
 * @param id ID number of the sent message.
 * @param rtr Remote Transmission Request.
 * @param xtd Extended Identifier.
 * @param esi Error State Indicator.
 * @param txts tx timestamp.
 * @param dlc Data Length Code.
 * 0-8= CAN + CAN FD: transmit frame has 0-8 data bytes.
 * 9-15= CAN: transmit frame has 8 data bytes.
 * 9-15=CAN FD: transmit frame has 12/16/20/24/32/48/64 data bytes.
 * @param brs Bit Rate Switch.
 * @param fdf FD Format. If set, the frame is CANFD format.
 * @param et Event Type.
 * @param mm Message Marker low 8 bit.(CCCR.WMM = '0' and TSU not used).
 * @param mm1 Message Marker high 8 bit.(CCCR.WMM = '1' or use TSU).
 * @param txtsp Tx Timestamp Pointer.Number of TSU Time Stamp register (TS0...15)
 * where the related timestamp is stored.
 * @param tsc Timestamp captured and stored in TSU Timestamp register
 * referenced by E1B.TXTSP.
 * @param res[0:1] Reserved for future use.
 * @param msgsize Array mapping Tx event buffer storage space.
 */
typedef union MCAN_TxEventFifo {
	struct {
		uint32_t id				: 29;
		uint32_t rtr			: 1;
		uint32_t xtd			: 1;
		uint32_t esi			: 1;
		union {
			/* Using TCU mode or CCCR.WMM = '1'. */
			struct {
				uint16_t txtsp	: 4;
				uint16_t tsc	: 1;
				uint16_t res1	: 3;
				uint16_t mm1	: 8;
			};

			/* Not using TCU mode or CCCR.WMM = '0'. */
			uint16_t txts;
		};

		uint16_t dlc			: 4;
		uint16_t brs			: 1;
		uint16_t fdf			: 1;
		uint16_t et				: 2;
		uint16_t mm				: 8;
	};

	volatile uint8_t msgsize[M_CAN_TX_EVENT_FIFO_SIZE];
} MCAN_TxEventFifo_t;

/**
 * @brief MCAN high priority status.
 *
 * @param bidx Buffer Index.(Index of Rx FIFO element to which the message was stored.
 * Only valid when MSI[1] = 1.)
 * @param msi Message Storage Indicator(@MCAN_HighPriorityMsgFifoStatus_t).
 * @param fidx Filter Index(Index of matching filter element.
 * Range is 0 to SIDFC.LSS - 1 resp. XIDFC.LSE - 1.).
 * @param flst Filter List (0- Standard Filter List, 1- Extended Filter List).
 */
typedef struct MCAN_HighPriorityMsgInfo {
	union {
		struct {
			uint16_t bidx	: 6;
			uint16_t msi	: 2;
			uint16_t fidx	: 7;
			uint16_t flst	: 1;
		};

		uint16_t hpm_status;
	};
} MCAN_HighPriorityMsgInfo_t;

typedef enum {
	MCAN_HIGH_PRIORITY_NO_FIFO_SEL	= 0U,		/* No FIFO selected. */
	MCAN_HIGH_PRIORITY_MSG_LOST		= 1U,		/* FIFO message lost. */
	MCAN_HIGH_PRIORITY_STORED_FIFO0	= 2U,		/* Message is stored in Rx FIFO 0. */
	MCAN_HIGH_PRIORITY_STORED_FIFO1	= 3U,		/* Message is stored in Rx FIFO 1. */
} MCAN_HighPriorityMsgFifoStatus_t;

/**
 * @brief MCAN interrupt source number.
 *
 * @param MCAN_INT_SRC_RxFIFO0_NEW_MSG New message written to Rx FIFO 0.
 * @param MCAN_INT_SRC_RxFIFO0_WATERMARK_REACHED Rx FIFO 0 fill level reached watermark.
 * @param MCAN_INT_SRC_RXFIFO0_FULL Rx FIFO 0 full.
 * @param MCAN_INT_SRC_RXFIFO0_MSG_LOSS Rx FIFO 0 message lost, also set after write
 * attempt to Rx FIFO 0 of size zero.
 * @param MCAN_INT_SRC_RxFIFO1_NEW_MSG New message written to Rx FIFO 1.
 * @param MCAN_INT_SRC_RxFIFO1_WATERMARK_REACHED Rx FIFO 1 fill level reached watermark.
 * @param MCAN_INT_SRC_RXFIFO1_FULL Rx FIFO 1 full.
 * @param MCAN_INT_SRC_RXFIFO1_MSG_LOSS Rx FIFO 1 message lost, also set after
 * write attempt to Rx FIFO 1 of size zero.
 * @param MCAN_INT_SRC_HIGH_PRIORITY_MSG High priority message received.
 * @param MCAN_INT_SRC_TRANSMISSION_COMPLETE Transmission completed.
 * @param MCAN_INT_SRC_TRANSMISSION_CANCLE_FINISHED Transmission cancellation finished.
 * @param MCAN_INT_SRC_TXFIFO_EMPTY Tx FIFO empty.
 * @param MCAN_INT_SRC_TXEVENT_FIFO_NEW_ENTRY Tx Handler wrote Tx Event FIFO element.
 * @param MCAN_INT_SRC_TXEVENT_FIFO_WATEMARK_REACHED Tx Event FIFO fill level
 * reached watermark.
 * @param MCAN_INT_SRC_TXEVENT_FIFO_FULL Tx Event FIFO full.
 * @param MCAN_INT_SRC_TXEVENT_FIFO_ELEMENT_LOST Tx Event FIFO element lost,
 * also set after write attempt to Tx Event FIFO of size zero.
 * @param MCAN_INT_SRC_TIMESTAMP_COUNTER_WRAPAROUND Timestamp counter wrapped around.
 * @param MCAN_INT_SRC_MESSAGERAM_ACCESS_FAILURE Message RAM Access Failure.
 * @param MCAN_INT_SRC_TIMEOUT_OCCURRED Timeout reached.
 * @param MCAN_INT_SRC_MESSAGE_STORED_TO_RXBUF At least one received message stored into
 * an Rx Buffer.
 * @param MCAN_INT_SRC_BIT_ERROR_CORRECTED Bit error detected and corrected (e.g. ECC).
 * @param MCAN_INT_SRC_BIT_ERROR_UNCORRECTED Bit error detected, uncorrected
 * (e.g. parity logic).
 * @param MCAN_INT_SRC_ERROR_LOGGING_OVERFLOW Overflow of CAN Error Logging
 * Counter occurred.
 * @param MCAN_INT_SRC_ERROR_PASSIVE Error_Passive status changed.
 * @param MCAN_INT_SRC_WARNING_STATUS Error_Warning status changed.
 * @param MCAN_INT_SRC_BUS_OFF_STATUS Bus_Off status changed.
 * @param MCAN_INT_SRC_WATCHDOG_INTERRUPT Message RAM Watchdog event due to missing READY.
 * @param MCAN_INT_SRC_PROTOCOL_ERROR_IN_ARBITRATION_PHASE Protocol error in arbitration
 * phase detected (PSR.LEC != 0,7).
 * @param MCAN_INT_SRC_PROTOCOL_ERROR_IN_DATA_PHASE Protocol error in data phase detected
 * (PSR.DLEC != 0,7).
 * @param MCAN_INT_SRC_ACCESS_RESERVED_ADDRESS Access to reserved address occurred.
 */
typedef enum {
	MCAN_INT_SRC_RxFIFO0_NEW_MSG						= M_CAN_IR_RF0N_M,
	MCAN_INT_SRC_RxFIFO0_WATERMARK_REACHED				= M_CAN_IR_RF0W_M,
	MCAN_INT_SRC_RXFIFO0_FULL							= M_CAN_IR_RF0F_M,
	MCAN_INT_SRC_RXFIFO0_MSG_LOSS						= M_CAN_IR_RF0L_M,
	MCAN_INT_SRC_RxFIFO1_NEW_MSG						= M_CAN_IR_RF1N_M,
	MCAN_INT_SRC_RxFIFO1_WATERMARK_REACHED				= M_CAN_IR_RF1W_M,
	MCAN_INT_SRC_RXFIFO1_FULL							= M_CAN_IR_RF1F_M,
	MCAN_INT_SRC_RXFIFO1_MSG_LOSS						= M_CAN_IR_RF1L_M,
	MCAN_INT_SRC_HIGH_PRIORITY_MSG						= M_CAN_IR_HPM_M,
	MCAN_INT_SRC_TRANSMISSION_COMPLETE					= M_CAN_IR_TC_M,
	MCAN_INT_SRC_TRANSMISSION_CANCLE_FINISHED			= M_CAN_IR_TCF_M,
	MCAN_INT_SRC_TXFIFO_EMPTY							= M_CAN_IR_TFE_M,
	MCAN_INT_SRC_TXEVENT_FIFO_NEW_ENTRY					= M_CAN_IR_TEFN_M,
	MCAN_INT_SRC_TXEVENT_FIFO_WATEMARK_REACHED			= M_CAN_IR_TEFW_M,
	MCAN_INT_SRC_TXEVENT_FIFO_FULL						= M_CAN_IR_TEFF_M,
	MCAN_INT_SRC_TXEVENT_FIFO_ELEMENT_LOST				= M_CAN_IR_TEFL_M,
	MCAN_INT_SRC_TIMESTAMP_COUNTER_WRAPAROUND			= M_CAN_IR_TSW_M,
	MCAN_INT_SRC_MESSAGERAM_ACCESS_FAILURE				= M_CAN_IR_MRAF_M,
	MCAN_INT_SRC_TIMEOUT_OCCURRED						= M_CAN_IR_TOO_M,
	MCAN_INT_SRC_MESSAGE_STORED_TO_RXBUF				= M_CAN_IR_DRX_M,
	MCAN_INT_SRC_BIT_ERROR_CORRECTED					= M_CAN_IR_BEC_M,
	MCAN_INT_SRC_BIT_ERROR_UNCORRECTED					= M_CAN_IR_BEU_M,
	MCAN_INT_SRC_ERROR_LOGGING_OVERFLOW					= M_CAN_IR_ELO_M,
	MCAN_INT_SRC_ERROR_PASSIVE							= M_CAN_IR_EP_M,
	MCAN_INT_SRC_WARNING_STATUS							= M_CAN_IR_EW_M,
	MCAN_INT_SRC_BUS_OFF_STATUS							= M_CAN_IR_BO_M,
	MCAN_INT_SRC_WATCHDOG_INTERRUPT						= M_CAN_IR_WDI_M,
	MCAN_INT_SRC_PROTOCOL_ERROR_IN_ARBITRATION_PHASE	= M_CAN_IR_PEA_M,
	MCAN_INT_SRC_PROTOCOL_ERROR_IN_DATA_PHASE			= M_CAN_IR_PED_M,
	MCAN_INT_SRC_ACCESS_RESERVED_ADDRESS				= M_CAN_IR_ARA_M,
} MCAN_InterruptSource_t;

/**
 * @brief Enum of MCAN interrupt line.
 *
 * @param MCAN_INTERRUPT_LINE_0 MCAN interrupt line 0.
 * @param MCAN_INTERRUPT_LINE_1 MCAN interrupt line 1.
 */
typedef enum {
	MCAN_INTERRUPT_LINE_0				= 1,
	MCAN_INTERRUPT_LINE_1				= 2,
} MCAN_InterruptLine_t;

/**
 * @brief Structure for MCAN new data flag for Rx buffer.
 *
 */
typedef struct {
	/* New data flag for Rx buffer no. 0 to 31 */
	uint32_t statusLow;
	/* New data flag for Rx buffer no. 32 to 63 */
	uint32_t statusHigh;
} MCAN_RxNewDataStatus_t;

/**
 * @brief  Structure for MCAN Tx Event FIFO Status.
 *
 * @param fillLvl Event FIFO Fill Level.
 * @param getIdx Event FIFO Gut Index.
 * @param putIdx Event FIFO Put Index.
 * @param fifoFull Event FIFO Full.
 * 0 = Tx Event FIFO not full.
 * 1 = Tx Event FIFO full.
 * @param eleLost Tx Event FIFO Element Lost.
 * 0 = No Tx Event FIFO element lost.
 * 1 = Tx Event FIFO element lost, also set after write attempt to
 * Tx Event FIFO of size zero.
 */
typedef struct MCAN_TxEventFIFOStatus {
	uint32_t fillLvl	: 6;
	uint32_t res0		: 2;
	uint32_t getIdx		: 5;
	uint32_t res1		: 3;
	uint32_t putIdx		: 5;
	uint32_t res2		: 3;
	uint32_t fifoFull	: 1;
	uint32_t eleLost	: 1;
	uint32_t res3		: 6;
} MCAN_TxEventFIFOStatus_t;

/**
 * @brief  Structure for MCAN error counter.
 *
 * @param transErrLogCnt Transmission Error Counter.
 * @param recErrCnt Receive Error Counter.
 * @param rpStatus RP Status.
 * 0 = Error Passive.
 * 1 = Error Warning.
 * @param canErrLogCnt CAN Error Logging.
 */
typedef struct MCAN_ErrCntStatus {
	union {
		struct {
			uint32_t transErrLogCnt		: 8;
			uint32_t recErrCnt			: 7;
			uint32_t rpStatus			: 1;
			uint32_t canErrLogCnt		: 8;
			uint32_t res				: 8;
		};

		uint32_t ecr_val;
	};
} MCAN_ErrCntStatus_t;

/**
 * @brief Structure for MCAN protocol status.
 *
 * @param lastErrCode Last Error Code.
 * The LEC indicates the type of the last error to occur on the CAN bus. This field will
 * be cleared to 0 when a message has been transferred (reception or transmission)
 * without error.0= No Error: No error occurred since LEC has been reset by successful
 * reception or transmission.
 * 1 - Stuff Error: More than 5 equal bits in a sequence have
 * occurred in a part of a received message where this is not allowed.
 * 2 - Form Error: A fixed format part of a received frame has the wrong format.
 * 3 - AckError: The message transmitted by the M_CAN was not acknowledged by another
 * node.
 * 4 - Bit1Error: During the transmission of a message (with the exception of the
 * arbitration field), the device wanted to send a recessive level (bit of logical value
 * 1), but the monitored bus value was dominant.
 * 5 - Bit0Error: During the transmission of a message (or acknowledge bit, or active
 * error flag, or overload flag), the device wanted to send a dominant level (data or
 * identifier bit logical value 0), but the monitored bus value was recessive. During
 * Bus_Off recovery this status is set each time a sequence of 11 recessive bits has been
 * monitored. This enables the CPU to monitor the proceeding of the Bus_Off recovery
 * sequence (indicating the bus is not stuck at dominant or continuously disturbed).
 * 6 - CRCError: The CRC check sum of a received message was incorrect. The CRC of an
 * incoming message does not match with the CRC calculated from the received data.
 * 7 - NoChange: Any read access to the Protocol Status Register re-initializes the LEC
 * to 7. When the LEC shows the value 7, no CAN bus event was detected since the last
 * CPU readaccess to the Protocol Status Register.
 * @param act Activity.
 * Monitors the module's CAN communication state.
 * 00 - Synchronizing	node is synchronizing on CAN communication.
 * 01 - Idle			node is neither receiver nor transmitter.
 * 10 - Receiver		node is operating as receiver.
 * 11 - Transmitter		node is operating as transmitter.
 * @param errPassive Error Passive.
 * 0 - The M_CAN is in the Error_Active state. It normally takes part in bus
 * communication and sends an active error flag when an error has been detected.
 * 1 - The M_CAN is in the Error_Passive state.
 * @param warningStatus Error Warning.
 * 0 - Both error counters are below the Error_Warning limit of 96.
 * 1 - At least one of error counter has reached the Error_Warning limit of 96.
 * @param busOffStatus Bus Off Status.
 * 0 - The M_CAN is not Bus_Off.
 * 1 - The M_CAN is in the Bus_Off state.
 * @param dlec Data Phase Last Error Code.
 * Type of last error that occurred in the data phase of a CAN FD format frame with its
 * BRS flag set. Coding is the same as for LEC. This field will be cleared to zero when a
 * CAN FD format frame with its BRS flag set has been transferred (reception or
 * transmission) without erro.
 * @param resi ESI flag of last received CAN FD Message.
 * This bit is set together with RFDF, independent of acceptance filtering.
 * 0 - Last received CAN FD message did not have its ESI flag set.
 * 1 - Last received CAN FD message had its ESI flag set.
 * @param rbrs BRS flag of last received CAN FD Message.
 * This bit is set together with RFDF, independent of acceptance filtering.
 * 0 - Last received CAN FD message did not have its BRS flag set.
 * 1 - Last received CAN FD message had its BRS flag set.
 * @param rfdf Received a CAN FD Message Flag.
 * This bit is set independent of acceptance filtering.
 * 0 - Since this bit was reset by the CPU, no CAN FD message has been received.
 * 1 - Message in CAN FD format with FDF flag set has been received.
 * @param pxe Protocol Exception Event.
 * 0 - No protocol exception event occurred since last read access.
 * 1 - Protocol exception event occurred.
 * @param tdcv Transmitter Delay Compensation Value.
 * Position of the secondary sample point, defined by the sum of the measured delay
 * from m_can_tx to m_can_rx and TDCR.TDCO. The SSP position is, in the data
 * phase, the number of mtq between the start of the transmitted bit and the secondary
 * sample point. Valid values are 0 to 127 mtq.
 */
typedef struct MCAN_ProtocolStatus {
	union {
		struct {
			uint32_t lastErrCode		: 3;
			uint32_t act				: 2;
			uint32_t errPassive			: 1;
			uint32_t warningStatus		: 1;
			uint32_t busOffStatus		: 1;
			uint32_t dlec				: 3;
			uint32_t resi				: 1;
			uint32_t rbrs				: 1;
			uint32_t rfdf				: 1;
			uint32_t pxe				: 1;
			uint32_t res0				: 1;
			uint32_t tdcv				: 7;
			uint32_t res1				: 9;
		};

		uint32_t psr_val;
	};
} MCAN_ProtocolStatus_t;

/**
 * @brief  Structure for accessing Revision ID and Core Release Info.
 *
 * @param scheme Scheme ID.
 * @param bu Business Unit: 10 = Processors.
 * @param modId Module ID.
 * @param rtlRev RTL revision.
 * @param major Major revision.
 * @param custom Custom revision.
 * @param minor Minor revision.
 * @param day Time Stamp Day. Two digits, BCD-coded.
 * @param mon Time Stamp Month. Two digits, BCD-coded.
 * @param year Time Stamp Year. Single digit, BCD-coded.
 * @param subStep Sub-step of Core Release Single digit, BCD-coded.
 * @param step Step of Core Release.Two digits, BCD-coded Single digit, BCD-coded.
 * @param rel Core Release. Single digit, BCD-coded.
 */
typedef struct MCAN_RevisionId {
	union {
		struct {
			uint32_t day		: 8;
			uint32_t mon		: 8;
			uint32_t year		: 4;
			uint32_t subStep	: 4;
			uint32_t step		: 4;
			uint32_t rel		: 4;
		};

		uint32_t crel_val;
	};

	union {
		struct {
			uint32_t minor		: 6;
			uint32_t custom		: 2;
			uint32_t major		: 3;
			uint32_t rtlRev		: 5;
			uint32_t modId		: 12;
			uint32_t bu			: 2;
			uint32_t scheme		: 2;
		};

		uint32_t pid_val;
	};
} MCAN_RevisionId_t;

/**
 * @brief Structure for accessing Revision ID of ECC AGGR.
 *
 * @param scheme Scheme ID.
 * @param bu Business Unit: 10 = Processors.
 * @param modId Module ID.
 * @param rtlRev RTL revision.
 * @param major Major revision.
 * @param custom Custom revision.
 * @param minor Minor revision.
 */
typedef struct MCAN_ECCAggrRevisionId {
	union {
		struct {
			uint32_t minor		: 6;
			uint32_t custom		: 2;
			uint32_t major		: 3;
			uint32_t rtlRev		: 5;
			uint32_t modId		: 12;
			uint32_t bu			: 2;
			uint32_t scheme		: 2;
		};

		uint32_t rev_val;
	};
} MCAN_ECCAggrRevisionId_t;

/**
 * @brief Structure for MCAN ECC configuration parameters.
 *
 * @param enable Enable/disable ECC.
 * @param enableChk Enable/disable ECC Check.
 * @param enableRdModWr Enable/disable Read Modify Write operation.
 */
typedef struct MCAN_ECCConfigParams {
	bool enable;
	bool enableChk;
	bool enableRdModWr;
} MCAN_ECCConfigParams_t;

/**
 * @brief Enum to represent the ECC Error Types
 *
 */
typedef enum {
	/* ECC Single Error Correction */
	MCAN_ECC_ERR_TYPE_SEC	= 0U,
	/* ECC Single Error Detection */
	MCAN_ECC_ERR_TYPE_DED	= 1U
} MCAN_ECCErrType_t;

/**
 * @brief Structure for ECC Error forcing.
 *
 * @param errType Error type to be forced(@MCAN_ECCErrType_t).
 * @param rowNum Row address where error needs to be applied.
 * @param bit1 Column/Data bit that needs to be flipped
 * when force_sec or force_ded is set.
 * @param bit2 Data bit that needs to be flipped when force_ded is set.
 * @param errOnce Force Error once.
 * 1: The error will inject an error to the specified row only once.
 * @param errForce Force error on the next RAM read.
 */
typedef struct MCAN_ECCErrForceParams {
	MCAN_ECCErrType_t errType;
	uint32_t rowNum;
	uint16_t bit1;
	uint16_t bit2;
	bool errOnce;
	bool errForce;
} MCAN_ECCErrForceParams_t;

/**
 * @brief Structure for ECC Error Status.
 *
 * @param Single Bit Error Status(true is Single Bit Error pending).
 * @param dedErr Double Bit Error Status(true is Double Bit Error pending).
 * @param row Indicates the row/address where the single or double bit error occurred.
 * @param bit1 Indicates the bit position in the ram data that is in error.
 */
typedef struct MCAN_ECCErrStatus {
	uint8_t secErr;
	uint8_t dedErr;
	uint16_t bit1;
	uint32_t row;
} MCAN_ECCErrStatus_t;

/**
 * @brief Check if the MCAN controller instance is exist.
 *
 * @param base indicates that the MCAN instance base address.
 * @return true indicates that the MCAN instance is exist.
 * @return false indicates that the MCAN instance does not exist.
 */
static inline bool
MCAN_isBaseValid(uint32_t base)
{
	for (uint8_t i = 0; i < M_CAN_NUMBER_OF_INSTANCES; i++) {
		if (MCAN_BaseAddressArray[i] == base)
			return true;
	}

	return false;
}

/**
 * @brief Check if the access to the MCAN register has timeout.
 *
 * @param addr indicates that the MCAN register address.
 * @param cmpval indicates that the compare value.
 * @param mask indicates that the mask value.
 * @param timeout indicates that the maximum time to wait for writing to a register.(n US)
 * @return true indicates that the register has been written successfully.
 * @return false indicates that the register write has timed out.
 */
static inline bool
MCAN_checkAccessTimeout(uint32_t addr, uint32_t cmpval, uint32_t mask, int32_t timeout)
{
	do {
		if ((HWREG(addr) & mask) == cmpval)
			return true;
		__asm__ __volatile__("nop");
	} while (timeout-- <= 0);

	return false;
}

static inline bool
MCAN_setOpMode(uint32_t base, MCAN_OperationMode_t op_mode)
{
	if (!MCAN_isBaseValid(base))
		return false;

	/* Set MCAN to enter op_mode mode. */
	if (op_mode == MCAN_OPERATION_SW_INIT_MODE)
		HWREG(base + M_CAN_CCCR) |= M_CAN_CCCR_INIT_M;
	else
		HWREG(base + M_CAN_CCCR) &= ~M_CAN_CCCR_INIT_M;

	/* Wait for MCAN to enter op_mode mode. */
	return MCAN_checkAccessTimeout(base + M_CAN_CCCR,
								op_mode,
								M_CAN_CCCR_INIT_M,
								0xffffffff);
}

static inline bool
MCAN_getOpMode(uint32_t base, MCAN_OperationMode_t *op_mode)
{
	if (!MCAN_isBaseValid(base))
		return false;

	*op_mode = (HWREG(base + M_CAN_CCCR) & M_CAN_CCCR_INIT_M) ?
			MCAN_OPERATION_SW_INIT_MODE : MCAN_OPERATION_NORMAL_MODE;

	return true;
}

static inline bool
MCAN_writeProtectedRegAccessUnlock(uint32_t base)
{
	uint32_t regval;

	if (!MCAN_isBaseValid(base))
		return false;

	regval = HWREG(base + M_CAN_CCCR);

	/* CCE register can only be written in initialization mode. */
	if (!(regval & M_CAN_CCCR_INIT_M))
		return false;

	/* The CPU has write access to the protected configuration registers. */
	regval |= M_CAN_CCCR_CCE_M;

	HWREG(base + M_CAN_CCCR) = regval;

	return true;
}

static inline bool
MCAN_writeProtectedRegAccessLock(uint32_t base)
{
	uint32_t regval;

	if (!MCAN_isBaseValid(base))
		return false;

	regval = HWREG(base + M_CAN_CCCR);

	/* CCE register can only be written in initialization mode. */
	if (!(regval & M_CAN_CCCR_INIT_M))
		return false;

	/* The CPU has no write access to the protected configuration registers. */
	regval &= ~M_CAN_CCCR_CCE_M;

	HWREG(base + M_CAN_CCCR) = regval;

	return true;
}

/**
 * @brief Enable/Disable MCAN transmit pause. If this bit is set, the M_CAN pauses for
 * two CAN bit times before starting the next transmission after itself has successfully
 * transmitted a frame.
 *
 * @param base indicates that the MCAN instance base address.
 * @param is_enable true is enable, else disable.
 */
static inline void
MCAN_setTxPause(uint32_t base, bool is_enable)
{
	if (is_enable)
		HWREG(base + M_CAN_CCCR) |= M_CAN_CCCR_TXP_M;
	else
		HWREG(base + M_CAN_CCCR) &= ~M_CAN_CCCR_TXP_M;
}

/**
 * @brief Enable/Disable MCAN Edge Filtering during Bus Integration. If enabled.
 * Two consecutive dominant tq required to detect an edge for hard synchronization.
 *
 * @param base indicates that the MCAN instance base address.
 * @param is_enable true is enable, else disable.
 */
static inline void
MCAN_setEFBI(uint32_t base, bool is_enable)
{
	if (is_enable)
		HWREG(base + M_CAN_CCCR) |= M_CAN_CCCR_EFBI_M;
	else
		HWREG(base + M_CAN_CCCR) &= ~M_CAN_CCCR_EFBI_M;
}

/**
 * @brief Enable/Disable MCAN Protocol Exception Handling.
 *
 * @param base indicates that the MCAN instance base address.
 * @param is_enable true is enable, else disable.
 */
static inline void
MCAN_setProtocolExceptionHandling(uint32_t base, bool is_enable)
{
	if (is_enable)
		HWREG(base + M_CAN_CCCR) &= ~M_CAN_CCCR_PXHD_M;
	else
		HWREG(base + M_CAN_CCCR) |= M_CAN_CCCR_PXHD_M;
}

/**
 * @brief Enable/Disable Automatic Retransmission. If enabled, Automatic retransmission
 * of messages not transmitted successfully enabled.
 *
 * @param base indicates that the MCAN instance base address.
 * @param is_enable true is enable, else disable.
 */
static inline void
MCAN_setAutomaticRetransmission(uint32_t base, bool is_enable)
{
	if (is_enable)
		HWREG(base + M_CAN_CCCR) &= ~M_CAN_CCCR_DAR_M;
	else
		HWREG(base + M_CAN_CCCR) |= M_CAN_CCCR_DAR_M;
}

/**
 * @brief Set the MCAN RAM Watchdog counter value.Start value of the
 * Message RAM Watchdog Counter. With the reset value of "00" the counter is disabled.
 *
 * @param base indicates that the MCAN instance base address.
 * @param counter Start value of the Message RAM Watchdog Counter.
 */
static inline void
MCAN_setRAMWatchdogCounter(uint32_t base, uint8_t counter)
{
	HWREG(base + M_CAN_RWD) = counter;
}

/**
 * @brief Set the Listen Only Mode.
 * For MON mode, It meets the ISO standard and only send recessive bits on the CAN BUS.
 * For ASM mode, It only receives valid data and send acknowledge bit.
 *
 * @param base indicates that the MCAN instance base address.
 * @param mode Select use Bus Monitoring Mode or MCAN use Restricted Operation Mode.
 */
static inline void
MCAN_setListenOnlyMode(uint32_t base, MCAN_ListenOnlyMode_t mode)
{
	switch (mode) {
		case MCAN_LISTEN_ONLY_MODE_DISABLE :
			HWREG(base + M_CAN_CCCR) &= ~(M_CAN_CCCR_MON_M);
			break;

		case MCAN_LISTEN_ONLY_MON_MODE :
			HWREG(base + M_CAN_CCCR) |= M_CAN_CCCR_MON_M;
			break;

		default :
			break;
	}
}

static inline bool
MCAN_setRestrictedOperationMode(uint32_t base, bool enable)
{
	if (enable) {
		/* The Restricted Operation Mode must not be combined
		with the Loop Back Mode (internalor external). */
		if (HWREG(base + M_CAN_TEST) & M_CAN_TEST_LBCK_M)
			return false;

		HWREG(base + M_CAN_CCCR) |= M_CAN_CCCR_ASM_M;
	} else
		HWREG(base + M_CAN_CCCR) &= ~M_CAN_CCCR_ASM_M;

	return true;
}

static inline void
MCAN_setBusMonitoringMode(uint32_t base, bool enable)
{
	if (enable)
		HWREG(base + M_CAN_CCCR) |= M_CAN_CCCR_MON_M;
	else
		HWREG(base + M_CAN_CCCR) &= ~M_CAN_CCCR_MON_M;
}

/**
 * @brief Enables the use of 16-bit Wide Message Markers.
 * When 16-bit Wide Message Markers are used(WMM = '1'),
 * 16-bit internal timestamp is disabled for the Tx Event FIFO.
 *
 * @param base indicates that the MCAN instance base address.
 * @param mode Select 16bit/8bit Message Marker mode.
 */
static inline void
MCAN_setWideMessageMarker(uint32_t base, MCAN_WideMessageMarker_t mode)
{
	if (MCAN_WMM_8BIT_MODE == mode)
		HWREG(base + M_CAN_CCCR) &= ~M_CAN_CCCR_WMM_M;
	else
		HWREG(base + M_CAN_CCCR) |= M_CAN_CCCR_WMM_M;
}

/**
 * @brief Configure timestamp counter.
 *
 * @param base indicates that the MCAN instance base address.
 * @param timestamp indicates that timestamp counter initialization parameters.
 */
static inline void
MCAN_setTimestampCounter(uint32_t base,
									MCAN_TimestampClk_t tsClock,
									uint8_t tsPrescalar,
									MCAN_TimestampSource_t tsSelect)
{
	/* Select the MCAN timestamp souce from internal clock or external clock. */
	if (MCAN_INTERNAL_TIMESTAMP == tsClock)
		HWREG(base + M_CAN_CCCR) &= ~M_CAN_CCCR_UTSU_M;
	else
		HWREG(base + M_CAN_CCCR) |= M_CAN_CCCR_UTSU_M;

	/* Configure timestamp counter perscaler. */
	HWREG(base + M_CAN_TSCC) |= (((uint32_t)tsPrescalar &
								(M_CAN_TSCC_TCP_M >> M_CAN_TSCC_TCP_S)) << M_CAN_TSCC_TCP_S);
	HWREG(base + M_CAN_TSCC) |= (tsSelect & M_CAN_TSCC_TSS_M);
}

/**
 * @brief Configure the base address and size of standard identifier filter.
 *
 * @param base indicates that the MCAN instance base address.
 * @param lss Standard ID filter size.
 * @param flssa Standard ID filter base address.
 */
static inline void
MCAN_setStdIDFilter(uint32_t base, uint8_t lss, uint16_t flssa)
{
	uint32_t regval;

	/* Configure the size of standard identifier filter. */
	regval = ((uint32_t)lss & (M_CAN_SIDFC_LSS_M >> M_CAN_SIDFC_LSS_S)) << M_CAN_SIDFC_LSS_S;
	/* Configure the base address of standard identifier filter. */
	regval |= (((uint32_t)flssa & (M_CAN_SIDFC_FLSSA_M >> M_CAN_SIDFC_FLSSA_S)) << M_CAN_SIDFC_FLSSA_S);

	HWREG(base + M_CAN_SIDFC) = regval;
}

/**
 * @brief Get the standard ID filter size.
 *
 * @param base indicates that the MCAN instance base address.
 * @return uint8_t Return value of the standard ID filter size.(1-128);
 */
static inline uint8_t
MCAN_getStdFilterSize(uint32_t base)
{
	return (uint8_t)((HWREG(base + M_CAN_SIDFC) & M_CAN_SIDFC_LSS_M) >> M_CAN_SIDFC_LSS_S);
}

/**
 * @brief Get the standard ID filter base address.
 *
 * @param base indicates that the MCAN instance base address.
 * @return uint32_t Return value of the standard ID filter base address.
 */
static inline uint32_t
MCAN_getStdFilterBaseAddress(uint32_t base)
{
	return (HWREG(base + M_CAN_SIDFC) & M_CAN_SIDFC_FLSSA_M);
}

/**
 * @brief Get the extended ID filter size.
 *
 * @param base indicates that the MCAN instance base address.
 * @return uint8_t Return value of the extended ID filter size.(1-128);
 */
static inline uint8_t
MCAN_getExtFilterSize(uint32_t base)
{
	return (uint8_t)((HWREG(base + M_CAN_XIDFC) & M_CAN_XIDFC_LSE_M) >> M_CAN_XIDFC_LSE_S);
}

/**
 * @brief Get the extended ID filter base address.
 *
 * @param base indicates that the MCAN instance base address.
 * @return uint32_t Return value of the extended ID filter base address.
 */
static inline uint32_t
MCAN_getExtFilterBaseAddress(uint32_t base)
{
	return (HWREG(base + M_CAN_XIDFC) & M_CAN_XIDFC_FLESA_M);
}

/**
 * @brief Configure the base address and size of extended identifier filter.
 *
 * @param base indicates that the MCAN instance base address.
 * @param lse Extended ID filter size.
 * @param flesa Extended ID filter base address.
 */
static inline void
MCAN_setExtIDFilter(uint32_t base, uint8_t lse, uint16_t flesa)
{
	uint32_t regval;

	/* Configure the size of extended identifier filter. */
	regval = ((uint32_t)lse & (M_CAN_XIDFC_LSE_M >> M_CAN_XIDFC_LSE_S)) << M_CAN_XIDFC_LSE_S;
	/* Configure the base address of extended identifier filter. */
	regval |= ((uint32_t)flesa & (M_CAN_XIDFC_FLESA_M >> M_CAN_XIDFC_FLESA_S));
	HWREG(base + M_CAN_XIDFC) = regval;
}

static inline bool
MCAN_setExtIDAndMask(uint32_t base, uint32_t mask)
{
	if (!MCAN_isBaseValid(base))
		return false;

	/* Unlock protected registers. */
	if (!MCAN_writeProtectedRegAccessUnlock(base))
		return false;

	HWREG(base + M_CAN_XIDAM) = mask & M_CAN_XIDAM_EIDM_M;

	/* Lock protected registers. */
	if (!MCAN_writeProtectedRegAccessLock(base))
		return false;

	return true;
}

/**
 * @brief Initialize Rx buffer.
 *
 * @param base indicates that the MCAN instance base address.
 * @param rxbuf indicates that the Rx buffer initialization parameters.
 */
static inline void
MCAN_setRxBuffer(uint32_t base, uint16_t rxBufStartAddr)
{
	HWREG(base + M_CAN_RXBC) = rxBufStartAddr & M_CAN_RXBC_RBSA_M;
}

static inline bool
MCAN_addClockStopRequest(uint32_t base, bool enable)
{
	if (!MCAN_isBaseValid(base))
		return false;

	if (enable)
		HWREG(base + M_CAN_CCCR) |= M_CAN_CCCR_CSR_M;
	else
		HWREG(base + M_CAN_CCCR) &= ~M_CAN_CCCR_CSR_M;

	return true;
}

static inline bool
MCAN_DeInit(uint32_t base, int32_t timeout)
{
	if (!MCAN_isBaseValid(base))
		return 0;

	if (!MCAN_addClockStopRequest(base, true))
		return false;

	return true;
}

static inline uint32_t
MCAN_getTSCounterVal(uint32_t base)
{
	if (!MCAN_isBaseValid(base))
		return 0;

	return HWREG(base + M_CAN_TSCV) & M_CAN_TSCV_TSC_M;
}

static inline uint32_t
MCAN_getTOCounterVal(uint32_t base)
{
	if (!MCAN_isBaseValid(base))
		return false;

	return HWREG(base + M_CAN_TOCV) & M_CAN_TOCV_TOC_M;
}

static inline bool
MCAN_enableIntr(uint32_t base, uint32_t intmask, bool enable)
{
	if (!MCAN_isBaseValid(base))
		return false;

	/* Enable MCAN interrupt. */
	if (enable)
		HWREG(base + M_CAN_IE) |= intmask;
	else
		HWREG(base + M_CAN_IE) &= ~intmask;

	return true;
}

static inline uint32_t
MCAN_getIntrStatus(uint32_t base)
{
	if (!MCAN_isBaseValid(base))
		return 0;

	return HWREG(base + M_CAN_IR);
}

static inline bool
MCAN_clearIntrStatus(uint32_t base, uint32_t intrMask)
{
	if (!MCAN_isBaseValid(base))
			return false;

	HWREG(base + M_CAN_IR) = intrMask;

	return true;
}

static inline bool
MCAN_selectIntrLine(uint32_t base, uint32_t intmask, MCAN_InterruptLine_t line)
{
	if (!MCAN_isBaseValid(base))
		return false;

	/* Select the MCAN interrupt link to line 0 or 1. */
	if (line == MCAN_INTERRUPT_LINE_0)
		HWREG(base + M_CAN_ILS) &= ~intmask;
	else if (line == MCAN_INTERRUPT_LINE_1)
		HWREG(base + M_CAN_ILS) |= intmask;
	else
		return false;

	return true;
}

static inline bool
MCAN_txBufTransIntrEnable(uint32_t base, uint32_t buf_num, bool enable)
{
	if (!MCAN_isBaseValid(base))
		return false;

	if (enable)
		HWREG(base + M_CAN_TXBTIE) |= (1U << buf_num);
	else
		HWREG(base + M_CAN_TXBTIE) &= ~(1U << buf_num);

	return true;
}

static inline uint32_t
MCAN_getIntrLineSelectStatus(uint32_t base)
{
	if (!MCAN_isBaseValid(base))
		return 0;

	return (HWREG(base + M_CAN_ILS));
}

static inline bool
MCAN_getNewDataStatus(uint32_t base, MCAN_RxNewDataStatus_t *newdat)
{
	if (!MCAN_isBaseValid(base))
		return false;

	if (newdat == NULL)
		return false;

	newdat->statusLow = HWREG(base + M_CAN_NDAT1);
	newdat->statusHigh = HWREG(base + M_CAN_NDAT2);

	return true;
}

static inline bool
MCAN_clearNewDataStatus(uint32_t base, MCAN_RxNewDataStatus_t *newdat)
{
	if (!MCAN_isBaseValid(base))
		return false;

	if (newdat == NULL)
		return false;

	HWREG(base + M_CAN_NDAT1) = newdat->statusLow;
	HWREG(base + M_CAN_NDAT1) = newdat->statusHigh;

	return true;
}

static inline bool
MCAN_txBufAddReq(uint32_t base, uint32_t txbuf_num)
{
	if (!MCAN_isBaseValid(base))
		return false;

	/* Request to start transmission of message. */
	HWREG(base + M_CAN_TXBAR) = (1U << txbuf_num);

	return true;
}

static inline bool
MCAN_getTxFIFOQueStatus(uint32_t base, MCAN_TxFIFOStatus_t *fifoStatus)
{
	if (!MCAN_isBaseValid(base))
		return false;

	if (fifoStatus == NULL)
		return false;

	fifoStatus->txfifo_stat = HWREG(base + M_CAN_TXFQS);

	return true;
}

static inline bool
MCAN_writeRxFIFOAck(uint32_t base, MCAN_RxFIFONum_t fifoNum, uint32_t idx)
{
	if (!MCAN_isBaseValid(base))
		return false;

	if (fifoNum == MCAN_RX_FIFO_NUM_0)
		HWREG(base + M_CAN_RXF0A) = idx;
	else if (fifoNum == MCAN_RX_FIFO_NUM_1)
		HWREG(base + M_CAN_RXF1A) = idx;

	return true;
}

static inline uint32_t
MCAN_getTxBufReqPend(uint32_t base)
{
	if (!MCAN_isBaseValid(base))
		return 0;

	return (HWREG(base + M_CAN_TXBRP));
}

static inline bool
MCAN_txBufCancellationReq(uint32_t base, uint8_t idx)
{
	if (!MCAN_isBaseValid(base))
		return false;

	if (idx >= MCAN_TX_BUFFER_MAX)
		return false;

	HWREG(base + M_CAN_TXBCR) = (1U << idx);

	return true;
}

static inline uint32_t
MCAN_getTxBufTransmissionStatus(uint32_t base)
{
	if (!MCAN_isBaseValid(base))
		return 0;

	return HWREG(base + M_CAN_TXBTO);
}

static inline bool
MCAN_getTxEventFIFOStatus(uint32_t base, MCAN_TxEventFIFOStatus_t *fifoStatus)
{
	if (!MCAN_isBaseValid(base))
		return false;

	if (fifoStatus == NULL)
		return false;

	*(uint32_t *)fifoStatus = HWREG(base + M_CAN_TXEFS);

	return true;
}

static inline int
MCAN_getMsgObjSize(MCAN_ElemSize_t size)
{
	if (size > MCAN_ELEM_SIZE_64BYTES)
		return -1;

	return MCAN_DataPhaseSizeArray[size];
}

static inline int
MCAN_getDataSize(MCAN_MsgDataLength_t len)
{
	if (len > MCAN_DATA_LENGTH_64)
		return -1;

	return MCAN_DataLengthArray[len];
}

static inline bool
MCAN_getErrCounters(uint32_t base, MCAN_ErrCntStatus_t *errCounter)
{
	if (!MCAN_isBaseValid(base))
		return false;

	errCounter->ecr_val = HWREG(base + M_CAN_ECR);

	return true;
}

static inline bool
MCAN_getProtocolStatus(uint32_t base, MCAN_ProtocolStatus_t *protocolStatus)
{
	if (!MCAN_isBaseValid(base))
		return false;

	protocolStatus->psr_val = HWREG(base + M_CAN_PSR);

	return true;
}

/*****************************************************************************
 *
 *							MCAN System Configuration
 *
 * ***************************************************************************/

static inline bool
MCAN_getRevisionId(uint32_t base, MCAN_RevisionId_t *revId)
{
	uint32_t regval;

	if (!MCAN_isBaseValid(base))
		return false;

	revId->crel_val = HWREG(base + M_CAN_CREL);

	base = MCAN_SSREG_ADDR(base);

	revId->pid_val = HWREG(base + MCANSS_PID);

	return true;
}

static inline bool
MCAN_extTSCounterEnable(uint32_t base, bool enable)
{
	if (!MCAN_isBaseValid(base))
		return false;

	base = MCAN_SSREG_ADDR(base);

	if (enable)
		HWREG(base + MCANSS_CTRL) |= MCANSS_CTRL_EXT_TS_CNTR_EN_M;
	else
		HWREG(base + MCANSS_CTRL) &= ~MCANSS_CTRL_EXT_TS_CNTR_EN_M;

	return true;
}

static inline bool
MCAN_extTSCounterConfig(uint32_t base, uint32_t prescalar)
{
	if (!MCAN_isBaseValid(base))
		return false;

	if (prescalar >= 0xFFFFFF)
		return false;

	base = MCAN_SSREG_ADDR(base);

	HWREG(base + MCANSS_EXT_TS_PRESCALER) = prescalar;

	return true;
}

static inline uint32_t
MCAN_extTSGetUnservicedIntrCount(uint32_t base)
{
	if (!MCAN_isBaseValid(base))
		return 0;

	base = MCAN_SSREG_ADDR(base);

	return (HWREG(base + MCANSS_EXT_TS_UNSERVIED_INTR_CNTR) &
			MCANSS_EXT_TS_UNSERVIED_INTR_CNTR_EXT_TS_INTR_CNTR_M);
}

static inline bool
MCAN_extTSClearOverflowIntrCount(uint32_t base, uint32_t count_val)
{
	if (!MCAN_isBaseValid(base))
		return false;

	base = MCAN_SSREG_ADDR(base);

	HWREG(base + MCANSS_EXT_TS_UNSERVIED_INTR_CNTR) = count_val;

	return true;
}

static inline bool
MCAN_extTSClearRawStatus(uint32_t base)
{
	if (!MCAN_isBaseValid(base))
		return false;

	base = MCAN_SSREG_ADDR(base);

	HWREG(base + MCANSS_ICS) = MCANSS_ICS_EXT_TS_CNTR_OVFL_M;

	return true;
}

static inline bool
MCAN_extTSWriteEOI(uint32_t base)
{
	if (!MCAN_isBaseValid(base))
		return false;

	base = MCAN_SSREG_ADDR(base);

	HWREG(base + MCANSS_EOI) = 0;

	return true;
}

static inline bool
MCAN_extTSSetRawStatus(uint32_t base)
{
	if (!MCAN_isBaseValid(base))
		return false;

	base = MCAN_SSREG_ADDR(base);

	HWREG(base + MCANSS_IRS) = MCANSS_IRS_EXT_TS_CNTR_OVFL_M;

	return true;
}

static inline bool
MCAN_extTSEnableIntr(uint32_t base, bool enable)
{
	if (!MCAN_isBaseValid(base))
		return false;

	base = MCAN_SSREG_ADDR(base);

	if (enable)
		HWREG(base + MCANSS_IE) = MCANSS_IE_EXT_TS_CNTR_OVFL_M;
	else
		HWREG(base + MCANSS_IECS) = MCANSS_ICS_EXT_TS_CNTR_OVFL_M;

	return true;
}

static inline bool
MCAN_extTSIsIntrEnable(uint32_t base)
{
	if (!MCAN_isBaseValid(base))
		return false;

	base = MCAN_SSREG_ADDR(base);

	return (HWREG(base + MCANSS_IES) & MCANSS_IES_EXT_TS_CNTR_OVFL_M)? true : false;
}

static inline bool
MCAN_enableWakupReq(uint32_t base, bool enable)
{
	if (!MCAN_isBaseValid(base))
		return false;

	base = MCAN_SSREG_ADDR(base);

	if (enable)
		HWREG(base + MCANSS_CTRL) |= MCANSS_CTRL_WAKEUPREQEN_M;
	else
		HWREG(base + MCANSS_CTRL) &= ~MCANSS_CTRL_WAKEUPREQEN_M;

	return true;
}

static inline bool
MCAN_enableAutoWakeup(uint32_t base, bool enable)
{
	if (!MCAN_isBaseValid(base))
		return false;

	base = MCAN_SSREG_ADDR(base);

	if (enable)
		HWREG(base + MCANSS_CTRL) |= MCANSS_CTRL_AUTOWAKEUP_M;
	else
		HWREG(base + MCANSS_CTRL) &= ~MCANSS_CTRL_AUTOWAKEUP_M;

	return true;
}

static inline bool
MCAN_enableDebugFree(uint32_t base, bool enable)
{
	if (!MCAN_isBaseValid(base))
		return false;

	base = MCAN_SSREG_ADDR(base);

	if (enable)
		HWREG(base + MCANSS_CTRL) |= MCANSS_CTRL_DBGSUSP_FREE_M;
	else
		HWREG(base + MCANSS_CTRL) &= ~MCANSS_CTRL_DBGSUSP_FREE_M;

	return true;
}

static inline bool
MCAN_isInReset(uint32_t base)
{
	if (!MCAN_isBaseValid(base))
			return false;

	base = MCAN_SSREG_ADDR(base);

	if (HWREG(base + MCANSS_STAT) & MCANSS_STAT_RESET_M)
		return true;
	else
		return false;
}

static inline bool
MCAN_isFDOpEnable(uint32_t base)
{
	if (!MCAN_isBaseValid(base))
		return false;

	base = MCAN_SSREG_ADDR(base);

	if (HWREG(base + MCANSS_STAT) & MCANSS_STAT_ENABLE_FDOE_M)
		return true;
	else
		return false;
}

static inline bool
MCAN_isMemInitDone(uint32_t base)
{
	if (!MCAN_isBaseValid(base))
		return false;

	base = MCAN_SSREG_ADDR(base);

	if (HWREG(base + MCANSS_STAT) & MCANSS_STAT_MEM_INIT_DONE_M)
		return true;
	else
		return false;
}

static inline bool
MCAN_eccAggrGetRevisionId(uint32_t base, MCAN_ECCAggrRevisionId_t *revId)
{
	if (!MCAN_isBaseValid(base))
		return false;

	base = MCAN_ERRSRPT_ADDR(base);

	revId->rev_val = HWREG(base + MCANERR_REV);

	return true;
}

static inline uint32_t
MCAN_eccAggrGetRamNumber(uint32_t base)
{
	if (!MCAN_isBaseValid(base))
		return 0;

	base = MCAN_ERRSRPT_ADDR(base);

	return (HWREG(base + MCANERR_STAT) & MCANERR_STAT_NUM_RAMS_M);
}

static inline bool
MCAN_eccWrapGetRevisionId(uint32_t base, MCAN_ECCAggrRevisionId_t *revId)
{
	if (!MCAN_isBaseValid(base))
		return false;

	base = MCAN_ERRSRPT_ADDR(base);

	revId->rev_val = HWREG(base + MCANERR_WRAP_REV);

	return true;
}

/*****************************************************************************
 *
 *							MCAN ECC Configuration
 *
 * ***************************************************************************/

static inline bool
MCAN_eccConfig(uint32_t base, MCAN_ECCConfigParams_t *configParams)
{
	uint32_t regval;

	if (!MCAN_isBaseValid(base))
		return false;

	base = MCAN_ERRSRPT_ADDR(base);

	regval = HWREG(base + MCANERR_CTRL);

	if (configParams->enable)
		regval |= MCANERR_CTRL_ECC_ENABLE_M;
	else
		regval &= ~MCANERR_CTRL_ECC_ENABLE_M;

	if (configParams->enableChk)
		regval |= MCANERR_CTRL_ECC_CHECK_M;
	else
		regval &= ~MCANERR_CTRL_ECC_CHECK_M;

	HWREG(base + MCANERR_CTRL) = regval;

	return true;
}

static inline bool
MCAN_eccGetErrorStatus(uint32_t base, MCAN_ECCErrStatus_t *eccErr)
{
	uint32_t regval;

	if (!MCAN_isBaseValid(base))
		return false;

	base = MCAN_ERRSRPT_ADDR(base);

	regval = HWREG(base + MCANERR_ERR_STAT1);

	eccErr->bit1 = regval & MCANERR_ERR_STAT1_ECC_BIT1_M;
	eccErr->dedErr = (regval & MCANERR_ERR_STAT1_ECC_DED_M) >> MCANERR_ERR_STAT1_ECC_DED_S;
	eccErr->secErr = regval & MCANERR_ERR_STAT1_ECC_SEC_M;
	eccErr->row = HWREG(base + MCANERR_ERR_STAT2);

	return true;
}

static inline bool
MCAN_clearEccError(uint32_t base, MCAN_ECCErrType_t err_type, uint32_t val)
{
	if (!MCAN_isBaseValid(base))
		return false;

	base = MCAN_ERRSRPT_ADDR(base);

	if (err_type == MCAN_ECC_ERR_TYPE_SEC)
		HWREG(base + MCANERR_ERR_STAT1) = ((val &
										(MCANERR_ERR_STAT1_CLR_ECC_SEC_M >> MCANERR_ERR_STAT1_CLR_ECC_SEC_S)) <<
										MCANERR_ERR_STAT1_CLR_ECC_SEC_S);
	else if (err_type == MCAN_ECC_ERR_TYPE_DED)
		HWREG(base + MCANERR_ERR_STAT1) = ((val &
										(MCANERR_ERR_STAT1_CLR_ECC_DED_M >> MCANERR_ERR_STAT1_CLR_ECC_DED_S)) <<
										MCANERR_ERR_STAT1_CLR_ECC_DED_S);
	else
		return false;

	return true;
}

static inline bool
MCAN_writeEccError(uint32_t base, MCAN_ECCErrType_t err_type, uint32_t val)
{
	if (!MCAN_isBaseValid(base))
		return false;

	base = MCAN_ERRSRPT_ADDR(base);

	if (err_type == MCAN_ECC_ERR_TYPE_DED)
		HWREG(base + MCANERR_ERR_STAT1) = ((val & (MCANERR_ERR_STAT1_ECC_DED_M >> MCANERR_ERR_STAT1_ECC_DED_S))
											<< MCANERR_ERR_STAT1_ECC_DED_S);
	else if (err_type == MCAN_ECC_ERR_TYPE_SEC)
		HWREG(base + MCANERR_ERR_STAT1) = val & MCANERR_ERR_STAT1_ECC_SEC_M;
	else
		return false;

	return true;
}

static inline bool
MCAN_eccForceError(uint32_t base, MCAN_ECCErrForceParams_t *eccErr)
{
	uint32_t regval;

	if (!MCAN_isBaseValid(base))
		return false;

	base = MCAN_ERRSRPT_ADDR(base);

	regval = HWREG(base + MCANERR_CTRL);

	if (eccErr->errType == MCAN_ECC_ERR_TYPE_SEC) {
		regval |= MCANERR_CTRL_FORCE_SEC_M;
	} else if (eccErr->errType == MCAN_ECC_ERR_TYPE_DED) {
		regval |= MCANERR_CTRL_FORCE_DED_M;
	} else
		return false;

	HWREG(base + MCANERR_CTRL) = regval;

	return true;
}

static inline bool
MCAN_eccClearForceError(uint32_t base, MCAN_ECCErrForceParams_t *eccErr)
{
	uint32_t regval;

	if (!MCAN_isBaseValid(base))
		return false;

	base = MCAN_ERRSRPT_ADDR(base);

	regval = HWREG(base + MCANERR_CTRL);

	if (eccErr->errType == MCAN_ECC_ERR_TYPE_SEC) {
		regval &= ~MCANERR_CTRL_FORCE_SEC_M;
	} else if (eccErr->errType == MCAN_ECC_ERR_TYPE_DED) {
		regval &= ~MCANERR_CTRL_FORCE_DED_M;
	} else
		return false;

	HWREG(base + MCANERR_CTRL) = regval;

	return true;
}

static inline bool
MCAN_eccGetIntrStatus(uint32_t base, MCAN_ECCErrType_t err_type)
{
	if (!MCAN_isBaseValid(base))
		return false;

	base = MCAN_ERRSRPT_ADDR(base);

	if (err_type == MCAN_ECC_ERR_TYPE_SEC)
		return ((HWREG(base + MCANERR_SEC_STATUS) & MCANERR_SEC_STATUS_MSGMEM_PEND_M) ? true : false);
	else if (err_type == MCAN_ECC_ERR_TYPE_DED)
		return ((HWREG(base + MCANERR_DED_STATUS) & MCANERR_DED_STATUS_MSGMEM_PEND_M) ? true : false);
	else
		return false;
}

static inline bool
MCAN_eccClearIntrStatus(uint32_t base, MCAN_ECCErrType_t err_type)
{
	if (!MCAN_isBaseValid(base))
		return false;

	base = MCAN_ERRSRPT_ADDR(base);

	if (err_type == MCAN_ECC_ERR_TYPE_SEC)
		HWREG(base + MCANERR_SEC_STATUS) = MCANERR_SEC_STATUS_MSGMEM_PEND_M;
	else if (err_type == MCAN_ECC_ERR_TYPE_DED)
		HWREG(base + MCANERR_DED_STATUS) = MCANERR_DED_STATUS_MSGMEM_PEND_M;
	else
		return false;

	return true;
}

static inline bool
MCAN_eccEnableIntr(uint32_t base, MCAN_ECCErrType_t err_type, bool enable)
{
	if (!MCAN_isBaseValid(base))
		return false;

	base = MCAN_ERRSRPT_ADDR(base);

	if (enable) {
		if (err_type == MCAN_ECC_ERR_TYPE_SEC)
			HWREG(base + MCANERR_SEC_ENABLE_SET) = MCANERR_SEC_ENABLE_SET_MSGMEM_ENABLE_SET_M;
		else if (err_type == MCAN_ECC_ERR_TYPE_DED)
			HWREG(base + MCANERR_DED_ENABLE_SET) = MCANERR_DED_ENABLE_SET_MSGMEM_ENABLE_SET_M;
		else
			return false;
	} else {
		if (err_type == MCAN_ECC_ERR_TYPE_SEC)
			HWREG(base + MCANERR_SEC_ENABLE_CLR) = MCANERR_SEC_ENABLE_CLR_MSGMEM_ENABLE_CLR_M;
		else if (err_type == MCAN_ECC_ERR_TYPE_DED)
			HWREG(base + MCANERR_DED_ENABLE_CLR) = MCANERR_DED_ENABLE_CLR_MSGMEM_ENABLE_CLR_M;
		else
			return false;
	}

	return true;
}

static inline bool
MCAN_eccWriteEOI(uint32_t base, MCAN_ECCErrType_t err_type)
{
	if (!MCAN_isBaseValid(base))
		return false;

	base = MCAN_ERRSRPT_ADDR(base);

	if (err_type == MCAN_ECC_ERR_TYPE_SEC)
		HWREG(base + MCANERR_SEC_EOI) = MCANERR_SEC_EOI_EOI_WR_M;
	else if (err_type == MCAN_ECC_ERR_TYPE_DED)
		HWREG(base + MCANERR_DED_EOI) = MCANERR_DED_EOI_EOI_WR_M;
	else
		return false;

	return true;
}

/**
 * @brief Set the MCAN to enter normal mode ro disable mode.
 *
 * @param base indicates that the MCAN instance base address.
 * @param op_mode indicates that the MCAN instance base address.
 * @return true indicates that MCAN has successfully set the operation mode.
 * @return false indicates that MCAN has unsuccessfully set the operation mode.
 */
bool MCAN_setOpMode(uint32_t base, MCAN_OperationMode_t op_mode);

/**
 * @brief Get the MCAN operation mode.
 *
 * @param base indicates that the MCAN instance base address.
 * @param op_mode indicates that the MCAN operation mode value.
 * @return true indicates that MCAN has successfully get the operation mode.
 * @return false indicates that MCAN has unsuccessfully get the operation mode.
 */
bool MCAN_getOpMode(uint32_t base, MCAN_OperationMode_t *op_mode);

/**
 * @brief Unlock write access to MCAN protected registers.
 *
 * @param base indicates that the MCAN instance base address.
 * @return true indicates that the protected registers has been unlocked.
 * @return false indicates that the protected registers is not unlocked.
 */
bool MCAN_writeProtectedRegAccessUnlock(uint32_t base);

/**
 * @brief Lock write access to MCAN protected registers.
 *
 * @param base indicates that the MCAN instance base address.
 * @return true indicates that the protected registers has been locked.
 * @return false indicates that the protected registers is not locked.
 */
bool MCAN_writeProtectedRegAccessLock(uint32_t base);

/**
 * @brief Add a new MCAN standard ID filter.
 *
 * @param base indicates that the MCAN instance base address.
 * @param filter_number Standard ID filter number.
 * @param std_filter Initialize the initialization parameters of standard ID filter.
 * @return true Successfully configure standard ID filter.
 * @return false Unsuccessfully configure standard ID filter.
 */
bool MCAN_addStdMsgIDFilter(uint32_t base,
							uint8_t filter_number,
							const MCAN_StdMsgIDFilterElement_t *std_filter);

/**
 * @brief Add a new MCAN Extended ID filter.
 *
 * @param base indicates that the MCAN instance base address.
 * @param filter_number Extended ID filter number.
 * @param ext_filter Initialize the initialization parameters of Extended ID filter.
 * @return true Successfully configure Extended ID filter.
 * @return false Unsuccessfully configure Extended ID filter.
 */
bool MCAN_addExtMsgIDFilter(uint32_t base,
							uint8_t filter_number,
							const MCAN_ExtMsgIDFilterElement_t *ext_filter);

/**
 * @brief Set MCAN extended ID and mask.
 *
 * @param base indicates that the MCAN instance base address.
 * @param mask Extended ID mask value.
 * For acceptance filtering of extended frames the Extended ID AND Mask
 * is ANDed with the Message ID of a received frame. Intended for
 * masking of 29-bit IDs in SAE J1939. With the reset value of all bits
 * set to one the mask is not active.
 */
bool MCAN_setExtIDAndMask(uint32_t base, uint32_t mask);

/**
 * @brief Initialize Rx fifo/buffer and Tx buffer/event.
 *
 * @param base indicates that the MCAN instance base address.
 * @param msg_buffer indicates that the message buffer initialization parameters.
 * @return true All message buffers initialized successfully.
 * @return false message buffers is not initialized.
 */
bool MCAN_msgRAMConfig(uint32_t base, MCAN_MsgRAMConfigParams_t *RAMConfig);

/**
 * @brief This function will initialize the MCAN module based on user configuration.
 *
 * @param base indicates that the MCAN instance base address.
 * @param initparams MCAN initialization configuration parameters.
 * @return true MCAN has been successfully initialized.
 * @return false MCAN initialization failed.
 */
bool MCAN_init(uint32_t base, MCAN_InitParams_t *initparams);

/**
 * @brief Release the MCAN.
 *
 * @param base indicates that the MCAN instance base address.
 * @return true indicates that MCAN has been released.
 * @return false indicates that MCAN has not been released.
 */
bool MCAN_DeInit(uint32_t base, int32_t timeout);

/**
 * @brief Configure MCAN module.
 *
 * @param base indicates that the MCAN instance base address.
 * @param configParams configuration parameters.(@MCAN_ConfigParams_t)
 * @return true M_CAN configure done.
 * @return false M_CAN configure error.
 */
bool MCAN_config(uint32_t base, MCAN_ConfigParams_t *configParams);

/**
 * @brief Set normal and data phase bitrate.
 *
 * @param base indicates that the MCAN instance base address.
 * @param configBitrate Bitrate configuration parameters.
 * Entering the subfunction allows you to view the specific bit rate calculation method.
 * @return true Bitrate configuration done.
 * @return false Bitrate configuration error.
 */
bool MCAN_setBitTime(uint32_t base, const MCAN_BitTimingParams_t *configBitrate);

/**
 * @brief Set the MCAN enter internal or external lookback mode.
 * NOTE: Loopback mode cannot be used in conjunction with listening only mode.
 *
 * @param base indicates that the MCAN instance base address.
 * @param mode Select internal/external loopback mode.
 */
bool MCAN_lpbkModeEnable(uint32_t base, MCAN_LpbkMode_t mode, bool enable);

/**
 * @brief Request stop MCAN clock.
 *
 * @param base indicates that the MCAN instance base address.
 * @param enable true is enable stop clock request, else disable.
 * @return true indicates enable stop clock request.
 * @return false base address input error.
 */
bool MCAN_addClockStopRequest(uint32_t base, bool enable);

/**
 * @brief Get timestamp counter value.
 *
 * @param base indicates that the MCAN instance base address.
 * @return non-zero Successfully read timestamp counter value.
 * @return 0 Unsuccessfully read timestamp counter value.
 */
uint32_t MCAN_getTSCounterVal(uint32_t base);

/**
 * @brief Get timeout counter value.
 *
 * @param base indicates that the MCAN instance base address.
 * @return non-zero Successfully read timeout counter value.
 * @return 0 Unsuccessfully read timeout counter value.
 */
uint32_t MCAN_getTOCounterVal(uint32_t base);

/**
 * @brief Enable MCAN interrupt.
 *
 * @param base indicates that the MCAN instance base address.
 * @param intmask MCAN interrupt source bit mask.(options:@MCAN_InterruptSource_t).
 * @param enable true is enable interrupt, else disable.
 * @return true MCAN interrupt is enabled.
 * @return false Failed to enable MCAN interrupt.
 */
bool MCAN_enableIntr(uint32_t base, uint32_t intmask, bool enable);

/**
 * @brief Get MCAN interrupt status.
 *
 * @param base indicates that the MCAN instance base address.
 * @return non-zero If the bit is set,it indicates that the interrupt has been triggered.
 * (options:@MCAN_InterruptSource_t)
 * @return 0 is No interrupt generated or incorrect input address.
 */
uint32_t MCAN_getIntrStatus(uint32_t base);

/**
 * @brief Clear MCAN interrupt status.
 *
 * @param base indicates that the MCAN instance base address.
 * @param intrMask indicates that need to clear interrupt status bit.
 * @return true Clear interrupt status successfully.
 * @return false Clear interrupt status failed.
 */
bool MCAN_clearIntrStatus(uint32_t base, uint32_t intrMask);

/**
 * @brief Choose which interrupt line to route MCAN interrupts to.
 *
 * @param base indicates that the MCAN instance base address.
 * @param intmask MCAN interrupt source bit mask.(options:@MCAN_InterruptSource_t).
 * @param line MCAN interrupt line.(options:@MCAN_InterruptLine_t).
 * @return true Set interrupt route success.
 * @return false Input parameters error.
 */
bool MCAN_selectIntrLine(uint32_t base, uint32_t intmask, MCAN_InterruptLine_t line);

/**
 * @brief Enable/Disable MCAN interrupt line.
 *
 * @param base indicates that the MCAN instance base address.
 * @param line MCAN interrupt line.(options:@MCAN_InterruptLine_t).
 * @param enable true is enable line, else disable.
 * @return true Successfully enbale MCAN interrupt line.
 * @return false Unsuccessfully enbale MCAN interrupt line.
 */
bool MCAN_enableIntrLine(uint32_t base, MCAN_InterruptLine_t line, bool enable);

/**
 * @brief Enable/Disable MCAN interrupt line.
 *
 * @param base indicates that the MCAN instance base address.
 * @param buf_num MCAN Transmission buffer number.
 * @param enable true is enable transmit interrupt, else disable.
 * @return true Successfully enbale MCAN transmit interrupt.
 * @return false Unsuccessfully enbale MCAN transmit interrupt.
 */
bool MCAN_txBufTransIntrEnable(uint32_t base, uint32_t buf_num, bool enable);

/**
 * @brief used to get interrupt line status selected for each interrupt.
 *
 * @param base indicates that the MCAN instance base address.
 * @return Interrupt Line Select Status.
 */
uint32_t MCAN_getIntrLineSelectStatus(uint32_t base);

/**
 * @brief Get status of the Rx buffer.
 *
 * @param base indicates that the MCAN instance base address.
 * @param newdat Pointing to the address space for save new data status(@MCAN_RxNewDataStatus_t).
 */
bool MCAN_getNewDataStatus(uint32_t base, MCAN_RxNewDataStatus_t *newdat);

/**
 * @brief clear New Data Message Status.
 *
 * @param base indicates that the MCAN instance base address.
 * @param newdat Clear the status of Rx buffer, high bit in newdat will not be cleared.
 */
bool MCAN_clearNewDataStatus(uint32_t base, MCAN_RxNewDataStatus_t *newdat);

/**
 * @brief Read message from Rx buffer.
 *
 * @param base indicates that the MCAN instance base address.
 * @param memType MCAN memory type.
 * @param bufNum Rx buffer index.
 * @param fifoNum Rx fifo index.
 * @param rxbuf Pointing to the address space for storing messages.
 * @return true Read message successfully from Rx buffer.
 * @return false Read message unsuccessfullyfrom Rx buffer.
 */
bool MCAN_readMsgRam(uint32_t base, MCAN_MemType_t memType,
					 uint32_t bufNum, MCAN_RxFIFONum_t fifoNum,
					 MCAN_RxMessage_t *rxbuf);

/**
 * @brief Receive message from Rx buffer.
 *
 * @param base indicates that the MCAN instance base address.
 * @param rxbuf Pointing to the address space for storing messages.
 * @param rxbuf_num Rx buffer index.
 * @return true Received message successfully from Rx buffer.
 * @return false Received message unsuccessfully from Rx buffer.
 */
bool MCAN_receiveMsgFromBuffer(uint32_t base, MCAN_RxMessage_t *rxbuf, uint8_t rxbuf_num);

/**
 * @brief Receive message from Rx fifo 0.
 *
 * @param base indicates that the MCAN instance base address.
 * @param rxbuf Pointing to the address space for storing messages.
 * @return true Received message successfully from Rx fifo 0.
 * @return false Received message unsuccessfully from Rx fifo 0.
 */
bool MCAN_receiveMsgFromFifo0(uint32_t base, MCAN_RxMessage_t *rxbuf);

/**
 * @brief Receive message from Rx fifo 1.
 *
 * @param base indicates that the MCAN instance base address.
 * @param rxbuf Pointing to the address space for storing messages.
 * @return true Received message successfully from Rx fifo 1.
 * @return false Received message unsuccessfully from Rx fifo 1.
 */
bool MCAN_receiveMsgFromFifo1(uint32_t base, MCAN_RxMessage_t *rxbuf);

/**
 * @brief Receive message with high priority.
 *
 * @param base indicates that the MCAN instance base address.
 * @param rxbuf Pointing to the address space for storing messages.
 * @return true Received message successfully with high priority.
 * @return false Received message unsuccessfully with high priority.
 */
bool MCAN_readHighPriorityMsgRam(uint32_t base, MCAN_RxMessage_t *rxbuf);

/**
 * @brief Using TX buffer to transmit messages.
 *
 * @param base indicates that the MCAN instance base address.
 * @param txbuf Pointing to the address space for storing messages.
 * @param txbuf_num Tx buffer index.(0 - maximum of Tx buffer).
 * @return true Successfully transmitted message.
 * @return false Unsuccessfully transmitted message.
 */
bool MCAN_transmitMsgBuffer(uint32_t base, MCAN_TxMessage_t *txbuf, uint8_t txbuf_num);

/**
 * @brief Using TX fifo/queue to transmit messages.
 * Tx FIFO mode sends according to write order,
 * while Tx queue mode sends according to ID priority.
 *
 * @param base indicates that the MCAN instance base address.
 * @param txbuf Pointing to the address space for storing messages.
 * @return true Successfully transmitted message.
 * @return false Unsuccessfully transmitted message.
 */
bool MCAN_transmitMsgFifoQueue(uint32_t base, MCAN_TxMessage_t *txbuf);

/**
 * @brief Write message to message RAM.
 *
 * @param base indicates that the MCAN instance base address.
 * @param memType indicates that the MCAN memory type.
 * @param txbuf_num Tx buffer index.
 * @param txbuf Pointing to the address space for storing messages.
 * @return true Successfully write message to message RAM.
 * @return false Unsuccessfully write message to message RAM.
 */
bool MCAN_writeMsgRam(uint32_t base, MCAN_MemType_t memType,
					  uint32_t txbuf_num, const MCAN_TxMessage_t *txbuf);

/**
 * @brief Add transmission request to Tx buffer.
 *
 * @param base indicates that the MCAN instance base address.
 * @param txbuf_num Tx buffer index.
 * @return true Successfully add transmission request to Tx buffer.
 * @return false Unsuccessfully add transmission request to Tx buffer.
 */
bool MCAN_txBufAddReq(uint32_t base, uint32_t txbuf_num);

/**
 * @brief Get Tx buffer pending status.
 *
 * @param base indicates that the MCAN instance base address.
 * @return non-zero Get Tx buffer pending status success.
 * @return 0 Tx Buffer is not pending or get pending status error.
 */
uint32_t MCAN_getTxBufReqPend(uint32_t base);

/**
 * @brief Get Tx fifo/queue status.
 *
 * @param base indicates that the MCAN instance base address.
 * @param fifoStatus Pointing to the address space for storing Tx fifo/queue status.
 * @return true Get Tx fifo/queue status success.
 * @return false Get Tx fifo/queue status error.
 */
bool MCAN_getTxFIFOQueStatus(uint32_t base, MCAN_TxFIFOStatus_t *fifoStatus);

/**
 * @brief Write Rx fifo Acknowledge Index.
 *
 * @param base indicates that the MCAN instance base address.
 * @param fifoNum Rx fifo number.
 * @param idx Rx fifo index.
 * @return true Write Rx fifo Acknowledge Index success.
 * @return false Write Rx fifo Acknowledge Index error.
 */
bool MCAN_writeRxFIFOAck(uint32_t base, MCAN_RxFIFONum_t fifoNum, uint32_t idx);

/**
 * @brief Get Rx fifo status.
 *
 * @param base indicates that the MCAN instance base address.
 * @param fifoStatus Pointing to the address space for storing Rx fifo status.
 * @return true Get Rx fifo status success.
 * @return false Get Rx fifo status error.
 */
bool MCAN_getRxFIFOStatus(uint32_t base, MCAN_RxFIFOStatus_t *fifoStatus);

/**
 * @brief Cancle transmission message request.
 *
 * @param base indicates that the MCAN instance base address.
 * @param idx Tx buffer index.
 * @return true Successfully cancellation of transmission message request.
 * @return false Unsuccessfully cancellation of transmission message request.
 */
bool MCAN_txBufCancellationReq(uint32_t base, uint8_t idx);

/**
 * @brief Get the status of transmission completion.
 *
 * @param base indicates that the MCAN instance base address.
 * @return non-zero Message transmission completed.
 * @return 0 Get message buffer status fail or no message transmission completed.
 */
uint32_t MCAN_getTxBufTransmissionStatus(uint32_t base);

/**
 * @brief Get Tx buffer transmission request flag status.
 *
 * @param base indicates that the MCAN instance base address.
 */
uint32_t MCAN_getTxBufTransmissionFlagStatus(uint32_t base);

/**
 * @brief Set Tx buffer transmission request flag.
 * @param base indicates that the MCAN instance base address.
 * @param txbuf_num Tx buffer index.(0 - maximum of Tx buffer).
 * @return true indicates successful setting of the transmission flag for sending.
 * @return false indicates fail setting of the transmission flag for sending.
 */
bool MCAN_setTxBufTransmissionFlag(uint32_t base, uint8_t txbuf_num);

/**
 * @brief Clear Tx buffer transmission request flag.
 * @param base indicates that the MCAN instance base address.
 * @param txbuf_num Tx buffer index.(0 - maximum of Tx buffer).
 * @return true indicates successful clearing of the transmission flag for sending.
 * @return false indicates fail clearing of the transmission flag for sending.
 */
bool MCAN_clearTxBufTransmissionFlag(uint32_t base, uint8_t txbuf_num);

/**
 * @brief Get transmission event message.
 * After the transmission is completed, the message can be read.
 *
 * @param base indicates that the MCAN instance base address.
 * @param eventbuf Pointing to the address space for storing Tx event messages.
 * @return true Successfully received Tx message.
 * @return false Unsuccessfully received Tx message.
 */
bool MCAN_readTxEventFIFO(uint32_t base, MCAN_TxEventFifo_t *eventbuf);

/**
 * @brief This API will Tx Event FIFO status.
 *
 * @param base indicates that the MCAN instance base address.
 * @param fifoStatus Tx Event FIFO Status(@MCAN_TxEventFIFOStatus_t).
 * @return true is get Tx event success, else failed.
 */
bool MCAN_getTxEventFIFOStatus(uint32_t base, MCAN_TxEventFIFOStatus_t *fifoStatus);

/**
 * @brief Generate message data size based on data format.
 *
 * @param size MCAN data phase data format size(@MCAN_ElemSize_t).
 * @return uint32_t Message data size.
 * @return Returning a value less than 0 is an error.
 */
int MCAN_getMsgObjSize(MCAN_ElemSize_t size);

/**
 * @brief Generate message data length.
 *
 * @param len MCAN data phase data length(@MCAN_MsgDataLength_t).
 * @return uint32_t Message data length.
 * @return Returning a value less than 0 is an error.
 */
int MCAN_getDataSize(MCAN_MsgDataLength_t len);

/**
 * @brief Get MCAN Error counter status.
 *
 * @param base indicates that the MCAN instance base address.
 * @param errCounter Contains MCAN Error counter(@MCAN_ErrCntStatus_t).
 * @return true Get MCAN Error counter success.
 * @return false Base address is error or read error counter error.
 */
bool MCAN_getErrCounters(uint32_t base, MCAN_ErrCntStatus_t *errCounter);

/**
 * @brief Get MCAN protocol status.
 *
 * @param base indicates that the MCAN instance base address.
 * @param protocolStatus Contains MCAN protocol status(@MCAN_ProtocolStatus_t).
 * @return true Get MCAN protocol status success.
 * @return false Base address is error or read protocol status error.
 */
bool MCAN_getProtocolStatus(uint32_t base, MCAN_ProtocolStatus_t *protocolStatus);

/**
 * @brief Get MCANSS PID information.
 *
 * @param base indicates that the MCAN instance base address.
 * @param revId Contains Revision ID of MCAN module(@MCAN_RevisionId_t).
 * @return true Get PID information.
 * @return false Base address is error or read PID error.
 */
bool MCAN_getRevisionId(uint32_t base, MCAN_RevisionId_t *revId);

/**
 * @brief Enable/Disable external timestamp counter.
 *
 * @param base indicates that the MCAN instance base address.
 * @param enable True is enable timestamp, else disable timestamp.
 * @return true Timestamp is enabled/disabled.
 * @return false Timestamp is enabled/disabled fail.
 */
bool MCAN_extTSCounterEnable(uint32_t base, bool enable);

/**
 * @brief Configure external timestamp counter for MCAN module.
 *
 * @param base indicates that the MCAN instance base address.
 * @param prescalar Timestamp Counter Prescaler. Range:[0x0-0xFFFFFF].
 * @return true Timestamp counter configure done.
 * @return false Timestamp counter configure failed.
 */
bool MCAN_extTSCounterConfig(uint32_t base, uint32_t prescalar);

/**
 * @brief Get external timestamp counter value.
 *
 * @param base indicates that the MCAN instance base address.
 * @return uint32_t non-zero value is timestamp counter value, 0 is error.
 */
uint32_t MCAN_extTSGetUnservicedIntrCount(uint32_t base);

/**
 * @brief Clear External timestamp overflow count,
 * write value to MCANSS_EXT_TS_UNSERVIED_INTR_CNTR register will subtraction value.
 *
 * @param base indicates that the MCAN instance base address.
 * @return true is write successful, else is error.
 * */
bool MCAN_extTSClearOverflowIntrCount(uint32_t base, uint32_t count_val);

/**
 * @brief Clear External TimeStamp Counter Overflow Interrupt.
 * If the API is used, IRS is clear 0, timestamp overflow count - 1.
 *
 * @param base indicates that the MCAN instance base address.
 * @return true Clear External TimeStamp Counter Overflow Interrupt successfully.
 * @return false Input base address is error.
 */
bool MCAN_extTSClearRawStatus(uint32_t base);

/**
 * @brief Clear Timestamp overflow interrupt.
 * If Timestamp overflow count > 0 and interrupt is enable,
 * the timestamp overflow interrupt will be triggered again.
 *
 * @param base indicates that the MCAN instance base address.
 * @return true Write Timestamp End of interrupt successfully.
 * @return false Input base address is error.
 * */
bool MCAN_extTSWriteEOI(uint32_t base);

/**
 * @brief External Timestamp Counter Overflow Interrupt Status.
 * MCANSS_EXT_TS_UNSERVICED_INTR_CNTR.EXT_TS_INTR_CNTR bit field will increment by 1.
 *
 * @param base indicates that the MCAN instance base address.
 * @return true Trigger External Timestamp Counter Overflow Interrupt successfully.
 * @return false Input base address is error.
 */
bool MCAN_extTSSetRawStatus(uint32_t base);

/**
 * @brief Enable/Disable External Timestamp Counter Overflow Interrupt.
 *
 * @param base indicates that the MCAN instance base address.
 * @param enable true is enbale external timestamp counter
 * overflow interrupt, else disable.
 * @return true Enable external timestamp counter overflow interrupt successfully.
 * @return false Input base address is error.
 */
bool MCAN_extTSEnableIntr(uint32_t base, bool enable);

/**
 * @brief Check if External Timestamp Counter overflow interrupt is triggered.
 *
 * @param base indicates that the MCAN instance base address.
 * @return true External timestamp counter overflow interrupt is triggered.
 * @return false External timestamp counter overflow interrupt not triggered.
 */
bool MCAN_extTSIsIntrEnable(uint32_t base);

/**
 * @brief Enable/Disable external timestamp counter.
 *
 * @param base indicates that the MCAN instance base address.
 * @param enable True is enable timestamp, else disable timestamp.
 * @return true Timestamp is enabled/disabled.
 * @return false Timestamp is enabled/disabled fail.
 */
bool MCAN_enableAutoWakeup(uint32_t base, bool enable);

/**
 * @brief Check if MCAN is reset completed.
 *
 * @param base indicates that the MCAN instance base address.
 * @return true indecates that MCAN Reset in progress.
 * @return false indecates that MCAN Reset finally or base address input error.
 */
bool MCAN_isInReset(uint32_t base);

/**
 * @brief Check if MCAN CANFD mode is enabled.
 *
 * @param base indicates that the MCAN instance base address.
 * @return true indecates that MCAN CANFD mode is enable.
 * @return false indecates that MCAN CANFD mode is disable.
 */
bool MCAN_isFDOpEnable(uint32_t base);

/**
 * @brief Check if MCAN message RAM is initialized.
 *
 * @param base indicates that the MCAN instance base address.
 * @return true indecates that MCAN message RAM is initialized.
 * @return false indecates that MCAN message RAM is not initialized.
 */
bool MCAN_isMemInitDone(uint32_t base);

/**
 * @brief Wakeup request enable.
 * Enables the MCANSS to wakeup on CAN RXD activity.
 *
 * @param base indicates that the MCAN instance base address.
 * @param enable true is enable wakeup on CAN RXD activity, else disable.
 * @return true Successfully enabled wake-up request.
 * @return false Enabling wake-up request failed.
 */
bool MCAN_enableWakupReq(uint32_t base, bool enable);

/**
 * @brief Auto wakeup request enable.
 * Automatic Wakeup Enable. Enables the MCANSS to automatically
 * clear the MCAN CCCR.INIT bit, fully waking the MCAN up,
 * on an enabled wakeup request.
 *
 * @param base indicates that the MCAN instance base address.
 * @param enable true is enable auto wakeup clear CCCR.INIT.
 * @return true Successfully enabled auto wake-up request.
 * @return false Enabling auto wake-up request failed.
 */
bool MCAN_enableAutoWakeup(uint32_t base, bool enable);

/**
 * @brief Enable debug suspend.
 *
 * @param base indicates that the MCAN instance base address.
 * @param enable true is enable debug suspend, else disable.
 * @return true Successfully enabled debug suspend.
 * @return false Enabling debug suspend failed.
 */
bool MCAN_enableDebugFree(uint32_t base, bool enable);

/**
 * @brief Get the ECC AGGR revision ID.
 *
 * @param base indicates that the MCAN instance base address.
 * @param revId Contains Revision ID of ECC AGGR.
 * @return true Get Revision ID.
 * @return false Input parameter error.
 */
bool MCAN_eccAggrGetRevisionId(uint32_t base, MCAN_ECCAggrRevisionId_t *revId);

/**
 * @brief Get the ECC AGGR RAM number.
 *
 * @param base indicates that the MCAN instance base address.
 * @return non-zero Get ECC RAM number.
 * @return zero Input parameter error.
 */
uint32_t MCAN_eccAggrGetRamNumber(uint32_t base);

/**
 * @brief Get the ECC Wrapper revision ID.
 *
 * @param base indicates that the MCAN instance base address.
 * @param revId Contains Revision ID of ECC Wrapper.
 * @return true Get Revision ID.
 * @return false Input parameter error.
 */
bool MCAN_eccWrapGetRevisionId(uint32_t base, MCAN_ECCAggrRevisionId_t *revId);

/**
 * @brief Enable/Disable ECC on the Message RAM.
 *
 * @param base indicates that the MCAN instance base address.
 * @param configParams MCAN ECC Configuration Parameters(@MCAN_ECCConfigParams_t).
 * @return true ECC configure successfully.
 * @return false Input parameter error.
 */
bool MCAN_eccConfig(uint32_t base, MCAN_ECCConfigParams_t *configParams);

/**
 * @brief Return ECC Error status.
 *
 * @param base indicates that the MCAN instance base address.
 * @param eccErr ECC error status(@MCAN_ECCErrStatus_t).
 * @return true Get ECC error status successfully.
 * @return false Input parameter error.
 */
bool MCAN_eccGetErrorStatus(uint32_t base, MCAN_ECCErrStatus_t *eccErr);

/**
 * @brief Clear ECC Error.
 *
 * @param base indicates that the MCAN instance base address.
 * @param err_type ECC error types(@MCAN_ECCErrType_t).
 * @return true Write clear ECC error successful.
 * @return false Input parameter error.
 */
bool MCAN_clearEccError(uint32_t base, MCAN_ECCErrType_t err_type, uint32_t val);

/**
 * @brief Inject ECC Error to DED or SEC.
 *
 * @param base indicates that the MCAN instance base address.
 * @param err_type ECC error types(@MCAN_ECCErrType_t).
 * @param val Error inject number.
 * @return true Write clear ECC error successful.
 * @return false Input parameter error.
 */
bool MCAN_writeEccError(uint32_t base, MCAN_ECCErrType_t err_type, uint32_t val);

/**
 * @brief Inject ECC Error.
 *
 * @param base indicates that the MCAN instance base address.
 * @param err_type ECC error types(@MCAN_ECCErrType_t).
 * @return true Force ECC error successful.
 * @return false Input parameter error.
 */
bool MCAN_eccForceError(uint32_t base, MCAN_ECCErrForceParams_t *eccErr);

/**
 * @brief Clear Force ECC Error.
 *
 * @param base indicates that the MCAN instance base address.
 * @param err_type ECC error types(@MCAN_ECCErrType_t).
 * @return true Clear Force ECC error successful.
 * @return false Input parameter error.
 */
bool MCAN_eccClearForceError(uint32_t base, MCAN_ECCErrForceParams_t *eccErr);

/**
 * @brief Get Ecc interrupt status.
 *
 * @param base indicates that the MCAN instance base address.
 * @param err_type ECC error types(@MCAN_ECCErrType_t).
 * @return true Get Ecc interrupt status and interrupt is pending.
 * @return false Input parameter error or interrupt is not pending.
 */
bool MCAN_eccGetIntrStatus(uint32_t base, MCAN_ECCErrType_t err_type);

/**
 * @brief Clear ECC interrupt status.
 *
 * @param base indicates that the MCAN instance base address.
 * @param err_type ECC error types(@MCAN_ECCErrType_t).
 * @return true ECC Error interrupt is cleared.
 * @return false Input parameter error.
 */
bool MCAN_eccClearIntrStatus(uint32_t base, MCAN_ECCErrType_t err_type);

/**
 * @brief Enable/Disable ECC error interrupt.
 *
 * @param base indicates that the MCAN instance base address.
 * @param err_type ECC error types(@MCAN_ECCErrType_t).
 * @param enable true is enable, else disable.
 * @return true is enabled, else input parameters error.
 */
bool MCAN_eccEnableIntr(uint32_t base, MCAN_ECCErrType_t err_type, bool enable);

/**
 * @brief used to write End of Interrupt for ECC interrupt.
 *
 * @param base indicates that the MCAN instance base address.
 * @param err_type ECC error types(@MCAN_ECCErrType_t).
 * @return true indicates successful EOI writing, else input parameters error.
 */
bool MCAN_eccWriteEOI(uint32_t base, MCAN_ECCErrType_t err_type);

#ifdef __cplusplus
}
#endif

#endif
