/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    qspi.h
*   @brief   
*
*/

#ifndef DEVICE_DRIVERLIB_QSPI_H_
#define DEVICE_DRIVERLIB_QSPI_H_


#ifdef __cplusplus
extern "C"{
#endif
/*==================================================================================================
*                                        INCLUDE FILES
* 1) system and project includes
* 2) needed interfaces from external units
* 3) internal and external interfaces from this unit
==================================================================================================*/

#include "inc/hw_types.h"
#include "inc/hw_memmap.h"
#include "inc/hw_qspi.h"
#include "debug.h"

#ifdef QSPIA_BASE
#define QSPIREG(base, offset)   (HWREG(base + offset))

#define QSPI_FIFO_TX_FULL 32
#define QSPI_FIFO_RX_FULL 32
#define QSPI_FIFO_EMPTY   (0)

//*****************************************************************************
//
//! Values that can config BUAD CTRLR0 SPI_CTRLR0
//
//*****************************************************************************
typedef struct
{
    uint32_t buad;              /*corresponding to REG[QSPI_BAUDR]*/
    uint32_t baseCtrl;          /*corresponding to REG[CTRLR0]*/
    uint32_t qspiCtrl;          /*corresponding to REG[SPI_CTRLR0]*/
}QSPI_CtrlParam;

//*****************************************************************************
//
//! Values that can be passed to SPI_setConfig() as the \e protocol parameter.
//
//*****************************************************************************
typedef enum
{
    //! Mode 0. Polarity 0, phase 0. Rising edge without delay.
    QSPI_PROT_POL0PHA0   = 0x0000U,
    //! Mode 1. Polarity 0, phase 1. Rising edge with delay.
    QSPI_PROT_POL0PHA1   = 0x0002U,
    //! Mode 2. Polarity 1, phase 0. Falling edge without delay.
    QSPI_PROT_POL1PHA0   = 0x0001U,
    //! Mode 3. Polarity 1, phase 1. Falling edge with delay.
    QSPI_PROT_POL1PHA1   = 0x0003U
} QSPI_TransferProtocol;

//*****************************************************************************
//
//! Values that can be passed to SPI_setConfig() as the \e mode parameter.
//
//*****************************************************************************
typedef enum
{
    QSPI_MODE_PERIPHERAL     = 0x0002U,   //!< SPI peripheral
    QSPI_MODE_CONTROLLER     = 0x0006U,   //!< SPI controller
    QSPI_MODE_PERIPHERAL_OD  = 0x0000U,   //!< SPI peripheral w/ output disabled
    QSPI_MODE_CONTROLLER_OD  = 0x0004U    //!< SPI controller w/ output disabled
}QSPI_Mode;

typedef enum{
    QSPI_FRF_SPI  = 0,/*Motorola SPI Frame Format*/
    QSPI_FRF_SSP  = 1,/*Texas Instruments SSP Frame Format*/
    QSPI_FRF_NSMW = 2,/*National Microwire Frame Format*/
    QSPI_FRF_RSVD = 3,/*Reserved value*/
}QSPI_FrameFormat;

typedef enum
{
    QSPI_DATA_LITTLE_ENDIAN   = 0U, //!< LITTLE ENDIAN
    QSPI_DATA_BIG_ENDIAN   = 1U,    //!< BIG ENDIAN
}QSPI_endianess;

/*==================================================================================================
*                                        QSPI registers definition
==================================================================================================*/

/* ----------------------------------------------------------------------------
   -- QSPI Register Masks
   ---------------------------------------------------------------------------- */

/* CTRLR0 */
#define QSPI_CTRLR0_SSI_IS_MST_MASK				(0x80000000U)

#define QSPI_CTRLR0_CLK_LOOP_EN_MASK			(0x04000000U)

#define QSPI_CTRLR0_SPI_DWS_EN_MASK				(0x02000000U)

#define QSPI_CTRLR0_SPI_HYPERBUS_EN_MASK		(0x01000000U)

#define QSPI_CTRLR0_SPI_FRF_MASK				(0x00C00000U)
#define QSPI_CTRLR0_SPI_FRF_SHIFT				(22U)
#define QSPI_CTRLR0_SPI_FRF(x)					(((uint32_t)(((uint32_t)(x)) << QSPI_CTRLR0_SPI_FRF_SHIFT)) & QSPI_CTRLR0_SPI_FRF_MASK)
#define QSPI_CTRLR0_SPI_FRF_STD					(0x00)
#define QSPI_CTRLR0_SPI_FRF_DUAL				(0x01)
#define QSPI_CTRLR0_SPI_FRF_QUAD				(0x02)


#define QSPI_CTRLR0_CFS_MASK					(0x000F0000U)
#define QSPI_CTRLR0_CFS_SHIFT					(16U)
#define QSPI_CTRLR0_CFS(x)						(((uint32_t)(((uint32_t)(x)) << QSPI_CTRLR0_CFS_SHIFT)) & QSPI_CTRLR0_CFS_MASK)

#define QSPI_CTRLR0_SSTE_MASK					(0x00004000U)

#define QSPI_CTRLR0_SRL_MASK					(0x00002000U)
#define QSPI_CTRLR0_SLV_OE_MASK					(0x00001000U)

#define QSPI_CTRLR0_TMOD_MASK					(0x00000C00U)
#define QSPI_CTRLR0_TMOD_SHIFT					(10U)
#define QSPI_CTRLR0_TMOD(x)						(((uint32_t)(((uint32_t)(x)) << QSPI_CTRLR0_TMOD_SHIFT)) & QSPI_CTRLR0_TMOD_MASK)
#define QSPI_CTRLR0_TMOD_TXRX					0
#define QSPI_CTRLR0_TMOD_TXONLY					1
#define QSPI_CTRLR0_TMOD_RXONLY					2
#define QSPI_CTRLR0_TMOD_EEPROM_READ			3



#define QSPI_CTRLR0_SCPOL_MASK					(0x00000200U)
#define QSPI_CTRLR0_SCPH_MASK					(0x00000100U)
#define QSPI_CTRLR0_FRF_MASK					(0x000000C0U)

#define QSPI_CTRLR0_DFS_MASK					(0x0000001FU)
#define QSPI_CTRLR0_DFS_SHIFT					(0U)
#define QSPI_CTRLR0_DFS(x)						(((uint32_t)(((uint32_t)(x)) << QSPI_CTRLR0_DFS_SHIFT)) & QSPI_CTRLR0_DFS_MASK)



/* CTRL1 register masks */
#define QSPI_CTRLR1_NDF_MASK					(0x0000FFFFU)
#define QSPI_CTRLR1_NDF_SHIFT					(0U)
#define QSPI_CTRLR1_NDF(x)						(((uint32_t)(((uint32_t)(x)) << QSPI_CTRLR1_NDF_SHIFT)) & QSPI_CTRLR1_NDF_MASK)


/* SSIENR register */
#define QSPI_SSIENR_SSIC_EN_MASK				(0x00000001U)


/* SER register */
#define QSPI_SER_SER_MASK						(0x0000000FU)
#define QSPI_SER_SER_SHIFT						(0U)
#define QSPI_SER_SER(x)							(((uint32_t)(((uint32_t)(x)) << QSPI_SER_SER_SHIFT)) & QSPI_SER_SER_MASK)


/* BAUDR register */
#define QSPI_BAUDR_SCKDV_MASK					(0x0000FFFEU)
#define QSPI_BAUDR_SCKDV_SHIFT					(1U)
#define QSPI_BAUDR_SCKDV(x)						(((uint32_t)(((uint32_t)(x)) << QSPI_BAUDR_SCKDV_SHIFT)) & QSPI_BAUDR_SCKDV_MASK)

/* TXFTLR register */
#define QSPI_TXFTLR_TXFTHR_MASK					(0x000F0000U)
#define QSPI_TXFTLR_TXFTHR_SHIFT				(16U)
#define QSPI_TXFTLR_TXFTHR(x)					(((uint32_t)(((uint32_t)(x)) << QSPI_TXFTLR_TXFTHR_SHIFT)) & QSPI_TXFTLR_TXFTHR_MASK)

#define QSPI_TXFTLR_TFT_MASK					(0x0000000FU)
#define QSPI_TXFTLR_TFT_SHIFT					(0U)
#define QSPI_TXFTLR_TFT(x)						(((uint32_t)(((uint32_t)(x)) << QSPI_TXFTLR_TFT_SHIFT)) & QSPI_TXFTLR_TFT_MASK)


/* RXFTLR register */
#define QSPI_RXFTLR_RFT_MASK					(0x0000000FU)
#define QSPI_RXFTLR_RFT_SHIFT					(0U)
#define QSPI_RXFTLR_RFT(x)						(((uint32_t)(((uint32_t)(x)) << QSPI_RXFTLR_RFT_SHIFT)) & QSPI_RXFTLR_RFT_MASK)


/* SR register */
#define QSPI_SR_CMPLTD_DF_MASK					(0xFFFF0000U)
#define QSPI_SR_DCOL_MASK						(0x00000040U)
#define QSPI_SR_TXE_MASK						(0x00000020U)
#define QSPI_SR_RFF_MASK						(0x00000010U)
#define QSPI_SR_RFNE_MASK						(0x00000008U)
#define QSPI_SR_TFE_MASK						(0x00000004U)
#define QSPI_SR_TFNF_MASK						(0x00000002U)
#define QSPI_SR_BUSY_MASK						(0x00000001U)


/* IMR Interrupts Mask register */
#define QSPI_IMR_RXFIM                          0x00000010U
#define QSPI_IMR_RXOIM                          0x00000008U
#define QSPI_IMR_RXUIM                          0x00000004U
#define QSPI_IMR_TXOIM                          0x00000002U
#define QSPI_IMR_TXEIM                          0x00000001U


#define QSPI_ISR_RXFIS                          0x00000010U
#define QSPI_ISR_RXOIS                          0x00000008U
#define QSPI_ISR_RXUIS                          0x00000004U
#define QSPI_ISR_TXOIS                          0x00000002U
#define QSPI_ISR_TXEIS                          0x00000001U


/* SPI_CTRLR0 register Mask */
#define QSPI_SPI_CTRLR0_CLK_STRETCH_EN_MASK		(0x40000000U)

#define QSPI_SPI_CTRLR0_XIP_PREFETCH_EN_MASK	(0x20000000U)

#define QSPI_SPI_CTRLR0_XIP_INST_EN_MASK		(0x00100000U)

#define QSPI_SPI_CTRLR0_XIP_DFS_HC_MASK			(0x00080000U)

#define QSPI_SPI_CTRLR0_WAIT_CYCLES_MASK		(0x0000F800U)
#define QSPI_SPI_CTRLR0_WAIT_CYCLES_SHIFT		(11U)
#define QSPI_SPI_CTRLR0_WAIT_CYCLES(x)			(((uint32_t)(((uint32_t)(x)) << QSPI_SPI_CTRLR0_WAIT_CYCLES_SHIFT)) & QSPI_SPI_CTRLR0_WAIT_CYCLES_MASK)

#define QSPI_SPI_CTRLR0_INST_L_MASK				(0x00000300U)
#define QSPI_SPI_CTRLR0_INST_L_SHIFT			(8U)
#define QSPI_SPI_CTRLR0_INST_L(x)				(((uint32_t)(((uint32_t)(x)) << QSPI_SPI_CTRLR0_INST_L_SHIFT)) & QSPI_SPI_CTRLR0_INST_L_MASK)
#define QSPI_SPI_CTRLR0_INST_L_4BIT				(0x01U)
#define QSPI_SPI_CTRLR0_INST_L_8BIT				(0x02U)
#define QSPI_SPI_CTRLR0_INST_L_16BIT			(0x03U)

#define QSPI_SPI_CTRLR0_ADDR_L_MASK				(0x0000003CU)
#define QSPI_SPI_CTRLR0_ADDR_L_SHIFT			(2U)
#define QSPI_SPI_CTRLR0_ADDR_L(x)				(((uint32_t)(((uint32_t)(x)) << QSPI_SPI_CTRLR0_ADDR_L_SHIFT)) & QSPI_SPI_CTRLR0_ADDR_L_MASK)
#define QSPI_SPI_CTRLR0_ADDR_L_16bit			(0x04U)
#define QSPI_SPI_CTRLR0_ADDR_L_24bit			(0x06U)
#define QSPI_SPI_CTRLR0_ADDR_L_32bit			(0x08U)

#define QSPI_SPI_CTRLR0_TRANS_TYPE_MASK			(0x00000003U)
#define QSPI_SPI_CTRLR0_TRANS_TYPE_SHIFT		(0U)
#define QSPI_SPI_CTRLR0_TRANS_TYPE(x)			(((uint32_t)(((uint32_t)(x)) << QSPI_SPI_CTRLR0_TRANS_TYPE_SHIFT)) & QSPI_SPI_CTRLR0_TRANS_TYPE_MASK)
#define QSPI_SPI_CTRLR0_TRANS_TYPE_TT0			(0U)
#define QSPI_SPI_CTRLR0_TRANS_TYPE_TT1			(1U)
#define QSPI_SPI_CTRLR0_TRANS_TYPE_TT2			(2U)


#define QSPI_SPI_CTRLR0_TRANS_TYPE_SHIFT		(0U)
#define QSPI_SPI_CTRLR0_TRANS_TYPE(x)			(((uint32_t)(((uint32_t)(x)) << QSPI_SPI_CTRLR0_TRANS_TYPE_SHIFT)) & QSPI_SPI_CTRLR0_TRANS_TYPE_MASK)
#define QSPI_SPI_CTRLR0_TRANS_TYPE_TT0			(0U)
#define QSPI_SPI_CTRLR0_TRANS_TYPE_TT1			(1U)
#define QSPI_SPI_CTRLR0_TRANS_TYPE_TT2			(2U)




/* XIP_CTRL register Mask */
#define QSPI_XIP_CTRL_RXDS_VL_EN_MASK			(0x40000000U)

#define QSPI_XIP_CTRL_XIP_PREFETCH_EN_MASK		(0x20000000U)

#define QSPI_XIP_CTRL_XIP_MBL_MASK				(0x0C000000U)

#define QSPI_XIP_CTRL_INST_EN_MASK				(1U<<22U)

#define QSPI_XIP_CTRL_XIP_DFS_HC_MASK			(1U<<18U)

#define QSPI_XIP_CTRL_WAIT_CYCLES_MASK			(0x1FU<<13U)
#define QSPI_XIP_CTRL_WAIT_CYCLES_SHIFT			(13U)
#define QSPI_XIP_CTRL_WAIT_CYCLES(x)			(((uint32_t)(((uint32_t)(x)) << QSPI_XIP_CTRL_WAIT_CYCLES_SHIFT)) & QSPI_XIP_CTRL_WAIT_CYCLES_MASK)

#define QSPI_XIP_CTRL_MB_BITS_EN_MASK			(1U<<12U)

#define QSPI_XIP_CTRL_INST_L_MASK				(0x03U<<9U)
#define QSPI_XIP_CTRL_INST_L_SHIFT				(9U)
#define QSPI_XIP_CTRL_INST_L(x)					(((uint32_t)(((uint32_t)(x)) << QSPI_XIP_CTRL_INST_L_SHIFT)) & QSPI_XIP_CTRL_INST_L_MASK)
#define QSPI_XIP_CTRL_INST_L_4BIT				(0x01U)
#define QSPI_XIP_CTRL_INST_L_8BIT				(0x02U)
#define QSPI_XIP_CTRL_INST_L_16BIT				(0x03U)

#define QSPI_XIP_CTRL_ADDR_L_MASK				(0x0FU<<4U)
#define QSPI_XIP_CTRL_ADDR_L_SHIFT				(4U)
#define QSPI_XIP_CTRL_ADDR_L(x)					(((uint32_t)(((uint32_t)(x)) << QSPI_XIP_CTRL_ADDR_L_SHIFT)) & QSPI_XIP_CTRL_ADDR_L_MASK)
#define QSPI_XIP_CTRL_ADDR_L_16bit				(0x04U)
#define QSPI_XIP_CTRL_ADDR_L_24bit				(0x06U)
#define QSPI_XIP_CTRL_ADDR_L_32bit				(0x08U)

#define QSPI_XIP_CTRL_TRANS_TYPE_MASK			(0x03U<<2U)
#define QSPI_XIP_CTRL_TRANS_TYPE_SHIFT			(2U)
#define QSPI_XIP_CTRL_TRANS_TYPE(x)				(((uint32_t)(((uint32_t)(x)) << QSPI_XIP_CTRL_TRANS_TYPE_SHIFT)) & QSPI_XIP_CTRL_TRANS_TYPE_MASK)
#define QSPI_XIP_CTRL_TRANS_TYPE_TT0			(0U)
#define QSPI_XIP_CTRL_TRANS_TYPE_TT1			(1U)
#define QSPI_XIP_CTRL_TRANS_TYPE_TT2			(2U)

#define QSPI_XIP_CTRL_FRF_MASK					(0x03U<<0U)
#define QSPI_XIP_CTRL_FRF_SHIFT					(0U)
#define QSPI_XIP_CTRL_FRF(x)					(((uint32_t)(((uint32_t)(x)) << QSPI_XIP_CTRL_FRF_SHIFT)) & QSPI_XIP_CTRL_FRF_MASK)
#define QSPI_XIP_CTRL_FRF_STANDARD				(0x0U)
#define QSPI_XIP_CTRL_FRF_DUAL					(0x1U)
#define QSPI_XIP_CTRL_FRF_QUAD					(0x2U)


/* QSPI default parameters */

#define QSPI_TX_FIFO_DEPTH						(0x20)
#define QSPI_FIFO_DEPTH						    (0x20)
#define QSPI_DUMMY_CYCLES_LEN					(10U)


typedef enum
{
    QSPI_STANDARD_FRAME   = 0x0,
    QSPI_DUAL_FRAME       = 0x1,
    QSPI_QUAD_FRAME       = 0x2,
    QSPI_OCTAL_FRAME      = 0x3,
} QSPI_SPI_FRAME_FORMAT;

typedef enum
{
    QSPI_SELECT_TOGGLE_DISABLE = 0,
    QSPI_SELECT_TOGGLE_ENABLE  = 1,
} QSPI_TARGET_SELECT_TOGGLE;

typedef enum
{
    QSPI_TX_AND_RX     = 0x0,
    QSPI_TX_ONLY       = 0x1,
    QSPI_RX_ONLY       = 0x2,
    QSPI_EEPROM_READ   = 0x3,
} QSPI_TRANSFER_MODE;

typedef enum
{
    QSPI_SCLK_LOW  = 0x0,
    QSPI_SCLK_HIGH = 0x1,
} QSPI_SERIAL_CLOCK_POLARITY;

typedef enum
{
    QSPI_SCPH_0    = 0x0,
    QSPI_SCPH_1    = 0x1,
} QSPI_SERIAL_CLOCK_PHASE;

typedef enum
{
    QSPI_MOTOROLA_SPI = 0x0,
    QSPI_TEXAS_SSP    = 0x1,
    QSPI_NS_MICROWIRE = 0x2,
    QSPI_RESERVED     = 0x3,
} QSPI_FRAME_FORMAT;

typedef enum
{
    QSPI_DFS_04_BIT     = 0x03,
    QSPI_DFS_05_BIT     = 0x04,
    QSPI_DFS_06_BIT     = 0x05,
    QSPI_DFS_07_BIT     = 0x06,
    QSPI_DFS_08_BIT     = 0x07,
    QSPI_DFS_09_BIT     = 0x08,
    QSPI_DFS_10_BIT     = 0x09,
    QSPI_DFS_11_BIT     = 0x0a,
    QSPI_DFS_12_BIT     = 0x0b,
    QSPI_DFS_13_BIT     = 0x0c,
    QSPI_DFS_14_BIT     = 0x0d,
    QSPI_DFS_15_BIT     = 0x0e,
    QSPI_DFS_16_BIT     = 0x0f,
    QSPI_DFS_17_BIT     = 0x10,
    QSPI_DFS_18_BIT     = 0x11,
    QSPI_DFS_19_BIT     = 0x12,
    QSPI_DFS_20_BIT     = 0x13,
    QSPI_DFS_21_BIT     = 0x14,
    QSPI_DFS_22_BIT     = 0x15,
    QSPI_DFS_23_BIT     = 0x16,
    QSPI_DFS_24_BIT     = 0x17,
    QSPI_DFS_25_BIT     = 0x18,
    QSPI_DFS_26_BIT     = 0x19,
    QSPI_DFS_27_BIT     = 0x1a,
    QSPI_DFS_28_BIT     = 0x1b,
    QSPI_DFS_29_BIT     = 0x1c,
    QSPI_DFS_30_BIT     = 0x1d,
    QSPI_DFS_31_BIT     = 0x1e,
    QSPI_DFS_32_BIT     = 0x1f,
} QSPI_DATA_FRAM_SIZE;


typedef enum
{
    QSPI_TRANS_TYPE_TT0 = 0x0,
	QSPI_TRANS_TYPE_TT1 = 0x1,
	QSPI_TRANS_TYPE_TT2 = 0x2,
	QSPI_TRANS_TYPE_TT3 = 0x3,
} QSPI_TRANS_TYPE;

typedef enum
{
    QSPI_ADDR_L0  = 0x0,
	QSPI_ADDR_L4  = 0x1,
	QSPI_ADDR_L8  = 0x2,
	QSPI_ADDR_L12 = 0x3,
	QSPI_ADDR_L16 = 0x4,
	QSPI_ADDR_L20 = 0x5,
	QSPI_ADDR_L24 = 0x6,
	QSPI_ADDR_L28 = 0x7,
	QSPI_ADDR_L32 = 0x8,
	QSPI_ADDR_L36 = 0x9,
	QSPI_ADDR_L40 = 0xa,
	QSPI_ADDR_L44 = 0xb,
	QSPI_ADDR_L48 = 0xc,
	QSPI_ADDR_L52 = 0xd,
	QSPI_ADDR_L56 = 0xe,
	QSPI_ADDR_L60 = 0xf,
} QSPI_ADDR_L;


typedef enum
{
    QSPI_INST_L0  = 0x0,
	QSPI_INST_L4  = 0x1,
	QSPI_INST_L8  = 0x2,
	QSPI_INST_L16 = 0x3,
} QSPI_INST_L;


typedef union
{
    uint32_t param;
    struct
    {
        uint32_t DFS_32          : 5; //[0:4]
        uint32_t reserved1       : 1; //5
        uint32_t FRF             : 2; //[6:7]
        uint32_t SCPH            : 1; //8
        uint32_t SCPOL           : 1; //9
        uint32_t TMOD            : 2; //[10:11]
        uint32_t SLV_OE          : 1; //12
        uint32_t SRL             : 1; //13
        uint32_t SSTE            : 1; //14
        uint32_t reserved2       : 1; //15
        uint32_t CFS             : 4; //[16:19]
        uint32_t reserved3       : 2; //[20:21]
        uint32_t SPI_FRF         : 2; //[22:23]
        uint32_t reserved4       : 7; //[24:30] This part of the function is not supported.
        uint32_t SSI_IS_MST      : 1; //31 Default to host mode, not configurable.
    } RegistValue;
} QSPI_CONTROL_PARAM;

typedef union
{
    uint32_t param;
    struct
    {
        uint32_t TRANS_TYPE             : 2; //[0:1]
        uint32_t ADDR_L                 : 4; //[2:5]
        uint32_t reserved1              : 1; //6
        uint32_t XIP_MD_BIT_EN          : 1; //7
        uint32_t INST_L                 : 2; //[8:9]
        uint32_t reserved2              : 1; //10
        uint32_t WAIT_CYCLES            : 5; //[11:15]
        uint32_t SPI_DDR_EN             : 1; //16
        uint32_t INST_DDR_EN            : 1; //17
        uint32_t SPI_RXDS_EN            : 1; //18
        uint32_t XIP_DFS_HC             : 1; //19
        uint32_t XIP_INST_EN            : 1; //20
        uint32_t SSIC_XIP_CONT_XFER_EN  : 1; //21
        uint32_t reserved3              : 1; //22
        uint32_t RXDS_VL_EN             : 1; //23
        uint32_t SPI_DM_EN              : 1; //24
        uint32_t SPI_RXDS_SIG_EN        : 1; //25
        uint32_t XIP_MBL                : 2; //[26:27]
        uint32_t reserved4              : 1; //28
        uint32_t XIP_PREFETCH_EN        : 1; //29
        uint32_t CLK_STRETCH_EN         : 1; //30
        uint32_t reserved5              : 1; //31
    } RegistValue;
} QSPI_SPI_CONTROL_PARAM;

typedef struct
{
    uint32_t baudrate;
    uint32_t ahbclk_freq;
    QSPI_TRANSFER_MODE transfer_mode;
    QSPI_SPI_FRAME_FORMAT frame_format;
    QSPI_SERIAL_CLOCK_PHASE scph;
    QSPI_SERIAL_CLOCK_POLARITY scpol;
    QSPI_DATA_FRAM_SIZE data_width;

    uint32_t clk_stretch_en;
    uint32_t xip_prefetch_en;
    uint32_t wait_cycles;
    QSPI_INST_L inst_l;
    QSPI_ADDR_L addr_l;
    QSPI_TRANS_TYPE trans_type;
} QSPI_INIT_PARAM;


//*****************************************************************************
//
//! \internal
//! Checks an QSPI base address.
//!
//! \param base specifies the QSPI module base address.
//!
//! This function determines if a QSPI module base address is valid.
//!
//! \return Returns \b true if the base address is valid and \b false
//! otherwise.
//
//*****************************************************************************
#ifdef DEBUG
__STATIC_INLINE boolean
QSPI_isBaseValid(uint32_t base)
{
    return( (base == QSPIA_BASE) );
}
#endif

//*****************************************************************************
//
//! Enables the serial peripheral interface.
//!
//! \param base specifies the QSPI module base address.
//!
//! This function enables operation of the serial peripheral interface.  The
//! serial peripheral interface must be configured before it is enabled.
//!
//! \return None.
//
//*****************************************************************************
__STATIC_INLINE void
QSPI_enableModule(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(QSPI_isBaseValid(base));

    QSPIREG(base, QSPI_SSIENR) |= QSPI_SSIENR_SSIC_EN_M;
}

//*****************************************************************************
//
//! Disables the serial peripheral interface.
//!
//! \param base specifies the QSPI module base address.
//!
//! This function disables operation of the serial peripheral interface. Call
//! this function before doing any configuration.
//!
//! \return None.
//
//*****************************************************************************
__STATIC_INLINE void
QSPI_disableModule(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(QSPI_isBaseValid(base));

    QSPIREG(base, QSPI_SSIENR) &= ~QSPI_SSIENR_SSIC_EN_M;
}

//*****************************************************************************
//
//! Sets the character length of QSPI transaction
//!
//! \param base specifies the QSPI module base address.
//! \param charLength specifies the character length of QSPI transaction
//!
//! This function configures the character length of QSPI transaction.
//! QSPI character length can be from anywhere between 1-bit word to 16 bit word
//! of character length
//!
//! \return None.
//
//*****************************************************************************
__STATIC_INLINE void
QSPI_setcharLength(uint32_t base, uint16_t charLength)
{
    ASSERT((charLength >= 4U) && (charLength <= 32U));
    QSPI_disableModule(base);
    uint32_t regValue = QSPIREG(base, QSPI_CTRLR0);
    QSPIREG(base, QSPI_CTRLR0) = (regValue & ~QSPI_CTRLR0_DFS_M) |
                               (charLength - 1);
    QSPI_enableModule(base);
}

//*****************************************************************************
//
//! Resets the FIFO.
//!
//! \param base is the base address of the QSPI port.
//!
//! This function resets the rx tx FIFO, setting the FIFO pointer back to
//! zero.
//!
//! \return None.
//
//*****************************************************************************
__STATIC_INLINE void
QSPI_resetFIFO(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(QSPI_isBaseValid(base));

    //
    // Reset the TX FIFO.
    //
    QSPI_disableModule(base);
    QSPI_enableModule(base);
}

//*****************************************************************************
//
//! Sets the FIFO level at which interrupts are generated.
//!
//! \param base is the base address of the QSPI port.
//! \param txLevel is the transmit FIFO interrupt level
//! \param rxLevel is the receive FIFO interrupt level
//!
//! This function sets the FIFO level at which transmit and receive interrupts
//! are generated.
//!
//! \return None.
//
//*****************************************************************************
__STATIC_INLINE void
QSPI_setFIFOInterruptLevel(uint32_t base, uint8_t txLevel, uint8_t rxLevel)
{
    //
    // Check the arguments.
    //
    ASSERT(QSPI_isBaseValid(base));

    //
    // Set the FIFO interrupt levels.
    //
    QSPIREG(base, QSPI_TXFTLR) = (QSPIREG(base, QSPI_TXFTLR) & (~QSPI_TXFTLR_TFT_M)) \
                            | txLevel;
    QSPIREG(base, QSPI_RXFTLR) = (QSPIREG(base, QSPI_RXFTLR) & (~QSPI_RXFTLR_RFT_M)) \
                            | rxLevel;
}

//*****************************************************************************
//
//! Gets the FIFO level at which interrupts are generated.
//!
//! \param base is the base address of the QSPI port.
//! \param txLevel is a pointer to storage for the transmit FIFO level,
//! \param rxLevel is a pointer to storage for the receive FIFO level,.
//!
//! This function gets the FIFO level at which transmit and receive interrupts
//! are generated.
//!
//! \return None.
//
//*****************************************************************************
__STATIC_INLINE void
QSPI_getFIFOInterruptLevel(uint32_t base, uint8_t *txLevel, uint8_t *rxLevel)
{
    //
    // Check the arguments.
    //
    ASSERT(QSPI_isBaseValid(base));

    //
    // Extract the transmit and receive FIFO levels.
    //
    *txLevel = (uint8_t)(QSPIREG(base, QSPI_TXFTLR) & QSPI_TXFTLR_TFT_M);
    *rxLevel = (uint8_t)(QSPIREG(base, QSPI_RXFTLR) & QSPI_RXFTLR_RFT_M);
}

//*****************************************************************************
//
//! Sets the FIFO level at which interrupts are generated.
//!
//! \param base is the base address of the QSPI port.
//! \param txLevel is the transmit FIFO interrupt level
//! \param rxLevel is the receive FIFO interrupt level
//!
//! This function sets the FIFO level at which transmit and receive interrupts
//! are generated.
//!
//! \return None.
//
//*****************************************************************************
__STATIC_INLINE void
QSPI_setFIFOTransferStartLevel(uint32_t base, uint8_t level)
{
    //
    // Check the arguments.
    //
    ASSERT(QSPI_isBaseValid(base));

    //
    // Set the FIFO interrupt levels.
    //
    QSPIREG(base, QSPI_TXFTLR) = (QSPIREG(base, QSPI_TXFTLR) & (~QSPI_TXFTLR_TXFTHR_M)) \
                            | (level << QSPI_TXFTLR_TXFTHR_S);
}

//*****************************************************************************
//
//! Gets the FIFO level at which interrupts are generated.
//!
//! \param base is the base address of the QSPI port.
//! \param txLevel is a pointer to storage for the transmit FIFO level,
//! \param rxLevel is a pointer to storage for the receive FIFO level,.
//!
//! This function gets the FIFO level at which transmit and receive interrupts
//! are generated.
//!
//! \return None.
//
//*****************************************************************************
__STATIC_INLINE void
QSPI_getFIFOTransferStartLevel(uint32_t base, uint8_t *level)
{
    //
    // Check the arguments.
    //
    ASSERT(QSPI_isBaseValid(base));

    //
    // Extract the transmit and receive FIFO levels.
    //
    *level = (uint8_t)(QSPIREG(base, QSPI_TXFTLR) & QSPI_TXFTLR_TXFTHR_M);
}

//*****************************************************************************
//
//! Get the transmit FIFO status
//!
//! \param base is the base address of the QSPI port.
//!
//! This function gets the current number of words in the transmit FIFO.
//!
//! \return Returns the current number of words in the transmit FIFO 
//
//*****************************************************************************
__STATIC_INLINE uint32_t QSPI_getTxFIFOStatus(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(QSPI_isBaseValid(base));

    //
    // Get the current FIFO status
    //
    return (QSPIREG(base, QSPI_TXFLR) & QSPI_TXFLR_TXTFL_M);
}

//*****************************************************************************
//
//! Get the receive FIFO status
//!
//! \param base is the base address of the QSPI port.
//!
//! This function gets the current number of words in the receive FIFO.
//!
//! \return Returns the current number of words in the receive FIFO
//
//*****************************************************************************
__STATIC_INLINE uint32_t
QSPI_getRxFIFOStatus(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(QSPI_isBaseValid(base));

    //
    // Get the current FIFO status
    //
	return (QSPIREG(base, QSPI_RXFLR) & QSPI_RXFLR_RXTFL_M);
}

//*****************************************************************************
//
//! Determines whether the QSPI transmitter is busy or not.
//!
//! \param base is the base address of the QSPI port.
//!
//! This function allows the caller to determine whether all transmitted bytes
//! have cleared the transmitter hardware.  If \b false is returned, then the
//! transmit FIFO is empty and all bits of the last transmitted word have left
//! the hardware shift register. This function is only valid when operating in
//! FIFO mode.
//!
//! \return Returns \b true if the QSPI is transmitting or \b false if all
//! transmissions are complete.
//
//*****************************************************************************
__STATIC_INLINE boolean
QSPI_isBusy(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(QSPI_isBaseValid(base));

    //
    // Determine if the QSPI is busy.
    //
    return((QSPIREG(base, QSPI_TXFLR) & QSPI_TXFLR_TXTFL_M) != 0U);
}

//*****************************************************************************
//
//! Puts a data element into the QSPI transmit buffer.
//!
//! \param base specifies the QSPI module base address.
//! \param data is the left-justified data to be transmitted over QSPI.
//!
//! This function places the supplied data into the transmit buffer of the
//! specified QSPI module.
//!
//! \note The data being sent must be left-justified in \e data. The lower
//! 16 - N bits will be discarded where N is the data width selected in
//! QSPI_setConfig(). For example, if configured for a 6-bit data width, the
//! lower 10 bits of data will be discarded.
//!
//! \return None.
//
//*****************************************************************************
__STATIC_INLINE void
QSPI_writeDataNonBlocking(uint32_t base, uint32_t data)
{
    //
    // Check the arguments.
    //
    ASSERT(QSPI_isBaseValid(base));

    //
    // Write data to the transmit buffer.
    //
    QSPIREG(base, QSPI_DR0) = data;
}

//*****************************************************************************
//
//! Gets a data element from the QSPI receive buffer.
//!
//! \param base specifies the QSPI module base address.
//!
//! This function gets received data from the receive buffer of the specified
//! QSPI module and returns it.
//!
//! \note Only the lower N bits of the value written to \e data contain valid
//! data, where N is the data width as configured by QSPI_setConfig(). For
//! example, if the interface is configured for 8-bit data width, only the
//! lower 8 bits of the value written to \e data contain valid data.
//!
//! \return Returns the word of data read from the QSPI receive buffer.
//
//*****************************************************************************
__STATIC_INLINE uint32_t
QSPI_readDataNonBlocking(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(QSPI_isBaseValid(base));

    //
    // Check for data to read.
    //
    return (QSPIREG(base, QSPI_DR0));
}

//*****************************************************************************
//
//! Waits for space in the FIFO and then puts data into the transmit buffer.
//!
//! \param base specifies the QSPI module base address.
//! \param data is the left-justified data to be transmitted over QSPI.
//!
//! This function places the supplied data into the transmit buffer of the
//! specified QSPI module once space is available in the transmit FIFO. This
//! function should only be used when the FIFO is enabled.
//!
//! \note The data being sent must be left-justified in \e data. The lower
//! 16 - N bits will be discarded where N is the data width selected in
//! QSPI_setConfig(). For example, if configured for a 6-bit data width, the
//! lower 10 bits of data will be discarded.
//!
//! \return None.
//
//*****************************************************************************
__STATIC_INLINE void
QSPI_writeDataBlockingFIFO(uint32_t base, uint32_t data)
{
    //
    // Check the arguments.
    //
    ASSERT(QSPI_isBaseValid(base));

    //
    // Wait until space is available in the receive FIFO.
    //
    while(QSPI_getTxFIFOStatus(base) == QSPI_FIFO_TX_FULL)
    {
    }

    //
    // Write data to the transmit buffer.
    //
    QSPIREG(base, QSPI_DR0) = data;
}

//*****************************************************************************
//
//! Waits for data in the FIFO and then reads it from the receive buffer.
//!
//! \param base specifies the QSPI module base address.
//!
//! This function waits until there is data in the receive FIFO and then reads
//! received data from the receive buffer.  This function should only be used
//! when FIFO mode is enabled.
//!
//! \note Only the lower N bits of the value written to \e data contain valid
//! data, where N is the data width as configured by QSPI_setConfig(). For
//! example, if the interface is configured for 8-bit data width, only the
//! lower 8 bits of the value written to \e data contain valid data.
//!
//! \return Returns the word of data read from the QSPI receive buffer.
//
//*****************************************************************************
__STATIC_INLINE uint32_t
QSPI_readDataBlockingFIFO(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(QSPI_isBaseValid(base));

    //
    // Wait until data is available in the receive FIFO.
    //
    while(QSPI_getRxFIFOStatus(base) == QSPI_FIFO_EMPTY)
    {
    }

    //
    // Check for data to read.
    //
    return (QSPIREG(base, QSPI_DR0));
}

//*****************************************************************************
//
//! Enables QSPI loopback mode.
//!
//! \param base is the base address of the QSPI port.
//!
//! This function enables loopback mode. This mode is only valid during
//! controller mode and is helpful during device testing as it internally
//! connects PICO and POCI.
//!
//! \return None.
//
//*****************************************************************************
__STATIC_INLINE void
QSPI_enableLoopback(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(QSPI_isBaseValid(base));

    //
    // Set the bit that enables loopback mode.
    //
    QSPI_disableModule(base);
    QSPIREG(base, QSPI_CTRLR0) |= QSPI_CTRLR0_SRL_M;
    QSPI_enableModule(base);
}

//*****************************************************************************
//
//! Disables QSPI loopback mode.
//!
//! \param base is the base address of the QSPI port.
//!
//! This function disables loopback mode. Loopback mode is disabled by default
//! after reset.
//!
//! \return None.
//
//*****************************************************************************
__STATIC_INLINE void
QSPI_disableLoopback(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(QSPI_isBaseValid(base));

    //
    // Clear the bit that enables loopback mode.
    //
    QSPI_disableModule(base);
    QSPIREG(base, QSPI_CTRLR0) &= ~QSPI_CTRLR0_SRL_M;
    QSPI_enableModule(base);
}

//*****************************************************************************
//
//! Enable Trasnmit
//!
//! \param base is the base address of the QSPI port.
//!
//! This function sets the TALK bit enabling the data trasnmission.
//! This bit is enabled by QSPI_setConfig if the parameter \r mode is selected as
//! target device.
//!
//! \return None
//
//*****************************************************************************
__STATIC_INLINE void
QSPI_enableTalk(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(QSPI_isBaseValid(base));

    //
    // Set the TALK bit
    //
    QSPI_disableModule(base);
    QSPIREG(base, QSPI_CTRLR0) &= ~QSPI_CTRLR0_SLV_OE_M;
    QSPI_enableModule(base);
}

//*****************************************************************************
//
//! Disable Trasnmit
//!
//! \param base is the base address of the QSPI port.
//!
//! This function clears the TALK bit disabling the data trasnmission. The
//! output pin will be put in high-impedance state.
//! This bit is enabled by QSPI_setConfig if the parameter \r mode is selected as
//! QSPI_MODE_PERIPHERAL or QSPI_MODE_CONTROLLER.
//!
//! \return None
//
//*****************************************************************************
__STATIC_INLINE void
QSPI_disableTalk(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(QSPI_isBaseValid(base));

    //
    // Set the TALK bit
    //
    QSPI_disableModule(base);
    QSPIREG(base, QSPI_CTRLR0) |= QSPI_CTRLR0_SLV_OE_M;
    QSPI_enableModule(base);
}

//*****************************************************************************
//
//! Reset QSPI transmit and receive channels
//!
//! \param base is the base address of the QSPI port.
//!
//! This function resets the QSPI transmit and receive channels.
//!
//! \return None
//
//*****************************************************************************
__STATIC_INLINE void
QSPI_reset(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(QSPI_isBaseValid(base));

    //
    // Write to SPRST bit the TX FIFO.
    //
    QSPI_disableModule(base);
    QSPI_enableModule(base);
}

__STATIC_INLINE void
QSPI_csEnable(uint32_t base)
{
    QSPI_disableModule(base);
    QSPIREG(base, QSPI_SER) |= QSPI_SER_SER_M;
    QSPI_enableModule(base);
}

__STATIC_INLINE void
QSPI_csDisable(uint32_t base)
{
    QSPIREG(base, QSPI_SER) &= ~QSPI_SER_SER_M;
}

__STATIC_INLINE void
QSPI_selectFrameFormat(uint32_t base, QSPI_FrameFormat format)
{
    QSPIREG(base, QSPI_CTRLR0) = (QSPIREG(base, QSPI_CTRLR0) & (~QSPI_CTRLR0_FRF_M)) | \
                            (format << QSPI_CTRLR0_FRF_S);
}

__STATIC_INLINE void
QSPI_setCtrlFrameSize(uint32_t base, uint8_t length)
{
    ASSERT((length >= 1U) && (length <= 16U));
    QSPI_disableModule(base);
    QSPIREG(base, QSPI_CTRLR0) = (QSPIREG(base, QSPI_CTRLR0) & (~QSPI_CTRLR0_CFS_M)) | \
                            ((length - 1) << QSPI_CTRLR0_CFS_S);
    QSPI_enableModule(base);
}

__STATIC_INLINE void
QSPI_setSpiModeSSTE(uint32_t base, uint8_t mode)
{
    QSPI_disableModule(base);
    QSPIREG(base, QSPI_CTRLR0) = (QSPIREG(base, QSPI_CTRLR0) & (~QSPI_CTRLR0_SSTE_M)) | \
                            (mode << QSPI_CTRLR0_SSTE_S);
    QSPI_enableModule(base);
}

__STATIC_INLINE void
QSPI_setMWCR(uint32_t base, uint8_t mode)
{
    QSPI_disableModule(base);
    QSPIREG(base, QSPI_MWCR) = mode;
    QSPI_enableModule(base);
}

__STATIC_INLINE void
QSPI_setDmaTxDataLevel(uint32_t base, uint8_t level)
{
    ASSERT((level >= 0) && (level <= 31));
    QSPIREG(base, QSPI_DMATDLR) = level;
}

__STATIC_INLINE void
QSPI_setDmaRxDataLevel(uint32_t base, uint8_t level)
{
    ASSERT((level >= 1) && (level <= 32));
    QSPIREG(base, QSPI_DMARDLR) = level - 1;
}

__STATIC_INLINE void
QSPI_ctrlDmaTrans(uint32_t base, boolean enable)
{
    if(enable)
    {
        QSPIREG(base, QSPI_DMACR) |= BIT(1);
    }
    else
    {
        QSPIREG(base, QSPI_DMACR) &= ~BIT(1);
    }
}

__STATIC_INLINE void
QSPI_ctrlDmaRecv(uint32_t base, boolean enable)
{
    if(enable)
    {
        QSPIREG(base, QSPI_DMACR) |= BIT(0);
    }
    else
    {
        QSPIREG(base, QSPI_DMACR) &= ~BIT(0);
    }
}


__STATIC_INLINE uint32_t
QSPI_clearAllInterruptStatus(uint32_t base)
{
    return QSPIREG(base, QSPI_ICR);
}

__STATIC_INLINE void
QSPI_disableAllInterrupt(uint32_t base)
{
    QSPIREG(base, QSPI_IMR) = 0x00000FFFU;
}

__STATIC_INLINE void
QSPI_clearIntFlagTxFlow(uint32_t base)
{
    (void)(QSPIREG(base, QSPI_TXEICR));
}

__STATIC_INLINE void
QSPI_clearIntFlagRxOverflow(uint32_t base)
{
    (void)(QSPIREG(base, QSPI_RXOICR));
}

__STATIC_INLINE void
QSPI_clearIntFlagRxUnderflow(uint32_t base)
{
    (void)(QSPIREG(base, QSPI_RXUICR));
}

__STATIC_INLINE void
QSPI_clearIntFlagMst(uint32_t base)
{
    (void)(QSPIREG(base, QSPI_MSTICR));
}

__STATIC_INLINE void
QSPI_enableInterrupt(uint32_t base, uint32_t intFlags)
{
    //
    // Check the arguments.
    //
    ASSERT(QSPI_isBaseValid(base));

    QSPIREG(base, QSPI_IMR) |= intFlags;
}

static inline void
QSPI_ControlRegisterConfig(uint32_t base, uint32_t value)
{
	ASSERT(QSPI_isBaseValid(base));

	QSPI_disableModule(base);
	QSPIREG(base, QSPI_CTRLR0) |= value;
	QSPI_enableModule(base);
}

static inline void
QSPI_spiControlRegisterConfig(uint32_t base, uint32_t value)
{
	ASSERT(QSPI_isBaseValid(base));

	QSPI_disableModule(base);
	QSPIREG(base, QSPI_SPI_CTRLR0) |= value;
	QSPI_enableModule(base);
}

//*****************************************************************************
//
// QSPI_disableInterrupt
//
//*****************************************************************************
__STATIC_INLINE void
QSPI_disableInterrupt(uint32_t base, uint32_t intFlags)
{
    uint32_t regValue;
    //
    // Check the arguments.
    //
    ASSERT(QSPI_isBaseValid(base));

    QSPIREG(base, QSPI_IMR) &= (~intFlags);
}

//*****************************************************************************
//
// QSPI_getInterruptStatus
//
//*****************************************************************************
__STATIC_INLINE uint32_t
QSPI_getInterruptStatus(uint32_t base)
{
    uint32_t temp = 0;

    //
    // Check the arguments.
    //
    ASSERT(QSPI_isBaseValid(base));

    return QSPIREG(base, QSPI_ISR);
}

static inline uint32_t
QSPI_getStatus(uint32_t base)
{
    return QSPIREG(base, QSPI_SR);
}

//*****************************************************************************
//
// QSPI xip operation need dw_ssi disabled
//
//*****************************************************************************
__STATIC_INLINE void
QSPI_setXipIncrInst(uint32_t base, uint16_t opcode)
{
    uint32_t regValue;
    //
    // Check the arguments.
    //
    ASSERT(QSPI_isBaseValid(base));

    QSPIREG(base, QSPI_XIP_INCR_INST) = opcode;
}

__STATIC_INLINE void
QSPI_setXipWrapInst(uint32_t base, uint16_t opcode)
{
    uint32_t regValue;
    //
    // Check the arguments.
    //
    ASSERT(QSPI_isBaseValid(base));

    QSPIREG(base, QSPI_XIP_WRAP_INST) = opcode;
}

//*****************************************************************************
//
//! This function can be used to config SPI CTRLR0
//!
//! \param base specifies the QSPI module base address.
//! \param lspclkHz is the rate of the clock supplied to the QSPI module
//! (LSPCLK) in Hz.
//! \param param include buad,ctrlr0,spi_ctrlr0
//!
//! \return None.
//
//*****************************************************************************
extern void
QSPI_setConfig(uint32_t base, uint32_t lspclkHz,  QSPI_CtrlParam *pCtrl);
//*****************************************************************************
//
//! Configures the baud rate of the serial peripheral interface.
//!
//! \param base specifies the QSPI module base address.
//! \param lspclkHz is the rate of the clock supplied to the QSPI module
//! (LSPCLK) in Hz.
//! \param bitRate specifies the clock rate in Hz.
//!
//! This function configures the QSPI baud rate. The \e bitRate parameter
//! defines the bit rate for the QSPI.  This bit rate must satisfy the following
//! clock ratio criteria:
//!
//! - \e lspclkHz / \e bitRate = any even value between 2 and 65534.
//!
//! The peripheral clock is the low speed peripheral clock.  This value is
//! returned by SysCtl_getLowSpeedClock(), or it can be explicitly hard coded
//! if it is constant and known (to save the code/execution overhead of a call
//! to SysCtl_getLowSpeedClock()).
//!
//! \note QSPI_setConfig() also sets the baud rate. Use QSPI_setBaudRate()
//! if you wish to configure it separately from protocol and mode.
//!
//! \return None.
//
//*****************************************************************************
extern void
QSPI_setBaudRate(uint32_t base, uint32_t lspclkHz, uint32_t bitRate);

//*****************************************************************************
//
//! This function can be used to transmit a 24-bit word of data
//!
//! \param base specifies the QSPI module base address.
//! \param txData is the data to be transmitted over QSPI
//! \param txDelay specifies the number of serial clock cycles delay time after
//!        completion of perious word
//!
//! This function can be used to transmit a 24-bit word of data.
//! 24-bit word data is divided into three bytes of data.
//!
//! This function uses QSPI_pollingFIFOTransaction function.
//! QSPI character length is hardcoded to 8 (8bits) of character length
//!
//! \return None.
//
//*****************************************************************************
extern void
QSPI_transmit24Bits(uint32_t base, uint32_t data, uint16_t txDelay);

//*****************************************************************************
//
//! This function can be used to transmit a 32-bit word of data
//!
//! \param base specifies the QSPI module base address.
//! \param txData is the data to be transmitted over QSPI
//! \param txDelay specifies the number of serial clock cycles delay time after
//!        completion of perious word
//!
//! This function can be used to transmit a 32-bit word of data.
//! 32-bit word data is divided into four bytes of data.
//!
//! This function uses QSPI_pollingFIFOTransaction function.
//! QSPI character length is hardcoded to 8 (8bits) of character length
//!
//! \return None.
//
//*****************************************************************************
extern void
QSPI_transmit32Bits(uint32_t base, uint32_t data, uint16_t txDelay);



//*****************************************************************************
//
//! This function is used to receive a 16-bit word of data
//!
//! \param base specifies the QSPI module base address.
//! \param endianness specifies the endianess of received data
//! \param dummyData is the data which is transmitted to initiate
//!        QSPI transaction to receive QSPI data
//! \param txDelay specifies the number of serial clock cycles delay time after
//!        completion of perious word
//!
//! This function is used to receive a 16-bit word of data.
//! This function uses SPIpolling_FIFO_Transaction function.
//! QSPI character length is hardcoded to 8 (1 byte = 8 bits)of character length
//!
//! \return the received 16-bit word.
//
//*****************************************************************************
extern uint16_t
QSPI_receive16Bits(uint32_t base, QSPI_endianess endianness, uint16_t dummyData,
                  uint16_t txDelay);

//*****************************************************************************
//
//! This function is used to receive a 24-bit word of data
//!
//! \param base specifies the QSPI module base address.
//! \param endianness specifies the endianess of received data
//! \param dummyData is the data which is transmitted to initiate
//!        QSPI transaction to receive QSPI data
//! \param txDelay specifies the number of serial clock cycles delay time after
//!        completion of perious word
//!
//! This function is used to receive a 24-bit word of data.
//! This function uses SPIpolling_FIFO_Transaction function.
//! QSPI character length is hardcoded to 8 (1 byte = 8 bits)of character length
//!
//! \return the received 24-bit word.
//
//*****************************************************************************
extern uint32_t
QSPI_receive24Bits(uint32_t base, QSPI_endianess endianness, uint16_t dummyData,
                  uint16_t txDelay);

//*****************************************************************************
//
//! This function is used to receive a 32-bit word of data
//!
//! \param base specifies the QSPI module base address.
//! \param endianness specifies the endianess of received data
//! \param dummyData is the data which is transmitted to initiate
//!        QSPI transaction to receive QSPI data
//! \param txDelay specifies the number of serial clock cycles delay time after
//!        completion of perious word
//!
//! This function is used to receive a 32-bit word of data.
//! This function uses SPIpolling_FIFO_Transaction function.
//! QSPI character length is hardcoded to 8 (1 byte = 8 bits)of character length
//!
//! \return the received 32-bit word.
//
//*****************************************************************************
extern uint32_t
QSPI_receive32Bits(uint32_t base, QSPI_endianess endianness, uint16_t dummyData,
                  uint16_t txDelay);

//*****************************************************************************
//
//! This function is used to initiate QSPI transaction of specified character
//! length and 'N' words of transaction
//!
//! \param base specifies the QSPI module base address.
//! \param charLength specifies the QSPI character length of QSPI transaction
//! \param pTxBuffer specifies the pointer to transmit buffer
//! \param pRxBuffer specifies the pointer to receive buffer
//! \param numOfWords specified the number of data to be transmitted / received
//!
//! QSPI character length is configurable using charLength variable
//!
//! \return none
//
//*****************************************************************************
extern void
QSPI_pollingFIFOTransaction32(uint32_t base, uint16_t dataWidth,
                           uint32_t *pTxBuffer, uint32_t *pRxBuffer,
                           uint16_t numOfWords, uint16_t dummyData);

/*==================================================================================================
*                                        QSPI driver function prototypes
==================================================================================================*/
void QSPI_initConfig(uint32_t qspi_base, QSPI_INIT_PARAM *init_param);
uint32_t QSPI_syncSendRecv(uint32_t base, uint8_t *pTxBuf, uint8_t *pRxBuf, uint32_t nSize);
uint32_t QSPI_syncSend(uint32_t base, uint8_t *pTxBuf, uint32_t nSize);
uint32_t QSPI_syncRecv(uint32_t base, uint8_t *pRxBuf, uint32_t nSize);
uint32_t QSPI_SendFlsCmd(uint32_t base, uint8_t *pTxBuf, uint32_t nSize);
uint32_t QSPI_SendFlsData(uint32_t base, uint8_t *pTxData, uint32_t nSize);
uint32_t QSPI_recvFlsData(uint32_t base, uint8_t *pRxData, uint32_t nSize);
uint32_t QSPI_readClearFlags(uint32_t base);

uint32_t QSPI_programFlash(uint32_t base, uint32_t nCmd, uint32_t nAddr, uint32_t *pU32Buf, uint32_t nSize);
uint32_t QSPI_recvWords(uint32_t base, uint32_t u32CmdAddr, uint32_t *pU32Buf, uint32_t nSize);
uint32_t QSPI_sendWords(uint32_t base, uint32_t u32CmdAddr, uint32_t *pU32Buf, uint32_t nSize);

void     QSPI_enableQuadRead(uint32_t base);
void     QSPI_enableStdRead(uint32_t base);

uint32_t QSPI_quadFastReadWords(uint32_t base, uint32_t nCmd, uint32_t nAddr, uint32_t *pRxBuf, uint32_t nSize);

uint32_t QSPI_quadFastReadWordsInit(uint32_t base, uint32_t nCmd, uint32_t nAddr, uint32_t nSize);

uint32_t QSPI_enableXipMode(uint32_t base, uint32_t nTransferMode, uint32_t nCmd);

#endif

#ifdef __cplusplus
}
#endif

#endif /* DEVICE_DRIVERLIB_QSPI_H_ */
