/*
 *   Copyright (c) Gejian Semiconductors 2025
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
 #ifndef __REGS_MCAN_H__
 #define __REGS_MCAN_H__
 
 #ifdef __cplusplus
 extern "C" {
 #endif

#include "inc/hw_types.h"

/**
 * @brief MCAN Subsystem PID Register bitfield definitions.
 *
 * @param MINOR      Minor Revision (bits 5:0).
 * @param rsvd1      Reserved bits (bits 7:6).
 * @param MAJOR      Major Revision (bits 10:8).
 * @param rsvd2      Reserved bits (bits 15:11).
 * @param MODULE_ID  Module Identification Number (bits 27:16).
 * @param rsvd3      Reserved bits (bits 29:28).
 * @param SCHEME     PID Register Scheme (bits 31:30).
 */
 struct MCANSS_PID_BITS {
    uint16_t MINOR     : 6;
    uint16_t rsvd1     : 2;
    uint16_t MAJOR     : 3;
    uint16_t rsvd2     : 5;
    uint32_t MODULE_ID : 12;
    uint32_t rsvd3     : 2;
    uint32_t SCHEME    : 2;
};
union MCANSS_PID_REG {
    uint32_t all;
    struct MCANSS_PID_BITS bit;
};

/**
 * @brief MCAN Subsystem Control Register bitfield definitions.
 *
 * @param rsvd1          Reserved bits (bits 2:0).
 * @param DBGSUSP_FREE   Debug Suspend Free control bit (bit 3).
 * @param WAKEUPREQEN    Wakeup Request Enable bit (bit 4).
 * @param AUTOWAKEUP     Automatic Wakeup Enable bit (bit 5).
 * @param EXT_TS_CNTR_EN External Timestamp Counter Enable bit (bit 6).
 */
struct MCANSS_CTRL_BITS {
    uint16_t rsvd1          : 3;
    uint16_t DBGSUSP_FREE   : 1;
    uint16_t WAKEUPREQEN    : 1;
    uint16_t AUTOWAKEUP     : 1;
    uint16_t EXT_TS_CNTR_EN : 1;
    uint16_t rsvd2          : 9;
    uint32_t rsvd3          : 16;
};
union MCANSS_CTRL_REG {
    uint32_t all;
    struct MCANSS_CTRL_BITS bit;
};

/**
 * @brief MCAN Subsystem Status Register bitfield definitions.
 *
 * @param RESET          Soft Reset Status bit (bit 0).
 * @param MEM_INIT_DONE  Memory Initialization Done bit (bit 1).
 * @param ENABLE_FDOE    Flexible Data-rate Operation Enable bit (bit 2).
 * @param rsvd1          Reserved bits (bits 15:3).
 * @param rsvd2          Reserved bits (bits 31:16).
 */
struct MCANSS_STAT_BITS {
    uint16_t RESET         : 1;
    uint16_t MEM_INIT_DONE : 1;
    uint16_t ENABLE_FDOE   : 1;
    uint16_t rsvd1         : 13;
    uint32_t rsvd2         : 16;
};
union MCANSS_STAT_REG {
    uint32_t all;
    struct MCANSS_STAT_BITS bit;
};

/**
 * @brief MCAN Subsystem Interrupt Clear Shadow Register bitfield definitions.
 *
 * @param EXT_TS_CNTR_OVFL External Timestamp Counter Overflow Interrupt Clear bit (bit 0).
 * @param rsvd1            Reserved bits (bits 15:1).
 * @param rsvd2            Reserved bits (bits 31:16).
 */
struct MCANSS_ICS_BITS {
    uint16_t EXT_TS_CNTR_OVFL : 1;
    uint16_t rsvd1           : 15;
    uint32_t rsvd2           : 16;
};
union MCANSS_ICS_REG {
    uint32_t all;
    struct MCANSS_ICS_BITS bit;
};

/**
 * @brief MCAN Subsystem Interrupt Raw Status Register bitfield definitions.
 *
 * @param EXT_TS_CNTR_OVFL External Timestamp Counter Overflow Interrupt Status bit (bit 0).
 * @param rsvd1            Reserved bits (bits 15:1).
 * @param rsvd2            Reserved bits (bits 31:16).
 */
struct MCANSS_IRS_BITS {
    uint16_t EXT_TS_CNTR_OVFL : 1;
    uint16_t rsvd1           : 15;
    uint32_t rsvd2           : 16;
};
union MCANSS_IRS_REG {
    uint32_t all;
    struct MCANSS_IRS_BITS bit;
};

/**
 * @brief MCAN Subsystem Interrupt Enable Clear Shadow Register bitfield definitions.
 *
 * @param EXT_TS_CNTR_OVFL External Timestamp Counter Overflow Interrupt Enable Clear bit (bit 0).
 * @param rsvd1            Reserved bits (bits 15:1).
 * @param rsvd2            Reserved bits (bits 31:16).
 */
struct MCANSS_IECS_BITS {
    uint16_t EXT_TS_CNTR_OVFL : 1;
    uint16_t rsvd1           : 15;
    uint32_t rsvd2           : 16;
};
union MCANSS_IECS_REG {
    uint32_t all;
    struct MCANSS_IECS_BITS bit;
};

/**
 * @brief MCAN Subsystem Interrupt Enable Register bitfield definitions.
 *
 * @param EXT_TS_CNTR_OVFL External Timestamp Counter Overflow Interrupt Enable bit (bit 0).
 * @param rsvd1            Reserved bits (bits 15:1).
 * @param rsvd2            Reserved bits (bits 31:16).
 */
struct MCANSS_IE_BITS {
    uint16_t EXT_TS_CNTR_OVFL : 1;
    uint16_t rsvd1           : 15;
    uint32_t rsvd2           : 16;
};
union MCANSS_IE_REG {
    uint32_t all;
    struct MCANSS_IE_BITS bit;
};

/**
 * @brief MCAN Subsystem Interrupt Enable Status Register bitfield definitions.
 *
 * @param EXT_TS_CNTR_OVFL External Timestamp Counter Overflow Interrupt Enable Status bit (bit 0).
 * @param rsvd1            Reserved bits (bits 15:1).
 * @param rsvd2            Reserved bits (bits 31:16).
 */
struct MCANSS_IES_BITS {
    uint16_t EXT_TS_CNTR_OVFL : 1;
    uint16_t rsvd1           : 15;
    uint32_t rsvd2           : 16;
};
union MCANSS_IES_REG {
    uint32_t all;
    struct MCANSS_IES_BITS bit;
};

/**
 * @brief MCAN Subsystem End of Interrupt Register bitfield definitions.
 *
 * @param EOI   External Timestamp Counter Overflow End of Interrupt value (bits 7:0).
 * @param rsvd1 Reserved bits (bits 15:8).
 * @param rsvd2 Reserved bits (bits 31:16).
 */
struct MCANSS_EOI_BITS {
    uint16_t EOI   : 8;
    uint16_t rsvd1 : 8;
    uint32_t rsvd2 : 16;
};
union MCANSS_EOI_REG {
    uint32_t all;
    struct MCANSS_EOI_BITS bit;
};

/**
 * @brief MCAN Subsystem External Timestamp Prescaler Register bitfield definitions.
 *
 * @param PRESCALER External Timestamp Prescaler value (bits 23:0).
 * @param rsvd1     Reserved bits (bits 31:24).
 */
struct MCANSS_EXT_TS_PRESCALER_BITS {
    uint32_t PRESCALER : 24;
    uint32_t rsvd1     : 8;
};
union MCANSS_EXT_TS_PRESCALER_REG {
    uint32_t all;
    struct MCANSS_EXT_TS_PRESCALER_BITS bit;
};

/**
 * @brief MCAN Subsystem External Timestamp Unserviced Interrupt Counter bitfield definitions.
 *
 * @param EXT_TS_INTR_CNTR Count of unserviced External Timestamp Counter Overflow Interrupts (bits 4:0).
 * @param rsvd1            Reserved bits (bits 15:5).
 * @param rsvd2            Reserved bits (bits 31:16).
 */
struct MCANSS_EXT_TS_UNSERVICED_INTR_CNTR_BITS {
    uint16_t EXT_TS_INTR_CNTR : 5;
    uint16_t rsvd1           : 11;
    uint32_t rsvd2           : 16;
};
union MCANSS_EXT_TS_UNSERVICED_INTR_CNTR_REG {
    uint32_t all;
    struct MCANSS_EXT_TS_UNSERVICED_INTR_CNTR_BITS bit;
};

/**
 * @brief MCAN Subsystem Register Map.
 *
 * @param MCANSS_PID                   MCAN Subsystem Revision Register.
 * @param MCANSS_CTRL                  MCAN Subsystem Control Register.
 * @param MCANSS_STAT                  MCAN Subsystem Status Register.
 * @param MCANSS_ICS                   MCAN Subsystem Interrupt Clear Shadow Register.
 * @param MCANSS_IRS                   MCAN Subsystem Interrupt Raw Status Register.
 * @param MCANSS_IECS                  MCAN Subsystem Interrupt Enable Clear Shadow Register.
 * @param MCANSS_IE                    MCAN Subsystem Interrupt Enable Register.
 * @param MCANSS_IES                   MCAN Subsystem Interrupt Enable Status Register.
 * @param MCANSS_EOI                   MCAN Subsystem End of Interrupt Register.
 * @param MCANSS_EXT_TS_PRESCALER      MCAN Subsystem External Timestamp Prescaler Register.
 * @param MCANSS_EXT_TS_UNSERVICED_INTR_CNTR MCAN Subsystem External Timestamp Unserviced Interrupt Counter Register.
 */
struct MCANSS_REGS {
    union MCANSS_PID_REG                   PID;
    union MCANSS_CTRL_REG                  CTRL;
    union MCANSS_STAT_REG                  STAT;
    union MCANSS_ICS_REG                   ICS;
    union MCANSS_IRS_REG                   IRS;
    union MCANSS_IECS_REG                  IECS;
    union MCANSS_IE_REG                    IE;
    union MCANSS_IES_REG                   IES;
    union MCANSS_EOI_REG                   EOI;
    union MCANSS_EXT_TS_PRESCALER_REG      EXT_TS_PRESCALER;
    union MCANSS_EXT_TS_UNSERVICED_INTR_CNTR_REG EXT_TS_UNSERVICED_INTR_CNTR;
};

/**
 * @brief MCAN Core Release Register bitfield definitions.
 *
 * @param DAY      Time Stamp Day (bits 7:0).
 * @param MON      Time Stamp Month (bits 15:8).
 * @param YEAR     Time Stamp Year (bits 19:16).
 * @param SUBSTEP  Sub-Step of Core Release (bits 23:20).
 * @param STEP     Step of Core Release (bits 27:24).
 * @param REL      Core Release (bits 31:28).
 */
struct MCAN_CREL_BITS {
    uint16_t DAY     : 8;
    uint16_t MON     : 8;
    uint32_t YEAR    : 4;
    uint32_t SUBSTEP : 4;
    uint32_t STEP    : 4;
    uint32_t REL     : 4;
};
union MCAN_CREL_REG {
    uint32_t all;
    struct MCAN_CREL_BITS bit;
};

/**
 * @brief MCAN Data Bit Timing and Prescaler Register bitfield definitions.
 *
 * @param DSJW   Data Resynchronization Jump Width (bits 3:0).
 * @param DTSEG2 Data Time Segment After Sample Point (bits 7:4).
 * @param DTSEG1 Data Time Segment Before Sample Point (bits 12:8).
 * @param rsvd1  Reserved bits (bits 15:13).
 * @param DBRP   Data Bit Rate Prescaler (bits 20:16).
 * @param rsvd2  Reserved bits (bits 22:21).
 * @param TDC    Transmitter Delay Compensation Enable (bit 23).
 * @param rsvd3  Reserved bits (bits 31:24).
 */
struct MCAN_DBTP_BITS {
    uint16_t DSJW   : 4;
    uint16_t DTSEG2 : 4;
    uint16_t DTSEG1 : 5;
    uint16_t rsvd1  : 3;
    uint32_t DBRP   : 5;
    uint32_t rsvd2  : 2;
    uint32_t TDC    : 1;
    uint32_t rsvd3  : 8;
};
union MCAN_DBTP_REG {
    uint32_t all;
    struct MCAN_DBTP_BITS bit;
};

/**
 * @brief MCAN Test Register bitfield definitions.
 *
 * @param rsvd1 Reserved bits (bits 3:0).
 * @param LBCK  Loop Back Mode Enable (bit 4).
 * @param TX    Control of Transmit Pin (bits 6:5).
 * @param RX    Control of Receive Pin (bit 7).
 * @param rsvd2 Reserved bits (bits 15:8).
 * @param rsvd3 Reserved bits (bits 31:16).
 */
struct MCAN_TEST_BITS {
    uint16_t rsvd1 : 4;
    uint16_t LBCK  : 1;
    uint16_t TX    : 2;
    uint16_t RX    : 1;
    uint16_t rsvd2 : 8;
    uint32_t rsvd3 : 16;
};
union MCAN_TEST_REG {
    uint32_t all;
    struct MCAN_TEST_BITS bit;
};

/**
 * @brief MCAN Watchdog Register bitfield definitions.
 *
 * @param WDC   Watchdog Configuration (bits 7:0).
 * @param WDV   Watchdog Value (bits 15:8).
 * @param rsvd1 Reserved bits (bits 31:16).
 */
struct MCAN_RWD_BITS {
    uint16_t WDC   : 8;
    uint16_t WDV   : 8;
    uint32_t rsvd1 : 16;
};
union MCAN_RWD_REG {
    uint32_t all;
    struct MCAN_RWD_BITS bit;
};
/**
 * @brief MCAN Configuration Control Register bitfield definitions.
 *
 * @param INIT   Initialization (bit 0).
 * @param CCE    Configuration Change Enable (bit 1).
 * @param ASM    Restricted Operation Mode (bit 2).
 * @param CSA    Clock Stop Acknowledge (bit 3).
 * @param CSR    Clock Stop Request (bit 4).
 * @param MON    Bus Monitoring Mode (bit 5).
 * @param DAR    Disable Automatic Retransmission (bit 6).
 * @param TEST   Test Mode Enable (bit 7).
 * @param FDOE   Flexible Datarate Operation Enable (bit 8).
 * @param BRSE   Bit Rate Switch Enable (bit 9).
 * @param rsvd1  Reserved bits (bits 10:11).
 * @param PXHD   Protocol Exception Handling Disable (bit 12).
 * @param EFBI   Edge Filtering During Bus Integration (bit 13).
 * @param TXP    Transmit Pause (bit 14).
 * @param NISO   Non-ISO Operation (bit 15).
 * @param rsvd2  Reserved bits (bits 16:31).
 */
 struct MCAN_CCCR_BITS {
    uint16_t INIT  : 1;
    uint16_t CCE   : 1;
    uint16_t ASM   : 1;
    uint16_t CSA   : 1;
    uint16_t CSR   : 1;
    uint16_t MON   : 1;
    uint16_t DAR   : 1;
    uint16_t TEST  : 1;
    uint16_t FDOE  : 1;
    uint16_t BRSE  : 1;
    uint16_t rsvd1 : 2;
    uint16_t PXHD  : 1;
    uint16_t EFBI  : 1;
    uint16_t TXP   : 1;
    uint16_t NISO  : 1;
    uint32_t rsvd2 : 16;
};
union MCAN_CCCR_REG {
    uint32_t all;
    struct MCAN_CCCR_BITS bit;
};

/**
 * @brief MCAN Nominal Bit Timing and Prescaler Register bitfield definitions.
 *
 * @param NTSEG2 Nominal Time Segment After Sample Point (bits 6:0).
 * @param rsvd1  Reserved bit (bit 7).
 * @param NTSEG1 Nominal Time Segment Before Sample Point (bits 15:8).
 * @param NBRP   Nominal Bit Rate Prescaler (bits 24:16).
 * @param NSJW   Nominal (Re)Synchronization Jump Width (bits 31:25).
 */
struct MCAN_NBTP_BITS {
    uint16_t NTSEG2 : 7;
    uint16_t rsvd1  : 1;
    uint16_t NTSEG1 : 8;
    uint32_t NBRP   : 9;
    uint32_t NSJW   : 7;
};
union MCAN_NBTP_REG {
    uint32_t all;
    struct MCAN_NBTP_BITS bit;
};

/**
 * @brief MCAN Timestamp Control Register bitfield definitions.
 *
 * @param TSS    Timestamp Select (bits 1:0).
 * @param rsvd1  Reserved bits (bits 15:2).
 * @param TCP    Timestamp Counter Prescaler (bits 19:16).
 * @param rsvd2  Reserved bits (bits 31:20).
 */
struct MCAN_TSCC_BITS {
    uint16_t TSS   : 2;
    uint16_t rsvd1 : 14;
    uint32_t TCP   : 4;
    uint32_t rsvd2 : 12;
};
union MCAN_TSCC_REG {
    uint32_t all;
    struct MCAN_TSCC_BITS bit;
};

/**
 * @brief MCAN Timestamp Counter Value Register bitfield definitions.
 *
 * @param TSC    Timestamp Counter (bits 15:0).
 * @param rsvd1  Reserved bits (bits 31:16).
 */
struct MCAN_TSCV_BITS {
    uint16_t TSC   : 16;
    uint32_t rsvd1 : 16;
};
union MCAN_TSCV_REG {
    uint32_t all;
    struct MCAN_TSCV_BITS bit;
};

/**
 * @brief MCAN Timeout Counter Control Register bitfield definitions.
 *
 * @param ETOC   Enable Timeout Counter (bit 0).
 * @param TOS    Timeout Select (bits 2:1).
 * @param rsvd1  Reserved bits (bits 15:3).
 * @param TOP    Timeout Period (bits 31:16).
 */
struct MCAN_TOCC_BITS {
    uint16_t ETOC  : 1;
    uint16_t TOS   : 2;
    uint16_t rsvd1 : 13;
    uint32_t TOP   : 16;
};
union MCAN_TOCC_REG {
    uint32_t all;
    struct MCAN_TOCC_BITS bit;
};

/**
 * @brief MCAN Timeout Counter Value Register bitfield definitions.
 *
 * @param TOC    Timeout Counter (bits 15:0).
 * @param rsvd1  Reserved bits (bits 31:16).
 */
struct MCAN_TOCV_BITS {
    uint16_t TOC   : 16;
    uint32_t rsvd1 : 16;
};
union MCAN_TOCV_REG {
    uint32_t all;
    struct MCAN_TOCV_BITS bit;
};

/**
 * @brief MCAN Error Counter Register bitfield definitions.
 *
 * @param TEC    Transmit Error Counter (bits 7:0).
 * @param REC    Receive Error Counter (bits 14:8).
 * @param RP     Receive Error Passive (bit 15).
 * @param CEL    CAN Error Logging (bits 23:16).
 * @param rsvd1  Reserved bits (bits 31:24).
 */
struct MCAN_ECR_BITS {
    uint16_t TEC   : 8;
    uint16_t REC   : 7;
    uint16_t RP    : 1;
    uint32_t CEL   : 8;
    uint32_t rsvd1 : 8;
};
union MCAN_ECR_REG {
    uint32_t all;
    struct MCAN_ECR_BITS bit;
};
/**
 * @brief MCAN Protocol Status Register bitfield definitions.
 *
 * @param LEC    Last Error Code (bits 2:0).
 * @param ACT    Node Activity (bits 4:3).
 * @param EP     Error Passive (bit 5).
 * @param EW     Warning Status (bit 6).
 * @param BO     Bus_Off Status (bit 7).
 * @param DLEC   Data Phase Last Error Code (bits 10:8).
 * @param RESI   ESI Flag of Last Received CAN FD Message (bit 11).
 * @param RBRS   BRS Flag of Last Received CAN FD Message (bit 12).
 * @param RFDF   Received a CAN FD Message (bit 13).
 * @param PXE    Protocol Exception Event (bit 14).
 * @param rsvd1  Reserved bit (bit 15).
 * @param TDCV   Transmitter Delay Compensation Value (bits 22:16).
 * @param rsvd2  Reserved bits (bits 31:23).
 */
 struct MCAN_PSR_BITS {
    uint16_t LEC    : 3;
    uint16_t ACT    : 2;
    uint16_t EP     : 1;
    uint16_t EW     : 1;
    uint16_t BO     : 1;
    uint16_t DLEC   : 3;
    uint16_t RESI   : 1;
    uint16_t RBRS   : 1;
    uint16_t RFDF   : 1;
    uint16_t PXE    : 1;
    uint16_t rsvd1  : 1;
    uint32_t TDCV   : 7;
    uint32_t rsvd2  : 9;
};
union MCAN_PSR_REG {
    uint32_t all;
    struct MCAN_PSR_BITS bit;
};

/**
 * @brief MCAN Transmitter Delay Compensation Register bitfield definitions.
 *
 * @param TDCF   Transmitter Delay Compensation Filter Window Length (bits 6:0).
 * @param rsvd1  Reserved bit (bit 7).
 * @param TDCO   Transmitter Delay Compensation Offset (bits 14:8).
 * @param rsvd2  Reserved bit (bit 15).
 * @param rsvd3  Reserved bits (bits 31:16).
 */
struct MCAN_TDCR_BITS {
    uint16_t TDCF   : 7;
    uint16_t rsvd1  : 1;
    uint16_t TDCO   : 7;
    uint16_t rsvd2  : 1;
    uint32_t rsvd3  : 16;
};
union MCAN_TDCR_REG {
    uint32_t all;
    struct MCAN_TDCR_BITS bit;
};

/**
 * @brief MCAN Interrupt Register bitfield definitions.
 *
 * @param RF0N   Rx FIFO 0 New Message (bit 0).
 * @param RF0W   Rx FIFO 0 Watermark Reached (bit 1).
 * @param RF0F   Rx FIFO 0 Full (bit 2).
 * @param RF0L   Rx FIFO 0 Message Lost (bit 3).
 * @param RF1N   Rx FIFO 1 New Message (bit 4).
 * @param RF1W   Rx FIFO 1 Watermark Reached (bit 5).
 * @param RF1F   Rx FIFO 1 Full (bit 6).
 * @param RF1L   Rx FIFO 1 Message Lost (bit 7).
 * @param HPM    High Priority Message (bit 8).
 * @param TC     Transmission Completed (bit 9).
 * @param TCF    Transmission Cancellation Finished (bit 10).
 * @param TFE    Tx FIFO Empty (bit 11).
 * @param TEFN   Tx Event FIFO New Entry (bit 12).
 * @param TEFW   Tx Event FIFO Watermark Reached (bit 13).
 * @param TEFF   Tx Event FIFO Full (bit 14).
 * @param TEFL   Tx Event FIFO Element Lost (bit 15).
 * @param TSW    Timestamp Wraparound (bit 16).
 * @param MRAF   Message RAM Access Failure (bit 17).
 * @param TOO    Timeout Occurred (bit 18).
 * @param DRX    Message Stored to Dedicated Rx Buffer (bit 19).
 * @param rsvd1  Reserved bit (bit 20).
 * @param BEU    Bit Error Uncorrected (bit 21).
 * @param ELO    Error Logging Overflow (bit 22).
 * @param EP     Error Passive (bit 23).
 * @param EW     Warning Status (bit 24).
 * @param BO     Bus_Off Status (bit 25).
 * @param WDI    Watchdog Interrupt (bit 26).
 * @param PEA    Protocol Error in Arbitration Phase (bit 27).
 * @param PED    Protocol Error in Data Phase (bit 28).
 * @param ARA    Access to Reserved Address (bit 29).
 * @param rsvd2  Reserved bit (bit 30).
 * @param rsvd3  Reserved bit (bit 31).
 */
struct MCAN_IR_BITS {
    uint16_t RF0N  : 1;
    uint16_t RF0W  : 1;
    uint16_t RF0F  : 1;
    uint16_t RF0L  : 1;
    uint16_t RF1N  : 1;
    uint16_t RF1W  : 1;
    uint16_t RF1F  : 1;
    uint16_t RF1L  : 1;
    uint16_t HPM   : 1;
    uint16_t TC    : 1;
    uint16_t TCF   : 1;
    uint16_t TFE   : 1;
    uint16_t TEFN  : 1;
    uint16_t TEFW  : 1;
    uint16_t TEFF  : 1;
    uint16_t TEFL  : 1;
    uint32_t TSW   : 1;
    uint32_t MRAF  : 1;
    uint32_t TOO   : 1;
    uint32_t DRX   : 1;
    uint32_t rsvd1 : 1;
    uint32_t BEU   : 1;
    uint32_t ELO   : 1;
    uint32_t EP    : 1;
    uint32_t EW    : 1;
    uint32_t BO    : 1;
    uint32_t WDI   : 1;
    uint32_t PEA   : 1;
    uint32_t PED   : 1;
    uint32_t ARA   : 1;
    uint32_t rsvd2 : 1;
    uint32_t rsvd3 : 1;
};
union MCAN_IR_REG {
    uint32_t all;
    struct MCAN_IR_BITS bit;
};

/**
 * @brief MCAN Interrupt Enable Register bitfield definitions.
 *
 * @param RF0NE  Rx FIFO 0 New Message Enable (bit 0).
 * @param RF0WE  Rx FIFO 0 Watermark Reached Enable (bit 1).
 * @param RF0FE  Rx FIFO 0 Full Enable (bit 2).
 * @param RF0LE  Rx FIFO 0 Message Lost Enable (bit 3).
 * @param RF1NE  Rx FIFO 1 New Message Enable (bit 4).
 * @param RF1WE  Rx FIFO 1 Watermark Reached Enable (bit 5).
 * @param RF1FE  Rx FIFO 1 Full Enable (bit 6).
 * @param RF1LE  Rx FIFO 1 Message Lost Enable (bit 7).
 * @param HPME   High Priority Message Enable (bit 8).
 * @param TCE    Transmission Completed Enable (bit 9).
 * @param TCFE   Transmission Cancellation Finished Enable (bit 10).
 * @param TFEE   Tx FIFO Empty Enable (bit 11).
 * @param TEFNE  Tx Event FIFO New Entry Enable (bit 12).
 * @param TEFWE  Tx Event FIFO Watermark Reached Enable (bit 13).
 * @param TEFFE  Tx Event FIFO Full Enable (bit 14).
 * @param TEFLE  Tx Event FIFO Element Lost Enable (bit 15).
 * @param TSWE   Timestamp Wraparound Enable (bit 16).
 * @param MRAFE  Message RAM Access Failure Enable (bit 17).
 * @param TOOE   Timeout Occurred Enable (bit 18).
 * @param DRXE   Message Stored to Dedicated Rx Buffer Enable (bit 19).
 * @param BECE   Bit Error Corrected Enable (bit 20).
 * @param BEUE   Bit Error Uncorrected Enable (bit 21).
 * @param ELOE   Error Logging Overflow Enable (bit 22).
 * @param EPE    Error Passive Enable (bit 23).
 * @param EWE    Warning Status Enable (bit 24).
 * @param BOE    Bus_Off Status Enable (bit 25).
 * @param WDIE   Watchdog Interrupt Enable (bit 26).
 * @param PEAE   Protocol Error in Arbitration Phase Enable (bit 27).
 * @param PEDE   Protocol Error in Data Phase Enable (bit 28).
 * @param ARAE   Access to Reserved Address Enable (bit 29).
 * @param rsvd1  Reserved bits (bits 31:30).
 */
struct MCAN_IE_BITS {
    uint16_t RF0NE : 1;
    uint16_t RF0WE : 1;
    uint16_t RF0FE : 1;
    uint16_t RF0LE : 1;
    uint16_t RF1NE : 1;
    uint16_t RF1WE : 1;
    uint16_t RF1FE : 1;
    uint16_t RF1LE : 1;
    uint16_t HPME  : 1;
    uint16_t TCE   : 1;
    uint16_t TCFE  : 1;
    uint16_t TFEE  : 1;
    uint16_t TEFNE : 1;
    uint16_t TEFWE : 1;
    uint16_t TEFFE : 1;
    uint16_t TEFLE : 1;
    uint32_t TSWE  : 1;
    uint32_t MRAFE : 1;
    uint32_t TOOE  : 1;
    uint32_t DRXE  : 1;
    uint32_t BECE  : 1;
    uint32_t BEUE  : 1;
    uint32_t ELOE  : 1;
    uint32_t EPE   : 1;
    uint32_t EWE   : 1;
    uint32_t BOE   : 1;
    uint32_t WDIE  : 1;
    uint32_t PEAE  : 1;
    uint32_t PEDE  : 1;
    uint32_t ARAE  : 1;
    uint32_t rsvd1 : 2;
};
union MCAN_IE_REG {
    uint32_t all;
    struct MCAN_IE_BITS bit;
};

/**
 * @brief MCAN Interrupt Line Select Register bitfield definitions.
 *
 * @param RF0NL  Rx FIFO 0 New Message Line (bit 0).
 * @param RF0WL  Rx FIFO 0 Watermark Reached Line (bit 1).
 * @param RF0FL  Rx FIFO 0 Full Line (bit 2).
 * @param RF0LL  Rx FIFO 0 Message Lost Line (bit 3).
 * @param RF1NL  Rx FIFO 1 New Message Line (bit 4).
 * @param RF1WL  Rx FIFO 1 Watermark Reached Line (bit 5).
 * @param RF1FL  Rx FIFO 1 Full Line (bit 6).
 * @param RF1LL  Rx FIFO 1 Message Lost Line (bit 7).
 * @param HPML   High Priority Message Line (bit 8).
 * @param TCL    Transmission Completed Line (bit 9).
 * @param TCFL   Transmission Cancellation Finished Line (bit 10).
 * @param TFEL   Tx FIFO Empty Line (bit 11).
 * @param TEFNL  Tx Event FIFO New Entry Line (bit 12).
 * @param TEFWL  Tx Event FIFO Watermark Reached Line (bit 13).
 * @param TEFFL  Tx Event FIFO Full Line (bit 14).
 * @param TEFLL  Tx Event FIFO Element Lost Line (bit 15).
 * @param TSWL   Timestamp Wraparound Line (bit 16).
 * @param MRAFL  Message RAM Access Failure Line (bit 17).
 * @param TOOL   Timeout Occurred Line (bit 18).
 * @param DRXL   Message Stored to Dedicated Rx Buffer Line (bit 19).
 * @param BECL   Bit Error Corrected Line (bit 20).
 * @param BEUL   Bit Error Uncorrected Line (bit 21).
 * @param ELOL   Error Logging Overflow Line (bit 22).
 * @param EPL    Error Passive Line (bit 23).
 * @param EWL    Warning Status Line (bit 24).
 * @param BOL    Bus_Off Status Line (bit 25).
 * @param WDIL   Watchdog Interrupt Line (bit 26).
 * @param PEAL   Protocol Error in Arbitration Phase Line (bit 27).
 * @param PEDL   Protocol Error in Data Phase Line (bit 28).
 * @param ARAL   Access to Reserved Address Line (bit 29).
 * @param rsvd1  Reserved bits (bits 31:30).
 */
struct MCAN_ILS_BITS {
    uint16_t RF0NL : 1;
    uint16_t RF0WL : 1;
    uint16_t RF0FL : 1;
    uint16_t RF0LL : 1;
    uint16_t RF1NL : 1;
    uint16_t RF1WL : 1;
    uint16_t RF1FL : 1;
    uint16_t RF1LL : 1;
    uint16_t HPML  : 1;
    uint16_t TCL   : 1;
    uint16_t TCFL  : 1;
    uint16_t TFEL  : 1;
    uint16_t TEFNL : 1;
    uint16_t TEFWL : 1;
    uint16_t TEFFL : 1;
    uint16_t TEFLL : 1;
    uint32_t TSWL  : 1;
    uint32_t MRAFL : 1;
    uint32_t TOOL  : 1;
    uint32_t DRXL  : 1;
    uint32_t BECL  : 1;
    uint32_t BEUL  : 1;
    uint32_t ELOL  : 1;
    uint32_t EPL   : 1;
    uint32_t EWL   : 1;
    uint32_t BOL   : 1;
    uint32_t WDIL  : 1;
    uint32_t PEAL  : 1;
    uint32_t PEDL  : 1;
    uint32_t ARAL  : 1;
    uint32_t rsvd1 : 2;
};
union MCAN_ILS_REG {
    uint32_t all;
    struct MCAN_ILS_BITS bit;
};

/**
 * @brief MCAN Interrupt Line Enable Register bitfield definitions.
 *
 * @param EINT0  Enable Interrupt Line 0 (bit 0).
 * @param EINT1  Enable Interrupt Line 1 (bit 1).
 * @param rsvd1  Reserved bits (bits 15:2).
 * @param rsvd2  Reserved bits (bits 31:16).
 */
struct MCAN_ILE_BITS {
    uint16_t EINT0 : 1;
    uint16_t EINT1 : 1;
    uint16_t rsvd1 : 14;
    uint32_t rsvd2 : 16;
};
union MCAN_ILE_REG {
    uint32_t all;
    struct MCAN_ILE_BITS bit;
};

/**
 * @brief MCAN Global Filter Configuration Register bitfield definitions.
 *
 * @param RRFE   Reject Remote Frames Extended (bit 0).
 * @param RRFS   Reject Remote Frames Standard (bit 1).
 * @param ANFE   Accept Non-matching Frames Extended (bits 3:2).
 * @param ANFS   Accept Non-matching Frames Standard (bits 5:4).
 * @param rsvd1  Reserved bits (bits 15:6).
 * @param rsvd2  Reserved bits (bits 31:16).
 */
struct MCAN_GFC_BITS {
    uint16_t RRFE  : 1;
    uint16_t RRFS  : 1;
    uint16_t ANFE  : 2;
    uint16_t ANFS  : 2;
    uint16_t rsvd1 : 10;
    uint32_t rsvd2 : 16;
};
union MCAN_GFC_REG {
    uint32_t all;
    struct MCAN_GFC_BITS bit;
};
/**
 * @brief MCAN Standard ID Filter Configuration Register bitfield definitions.
 *
 * @param rsvd1      Reserved bits (bits 1:0).
 * @param FLSSA      Filter List Standard Start Address (bits 15:2).
 * @param LSS        List Size Standard (bits 23:16).
 * @param rsvd2      Reserved bits (bits 31:24).
 */
 struct MCAN_SIDFC_BITS {
    uint16_t rsvd1 : 2;
    uint16_t FLSSA : 14;
    uint32_t LSS   : 8;
    uint32_t rsvd2 : 8;
};
union MCAN_SIDFC_REG {
    uint32_t all;
    struct MCAN_SIDFC_BITS bit;
};

/**
 * @brief MCAN Extended ID Filter Configuration Register bitfield definitions.
 *
 * @param rsvd1      Reserved bits (bits 1:0).
 * @param FLESA      Filter List Extended Start Address (bits 15:2).
 * @param LSE        List Size Extended (bits 22:16).
 * @param rsvd2      Reserved bits (bits 31:23).
 */
struct MCAN_XIDFC_BITS {
    uint16_t rsvd1 : 2;
    uint16_t FLESA : 14;
    uint32_t LSE   : 7;
    uint32_t rsvd2 : 9;
};
union MCAN_XIDFC_REG {
    uint32_t all;
    struct MCAN_XIDFC_BITS bit;
};

/**
 * @brief MCAN Extended ID Mask Register bitfield definitions.
 *
 * @param EIDM       Extended ID Mask (bits 28:0).
 * @param rsvd1      Reserved bits (bits 31:29).
 */
struct MCAN_XIDAM_BITS {
    uint32_t EIDM  : 29;
    uint32_t rsvd1 : 3;
};
union MCAN_XIDAM_REG {
    uint32_t all;
    struct MCAN_XIDAM_BITS bit;
};

/**
 * @brief MCAN High Priority Message Status Register bitfield definitions.
 *
 * @param BIDX       Buffer Index (bits 5:0).
 * @param MSI        Message Storage Indicator (bits 7:6).
 * @param FIDX       Filter Index (bits 14:8).
 * @param FLST       Filter List (bit 15).
 * @param rsvd1      Reserved bits (bits 31:16).
 */
struct MCAN_HPMS_BITS {
    uint16_t BIDX  : 6;
    uint16_t MSI   : 2;
    uint16_t FIDX  : 7;
    uint16_t FLST  : 1;
    uint32_t rsvd1 : 16;
};
union MCAN_HPMS_REG {
    uint32_t all;
    struct MCAN_HPMS_BITS bit;
};

/**
 * @brief MCAN New Data RX Buffer Register 1 bitfield definitions.
 *
 * @param ND0        New Data RX Buffer 0 (bit 0).
 * @param ND1        New Data RX Buffer 1 (bit 1).
 * @param ND2        New Data RX Buffer 2 (bit 2).
 * @param ND3        New Data RX Buffer 3 (bit 3).
 * @param ND4        New Data RX Buffer 4 (bit 4).
 * @param ND5        New Data RX Buffer 5 (bit 5).
 * @param ND6        New Data RX Buffer 6 (bit 6).
 * @param ND7        New Data RX Buffer 7 (bit 7).
 * @param ND8        New Data RX Buffer 8 (bit 8).
 * @param ND9        New Data RX Buffer 9 (bit 9).
 * @param ND10       New Data RX Buffer 10 (bit 10).
 * @param ND11       New Data RX Buffer 11 (bit 11).
 * @param ND12       New Data RX Buffer 12 (bit 12).
 * @param ND13       New Data RX Buffer 13 (bit 13).
 * @param ND14       New Data RX Buffer 14 (bit 14).
 * @param ND15       New Data RX Buffer 15 (bit 15).
 * @param ND16       New Data RX Buffer 16 (bit 16).
 * @param ND17       New Data RX Buffer 17 (bit 17).
 * @param ND18       New Data RX Buffer 18 (bit 18).
 * @param ND19       New Data RX Buffer 19 (bit 19).
 * @param ND20       New Data RX Buffer 20 (bit 20).
 * @param ND21       New Data RX Buffer 21 (bit 21).
 * @param ND22       New Data RX Buffer 22 (bit 22).
 * @param ND23       New Data RX Buffer 23 (bit 23).
 * @param ND24       New Data RX Buffer 24 (bit 24).
 * @param ND25       New Data RX Buffer 25 (bit 25).
 * @param ND26       New Data RX Buffer 26 (bit 26).
 * @param ND27       New Data RX Buffer 27 (bit 27).
 * @param ND28       New Data RX Buffer 28 (bit 28).
 * @param ND29       New Data RX Buffer 29 (bit 29).
 * @param ND30       New Data RX Buffer 30 (bit 30).
 * @param ND31       New Data RX Buffer 31 (bit 31).
 */
struct MCAN_NDAT1_BITS {
    uint16_t ND0  : 1;
    uint16_t ND1  : 1;
    uint16_t ND2  : 1;
    uint16_t ND3  : 1;
    uint16_t ND4  : 1;
    uint16_t ND5  : 1;
    uint16_t ND6  : 1;
    uint16_t ND7  : 1;
    uint16_t ND8  : 1;
    uint16_t ND9  : 1;
    uint16_t ND10 : 1;
    uint16_t ND11 : 1;
    uint16_t ND12 : 1;
    uint16_t ND13 : 1;
    uint16_t ND14 : 1;
    uint16_t ND15 : 1;
    uint32_t ND16 : 1;
    uint32_t ND17 : 1;
    uint32_t ND18 : 1;
    uint32_t ND19 : 1;
    uint32_t ND20 : 1;
    uint32_t ND21 : 1;
    uint32_t ND22 : 1;
    uint32_t ND23 : 1;
    uint32_t ND24 : 1;
    uint32_t ND25 : 1;
    uint32_t ND26 : 1;
    uint32_t ND27 : 1;
    uint32_t ND28 : 1;
    uint32_t ND29 : 1;
    uint32_t ND30 : 1;
    uint32_t ND31 : 1;
};
union MCAN_NDAT1_REG {
    uint32_t all;
    struct MCAN_NDAT1_BITS bit;
};

/**
 * @brief MCAN New Data RX Buffer Register 2 bitfield definitions.
 *
 * @param ND32       New Data RX Buffer 32 (bit 0).
 * @param ND33       New Data RX Buffer 33 (bit 1).
 * @param ND34       New Data RX Buffer 34 (bit 2).
 * @param ND35       New Data RX Buffer 35 (bit 3).
 * @param ND36       New Data RX Buffer 36 (bit 4).
 * @param ND37       New Data RX Buffer 37 (bit 5).
 * @param ND38       New Data RX Buffer 38 (bit 6).
 * @param ND39       New Data RX Buffer 39 (bit 7).
 * @param ND40       New Data RX Buffer 40 (bit 8).
 * @param ND41       New Data RX Buffer 41 (bit 9).
 * @param ND42       New Data RX Buffer 42 (bit 10).
 * @param ND43       New Data RX Buffer 43 (bit 11).
 * @param ND44       New Data RX Buffer 44 (bit 12).
 * @param ND45       New Data RX Buffer 45 (bit 13).
 * @param ND46       New Data RX Buffer 46 (bit 14).
 * @param ND47       New Data RX Buffer 47 (bit 15).
 * @param ND48       New Data RX Buffer 48 (bit 16).
 * @param ND49       New Data RX Buffer 49 (bit 17).
 * @param ND50       New Data RX Buffer 50 (bit 18).
 * @param ND51       New Data RX Buffer 51 (bit 19).
 * @param ND52       New Data RX Buffer 52 (bit 20).
 * @param ND53       New Data RX Buffer 53 (bit 21).
 * @param ND54       New Data RX Buffer 54 (bit 22).
 * @param ND55       New Data RX Buffer 55 (bit 23).
 * @param ND56       New Data RX Buffer 56 (bit 24).
 * @param ND57       New Data RX Buffer 57 (bit 25).
 * @param ND58       New Data RX Buffer 58 (bit 26).
 * @param ND59       New Data RX Buffer 59 (bit 27).
 * @param ND60       New Data RX Buffer 60 (bit 28).
 * @param ND61       New Data RX Buffer 61 (bit 29).
 * @param ND62       New Data RX Buffer 62 (bit 30).
 * @param ND63       New Data RX Buffer 63 (bit 31).
 */
struct MCAN_NDAT2_BITS {
    uint16_t ND32 : 1;
    uint16_t ND33 : 1;
    uint16_t ND34 : 1;
    uint16_t ND35 : 1;
    uint16_t ND36 : 1;
    uint16_t ND37 : 1;
    uint16_t ND38 : 1;
    uint16_t ND39 : 1;
    uint16_t ND40 : 1;
    uint16_t ND41 : 1;
    uint16_t ND42 : 1;
    uint16_t ND43 : 1;
    uint16_t ND44 : 1;
    uint16_t ND45 : 1;
    uint16_t ND46 : 1;
    uint16_t ND47 : 1;
    uint32_t ND48 : 1;
    uint32_t ND49 : 1;
    uint32_t ND50 : 1;
    uint32_t ND51 : 1;
    uint32_t ND52 : 1;
    uint32_t ND53 : 1;
    uint32_t ND54 : 1;
    uint32_t ND55 : 1;
    uint32_t ND56 : 1;
    uint32_t ND57 : 1;
    uint32_t ND58 : 1;
    uint32_t ND59 : 1;
    uint32_t ND60 : 1;
    uint32_t ND61 : 1;
    uint32_t ND62 : 1;
    uint32_t ND63 : 1;
};
union MCAN_NDAT2_REG {
    uint32_t  all;
    struct  MCAN_NDAT2_BITS  bit;
};
/**
 * @brief MCAN RX FIFO 0 Control Register bitfield definitions.
 *
 * @param rsvd1  Reserved bits (bits 1:0).
 * @param F0SA   Rx FIFO 0 Start Address (bits 15:2).
 * @param F0S    Rx FIFO 0 Size (bits 22:16).
 * @param rsvd2  Reserved bit (bit 23).
 * @param F0WM   Rx FIFO 0 Watermark (bits 30:24).
 * @param F0OM   FIFO 0 Operation Mode (bit 31).
 */
 struct MCAN_RXF0C_BITS {
    uint16_t rsvd1:2;
    uint16_t F0SA:14;
    uint32_t F0S:7;
    uint32_t rsvd2:1;
    uint32_t F0WM:7;
    uint32_t F0OM:1;
};
union MCAN_RXF0C_REG {
    uint32_t all;
    struct MCAN_RXF0C_BITS bit;
};

/**
 * @brief MCAN RX FIFO 0 Status Register bitfield definitions.
 *
 * @param F0FL   Rx FIFO 0 Fill Level (bits 6:0).
 * @param rsvd1  Reserved bit (bit 7).
 * @param F0GI   Rx FIFO 0 Get Index (bits 13:8).
 * @param rsvd2  Reserved bits (bits 15:14).
 * @param F0PI   Rx FIFO 0 Put Index (bits 21:16).
 * @param rsvd3  Reserved bits (bits 23:22).
 * @param F0F    Rx FIFO 0 Full (bit 24).
 * @param RF0L   Rx FIFO 0 Message Lost (bit 25).
 * @param rsvd4  Reserved bits (bits 31:26).
 */
struct MCAN_RXF0S_BITS {
    uint16_t F0FL:7;
    uint16_t rsvd1:1;
    uint16_t F0GI:6;
    uint16_t rsvd2:2;
    uint32_t F0PI:6;
    uint32_t rsvd3:2;
    uint32_t F0F:1;
    uint32_t RF0L:1;
    uint32_t rsvd4:6;
};
union MCAN_RXF0S_REG {
    uint32_t all;
    struct MCAN_RXF0S_BITS bit;
};

/**
 * @brief MCAN RX FIFO 0 Acknowledge Register bitfield definitions.
 *
 * @param F0AI   Rx FIFO 0 Acknowledge Index (bits 5:0).
 * @param rsvd1  Reserved bits (bits 15:6).
 * @param rsvd2  Reserved bits (bits 31:16).
 */
struct MCAN_RXF0A_BITS {
    uint16_t F0AI:6;
    uint16_t rsvd1:10;
    uint32_t rsvd2:16;
};
union MCAN_RXF0A_REG {
    uint32_t all;
    struct MCAN_RXF0A_BITS bit;
};

/**
 * @brief MCAN RX Buffer Configuration Register bitfield definitions.
 *
 * @param rsvd1  Reserved bits (bits 1:0).
 * @param RBSA   Rx Buffer Start Address (bits 15:2).
 * @param rsvd2  Reserved bits (bits 31:16).
 */
struct MCAN_RXBC_BITS {
    uint16_t rsvd1:2;
    uint16_t RBSA:14;
    uint32_t rsvd2:16;
};
union MCAN_RXBC_REG {
    uint32_t all;
    struct MCAN_RXBC_BITS bit;
};

/**
 * @brief MCAN RX FIFO 1 Control Register bitfield definitions.
 *
 * @param rsvd1  Reserved bits (bits 1:0).
 * @param F1SA   Rx FIFO 1 Start Address (bits 15:2).
 * @param F1S    Rx FIFO 1 Size (bits 22:16).
 * @param rsvd2  Reserved bit (bit 23).
 * @param F1WM   Rx FIFO 1 Watermark (bits 30:24).
 * @param F1OM   FIFO 1 Operation Mode (bit 31).
 */
struct MCAN_RXF1C_BITS {
    uint16_t rsvd1:2;
    uint16_t F1SA:14;
    uint32_t F1S:7;
    uint32_t rsvd2:1;
    uint32_t F1WM:7;
    uint32_t F1OM:1;
};
union MCAN_RXF1C_REG {
    uint32_t all;
    struct MCAN_RXF1C_BITS bit;
};

/**
 * @brief MCAN RX FIFO 1 Status Register bitfield definitions.
 *
 * @param F1FL   Rx FIFO 1 Fill Level (bits 6:0).
 * @param rsvd1  Reserved bit (bit 7).
 * @param F1GI   Rx FIFO 1 Get Index (bits 13:8).
 * @param rsvd2  Reserved bits (bits 15:14).
 * @param F1PI   Rx FIFO 1 Put Index (bits 21:16).
 * @param rsvd3  Reserved bits (bits 23:22).
 * @param F1F    Rx FIFO 1 Full (bit 24).
 * @param RF1L   Rx FIFO 1 Message Lost (bit 25).
 * @param rsvd4  Reserved bits (bits 31:26).
 * @param DMS    Debug Message Status (bits 31:30).
 */
struct MCAN_RXF1S_BITS {
    uint16_t F1FL:7;
    uint16_t rsvd1:1;
    uint16_t F1GI:6;
    uint16_t rsvd2:2;
    uint32_t F1PI:6;
    uint32_t rsvd3:2;
    uint32_t F1F:1;
    uint32_t RF1L:1;
    uint32_t rsvd4:4;
    uint32_t DMS:2;
};
union MCAN_RXF1S_REG {
    uint32_t all;
    struct MCAN_RXF1S_BITS bit;
};

/**
 * @brief MCAN RX FIFO 1 Acknowledge Register bitfield definitions.
 *
 * @param F1AI   Rx FIFO 1 Acknowledge Index (bits 5:0).
 * @param rsvd1  Reserved bits (bits 15:6).
 * @param rsvd2  Reserved bits (bits 31:16).
 */
struct MCAN_RXF1A_BITS {
    uint16_t F1AI:6;
    uint16_t rsvd1:10;
    uint32_t rsvd2:16;
};
union MCAN_RXF1A_REG {
    uint32_t all;
    struct MCAN_RXF1A_BITS bit;
};

/**
 * @brief MCAN RX FIFO and Buffer Control Register bitfield definitions.
 *
 * @param F0DS   Rx FIFO 0 Data Field Size (bits 2:0).
 * @param rsvd1  Reserved bit (bit 3).
 * @param F1DS   Rx FIFO 1 Data Field Size (bits 6:4).
 * @param rsvd2  Reserved bit (bit 7).
 * @param RBDS   Rx Buffer Data Field Size (bits 10:8).
 * @param rsvd3  Reserved bits (bits 15:11).
 * @param rsvd4  Reserved bits (bits 31:16).
 */
struct MCAN_RXESC_BITS {
    uint16_t F0DS:3;
    uint16_t rsvd1:1;
    uint16_t F1DS:3;
    uint16_t rsvd2:1;
    uint16_t RBDS:3;
    uint16_t rsvd3:5;
    uint32_t rsvd4:16;
};
union MCAN_RXESC_REG {
    uint32_t all;
    struct MCAN_RXESC_BITS bit;
};

/**
 * @brief MCAN Transmit Buffer Configuration Register bitfield definitions.
 *
 * @param rsvd1  Reserved bits (bits 1:0).
 * @param TBSA   Tx Buffers Start Address (bits 15:2).
 * @param NDTB   Number of Dedicated Transmit Buffers (bits 21:16).
 * @param rsvd2  Reserved bits (bits 23:22).
 * @param TFQS   Transmit FIFO/Queue Size (bits 29:24).
 * @param TFQM   Tx FIFO/Queue Mode (bit 30).
 * @param rsvd3  Reserved bit (bit 31).
 */
struct MCAN_TXBC_BITS {
    uint16_t rsvd1:2;
    uint16_t TBSA:14;
    uint32_t NDTB:6;
    uint32_t rsvd2:2;
    uint32_t TFQS:6;
    uint32_t TFQM:1;
    uint32_t rsvd3:1;
};
union MCAN_TXBC_REG {
    uint32_t all;
    struct MCAN_TXBC_BITS bit;
};

/**
 * @brief MCAN Transmit FIFO/Queue Status Register bitfield definitions.
 *
 * @param TFFL   Tx FIFO Free Level (bits 5:0).
 * @param rsvd1  Reserved bits (bits 7:6).
 * @param TFGI   Tx FIFO Get Index (bits 12:8).
 * @param rsvd2  Reserved bits (bits 15:13).
 * @param TFQP   Tx FIFO/Queue Put Index (bits 20:16).
 * @param TFQF   Tx FIFO/Queue Full (bit 21).
 * @param rsvd3  Reserved bits (bits 31:22).
 */
struct MCAN_TXFQS_BITS {
    uint16_t TFFL:6;
    uint16_t rsvd1:2;
    uint16_t TFGI:5;
    uint16_t rsvd2:3;
    uint32_t TFQP:5;
    uint32_t TFQF:1;
    uint32_t rsvd3:10;
};
union MCAN_TXFQS_REG {
    uint32_t all;
    struct MCAN_TXFQS_BITS bit;
};

/**
 * @brief MCAN Transmit Buffer Data Field Size Register bitfield definitions.
 *
 * @param TBDS   Tx Buffer Data Field Size (bits 2:0).
 * @param rsvd1  Reserved bits (bits 15:3).
 * @param rsvd2  Reserved bits (bits 31:16).
 */
struct MCAN_TXESC_BITS {
    uint16_t TBDS:3;
    uint16_t rsvd1:13;
    uint32_t rsvd2:16;
};
union MCAN_TXESC_REG {
    uint32_t all;
    struct MCAN_TXESC_BITS bit;
};

/**
 * @brief MCAN Transmission Request Pending Register bitfield definitions.
 *
 * @param TRP0   Transmission Request Pending 0 (bit 0).
 * @param TRP1   Transmission Request Pending 1 (bit 1).
 * @param TRP31  Transmission Request Pending 31 (bit 31).
 */
struct MCAN_TXBRP_BITS {
    uint16_t TRP0:1;
    uint16_t TRP1:1;
    uint16_t TRP2:1;
    uint16_t TRP3:1;
    uint16_t TRP4:1;
    uint16_t TRP5:1;
    uint16_t TRP6:1;
    uint16_t TRP7:1;
    uint16_t TRP8:1;
    uint16_t TRP9:1;
    uint16_t TRP10:1;
    uint16_t TRP11:1;
    uint16_t TRP12:1;
    uint16_t TRP13:1;
    uint16_t TRP14:1;
    uint16_t TRP15:1;
    uint32_t TRP16:1;
    uint32_t TRP17:1;
    uint32_t TRP18:1;
    uint32_t TRP19:1;
    uint32_t TRP20:1;
    uint32_t TRP21:1;
    uint32_t TRP22:1;
    uint32_t TRP23:1;
    uint32_t TRP24:1;
    uint32_t TRP25:1;
    uint32_t TRP26:1;
    uint32_t TRP27:1;
    uint32_t TRP28:1;
    uint32_t TRP29:1;
    uint32_t TRP30:1;
    uint32_t TRP31:1;
};
union MCAN_TXBRP_REG {
    uint32_t all;
    struct MCAN_TXBRP_BITS bit;
};

/**
 * @brief MCAN Transmit Add Request Register bitfield definitions.
 *
 * @param AR0    Add Request 0 (bit 0).
 * @param AR1    Add Request 1 (bit 1).
 * @param AR31   Add Request 31 (bit 31).
 */
struct MCAN_TXBAR_BITS {
    uint16_t AR0:1;
    uint16_t AR1:1;
    uint16_t AR2:1;
    uint16_t AR3:1;
    uint16_t AR4:1;
    uint16_t AR5:1;
    uint16_t AR6:1;
    uint16_t AR7:1;
    uint16_t AR8:1;
    uint16_t AR9:1;
    uint16_t AR10:1;
    uint16_t AR11:1;
    uint16_t AR12:1;
    uint16_t AR13:1;
    uint16_t AR14:1;
    uint16_t AR15:1;
    uint32_t AR16:1;
    uint32_t AR17:1;
    uint32_t AR18:1;
    uint32_t AR19:1;
    uint32_t AR20:1;
    uint32_t AR21:1;
    uint32_t AR22:1;
    uint32_t AR23:1;
    uint32_t AR24:1;
    uint32_t AR25:1;
    uint32_t AR26:1;
    uint32_t AR27:1;
    uint32_t AR28:1;
    uint32_t AR29:1;
    uint32_t AR30:1;
    uint32_t AR31:1;
};
union MCAN_TXBAR_REG {
    uint32_t all;
    struct MCAN_TXBAR_BITS bit;
};
/**
 * @brief MCAN Tx Buffer Cancellation Request Register bitfield definitions.
 *
 * @param CR0    Cancellation Request 0 (bit 0).
 * @param CR1    Cancellation Request 1 (bit 1).
 * ...
 * @param CR31   Cancellation Request 31 (bit 31).
 */
 struct MCAN_TXBCR_BITS {
    uint16_t CR0   : 1;
    uint16_t CR1   : 1;
    uint16_t CR2   : 1;
    uint16_t CR3   : 1;
    uint16_t CR4   : 1;
    uint16_t CR5   : 1;
    uint16_t CR6   : 1;
    uint16_t CR7   : 1;
    uint16_t CR8   : 1;
    uint16_t CR9   : 1;
    uint16_t CR10  : 1;
    uint16_t CR11  : 1;
    uint16_t CR12  : 1;
    uint16_t CR13  : 1;
    uint16_t CR14  : 1;
    uint16_t CR15  : 1;
    uint32_t CR16  : 1;
    uint32_t CR17  : 1;
    uint32_t CR18  : 1;
    uint32_t CR19  : 1;
    uint32_t CR20  : 1;
    uint32_t CR21  : 1;
    uint32_t CR22  : 1;
    uint32_t CR23  : 1;
    uint32_t CR24  : 1;
    uint32_t CR25  : 1;
    uint32_t CR26  : 1;
    uint32_t CR27  : 1;
    uint32_t CR28  : 1;
    uint32_t CR29  : 1;
    uint32_t CR30  : 1;
    uint32_t CR31  : 1;
};
union MCAN_TXBCR_REG {
    uint32_t all;
    struct MCAN_TXBCR_BITS bit;
};

/**
 * @brief MCAN Tx Buffer Transmission Occurred Register bitfield definitions.
 *
 * @param TO0     Transmission Occurred 0 (bit 0).
 * @param TO1     Transmission Occurred 1 (bit 1).
 * ...
 * @param TO31    Transmission Occurred 31 (bit 31).
 */
struct MCAN_TXBTO_BITS {
    uint16_t TO0   : 1;
    uint16_t TO1   : 1;
    uint16_t TO2   : 1;
    uint16_t TO3   : 1;
    uint16_t TO4   : 1;
    uint16_t TO5   : 1;
    uint16_t TO6   : 1;
    uint16_t TO7   : 1;
    uint16_t TO8   : 1;
    uint16_t TO9   : 1;
    uint16_t TO10  : 1;
    uint16_t TO11  : 1;
    uint16_t TO12  : 1;
    uint16_t TO13  : 1;
    uint16_t TO14  : 1;
    uint16_t TO15  : 1;
    uint32_t TO16  : 1;
    uint32_t TO17  : 1;
    uint32_t TO18  : 1;
    uint32_t TO19  : 1;
    uint32_t TO20  : 1;
    uint32_t TO21  : 1;
    uint32_t TO22  : 1;
    uint32_t TO23  : 1;
    uint32_t TO24  : 1;
    uint32_t TO25  : 1;
    uint32_t TO26  : 1;
    uint32_t TO27  : 1;
    uint32_t TO28  : 1;
    uint32_t TO29  : 1;
    uint32_t TO30  : 1;
    uint32_t TO31  : 1;
};
union MCAN_TXBTO_REG {
    uint32_t all;
    struct MCAN_TXBTO_BITS bit;
};
/**
 * @brief MCAN Tx Buffer Cancellation Finished Register bitfield definitions.
 * 
 * @param CF0    Cancellation Finished 0 (bit 0)
 * @param CF1    Cancellation Finished 1 (bit 1)
 * ...
 * @param CF31   Cancellation Finished 31 (bit 31)
 */
 struct MCAN_TXBCF_BITS {
    uint16_t CF0   : 1;
    uint16_t CF1   : 1;
    uint16_t CF2   : 1;
    uint16_t CF3   : 1;
    uint16_t CF4   : 1;
    uint16_t CF5   : 1;
    uint16_t CF6   : 1;
    uint16_t CF7   : 1;
    uint16_t CF8   : 1;
    uint16_t CF9   : 1;
    uint16_t CF10  : 1;
    uint16_t CF11  : 1;
    uint16_t CF12  : 1;
    uint16_t CF13  : 1;
    uint16_t CF14  : 1;
    uint16_t CF15  : 1;
    uint32_t CF16  : 1;
    uint32_t CF17  : 1;
    uint32_t CF18  : 1;
    uint32_t CF19  : 1;
    uint32_t CF20  : 1;
    uint32_t CF21  : 1;
    uint32_t CF22  : 1;
    uint32_t CF23  : 1;
    uint32_t CF24  : 1;
    uint32_t CF25  : 1;
    uint32_t CF26  : 1;
    uint32_t CF27  : 1;
    uint32_t CF28  : 1;
    uint32_t CF29  : 1;
    uint32_t CF30  : 1;
    uint32_t CF31  : 1;
};
union MCAN_TXBCF_REG {
    uint32_t all;
    struct MCAN_TXBCF_BITS bit;
};

/**
 * @brief MCAN Tx Buffer Transmission Interrupt Enable Register bitfield definitions.
 * 
 * @param TIE0    Transmission Interrupt Enable 0 (bit 0)
 * @param TIE1    Transmission Interrupt Enable 1 (bit 1)
 * ...
 * @param TIE31   Transmission Interrupt Enable 31 (bit 31)
 */
struct MCAN_TXBTIE_BITS {
    uint16_t TIE0  : 1;
    uint16_t TIE1  : 1;
    uint16_t TIE2  : 1;
    uint16_t TIE3  : 1;
    uint16_t TIE4  : 1;
    uint16_t TIE5  : 1;
    uint16_t TIE6  : 1;
    uint16_t TIE7  : 1;
    uint16_t TIE8  : 1;
    uint16_t TIE9  : 1;
    uint16_t TIE10 : 1;
    uint16_t TIE11 : 1;
    uint16_t TIE12 : 1;
    uint16_t TIE13 : 1;
    uint16_t TIE14 : 1;
    uint16_t TIE15 : 1;
    uint32_t TIE16 : 1;
    uint32_t TIE17 : 1;
    uint32_t TIE18 : 1;
    uint32_t TIE19 : 1;
    uint32_t TIE20 : 1;
    uint32_t TIE21 : 1;
    uint32_t TIE22 : 1;
    uint32_t TIE23 : 1;
    uint32_t TIE24 : 1;
    uint32_t TIE25 : 1;
    uint32_t TIE26 : 1;
    uint32_t TIE27 : 1;
    uint32_t TIE28 : 1;
    uint32_t TIE29 : 1;
    uint32_t TIE30 : 1;
    uint32_t TIE31 : 1;
};
union MCAN_TXBTIE_REG {
    uint32_t all;
    struct MCAN_TXBTIE_BITS bit;
};

/**
 * @brief MCAN Tx Buffer Cancellation Finished Interrupt Enable Register bitfield definitions.
 * 
 * @param CFIE0   Cancellation Finished Interrupt Enable 0 (bit 0)
 * @param CFIE1   Cancellation Finished Interrupt Enable 1 (bit 1)
 * ...
 * @param CFIE31  Cancellation Finished Interrupt Enable 31 (bit 31)
 */
struct MCAN_TXBCIE_BITS {
    uint16_t CFIE0  : 1;
    uint16_t CFIE1  : 1;
    uint16_t CFIE2  : 1;
    uint16_t CFIE3  : 1;
    uint16_t CFIE4  : 1;
    uint16_t CFIE5  : 1;
    uint16_t CFIE6  : 1;
    uint16_t CFIE7  : 1;
    uint16_t CFIE8  : 1;
    uint16_t CFIE9  : 1;
    uint16_t CFIE10 : 1;
    uint16_t CFIE11 : 1;
    uint16_t CFIE12 : 1;
    uint16_t CFIE13 : 1;
    uint16_t CFIE14 : 1;
    uint16_t CFIE15 : 1;
    uint32_t CFIE16 : 1;
    uint32_t CFIE17 : 1;
    uint32_t CFIE18 : 1;
    uint32_t CFIE19 : 1;
    uint32_t CFIE20 : 1;
    uint32_t CFIE21 : 1;
    uint32_t CFIE22 : 1;
    uint32_t CFIE23 : 1;
    uint32_t CFIE24 : 1;
    uint32_t CFIE25 : 1;
    uint32_t CFIE26 : 1;
    uint32_t CFIE27 : 1;
    uint32_t CFIE28 : 1;
    uint32_t CFIE29 : 1;
    uint32_t CFIE30 : 1;
    uint32_t CFIE31 : 1;
};
union MCAN_TXBCIE_REG {
    uint32_t all;
    struct MCAN_TXBCIE_BITS bit;
};
/**
 * @brief MCAN Tx Event FIFO Configuration Register bitfield definitions
 * 
 * @param rsvd1     Reserved bits (bits 1:0)
 * @param EFSA      Event FIFO Start Address (bits 15:2)
 * @param EFS       Event FIFO Size (bits 21:16)
 * @param rsvd2     Reserved bits (bits 23:22)
 * @param EFWM      Event FIFO Watermark (bits 29:24)
 * @param rsvd3     Reserved bits (bits 31:30)
 *
 */
 struct MCAN_TXEFC_BITS {
    uint16_t rsvd1  : 2;
    uint16_t EFSA    : 14;
    uint32_t EFS     : 6;
    uint32_t rsvd2   : 2;
    uint32_t EFWM    : 6;
    uint32_t rsvd3   : 2;
};
union MCAN_TXEFC_REG {
    uint32_t all;
    struct MCAN_TXEFC_BITS bit;
};

/**
 * @brief MCAN Tx Event FIFO Status Register bitfield definitions
 * 
 * @param EFFL      Event FIFO Fill Level (bits 5:0)
 * @param rsvd1     Reserved bits (bits 7:6)
 * @param EFGI      Event FIFO Get Index (bits 12:8)
 * @param rsvd2     Reserved bits (bits 15:13)
 * @param EFPI      Event FIFO Put Index (bits 20:16)
 * @param rsvd3     Reserved bits (bits 23:21)
 * @param EFF       Event FIFO Full (bit 24)
 * @param TEFL      Tx Event FIFO Element Lost (bit 25)
 * @param rsvd4     Reserved bits (bits 31:26)
 * 
 */
struct MCAN_TXEFS_BITS {
    uint16_t EFFL   : 6;
    uint16_t rsvd1  : 2;
    uint16_t EFGI   : 5;
    uint16_t rsvd2  : 3;
    uint32_t EFPI   : 5;
    uint32_t rsvd3  : 3;
    uint32_t EFF    : 1;
    uint32_t TEFL   : 1;
    uint32_t rsvd4  : 6;
};
union MCAN_TXEFS_REG {
    uint32_t all;
    struct MCAN_TXEFS_BITS bit;
};

/**
 * @brief MCAN Tx Event FIFO Acknowledge Register bitfield definitions
 * 
 * @param EFAI      Event FIFO Acknowledge Index (bits 4:0)
 * @param rsvd1     Reserved bits (bits 15:5)
 * @param rsvd2     Reserved bits (bits 31:16)
 * 
 */
struct MCAN_TXEFA_BITS {
    uint16_t EFAI   : 5;
    uint16_t rsvd1  : 11;
    uint32_t rsvd2  : 16;
};
union MCAN_TXEFA_REG {
    uint32_t all;
    struct MCAN_TXEFA_BITS bit;
};
/**
 * @brief MCAN Core Register Map.
 *
 * @param MCAN_CREL                   MCAN Core Release Register.
 * @param MCAN_ENDN                   MCAN Endian Register.
 * @param MCAN_DBTP                   MCAN Data Bit Timing and Prescaler Register.
 * @param MCAN_TEST                   MCAN Test Register.
 * @param MCAN_RWD                    MCAN RAM Watchdog Register.
 * @param MCAN_CCCR                   MCAN CC Control Register.
 * @param MCAN_NBTP                   MCAN Nominal Bit Timing and Prescaler Register.
 * @param MCAN_TSCC                   MCAN Timestamp Counter Configuration Register.
 * @param MCAN_TSCV                   MCAN Timestamp Counter Value Register.
 * @param MCAN_TOCC                   MCAN Timeout Counter Configuration Register.
 * @param MCAN_TOCV                   MCAN Timeout Counter Value Register.
 * @param MCAN_ECR                    MCAN Error Counter Register.
 * @param MCAN_PSR                    MCAN Protocol Status Register.
 * @param MCAN_TDCR                   MCAN Transmitter Delay Compensation Register.
 * @param MCAN_IR                     MCAN Interrupt Register.
 * @param MCAN_IE                     MCAN Interrupt Enable Register.
 * @param MCAN_ILS                    MCAN Interrupt Line Select Register.
 * @param MCAN_ILE                    MCAN Interrupt Line Enable Register.
 * @param MCAN_GFC                    MCAN Global Filter Configuration Register.
 * @param MCAN_SIDFC                  MCAN Standard ID Filter Configuration Register.
 * @param MCAN_XIDFC                  MCAN Extended ID Filter Configuration Register.
 * @param MCAN_XIDAM                  MCAN Extended ID and Mask Register.
 * @param MCAN_HPMS                   MCAN High Priority Message Status Register.
 * @param MCAN_NDAT1                  MCAN New Data 1 Register.
 * @param MCAN_NDAT2                  MCAN New Data 2 Register.
 * @param MCAN_RXF0C                  MCAN Rx FIFO 0 Configuration Register.
 * @param MCAN_RXF0S                  MCAN Rx FIFO 0 Status Register.
 * @param MCAN_RXF0A                  MCAN Rx FIFO 0 Acknowledge Register.
 * @param MCAN_RXBC                   MCAN Rx Buffer Configuration Register.
 * @param MCAN_RXF1C                  MCAN Rx FIFO 1 Configuration Register.
 * @param MCAN_RXF1S                  MCAN Rx FIFO 1 Status Register.
 * @param MCAN_RXF1A                  MCAN Rx FIFO 1 Acknowledge Register.
 * @param MCAN_RXESC                  MCAN Rx Buffer / FIFO Element Size Configuration Register.
 * @param MCAN_TXBC                   MCAN Tx Buffer Configuration Register.
 * @param MCAN_TXFQS                  MCAN Tx FIFO / Queue Status Register.
 * @param MCAN_TXESC                  MCAN Tx Buffer Element Size Configuration Register.
 * @param MCAN_TXBRP                  MCAN Tx Buffer Request Pending Register.
 * @param MCAN_TXBAR                  MCAN Tx Buffer Add Request Register.
 * @param MCAN_TXBCR                  MCAN Tx Buffer Cancellation Request Register.
 * @param MCAN_TXBTO                  MCAN Tx Buffer Transmission Occurred Register.
 * @param MCAN_TXBCF                  MCAN Tx Buffer Cancellation Finished Register.
 * @param MCAN_TXBTIE                 MCAN Tx Buffer Transmission Interrupt Enable Register.
 * @param MCAN_TXBCIE                 MCAN Tx Buffer Cancellation Finished Interrupt Enable Register.
 * @param MCAN_TXEFC                  MCAN Tx Event FIFO Configuration Register.
 * @param MCAN_TXEFS                  MCAN Tx Event FIFO Status Register.
 * @param MCAN_TXEFA                  MCAN Tx Event FIFO Acknowledge Register.
 */
 struct MCAN_REGS {
    union   MCAN_CREL_REG                    CREL;
    uint32_t                                 ENDN;
    uint32_t                                 rsvd1[1];
    union   MCAN_DBTP_REG                    DBTP;
    union   MCAN_TEST_REG                    TEST;
    union   MCAN_RWD_REG                     RWD;
    union   MCAN_CCCR_REG                    CCCR;
    union   MCAN_NBTP_REG                    NBTP;
    union   MCAN_TSCC_REG                    TSCC;
    union   MCAN_TSCV_REG                    TSCV;
    union   MCAN_TOCC_REG                    TOCC;
    union   MCAN_TOCV_REG                    TOCV;
    uint32_t                                 rsvd2[4];
    union   MCAN_ECR_REG                     ECR;
    union   MCAN_PSR_REG                     PSR;
    union   MCAN_TDCR_REG                    TDCR;
    uint32_t                                 rsvd3[1];
    union   MCAN_IR_REG                      IR;
    union   MCAN_IE_REG                      IE;
    union   MCAN_ILS_REG                     ILS;
    union   MCAN_ILE_REG                     ILE;
    uint32_t                                 rsvd4[8];
    union   MCAN_GFC_REG                     GFC;
    union   MCAN_SIDFC_REG                   SIDFC;
    union   MCAN_XIDFC_REG                   XIDFC;
    uint32_t                                 rsvd5[1];
    union   MCAN_XIDAM_REG                   XIDAM;
    union   MCAN_HPMS_REG                    HPMS;
    union   MCAN_NDAT1_REG                   NDAT1;
    union   MCAN_NDAT2_REG                   NDAT2;
    union   MCAN_RXF0C_REG                   RXF0C;
    union   MCAN_RXF0S_REG                   RXF0S;
    union   MCAN_RXF0A_REG                   RXF0A;
    union   MCAN_RXBC_REG                    RXBC;
    union   MCAN_RXF1C_REG                   RXF1C;
    union   MCAN_RXF1S_REG                   RXF1S;
    union   MCAN_RXF1A_REG                   RXF1A;
    union   MCAN_RXESC_REG                   RXESC;
    union   MCAN_TXBC_REG                    TXBC;
    union   MCAN_TXFQS_REG                   TXFQS;
    union   MCAN_TXESC_REG                   TXESC;
    union   MCAN_TXBRP_REG                   TXBRP;
    union   MCAN_TXBAR_REG                   TXBAR;
    union   MCAN_TXBCR_REG                   TXBCR;
    union   MCAN_TXBTO_REG                   TXBTO;
    union   MCAN_TXBCF_REG                   TXBCF;
    union   MCAN_TXBTIE_REG                  TXBTIE;
    union   MCAN_TXBCIE_REG                  TXBCIE;
    uint32_t                                 rsvd6[2];
    union   MCAN_TXEFC_REG                   TXEFC;
    union   MCAN_TXEFS_REG                   TXEFS;
    union   MCAN_TXEFA_REG                   TXEFA;
};
/**
 * @brief MCAN Error Aggregator Revision Register bitfield definitions
 * 
 * @param REVMIN    Minor Revision (bits 5:0)
 * @param rsvd1     Reserved bits (bits 7:6)
 * @param REVMAJ    Major Revision (bits 10:8)
 * @param rsvd2     Reserved bits (bits 15:11)
 * @param MODULE_ID Module Identification Number (bits 27:16)
 * @param rsvd3     Reserved bits (bits 29:28)
 * @param SCHEME    PID Register Scheme (bits 31:30)
 * 
 */
 struct MCANERR_REV_BITS {
    uint16_t REVMIN:6;                    
    uint16_t rsvd1:2;                     
    uint16_t REVMAJ:3;                     
    uint16_t rsvd2:5;                      
    uint32_t MODULE_ID:12;                 
    uint32_t rsvd3:2;                      
    uint32_t SCHEME:2;                     
};

/**
 * @brief MCAN Error Aggregator Revision Register union
 * 
 * @param all     32-bit register value
 * @param bit     Bitfield structure representing the register
 * 
 */
union MCANERR_REV_REG {
    uint32_t  all;
    struct  MCANERR_REV_BITS  bit;
};

/**
 * @brief MCAN ECC Vector Register bitfield definitions
 * 
 * @param ECC_VECTOR       ECC RAM ID (bits 10:0)
 * @param rsvd1            Reserved bits (bits 14:11)
 * @param RD_SVBUS         Read Trigger (bit 15)
 * @param RD_SVBUS_ADDRESS Read Address Offset (bits 23:16)
 * @param RD_SVBUS_DONE    Read Completion Flag (bit 24)
 * @param rsvd2            Reserved bits (bits 31:25)
 * 
 */
struct MCANERR_VECTOR_BITS {
    uint16_t ECC_VECTOR:11;                
    uint16_t rsvd1:4;                      
    uint16_t RD_SVBUS:1;                   
    uint32_t RD_SVBUS_ADDRESS:8;           
    uint32_t RD_SVBUS_DONE:1;              
    uint32_t rsvd2:7;                      
};

/**
 * @brief MCAN ECC Vector Register union
 * 
 * @param all     32-bit register value
 * @param bit     Bitfield structure representing the register
 * 
 */
union MCANERR_VECTOR_REG {
    uint32_t  all;
    struct  MCANERR_VECTOR_BITS  bit;
};

/**
 * @brief MCAN Error Misc Status Register bitfield definitions
 * 
 * @param NUM_RAMS    Number of RAMs (bits 10:0)
 * @param rsvd1       Reserved bits (bits 15:11)
 * @param rsvd2       Reserved bits (bits 31:16)
 * 
 */
struct MCANERR_STAT_BITS {
    uint16_t NUM_RAMS:11;                  
    uint16_t rsvd1:5;                      
    uint32_t rsvd2:16;                     
};

/**
 * @brief MCAN Error Misc Status Register union
 * 
 * @param all     32-bit register value
 * @param bit     Bitfield structure representing the register
 * 
 */
union MCANERR_STAT_REG {
    uint32_t  all;
    struct  MCANERR_STAT_BITS  bit;
};
/**
 * @brief MCAN Error Wrapper Revision Register bitfield definitions
 *
 * @param REVMIN      Minor revision number (bits 5:0)
 * @param rsvd1       Reserved bits (bits 7:6)
 * @param REVMAJ      Major revision number (bits 10:8)
 * @param rsvd2       Reserved bits (bits 15:11)
 * @param MODULE_ID   Module Identification Number (bits 27:16)
 * @param rsvd3       Reserved bits (bits 29:28)
 * @param SCHEME      PID Register Scheme (bits 31:30)
 *
 */
struct MCANERR_WRAP_REV_BITS {
    uint16_t REVMIN:6;
    uint16_t rsvd1:2;
    uint16_t REVMAJ:3;
    uint16_t rsvd2:5;
    uint32_t MODULE_ID:12;
    uint32_t rsvd3:2;
    uint32_t SCHEME:2;
};

/**
 * @brief MCAN Error Wrapper Revision Register union
 *
 * @param all     32-bit register value
 * @param bit     Bitfield structure representing the register
 *
 */
union MCANERR_WRAP_REV_REG {
    uint32_t  all;
    struct  MCANERR_WRAP_REV_BITS  bit;
};
/**
 * @brief MCAN Error Control Register 1 bitfield definitions
 * 
 * @param ECC_ENABLE    Enable ECC Generation (bit 0)
 * @param ECC_CHECK     Enable ECC Check (bit 1)
 * @param ENABLE_RMW    Enable Read-Modify-Write (bit 2)
 * @param FORCE_SEC     Force Single Bit Error Corrected Error (bit 3)
 * @param FORCE_DED     Force Double Bit Error Detected Error (bit 4)
 * @param FORCE_N_ROW   Force Next Single/Double Bit Error (bit 5)
 * @param ERROR_ONCE    Force Error Only Once Enable (bit 6)
 * @param rsvd1         Reserved bits (bit 7)
 * @param CHECK_SVBUS_TIMEOUT Enable SVBUS Timeout (bit 8)
 * @param rsvd2         Reserved bits (bits 15:9)
 * @param rsvd3         Reserved bits (bits 31:16)
 * 
 */
struct MCANERR_CTRL_BITS {
    uint16_t ECC_ENABLE:1;                 
    uint16_t ECC_CHECK:1;                  
    uint16_t ENABLE_RMW:1;                 
    uint16_t FORCE_SEC:1;                  
    uint16_t FORCE_DED:1;             
    uint16_t FORCE_N_ROW:1;         
    uint16_t ERROR_ONCE:1;              
    uint16_t rsvd1:1;                    
    uint16_t CHECK_SVBUS_TIMEOUT:1;       
    uint16_t rsvd2:7;                      
    uint32_t rsvd3:16;                     
};

/**
 * @brief MCAN Error Control Register 1 union
 * 
 * @param all     32-bit register value
 * @param bit     Bitfield structure representing the register
 * 
 */
union MCANERR_CTRL_REG {
    uint32_t  all;
    struct  MCANERR_CTRL_BITS  bit;
};

/**
 * @brief MCAN Error Control 2 Register bitfield definitions
 * 
 * @param ECC_BIT1    Force Error Bit1 Column Index (bits 15:0)
 * @param ECC_BIT2    Force Error Bit2 Column Index (bits 31:16)
 * 
 */
struct MCANERR_ERR_CTRL2_BITS {
    uint16_t ECC_BIT1:16;                  
    uint32_t ECC_BIT2:16;                  
};

/**
 * @brief MCAN Error Control 2 Register union
 * 
 * @param all     32-bit register value
 * @param bit     Bitfield structure representing the register
 * 
 */
union MCANERR_ERR_CTRL2_REG {
    uint32_t  all;
    struct  MCANERR_ERR_CTRL2_BITS  bit;
};

/**
 * @brief MCAN ECC Error Status 1 Register bitfield definitions
 * 
 * @param ECC_SEC        Single Bit Error Corrected Status (bits 1:0)
 * @param ECC_DED        Double Bit Error Detected Status (bits 3:2)
 * @param ECC_OTHER      SEC While Writeback Error Status (bit 4)
 * @param rsvd1          Reserved bits (bits 6:5)
 * @param CTRL_REG_ERROR Control Register Error (bit 7)
 * @param CLR_ECC_SEC    Clear ECC_SEC (bits 9:8)
 * @param CLR_ECC_DED    Clear ECC_DED (bits 11:10)
 * @param CLR_ECC_OTHER  Clear ECC_OTHER (bit 12)
 * @param rsvd2          Reserved bits (bits 14:13)
 * @param CLR_CTRL_REG_ERROR Clear Control Register Error (bit 15)
 * @param ECC_BIT1       ECC Error Bit Position (bits 31:16)
 * 
 */
struct MCANERR_ERR_STAT1_BITS {
    uint16_t ECC_SEC:2;                    
    uint16_t ECC_DED:2;                    
    uint16_t ECC_OTHER:1;                 
    uint16_t rsvd1:2;                      
    uint16_t CTRL_REG_ERROR:1;             
    uint16_t CLR_ECC_SEC:2;                
    uint16_t CLR_ECC_DED:2;                
    uint16_t CLR_ECC_OTHER:1;              
    uint16_t rsvd2:2;                      
    uint16_t CLR_CTRL_REG_ERROR:1;         
    uint32_t ECC_BIT1:16;                  
};
union MCANERR_ERR_STAT1_REG {
    uint32_t  all;
    struct  MCANERR_ERR_STAT1_BITS  bit;
};
/**
 * @brief MCAN Error Status 3 Register bitfield definitions
 * 
 * @param WB_PEND            Delayed Write Back Pending Status (bit 0)
 * @param SVBUS_TIMEOUT      Serial VBUS Timeout Flag (bit 1)
 * @param rsvd1              Reserved bits (bits 8:2)
 * @param CLR_SVBUS_TIMEOUT  Clear Serial VBUS Timeout (bit 9)
 * @param rsvd2              Reserved bits (bits 15:10)
 * @param rsvd3              Reserved bits (bits 31:16)
 */
 struct MCANERR_ERR_STAT3_BITS {
    uint16_t WB_PEND:1;
    uint16_t SVBUS_TIMEOUT:1;
    uint16_t rsvd1:7;
    uint16_t CLR_SVBUS_TIMEOUT:1;
    uint16_t rsvd2:6;
    uint32_t rsvd3:16;
};

/**
 * @brief MCAN Error Status 3 Register union
 * 
 * @param all     32-bit register value
 * @param bit     Bitfield structure representing the register
 */
union MCANERR_ERR_STAT3_REG {
    uint32_t  all;
    struct  MCANERR_ERR_STAT3_BITS  bit;
};

/**
 * @brief MCAN SEC End of Interrupt Register bitfield definitions
 * 
 * @param EOI_WR   End of Interrupt (bit 0)
 * @param rsvd1    Reserved bits (bits 15:1)
 * @param rsvd2    Reserved bits (bits 31:16)
 */
struct MCANERR_SEC_EOI_BITS {
    uint16_t EOI_WR:1;
    uint16_t rsvd1:15;
    uint32_t rsvd2:16;
};

/**
 * @brief MCAN SEC End of Interrupt Register union
 * 
 * @param all     32-bit register value
 * @param bit     Bitfield structure representing the register
 */
union MCANERR_SEC_EOI_REG {
    uint32_t  all;
    struct  MCANERR_SEC_EOI_BITS  bit;
};

/**
 * @brief MCAN SEC Status Register bitfield definitions
 * 
 * @param MSGMEM_PEND   Message RAM SEC Interrupt Pending (bit 0)
 * @param rsvd1         Reserved bit (bit 1)
 * @param rsvd2         Reserved bits (bits 15:2)
 * @param rsvd3         Reserved bits (bits 31:16)
 */
struct MCANERR_SEC_STATUS_BITS {
    uint16_t MSGMEM_PEND:1;
    uint16_t rsvd1:1;
    uint16_t rsvd2:14;
    uint32_t rsvd3:16;
};

/**
 * @brief MCAN SEC Status Register union
 * 
 * @param all     32-bit register value
 * @param bit     Bitfield structure representing the register
 */
union MCANERR_SEC_STATUS_REG {
    uint32_t  all;
    struct  MCANERR_SEC_STATUS_BITS  bit;
};

/**
 * @brief MCAN SEC Enable Set Register bitfield definitions
 * 
 * @param MSGMEM_ENABLE_SET   Message RAM SEC Interrupt Pending Enable Set (bit 0)
 * @param rsvd1               Reserved bit (bit 1)
 * @param rsvd2               Reserved bits (bits 15:2)
 * @param rsvd3               Reserved bits (bits 31:16)
 */
struct MCANERR_SEC_ENABLE_SET_BITS {
    uint16_t MSGMEM_ENABLE_SET:1;
    uint16_t rsvd1:1;
    uint16_t rsvd2:14;
    uint32_t rsvd3:16;
};

/**
 * @brief MCAN SEC Enable Set Register union
 * 
 * @param all     32-bit register value
 * @param bit     Bitfield structure representing the register
 */
union MCANERR_SEC_ENABLE_SET_REG {
    uint32_t  all;
    struct  MCANERR_SEC_ENABLE_SET_BITS  bit;
};

/**
 * @brief MCAN SEC Enable Clear Register bitfield definitions
 * 
 * @param MSGMEM_ENABLE_CLR   Message RAM SEC Interrupt Pending Enable Clear (bit 0)
 * @param rsvd1               Reserved bit (bit 1)
 * @param rsvd2               Reserved bits (bits 15:2)
 * @param rsvd3               Reserved bits (bits 31:16)
 */
struct MCANERR_SEC_ENABLE_CLR_BITS {
    uint16_t MSGMEM_ENABLE_CLR:1;
    uint16_t rsvd1:1;
    uint16_t rsvd2:14;
    uint32_t rsvd3:16;
};

/**
 * @brief MCAN SEC Enable Clear Register union
 * 
 * @param all     32-bit register value
 * @param bit     Bitfield structure representing the register
 */
union MCANERR_SEC_ENABLE_CLR_REG {
    uint32_t  all;
    struct  MCANERR_SEC_ENABLE_CLR_BITS  bit;
};

/**
 * @brief MCAN DED End of Interrupt Register bitfield definitions
 * 
 * @param EOI_WR   End of Interrupt (bit 0)
 * @param rsvd1    Reserved bits (bits 15:1)
 * @param rsvd2    Reserved bits (bits 31:16)
 */
struct MCANERR_DED_EOI_BITS {
    uint16_t EOI_WR:1;
    uint16_t rsvd1:15;
    uint32_t rsvd2:16;
};

/**
 * @brief MCAN DED End of Interrupt Register union
 * 
 * @param all     32-bit register value
 * @param bit     Bitfield structure representing the register
 */
union MCANERR_DED_EOI_REG {
    uint32_t  all;
    struct  MCANERR_DED_EOI_BITS  bit;
};
/**
 * @brief Bit definitions for MCAN Double Error Detected Status Register
 * 
 * @param MSGMEM_PEND   Bit 0 - Message RAM DED Interrupt Pending
 * @param rsvd1         Bit 1 - Reserved
 * @param rsvd2         Bits 15:2 - Reserved
 * @param rsvd3         Bits 31:16 - Reserved
 */
 struct MCANERR_DED_STATUS_BITS {
    uint16_t MSGMEM_PEND:1;
    uint16_t rsvd1:1;
    uint16_t rsvd2:14;
    uint32_t rsvd3:16;
};

/**
 * @brief Union for MCAN DED Status Register
 * 
 * @param all   Full 32-bit access
 * @param bit   Bit-field level access
 */
union MCANERR_DED_STATUS_REG {
    uint32_t  all;
    struct  MCANERR_DED_STATUS_BITS  bit;
};

/**
 * @brief Bit definitions for MCAN DED Interrupt Enable Set Register
 * 
 * @param MSGMEM_ENABLE_SET   Bit 0 - Enable DED Interrupt
 * @param rsvd1               Bit 1 - Reserved
 * @param rsvd2               Bits 15:2 - Reserved
 * @param rsvd3               Bits 31:16 - Reserved
 */
struct MCANERR_DED_ENABLE_SET_BITS {
    uint16_t MSGMEM_ENABLE_SET:1;
    uint16_t rsvd1:1;
    uint16_t rsvd2:14;
    uint32_t rsvd3:16;
};

/**
 * @brief Union for MCAN DED Interrupt Enable Set Register
 */
union MCANERR_DED_ENABLE_SET_REG {
    uint32_t  all;
    struct  MCANERR_DED_ENABLE_SET_BITS  bit;
};

/**
 * @brief Bit definitions for MCAN DED Interrupt Enable Clear Register
 * 
 * @param MSGMEM_ENABLE_CLR   Bit 0 - Clear DED Interrupt Enable
 * @param rsvd1               Bit 1 - Reserved
 * @param rsvd2               Bits 15:2 - Reserved
 * @param rsvd3               Bits 31:16 - Reserved
 */
struct MCANERR_DED_ENABLE_CLR_BITS {
    uint16_t MSGMEM_ENABLE_CLR:1;
    uint16_t rsvd1:1;
    uint16_t rsvd2:14;
    uint32_t rsvd3:16;
};

/**
 * @brief Union for MCAN DED Interrupt Enable Clear Register
 */
union MCANERR_DED_ENABLE_CLR_REG {
    uint32_t  all;
    struct  MCANERR_DED_ENABLE_CLR_BITS  bit;
};

/**
 * @brief Bit definitions for MCAN Aggregator Enable Set Register
 * 
 * @param ENABLE_PARITY_SET   Bit 0 - Enable Parity Error Reporting
 * @param ENABLE_TIMEOUT_SET  Bit 1 - Enable Timeout Error Reporting
 * @param rsvd1               Bits 15:2 - Reserved
 * @param rsvd2               Bits 31:16 - Reserved
 */
struct MCANERR_AGGR_ENABLE_SET_BITS {
    uint16_t ENABLE_PARITY_SET:1;
    uint16_t ENABLE_TIMEOUT_SET:1;
    uint16_t rsvd1:14;
    uint32_t rsvd2:16;
};

/**
 * @brief Union for MCAN Aggregator Enable Set Register
 */
union MCANERR_AGGR_ENABLE_SET_REG {
    uint32_t  all;
    struct  MCANERR_AGGR_ENABLE_SET_BITS  bit;
};

/**
 * @brief Bit definitions for MCAN Aggregator Enable Clear Register
 * 
 * @param ENABLE_PARITY_CLR   Bit 0 - Disable Parity Error Reporting
 * @param ENABLE_TIMEOUT_CLR  Bit 1 - Disable Timeout Error Reporting
 * @param rsvd1               Bits 15:2 - Reserved
 * @param rsvd2               Bits 31:16 - Reserved
 */
struct MCANERR_AGGR_ENABLE_CLR_BITS {
    uint16_t ENABLE_PARITY_CLR:1;
    uint16_t ENABLE_TIMEOUT_CLR:1;
    uint16_t rsvd1:14;
    uint32_t rsvd2:16;
};

/**
 * @brief Union for MCAN Aggregator Enable Clear Register
 */
union MCANERR_AGGR_ENABLE_CLR_REG {
    uint32_t  all;
    struct  MCANERR_AGGR_ENABLE_CLR_BITS  bit;
};

/**
 * @brief Bit definitions for MCAN Aggregator Status Set Register
 * 
 * @param AGGR_PARITY_ERR   Bits 1:0 - Aggregator Parity Error Status
 * @param SVBUS_TIMEOUT     Bits 3:2 - Serial VBUS Timeout Error Status
 * @param rsvd1             Bits 15:4 - Reserved
 * @param rsvd2             Bits 31:16 - Reserved
 */
struct MCANERR_AGGR_STATUS_SET_BITS {
    uint16_t AGGR_PARITY_ERR:2;
    uint16_t SVBUS_TIMEOUT:2;
    uint16_t rsvd1:12;
    uint32_t rsvd2:16;
};

/**
 * @brief Union for MCAN Aggregator Status Set Register
 */
union MCANERR_AGGR_STATUS_SET_REG {
    uint32_t  all;
    struct  MCANERR_AGGR_STATUS_SET_BITS  bit;
};
/**
 * @brief Bit definitions for MCAN Aggregator Status Clear Register
 * 
 * @param AGGR_PARITY_ERR   Bits 1:0 - Aggregator Parity Error Clear
 * @param SVBUS_TIMEOUT     Bits 3:2 - Aggregator Serial VBUS Timeout Error Clear
 * @param rsvd1             Bits 15:4 - Reserved
 * @param rsvd2             Bits 31:16 - Reserved
 */
 struct MCANERR_AGGR_STATUS_CLR_BITS {
    uint16_t AGGR_PARITY_ERR:2;
    uint16_t SVBUS_TIMEOUT:2;
    uint16_t rsvd1:12;
    uint32_t rsvd2:16;
};

/**
 * @brief Union for MCAN Aggregator Status Clear Register
 */
union MCANERR_AGGR_STATUS_CLR_REG {
    uint32_t  all;
    struct  MCANERR_AGGR_STATUS_CLR_BITS  bit;
};
/**
 * @brief MCAN ECC Error Aggregator Register Map
 * 
 * This structure defines the full register layout of the MCAN Error Aggregator,
 * including ECC vector, status, control, and interrupt management registers.
 * 
 * Members:
 * - MCANERR_REV                : ECC Aggregator Revision Register
 * - rsvd1[2]                   : Reserved
 * - MCANERR_VECTOR             : ECC Error Vector Register
 * - MCANERR_STAT               : ECC Miscellaneous Status Register
 * - MCANERR_WRAP_REV           : ECC Wrapper Revision Register
 * - MCANERR_CTRL               : ECC Control Register
 * - MCANERR_ERR_CTRL1          : ECC Error Control Register 1
 * - MCANERR_ERR_CTRL2          : ECC Error Control Register 2
 * - MCANERR_ERR_STAT1          : ECC Error Status Register 1
 * - MCANERR_ERR_STAT2          : ECC Error Status Register 2
 * - MCANERR_ERR_STAT3          : ECC Error Status Register 3
 * - rsvd2[8]                   : Reserved
 * - MCANERR_SEC_EOI            : Single-bit Error Corrected End of Interrupt Register
 * - MCANERR_SEC_STATUS         : Single-bit Error Corrected Interrupt Status Register
 * - rsvd3[30]                  : Reserved
 * - MCANERR_SEC_ENABLE_SET     : Single-bit Error Corrected Interrupt Enable Set Register
 * - rsvd4[30]                  : Reserved
 * - MCANERR_SEC_ENABLE_CLR     : Single-bit Error Corrected Interrupt Enable Clear Register
 * - rsvd5[60]                  : Reserved
 * - MCANERR_DED_EOI            : Double-bit Error Detected End of Interrupt Register
 * - MCANERR_DED_STATUS         : Double-bit Error Detected Interrupt Status Register
 * - rsvd6[30]                  : Reserved
 * - MCANERR_DED_ENABLE_SET     : Double-bit Error Detected Interrupt Enable Set Register
 * - rsvd7[30]                  : Reserved
 * - MCANERR_DED_ENABLE_CLR     : Double-bit Error Detected Interrupt Enable Clear Register
 * - rsvd8[30]                  : Reserved
 * - MCANERR_AGGR_ENABLE_SET    : Aggregator Error Enable Set Register
 * - MCANERR_AGGR_ENABLE_CLR    : Aggregator Error Enable Clear Register
 * - MCANERR_AGGR_STATUS_SET    : Aggregator Error Status Set Register
 * - MCANERR_AGGR_STATUS_CLR    : Aggregator Error Status Clear Register
 */
 struct MCAN_ERROR_REGS {
    union   MCANERR_REV_REG                  REV;
    uint32_t                                 rsvd1[1];
    union   MCANERR_VECTOR_REG               VECTOR;
    union   MCANERR_STAT_REG                 STAT;
    union   MCANERR_WRAP_REV_REG             WRAP_REV;
    union   MCANERR_CTRL_REG                 CTRL;
    uint32_t                                 CTRL1;
    union   MCANERR_ERR_CTRL2_REG            CTRL2;
    union   MCANERR_ERR_STAT1_REG            STAT1;
    uint32_t                                 STAT2;
    union   MCANERR_ERR_STAT3_REG            STAT3;
    uint32_t                                 rsvd2[4];
    union   MCANERR_SEC_EOI_REG              SEC_EOI;
    union   MCANERR_SEC_STATUS_REG           SEC_STATUS;
    uint32_t                                 rsvd3[15];
    union   MCANERR_SEC_ENABLE_SET_REG       SEC_ENABLE_SET;
    uint32_t                                 rsvd4[15];
    union   MCANERR_SEC_ENABLE_CLR_REG       SEC_ENABLE_CLR;
    uint32_t                                 rsvd5[30];
    union   MCANERR_DED_EOI_REG              DED_EOI;
    union   MCANERR_DED_STATUS_REG           DED_STATUS;
    uint32_t                                 rsvd6[15];
    union   MCANERR_DED_ENABLE_SET_REG       ENABLE_SET;
    uint32_t                                 rsvd7[15];
    union   MCANERR_DED_ENABLE_CLR_REG       ENABLE_CLR;
    uint32_t                                 rsvd8[15];
    union   MCANERR_AGGR_ENABLE_SET_REG      AGGR_ENABLE_SET;
    union   MCANERR_AGGR_ENABLE_CLR_REG      AGGR_ENABLE_CLR;
    union   MCANERR_AGGR_STATUS_SET_REG      AGGR_STATUS_SET;
    union   MCANERR_AGGR_STATUS_CLR_REG      AGGR_STATUS_CLR;
};
extern volatile struct MCANSS_REGS McanaSsRegs,McanbSsRegs;
extern volatile struct MCAN_REGS McanaRegs,McanbRegs;
extern volatile struct MCAN_ERROR_REGS McanaErrorRegs,McanbErrorRegs;
#ifdef __cplusplus
}
#endif                                  /* extern "C" */

#endif

//===========================================================================
// End of file.
//===========================================================================

