/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 * @file  regs_spi_v30.h
 * @brief v3.0 SPI Individual Register Bit Definitions:
 */

#ifndef DRIVERLIB_REGS_REGS_SPI_V30_H_
#define DRIVERLIB_REGS_REGS_SPI_V30_H_

#ifdef __cplusplus
extern "C"{
#endif

#if (IS_GS32F00xx(0x30))

struct SPICCR_BITS {                      /** @brief bits description.             */
    uint16_t SPICHAR:4;                   /** @brief 3:0 Character Length Control. */
    uint16_t SPILBK:1;                    /** @brief 4 SPI loop back.              */
    uint16_t HS_MODE:1;                   /** @brief 5 High Speed mode control.    */
    uint16_t CLKPOLARITY:1;               /** @brief 6 Shift Clock Polarity.       */
    uint16_t SPISWRESET:1;                /** @brief 7 SPI Software Reset.         */
    uint16_t rsvd1:8;                     /** @brief 15:8 Reserved.                */
};

union SPICCR_REG {
    uint16_t all;
    struct   SPICCR_BITS  bit;
};

struct SPICTL_BITS {                      /** @brief bits description.               */
    uint16_t SPIINTENA:1;                 /** @brief 0 SPI Interupt Enable.          */
    uint16_t TALK:1;                      /** @brief 1 Master/Slave Transmit Enable. */
    uint16_t MASTER_SLAVE:1;              /** @brief 2 SPI Network Mode Control.     */
    uint16_t CLK_PHASE:1;                 /** @brief 3 SPI Clock Phase.              */
    uint16_t OVERRUNINTENA:1;             /** @brief 4 Overrun Interrupt Enable.     */
    uint16_t rsvd1:11;                    /** @brief 15:5 Reserved.                  */
};

union SPICTL_REG {
    uint16_t all;
    struct   SPICTL_BITS  bit;
};

struct SPISTS_BITS {                      /** @brief bits description.                */
    uint16_t rsvd1:5;                     /** @brief 4:0 Reserved.                    */
    uint16_t BUFFULL_FLAG:1;              /** @brief 5 SPI Transmit Buffer Full Flag. */
    uint16_t INT_FLAG:1;                  /** @brief 6 SPI Interrupt Flag.            */
    uint16_t OVERRUN_FLAG:1;              /** @brief 7 SPI Receiver Overrun Flag.     */
    uint16_t rsvd2:8;                     /** @brief 15:8 Reserved.                   */
};

union SPISTS_REG {
    uint16_t all;
    struct   SPISTS_BITS  bit;
};

struct SPIBRR_BITS {                      /** @brief bits description.         */
    uint16_t SPI_BIT_RATE:7;              /** @brief 6:0 SPI Bit Rate Control. */
    uint16_t rsvd1:9;                     /** @brief 15:7 Reserved.            */
};

union SPIBRR_REG {
    uint16_t all;
    struct   SPIBRR_BITS  bit;
};

struct SPIFFTX_BITS {                     /** @brief bits description.            */
    uint16_t TXFFIL:5;                    /** @brief 4:0 TXFIFO Interrupt Level.  */
    uint16_t TXFFIENA:1;                  /** @brief 5 TXFIFO Interrupt Enable.   */
    uint16_t TXFFINTCLR:1;                /** @brief 6 TXFIFO Interrupt Clear.    */
    uint16_t TXFFINT:1;                   /** @brief 7 TXFIFO Interrupt Flag.     */
    uint16_t TXFFST:5;                    /** @brief 12:8 Transmit FIFO Status.   */
    uint16_t TXFIFO:1;                    /** @brief 13 TXFIFO Reset.             */
    uint16_t SPIFFENA:1;                  /** @brief 14 FIFO Enhancements Enable. */
    uint16_t SPIRST:1;                    /** @brief 15 SPI Reset.                */
};

union SPIFFTX_REG {
    uint16_t  all;
    struct  SPIFFTX_BITS  bit;
};

struct SPIFFRX_BITS {                     /** @brief bits description.               */
    uint16_t RXFFIL:5;                    /** @brief 4:0 RXFIFO Interrupt Level.     */
    uint16_t RXFFIENA:1;                  /** @brief 5 RXFIFO Interrupt Enable.      */
    uint16_t RXFFINTCLR:1;                /** @brief 6 RXFIFO Interrupt Clear.       */
    uint16_t RXFFINT:1;                   /** @brief 7 RXFIFO Interrupt Flag.        */
    uint16_t RXFFST:5;                    /** @brief 12:8 Receive FIFO Status.       */
    uint16_t RXFIFORESET:1;               /** @brief 13 RXFIFO Reset.                */
    uint16_t RXFFOVFCLR:1;                /** @brief 14 Receive FIFO Overflow Clear. */
    uint16_t RXFFOVF:1;                   /** @brief 15 Receive FIFO Overflow Flag.  */
};

union SPIFFRX_REG {
    uint16_t all;
    struct   SPIFFRX_BITS  bit;
};

struct SPIFFCT_BITS {                     /** @brief bits description.             */
    uint16_t TXDLY:8;                     /** @brief 7:0 FIFO Transmit Delay Bits. */
    uint16_t rsvd1:8;                     /** @brief 15:8 Reserved.                */
};

union SPIFFCT_REG {
    uint16_t all;
    struct   SPIFFCT_BITS  bit;
};

struct SPIPRI_BITS {                      /** @brief bits description.         */
    uint16_t TRIWIRE:1;                   /** @brief 0 3-wire mode select bit. */
    uint16_t STEINV:1;                    /** @brief 1 SPISTE inversion bit.   */
    uint16_t rsvd1:2;                     /** @brief 3:2 Reserved.             */
    uint16_t FREE:1;                      /** @brief 4 Free emulation mode.    */
    uint16_t SOFT:1;                      /** @brief 5 Soft emulation mode.    */
    uint16_t rsvd2:1;                     /** @brief 6 Reserved.               */
    uint16_t rsvd3:9;                     /** @brief 15:7 Reserved.            */
};

union SPIPRI_REG {
    uint16_t all;
    struct   SPIPRI_BITS  bit;
};

struct SPI_REGS {
    union    SPICCR_REG  SPICCR;          /** @brief SPI Configuration Control Register. */
    union    SPICTL_REG  SPICTL;          /** @brief SPI Operation Control Register.     */
    union    SPISTS_REG  SPISTS;          /** @brief SPI Status Register.                */
    uint16_t rsvd1;                       /** @brief Reserved.                           */
    union    SPIBRR_REG  SPIBRR;          /** @brief SPI Baud Rate Register.             */
    uint16_t rsvd2;                       /** @brief Reserved.                           */
    uint16_t SPIRXEMU;                    /** @brief SPI Emulation Buffer Register.      */
    uint16_t SPIRXBUF;                    /** @brief SPI Serial Input Buffer Register.   */
    uint16_t SPITXBUF;                    /** @brief SPI Serial Output Buffer Register.  */
    uint16_t SPIDAT;                      /** @brief SPI Serial Data Register.           */
    union    SPIFFTX_REG SPIFFTX;         /** @brief SPI FIFO Transmit Register.         */
    union    SPIFFRX_REG SPIFFRX;         /** @brief SPI FIFO Receive Register.          */
    union    SPIFFCT_REG SPIFFCT;         /** @brief SPI FIFO Control Register.          */
    uint16_t rsvd3[2];                    /** @brief Reserved.                           */
    union    SPIPRI_REG  SPIPRI;          /** @brief SPI Priority Control Register.      */
};

/**
 * @brief SPI External References & Function Declarations:
 */
extern volatile struct SPI_REGS SpiaRegs;
extern volatile struct SPI_REGS SpibRegs;

#endif /** @brief version */

#ifdef __cplusplus
}
#endif

#endif /** @brief DRIVERLIB_REGS_REGS_SPI_V30_H_ */
