/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/*
 * commit history
 * 20240620,WenGuangYong,add filter 5,Any API related to filterNumber has changed
 */
#ifndef SDFM_H
#define SDFM_H

/**
 * @brief If building with a C++ compiler, make all of the definitions in this header
 * have a C binding.
 */
#ifdef __cplusplus
extern "C"
{
#endif

#include <stdbool.h>
#include <stdint.h>
#include "inc/hw_sdfm.h"
#include "inc/hw_memmap.h"
#include "cpu.h"
#include "debug.h"

#if (defined(SDFM_VERSION))

/**
 * @brief Macro to get the low threshold
 */
#define SDFM_GET_LOW_THRESHOLD(C)    ((uint16_t)(C))

/**
 * @brief Macro to get the high threshold
 */
#define SDFM_GET_HIGH_THRESHOLD(C)   ((uint16_t)((uint32_t)(C) >> 16U))
/* #define SDFM_GET_HIGH_THRESHOLD(C)    ((uint16_t)(C)) */

/**
 * @brief Macro to get the high threshold 1 & 2 to be passed as lowThreshold
 * parameter to SDFM_setCompFilterLowThreshold().
 */
#define SDFM_GET_LOW_THRESHOLD_BOTH(C1, C2)                                    \
                        ((((uint32_t)(SDFM_GET_LOW_THRESHOLD(C2))) << 16U) |   \
                         ((uint32_t)(SDFM_GET_LOW_THRESHOLD(C1))))

/**
 * @brief Macro to get the high threshold 1 & 2 to be passed as highThreshold
 * parameter to SDFM_setCompFilterHighThreshold().
 */
#define SDFM_GET_HIGH_THRESHOLD_BOTH(C1, C2)                                   \
                        ((((uint32_t)(SDFM_GET_HIGH_THRESHOLD(C2))) << 16U) |  \
                         ((uint32_t)(SDFM_GET_HIGH_THRESHOLD(C1))))

/**
 * @brief Macro to convert comparator over sampling ratio to acceptable bit location
 */
#define SDFM_SET_OSR(X)         (((X) - 1) << 8U)

/**
 * @brief Macro to convert the data shift bit values to acceptable bit location
 */
#define SDFM_SHIFT_VALUE(X)     ((X) << 2U)

/**
 * @brief Macro to combine high threshold and low threshold values
 */
#define SDFM_THRESHOLD(H, L)     ((((uint32_t)(H)) << 16U) | (L))

/**
 * @brief Macro to set the FIFO level to acceptable bit location
 */
#define SDFM_SET_FIFO_LEVEL(X)  ((X) << 7U)

/**
 * @brief Macro to set and enable the zero cross threshold value.
 */
#define SDFM_SET_ZERO_CROSS_THRESH_VALUE(X) (0x8000 | (X))

/**
 * @brief Macros to enable or disable filter.
 */
#define SDFM_FILTER_DISABLE     0x0U
#define SDFM_FILTER_ENABLE      0x2U

/**
 * @brief The offsets of each channel register in the data filter module
 */
#define SDFM_SDFIL_OFFSET       (SDFM_O_SDCTLPARM2 - SDFM_O_SDCTLPARM1)

/**
 * @brief The offsets of each channel register of the comparator filter module
 */
#define SDFM_DIGFIL_OFFSET      (SDFM_O_SDCOMP2CTL - SDFM_O_SDCOMP1CTL)

/**
 * @brief The offset of the double high-threshold registers within the channel
 */
#define SDFM_SDFLT1CMPHx_OFFSET    (SDFM_O_SDFLT1CMPH2 - SDFM_O_SDFLT1CMPH1)

/**
 * @brief The offset of the double low-threshold registers within the channel
 */
#define SDFM_SDFLT1CMPLx_OFFSET    (SDFM_O_SDFLT1CMPL2 - SDFM_O_SDFLT1CMPL1)

/**
 * @brief Define to mask out the bits in the SDCOMPHFILCTL register that aren't
 * associated with comparator event filter configurations. Added for internal
 * use, not to be used in application code.
 */
#define SDFM_COMPEVT_FILTER_CONFIG_M  (SDFM_SDCOMP1EVT1FLTCTL_SAMPWIN_M  |     \
                                       SDFM_SDCOMP1EVT1FLTCTL_THRESH_M)

/**
 * @brief Define to mask out the bits in the SDCOMPLOCK register that aren't
 * associated with lock configuration.  Added for internal use, not to
 * be used in application code.
 *
 */
#define SDFM_COMPEVT_FILTER_LOCK_M  (SDFM_SDCOMPLOCK_SDCOMPCTL |               \
                                     SDFM_SDCOMPLOCK_COMP)

/**
 * @brief Values that can be passed to SDFM_enableSynchronizer() or
 * SDFM_disableSynchronizer() as syncConfig parameter.
 * 
 * @param SDFM_CLOCK_SYNCHRONIZER Define for Clock synchronizer Configuration
 * @param SDFM_DATA_SYNCHRONIZER Define for Data Synchronizer Configuration
 */
#define SDFM_CLOCK_SYNCHRONIZER    SDFM_SDCTLPARM1_SDCLKSYNC
#define SDFM_DATA_SYNCHRONIZER     SDFM_SDCTLPARM1_SDDATASYNC

#define SDFM_SDCPARMx_CEVTxSEL_MASK  0x2

/**
 * @brief Values that can be returned from SDFM_getThresholdStatus()
 * 
 * @param SDFM_OUTPUT_WITHIN_THRESHOLD SDFM output is within threshold
 * @param SDFM_OUTPUT_ABOVE_THRESHOLD SDFM output is above threshold
 * @param SDFM_OUTPUT_BELOW_THRESHOLD SDFM output is below threshold
 */
typedef enum
{
    SDFM_OUTPUT_WITHIN_THRESHOLD = 0,
    SDFM_OUTPUT_ABOVE_THRESHOLD  = 1,
    SDFM_OUTPUT_BELOW_THRESHOLD  = 2
} SDFM_OutputThresholdStatus;

/**
 * @brief Values that can be passed to all functions as the filterNumber parameter.
 * 
 * @param SDFM_FILTER_1 Digital filter 1
 * @param SDFM_FILTER_2 Digital filter 2
 * @param SDFM_FILTER_3 Digital filter 3
 * @param SDFM_FILTER_4 Digital filter 4
 * @param SDFM_FILTER_5 Digital filter 5
 */
typedef enum
{
    SDFM_FILTER_1 = 0,
    SDFM_FILTER_2 = 1,
    SDFM_FILTER_3 = 2,
    SDFM_FILTER_4 = 3,
#if SDFM_VERSION == 300
	SDFM_FILTER_5 = 4
#endif
} SDFM_FilterNumber;

/**
 * @brief Values that can be passed to SDFM_setFilterType(),
 * SDFM_setComparatorFilterType() as the filterType parameter.
 * 
 * @param SDFM_FILTER_SINC_FAST Digital filter with SincFast structure.
 * @param SDFM_FILTER_SINC_2 Digital filter with Sinc3 structure.
 * @param SDFM_FILTER_SINC_1 Digital filter with Sinc1 structure.
 * @param SDFM_FILTER_SINC_3 Digital filter with Sinc4 structure.
 */
typedef enum
{
    SDFM_FILTER_SINC_FAST = 0x00,
    SDFM_FILTER_SINC_1    = 0x10,
    SDFM_FILTER_SINC_2    = 0x20,
    SDFM_FILTER_SINC_3    = 0x30
} SDFM_FilterType;

/**
 * @brief Values that can be passed to SDFM_setupModulatorClock(),as the
 * clockMode parameter.
 * 
 * @param SDFM_MODULATOR_CLK_EQUAL_DATA_RATE Modulator clock is identical to the data rate.
 * @param SDFM_MODULATOR_CLK_HALF_DATA_RATE Modulator clock is half the data rate.
 * @param SDFM_MODULATOR_CLK_OFF Modulator clock is off. Data is Manchester coded.
 * @param SDFM_MODULATOR_CLK_DOUBLE_DATA_RATE Modulator clock is double the data rate.
 */
typedef enum
{
	SDFM_MODULATOR_CLK_EQUAL_DATA_RATE  = 0,
	SDFM_MODULATOR_CLK_HALF_DATA_RATE   = 1,
	SDFM_MODULATOR_CLK_OFF              = 2,
	SDFM_MODULATOR_CLK_DOUBLE_DATA_RATE = 3
} SDFM_ModulatorClockMode;

/**
 * @brief Values that can be passed to SDFM_setOutputDataFormat(),as the
 * dataFormat parameter.
 * 
 * @param SDFM_DATA_FORMAT_16_BIT Filter output is in 16 bits 2's complement format.
 * @param SDFM_DATA_FORMAT_32_BIT Filter output is in 32 bits 2's complement format.
 */
typedef enum
{
   SDFM_DATA_FORMAT_16_BIT = 0,
   SDFM_DATA_FORMAT_32_BIT = 1
} SDFM_OutputDataFormat;

/**
 * @brief Values that can be passed to SDFM_setDataReadyInterruptSource(),as the
 * dataReadySource parameter.
 * 
 * @param SDFM_DATA_READY_SOURCE_DIRECT Data ready interrupt source is direct (non -FIFO).
 * @param SDFM_DATA_READY_SOURCE_FIFO Data ready interrupt source is FIFO.
 */
typedef enum
{
   SDFM_DATA_READY_SOURCE_DIRECT = 0,
   SDFM_DATA_READY_SOURCE_FIFO = 1
} SDFM_DataReadyInterruptSource;

/**
 * @brief Values that can be passed to SDFM_setPWMSyncSource(),as the
 * syncSource parameter.
 * 
 * @param SDFM_SYNC_PWM1_SOCA SDFM sync source is PWM1 SOCA
 * @param SDFM_SYNC_PWM1_SOCB SDFM sync source is PWM1 SOCB
 * @param SDFM_SYNC_PWM2_SOCA SDFM sync source is PWM2 SOCA
 * @param SDFM_SYNC_PWM2_SOCB SDFM sync source is PWM2 SOCB
 * @param SDFM_SYNC_PWM3_SOCA SDFM sync source is PWM3 SOCA
 * @param SDFM_SYNC_PWM3_SOCB SDFM sync source is PWM3 SOCB
 * @param SDFM_SYNC_PWM4_SOCA SDFM sync source is PWM4 SOCA
 * @param SDFM_SYNC_PWM4_SOCB SDFM sync source is PWM4 SOCB
 * @param SDFM_SYNC_PWM5_SOCA SDFM sync source is PWM5 SOCA
 * @param SDFM_SYNC_PWM5_SOCB SDFM sync source is PWM5 SOCB
 * @param SDFM_SYNC_PWM6_SOCA SDFM sync source is PWM6 SOCA
 * @param SDFM_SYNC_PWM6_SOCB SDFM sync source is PWM6 SOCB
 * @param SDFM_SYNC_PWM7_SOCA SDFM sync source is PWM7 SOCA
 * @param SDFM_SYNC_PWM7_SOCB SDFM sync source is PWM7 SOCB
 * @param SDFM_SYNC_PWM8_SOCA SDFM sync source is PWM8 SOCA
 * @param SDFM_SYNC_PWM8_SOCB SDFM sync source is PWM8 SOCB
 */
typedef enum
{
   SDFM_SYNC_PWM1_SOCA = 0,
   SDFM_SYNC_PWM1_SOCB = 1,
   SDFM_SYNC_PWM2_SOCA = 4,
   SDFM_SYNC_PWM2_SOCB = 5,
   SDFM_SYNC_PWM3_SOCA = 8,
   SDFM_SYNC_PWM3_SOCB = 9,
   SDFM_SYNC_PWM4_SOCA = 12,
   SDFM_SYNC_PWM4_SOCB = 13,
   SDFM_SYNC_PWM5_SOCA = 16,
   SDFM_SYNC_PWM5_SOCB = 17,
   SDFM_SYNC_PWM6_SOCA = 20,
   SDFM_SYNC_PWM6_SOCB = 21,
   SDFM_SYNC_PWM7_SOCA = 24,
   SDFM_SYNC_PWM7_SOCB = 25,
   SDFM_SYNC_PWM8_SOCA = 28,
   SDFM_SYNC_PWM8_SOCB = 29
} SDFM_PWMSyncSource;

/**
 * @brief Values that can be passed to SDFM_setFIFOClearOnSyncMode(),as the
 * fifoClearSyncMode parameter.
 * 
 * @param SDFM_FIFO_NOT_CLEARED_ON_SYNC SDFM FIFO buffer is not cleared on Sync signal
 * @param SDFM_FIFO_CLEARED_ON_SYNC SDFM FIFO buffer is cleared on Sync signal
 */
typedef enum
{
   SDFM_FIFO_NOT_CLEARED_ON_SYNC = 0,
   SDFM_FIFO_CLEARED_ON_SYNC     = 1
} SDFM_FIFOClearSyncMode;

/**
 * @brief Values that can be passed to SDFM_setWaitForSyncClearMode(),as the
 * syncClearMode parameter.
 * 
 * @param SDFM_MANUAL_CLEAR_WAIT_FOR_SYNC Wait for sync cleared using software.
 * @param SDFM_AUTO_CLEAR_WAIT_FOR_SYNC Wait for sync cleared automatically on SDFFINT.
 */
typedef enum
{
   SDFM_MANUAL_CLEAR_WAIT_FOR_SYNC = 0,
   SDFM_AUTO_CLEAR_WAIT_FOR_SYNC   = 1
} SDFM_WaitForSyncClearMode;

/**
 * @brief Values that can be passed to SDFM_selectCompEventSource() as the
 * compEventNum parameter.
 * 
 * @param SDFM_COMP_EVENT_1 Selects CEVT1,Default select SDFM_COMP_EVENT_SRC_COMPH1
 * @param SDFM_COMP_EVENT_2 Selects CEVT2,Default select SDFM_COMP_EVENT_SRC_COMPL1
 */
typedef enum
{
    SDFM_COMP_EVENT_1 = SDFM_SDCPARM1_CEVT1SEL_S,
    SDFM_COMP_EVENT_2 = SDFM_SDCPARM1_CEVT2SEL_S
} SDFM_CompEventNumber;

/**
 * @brief Values that can be passed to SDFM_selectCompEventSource() as the
 * compEventSource parameter.
 * 
 * @param SDFM_COMP_EVENT_SRC_COMPH1 COMPH1 event is the source
 * @param SDFM_COMP_EVENT_SRC_COMPH1_L1 Either of COMPH1 or COMPL1 event
 * @param SDFM_COMP_EVENT_SRC_COMPH2 COMPH2 event is the source
 * @param SDFM_COMP_EVENT_SRC_COMPH2_L2 Either of COMPH2 or COMPL2 event
 * @param SDFM_COMP_EVENT_SRC_COMPL1 COMPL1 event is the source
 * @param SDFM_COMP_EVENT_SRC_COMPL2 COMPL2 event is the source
 */
typedef enum
{
    SDFM_COMP_EVENT_SRC_COMPH1    = 0,
    SDFM_COMP_EVENT_SRC_COMPH1_L1 = 1,
    SDFM_COMP_EVENT_SRC_COMPH2    = 2,
    SDFM_COMP_EVENT_SRC_COMPH2_L2 = 3,
    SDFM_COMP_EVENT_SRC_COMPL1    = 0,
    SDFM_COMP_EVENT_SRC_COMPL2    = 2
} SDFM_CompEventSource;

/**
 * @brief Values that can be passed to SDFM_selectClockSource() as the clkSource
 * parameter.
 * 
 * @param SDFM_CLK_SOURCE_CHANNEL_CLK Source is respective channel clock
 * @param SDFM_CLK_SOURCE_SD1_CLK Source is SD1 channel clock is the source
 */
typedef enum
{
    SDFM_CLK_SOURCE_CHANNEL_CLK = 0x0,
    SDFM_CLK_SOURCE_SD1_CLK = SDFM_SDCTLPARM1_SDCLKSEL
} SDFM_ClockSource;

/**
 * @brief Values that can be passed to SDFM_selectCompEventHighSource() as the
 * source parameter.
 * 
 * @param SDFM_COMPHOUT_SOURCE_COMPHIN Comparator event high source is unfiltered event
 * @param SDFM_COMPHOUT_SOURCE_FILTER Comparator event high source is filtered event
 */
typedef enum
{
    SDFM_COMPHOUT_SOURCE_COMPHIN = 0x0,
    SDFM_COMPHOUT_SOURCE_FILTER  = 0x8
} SDFM_CompEventHighSource;

/**
 * @brief Values that can be passed to SDFM_selectCompEventLowSource() as the
 * source parameter.
 * 
 * @param SDFM_COMPLOUT_SOURCE_COMPLIN Comparator event low source is unfiltered event
 * @param SDFM_COMPLOUT_SOURCE_FILTER Comparator event low source is filtered event
 */
typedef enum
{
    SDFM_COMPLOUT_SOURCE_COMPLIN = 0x000,
    SDFM_COMPLOUT_SOURCE_FILTER  = 0x800
} SDFM_CompEventLowSource;

/**
 * @brief Values that can be passed to SDFM_configCompEventLowFilter() &
 * SDFM_configCompEventHighFilter() as the filterNumber.
 * 
 * @param sampleWindow   Sample window size
 * @param threshold   Majority voting threshold
 * @param clkPrescale   Sample clock pre-scale
 */
typedef struct
{
    uint16_t sampleWindow;
    uint16_t threshold;
    uint16_t clkPrescale;
} SDFM_CompEventFilterConfig;

/**
 * @brief Values that can be passed to SDFM_enableInterrupt and
 * SDFM_disableInterrupt as intFlags parameter.
 * 
 * @param
 * SDFM_MODULATOR_FAILURE_INTERRUPT Interrupt is generated if Modulator fails.
 * @param
 * SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT Interrupt on Comparator low-level threshold.
 * @param
 * SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT Interrupt on Comparator high-level threshold.
 * @param
 * SDFM_DATA_FILTER_ACKNOWLEDGE_INTERRUPT Interrupt on Acknowledge flag
 * @param SDFM_FIFO_INTERRUPT Interrupt on FIFO level
 * @param SDFM_FIFO_OVERFLOW_INTERRUPT Interrupt on FIFO overflow
 */
#define SDFM_MODULATOR_FAILURE_INTERRUPT       0x200U
#define SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT     0x40U
#define SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT    0x20U
#define SDFM_DATA_FILTER_ACKNOWLEDGE_INTERRUPT 0x1U
#define SDFM_FIFO_INTERRUPT                    0x1000U
#define SDFM_FIFO_OVERFLOW_INTERRUPT           0x8000U

/**
 * @brief Values that can be passed to SDFM_clearInterruptFlag flags parameter.
 * Main interrupt flag.
 * @param SDFM_MAIN_INTERRUPT_FLAG Main interrupt flag
 * @param SDFM_FILTER_1_HIGH_THRESHOLD_FLAG Filter 1 high -level threshold flag
 * @param SDFM_FILTER_1_LOW_THRESHOLD_FLAG Filter 1 low -level threshold flag
 * @param SDFM_FILTER_2_HIGH_THRESHOLD_FLAG Filter 2 high -level threshold flag
 * @param SDFM_FILTER_2_LOW_THRESHOLD_FLAG Filter 2 low -level threshold flag
 * 
 * @param SDFM_FILTER_3_HIGH_THRESHOLD_FLAG Filter 3 high -level threshold fla
 * @param SDFM_FILTER_3_LOW_THRESHOLD_FLAG Filter 3 low -level threshold flag
 * @param SDFM_FILTER_4_HIGH_THRESHOLD_FLAG Filter 4 high -level threshold flag
 * @param SDFM_FILTER_4_LOW_THRESHOLD_FLAG Filter 4 low -level threshold flag
 * @param SDFM_FILTER_1_MOD_FAILED_FLAG Filter 1 modulator failed flag
 * 
 * @param SDFM_FILTER_2_MOD_FAILED_FLAG Filter 2 modulator failed flag
 * @param SDFM_FILTER_3_MOD_FAILED_FLAG Filter 3 modulator failed flag
 * @param SDFM_FILTER_4_MOD_FAILED_FLAG Filter 4 modulator failed flag
 * @param SDFM_FILTER_1_NEW_DATA_FLAG Filter 1 new data flag
 * @param SDFM_FILTER_2_NEW_DATA_FLAG Filter 2 new data flag
 * 
 * @param SDFM_FILTER_3_NEW_DATA_FLAG Filter 3 new data flag
 * @param SDFM_FILTER_4_NEW_DATA_FLAG Filter 4 new data flag
 * @param SDFM_FILTER_1_FIFO_OVERFLOW_FLAG Filter 1 FIFO overflow flag
 * @param SDFM_FILTER_2_FIFO_OVERFLOW_FLAG Filter 2 FIFO overflow flag
 * @param SDFM_FILTER_3_FIFO_OVERFLOW_FLAG Filter 3 FIFO overflow flag
 * 
 * @param SDFM_FILTER_4_FIFO_OVERFLOW_FLAG Filter 4 FIFO overflow flag
 * @param SDFM_FILTER_1_FIFO_INTERRUPT_FLAG Filter 1 FIFO interrupt flag
 * @param SDFM_FILTER_2_FIFO_INTERRUPT_FLAG Filter 2 FIFO interrupt flag
 * @param SDFM_FILTER_3_FIFO_INTERRUPT_FLAG Filter 3 FIFO interrupt flag
 * @param SDFM_FILTER_4_FIFO_INTERRUPT_FLAG Filter 4 FIFO interrupt flag
 * 
 * @param SDFM_FILTER_5_HIGH_THRESHOLD_FLAG Filter 5 high -level threshold flag
 * @param SDFM_FILTER_5_LOW_THRESHOLD_FLAG Filter 5 low -level threshold flag
 * @param SDFM_FILTER_5_MOD_FAILED_FLAG Filter 5 modulator failed flag
 * @param SDFM_FILTER_5_MOD_FAILED_FLAG Filter 5 new data flag
 * @param SDFM_FILTER_5_FIFO_OVERFLOW_FLAG Filter 5 FIFO overflow flag
 * @param SDFM_FILTER_5_FIFO_INTERRUPT_FLAG Filter 5 FIFO interrupt flag
 */
#define SDFM_MAIN_INTERRUPT_FLAG           0x80000000U
#define SDFM_FILTER_1_HIGH_THRESHOLD_FLAG  0x1U
#define SDFM_FILTER_1_LOW_THRESHOLD_FLAG   0x2U
#define SDFM_FILTER_2_HIGH_THRESHOLD_FLAG  0x4U
#define SDFM_FILTER_2_LOW_THRESHOLD_FLAG   0x8U

#define SDFM_FILTER_3_HIGH_THRESHOLD_FLAG  0x10U
#define SDFM_FILTER_3_LOW_THRESHOLD_FLAG   0x20U
#define SDFM_FILTER_4_HIGH_THRESHOLD_FLAG  0x40U
#define SDFM_FILTER_4_LOW_THRESHOLD_FLAG   0x80U
#define SDFM_FILTER_1_MOD_FAILED_FLAG      0x100U


#define SDFM_FILTER_2_MOD_FAILED_FLAG      0x200U
#define SDFM_FILTER_3_MOD_FAILED_FLAG      0x400U
#define SDFM_FILTER_4_MOD_FAILED_FLAG      0x800U
#define SDFM_FILTER_1_NEW_DATA_FLAG        0x1000U
#define SDFM_FILTER_2_NEW_DATA_FLAG        0x2000U


#define SDFM_FILTER_3_NEW_DATA_FLAG        0x4000U
#define SDFM_FILTER_4_NEW_DATA_FLAG        0x8000U
#define SDFM_FILTER_1_FIFO_OVERFLOW_FLAG   0x10000U
#define SDFM_FILTER_2_FIFO_OVERFLOW_FLAG   0x20000U
#define SDFM_FILTER_3_FIFO_OVERFLOW_FLAG   0x40000U

#define SDFM_FILTER_4_FIFO_OVERFLOW_FLAG   0x80000U
#define SDFM_FILTER_1_FIFO_INTERRUPT_FLAG  0x100000U
#define SDFM_FILTER_2_FIFO_INTERRUPT_FLAG  0x200000U
#define SDFM_FILTER_3_FIFO_INTERRUPT_FLAG  0x400000U
#define SDFM_FILTER_4_FIFO_INTERRUPT_FLAG  0x800000U

#if SDFM_VERSION == 300
#define SDFM_FILTER_5_HIGH_THRESHOLD_FLAG  SDFM_SDIFLGCLR_FLT5_FLG_CEVT1
#define SDFM_FILTER_5_LOW_THRESHOLD_FLAG   SDFM_SDIFLGCLR_FLT5_FLG_CEVT2
#define SDFM_FILTER_5_MOD_FAILED_FLAG      SDFM_SDIFLGCLR_MF5
#define SDFM_FILTER_5_NEW_DATA_FLAG        SDFM_SDIFLGCLR_AF5
#define SDFM_FILTER_5_FIFO_OVERFLOW_FLAG   SDFM_SDIFLGCLR_SDFFOVF5
#define SDFM_FILTER_5_FIFO_INTERRUPT_FLAG  SDFM_SDIFLGCLR_SDFFINT5
#endif

/**
 * @brief Checks SDFM base address.
 *
 * @details
 * This function determines if SDFM module base address is valid.
 *
 * @param[in] base  Specifies the SDFM module base address.
 *
 * @return bool     true if base is the valid SDFM module base address
 */
#ifdef DEBUG
static inline bool
SDFM_isBaseValid(uint32_t base)
{
    return(
           (base == SDFM1_BASE) ||
           (base == SDFM2_BASE)
          );
}
#endif

/**
 * @brief Enable external reset
 *
 * @details
 * This function enables data filter to be reset by an external source
 * (PWM compare output).
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 */
static inline void
SDFM_enableExternalReset(uint32_t base, SDFM_FilterNumber filterNumber)
{
    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        /* Set the SDSYNCEN bit */
        HWREGH(base + SDFM_O_SDDFPARM1 + ((uint32_t)filterNumber * 32U)) |=
            SDFM_SDDFPARM1_SDSYNCEN;

    #elif (SDFM_VERSION == 300)
        ASSERT(SDFM_isBaseValid(base));
        ASSERT(filterNumber <= SDFM_FILTER_5);

        /* Set the SDSYNCEN bit */
        if(filterNumber != SDFM_FILTER_5)
        {
            HWREGH(base + SDFM_O_SDDFPARM1 + ((uint32_t)filterNumber * 32U)) |=
                SDFM_SDDFPARM1_SDSYNCEN;
        }else
        {
            HWREGH(base + SDFM_O_SDDFPARM5) |=
                SDFM_SDDFPARM5_SDSYNCEN;
        }
    #endif
}

/**
 * @brief Disable external reset
 *
 * @details
 * This function disables data filter from being reset by an external source
 * (PWM compare output).
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 */
static inline void
SDFM_disableExternalReset(uint32_t base, SDFM_FilterNumber filterNumber)
{
    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        /* Clear the SDSYNCEN bit */
        HWREGH(base + SDFM_O_SDDFPARM1 + ((uint32_t)filterNumber * 32U)) &=
            ~SDFM_SDDFPARM1_SDSYNCEN;

    #elif (SDFM_VERSION == 300)
        ASSERT(SDFM_isBaseValid(base));
        ASSERT(filterNumber <= SDFM_FILTER_5);

        /* Clear the SDSYNCEN bit */
        if(filterNumber != SDFM_FILTER_5)
        {
            HWREGH(base + SDFM_O_SDDFPARM1 + ((uint32_t)filterNumber * 32U)) &=
                ~SDFM_SDDFPARM1_SDSYNCEN;
        }else
        {
            HWREGH(base + SDFM_O_SDDFPARM5 ) &= ~SDFM_SDDFPARM5_SDSYNCEN;
        }
    #endif
}

/**
 * @brief Enable filter
 *
 * @details
 * This function enables the filter specified by the filterNumber variable.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 */
static inline void
SDFM_enableFilter(uint32_t base, SDFM_FilterNumber filterNumber)
{
    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        /* Set the FEN bit */
        HWREGH(base + SDFM_O_SDDFPARM1 + ((uint32_t)filterNumber * 32U)) |=
            SDFM_SDDFPARM1_FEN;
        EDIS;
    #elif (SDFM_VERSION == 300)
        ASSERT(SDFM_isBaseValid(base));
        ASSERT(filterNumber <= SDFM_FILTER_5);
 
        /* Set the FEN bit */
        if(filterNumber != SDFM_FILTER_5)
        {
            HWREGH(base + SDFM_O_SDDFPARM1 + ((uint32_t)filterNumber * 32U)) |=
                SDFM_SDDFPARM1_FEN;
        }else
        {
            HWREGH(base + SDFM_O_SDDFPARM5) |= SDFM_SDDFPARM5_FEN;
        }
    #endif
}

/**
 * @brief Disable filter
 *
 * @details
 * This function disables the filter specified by the filterNumber variable.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 */
static inline void
SDFM_disableFilter(uint32_t base, SDFM_FilterNumber filterNumber)
{
    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        /* Set the FEN bit */
        HWREGH(base + SDFM_O_SDDFPARM1 + ((uint32_t)filterNumber * 32U)) &=
            ~SDFM_SDDFPARM1_FEN;
    #elif (SDFM_VERSION == 300)
        ASSERT(SDFM_isBaseValid(base));
        ASSERT(filterNumber <= SDFM_FILTER_5);
 
        /* Set the FEN bit */
        if(filterNumber != SDFM_FILTER_5)
        {
            HWREGH(base + SDFM_O_SDDFPARM1 + ((uint32_t)filterNumber * 32U)) &=
                ~SDFM_SDDFPARM1_FEN;
        }else
        {
            HWREGH(base + SDFM_O_SDDFPARM5) &= ~SDFM_SDDFPARM5_FEN;
        }
    #endif
}

/**
 * @brief Enable FIFO buffer
 *
 * @details
 * This function enables the filter FIFO buffer specified by the
 * filterNumber variable.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 */
static inline void
SDFM_enableFIFOBuffer(uint32_t base, SDFM_FilterNumber filterNumber)
{
    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        /* Set the FFEN bit */
        HWREGH(base + SDFM_O_SDFIFOCTL1 + ((uint32_t)filterNumber * 32U)) |=
            SDFM_SDFIFOCTL1_FFEN;
    #elif (SDFM_VERSION == 300)
        ASSERT(SDFM_isBaseValid(base));
        ASSERT(filterNumber <= SDFM_FILTER_5);
 

        /* Set the FFEN bit */
        if(filterNumber != SDFM_FILTER_5)
        {
            HWREGH(base + SDFM_O_SDFIFOCTL1 + ((uint32_t)filterNumber * 32U)) |=
                SDFM_SDFIFOCTL1_FFEN;
        }else
        {
            HWREGH(base + SDFM_O_SDFIFOCTL5 ) |= SDFM_SDFIFOCTL5_FFEN;
        }
    #endif
}

/**
 * @brief Disable FIFO buffer
 *
 * @details
 * This function disables the filter FIFO buffer specified by the
 * filterNumber variable.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 */
static inline void
SDFM_disableFIFOBuffer(uint32_t base, SDFM_FilterNumber filterNumber)
{
    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        /* Clear the FFEN bit */
        HWREGH(base + SDFM_O_SDFIFOCTL1 + ((uint32_t)filterNumber * 32U)) &=
            ~SDFM_SDFIFOCTL1_FFEN;
    #elif (SDFM_VERSION == 300)
        ASSERT(SDFM_isBaseValid(base));
        ASSERT(filterNumber <= SDFM_FILTER_5);
 
        /* Clear the FFEN bit */
        if(filterNumber != SDFM_FILTER_5)
        {
            HWREGH(base + SDFM_O_SDFIFOCTL1 + ((uint32_t)filterNumber * 32U)) &=
                ~SDFM_SDFIFOCTL1_FFEN;
        }else
        {
            HWREGH(base + SDFM_O_SDFIFOCTL5 ) &=~ SDFM_SDFIFOCTL5_FFEN;
        }
    #endif
}

/**
 * @brief Return the Zero Cross Trip status
 *
 * @details
 * This function returns the Zero Cross Trip status for the filter
 * specified by filterNumber variable.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 *
 * @return true if Comparator filter output >= High-level threshold (Z)
 * false if Comparator filter output < High-level threshold (Z)
 */
static inline bool
SDFM_getZeroCrossTripStatus(uint32_t base, SDFM_FilterNumber filterNumber)
{
    ASSERT(SDFM_isBaseValid(base));

    return(((HWREGH(base + SDFM_O_SDSTATUS) >> (uint16_t)filterNumber) &
            0x1U) == 1U);
}

/**
 * @brief Clear the Zero Cross Trip status
 *
 * @details
 * This function clears the Zero Cross Trip status for the filter
 * specified by filterNumber variable.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 */
static inline void
SDFM_clearZeroCrossTripStatus(uint32_t base, SDFM_FilterNumber filterNumber)
{
    ASSERT(SDFM_isBaseValid(base));

    /* Set SDCTL MIE bit */
    HWREGH(base + SDFM_O_SDCTL) |= ((uint16_t)1U << (uint16_t)filterNumber);
}

/**
 * @brief Enable Comparator.
 *
 * @details
 * This function enables the Comparator for the selected filter.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 */
static inline void
SDFM_enableComparator(uint32_t base, SDFM_FilterNumber filterNumber)
{
    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        /* Set CEN bit */
        HWREGH(base + SDFM_O_SDCPARM1 + ((uint32_t)filterNumber * 32U)) |=
            SDFM_SDCPARM1_CEN;
    #elif (SDFM_VERSION == 300)
        ASSERT(SDFM_isBaseValid(base));
        ASSERT(filterNumber <= SDFM_FILTER_5);
 
        /* Set CEN bit */
        if(filterNumber != SDFM_FILTER_5)
        {
            HWREGH(base + SDFM_O_SDCPARM1 + ((uint32_t)filterNumber * 32U)) |=
                SDFM_SDCPARM1_CEN;
        }else
        {
            HWREGH(base + SDFM_O_SDCPARM5) |= SDFM_SDCPARM5_CEN;
        }
    #endif
}

/**
 * @brief Disable Comparator.
 *
 * @details
 * This function disables the Comparator for the selected filter.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 */
static inline void
SDFM_disableComparator(uint32_t base, SDFM_FilterNumber filterNumber)
{
    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        /* Clear CEN bit */
        HWREGH(base + SDFM_O_SDCPARM1 + ((uint32_t)filterNumber * 32U)) &=
            ~SDFM_SDCPARM1_CEN;
    #elif (SDFM_VERSION == 300)
        ASSERT(SDFM_isBaseValid(base));
        ASSERT(filterNumber <= SDFM_FILTER_5);
 
        /* Clear CEN bit */
        if(filterNumber != SDFM_FILTER_5)
        {
            HWREGH(base + SDFM_O_SDCPARM1 + ((uint32_t)filterNumber * 32U)) &=
                ~SDFM_SDCPARM1_CEN;
        }else
        {
            HWREGH(base + SDFM_O_SDCPARM5) &=~SDFM_SDCPARM5_CEN;
        }
    #endif
}

/**
 * @brief Selects Comparator Event Source.
 *
 * @details
 * This function selects the comparator event source.
 *
 * @param[in] base              Specifies the SDFM module base address.
 * @param[in] filterNumber      The filter number.
 * @param[in] compEventNum      The event number
 * @param[in] compEventSource   The event source
 *
 * @remarks
 *   Valid values for compEventNum are:
 *     SDFM_COMP_EVENT_1 - Selects comparator event 1
 *     SDFM_COMP_EVENT_2 - Selects comparator event 2
 *   Valid values for SDFM_COMP_EVENT_1 are:
 *     SDFM_COMP_EVENT_SRC_COMPH1 	 - COMPH1 event is the source for selected event
 *     SDFM_COMP_EVENT_SRC_COMPH1_L1 - Either of COMPH1 or COMPL1 event can be
 *                                     the source for selected event
 *     SDFM_COMP_EVENT_SRC_COMPH2 	 - COMPH2 event is the source for selected event
 *     SDFM_COMP_EVENT_SRC_COMPH2_L2 - Either of COMPH2 or COMPL2 event can be
 *                                     the source for selected event
 *   Valid values for SDFM_COMP_EVENT_2 are:
 *     SDFM_COMP_EVENT_SRC_COMPL1 	 - COMPL1 event is the source for selected event
 *     SDFM_COMP_EVENT_SRC_COMPH1_L1 - Either of COMPH1 or COMPL1 event can be
 *                                     the source for selected event
 *     SDFM_COMP_EVENT_SRC_COMPL2 	 - COMPL2 event is the source for selected event
 *     SDFM_COMP_EVENT_SRC_COMPH2_L2 - Either of COMPH2 or COMPL2 event can be
 *                                     the source for selected event
 */
static inline void
SDFM_selectCompEventSource(uint32_t base, SDFM_FilterNumber filterNumber,
                           SDFM_CompEventNumber compEventNum,
                           SDFM_CompEventSource compEventSource)
{
    #if (SDFM_VERSION == 200)
        uint32_t address;
        ASSERT(SDFM_isBaseValid(base));
        address = base + SDFM_O_SDCPARM1 + ((uint32_t)filterNumber *
                                            SDFM_SDFIL_OFFSET);

        /* Select source for selected comparator event */
        HWREGH(address) = (HWREGH(address) & ~((uint16_t)SDFM_SDCPARMx_CEVTxSEL_MASK <<
                        (uint16_t)compEventNum)) |
                        ((uint16_t)compEventSource << (uint16_t)compEventNum) ;

    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        uint32_t address;
        ASSERT(SDFM_isBaseValid(base));
        if(filterNumber != SDFM_FILTER_5)
        {
            address = base + SDFM_O_SDCPARM1 + ((uint32_t)filterNumber *
                                                SDFM_SDFIL_OFFSET);

            /* Select source for selected comparator event */
            EALLOW;
            HWREGH(address) = (HWREGH(address) & ~((uint16_t)SDFM_SDCPARMx_CEVTxSEL_MASK <<
                            (uint16_t)compEventNum)) |
                            ((uint16_t)compEventSource << (uint16_t)compEventNum) ;
            EDIS;
        }
        else
        {
            address = base + SDFM_O_SDCPARM5;
            /* Select source for selected comparator event */
            HWREGH(address) = (HWREGH(address) & ~((uint16_t)SDFM_SDCPARMx_CEVTxSEL_MASK <<
                            (uint16_t)compEventNum)) |
                            ((uint16_t)compEventSource << (uint16_t)compEventNum) ;  //TODO confirm 0x2U
        }
    #endif
}

/**
 * @brief Set filter type.
 *
 * @details
 * This function sets the filter type or structure to be used as specified by
 * filterType for the selected filter number as specified by filterNumber.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 * @param[in] filterType    The filter type or structure.
 */
static inline void
SDFM_setFilterType(uint32_t base, SDFM_FilterNumber filterNumber,
                   SDFM_FilterType filterType)
{
    #if (SDFM_VERSION == 200)
    uint32_t address;

    ASSERT(SDFM_isBaseValid(base));

    address = base + SDFM_O_SDDFPARM1 + ((uint32_t)filterNumber * 32U);

    /* Write to SST bits */
    HWREGH(address) = (HWREGH(address) & (~SDFM_SDDFPARM1_SST_M)) |
                      ((uint16_t)filterType << 6U);
    #elif (SDFM_VERSION == 300)
        ASSERT(SDFM_isBaseValid(base));
        ASSERT(filterNumber <= SDFM_FILTER_5);
        uint32_t address;

        ASSERT(SDFM_isBaseValid(base));
        if(filterNumber != SDFM_FILTER_5)
        {

            address = base + SDFM_O_SDDFPARM1 + ((uint32_t)filterNumber * 32U);

            /* Write to SST bits */
            HWREGH(address) = (HWREGH(address) & (~SDFM_SDDFPARM1_SST_M)) |
                            ((uint16_t)filterType << 6U);
        }else
        {


            address = base + SDFM_O_SDDFPARM5;

            /* Write to SST bits */
            HWREGH(address) = (HWREGH(address) & (~SDFM_SDDFPARM5_SST_M)) |
                            ((uint16_t)filterType << 6U);
        }
    #endif
}

/**
 * @brief Set data filter over sampling ratio.
 *
 * @details
 * This function sets the filter oversampling ratio for the filter specified
 * by the filterNumber variable.Valid values for the variable overSamplingRatio
 * are 0 to 255 inclusive. The actual oversampling ratio will be this value
 * plus one.
 *
 * @param[in] base              Specifies the SDFM module base address.
 * @param[in] filterNumber      The filter number.
 * @param[in] overSamplingRatio The data filter over sampling ratio.
 */
static inline void
SDFM_setFilterOverSamplingRatio(uint32_t base, SDFM_FilterNumber filterNumber,
                                uint16_t overSamplingRatio)
{

    #if (SDFM_VERSION == 200)
    uint32_t address;

    ASSERT(SDFM_isBaseValid(base));
    ASSERT(overSamplingRatio < 256U);

    address = base + SDFM_O_SDDFPARM1 + ((uint32_t)filterNumber * 32U);

    /* Write to DOSR bits */
    HWREGH(address) = (HWREGH(address) & (~SDFM_SDDFPARM1_DOSR_M)) |
                      overSamplingRatio;
    #elif (SDFM_VERSION == 300)
        ASSERT(SDFM_isBaseValid(base));
        ASSERT(filterNumber <= SDFM_FILTER_5);
        uint32_t address;
        ASSERT(overSamplingRatio < 256U);

        if(filterNumber != SDFM_FILTER_5)
        {
            address = base + SDFM_O_SDDFPARM1 + ((uint32_t)filterNumber * 32U);

            /* Write to DOSR bits */
            HWREGH(address) = (HWREGH(address) & (~SDFM_SDDFPARM1_DOSR_M)) |
                            overSamplingRatio;
        }else
        {
            
            address = base + SDFM_O_SDDFPARM5;

            /* Write to DOSR bits */
            HWREGH(address) = (HWREGH(address) & (~SDFM_SDDFPARM5_DOSR_M)) |
                            overSamplingRatio;
        }

    #endif
}

/**
 * @brief Set modulator clock mode.
 *
 * @details
 * This function sets the modulator clock mode specified by clockMode
 * for the filter specified by filterNumber.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 * @param[in] clockMode     The modulator clock mode.
 *
 * @note
 * This function also enables the data and clock synchronizers for
 * the specified filter.
 */
static inline void
SDFM_setupModulatorClock(uint32_t base, SDFM_FilterNumber filterNumber,
                         SDFM_ModulatorClockMode clockMode)
{

    #if (SDFM_VERSION == 200)
        uint32_t address;

        ASSERT(SDFM_isBaseValid(base));

        address = base + SDFM_O_SDCTLPARM1 + ((uint32_t)filterNumber * 32U);

        /* Write to MOD bits */
        HWREGH(address) = (HWREGH(address) & (~SDFM_SDCTLPARM1_MOD_M)) |
                        (uint16_t)clockMode;

        /* Enable data and clock synchronizer */
        HWREGH(base + SDFM_O_SDCTLPARM1 +
            ((uint32_t)filterNumber * SDFM_SDFIL_OFFSET)) |=
            (SDFM_CLOCK_SYNCHRONIZER | SDFM_DATA_SYNCHRONIZER);
    #elif (SDFM_VERSION == 300)
        ASSERT(SDFM_isBaseValid(base));
        ASSERT(filterNumber <= SDFM_FILTER_5);
        uint32_t address;

        ASSERT(SDFM_isBaseValid(base));

        if(filterNumber != SDFM_FILTER_5)
        {
            address = base + SDFM_O_SDCTLPARM1 + ((uint32_t)filterNumber * 32U);

            /* Write to MOD bits */
            HWREGH(address) = (HWREGH(address) & (~SDFM_SDCTLPARM1_MOD_M)) |
                            (uint16_t)clockMode;

            /* Enable data and clock synchronizer */
            HWREGH(base + SDFM_O_SDCTLPARM1 +
                ((uint32_t)filterNumber * SDFM_SDFIL_OFFSET)) |=
                (SDFM_CLOCK_SYNCHRONIZER | SDFM_DATA_SYNCHRONIZER);
        }else
        {
            address = base + SDFM_O_SDCTLPARM5;

            /* Write to MOD bits */
            HWREGH(address) = (HWREGH(address) & (~SDFM_SDCTLPARM5_MOD_M)) |
                            (uint16_t)clockMode;

            /* Enable data and clock synchronizer */
            HWREGH(base + SDFM_O_SDCTLPARM5) |= (SDFM_CLOCK_SYNCHRONIZER | SDFM_DATA_SYNCHRONIZER);
        }

    #endif
}

/**
 * @brief Set the output data format
 *
 * @details
 * This function sets the output data format for the filter specified by
 * filterNumber.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 * @param[in] dataFormat    The output data format.
 */
static inline void
SDFM_setOutputDataFormat(uint32_t base, SDFM_FilterNumber filterNumber,
                         SDFM_OutputDataFormat dataFormat)
{
    #if (SDFM_VERSION == 200)
        uint32_t address;

        ASSERT(SDFM_isBaseValid(base));

        address = base + SDFM_O_SDDPARM1 + ((uint32_t)filterNumber * 32U);

        /* Write to DR bit */
        HWREGH(address) = (HWREGH(address) & (~SDFM_SDDPARM1_DR)) |
                        ((uint16_t)dataFormat << SDFM_SDDPARM1_DR_S);
    #elif (SDFM_VERSION == 300)
        ASSERT(SDFM_isBaseValid(base));
        ASSERT(filterNumber <= SDFM_FILTER_5);
        uint32_t address;

        if(filterNumber != SDFM_FILTER_5)
        {
            address = base + SDFM_O_SDDPARM1 + ((uint32_t)filterNumber * 32U);

            /* Write to DR bit */
            HWREGH(address) = (HWREGH(address) & (~SDFM_SDDPARM1_DR)) |
                            ((uint16_t)dataFormat << 10U);
        }
        else
        {
            address = base + SDFM_O_SDDPARM5;

            /* Write to DR bit */
            HWREGH(address) = (HWREGH(address) & (~SDFM_SDDPARM5_DR)) |
                            ((uint16_t)dataFormat << 10U);
        }

    #endif
}

/**
 * @brief Set data shift value.
 *
 * @details
 * This function sets the shift value for the 16 bit 2's complement data
 * format. The valid maximum value for shiftValue is 31.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 * @param[in] shiftValue    The data shift value.
 *
 * @note
 *   Use this function with 16 bit 2's complement data format only.
 */
static inline void
SDFM_setDataShiftValue(uint32_t base, SDFM_FilterNumber filterNumber,
                       uint16_t shiftValue)
{
    #if (SDFM_VERSION == 200)
        uint32_t address;

        ASSERT(SDFM_isBaseValid(base));
        ASSERT(shiftValue < 32U);

        address = base + SDFM_O_SDDPARM1 + ((uint32_t)filterNumber * 32U);

        /* Write to SH bit */
        HWREGH(address) = (HWREGH(address) & (~SDFM_SDDPARM1_SH_M)) |
                        (shiftValue << SDFM_SDDPARM1_SH_S);
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        uint32_t address;

        ASSERT(SDFM_isBaseValid(base));
        ASSERT(shiftValue < 32U);

        if(filterNumber != SDFM_FILTER_5)
        {

            address = base + SDFM_O_SDDPARM1 + ((uint32_t)filterNumber * 32U);

            /* Write to SH bit */
            HWREGH(address) = (HWREGH(address) & (~SDFM_SDDPARM1_SH_M)) |
                            (shiftValue << SDFM_SDDPARM1_SH_S);
        }
        else
        {


            address = base + SDFM_O_SDDPARM5;

            /* Write to SH bit */
            EALLOW;
            HWREGH(address) = (HWREGH(address) & (~SDFM_SDDPARM5_SH_M)) |
                            (shiftValue << SDFM_SDDPARM5_SH_S);
            EDIS;
        }

    #endif
}

/**
 * @brief Set Filter output high-level threshold.
 *
 * @details
 * This function sets the unsigned high-level threshold value for the
 * Comparator filter output. If the output value of the filter exceeds
 * highThreshold and interrupt generation is enabled, an interrupt will be
 * issued. The param highThreshold takes both high threshold 1 & 2 values.
 * The upper 16-bits represent the high threshold 2 value while lower 16-bits
 * represent the threshold 1 values.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 */
static inline void
SDFM_setCompFilterHighThreshold(uint32_t base, SDFM_FilterNumber filterNumber,
                                uint32_t highThreshold)
{
    #if (SDFM_VERSION == 200)
        uint32_t address;

        ASSERT(SDFM_isBaseValid(base));
        ASSERT((uint16_t)highThreshold <= SDFM_SDFLT1CMPH1_HLT_M);
        ASSERT((uint16_t)(highThreshold >> 16U) <= SDFM_SDFLT1CMPH2_HLT2_M);

        address = base + SDFM_O_SDFLT1CMPH1 +
                ((uint32_t)filterNumber * SDFM_SDFIL_OFFSET);

        /* Write to HLT bit */
        HWREGH(address) = (HWREGH(address) & ~SDFM_SDFLT1CMPH1_HLT_M) |
                        (uint16_t)highThreshold;
        HWREGH(address + SDFM_SDFLT1CMPHx_OFFSET) =
                                    (HWREGH(address + SDFM_SDFLT1CMPHx_OFFSET) &
                                    ~SDFM_SDFLT1CMPH2_HLT2_M) |
                                    (uint16_t)(highThreshold >> 16U);
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        uint32_t address;

        ASSERT(SDFM_isBaseValid(base));
        ASSERT((uint16_t)highThreshold <= SDFM_SDFLT1CMPH1_HLT_M);
        ASSERT((uint16_t)(highThreshold >> 16U) <= SDFM_SDFLT1CMPH2_HLT2_M);

        if(filterNumber != SDFM_FILTER_5)
        {
            address = base + SDFM_O_SDFLT1CMPH1 +
                    ((uint32_t)filterNumber * SDFM_SDFIL_OFFSET);

            /* Write to HLT bit */
            HWREGH(address) = (HWREGH(address) & ~SDFM_SDFLT1CMPH1_HLT_M) |
                            (uint16_t)highThreshold;
            HWREGH(address + SDFM_SDFLT1CMPHx_OFFSET) =
                                        (HWREGH(address + SDFM_SDFLT1CMPHx_OFFSET) &
                                        ~SDFM_SDFLT1CMPH2_HLT2_M) |
                                        (uint16_t)(highThreshold >> 16U);
        }else
        {

            address = base + SDFM_O_SDFLT5CMPH1;

            /* Write to HLT bit */
            HWREGH(address) = (HWREGH(address) & ~SDFM_SDFLT5CMPH1_HLT_M) |
                            (uint16_t)highThreshold;
            HWREGH(address + SDFM_O_SDFLT5CMPH2) =
                                        (HWREGH(address + SDFM_O_SDFLT5CMPH2) &
                                        ~SDFM_SDFLT5CMPH2_HLT2_M) |
                                        (uint16_t)(highThreshold >> 16U);
        }

    #endif
}

/**
 * @brief Set Filter output low-level threshold.
 *
 * @details
 * This function sets the unsigned low-level threshold value 1 or 2 for the
 * Comparator filter output. If the output value of the filter gets below
 * lowThreshold and interrupt generation is enabled, an interrupt will be
 * issued. The param lowThreshold takes both low threshold 1 & 2 values.
 * The upper 16-bits represent the low threshold 2 value while lower 16-bits
 * represent the threshold 1 values.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 */
static inline void
SDFM_setCompFilterLowThreshold(uint32_t base, SDFM_FilterNumber filterNumber,
                               uint32_t lowThreshold)
{
    #if (SDFM_VERSION == 200)
        uint32_t address;

        ASSERT(SDFM_isBaseValid(base));
        ASSERT((uint16_t)lowThreshold <= SDFM_SDFLT1CMPL1_LLT_M);
        ASSERT((uint16_t)(lowThreshold >> 16U) <= SDFM_SDFLT1CMPL2_LLT2_M);

        address = base + SDFM_O_SDFLT1CMPL1 +
                ((uint32_t)filterNumber * SDFM_SDFIL_OFFSET);

        /* Write to LLT bit. */
        HWREGH(address) = (HWREGH(address) & ~SDFM_SDFLT1CMPL1_LLT_M) |
                        (uint16_t)lowThreshold;
        HWREGH(address + SDFM_SDFLT1CMPLx_OFFSET) =
                                    (HWREGH(address + SDFM_SDFLT1CMPLx_OFFSET) &
                                        ~SDFM_SDFLT1CMPL2_LLT2_M) |
                                    (uint16_t)(lowThreshold >> 16U);
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        uint32_t address;

        ASSERT(SDFM_isBaseValid(base));
        ASSERT((uint16_t)lowThreshold <= SDFM_SDFLT1CMPL1_LLT_M);
        ASSERT((uint16_t)(lowThreshold >> 16U) <= SDFM_SDFLT1CMPL2_LLT2_M);

        if(filterNumber != SDFM_FILTER_5)
        {
            address = base + SDFM_O_SDFLT1CMPL1 +
                    ((uint32_t)filterNumber * SDFM_SDFIL_OFFSET);

            /* Write to LLT bit. */
            HWREGH(address) = (HWREGH(address) & ~SDFM_SDFLT1CMPL1_LLT_M) |
                            (uint16_t)lowThreshold;
            HWREGH(address + SDFM_SDFLT1CMPLx_OFFSET) =
                                        (HWREGH(address + SDFM_SDFLT1CMPLx_OFFSET) &
                                            ~SDFM_SDFLT1CMPL2_LLT2_M) |
                                        (uint16_t)(lowThreshold >> 16U);
        }
        else
        {
            address = base + SDFM_O_SDFLT5CMPL1;

            /* Write to LLT bit. */
            HWREGH(address) = (HWREGH(address) & ~SDFM_SDFLT5CMPL1_LLT_M) |
                            (uint16_t)lowThreshold;
            HWREGH(address + SDFM_O_SDFLT5CMPL2) =
                                        (HWREGH(address + SDFM_O_SDFLT5CMPL2) &
                                            ~SDFM_SDFLT5CMPL2_LLT2_M) |
                                        (uint16_t)(lowThreshold >> 16U);
        }

    #endif
}

/**
 * @brief Set Filter output zero-cross threshold.
 *
 * @details
 * This function sets the unsigned zero-cross threshold value for the
 * Comparator filter output.
 *
 * @param[in] base               Specifies the SDFM module base address.
 * @param[in] filterNumber       The filter number.
 * @param[in] zeroCrossThreshold The zero-cross threshold.
 */
static inline void
SDFM_setCompFilterZeroCrossThreshold(uint32_t base,
                                     SDFM_FilterNumber filterNumber,
                                     uint16_t zeroCrossThreshold)
{

    #if (SDFM_VERSION == 200)
        uint32_t address;

        ASSERT(SDFM_isBaseValid(base));
        ASSERT(zeroCrossThreshold < 0x7FFFU);

        address = base + SDFM_O_SDFLT1CMPHZ + ((uint32_t)filterNumber * 32U);

        /* Write to ZCT bit */
        HWREGH(address) = (HWREGH(address) & ~SDFM_SDFLT1CMPHZ_HLTZ_M) |
                        zeroCrossThreshold;

    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        uint32_t address;

        ASSERT(SDFM_isBaseValid(base));
        ASSERT(zeroCrossThreshold < 0x7FFFU);

        if(filterNumber != SDFM_FILTER_5)
        {

            address = base + SDFM_O_SDFLT1CMPHZ + ((uint32_t)filterNumber * 32U);

            /* Write to ZCT bit */
            HWREGH(address) = (HWREGH(address) & ~SDFM_SDFLT1CMPHZ_HLTZ_M) |
                            zeroCrossThreshold;

        }
        else
        {

            address = base + SDFM_O_SDFLT5CMPHZ;

            /* Write to ZCT bit */
            HWREGH(address) = (HWREGH(address) & ~SDFM_SDFLT5CMPHZ_HLTZ_M) |
                            zeroCrossThreshold;

        }

    #endif
}

/**
 * @brief Enable zero-cross Edge detect mode.
 *
 * @details
 * This function enables Zero Cross Edge detection.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 */
static inline void
SDFM_enableZeroCrossEdgeDetect(uint32_t base, SDFM_FilterNumber filterNumber)
{

    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        /* Set ZCEN bit */
        HWREGH(base + SDFM_O_SDCPARM1 + ((uint32_t)filterNumber * 32U)) |=
            SDFM_SDCPARM1_HZEN;

    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        ASSERT(SDFM_isBaseValid(base));

        if(filterNumber != SDFM_FILTER_5)
        {
            /* Set ZCEN bit */
            HWREGH(base + SDFM_O_SDCPARM1 + ((uint32_t)filterNumber * 32U)) |=
                SDFM_SDCPARM1_HZEN;
        }else
        {
            /* Set ZCEN bit */
            HWREGH(base + SDFM_O_SDCPARM5) |= SDFM_SDCPARM5_HZEN;
        }

    #endif
}

/**
 * @brief Disable zero-cross Edge detect mode.
 *
 * @details
 * This function disables Zero Cross Edge detection.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 */
static inline void
SDFM_disableZeroCrossEdgeDetect(uint32_t base, SDFM_FilterNumber filterNumber)
{

    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        /* Clear ZCEN bit */
        HWREGH(base + SDFM_O_SDCPARM1 + ((uint32_t)filterNumber * 32U)) &=
            ~SDFM_SDCPARM1_HZEN;

    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
 
        if(filterNumber != SDFM_FILTER_5)
        {
            ASSERT(SDFM_isBaseValid(base));

            /* Clear ZCEN bit */
            HWREGH(base + SDFM_O_SDCPARM1 + ((uint32_t)filterNumber * 32U)) &=
                ~SDFM_SDCPARM1_HZEN;
        }
        else
        {
            ASSERT(SDFM_isBaseValid(base));

            /* Clear ZCEN bit */
            HWREGH(base + SDFM_O_SDCPARM5) &=~SDFM_SDCPARM5_HZEN;
        }

    #endif
}

/**
 * @brief Enable SDFM interrupts.
 *
 * @details
 * This function enables the low threshold , high threshold or modulator
 * failure interrupt as determined by intFlags for the filter specified
 * by filterNumber.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 * @param[in] intFlags      The interrupt source.
 *
 * @remarks
 *   Valid values for intFlags are:
 *     SDFM_MODULATOR_FAILURE_INTERRUPT
 *     SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT
 *     SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT
 *     SDFM_FIFO_INTERRUPT
 *     SDFM_FIFO_OVERFLOW_INTERRUPT
 *     SDFM_DATA_FILTER_ACKNOWLEDGE_INTERRUPT
 */
static inline void
SDFM_enableInterrupt(uint32_t base, SDFM_FilterNumber filterNumber,
                     uint16_t intFlags)
{
    #if (SDFM_VERSION == 200)
        uint16_t offset;

        ASSERT(SDFM_isBaseValid(base));

        offset = (uint16_t)filterNumber * 32U;

        /* Low, high threshold, Modulator failure */
        if((intFlags & (SDFM_MODULATOR_FAILURE_INTERRUPT |
                        SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT |
                        SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT)) != 0U)
        {
            /* Set IEL or IEH or MFIE bit of SDFM_O_SDCPARMx */
            HWREGH(base + SDFM_O_SDCPARM1 + offset) |=
                    (intFlags & (SDFM_MODULATOR_FAILURE_INTERRUPT |
                                SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT |
                                SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT));
        }

        /* Data filter acknowledge interrupt */
        if((intFlags & SDFM_DATA_FILTER_ACKNOWLEDGE_INTERRUPT) != 0U)
        {
            HWREGH(base + SDFM_O_SDDFPARM1 + offset) |= SDFM_SDDFPARM1_AE;
        }

        /* FIFO , FIFO overflow interrupt */
        if((intFlags & (SDFM_FIFO_INTERRUPT | SDFM_FIFO_OVERFLOW_INTERRUPT)) != 0U)
        {
            /* Set OVFIEN or FFIEN bits of SDFM_O_SDFIFOCTLx */
            HWREGH(base + SDFM_O_SDFIFOCTL1 + offset) |=
                    (intFlags & (SDFM_FIFO_INTERRUPT |
                                SDFM_FIFO_OVERFLOW_INTERRUPT));
        }
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        uint16_t offset;

        ASSERT(SDFM_isBaseValid(base));

        offset = (uint16_t)filterNumber * 32U;

        if(filterNumber != SDFM_FILTER_5)
        {
            /* Low, high threshold, Modulator failure */
            if((intFlags & (SDFM_MODULATOR_FAILURE_INTERRUPT |
                            SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT |
                            SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT)) != 0U)
            {
                /* Set IEL or IEH or MFIE bit of SDFM_O_SDCPARMx */
                HWREGH(base + SDFM_O_SDCPARM1 + offset) |=
                        (intFlags & (SDFM_MODULATOR_FAILURE_INTERRUPT |
                                    SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT |
                                    SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT));
            }

            /* Data filter acknowledge interrupt */
            if((intFlags & SDFM_DATA_FILTER_ACKNOWLEDGE_INTERRUPT) != 0U)
            {
                HWREGH(base + SDFM_O_SDDFPARM1 + offset) |= SDFM_SDDFPARM1_AE;
            }

            /* FIFO , FIFO overflow interrupt */
            if((intFlags & (SDFM_FIFO_INTERRUPT | SDFM_FIFO_OVERFLOW_INTERRUPT)) != 0U)
            {
                /* Set OVFIEN or FFIEN bits of SDFM_O_SDFIFOCTLx */
                HWREGH(base + SDFM_O_SDFIFOCTL1 + offset) |=
                        (intFlags & (SDFM_FIFO_INTERRUPT |
                                    SDFM_FIFO_OVERFLOW_INTERRUPT));
            }
        }
        else
        {
            /* Low, high threshold, Modulator failure */
            if((intFlags & (SDFM_MODULATOR_FAILURE_INTERRUPT |
                            SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT |
                            SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT)) != 0U)
            {
                /* Set IEL or IEH or MFIE bit of SDFM_O_SDCPARMx */
                HWREGH(base + SDFM_O_SDCPARM5) |=
                        (intFlags & (SDFM_MODULATOR_FAILURE_INTERRUPT |
                                    SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT |
                                    SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT));
            }

            /* Data filter acknowledge interrupt */
            if((intFlags & SDFM_DATA_FILTER_ACKNOWLEDGE_INTERRUPT) != 0U)
            {
                HWREGH(base + SDFM_O_SDDFPARM5) |= SDFM_SDDFPARM5_AE;
            }

            /* FIFO , FIFO overflow interrupt */
            if((intFlags & (SDFM_FIFO_INTERRUPT | SDFM_FIFO_OVERFLOW_INTERRUPT)) != 0U)
            {
                /* Set OVFIEN or FFIEN bits of SDFM_O_SDFIFOCTLx */
                HWREGH(base + SDFM_O_SDFIFOCTL5) |=
                        (intFlags & (SDFM_FIFO_INTERRUPT |
                                    SDFM_FIFO_OVERFLOW_INTERRUPT));
            }
        }

    #endif
}

/**
 * @brief Disable SDFM interrupts.
 *
 * @details
 * This function disables the low threshold , high threshold or modulator
 * failure interrupt as determined by intFlags for the filter specified by
 * filterNumber.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 * @param[in] intFlags      The interrupt source.
 *
 * @remarks
 *   Valid values for intFlags are:
 *     SDFM_MODULATOR_FAILURE_INTERRUPT
 *     SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT
 *     SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT
 *     SDFM_FIFO_INTERRUPT
 *     SDFM_FIFO_OVERFLOW_INTERRUPT
 *     SDFM_DATA_FILTER_ACKNOWLEDGE_INTERRUPT
 */
static inline void
SDFM_disableInterrupt(uint32_t base, SDFM_FilterNumber filterNumber,
                      uint16_t intFlags)
{
    #if (SDFM_VERSION == 200)
        uint16_t offset;

        ASSERT(SDFM_isBaseValid(base));

        offset = (uint16_t)filterNumber * 32U;

        /* Low, high threshold, modulator failure interrupts */
        if((intFlags & (SDFM_MODULATOR_FAILURE_INTERRUPT |
                        SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT |
                        SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT)) != 0U)
        {
            /* Set IEL or IEH or MFIE bit of SDFM_O_SDCPARMx */
            HWREGH(base + SDFM_O_SDCPARM1 + offset) &=
                ~(intFlags & (SDFM_MODULATOR_FAILURE_INTERRUPT |
                            SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT |
                            SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT));
        }

        /* Data filter acknowledge interrupt */
        if((intFlags & SDFM_DATA_FILTER_ACKNOWLEDGE_INTERRUPT) != 0U)
        {
            HWREGH(base + SDFM_O_SDDFPARM1 + offset) &= ~SDFM_SDDFPARM1_AE;
        }

        /* FIFO , FIFO overflow interrupt */
        if((intFlags & (SDFM_FIFO_INTERRUPT | SDFM_FIFO_OVERFLOW_INTERRUPT)) != 0U)
        {
            /* Set OVFIEN or FFIEN bits of SDFM_O_SDFIFOCTLx */
            HWREGH(base + SDFM_O_SDFIFOCTL1 + offset) &=
                ~(intFlags & (SDFM_FIFO_INTERRUPT | SDFM_FIFO_OVERFLOW_INTERRUPT));
        }
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        uint16_t offset;

        ASSERT(SDFM_isBaseValid(base));
        if(filterNumber != SDFM_FILTER_5)
        {

            offset = (uint16_t)filterNumber * 32U;

            /* Low, high threshold, modulator failure interrupts */
            if((intFlags & (SDFM_MODULATOR_FAILURE_INTERRUPT |
                            SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT |
                            SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT)) != 0U)
            {
                /* Set IEL or IEH or MFIE bit of SDFM_O_SDCPARMx */
                HWREGH(base + SDFM_O_SDCPARM1 + offset) &=
                    ~(intFlags & (SDFM_MODULATOR_FAILURE_INTERRUPT |
                                SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT |
                                SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT));
            }

            /* Data filter acknowledge interrupt */
            if((intFlags & SDFM_DATA_FILTER_ACKNOWLEDGE_INTERRUPT) != 0U)
            {
                HWREGH(base + SDFM_O_SDDFPARM1 + offset) &= ~SDFM_SDDFPARM1_AE;
            }

            /* FIFO , FIFO overflow interrupt */
            if((intFlags & (SDFM_FIFO_INTERRUPT | SDFM_FIFO_OVERFLOW_INTERRUPT)) != 0U)
            {
                /* Set OVFIEN or FFIEN bits of SDFM_O_SDFIFOCTLx */
                HWREGH(base + SDFM_O_SDFIFOCTL1 + offset) &=
                    ~(intFlags & (SDFM_FIFO_INTERRUPT | SDFM_FIFO_OVERFLOW_INTERRUPT));
            }
        }
        else
        {

            /* Low, high threshold, modulator failure interrupts */
            if((intFlags & (SDFM_MODULATOR_FAILURE_INTERRUPT |
                            SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT |
                            SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT)) != 0U)
            {
                // Set IEL or IEH or MFIE bit of SDFM_O_SDCPARMx
                HWREGH(base + SDFM_O_SDCPARM5) &=
                    ~(intFlags & (SDFM_MODULATOR_FAILURE_INTERRUPT |
                                SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT |
                                SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT));
            }

            /* Data filter acknowledge interrupt */
            if((intFlags & SDFM_DATA_FILTER_ACKNOWLEDGE_INTERRUPT) != 0U)
            {
                HWREGH(base + SDFM_O_SDDFPARM5) &= ~SDFM_SDDFPARM5_AE;
            }

            /* FIFO , FIFO overflow interrupt */
            if((intFlags & (SDFM_FIFO_INTERRUPT | SDFM_FIFO_OVERFLOW_INTERRUPT)) != 0U)
            {
                /* Set OVFIEN or FFIEN bits of SDFM_O_SDFIFOCTLx */
                HWREGH(base + SDFM_O_SDFIFOCTL5) &=
                    ~(intFlags & (SDFM_FIFO_INTERRUPT | SDFM_FIFO_OVERFLOW_INTERRUPT));
            }
        }

    #endif
}

/**
 * @brief Set the comparator filter type.
 *
 * @details
 * This function sets the Comparator filter type or structure to be used as
 * specified by filterType for the selected filter number as specified by
 * filterNumber.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 * @param[in] filterType    The comparator filter type or structure.
 */
static inline void
SDFM_setComparatorFilterType(uint32_t base, SDFM_FilterNumber filterNumber,
                             SDFM_FilterType filterType)
{
    #if (SDFM_VERSION == 200)
        uint32_t address;

        ASSERT(SDFM_isBaseValid(base));

        address = base + SDFM_O_SDCPARM1 + ((uint32_t)filterNumber * 32U);

        /* Write to CS1_CS0 bits */
        HWREGH(address) = (HWREGH(address) & (~SDFM_SDCPARM1_CS1_CS0_M)) |
                        ((uint16_t)filterType << 3U);
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        uint32_t address;

        ASSERT(SDFM_isBaseValid(base));

        if(filterNumber != SDFM_FILTER_5)
        {
            address = base + SDFM_O_SDCPARM1 + ((uint32_t)filterNumber * 32U);

            /* Write to CS1_CS0 bits */
            HWREGH(address) = (HWREGH(address) & (~SDFM_SDCPARM1_CS1_CS0_M)) |
                            ((uint16_t)filterType << 3U);
        }else
        {
            address = base + SDFM_O_SDCPARM5;

            /* Write to CS1_CS0 bits */
            HWREGH(address) = (HWREGH(address) & (~SDFM_SDCPARM5_CS1_CS0_M)) |
                            ((uint16_t)filterType << 3U);
        }

    #endif
}

/**
 * @brief Set Comparator filter over sampling ratio.
 *
 * @details
 * This function sets the comparator filter oversampling ratio for the filter
 * specified by the filterNumber.Valid values for the variable overSamplingRatio
 * are 0 to 31 inclusive.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 *
 * @remarks
 *   The actual oversampling ratio will be this value plus one.
 */
static inline void
SDFM_setCompFilterOverSamplingRatio(uint32_t base,
                                    SDFM_FilterNumber filterNumber,
                                    uint16_t overSamplingRatio)
{
    #if (SDFM_VERSION == 200)
        uint32_t address;

        ASSERT(SDFM_isBaseValid(base));
        ASSERT(overSamplingRatio < 32U);

        address = base + SDFM_O_SDCPARM1 + ((uint32_t)filterNumber * 32U);

        /* Write to COSR bits */
        HWREGH(address) = (HWREGH(address) & (~SDFM_SDCPARM1_COSR_M)) |
                        overSamplingRatio;
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        uint32_t address;

        ASSERT(SDFM_isBaseValid(base));
        ASSERT(overSamplingRatio < 32U);

        if(filterNumber != SDFM_FILTER_5)
        {
            address = base + SDFM_O_SDCPARM1 + ((uint32_t)filterNumber * 32U);

            /* Write to COSR bits */
            HWREGH(address) = (HWREGH(address) & (~SDFM_SDCPARM1_COSR_M)) |
                            overSamplingRatio;
        }else
        {
            address = base + SDFM_O_SDCPARM5;

            /* Write to COSR bits */
            HWREGH(address) = (HWREGH(address) & (~SDFM_SDCPARM5_COSR_M)) |
                            overSamplingRatio;
        }

    #endif
}

/**
 * @brief Get the filter data output.
 *
 * @details
 * This function returns the latest data filter output. Depending on the
 * filter data output format selected, the valid value will be the lower 16
 * bits or the whole 32 bits of the returned value.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 *
 * @return Returns the latest data filter output.
 */
static inline uint32_t
SDFM_getFilterData(uint32_t base, SDFM_FilterNumber filterNumber)
{

    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        /* Read SDDATA bits */
        return(HWREG(base + SDFM_O_SDDATA1 + ((uint32_t)filterNumber * 32U)));
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        ASSERT(SDFM_isBaseValid(base));
        if(filterNumber != SDFM_FILTER_5)
        {
            /* Read SDDATA bits */
            return(HWREG(base + SDFM_O_SDDATA1 + ((uint32_t)filterNumber * 32U)));
        }else
        {

            /* Read SDDATA bits */
            return(HWREG(base + SDFM_O_SDDATA5 ));
        }

    #endif
}

/**
 * @brief Get the Comparator threshold status.
 *
 * @details
 * This function returns the Comparator output threshold status for the given
 * filterNumber.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 *
 * @return
 * SDFM_OUTPUT_WITHIN_THRESHOLD   if the output is within the specified threshold.
 * SDFM_OUTPUT_ABOVE_THRESHOLD    if the output is above the high threshold.
 * SDFM_OUTPUT_BELOW_THRESHOLD    if the output is below the low threshold.
 *
 */
static inline SDFM_OutputThresholdStatus
SDFM_getThresholdStatus(uint32_t base, SDFM_FilterNumber filterNumber)
{
    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        /* Read SDIFLG high/low threshold bits */
        return((SDFM_OutputThresholdStatus)((HWREG(base + SDFM_O_SDIFLG) >>
                                            (2U * (uint16_t)filterNumber)) & 0x3U));
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        ASSERT(SDFM_isBaseValid(base));
        if(filterNumber != SDFM_FILTER_5)
        {

            /* Read SDIFLG high/low threshold bits */
            return((SDFM_OutputThresholdStatus)((HWREG(base + SDFM_O_SDIFLG) >>
                                                (2U * (uint16_t)filterNumber)) & 0x3U));
        }else
        {
            /* Read SDIFLG high/low threshold bits */
            return((SDFM_OutputThresholdStatus)((HWREG(base + SDFM_O_SDIFLG) >>(24U)) & 0x3U));
        }

    #endif
}

/**
 * @brief Get the Modulator status.
 *
 * @details
 * This function returns the Modulator status.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 *
 * @return
 * true     if the Modulator is operating normally
 * false    if the Modulator has failed
 */
static inline bool
SDFM_getModulatorStatus(uint32_t base, SDFM_FilterNumber filterNumber)
{
    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        /* Read SDIFLG MF1, MF2, MF3 OR MF4 bits */
        return(((HWREG(base + SDFM_O_SDIFLG) >> ((uint16_t)filterNumber + 8U)) &
                0x1U) != 0x1U);
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        ASSERT(SDFM_isBaseValid(base));

        if(filterNumber != SDFM_FILTER_5)
        {
            /* Read SDIFLG MF1, MF2, MF3 OR MF4 bits */
            return(((HWREG(base + SDFM_O_SDIFLG) >> ((uint16_t)filterNumber + 8U)) &
                    0x1U) != 0x1U);
        }else
        {
            /* Read SDIFLG MF5 bits */
            return(((HWREG(base + SDFM_O_SDIFLG) >> 26)) & 0x1U) != 0x1U;
        }

    #endif
}

/**
 * @brief Check if new Filter data is available.
 *
 * @details
 * This function returns new filter data status.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 *
 * @return
 * true    if new filter data is available
 * false   if no new filter data is available
 */
static inline bool
SDFM_getNewFilterDataStatus(uint32_t base, SDFM_FilterNumber filterNumber)
{
    #if (SDFM_VERSION == 200)
    ASSERT(SDFM_isBaseValid(base));

    /* Read SDIFLG AF1, AF2, AF3 OR AF4 bits */
    return(((HWREG(base + SDFM_O_SDIFLG) >> ((uint16_t)filterNumber + 12U)) &
            0x1U) == 0x1U);
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        ASSERT(SDFM_isBaseValid(base));

        if(filterNumber != SDFM_FILTER_5)
        {
            /* Read SDIFLG AF1, AF2, AF3 OR AF4 bits */
            return(((HWREG(base + SDFM_O_SDIFLG) >> ((uint16_t)filterNumber + 12U)) &
                    0x1U) == 0x1U);
        }else
        {
            /* Read SDIFLG AF5 bits */
            return(((HWREG(base + SDFM_O_SDIFLG) >> (27U)) & 0x1U) == 0x1U);
        }

    #endif
}

/**
 * @brief Check if FIFO buffer is overflowed.
 *
 * @details
 * This function returns the status of the FIFO buffer overflow for the given
 * filter value.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 *
 * @return
 * true    if FIFO buffer is overflowed
 * false   if FIFO buffer is not overflowed
 */
static inline bool
SDFM_getFIFOOverflowStatus(uint32_t base, SDFM_FilterNumber filterNumber)
{
    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        /* Read SDIFLG SDFFOVF1, SDFFOVF2, SDFFOVF3 OR SDFFOVF4 bits */
        return(((HWREG(base + SDFM_O_SDIFLG) >> ((uint16_t)filterNumber + 16U)) &
                0x1U) == 0x1U);
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
 
        if(filterNumber != SDFM_FILTER_5)
        {
            ASSERT(SDFM_isBaseValid(base));

            /* Read SDIFLG SDFFOVF1, SDFFOVF2, SDFFOVF3 OR SDFFOVF4 bits */
            return(((HWREG(base + SDFM_O_SDIFLG) >> ((uint16_t)filterNumber + 16U)) &
                    0x1U) == 0x1U);
        }else
        {
            ASSERT(SDFM_isBaseValid(base));

            /* Read SDIFLG  SDFFOVF5 bits */
            return(((HWREG(base + SDFM_O_SDIFLG) >> (28U)) & 0x1U) == 0x1U);  
        }

    #endif
}

/**
 * @brief Check FIFO buffer interrupt status.
 *
 * @details
 * This function returns the status of the FIFO buffer interrupt for the given
 * filter.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 *
 * @return
 * true    if FIFO buffer interrupt has occurred.
 * false   if FIFO buffer interrupt has not occurred.
 */
static inline bool
SDFM_getFIFOISRStatus(uint32_t base, SDFM_FilterNumber filterNumber)
{
    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        /* Read SDIFLG SDFFINT1, SDFFINT2, SDFFINT3 OR SDFFINT4 bits */
        return(((HWREG(base + SDFM_O_SDIFLG) >> ((uint16_t)filterNumber + 20U)) &
                0x1U) == 0x1U);
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        ASSERT(SDFM_isBaseValid(base));

        if(filterNumber != SDFM_FILTER_5)
        {
            /* Read SDIFLG SDFFINT1, SDFFINT2, SDFFINT3 OR SDFFINT4 bits */
            return(((HWREG(base + SDFM_O_SDIFLG) >> ((uint16_t)filterNumber + 20U)) &
                    0x1U) == 0x1U);
        }else
        {
            /* Read SDIFLG SDFFINT5 bits */
            return(((HWREG(base + SDFM_O_SDIFLG) >> (29U)) & 0x1U) == 0x1U);
        }

    #endif
}

/**
 * @brief Get pending interrupt.
 *
 * @details
 * This function returns any pending interrupt status.
 *
 * @param[in] base          Specifies the SDFM module base address.
 *
 * @return
 * true    if there is a pending interrupt.
 * false   if no interrupt is pending.
 */
static inline bool
SDFM_getIsrStatus(uint32_t base)
{
    ASSERT(SDFM_isBaseValid(base));

    /* Read SDIFLG MIF */
    return((HWREG(base + SDFM_O_SDIFLG) >> 31U) == 0x1U);
}

/**
 * @brief Clear pending flags.
 *
 * @details
 * This function clears the specified pending interrupt flag.
 *
 * @param[in] base  Specifies the SDFM module base address.
 * @param[in] flag  The SDFM status.
 *
 * @remarks
 * - SDFM_MAIN_INTERRUPT_FLAG
 * - SDFM_FILTER_1_FIFO_INTERRUPT_FLAG
 * - SDFM_FILTER_2_FIFO_INTERRUPT_FLAG
 * - SDFM_FILTER_3_FIFO_INTERRUPT_FLAG
 * - SDFM_FILTER_4_FIFO_INTERRUPT_FLAG
 * - SDFM_FILTER_1_FIFO_OVERFLOW_FLAG
 * - SDFM_FILTER_2_FIFO_OVERFLOW_FLAG
 * - SDFM_FILTER_3_FIFO_OVERFLOW_FLAG
 * - SDFM_FILTER_4_FIFO_OVERFLOW_FLAG
 * - SDFM_FILTER_1_NEW_DATA_FLAG
 * - SDFM_FILTER_2_NEW_DATA_FLAG
 * - SDFM_FILTER_3_NEW_DATA_FLAG
 * - SDFM_FILTER_4_NEW_DATA_FLAG
 * - SDFM_FILTER_1_MOD_FAILED_FLAG
 * - SDFM_FILTER_2_MOD_FAILED_FLAG
 * - SDFM_FILTER_3_MOD_FAILED_FLAG
 * - SDFM_FILTER_4_MOD_FAILED_FLAG
 * - SDFM_FILTER_1_HIGH_THRESHOLD_FLAG
 * - SDFM_FILTER_1_LOW_THRESHOLD_FLAG
 * - SDFM_FILTER_2_HIGH_THRESHOLD_FLAG
 * - SDFM_FILTER_2_LOW_THRESHOLD_FLAG
 * - SDFM_FILTER_3_HIGH_THRESHOLD_FLAG
 * - SDFM_FILTER_3_LOW_THRESHOLD_FLAG
 * - SDFM_FILTER_4_HIGH_THRESHOLD_FLAG
 * - SDFM_FILTER_4_LOW_THRESHOLD_FLAG
 * or any combination of the above flags.
 * - SDFM_FILTER_5_FIFO_INTERRUPT_FLAG
 * - SDFM_FILTER_5_FIFO_OVERFLOW_FLAG
 * - SDFM_FILTER_5_NEW_DATA_FLAG
 * - SDFM_FILTER_5_MOD_FAILED_FLAG
 * - SDFM_FILTER_5_LOW_THRESHOLD_FLAG
 * - SDFM_FILTER_5_HIGH_THRESHOLD_FLAG
 */
static inline void
SDFM_clearInterruptFlag(uint32_t base, uint32_t flag)
{
    ASSERT(SDFM_isBaseValid(base));
    ASSERT((flag & 0xBFFFFFFFU) == flag);

    /* Write to  SDIFLGCLR register */
    HWREG(base + SDFM_O_SDIFLGCLR) |= flag;
}

/**
 * @brief Enable main interrupt.
 *
 * @details
 * This function enables the main SDFM interrupt.
 *
 * @param[in] base          Specifies the SDFM module base address.
 */
static inline void
SDFM_enableMainInterrupt(uint32_t base)
{
    ASSERT(SDFM_isBaseValid(base));

    /* Set SDCTL MIE bit */
    HWREGH(base + SDFM_O_SDCTL) |= SDFM_SDCTL_MIE;
}

/**
 * @brief Disable main interrupt.
 *
 * @details
 * This function disables the main SDFM interrupt.
 *
 * @param[in] base          Specifies the SDFM module base address.
 */
static inline void
SDFM_disableMainInterrupt(uint32_t base)
{
    ASSERT(SDFM_isBaseValid(base));

    /* Clear SDCTL MIE bit */
    HWREGH(base + SDFM_O_SDCTL) &= ~SDFM_SDCTL_MIE;
}

/**
 * @brief Enable main filter.
 *
 * @details
 * This function enables main filter.
 *
 * @param[in] base  Specifies the SDFM module base address.
 */
static inline void
SDFM_enableMainFilter(uint32_t base)
{
    ASSERT(SDFM_isBaseValid(base));

    /* Set SDMFILEN MFE bit */
    HWREGH(base + SDFM_O_SDMFILEN) |= SDFM_SDMFILEN_MFE;
}

/**
 * @brief Disable main filter.
 *
 * @details
 * This function disables main filter.
 *
 * @param[in] base  Specifies the SDFM module base address.
 */
static inline void
SDFM_disableMainFilter(uint32_t base)
{
    ASSERT(SDFM_isBaseValid(base));

    /* Clear SDMFILEN MFE bit */
    HWREGH(base + SDFM_O_SDMFILEN) &= ~SDFM_SDMFILEN_MFE;
}

/**
 * @brief Return the FIFO data count
 *
 * @details
 * This function returns the FIFO data count.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 *
 * @return
 * Returns the number of data words available in FIFO buffer.
 */
static inline uint16_t
SDFM_getFIFODataCount(uint32_t base, SDFM_FilterNumber filterNumber)
{
    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        /* Read SDFFST */
        return((HWREGH(base + SDFM_O_SDFIFOCTL1 +
            ((uint32_t)filterNumber * 32U)) & SDFM_SDFIFOCTL1_SDFFST_M) >>
            SDFM_SDFIFOCTL1_SDFFST_S);
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
 
        if(filterNumber != SDFM_FILTER_5)
        {
            ASSERT(SDFM_isBaseValid(base));

            /* Read SDFFST */
            return((HWREGH(base + SDFM_O_SDFIFOCTL1 +
                ((uint32_t)filterNumber * 32U)) & SDFM_SDFIFOCTL1_SDFFST_M) >>
                SDFM_SDFIFOCTL1_SDFFST_S);
        }else
        {
            ASSERT(SDFM_isBaseValid(base));

            /* Read SDFFST */
            return((HWREGH(base + SDFM_O_SDFIFOCTL5) & SDFM_SDFIFOCTL5_SDFFST_M) >>
                SDFM_SDFIFOCTL5_SDFFST_S);
        }

    #endif
}

/**
 * @brief Return the Comparator sinc filter data
 *
 * @details
 * This function returns the Comparator sinc filter data output.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 *
 * @return
 * Returns the Comparator sinc filter data output.
 *
 */
static inline uint16_t
SDFM_getComparatorSincData(uint32_t base, SDFM_FilterNumber filterNumber)
{
    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        /* Read SDCDATA */
        return(HWREGH(base + SDFM_O_SDCDATA1 + ((uint32_t)filterNumber * 32U)));
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        ASSERT(SDFM_isBaseValid(base));

        if(filterNumber != SDFM_FILTER_5)
        {
        	/* Read SDCDATA */
            return(HWREGH(base + SDFM_O_SDCDATA1 + ((uint32_t)filterNumber * 32U)));
        }else
        {
        	/* Read SDCDATA */
            return(HWREGH(base + SDFM_O_SDCDATA5));  
        }

    #endif

}

/**
 * @brief Return the FIFO data
 *
 * @details
 * This function returns the latest FIFO data.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 *
 * @remarks
 *   Discard the upper 16 bits if the output data format is 16bits.
 */
static inline uint32_t
SDFM_getFIFOData(uint32_t base, SDFM_FilterNumber filterNumber)
{
    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        /* Read SDDATFIFO */
        return(HWREG(base + SDFM_O_SDDATFIFO1 + ((uint32_t)filterNumber * 32U)));
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        ASSERT(SDFM_isBaseValid(base));

        if(filterNumber != SDFM_FILTER_5)
        {
            /* Read SDDATFIFO */
            return(HWREG(base + SDFM_O_SDDATFIFO1 + ((uint32_t)filterNumber * 32U)));
        }else
        {

            /* Read SDDATFIFO */
            return(HWREG(base + SDFM_O_SDDATFIFO5));
        }

    #endif
}

/**
 * @brief Set the FIFO interrupt level.
 *
 * @details
 * This function sets the FIFO interrupt level.Interrupt is generated when
 * the FIFO buffer word count gets to or exceeds the value of fifoLevel.
 * Maximum value for fifoLevel is 16.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 * @param[in] fifoLevel     The FIFO interrupt level.
 */
static inline void
SDFM_setFIFOInterruptLevel(uint32_t base, SDFM_FilterNumber filterNumber,
                           uint16_t fifoLevel)
{
    #if (SDFM_VERSION == 200)
        uint32_t address;

        ASSERT(SDFM_isBaseValid(base));
        ASSERT(fifoLevel <= 16U);

        address = base + SDFM_O_SDFIFOCTL1 + ((uint32_t)filterNumber * 32U);

        /* Write to SDFFIL bit */
        HWREGH(address) =
            ((HWREGH(address) & (~SDFM_SDFIFOCTL1_SDFFIL_M)) | fifoLevel);
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        uint32_t address;

        ASSERT(SDFM_isBaseValid(base));
        ASSERT(fifoLevel <= 16U);

        if(filterNumber != SDFM_FILTER_5)
        {

            address = base + SDFM_O_SDFIFOCTL1 + ((uint32_t)filterNumber * 32U);

            /* Write to SDFFIL bit */
            HWREGH(address) =
                ((HWREGH(address) & (~SDFM_SDFIFOCTL1_SDFFIL_M)) | fifoLevel);
        }else
        {

            address = base + SDFM_O_SDFIFOCTL5;

            /* Write to SDFFIL bit */
            HWREGH(address) =
                ((HWREGH(address) & (~SDFM_SDFIFOCTL5_SDFFIL_M)) | fifoLevel);
        }

    #endif
}

/**
 * @brief Set data ready interrupt source.
 *
 * @details
 * This function sets the data ready interrupt source.
 *
 * @param[in] base            Specifies the SDFM module base address.
 * @param[in] filterNumber    The filter number.
 * @param[in] dataReadySource The data ready interrupt source.
 *
 * @remarks
 *   Valid values for dataReadySource:
 *     SDFM_DATA_READY_SOURCE_DIRECT   Direct data ready.
 *     SDFM_DATA_READY_SOURCE_FIFO    FIFO data ready.
 */
static inline void
SDFM_setDataReadyInterruptSource(uint32_t base, SDFM_FilterNumber filterNumber,
                                 SDFM_DataReadyInterruptSource dataReadySource)
{
    #if (SDFM_VERSION == 200)
        uint32_t address;

        ASSERT(SDFM_isBaseValid(base));

        address = base + SDFM_O_SDFIFOCTL1 + ((uint32_t)filterNumber * 32U);

        /* Write to DRINTSEL */
        HWREGH(address) = (HWREGH(address) & ~SDFM_SDFIFOCTL1_DRINTSEL) |
                        ((uint16_t)dataReadySource << 14U);
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        uint32_t address;

        ASSERT(SDFM_isBaseValid(base));

        if(filterNumber != SDFM_FILTER_5)
        {

            address = base + SDFM_O_SDFIFOCTL1 + ((uint32_t)filterNumber * 32U);

            /* Write to DRINTSEL */
            HWREGH(address) = (HWREGH(address) & ~SDFM_SDFIFOCTL1_DRINTSEL) |
                            ((uint16_t)dataReadySource << 14U);
        }else
        {

            address = base + SDFM_O_SDFIFOCTL5;

            /* Write to DRINTSEL */
            HWREGH(address) = (HWREGH(address) & ~SDFM_SDFIFOCTL5_DRINTSEL) |
                            ((uint16_t)dataReadySource << 14U);
        }

    #endif
}

/**
 * @brief Get the wait-for-sync event status.
 *
 * @details
 * This function returns the Wait-for-Sync event status.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 *
 * @return true   if sync event has occurred.
 *         false  if sync event has not occurred.
 */
static inline bool
SDFM_getWaitForSyncStatus(uint32_t base, SDFM_FilterNumber filterNumber)
{
    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        /* Read WTSYNFLG bit */
        return(((HWREGH(base + SDFM_O_SDSYNC1 + ((uint32_t)filterNumber * 32U)) &
                SDFM_SDSYNC1_WTSYNFLG) >> 7U) == 0x1U);
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        ASSERT(SDFM_isBaseValid(base));

        if(filterNumber != SDFM_FILTER_5)
        {

            /* Read WTSYNFLG bit */
            return(((HWREGH(base + SDFM_O_SDSYNC1 + ((uint32_t)filterNumber * 32U)) &
                    SDFM_SDSYNC1_WTSYNFLG) >> 7U) == 0x1U);
        }else
        {

            /* Read WTSYNFLG bit */
            return(((HWREGH(base + SDFM_O_SDSYNC5 ) & SDFM_SDSYNC5_WTSYNFLG) >> 7U) == 0x1U);
        }

    #endif
}

/**
 * @brief Clear the Wait-for-sync event status.
 *
 * @details
 * This function clears the Wait-for-sync event status.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 */
static inline void
SDFM_clearWaitForSyncFlag(uint32_t base, SDFM_FilterNumber filterNumber)
{

    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        EALLOW;

        /* Clear WTSYNCLR bit */
        HWREGH(base + SDFM_O_SDSYNC1 + ((uint32_t)filterNumber * 32U)) |=
            SDFM_SDSYNC1_WTSYNCLR;
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        ASSERT(SDFM_isBaseValid(base));

        if(filterNumber != SDFM_FILTER_5)
        {

            /* Clear WTSYNCLR bit */
            HWREGH(base + SDFM_O_SDSYNC1 + ((uint32_t)filterNumber * 32U)) |=
                SDFM_SDSYNC1_WTSYNCLR;
        }
        else
        {
            /* Clear WTSYNCLR bit */
            HWREGH(base + SDFM_O_SDSYNC5) |= SDFM_SDSYNC5_WTSYNCLR;
        }

    #endif
}

/**
 * @brief Enable wait for sync mode.
 *
 * @details
 * This function enables the wait for sync mode. Data to FIFO will be written
 * only after PWM sync event.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 */
static inline void
SDFM_enableWaitForSync(uint32_t base, SDFM_FilterNumber filterNumber)
{

    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        /* Set WTSYNCEN bit */
        HWREGH(base + SDFM_O_SDSYNC1 + ((uint32_t)filterNumber * 32U)) |=
            SDFM_SDSYNC1_WTSYNCEN;
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        ASSERT(SDFM_isBaseValid(base));

        if(filterNumber != SDFM_FILTER_5)
        {

            /* Set WTSYNCEN bit */
            HWREGH(base + SDFM_O_SDSYNC1 + ((uint32_t)filterNumber * 32U)) |=
                SDFM_SDSYNC1_WTSYNCEN;
        }else
        {
            /* Set WTSYNCEN bit */
            HWREGH(base + SDFM_O_SDSYNC5 ) |= SDFM_SDSYNC5_WTSYNCEN;
        }

    #endif
}

/**
 * @brief Disable wait for sync mode.
 *
 * @details
 * This function disables the wait for sync mode. Data to FIFO will be written
 * every Data ready event.
 *
 * @param[in] base           Specifies the SDFM module base address.
 * @param[in] filterNumber   The filter number.
 */
static inline void
SDFM_disableWaitForSync(uint32_t base, SDFM_FilterNumber filterNumber)
{
    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        /* Clear WTSYNCEN bit */
        HWREGH(base + SDFM_O_SDSYNC1 + ((uint32_t)filterNumber * 32U)) &=
            ~SDFM_SDSYNC1_WTSYNCEN;
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        ASSERT(SDFM_isBaseValid(base));

        if(filterNumber != SDFM_FILTER_5)
        {
            /* Clear WTSYNCEN bit */
            HWREGH(base + SDFM_O_SDSYNC1 + ((uint32_t)filterNumber * 32U)) &=
                ~SDFM_SDSYNC1_WTSYNCEN;
        }else
        {
            /* Clear WTSYNCEN bit */
            HWREGH(base + SDFM_O_SDSYNC5 ) &= ~SDFM_SDSYNC5_WTSYNCEN;
        }

    #endif
}

/**
 * @brief Set the PWM sync mode.
 *
 * @details
 * This function sets the PWM sync source for the specific SDFM filter.Valid
 * values for syncSource are SDFM_SYNC_PWMx_CMPy. Where x ranges from 1 to 8
 * Representing PWM1 to PWM8 respectively and y ranges from A to D representing
 * PWM comparators A to D.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 * @param[in] syncSource    The PWM sync source.
 */
static inline void
SDFM_setPWMSyncSource(uint32_t base, SDFM_FilterNumber filterNumber,
                      SDFM_PWMSyncSource syncSource)
{
    #if (SDFM_VERSION == 200)
        uint32_t address;

        ASSERT(SDFM_isBaseValid(base));

        address = base + SDFM_O_SDSYNC1 + ((uint32_t)filterNumber * 32U);

        /* Write to SYNCSEL bits */
        HWREGH(address) =
            (HWREGH(address) & ~SDFM_SDSYNC1_SYNCSEL_M) | (uint16_t)syncSource;

        /* Enable SDSYNC reset to data filter */
        SDFM_enableExternalReset(base, filterNumber);
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        uint32_t address;

        ASSERT(SDFM_isBaseValid(base));

        if(filterNumber != SDFM_FILTER_5)
        {
            address = base + SDFM_O_SDSYNC1 + ((uint32_t)filterNumber * 32U);

            /* Write to SYNCSEL bits */
            HWREGH(address) =
                (HWREGH(address) & ~SDFM_SDSYNC1_SYNCSEL_M) | (uint16_t)syncSource;
            /* Enable SDSYNC reset to data filter */
            SDFM_enableExternalReset(base, filterNumber);
        }else
        {
            address = base + SDFM_O_SDSYNC5;

            /* Write to SYNCSEL bits */
            HWREGH(address) =
                (HWREGH(address) & ~SDFM_SDSYNC5_SYNCSEL_M) | (uint16_t)syncSource;

            /* Enable SDSYNC reset to data filter */
            SDFM_enableExternalReset(base, filterNumber);  
        }

    #endif
}

/**
 * @brief Set FIFO clear on sync mode.
 *
 * @details
 * This function sets the FIFO clear mode for the specified filter when a sync
 * happens depending on the value of fifoClearSyncMode.
 *
 * @param[in] base              Specifies the SDFM module base address.
 * @param[in] filterNumber      The filter number.
 * @param[in] fifoClearSyncMode The FIFO clear on sync mode.
 *
 * @remarks
 *   Valid values for fifoClearSyncMode are:
 *     SDFM_FIFO_NOT_CLEARED_ON_SYNC   FIFO is not cleared on sync.
 *     SDFM_FIFO_CLEARED_ON_SYNC   FIFO is cleared on sync.
 */
static inline void
SDFM_setFIFOClearOnSyncMode(uint32_t base, SDFM_FilterNumber filterNumber,
                            SDFM_FIFOClearSyncMode fifoClearSyncMode)
{
    #if (SDFM_VERSION == 200)
        uint32_t address;

        ASSERT(SDFM_isBaseValid(base));

        address = base + SDFM_O_SDSYNC1 + ((uint32_t)filterNumber * 32U);

        /* Write to FFSYNCCLREN bit */
        HWREGH(address) = (HWREGH(address) & ~SDFM_SDSYNC1_FFSYNCCLREN) |
                        ((uint16_t)fifoClearSyncMode << 9U);
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        uint32_t address;
        ASSERT(SDFM_isBaseValid(base));

        if(filterNumber != SDFM_FILTER_5)
        {
            address = base + SDFM_O_SDSYNC1 + ((uint32_t)filterNumber * 32U);

            /* Write to FFSYNCCLREN bit */
            HWREGH(address) = (HWREGH(address) & ~SDFM_SDSYNC1_FFSYNCCLREN) |
                            ((uint16_t)fifoClearSyncMode << 9U);
        }else
        {
            address = base + SDFM_O_SDSYNC5;

            /* Write to FFSYNCCLREN bit */
            HWREGH(address) = (HWREGH(address) & ~SDFM_SDSYNC5_FFSYNCCLREN) |
                            ((uint16_t)fifoClearSyncMode << 9U);
        }

    #endif
}

/**
 * @brief Set Wait-for-sync clear mode.
 *
 * @details
 * This function sets the Wait-For-sync clear mode depending on the value of
 * syncClearMode.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 * @param[in] syncClearMode The wait-for-sync clear mode.
 *
 * @remarks
 *   Valid values for syncClearMode are:
 *     SDFM_MANUAL_CLEAR_WAIT_FOR_SYNC   Wait-for-sync flag is cleared by
 *                                       invoking SDFM_clearWaitForSyncFlag().
 *   - SDFM_AUTO_CLEAR_WAIT_FOR_SYNC     Wait-for-sync flag is cleared
 *                                       automatically on FIFO interrupt.
 */
static inline void
SDFM_setWaitForSyncClearMode(uint32_t base, SDFM_FilterNumber filterNumber,
                             SDFM_WaitForSyncClearMode syncClearMode)
{
    #if (SDFM_VERSION == 200)
        uint32_t address;

        ASSERT(SDFM_isBaseValid(base));

        address = base + SDFM_O_SDSYNC1 + ((uint32_t)filterNumber * 32U);

        /* Write to WTSCLREN  bit */
        EALLOW;

        HWREGH(address) = (HWREGH(address) & ~SDFM_SDSYNC1_WTSCLREN) |
                        ((uint16_t)syncClearMode << 10U);
        EDIS;
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        uint32_t address;
        ASSERT(SDFM_isBaseValid(base));

        if(filterNumber != SDFM_FILTER_5)
        {
            address = base + SDFM_O_SDSYNC1 + ((uint32_t)filterNumber * 32U);

            EALLOW;

            /* Write to WTSCLREN  bit */
            HWREGH(address) = (HWREGH(address) & ~SDFM_SDSYNC1_WTSCLREN) |
                            ((uint16_t)syncClearMode << 10U);
            EDIS;
        }else
        {
            address = base + SDFM_O_SDSYNC5;

            EALLOW;

            /* Write to WTSCLREN  bit */
            HWREGH(address) = (HWREGH(address) & ~SDFM_SDSYNC5_WTSCLREN) |
                            ((uint16_t)syncClearMode << 10U);
            EDIS; 
        }

    #endif
}

/**
 * @brief Selects clock source for SDFM channels.
 *
 * @details
 * This function selects the clock for SDFM module filter channels.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 * @param[in] clkSource     The clock source
 *
 * @remarks
 *   Valid values for clkSource are:
 *     SDFM_CLK_SOURCE_CHANNEL_CLK   Respective channel's clk is the source
 *     SDFM_CLK_SOURCE_SD1_CLK       Filter 1 clock is the source
 */
static inline void
SDFM_selectClockSource(uint32_t base, SDFM_FilterNumber filterNumber,
                       SDFM_ClockSource clkSource)
{
    #if (SDFM_VERSION == 200)
        uint32_t address;

        ASSERT(SDFM_isBaseValid(base));
        address = base + SDFM_O_SDCTLPARM1 +
                ((uint32_t)filterNumber * SDFM_SDFIL_OFFSET);

        /* Select SDFM clock source. */
        EALLOW;

        HWREGH(address) = (HWREGH(address) & ~(SDFM_SDCTLPARM1_SDCLKSEL)) |
                        (uint16_t)clkSource;
        EDIS;
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
 
        if(filterNumber != SDFM_FILTER_5)
        {
            uint32_t address;

            ASSERT(SDFM_isBaseValid(base));
            address = base + SDFM_O_SDCTLPARM1 +
                    ((uint32_t)filterNumber * SDFM_SDFIL_OFFSET);

            /* Select SDFM clock source. */
            EALLOW;

            HWREGH(address) = (HWREGH(address) & ~(SDFM_SDCTLPARM1_SDCLKSEL)) |
                            (uint16_t)clkSource;
            EDIS;
        }else
        {
            uint32_t address;

            ASSERT(SDFM_isBaseValid(base));
            address = base + SDFM_O_SDCTLPARM5;

            /* Select SDFM clock source. */
            EALLOW;

            HWREGH(address) = (HWREGH(address) & ~(SDFM_SDCTLPARM5_SDCLKSEL)) | (uint16_t)clkSource;

            EDIS;  
        }

    #endif
}

/**
 * @brief Enables Input Synchronizer.
 *
 * @details
 * This function enables either data or clock or both synchronizer.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 * @param[in] syncConfig    Defines which synchronizer to be enabled
 *
 * @remarks
 *   Valid values for syncConfig can be the logical OR of any of the values:
 *     SDFM_CLOCK_SYNCHRONIZER   Enable SDFM input clock synchronizer
 *     SDFM_DATA_SYNCHRONIZER    Enable SDFM input data synchronizer
 */
static inline void
SDFM_enableSynchronizer(uint32_t base, SDFM_FilterNumber filterNumber,
                        uint16_t syncConfig)
{
    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        /* Select SDFM clock source. */
        EALLOW;

        HWREGH(base + SDFM_O_SDCTLPARM1 +
            ((uint32_t)filterNumber * SDFM_SDFIL_OFFSET)) |= syncConfig;
        EDIS;
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        ASSERT(SDFM_isBaseValid(base));

        if(filterNumber != SDFM_FILTER_5)
        {
            /* Select SDFM clock source. */
            EALLOW;

            HWREGH(base + SDFM_O_SDCTLPARM1 +
                ((uint32_t)filterNumber * SDFM_SDFIL_OFFSET)) |= syncConfig;
            EDIS;
        }else
        {
            /* Select SDFM clock source. */
            EALLOW;

            HWREGH(base + SDFM_O_SDCTLPARM5) |= syncConfig;

            EDIS;   
        }

    #endif
}

/**
 * @brief Disables Input Synchronizer.
 *
 * @details
 * This function disables either data or clock or both synchronizer.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 * @param[in] syncConfig    Defines which synchronizer to be disabled
 *
 * @remarks
 *   Valid values for syncConfig can be the logical OR of any of the values:
 *     SDFM_CLOCK_SYNCHRONIZER   Disable SDFM input clock synchronizer
 *     SDFM_DATA_SYNCHRONIZER    Disable SDFM input data synchronizer
 */
static inline void
SDFM_disableSynchronizer(uint32_t base, SDFM_FilterNumber filterNumber,
                         uint16_t syncConfig)
{
    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        /* Select SDFM clock source. */
        EALLOW;

        HWREGH(base + SDFM_O_SDCTLPARM1 +
            ((uint32_t)filterNumber * SDFM_SDFIL_OFFSET)) &= ~syncConfig;
        EDIS;
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        ASSERT(SDFM_isBaseValid(base));
        if(filterNumber != SDFM_FILTER_5)
        {
            /* Select SDFM clock source. */
            EALLOW;

            HWREGH(base + SDFM_O_SDCTLPARM1 +
                ((uint32_t)filterNumber * SDFM_SDFIL_OFFSET)) &= ~syncConfig;
            EDIS;
        }else
        {
            /* Select SDFM clock source. */
            EALLOW;

            HWREGH(base + SDFM_O_SDCTLPARM5) &= ~syncConfig;

            EDIS;  
        }

    #endif
}

/**
 * @brief Selects comparator event high source.
 *
 * @details
 * This function selects the source for comparator event high.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 * @param[in] source        The comparator event high source
 *
 * @remarks
 *   Valid values for source are:
 *      SDFM_COMPHOUT_SOURCE_COMPHIN   Original COMPHIN/CEVT1 signal is source
 *      SDFM_COMPHOUT_SOURCE_FILTER    Filtered COMPHIN/CEVT1 signal is source
 */
static inline void
SDFM_selectCompEventHighSource(uint32_t base, SDFM_FilterNumber filterNumber,
                               SDFM_CompEventHighSource source)
{
    #if (SDFM_VERSION == 200)
        uint32_t address;
        ASSERT(SDFM_isBaseValid(base));
        address = base + SDFM_O_SDCOMP1CTL +
                ((uint32_t)filterNumber * SDFM_DIGFIL_OFFSET);

        /* Set COMPHOUT source. */
        EALLOW;

        HWREGH(address) = (HWREGH(address) & ~SDFM_SDCOMP1CTL_CEVT1DIGFILTSEL_M) |
                          (uint16_t)source;
        EDIS;
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        uint32_t address;
        ASSERT(SDFM_isBaseValid(base));
        if(filterNumber != SDFM_FILTER_5)
        {
            address = base + SDFM_O_SDCOMP1CTL +
                    ((uint32_t)filterNumber * SDFM_DIGFIL_OFFSET);

            /* Set COMPHOUT source. */
            EALLOW;

            HWREGH(address) = (HWREGH(address) & ~SDFM_SDCOMP1CTL_CEVT1DIGFILTSEL_M) |
                            (uint16_t)source;
            EDIS;
        }else
        {

            address = base + SDFM_O_SDCOMP5CTL;

            /* Set COMPHOUT source. */
            EALLOW;

            HWREGH(address) = (HWREGH(address) & ~SDFM_SDCOMP5CTL_CEVT1DIGFILTSEL_M) |
            				  (uint16_t)source;

            EDIS;
        }

    #endif
}

/**
 * @brief Selects comparator event low source.
 *
 * @details
 * This function selects the source for comparator event low.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 * @param[in] source        The comparator event low source
 *
 * @remarks
 *   Valid values for source are:
 *     SDFM_COMPLOUT_SOURCE_COMPLIN   Original COMPLIN/CEVT2 signal is source
 *     SDFM_COMPHOUT_SOURCE_FILTER    Filtered COMPLIN/CEVT2 signal is source
 */
static inline void
SDFM_selectCompEventLowSource(uint32_t base, SDFM_FilterNumber filterNumber,
                              SDFM_CompEventLowSource source)
{
    #if (SDFM_VERSION == 200)
        uint32_t address;
        ASSERT(SDFM_isBaseValid(base));
        address = base + SDFM_O_SDCOMP1CTL +
                ((uint32_t)filterNumber * SDFM_DIGFIL_OFFSET);

        /* Set COMPLOUT source. */
        EALLOW;

        HWREGH(address) = (HWREGH(address) & ~SDFM_SDCOMP1CTL_CEVT2DIGFILTSEL_M) |
                        (uint16_t)source;
        EDIS;
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        uint32_t address;
        ASSERT(SDFM_isBaseValid(base));

        if(filterNumber != SDFM_FILTER_5)
        {
            uint32_t address;
            ASSERT(SDFM_isBaseValid(base));
            address = base + SDFM_O_SDCOMP1CTL +
                    ((uint32_t)filterNumber * SDFM_DIGFIL_OFFSET);

            /* Set COMPLOUT source. */
            EALLOW;

            HWREGH(address) = (HWREGH(address) & ~SDFM_SDCOMP1CTL_CEVT2DIGFILTSEL_M) |
                            (uint16_t)source;
            EDIS;
        }else
        {

            address = base + SDFM_O_SDCOMP5CTL;

            /* Set COMPLOUT source. */
            EALLOW;

            HWREGH(address) = (HWREGH(address) & ~SDFM_SDCOMP5CTL_CEVT2DIGFILTSEL_M) | (uint16_t)source;

            EDIS;
        }

    #endif
}

/**
 * @brief Initializes Comparator Event Low Filter.
 *
 * @details
 * This function initializes Comparator Event Low Filter.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 */
static inline void
SDFM_initCompEventLowFilter(uint32_t base, SDFM_FilterNumber filterNumber)
{
    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        /* Initialize comparator event low filter. */
        EALLOW;

        HWREGH(base + SDFM_O_SDCOMP1EVT2FLTCTL +
            ((uint32_t)filterNumber * SDFM_DIGFIL_OFFSET)) |=
                                        (uint16_t)SDFM_SDCOMP1EVT2FLTCTL_FILINIT;
        EDIS;
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        ASSERT(SDFM_isBaseValid(base));
        if(filterNumber != SDFM_FILTER_5)
        {
            ASSERT(SDFM_isBaseValid(base));

            /* Initialize comparator event low filter. */
            EALLOW;

            HWREGH(base + SDFM_O_SDCOMP1EVT2FLTCTL +
                ((uint32_t)filterNumber * SDFM_DIGFIL_OFFSET)) |=
                                            (uint16_t)SDFM_SDCOMP1EVT2FLTCTL_FILINIT;
            EDIS;
        }else
        {
            /* Initialize comparator event low filter. */
            EALLOW;

            HWREGH(base + SDFM_O_SDCOMP5EVT2FLTCTL) |=
                                            (uint16_t)SDFM_SDCOMP5EVT2FLTCTL_FILINIT;
            EDIS;
        }

    #endif
}

/**
 * @brief Initializes Comparator Event High Filter.
 *
 * @details
 * This function initializes Comparator Event High Filter.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 */
static inline void
SDFM_initCompEventHighFilter(uint32_t base, SDFM_FilterNumber filterNumber)
{
    #if (SDFM_VERSION == 200)
        ASSERT(SDFM_isBaseValid(base));

        /* Initialize comparator event high filter. */
        EALLOW;

        HWREGH(base + SDFM_O_SDCOMP1EVT1FLTCTL +
            ((uint32_t)filterNumber * SDFM_DIGFIL_OFFSET)) |=
                                        (uint16_t)SDFM_SDCOMP1EVT1FLTCTL_FILINIT;
        EDIS;
    #elif (SDFM_VERSION == 300)
        ASSERT(filterNumber <= SDFM_FILTER_5);
        ASSERT(SDFM_isBaseValid(base));

        if(filterNumber != SDFM_FILTER_5)
        {
            /* Initialize comparator event high filter. */
            EALLOW;

            HWREGH(base + SDFM_O_SDCOMP1EVT1FLTCTL +
                ((uint32_t)filterNumber * SDFM_DIGFIL_OFFSET)) |=
                                            (uint16_t)SDFM_SDCOMP1EVT1FLTCTL_FILINIT;
            EDIS;
        }else
        {
            /* Initialize comparator event high filter. */
            EALLOW;

            HWREGH(base + SDFM_O_SDCOMP5EVT1FLTCTL) |= (uint16_t)SDFM_SDCOMP5EVT1FLTCTL_FILINIT;

            EDIS;
        }

    #endif
}

/**
 * @brief Configures SDFM comparator for filter config & threshold values
 *
 * @details
 * This function configures the comparator filter for filter config and
 * threshold values based on provided inputs.
 *
 * @param[in] base	  Specifies the SDFM module base address.
 * @param[in] config1 The filter number, filter type and over sampling ratio.
 * @param[in] config2 Is high-level and low-level threshold 1 values.
 * @param[in] config3 is the zero-cross threshold value.
 *
 * @remarks
 *   The config1 parameter is the logical OR of the filter number, filter type
 *   and oversampling ratio.
 *   The bit definitions for config1 are as follow:
 *       config1.[3:0]  filter number
 *       config1.[7:4]  filter type
 *       config1.[15:8] Over sampling Ratio
 *   Valid values for filter number and filter type are defined in
 *   SDFM_FilterNumber and SDFM_FilterType enumerations respectively.
 *   SDFM_SET_OSR(X) macro can be used to set the value of the oversampling
 *   ratio ,which ranges [1,32] inclusive, in the appropriate bit location.
 *   For example the value
 *   (SDFM_FILTER_1 | SDFM_FILTER_SINC_2 | SDFM_SET_OSR(16))
 *   will select Filter 1, SINC 2 type with an oversampling ratio of 16.
 *
 *   The config2 parameter is the logical OR of the filter high and low
 *   threshold 1 values.
 *   The bit definitions for config2 are as follow:
 *     - config2.[15:0]  low threshold 1
 *     - config2.[31:16] high threshold 1
 *   The upper 16 bits define the high threshold 1 and the lower 16 bits define
 *   the low threshold 1. SDFM_THRESHOLD(H,L) can be used to combine the high
 *   and low thresholds.
 *   The config3 parameter is the logical OR of the zero cross threshold
 *   enable flag and the zero-cross threshold value.
 *   The bit definitions for config3 are as follow:
 *     - config3.[15] - Enable or disable zero cross threshold. Valid values
 *                      are 1 or 0 to enable or disable the zero cross threshold
 *                      respectively.
 *     -config3.[14:0] - Zero Cross Threshold value.
 *   The SDFM_SET_ZERO_CROSS_THRESH_VALUE(X) macro can be used to specify the
 *   zero-cross threshold value and OR the 1 to enable it.
 */
extern void
SDFM_configComparator(uint32_t base, uint16_t config1,
                      uint32_t config2, uint16_t config3);

/**
 * @brief
 * Configure SDFM enhanced comparator for filter config & threshold values
 *
 * @details
 * This function configures the comparator filter for filter config and
 * threshold values based on input parameters.
 *
 * @param[in] base               Specifies the SDFM module base address.
 * @param[in] filterNumber       The filter number.
 * @param[in] highLowThreshold1  Is high-level and low-level threshold 1 values.
 * @param[in] highLowThreshold2  Is high-level and low-level threshold 2 values.
 * @param[in] zeroCrossThreshold Is the zero-cross threshold value.
 *
 * @remarks
 *   The filterConfig parameter is the logical OR of the filter number, filter
 *   type and oversampling ratio.
 *   The bit definitions for filterConfig are as follow:
 *       filterConfig.[3:0]  filter number
 *       filterConfig.[7:4]  filter type
 *       filterConfig.[15:8] Over sampling Ratio
 *   Valid values for filter number and filter type are defined in
 *   SDFM_FilterNumber and SDFM_FilterType enumerations respectively.
 *   SDFM_SET_OSR(X) macro can be used to set the value of the oversampling
 *   ratio ,which ranges [1,32] inclusive, in the appropriate bit location.
 *   For example the value
 *   (SDFM_FILTER_1 | SDFM_FILTER_SINC_2 | SDFM_SET_OSR(16))
 *   will select Filter 1, SINC 2 type with an oversampling ratio of 16.
 *
 *   The highLowThreshold1 parameter is the logical OR of the filter high & low
 *   threshold 1 values.
 *   The bit definitions for highLowThreshold1 are as follow:
 *       highLowThreshold1.[15:0]  low threshold 1
 *       highLowThreshold1.[31:16] high threshold 1
 *   The upper 16 bits define the high threshold and the lower 16 bits define
 *   the low threshold. SDFM_THRESHOLD(H,L) can be used to combine the high and
 *   low thresholds.
 *
 *   The highLowThreshold2 parameter is the logical OR of the filter high & low
 *   threshold 2 values.
 *   The bit definitions for highLowThreshold2 are as follow:
 *       highLowThreshold2.[15:0]  low threshold 2
 *       highLowThreshold2.[31:16] high threshold 2
 *   The upper 16 bits define the high threshold and the lower 16 bits define
 *   the low threshold. SDFM_THRESHOLD(H,L) can be used to combine the high &
 *   low thresholds.
 *
 *   The zeroCrossThreshold parameter is the logical OR of the zero cross
 *   threshold enable flag and the zero-cross threshold value. The bit
 *   definitions for zeroCrossThreshold are as follows:
 *       zeroCrossThreshold.[15] - Enable or disable zero cross threshold. Valid
 *       values are 1 or 0 to enable or disable the zero cross threshold
 *       respectively.
 *       zeroCrossThreshold.[14:0] - Zero Cross Threshold value.
 *   The SDFM_SET_ZERO_CROSS_THRESH_VALUE(X) macro can be used as parameter
 *   zeroCrossThreshold to enable & specify the zero-cross threshold value.
 */
extern void
SDFM_configEnhancedComparator(uint32_t base, uint16_t filterConfig,
                              uint32_t highLowThreshold1,
                              uint32_t highLowThreshold2,
                              uint16_t zeroCrossThreshold);

/**
 * @brief Configure SDFM data filter
 *
 * @details
 * This function configures the data filter based on configurations
 * config1 and config2.
 *
 * @param[in] base      Specifies the SDFM module base address.
 * @param[in] config1   The filter number, filter type and over sampling ratio
 *                      configuration.
 * @param[in] config2   Is filter switch, data representation and data shift values
 *                      configuration.
 *
 * @remarks
 *   The config1 parameter is the logical OR of the filter number, filter type
 *   and oversampling ratio.
 *   The bit definitions for config1 are as follow:
 *       config1.[3:0]  Filter number
 *       config1.[7:4]  Filter type
 *       config1.[15:8] Over sampling Ratio
 *   Valid values for filter number and filter type are defined in
 *   SDFM_FilterNumber and SDFM_FilterType enumerations respectively.
 *   SDFM_SET_OSR(X) macro can be used to set the value of the oversampling
 *   ratio , which ranges [1,256] inclusive , in the appropriate bit location
 *   for config1. For example the value
 *   (SDFM_FILTER_2 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(64))
 *   will select Filter 2 , SINC 3 type with an oversampling ratio of 64.
 *
 *   The config2 parameter is the logical OR of data representation, filter
 *   switch, and data shift values
 *   The bit definitions for config2 are as follow:
 *       config2.[0]  Data representation
 *       config2.[1]  Filter switch
 *       config2.[15:2]  Shift values
 *   Valid values for data representation are given in SDFM_OutputDataFormat
 *   enumeration. SDFM_FILTER_DISABLE or SDFM_FILTER_ENABLE will define the
 *   filter switch values.SDFM_SHIFT_VALUE(X) macro can be used to set the value
 *   of the data shift value,which ranges [0,31] inclusive, in the appropriate
 *   bit location for config2.
 *   The shift value is valid only in SDFM_DATA_FORMAT_16_BIT data
 *   representation format.
 */
extern void
SDFM_configDataFilter(uint32_t base, uint16_t config1, uint16_t config2);

/**
 * @brief Configure SDFM comparator Zero Cross threshold
 *
 * @details
 * This function configures the comparator filter zero cross threshold values
 * based on configurations config1 and config2.
 *
 * @param[in] base	  Specifies the SDFM module base address.
 * @param[in] config1 The filter number, filter type and over sampling ratio.
 * @param[in] config2 The zero cross threshold value.
 *
 * @remarks
 *   The config1 parameter is the logical OR of the filter number, filter type
 *   and oversampling ratio.
 *   The bit definitions for config1 are as follow:
 *       config1.[3:0]  filter number
 *       config1.[7:4]  filter type
 *       config1.[15:8] Over sampling Ratio
 *   Valid values for filter number and filter type are defined in
 *   SDFM_FilterNumber and SDFM_FilterType enumerations respectively.
 *   SDFM_SET_OSR(X) macro can be used to set the value of the oversampling
 *   ratio ,which ranges [1,32] inclusive, in the appropriate bit location.
 *   For example the value
 *   (SDFM_FILTER_1 | SDFM_FILTER_SINC_2 | SDFM_SET_OSR(16))
 *   will select Filter 1 , SINC 2 type with an oversampling ratio of 16.
 *
 *   The config2 parameter is the value of the zero cross threshold. The maximum
 *   acceptable value is 32767.
 */
extern void
SDFM_configZeroCrossComparator(uint32_t base, uint16_t config1,
                               uint16_t config2);

/**
 * @brief Configure SDFM data filter FIFO
 *
 * @details
 * This function enables and configures the data filter FIFO based on
 * configurations config1 and config2.
 *
 * @param base     Specifies the SDFM module base address.
 * @param config1  The filter number, filter type and over sampling ratio
 *                 configuration.
 * @param config2  Is filter switch, data representation and data shift values
 *                 and FIFO level configuration.
 *
 * @remarks
 *   The config1 parameter is the logical OR of the filter number, filter type
 *   and oversampling ratio.
 *   The bit definitions for config1 are as follow:
 *       config1.[3:0]  filter number
 *       config1.[7:4]  filter type
 *       config1.[15:8] Over sampling Ratio
 *   Valid values for filter number and filter type are defined in
 *   SDFM_FilterNumber and SDFM_FilterType enumerations respectively.
 *   SDFM_SET_OSR(X) macro can be used to set the value of the oversampling
 *   ratio , which ranges [1,256] inclusive , in the appropriate bit location
 *   for config1. For example the value
 *   (SDFM_FILTER_2 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(64))
 *   will select Filter 2 , SINC 3 type with an oversampling ratio of 64.
 *
 *   The config2 parameter is the logical OR of data representation, filter
 *   switch, data shift value, and FIFO level
 *   The bit definitions for config2 are as follow:
 *       config2.[0]  Data representation
 *       config2.[1]  filter switch.
 *       config2.[6:2]  shift values.
 *       config2.[15:7] FIFO level
 *   Valid values for data representation are given in SDFM_OutputDataFormat
 *   enumeration. SDFM_FILTER_DISABLE or SDFM_FILTER_ENABLE will define the
 *   filter switch values.SDFM_SHIFT_VALUE(X) macro can be used to set the value
 *   of the data shift value,which ranges [0,31] inclusive, in the appropriate
 *   bit location for config2.
 *   The value of FIFO level ranges [1,16] inclusive. The macro
 *   SDFM_SET_FIFO_LEVEL(X) can be used to set the value of the FIFO level.
 */

extern void
SDFM_configDataFilterFIFO(uint32_t base, uint16_t config1, uint16_t config2);

/**
 * @brief Configure Comparator Event Low Filter
 *
 * @details
 * This function configures the sample window, threshold and clock prescale
 * configurations for the comparator event low filter.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 * @param[in] onfig         Is the comparator event low source
 */
extern void
SDFM_configCompEventLowFilter(uint32_t base, SDFM_FilterNumber filterNumber,
                              const SDFM_CompEventFilterConfig *config);

/**
 * @brief Configure Comparator Event High Filter
 *
 * @details
 * This function configures the sample window, threshold and clock prescale
 * configurations for the comparator event high filter.
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 * @param[in] config        The comparator event high source
 */
extern void
SDFM_configCompEventHighFilter(uint32_t base, SDFM_FilterNumber filterNumber,
                               const SDFM_CompEventFilterConfig *config);

/**
 * @brief configure SDFM Type
 *
 * @param[in] base          Specifies the SDFM module base address.
 * @param[in] filterNumber  The filter number.
 *
 */
static inline void SysCtl_configureType(uint16_t config)
{
    uint32_t address;

    address = SYSCTL_BASE + SDFM_TYPE_CTRL;

    EALLOW;

    HWREGH(address) = config;

    EDIS;
}

/* Mark the end of the C bindings section for C++ compilers. */

#ifdef __cplusplus
}
#endif

#endif
#endif
