/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    asysctl_v12.c
*   @brief   digital system top level configuration
*   @details
*
*/
/*
 * commit history
 * 20240306, Jason Yang, Add calling ADC_setINLTrim in ASysCtl_initConfig.
 *                       This is a must to get accurate ADC results.
 * 20240324, Jason Yang, add code to init AnaTop for CMPSS and enable CMPSS.
 * *
 * 20240619, CaoHan, add code to PintoPin CMPSS
 * 20241209, Zhaolei, fix v6 DIR,add v7 beta
 * 20241211, Zhaolei, revert 20241211 v6 DIR fix
 * 20241219, Zhaolei, v7 trim
 * 20241223, Zhaolei, using v7 trim for new version (> v7) trim param
 */

#include "gs32_version.h"

#if IS_GS32F00xx(0x12)      //used for 1.2CS

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "sysctl/asysctl_v12.h"
#include "adc.h"
#include "cmpss.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

GS32_DRIVER_ROM_DATA_T AnaTrimTable_t anaTrimTable_v0[] = {
    [V0_TRIM_ID_LDO_1V2           ] = {0x400CD308, REG_WIDTH_32BITS,  4,  0,    22,   0x1FFF4200,  4,  0},     //TOP_LDO            0   31
    [V0_TRIM_ID_ANA_LDO           ] = {0x400CD534, REG_WIDTH_32BITS,  2,  0,     0,   0x1FFF4210,  2,  0},     //LDO_ANA            0   7
    [V0_TRIM_ID_OSC1              ] = {0x400CD314, REG_WIDTH_32BITS, 14,  0,  1397,   0x1FFF4230, 14,  0},     //OSC                0   2047
    [V0_TRIM_ID_OSC2              ] = {0x400CD314, REG_WIDTH_32BITS, 30, 16,  1397,   0x1FFF4230, 30, 16},     //OSC                0   2047
    [V0_TRIM_ID_VREFHI_COARSE_3V3 ] = {0x400CD408, REG_WIDTH_32BITS, 13, 10,     7,   0x1FFF4250, 15, 12},     //VREFHI             0   15 //coarse
    [V0_TRIM_ID_VREFHI_FINE_3V3   ] = {0x400CD408, REG_WIDTH_32BITS, 17, 15,     3,   0x1FFF4250, 11,  8},     //VREFHI             0   7  //fine
    [V0_TRIM_ID_VREFHI_COARSE_2V5 ] = {0x400CD408, REG_WIDTH_32BITS, 13, 10,     7,   0x1FFF4250,  7,  4},     //VREFHI             0   15 //coarse
    [V0_TRIM_ID_VREFHI_FINE_2V5   ] = {0x400CD408, REG_WIDTH_32BITS, 17, 15,     3,   0x1FFF4250,  3,  0},     //VREFHI             0   7  //fine
    [V0_TRIM_ID_ADCA_VCM_INT      ] = {0x400CD408, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF4260, 24, 21},     //VCMA               0   15
    [V0_TRIM_ID_ADCA_VCM_EXT      ] = {0x400CD408, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF4260, 28, 26},     //VCMA               0   7
    [V0_TRIM_ID_ADCA_OFF_DIR      ] = {0x400CD408, REG_WIDTH_32BITS,  9,  8,     1,   0x1FFF4240,  9,  8},     //ADCA Offset direction    0  3
    [V0_TRIM_ID_ADCB_VCM_INT      ] = {0x400CD40C, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF4290, 24, 21},     //VCMB               0   15
    [V0_TRIM_ID_ADCB_VCM_EXT      ] = {0x400CD40C, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF4290, 28, 26},     //VCMB               0   7
    [V0_TRIM_ID_ADCB_OFF_DIR      ] = {0x400CD40C, REG_WIDTH_32BITS,  9,  8,     1,   0x1FFF4280,  9,  8},     //ADCB Offset direction    0  3
    [V0_TRIM_ID_ADCC_VCM_INT      ] = {0x400CD410, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF42C0, 24, 21},     //VCMC               0   15
    [V0_TRIM_ID_ADCC_VCM_EXT      ] = {0x400CD410, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF42C0, 28, 26},     //VCMC               0   7
    [V0_TRIM_ID_ADCC_OFF_DIR      ] = {0x400CD410, REG_WIDTH_32BITS,  9,  8,     1,   0x1FFF42B0,  9,  8},     //ADCC Offset direction    0  3
    [V0_TRIM_ID_ADCA_OFFSET       ] = {0x40000076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF4270,  7,  0},     //ADCA Offset    0   255
    [V0_TRIM_ID_ADCB_OFFSET       ] = {0x40000876, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF42A0,  7,  0},     //ADCB Offset    0   255
    [V0_TRIM_ID_ADCC_OFFSET       ] = {0x40001076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF42D0,  7,  0},     //ADCC Offset    0   255
};

GS32_DRIVER_ROM_DATA_T AnaTrimTable_t anaTrimTable_v5[] = {
    [V5_TRIM_ID_LDO_1V2               ] = {0x400CD308, REG_WIDTH_32BITS,  4,  0,    22,   0x1FFF4200,  4,  0},     //TOP_LDO            0   31
    [V5_TRIM_ID_ANA_LDO               ] = {0x400CD534, REG_WIDTH_32BITS,  2,  0,     0,   0x1FFF4210,  2,  0},     //LDO_ANA            0   7
    [V5_TRIM_ID_OSC1                  ] = {0x400CD314, REG_WIDTH_32BITS, 14,  0,  1397,   0x1FFF4230, 14,  0},     //OSC                0   2047
    [V5_TRIM_ID_OSC2                  ] = {0x400CD314, REG_WIDTH_32BITS, 30, 16,  1397,   0x1FFF4230, 30, 16},     //OSC                0   2047

    [V5_TRIM_ID_ADCA_OFF_DIR          ] = {0x400CD408, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF4240,  9,  8},     //ADCA Offset direction    0  3

    [V5_TRIM_ID_VREFHI_COARSE_1P65V   ] = {0x400CD408, REG_WIDTH_32BITS, 13, 10,     7,   0x1FFF4250, 15, 12},     //VREFHI             0   15 //coarse
    [V5_TRIM_ID_VREFHI_FINE_1P65V     ] = {0x400CD408, REG_WIDTH_32BITS, 17, 15,     3,   0x1FFF4250, 11,  8},     //VREFHI             0   7  //fine
    [V5_TRIM_ID_VREFHI_COARSE_2P5V    ] = {0x400CD408, REG_WIDTH_32BITS, 13, 10,     7,   0x1FFF4250,  7,  4},     //VREFHI             0   15 //coarse
    [V5_TRIM_ID_VREFHI_FINE_2P5V      ] = {0x400CD408, REG_WIDTH_32BITS, 17, 15,     3,   0x1FFF4250,  3,  0},     //VREFHI             0   7  //fine

    [V5_TRIM_ID_ADC_VCM_COARSE_1P65V  ] = {0x400CD408, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF4260, 24, 21},
    [V5_TRIM_ID_ADC_VCM_FINE_1P65V    ] = {0x400CD408, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF4260, 28, 26},

    [V5_TRIM_ID_ADCA_OFFSET           ] = {0x40000076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF4270,  7,  0},     //ADCA Offset    0   255

    [V5_TRIM_ID_ADCB_OFF_DIR          ] = {0x400CD40C, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF4280,  9,  8},     //ADCB Offset direction    0  3

    [V5_TRIM_ID_ADC_VCM_COARSE_2P475V ] = {0x400CD408, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF4290, 24, 21},
    [V5_TRIM_ID_ADC_VCM_FINE_2P475V   ] = {0x400CD408, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF4290, 28, 26},

    [V5_TRIM_ID_ADCB_OFFSET           ] = {0x40000876, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF42A0,  7,  0},     //ADCB Offset    0   255

    [V5_TRIM_ID_ADCC_OFF_DIR          ] = {0x400CD410, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF42B0,  9,  8},     //ADCC Offset direction    0  3

    [V5_TRIM_ID_ADC_VCM_COARSE_1P25V  ] = {0x400CD408, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF42C0, 24, 21},
    [V5_TRIM_ID_ADC_VCM_FINE_1P25V    ] = {0x400CD408, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF42C0, 28, 26},

    [V5_TRIM_ID_ADCC_OFFSET           ] = {0x40001076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF42D0,  7,  0},     //ADCC Offset    0   255
};

GS32_DRIVER_ROM_DATA_T AnaTrimTable_t anaTrimTable_v6[] = {
    [V6_TRIM_ID_LDO_1V2                ] = {0x400CD308, REG_WIDTH_32BITS,  4,  0,    22,   0x1FFF4200,  4,  0},     //TOP_LDO            0   31
    [V6_TRIM_ID_ANA_LDO                ] = {0x400CD534, REG_WIDTH_32BITS,  2,  0,     0,   0x1FFF4210,  2,  0},     //LDO_ANA            0   7
    [V6_TRIM_ID_OSC1                   ] = {0x400CD314, REG_WIDTH_32BITS, 14,  0,  1397,   0x1FFF4230, 14,  0},     //OSC                0   2047
    [V6_TRIM_ID_OSC2                   ] = {0x400CD314, REG_WIDTH_32BITS, 30, 16,  1397,   0x1FFF4230, 30, 16},     //OSC                0   2047

    [V6_TRIM_ID_ADCA_OFF_DIR           ] = {0x400CD408, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF4240,  9,  8},     //ADCA Offset direction    0  3

    [V6_TRIM_ID_VREFHI_COARSE_2P5V     ] = {0x400CD408, REG_WIDTH_32BITS, 13, 10,     7,   0x1FFF4250, 13, 10},     //VREFHI             0   15 //coarse
    [V6_TRIM_ID_VREFHI_FINE_2P5V       ] = {0x400CD408, REG_WIDTH_32BITS, 17, 15,     3,   0x1FFF4250, 17, 15},     //VREFHI             0   7  //fine
    [V6_TRIM_ID_VREFHI_COARSE_1P65V    ] = {0x400CD408, REG_WIDTH_32BITS, 13, 10,     7,   0x1FFF4254, 13, 10},     //VREFHI             0   15 //coarse
    [V6_TRIM_ID_VREFHI_FINE_1P65V      ] = {0x400CD408, REG_WIDTH_32BITS, 17, 15,     3,   0x1FFF4254, 17, 15},     //VREFHI             0   7  //fine

    [V6_TRIM_ID_ADCA_VCM_COARSE_1P65V  ] = {0x400CD408, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF4260, 24, 21},
    [V6_TRIM_ID_ADCA_VCM_FINE_1P65V    ] = {0x400CD408, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF4260, 28, 26},
    [V6_TRIM_ID_ADCA_VCM_COARSE_2P475V ] = {0x400CD408, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF4264, 24, 21},
    [V6_TRIM_ID_ADCA_VCM_FINE_2P475V   ] = {0x400CD408, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF4264, 28, 26},
    [V6_TRIM_ID_ADCA_VCM_COARSE_1P25V  ] = {0x400CD408, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF4268, 24, 21},
    [V6_TRIM_ID_ADCA_VCM_FINE_1P25V    ] = {0x400CD408, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF4268, 28, 26},
    [V6_TRIM_ID_ADCA_OFFSET            ] = {0x40000076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF4270,  7,  0},     //ADCA Offset    0   255

    [V6_TRIM_ID_ADCB_OFF_DIR           ] = {0x400CD40C, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF4280,  9,  8},     //ADCB Offset direction    0  3

    [V6_TRIM_ID_ADCB_VCM_COARSE_1P65V  ] = {0x400CD40C, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF4290, 24, 21},
    [V6_TRIM_ID_ADCB_VCM_FINE_1P65V    ] = {0x400CD40C, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF4290, 28, 26},
    [V6_TRIM_ID_ADCB_VCM_COARSE_2P475V ] = {0x400CD40C, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF4294, 24, 21},
    [V6_TRIM_ID_ADCB_VCM_FINE_2P475V   ] = {0x400CD40C, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF4294, 28, 26},
    [V6_TRIM_ID_ADCB_VCM_COARSE_1P25V  ] = {0x400CD40C, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF4298, 24, 21},
    [V6_TRIM_ID_ADCB_VCM_FINE_1P25V    ] = {0x400CD40C, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF4298, 28, 26},
    [V6_TRIM_ID_ADCB_OFFSET            ] = {0x40000876, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF42A0,  7,  0},     //ADCB Offset    0   255

    [V6_TRIM_ID_ADCC_OFF_DIR           ] = {0x400CD410, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF42B0,  9,  8},     //ADCC Offset direction    0  3

    [V6_TRIM_ID_ADCC_VCM_COARSE_1P65V  ] = {0x400CD410, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF42C0, 24, 21},
    [V6_TRIM_ID_ADCC_VCM_FINE_1P65V    ] = {0x400CD410, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF42C0, 28, 26},
    [V6_TRIM_ID_ADCC_VCM_COARSE_2P475V ] = {0x400CD410, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF42C4, 24, 21},
    [V6_TRIM_ID_ADCC_VCM_FINE_2P475V   ] = {0x400CD410, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF42C4, 28, 26},
    [V6_TRIM_ID_ADCC_VCM_COARSE_1P25V  ] = {0x400CD410, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF42C8, 24, 21},
    [V6_TRIM_ID_ADCC_VCM_FINE_1P25V    ] = {0x400CD410, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF42C8, 28, 26},
    [V6_TRIM_ID_ADCC_OFFSET            ] = {0x40001076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF42D0,  7,  0},     //ADCC Offset    0   255
};

GS32_DRIVER_ROM_DATA_T AnaTrimTable_t anaTrimTable_v7[] = {
    [V7_TRIM_ID_LDO_1V2                          ] = {0x400CD308, REG_WIDTH_32BITS,  4,  0,    22,   0x1FFF4200,  4,  0},     //TOP_LDO            0   31
    [V7_TRIM_ID_TOP_BOR                          ] = {0x400CD304, REG_WIDTH_32BITS,  6,  4,     2,   0x1FFF4208,  6,  4},     //TOP BOR            0   7
    [V7_TRIM_ID_ANA_LDO                          ] = {0x400CD534, REG_WIDTH_32BITS,  2,  0,     0,   0x1FFF4210,  2,  0},     //LDO_ANA            0   7
    [V7_TRIM_ID_OSC1                             ] = {0x400CD314, REG_WIDTH_32BITS, 14,  0,  1397,   0x1FFF4230, 14,  0},     //OSC                0   2047
    [V7_TRIM_ID_OSC2                             ] = {0x400CD314, REG_WIDTH_32BITS, 30, 16,  1397,   0x1FFF4230, 30, 16},     //OSC                0   2047

    //ADCA
    [V7_TRIM_ID_ADCA_OFF_DIR_VREFHI_3P3V_IN      ] = {0x400CD408, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF4240,  9,  8},     //ADCA Offset direction    0  3
    [V7_TRIM_ID_ADCA_OFF_DIR_VREFHI_2P5V_IN      ] = {0x400CD408, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF4240, 25, 24},     //ADCA Offset direction    0  3
    [V7_TRIM_ID_ADCA_OFF_DIR_VREFHI_3P3V_OUT     ] = {0x400CD408, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF4244,  9,  8},     //ADCA Offset direction    0  3
    [V7_TRIM_ID_ADCA_OFF_DIR_VREFHI_2P5V_OUT     ] = {0x400CD408, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF4244, 25, 24},     //ADCA Offset direction    0  3
    [V7_TRIM_ID_ADCA_OFF_DIR_VREFHI_3P0V_OUT     ] = {0x400CD408, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF4248,  9,  8},     //ADCA Offset direction    0  3

    [V7_TRIM_ID_VREFHI_COARSE_2P5V               ] = {0x400CD408, REG_WIDTH_32BITS, 13, 10,     7,   0x1FFF4250, 13, 10},     //VREFHI             0   15 //coarse
    [V7_TRIM_ID_VREFHI_FINE_2P5V                 ] = {0x400CD408, REG_WIDTH_32BITS, 17, 15,     3,   0x1FFF4250, 17, 15},     //VREFHI             0   7  //fine
    [V7_TRIM_ID_VREFHI_COARSE_1P65V              ] = {0x400CD408, REG_WIDTH_32BITS, 13, 10,     7,   0x1FFF4254, 13, 10},     //VREFHI             0   15 //coarse
    [V7_TRIM_ID_VREFHI_FINE_1P65V                ] = {0x400CD408, REG_WIDTH_32BITS, 17, 15,     3,   0x1FFF4254, 17, 15},     //VREFHI             0   7  //fine

    [V7_TRIM_ID_ADCA_VCM_COARSE_1P65V            ] = {0x400CD408, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF4260, 24, 21},
    [V7_TRIM_ID_ADCA_VCM_FINE_1P65V              ] = {0x400CD408, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF4260, 28, 26},
    [V7_TRIM_ID_ADCA_VCM_COARSE_2P475V           ] = {0x400CD408, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF4264, 24, 21},
    [V7_TRIM_ID_ADCA_VCM_FINE_2P475V             ] = {0x400CD408, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF4264, 28, 26},
    [V7_TRIM_ID_ADCA_VCM_COARSE_1P25V            ] = {0x400CD408, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF4268, 24, 21},
    [V7_TRIM_ID_ADCA_VCM_FINE_1P25V              ] = {0x400CD408, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF4268, 28, 26},
    [V7_TRIM_ID_ADCA_VCM_COARSE_1P5V             ] = {0x400CD408, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF426C, 24, 21},
    [V7_TRIM_ID_ADCA_VCM_FINE_1P5V               ] = {0x400CD408, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF426C, 28, 26},

    [V7_TRIM_ID_ADCA_OFFSET_VREFHI_3P3V_IN       ] = {0x40000076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF4270,  7,  0},     //ADCA Offset    0   255
    [V7_TRIM_ID_ADCA_OFFSET_VREFHI_2P5V_IN       ] = {0x40000076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF4272,  7,  0},     //ADCA Offset    0   255
    [V7_TRIM_ID_ADCA_OFFSET_VREFHI_3P3V_OUT      ] = {0x40000076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF4274,  7,  0},     //ADCA Offset    0   255
    [V7_TRIM_ID_ADCA_OFFSET_VREFHI_2P5V_OUT      ] = {0x40000076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF4276,  7,  0},     //ADCA Offset    0   255
    [V7_TRIM_ID_ADCA_OFFSET_VREFHI_3P0V_OUT      ] = {0x40000076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF4278,  7,  0},     //ADCA Offset    0   255

    //ADCB
    [V7_TRIM_ID_ADCB_OFF_DIR_VREFHI_3P3V_IN      ] = {0x400CD40C, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF4280,  9,  8},     //ADCB Offset direction    0  3
    [V7_TRIM_ID_ADCB_OFF_DIR_VREFHI_2P5V_IN      ] = {0x400CD40C, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF4280, 25, 24},     //ADCB Offset direction    0  3
    [V7_TRIM_ID_ADCB_OFF_DIR_VREFHI_3P3V_OUT     ] = {0x400CD40C, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF4284,  9,  8},     //ADCB Offset direction    0  3
    [V7_TRIM_ID_ADCB_OFF_DIR_VREFHI_2P5V_OUT     ] = {0x400CD40C, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF4284, 25, 24},     //ADCB Offset direction    0  3
    [V7_TRIM_ID_ADCB_OFF_DIR_VREFHI_3P0V_OUT     ] = {0x400CD40C, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF4288,  9,  8},     //ADCB Offset direction    0  3

    [V7_TRIM_ID_ADCB_VCM_COARSE_1P65V            ] = {0x400CD40C, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF4290, 24, 21},
    [V7_TRIM_ID_ADCB_VCM_FINE_1P65V              ] = {0x400CD40C, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF4290, 28, 26},
    [V7_TRIM_ID_ADCB_VCM_COARSE_2P475V           ] = {0x400CD40C, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF4294, 24, 21},
    [V7_TRIM_ID_ADCB_VCM_FINE_2P475V             ] = {0x400CD40C, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF4294, 28, 26},
    [V7_TRIM_ID_ADCB_VCM_COARSE_1P25V            ] = {0x400CD40C, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF4298, 24, 21},
    [V7_TRIM_ID_ADCB_VCM_FINE_1P25V              ] = {0x400CD40C, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF4298, 28, 26},
    [V7_TRIM_ID_ADCB_VCM_COARSE_1P5V             ] = {0x400CD40C, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF429C, 24, 21},
    [V7_TRIM_ID_ADCB_VCM_FINE_1P5V               ] = {0x400CD40C, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF429C, 28, 26},

    [V7_TRIM_ID_ADCB_OFFSET_VREFHI_3P3V_IN       ] = {0x40000876, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF42A0,  7,  0},     //ADCB Offset    0   255
    [V7_TRIM_ID_ADCB_OFFSET_VREFHI_2P5V_IN       ] = {0x40000876, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF42A2,  7,  0},     //ADCB Offset    0   255
    [V7_TRIM_ID_ADCB_OFFSET_VREFHI_3P3V_OUT      ] = {0x40000876, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF42A4,  7,  0},     //ADCB Offset    0   255
    [V7_TRIM_ID_ADCB_OFFSET_VREFHI_2P5V_OUT      ] = {0x40000876, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF42A6,  7,  0},     //ADCB Offset    0   255
    [V7_TRIM_ID_ADCB_OFFSET_VREFHI_3P0V_OUT      ] = {0x40000876, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF42A8,  7,  0},     //ADCB Offset    0   255

    //ADCC
    [V7_TRIM_ID_ADCC_OFF_DIR_VREFHI_3P3V_IN      ] = {0x400CD410, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF42B0,  9,  8},     //ADCC Offset direction    0  3
    [V7_TRIM_ID_ADCC_OFF_DIR_VREFHI_2P5V_IN      ] = {0x400CD410, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF42B0, 25, 24},     //ADCC Offset direction    0  3
    [V7_TRIM_ID_ADCC_OFF_DIR_VREFHI_3P3V_OUT     ] = {0x400CD410, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF42B4,  9,  8},     //ADCC Offset direction    0  3
    [V7_TRIM_ID_ADCC_OFF_DIR_VREFHI_2P5V_OUT     ] = {0x400CD410, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF42B4, 25, 24},     //ADCC Offset direction    0  3
    [V7_TRIM_ID_ADCC_OFF_DIR_VREFHI_3P0V_OUT     ] = {0x400CD410, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF42B8,  9,  8},     //ADCC Offset direction    0  3

    [V7_TRIM_ID_ADCC_VCM_COARSE_1P65V            ] = {0x400CD410, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF42C0, 24, 21},
    [V7_TRIM_ID_ADCC_VCM_FINE_1P65V              ] = {0x400CD410, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF42C0, 28, 26},
    [V7_TRIM_ID_ADCC_VCM_COARSE_2P475V           ] = {0x400CD410, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF42C4, 24, 21},
    [V7_TRIM_ID_ADCC_VCM_FINE_2P475V             ] = {0x400CD410, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF42C4, 28, 26},
    [V7_TRIM_ID_ADCC_VCM_COARSE_1P25V            ] = {0x400CD410, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF42C8, 24, 21},
    [V7_TRIM_ID_ADCC_VCM_FINE_1P25V              ] = {0x400CD410, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF42C8, 28, 26},
    [V7_TRIM_ID_ADCC_VCM_COARSE_1P5V             ] = {0x400CD410, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF42CC, 24, 21},
    [V7_TRIM_ID_ADCC_VCM_FINE_1P5V               ] = {0x400CD410, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF42CC, 28, 26},

    [V7_TRIM_ID_ADCC_OFFSET_VREFHI_3P3V_IN       ] = {0x40001076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF42D0,  7,  0},     //ADCC Offset    0   255
    [V7_TRIM_ID_ADCC_OFFSET_VREFHI_2P5V_IN       ] = {0x40001076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF42D2,  7,  0},     //ADCC Offset    0   255
    [V7_TRIM_ID_ADCC_OFFSET_VREFHI_3P3V_OUT      ] = {0x40001076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF42D4,  7,  0},     //ADCC Offset    0   255
    [V7_TRIM_ID_ADCC_OFFSET_VREFHI_2P5V_OUT      ] = {0x40001076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF42D6,  7,  0},     //ADCC Offset    0   255
    [V7_TRIM_ID_ADCC_OFFSET_VREFHI_3P0V_OUT      ] = {0x40001076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF42D8,  7,  0},     //ADCC Offset    0   255

    //Digital gain offset
    [V7_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_3P3V_IN   ] = {0x40000248, REG_WIDTH_32BITS,  9,  0,     0,   0x1FFF42E0,  9,  0},
    [V7_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_3P3V_IN ] = {0x40000244, REG_WIDTH_32BITS, 11,  0, 0x800,   0x1FFF42E0, 27, 16},
    [V7_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_2P5V_IN   ] = {0x40000248, REG_WIDTH_32BITS,  9,  0,     0,   0x1FFF42E4,  9,  0},
    [V7_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_2P5V_IN ] = {0x40000244, REG_WIDTH_32BITS, 11,  0, 0x800,   0x1FFF42E4, 27, 16},
    [V7_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_3P3V_OUT  ] = {0x40000248, REG_WIDTH_32BITS,  9,  0,     0,   0x1FFF42E8,  9,  0},
    [V7_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_3P3V_OUT] = {0x40000244, REG_WIDTH_32BITS, 11,  0, 0x800,   0x1FFF42E8, 27, 16},
    [V7_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_2P5V_OUT  ] = {0x40000248, REG_WIDTH_32BITS,  9,  0,     0,   0x1FFF42EC,  9,  0},
    [V7_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_2P5V_OUT] = {0x40000244, REG_WIDTH_32BITS, 11,  0, 0x800,   0x1FFF42EC, 27, 16},
    [V7_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_3P0V_OUT  ] = {0x40000248, REG_WIDTH_32BITS,  9,  0,     0,   0x1FFF42F0,  9,  0},
    [V7_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_3P0V_OUT] = {0x40000244, REG_WIDTH_32BITS, 11,  0, 0x800,   0x1FFF42F0, 27, 16},

    [V7_TRIM_ID_ADCB_DIG_OFFSET_VREFHI_3P3V_IN   ] = {0x40000A48, REG_WIDTH_32BITS,  9,  0,     0,   0x1FFF4300,  9,  0},
    [V7_TRIM_ID_ADCB_DIG_GAIN_ERR_VREFHI_3P3V_IN ] = {0x40000A44, REG_WIDTH_32BITS, 11,  0, 0x800,   0x1FFF4300, 27, 16},
    [V7_TRIM_ID_ADCB_DIG_OFFSET_VREFHI_2P5V_IN   ] = {0x40000A48, REG_WIDTH_32BITS,  9,  0,     0,   0x1FFF4304,  9,  0},
    [V7_TRIM_ID_ADCB_DIG_GAIN_ERR_VREFHI_2P5V_IN ] = {0x40000A44, REG_WIDTH_32BITS, 11,  0, 0x800,   0x1FFF4304, 27, 16},
    [V7_TRIM_ID_ADCB_DIG_OFFSET_VREFHI_3P3V_OUT  ] = {0x40000A48, REG_WIDTH_32BITS,  9,  0,     0,   0x1FFF4308,  9,  0},
    [V7_TRIM_ID_ADCB_DIG_GAIN_ERR_VREFHI_3P3V_OUT] = {0x40000A44, REG_WIDTH_32BITS, 11,  0, 0x800,   0x1FFF4308, 27, 16},
    [V7_TRIM_ID_ADCB_DIG_OFFSET_VREFHI_2P5V_OUT  ] = {0x40000A48, REG_WIDTH_32BITS,  9,  0,     0,   0x1FFF430C,  9,  0},
    [V7_TRIM_ID_ADCB_DIG_GAIN_ERR_VREFHI_2P5V_OUT] = {0x40000A44, REG_WIDTH_32BITS, 11,  0, 0x800,   0x1FFF430C, 27, 16},
    [V7_TRIM_ID_ADCB_DIG_OFFSET_VREFHI_3P0V_OUT  ] = {0x40000A48, REG_WIDTH_32BITS,  9,  0,     0,   0x1FFF4310,  9,  0},
    [V7_TRIM_ID_ADCB_DIG_GAIN_ERR_VREFHI_3P0V_OUT] = {0x40000A44, REG_WIDTH_32BITS, 11,  0, 0x800,   0x1FFF4310, 27, 16},

    [V7_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_3P3V_IN   ] = {0x40001248, REG_WIDTH_32BITS,  9,  0,     0,   0x1FFF4320,  9,  0},
    [V7_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_3P3V_IN ] = {0x40001244, REG_WIDTH_32BITS, 11,  0, 0x800,   0x1FFF4320, 27, 16},
    [V7_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_2P5V_IN   ] = {0x40001248, REG_WIDTH_32BITS,  9,  0,     0,   0x1FFF4324,  9,  0},
    [V7_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_2P5V_IN ] = {0x40001244, REG_WIDTH_32BITS, 11,  0, 0x800,   0x1FFF4324, 27, 16},
    [V7_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_3P3V_OUT  ] = {0x40001248, REG_WIDTH_32BITS,  9,  0,     0,   0x1FFF4328,  9,  0},
    [V7_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_3P3V_OUT] = {0x40001244, REG_WIDTH_32BITS, 11,  0, 0x800,   0x1FFF4328, 27, 16},
    [V7_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_2P5V_OUT  ] = {0x40001248, REG_WIDTH_32BITS,  9,  0,     0,   0x1FFF432C,  9,  0},
    [V7_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_2P5V_OUT] = {0x40001244, REG_WIDTH_32BITS, 11,  0, 0x800,   0x1FFF432C, 27, 16},
    [V7_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_3P0V_OUT  ] = {0x40001248, REG_WIDTH_32BITS,  9,  0,     0,   0x1FFF4330,  9,  0},
    [V7_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_3P0V_OUT] = {0x40001244, REG_WIDTH_32BITS, 11,  0, 0x800,   0x1FFF4330, 27, 16},
};

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

GS32_DRIVER_INIT_FUNC_T void
ASysCtl_initConfig(void)
{
    ASysCtl_setTopBgTestMuxSel(2);
	ASysCtl_setTopBgChopEn(1);

	ASysCtl_setEflashBorTestEn(1);

	//
	//ASysCtl init for CMPSS
	//
	ASysCtl_setAnaClkTrim(0x49);

	ASysCtl_setL2HCfg(0x01);

	ASysCtl_setAdcTopSpare(0X777B5F);

	//
	//Set ADC to external reference mode by default
	//
	ASysCtl_setAnalogReferenceExternal(0x01);

	//HWREG(0x400008E0) = 0X5204;			//adca_inl_trim_31to0
	//ADC_setINLTrim(ADCA_BASE);
	//ADC_setINLTrim(ADCB_BASE);
	//ADC_setINLTrim(ADCC_BASE);

	//
	//Configure ADC trim value for external reference mode, they can be reconfigured later
	//
//	ASysCtl_setAdcSpare(0, 0x0E625457);     //ADCA spare
//	ASysCtl_setAdcSpare(1, 0x0E625457);     //ADCB spare
//	ASysCtl_setAdcSpare(2, 0x0E625457);     //ADCC spare

    //Let ADC leave reset state
	ASysCtl_setAdcReset(0x00);              //release ADCA/B/C


    //Avoid user trigger default SOC of too narrow acqps
    for(int i = 0; i < 16 ; i++)
    {
    	ADC_setupSOC(ADCA_BASE, i, ADC_TRIGGER_SW_ONLY, ADC_CH_ADCIN0, 10U);
    	ADC_setupSOC(ADCB_BASE, i, ADC_TRIGGER_SW_ONLY, ADC_CH_ADCIN0, 10U);
    	ADC_setupSOC(ADCC_BASE, i, ADC_TRIGGER_SW_ONLY, ADC_CH_ADCIN0, 10U);
    }

	HWREGH(ADCA_BASE + ADC_O_CTL1) |= ADC_CTL1_ADCPWDNZ; 	//B C VrefHi came from A

	//
	//ASysCtl init for CMPSS
	//
    //HWREG(0x400CD4B4) = 0x0F000000;
    ASysCtl_setCmpSpare(0x0FFFFFFF);        //enable hysteresis on all channels
    //HWREG(0x400CD41C) = 0x1;
    ASysCtl_setCmpEnable(0x01);
    //HWREG(0x400CD420) = 0x0;
    ASysCtl_setCmpReset(0x0);
    //set all LP source to external input by default
    CMPSS_setCmpLpSrc(CMPSS1_BASE, 1);
    CMPSS_setCmpLpSrc(CMPSS2_BASE, 1);
    CMPSS_setCmpLpSrc(CMPSS3_BASE, 1);
    CMPSS_setCmpLpSrc(CMPSS4_BASE, 1);
    CMPSS_setCmpLpSrc(CMPSS5_BASE, 1);
    CMPSS_setCmpLpSrc(CMPSS6_BASE, 1);
    CMPSS_setCmpLpSrc(CMPSS7_BASE, 1);

	//
	//ASysCtl init for DAC
	//
    //HWREG(ANALOGSUBSYS_BASE + 0X4B8) = 0X1;
    ASysCtl_setDacTopEnable(0x01);
	//HWREG(0x40080002) = 0X1; //select VREFHI as DAC reference
	//HWREG(ANALOGSUBSYS_BASE + 0X4B8) = 0X1;
    ASysCtl_setDacTopEnable(0x01);
	//HWREG(ANALOGSUBSYS_BASE + 0x4BC) = 0X0;
    ASysCtl_setDacReset(0x00);
}

GS32_DRIVER_ASYSCTL_FUNC_T static void
AsysCtl_trimPeri(AnaTrimTable_t *trim)
{
    uint32_t shift, mask;
    shift = trim->startbit;
    mask = (1<<(trim->endbit - trim->startbit + 1)) - 1;

    if (HWREG(trim->efuseAddr) == 0xFFFFFFFF) {     //do nothing, using register's default value
//        if (trim->regWidth == 16) {
//            HWREGH(trim->regAddr) = (HWREGH(trim->regAddr) & ~(mask << shift)) | (trim->defaultTrimVal << shift);
//        } else {
//            HWREG(trim->regAddr) = (HWREG(trim->regAddr) & ~(mask << shift)) | (trim->defaultTrimVal << shift);
//        }
    } else {
        uint32_t efuse_shift, efuse_mask;
        efuse_shift = trim->efuseStartbit;
        efuse_mask = (1<<(trim->efuseEndbit - trim->efuseStartbit + 1)) - 1;

        if(trim->regWidth == REG_WIDTH_16BITS)
            HWREGH(trim->regAddr) = (HWREGH(trim->regAddr) & ~(mask << shift)) | (((HWREGH(trim->efuseAddr) >> efuse_shift) & efuse_mask) << shift);
        else
            HWREG(trim->regAddr) = (HWREG(trim->regAddr) & ~(mask << shift)) | (((HWREG(trim->efuseAddr) >> efuse_shift) & efuse_mask) << shift);
    }
}

GS32_DRIVER_ASYSCTL_FUNC_T static
void AsysCtl_trimPeri_v0(AnaTrimPeriId_v0_t id)
{
    AsysCtl_trimPeri(&anaTrimTable_v0[id]);
}

GS32_DRIVER_ASYSCTL_FUNC_T static
void AsysCtl_trimPeri_v5(AnaTrimPeriId_v5_t id)
{
    AsysCtl_trimPeri(&anaTrimTable_v5[id]);
}

GS32_DRIVER_ASYSCTL_FUNC_T static
void AsysCtl_trimPeri_v6(AnaTrimPeriId_v6_t id)
{
    AsysCtl_trimPeri(&anaTrimTable_v6[id]);
}

GS32_DRIVER_ASYSCTL_FUNC_T static
void AsysCtl_trimPeri_v7(AnaTrimPeriId_v6_t id)
{
    AsysCtl_trimPeri(&anaTrimTable_v7[id]);
}

GS32_DRIVER_ASYSCTL_FUNC_T void
AsysCtl_trimBor(void)
{
    //SysCtl_setBorLvmonDisable(1);

    AsysCtl_trimPeri_v7(V7_TRIM_ID_TOP_BOR);

    //SysCtl_setBorLvmonDisable(0);
}

GS32_DRIVER_INIT_FUNC_T void
ASysCtl_trim(void)
{
    uint32_t version = *((uint32_t *)ANALOG_TRIM_EFUSE_VERSION_ADDR);
    uint32_t id;

    SysCtl_setLdoTestMode(2);
    SysCtl_setLdoChopEn(1);     //enable internal vcore ldo chop

    if (version == 0xFFFFFFFF) {
        if (*((uint32_t *)anaTrimTable_v0[V0_TRIM_ID_LDO_1V2].efuseAddr) != 0xFFFFFFFF) {
            for (id=0; id<V0_TRIM_ID_MAX; id+=1) {
                AsysCtl_trimPeri_v0(id);
            }
        }
    } else if (version == 5){
        for (id=0; id<V5_TRIM_ID_MAX; id+=1) {
            AsysCtl_trimPeri_v5(id);
        }
    } else if (version == 6){
        AsysCtl_trimPeri_v6(V6_TRIM_ID_LDO_1V2);
        AsysCtl_trimPeri_v6(V6_TRIM_ID_ANA_LDO);
        AsysCtl_trimPeri_v6(V6_TRIM_ID_OSC1);
        AsysCtl_trimPeri_v6(V6_TRIM_ID_OSC2);
    } else if (version >= 7) {
        AsysCtl_trimPeri_v7(V7_TRIM_ID_LDO_1V2);
        AsysCtl_trimPeri_v7(V7_TRIM_ID_ANA_LDO);
        AsysCtl_trimPeri_v7(V7_TRIM_ID_OSC1);
        AsysCtl_trimPeri_v7(V7_TRIM_ID_OSC2);

        AsysCtl_trimBor();
    }

#ifdef TOP_LDO_CAL_MODE
    uint16 top_ldo_val = 0;
    top_ldo_val = HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG)&0x1f;

    for(uint16_t data = top_ldo_val;data <= 0x1f;data++)
    {
    	HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) = (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) &~(0x1f)) | data;

    }
#endif
}

GS32_DRIVER_ASYSCTL_FUNC_T static void
AsysCtl_trimAdc_v0(uint32_t refMode, uint32_t refVolt)
{
    if(refMode == ADC_REFERENCE_INTERNAL)
    {
        if (refVolt == ADC_REFERENCE_3_3V) {
            AsysCtl_trimPeri_v0(V0_TRIM_ID_VREFHI_COARSE_3V3);    //bit17:15 bit13:10 VREFHI trim for 1.65V mode
            AsysCtl_trimPeri_v0(V0_TRIM_ID_VREFHI_FINE_3V3);
        } else if (refVolt == ADC_REFERENCE_2_5V) {
            AsysCtl_trimPeri_v0(V0_TRIM_ID_VREFHI_COARSE_2V5);    //bit17:15 bit13:10 VREFHI trim for 2.5V mode
            AsysCtl_trimPeri_v0(V0_TRIM_ID_VREFHI_FINE_2V5);
        }
    }
    else
    {
        AsysCtl_trimPeri_v0(V0_TRIM_ID_ADCA_VCM_EXT);     //bit28:26, VCM trim for external mode
#if defined(ADCB_BASE)
        AsysCtl_trimPeri_v0(V0_TRIM_ID_ADCB_VCM_EXT);
#endif
        AsysCtl_trimPeri_v0(V0_TRIM_ID_ADCC_VCM_EXT);
    }


    AsysCtl_trimPeri_v0(V0_TRIM_ID_ADCA_VCM_INT);         //bit24:21, VCM trim for internal mode
#if defined(ADCB_BASE)
    AsysCtl_trimPeri_v0(V0_TRIM_ID_ADCB_VCM_INT);
#endif
    AsysCtl_trimPeri_v0(V0_TRIM_ID_ADCC_VCM_INT);

    AsysCtl_trimPeri_v0(V0_TRIM_ID_ADCA_OFFSET);            //Individual ADC OffsetTrim register
#if defined(ADCB_BASE)
    AsysCtl_trimPeri_v0(V0_TRIM_ID_ADCB_OFFSET);
#endif
    AsysCtl_trimPeri_v0(V0_TRIM_ID_ADCC_OFFSET);

    AsysCtl_trimPeri_v0(V0_TRIM_ID_ADCA_OFF_DIR);            //bit9:8, trim offset direction
#if defined(ADCB_BASE)
    AsysCtl_trimPeri_v0(V0_TRIM_ID_ADCB_OFF_DIR);
#endif
    AsysCtl_trimPeri_v0(V0_TRIM_ID_ADCC_OFF_DIR);
}

GS32_DRIVER_ASYSCTL_FUNC_T static void
AsysCtl_trimAdc_v5(uint32_t refMode, uint32_t refVolt)
{
    if(refMode == ADC_REFERENCE_INTERNAL)
    {
        if (refVolt == ADC_REFERENCE_3_3V) {
            AsysCtl_trimPeri_v5(V5_TRIM_ID_VREFHI_COARSE_1P65V);    //bit17:15 bit13:10 VREFHI trim for 1.65V mode
            AsysCtl_trimPeri_v5(V5_TRIM_ID_VREFHI_FINE_1P65V);

            AsysCtl_trimPeri_v5(V5_TRIM_ID_ADC_VCM_COARSE_2P475V);
            AsysCtl_trimPeri_v5(V5_TRIM_ID_ADC_VCM_FINE_2P475V);
        } else if (refVolt == ADC_REFERENCE_2_5V) {
            AsysCtl_trimPeri_v5(V5_TRIM_ID_VREFHI_COARSE_2P5V);    //bit17:15 bit13:10 VREFHI trim for 2.5V mode
            AsysCtl_trimPeri_v5(V5_TRIM_ID_VREFHI_FINE_2P5V);

            AsysCtl_trimPeri_v5(V5_TRIM_ID_ADC_VCM_COARSE_1P25V);
            AsysCtl_trimPeri_v5(V5_TRIM_ID_ADC_VCM_FINE_1P25V);
        }
    }
    else
    {
        if (refVolt == ADC_REFERENCE_3_3V) {
            AsysCtl_trimPeri_v5(V5_TRIM_ID_ADC_VCM_COARSE_1P65V);
            AsysCtl_trimPeri_v5(V5_TRIM_ID_ADC_VCM_FINE_1P65V);
        } else if (refVolt == ADC_REFERENCE_2_5V) {
            AsysCtl_trimPeri_v5(V5_TRIM_ID_ADC_VCM_COARSE_1P25V);
            AsysCtl_trimPeri_v5(V5_TRIM_ID_ADC_VCM_FINE_1P25V);
        }
    }

    AsysCtl_trimPeri_v5(V5_TRIM_ID_ADCA_OFFSET);            //Individual ADC OffsetTrim register
#if defined(ADCB_BASE)
    AsysCtl_trimPeri_v5(V5_TRIM_ID_ADCB_OFFSET);
#endif
    AsysCtl_trimPeri_v5(V5_TRIM_ID_ADCC_OFFSET);

    AsysCtl_trimPeri_v5(V5_TRIM_ID_ADCA_OFF_DIR);            //bit9:8, trim offset direction
#if defined(ADCB_BASE)
    AsysCtl_trimPeri_v5(V5_TRIM_ID_ADCB_OFF_DIR);
#endif
    AsysCtl_trimPeri_v5(V5_TRIM_ID_ADCC_OFF_DIR);
}

GS32_DRIVER_ASYSCTL_FUNC_T static void
AsysCtl_trimAdc_v6(uint32_t adcBase, uint32_t refMode, uint32_t refVolt)
{
    if(refMode == ADC_REFERENCE_INTERNAL)
    {
        if (refVolt == ADC_REFERENCE_3_3V) {
            AsysCtl_trimPeri_v6(V6_TRIM_ID_VREFHI_COARSE_1P65V);
            AsysCtl_trimPeri_v6(V6_TRIM_ID_VREFHI_FINE_1P65V);
        } else if (refVolt == ADC_REFERENCE_2_5V) {
            AsysCtl_trimPeri_v6(V6_TRIM_ID_VREFHI_COARSE_2P5V);
            AsysCtl_trimPeri_v6(V6_TRIM_ID_VREFHI_FINE_2P5V);
        }
    }

    if (refVolt == ADC_REFERENCE_2_5V) {
        switch (adcBase) {
            case ADCA_BASE:
                AsysCtl_trimPeri_v6(V6_TRIM_ID_ADCA_VCM_COARSE_1P25V);
                AsysCtl_trimPeri_v6(V6_TRIM_ID_ADCA_VCM_FINE_1P25V);
                break;
            case ADCB_BASE:
                AsysCtl_trimPeri_v6(V6_TRIM_ID_ADCB_VCM_COARSE_1P25V);
                AsysCtl_trimPeri_v6(V6_TRIM_ID_ADCB_VCM_FINE_1P25V);
                break;
            case ADCC_BASE:
                AsysCtl_trimPeri_v6(V6_TRIM_ID_ADCC_VCM_COARSE_1P25V);
                AsysCtl_trimPeri_v6(V6_TRIM_ID_ADCC_VCM_FINE_1P25V);
                break;
        }
    } else {
        if(refMode == ADC_REFERENCE_INTERNAL) {
            switch (adcBase) {
                case ADCA_BASE:
                    AsysCtl_trimPeri_v6(V6_TRIM_ID_ADCA_VCM_COARSE_2P475V);
                    AsysCtl_trimPeri_v6(V6_TRIM_ID_ADCA_VCM_FINE_2P475V);
                    break;
                case ADCB_BASE:
                    AsysCtl_trimPeri_v6(V6_TRIM_ID_ADCB_VCM_COARSE_2P475V);
                    AsysCtl_trimPeri_v6(V6_TRIM_ID_ADCB_VCM_FINE_2P475V);
                    break;
                case ADCC_BASE:
                    AsysCtl_trimPeri_v6(V6_TRIM_ID_ADCC_VCM_COARSE_2P475V);
                    AsysCtl_trimPeri_v6(V6_TRIM_ID_ADCC_VCM_FINE_2P475V);
                    break;
            }
        } else {
            switch (adcBase) {
                case ADCA_BASE:
                    AsysCtl_trimPeri_v6(V6_TRIM_ID_ADCA_VCM_COARSE_1P65V);
                    AsysCtl_trimPeri_v6(V6_TRIM_ID_ADCA_VCM_FINE_1P65V);
                    break;
                case ADCB_BASE:
                    AsysCtl_trimPeri_v6(V6_TRIM_ID_ADCB_VCM_COARSE_1P65V);
                    AsysCtl_trimPeri_v6(V6_TRIM_ID_ADCB_VCM_FINE_1P65V);
                    break;
                case ADCC_BASE:
                    AsysCtl_trimPeri_v6(V6_TRIM_ID_ADCC_VCM_COARSE_1P65V);
                    AsysCtl_trimPeri_v6(V6_TRIM_ID_ADCC_VCM_FINE_1P65V);
                    break;
            }
        }
    }

    switch (adcBase) {
        case ADCA_BASE:
            AsysCtl_trimPeri_v6(V6_TRIM_ID_ADCA_OFFSET);            //Individual ADC OffsetTrim register
            AsysCtl_trimPeri_v6(V6_TRIM_ID_ADCA_OFF_DIR);
            break;
        case ADCB_BASE:
            AsysCtl_trimPeri_v6(V6_TRIM_ID_ADCB_OFFSET);            //Individual ADC OffsetTrim register
            AsysCtl_trimPeri_v6(V6_TRIM_ID_ADCB_OFF_DIR);
            break;
        case ADCC_BASE:
            AsysCtl_trimPeri_v6(V6_TRIM_ID_ADCC_OFFSET);            //Individual ADC OffsetTrim register
            AsysCtl_trimPeri_v6(V6_TRIM_ID_ADCC_OFF_DIR);
            break;
    }
}

GS32_DRIVER_ASYSCTL_FUNC_T static void
AsysCtl_trimAdcVref_v7(uint32_t adcBase, uint32_t refMode, uint32_t refVolt)
{
    if(refMode == ADC_REFERENCE_INTERNAL) {
        if (refVolt == ADC_REFERENCE_3_3V) {
            AsysCtl_trimPeri_v7(V7_TRIM_ID_VREFHI_COARSE_1P65V);
            AsysCtl_trimPeri_v7(V7_TRIM_ID_VREFHI_FINE_1P65V);
        } else if (refVolt == ADC_REFERENCE_2_5V) {
            AsysCtl_trimPeri_v7(V7_TRIM_ID_VREFHI_COARSE_2P5V);
            AsysCtl_trimPeri_v7(V7_TRIM_ID_VREFHI_FINE_2P5V);
        }
    }
}

GS32_DRIVER_ASYSCTL_FUNC_T static void
AsysCtl_trimAdcVcm_v7(uint32_t adcBase, uint32_t refMode, uint32_t refVolt)
{
    if (refVolt == ADC_REFERENCE_2_5V) {     //internal external 2.5V
        switch (adcBase) {
            case ADCA_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_VCM_COARSE_1P25V);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_VCM_FINE_1P25V);
                break;
            case ADCB_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_VCM_COARSE_1P25V);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_VCM_FINE_1P25V);
                break;
            case ADCC_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_VCM_COARSE_1P25V);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_VCM_FINE_1P25V);
                break;
        }
    } else if (refVolt == ADC_REFERENCE_3_0V) {  //external 3.0V
         switch (adcBase) {
             case ADCA_BASE:
                 AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_VCM_COARSE_1P5V);
                 AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_VCM_FINE_1P5V);
             break;
             case ADCB_BASE:
                 AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_VCM_COARSE_1P5V);
                 AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_VCM_FINE_1P5V);
             break;
             case ADCC_BASE:
                 AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_VCM_COARSE_1P5V);
                 AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_VCM_FINE_1P5V);
             break;
         }
    } else {
        if(refMode == ADC_REFERENCE_INTERNAL) {  //internal 3.3V
            switch (adcBase) {
                case ADCA_BASE:
                    AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_VCM_COARSE_2P475V);
                    AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_VCM_FINE_2P475V);
                    break;
                case ADCB_BASE:
                    AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_VCM_COARSE_2P475V);
                    AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_VCM_FINE_2P475V);
                    break;
                case ADCC_BASE:
                    AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_VCM_COARSE_2P475V);
                    AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_VCM_FINE_2P475V);
                    break;
            }
        } else {
            switch (adcBase) {       //external 3.3V
                case ADCA_BASE:
                    AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_VCM_COARSE_1P65V);
                    AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_VCM_FINE_1P65V);
                    break;
                case ADCB_BASE:
                    AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_VCM_COARSE_1P65V);
                    AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_VCM_FINE_1P65V);
                    break;
                case ADCC_BASE:
                    AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_VCM_COARSE_1P65V);
                    AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_VCM_FINE_1P65V);
                    break;
            }
        }
    }
}

GS32_DRIVER_ASYSCTL_FUNC_T static void
AsysCtl_trimAdcOffset_v7(uint32_t adcBase, uint32_t refMode, uint32_t refVolt)
{
    if(refMode == ADC_REFERENCE_INTERNAL && refVolt == ADC_REFERENCE_2_5V) {
        switch (adcBase) {
            case ADCA_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_OFF_DIR_VREFHI_2P5V_IN);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_OFFSET_VREFHI_2P5V_IN);
                break;
            case ADCB_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_OFF_DIR_VREFHI_2P5V_IN);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_OFFSET_VREFHI_2P5V_IN);
                break;
            case ADCC_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_OFF_DIR_VREFHI_2P5V_IN);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_OFFSET_VREFHI_2P5V_IN);
                break;
        }
    } else if(refMode == ADC_REFERENCE_INTERNAL && refVolt == ADC_REFERENCE_3_3V) {
        switch (adcBase) {
            case ADCA_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_OFF_DIR_VREFHI_3P3V_IN);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_OFFSET_VREFHI_3P3V_IN);
                break;
            case ADCB_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_OFF_DIR_VREFHI_3P3V_IN);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_OFFSET_VREFHI_3P3V_IN);
                break;
            case ADCC_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_OFF_DIR_VREFHI_3P3V_IN);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_OFFSET_VREFHI_3P3V_IN);
                break;
        }
    } if(refMode == ADC_REFERENCE_EXTERNAL && refVolt == ADC_REFERENCE_2_5V) {
        switch (adcBase) {
            case ADCA_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_OFF_DIR_VREFHI_2P5V_OUT);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_OFFSET_VREFHI_2P5V_OUT);
                break;
            case ADCB_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_OFF_DIR_VREFHI_2P5V_OUT);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_OFFSET_VREFHI_2P5V_OUT);
                break;
            case ADCC_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_OFF_DIR_VREFHI_2P5V_OUT);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_OFFSET_VREFHI_2P5V_OUT);
                break;
        }
    } else if(refMode == ADC_REFERENCE_EXTERNAL && refVolt == ADC_REFERENCE_3_0V) {
        switch (adcBase) {
            case ADCA_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_OFF_DIR_VREFHI_3P0V_OUT);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_OFFSET_VREFHI_3P0V_OUT);
                break;
            case ADCB_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_OFF_DIR_VREFHI_3P0V_OUT);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_OFFSET_VREFHI_3P0V_OUT);
                break;
            case ADCC_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_OFF_DIR_VREFHI_3P0V_OUT);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_OFFSET_VREFHI_3P0V_OUT);
                break;
        }
    } else if(refMode == ADC_REFERENCE_EXTERNAL && refVolt == ADC_REFERENCE_3_3V) {
        switch (adcBase) {
            case ADCA_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_OFF_DIR_VREFHI_3P3V_OUT);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_OFFSET_VREFHI_3P3V_OUT);
                break;
            case ADCB_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_OFF_DIR_VREFHI_3P3V_OUT);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_OFFSET_VREFHI_3P3V_OUT);
                break;
            case ADCC_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_OFF_DIR_VREFHI_3P3V_OUT);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_OFFSET_VREFHI_3P3V_OUT);
                break;
        }
    }
}

GS32_DRIVER_ASYSCTL_FUNC_T static void
AsysCtl_trimAdcDigOffsetGain_v7(uint32_t adcBase, uint32_t refMode, uint32_t refVolt)
{
    //enable digital trim
    HWREG(adcBase + 0x248) |= 1<<16;        //adc_pre_pro_en

    if(refMode == ADC_REFERENCE_INTERNAL && refVolt == ADC_REFERENCE_2_5V) {
        switch (adcBase) {
            case ADCA_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_2P5V_IN);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_2P5V_IN);
                break;
            case ADCB_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_DIG_OFFSET_VREFHI_2P5V_IN);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_DIG_GAIN_ERR_VREFHI_2P5V_IN);
                break;
            case ADCC_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_2P5V_IN);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_2P5V_IN);
                break;
        }
    } else if(refMode == ADC_REFERENCE_INTERNAL && refVolt == ADC_REFERENCE_3_3V) {
        switch (adcBase) {
            case ADCA_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_3P3V_IN);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_3P3V_IN);
                break;
            case ADCB_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_DIG_OFFSET_VREFHI_3P3V_IN);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_DIG_GAIN_ERR_VREFHI_3P3V_IN);
                break;
            case ADCC_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_3P3V_IN);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_3P3V_IN);
                break;
        }
    } if(refMode == ADC_REFERENCE_EXTERNAL && refVolt == ADC_REFERENCE_2_5V) {
        switch (adcBase) {
            case ADCA_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_2P5V_OUT);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_2P5V_OUT);
                break;
            case ADCB_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_DIG_OFFSET_VREFHI_2P5V_OUT);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_DIG_GAIN_ERR_VREFHI_2P5V_OUT);
                break;
            case ADCC_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_2P5V_OUT);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_2P5V_OUT);
                break;
        }
    } else if(refMode == ADC_REFERENCE_EXTERNAL && refVolt == ADC_REFERENCE_3_0V) {
        switch (adcBase) {
            case ADCA_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_3P0V_OUT);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_3P0V_OUT);
                break;
            case ADCB_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_DIG_OFFSET_VREFHI_3P0V_OUT);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_DIG_GAIN_ERR_VREFHI_3P0V_OUT);
                break;
            case ADCC_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_3P0V_OUT);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_3P0V_OUT);
                break;
        }
    } else if(refMode == ADC_REFERENCE_EXTERNAL && refVolt == ADC_REFERENCE_3_3V) {
        switch (adcBase) {
            case ADCA_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_3P3V_OUT);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_3P3V_OUT);
                break;
            case ADCB_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_DIG_OFFSET_VREFHI_3P3V_OUT);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCB_DIG_GAIN_ERR_VREFHI_3P3V_OUT);
                break;
            case ADCC_BASE:
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_3P3V_OUT);
                AsysCtl_trimPeri_v7(V7_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_3P3V_OUT);
                break;
        }
    }
}

GS32_DRIVER_ASYSCTL_FUNC_T static void
AsysCtl_trimAdc_v7(uint32_t adcBase, uint32_t refMode, uint32_t refVolt)
{
    AsysCtl_trimAdcVref_v7(adcBase, refMode, refVolt);
    AsysCtl_trimAdcVcm_v7(adcBase, refMode, refVolt);
    AsysCtl_trimAdcOffset_v7(adcBase, refMode, refVolt);
    AsysCtl_trimAdcDigOffsetGain_v7(adcBase, refMode, refVolt);
}

GS32_DRIVER_ASYSCTL_FUNC_T void
AsysCtl_trimAdc(uint32_t adcBase, uint32_t refMode, uint32_t refVolt)
{
    uint32_t version = *((uint32_t *)ANALOG_TRIM_EFUSE_VERSION_ADDR);

    if (version == 0xFFFFFFFF) {
        if (*((uint32_t *)anaTrimTable_v0[V0_TRIM_ID_LDO_1V2].efuseAddr) != 0xFFFFFFFF) {
            AsysCtl_trimAdc_v0(refMode, refVolt);
        }
    } else if (version == 5) {
        AsysCtl_trimAdc_v5(refMode, refVolt);
    } else if (version == 6) {
        AsysCtl_trimAdc_v6(adcBase, refMode, refVolt);
    } else if (version >= 7) {
        AsysCtl_trimAdc_v7(adcBase, refMode, refVolt);
    }
}
#endif

#ifdef __cplusplus
}
#endif
