/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    asysctl_v12.h
*   @brief   analog system top level configuration
*            do NOT include this file directly, include "sysctl.h" instead
*
*/
/**
 * commit history
 * 20240627,WenGuangYong,Add register bit-field descriptions and API for 1.2 CS asysctl
 * 20240708.WenGuangYong,refine ASysCtl_setAnalogReference API
 * 20241218,Hezhiyuan,Modify comments
*/
#ifndef DEVICE_DRIVERLIB_SYSCTL_ANA_V12_H_
#define DEVICE_DRIVERLIB_SYSCTL_ANA_V12_H_

#ifdef __cplusplus
extern "C"{
#endif


/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "inc/hw_types.h"
#include "gs32_version.h"

#if IS_GS32F00xx(0x12)
#include "debug.h"
#include "inc/hw_types.h"
#include "inc/hw_memmap.h"
#include "inc/hw_asysctl_v12.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/*****************************************************************************
 * Analog pin configuration
 * The following MACRO is input parameter for GPIO_setAnalogPin(...) or SysCtl_setAnalogPin(...)
 * bit16~19 is ASEL offset in asysctl, bit0~3 is ASEL value.
 * User can add DEVICE_TYPE_0049_100PIN or DEVICE_TYPE_0039_100PIN MACRO define in the project settings
 *****************************************************************************/
#if (GS32_PART_NUM==0x0049)
//A0_2
#define AIO_A0_DACA_OUT    0x00000003
#else
#define AIO_A0_DACA_OUT    0x00000003
#define AIO_A0_CMP3_HP2    0x00000004
#define AIO_A0_CMP3_LP2    0x00000005
#endif


#define AIO_A1_DACB_OUT    0x00010002
#define AIO_A1_CMP1_HP4    0x00010003
#define AIO_A1_CMP1_LP4    0x00010004

#define AIO_A2_CMP1_HP0    0x00020003
#define AIO_A2_CMP1_LP0    0x00020004

#if (GS32_PART_NUM==0x0039)
//A3_2
#define AIO_A3_CMP3_HP5    0x00030003
#define AIO_A3_CMP3_LP5    0x00030004
#define AIO_A3_CMP1_HP3    0x00030005
#define AIO_A3_CMP1_HN0    0x00030006
#define AIO_A3_CMP1_LP3    0x00030007
#define AIO_A3_CMP1_LN0    0x00030008
#elif (GS32_PART_NUM==0x0049)
//A3_2
#define AIO_A3_CMP3_HP5    0x00030003
#define AIO_A3_CMP3_LP5    0x00030004
#define AIO_A3_CMP1_HP3    0x00030005
#define AIO_A3_CMP1_HN0    0x00030006
#define AIO_A3_CMP1_LP3    0x00030007
#define AIO_A3_CMP1_LN0    0x00030008
#else
#define AIO_A3_CMP3_HP3    0x00030003
#define AIO_A3_CMP3_HN0    0x00030004
#define AIO_A3_CMP3_LP3    0x00030005
#define AIO_A3_CMP3_LN0    0x00030006
#define AIO_A3_CMP3_HP5    0x00030007
#define AIO_A3_CMP3_LP5    0x00030008
#endif

#if ((GS32_PART_NUM==0x0039) || (GS32_PART_NUM==0x0049))
#define AIO_A4_CMP2_HP0    0x00040002
#define AIO_A4_CMP2_LP0    0x00040003
#define AIO_A4_CMP5_HP3    0x00040004
#define AIO_A4_CMP5_LP3    0x00040005
#endif

#if (GS32_PART_NUM==0x0039)
//A5_3
#define AIO_A5_CMP2_HP5    0x00050001
#define AIO_A5_CMP2_LP5    0x00050002
#define AIO_A5_CMP5_HP5    0x00050003
#define AIO_A5_CMP5_LP5    0x00050004
#elif (GS32_PART_NUM==0x0049)
//A5_4
#define AIO_A5_CMP2_HP3    0x00050001
#define AIO_A5_CMP2_HN0    0x00050002
#define AIO_A5_CMP2_LP3    0x00050003
#define AIO_A5_CMP2_LN0    0x00050004
#else
#define AIO_A5_CMP2_HP3    0x00050001
#define AIO_A5_CMP2_HN0    0x00050002
#define AIO_A5_CMP2_LP3    0x00050003
#define AIO_A5_CMP2_LN0    0x00050004
#endif

#if (GS32_PART_NUM==0x0039)
#define AIO_A6_CMP1_HP2    0x00060001
#define AIO_A6_CMP1_LP2    0x00060002
#define AIO_A6_CMP5_HP0    0x00060003
#define AIO_A6_CMP5_LP0    0x00060004
#elif (GS32_PART_NUM==0x0049)
//A6_2 efuse
#define AIO_A6_CMP5_HP0    0x00060001
#define AIO_A6_CMP5_LP0    0x00060002
#endif

/* There is no A7 ******************/
#if ((GS32_PART_NUM==0x0039) || (GS32_PART_NUM==0x0049))
//A8_2
#define AIO_A8_CMP4_HP4    0x00070001
#define AIO_A8_CMP4_LP4    0x00070002
#define AIO_A8_CMP6_HP0    0x00070003
#define AIO_A8_CMP6_LP0    0x00060004
#endif

#if (GS32_PART_NUM==0x0039)
#define AIO_A9_CMP2_HP2    0x00080001
#define AIO_A9_CMP2_LP2    0x00080002
#define AIO_A9_CMP6_HP3    0x00080003
#define AIO_A9_CMP6_HN0    0x00080004
#define AIO_A9_CMP6_LP3    0x00080005
#define AIO_A9_CMP6_LN0    0x00080006
#else
#define AIO_A9_CMP6_HP3    0x00080001
#define AIO_A9_CMP6_HN0    0x00080002
#define AIO_A9_CMP6_LP3    0x00080003
#define AIO_A9_CMP6_LN0    0x00080004
#endif

#if (GS32_PART_NUM==0x0039)
#define AIO_A10_CMP2_HP3  0x00090003
#define AIO_A10_CMP2_HN0  0x00090004
#define AIO_A10_CMP2_LP3  0x00090005
#define AIO_A10_CMP2_LN0  0x00090006
#define AIO_A10_CMP7_HP0  0x00090007
#define AIO_A10_CMP7_LP0  0x00090008
#endif

#if ((GS32_PART_NUM==0x0049) || (GS32_PART_NUM==0x0039))
#define AIO_A11_CMP1_HP1   0x000A0003
#define AIO_A11_CMP1_HN1   0x000A0004
#define AIO_A11_CMP1_LP1   0x000A0005
#define AIO_A11_CMP1_LN1   0x000A0006
#define AIO_A11_CMP1_HP2   0x000A0007
#define AIO_A11_CMP1_LP2   0x000A0008
#endif

#if (GS32_PART_NUM==0x0039)
//A12_1
#define AIO_A12_CMP2_HP1   0x000B0002
#define AIO_A12_CMP2_HN1   0x000B0003
#define AIO_A12_CMP2_LP1   0x000B0004
#define AIO_A12_CMP2_LN1   0x000B0005
#elif (GS32_PART_NUM==0x0049)
//A12_2
#define AIO_A12_CMP6_HP1   0x000B0002
#define AIO_A12_CMP6_HN1   0x000B0003
#define AIO_A12_CMP6_LP1   0x000B0004
#define AIO_A12_CMP6_LN1   0x000B0005
#define AIO_A12_CMP6_HP2   0x000B0006
#define AIO_A12_CMP6_LP2   0x000B0007
#else
#endif

/* There is no A13 *****************/

#if ((GS32_PART_NUM==0x0049) || (GS32_PART_NUM==0x0039))
//A14_2
#define AIO_A14_CMP3_HP4   0x000C0003
#define AIO_A14_CMP3_LP4   0x000C0004
#define AIO_A14_CMP5_HP1   0x000C0005
#define AIO_A14_CMP5_HN1   0x000C0006
#define AIO_A14_CMP5_LP1   0x000C0007
#define AIO_A14_CMP5_LN1   0x000C0008
#define AIO_A14_CMP5_HP2   0x000C0009
#define AIO_A14_CMP5_LP2   0x000C000A
#endif

/* A15 is GPIO13 on 0039 */
#if ((GS32_PART_NUM==0x0049) || (GS32_PART_NUM==0x0039))
#else
#define AIO_A15_CMP1_HP3   0x000D0003
#define AIO_A15_CMP1_HN0   0x000D0004
#define AIO_A15_CMP1_LP3   0x000D0005
#define AIO_A15_CMP1_LN0   0x000D0006
#define AIO_A15_CMP3_LP1   0x000D0007
#define AIO_A15_CMP3_LN1   0x000D0008
#endif

#define AIO_B0_CMP2_HP4    0x000E0002
#define AIO_B0_CMP2_LP4    0x000E0003
#define AIO_B0_CMP7_HP3    0x000E0004
#define AIO_B0_CMP7_HN0    0x000E0005
#define AIO_B0_CMP7_LP3    0x000E0006
#define AIO_B0_CMP7_LN0    0x000E0007

#define AIO_B2_CMP3_HP0    0x000F0002
#define AIO_B2_CMP3_LP0    0x000F0003

#define AIO_B3_CMP3_HP3    0x00100002
#define AIO_B3_CMP3_HN0    0x00100003
#define AIO_B3_CMP3_LP3    0x00100004
#define AIO_B3_CMP3_LN0    0x00100005

#define AIO_B4_CMP4_HP0    0x00110002
#define AIO_B4_CMP4_LP0    0x00110003

#define AIO_B5_CMP1_HP5    0x00120001
#define AIO_B5_CMP1_LP5    0x00120002

#if (GS32_PART_NUM==0x0039)
#define AIO_B11_CMP4_HP5   0x00130001
#define AIO_B11_CMP4_LP5   0x00130002
#elif (GS32_PART_NUM==0x0049)
//B11_2
#define AIO_B11_CMP4_HP5   0x00130001
#define AIO_B11_CMP4_LP5   0x00130002
#endif

#define AIO_B12_CMP3_HP1   0x00140002
#define AIO_B12_CMP3_HN1   0x00140003
#define AIO_B12_CMP3_LP1   0x00140004
#define AIO_B12_CMP3_LN1   0x00140005

#if (GS32_PART_NUM==0x0039)
//C1_1
#define AIO_C1_CMP4_HP2    0x00150001
#define AIO_C1_CMP4_LP2    0x00150002
#elif (GS32_PART_NUM==0x0049)
//C1_2
#define AIO_C1_CMP2_HP1    0x00150001
#define AIO_C1_CMP2_HN1    0x00150002
#define AIO_C1_CMP2_LP1    0x00150003
#define AIO_C1_CMP2_LN1    0x00150004
#endif

#if (GS32_PART_NUM==0x0039)
//C3_1
#define AIO_C3_CMP4_HP1    0x00160002
#define AIO_C3_CMP4_HN1    0x00160003
#define AIO_C3_CMP4_LP1    0x00160004
#define AIO_C3_CMP4_LN1    0x00160005
#elif (GS32_PART_NUM==0x0049)
//C3_2
#define AIO_C3_CMP4_HP1    0x00160002
#define AIO_C3_CMP4_HN1    0x00160003
#define AIO_C3_CMP4_LP1    0x00160004
#define AIO_C3_CMP4_LN1    0x00160005
#define AIO_C3_CMP4_HP2    0x00160006
#define AIO_C3_CMP4_LP2    0x00160007
#endif

#define AIO_C14_CMP4_HP3   0x00170001
#define AIO_C14_CMP4_HN0   0x00170002
#define AIO_C14_CMP4_LP3   0x00170003
#define AIO_C14_CMP4_LN0   0x00170004
#define AIO_C14_CMP7_HP1   0x00170005
#define AIO_C14_CMP7_HN1   0x00170006
#define AIO_C14_CMP7_LP1   0x00170007
#define AIO_C14_CMP7_LN1   0x00170008

#if (GS32_PART_NUM==0x0049)
//pin
#define AIO_C1H2_CMP1_HP2  0x00180001
#define AIO_C1H2_CMP1_LP2  0x00180002
#endif

#if (GS32_PART_NUM==0x0049)
#define AIO_C3H2_CMP3_HP2  0x00190001
#define AIO_C3H2_CMP3_LP2  0x00190002
#endif

#if (GS32_PART_NUM==0x0049)
#define AIO_C5H2_CMP5_HP2  0x001A0001
#define AIO_C5H2_CMP5_LP2  0x001A0002
#endif

#if (GS32_PART_NUM==0x0049)
#define AIO_C7H2_CMP7_HP2  0x001B0001
#define AIO_C7H2_CMP7_LP2  0x001B0002
#endif

/**
 * The following MACRO is input parameter for GPIO_setAnalogPin(...)
 * bit24==1 means it's AGPIO in sysctl_h, (not ASEL in asysctl)
 * bit16~19 is AGPIO offset, (AGPIO13==0, AGPIO20==1, AGPIO21==2)
 * bit0~3 is AGPIO analog function select, 0=1st option, 1=2nd option, 2=3rd option, ....
 */

#define AIO_GP13_A15       0x01000000

#define AIO_GP20_B5        0x01010000
#define AIO_GP20_CMP1_HP5  0x01010001
#define AIO_GP20_CMP1_LP5  0x01010002

#define AIO_GP21_B11       0x01020000
#define AIO_GP21_CMP4_HP5  0x01020001
#define AIO_GP21_CMP4_LP5  0x01020002

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/**
 * 
 * Values used for function ASysCtl_selectCMPHNMux() and
 * ASysCtl_selectCMPHNMuxValue().  These values can be OR-ed together and
 * passed to ASysCtl_selectCMPHNMux().
 * 
 */
#define ASYSCTL_CMPHNMUX_SELECT_1 0x1U //< CMPHNMUX select 1
#define ASYSCTL_CMPHNMUX_SELECT_2 0x2U //< CMPHNMUX select 2
#define ASYSCTL_CMPHNMUX_SELECT_3 0x4U //< CMPHNMUX select 3
#define ASYSCTL_CMPHNMUX_SELECT_4 0x8U //< CMPHNMUX select 4
#define ASYSCTL_CMPHNMUX_SELECT_5 0x10U //< CMPHNMUX select 5
#define ASYSCTL_CMPHNMUX_SELECT_6 0x20U //< CMPHNMUX select 6
#define ASYSCTL_CMPHNMUX_SELECT_7 0x40U //< CMPHNMUX select 7


/**
 * 
 *  Values used for function ASysCtl_selectCMPLNMux() and
 *  ASysCtl_selectCMPLNMuxValue().  These values can be OR-ed together and
 *  passed to ASysCtl_selectCMPLNMux().
 * 
 */
#define ASYSCTL_CMPLNMUX_SELECT_1 0x1U //< CMPLNMUX select 1
#define ASYSCTL_CMPLNMUX_SELECT_2 0x2U //< CMPLNMUX select 2
#define ASYSCTL_CMPLNMUX_SELECT_3 0x4U //< CMPLNMUX select 3
#define ASYSCTL_CMPLNMUX_SELECT_4 0x8U //< CMPLNMUX select 4
#define ASYSCTL_CMPLNMUX_SELECT_5 0x10U //< CMPLNMUX select 5
#define ASYSCTL_CMPLNMUX_SELECT_6 0x20U //< CMPLNMUX select 6
#define ASYSCTL_CMPLNMUX_SELECT_7 0x40U //< CMPLNMUX select 7

/**
 * 
 *  ASysCtl_CMPHPMuxSelect used for function ASysCtl_selectCMPHPMux().
 * 
 */
typedef enum
{
    ASYSCTL_CMPHPMUX_SELECT_1 = 0U, //< CMPHPMUX select 1
    ASYSCTL_CMPHPMUX_SELECT_2 = 3U, //< CMPHPMUX select 2
    ASYSCTL_CMPHPMUX_SELECT_3 = 6U, //< CMPHPMUX select 3
    ASYSCTL_CMPHPMUX_SELECT_4 = 9U, //< CMPHPMUX select 4
	ASYSCTL_CMPHPMUX_SELECT_5 = 12U,//< CMPHPMUX select 5
	ASYSCTL_CMPHPMUX_SELECT_6 = 16U,//< CMPHPMUX select 6
	ASYSCTL_CMPHPMUX_SELECT_7 = 19U //< CMPHPMUX select 7
} ASysCtl_CMPHPMuxSelect;

/**
 * 
 *  ASysCtl_CMPLPMuxSelect used for function ASysCtl_selectCMPLPMux().
 * 
 */
typedef enum
{
    ASYSCTL_CMPLPMUX_SELECT_1 = 0U, //< CMPLPMUX select 1
    ASYSCTL_CMPLPMUX_SELECT_2 = 3U, //< CMPLPMUX select 2
    ASYSCTL_CMPLPMUX_SELECT_3 = 6U, //< CMPLPMUX select 3
    ASYSCTL_CMPLPMUX_SELECT_4 = 9U, //< CMPLPMUX select 4
	ASYSCTL_CMPLPMUX_SELECT_5 = 12U,//< CMPLPMUX select 5
	ASYSCTL_CMPLPMUX_SELECT_6 = 16U,//< CMPLPMUX select 6
	ASYSCTL_CMPLPMUX_SELECT_7 = 19U //< CMPLPMUX select 7
} ASysCtl_CMPLPMuxSelect;

typedef enum
{
    ASYSCTL_CMP_SELECT_1 = 0U,  //< CMP select 1
    ASYSCTL_CMP_SELECT_2 = 4U,  //< CMP select 2
    ASYSCTL_CMP_SELECT_3 = 8U,  //< CMP select 3
    ASYSCTL_CMP_SELECT_4 = 12U, //< CMP select 4
	ASYSCTL_CMP_SELECT_5 = 16U, //< CMP select 5
	ASYSCTL_CMP_SELECT_6 = 20U, //< CMP select 6
	ASYSCTL_CMP_SELECT_7 = 24U  //< CMP select 7
} ASysCtl_CMPSelect;


#define ANALOG_TRIM_EFUSE_INFO_BASE     	0x1FFF4200
#define ANALOG_TRIM_EFUSE_VERSION_ADDR    	0x1FFF4100

typedef enum AnaTrimPeriId_v0 {
    V0_TRIM_ID_LDO_1V2 = 0,
    V0_TRIM_ID_ANA_LDO,
    V0_TRIM_ID_OSC1,
    V0_TRIM_ID_OSC2,

    V0_TRIM_ID_VREFHI_COARSE_2V5,
    V0_TRIM_ID_VREFHI_FINE_2V5,

    V0_TRIM_ID_VREFHI_COARSE_3V3,
    V0_TRIM_ID_VREFHI_FINE_3V3,

//    TRIM_ID_ADCA_OFF,
    V0_TRIM_ID_ADCA_OFFSET,
    V0_TRIM_ID_ADCA_VCM_INT,
    V0_TRIM_ID_ADCA_VCM_EXT,
	V0_TRIM_ID_ADCA_OFF_DIR,

//  V0_  TRIM_ID_ADCB_OFF,
    V0_TRIM_ID_ADCB_OFFSET,
    V0_TRIM_ID_ADCB_VCM_INT,
    V0_TRIM_ID_ADCB_VCM_EXT,
	V0_TRIM_ID_ADCB_OFF_DIR,

//  V0_  TRIM_ID_ADCC_OFF,
    V0_TRIM_ID_ADCC_OFFSET,
    V0_TRIM_ID_ADCC_VCM_INT,
    V0_TRIM_ID_ADCC_VCM_EXT,
	V0_TRIM_ID_ADCC_OFF_DIR,

    V0_TRIM_ID_MAX,
}AnaTrimPeriId_v0_t;

typedef enum AnaTrimPeriId_v5 {
    V5_TRIM_ID_LDO_1V2 = 0,
    V5_TRIM_ID_ANA_LDO,
    V5_TRIM_ID_OSC1,
    V5_TRIM_ID_OSC2,

    V5_TRIM_ID_VREFHI_COARSE_2P5V,
    V5_TRIM_ID_VREFHI_FINE_2P5V,

    V5_TRIM_ID_VREFHI_COARSE_1P65V,
    V5_TRIM_ID_VREFHI_FINE_1P65V,

    V5_TRIM_ID_ADCA_OFFSET,
    V5_TRIM_ID_ADCA_OFF_DIR,

    V5_TRIM_ID_ADCB_OFFSET,
    V5_TRIM_ID_ADCB_OFF_DIR,

    V5_TRIM_ID_ADCC_OFFSET,
    V5_TRIM_ID_ADCC_OFF_DIR,

    V5_TRIM_ID_ADC_VCM_COARSE_1P65V,
    V5_TRIM_ID_ADC_VCM_FINE_1P65V,
    V5_TRIM_ID_ADC_VCM_COARSE_2P475V,
    V5_TRIM_ID_ADC_VCM_FINE_2P475V,
    V5_TRIM_ID_ADC_VCM_COARSE_1P25V,
    V5_TRIM_ID_ADC_VCM_FINE_1P25V,

    V5_TRIM_ID_MAX,
}AnaTrimPeriId_v5_t;

typedef enum AnaTrimPeriId_v6 {
    V6_TRIM_ID_LDO_1V2 = 0,
    V6_TRIM_ID_ANA_LDO,
    V6_TRIM_ID_OSC1,
    V6_TRIM_ID_OSC2,

    V6_TRIM_ID_VREFHI_COARSE_2P5V,
    V6_TRIM_ID_VREFHI_FINE_2P5V,

    V6_TRIM_ID_VREFHI_COARSE_1P65V,
    V6_TRIM_ID_VREFHI_FINE_1P65V,

    V6_TRIM_ID_ADCA_VCM_COARSE_1P65V,
    V6_TRIM_ID_ADCA_VCM_FINE_1P65V,
    V6_TRIM_ID_ADCA_VCM_COARSE_2P475V,
    V6_TRIM_ID_ADCA_VCM_FINE_2P475V,
    V6_TRIM_ID_ADCA_VCM_COARSE_1P25V,
    V6_TRIM_ID_ADCA_VCM_FINE_1P25V,
    V6_TRIM_ID_ADCA_OFFSET,
    V6_TRIM_ID_ADCA_OFF_DIR,

    V6_TRIM_ID_ADCB_VCM_COARSE_1P65V,
    V6_TRIM_ID_ADCB_VCM_FINE_1P65V,
    V6_TRIM_ID_ADCB_VCM_COARSE_2P475V,
    V6_TRIM_ID_ADCB_VCM_FINE_2P475V,
    V6_TRIM_ID_ADCB_VCM_COARSE_1P25V,
    V6_TRIM_ID_ADCB_VCM_FINE_1P25V,
    V6_TRIM_ID_ADCB_OFFSET,
    V6_TRIM_ID_ADCB_OFF_DIR,

    V6_TRIM_ID_ADCC_VCM_COARSE_1P65V,
    V6_TRIM_ID_ADCC_VCM_FINE_1P65V,
    V6_TRIM_ID_ADCC_VCM_COARSE_2P475V,
    V6_TRIM_ID_ADCC_VCM_FINE_2P475V,
    V6_TRIM_ID_ADCC_VCM_COARSE_1P25V,
    V6_TRIM_ID_ADCC_VCM_FINE_1P25V,
    V6_TRIM_ID_ADCC_OFFSET,
    V6_TRIM_ID_ADCC_OFF_DIR,

    V6_TRIM_ID_MAX,
}AnaTrimPeriId_v6_t;

typedef enum AnaTrimPeriId_v7 {
    V7_TRIM_ID_LDO_1V2 = 0,
    V7_TRIM_ID_TOP_BOR,
    V7_TRIM_ID_ANA_LDO,
    V7_TRIM_ID_OSC1,
    V7_TRIM_ID_OSC2,

    V7_TRIM_ID_VREFHI_COARSE_2P5V,
    V7_TRIM_ID_VREFHI_FINE_2P5V,

    V7_TRIM_ID_VREFHI_COARSE_1P65V,
    V7_TRIM_ID_VREFHI_FINE_1P65V,

    //ADCA
    V7_TRIM_ID_ADCA_VCM_COARSE_1P65V,
    V7_TRIM_ID_ADCA_VCM_FINE_1P65V,
    V7_TRIM_ID_ADCA_VCM_COARSE_2P475V,
    V7_TRIM_ID_ADCA_VCM_FINE_2P475V,
    V7_TRIM_ID_ADCA_VCM_COARSE_1P25V,
    V7_TRIM_ID_ADCA_VCM_FINE_1P25V,
    V7_TRIM_ID_ADCA_VCM_COARSE_1P5V,
    V7_TRIM_ID_ADCA_VCM_FINE_1P5V,

    V7_TRIM_ID_ADCA_OFF_DIR_VREFHI_3P3V_IN,
    V7_TRIM_ID_ADCA_OFF_DIR_VREFHI_2P5V_IN,
    V7_TRIM_ID_ADCA_OFF_DIR_VREFHI_3P3V_OUT,
    V7_TRIM_ID_ADCA_OFF_DIR_VREFHI_2P5V_OUT,
    V7_TRIM_ID_ADCA_OFF_DIR_VREFHI_3P0V_OUT,

    V7_TRIM_ID_ADCA_OFFSET_VREFHI_3P3V_IN,
    V7_TRIM_ID_ADCA_OFFSET_VREFHI_2P5V_IN,
    V7_TRIM_ID_ADCA_OFFSET_VREFHI_3P3V_OUT,
    V7_TRIM_ID_ADCA_OFFSET_VREFHI_2P5V_OUT,
    V7_TRIM_ID_ADCA_OFFSET_VREFHI_3P0V_OUT,

    //ADCB
    V7_TRIM_ID_ADCB_VCM_COARSE_1P65V,
    V7_TRIM_ID_ADCB_VCM_FINE_1P65V,
    V7_TRIM_ID_ADCB_VCM_COARSE_2P475V,
    V7_TRIM_ID_ADCB_VCM_FINE_2P475V,
    V7_TRIM_ID_ADCB_VCM_COARSE_1P25V,
    V7_TRIM_ID_ADCB_VCM_FINE_1P25V,
    V7_TRIM_ID_ADCB_VCM_COARSE_1P5V,
    V7_TRIM_ID_ADCB_VCM_FINE_1P5V,

    V7_TRIM_ID_ADCB_OFF_DIR_VREFHI_3P3V_IN,
    V7_TRIM_ID_ADCB_OFF_DIR_VREFHI_2P5V_IN,
    V7_TRIM_ID_ADCB_OFF_DIR_VREFHI_3P3V_OUT,
    V7_TRIM_ID_ADCB_OFF_DIR_VREFHI_2P5V_OUT,
    V7_TRIM_ID_ADCB_OFF_DIR_VREFHI_3P0V_OUT,

    V7_TRIM_ID_ADCB_OFFSET_VREFHI_3P3V_IN,
    V7_TRIM_ID_ADCB_OFFSET_VREFHI_2P5V_IN,
    V7_TRIM_ID_ADCB_OFFSET_VREFHI_3P3V_OUT,
    V7_TRIM_ID_ADCB_OFFSET_VREFHI_2P5V_OUT,
    V7_TRIM_ID_ADCB_OFFSET_VREFHI_3P0V_OUT,

    //ADCC
    V7_TRIM_ID_ADCC_VCM_COARSE_1P65V,
    V7_TRIM_ID_ADCC_VCM_FINE_1P65V,
    V7_TRIM_ID_ADCC_VCM_COARSE_2P475V,
    V7_TRIM_ID_ADCC_VCM_FINE_2P475V,
    V7_TRIM_ID_ADCC_VCM_COARSE_1P25V,
    V7_TRIM_ID_ADCC_VCM_FINE_1P25V,
    V7_TRIM_ID_ADCC_VCM_COARSE_1P5V,
    V7_TRIM_ID_ADCC_VCM_FINE_1P5V,

    V7_TRIM_ID_ADCC_OFF_DIR_VREFHI_3P3V_IN,
    V7_TRIM_ID_ADCC_OFF_DIR_VREFHI_2P5V_IN,
    V7_TRIM_ID_ADCC_OFF_DIR_VREFHI_3P3V_OUT,
    V7_TRIM_ID_ADCC_OFF_DIR_VREFHI_2P5V_OUT,
    V7_TRIM_ID_ADCC_OFF_DIR_VREFHI_3P0V_OUT,

    V7_TRIM_ID_ADCC_OFFSET_VREFHI_3P3V_IN,
    V7_TRIM_ID_ADCC_OFFSET_VREFHI_2P5V_IN,
    V7_TRIM_ID_ADCC_OFFSET_VREFHI_3P3V_OUT,
    V7_TRIM_ID_ADCC_OFFSET_VREFHI_2P5V_OUT,
    V7_TRIM_ID_ADCC_OFFSET_VREFHI_3P0V_OUT,

    //digital
    V7_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_3P3V_IN,
    V7_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_3P3V_IN,
    V7_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_2P5V_IN,
    V7_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_2P5V_IN,
    V7_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_3P3V_OUT,
    V7_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_3P3V_OUT,
    V7_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_2P5V_OUT,
    V7_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_2P5V_OUT,
    V7_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_3P0V_OUT,
    V7_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_3P0V_OUT,

    V7_TRIM_ID_ADCB_DIG_OFFSET_VREFHI_3P3V_IN,
    V7_TRIM_ID_ADCB_DIG_GAIN_ERR_VREFHI_3P3V_IN,
    V7_TRIM_ID_ADCB_DIG_OFFSET_VREFHI_2P5V_IN,
    V7_TRIM_ID_ADCB_DIG_GAIN_ERR_VREFHI_2P5V_IN,
    V7_TRIM_ID_ADCB_DIG_OFFSET_VREFHI_3P3V_OUT,
    V7_TRIM_ID_ADCB_DIG_GAIN_ERR_VREFHI_3P3V_OUT,
    V7_TRIM_ID_ADCB_DIG_OFFSET_VREFHI_2P5V_OUT,
    V7_TRIM_ID_ADCB_DIG_GAIN_ERR_VREFHI_2P5V_OUT,
    V7_TRIM_ID_ADCB_DIG_OFFSET_VREFHI_3P0V_OUT,
    V7_TRIM_ID_ADCB_DIG_GAIN_ERR_VREFHI_3P0V_OUT,

    V7_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_3P3V_IN,
    V7_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_3P3V_IN,
    V7_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_2P5V_IN,
    V7_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_2P5V_IN,
    V7_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_3P3V_OUT,
    V7_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_3P3V_OUT,
    V7_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_2P5V_OUT,
    V7_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_2P5V_OUT,
    V7_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_3P0V_OUT,
    V7_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_3P0V_OUT,

    V7_TRIM_ID_MAX,
}AnaTrimPeriId_v7_t;

#define REG_WIDTH_16BITS     2
#define REG_WIDTH_32BITS     4

typedef struct AnaTrimTable {
    uint32_t regAddr;
    uint32_t regWidth : 3;  //bytes
    uint32_t endbit : 5;
    uint32_t startbit : 5;
    uint32_t defaultTrimVal : 19;
    uint32_t efuseAddr;
    uint16_t efuseEndbit;
    uint16_t efuseStartbit;
}AnaTrimTable_t;

typedef struct strAdcSpareReg {
	uint32_t    rsvd0_3:    4;     // 3:0
	uint32_t    rsvd4_5:    2;     // 5:4
	uint32_t    MODE_SEL:   3;     // 6
	uint32_t    rsvd7_9:    3;     // 9:7
	uint32_t    VREFH_CRS:  3;     // 13:10
	uint32_t    rsvd14:     1;     // 14
	uint32_t    VREFH_FINE: 3;     // 17:15
	uint32_t    rsvd18_20:  4;     // 20:18
	uint32_t    VCM_TRM:    4;     // 24:21
	uint32_t	rsvd25:     1;	   // 25
	uint32_t	VCM_CTL:    3;	   // 28:26
	uint32_t	rsvd29_31:  3;	   // 31:29
}AdcSpareReg_t;

#define ADCA_SPARE_REG_PTR  ((volatile AdcSpareReg_t*)(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_SPARE))
#define ADCB_SPARE_REG_PTR  ((volatile AdcSpareReg_t*)(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCB_SPARE))
#define ADCC_SPARE_REG_PTR  ((volatile AdcSpareReg_t*)(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCC_SPARE))

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */

void ASysCtl_initConfig(void);

/*
 *  ANA_SYSCTRL_LOCK   Locking Register   Offset 0x0000
 *
 *  Register Field       Field Range
 *  ana_sysctrl_lock     31:0
 *                
 *  Writing the unlock value to this register allows writing to all registers protected by lock.
 *  Writing any other value will lock all registers protected by lock.
 *  Unlock value: 0x5A5A5A5A.
 * 
 */
__STATIC_INLINE void ASysCtl_setAnaSysctrlLock(void)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_CFG_LOCK) = 0x01;
}

__STATIC_INLINE void ASysCtl_setAnaSysctrlUnLock(void)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_CFG_LOCK) = 0x5A5A5A5A;
}

__STATIC_INLINE uint32_t ASysCtl_getAnaSysctrlLock(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_CFG_LOCK);
}

/**
 * VMONCTL      Voltage Monitor Control Register    Offset 0x0E0
 *
 *  reserved        15:9
 *  borlvmondis     8
 *  reserved        7:0
 *
 */
__STATIC_INLINE void SysCtl_setBorLvmonDisable(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_VMONCTL) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_VMONCTL) & ~(0x01<<8)) | (value & 0x01) << 8;
}

__STATIC_INLINE uint32_t SysCtl_getBorLvmonDisable(void)
{
    return ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_VMONCTL)) >> 8) & 0x01;
}

/**
 * ANA_BOR_CFG      Brown-out Reset Configuration Register    Offset 0x304
 *
 *  Register Field     Field Range
 *  reserved           31:12
 *  bor_test_en        11:11
 *  bor_trim           10:7
 *  bor_rising_sel     6:4
 *  bor_falling_sel    3:1
 *  reserved             0
 *
 */
__STATIC_INLINE void SysCtl_setBorVfallDSel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) &(~(0x07 << 1))) | ((value & 0x07) << 1);
}

__STATIC_INLINE uint32_t SysCtl_getBorVfallDSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) >> 1) & 0x07;
}

__STATIC_INLINE void SysCtl_setBorVriseDSel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) & (~(0x07 << 4))) | ((value & 0x07) << 4);
}

__STATIC_INLINE uint32_t SysCtl_getBorVriseDSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) >> 4) & 0x07;
}

__STATIC_INLINE void SysCtl_setBorTrim(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) &(~(0x0F << 7))) | ((value & 0x0F) << 7); ;
}

__STATIC_INLINE uint32_t SysCtl_getBorTrim(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) >> 7) & 0x0F;
}

__STATIC_INLINE void SysCtl_setBorTestEn(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) & (1<<11)) | (value & 0x01);
}

__STATIC_INLINE uint32_t SysCtl_getBorTestEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG)) & (1<<11);
}

/**
 * ANA_LDO_1V1_CFG      1V1 LDO Configuration Register    Offset 0x308
 *
 *  Register Field            Field Range
 *  ldo_1v1_test_en           11
 *  ldo_1v1_test_mux_sel      10:8
 *  ldo_1v1_chop_en           7
 *  ldo_1v1_test_mode         6:5
 *  ldo_1v1_vsel              4:0
 *
 */
__STATIC_INLINE void SysCtl_setLdoVsel(uint32_t vsel)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) & (~(0x1F << 0))) | (vsel << 0);
}

__STATIC_INLINE uint32_t SysCtl_getLdoVsel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) >> 0) & 0x1F;
}

__STATIC_INLINE void SysCtl_setLdoTestMode(uint32_t mode)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) & (~(0x3 << 5))) | (mode << 5);
}

__STATIC_INLINE uint32_t SysCtl_getLdoTestMode(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) >> 5) & 0x03;
}

__STATIC_INLINE void SysCtl_setLdoChopEn(uint32_t en)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) & (~(0x1 << 7))) | (en << 7);
}

__STATIC_INLINE uint32_t SysCtl_getLdoChopEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) >> 7) & 0x01;
}

/**
 * ANA_EFLASH_BOR_CFG      EFLASH_BOR Configuration Register    Offset 0x30C
 *
 *  Register Field            Field Range
 *  reserved                    31:3
 *  eflash_bor_test_sel         2:1
 *  eflash_bor_test_en          0
 *
 */
__STATIC_INLINE void ASysCtl_setEflashBorTestEn(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_EFLASH_BOR_CFG) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_EFLASH_BOR_CFG) & (~(0x1 << 0))) | (enable << 0);
}

__STATIC_INLINE uint32_t ASysCtl_getEflashBorTestEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_EFLASH_BOR_CFG) >> 0) & 0x1F;
}

__STATIC_INLINE void ASysCtl_setEflashBorTestSel(uint32_t vsel)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_EFLASH_BOR_CFG) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_EFLASH_BOR_CFG) & (~(0x3 << 1))) | ((vsel & 0x03) << 1);
}

__STATIC_INLINE uint32_t ASysCtl_getEflashBorTestSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_EFLASH_BOR_CFG) >> 1) & 0x3;
}

/**
 * ANA_OSC_EN      OSC Enable Configuration Register    Offset 0x10
 *
 *  Register Field    Field Range
 *  osc2_en           1
 *  osc1_en           0
 *  Reserved          (No description provided)
 *
 */
//__STATIC_INLINE uint32_t ASysCtl_getOsc1En(void)
//{
//    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_OSC_EN)) & 0x01;
//}
//
//__STATIC_INLINE uint32_t ASysCtl_getOsc2En(void)
//{
//    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_OSC_EN) >> 1) & 0x01;
//}

/**
 *  ANA_TOP_BG_CFG   BG Configuration        0x310
 *
 *  Register Field              Field Range
 *  reserved                    31:5
 *  top_bg_test_mux_sel         4:2
 *  top_bg_test_mux_en          1
 *  top_bg_chop_en              0
 *
 */
__STATIC_INLINE void ASysCtl_setTopBgChopEn(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_BG_CFG) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_BG_CFG) & ~(0x01<<0)) | (enable & 0x01);
}

__STATIC_INLINE uint32_t ASysCtl_getTopBgChopEn(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_BG_CFG) & 0x01;
}

__STATIC_INLINE void ASysCtl_setTopBgTestMuxEn(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_BG_CFG) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_BG_CFG) & ~(0x01<<1)) | ((enable & 0x01) << 1);
}

__STATIC_INLINE uint32_t ASysCtl_getTopBgTestMuxEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_BG_CFG) >> 1) & 0x01;
}

__STATIC_INLINE void ASysCtl_setTopBgTestMuxSel(uint32_t mux)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_BG_CFG) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_BG_CFG) & ~(0x07<<2)) | ((mux & 0x07) << 2);
}

__STATIC_INLINE uint32_t ASysCtl_getTopBgTestMuxSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_BG_CFG) >> 2) & 0x07;
}

/**
 * ANA_OSC_TRIM      OSC Trim Configuration Register    Offset 0x314
 *
 *  Register Field    Field Range
 *  osc2_freqtrim     30 : 16
 *  reserved          15
 *  osc1_freqtrim     14:0
 */
__STATIC_INLINE void ASysCtl_setOsc1Trim(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_OSC_TRIM) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_OSC_TRIM) & (~0x7FFF)) | (value & 0x7FFF);
}

__STATIC_INLINE uint32_t ASysCtl_getOsc1Trim(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_OSC_TRIM) & 0x7FFF;
}

__STATIC_INLINE void ASysCtl_setOsc2Trim(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_OSC_TRIM) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_OSC_TRIM) & (~(0x7FFF << 16))) | ((value & 0x7FFF) << 16);
}

__STATIC_INLINE uint32_t ASysCtl_getOsc2Trim(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_OSC_TRIM) >> 16) & 0x7FFF;
}

/*
 *  PVM_EN   PVM Enable Configuration Register  Offset 0x318
 *
 *  Register Field     Field Range
 *  reserved           31:12
 *  pvm_test_en        11
 *  pvm_trim           10:7
 *  pvm_rising_sel     6:4
 *  pvm_falling_sel    3:1
 *  pvm_en             0
 *
 */
/*
 * TODO:
 */
__STATIC_INLINE void ASysCtl_setPvmEn(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getPvmEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) & 0x1);
}

__STATIC_INLINE void ASysCtl_setPvmFallingSel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) =
            ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG)&(~(0x07<<1)))|(value << 1));
}

__STATIC_INLINE uint32_t ASysCtl_getPvmFallingSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) >> 1) & 0x07;
}

__STATIC_INLINE void ASysCtl_setPvmRisingSel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) =
            ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG)&(~(0x07<<4)))|(value << 4));
}

__STATIC_INLINE uint32_t ASysCtl_getPvmRisingSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) >> 4) & 0x07;
}

__STATIC_INLINE void ASysCtl_setPvmTrim(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) =
                ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG)&(~(0x0F<<7)))|(value << 7));
}

__STATIC_INLINE uint32_t ASysCtl_getPvmTrim(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) >> 7) & 0xf;
}

__STATIC_INLINE void ASysCtl_setPvmTestEn(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) =
                    ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG)&(~(0x01<<11)))|(value << 11));
}

__STATIC_INLINE uint32_t ASysCtl_getPvmTestEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) >> 11) & 0x1;
}

/*
 *  ANA_TOP_SPARE1 ~ ANA_TOP_SPARE4   ANA Top Spare Registers  Offset 0x554+0x4*m   m = 0~1
 *
 *  Register Field    Field Range
 *  ana_top_spare     31 : 0
 *                
 * TODO:
 */
__STATIC_INLINE void ASysCtl_setAnaTopSpare(uint32_t m, uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_SPARE1 + 0x4*m) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTopSpare(uint32_t m)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_SPARE1 + 0x4*m);
}

/*
 *  ADC_SAMP_SEL   ADCA Sampling Selection Control Register  Offset 0x404
 *
 *  Register Field    Field Range
 *  reserved          31:11
 *  adcc_samp_sel     10:8
 *  reserved          7
 *  adcb_samp_sel     6:4
 *  reserved          3
 *  adca_samp_sel     2:0
 */

__STATIC_INLINE void ASysCtl_setAdcSampSel(uint32_t n, uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_SAMP_SEL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_SAMP_SEL) & ~(0x07<<(n*4))) | (value & 0x7) << (n*4);
}

__STATIC_INLINE uint32_t ASysCtl_getAdcSampSel(uint32_t n)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_SAMP_SEL) >> (n*4)) & 0x7;
}

/*
 *  ADC_SPARE   ADC Spare Registers  Offset 0x408+0x4*n   n = 0~2
 *
 *  Register Field    Field Range
 *  adc_spare         31 : 0
 *                
 * TODO:
 */
__STATIC_INLINE void ASysCtl_setAdcSpare(uint32_t n, uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_SPARE + 0x4*n) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAdcSpare(uint32_t n)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_SPARE + 0x4*n);
}


/*
 *  ADC_TOP_SPARE   ADC Top-Level Spare Register  Offset 0x418
 *
 *  Register Field    Field Range
 *  adc_spare         31 : 0
 *                
 * 
 */
__STATIC_INLINE void ASysCtl_setAdcTopSpare(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_TOP_SPARE) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAdcTopSpare(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_TOP_SPARE);
}

/*
 *  CMP_SPARE   Comparator Spare Register  Offset 0x4B4
 *
 *  Register Field    Field Range
 *  cmp_spare         31 : 0
 *  set to 0x0FFFFFFF to enable hysteresis for all CMPSS channels
 *                
 * 
 */
__STATIC_INLINE void ASysCtl_setCmpSpare(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_TOP_SPARE) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getCmpSpare(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_TOP_SPARE);
}

/*
 *  CMP_EN   Comparator Enable Register  Offset 0x41C
 *
 *  Register Field    Field Range
 *  cmp_en            0
 *
 *
 */
__STATIC_INLINE void ASysCtl_setCmpEnable(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_EN) = value & 0x01;
}

__STATIC_INLINE uint32_t ASysCtl_getCmpEnable(uint32_t value)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_EN) & 0x01);
}

/*
 *  CMP_RST   Comparator Reset Register  Offset 0x420
 *
 *  Register Field    Field Range
 *  cmp_rst           0
 *
 *
 */
__STATIC_INLINE void ASysCtl_setCmpReset(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_RST) = value & 0x01;
}

__STATIC_INLINE uint32_t ASysCtl_getCmpReset(uint32_t value)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_RST) & 0x01);
}

/*
 *  DAC_TOP_EN   DAC top enable register  Offset 0x4B8
 *
 *  Register Field    Field Range
 *  dac_top_en           0
 *
 *
 */
__STATIC_INLINE void ASysCtl_setDacTopEnable(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DAC_TOP_EN) = value & 0x01;
}

__STATIC_INLINE uint32_t ASysCtl_getDacTopEnable(uint32_t value)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DAC_TOP_EN) & 0x01);
}

/*
 *  DAC_RST   DAC Reset Register  Offset 0x4BC
 *
 *  Register Field    Field Range
 *  dac_rst           0
 *
 *
 */
__STATIC_INLINE void ASysCtl_setDacReset(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DAC_RST) = value & 0x01;
}

__STATIC_INLINE uint32_t ASysCtl_getDacReset(uint32_t value)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DAC_RST) & 0x01);
}

/*
 *  DACA_TRIM   DACA Trim Value  Offset 0x580
 *
 *  Register Field    Field Range
 *  daca_trim         31 : 0
 *
 *
 */
__STATIC_INLINE void ASysCtl_setDacaTrim(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_TRIM) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getDacaTrim(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_TRIM);
}


/*
 *  DACB_TRIM   DACB Trim Value  Offset 0x584
 *
 *  Register Field    Field Range
 *  dacb_trim         31 : 0
 *                
 * 
 */
__STATIC_INLINE void ASysCtl_setDacbTrim(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_TRIM) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getDacbTrim(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_TRIM);
}


/*
 *  DACA_BUF_TRIM   DACA Buffer Trim Value  Offset 0x4C0
 *
 *  Register Field    Field Range
 *  daca_buf_trim     5 : 0
 *                
 * TODO:
 */
__STATIC_INLINE void ASysCtl_setDacaBufTrim(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_BUF_TRIM) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getDacaBufTrim(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_BUF_TRIM);
}


/*
 *  DACB_BUF_TRIM   DACB Buffer Trim Value  Offset 0x4C4
 *
 *  Register Field    Field Range
 *  dacb_buf_trim     31 : 0
 *                
 * 
 */
__STATIC_INLINE void ASysCtl_setDacbBufTrim(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_BUF_TRIM) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getDacbBufTrim(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_BUF_TRIM);
}


/*
 *  DACA_SPARE   DACA Spare Register  Offset 0x0100
 *
 *  Register Field    Field Range
 *  daca_spare        31 : 0
 *
 */
__STATIC_INLINE void ASysCtl_setDacaSpare(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_SPARE) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getDacaSpare(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_SPARE);
}


/*
 *  DACB_SPARE   DACB Spare Register  Offset 0x0100
 *
 *  Register Field    Field Range
 *  dacb_spare        31 : 0
 *
 *
 */
__STATIC_INLINE void ASysCtl_setDacbSpare(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_SPARE) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getDacbSpare(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_SPARE);
}


/*
 *  DAC_TOP_SPARE   DAC Top Spare Register  Offset 0x0100
 *
 *  Register Field    Field Range
 *  dac_spare         31 : 0
 *                
 * 
 */
__STATIC_INLINE void ASysCtl_setDacTopSpare(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DAC_TOP_SPARE) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getDacTopSpare(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DAC_TOP_SPARE);
}


/*
 *  TMUX   TMUX Configuration Register  Offset 0x0120
 *
 *  Register Field    Field Range
 *  tmux              31 : 0
 *                
 * 
 */
__STATIC_INLINE void ASysCtl_setTmux(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_TMUX) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getTmux(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_TMUX) >> 0);
}


/*
 *  V2I_A   V2I_A Configuration Register  Offset 0x0124
 *
 *  Register Field    Field Range
 *  v2i_a             31 : 0
 *                
 * 
 */
__STATIC_INLINE void ASysCtl_setV2iA(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_V2I_A) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getV2iA(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_V2I_A);
}


/*
 *  V2I_B   V2I_B Configuration Register  Offset 0x0128
 *
 *  Register Field    Field Range
 *  v2i_b             31 : 0
 *                
 * 
 */
__STATIC_INLINE void ASysCtl_setV2iB(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_V2I_B) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getV2iB(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_V2I_B) >> 0);
}



/*
 *  BG_REG   Band Gap Configuration Register  Offset 0x0130
 *
 *  Register Field   Field Range
 *  reserved         31 : 4
 *  bg_meas          3
 *  bg_en            2
 *  reserved         1
 *  bg_chop_en       0
 * 
 */
__STATIC_INLINE void ASysCtl_setBgMeas(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG) = ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG)&(~BIT(3)))|(value << 3));
}

__STATIC_INLINE uint32_t ASysCtl_getBgMeas(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG) >> 3) & 0x01;
}

__STATIC_INLINE void ASysCtl_setBgEn(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG) = ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG)&(~BIT(2)))|(value << 2));
}

__STATIC_INLINE uint32_t ASysCtl_getBgEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG) >> 2) & 0x01;
}

__STATIC_INLINE void ASysCtl_setBgChopEn(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG) = ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG)&(~BIT(0)))|(value << 0));
}

__STATIC_INLINE uint32_t ASysCtl_getBgChopEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG) >> 0) & 0x01;
}


/*
 *  ANA_CLK_TRIM   Ana_clk Trim Configuration Register  Offset 0x4EC
 *
 *  Register Field   Field Range
 *  reserved         31 : 16
 *  ana_clk_trim     15 : 0
 * 
 */
__STATIC_INLINE void ASysCtl_setAnaClkTrim(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_CLK_TRIM) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAnaClkTrim(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_CLK_TRIM) >> 0) & 0xffff;
}


/*
 *  ANA_CLK_SPARE   Ana_clk_spare Configuration Register  Offset 0x0150
 *
 *  Register Field   Field Range
 *  ana_clk_spare    31 : 16
 * 
 */
__STATIC_INLINE void ASysCtl_setAnaClkSpare(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_CLK_SPARE) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAnaClkSpare(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_CLK_SPARE) >> 0) & 0xffff0000;
}


/*
 *  ANA_TS_CHOP_EN   Ana_ts_chop_en Configuration Register  Offset 0x0168
 *
 *  Register Field   Field Range
 *  reserved         31 : 1
 *  ana_ts_chop_en   0
 * 
 */
__STATIC_INLINE void ASysCtl_setAnaTsChopEn(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TS_CHOP_EN) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTsChopEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TS_CHOP_EN) >> 0) & 0x1;
}


/*
 *  ANA_TS_MEAS_EN   Ana_ts_meas_en Configuration Register  Offset 0x016C
 *
 *  Register Field   Field Range
 *  reserved         31 : 1
 *  ana_ts_meas_en   0
 * 
 */
__STATIC_INLINE void ASysCtl_setAnaTsMeasEn(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TS_MEAS_EN) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTsMeasEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TS_MEAS_EN) >> 0) & 0x1;
}


/*
 *  ANA_TS_SPARE   Ana_ts_spare Configuration Register  Offset 0x0174
 *
 *  Register Field   Field Range
 *  reserved         31 : 10
 *  ana_ts_spare     9 : 0
 * 
 */
__STATIC_INLINE void ASysCtl_setAnaTsSpare(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TS_SPARE) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTsSpare(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TS_SPARE) >> 0) & 0x3ff;
}

/*
 *  ANA_TEST_MUX_SEL   Ana_test_mux_sel Configuration Register  Offset 0x0198
 *
 *  Register Field    Field Range
 *  reserved          31 : 12
 *  ana_test_mux3_sel 15 : 12
 *  ana_test_mux2_sel 11 : 8
 *  ana_test_mux1_sel 7 : 4
 *  ana_test_mux0_sel 3 : 0
 */

__STATIC_INLINE void ASysCtl_setAnaTestMux3Sel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) =
            ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL)) & (~(0x0f << 12))) |(value << 12);
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTestMux3Sel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) >> 12) & 0xf;
}

__STATIC_INLINE void ASysCtl_setAnaTestMux2Sel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) =
            ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL)) & (~(0x0f << 8))) |(value << 8);
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTestMux2Sel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) >> 8) & 0xf;
}

__STATIC_INLINE void ASysCtl_setAnaTestMux1Sel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) =
            ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL)) & (~(0x0f << 4))) |(value << 4);
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTestMux1Sel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) >> 4) & 0xf;
}

__STATIC_INLINE void ASysCtl_setAnaTestMux0Sel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) =
            ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) & (~(0x0f << 0))) |(value << 0));
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTestMux0Sel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) >> 0) & 0xf;
}


/*
 *  ANA_TEST_MUX0_MODE   ANA TEST MUX Mode Configuration Register  Offset 0x0270
 *
 *  Register Field      Field Range
 *  ana_test_mux3_mode  7
 *  ana_test_mux2_mode  6
 *  ana_test_mux1_mode  5
 *  ana_test_mux0_mode  4
 *  ana_test_mux3_en    3
 *  ana_test_mux2_en    2
 *  ana_test_mux1_en    1
 *  ana_test_mux0_en    0
 *  muxId: 0, 1, 2, 3
 */
__STATIC_INLINE void ASysCtl_setAnaTestMuxMode(uint32_t muxId, uint32_t mode)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) & ~(1<<(muxId+4))) | mode << (muxId+4);
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTestMuxMode(uint32_t muxId)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) >> (muxId+4)) & 0x1;
}

__STATIC_INLINE void ASysCtl_setAnaTestMuxEn(uint32_t muxId, uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) & ~(1<<muxId)) | enable << muxId;
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTestMuxEn(uint32_t muxId)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) >> muxId) & 0x1;
}


__STATIC_INLINE void ASysCtl_setCMPEn(uint32_t enable)
{
   HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_EN) = (enable & 0x01);
}

__STATIC_INLINE uint32_t ASysCtl_getCMPEn(void)
{
   return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_EN) & 0X01;
}

//__STATIC_INLINE uint32_t ASysCtl_setCMP_HysteresisMod(uint32_t enable)
//{
//   HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_EN) |=0X01;
//}


__STATIC_INLINE void ASysCtl_setCMP_Mux_AllDisable()
{
   HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) = 0x00;
}


__STATIC_INLINE void ASysCtl_setCMP_MuxEn(ASysCtl_CMPSelect select,uint8_t hp_en,uint8_t hn_en,uint8_t lp_en,uint8_t ln_en)
{
	uint16_t value = (lp_en  & 0X01) | ((ln_en  & 0X01)<<1) | ((hn_en  & 0X01)<<2) | ((hp_en  & 0X01)<<3);

   HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) = (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) &
   ~(0xf << (uint32_t)select)) | (value << (uint32_t)select);
}

/**
 * 
 *  Select the value for CMPHNMXSEL.
 * 
 *  @param [in] select is a combination of CMPHNMXSEL values.
 * 
 *  The parameter  select can be a bitwise OR of the below values:
 * 
 *  -  ASYSCTL_CMPHNMUX_SELECT_1
 *  -  ASYSCTL_CMPHNMUX_SELECT_2
 *  -  ASYSCTL_CMPHNMUX_SELECT_3
 *  -  ASYSCTL_CMPHNMUX_SELECT_4
 * 
 *  @return None.
 * 
 */
static inline void ASysCtl_selectCMPHNMux(uint16_t select)
{
    ASSERT(select <= 0x7FU);

    EALLOW;

    //
    // Write a select to the mux select bit.
    //
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHNMXSEL) = select;

    EDIS;
}

/**
 *  
 *  Select the value for individual CMPxHNMXSEL.
 * 
 *  @param [in] select is the CMPxHNMXSEL to be set.
 *  @param [in] value is 0 or 1.
 * 
 *  The parameter  select can be one of the below values:
 * 
 *  -  ASYSCTL_CMPHNMUX_SELECT_1
 *  -  ASYSCTL_CMPHNMUX_SELECT_2
 *  -  ASYSCTL_CMPHNMUX_SELECT_3
 *  -  ASYSCTL_CMPHNMUX_SELECT_4
 * 
 *  @return None.
 *  
 */
static inline void ASysCtl_selectCMPHNMuxValue(uint16_t select, uint16_t value)
{
    ASSERT(select <= 0x7FU);
    ASSERT(value <= 1U);
    uint16_t CMPHNselect=select;
    EALLOW;

    //
    // Write a select to the mux select bit.
    //
    if(value == 0U)
    {
        HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHNMXSEL) &= ~select;
    }
    else if(value == 1U)
    {
        HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHNMXSEL) |= select;
    }
    else
    {
        //Do nothing. Not a valid value
    }
    switch (CMPHNselect) {
     case 1:
      HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) = (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN)  | (ANA_SYSCTRL_PARA_CMPX_HN_EN_M << (uint32_t)ANA_SYSCTRL_PARA_CMPX_LN_SELECT_1_S));
      break;
     case 2:
      HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) = (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN)  | (ANA_SYSCTRL_PARA_CMPX_HN_EN_M << (uint32_t)ANA_SYSCTRL_PARA_CMPX_LN_SELECT_2_S));
      break;
     case 4:
      HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) = (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN)  | (ANA_SYSCTRL_PARA_CMPX_HN_EN_M << (uint32_t)ANA_SYSCTRL_PARA_CMPX_LN_SELECT_3_S));
      break;
     case 8:
      HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) = (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN)  | (ANA_SYSCTRL_PARA_CMPX_HN_EN_M << (uint32_t)ANA_SYSCTRL_PARA_CMPX_LN_SELECT_4_S));
      break;
     case 16:
      HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) = (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN)  | (ANA_SYSCTRL_PARA_CMPX_HN_EN_M << (uint32_t)ANA_SYSCTRL_PARA_CMPX_LN_SELECT_5_S));
      break;
     case 32:
      HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) = (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN)  | (ANA_SYSCTRL_PARA_CMPX_HN_EN_M << (uint32_t)ANA_SYSCTRL_PARA_CMPX_LN_SELECT_6_S));
      break;
    case 64:
      HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) = (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN)  | (ANA_SYSCTRL_PARA_CMPX_HN_EN_M << (uint32_t)ANA_SYSCTRL_PARA_CMPX_LN_SELECT_7_S));
      break;
    default:
      break;
               }
    EDIS;
}

/**
 *  
 *  Select the value for individual CMPxLNMXSEL.
 * 
 *  @param [in] select is a combination of CMPLNMXSEL values.
 * 
 *  The parameter  select can be the bitwise OR of the below values:
 * 
 *  -  ASYSCTL_CMPLNMUX_SELECT_1
 *  -  ASYSCTL_CMPLNMUX_SELECT_2
 *  -  ASYSCTL_CMPLNMUX_SELECT_3
 *  -  ASYSCTL_CMPLNMUX_SELECT_4
 * 
 *  @return None.
 *  
 */
static inline void ASysCtl_selectCMPLNMux(uint16_t select)
{
    ASSERT(select <= 0x7FU);

    EALLOW;

    //
    // Write a select to the mux select bit.
    //
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPLNMXSEL) = select;

    EDIS;
}

/**
 *  
 *  Select the value for CMPLNMXSEL.
 * 
 *  @param [in] select is the CMPxHNMXSEL to be set.
 *  @param [in] value is 0 or 1.
 * 
 *  The parameter  select can be one of the below values:
 * 
 *  -  ASYSCTL_CMPLNMUX_SELECT_1
 *  -  ASYSCTL_CMPLNMUX_SELECT_2
 *  -  ASYSCTL_CMPLNMUX_SELECT_3
 *  -  ASYSCTL_CMPLNMUX_SELECT_4
 * 
 *  @return None.
 *  
 */
static inline void ASysCtl_selectCMPLNMuxValue(uint32_t select, uint16_t value)
{
    ASSERT(select <= 0x7FU);
    ASSERT(value <= 1U);
    uint32_t CMPLNselect=select;
    EALLOW;
    select = select<<16;
    //
    // Write a select to the mux select bit.
    //
    if(value == 0U)
    {
        HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHNMXSEL) &= ~select;
    }
    else if(value == 1U)
    {
        HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHNMXSEL) |= select;
    }
    else
    {
        //Do nothing. Not a valid value
    }
    switch (CMPLNselect) {
      case 1:
       HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) = (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN)  | (ANA_SYSCTRL_PARA_CMPX_LN_EN_M << (uint32_t)ANA_SYSCTRL_PARA_CMPX_HN_SELECT_1_S));
      break;
      case 2:
       HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) = (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN)  | (ANA_SYSCTRL_PARA_CMPX_LN_EN_M << (uint32_t)ANA_SYSCTRL_PARA_CMPX_HN_SELECT_2_S));
      break;
      case 4:
       HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) = (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN)  | (ANA_SYSCTRL_PARA_CMPX_LN_EN_M << (uint32_t)ANA_SYSCTRL_PARA_CMPX_HN_SELECT_3_S));
      break;
      case 8:
       HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) = (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN)  | (ANA_SYSCTRL_PARA_CMPX_LN_EN_M << (uint32_t)ANA_SYSCTRL_PARA_CMPX_HN_SELECT_4_S));
      break;
      case 16:
       HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) = (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN)  | (ANA_SYSCTRL_PARA_CMPX_LN_EN_M << (uint32_t)ANA_SYSCTRL_PARA_CMPX_HN_SELECT_5_S));
      break;
      case 32:
       HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) = (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN)  | (ANA_SYSCTRL_PARA_CMPX_LN_EN_M << (uint32_t)ANA_SYSCTRL_PARA_CMPX_HN_SELECT_6_S));
      break;
      case 64:
       HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) = (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN)  | (ANA_SYSCTRL_PARA_CMPX_LN_EN_M << (uint32_t)ANA_SYSCTRL_PARA_CMPX_HN_SELECT_7_S));
      break;
       default:
            break;
                  }
    EDIS;
}

/**
 *  
 *  Select the value for CMPHPMXSEL.
 * 
 *  @param [in] select is of type ASysCtl_CMPHPMuxSelect.
 *  @param [in] value is 0, 1, 2, 3, 4 or 5.
 * 
 *  This function is used to write a value to one mux select at a time.
 *  The parameter  select can be one of the following values:
 * 
 *  -  ASYSCTL_CMPHPMUX_SELECT_1
 *  -  ASYSCTL_CMPHPMUX_SELECT_2
 *  -  ASYSCTL_CMPHPMUX_SELECT_3
 *  -  ASYSCTL_CMPHPMUX_SELECT_4
 *  -  ASYSCTL_CMPHPMUX_SELECT_5
 *  -  ASYSCTL_CMPHPMUX_SELECT_6
 *  -  ASYSCTL_CMPHPMUX_SELECT_7
 * 
 *  @return None.
 *  
 */

__STATIC_INLINE void ASysCtl_selectCMPHPMux(ASysCtl_CMPHPMuxSelect select, uint32_t value)
{
//    ASSERT(value <= 5U);
//    EALLOW;
    //
    // Set the value for the appropriate Mux Select.
    //
        HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHPMXSEL) =
        (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHPMXSEL) &
        ~((uint32_t)ASYSCTL_CMPHPMXSEL_CMP1HPMXSEL_M << (uint32_t)select)) |
        (value << (uint32_t)select);


        HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) |=
                			(ANA_SYSCTRL_PARA_CMPX_HP_EN_M<<(select/3*ANA_SYSCTRL_PARA_CMPMUX_STEP ));
//    EDIS;
}
/**
 *  
 *  Select the value for CMPLPMXSEL.
 * 
 *  @param [in] select is of type ASysCtl_CMPLPMuxSelect.
 *  @param [in] value is 0, 1, 2, 3, 4 or 5.
 * 
 *  This function is used to write a value to one mux select at a time.
 *  The parameter  select can be one of the following values:
 * 
 *  -  ASYSCTL_CMPLPMUX_SELECT_1
 *  -  ASYSCTL_CMPLPMUX_SELECT_2
 *  -  ASYSCTL_CMPLPMUX_SELECT_3
 *  -  ASYSCTL_CMPLPMUX_SELECT_4
 *  -  ASYSCTL_CMPLPMUX_SELECT_5
 *  -  ASYSCTL_CMPLPMUX_SELECT_6
 *  -  ASYSCTL_CMPLPMUX_SELECT_7
 * 
 *  @return None.
 *  
 */
__STATIC_INLINE void ASysCtl_selectCMPLPMux(ASysCtl_CMPLPMuxSelect select, uint32_t value)
{
//    ASSERT(value <= 5U);
//
//    EALLOW;

    //
    // Set the value for the appropriate Mux Select.
    //
        HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPLPMXSEL) =
        (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPLPMXSEL) &
        ~((uint32_t)ASYSCTL_CMPLPMXSEL_CMP1LPMXSEL_M << (uint32_t)select)) |
        (value << (uint32_t)select);

        HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) |=
                			(ANA_SYSCTRL_PARA_CMPX_LP_EN_M<<(select/3*ANA_SYSCTRL_PARA_CMPMUX_STEP ));
//    EDIS;
}



/**
 *  
 *  Defines used for setting AnalogReference functions.
 *  ASysCtl_setAnalogReferenceInternal()
 *  ASysCtl_setAnalogReferenceExternal()
 *  ASysCtl_setAnalogReference2P5()
 *  ASysCtl_setAnalogReference1P65()
 *  
 */
#define ASYSCTL_VREFHI  0x1U //< VREFHI
//from p65(repeater) function needed.               according to latest register list not support below field
#define ASYSCTL_VREFHIA  0x1U //< VREFHIA
#define ASYSCTL_VREFHIB  0x2U //< VREFHIB
#define ASYSCTL_VREFHIC  0x4U //< VREFHIC

/**
 *  
 *  Disable temperature sensor.
 * 
 *  This function disables the temperature sensor output to the ADC.
 * 
 *  @return None.
 *  
 */
static inline void
ASysCtl_disableTemperatureSensor(void)
{
    //
    // Clear the temperature sensor enable bit.
    //
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_TSNSCTL) &= ~(ASYSCTL_TSNSCTL_ENABLE);
}

/**
 *  
 *  Enable temperature sensor.
 * 
 *  This function enables the temperature sensor output to the ADC.
 * 
 *  @return None.
 *  
 */
static inline void
ASysCtl_enableTemperatureSensor(void)
{
    EALLOW;
    HWREG(0x400CD4EC) |= (0x1<<2);
    HWREG(0x400CD0C0) |= (0x1<<0);
    HWREG(0x400CD504) |= (0x1<<0);
    HWREG(0x400CD508) |= (0x1<<0);
    HWREG(0x400CD510) |= (0x1<<0);
    //
    // Set the temperature sensor enable bit.
    //
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_TSNSCTL) |= ASYSCTL_TSNSCTL_ENABLE;

    EDIS;
}

/**
 * 
 *  Set the analog voltage reference selection to internal.
 * 
 *  @param [in] reference is the analog reference.
 * 
 *  The parameter  reference can be a combination of the following values:
 * 
 *  -  ASYSCTL_VREFHI
 * 
 *  @return None.
 * 
 */
static inline void
ASysCtl_setAnalogReferenceInternal(uint16_t reference)
{
    ASSERT((reference & (
                         ASYSCTL_VREFHI
                        )) == reference);

    //
    // Write selection to the Analog Internal Reference Select bit.
    //

    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) |= reference;


}

/**
 * 
 *  Set the analog voltage reference selection to external.
 * 
 *  @param [in] reference is the analog reference.
 * 
 *  The parameter  reference can be a combination of the following values:
 * 
 *  -  ASYSCTL_VREFHI
 * 
 *  @return None.
 * 
 */
static inline void
ASysCtl_setAnalogReferenceExternal(uint16_t reference)
{
    ASSERT((reference & (
                         ASYSCTL_VREFHI
                        )) == reference);


    ASysCtl_setAdcSpare(0, 0x06625757); //bit21:24 VCM, bit10:13 VREFHI
    ASysCtl_setAdcSpare(1, 0x06625757);
    ASysCtl_setAdcSpare(2, 0x06625757);

    //
    // Write selection to the Analog External Reference Select bit.
    //
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) &= ~reference; 					//reverse

}

/**
 * 
 *  Set the internal analog voltage reference selection to 1.65V.
 * 
 *  @param [in] reference is the analog reference.
 * 
 *  The parameter  reference can be a combination of the following values:
 * 
 *  -  ASYSCTL_VREFHI
 * 
 *  @return None.
 * 
 */
static inline void ASysCtl_setAnalogReference1P65(uint16_t reference)
{
    ASSERT((reference & (
                         ASYSCTL_VREFHI
                        )) == reference);

    if(HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL))//for internal
    {
        ASysCtl_setAdcSpare(0, 0x1662D717);
        ASysCtl_setAdcSpare(1, 0x1662D717);
        ASysCtl_setAdcSpare(2, 0x1662D717);
    }


    //
    // Write selection to the Analog Voltage Reference Select bit.
    //
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) &= ~(reference << 8U);

}

/**
 * 
 *  Set the internal analog voltage reference selection to 2.5V.
 * 
 *  @param [in] reference is the analog reference.
 * 
 *  The parameter  reference can be a combination of the following values:
 * 
 *  - \b ASYSCTL_VREFHI
 * 
 *  @return None.
 * 
 */
static inline void ASysCtl_setAnalogReference2P5(uint16_t reference)
{
    ASSERT((reference & (
                         ASYSCTL_VREFHI
                        )) == reference);

    if(HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL))//for internal
    {
        ASysCtl_setAdcSpare(0, 0x1AE1DF57);
        ASysCtl_setAdcSpare(1, 0x1AE1DF57);
        ASysCtl_setAdcSpare(2, 0x1AE1DF57);
    }

    //
    // Write selection to the Analog Voltage Reference Select bit.
    //
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) |= (reference << 8U);

}

/**
 *  
 *  Set L2H enable bit
 * 
 *  @param [in]: 0 disable, 1 enable;
 * 
 *  @return None.
 *  
 */
static inline void ASysCtl_setL2HCfg(uint32_t value)
{
    //
    // Write selection to the ADC L2H enable bit
    //
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_L2H_CFG) = value;


}

/**
 *  
 *  Set adc a/b/c top spare
 * 
 *  @param [in]: adc_base
 * 
 *  @return None.
 *  
 */
//static inline void ASysCtl_setAdcSpare(uint32_t adc_base, uint32_t value)
//{
//	uint32_t reg_offset;
//    //
//    // Write selection to the Analog Voltage Reference Select bit.
//    //
//	reg_offset = (adc_base - ADCA_BASE)/(ADCB_BASE-ADCA_BASE)
//    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_SPARE + (reg_offset * 4)) = value;
//
//}

static inline void ASysCtl_setAdcReset(uint32_t value)
{
	uint32_t reg_offset;
    //
    // Write selection to the Analog Voltage Reference Select bit.
    //
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_RST) = value;

}

void ASysCtl_initConfig(void);
void ASysCtl_trim(void);
void AsysCtl_trimAdc(uint32_t adcBase, uint32_t refMode, uint32_t refVoltage);

#endif

#ifdef __cplusplus
}
#endif


#endif /* DEVICE_DRIVERLIB_SYSCTL_ANA_H_ */
