/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    asysctl_v22.c
*   @brief   digital system top level configuration
*   @details
*
*/
/*
 * commit history
 * 20240306, Jason Yang, Add calling ADC_setINLTrim in ASysCtl_initConfig.
 *                       This is a must to get accurate ADC results.
 * 20240324, Jason Yang, add code to init AnaTop for CMPSS and enable CMPSS.
 */

#include "gs32_version.h"

#ifdef __cplusplus
extern "C"{
#endif

#if IS_GS32F3xx(0x22)          //used for 2.2CS

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "sysctl/asysctl_v22.h"
#include "adc.h"
#include "cmpss.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

uint32_t chipRevision = 0;

GS32_DRIVER_ROM_DATA_T AnaTopTrimTable_t anaTopTrimTable[] = {
    [TRIM_ID_LDO_1V2 ] = {0x400CFF00, REG_WIDTH_32BITS,  6,  2,    22,   0x1FFF8210,  4,  0},     //TOP_LDO            0   31
    [TRIM_ID_TOP_BOR ] = {0x400CD304, REG_WIDTH_32BITS,  6,  4,     2,   0x1FFF8218,  6,  4},     //TOP BOR            0   7
    [TRIM_ID_ANA_LDO ] = {0x400CD534, REG_WIDTH_32BITS,  2,  0,     0,   0x1FFF8110,  2,  0},     //LDO_ANA            0   7
    [TRIM_ID_OSC1    ] = {0x400CD314, REG_WIDTH_32BITS, 14,  0,  1397,   0x1FFF8120, 14,  0},     //OSC                0   2047
    [TRIM_ID_OSC2    ] = {0x400CD314, REG_WIDTH_32BITS, 30, 16,  1397,   0x1FFF8120, 30, 16},     //OSC                0   2047

};

/* v0 */
GS32_DRIVER_ROM_DATA_T AnaTopTrimTable_t anaTopTrimTable_v0[] = {
    [V0_TRIM_ID_ADCA_VCM_INT      ] = {0x400CD408, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF8160, 24, 21},     //VCMA               0   15
    [V0_TRIM_ID_ADCA_VCM_EXT      ] = {0x400CD408, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF8160, 28, 26},     //VCMA               0   7
    [V0_TRIM_ID_ADCA_OFF_DIR      ] = {0x400CD408, REG_WIDTH_32BITS,  9,  8,     1,   0x1FFF8140,  9,  8},     //ADCA Offset direction    0  3
    [V0_TRIM_ID_ADCA_OFFSET       ] = {0x40000076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF8170,  7,  0},     //ADCA Offset    0   255

    [V0_TRIM_ID_ADCB_VCM_INT      ] = {0x400CD40C, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF8190, 24, 21},     //VCMB               0   15
    [V0_TRIM_ID_ADCB_VCM_EXT      ] = {0x400CD40C, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF8190, 28, 26},     //VCMB               0   7
    [V0_TRIM_ID_ADCB_OFF_DIR      ] = {0x400CD40C, REG_WIDTH_32BITS,  9,  8,     1,   0x1FFF8180,  9,  8},     //ADCA Offset direction    0  3
    [V0_TRIM_ID_ADCB_OFFSET       ] = {0x40000876, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF81A0,  7,  0},     //ADCB Offset    0   255

    [V0_TRIM_ID_ADCC_VCM_INT      ] = {0x400CD410, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF81C0, 24, 21},     //VCMC               0   15
    [V0_TRIM_ID_ADCC_VCM_EXT      ] = {0x400CD410, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF81C0, 28, 26},     //VCMC               0   7
    [V0_TRIM_ID_ADCC_OFF_DIR      ] = {0x400CD410, REG_WIDTH_32BITS,  9,  8,     1,   0x1FFF81B0,  9,  8},     //ADCA Offset direction    0  3
    [V0_TRIM_ID_ADCC_OFFSET       ] = {0x40001076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF81D0,  7,  0},     //ADCC Offset    0   255

    [V0_TRIM_ID_ADCD_VCM_INT      ] = {0x400CD414, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF81F0, 24, 21},     //VCMC               0   15
    [V0_TRIM_ID_ADCD_VCM_EXT      ] = {0x400CD414, REG_WIDTH_32BITS, 28, 26,     3,   0x1FFF81F0, 28, 26},     //VCMC               0   7
    [V0_TRIM_ID_ADCD_OFF_DIR      ] = {0x400CD414, REG_WIDTH_32BITS,  9,  8,     1,   0x1FFF81E0,  9,  8},     //ADCA Offset direction    0  3
    [V0_TRIM_ID_ADCD_OFFSET       ] = {0x40001876, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF8200,  7,  0},     //ADCC Offset    0   255
};

/* v5 v7 */
GS32_DRIVER_ROM_DATA_T AdcTrim_t adcTrimTable = {
    .vrefhiCoarse = { /* checked */
        .addrMap = {
            [ADCA_ID] = {.regAddr = 0x400CD408, .efuseAddr = 0x1FFF8220},
            [ADCB_ID] = {.regAddr = 0x400CD40C, .efuseAddr = 0x1FFF8230},
            [ADCC_ID] = {.regAddr = 0x400CD410, .efuseAddr = 0x1FFF8240},
            [ADCD_ID] = {.regAddr = 0x400CD414, .efuseAddr = 0x1FFF8250},
        },

        .regWidth = REG_WIDTH_32BITS, .endbit = 13, .startbit = 10, .defaultTrimVal = 7,

        .mode = {
            [VREFHI_2P5V]   = {.efuseEndbit = 13, .efuseStartbit = 10},
            [VREFHI_1P65V]  = {.efuseEndbit = 45, .efuseStartbit = 42},
        }
    },

    .vrefhiFine = { /* checked */
        .addrMap = {
            [ADCA_ID] = {.regAddr = 0x400CD408, .efuseAddr = 0x1FFF8220},
            [ADCB_ID] = {.regAddr = 0x400CD40C, .efuseAddr = 0x1FFF8230},
            [ADCC_ID] = {.regAddr = 0x400CD410, .efuseAddr = 0x1FFF8240},
            [ADCD_ID] = {.regAddr = 0x400CD414, .efuseAddr = 0x1FFF8250},
        },

        .regWidth = REG_WIDTH_32BITS, .endbit = 17, .startbit = 15, .defaultTrimVal = 3,

        .mode = {
            [VREFHI_2P5V]   = {.efuseEndbit = 17, .efuseStartbit = 15},
            [VREFHI_1P65V]  = {.efuseEndbit = 49, .efuseStartbit = 47},
        }
    },

    .offsetDir = {  /* checked */
        .addrMap = {
            [ADCA_ID] = {.regAddr = 0x400CD408, .efuseAddr = 0x1FFF8140},
            [ADCB_ID] = {.regAddr = 0x400CD40C, .efuseAddr = 0x1FFF8180},
            [ADCC_ID] = {.regAddr = 0x400CD410, .efuseAddr = 0x1FFF81B0},
            [ADCD_ID] = {.regAddr = 0x400CD414, .efuseAddr = 0x1FFF81E0},
        },

        .regWidth = REG_WIDTH_32BITS, .endbit = 9, .startbit = 8, .defaultTrimVal = 1,

        .mode = {
            [ADC_VREFHI_MODE_3P3V_IN]  = {.efuseEndbit =  9, .efuseStartbit =  8},
            [ADC_VREFHI_MODE_2P5V_IN]  = {.efuseEndbit = 25, .efuseStartbit = 24},
            [ADC_VREFHI_MODE_3P3V_OUT] = {.efuseEndbit = 41, .efuseStartbit = 40},
            [ADC_VREFHI_MODE_2P5V_OUT] = {.efuseEndbit = 57, .efuseStartbit = 56},
            [ADC_VREFHI_MODE_3P0V_OUT] = {.efuseEndbit = 73, .efuseStartbit = 72},
        }
    },

    .offset = {/* checked */
        .addrMap = {
            [ADCA_ID] = {.regAddr = 0x40000076, .efuseAddr = 0x1FFF8170},
            [ADCB_ID] = {.regAddr = 0x40000876, .efuseAddr = 0x1FFF81A0},
            [ADCC_ID] = {.regAddr = 0x40001076, .efuseAddr = 0x1FFF81D0},
            [ADCD_ID] = {.regAddr = 0x40001876, .efuseAddr = 0x1FFF8200},
        },

        .regWidth = REG_WIDTH_16BITS, .endbit = 7, .startbit = 0, .defaultTrimVal = 0,

        .mode = {
            [ADC_VREFHI_MODE_3P3V_IN]  = {.efuseEndbit =  7, .efuseStartbit =  0},
            [ADC_VREFHI_MODE_2P5V_IN]  = {.efuseEndbit = 23, .efuseStartbit = 16},
            [ADC_VREFHI_MODE_3P3V_OUT] = {.efuseEndbit = 39, .efuseStartbit = 32},
            [ADC_VREFHI_MODE_2P5V_OUT] = {.efuseEndbit = 55, .efuseStartbit = 48},
            [ADC_VREFHI_MODE_3P0V_OUT] = {.efuseEndbit = 71, .efuseStartbit = 64},
        }
    },

    .vcmFine = {/* checked */
        .addrMap = {
            [ADCA_ID] = {.regAddr = 0x400CD408, .efuseAddr = 0x1FFF8160},
            [ADCB_ID] = {.regAddr = 0x400CD40C, .efuseAddr = 0x1FFF8190},
            [ADCC_ID] = {.regAddr = 0x400CD410, .efuseAddr = 0x1FFF81C0},
            [ADCD_ID] = {.regAddr = 0x400CD414, .efuseAddr = 0x1FFF81F0},
        },

        .regWidth = REG_WIDTH_32BITS, .endbit = 24, .startbit = 21, .defaultTrimVal = 7,

        .mode = {
            [ADC_VCM_MODE_1P65V]  = {.efuseEndbit =  24, .efuseStartbit =  21},
            [ADC_VCM_MODE_2P475V] = {.efuseEndbit =  56, .efuseStartbit =  53},
            [ADC_VCM_MODE_1P25V]  = {.efuseEndbit =  88, .efuseStartbit =  85},
            [ADC_VCM_MODE_1P5V]   = {.efuseEndbit = 120, .efuseStartbit = 117},
        }
    },

    .vcmCoarse = {/* checked */
        .addrMap = {
            [ADCA_ID] = {.regAddr = 0x400CD408, .efuseAddr = 0x1FFF8160},
            [ADCB_ID] = {.regAddr = 0x400CD40C, .efuseAddr = 0x1FFF8190},
            [ADCC_ID] = {.regAddr = 0x400CD410, .efuseAddr = 0x1FFF81C0},
            [ADCD_ID] = {.regAddr = 0x400CD414, .efuseAddr = 0x1FFF81F0},
        },

        .regWidth = REG_WIDTH_32BITS, .endbit = 28, .startbit = 26, .defaultTrimVal = 3,

        .mode = {
            [ADC_VCM_MODE_1P65V]  = {.efuseEndbit =  28, .efuseStartbit =  26},
            [ADC_VCM_MODE_2P475V] = {.efuseEndbit =  60, .efuseStartbit =  58},
            [ADC_VCM_MODE_1P25V]  = {.efuseEndbit =  92, .efuseStartbit =  90},
            [ADC_VCM_MODE_1P5V]   = {.efuseEndbit = 124, .efuseStartbit = 122},
        }
    },

    .digOffset = {
        .addrMap = {
            [ADCA_ID] = {.regAddr = 0x40000248, .efuseAddr = 0x1FFF8260},
            [ADCB_ID] = {.regAddr = 0x40000A48, .efuseAddr = 0x1FFF8280},
            [ADCC_ID] = {.regAddr = 0x40001248, .efuseAddr = 0x1FFF82A0},
            [ADCD_ID] = {.regAddr = 0x40001A48, .efuseAddr = 0x1FFF82C0},
        },

        .regWidth = REG_WIDTH_32BITS, .endbit = 9, .startbit = 0, .defaultTrimVal = 0,

        .mode = {
            [ADC_VREFHI_MODE_3P3V_IN]  = {.efuseEndbit =   9, .efuseStartbit =   0},
            [ADC_VREFHI_MODE_2P5V_IN]  = {.efuseEndbit =  41, .efuseStartbit =  32},
            [ADC_VREFHI_MODE_3P3V_OUT] = {.efuseEndbit =  73, .efuseStartbit =  64},
            [ADC_VREFHI_MODE_2P5V_OUT] = {.efuseEndbit = 105, .efuseStartbit =  96},
            [ADC_VREFHI_MODE_3P0V_OUT] = {.efuseEndbit = 137, .efuseStartbit = 128},
        }
    },

    .digGainErr= {
        .addrMap = {
            [ADCA_ID] = {.regAddr = 0x40000244, .efuseAddr = 0x1FFF8260},
            [ADCB_ID] = {.regAddr = 0x40000A44, .efuseAddr = 0x1FFF8280},
            [ADCC_ID] = {.regAddr = 0x40001244, .efuseAddr = 0x1FFF82A0},
            [ADCD_ID] = {.regAddr = 0x40001A44, .efuseAddr = 0x1FFF82C0},
        },

        .regWidth = REG_WIDTH_32BITS, .endbit = 11, .startbit = 0, .defaultTrimVal = 0x800,

        .mode = {
            [ADC_VREFHI_MODE_3P3V_IN]  = {.efuseEndbit =  27, .efuseStartbit =  16},
            [ADC_VREFHI_MODE_2P5V_IN]  = {.efuseEndbit =  59, .efuseStartbit =  48},
            [ADC_VREFHI_MODE_3P3V_OUT] = {.efuseEndbit =  91, .efuseStartbit =  80},
            [ADC_VREFHI_MODE_2P5V_OUT] = {.efuseEndbit = 123, .efuseStartbit = 112},
            [ADC_VREFHI_MODE_3P0V_OUT] = {.efuseEndbit = 155, .efuseStartbit = 144},
        }
    },
};


/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

GS32_DRIVER_INIT_FUNC_T void
ASysCtl_initConfig(void)
{
    ASysCtl_setTopBgTestMuxSel(2);
	ASysCtl_setTopBgChopEn(1);

    ASysCtl_setEflashBorTestEn(1);

	//
	//ASysCtl init for CMPSS
	//
	ASysCtl_setAnaClkTrim(0x45);        //ANA_CLK_TRIM

	ASysCtl_setL2HCfg(0x1);				//adc_12h_en

	ASysCtl_setAdcTopSpare(0xC1001);	//ADC_TOP_SPARE

	ASysCtl_setAdcReset(0x0);			// reset adc abcd

	////	ASysCtl_setAdcSpare(0, 0x0E625457);     //ADCA spare
	////	ASysCtl_setAdcSpare(1, 0x0E625457);     //ADCB spare
	////	ASysCtl_setAdcSpare(2, 0x0E625457);     //ADCC spare
	//based on internal 2.5V
	ASysCtl_setAdcaSpare(0x1AE1DC57);   //adca_spare
	ASysCtl_setAdcbSpare(0x1AE1DC57);   //adcb_spare
	ASysCtl_setAdccSpare(0x1AE1DC57);   //adcc_spare
	ASysCtl_setAdcdSpare(0x1AE1DC57);   //adcd_spare


	//
	//Set ADC to external reference mode by default
	//
	ASysCtl_setAnalogReferenceExternal(ASYSCTL_VREFHIA | ASYSCTL_VREFHIB | ASYSCTL_VREFHIC | ASYSCTL_VREFHID);

	//HWREG(0x400008E0) = 0X5204;			//adca_inl_trim_31to0
	//ADC_setINLTrim(ADCA_BASE);
	//ADC_setINLTrim(ADCB_BASE);
	//ADC_setINLTrim(ADCC_BASE);

    //Avoid user trigger default SOC of too narrow acqps
    for(int i = 0; i < 16 ; i++)
    {
    	ADC_setupSOC(ADCA_BASE, i, ADC_TRIGGER_SW_ONLY, ADC_CH_ADCIN0, 3U);
    	ADC_setupSOC(ADCB_BASE, i, ADC_TRIGGER_SW_ONLY, ADC_CH_ADCIN0, 3U);
    	ADC_setupSOC(ADCC_BASE, i, ADC_TRIGGER_SW_ONLY, ADC_CH_ADCIN0, 3U);
    }


	//
	//Configure ADC trim value for external reference mode, they can be reconfigured later
	//
//	ASysCtl_setAdcSpare(0, 0x0E625457);     //ADCA spare
//	ASysCtl_setAdcSpare(1, 0x0E625457);     //ADCB spare
//	ASysCtl_setAdcSpare(2, 0x0E625457);     //ADCC spare

    //Let ADC leave reset state
	ASysCtl_setAdcReset(0x00);              //release ADCA/B/C

	//
	//ASysCtl init for CMPSS
	//
    //ASysCtl_setCmpSpare(0x0FFFFFFF);        //enable hysteresis on all channels
	//ASysCtl_setCmpSpare(CMP_ID_1, 0x0FFFFFFF);
	//ASysCtl_setCmpSpare(CMP_ID_2, 0x0FFFFFFF);

    ASysCtl_setCmpEnable(0x01);

    ASysCtl_setCmpReset(0x0);

    //set all LP source to external input by default
    CMPSS_setCmpLpSrc(CMPSS1_BASE, 1);
    CMPSS_setCmpLpSrc(CMPSS2_BASE, 1);
    CMPSS_setCmpLpSrc(CMPSS3_BASE, 1);
    CMPSS_setCmpLpSrc(CMPSS4_BASE, 1);
    CMPSS_setCmpLpSrc(CMPSS5_BASE, 1);
    CMPSS_setCmpLpSrc(CMPSS6_BASE, 1);
    CMPSS_setCmpLpSrc(CMPSS7_BASE, 1);
    CMPSS_setCmpLpSrc(CMPSS8_BASE, 1);
    CMPSS_setCmpLpSrc(CMPSS9_BASE, 1);
    CMPSS_setCmpLpSrc(CMPSS10_BASE, 1);
    CMPSS_setCmpLpSrc(CMPSS11_BASE, 1);
    CMPSS_setCmpLpSrc(CMPSS12_BASE, 1);

	//
	//ASysCtl init for DAC
	//
    //HWREG(ANALOGSUBSYS_BASE + 0X4B8) = 0X1;
    ASysCtl_setDacTopEnable(0x01);
	//HWREG(0x40080002) = 0X1; //select VREFHI as DAC reference
	//HWREG(ANALOGSUBSYS_BASE + 0X4B8) = 0X1;
    ASysCtl_setDacTopEnable(0x01);
	//HWREG(ANALOGSUBSYS_BASE + 0x4BC) = 0X0;
    ASysCtl_setDacReset(0x00);
}

GS32_DRIVER_ASYSCTL_FUNC_T static void
AsysCtl_trimTop(AnaTopTrimTable_t *trim)
{
    uint32_t shift, mask;
    shift = trim->startbit;
    mask = (1<<(trim->endbit - trim->startbit + 1)) - 1;

    if (HWREG(trim->efuseAddr) != 0xFFFFFFFF) {
        uint32_t efuse_shift, efuse_mask;
        efuse_shift = trim->efuseStartbit;
        efuse_mask = (1<<(trim->efuseEndbit - trim->efuseStartbit + 1)) - 1;

        if((trim->regWidth == REG_WIDTH_16BITS))
            HWREGH(trim->regAddr) = (HWREGH(trim->regAddr) & ~(mask << shift)) | (((HWREGH(trim->efuseAddr) >> efuse_shift) & efuse_mask) << shift);
        else
            HWREG(trim->regAddr) = (HWREG(trim->regAddr) & ~(mask << shift)) | (((HWREG(trim->efuseAddr) >> efuse_shift) & efuse_mask) << shift);
    }
}

GS32_DRIVER_ASYSCTL_FUNC_T static void
AsysCtl_trimAdcParam(uint32_t id, AdcModeParam_t *trim, uint32_t mode)
{
    uint32_t shift, mask;
    shift = trim->startbit;
    mask = (1<<(trim->endbit - trim->startbit + 1)) - 1;

    if (HWREG(trim->addrMap[id].efuseAddr) != 0xFFFFFFFF) {
        uint32_t efuse_shift, efuse_mask;
        uint32_t efuseAddr;
        efuse_mask = (1<<(trim->mode[mode].efuseEndbit - trim->mode[mode].efuseStartbit + 1)) - 1;

        if((trim->regWidth == REG_WIDTH_16BITS)) {
            efuseAddr = trim->addrMap[id].efuseAddr + (trim->mode[mode].efuseStartbit/16) *2;
            efuse_shift = trim->mode[mode].efuseStartbit % 16;

            HWREGH(trim->addrMap[id].regAddr) = (HWREGH(trim->addrMap[id].regAddr) & ~(mask << shift)) | (((HWREGH(efuseAddr) >> efuse_shift) & efuse_mask) << shift);
        } else {
            efuseAddr = trim->addrMap[id].efuseAddr + (trim->mode[mode].efuseStartbit/32) *4;
            efuse_shift = trim->mode[mode].efuseStartbit % 32;

            HWREG(trim->addrMap[id].regAddr) = (HWREG(trim->addrMap[id].regAddr) & ~(mask << shift)) | (((HWREG(efuseAddr) >> efuse_shift) & efuse_mask) << shift);
        }
    }
}

GS32_DRIVER_ASYSCTL_FUNC_T static void
AsysCtl_trimPeri_v0(uint32_t id)
{
    AsysCtl_trimTop(&anaTopTrimTable_v0[id]);
}

GS32_DRIVER_ASYSCTL_FUNC_T static void
AsysCtl_trimAdc_v0(uint32_t adcBase, uint32_t refMode, uint32_t refVolt)
{
	if(refMode == ADC_REFERENCE_INTERNAL)
	{
//		if (refVolt == ADC_REFERENCE_3_3V) {
//			AsysCtl_trimPeri(TRIM_ID_VREFHI_COARSE_3V3);    //bit17:15 bit13:10 VREFHI trim for 1.65V mode
//			AsysCtl_trimPeri(TRIM_ID_VREFHI_FINE_3V3);
//		} else if (refVolt == ADC_REFERENCE_2_5V) {
//			AsysCtl_trimPeri(TRIM_ID_VREFHI_COARSE_2V5);    //bit17:15 bit13:10 VREFHI trim for 2.5V mode
//			AsysCtl_trimPeri(TRIM_ID_VREFHI_FINE_2V5);
//		}
	}
	else
	{
		AsysCtl_trimPeri_v0(V0_TRIM_ID_ADCA_VCM_EXT);     //bit28:26, VCM trim for external mode
		AsysCtl_trimPeri_v0(V0_TRIM_ID_ADCB_VCM_EXT);
		AsysCtl_trimPeri_v0(V0_TRIM_ID_ADCC_VCM_EXT);
		AsysCtl_trimPeri_v0(V0_TRIM_ID_ADCD_VCM_EXT);
	}

	AsysCtl_trimPeri_v0(V0_TRIM_ID_ADCA_VCM_INT);         //bit24:21, VCM trim for internal mode
	AsysCtl_trimPeri_v0(V0_TRIM_ID_ADCB_VCM_INT);
	AsysCtl_trimPeri_v0(V0_TRIM_ID_ADCC_VCM_INT);
	AsysCtl_trimPeri_v0(V0_TRIM_ID_ADCD_VCM_INT);

    AsysCtl_trimPeri_v0(V0_TRIM_ID_ADCA_OFFSET);            //Individual ADC OffsetTrim register
    AsysCtl_trimPeri_v0(V0_TRIM_ID_ADCB_OFFSET);
    AsysCtl_trimPeri_v0(V0_TRIM_ID_ADCC_OFFSET);
	AsysCtl_trimPeri_v0(V0_TRIM_ID_ADCD_OFFSET);
}

GS32_DRIVER_ASYSCTL_FUNC_T static void
AsysCtl_trimAdc_v5(uint32_t adcBase, uint32_t refMode, uint32_t refVolt)
{
    uint32_t adcId = ADCA_ID;
    uint32_t vcmMode = ADC_VCM_MODE_1P65V;

    switch (adcBase) {
        case ADCA_BASE: adcId = ADCA_ID; break;
        case ADCB_BASE: adcId = ADCB_ID; break;
        case ADCC_BASE: adcId = ADCC_ID; break;
        case ADCD_BASE: adcId = ADCD_ID; break;
    }

    if (refMode == ADC_REFERENCE_INTERNAL && refVolt == ADC_REFERENCE_3_3V) {
        vcmMode = ADC_VCM_MODE_2P475V;
    } else if (refMode == ADC_REFERENCE_INTERNAL && refVolt == ADC_REFERENCE_2_5V) {
        vcmMode = ADC_VCM_MODE_1P25V;
    } else if (refMode == ADC_REFERENCE_EXTERNAL && refVolt == ADC_REFERENCE_3_3V) {
        vcmMode = ADC_VCM_MODE_1P65V;
    } else if (refMode == ADC_REFERENCE_EXTERNAL && refVolt == ADC_REFERENCE_2_5V) {
        vcmMode = ADC_VCM_MODE_1P25V;
    }

    /* VREFHI */
    if (refMode == ADC_REFERENCE_INTERNAL) {
        if (refVolt == ADC_REFERENCE_2_5V) {
            AsysCtl_trimAdcParam(adcId, &adcTrimTable.vrefhiCoarse, VREFHI_2P5V);
            AsysCtl_trimAdcParam(adcId, &adcTrimTable.vrefhiFine, VREFHI_2P5V);
        } else {
            AsysCtl_trimAdcParam(adcId, &adcTrimTable.vrefhiCoarse, VREFHI_1P65V);
            AsysCtl_trimAdcParam(adcId, &adcTrimTable.vrefhiFine, VREFHI_1P65V);
        }
    }

    /* VCM */
    AsysCtl_trimAdcParam(adcId, &adcTrimTable.vcmCoarse, vcmMode);
    AsysCtl_trimAdcParam(adcId, &adcTrimTable.vcmFine, vcmMode);

    /* offset */
    AsysCtl_trimAdcParam(adcId, &adcTrimTable.offsetDir, 0);
    AsysCtl_trimAdcParam(adcId, &adcTrimTable.offset, 0);
}

GS32_DRIVER_ASYSCTL_FUNC_T static void
AsysCtl_trimAdc_v7(uint32_t adcBase, uint32_t refMode, uint32_t refVolt)
{
    uint32_t adcId = ADCA_ID;
    uint32_t adcMode = ADC_VREFHI_MODE_3P3V_IN;
    uint32_t vcmMode = ADC_VCM_MODE_1P65V;

    switch (adcBase) {
        case ADCA_BASE: adcId = ADCA_ID; break;
        case ADCB_BASE: adcId = ADCB_ID; break;
        case ADCC_BASE: adcId = ADCC_ID; break;
        case ADCD_BASE: adcId = ADCD_ID; break;
    }

    if (refMode == ADC_REFERENCE_INTERNAL && refVolt == ADC_REFERENCE_3_3V) {
        adcMode = ADC_VREFHI_MODE_3P3V_IN;
        vcmMode = ADC_VCM_MODE_2P475V;
    } else if (refMode == ADC_REFERENCE_INTERNAL && refVolt == ADC_REFERENCE_2_5V) {
        adcMode = ADC_VREFHI_MODE_2P5V_IN;
        vcmMode = ADC_VCM_MODE_1P25V;
    } else if (refMode == ADC_REFERENCE_EXTERNAL && refVolt == ADC_REFERENCE_3_3V) {
        adcMode = ADC_VREFHI_MODE_3P3V_OUT;
        vcmMode = ADC_VCM_MODE_1P65V;
    } else if (refMode == ADC_REFERENCE_EXTERNAL && refVolt == ADC_REFERENCE_2_5V) {
        adcMode = ADC_VREFHI_MODE_2P5V_OUT;
        vcmMode = ADC_VCM_MODE_1P25V;
    } else if (refMode == ADC_REFERENCE_EXTERNAL && refVolt == ADC_REFERENCE_3_0V) {
        adcMode = ADC_VREFHI_MODE_3P0V_OUT;
        vcmMode = ADC_VCM_MODE_1P5V;
    }

    /* VREFHI */
    if (refMode == ADC_REFERENCE_INTERNAL) {
        if (refVolt == ADC_REFERENCE_2_5V) {
            AsysCtl_trimAdcParam(adcId, &adcTrimTable.vrefhiCoarse, VREFHI_2P5V);
            AsysCtl_trimAdcParam(adcId, &adcTrimTable.vrefhiFine, VREFHI_2P5V);
        } else {
            AsysCtl_trimAdcParam(adcId, &adcTrimTable.vrefhiCoarse, VREFHI_1P65V);
            AsysCtl_trimAdcParam(adcId, &adcTrimTable.vrefhiFine, VREFHI_1P65V);
        }
    }

    /* VCM */
    AsysCtl_trimAdcParam(adcId, &adcTrimTable.vcmCoarse, vcmMode);
    AsysCtl_trimAdcParam(adcId, &adcTrimTable.vcmFine, vcmMode);

    /* offset */
    AsysCtl_trimAdcParam(adcId, &adcTrimTable.offsetDir, adcMode);
    AsysCtl_trimAdcParam(adcId, &adcTrimTable.offset, adcMode);

    //enable digital trim
    HWREG(adcBase + 0x248) |= 1<<16;        //adc_pre_pro_en
    /* digital */
    AsysCtl_trimAdcParam(adcId, &adcTrimTable.digOffset, adcMode);
    AsysCtl_trimAdcParam(adcId, &adcTrimTable.digGainErr, adcMode);
}

GS32_DRIVER_INIT_FUNC_T
void AsysCtl_trimBor(void)
{
    //SysCtl_setBorLvmonDisable(1);

    AsysCtl_trimTop(&anaTopTrimTable[TRIM_ID_TOP_BOR]);

    //SysCtl_setBorLvmonDisable(0);
}

GS32_DRIVER_INIT_FUNC_T
void ASysCtl_trim(void)
{
    uint32_t id;
    uint32_t version = *((uint32_t *)ANALOG_TRIM_EFUSE_VERSION_ADDR);

    ASysCtl_setLdo1v1TestMode(2);
    ASysCtl_setLdoChopEn(1);     //enable internal vcore ldo chop

    AsysCtl_trimTop(&anaTopTrimTable[TRIM_ID_LDO_1V2]);
    AsysCtl_trimTop(&anaTopTrimTable[TRIM_ID_ANA_LDO]);
    AsysCtl_trimTop(&anaTopTrimTable[TRIM_ID_OSC1]);
    AsysCtl_trimTop(&anaTopTrimTable[TRIM_ID_OSC2]);

    AsysCtl_trimBor();

    if (HWREG(anaTopTrimTable[TRIM_ID_TOP_BOR].efuseAddr) != 0xFFFFFFFF) {
        chipRevision = 3;
    }

#ifdef TOP_LDO_CAL_MODE
    uint16 top_ldo_val = 0;
    top_ldo_val = HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG)&0x1f;

    for(uint16_t data = top_ldo_val;data <= 0x1f;data++)
    {
        HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) = (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) &~(0x1f)) | data;

    }
#endif
}

GS32_DRIVER_ASYSCTL_FUNC_T void
AsysCtl_trimAdc(uint32_t adcBase, uint32_t refMode, uint32_t refVolt)
{
	 uint32_t version = *((uint32_t *)ANALOG_TRIM_EFUSE_VERSION_ADDR);

	 if (version == 0xFFFFFFFF) {
		 AsysCtl_trimAdc_v0(adcBase, refMode, refVolt);
	 } else if (version == 5) {
		 AsysCtl_trimAdc_v5(adcBase, refMode, refVolt);
	 } else if (version >= 7) {
         AsysCtl_trimAdc_v7(adcBase, refMode, refVolt);
     }
}

#endif

#ifdef __cplusplus
}
#endif
