/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    asysctl_v30.h
*   @brief   analog system top level configuration
*            do NOT include this file directly, include "sysctl.h" instead
*
*/
/**
 * commit history
 * 20241218,Hezhiyuan,Modify comments
*/
#ifndef DEVICE_DRIVERLIB_SYSCTL_ANA_V30_H_
#define DEVICE_DRIVERLIB_SYSCTL_ANA_V30_H_

#include "gs32_version.h"

#ifdef __cplusplus
extern "C"{
#endif

#if IS_GS32F00xx(0x30)
/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "inc/hw_types.h"
#include "inc/hw_memmap.h"
#include "inc/hw_asysctl_v30.h"
#include "debug.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/**
 * @brief OSCSTATUS value on completion
 */
#define ASYSCTL_EXTR_ENABLE_COMPLETE	(0xE7U)

/*
 * @brief Defines used for setting AnalogReference functions.
 */
#define ASYSCTL_VREFHI  0x1U

/**
 * @brief This macro definition can be used for functions
 * 		  ASysCtl_enableCMPSSExternalDAC() and
 * 		  ASysCtl_disableCMPSSExternalDAC().
 */
#define ASYSCTL_CMPSS1_DACL 0x1U

/**
 * @brief This macro definition can be used for functions
 * 		  ASysCtl_selectCMPHNMux() and
 * 		  ASysCtl_selectCMPHNMuxValue().
 */
#define ASYSCTL_CMPHNMUX_SELECT_1 0x1U /* CMPHNMUX select 1 */
#define ASYSCTL_CMPHNMUX_SELECT_2 0x2U /* CMPHNMUX select 2 */
#define ASYSCTL_CMPHNMUX_SELECT_3 0x4U /* CMPHNMUX select 3 */
#define ASYSCTL_CMPHNMUX_SELECT_4 0x8U /* CMPHNMUX select 4 */

/**
 * @brief This macro definition can be used for functions
 * 		  ASysCtl_selectCMPLNMux() and
 * 		  ASysCtl_selectCMPLNMuxValue().
 */
#define ASYSCTL_CMPLNMUX_SELECT_1 0x1U /* CMPLNMUX select 1 */
#define ASYSCTL_CMPLNMUX_SELECT_2 0x2U /* CMPLNMUX select 2 */
#define ASYSCTL_CMPLNMUX_SELECT_3 0x4U /* CMPLNMUX select 3 */
#define ASYSCTL_CMPLNMUX_SELECT_4 0x8U /* CMPLNMUX select 4 */

/**
 * @brief ASysCtl_CMPHPMuxSelect used for function ASysCtl_selectCMPHPMux().
 */
typedef enum
{
    ASYSCTL_CMPHPMUX_SELECT_1 = 0U, /* CMPHPMUX select 1 */
    ASYSCTL_CMPHPMUX_SELECT_2 = 3U, /* CMPHPMUX select 2 */
    ASYSCTL_CMPHPMUX_SELECT_3 = 6U, /* CMPHPMUX select 3 */
    ASYSCTL_CMPHPMUX_SELECT_4 = 9U  /* CMPHPMUX select 4 */
} ASysCtl_CMPHPMuxSelect;

/**
 * @brief ASysCtl_CMPLPMuxSelect used for function ASysCtl_selectCMPLPMux().
 */
typedef enum
{
    ASYSCTL_CMPLPMUX_SELECT_1 = 0U, /* CMPLPMUX select 1 */
    ASYSCTL_CMPLPMUX_SELECT_2 = 3U, /* CMPLPMUX select 2 */
    ASYSCTL_CMPLPMUX_SELECT_3 = 6U, /* CMPLPMUX select 3 */
    ASYSCTL_CMPLPMUX_SELECT_4 = 9U  /* CMPLPMUX select 4 */
} ASysCtl_CMPLPMuxSelect;

/**
 * @brief ASysCtl_TestSelect used for function ASysCtl_selectInternalTestNode().
 */
typedef enum
{
    ASYSCTL_TEST_NODE_NO_CONN       	  = 0U,  /* No Internal Connection       */
    ASYSCTL_TEST_NODE_VDDCORE         	  = 1U,  /* Core VDD (1.2V) voltage      */
    ASYSCTL_TEST_NODE_VREFLO              = 2U,  /* VREFLO pin voltage           */
    ASYSCTL_TEST_NODE_CDAC1H              = 4U,  /* CMPSS1 High DAC output       */
    ASYSCTL_TEST_NODE_CDAC1L              = 5U,  /* CMPSS1 Low DAC output        */
    ASYSCTL_TEST_NODE_CDAC2H              = 6U,  /* CMPSS2 High DAC output       */
    ASYSCTL_TEST_NODE_CDAC2L              = 7U,  /* CMPSS2 Low DAC output        */
    ASYSCTL_TEST_NODE_CDAC3H              = 8U,  /* CMPSS3 High DAC output       */
    ASYSCTL_TEST_NODE_CDAC3L              = 9U,  /* CMPSS3 Low DAC output        */
    ASYSCTL_TEST_NODE_CDAC4H              = 10U, /* CMPSS4 High DAC output       */
    ASYSCTL_TEST_NODE_CDAC4L              = 11U, /* CMPSS4 Low DAC output        */
    ASYSCTL_TEST_NODE_VDDA                = 12U, /* VDDA voltage                 */
    ASYSCTL_TEST_NODE_VSSA                = 13U, /* VSSA - Analog ground pin     */
    ASYSCTL_TEST_NODE_ENZ_CALIB_GAIN_3P3V = 19U  /* All ADCs are placed in gain  */
                                                 /* calibration mode             */
} ASysCtl_TestSelect;

typedef enum
{
    ASYSCTL_CMP_SELECT_1 = 0U,  //< CMP select 1
    ASYSCTL_CMP_SELECT_2 = 4U,  //< CMP select 2
    ASYSCTL_CMP_SELECT_3 = 8U,  //< CMP select 3
    ASYSCTL_CMP_SELECT_4 = 12U, //< CMP select 4
} ASysCtl_CMPSelect;

#define CMP_HP_LP_0		(0U)
#define CMP_HP_LP_1		(1U)
#define CMP_HP_LP_2		(2U)
#define CMP_HP_LP_3		(3U)
#define CMP_HP_LP_4		(4U)

#define CMP_HN_LN_0		(0U)
#define CMP_HN_LN_1		(1U)

/**
 * @brief  The following macro definitions are used for the ASysCtl_setCMP_MuxEn().
 */
#define CMPSS_DISABLE_ALL (0U)
#define CMPSS_LN_EN		(1U)
#define CMPSS_LP_EN		(2U)
#define CMPSS_HN_EN		(3U)
#define CMPSS_HP_EN		(8U)

#define EFUSE_ADCA_OFFSET_DIR	(0x1FFF4240U)
#define EFUSE_ADC_VREFHI		(0x1FFF4250U)
#define EFUSE_VCMA				(0x1FFF4260U)
#define EFUSE_ADCA_OFFSET_VAL	(0x1FFF4270U)
#define EFUSE_ADCC_OFFSET_DIR	(0x1FFF4280U)
#define EFUSE_VCMB				(0x1FFF4290U)
#define EFUSE_ADCC_OFFSET_VAL	(0x1FFF42A0U)
#define EFUSE_ADCA_DIG_TRIM		(0x1FFF42B0U)
#define EFUSE_ADCB_DIG_TRIM		(0x1FFF42D0U)

#define ADC_VREFHI_MASK		(0xFFF843FFU)
#define ADC_VCM_MASK		(0xC21FFFFFU)
#define ADC_OFFDIR_MASK		(0xFFFFFCFFU)
#define ADC_OFFSET_MASK		(0xFF00U)
#define ADC_DIG_GAIN_MASK	(0xFFFFF000U)
#define ADC_DIG_OFF_MASK	(0xFFFFFC00U)


#define ANALOG_TRIM_EFUSE_INFO_BASE     	0x1FFF4200
#define ANALOG_TRIM_EFUSE_VERSION_ADDR    	0x1FFF4100

typedef enum AnaTrimPeriId_v1 {
    V1_TRIM_ID_LDO_1V2 = 0,
    V1_TRIM_ID_ANA_LDO,
    V1_TRIM_ID_OSC1,
    V1_TRIM_ID_OSC2_FINE_IN,
	V1_TRIM_ID_OSC2_FINE_OUT,
	V1_TRIM_ID_OSC2_COARSE_IN,

    V1_TRIM_ID_VREFHI_COARSE_2P5V,
    V1_TRIM_ID_VREFHI_FINE_2P5V,

    V1_TRIM_ID_VREFHI_COARSE_1P65V,
    V1_TRIM_ID_VREFHI_FINE_1P65V,

    //ADCA
    V1_TRIM_ID_ADCA_VCM_COARSE_1P65V,
    V1_TRIM_ID_ADCA_VCM_FINE_1P65V,
    V1_TRIM_ID_ADCA_VCM_COARSE_2P475V,
    V1_TRIM_ID_ADCA_VCM_FINE_2P475V,
    V1_TRIM_ID_ADCA_VCM_COARSE_1P25V,
    V1_TRIM_ID_ADCA_VCM_FINE_1P25V,
    V1_TRIM_ID_ADCA_VCM_COARSE_1P5V,
    V1_TRIM_ID_ADCA_VCM_FINE_1P5V,

    V1_TRIM_ID_ADCA_OFF_DIR_VREFHI_3P3V_IN,
    V1_TRIM_ID_ADCA_OFF_DIR_VREFHI_2P5V_IN,
    V1_TRIM_ID_ADCA_OFF_DIR_VREFHI_3P3V_OUT,
    V1_TRIM_ID_ADCA_OFF_DIR_VREFHI_2P5V_OUT,
    V1_TRIM_ID_ADCA_OFF_DIR_VREFHI_3P0V_OUT,

    V1_TRIM_ID_ADCA_OFFSET_VREFHI_3P3V_IN,
    V1_TRIM_ID_ADCA_OFFSET_VREFHI_2P5V_IN,
    V1_TRIM_ID_ADCA_OFFSET_VREFHI_3P3V_OUT,
    V1_TRIM_ID_ADCA_OFFSET_VREFHI_2P5V_OUT,
    V1_TRIM_ID_ADCA_OFFSET_VREFHI_3P0V_OUT,

    //ADCC
    V1_TRIM_ID_ADCC_VCM_COARSE_1P65V,
    V1_TRIM_ID_ADCC_VCM_FINE_1P65V,
    V1_TRIM_ID_ADCC_VCM_COARSE_2P475V,
    V1_TRIM_ID_ADCC_VCM_FINE_2P475V,
    V1_TRIM_ID_ADCC_VCM_COARSE_1P25V,
    V1_TRIM_ID_ADCC_VCM_FINE_1P25V,
    V1_TRIM_ID_ADCC_VCM_COARSE_1P5V,
    V1_TRIM_ID_ADCC_VCM_FINE_1P5V,

    V1_TRIM_ID_ADCC_OFF_DIR_VREFHI_3P3V_IN,
    V1_TRIM_ID_ADCC_OFF_DIR_VREFHI_2P5V_IN,
    V1_TRIM_ID_ADCC_OFF_DIR_VREFHI_3P3V_OUT,
    V1_TRIM_ID_ADCC_OFF_DIR_VREFHI_2P5V_OUT,
    V1_TRIM_ID_ADCC_OFF_DIR_VREFHI_3P0V_OUT,

    V1_TRIM_ID_ADCC_OFFSET_VREFHI_3P3V_IN,
    V1_TRIM_ID_ADCC_OFFSET_VREFHI_2P5V_IN,
    V1_TRIM_ID_ADCC_OFFSET_VREFHI_3P3V_OUT,
    V1_TRIM_ID_ADCC_OFFSET_VREFHI_2P5V_OUT,
    V1_TRIM_ID_ADCC_OFFSET_VREFHI_3P0V_OUT,

    //digital
    V1_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_3P3V_IN,
    V1_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_3P3V_IN,
    V1_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_2P5V_IN,
    V1_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_2P5V_IN,
    V1_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_3P3V_OUT,
    V1_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_3P3V_OUT,
    V1_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_2P5V_OUT,
    V1_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_2P5V_OUT,
    V1_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_3P0V_OUT,
    V1_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_3P0V_OUT,

    V1_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_3P3V_IN,
    V1_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_3P3V_IN,
    V1_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_2P5V_IN,
    V1_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_2P5V_IN,
    V1_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_3P3V_OUT,
    V1_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_3P3V_OUT,
    V1_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_2P5V_OUT,
    V1_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_2P5V_OUT,
    V1_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_3P0V_OUT,
    V1_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_3P0V_OUT,

    V1_TRIM_ID_MAX,
}AnaTrimPeriId_v1_t;

#define REG_WIDTH_16BITS	2
#define REG_WIDTH_32BITS	4

typedef struct AnaTrimTable {
    uint32_t regAddr;
    uint32_t regWidth : 3;  //bytes
    uint32_t endbit : 5;
    uint32_t startbit : 5;
    uint32_t defaultTrimVal : 19;
    uint32_t efuseAddr;
    uint16_t efuseEndbit;
    uint16_t efuseStartbit;
}AnaTrimTable_t;

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

//}ADC_SPARE_STR;

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */

void ASysCtl_initConfig(void);


/**
 * @brief Disable write permission on all registers protected by lock.
 */
__STATIC_INLINE void ASysCtl_setAnaSysctrlLock(void)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_CFG_LOCK) = 0x01;
}

/**
 * @brief Enable write permission on all registers protected by lock.
 */
__STATIC_INLINE void ASysCtl_setAnaSysctrlUnLock(void)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_CFG_LOCK) = 0x5A5A5A5A;
}

/**
 * @brief Gets the current value of the ANA_SYSCTRL_LOCK register.
 */
__STATIC_INLINE uint32_t ASysCtl_getAnaSysctrlLock(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_CFG_LOCK);
}


/* Here are the APIs related to the Temperature Sensor */
/*
 * @brief Enable temperature sensor.
 *
 * @details This function enables the temperature sensor output to the ADC.
 *
 */
__STATIC_INLINE void
ASysCtl_enableTemperatureSensor(void)
{
    HWREG(0x400D6508) |= (0x1<<0);//ana_ts_meas_en
    HWREG(0x400D6510) |= (0x1<<0);//ana_ts_spare
//    HWREG(0x400D6528) = 0x22;//ana_test_mux1_en,ana_test_mux1_mode
//    HWREG(0x400D6524) &= ~(0X7);//ana_test_mux1_sel
	/* Set the temperature sensor enable bit. */
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_TSNSCTL) |= ANA_SYSCTRL_PARA_TSNSCTL_TSNS_ENABLE;
}

/*
 * @brief Disable temperature sensor.
 *
 * @details This function disables the temperature sensor output to the ADC.
 *
 */
__STATIC_INLINE void
ASysCtl_disableTemperatureSensor(void)
{
	HWREG(ANALOGSUBSYS_BASE+ANA_CFG_O_ANA_TS_MEAS_EN) &= (~ANA_SYSCTRL_PARA_ANA_TS_MEAS_EN);
	HWREG(ANALOGSUBSYS_BASE+ANA_CFG_O_ANA_TS_SPARE) &= (~0x3U);
    /* Clear the temperature sensor enable bit. */
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_TSNSCTL) &= ~(ANA_SYSCTRL_PARA_TSNSCTL_TSNS_ENABLE);

}

/**
 * @brief Set the analog voltage reference selection to internal.
 *
 * @param[in] reference is the analog reference.
 *
 * The parameter reference can be a combination of the following values:
 * 	- ASYSCTL_VREFHI
 * @note:
 * Internal reference mode is not available for packages lacking VREFHI and VREFLO pins.
 * Please see device datasheet for packages with VREFHI and VREFLO pins available.
 *
 */
__STATIC_INLINE void
ASysCtl_setAnalogReferenceInternal(uint16_t reference)
{
    ASSERT((reference & (ASYSCTL_VREFHI)) == reference);

    /* Write selection to the Analog Internal Reference Select bit. */
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) |= reference;
}

/**
 * @brief Set the analog voltage reference selection to external.
 *
 * @param[in] reference is the analog reference.
 *
 * The parameter reference can be a combination of the following values:
 * 	- ASYSCTL_VREFHI
 *
 */
__STATIC_INLINE void
ASysCtl_setAnalogReferenceExternal(uint16_t reference)
{
    ASSERT((reference & (ASYSCTL_VREFHI
                        )) == reference);

    /* Write selection to the Analog External Reference Select bit. */
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) &= (~reference);
}

/**
 * @brief Set the internal analog voltage reference selection to 2.5V.
 *
 * @param[in] reference is the analog reference.
 *
 * The parameter reference can be a combination of the following values:
 * 	- ASYSCTL_VREFHI
 *
 */
__STATIC_INLINE void ASysCtl_setAnalogReference2P5(uint16_t reference)
{
    ASSERT((reference & (ASYSCTL_VREFHI)) == reference);

    /* Write selection to the Analog Voltage Reference Select bit. */
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) |= (reference << 8U);

}

/**
 * @brief Set the internal analog voltage reference selection to 1.65V.
 *
 * @param[in] reference is the analog reference.
 *
 * The parameter reference can be a combination of the following values:
 * 	- ASYSCTL_VREFHI
 *
 */
__STATIC_INLINE void ASysCtl_setAnalogReference1P65(uint16_t reference)
{
    ASSERT((reference & (ASYSCTL_VREFHI)) == reference);

    /* Write selection to the Analog Voltage Reference Select bit. */
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) &= ~(reference << 8U);

}

__STATIC_INLINE void ASysCtl_setCMP_MuxEn(ASysCtl_CMPSelect select,uint16_t chl_en)
{
   HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) = (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) &
   ~(0xf << (uint32_t)select)) | (chl_en << (uint32_t)select);
}

__STATIC_INLINE void ASysCtl_disableAllCMPMUX(void)
{
	HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN)= 0;
}

__STATIC_INLINE void ASysCtl_enableCMPMUXIO(void)
{
	HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MUX_IO_EN)|=
			ANA_SYSCTRL_PARA_CMP_MUX_IO_EN;
}

__STATIC_INLINE void ASysCtl_disableCMPMUXIO(void)
{
	HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MUX_IO_EN)&=
			(~ANA_SYSCTRL_PARA_CMP_MUX_IO_EN);
}


/* Here are the APIs related to the function of CMPSS */
/**
 * @brief Select the value for CMPHNMXSEL.
 *
 * @param[in]  select is a combination of CMPHNMXSEL values.
 *
 * The parameter select can be a bitwise OR of the below values:
 * - ASYSCTL_CMPHNMUX_SELECT_1
 * - ASYSCTL_CMPHNMUX_SELECT_2
 * - ASYSCTL_CMPHNMUX_SELECT_3
 * - ASYSCTL_CMPHNMUX_SELECT_4
 */
__STATIC_INLINE void ASysCtl_selectCMPHNMux(uint16_t select)
{
    ASSERT(select <= 0x7FU);

    EALLOW;

    /* Write a select to the mux select bit. */
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHNMXSEL) = select;

    EDIS;
}

/**
 * @brief Select the value for individual CMPxHNMXSEL.
 *
 * param[in] select is the CMPxHNMXSEL to be set.
 * param[in] value is CMP HN channel selections.
 *
 * The parameter select can be one of the below values:
 * - ASYSCTL_CMPHNMUX_SELECT_1
 * - ASYSCTL_CMPHNMUX_SELECT_2
 * - ASYSCTL_CMPHNMUX_SELECT_3
 * - ASYSCTL_CMPHNMUX_SELECT_4
 *
 * The parameter value can be one of the below values:
 * - CMP_HN_LN_0
 * - CMP_HN_LN_1
 */
__STATIC_INLINE void ASysCtl_selectCMPHNMuxValue(uint16_t select, uint16_t value)
{
    ASSERT(select <= 0x7FU);
    ASSERT(value <= 1U);

    EALLOW;

    /* Write a select to the mux select bit. */
    if(value == 0U)
    {
        HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHNMXSEL) &= ~select;
    }
    else if(value == 1U)
    {
        HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHNMXSEL) |= select;
    }
    else
    {
        /* Do nothing. */
    }
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) |=
    			( ANA_SYSCTRL_PARA_CMPX_HN_EN_M<<(select/3*ANA_SYSCTRL_PARA_CMPMUX_STEP ));
    switch (select) {
            case 1:
            	HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) |= (0x04 << 0);
                break;
            case 2:
            	HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) |= (0x04 << 4);
                break;
            case 4:
            	HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) |= (0x04 << 8);
                break;
            case 8:
            	HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) |= (0x04 << 12);
                break;
            default:
                break;

        }

    EDIS;
}

/*
 * @brief Select the value for CMPHNMXSEL.
 *
 * @param[in] select  is a combination of CMPLNMXSEL values.
 *
 * The parameter select can be the bitwise OR of the below values:
 * - ASYSCTL_CMPLNMUX_SELECT_1
 * - ASYSCTL_CMPLNMUX_SELECT_2
 * - ASYSCTL_CMPLNMUX_SELECT_3
 * - ASYSCTL_CMPLNMUX_SELECT_4
 */
__STATIC_INLINE void ASysCtl_selectCMPLNMux(uint16_t select)
{
    ASSERT(select <= 0x7FU);

    /* Write a select to the mux select bit. */
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPLNMXSEL) = select;

}

/**
 * @brief Select the value for individual CMPxLNMXSEL.
 *
 * param[in] select is the CMPxLNMXSEL to be set.
 * param[in] value is CMP LN channel selections.
 *
 * The parameter select can be one of the below values:
 * - ASYSCTL_CMPHNMUX_SELECT_1
 * - ASYSCTL_CMPHNMUX_SELECT_2
 * - ASYSCTL_CMPHNMUX_SELECT_3
 * - ASYSCTL_CMPHNMUX_SELECT_4
 *
 * The parameter value can be one of the below values:
 * - CMP_HN_LN_0
 * - CMP_HN_LN_1
 */
__STATIC_INLINE void ASysCtl_selectCMPLNMuxValue(uint16_t select, uint16_t value)
{
    ASSERT(select <= 0x7FU);
    ASSERT(value <= 1U);

    /* Write a select to the mux select bit. */
    if(value == 0U)
    {
        HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPLNMXSEL) &= ~select;
    }
    else if(value == 1U)
    {
        HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPLNMXSEL) |= select;
    }
    else
    {
        //Do nothing. Not a valid value
    }
//    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) |=
//        			( ANA_SYSCTRL_PARA_CMPX_LN_EN_M<<(select/3*ANA_SYSCTRL_PARA_CMPMUX_STEP ));

    switch (select) {
            case 1:
            	HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) |= (0x01 << 0);
                break;
            case 2:
            	HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) |= (0x01 << 4);
                break;
            case 4:
            	HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) |= (0x01 << 8);
                break;
            case 8:
            	HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) |= (0x01 << 12);
                break;
            default:
                break;
        }
}

/**
 * @brief Select the value for CMPHPMXSEL.
 *
 * @detail
 * This function is used to write a value to one mux select at a time.
 *
 * @param[in] select is of type ASysCtl_CMPHPMuxSelect.
 * @param[in] value is HP channel select.
 *
 * The parameter select can be one of the following values:
 * - ASYSCTL_CMPHPMUX_SELECT_1
 * - ASYSCTL_CMPHPMUX_SELECT_2
 * - ASYSCTL_CMPHPMUX_SELECT_3
 * - ASYSCTL_CMPHPMUX_SELECT_4
 *
 * The parameter value can be one of the following values:
 * - CMP_HP_LP_0
 * - CMP_HP_LP_1
 * - CMP_HP_LP_2
 * - CMP_HP_LP_3
 * - CMP_HP_LP_4
 *
 */
__STATIC_INLINE void
ASysCtl_selectCMPHPMux(ASysCtl_CMPHPMuxSelect select, uint32_t value)
{
    ASSERT(value <= 5U);

    /* Set the value for the appropriate Mux Select. */
	HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHPMXSEL) =
	(HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHPMXSEL) &
	~((uint32_t)ANA_SYSCTRL_PARA_CMPHPMXSEL_CMP1HPMXSEL_M << (uint32_t)select)) |
	(value << (uint32_t)select);

	HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) |=
			(ANA_SYSCTRL_PARA_CMPX_HP_EN_M<<(select/3*ANA_SYSCTRL_PARA_CMPMUX_STEP ));

}

/**
 * @brief Select the value for CMPLPMXSEL.
 *
 * @detail
 * This function is used to write a value to one mux select at a time.
 *
 * @param[in] select is of type ASysCtl_CMPHPMuxSelect.
 * @param[in] value is HP channel select.
 *
 * The parameter select can be one of the following values:
 * - ASYSCTL_CMPLPMUX_SELECT_1
 * - ASYSCTL_CMPLPMUX_SELECT_2
 * - ASYSCTL_CMPLPMUX_SELECT_3
 * - ASYSCTL_CMPLPMUX_SELECT_4
 *
 * The parameter value can be one of the following values:
 * - CMP_HP_LP_0
 * - CMP_HP_LP_1
 * - CMP_HP_LP_2
 * - CMP_HP_LP_3
 * - CMP_HP_LP_4
 *
 */
__STATIC_INLINE void
ASysCtl_selectCMPLPMux(ASysCtl_CMPLPMuxSelect select, uint32_t value)
{
    ASSERT(value <= 5U);

    /* Set the value for the appropriate Mux Select. */
	HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPLPMXSEL) =
	(HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPLPMXSEL) &
	~((uint32_t)ANA_SYSCTRL_PARA_CMPLPMXSEL_CMP1LPMXSEL_M << (uint32_t)select)) |
	(value << (uint32_t)select);

	HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) |=
			(ANA_SYSCTRL_PARA_CMPX_LP_EN_M<<(select/3*ANA_SYSCTRL_PARA_CMPMUX_STEP ));

}

/**
 * @brief Enables the CMPSSx.DACL output to be routed to a pin.
 *
 * @param[in] select is CMPSS instance.
 *
 * This function disables CMPSSx.COMPL and the associated DAC will act as a
 * general purpose DAC with 11 bit resolution.
 * The parameter select can be OR of the following values:
 * - ASYSCTL_CMPSS1_DACL
 *
 * @note
 * In this case, all other CMPSS module functionality is not useable,
 * including the high DAC, both comparators, ramp generation, and the digital
 * filters.
 */
__STATIC_INLINE void
ASysCtl_enableCMPSSExternalDAC(uint32_t select)
{
    ASSERT((select & (ASYSCTL_CMPSS1_DACL)) == select);

    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPSSCTL) &= (~ANA_SYSCTRL_PARA_CMPSSCTL_CMPSSCTLEN);

    /* Set the specified bits and enable CMPSSCTL register */
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPSSCTL) =
                    (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPSSCTL) | select) |
					ANA_SYSCTRL_PARA_CMPSSCTL_CMPSSCTLEN;
}

/**
 * @brief Disbales the CMPSSx.DACL output to be routed to a pin.
 *
 * @param[in] select is CMPSS instance
 *
 * This function enables CMPSSx.COMPL and the associated DAC will act as
 * reference for comparator.
 * The parameter select can be OR of the following values:
 * - ASYSCTL_CMPSS1_DACL
 */
__STATIC_INLINE void
ASysCtl_disableCMPSSExternalDAC(uint32_t select)
{
    ASSERT((select & (
                      ASYSCTL_CMPSS1_DACL
                     )) == select);

    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPSSCTL) &= (~ANA_SYSCTRL_PARA_CMPSSCTL_CMPSSCTLEN);

    /* Set the specified bits and enable CMPSSCTL register */
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPSSCTL) =
                    (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPSSCTL) & ~select) |
					ANA_SYSCTRL_PARA_CMPSSCTL_CMPSSCTLEN;
}

__STATIC_INLINE void
ASysCtl_enableCMPSS(void)
{
	HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_EN) |=  ANA_SYSCTRL_PARA_CMP_EN;
}

__STATIC_INLINE void
ASysCtl_disableCMPSS(void)
{
	HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_EN) &=  (~ANA_SYSCTRL_PARA_CMP_EN);
}

__STATIC_INLINE void
ASysCtl_resetCMPSS(void)
{
	HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_RST) |=  ANA_SYSCTRL_PARA_CMP_EN;
}

__STATIC_INLINE void
ASysCtl_unresetCMPSS(void)
{
	HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_RST) &=  (~ANA_SYSCTRL_PARA_CMP_EN);
}

/*
 * @brief Enable loo pback from CMPDAC to ADCs
 *
 */
/**
 * @brief Enable loopback from DAC to ADCs.
 *
 * @param[in] onfig can be bitwise OR of the following values:
 * - ANA_SYSCTRL_PARA_ADCDACLOOPBACK_ENLB2ADCA
 * - ANA_SYSCTRL_PARA_ADCDACLOOPBACK_ENLB2ADCC
 */
__STATIC_INLINE void ASysCtl_enableADCDACLoopback(uint32_t config)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCDACLOOPBACK) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCDACLOOPBACK) | config);
}

/**
 * @brief Disable loopback from DAC to ADCs.
 *
 * @param[in] onfig can be bitwise OR of the following values:
 * - ANA_SYSCTRL_PARA_ADCDACLOOPBACK_ENLB2ADCA
 * - ANA_SYSCTRL_PARA_ADCDACLOOPBACK_ENLB2ADCC
 */
__STATIC_INLINE void ASysCtl_disableADCDACLoopback(uint32_t config)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCDACLOOPBACK) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCDACLOOPBACK) & ~config);
}

/**
 * @brief Configure the ADCs for Global SW Trigger
 *
 * @param adcSelect can be bitwsie OR of the following :
 * ADC_FORCE_SOCn (n=[0,20])
 */
static inline void AsysCtl_configADCGlobalSOC(uint32_t adcSelect)
{
    EALLOW;
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCSOCFRCGBSEL) = adcSelect;
    EDIS;
}

/**
 * @brief Generate Global SOC force for specified SOCs
 *
 * @param socSelect can be bitwsie OR of the following :
 * ADC_FORCE_SOCn (n=[0,20])
 */
static inline void AsysCtl_forceADCGlobalSOC(uint32_t socSelect)
{
    EALLOW;
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCSOCFRCGB) = socSelect;
    EDIS;
}


/* Here are the APIs related to the function of Voltage Monitor */

/**
 * @brief BORL is disabled on VDDIO.
 *
 */
__STATIC_INLINE void ASysCtl_disableBOR(void)
{
    EALLOW;

    /* Clear the temperature sensor enable bit. */
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_VMONCTL) |= ANA_SYSCTRL_PARA_VMONCTL_BORLVMONDIS;

    EDIS;
}

/**
 * @brief BORL is enabled on VDDIO.
 *
 * @details
 * BOR circuit will be triggered if VDDIO goes lower than the lower BOR threshold of VDDIO.
 */
__STATIC_INLINE void ASysCtl_enableBOR(void)
{
    EALLOW;

    /* Clear the temperature sensor enable bit. */
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_VMONCTL) &= (~ANA_SYSCTRL_PARA_VMONCTL_BORLVMONDIS);

    EDIS;
}

/*
 * @brief Dereset ADC.
 */
__STATIC_INLINE void ASysCtl_deresetADC(uint32_t adcBase)
{
	if (adcBase == ADCA_BASE)
	{
		HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_RST) =
				HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_RST) &	\
				(~(uint32_t)ANA_SYSCTRL_PARA_ADC_RST_ADCA_RST);
	}
	else if(adcBase == ADCC_BASE)
	{
		HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_RST) =
				HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_RST) &	\
				(~(uint32_t)ANA_SYSCTRL_PARA_ADC_RST_ADCC_RST);
	}
	else
	{
		/* error */
	}
}

/*
 * @brief Reset ADC.
 */
__STATIC_INLINE void ASysCtl_resetADC(uint32_t adcBase)
{
	if (adcBase == ADCA_BASE)
	{
		HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_RST) =
				HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_RST) |	\
				((uint32_t)ANA_SYSCTRL_PARA_ADC_RST_ADCA_RST);
	}
	else if(adcBase == ADCC_BASE)
	{
		HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_RST) =
				HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_RST) |	\
				((uint32_t)ANA_SYSCTRL_PARA_ADC_RST_ADCC_RST);
	}
	else
	{
		/* error */
	}
}

/*
 * @brief Reset ADC.
 */
__STATIC_INLINE void ASysCtl_resetAllADC(void)
{
	HWREG(0x400D6400U) = 0U;
}
/**
 * ANA_BOR_CFG      BOR Configuration Register    Offset 0x304
 *
 *  Register Field          Field Range
 *  reserved                31:12
 *  bor_test_en             11:11
 *  bor_trim                10:7
 *  bor_rising_sel          6:4
 *  bor_falling_sel         3:1
 *  bor_en                  0
 *
 */
__STATIC_INLINE void SysCtl_setBorEn(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) & ~0x01) | (value & 0x01);
}

__STATIC_INLINE uint32_t SysCtl_getBorEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG)) & 0x01;
}

__STATIC_INLINE void SysCtl_setBorVfallDSel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) &(~(0x07 << 1))) | ((value & 0x07) << 1);
}

__STATIC_INLINE uint32_t SysCtl_getBorVfallDSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) >> 1) & 0x07;
}

__STATIC_INLINE void SysCtl_setBorVriseDSel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) & (~(0x07 << 4))) | ((value & 0x07) << 4);
}

__STATIC_INLINE uint32_t SysCtl_getBorVriseDSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) >> 4) & 0x07;
}

__STATIC_INLINE void SysCtl_setBorTrim(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) &(~(0x0F << 7))) | ((value & 0x0F) << 7); ;
}

__STATIC_INLINE uint32_t SysCtl_getBorTrim(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) >> 7) & 0x0F;
}

__STATIC_INLINE void SysCtl_setBorTestEn(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) & (1<<11)) | (value & 0x01);
}

__STATIC_INLINE uint32_t SysCtl_getBorTestEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG)) & (1<<11);
}

/**
 * ANA_LDO_1V1_CFG      1V1 LDO Configuration Register    Offset 0x308
 *
 *  Register Field              Field Range
 *  ldo_1v1_test_en             11
 *  ldo_1v1_test_mux_sel        10:8
 *  ldo_1v1_chop_en             7
 *  ldo_1v1_test_mode           6:5
 *  ldo_1v1_vsel                4:0
 *
 */
__STATIC_INLINE void SysCtl_setLdoVsel(uint32_t vsel)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) & (~(0x1F << 0))) | (vsel << 0);
}

__STATIC_INLINE uint32_t SysCtl_getLdoVsel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) >> 0) & 0x1F;
}

__STATIC_INLINE void SysCtl_setLdoTestMode(uint32_t mode)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) & (~(0x3 << 5))) | (mode << 5);
}

__STATIC_INLINE uint32_t SysCtl_getLdoTestMode(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) >> 5) & 0x03;
}

__STATIC_INLINE void SysCtl_setLdoChopEn(uint32_t en)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) & (~(0x1 << 7))) | (en << 7);
}

__STATIC_INLINE uint32_t SysCtl_getLdoChopEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) >> 7) & 0x01;
}

/**
 * ANA_OSC_TRIM      OSC Trim Configuration Register    Offset 0x314
 *
 *  Register Field      Field Range
 *  osc2_freqtrim       30:16
 *  reserved            15
 *  osc1_freqtrim       14:0
 *
 */
__STATIC_INLINE void ASysCtl_setOsc1Trim(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_OSC_TRIM) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_OSC_TRIM) & (~0x7FFF)) | (value & 0x7FFF);
}

__STATIC_INLINE uint32_t ASysCtl_getOsc1Trim(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_OSC_TRIM) & 0x7FFF;
}

__STATIC_INLINE void ASysCtl_setOsc2Trim(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_OSC_TRIM) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_OSC_TRIM) & (~(0x7FFF << 16))) | ((value & 0x7FFF) << 16);
}

__STATIC_INLINE uint32_t ASysCtl_getOsc2Trim(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_OSC_TRIM) >> 16) & 0x7FFF;
}

__STATIC_INLINE bool ASysCtl_getOSC2ResSel(void)
{
	if (HWREG(ANALOGSUBSYS_BASE+ANA_CFG_O_ANA_OSC_CFG)&ANA_SYSCTRL_PARA_ANA_OSC_CFG_OSC2_RES_SEL)
	{
		return TRUE;
	}

	return FALSE;
}

/*
 *  PVM_EN   PVM Enable Configuration Register  Offset  0x318
 *
 *  Register Field      Field Range
 *  reserved            31:12
 *  pvm_test_en         11
 *  pvm_trim            10:7
 *  pvm_rising_sel      6:4
 *  pvm_falling_sel     3:1
 *  pvm_en              0
 *
 */
__STATIC_INLINE void ASysCtl_setPvmEn(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getPvmEn(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) & 0x1;
}

__STATIC_INLINE void ASysCtl_setPvmFallingSel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) =
            ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG)&(~(0x07<<1)))|(value << 1));
}

__STATIC_INLINE uint32_t ASysCtl_getPvmFallingSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) >> 1) & 0x07;
}

__STATIC_INLINE void ASysCtl_setPvmRisingSel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) =
            ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG)&(~(0x07<<4)))|(value << 4));
}

__STATIC_INLINE uint32_t ASysCtl_getPvmRisingSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) >> 4) & 0x07;
}

__STATIC_INLINE void ASysCtl_setPvmTrim(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) =
                ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG)&(~(0x0F<<7)))|(value << 7));
}

__STATIC_INLINE uint32_t ASysCtl_getPvmTrim(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) >> 7) & 0xf;
}

__STATIC_INLINE void ASysCtl_setPvmTestEn(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) =
                    ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG)&(~(0x01<<11)))|(value << 11));
}

__STATIC_INLINE uint32_t ASysCtl_getPvmTestEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) >> 11) & 0x1;
}

/* ANA_OSC_CFG 		OSC Resistor Selection Config Register Offset 0x324
 *
 *  Register Field      Field Range
 *  reserved              31:12
 *  osc2_res_sel        	1            0:internal resistance, 1:External resistance
 *  osc1_res_sel        	0(invalid)
*/
__STATIC_INLINE void setOsc2ResistorSelection(uint32_t value)
{
	HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_OSC_CFG) = ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_OSC_CFG)&(~(0x01<<1)))|(value << 1));
}


/* ANA_OSC2_SPARE OSC2 Config Register Offset 0x328
 *
 *  Register Field      Field Range
 *  ana_top_spare       31:11
 *  freq_sel            10:9
 *  ldo trim            5:2
 *  test_vol_sel        1            0:ldo_vref, 1:ldo_vout
 *  test_en             0
*/

typedef enum OSC2_Freq_Sel {
    OSC2_FREQ_HIGH = 2,         //  ~10MHz
    OSC2_FREQ_MID = 0,         //  ~5MHz
    OSC2_FREQ_LOW = 1,         //  ~4MHz
}OSC2_Freq_Sel_t;

__STATIC_INLINE void ASysCtl_setOsc2FreqRange(uint32_t val)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_OSC2_SPARE) = (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_OSC2_SPARE) & (~(3<<9))) | ((val &0x03) << 9);
}

/*
 *  ANA_TOP_SPARE1 ~ ANA_TOP_SPARE4   ANA Top-Level Reserved Registers  Offset  0x554+0x4*m   m = 0~1
 *
 *  Register Field      Field Range
 *  ana_top_spare       31:0
 *                
 * 
 */
__STATIC_INLINE void ASysCtl_setAnaTopSpare(uint32_t m, uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_SPARE1 + 0x4*m) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTopSpare(uint32_t m)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_SPARE1 + 0x4*m);
}

/*
 *  ADC_SAMP_SEL   ADC Sampling Selection Control Register  Offset  0x404
 *
 *  Register Field      Field Range
 *  reserved            31:11
 *  adcc_samp_sel       10:8
 *  reserved            7
 *  adcb_samp_sel       6:4
 *  reserved            3
 *  adca_samp_sel       2:0
 */

__STATIC_INLINE void ASysCtl_setAdcSampSel(uint32_t n, uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_SAMP_SEL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_SAMP_SEL) & ~(0x07<<(n*4))) | (value & 0x7) << (n*4);
}

__STATIC_INLINE uint32_t ASysCtl_getAdcSampSel(uint32_t n)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_SAMP_SEL) >> (n*4)) & 0x7;
}

/*
 *  ADC_SPARE   ADC Reserved Registers  Offset  0x408+0x4*n   n = 0~2
 *
 *  Register Field      Field Range
 *  adc_spare           31:0
 *                
 * 
 */
__STATIC_INLINE void ASysCtl_setAdcSpare(uint32_t n, uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_SPARE + 0x4*n) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAdcSpare(uint32_t n)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_SPARE + 0x4*n);
}

/**
 *
 *  Set L2H enable bit
 *
 *  @param [in]: 0 disable, 1 enable;
 *
 *  @return None.
 *
 */
__STATIC_INLINE void ASysCtl_setL2HCfg(uint32_t value)
{
    //
    // Write selection to the ADC L2H enable bit
    //
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_L2H_CFG) = value;


}

/*
 *  ADC_TOP_SPARE   ADC Top-Level Reserved Register  Offset  0x418
 *
 *  Register Field      Field Range
 *  adc_spare           31:0
 *                
 * 
 */
__STATIC_INLINE void ASysCtl_setAdcTopSpare(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_TOP_SPARE) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAdcTopSpare(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_TOP_SPARE);
}

/*
 *  CMP_SPARE   CPM Reserved Register  Offset  0x4B4
 *
 *  Register Field      Field Range
 *  cmp_spare           31:0
 *                
 * 
 */
__STATIC_INLINE void ASysCtl_setCmpSpare(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_TOP_SPARE) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getCmpSpare(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_TOP_SPARE);
}



/*
 *  DACA_BUF_TRIM   DACA Buffer Trim Value  Offset  0x4C0
 *
 *  Register Field          Field Range
 *  daca_buf_trim           31:0
 *                
 * 
 */
__STATIC_INLINE void ASysCtl_setDacaBufTrim(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_BUF_TRIM) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getDacaBufTrim(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_BUF_TRIM);
}


/*
 *  DACB_BUF_TRIM   DACB Buffer Trim Value  Offset  0x4C4
 *
 *  Register Field          Field Range
 *  dacb_buf_trim           31:0
 *                
 * 
 */
__STATIC_INLINE void ASysCtl_setDacbBufTrim(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_BUF_TRIM) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getDacbBufTrim(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_BUF_TRIM);
}


/*
 *  DACA_SPARE   DAC A Spare Register  Offset  0x0100
 *
 *  Register Field          Field Range
 *  daca_spare              31:0
 *
 *
 */
__STATIC_INLINE void ASysCtl_setDacaSpare(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_SPARE) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getDacaSpare(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_SPARE);
}


/*
 *  DACB_SPARE   DAC B Spare Register  Offset  0x0100
 *
 *  Register Field          Field Range
 *  dacb_spare              31:0
 *
 *
 */
__STATIC_INLINE void ASysCtl_setDacbSpare(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_SPARE) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getDacbSpare(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_SPARE);
}


/*
 *  DAC_TOP_SPARE   DAC Top Spare Register  Offset  0x0100
 *
 *  Register Field          Field Range
 *  dac_spare               31:0
 *                
 * 
 */
__STATIC_INLINE void ASysCtl_setDacTopSpare(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DAC_TOP_SPARE) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getDacTopSpare(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DAC_TOP_SPARE);
}


/*
 *  TMUX   TMUX Configuration Register  Offset  0x0120
 *
 *  Register Field          Field Range
 *  tmux                    31:0
 *                
 * 
 */
__STATIC_INLINE void ASysCtl_setTmux(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_TMUX) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getTmux(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_TMUX) >> 0);
}


/*
 *  V2I_A   V2I_A Configuration Register  Offset  0x0124
 *
 *  Register Field          Field Range
 *  v2i_a                   31:0
 *                
 * 
 */
__STATIC_INLINE void ASysCtl_setV2iA(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_V2I_A) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getV2iA(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_V2I_A);
}


/*
 *  V2I_B   V2I_B Configuration Register  Offset  0x0128
 *
 *  Register Field          Field Range
 *  v2i_b                   31:0
 *                
 * 
 */
__STATIC_INLINE void ASysCtl_setV2iB(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_V2I_B) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getV2iB(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_V2I_B) >> 0);
}



/*
 *  BG_REG   Band Gap Configuration Register  Offset  0x0130
 *
 *  Register Field   Field Range
 *  reserved         31:4
 *  bg_meas          3
 *  bg_en            2
 *  reserved         1
 *  bg_chop_en       0
 * 
 */
__STATIC_INLINE void ASysCtl_setBgMeas(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG) = ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG)&(~BIT(3)))|(value << 3));
}

__STATIC_INLINE uint32_t ASysCtl_getBgMeas(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG) >> 3) & 0x01;
}

__STATIC_INLINE void ASysCtl_setBgEn(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG) = ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG)&(~BIT(2)))|(value << 2));
}

__STATIC_INLINE uint32_t ASysCtl_getBgEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG) >> 2) & 0x01;
}

__STATIC_INLINE void ASysCtl_setBgChopEn(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG) = ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG)&(~BIT(0)))|(value << 0));
}

__STATIC_INLINE uint32_t ASysCtl_getBgChopEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG) >> 0) & 0x01;
}


/*
 *  ANA_CLK_TRIM   Ana_clk Trim Configuration Register  Offset  0x4EC
 *
 *  Register Field   Field Range
 *  reserved         31:16
 *  ana_clk_trim     15:0
 * 
 */
__STATIC_INLINE void ASysCtl_setAnaClkTrim(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_CLK_TRIM) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAnaClkTrim(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_CLK_TRIM) >> 0) & 0xffff;
}


/*
 *  ANA_CLK_SPARE   Ana_clk_spare Configuration Register  Offset  0x0150
 *
 *  Register Field   Field Range
 *  ana_clk_spare    31:16
 * 
 */
__STATIC_INLINE void ASysCtl_setAnaClkSpare(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_CLK_SPARE) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAnaClkSpare(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_CLK_SPARE) >> 0) & 0xffff0000;
}


/*
 *  ANA_TS_CHOP_EN   Ana_ts_chop_en Configuration Register  Offset  0x0168
 *
 *  Register Field   Field Range
 *  reserved         31:1
 *  ana_ts_chop_en   0
 * 
 */
__STATIC_INLINE void ASysCtl_setAnaTsChopEn(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TS_CHOP_EN) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTsChopEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TS_CHOP_EN) >> 0) & 0x1;
}


/*
 *  ANA_TS_MEAS_EN   Ana_ts_meas_en Configuration Register  Offset  0x016C
 *
 *  Register Field   Field Range
 *  reserved         31:1
 *  ana_ts_meas_en   0
 * 
 */
__STATIC_INLINE void ASysCtl_setAnaTsMeasEn(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TS_MEAS_EN) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTsMeasEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TS_MEAS_EN) >> 0) & 0x1;
}


/*
 *  ANA_TS_SPARE   Ana_ts_spare Configuration Register  Offset  0x0174
 *
 *  Register Field   Field Range
 *  reserved         31:10
 *  ana_ts_spare     9:0
 * 
 */
__STATIC_INLINE void ASysCtl_setAnaTsSpare(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TS_SPARE) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTsSpare(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TS_SPARE) >> 0) & 0x3FF;
}

/*
 *  ANA_TEST_MUX_SEL   Ana_test_mux_sel Configuration Register  Offset  0x0198
 *
 *  Register Field   Field Range
 *  reserved         	31:12
 *  ana_test_mux3_sel   15:12
 *  ana_test_mux2_sel   11:8
 *  ana_test_mux1_sel   7:4
 *  ana_test_mux0_sel   3:0
 */

__STATIC_INLINE void ASysCtl_setAnaTestMux3Sel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) =
            ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL)) & (~(0x0f << 12))) |(value << 12);
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTestMux3Sel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) >> 12) & 0xf;
}

__STATIC_INLINE void ASysCtl_setAnaTestMux2Sel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) =
            ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL)) & (~(0x0f << 8))) |(value << 8);
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTestMux2Sel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) >> 8) & 0xf;
}

__STATIC_INLINE void ASysCtl_setAnaTestMux1Sel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) =
            ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL)) & (~(0x0f << 4))) |(value << 4);
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTestMux1Sel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) >> 4) & 0xf;
}

__STATIC_INLINE void ASysCtl_setAnaTestMux0Sel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) =
            ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) & (~(0x0f << 0))) |(value << 0));
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTestMux0Sel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) >> 0) & 0xf;
}


/*
 *  ANA_TEST_MUX0_MODE   ANA TEST MUX Mode Configuration Register  Offset  0x0270
 *
 *  Register Field       Field Range
 *  ana_test_mux3_mode   7
 *  ana_test_mux2_mode   6
 *  ana_test_mux1_mode   5
 *  ana_test_mux0_mode   4
 *  ana_test_mux3_en     3
 *  ana_test_mux2_en     2
 *  ana_test_mux1_en     1
 *  ana_test_mux0_en     0
 *  muxId: 0, 1, 2, 3
 */
__STATIC_INLINE void ASysCtl_setAnaTestMuxMode(uint32_t muxId, uint32_t mode)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) & ~(1<<(muxId+4))) | mode << (muxId+4);
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTestMuxMode(uint32_t muxId)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) >> (muxId+4)) & 0x1;
}

__STATIC_INLINE void ASysCtl_setAnaTestMuxEn(uint32_t muxId, uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) & ~(1<<muxId)) | enable << muxId;
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTestMuxEn(uint32_t muxId)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) >> muxId) & 0x1;
}

#endif


void ASysCtl_initConfig(void);
void ASysCtl_trim(void);
void AsysCtl_trimAdc(uint32_t adcBase, uint32_t refMode, uint32_t refVoltage);

#ifdef __cplusplus
}
#endif


#endif /* DEVICE_DRIVERLIB_SYSCTL_ANA_H_ */
