/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    sysctl_crg_v12.h
*   @brief   do NOT include this file directly, include "sysctl.h" instead
*
*/
/**
 * commit history
 * 20241218,Hezhiyuan,Modify comments
*/
#ifndef DEVICE_DRIVERLIB_SYSCTL_CRG_V12_H_
#define DEVICE_DRIVERLIB_SYSCTL_CRG_V12_H_

#include "gs32_version.h"

#if IS_GS32F00xx(0x12)

#ifdef __cplusplus
extern "C"{
#endif

#include "inc/hw_types.h"
#include "inc/hw_memmap.h"
#include "inc/hw_sysctl_crg_v12.h"
#include "sysctl/sysctl_dig_v12.h"

typedef enum CRG_DIV {
    CRG_DIV_1 = 1,
    CRG_DIV_2 = 2,
    CRG_DIV_4 = 4,
    CRG_DIV_8 = 8,
    CRG_DIV_16 = 16,
}crg_div_t;

typedef enum {
    SYS_MAX_CLK 	= 0,    //0:Clock before system frequency division;
    SMW_CLK			= 1,    //1:flash smw clock;
    CAN_CCLK		= 2,    //2:can core clk;
    APB_CLK			= 3,    //3:apb_clk;
    DSP_CLK			= 4,    //4:dsp_clk;
    PLL_OUT_DIV2	= 8,    //8:PLL output clock frequency division coefficient 2;
    PLL_OUT			= 9,    //9:PLL export clock;
    REF_PIN			= 10,	//10:xtal
    OSC1			= 11,   //11:OSC1;
    OSC2			= 12,   //12:OSC2;
    PLL_REF			= 13,   //13:PLL input reference clock
}OBS_SIG_Type;

/**
 * CRG_CFG_LOCK   lock control register   offset 0x0000
 *
 *  Register Field            Field Range
 *  CRG_CFG_lock              31 : 0
 *
 * Locking register. Writing unlock value to this register allows writing to all registers protected by lock.
 * Writing any other value will lock all registers protected by lock;
 * unlock value: 0x5A5A_5A5A
 * *
 */
__STATIC_INLINE void SysCtl_setSysctrlHLock(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CRG_CFG_LOCK) = 0x01;
}

__STATIC_INLINE void SysCtl_setSysctrlHUnlock(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CRG_CFG_LOCK) = 0x5A5A5A5A;
}

__STATIC_INLINE uint32_t SysCtl_getSysctrlHLock(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_CRG_CFG_LOCK);
}

/**
 * REF_CLK_SEL   PLL reference clock selection   offset 0x0004
 *
 *  Register Field               Field Range
 *  ref_clk_src_sel              1 : 0
 *  00: Internal backup 10M
 *  01: Internal 10M (default)
 *  10: Oscillation circuit clock
 *  11: Illegal configuration, configuration prohibited
 * *
 */
typedef enum REF_CLK_TYPE {
    REF_CLK_TYPE_INTRENAL_OSC1_10MHZ = 0,
    REF_CLK_TYPE_INTERNAL_OSC2_10MHZ = 1,
    REF_CLK_TYPE_EXTERNAL_XTAL = 2,
}RefClkType_t;

__STATIC_INLINE void SysCtl_setRefClkSel(RefClkType_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_REF_CLK_SEL) = value;
}

/**
 * SYSCLK_S2N_EN   System clock SLOW to NORMAL switch   offset 0x0008
 *
 *  Register Field                Field Range
 *  reserved                      31 :  1
 *  sysclk_s2n_en                 0
 *
 *
 * *
 */
__STATIC_INLINE void SysCtl_setSysclkS2nEn(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_S2N_CLK_SEL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getSysclkS2nEn(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_S2N_CLK_SEL) >> 0) & 0x01;
}

/**
* CAN_CLK_SRC_SEL   CAN clock source selection configuration register   offset 0x000C
 *
 *  Register Field              Field Range
 *  reserved                    31 :  2
 *  can_clk_src_sel             1  :  0
 *
 *  Bit1:Bit0
 *  00: HSE clock
 *  01: AUXCLKIN
 *  10: Divided clock
 *  11: Illegal configuration, configuration prohibited.
 * *
 */
typedef enum CAN_CLK_SRC_TYPE {
    CAN_CLK_TYPE_HSE = 0,
    CAN_CLK_TYPE_AUXCLK_IN = 1,
    CAN_CLK_TYPE_PLL = 2,
}CanClkSrcType_t;

__STATIC_INLINE void SysCtl_setCanClkSrcSel(CanClkSrcType_t type)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CAN_CORE_CLK_SEL) = type;
}

__STATIC_INLINE uint32_t SysCtl_getCanClkSrcSel(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_CAN_CORE_CLK_SEL) >> 0) & 0x03;
}

/**
* CFG_DIV_SLV_SYS   Slave clock division configuration register   offset 0x0010
 *
 *  Register Field              Field Range
 *  reserved                    31 : 16
 *  cfg_div_slv_sys             15 : 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setSlvSysClkDiv(crg_div_t value)
{
    /* switch (value) {
        case CRG_DIV_1: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_SLV_SYS) = 0xFFFF; break;
        case CRG_DIV_2: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_SLV_SYS) = 0x5555; break;
        case CRG_DIV_4: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_SLV_SYS) = 0x1111; break;
        case CRG_DIV_8: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_SLV_SYS) = 0x0101; break;
        default: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_SLV_SYS) = 0xFFFF; break;
    }*/

    if (value == CRG_DIV_2) {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_SLV_SYS) = 0x5555;
    } else if (value == CRG_DIV_4) {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_SLV_SYS) = 0x1111;
    } else if (value == CRG_DIV_8) {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_SLV_SYS) = 0x0101;
    } else {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_SLV_SYS) = 0xFFFF;
    }
}

/**
* CFG_DIV_DSP_SYS   DSP clock division configuration register   offset 0x0014
 *
 *  Register Field              Field Range
 *  reserved                    31 : 16
 *  cfg_div_dsp_sys             15 : 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setDspSysClkDiv(crg_div_t value)
{
    /* switch (value) {
        case CRG_DIV_1: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_SYS) = 0xFFFF; break;
        case CRG_DIV_2: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_SYS) = 0x5555; break;
        case CRG_DIV_4: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_SYS) = 0x1111; break;
        case CRG_DIV_8: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_SYS) = 0x0101; break;
        default: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_SYS) = 0xFFFF; break;
    } */

    if (value == CRG_DIV_2) {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_SYS) = 0x5555;
    } else if (value == CRG_DIV_4) {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_SYS) = 0x1111;
    } else if (value == CRG_DIV_8) {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_SYS) = 0x0101;
    } else {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_SYS) = 0xFFFF;
    }
}

__STATIC_INLINE crg_div_t SysCtl_getDspSysClkDiv(void)
{
    uint16_t crg = (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_SYS) >> 0) & 0xFFFF;

    /* switch (crg){
        case 0xFFFF : return CRG_DIV_1;
        case 0x5555 : return CRG_DIV_2;
        case 0x1111 : return CRG_DIV_4;
        case 0x0101 : return CRG_DIV_8;
        default: return CRG_DIV_1;
    }*/

    if (crg == 0x5555) {
        return CRG_DIV_2;
    } else if (crg == 0x1111) {
        return CRG_DIV_4;
    } else if (crg == 0x0101) {
        return CRG_DIV_8;
    } else {
        return CRG_DIV_1;
    }
}

/**
* CFG_DIV_APB_SYS   APB clock division configuration register   offset 0x0018
 *
 *  Register Field              Field Range
 *  reserved                    31 : 16
 *  cfg_div_apb_sys             15 : 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setApbClkDiv(crg_div_t value)
{
    /*switch (value) {
        case CRG_DIV_1: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0xFFFF; break;
        case CRG_DIV_2: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0x5555; break;
        case CRG_DIV_4: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0x1111; break;
        case CRG_DIV_8: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0x0101; break;
        case CRG_DIV_16: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0x0001; break;
        default: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0xFFFF; break;
    }*/

    if (value == CRG_DIV_2) {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0x5555;
    } else if (value == CRG_DIV_4) {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0x1111;
    } else if (value == CRG_DIV_8) {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0x0101;
    } else if (value == CRG_DIV_16) {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0x0001;
    } else {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0xFFFF;
    }
}

__STATIC_INLINE crg_div_t SysCtl_getApbClkDiv(void)
{
    uint16_t crg = (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) >> 0) & 0xFFFF;

    /* switch (crg){
        case 0xFFFF : return CRG_DIV_1;
        case 0x5555 : return CRG_DIV_2;
        case 0x1111 : return CRG_DIV_4;
        case 0x0101 : return CRG_DIV_8;
        case 0x0001 : return CRG_DIV_16;
        default: return CRG_DIV_1;
    }*/

    if (crg == 0x5555) {
        return CRG_DIV_2;
    } else if (crg == 0x1111) {
        return CRG_DIV_4;
    } else if (crg == 0x0101) {
        return CRG_DIV_8;
    } else if (crg == 0x0001) {
        return CRG_DIV_16;
    } else {
        return CRG_DIV_1;
    }
}

/**
* CFG_SYS_DIV_LOAD_POS   Clock division configuration synchronization update register   offset 0x001C
 *
 *  Register Field              Field Range
 *  reserved                    31 : 1
 *  cfg_sys_div_load_pos        0
 *
 *
 * *
 */
__STATIC_INLINE void SysCtl_setSysDivLoad(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_SYS_DIV_LOAD_POS) = 1;
}

/**
 * TRACE_CLK_DIV   TRACE clock division configuration register   offset 0x0020
 *
 *  Register Field              Field Range
 *  reserved                    31 :  7
 *  trace_clk_div               6  :  0
 *
 *
 * *
 */
__STATIC_INLINE void SysCtl_setTraceClkDiv(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TRACE_CLK_DIV) = value;
}

__STATIC_INLINE uint32_t SysCtl_getTraceClkDiv(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_TRACE_CLK_DIV) >> 0) & 0x7f;
}

/**
 * SMW_CLK_DIV   SMW clock division configuration register   offset 0x0024
 *
 *  Register Field              Field Range
 *  reserved                    31 : 7
 *  cfg_smw_clk_div             6 : 0   Counter divider configuration divisor value is twice the configured value, (e.g., configuring 1 indicates a 2-divider)
 *
 * *
 */
__STATIC_INLINE void SysCtl_setSmwClkDiv(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SMW_CLK_DIV) = value;
}

__STATIC_INLINE uint32_t SysCtl_getSmwClkDiv(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_SMW_CLK_DIV) >> 0) & 0x7f;
}

/**
* CAN_CLK_DIV   CAN clock division configuration register   offset 0x0028
 *
 *  Register Field              Field Range
 *  crg_len                     31  :  27
 *  crg_patten                  26  :  0
 *  default                     0x47FFFE01  (divider value) div by 9
 * *
 */
__STATIC_INLINE void SysCtl_setCanClkDiv(uint32_t div)
{
    if (div > 1 && div <= 27) {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_CAN) = ((div-1)<<27) | (((0x7FFFFFF << div) & 0x7FFFFFF) | 0x01);        //TODO:
    } else {

    }
}

__STATIC_INLINE uint32_t SysCtl_getCanClkDiv(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_CAN);
}

typedef enum WDG_PRE_DIV_VAL {
    WDG_PRE_DIV_16 = 4,
    WDG_PRE_DIV_32 = 5,
    WDG_PRE_DIV_64 = 6,
    WDG_PRE_DIV_128 = 7,
    WDG_PRE_DIV_256 = 8,
    WDG_PRE_DIV_512 = 9,
    WDG_PRE_DIV_1024 = 10,
    WDG_PRE_DIV_2048 = 11,
    WDG_PRE_DIV_4096 = 12,
}WDGPreDivVal_t;

typedef enum WDG_SCALER_DIV_VAL {
    WDG_SCALER_DIV_1 = 0,
    WDG_SCALER_DIV_2 = 1,
    WDG_SCALER_DIV_4 = 2,
    WDG_SCALER_DIV_8 = 3,
    WDG_SCALER_DIV_16 = 4,
    WDG_SCALER_DIV_32 = 5,
    WDG_SCALER_DIV_64 = 6,
}WDGScalerDivVal_t;

/**
 * DSP_WDG_CLK_PRE_DIV   DSP Watchdog clock pre-divider configuration register   offset 0x0038
 *
 *  Register Field              Field Range
 *  reserved                    31 : 7
 *  dsp_wdg_clk_div             6 : 0
 *
 *
 * *
 */
__STATIC_INLINE void SysCtl_setDspWdgClkPreDiv(WDGPreDivVal_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DSP_WDG_CLK_PRE_DIV) = value;

    /* Update watchdog frequency division value */
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE2) = HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE2) & ~(0x1 << 1);
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE2) |= (0x1 << 1);
}

__STATIC_INLINE uint32_t SysCtl_getDspWdgClkPreDiv(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DSP_WDG_CLK_PRE_DIV) >> 0) & 0x7f;
}

/**
 * CFG_DSP_WDG_CLK_SCALER_DIV   DSP Watchdog Scaler clock division configuration register   offset 0x003C
 *
 *  Register Field              Field Range
 *  reserved                    31 : 4
 *  cfg_dsp_wdg_clk_pre_div     3 : 0       default 9
 *
 *
 * *
 */
__STATIC_INLINE void SysCtl_setDspWdgClkScalerDiv(WDGScalerDivVal_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DSP_WDG_CLK_SCALER_DIV) = value;

    /* Update watchdog frequency division value */
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE2) = HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE2) & ~(0x1 << 1);
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE2) |= (0x1 << 1);
}

__STATIC_INLINE uint32_t SysCtl_getDspWdgClkScalerDiv(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DSP_WDG_CLK_SCALER_DIV) >> 0) & 0x0F;
}

/**
 * BOOT_PIN_LATCH_SYSRST_MASK   Boot pin latch system reset influence configuration register   offset 0x002C
 *
 *  Register Field              Field Range
 *  reserved                    31 : 1
 *  boot_pin_latch_sysrst_mask  0
 *
 * bit0: Set this register to 0 to latch boot pin values when a system reset is needed.
 * *
 */
//__STATIC_INLINE void SysCtl_setBootPinLatchSysrstMask(uint32_t value)
//{
//    HWREG(CRG_CFG_BASE + CRG_CFG_O_BOOT_PIN_LATCH_SYSRST_MASK) = value ;
//}
//
//__STATIC_INLINE uint32_t SysCtl_getBootPinLatchSysrstMask(void)
//{
//    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_BOOT_PIN_LATCH_SYSRST_MASK) >> 0) & 0x01;
//}

/**
 * BOOT_PIN_LATCH_RPT   Boot pin feature word latch value reporting   offset 0x0030
 *
 *  Register Field         Field Range
 *  reserved               31 : 2
 *  boot_pin_lck_rpt       1 : 0
 * *
 */
//__STATIC_INLINE uint32_t SysCtl_getBootPinLatchRpt(void)
//{
//    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_BOOT_PIN_LATCH_RPT) >> 0) & 0x02;
//}

/**
 * CFG_DBG_WDT_HOLD_EN   Debug WDG Hold   offset 0x0040
 *
 *  Register Field             Field Range
 *  reserved                   31 : 4
 *  cfg_debug_wdt_hold_en      1 : 0
 *  bit 1: DSP WDG counter paused when core halted
 * *
 */
__STATIC_INLINE void SysCtl_setDspWdgDebugHoldEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DBG_WDT_HOLD_EN) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DBG_WDT_HOLD_EN) & (~(1<<1))) | ((enable & 0x01) << 1);
}

__STATIC_INLINE uint32_t SysCtl_getDspWdgDebugHoldEn(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DBG_WDT_HOLD_EN) >> 1) & 0x01;
}

/**
 * DSP_WDG_RST_EN   DSP Watchdog Reset Enable Configuration   offset 0x0048
 *
 *  Register Field            Field Range
 *  reserved                  31 : 2
 *  cfg_dsp_wdg_rst_en        1 : 0
 *  bit[1]: Windowed watchdog
 *  bit[0]: Normal watchdog
 *  1: Enable, can trigger reset
 *  0: Disable, does not trigger reset
 * *
 */
__STATIC_INLINE void SysCtl_setDspWdgResetEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_WDG_RST_EN) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_WDG_RST_EN) & (~(1<<0))) | (enable & 0x01);
}

__STATIC_INLINE uint32_t SysCtl_getDspWdgResetEn(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_WDG_RST_EN) & 0x01;
}

__STATIC_INLINE void SysCtl_setDspWwdgResetEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_WDG_RST_EN) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_WDG_RST_EN) & (~(1<<1))) | ((enable & 0x01) << 1);
}

__STATIC_INLINE uint32_t SysCtl_getDspWwdgResetEn(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_WDG_RST_EN) >> 1) & 0x01;
}

/**
 * SLAVE_CORE_RST_ON_SYS   Slave Core Reset Trigger System Reset Enable   offset 0x004C
 *
 *  Register Field                Field Range
 *  reserved                      31 : 1
 *  cfg_slave_core_rst_on_sys     0
 *  Whether slave core reset triggers system reset
 *  0: Reset does not trigger system reset
 *  1: Reset triggers system reset
 * *
 */
__STATIC_INLINE void SysCtl_setSlaveCoreRstOnSysEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SLAVE_CORE_RST_ON_SYS) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_SLAVE_CORE_RST_ON_SYS) & (~(1<<0))) | (enable & 0x01);
}

__STATIC_INLINE uint32_t SysCtl_getSlaveCoreRstOnSysEn(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_SLAVE_CORE_RST_ON_SYS) & 0x01;
}

/**
 * HSE_CMD_EN  HSE Detection Enable Configuration Register   offset 0x0050
 *
 *  Register Field          Field Range
 *  reserved                31 : 1
 *  hse_cmd_en              0
 * *
 */
__STATIC_INLINE void SysCtl_setHseCmdEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CMD_EN) = enable ;
}

__STATIC_INLINE uint32_t SysCtl_getHseCmdEn(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CMD_EN) & 0x01;
}

/**
 * CFG_CMD_ST_CLR  CMD_ST_CLR Enable   offset 0x0054
 *
 *  Register Field          Field Range
 *  reserved                31 : 1
 *  cfg_cmd_st_clr          0
 * *
 */
__STATIC_INLINE void SysCtl_setCmdStClr(uint32_t val)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CMD_ST_CLR) = val;
}

__STATIC_INLINE uint32_t SysCtl_getCmdStClr(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CMD_ST_CLR) & 0x01;
}

/**
 * CMD_CHECK_FAIL_RPT  Command Detection Result Reporting   offset 0x0058
 *
 *  Register Field          Field Range
 *  reserved                31 : 1
 *  cmd_check_fail_rpt      0
 * *
 */
__STATIC_INLINE uint32_t SysCtl_getCmdCheckFailRpt(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_CMD_CHECK_FAIL_RPT) & 0x01;
}

/**
 * TEST_MUX_EN   Test Signal Output Selection Configuration Register   offset 0x60
 *
 *  Register Field   Field Range
 *  reserved         31 : 1
 *  test_mux_en      0
 */
__STATIC_INLINE void SysCtl_setTestMuxEn(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TEST_MUX_EN) = value;
}

__STATIC_INLINE uint32_t SysCtl_getTestMuxEn(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_TEST_MUX_EN) & 0x01;
}

/**
 * TEST_MUX_SEL   Test Observation Signal Selection Register   offset 0x64
 *
 *  Register Field   Field Range
 *  reserved         31 : 4
 *  test_mux_sel     3 : 0
 */
__STATIC_INLINE void SysCtl_setTestSigSel(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TEST_MUX_SEL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getTestSigSel(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_TEST_MUX_SEL)) & 0x0f;
}

/**
 * OBS_SIG_SEL   Observation Signal Clock Selection Configuration Register   offset 0x68
 *
 *  Register Field   Field Range
 *  reserved         31 : 4
 *  obs_sig_sel      3 : 0
 */
__STATIC_INLINE void SysCtl_setObsSigSel(OBS_SIG_Type value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OBS_SIG_SEL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getObsSigSel(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_OBS_SIG_SEL) & 0x0f;
}

/**
 * OBS_SIG_DIV   Observation Signal Clock Divider Configuration Register   offset 0x6C
 *
 *  Register Field   Field Range
 *  reserved         31 : 7
 *  obs_sig_div      6 : 0
 */
__STATIC_INLINE void SysCtl_setObsSigDiv(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OBS_SIG_DIV) = value;
}

__STATIC_INLINE uint32_t SysCtl_getObsSigDiv(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OBS_SIG_DIV) & 0x3f);
}

/**
 * X1_CNT   X1 Clock Self-Test Register   offset 0x70
 *
 *  Register Field              Field Range
 *  cfg_ref_pin_clk_cnt_clr    16
 *  reserved                   15 : 11
 *  ref_pin_clk_cnt            10 : 0
 */
__STATIC_INLINE void SysCtl_setX1CntClr(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_X1_CNT) = 1<<16;
}

__STATIC_INLINE uint32_t SysCtl_getX1Cnt(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_X1_CNT) & 0x7FF;
}

/**
 * MASTER_CORE_RST_RECORD   Master Core System Reset Reason Record   offset 0x74
 *
 *  Register Field           Field Range
 *  reserved                 31 : 7
 *  master_core_rst_record   6 : 0
 */
typedef enum {
    RESET_TYPE_WWDG         = 1<<0,
    RESET_TYPE_WDG          = 1<<1,
    RESET_TYPE_NMIWDG       = 1<<2,
    RESET_TYPE_CORE         = 1<<3,
    RESET_TYPE_EFUSE_LAUNCH = 1<<4,
    RESET_TYPE_XRST         = 1<<5,
    RESET_TYPE_SOFT         = 1<<6,
    RESET_TYPE_POR          = 1<<8,
}CoreResetType_t;

__STATIC_INLINE CoreResetType_t SysCtl_getCoreRstRecord(void)
{
    return (CoreResetType_t)(
        (HWREG(CRG_CFG_BASE + CRG_CFG_O_MASTER_CORE_RST_RECORD) & 0x7F) |
        (SysCtl_getPorFlag() << 8)
    );
}

/**
 * @brief Clear Master Core Reset Record
 * @param resetType: Reset type(s) to clear, can be OR'ed together
 *                  e.g. RESET_TYPE_WWDG | RESET_TYPE_WDG
 * @return None
 * 
 * Note: This register is write-1-to-clear
 */
__STATIC_INLINE void SysCtl_clearCoreRstRecord(CoreResetType_t resetType)
{
    // Write 1 to clear the specified reset type bits
    HWREG(CRG_CFG_BASE + CRG_CFG_O_MASTER_CORE_RST_RECORD) = resetType & 0x7F;
    if(resetType & RESET_TYPE_POR)
    {
    	SysCtl_clearPorFlag();
    }
}


/**
 * PLL_PD  PLL Power Down Configuration   offset 0x0200
 *
 *  Register Field   Field Range
 *  reserved         31 : 1
 *  bpll_pd          0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setPllPd(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_PD) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllPd(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_PD) >> 0) & 0x01;
}

/**
 * PLL_BYPASS  PLL Bypass Configuration   offset 0x0204
 *
 *  Register Field   Field Range
 *  reserved         31 : 1
 *  bpll_bypass      0
 * *
 */
__STATIC_INLINE void SysCtl_setPllBypass(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_BYPASS) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllBypass(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_BYPASS) >> 0) & 0x01;
}

/**
 * PLL_OTHPD_CFG  PLL Function Power Down Configuration   offset 0x0208
 *
 *  Register Field   Field Range
 *  reserved         31 : 5
 *  bpll_othpd_cfg   4  : 0
 *
 *  [0]: DACPD
 *  [1]: DSMPD
 *  [2]: FOUT4PHASEPD
 *  [3]: FOUTPOSTIDVPD
 *  [4]: FOUTVCOPD
 * *
 */
__STATIC_INLINE void SysCtl_setPllOthpdCfg(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_OTHPD_CFG) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllOthpdCfg(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_OTHPD_CFG) >> 0) & 0x1f;
}

/**
 * PLL_DIV_CFG  PLL Divider Configuration   offset 0x020C
 *
 *  Register Field   Field Range
 *  reserved         31 : 12
 *  bpll_div_cfg     11 : 0
 *
 *  [2:0]: POSTDIV1
 *  [5:3]: POSTDIV2
 *  [11:6]: REFDIV
 * *
 */
__STATIC_INLINE void SysCtl_setPllDivCfg(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_DIV_CFG) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllDivCfg(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_DIV_CFG) >> 0) & 0xfff;
}

/**
 * PLL_FBDIV_CFG  PLL Feedback Divider Configuration   offset 0x0210
 *
 *  Register Field   Field Range
 *  reserved         31 : 12
 *  bpll_fbdiv_cfg   11 : 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setPllFbdivCfg(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_FBDIV_CFG) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllFbdivCfg(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_FBDIV_CFG) >> 0) & 0xfff;
}

/**
 * PLL_FRAC_CFG  PLL Fractional Divider Configuration   offset 0x0214
 *
 *  Register Field   Field Range
 *  reserved         31 : 24
 *  bpll_frac_cfg    23 : 0
 * *
 */
__STATIC_INLINE void SysCtl_setPllFracCfg(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_FRAC_CFG) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllFracCfg(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_FRAC_CFG) >> 0) & 0xffffff;
}

/**
 * PLL_DIVVAL  PLL Spread Spectrum DIVVAL Configuration   offset 0x021C
 *
 *  Register Field   Field Range
 *  reserved         31 : 6
 *  bpll_divval      5  : 0
 * *
 */
__STATIC_INLINE void SysCtl_setPllDivval(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_DIVVAL) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllDivval(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_DIVVAL) >> 0) & 0x3f;
}

/**
 * PLL_SPREAD  PLL Spread Spectrum Configuration   offset 0x0220
 *
 *  Register Field   Field Range
 *  reserved         31 : 6
 *  bpll_spread      5  : 0
 * *
 */
__STATIC_INLINE void SysCtl_setPllSpread(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_SPREAD) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllSpread(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_SPREAD) >> 0) & 0x3f;
}

/**
 * PLL_DOWNSPREAD  PLL Spread Spectrum Down Spread Configuration   offset 0x0224
 *
 *  Register Field   Field Range
 *  reserved         31 : 1
 *  bpll_downspread  0
 * *
 */
__STATIC_INLINE void SysCtl_setPllDownspread(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_DOWNSPREAD) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllDownspread(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_DOWNSPREAD) >> 0) & 0x01;
}

/**
 * PLL_EXT_MAXADDR  PLL Spread Spectrum EXT MAXADDR Configuration   offset 0x0228
 *
 *  Register Field     Field Range
 *  reserved           31 : 8
 *  bpll_ext_maxaddr   7  : 0
 * *
 */
__STATIC_INLINE void SysCtl_setPllExtMaxaddr(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_EXT_MAXADDR) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllExtMaxaddr(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_EXT_MAXADDR) >> 0) & 0xff;
}

/**
 * PLL_SEL_EXTWAVE  PLL Spread Spectrum EXT WAVE Configuration Enable   offset 0x022C
 *
 *  Register Field     Field Range
 *  reserved           31 : 1
 *  bpll_sel_extwave   0
 * *
 */
__STATIC_INLINE void SysCtl_setPllSelExtwave(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_SEL_EXTWAVE) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllSelExtwave(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_SEL_EXTWAVE) >> 0) & 0x01;
}

/**
 * PLL_RESETPTR  PLL Spread Spectrum Reset Pointer Configuration   offset 0x0230
 *
 *  Register Field   Field Range
 *  reserved         31 : 1
 *  bpll_resetptr    0
 * *
 */
__STATIC_INLINE void SysCtl_setPllResetptr(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_RESETPTR) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllResetptr(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_RESETPTR) >> 0) & 0x01;
}

/**
 * PLL_DISABLE_SSCG  PLL Spread Spectrum Function Disable Configuration   offset 0x0234
 *
 *  Register Field       Field Range
 *  reserved             31 : 1
 *  bpll_disable_sscg    0
 * *
 */
__STATIC_INLINE void SysCtl_setPllDisableSscg(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_DISABLE_SSCG) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllDisableSscg(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_DISABLE_SSCG) >> 0) & 0x01;
}

/**
 * PLL_LCK  PLL Lock Status   offset 0x0238
 *
 *  Register Field   Field Range
 *  reserved         31 : 1
 *  bpll_unlck_his   0
 * *
 */
__STATIC_INLINE void SysCtl_clearPllUnlckHis(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_UNLCK) = 1;
}

__STATIC_INLINE uint32_t SysCtl_getPllUnlckHis(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_UNLCK) & 0x01;
}

/**
 * PLL_LCK  PLL Lock Status   offset 0x023C
 *
 *  Register Field   Field Range
 *  reserved         31 : 1
 *  pll_lck          0
 * *
 */
__STATIC_INLINE uint32_t SysCtl_getPllLck(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_LCK) & 0x01;
}


/**
 * OSC1_TRIM_OSC_EN   OSC1 Calibration Enable   offset 0x0300
 *
 *  Register Field       Field Range
 *  reserved             31 : 1
 *  hsi_trim_osc_en      0
 *
 *  bit0: When calibration is enabled,
 *  the calibration value of OSC comes from the real-time calibration value,
 *  not from the cfg_osc1_trim_value configuration.
 * *
 */
__STATIC_INLINE void SysCtl_setOsc1TrimOscEn(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_OSC_EN) = value;
}

__STATIC_INLINE uint32_t SysCtl_getOsc1TrimOscEn(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_OSC_EN) >> 0) & 0x01;
}

/**
 * OSC1_TRIM_OSC_STEP_NUM   OSC Calibration Step Configuration   offset 0x0304
 *
 *  Register Field         Field Range
 *  reserved               31 : 4
 *  osc1_trim_osc_step_num  3 : 0
 *
 *  bit0-3: n represents 2^n steps.
 * *
 */
__STATIC_INLINE void SysCtl_setOsc1TrimOscStepNum(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_OSC_STEP_NUM) = value;
}

__STATIC_INLINE uint32_t SysCtl_getOsc1TrimOscStepNum(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_OSC_STEP_NUM) >> 0) & 0x0f;
}

/**
 * OSC1_TRIM_OSC_CALC_NUM   OSC1 Calibration Calculation Length   offset 0x0308
 *
 *  Register Field            Field Range
 *  reserved                  31 : 24
 *  osc1_trim_osc_calc_num    23 : 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setOsc1TrimOscCalcNum(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_OSC_CALC_NUM) = value;
}

__STATIC_INLINE uint32_t SysCtl_getOsc1TrimOscCalcNum(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_OSC_CALC_NUM) >> 0) & 0xffffff;
}

/**
 * OSC1_TRIM_OSC_WAIT_NUM   OSC1 Wait Refresh Time   offset 0x030C
 *
 *  Register Field              Field Range
 *  reserved                    31 : 24
 *  osc1_trim_osc_wait_num      23 : 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setOsc1TrimOscWaitNum(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_OSC_WAIT_NUM) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getOsc1TrimOscWaitNum(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_OSC_WAIT_NUM) >> 0) & 0xffffff;
}

/**
 * OSC2_TRIM_OSC_EN   OSC2 Calibration Enable   offset 0x0320
 *
 *  Register Field       Field Range
 *  reserved             31 : 1
 *  hsi_trim_osc_en      0
 *
 *  bit0: When calibration is enabled,
 *  the calibration value of OSC comes from the real-time calibration value,
 *  not from the cfg_osc2_trim_value configuration.
 * *
 */
__STATIC_INLINE void SysCtl_setOsc2TrimOscEn(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_OSC_EN) = value;
}

__STATIC_INLINE uint32_t SysCtl_getOsc2TrimOscEn(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_OSC_EN) >> 0) & 0x01;
}

/**
 * OSC2_TRIM_OSC_STEP_NUM   OSC2 Calibration Step Configuration   offset 0x0324
 *
 *  Register Field              Field Range
 *  reserved                    31 : 4
 *  osc2_trim_osc_step_num      3  : 0
 *
 *  bit0-3: 'n' represents 2^n calibration steps.
 * *
 */
__STATIC_INLINE void SysCtl_setOsc2TrimOscStepNum(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_OSC_STEP_NUM) = value;
}

__STATIC_INLINE uint32_t SysCtl_getOsc2TrimOscStepNum(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_OSC_STEP_NUM) >> 0) & 0x0f;
}

/**
 * OSC2_TRIM_OSC_CALC_NUM   OSC2 Calibration Calculation Length   offset 0x0328
 *
 *  Register Field              Field Range
 *  reserved                    31 : 24
 *  osc2_trim_osc_calc_num      23 : 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setOsc2TrimOscCalcNum(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_OSC_CALC_NUM) = value;
}

__STATIC_INLINE uint32_t SysCtl_getOsc2TrimOscCalcNum(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_OSC_CALC_NUM) >> 0) & 0xffffff;
}

/**
 * OSC2_TRIM_OSC_WAIT_NUM   OSC2 Wait Refresh Time   offset 0x032C
 *
 *  Register Field              Field Range
 *  reserved                    31 : 24
 *  osc2_trim_osc_wait_num      23 : 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setOsc2TrimOscWaitNum(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_OSC_WAIT_NUM) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getOsc2TrimOscWaitNum(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_OSC_WAIT_NUM) >> 0) & 0xffffff;
}

/**
 * OSC_TRIM_VALUE_RPT   OSC Calibration Value Reporting   offset 0x340
 *
 *  Register Field          Field Range
 *  reserved                31
 *  osc2_trim_value_rpt     30 : 16
 *  reserved                15
 *  osc1_trim_value_rpt     14 : 0
 *
 *  bit0-14: Calibration values reported for efuse burning.
 * *
 */
__STATIC_INLINE uint32_t SysCtl_getOsc1TrimValueRpt(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC_TRIM_VALUE_RPT) >> 0) & 0x7fff;
}

__STATIC_INLINE uint32_t SysCtl_getOsc2TrimValueRpt(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC_TRIM_VALUE_RPT) >> 16) & 0x7fff;
}


/**
 * SRST_N      Global Software Reset   offset 0x0400
 *
 *  Register Field        Field Range
 *  reserved              31 : 1
 *  srst_n                0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setSrstN(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_SYS) = value ;
}

__STATIC_INLINE void SysCtl_resetDevice(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_SYS) = 1 ;
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_SYS) = 0 ;
}

__STATIC_INLINE uint32_t SysCtl_getSrstN(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_SYS) >> 0) & 0x01;
}

/**
 * SRST_SLV_CORE_N      Slave Core Software Reset Configuration   offset 0x0404
 *
 *  Register Field              Field Range
 *  reserved                    31 : 2
 *  srst_slv_core_por_or_sys   	1
 *  srst_slv_core_n             0
 *
 *  bit 1: Configuration of the reset domain when resetting the slave core:
 *          0: Slave core POR domain
 *          1: Slave core SYS domain
 *          (Default resets the entire slave core POR domain)
 *  bit 0: Slave core software reset configuration. 1: Release reset; 0: Reset
 *
 * *
 */
__STATIC_INLINE void SysCtl_setSlaveCoreSrstN(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_SLV_CORE_N) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_SLV_CORE_N) & (~(1<<0))) | (value & 0x01) ;
}

__STATIC_INLINE void SysCtl_resetSlaveCore(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_SLV_CORE_N) = 0;
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_SLV_CORE_N) = 1;
}

__STATIC_INLINE uint32_t SysCtl_getSlaveCoreSrstN(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_SLV_CORE_N) & 0x01;
}

typedef enum {
    SLAVE_RESET_RANGE_POR = 0,
    SLAVE_RESET_RANGE_SYS = 1,
}SlaveCoreResetType_t;

__STATIC_INLINE void SysCtl_setSlaveCoreResetRange(SlaveCoreResetType_t range)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_SLV_CORE_N) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_SLV_CORE_N) & (~(1<<1))) | (range << 1);
}

__STATIC_INLINE uint32_t SysCtl_getSlaveCoreResetRange(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_SLV_CORE_N) >> 1) & 0x01;
}


/**
 * FORCE_AHB_ISO    Slave Core AHB MASTER Forced Takeover Configuration   0x408
 *
 *  Register Field              Field Range
 *  reserved                    31 : 20
 *  dsp_m_iso_ok_st             19
 *  cm4_system_m_iso_ok_st      18
 *  cm4_dcode_m_iso_ok_st       17
 *  cm4_icode_m_iso_ok_st       16
 *  reserved                    15 : 10
 *  dsp_iso_bypass              9
 *  reserved                    7 : 1
 *  slv_core_force_iso          0
 *
 */
__STATIC_INLINE void SysCtl_setSlaveCoreForceIso(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_FORCE_AHB_ISO) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_FORCE_AHB_ISO) & (~(1<<0))) | (value & 0x01);
}

__STATIC_INLINE uint32_t SysCtl_getSlaveCoreForceIso(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_FORCE_AHB_ISO) & 0x01;
}

__STATIC_INLINE void SysCtl_setDspIsoBypass(uint32_t bypass)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_FORCE_AHB_ISO) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_FORCE_AHB_ISO) & (~(1<<9))) | ((bypass & 0x01) << 9);
}

__STATIC_INLINE uint32_t SysCtl_getDspIsoBypass(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_FORCE_AHB_ISO) >> 9) & 0x01;
}

/*--- status ---*/
__STATIC_INLINE uint32_t SysCtl_getDspIsoSt(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_FORCE_AHB_ISO) >> 19) & 0x01;
}

/**
 * ISO_OVF_CNT_TH  Takeover Threshold Configuration   0x40C
 *
 *  Register Field       Field Range
 *  dsp_ovf_cnt_th       31 : 16
 *
 */
__STATIC_INLINE void SysCtl_setDspOvfCntTh(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_ISO_OVF_CNT_TH) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_ISO_OVF_CNT_TH) & (~(0xFFFF<<16))) | ((value & 0xFFFF) << 16);
}

__STATIC_INLINE uint32_t SysCtl_getDspOvfCntTh(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_ISO_OVF_CNT_TH) >> 16) & 0xFFFF;
}


/**
 * SLV_CORE_STOP_WITH_SLV  SLV_CORE_STOP_WITH_SLV Configuration   0x410
 *
 *  Register Field          Field Range
 *  reserved                31 : 1
 *  slv_core_stop_with_slv  0
 *
 */
__STATIC_INLINE void SysCtl_setSlaveCoreStopWithSlv(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SLV_CORE_STOP_WITH_SLV) = value & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_getSlaveCoreStopWithSlv(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_SLV_CORE_STOP_WITH_SLV) & 0x01;
}


/**
 * SLV_CORE_STOP_ON_SET  SLV_CORE_STOP_ON_SET Configuration   0x414
 *
 *  Register Field        Field Range
 *  reserved              31 : 1
 *  slv_core_stop_on_set  0
 */
__STATIC_INLINE void SysCtl_setSlaveCoreStopOnSet(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SLV_CORE_STOP_ON_SET) = 1;
}

/**
 * SLV_CORE_STOP_ON_CLR  SLV_CORE_STOP_ON_CLR Configuration   0x418
 *
 *  Register Field        Field Range
 *  reserved              31 : 1
 *  slv_core_stop_on_clr  0
 */
__STATIC_INLINE void SysCtl_setSlaveCoreStopOnClr(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SLV_CORE_STOP_ON_CLR) = 1;
}

/**
 * SLV_CORE_STOP_ON_RPT  SLV_CORE_STOP_ON_RPT Configuration   0x41C
 *
 *  Register Field        Field Range
 *  reserved              31 : 1
 *  slv_core_stop_on_rpt  0
 */
__STATIC_INLINE uint32_t SysCtl_getSlaveCoreStopOnRpt(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_SLV_CORE_STOP_ON_RPT) & 0x01;
}


/**
 * SRST_EFLASH_CTRL_N  EFLASH CTRL Software Reset  Offset 0x0504
 *
 *  Register Field       Field Range
 *  reserved             31 : 1
 *  srst_eflash_ctrl_n   0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setSrstEflashCtrlN(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_EFLASH_CTRL_N) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getSrstEflashCtrlN(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_EFLASH_CTRL_N) >> 0) & 0x01;
}

__STATIC_INLINE void SysCtl_resetEflash(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_EFLASH_CTRL_N) = 0 ;
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_EFLASH_CTRL_N) = 1 ;
}


/**
 * SRST_QSPI_N  QSPI Software Reset  Offset 0x0508
 *
 *  Register Field   Field Range
 *  reserved         31 : 1
 *  srst_qspi_n      0
 * *
 */
__STATIC_INLINE void SysCtl_setSrstQspiN(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_QSPI_N) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getSrstQspiN(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_QSPI_N) >> 0) & 0x01;
}

__STATIC_INLINE void SysCtl_resetQspi(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_QSPI_N) = 0 ;
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_QSPI_N) = 1 ;
}

/**
 * SRST_DMA_N  DMA Software Reset  Offset 0x050C
 *
 *  Register Field   Field Range
 *  reserved         31 : 1
 *  srst_dma_n       0
 * *
 */
__STATIC_INLINE void SysCtl_setSrstDmaN(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_DMA_N) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getSrstDmaN(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_DMA_N) >> 0) & 0x01;
}

__STATIC_INLINE void SysCtl_resetDma(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_DMA_N) = 0 ;
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_DMA_N) = 1 ;
}

/**
 * SRST_STC_N  STC Software Reset  Offset 0x0510
 *
 *  Register Field   Field Range
 *  reserved         31 : 1
 *  srst_stc_n       0
 * *
 */
__STATIC_INLINE void SysCtl_setSrstStcN(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_STC_N) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getSrstStcN(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_STC_N) >> 0) & 0x01;
}

__STATIC_INLINE void SysCtl_resetStc(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_STC_N) = 0 ;
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_STC_N) = 1 ;
}

/**
 * SRST_AES_N  AES Software Reset  Offset 0x0514
 *
 *  Register Field   Field Range
 *  reserved         31 : 1
 *  srst_aes_n       0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setSrstAesN(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_AES_N) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getSrstAesN(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_AES_N) >> 0) & 0x01;
}

__STATIC_INLINE void SysCtl_resetAes(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_AES_N) = 0 ;
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_AES_N) = 1 ;
}

/**
 * EFLASH_CTRL_CLK_EN  EFLASH CTRL Clock Enable Configuration Register  Offset 0x0600
 *
 *  Register Field       Field Range
 *  reserved             31 : 1
 *  eflash_ctrl_clk_en   0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setEflashCtrlClkEn(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_EFLASH_CTRL_CLK_EN) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getEflashCtrlClkEn(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_EFLASH_CTRL_CLK_EN) >> 0) & 0x01;
}

/**
 * QSPI_CLK_ICG_BYPASS  QSPI Clock Automatic CG Bypass Configuration  Offset 0x0604
 *
 *  Register Field           Field Range
 *  reserved                 31 : 1
 *  cfg_qspi_clk_icg_bypass  0
 * *
 */
__STATIC_INLINE void SysCtl_setQspiIcgBypassEn(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_QSPI_CLK_ICG_BYPASS) = value ;
}

/**
 * DMA_CLK_ICG_BYPASS   DMA Clock Enable Configuration Register   Offset 0x0608
 *
 *  Register Field               Field Range
 *  reserved                     31 : 2
 *  cfg_dsp_dma_clk_icg_bypass   1
 * *
 */
__STATIC_INLINE void SysCtl_setDspDmaIcgBypass(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_DMA_CLK_ICG_BYPASS) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_DMA_CLK_ICG_BYPASS) & (~(1<<1))) | ((value & 0x01) << 1) ;
}

__STATIC_INLINE uint32_t SysCtl_getDspDmaIcgBypass(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_DMA_CLK_ICG_BYPASS) >> 1) & 0x01;
}

/**
 * DSP_CLK_ICG_BYPASS   DSP Clock Automatic CG Bypass Configuration   Offset 0x060C
 *
 *  Register Field               Field Range
 *  reserved                     31 : 2
 *  cfg_dsp_core_clk_icg_bypass  1
 *  cfg_dsp_nice_clk_icg_bypass  0
 * *
 */
__STATIC_INLINE void SysCtl_setDspNiceIcgBypass(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_CLK_ICG_BYPASS) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_CLK_ICG_BYPASS) & (~(1<<0))) | (value & 0x01) ;
}

__STATIC_INLINE uint32_t SysCtl_getDspNiceIcgBypass(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_CLK_ICG_BYPASS) & 0x01;
}

__STATIC_INLINE void SysCtl_setDspCoreIcgBypass(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_CLK_ICG_BYPASS) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_CLK_ICG_BYPASS) & (~(1<<1))) | ((value & 0x01) << 1) ;
}

__STATIC_INLINE uint32_t SysCtl_getDspCoreIcgBypass(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_CLK_ICG_BYPASS) >> 1) & 0x01;
}

/**
 * CFG_STANDBY   Configuration for Entering Low Power State During Standby   Offset 0x0614
 *
 *  Register Field        Field Range
 *  reserved              31 : 1
 *  cfg_standby           0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setStandby(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_STANDBY) = (value & 0x01) ;
}

__STATIC_INLINE uint32_t SysCtl_getStandby(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_STANDBY) & 0x01;
}

/**
 * CFG_WDG_WAKE_EN   Main Core Watchdog Wakeup Enable   Offset 0x0618
 *
 *  Register Field      Field Range
 *  reserved            31 : 1
 *  cfg_wdg_wake_en     0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setWdgWakeEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_WDG_WAKE_EN) = enable;
}

__STATIC_INLINE uint32_t SysCtl_getWdgWakeEn(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_WDG_WAKE_EN) & 0x01;
}

/**
 * CFG_GPIO_WAKE_EN   GPIO Wakeup Enable   Offset 0x061C
 *
 *  Register Field       Field Range
 *  reserved             31 : 1
 *  cfg_gpio_wake_en     0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setGpioWakeEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_GPIO_WAKE_EN) = enable;
}

__STATIC_INLINE uint32_t SysCtl_getGpioWakeEn(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_GPIO_WAKE_EN) & 0x01;
}

/**
 * AES_CLK_EN   AES Clock Enable Configuration Register   Offset 0x0620
 *
 *  Register Field     Field Range
 *  reserved           31 : 1
 *  cfg_aes_clk_en     0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setAesClkEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_AES_CLK_EN) = enable;
}

__STATIC_INLINE uint32_t SysCtl_getAesClkEn(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_AES_CLK_EN) & 0x01;
}

/**
 * PARA_ICG_BYPASS   PARA_ICG	   Offset 0x06FC
 *
 *  Register Field    			 Field Range
 *  reserved						31 : 4
 *  io_cfg_para_icg_bypass			3		Clock enable
 *  											1: Clock gating function turned off, clock normally open
 *  											0: The clock gating function is turned on, and the clock switches with the working status of the core"
 *  ana_cfg_para_icg_bypass			2		Clock enable
 *  											1: Clock gating function turned off, clock normally open
 *  											0: The clock gating function is turned on, and the clock follows the nice working status switch"
 *  crg_cfg_para_icg_bypass			1		Clock enable
 *  											1: Clock gating function turned off, clock normally open
 *  											0: The clock gating function is turned on, and the clock switches with the working status of the core"
 *  sys_ctrl_para_icg_bypass		0		Clock enable
 *  											1: Clock gating function turned off, clock normally open
 *  											0: The clock gating function is turned on, and the clock follows the nice working status switch"
 *
 * 
 */
__STATIC_INLINE void SysCtl_setSysCtrlParaIcgBypass(uint32_t bypass)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) & (~(1<<0))) | (bypass & 0x01);
}

__STATIC_INLINE uint32_t SysCtl_getSysCtrlParaIcgBypass(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) & 0x01;
}

__STATIC_INLINE void SysCtl_setCrgCfgParaIcgBypass(uint32_t bypass)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) & (~(1<<1))) | ((bypass & 0x01) << 1);
}

__STATIC_INLINE uint32_t SysCtl_getCrgCfgParaIcgBypass(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) >> 1) & 0x01;
}

__STATIC_INLINE void SysCtl_setAnaCfgParaIcgBypass(uint32_t bypass)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) & (~(1<<2))) | ((bypass & 0x01) << 2);
}

__STATIC_INLINE uint32_t SysCtl_getAnaCfgParaIcgBypass(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) >> 2) & 0x01;
}

__STATIC_INLINE void SysCtl_setIoCfgParaIcgBypass(uint32_t bypass)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) & (~(1<<3))) | ((bypass & 0x01) << 3);
}

__STATIC_INLINE uint32_t SysCtl_getIoCfgParaIcgBypass(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) >> 3) & 0x01;
}

/**
 * TOP_CRG_SPARE0   Reserved Register 0   Offset 0xF00
 *
 *  Register Field      Field Range
 *  top_crg_spare0      31 : 0
 *
 * 
 */
__STATIC_INLINE void SysCtl_setTopCrgSpare0(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE0) = value;
}

__STATIC_INLINE uint32_t SysCtl_getTopCrgSpare0(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE0);
}

/**
 * TOP_CRG_SPARE1   Reserved Register 1   Offset 0xF04
 *
 *  Register Field      Field Range
 *  top_crg_spare1      31 : 0
 *
 * 
 */
__STATIC_INLINE void SysCtl_setTopCrgSpare1(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE1) = value;
}

__STATIC_INLINE uint32_t SysCtl_getTopCrgSpare1(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE1);
}

/**
 * TOP_CRG_SPARE2   Reserved Register 2   Offset 0xF08
 *
 *  Register Field      Field Range
 *  top_crg_spare2      31 : 0
 *
 * 
 */
__STATIC_INLINE void SysCtl_setTopCrgSpare2(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE2) = value;
}

__STATIC_INLINE uint32_t SysCtl_getTopCrgSpare2(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE2);
}

#ifdef __cplusplus
}
#endif

#endif

#endif /* DEVICE_DRIVERLIB_SYSCTL_CRG_H_ */
