/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    sysctl_crg_v22.h
*   @brief   do NOT include this file directly, include "sysctl.h" instead
*
*/
/**
 * commit history
 * 20241218,Hezhiyuan,Modify comments
*/
#ifndef DEVICE_DRIVERLIB_SYSCTL_CRG_V22_H_
#define DEVICE_DRIVERLIB_SYSCTL_CRG_V22_H_

#include "gs32_version.h"

#if IS_GS32F3xx(0x22)

#ifdef __cplusplus
extern "C"{
#endif

#include "inc/hw_types.h"
#include "inc/hw_memmap.h"
#include "inc/hw_sysctl_crg_v22.h"
#include "sysctl/sysctl_dig_v22.h"

typedef enum CRG_DIV {
    CRG_DIV_1 = 1,
    CRG_DIV_2 = 2,
    CRG_DIV_4 = 4,
    CRG_DIV_8 = 8,
    CRG_DIV_16 = 16,
}crg_div_t;

/***
 * CRG_CFG_LOCK   Lock Control Register   Offset 0x0000
 *
 *  Register Field       Field Range
 *  CRG_CFG_lock         31 : 0
 *
 * Locking register. After writing unlock value to this register, all registers protected by lock can be written to.
 * Writing other values will lock all registers protected by lock;
 * Unlock value: 0x5A5A_5A5A.
 * *
 */
__STATIC_INLINE void SysCtl_setSysctrlHLock(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CRG_CFG_LOCK) = 0x01;
}

__STATIC_INLINE void SysCtl_setSysctrlHUnlock(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CRG_CFG_LOCK) = 0x5A5A5A5A;
}

__STATIC_INLINE uint32_t SysCtl_getSysctrlHLock(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_CRG_CFG_LOCK);
}

/**
 * REF_CLK_SEL   PLL Reference Clock Selection   Offset 0x0004
 *
 *  Register Field         Field Range
 *  ref_clk_src_sel        1 : 0
 *  00: Internal Backup 10M
 *  01: Internal 10M (Default)
 *  10: Oscillator Circuit Clock
 *  11: Illegal Configuration, Configuration Forbidden
 * *
 */
typedef enum REF_CLK_TYPE {
    REF_CLK_TYPE_INTRENAL_OSC1_10MHZ = 0,
    REF_CLK_TYPE_INTERNAL_OSC2_10MHZ = 1,
    REF_CLK_TYPE_EXTERNAL_XTAL = 2,
}RefClkType_t;

__STATIC_INLINE void SysCtl_setRefClkSel(RefClkType_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_REF_CLK_SEL) = value;
}

/**
 * SYSCLK_S2N_EN   System Clock SLOW to NORMAL Switching   Offset 0x0008
 *
 *  Register Field     Field Range
 *  reserved           31 :  1
 *  sysclk_s2n_en      0
 * *
 */
__STATIC_INLINE void SysCtl_setSysclkS2nEn(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_S2N_CLK_SEL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getSysclkS2nEn(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_S2N_CLK_SEL) >> 0) & 0x01;
}

/**
 * CAN_CLK_SRC_SEL   CAN Clock Source Selection Configuration Register   Offset 0x000C
 *
 *  Register Field     Field Range
 *  reserved           31 :  2
 *  can_clk_src_sel    1  :  0
 *
 *  Bit 1: Bit 0
 *  00: HSE Clock
 *  01: AUXCLKIN
 *  10: Divided Clock
 *  11: Illegal Configuration, Configuration Forbidden
 * *
 */
typedef enum CAN_CLK_SRC_TYPE {
    CAN_CLK_TYPE_HSE = 0,
    CAN_CLK_TYPE_AUXCLK_IN = 1,
    CAN_CLK_TYPE_PLL = 2,
}CanClkSrcType_t;

__STATIC_INLINE void SysCtl_setCanClkSrcSel(CanClkSrcType_t type)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CAN_CORE_CLK_SEL) = type;
}

__STATIC_INLINE uint32_t SysCtl_getCanClkSrcSel(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_CAN_CORE_CLK_SEL) >> 0) & 0x03;
}

/**
 * CFG_DIV_DSP_SYS   DSP Clock Division Configuration Register   Offset 0x0010
 *
 *  Register Field     Field Range
 *  reserved           31 : 16
 *  cfg_div_dsp_sys    15 : 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setDspSysClkDiv(crg_div_t value)
{
    /*switch (value) {
        case CRG_DIV_1: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_CORE) = 0xFFFF; break;
        case CRG_DIV_2: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_CORE) = 0x5555; break;
        case CRG_DIV_4: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_CORE) = 0x1111; break;
        case CRG_DIV_8: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_CORE) = 0x0101; break;
        case CRG_DIV_16: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_CORE) = 0x0001; break;
        default: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_CORE) = 0xFFFF; break;
    }*/

    if (value == CRG_DIV_2) {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_CORE) = 0x5555;
    } else if (value == CRG_DIV_4) {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_CORE) = 0x1111;
    } else if (value == CRG_DIV_8) {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_CORE) = 0x0101;
    } else if (value == CRG_DIV_16) {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_CORE) = 0x0001;
    } else {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_CORE) = 0xFFFF;
    }
}

__STATIC_INLINE crg_div_t SysCtl_getDspSysClkDiv(void)
{
    uint16_t crg = (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_CORE) >> 0) & 0xFFFF;

    /*switch (crg){
        case 0xFFFF : return CRG_DIV_1;
        case 0x5555 : return CRG_DIV_2;
        case 0x1111 : return CRG_DIV_4;
        case 0x0101 : return CRG_DIV_8;
        case 0x0001 : return CRG_DIV_16;
        default: return CRG_DIV_1;
    }*/

    if (crg == 0x5555) {
        return CRG_DIV_2;
    } else if (crg == 0x1111) {
        return CRG_DIV_4;
    } else if (crg == 0x0101) {
        return CRG_DIV_8;
    } else if (crg == 0x0001) {
        return CRG_DIV_16;
    } else {
        return CRG_DIV_1;
    }
}

/**
 * CFG_DIV_PERI_SYS    Peripheral Clock Division Configuration Register   Offset 0x014
 *
 *  Register Field     Field Range
 *  reserved           31 : 16
 *  cfg_div_dsp_peri   15 : 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setPeriClkDiv(crg_div_t value)
{
    /*switch (value) {
        case CRG_DIV_1: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_PERI_SYS) = 0xFFFF; break;
        case CRG_DIV_2: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_PERI_SYS) = 0x5555; break;
        case CRG_DIV_4: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_PERI_SYS) = 0x1111; break;
        case CRG_DIV_8: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_PERI_SYS) = 0x0101; break;
        case CRG_DIV_16: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_PERI_SYS) = 0x0001; break;
        default: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_PERI_SYS) = 0xFFFF; break;
    }*/

    if (value == CRG_DIV_2) {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_PERI_SYS) = 0x5555;
    } else if (value == CRG_DIV_4) {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_PERI_SYS) = 0x1111;
    } else if (value == CRG_DIV_8) {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_PERI_SYS) = 0x0101;
    } else if (value == CRG_DIV_16) {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_PERI_SYS) = 0x0001;
    } else {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_PERI_SYS) = 0xFFFF;
    }
}

__STATIC_INLINE crg_div_t SysCtl_getPeriClkDiv(void)
{
    uint16_t crg = (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_PERI_SYS) >> 0) & 0xFFFF;
	
    /*switch (crg){
        case 0xFFFF : return CRG_DIV_1;
        case 0x5555 : return CRG_DIV_2;
        case 0x1111 : return CRG_DIV_4;
        case 0x0101 : return CRG_DIV_8;
        case 0x0001 : return CRG_DIV_16;
        default: return CRG_DIV_1;
    }*/

    if (crg == 0x5555) {
        return CRG_DIV_2;
    } else if (crg == 0x1111) {
        return CRG_DIV_4;
    } else if (crg == 0x0101) {
        return CRG_DIV_8;
    } else if (crg == 0x0001) {
        return CRG_DIV_16;
    } else {
        return CRG_DIV_1;
    }
}

/**
 * CFG_DIV_APB_SYS   APB Clock Division Configuration Register   Offset 0x0018
 *
 *  Register Field     Field Range
 *  reserved           31 : 16
 *  cfg_div_apb_sys    15 : 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setApbClkDiv(crg_div_t value)
{
    /*switch (value) {
        case CRG_DIV_1: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0xFFFF; break;
        case CRG_DIV_2: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0x5555; break;
        case CRG_DIV_4: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0x1111; break;
        case CRG_DIV_8: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0x0101; break;
        case CRG_DIV_16: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0x0001; break;
        default: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0xFFFF; break;
    }*/

    if (value == CRG_DIV_2) {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0x5555;
    } else if (value == CRG_DIV_4) {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0x1111;
    } else if (value == CRG_DIV_8) {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0x0101;
    } else if (value == CRG_DIV_16) {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0x0001;
    } else {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0xFFFF;
    }
}

__STATIC_INLINE crg_div_t SysCtl_getApbClkDiv(void)
{
    uint16_t crg = (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) >> 0) & 0xFFFF;

    /*switch (crg){
        case 0xFFFF : return CRG_DIV_1;
        case 0x5555 : return CRG_DIV_2;
        case 0x1111 : return CRG_DIV_4;
        case 0x0101 : return CRG_DIV_8;
        case 0x0001 : return CRG_DIV_16;
        default: return CRG_DIV_1;
    }*/

    if (crg == 0x5555) {
        return CRG_DIV_2;
    } else if (crg == 0x1111) {
        return CRG_DIV_4;
    } else if (crg == 0x0101) {
        return CRG_DIV_8;
    } else if (crg == 0x0001) {
        return CRG_DIV_16;
    } else {
        return CRG_DIV_1;
    }
}

/**
 * CFG_SYS_DIV_LOAD_POS   Clock Division Configuration Synchronization Update Register   Offset 0x001C
 *
 *  Register Field         Field Range
 *  reserved               31 : 1
 *  cfg_sys_div_load_pos   0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setSysDivLoad(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_SYS_DIV_LOAD_POS) = 1;
}

/**
 * TRACE_CLK_DIV   TRACE Clock Division Configuration Register   Offset 0x0020
 *
 *  Register Field   Field Range
 *  reserved         31 : 7
 *  trace_clk_div    6 : 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setTraceClkDiv(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TRACE_CLK_DIV) = value;
}

__STATIC_INLINE uint32_t SysCtl_getTraceClkDiv(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_TRACE_CLK_DIV) >> 0) & 0x7f;
}

/**
 * SMW_CLK_DIV   SMW Clock Division Configuration Register   Offset 0x0024
 *
 *  Register Field     Field Range
 *  reserved           31 : 7
 *  cfg_smw_clk_div    6 : 0
 *  Counter Divider Configuration, Divider value configured as double the specified value
 *  (e.g., config 1 implies 2-divider)
 *
 * *
 */
__STATIC_INLINE void SysCtl_setSmwClkDiv(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SMW_CLK_DIV) = value;
}

__STATIC_INLINE uint32_t SysCtl_getSmwClkDiv(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_SMW_CLK_DIV) >> 0) & 0x7f;
}

/**
 * CAN_CLK_DIV   CAN Clock Division Configuration Register   Offset 0x0028
 *
 *  Register Field   Field Range
 *  crg_len           31 : 27
 *  crg_patten        26 : 0
 *  default           0x47FFFE01  div by 9
 *
 *  @brief set the system clock divider for CAN_CLK
 *  @param[in]   div   2~27, clock divider from clock source to CAN_CLK.
 *  @remarks       
 *  - if PLL is selected as clock source, PLL frequency is 2x of sysclk
 *    and it's recommended to generate CAN_CLK at 20MHz/40MHz/80MHz.
 * *
 */
__STATIC_INLINE void SysCtl_setCanClkDiv(uint32_t div)
{
    if (div > 1 && div <= 27) {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_CAN) = ((div-1)<<27) | (((0x7FFFFFF << div) & 0x7FFFFFF) | 0x01);        //TODO:
    } else {

    }
}

__STATIC_INLINE uint32_t SysCtl_getCanClkDiv(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_CAN);
}

typedef enum WDG_PRE_DIV_VAL {
    WDG_PRE_DIV_16 = 4,
    WDG_PRE_DIV_32 = 5,
    WDG_PRE_DIV_64 = 6,
    WDG_PRE_DIV_128 = 7,
    WDG_PRE_DIV_256 = 8,
    WDG_PRE_DIV_512 = 9,
    WDG_PRE_DIV_1024 = 10,
    WDG_PRE_DIV_2048 = 11,
    WDG_PRE_DIV_4096 = 12,
}WDGPreDivVal_t;

typedef enum WDG_SCALER_DIV_VAL {
    WDG_SCALER_DIV_1 = 0,
    WDG_SCALER_DIV_2 = 1,
    WDG_SCALER_DIV_4 = 2,
    WDG_SCALER_DIV_8 = 3,
    WDG_SCALER_DIV_16 = 4,
    WDG_SCALER_DIV_32 = 5,
    WDG_SCALER_DIV_64 = 6,
}WDGScalerDivVal_t;

/**
 * CFG_CPU1_WDG_CLK_PRE_DIV   DSP Cpu1 Watchdog Clock Prescaler Division Configuration Register   Offset 0x0038
 *
 *  Register Field             Field Range
 *  reserved                    31 : 7
 *  dsp_wdg_clk_div             6 : 0
 * *
 */
__STATIC_INLINE void SysCtl_setDspCpu1WdgClkPreDiv(WDGPreDivVal_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CPU1_WDG_CLK_PRE_DIV) = value;

    /* Update watchdog frequency division value */
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE2) = HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE2) & ~(0x1 << 0);
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE2) |= (0x1 << 0);
}

__STATIC_INLINE uint32_t SysCtl_getDspCpu1WdgClkPreDiv(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CPU1_WDG_CLK_PRE_DIV) >> 0) & 0x7f;
}

/**
 * CFG_CPU1_WDG_CLK_SCALER_DIV   DSP Cpu1 Watchdog Scaler Clock Division Configuration Register   Offset 0x003C
 *
 *  Register Field                   Field Range
 *  reserved                          31 : 4
 *  cfg_dsp_wdg_clk_pre_div           3 : 0       Default 9
 *
 * *
 */
__STATIC_INLINE void SysCtl_setDspCpu1WdgClkScalerDiv(WDGScalerDivVal_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CPU1_WDG_CLK_SCALER_DIV) = value;

    /* Update watchdog frequency division value */
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE2) = HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE2) & ~(0x1 << 0);
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE2) |= (0x1 << 0);
}

__STATIC_INLINE uint32_t SysCtl_getDspCpu1WdgClkScalerDiv(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CPU1_WDG_CLK_SCALER_DIV) >> 0) & 0x0F;
}

/**
 * CFG_CPU2_WDG_CLK_PRE_DIV   DSP Cpu2 Watchdog Clock Prescaler Division Configuration Register   Offset 0x0040
 *
 *  Register Field             Field Range
 *  reserved                    31 : 7
 *  dsp_wdg_clk_div             6 : 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setDspCpu2WdgClkPreDiv(WDGPreDivVal_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CPU2_WDG_CLK_PRE_DIV) = value;

    /* Update watchdog frequency division value */
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE2) = HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE2) & ~(0x1 << 1);
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE2) |= (0x1 << 1);
}

__STATIC_INLINE uint32_t SysCtl_getDspCpu2WdgClkPreDiv(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CPU2_WDG_CLK_PRE_DIV) >> 0) & 0x7f;
}

/**
 * CFG_CPU2_WDG_CLK_SCALER_DIV   DSP Cpu2 Watchdog Scaler Clock Division Configuration Register   Offset 0x0044
 *
 *  Register Field                   Field Range
 *  reserved                          31 : 4
 *  cfg_dsp_wdg_clk_pre_div           3 : 0       Default 9
 *
 * *
 */
__STATIC_INLINE void SysCtl_setDspCpu2WdgClkScalerDiv(WDGScalerDivVal_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CPU2_WDG_CLK_SCALER_DIV) = value;

    /* Update watchdog frequency division value */
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE2) = HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE2) & ~(0x1 << 1);
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE2) |= (0x1 << 1);
}

__STATIC_INLINE uint32_t SysCtl_getDspCpu2WdgClkScalerDiv(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CPU2_WDG_CLK_SCALER_DIV) >> 0) & 0x0F;
}

/**
 * BOOT_PIN_LATCH_SYSRST_MASK   Boot Pin Feature Word Latch Whether Controlled by SYSRST Configuration Register   Offset 0x002C
 *
 *  Register Field                 Field Range
 *  reserved                        31 : 1
 *  boot_pin_latch_sysrst_mask      0
 *
 * bit0: When it is necessary to latch the boot pin value during system reset, configure this register to 0.
 * *
 */
//__STATIC_INLINE void SysCtl_setBootPinLatchSysrstMask(uint32_t value)
//{
//    HWREG(CRG_CFG_BASE + CRG_CFG_O_BOOT_PIN_LATCH_SYSRST_MASK) = value ;
//}
//
//__STATIC_INLINE uint32_t SysCtl_getBootPinLatchSysrstMask(void)
//{
//    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_BOOT_PIN_LATCH_SYSRST_MASK) >> 0) & 0x01;
//}

/**
 * BOOT_PIN_LATCH_RPT   Boot Pin Feature Word Latch Value Reporting   Offset 0x0030
 *
 *  Register Field         Field Range
 *  reserved               31 : 2
 *  boot_pin_lck_rpt       1 : 0
 *
 * *
 */
//__STATIC_INLINE uint32_t SysCtl_getBootPinLatchRpt(void)
//{
//    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_BOOT_PIN_LATCH_RPT) >> 0) & 0x02;
//}

/**
 * CFG_DBG_WDT_HOLD_EN   Debug Watchdog Hold   Offset 0x0040
 *
 *  Register Field           Field Range
 *  reserved                 31 : 4
 *  cfg_debug_wdt_hold_en    2 : 0
 *  bit 2: CPU2 watchdog counter pause follows core halted
 *  bit 1: CPU1 watchdog counter pause follows core halted
 *
 * *
 */

__STATIC_INLINE void SysCtl_setDspCpu1WdgDebugHoldEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DBG_WDT_HOLD_EN) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DBG_WDT_HOLD_EN) & (~(1<<1))) | ((enable & 0x01) << 1);
}

__STATIC_INLINE uint32_t SysCtl_getDspCpu1WdgDebugHoldEn(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DBG_WDT_HOLD_EN) >> 1) & 0x01;
}

__STATIC_INLINE void SysCtl_setDspCpu2WdgDebugHoldEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DBG_WDT_HOLD_EN) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DBG_WDT_HOLD_EN) & (~(1<<2))) | ((enable & 0x01) << 2);
}

__STATIC_INLINE uint32_t SysCtl_getDspCpu2WdgDebugHoldEn(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DBG_WDT_HOLD_EN) >> 2) & 0x01;
}

/**
 * CPU1_WDG_RST_EN   DSP Watchdog Reset Enable Configuration   Offset 0x0050
 *
 *  Register Field          Field Range
 *  reserved                31 : 2
 *  cfg_dsp_wdg_rst_en      1 : 0
 *  bit[1]: Window watchdog
 *  bit[0]: Normal watchdog
 *  1: Enable, can trigger reset
 *  0: Disable, does not trigger reset
 * *
 */
__STATIC_INLINE void SysCtl_setDspCpu1WdgResetEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CPU1_WDG_RST_EN) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_CPU1_WDG_RST_EN) & (~(1<<0))) | (enable & 0x01);
}

__STATIC_INLINE uint32_t SysCtl_getDspCpu1WdgResetEn(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_CPU1_WDG_RST_EN) & 0x01;
}

__STATIC_INLINE void SysCtl_setDspCpu1WwdgResetEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CPU1_WDG_RST_EN) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_CPU1_WDG_RST_EN) & (~(1<<1))) | ((enable & 0x01) << 1);
}

__STATIC_INLINE uint32_t SysCtl_getDspCpu1WwdgResetEn(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_CPU1_WDG_RST_EN) >> 1) & 0x01;
}

/**
 * CPU2_WDG_RST_EN   DSP Watchdog Reset Enable Configuration   Offset 0x0054
 *
 *  Register Field          Field Range
 *  reserved                31 : 2
 *  cfg_dsp_wdg_rst_en      1 : 0
 *  bit[1]: Window watchdog
 *  bit[0]: Normal watchdog
 *  1: Enable, can trigger reset
 *  0: Disable, does not trigger reset
 * *
 */
__STATIC_INLINE void SysCtl_setDspCpu2WdgResetEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CPU2_WDG_RST_EN) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_CPU2_WDG_RST_EN) & (~(1<<0))) | (enable & 0x01);
}

__STATIC_INLINE uint32_t SysCtl_getDspCpu2WdgResetEn(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_CPU2_WDG_RST_EN) & 0x01;
}

__STATIC_INLINE void SysCtl_setDspCpu2WwdgResetEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CPU2_WDG_RST_EN) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_CPU2_WDG_RST_EN) & (~(1<<1))) | ((enable & 0x01) << 1);
}

__STATIC_INLINE uint32_t SysCtl_getDspCpu2WwdgResetEn(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_CPU2_WDG_RST_EN) >> 1) & 0x01;
}

/**
 * HSE_CMD_EN  HSE Detection Enable Configuration Register   Offset 0x0100
 *
 *  Register Field    Field Range
 *  reserved          31 : 1
 *  hse_cmd_en        0
 * *
 */
__STATIC_INLINE void SysCtl_setHseCmdEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CMD_EN) = enable ;
}

__STATIC_INLINE uint32_t SysCtl_getHseCmdEn(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CMD_EN) & 0x01;
}

/**
 * CFG_CMD_ST_CLR  CMD_ST_CLR Enable Configuration Register   Offset 0x0104
 *
 *  Register Field    Field Range
 *  reserved          31 : 1
 *  cfg_cmd_st_clr    0
 * *
 */
__STATIC_INLINE void SysCtl_setCmdStClr(uint32_t val)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CMD_ST_CLR) = val;
}

__STATIC_INLINE uint32_t SysCtl_getCmdStClr(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CMD_ST_CLR) & 0x01;
}

/**
 * CMD_CHECK_FAIL_RPT  CMD Detection Result Reporting Register   Offset 0x0108
 *
 *  Register Field       Field Range
 *  reserved             31 : 1
 *  cmd_check_fail_rpt   0
 * *
 */
__STATIC_INLINE uint32_t SysCtl_getCmdCheckFailRpt(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_CMD_CHECK_FAIL_RPT) & 0x01;
}

/**
 * TEST_MUX_EN   Test Signal Output Selection Configuration Register   Offset 0x10C
 *
 *  Register Field   Field Range
 *  reserved         31 : 1
 *  test_mux_en      0
 */
__STATIC_INLINE void SysCtl_setTestMuxEn(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TEST_MUX_EN) = value;
}

__STATIC_INLINE uint32_t SysCtl_getTestMuxEn(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_TEST_MUX_EN) & 0x01;
}

/**
 * TEST_MUX_SEL   Test Observation Signal Selection Register   Offset 0x110
 *
 *  Register Field   Field Range
 *  reserved         31 : 4
 *  test_mux_sel     3 : 0
 */
__STATIC_INLINE void SysCtl_setTestSigSel(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TEST_MUX_SEL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getTestSigSel(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_TEST_MUX_SEL)) & 0x0f;
}

/**
 * OBS_SIG_SEL   Observation Clock Selection Configuration Register   Offset 0x114
 *
 *  Register Field   Field Range
 *  reserved         31 : 4
 *  obs_sig_sel      3 : 0
 */
__STATIC_INLINE void SysCtl_setObsSigSel(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OBS_SIG_SEL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getObsSigSel(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_OBS_SIG_SEL) & 0x0f;
}

/**
 * OBS_SIG_DIV   Observation Clock Divider Configuration Register   Offset 0x118
 *
 *  Register Field   Field Range
 *  reserved         31 : 7
 *  obs_sig_div      6 : 0
 */
__STATIC_INLINE void SysCtl_setObsSigDiv(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OBS_SIG_DIV) = value;
}

__STATIC_INLINE uint32_t SysCtl_getObsSigDiv(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_OBS_SIG_DIV) & 0x3f;
}

/**
 * X1_CNT   X1 Clock Self-Test Register   Offset 0x11C
 *
 *  Register Field            Field Range
 *  cfg_ref_pin_clk_cnt_clr  16
 *  reserved                  15 : 11
 *  ref_pin_clk_cnt           10 : 0
 */
__STATIC_INLINE void SysCtl_setX1CntClr(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_X1_CNT) = 1<<16;
}

__STATIC_INLINE uint32_t SysCtl_getX1Cnt(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_X1_CNT) & 0x7FF;
}

/**
 * CPU1_RST_RECORD   Main Core System Reset Reason Record   Offset 0x120
 *
 *  Register Field      Field Range
 *  reserved            31 : 8
 *  cpu1_rst_record     7 : 0
 *
 */
typedef enum {
    RESET_TYPE_WWDG         = 1<<0,
    RESET_TYPE_WDG          = 1<<1,
    RESET_TYPE_NMIWDG       = 1<<2,
    RESET_TYPE_CORE         = 1<<3,
    RESET_TYPE_EFUSE_LAUNCH = 1<<4,
    RESET_TYPE_XRST         = 1<<5,
    RESET_TYPE_SOFT         = 1<<6,
    RESET_TYPE_JTAG_AUTH    = 1<<7,/* Only cpu2 */
    RESET_TYPE_POR          = 1<<8,
}CoreResetType_t;

__STATIC_INLINE CoreResetType_t SysCtl_getDspCpu1RstRecord(void)
{
    return (CoreResetType_t)(
        (HWREG(CRG_CFG_BASE + CRG_CFG_O_CPU1_RST_RECORD) & 0x7F) |
        (SysCtl_getPorFlag() << 8)
    );
}

/**
 * @brief Clear Master Core1 Reset Record
 * @param resetType: Reset type(s) to clear, can be OR'ed together
 *                  e.g. RESET_TYPE_WWDG | RESET_TYPE_WDG
 * @return None
 *
 * Note: This register is write-1-to-clear
 */
__STATIC_INLINE void SysCtl_clearDspCpu1RstRecord(CoreResetType_t resetType)
{
    // Write 1 to clear the specified reset type bits
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CPU1_RST_RECORD) = resetType & 0x7F;
    if(resetType & RESET_TYPE_POR)
    {
    	SysCtl_clearPorFlag();
    }

}

/**
 * CPU2_RST_RECORD   Secondary Core System Reset Reason Record   Offset 0x124
 *
 *  Register Field      Field Range
 *  reserved            31 : 3
 *  cpu2_rst_record    6 : 0
 *
 */
__STATIC_INLINE CoreResetType_t SysCtl_getDspCpu2RstRecord(void)
{
    return (CoreResetType_t)(
        (HWREG(CRG_CFG_BASE + CRG_CFG_O_CPU2_RST_RECORD) & 0xFF) |
        (SysCtl_getPorFlag() << 8)
    );
}

/**
 * @brief Clear Master Core2 Reset Record
 * @param resetType: Reset type(s) to clear, can be OR'ed together
 *                  e.g. RESET_TYPE_WWDG | RESET_TYPE_WDG
 * @return None
 *
 * Note: This register is write-1-to-clear
 */
__STATIC_INLINE void SysCtl_clearDspCpu2RstRecord(CoreResetType_t resetType)
{
    // Write 1 to clear the specified reset type bits
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CPU2_RST_RECORD) = resetType & 0xFF;
    if(resetType & RESET_TYPE_POR)
    {
    	SysCtl_clearPorFlag();
    }

}

/**
 * PLL_PD  PLL Power Down Configuration   Offset 0x0200
 *
 *  Register Field      Field Range
 *  reserved            31 : 1
 *  bpll_pd             0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setPllPd(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_PD) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllPd(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_PD) >> 0) & 0x01;
}

/**
 * PLL_BYPASS  PLL Bypass Configuration   Offset 0x0204
 *
 *  Register Field      Field Range
 *  reserved            31 : 1
 *  bpll_bypass         0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setPllBypass(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_BYPASS) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllBypass(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_BYPASS) >> 0) & 0x01;
}

/**
 * PLL_OTHPD_CFG  PLL Function Power Down Configuration   Offset 0x0208
 *
 *  Register Field      Field Range
 *  reserved            31 : 5
 *  bpll_othpd_cfg      4 : 0
 *
 *  [0]: DACPD
 *  [1]: DSMPD
 *  [2]: FOUT4PHASEPD
 *  [3]: FOUTPOSTIDVPD
 *  [4]: FOUTVCOPD
 * *
 */
__STATIC_INLINE void SysCtl_setPllOthpdCfg(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_OTHPD_CFG) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllOthpdCfg(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_OTHPD_CFG) >> 0) & 0x1f;
}

/**
 * PLL_DIV_CFG  PLL Divider Configuration   Offset 0x020C
 *
 *  Register Field      Field Range
 *  reserved            31 : 12
 *  bpll_div_cfg        11 : 0
 *
 *  [2:0]: POSTDIV1
 *  [5:3]: POSTDIV2
 *  [11:6]: REFDIV
 * *
 */
__STATIC_INLINE void SysCtl_setPllDivCfg(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_DIV_CFG) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllDivCfg(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_DIV_CFG) >> 0) & 0xfff;
}

/**
 * PLL_FBDIV_CFG  PLL Feedback Divider Configuration   Offset 0x0210
 *
 *  Register Field      Field Range
 *  reserved            31 : 12
 *  bpll_fbdiv_cfg      11 : 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setPllFbdivCfg(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_FBDIV_CFG) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllFbdivCfg(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_FBDIV_CFG) >> 0) & 0xfff;
}

/**
 * PLL_FRAC_CFG  PLL Fractional Divider Configuration   Offset 0x0214
 *
 *  Register Field      Field Range
 *  reserved            31 : 24
 *  bpll_frac_cfg       23 : 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setPllFracCfg(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_FRAC_CFG) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllFracCfg(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_FRAC_CFG) >> 0) & 0xffffff;
}

/**
 * PLL_DIVVAL  PLL Spread Spectrum DIVVAL Configuration   Offset 0x021C
 *
 *  Register Field      Field Range
 *  reserved            31 : 6
 *  bpll_divval         5 : 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setPllDivval(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_DIVVAL) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllDivval(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_DIVVAL) >> 0) & 0x3f;
}

/**
 * PLL_SPREAD PLL Spread Spectrum Configuration   Offset 0x0220
 *
 *  Register Field      Field Range
 *  reserved            31 : 6
 *  bpll_spread         5 : 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setPllSpread(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_SPREAD) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllSpread(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_SPREAD) >> 0) & 0x3f;
}

/**
 * PLL_DOWNSPREAD PLL Spread Spectrum Down Spread Configuration   Offset 0x0224
 *
 *  Register Field      Field Range
 *  reserved            31 : 1
 *  bpll_downspread     0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setPllDownspread(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_DOWNSPREAD) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllDownspread(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_DOWNSPREAD) >> 0) & 0x01;
}

/**
 * PLL_EXT_MAXADDR PLL Spread Spectrum EXT MAXADDR Configuration   Offset 0x0228
 *
 *  Register Field      Field Range
 *  reserved            31 : 8
 *  bpll_ext_maxaddr    7 : 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setPllExtMaxaddr(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_EXT_MAXADDR) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllExtMaxaddr(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_EXT_MAXADDR) >> 0) & 0xff;
}

/**
 * PLL_SEL_EXTWAVE PLL Spread Spectrum EXT WAVE Configuration Enable   Offset 0x022C
 *
 *  Register Field      Field Range
 *  reserved            31 : 1
 *  bpll_sel_extwave    0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setPllSelExtwave(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_SEL_EXTWAVE) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllSelExtwave(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_SEL_EXTWAVE) >> 0) & 0x01;
}

/**
 * PLL_RESETPTR PLL Spread Spectrum Reset Pointer   Offset 0x0230
 *
 *  Register Field      Field Range
 *  reserved            31 : 1
 *  bpll_resetptr       0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setPllResetptr(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_RESETPTR) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllResetptr(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_RESETPTR) >> 0) & 0x01;
}

/**
 * PLL_DISABLE_SSCG PLL Spread Spectrum Function Disable Configuration   Offset 0x0234
 *
 *  Register Field      Field Range
 *  reserved            31 : 1
 *  bpll_disable_sscg   0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setPllDisableSscg(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_DISABLE_SSCG) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllDisableSscg(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_DISABLE_SSCG) >> 0) & 0x01;
}

/**
 ** PLL_LCK PLL Lock Status   Offset 0x0238
 *
 *  Register Field       Field Range
 *  reserved             31 : 1
 *  bpll_unlock_hist     0
 *
 * *
 */
__STATIC_INLINE void SysCtl_clearPllUnlckHis(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_UNLCK) = 1;
}

__STATIC_INLINE uint32_t SysCtl_getPllUnlckHis(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_UNLCK) & 0x01;
}

/**
 * PLL_LCK PLL Lock Status   Offset 0x023C
 *
 *  Register Field   Field Range
 *  reserved         31 : 1
 *  pll_lck          0
 *
 * *
 */
__STATIC_INLINE uint32_t SysCtl_getPllLck(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_LCK) & 0x01;
}


/**
 * OSC1_TRIM_OSC_EN   OSC1 Calibration Enable   Offset 0x0300
 *
 *  Register Field      Field Range
 *  reserved            31 : 1
 *  hsi_trim_osc_en     0
 *
 *  Bit 0: When calibration is enabled,
 *  the calibration value of the OSC comes from the real-time calibration value,
 *  not from the cfg_osc1_trim_value configuration.
 * *
 */
__STATIC_INLINE void SysCtl_setOsc1TrimOscEn(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_OSC_EN) = value;
}

__STATIC_INLINE uint32_t SysCtl_getOsc1TrimOscEn(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_OSC_EN) >> 0) & 0x01;
}

/**
 * OSC1_TRIM_OSC_STEP_NUM   OSC Calibration Step Configuration   Offset 0x0304
 *
 *  Register Field          Field Range
 *  reserved                31 : 4
 *  osc1_trim_osc_step_num  3  : 0
 *
 *  Bit 0-3: Represent 2^n steps.
 * *
 */
__STATIC_INLINE void SysCtl_setOsc1TrimOscStepNum(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_OSC_STEP_NUM) = value;
}

__STATIC_INLINE uint32_t SysCtl_getOsc1TrimOscStepNum(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_OSC_STEP_NUM) >> 0) & 0x0f;
}

/**
 * OSC1_TRIM_OSC_CALC_NUM   OSC1 Calibration Calculation Length   Offset 0x0308
 *
 *  Register Field              Field Range
 *  reserved                    31 : 24
 *  osc1_trim_osc_calc_num      23 : 0
 * *
 */
__STATIC_INLINE void SysCtl_setOsc1TrimOscCalcNum(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_OSC_CALC_NUM) = value;
}

__STATIC_INLINE uint32_t SysCtl_getOsc1TrimOscCalcNum(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_OSC_CALC_NUM) >> 0) & 0xffffff;
}

/**
 * OSC1_TRIM_OSC_WAIT_NUM   OSC1 Wait Refresh Time   Offset 0x030C
 *
 *  Register Field             Field Range
 *  reserved                   31 : 24
 *  osc1_trim_osc_wait_num     23 : 0
 * *
 */
__STATIC_INLINE void SysCtl_setOsc1TrimOscWaitNum(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_OSC_WAIT_NUM) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getOsc1TrimOscWaitNum(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_OSC_WAIT_NUM) >> 0) & 0xffffff;
}

/**
 * OSC2_TRIM_OSC_EN   OSC2 Calibration Enable   Offset 0x0320
 *
 *  Register Field           Field Range
 *  reserved                 31 : 1
 *  hsi_trim_osc_en          0
 *
 *  bit0: When calibration is enabled,
 *  the calibration value of OSC comes from real-time calibration value,
 *  i.e., not from cfg_osc2_trim_value configuration.
 * *
 */
__STATIC_INLINE void SysCtl_setOsc2TrimOscEn(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_OSC_EN) = value;
}

__STATIC_INLINE uint32_t SysCtl_getOsc2TrimOscEn(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_OSC_EN) >> 0) & 0x01;
}

/**
 * OSC2_TRIM_OSC_STEP_NUM   OSC2 Calibration Step Configuration   Offset 0x0324
 *
 *  Register Field             Field Range
 *  reserved                   31 : 4
 *  osc2_trim_osc_step_num     3  : 0
 *
 *  bit0-3: Representing 2^n steps.
 * *
 */
__STATIC_INLINE void SysCtl_setOsc2TrimOscStepNum(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_OSC_STEP_NUM) = value;
}

__STATIC_INLINE uint32_t SysCtl_getOsc2TrimOscStepNum(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_OSC_STEP_NUM) >> 0) & 0x0f;
}

/**
 * OSC2_TRIM_OSC_CALC_NUM   OSC2 Calibration Calculation Length   Offset 0x0328
 *
 *  Register Field              Field Range
 *  reserved                    31 : 24
 *  osc2_trim_osc_calc_num      23 : 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setOsc2TrimOscCalcNum(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_OSC_CALC_NUM) = value;
}

__STATIC_INLINE uint32_t SysCtl_getOsc2TrimOscCalcNum(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_OSC_CALC_NUM) >> 0) & 0xffffff;
}

/**
 * OSC2_TRIM_OSC_WAIT_NUM   OSC2 Wait Refresh Time   Offset 0x032C
 *
 *  Register Field             Field Range
 *  reserved                   31 : 24
 *  osc2_trim_osc_wait_num     23 : 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setOsc2TrimOscWaitNum(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_OSC_WAIT_NUM) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getOsc2TrimOscWaitNum(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_OSC_WAIT_NUM) >> 0) & 0xffffff;
}

/**
 * OSC_TRIM_VALUE_RPT   OSC Calibration Value Report   Offset 0x0340
 *
 *  Register Field          Field Range
 *  reserved                31
 *  osc2_trim_value_rpt     30 : 16
 *  reserved                15
 *  osc1_trim_value_rpt     14 :  0
 *
 *  Bit 0-14: Calibration value report, used for efuse burning.
 * *
 */
__STATIC_INLINE uint32_t SysCtl_getOsc1TrimValueRpt(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC_TRIM_VALUE_RPT) >> 0) & 0x7fff;
}

__STATIC_INLINE uint32_t SysCtl_getOsc2TrimValueRpt(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC_TRIM_VALUE_RPT) >> 16) & 0x7fff;
}


/**
 * SRST_N         Global Soft Reset   Offset 0x0400
 *
 *  Register Field           Field Range
 *  reserved                 31 : 1
 *  srst_n                   0         W1SR0
 * *
 */
__STATIC_INLINE void SysCtl_setSrstN(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_SYS) = value ;
}

__STATIC_INLINE void SysCtl_resetDevice(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_SYS) = 1 ;
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_SYS) = 0 ;
}

__STATIC_INLINE uint32_t SysCtl_getSrstN(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_SYS) >> 0) & 0x01;
}

/**
 * SRST_CLUSTER_N   Cluster Soft Reset Configuration   Offset 0x0404
 *
 *  Register Field       Field Range
 *  reserved             31 : 1
 *  srst_cluster_n       0
 * *
 */
__STATIC_INLINE void SysCtl_setDspClusterSrstN(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_CLUSTER_N) = value;
}

__STATIC_INLINE void SysCtl_resetDspCluster(void)
{
    SysCtl_setDspClusterSrstN(1);
    SysCtl_setDspClusterSrstN(0);
}

__STATIC_INLINE uint32_t SysCtl_getSlaveCoreSrstN(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_CLUSTER_N) & 0x01;
}

/**
 * SRST_DSP_CPU2_N    DSP Core 1 Soft Reset Configuration   Offset 0x0408
 *
 *  Register Field         Field Range
 *  reserved               31 : 1
 *  srst_dsp_cpu2_n       0
 * *
 */
__STATIC_INLINE void SysCtl_setCpu2SrstN(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_DSP_CPU2_N) = value;
}

__STATIC_INLINE void SysCtl_resetCpu2(void)
{
    SysCtl_setCpu2SrstN(0);
    SysCtl_setCpu2SrstN(1);
}

__STATIC_INLINE uint32_t SysCtl_getCpu2SrstN(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_DSP_CPU2_N) & 0x01;
}

/**
 ```plaintext
* SLV_CORE_RST_ON_SLV     SLV_CORE_RST_ON_SLV     Offset 0x0410
 *
 *  Register Field               Field Range
 *  reserved                     31 : 3
 *  cfg_cpu2_nmi_rst_on_sys   	 2   Whether system reset follows CPU2's NMI_WD reset
 *                                      0: Does not follow slave core reset
 *                                      1: Follows slave core reset
 *  cfg_cpu2_rst_on_sys       	 1   Whether system reset follows CPU2's watchdog reset
 *                                      0: Does not follow slave core reset
 *                                      1: Follows slave core reset
```
 * *
 */

__STATIC_INLINE void SysCtl_setCpu2ResetOnSys(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SLV_CORE_RST_ON_SLV) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_SLV_CORE_RST_ON_SLV) & (~(1<<1))) | ((enable & 0x01) << 1);
}

__STATIC_INLINE uint32_t SysCtl_getCpu2ResetOnSys(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_SLV_CORE_RST_ON_SLV) >> 1) & 0x01;
}

__STATIC_INLINE void SysCtl_setCpu2NmiResetOnSys(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SLV_CORE_RST_ON_SLV) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_SLV_CORE_RST_ON_SLV) & (~(1<<2))) | ((enable & 0x01) << 2);
}

__STATIC_INLINE uint32_t SysCtl_getCpu2NmiResetOnSys(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_SLV_CORE_RST_ON_SLV) >> 2) & 0x01;
}

/**
 * SLV_CORE_STOP_WITH_SLV  SLV_CORE_STOP_WITH_SLV      Offset 0x0414
 *
 *  Register Field               Field Range
 *  reserved                     31 : 2
 *  dsp_cpu2_stop_with_self   	 1   Whether slave core stops on reset follows core reset
 *                                      0: Does not follow slave core stop on reset
 *                                      1: Follows slave core system reset stop
 *
 */

__STATIC_INLINE void SysCtl_setDspCoreStopWithSlv(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SLV_CORE_STOP_WITH_SLV) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_SLV_CORE_STOP_WITH_SLV) & (~(1<<1))) | ((value & 0x01) << 1);
}

__STATIC_INLINE uint32_t SysCtl_getDspCoreStopWithSlv(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_SLV_CORE_STOP_WITH_SLV) >> 1) & 0x01;
}

/**
 * DSP_CORE_STOP_ON_SET    DSP_CORE_STOP_ON_SET Configuration     Offset 0x044C
 *
 *  Register Field         Field Range
 *  reserved              31 : 1
 *  dsp_core_stop_on_set   0
 */
__STATIC_INLINE void SysCtl_setDspCpu2StopOnSet(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_CORE_STOP_ON_SET) = 1;
}

/**
 * DSP_CORE_STOP_ON_CLR    DSP_CORE_STOP_ON_CLR Configuration     Offset 0x0450
 *
 *  Register Field         Field Range
 *  reserved              31 : 1
 *  dsp_core_stop_on_clr   0
 */
__STATIC_INLINE void SysCtl_setDspCpu2StopOnClr(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_CORE_STOP_ON_CLR) = 1;
}

/**
 * DSP_CORE_STOP_ON_RPT    DSP_CORE_STOP_ON_RPT Configuration     Offset 0x0454
 *
 *  Register Field         Field Range
 *  reserved              31 : 1
 *  dsp_core_stop_on_rpt   0
 */
__STATIC_INLINE uint32_t SysCtl_getDspCpu2StopOnRpt(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_CORE_STOP_ON_RPT) & 0x01;
}


/**
 * SRST_EFLASH_CTRL_N      eFlash controller soft reset Configuration     Offset 0x0504
 *
 *  Register Field         Field Range
 *  reserved               31 : 1
 *  srst_eflash_ctrl_n     0
 *
 */
__STATIC_INLINE void SysCtl_setSrstEflashCtrlN(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_EFLASH_CTRL_N) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getSrstEflashCtrlN(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_EFLASH_CTRL_N) >> 0) & 0x01;
}

__STATIC_INLINE void SysCtl_resetEflash(void)       //run in ram
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_EFLASH_CTRL_N) = 0 ;
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_EFLASH_CTRL_N) = 1 ;
}

/**
 * EFLASH_CTRL_CLK_EN      eFlash controller clock enable configuration register     Offset 0x0600
 *
 *  Register Field         Field Range
 *  reserved               31 : 1
 *  eflash_ctrl_clk_en     0
 *
 */

__STATIC_INLINE void SysCtl_setEflashCtrlClkEn(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_EFLASH_CTRL_CLK_EN) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getEflashCtrlClkEn(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_EFLASH_CTRL_CLK_EN) >> 0) & 0x01;
}

/**
 * TRACE_CLK_EN           Trace clock enable configuration register     Offset 0x0604
 *
 *  Register Field         Field Range
 *  reserved               31 : 1
 *  cfg_trace_clk_en       0
 *
 */
__STATIC_INLINE void SysCtl_setTraceClkEn(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_EFLASH_CTRL_CLK_EN) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getTraceClkEn(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_EFLASH_CTRL_CLK_EN) >> 0) & 0x01;
}

/**
 * DMA_CLK_ICG_BYPASS     DMA clock enable configuration register     Offset 0x0608
 *
 *  Register Field             Field Range
 *  reserved                   31 : 2
 *  dsp_dma1_clk_icg_bypass    2
 *  dsp_dma0_clk_icg_bypass    1
 *  arm_dma_clk_icg_bypass     0
 *
 */
__STATIC_INLINE void SysCtl_setArmDmaIcgBypass(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_DMA_CLK_ICG_BYPASS) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_DMA_CLK_ICG_BYPASS) & (~(1<<0))) | (value & 0x01) ;
}

__STATIC_INLINE uint32_t SysCtl_getArmDmaIcgBypass(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_DMA_CLK_ICG_BYPASS) & 0x01;
}

__STATIC_INLINE void SysCtl_setDspCpu1DmaIcgBypass(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_DMA_CLK_ICG_BYPASS) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_DMA_CLK_ICG_BYPASS) & (~(1<<1))) | ((value & 0x01) << 1) ;
}

__STATIC_INLINE uint32_t SysCtl_getDspCpu1DmaIcgBypass(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_DMA_CLK_ICG_BYPASS) >> 1) & 0x01;
}

__STATIC_INLINE void SysCtl_setDspCpu2DmaIcgBypass(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_DMA_CLK_ICG_BYPASS) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_DMA_CLK_ICG_BYPASS) & (~(1<<2))) | ((value & 0x01) << 2) ;
}

__STATIC_INLINE uint32_t SysCtl_getDspCpu2DmaIcgBypass(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_DMA_CLK_ICG_BYPASS) >> 2) & 0x01;
}

/**
 * DSP_CLK_ICG_BYPASS     DSP clock auto-CG function bypass configuration     Offset 0x060C
 *
 *  Register Field                 Field Range
 *  reserved                       31 : 2
 *  dsp_cpu2_nice_clk_icg_bypass   1
 *  dsp_cpu1_nice_clk_icg_bypass   0
 *
 */
__STATIC_INLINE void SysCtl_setDspCpu1NiceIcgBypass(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_CLK_ICG_BYPASS) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_CLK_ICG_BYPASS) & (~(1<<0))) | (value & 0x01) ;
}

__STATIC_INLINE uint32_t SysCtl_getDspCpu1NiceIcgBypass(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_CLK_ICG_BYPASS) & 0x01;
}

__STATIC_INLINE void SysCtl_setDspCpu2NiceIcgBypass(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_CLK_ICG_BYPASS) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_CLK_ICG_BYPASS) & (~(1<<1))) | ((value & 0x01) << 1) ;
}

__STATIC_INLINE uint32_t SysCtl_getDspCpu2NiceIcgBypass(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_CLK_ICG_BYPASS) >> 1) & 0x01;
}

/**
 * PARA_ICG_BYPASS         PARA_ICG    Configuration       Offset 0x06FC
 *
 *  Register Field         Field Range
 *  reserved              31 : 4
 *  io_cfg_para_icg_bypass          3
 *  ana_cfg_para_icg_bypass         2
 *  crg_cfg_para_icg_bypass         1
 *  sys_ctrl_para_icg_bypass        0
 * *
 */
__STATIC_INLINE void SysCtl_setSysCtrlParaIcgBypass(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) & (~(1<<0))) | (value & 0x01);
}

__STATIC_INLINE uint32_t SysCtl_getSysCtrlParaIcgBypass(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) & 0x01;
}

__STATIC_INLINE void SysCtl_setCrgCfgParaIcgBypass(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) & (~(1<<1))) | ((value & 0x01) << 1);
}

__STATIC_INLINE uint32_t SysCtl_getCrgCfgParaIcgBypass(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) >> 1) & 0x01;
}

__STATIC_INLINE void SysCtl_setAnaCfgParaIcgBypass(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) & (~(1<<2))) | ((value & 0x01) << 2) ;
}

__STATIC_INLINE uint32_t SysCtl_getAnaCfgParaIcgBypass(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) >> 2) & 0x01;
}

__STATIC_INLINE void SysCtl_setIoCfgParaIcgBypass(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) & (~(1<<3))) | ((value & 0x01) << 3);
}

__STATIC_INLINE uint32_t SysCtl_getIoCfgParaIcgBypass(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) >> 3) & 0x01;
}
/**
 * TOP_CRG_SPARE0    Reserved Register 0    Offset 0xF00
 *
 *  Register Field         Field Range
 *  top_crg_spare0        31 : 0
 * *
 */
__STATIC_INLINE void SysCtl_setTopCrgSpare0(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE0) = value;
}

__STATIC_INLINE uint32_t SysCtl_getTopCrgSpare0(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE0);
}

/**
 * TOP_CRG_SPARE1    Reserved Register 1    Offset 0xF04
 *
 *  Register Field         Field Range
 *  top_crg_spare1        31 : 0
 * *
 */
__STATIC_INLINE void SysCtl_setTopCrgSpare1(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE1) = value;
}

__STATIC_INLINE uint32_t SysCtl_getTopCrgSpare1(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE1);
}

/**
 * TOP_CRG_SPARE2    Reserved Register 2    Offset 0xF08
 *
 *  Register Field         Field Range
 *  top_crg_spare2        31 : 0
 * *
 */
__STATIC_INLINE void SysCtl_setTopCrgSpare2(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE2) = value;
}

__STATIC_INLINE uint32_t SysCtl_getTopCrgSpare2(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SPARE2);
}

#ifdef __cplusplus
}
#endif

#endif

#endif /* DEVICE_DRIVERLIB_SYSCTL_CRG_H_ */
