/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    sysctl_crg_v30.h
*   @brief   do NOT include this file directly, include "sysctl.h" instead
*
*/
/**
 * commit history
 * 20241218,Hezhiyuan,Modify comments
*/
#ifndef DEVICE_DRIVERLIB_SYSCTL_CRG_V30_H_
#define DEVICE_DRIVERLIB_SYSCTL_CRG_V30_H_

#include "gs32_version.h"

#if IS_GS32F00xx(0x30)

#ifdef __cplusplus
extern "C"{
#endif

#include "inc/hw_types.h"
#include "inc/hw_memmap.h"
#include "inc/hw_sysctl_crg_v30.h"
#include "sysctl/sysctl_dig_v30.h"

typedef enum CRG_DIV {
    CRG_DIV_1 = 1,
    CRG_DIV_2 = 2,
    CRG_DIV_4 = 4,
    CRG_DIV_8 = 8,
}crg_div_t;

/**
 * CRG_CFG_LOCK    Lock Control Register    Offset 0x0000
 *
 *  Register Field      Field Range
 *  CRG_CFG_lock        31 : 0
 *
 * Lock control register. Writing the unlock value to this register allows writing to all registers protected by lock. Writing any other value locks all registers protected by lock.
 * Unlock value: 0x5A5A_5A5A.
 * *
 */
__STATIC_INLINE void SysCtl_setSysctrlHLock(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CRG_CFG_LOCK) = 0x01;
}

__STATIC_INLINE void SysCtl_setSysctrlHUnlock(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CRG_CFG_LOCK) = 0x5A5A5A5A;
}

__STATIC_INLINE uint32_t SysCtl_getSysctrlHLock(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_CRG_CFG_LOCK);
}

/**
 * REF_CLK_SEL    PLL Reference Clock Selection    Offset 0x0004
 *
 *  Register Field      Field Range
 *  Reserved                31:3
 *  cfg_ext_clk_ref_sel     2
 *  cfg_ref_clk_src_sel     1 : 0
 *
 * 00: Internal Backup 10M
 * 01: Internal 10M (Default)
 * 10: Oscillator Circuit Clock
 * 11: Illegal Configuration, Configuration Disabled
 * *
 */

typedef enum EXT_REF_CLK_SEL {
    EXT_REF_CLK_SEL_XTAL = 0,
    EXT_REF_CLK_SEL_AUXIN = 1,
}ExtRefClkSel_t;

__STATIC_INLINE void SysCtl_setExtRefClkSel(ExtRefClkSel_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_REF_CLK_SEL) = (HWREG(CRG_CFG_BASE + CRG_CFG_O_REF_CLK_SEL) & (~(1<<2))) | ((value & 0x01) << 2);
}

typedef enum REF_CLK_TYPE {
    REF_CLK_TYPE_INTRENAL_OSC1_10MHZ = 0,
    REF_CLK_TYPE_INTERNAL_OSC2_10MHZ = 1,
    REF_CLK_TYPE_EXTERNAL_XTAL = 2,
}RefClkType_t;

__STATIC_INLINE void SysCtl_setRefClkSel(RefClkType_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_REF_CLK_SEL) = (HWREG(CRG_CFG_BASE + CRG_CFG_O_REF_CLK_SEL) & (~(3<<0))) | ((value & 0x03) << 0);
}

/**
 * SYSCLK_S2N_EN    System Clock SLOW to NORMAL Switch    Offset 0x0008
 *
 *  Register Field     Field Range
 *  reserved           31 : 1
 *  sysclk_s2n_en      0
 * *
 */
__STATIC_INLINE void SysCtl_setSysclkS2nEn(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_S2N_CLK_SEL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getSysclkS2nEn(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_S2N_CLK_SEL) >> 0) & 0x01;
}

/**
 * CAN_CLK_SRC_SEL    CAN Clock Source Selection Configuration Register    Offset 0x000C
 *
 *  Register Field      Field Range
 *  reserved            31 : 2
 *  can_clk_src_sel     1 : 0
 *
 *  Bit 1 : Bit 0
 *  00: SYS_MAX_CLK
 *  01: DSP_CLK
 *  10: AUXCLKIN
 *  11: Divided Clock
 * *
 */
typedef enum CAN_CLK_SRC_TYPE {
    CAN_CLK_TYPE_PLL = 0,
    CAN_CLK_TYPE_DSP = 1,
    CAN_CLK_TYPE_AUXCLK_IN = 2,
    CAN_CLK_TYPE_HSE = 3,
}CanClkSrcType_t;

__STATIC_INLINE void SysCtl_setCanClkSrcSel(CanClkSrcType_t type)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CAN_CORE_CLK_SEL) = type;
}

__STATIC_INLINE uint32_t SysCtl_getCanClkSrcSel(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_CAN_CORE_CLK_SEL) >> 0) & 0x03;
}

__STATIC_INLINE void SysCtl_setDspSysClkDiv(crg_div_t value)
{
    switch (value) {
        case CRG_DIV_1: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_SYS) = 0xFFFF; break;
        case CRG_DIV_2: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_SYS) = 0x5555; break;
        case CRG_DIV_4: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_SYS) = 0x1111; break;
        case CRG_DIV_8: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_SYS) = 0x0101; break;
        default: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_SYS) = 0xFFFF; break;
    }
}

__STATIC_INLINE crg_div_t SysCtl_getDspSysClkDiv(void)
{
    uint16_t crg = (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_DSP_SYS) >> 0) & 0xFFFF;

    switch (crg){
        case 0xFFFF : return CRG_DIV_1;
        case 0x5555 : return CRG_DIV_2;
        case 0x1111 : return CRG_DIV_4;
        case 0x0101 : return CRG_DIV_8;
        default: return CRG_DIV_1;
    }
}

/**
 * CFG_DIV_DSP_SYS    DSP Clock Division Configuration Register    Offset 0x0014
 *
 *  Register Field       Field Range
 *  reserved             31 : 16
 *  cfg_div_dsp_sys      15 : 0
 * *
 */
__STATIC_INLINE void SysCtl_setPeriClkDiv(crg_div_t value)
{
    switch (value) {
        case CRG_DIV_1: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_PERI_SYS) = 0xFFFF; break;
        case CRG_DIV_2: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_PERI_SYS) = 0x5555; break;
        case CRG_DIV_4: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_PERI_SYS) = 0x1111; break;
        case CRG_DIV_8: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_PERI_SYS) = 0x0101; break;
        default: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_PERI_SYS) = 0xFFFF; break;
    }
}

__STATIC_INLINE crg_div_t SysCtl_getPeriClkDiv(void)
{
    uint16_t crg = (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_PERI_SYS) >> 0) & 0xFFFF;
	
    switch (crg){
        case 0xFFFF : return CRG_DIV_1;
        case 0x5555 : return CRG_DIV_2;
        case 0x1111 : return CRG_DIV_4;
        case 0x0101 : return CRG_DIV_8;
        default: return CRG_DIV_1;
    }
}

/**
 * CFG_DIV_APB_SYS    APB Clock Division Configuration Register    Offset 0x0018
 *
 *  Register Field       Field Range
 *  reserved             31 : 16
 *  cfg_div_apb_sys      15 : 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setApbClkDiv(crg_div_t value)
{
    switch (value) {
        case CRG_DIV_1: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0xFFFF; break;
        case CRG_DIV_2: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0x5555; break;
        case CRG_DIV_4: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0x1111; break;
        case CRG_DIV_8: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0x0101; break;
        default: HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) = 0xFFFF; break;
    }
}

__STATIC_INLINE crg_div_t SysCtl_getApbClkDiv(void)
{
    uint16_t crg = (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_APB_SYS) >> 0) & 0xFFFF;

    switch (crg){
        case 0xFFFF : return CRG_DIV_1;
        case 0x5555 : return CRG_DIV_2;
        case 0x1111 : return CRG_DIV_4;
        case 0x0101 : return CRG_DIV_8;
        default: return CRG_DIV_1;
    }
}

/**
 * CFG_SYS_DIV_LOAD_POS    Clock Division Configuration Synchronization Update Register    Offset 0x001C
 *
 *  Register Field              Field Range
 *  reserved                    31 : 1
 *  cfg_sys_div_load_pos        0
 * *
 */
__STATIC_INLINE void SysCtl_setSysDivLoad(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_SYS_DIV_LOAD_POS) = 1;
}

/**
 * LIN_CLK_DIV    Lin Clock Division Configuration Register    Offset 0x0020
 *
 *  Register Field    Field Range
 *  reserved          31 : 4
 *  cfg_lin_clk_div    3 : 0   Counter Divider Configuration Divider Value is twice the configured value,
 *                              (e.g., setting 1 indicates a divide by 2)
 * *
 */
__STATIC_INLINE void SysCtl_setLinClkDiv(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_LIN_CLK_DIV) = value;
}

__STATIC_INLINE uint32_t SysCtl_getLinClkDiv(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_LIN_CLK_DIV) >> 0) & 0x0f;
}

/**
 * SMW_CLK_DIV    SMW Clock Division Configuration Register    Offset 0x0024
 *
 *  Register Field     Field Range
 *  reserved           31 : 4
 *  cfg_smw_clk_div     3 : 0   Counter Divider Configuration Divider Value is twice the configured value,
 *  							(e.g., setting 1 indicates a divide by 2)
 * *
 */
__STATIC_INLINE void SysCtl_setSmwClkDiv(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SMW_CLK_DIV) = value;
}

__STATIC_INLINE uint32_t SysCtl_getSmwClkDiv(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_SMW_CLK_DIV) >> 0) & 0x0f;
}

/**
 * CAN_CLK_DIV    CAN Clock Division Configuration Register    Offset 0x0028
 *
 *  Register Field   Field Range
 *  crg_len          31 : 27
 *  crg_patten       26 : 0
 *  default          0x47FFFE01  divider by 9
 * *
 */
__STATIC_INLINE void SysCtl_setCanClkDiv(uint32_t div)
{
    if (div > 1 && div <= 27) {
        HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_CAN) = ((div-1)<<27) | (((0x7FFFFFF << div) & 0x7FFFFFF) | 0x01);
    } else {

    }
}

__STATIC_INLINE uint32_t SysCtl_getCanClkDiv(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_CAN);
}

/**
 * SMW_CLK_DIV    SMW Clock Division Configuration Register    Offset 0x0024
 *
 *  Register Field     Field Range
 *  reserved           31 : 4
 *  cfg_hrcap_clk_div   3 : 0   Counter Divider Configuration Divider Value is twice the configured value,
 *                              (e.g., setting 1 indicates a divide by 2)
 * *
 */
__STATIC_INLINE void SysCtl_setHrcapClkDiv(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_HRCAP) = value;
}

__STATIC_INLINE uint32_t SysCtl_getHrcapClkDiv(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DIV_HRCAP) >> 0) & 0x0f;
}

typedef enum WDG_PRE_DIV_VAL {
    WDG_PRE_DIV_1 = 0,
    WDG_PRE_DIV_2 = 1,
    WDG_PRE_DIV_4 = 2,
    WDG_PRE_DIV_8 = 3,
    WDG_PRE_DIV_16 = 4,
    WDG_PRE_DIV_32 = 5,
    WDG_PRE_DIV_64 = 6,
    WDG_PRE_DIV_128 = 7,
    WDG_PRE_DIV_256 = 8,
    WDG_PRE_DIV_512 = 9,
    WDG_PRE_DIV_1024 = 10,
    WDG_PRE_DIV_2048 = 11,
    WDG_PRE_DIV_4096 = 12,
}WDGPreDivVal_t;

typedef enum WDG_SCALER_DIV_VAL {
    WDG_SCALER_DIV_1 = 0,
    WDG_SCALER_DIV_2 = 1,
    WDG_SCALER_DIV_4 = 2,
    WDG_SCALER_DIV_8 = 3,
    WDG_SCALER_DIV_16 = 4,
    WDG_SCALER_DIV_32 = 5,
    WDG_SCALER_DIV_64 = 6,
}WDGScalerDivVal_t;


/**
 * DSP_WDG_CLK_PRE_DIV    DSP Watchdog Clock Prescaler Division Configuration Register    Offset 0x0038
 *
 *  Register Field         Field Range
 *  reserved               31 : 7
 *  dsp_wdg_clk_div        6 : 0
 * *
 */
__STATIC_INLINE void SysCtl_setDspWdgClkPreDiv(WDGPreDivVal_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DSP_WDG_CLK_PRE_DIV) = value;
}

__STATIC_INLINE uint32_t SysCtl_getDspWdgClkPreDiv(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DSP_WDG_CLK_PRE_DIV) >> 0) & 0x7f;
}

/**
 * CFG_DSP_WDG_CLK_SCALER_DIV    DSP Watchdog Scaler Clock Division Configuration Register    Offset 0x003C
 *
 *  Register Field                Field Range
 *  reserved                      31 : 4
 *  cfg_dsp_wdg_clk_pre_div       3 : 0       Default: 9
 * *
 */
__STATIC_INLINE void SysCtl_setDspWdgClkScalerDiv(WDGScalerDivVal_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DSP_WDG_CLK_SCALER_DIV) = value;
}

__STATIC_INLINE uint32_t SysCtl_getDspWdgClkScalerDiv(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DSP_WDG_CLK_SCALER_DIV) >> 0) & 0x0F;
}

/**
 * CFG_DBG_WDT_HOLD_EN    Debug Watchdog Hold Enable Register    Offset 0x0040
 *
 *  Register Field           Field Range
 *  reserved                 31 : 1
 *  cfg_debug_wdt_hold_en    0
 *  bit 1: DSP Watchdog counter pause follows core halted
 * *
 */
__STATIC_INLINE void SysCtl_setDspWdgDebugHoldEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DBG_WDT_HOLD_EN) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DBG_WDT_HOLD_EN) & ~1U) | (enable & 0x01U);
}

__STATIC_INLINE uint32_t SysCtl_getDspWdgDebugHoldEn(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_DBG_WDT_HOLD_EN) & 0x01);
}

/**
 * DSP_WDG_RST_EN    DSP Watchdog Reset Enable Configuration    Offset 0x0048
 *
 *  Register Field           Field Range
 *  reserved                 31 : 2
 *  cfg_dsp_wdg_rst_en       1 : 0
 *  bit[1]: Window Watchdog
 *  bit[0]: Normal Watchdog
 *  1: Enable, can trigger reset
 *  0: Disable, does not trigger reset
 * *
 */
__STATIC_INLINE void SysCtl_setDspWdgResetEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_WDG_RST_EN) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_WDG_RST_EN) & ~(1<<0)) | (enable & 0x01);
}

__STATIC_INLINE uint32_t SysCtl_getDspWdgResetEn(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_WDG_RST_EN) & 0x01;
}

__STATIC_INLINE void SysCtl_setDspWwdgResetEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_WDG_RST_EN) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_WDG_RST_EN) & ~(1<<1)) | ((enable & 0x01) << 1);
}

__STATIC_INLINE uint32_t SysCtl_getDspWwdgResetEn(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_WDG_RST_EN) >> 1) & 0x01;
}

/**
 * HSE_CMD_EN    HSE Detection Enable Configuration Register    Offset 0x0050
 *
 *  Register Field         Field Range
 *  reserved               31 : 1
 *  hse_cmd_en             0
 *  HSE Detection Enable
 *  0: Detection disabled
 *
 * *
 */
__STATIC_INLINE void SysCtl_setHseCmdEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CMD_EN) = enable ;
}

__STATIC_INLINE uint32_t SysCtl_getHseCmdEn(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CMD_EN) & 0x01;
}

/**
 * CFG_CMD_ST_CLR    CMD_ST_CLR Enable Configuration Register    Offset 0x0054
 *
 *  Register Field         Field Range
 *  reserved               31 : 1
 *  cfg_cmd_st_clr         0
 *  CMD_ST_CLR Enable
 *  0: Disable CMD_ST_CLR
 *
 * *
 */
__STATIC_INLINE void SysCtl_setCmdStClr(uint32_t val)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CMD_ST_CLR) = val;
}

__STATIC_INLINE uint32_t SysCtl_getCmdStClr(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CMD_ST_CLR) & 0x01;
}

/**
 * CMD_CHECK_FAIL_RPT    CMD Check Result Reporting    Offset 0x0058
 *
 *  Register Field         Field Range
 *  reserved               31 : 1
 *  cmd_check_fail_rpt     0
 *  CMD Check Result Reporting
 *  0: Reporting disabled
 *
 * *
 */
__STATIC_INLINE uint32_t SysCtl_getCmdCheckFailRpt(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_CMD_CHECK_FAIL_RPT) & 0x01;
}

/**
 * TEST_MUX_EN    Test Signal Output Selection Configuration Register    Offset 0x0060
 *
 *  Register Field    Field Range
 *  reserved          31 : 1
 *  test_mux_en       0
 */
__STATIC_INLINE void SysCtl_setTestMuxEn(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TEST_MUX_EN) = value;
}

__STATIC_INLINE uint32_t SysCtl_getTestMuxEn(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_TEST_MUX_EN) & 0x01;
}

/**
 * TEST_MUX_SEL   Test Observation Signal Selection Register   Offset 0x0064
 *
 *  Register Field    Field Range
 *  reserved          31 : 4
 *  test_mux_sel      3 : 0
 */
typedef enum TestSigSel {
    TEST_SIG_PLL_UNLOCK = 0,
    TEST_SIG_CMD_CHECK_FAIL = 1,
    TEST_SIG_CLK_SHUT = 2,
    TEST_SIG_SYS_RSTN = 3,
    TEST_SIG_XRS_RSTN = 4,
    TEST_SIG_DSP_SYS_RSTN = 5,
    TEST_SIG_DSP_POR_RSTN = 6,
}TestSigSel_t;

__STATIC_INLINE void SysCtl_setTestSigSel(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TEST_MUX_SEL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getTestSigSel(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_TEST_MUX_SEL)) & 0x0f;
}

/**
 * OBS_SIG_SEL    Observation Clock Selection Configuration Register   Offset 0x0068
 *
 *  Register Field    Field Range
 *  reserved          31 : 4
 *  obs_sig_sel       3 : 0
 */
typedef enum OBSSigSel {
    OBS_SIG_SYS_MAX = 0,
    OBS_SIG_SMW_CLK = 1,
    OBS_SIG_CAN_CORE_CLK = 2,
    OBS_SIG_APB_CLK = 3,
    OBS_SIG_PERI_CLK = 4,
    OBS_SIG_DSP_CLK = 5,
    OBS_SIG_PLL_DIV2 = 8,
    OBS_SIG_PLL = 9,
    OBS_SIG_REF_PIN = 10,
    OBS_SIG_OSC1 = 11,
    OBS_SIG_OSC2 = 12,
    OBS_SIG_PLL_REF = 13,
}OSBSigSel_t;

__STATIC_INLINE void SysCtl_setObsSigSel(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OBS_SIG_SEL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getObsSigSel(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_OBS_SIG_SEL) & 0x0f;
}

/**
 * OBS_SIG_DIV    Observation Clock Divider Configuration Register   Offset 0x006C
 *
 *  Register Field    Field Range
 *  reserved          31 : 7
 *  obs_sig_div       6 : 0
 */
__STATIC_INLINE void SysCtl_setObsSigDiv(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OBS_SIG_DIV) = value;
}

__STATIC_INLINE uint32_t SysCtl_getObsSigDiv(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_OBS_SIG_DIV) & 0x3f;
}

/**
 * X1_CNT    X1 Clock Self-Test Register   Offset 0x0070
 *
 *  Register Field            Field Range
 *  cfg_ref_pin_clk_cnt_clr  16
 *  reserved                  15 : 11
 *  ref_pin_clk_cnt           10 : 0
 */
__STATIC_INLINE void SysCtl_setX1CntClr(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_X1_CNT) = 1<<16;
}

__STATIC_INLINE uint32_t SysCtl_getX1Cnt(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_X1_CNT) & 0x7FF;
}

/**
 * DSP_CORE_RST_RECORD   System Reset Cause Record   Offset 0x0074
 *
 *  Register Field           Field Range
 *  reserved                 31 : 8
 *  core_rst_record          7 : 0
 *
 */
typedef enum {
    RESET_TYPE_WWDG         = 1<<0,
    RESET_TYPE_WDG          = 1<<1,
    RESET_TYPE_NMIWDG       = 1<<2,
    RESET_TYPE_CORE         = 1<<3,
    RESET_TYPE_EFUSE_LAUNCH = 1<<4,
    RESET_TYPE_XRST         = 1<<5,
    RESET_TYPE_SOFT         = 1<<6,
    RESET_TYPE_JTAG_AUTH    = 1<<7,
    RESET_TYPE_POR          = 1<<8,
}CoreResetType_t;

__STATIC_INLINE CoreResetType_t SysCtl_getCoreRstRecord(void)
{
    return (CoreResetType_t)(
        (HWREG(CRG_CFG_BASE + CRG_CFG_O_CORE_RST_RECORD) & 0xFF) |
        (SysCtl_getPorFlag() << 8)
    );
}

/**
 * @brief Clear Master Core Reset Record
 * @param resetType: Reset type(s) to clear, can be OR'ed together
 *                  e.g. RESET_TYPE_WWDG | RESET_TYPE_WDG
 * @return None
 *
 * Note: This register is write-1-to-clear
 */
__STATIC_INLINE void SysCtl_clearCoreRstRecord(CoreResetType_t resetType)
{
    // Write 1 to clear the specified reset type bits
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CORE_RST_RECORD) = resetType & 0xFF;
    if(resetType & RESET_TYPE_POR)
    {
    	SysCtl_clearPorFlag();
    }
}

/**
 * PLL_PD  PLL Power Down Configuration   Offset 0x0200
 *
 *  Register Field          Field Range
 *  reserved                31 : 1
 *  bpll_pd                 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setPllPd(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_PD) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllPd(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_PD) >> 0) & 0x01;
}

/**
 * PLL_BYPASS  PLL Bypass Configuration   Offset 0x0204
 *
 *  Register Field          Field Range
 *  reserved                31 : 1
 *  bpll_bypass             0
 * *
 */
__STATIC_INLINE void SysCtl_setPllBypass(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_BYPASS) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllBypass(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_BYPASS) >> 0) & 0x01;
}

/**
 * PLL_OTHPD_CFG  PLL Function Power Down Configuration   Offset 0x0208
 *
 *  Register Field          Field Range
 *  reserved                31 : 5
 *  bpll_othpd_cfg          4 : 0
 *
 *  [0]: DACPD
 *  [1]: DSMPD
 *  [2]: FOUT4PHASEPD
 *  [3]: FOUTPOSTIDVPD
 *  [4]: FOUTVCOPD
 * *
 */
__STATIC_INLINE void SysCtl_setPllOthpdCfg(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_OTHPD_CFG) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllOthpdCfg(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_OTHPD_CFG) >> 0) & 0x1f;
}

/**
 * PLL_DIV_CFG  PLL Divider Configuration   Offset 0x020C
 *
 *  Register Field          Field Range
 *  reserved                31 : 12
 *  bpll_div_cfg            11 : 0
 *
 *  [2:0]: POSTDIV1
 *  [5:3]: POSTDIV2
 *  [11:6]: REFDIV
 * *
 */
__STATIC_INLINE void SysCtl_setPllDivCfg(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_DIV_CFG) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllDivCfg(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_DIV_CFG) >> 0) & 0xfff;
}

/**
 * PLL_FBDIV_CFG  PLL Feedback Divider Configuration   Offset 0x0210
 *
 *  Register Field          Field Range
 *  reserved                31 : 12
 *  bpll_fbdiv_cfg          11 : 0
 * *
 */
__STATIC_INLINE void SysCtl_setPllFbdivCfg(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_FBDIV_CFG) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllFbdivCfg(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_FBDIV_CFG) >> 0) & 0xfff;
}

/**
 * PLL_FRAC_CFG  PLL Fractional Configuration   Offset 0x0214
 *
 *  Register Field      Field Range
 *  reserved            31 : 24
 *  bpll_frac_cfg       23 : 0
 * *
 */
__STATIC_INLINE void SysCtl_setPllFracCfg(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_FRAC_CFG) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllFracCfg(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_FRAC_CFG) >> 0) & 0xffffff;
}

/**
 * PLL_DIVVAL  PLL Spread Spectrum DIVVAL Configuration   Offset 0x021C
 *
 *  Register Field      Field Range
 *  reserved            31 : 6
 *  bpll_divval         5  : 0
 * *
 */
__STATIC_INLINE void SysCtl_setPllDivval(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_DIVVAL) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllDivval(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_DIVVAL) >> 0) & 0x3f;
}

/**
 * PLL_SPREAD PLL Spread Spectrum Configuration   Offset 0x0220
 *
 *  Register Field      Field Range
 *  reserved            31 : 6
 *  bpll_spread         5  : 0
 * *
 */
__STATIC_INLINE void SysCtl_setPllSpread(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_SPREAD) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllSpread(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_SPREAD) >> 0) & 0x3f;
}

/**
 * PLL_DOWNSPREAD PLL Down Spread Configuration   Offset 0x0224
 *
 *  Register Field      Field Range
 *  reserved            31 : 1
 *  bpll_downspread     0
 * *
 */
__STATIC_INLINE void SysCtl_setPllDownspread(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_DOWNSPREAD) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllDownspread(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_DOWNSPREAD) >> 0) & 0x01;
}

/**
 * PLL_EXT_MAXADDR PLL Spread Spectrum EXT MAXADDR Configuration   Offset 0x0228
 *
 *  Register Field      Field Range
 *  reserved            31 : 8
 *  bpll_ext_maxaddr    7  : 0
 * *
 */
__STATIC_INLINE void SysCtl_setPllExtMaxaddr(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_EXT_MAXADDR) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllExtMaxaddr(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_EXT_MAXADDR) >> 0) & 0xff;
}

/**
 * PLL_SEL_EXTWAVE PLL Spread Spectrum EXT WAVE Configuration Enable   Offset 0x022C
 *
 *  Register Field      Field Range
 *  reserved            31 : 1
 *  bpll_sel_extwave    0
 * *
 */
__STATIC_INLINE void SysCtl_setPllSelExtwave(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_SEL_EXTWAVE) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllSelExtwave(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_SEL_EXTWAVE) >> 0) & 0x01;
}

/**
 * PLL_RESETPTR PLL Spread Spectrum Reset Pointer   Offset 0x0230
 *
 *  Register Field      Field Range
 *  reserved            31 : 1
 *  bpll_resetptr       0
 * *
 */
__STATIC_INLINE void SysCtl_setPllResetptr(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_RESETPTR) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllResetptr(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_RESETPTR) >> 0) & 0x01;
}

/**
 * PLL_DISABLE_SSCG PLL Spread Spectrum Function Disable Configuration   Offset 0x0234
 *
 *  Register Field      Field Range
 *  reserved            31 : 1
 *  bpll_disable_sscg   0
 * *
 */
__STATIC_INLINE void SysCtl_setPllDisableSscg(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_DISABLE_SSCG) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getPllDisableSscg(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_DISABLE_SSCG) >> 0) & 0x01;
}

/**
 * PLL_LCK PLL Lock Status   Offset 0x0238
 *
 *  Register Field      Field Range
 *  reserved            31 : 1
 *  bpll_unlck_his      0
 * *
 */
__STATIC_INLINE void SysCtl_clearPllUnlckHis(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_UNLCK) = 1;
}

__STATIC_INLINE uint32_t SysCtl_getPllUnlckHis(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_UNLCK) & 0x01;
}

/**
 * PLL_LCK PLL Lock Status   Offset 0x023C
 *
 *  Register Field      Field Range
 *  reserved            31 : 1
 *  pll_lck             0
 * *
 */
__STATIC_INLINE uint32_t SysCtl_getPllLck(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_PLL_LCK) & 0x01;
}


/**
 * OSC1_TRIM_OSC_EN   OSC1 Calibration Enable   Offset 0x0300
 *
 *  Register Field        Field Range
 *  reserved              31 : 1
 *  hsi_trim_osc_en       0
 *
 *  bit0: When calibration is enabled,
 *  the calibration value of OSC1 comes from the real-time calibration value,
 *  not from the cfg_osc1_trim_value configuration.
 * *
 */
__STATIC_INLINE void SysCtl_setOsc1TrimOscEn(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_OSC_EN) = value;
}

__STATIC_INLINE uint32_t SysCtl_getOsc1TrimOscEn(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_OSC_EN) >> 0) & 0x01;
}

/**
 * OSC1_TRIM_OSC_STEP_NUM   OSC Calibration Step Configuration   Offset 0x0304
 *
 *  Register Field            Field Range
 *  reserved                  31 : 4
 *  osc1_trim_osc_step_num    3  : 0
 *
 *  bit0-3: "n" represents 2^n steps.
 * *
 */
__STATIC_INLINE void SysCtl_setOsc1TrimOscStepNum(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_OSC_STEP_NUM) = value;
}

__STATIC_INLINE uint32_t SysCtl_getOsc1TrimOscStepNum(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_OSC_STEP_NUM) >> 0) & 0x0f;
}

/**
 * OSC1_TRIM_OSC_CALC_NUM   OSC1 Calibration Calculation Length   Offset 0x0308
 *
 *  Register Field             Field Range
 *  reserved                   31 : 24
 *  osc1_trim_osc_calc_num     23 : 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setOsc1TrimOscCalcNum(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_OSC_CALC_NUM) = value;
}

__STATIC_INLINE uint32_t SysCtl_getOsc1TrimOscCalcNum(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_OSC_CALC_NUM) >> 0) & 0xffffff;
}

/**
 * OSC1_TRIM_OSC_WAIT_NUM   OSC1 Waiting Refresh Time   Offset 0x030C
 *
 *  Register Field             Field Range
 *  reserved                   31 : 24
 *  osc1_trim_osc_wait_num     23 : 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setOsc1TrimOscWaitNum(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_OSC_WAIT_NUM) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getOsc1TrimOscWaitNum(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_OSC_WAIT_NUM) >> 0) & 0xffffff;
}

/**
 * OSC1_TRIM_REF_CALC_NUM   osc1_trim_ref_calc_num   Offset 0x0310
 *
 *  Register Field             Field Range
 *  reserved                   31 : 24
 *  osc1_trim_ref_calc_num     23 : 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setOsc1TrimRefCalcNum(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_REF_CALC_NUM) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getOsc1TrimRefCalcNum(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_REF_CALC_NUM) >> 0) & 0xffffff;
}

/**
 * OSC1_TRIM_OVER   OSC1 trim over   Offset 0x0314
 *
 *  Register Field         Field Range
 *  reserved              31 : 1
 *  osc1_trim_over_ind       0
 * *
 */
__STATIC_INLINE uint32_t SysCtl_getOsc1TrimOverInd(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC1_TRIM_OVER) >> 0) & 0x01;
}

/**
 * OSC2_TRIM_OSC_EN   OSC2 Calibration Enable   Offset 0x0320
 *
 *  Register Field         Field Range
 *  reserved              31 : 1
 *  hsi_trim_osc_en       0
 *
 *  bit0: When calibration is enabled,
 *  the calibration value of OSC comes from the real-time calibration value,
 *  i.e., not from the cfg_osc2_trim_value configuration.
 * *
 */
__STATIC_INLINE void SysCtl_setOsc2TrimOscEn(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_OSC_EN) = value;
}

__STATIC_INLINE uint32_t SysCtl_getOsc2TrimOscEn(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_OSC_EN) >> 0) & 0x01;
}

/**
 * OSC2_TRIM_OSC_STEP_NUM   OSC2 Calibration Step Configuration   Offset 0x0324
 *
 *  Register Field             Field Range
 *  reserved                   31 : 4
 *  osc2_trim_osc_step_num     3  : 0
 *
 *  bit0-3: 'n' represents 2^n steps.
 * *
 */
__STATIC_INLINE void SysCtl_setOsc2TrimOscStepNum(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_OSC_STEP_NUM) = value;
}

__STATIC_INLINE uint32_t SysCtl_getOsc2TrimOscStepNum(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_OSC_STEP_NUM) >> 0) & 0x0f;
}

/**
 * OSC2_TRIM_OSC_CALC_NUM   OSC2 Calibration Calculation Length   Offset 0x0328
 *
 *  Register Field             Field Range
 *  reserved                   31 : 24
 *  osc2_trim_osc_calc_num     23 : 0
 * *
 */
__STATIC_INLINE void SysCtl_setOsc2TrimOscCalcNum(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_OSC_CALC_NUM) = value;
}

__STATIC_INLINE uint32_t SysCtl_getOsc2TrimOscCalcNum(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_OSC_CALC_NUM) >> 0) & 0xffffff;
}

/**
 * OSC2_TRIM_OSC_WAIT_NUM   OSC2 Waiting Refresh Time   Offset 0x032C
 *
 *  Register Field             Field Range
 *  reserved                   31 : 24
 *  osc2_trim_osc_wait_num     23 : 0
 * *
 */
__STATIC_INLINE void SysCtl_setOsc2TrimOscWaitNum(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_OSC_WAIT_NUM) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getOsc2TrimOscWaitNum(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_OSC_WAIT_NUM) >> 0) & 0xffffff;
}

/**
 * OSC2_TRIM_REF_CALC_NUM   osc2_trim_ref_calc_num   Offset 0x0330
 *
 *  Register Field             Field Range
 *  reserved                   31 : 24
 *  osc2_trim_ref_calc_num     23 : 0
 *
 * *
 */
__STATIC_INLINE void SysCtl_setOsc2TrimRefCalcNum(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_REF_CALC_NUM) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getOsc2TrimRefCalcNum(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_REF_CALC_NUM) >> 0) & 0xffffff;
}

/**
 * OSC2_TRIM_OVER   OSC2 trim over   Offset 0x0334
 *
 *  Register Field         Field Range
 *  reserved              31 : 1
 *  osc2_trim_over_ind       0
 * *
 */
__STATIC_INLINE uint32_t SysCtl_getOsc2TrimOverInd(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC2_TRIM_OVER) >> 0) & 0x01;
}

/**
 * OSC_TRIM_VALUE_RPT   OSC Calibration Value Report   Offset 0x0340
 *
 *  Register Field         Field Range
 *  reserved               31
 *  osc2_trim_value_rpt    30 : 16
 *  reserved               15
 *  osc1_trim_value_rpt    14 :  0
 *
 *  bit0-14: Calibration value report, used for efuse burning.
 * *
 */
__STATIC_INLINE uint32_t SysCtl_getOsc1TrimValueRpt(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC_TRIM_VALUE_RPT) >> 0) & 0x7fff;
}

__STATIC_INLINE uint32_t SysCtl_getOsc2TrimValueRpt(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC_TRIM_VALUE_RPT) >> 16) & 0x7fff;
}

/**
 * OSC_TRIM_REFCLK_SEL   OSC Trim reference clock selection   Offset 0x0344
 *
 *  Register Field         Field Range
 *  reserved               31 : 1
 *  osc_trim_refclk_sel    0
 *
 *  bit0-14: Calibration value report, used for efuse burning.
 * *
 */
typedef enum {
    OSC_TRIM_CLK_SEL_X1      = 0,
    OSC_TRIM_CLK_SEL_AUXCLK  = 1,
}OSCTrimClkSel_t;

__STATIC_INLINE void SysCtl_setOscTrimRefClkSel(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC_TRIM_REFCLK_SEL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getOscTrimRefClkSel(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_OSC_TRIM_REFCLK_SEL) >> 0) & 0x1;
}


/**
 * SRST_N         Global Software Reset    Offset 0x0400
 *
 *  Register Field       Field Range
 *  reserved             31 : 1
 *  srst_n               0
 * *
 */
__STATIC_INLINE void SysCtl_setSrstN(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_SYS) = value ;
}

__STATIC_INLINE void SysCtl_resetDevice(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_SYS) = 1 ;
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_SYS) = 0 ;
}

__STATIC_INLINE uint32_t SysCtl_getSrstN(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_SYS) >> 0) & 0x01;
}

/**
 * SRST_EFLASH_CTRL_N        EFlash Control Software Reset      offset 0x0504
 *
 *  Register Field                 Field Range
 *  reserved                          31 : 1
 *  srst_eflash_ctrl_n                0
 * *
 */
__STATIC_INLINE void SysCtl_setSrstEflashCtrlN(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_EFLASH_CTRL_N) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getSrstEflashCtrlN(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_EFLASH_CTRL_N) >> 0) & 0x01;
}

__STATIC_INLINE void SysCtl_resetEflash(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_EFLASH_CTRL_N) = 0 ;
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_EFLASH_CTRL_N) = 1 ;
}

/**
 * SRST_DMA_N        DMA Software Reset      offset 0x50C
 *
 *  Register Field         Field Range
 *  reserved                  31 : 1
 *  srst_dma_n                   0
 * *
 */
__STATIC_INLINE void SysCtl_setSrstDmaN(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_DMA_N) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getSrstDmaN(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_DMA_N) >> 0) & 0x01;
}

__STATIC_INLINE void SysCtl_resetDma(void)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_DMA_N) = 0 ;
    HWREG(CRG_CFG_BASE + CRG_CFG_O_SRST_DMA_N) = 1 ;
}

/**
 * EFLASH_CTRL_CLK_EN      EFlash Controller Clock Enable Configuration Register   offset 0x600
 *
 *  Register Field             Field Range
 *  reserved                      31 : 1
 *  eflash_ctrl_clk_en               0
 * *
 */
__STATIC_INLINE void SysCtl_setEflashCtrlClkEn(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_EFLASH_CTRL_CLK_EN) = value ;
}

__STATIC_INLINE uint32_t SysCtl_getEflashCtrlClkEn(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_EFLASH_CTRL_CLK_EN) >> 0) & 0x01;
}

/**
 * DMA_CLK_ICG_BYPASS      DMA Clock Enable Configuration Register   offset 0x608
 *
 *  Register Field                  Field Range
 *  reserved                           31 : 1
 *  cfg_dsp_dma_clk_icg_bypass         0
 * *
 */

__STATIC_INLINE void SysCtl_setDspDmaIcgBypass(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_DMA_CLK_ICG_BYPASS) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_DMA_CLK_ICG_BYPASS) & ~(1<<0)) | ((value & 0x01) << 1) ;
}

__STATIC_INLINE uint32_t SysCtl_getDspDmaIcgBypass(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_DMA_CLK_ICG_BYPASS) >> 0) & 0x01;
}

/**
 * DSP_CLK_ICG_BYPASS      DSP Clock Automatic CG Function Bypass Configuration   offset 0x60C
 *
 *  Register Field                        Field Range
 *  reserved                                  31 : 2
 *  cfg_dsp_core_clk_icg_bypass                1
 *  cfg_dsp_nice_clk_icg_bypass                0
 * *
 */
__STATIC_INLINE void SysCtl_setDspNiceIcgBypass(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_CLK_ICG_BYPASS) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_CLK_ICG_BYPASS) & ~(1<<0)) | (value & 0x01) ;
}

__STATIC_INLINE uint32_t SysCtl_getDspNiceIcgBypass(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_CLK_ICG_BYPASS) & 0x01;
}

__STATIC_INLINE void SysCtl_setDspCoreIcgBypass(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_CLK_ICG_BYPASS) =
            (HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_CLK_ICG_BYPASS) & ~(1<<1)) | ((value & 0x01) << 1) ;
}

__STATIC_INLINE uint32_t SysCtl_getDspCoreIcgBypass(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_DSP_CLK_ICG_BYPASS) >> 1) & 0x01;
}

/**
 * CFG_STANDBY    Configuration for Entering Low Power State During Sleep   offset 0x614
 *
 *  Register Field                        Field Range
 *  reserved                                  31 : 1
 *  cfg_standby                               0
 * *
 */
__STATIC_INLINE void SysCtl_setStandby(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_STANDBY) = (value & 0x01) ;
}

__STATIC_INLINE uint32_t SysCtl_getStandby(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_STANDBY) & 0x01;
}

/**
 * CFG_WDG_WAKE_EN    Main Core Watchdog Wakeup Enable   offset 0x618
 *
 *  Register Field                        Field Range
 *  reserved                                  31 : 1
 *  cfg_wdg_wake_en                           0
 * *
 */
__STATIC_INLINE void SysCtl_setWdgWakeEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_WDG_WAKE_EN) = enable;
}

__STATIC_INLINE uint32_t SysCtl_getWdgWakeEn(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_WDG_WAKE_EN) & 0x01;
}

/**
 * CFG_GPIO_WAKE_EN    GPIO Wakeup Enable   offset 0x61C
 *
 *  Register Field                        Field Range
 *  reserved                                  31 : 1
 *  cfg_gpio_wake_en                         0
 * *
 */
__STATIC_INLINE void SysCtl_setGpioWakeEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_GPIO_WAKE_EN) = enable;
}

__STATIC_INLINE uint32_t SysCtl_getGpioWakeEn(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_GPIO_WAKE_EN) & 0x01;
}

/**
 * CFG_CMP_WAKE_EN    CMPSS Wakeup Enable   offset 0x620
 *
 *  Register Field                        Field Range
 *  reserved                                  31 : 8
 *  cfg_cmp_wake_en                           7 : 0
 * *
 */
typedef enum CmpssWakeEn{
    CMPSS_WAKE_CMPSS1H = 1<<0,
    CMPSS_WAKE_CMPSS1L = 1<<1,
    CMPSS_WAKE_CMPSS2H = 1<<2,
    CMPSS_WAKE_CMPSS2L = 1<<3,
    CMPSS_WAKE_CMPSS3H = 1<<4,
    CMPSS_WAKE_CMPSS3L = 1<<5,
    CMPSS_WAKE_CMPSS4H = 1<<6,
    CMPSS_WAKE_CMPSS4L = 1<<7,
}CmpssWakeEn_t;

__STATIC_INLINE void SysCtl_setCmpssWakeEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CMP_WAKE_EN) = enable;
}

__STATIC_INLINE uint32_t SysCtl_getCmpssWakeEn(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CMP_WAKE_EN) & 0xFF;
}

/**
 * CFG_CMP_WAKE_EN    CMPSS Wakeup Enable   offset 0x620
 *
 *  Register Field            Field Range
 *  reserved                    31 : 4
 *  io_cfg_para_icg_bypass      3
 *  ana_cfg_para_icg_bypass     2
 *  crg_cfg_para_icg_bypass     1
 *  sys_ctrl_para_icg_bypass    0
 * *
 */
__STATIC_INLINE void SysCtl_setSysCtrlParaIcgEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) = (HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) & (1<<0)) | ((enable & 0x01) << 0);
}

__STATIC_INLINE void SysCtl_setCrgCfgParaIcgEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) = (HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) & (1<<1)) | ((enable & 0x01) << 1);
}

__STATIC_INLINE void SysCtl_setAnaCfgParaIcgEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) = (HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) & (1<<2)) | ((enable & 0x01) << 2);
}

__STATIC_INLINE void SysCtl_setIoCfgParaIcgEn(uint32_t enable)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) = (HWREG(CRG_CFG_BASE + CRG_CFG_O_PARA_ICG_BYPASS) & (1<<3)) | ((enable & 0x01) << 3);
}

__STATIC_INLINE uint32_t SysCtl_getSysCtrlParaIcgEn(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CMP_WAKE_EN) >> 0) & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_getCrgCfgParaIcgEn(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CMP_WAKE_EN) >> 1) & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_getAnaCfgParaIcgEn(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CMP_WAKE_EN) >> 2) & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_getIoCfgParaIcgEn(void)
{
    return (HWREG(CRG_CFG_BASE + CRG_CFG_O_CFG_CMP_WAKE_EN) >> 3) & 0x01;
}

/**
 * TOP_CRG_SYS_SPARE0~2    Spare Register 0~2   offset 0xF00 ~ 0xF08
 *
 *  Register Field                        Field Range
 *  top_crg_spare                         31 : 0
 * *
 */
__STATIC_INLINE void SysCtl_setTopCrgSysSpare0(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SYS_SPARE0) = value;
}

__STATIC_INLINE uint32_t SysCtl_getTopCrgSysSpare0(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SYS_SPARE0);
}

__STATIC_INLINE void SysCtl_setTopCrgSysSpare1(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SYS_SPARE1) = value;
}

__STATIC_INLINE uint32_t SysCtl_getTopCrgSysSpare1(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SYS_SPARE1);
}

__STATIC_INLINE void SysCtl_setTopCrgSysSpare2(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SYS_SPARE2) = value;
}

__STATIC_INLINE uint32_t SysCtl_getTopCrgSysSpare2(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_SYS_SPARE2);
}

/**
 * TOP_CRG_POR_SPARE0~2    Spare Register 0~2   offset 0xF0C ~ 0xF14
 *
 *  Register Field                        Field Range
 *  top_crg_spare                         31 : 0
 * *
 */
__STATIC_INLINE void SysCtl_setTopCrgPorSpare0(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_POR_SPARE0) = value;
}

__STATIC_INLINE uint32_t SysCtl_getTopCrgPorSpare0(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_POR_SPARE0);
}

__STATIC_INLINE void SysCtl_setTopCrgPorSpare1(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_POR_SPARE1) = value;
}

__STATIC_INLINE uint32_t SysCtl_getTopCrgPorSpare1(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_POR_SPARE1);
}

__STATIC_INLINE void SysCtl_setTopCrgPorSpare2(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_POR_SPARE2) = value;
}

__STATIC_INLINE uint32_t SysCtl_getTopCrgPorSpare2(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_POR_SPARE2);
}

/**
 * TOP_CRG_XRS_SPARE0~2    Spare Register 0~2   offset 0xF18 ~ 0xF20
 *
 *  Register Field                        Field Range
 *  top_crg_spare                         31 : 0
 * *
 */
__STATIC_INLINE void SysCtl_setTopCrgXrsSpare0(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_XRS_SPARE0) = value;
}

__STATIC_INLINE uint32_t SysCtl_getTopCrgXrsSpare0(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_XRS_SPARE0);
}

__STATIC_INLINE void SysCtl_setTopCrgXrsSpare1(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_XRS_SPARE1) = value;
}

__STATIC_INLINE uint32_t SysCtl_getTopCrgXrsSpare1(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_XRS_SPARE1);
}

__STATIC_INLINE void SysCtl_setTopCrgXrsSpare2(uint32_t value)
{
    HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_XRS_SPARE2) = value;
}

__STATIC_INLINE uint32_t SysCtl_getTopCrgXrsSpare2(void)
{
    return HWREG(CRG_CFG_BASE + CRG_CFG_O_TOP_CRG_XRS_SPARE2);
}

#ifdef __cplusplus
}
#endif

#endif

#endif /* DEVICE_DRIVERLIB_SYSCTL_CRG_H_ */
