/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    sysctl_dig_v12.h
*   @brief   digital system top level configuration
*            do NOT include this file directly, include "sysctl.h" instead
*
*/

#ifndef DEVICE_DRIVERLIB_SYSCTL_DIG_V12_H_
#define DEVICE_DRIVERLIB_SYSCTL_DIG_V12_H_

#include "gs32_version.h"

#if IS_GS32F00xx(0x12)

#ifdef __cplusplus
extern "C"{
#endif

#include "inc/hw_types.h"
#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include <inc/hw_sysctl_dig_v12.h>

/*
 * DIG_SYSCTRL_LOCK   Lock control register   Offset 0x0000
 *
 * Register Field   Field Range
 * dig_sysctrl_lock   31 : 0
 * 
 * Writing unlock value 0x5A5A5A5A unlocks the register, allowing writes to all registers protected by lock.
 * Writing any other value locks the registers protected by lock.
 *
 */
__STATIC_INLINE void SysCtl_setDigSysctrlLock(void)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_SYSCTRL_LOCK) = 0x01;
}

__STATIC_INLINE void SysCtl_setDigSysctrlUnlock(void)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_SYSCTRL_LOCK) = 0x5A5A5A5A;
}

__STATIC_INLINE uint32_t SysCtl_getDigSysctrlLock(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_SYSCTRL_LOCK);
}


/*
 * CCM_MEM_CTRL   CCM parameter configuration register   0x200
 *
 * Register Field   Field Range
 * ccm_mem_ctrl     31 : 0
 *
 */
__STATIC_INLINE void SysCtl_setCcmMemCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_CCM_MEM_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getCcmMemCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_CCM_MEM_CTRL);
}



/*
 * CCM_MEM_PWR_CTRL   CCM memory low-power configuration register   Offset 0x0204
 *
 * Register Field   Field Range
 * ccm_mem_pwr_ctrl     31 : 0
 *
 */
__STATIC_INLINE void SysCtl_setCcmMemPwrCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_CCM_MEM_PWR_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getCcmMemPwrCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_CCM_MEM_PWR_CTRL);
}


/*
 * FLASH_MEM_CTRL   Flash memory parameter configuration register   Offset 0x208
 *
 * Register Field   Field Range
 * eflash_cache_mem_ctrl     31 : 0
 *
 */
__STATIC_INLINE void SysCtl_setFlashMemCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_FLASH_MEM_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getFlashMemCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_FLASH_MEM_CTRL);
}

/*
 * FLASH_MEM_PWR_CTRL   Flash memory low-power configuration register   Offset 0x20C
 *
 * Register Field   Field Range
 * eflash_mem_pwr_ctrl     31 : 0
 *
 */
__STATIC_INLINE void SysCtl_setFlashMemPwrCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_FLASH_MEM_PWR_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getFlashMemPwrCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_FLASH_MEM_PWR_CTRL);
}

/*
 * GS_MEM_CTRL   GS memory parameter configuration register   Offset 0x0210
 *
 * Register Field   Field Range
 * gs_mem_ctrl     31 : 0
 *
 */
__STATIC_INLINE void SysCtl_setGsMemCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_GS_MEM_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getGsMemCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_GS_MEM_CTRL);
}


/*
 * GS_MEM_PWR_CTRL   GS memory RAM low-power configuration register   Offset 0x0214
 *
 * Register Field   Field Range
 * gs_mem_pwr_ctrl     31 : 0
 *
 */
__STATIC_INLINE void SysCtl_setGsMemPwrCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_GS_MEM_PWR_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getGsMemPwrCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_GS_MEM_PWR_CTRL);
}

/*
 * SRAM1_MEM_CTRL   SRAM1 memory parameter configuration register   Offset 0x218
 *
 * Register Field   Field Range
 * sram1_mem_ctrl     31 : 0
 *
 */
__STATIC_INLINE void SysCtl_setSram1MemCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_SRAM1_MEM_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getSram1MemCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_SRAM1_MEM_CTRL);
}


/*
 * SRAM1_MEM_PWR_CTRL   SRAM1 memory low-power configuration register   Offset 0x21C
 *
 * Register Field   Field Range
 * sram1_mem_pwr_ctrl     31 : 0
 *
 */
__STATIC_INLINE void SysCtl_setSram1MemPwrCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_SRAM1_MEM_PWR_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getSram1MemPwrCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_SRAM1_MEM_PWR_CTRL);
}

/*
 * DSP_MEM_CTRL   DSP memory parameter configuration register   Offset 0x0220
 *
 * Register Field   Field Range
 * dsp_mem_ctrl     31 : 0
 *
 */

/*
 * Set the value of field dsp_mem_ctrl
 */
__STATIC_INLINE void SysCtl_setDspMemCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DSP_MEM_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getDspMemCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DSP_MEM_CTRL);
}

/*
 * DSP_MEM_PWR_CTRL   DSP memory low-power configuration register   Offset 0x0224
 *
 * Register Field   Field Range
 * dsp_mem_pwr_ctrl     31 : 0
 *
 */
__STATIC_INLINE void SysCtl_setDspMemPwrCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DSP_MEM_PWR_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getDspMemPwrCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DSP_MEM_PWR_CTRL);
}


/*
 * CCM_ECC_ERR_RPT     CCM Status Reporting Register   Offset 0x0300
 *
 * Register Field           Field Range
 * reserved                 31 : 2
 * ccm_ecc_rd_err          1        ECC Correctable Error Reporting
 * ccm_ecc_real_bad_err    0        ECC Uncorrectable Error Reporting
 * 
 */
__STATIC_INLINE uint32_t SysCtl_getCcmRamRdErr(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_CCM_ECC_ERR_RPT) >> 1) & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_clearCcmRamRdErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_CCM_ECC_ERR_RPT) = 0x01 << 1;
}

__STATIC_INLINE uint32_t SysCtl_getCcmRamRealBadRdErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_CCM_ECC_ERR_RPT) & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_clearCcmRamRealBadRdErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_CCM_ECC_ERR_RPT) = 0x01;
}

/*
 * CCM_ECC_SYNDROME_RPT   ECC information reporting   Offset 0x304
 * 
 * Register Field   			Field Range
 * reserved		       			31 : 7
 * ccm_ecc_err_syndrome_rpt		6: 0			ECC information reporting
 *
 */
__STATIC_INLINE uint32_t SysCtl_getCcmRamSyndrome(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_CCM_ECC_SYNDROME_RPT) & 0x7F;
}

__STATIC_INLINE uint32_t SysCtl_clearCcmRamSyndrome(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_CCM_ECC_SYNDROME_RPT) = 0x7F;
}


/*
 * SRAM1_PARITY_RPT   SRAM1 odd parity error reporting.   Offset 0x308
 *
 * Register Field   Field Range
 * reserved                    31 : 4
 * sram1_parity_err            3 : 0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getSram1ParityErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_SRAM1_PARITY_RPT) & 0xF;
}

__STATIC_INLINE uint32_t SysCtl_clearSram1ParityErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_SRAM1_PARITY_RPT) = 0xF;
}


/*
 * GS_PARITY_RPT   GS RAM odd parity error reporting.   Offset 0x030C
 * 
 * Register Field   Field Range
 * reserved                31 : 29
 * gs_sram8_parity_err     7
 * gs_sram7_parity_err     6
 * gs_sram6_parity_err     5
 * gs_sram5_parity_err     4
 * gs_sram4_parity_err     3
 * gs_sram3_parity_err     2
 * gs_sram2_parity_err     1
 * gs_sram1_parity_err     0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getGsRamParityErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_GS_PARITY_RPT) & 0xFF;
}

__STATIC_INLINE uint32_t SysCtl_clearGsRamParityErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_GS_PARITY_RPT) = 0xFF;
}


/*
 * LM_PARITY_RPT   DSP LM odd parity error reporting.  Offset 0x0310
 * 
 * Register Field   Field Range
 * reserved                    31 : 4
 * lm_sram_parity_err          3 : 0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getLsRamParityErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_LM_PARITY_RPT) & 0xF;
}

__STATIC_INLINE uint32_t SysCtl_clearLsRamParityErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_LM_PARITY_RPT) = 0xF;
}


/*
 * FLASH_ICACHE_ECC_ERR_RPT   ECC verification error reporting.  Offset 0x0314
 *
 * Register Field   Field Range
 * reserved                          31 : 2
 * flash_icache_ecc_rd_err           1
 * flash_icache_ecc_real_bad_err     0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getFlashIcacheRdErr(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_FLASH_ICACHE_ECC_ERR_RPT) >> 1) & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_clearFlashIcacheRdErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_FLASH_ICACHE_ECC_ERR_RPT) = 0x01 << 1;
}

__STATIC_INLINE uint32_t SysCtl_getFlashIcacheRealBadRdErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_FLASH_ICACHE_ECC_ERR_RPT) & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_clearFlashIcacheRealBadRdErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_FLASH_ICACHE_ECC_ERR_RPT) = 0x01;
}


/*
 * PVM_INT_RPT   PVM reporting  Offset 0x0318
 *
 * Register Field   Field Range
 * reserved                    31 : 1
 * pvm_rpt                     0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getPvmRpt(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_PVM_INT_RPT) & 0x1;
}

__STATIC_INLINE uint32_t SysCtl_clearPvmRpt(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_PVM_INT_RPT) = 0x1;
}


/*
 * SMP_MODE   Data Acquisition Mode Configuration Register   Offset 0x400
 *
 * Register Field   Field Range
 * reserved                    31 : 1
 * gs_ram_smp_mode             0
 *
 * 0: Functional mode
 * 1: Data acquisition mode, CPU can only perform read accesses (initiating write accesses will be automatically masked and ineffective), write accesses are performed by the data acquisition module.
 */
__STATIC_INLINE void SysCtl_setGsramSmpMode(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_GS_SRAM_SMP_MODE) = value & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_getGsramSmpMode(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_GS_SRAM_SMP_MODE) & 0x01;
}


/*
 * FFT_RAM_MODE   Configuration Register for Using the Last 24KB of GS_RAM as FFT RAM.   Offset 0x404
 *
 * Register Field   Field Range
 * reserved                    31 : 1
 * fft_ram_en                  0
 * 0: Not FFT RAM
 * 1: FFT RAM
 */
__STATIC_INLINE void SysCtl_setGsramFftMode(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_FFT_RAM_MODE) = value & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_getGsramFftMode(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_FFT_RAM_MODE) & 0x01;
}


/*
 * BOOT_PIN_LATCH_MSK   Indicates if the Boot Pin Latch is Affected by System Reset.   Offset: 0x500
 *
 * Register Field   Field Range
 * reserved                    31 : 1
 * boot_pin_latch_msk          0
 *
 * 0: Bootpin is locked again after system deasserts reset.
 * 1: Bootpin remains unlocked after system deasserts reset, only latched during power-up sequence.
 */
__STATIC_INLINE void SysCtl_setBootPinLatchMask(uint32_t mask)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_BOOT_PIN_LATCH_MSK) = mask & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_getBootPinLatchMask(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_BOOT_PIN_LATCH_MSK) & 0x01;
}


/*
 * BOOT_PIN_LATCH_RPT   Boot Pin Latch Value   Offset: 0x504
 *
 * Register Field   Field Range
 * reserved                    31 : 2
 * boot_pin_lock_data_rpt      1 : 0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getBootPinLatchRpt(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_BOOT_PIN_LATCH_RPT) & 0x03;
}


/*
 * BUS_REMAP_CFG   Software Configurable Bus Matrix Remap Configuration   Offset: 0x508
 *
 * Register Field   Field Range
 * reserved                    31 : 4
 * cfg_ccm_remap               3
 * cfg_mem_remap_mode          2 : 1         00: Flash Boot
 *                                           01: GS RAM Boot
 *                                           Others: Configuration Disabled
 * cfg_soft_ctrl_remap_en      0
 *
 */
__STATIC_INLINE void SysCtl_setSoftCtrlRemapEn(uint32_t enable)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_BUS_REMAP_CFG) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_BUS_REMAP_CFG) & ~(1<<0)) | (enable & 0x01);
}

__STATIC_INLINE uint32_t SysCtl_getSoftCtrlRemapEn(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_BUS_REMAP_CFG) & 0x01;
}

typedef enum {
    MEM_REMAP_FLASH = 0,
    MEM_REMAP_GSRAM = 1,
}MemRemapModeType_t;

__STATIC_INLINE void SysCtl_setMemMapMode(MemRemapModeType_t mode)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_BUS_REMAP_CFG) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_BUS_REMAP_CFG) & ~(3<<1)) | ((mode & 0x03) << 1);
}

__STATIC_INLINE uint32_t SysCtl_getMemMapMode(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_BUS_REMAP_CFG) >> 1) & 0x03;
}

__STATIC_INLINE void SysCtl_setCcmRemap(uint32_t val)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_BUS_REMAP_CFG) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_BUS_REMAP_CFG) & ~(1<<3)) | ((val & 0x01) << 3);
}

__STATIC_INLINE uint32_t SysCtl_getCcmRemap(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_BUS_REMAP_CFG) >> 3) & 0x01;
}


/*
 * BUS_REMAP_RPT   BUS MATRIX Remap Report   Offset: 0x50C
 *
 * Register Field   Field Range
 * reserved                    31 : 4
 * bus_remap_rpt               3 : 0         0000: Flash Aliased
 *                                           0001: GS Aliased
 *                                           100x: CCM Remap, where x follows the above meanings
 */
__STATIC_INLINE uint32_t SysCtl_getBusRemapRpt(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_BUS_REMAP_RPT) & 0x0F;
}


/*
 * DSP_RESET_VECTOR_RPT   DSP Reset Vector Report   Offset: 0x0510
 *
 * Register Field   Field Range
 * dsp_reset_vector_rpt        31 : 0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getDspResetVectorRpt(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DSP_RESET_VECTOR_RPT);
}

/*
 * DSP_STAT_REG DSP Core Status Report Register     0x608
 *
 *  Register Field       		Field Range
 *  reserved                    31 : 2
 *  dsp_core_wfi_mode           1
 *  dsp_core_sleep_value        0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getDspSleepValue(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DSP_STAT_REG) & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_getWfiMode(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_DSP_STAT_REG) >> 1) & 0x01;
}


/*
 *  DSP_RESET_VECTOR        DSP Reset Initial Vector Register    0x60C
 *
 *  Register Field       		Field Range
 *  cfg_dsp_reset_vector       31 : 0
 *
 */
__STATIC_INLINE void SysCtl_setDspResetVector(uint32_t vector)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DSP_RESET_VECTOR) = vector;
}

__STATIC_INLINE uint32_t SysCtl_getDspResetVector(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DSP_RESET_VECTOR);
}

__STATIC_INLINE void SysCtl_setDspDeadLoop(uint32_t addr)
{
    HWREG(addr) = (0x0000A001U);        //riscv instruction "j ."
}

/*
 * DSP_HALT_RPT         DSP Halt Report Register        0x610
 *
 *  Register Field       Field Range
 *  reserved                    31 : 1
 *  dsp_hart_halted             0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getDspHaltedSta(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DSP_HALT_RPT) & 0x01;
}


/*
 *  DBG_SWEN                Debug Software Enable Register    0x614
 *
 *  Register Field       Field Range
 *  reserved                    31 : 1
 *  dbg_swen                    0
 *
 */
__STATIC_INLINE uint32_t SysCtl_setDbgSwen(uint32_t enable)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DBG_SWEN) = enable;
}

__STATIC_INLINE uint32_t SysCtl_getDbgSwen(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DBG_SWEN);
}


/*
 *  SLV_CORE_JTAG_CFG    Slave Core JTAG Configuration Register     0x618
 *
 *  Register Field              Field Range
 *  reserved                            31 : 2
 *  slv_core_jtag_2wire_reuse           1       Enables reuse of 2-wire mode for debugging slave core,
 *                                               0: Disabled, requires separate JTAG connection for slave core (GPIO MUX)
 *                                               1: Enabled, reuses TDI/TDO interface of master core's JTAG as 2-wire debug interface for slave core.
 *  slv_core_jtag_discrete_en           0       Enables discrete debug for slave core,
 *                                               0: Disabled, slave core debug through daisy chain
 *                                               1: Enabled, slave core supports independent 2-wire, reused 2-wire, or independent 4-wire debug.
 * Note: For independent 2-wire or independent 4-wire debug, GPIO MUX configuration is also required to multiplex functionality to slave core JTAG.
 */
__STATIC_INLINE void SysCtl_setSlvCoreJtagDiscreteEn(uint32_t enable)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_SLV_CORE_JTAG_CFG) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_SLV_CORE_JTAG_CFG) & ~(1<<0)) | (enable & 0x01);
}

__STATIC_INLINE uint32_t SysCtl_getSlvCoreJtagDiscreteEn(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_SLV_CORE_JTAG_CFG) & 0x01;
}

__STATIC_INLINE void SysCtl_setSlvCoreJtag2WireReuse(uint32_t enable)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_SLV_CORE_JTAG_CFG) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_SLV_CORE_JTAG_CFG) & ~(1<<1)) | ((enable & 0x01) << 1);
}

__STATIC_INLINE uint32_t SysCtl_getSlvCoreJtag2WireReuse(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_SLV_CORE_JTAG_CFG) >> 1) & 0x01;
}


/*
 *  EFLASH_JTAG_CFG     Embedded Flash JTAG Configuration Register     0x61C
 *
 *  Register Field             Field Range
 *  bist_enable					2			Bist enable
 *  eflash_jtag_macro_sel       1  			0: macro0;   1: macro1
 *  eflash_jtag_reuse_en        0   		0: Uses dedicated JTAG pins for Flash
 *                                          1: Reuses chip JTAG pins (JTAG functionality disconnected)
 */



__STATIC_INLINE void SysCtl_setEflashJtagReuseEn(uint32_t enable)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_EFLASH_JTAG_CFG) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_EFLASH_JTAG_CFG) & ~(1<<0)) | (enable & 0x01);
}

__STATIC_INLINE uint32_t SysCtl_getEflashJtagReuseEn(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_EFLASH_JTAG_CFG) & 0x01;
}

__STATIC_INLINE void SysCtl_setEflashJtagMacroSel(uint32_t enable)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_EFLASH_JTAG_CFG) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_EFLASH_JTAG_CFG) & ~(1<<1)) | ((enable & 0x01) << 1);
}

__STATIC_INLINE uint32_t SysCtl_getEflashJtagMacroSel(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_EFLASH_JTAG_CFG) >> 1) & 0x01;
}

__STATIC_INLINE void SysCtl_setBistEnable(uint32_t enable)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_EFLASH_JTAG_CFG) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_EFLASH_JTAG_CFG) & ~(1<<2)) | (enable & 0x01 << 2);
}

__STATIC_INLINE uint32_t SysCtl_getBistEnable(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_EFLASH_JTAG_CFG >> 2) & 0x01;
}


/*
 *  BUS_AUSER_ID_CFG    Bus User ID Configuration Register     0x628
 *
 *  Register Field         Field Range
 *  reserved                       31 : 15
 *  cfg_dsp_dma_hauser             14 : 12
 *  reserved                       11
 *  reserved                       7
 *  cfg_dsp_hauser                 6 : 4
 *  reserved                       3
 */
typedef enum {
    DSP_HAUSER = 4,
    DSP_DMA_HAUSER = 12,
}BusHauserType_t;

__STATIC_INLINE void SysCtl_setBusAuserIdCfg(BusHauserType_t bus, uint32_t id)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_BUS_AUSER_ID_CFG) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_BUS_AUSER_ID_CFG) & ~(7<<bus)) | ((id & 0x07) << bus);
}

__STATIC_INLINE uint32_t SysCtl_getBusAuserIdCfg(BusHauserType_t bus)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_BUS_AUSER_ID_CFG) >> bus) & 0x07;
}

/*
 *  DSP_NMI_MASK    DSP Non-Maskable Interrupt Selection Register     0x644
 *
 *  Register Field     Field Range
 *  reserved                  31 : 16
 *  dsp_nmi_mask              15 : 0
 */
__STATIC_INLINE void SysCtl_setDspNmiMask(uint32_t mask)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DSP_NMI_MASK) = mask & 0xFFFF;
}

__STATIC_INLINE uint32_t SysCtl_getDspNmiMask(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DSP_NMI_MASK) & 0xFFFF;
}

/*
 *  DSP_SYS_ERR_MASK    DSP System Error Interrupt Selection Register    0x64C
 *
 *  Register Field     Field Range
 *  reserved                  31 : 16
 *  dsp_sys_err_mask          15 : 0
 */
__STATIC_INLINE void SysCtl_setDspSysErrMask(uint32_t mask)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DSP_SYS_ERR_MASK) = mask & 0xFFFF;
}

__STATIC_INLINE uint32_t SysCtl_getDspSysErrMask(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DSP_SYS_ERR_MASK) & 0xFFFF;
}


/*
 *  PWM_ECCERR_MASK         PWM ECC Error Mask Register       0x650
 *
 *  Register Field         Field Range
 *  reserved                      31 : 16
 *  cfg_pwm_ecc_err_mask          15 : 0
 */
__STATIC_INLINE void SysCtl_setPwmEccErrMask(uint32_t mask)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_PWM_ECCERR_MASK) = mask & 0xFFFF;
}

__STATIC_INLINE uint32_t SysCtl_getPwmEccErrMask(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_PWM_ECCERR_MASK) & 0xFFFF;
}


/*
 *  PWM_SYSFAULT_MASK      PWM System Fault Mask Register       0x654
 *
 *  Register Field         		  Field Range
 *  reserved                      31 : 16
 *  cfg_pwm_sys_err_mask          15 : 0
 */
__STATIC_INLINE void SysCtl_setPwmSysErrMask(uint32_t mask)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_PWM_SYSFAULT_MASK) = mask & 0xFFFF;
}

__STATIC_INLINE uint32_t SysCtl_getPwmSysErrMask(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_PWM_SYSFAULT_MASK) & 0xFFFF;
}


/*
 *  MEM_INI_REQ            Memory Initialization Request Configuration Register       0x700
 *
 *  Register Field         Field Range
 *  reserved                      31 : 5
 *  ilm_mem_ini_req               4
 *  dlm_mem_ini_req               3
 *  gs_mem_ini_req                2
 *  ccm_mem_ini_req               1
 *  sram1_mem_ini_req             0
 */
typedef enum {
    MEM_SRAM1 = 1<<0,
    MEM_CCM = 1<<1,
    MEM_GS = 1<<2,
    MEM_DLM = 1<<3,
    MEM_ILM = 1<<4,
}MemIniType_t;
__STATIC_INLINE void SysCtl_setMemIniReq(MemIniType_t memTypes)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_MEM_INI_REQ) = memTypes;
}


/*
 *  MEM_INI_DONE_RPT       Memory Initialization Done Report Register       0x704
 *
 *  Register Field         Field Range
 *  reserved                      31 : 5
 *  ilm_mem_ini_done_rpt           4
 *  dlm_mem_ini_done_rpt           3
 *  gs_mem_ini_done_rpt            2
 *  ccm_mem_ini_done_rpt           1
 *  sram1_mem_ini_done_rpt         0
 */
__STATIC_INLINE uint32_t SysCtl_getMemIniDoneRpt(MemIniType_t memTypes)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_MEM_INI_DONE_RPT) & memTypes;
}

__STATIC_INLINE void SysCtl_clearMemIniDoneRpt(MemIniType_t memTypes)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_MEM_INI_DONE_RPT) = memTypes;
}


/*
 *  PMBUS_CTL_OEN           PMBus Control Output Enable Register       0x708
 *
 *  Register Field         Field Range
 *  reserved                      31 : 1
 *  pmbus_ctl_oen                 0
 */
__STATIC_INLINE void SysCtl_setPmbusCtlOen(uint32_t val)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_PMBUS_CTL_OEN) = val;
}

__STATIC_INLINE uint32_t SysCtl_getPmbusCtlOen(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_PMBUS_CTL_OEN) & 0x01;
}


/*
 *  SDFM_TYPE_CFG           SDFM Type Configuration Register       0x70C
 *
 *  Register Field         Field Range
 *  reserved                      31 : 2
 *  sdfm_type_cfg                 1 : 0
 */
__STATIC_INLINE void SysCtl_setSdfmTypeCfg(uint32_t type)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_SDFM_TYPE_CFG) = type;
}

__STATIC_INLINE uint32_t SysCtl_getSdfmTypeCfg(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_SDFM_TYPE_CFG) & 0x03;
}


/*
 *  SEC_EN_CFG              Security Enable Configuration Register       0x710
 *
 *  Register Field         Field Range
 *  reserved                      31 : 1
 *  sec_en_cfg                     0     Security Enable Configuration. Writing once triggers sec_en to be set to 1.
 */
__STATIC_INLINE void SysCtl_setSecEnCfg(uint32_t enable)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_SEC_EN_CFG) = enable;
}

__STATIC_INLINE uint32_t SysCtl_getSecEnCfg(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_SEC_EN_CFG) & 0x01;
}


/*
 *  SEC_EN_RPT              Security Enable Report Register      0x714
 *
 *  Register Field         Field Range
 *  reserved                      31 : 1
 *  sec_en_cfg                     0     Security Enable Configuration. Writing once triggers sec_en to be set to 1.
 */
__STATIC_INLINE uint32_t SysCtl_getSecEnRpt(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_SEC_EN_RPT) & 0x01;
}


/*
 *  AES_LITTLE_ENDIAN_AHB   AES Little Endian Configuration      0x71C
 *
 *  Register Field         Field Range
 *  reserved                      31 : 1
 *  aes_little_endian_ahb         0     AES Little Endian Configuration.
 */
__STATIC_INLINE void SysCtl_setAesLittleEndianAhb(uint32_t enable)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_AES_LITTLE_ENDIAN_AHB) = enable;
}

__STATIC_INLINE uint32_t SysCtl_getAesLittleEndianAhb(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_AES_LITTLE_ENDIAN_AHB) & 0x01;
}

/*
 *  MBIST_CFG_PROG_END_IDX   MBIST-CFG0 register      0x0800
 *
 *  Register Field         Field Range
 *  reserved                 31 : 2
 *  cfg_prog_end_idx         1  : 0			Mbist end idx
 */
__STATIC_INLINE void SysCtl_setCfgProgEndIdx(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_MBIST_CFG_PROG_END_IDX) = value;
}

__STATIC_INLINE uint32_t SysCtl_getCfgProgEndIdx(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_MBIST_CFG_PROG_END_IDX) & 0x03;
}


/*
 *  MBIST_CFG_PROG_STA_IDX   MBIST-CFG0 register      0x0804
 *
 *  Register Field         Field Range
 *  reserved                 31 : 2
 *  cfg_prog_sta_idx         1  : 0			MBIST Start IDX
 */
__STATIC_INLINE void SysCtl_setCfgProgStaIdx(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_MBIST_CFG_PROG_STA_IDX) = value;
}

__STATIC_INLINE uint32_t SysCtl_getCfgProgStaIdx(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_MBIST_CFG_PROG_STA_IDX) & 0x03;
}

/*
 *  MBIST_CFG_PROG_IDX   MBIST-CFG0 register      0x0808
 *
 *  Register Field         Field Range
 *  reserved                 31 : 2
 *  cfg_prog_idx         	 1  : 0			MBIST Access IDX
 */
__STATIC_INLINE void SysCtl_setCfgProgIdx(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_MBIST_CFG_PROG_IDX) = value;
}

__STATIC_INLINE uint32_t SysCtl_getCfgProgIdx(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_MBIST_CFG_PROG_IDX) & 0x03;
}

/*
 *  MBIST_CFG_MAX_VALUE   MBIST-CFG1 register      0x080C
 *
 *  Register Field         Field Range
 *  cfg_max_value             31 : 0		Maximum working time for mbist
 */
__STATIC_INLINE void SysCtl_setCfgMaxValue(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_MBIST_CFG_MAX_VALUE) = value;
}

__STATIC_INLINE uint32_t SysCtl_getCfgMaxValue(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_MBIST_CFG_MAX_VALUE) & 0xFFFFFFFF;
}


/*
 *  MBIST_CFG_MBIST_TRIG   MBIST-CFG2 register      0x0810
 *
 *  Register Field         Field Range
 *  cfg_mbist_trig             31 : 0		STC start signal (if it matches the preset value)
 */
__STATIC_INLINE void SysCtl_setCfgMbistTrig(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_MBIST_CFG_MBIST_TRIG) = value;
}

__STATIC_INLINE uint32_t SysCtl_getCfgMbistTrig(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_MBIST_CFG_MBIST_TRIG) & 0xFFFFFFFF;
}

/*
 *  MBIST_CFG_SC1   MBIST-CFG3 register      0x0814
 *
 *  Register Field         Field Range
 *  cfg_sc1		             31 : 0		SC1 unlock (matches preset value)
 */
__STATIC_INLINE void SysCtl_setCfgSc1(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_MBIST_CFG_SC1) = value;
}

__STATIC_INLINE uint32_t SysCtl_getCfgSc1(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_MBIST_CFG_SC1) & 0xFFFFFFFF;
}

/*
 *  MBIST_CFG_SC2   MBIST-CFG4 register      0x0818
 *
 *  Register Field         Field Range
 *  cfg_sc2		             31 : 0		SC2 unlock (register can only be configured after unlocking)
 */
__STATIC_INLINE void SysCtl_setCfgSc2(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_MBIST_CFG_SC2) = value;
}

__STATIC_INLINE uint32_t SysCtl_getCfgSc2(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_MBIST_CFG_SC2) & 0xFFFFFFFF;
}

/*
 *  MBIST_CFG_LOCK_MAX_VALUE   MBIST-CFG5 register      0x081C
 *
 *  Register Field         Field Range
 *  reserved				31 : 16
 *  cfg_lock_max_value		 15 : 0		Maximum time for configuring registers
 */
__STATIC_INLINE void SysCtl_setCfgLockMaxValue(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_MBIST_CFG_LOCK_MAX_VALUE) = value;
}

__STATIC_INLINE uint32_t SysCtl_getCfgLockMaxValue(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_MBIST_CFG_LOCK_MAX_VALUE) & 0xFFFF;
}

/*
 * MBIST_RPT		     MBIST_RPT register      0x0820
 *
 *  Register Field         Field Range
 *  state_data_out           31 : 0		MBIST result reporting
 */

__STATIC_INLINE uint32_t SysCtl_getStateDataOut(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_MBIST_RPT) & 0xFFFFFFFF;
}


/*
 * 	MBIST_ACT_RPT		MBIST-RPT register   0x824
 *
 *  Register Field         Field Range
 *  reserved				31 : 2
 *  stc_cfg_act_rpt			1		Configuration time status reporting (1: indicates that it is within the configuration time and can be configured as a register)
 *  stc_mbist_act_rpt		0		MBIST work status report (1: indicates ongoing MBIST)
 *
 */

__STATIC_INLINE uint32_t SysCtl_getStcMbistActRpt(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_MBIST_ACT_RPT) >> 0) & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_getStcCfgActRpt(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_MBIST_ACT_RPT) >> 1) & 0x01;
}


/*	STC_INTR_RAW	STC interrupt source register		0x828

 *  Register Field         Field Range
 *  reserved				31 : 1
 *  stc_intr_raw			0			interrupt source
 *
 */

__STATIC_INLINE uint32_t SysCtl_getStcIntrRaw(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_STC_INTR_RAW) & 0x01;
}

__STATIC_INLINE void SysCtl_clearStcIntrRaw(void)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_STC_INTR_RAW) = 0x00;
}

/*
 * 	STC_INTR_MASK	STC interrupt source masking		0x82C
 *
 *  Register Field         Field Range
 *  reserved				31 : 1
 *  stc_intr_mask			0			interrupt mask
 *
 */
__STATIC_INLINE void SysCtl_setStcIntrMask(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_STC_INTR_MASK) = value;
}

__STATIC_INLINE uint32_t SysCtl_getStcIntrMask(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_STC_INTR_MASK) & 0x01;
}

/*
 * 	STC_INTR			STC interrupt reporting register	0x830
 *
 *  Register Field         Field Range
 *  reserved				31 : 1
 *  stc_intr				0				Interrupt query register
 *
 */

__STATIC_INLINE uint32_t SysCtl_getStcIntr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_STC_INTR) & 0x01;
}

__STATIC_INLINE void SysCtl_clearStcIntr(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_STC_INTR) = value;
}

/*
 * DIG_SYSCTL_POR_SPARE0    Reserved Register       0xF00
 * 
 *  Register Field              Field Range
 *  dig_sysctrl_por_spare0      31 : 0
 */
__STATIC_INLINE void SysCtl_setDigSysctlPorSpare0(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE0) = value;
}

__STATIC_INLINE uint32_t SysCtl_getDigSysctlPorSpare0(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE0);
}


/*
 * DIG_SYSCTL_POR_SPARE1    Reserved Register       0xF04
 * 
 *  Register Field              Field Range
 *  dig_sysctrl_por_spare1      31 : 0
 */
__STATIC_INLINE void SysCtl_setDigSysctlPorSpare1(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE1) = value;
}

__STATIC_INLINE uint32_t SysCtl_getDigSysctlPorSpare1(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE1);
}

/**
 * @brief Checks the Power-On Reset (POR) flag status.
 *
 * @return bool Returns true if both bits 31 and 30 are set (POR flag is active), false otherwise.
 */
__STATIC_INLINE bool SysCtl_getPorFlag(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE1) & 0xC0000000) == 0xC0000000;
}

/**
 * @brief Clears the Power-On Reset (POR) flag.
 *
 * @param value Unused parameter (included for API consistency).
 */
__STATIC_INLINE void SysCtl_clearPorFlag(void)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE1) &= ~0xC0000000;
}


/*
 * DIG_SYSCTL_POR_SPARE2    Reserved Register       0xF00
 * 
 *  Register Field              Field Range
 *  dig_sysctrl_por_spare2      31 : 0
 */
__STATIC_INLINE void SysCtl_setDigSysctlPorSpare2(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE2) = value;
}

__STATIC_INLINE uint32_t SysCtl_getDigSysctlPorSpare2(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE2);
}


/*
 * DIG_SYSCTL_SYS_SPARE0    Reserved Register       0xF10
 * 
 *  Register Field          Field Range
 *  reserved                31 : 1
 *  dig_sysctrl_sys_spare0  0      System Spare 0 Configuration.
 */
__STATIC_INLINE void SysCtl_setDigSysctlSysSpare0(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_SYS_SPARE0) = value;
}

__STATIC_INLINE uint32_t SysCtl_getDigSysctlSysSpare0(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_SYS_SPARE0);
}

/*
 * DIG_SYSCTL_SYS_SPARE1    Reserved Register       0xF10
 *
 *  Register Field          Field Range
 *  reserved                31 : 1
 *  dig_sysctrl_sys_spare1  0
 */
__STATIC_INLINE void SysCtl_setDigSysctlSysSpare1(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_SYS_SPARE1) = value;
}

__STATIC_INLINE uint32_t SysCtl_getDigSysctlSysSpare1(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_SYS_SPARE1);
}

/*
 * DIG_SYSCTL_SYS_SPARE2    Reserved Register       0xF10
 *
 *  Register Field          Field Range
 *  reserved                31 : 1
 *  dig_sysctrl_sys_spare2  0
 */
__STATIC_INLINE void SysCtl_setDigSysctlSysSpare2(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_SYS_SPARE2) = value;
}

__STATIC_INLINE uint32_t SysCtl_getDigSysctlSysSpare2(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_SYS_SPARE2);
}


#ifdef __cplusplus
}
#endif

#endif

#endif /* DEVICE_DRIVERLIB_SYSCTL_DIG_H_ */
