/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    sysctl_dig_v22.h
*   @brief   digital system top level configuration
*            do NOT include this file directly, include "sysctl.h" instead
*
*/

#ifndef DEVICE_DRIVERLIB_SYSCTL_DIG_V22_H_
#define DEVICE_DRIVERLIB_SYSCTL_DIG_V22_H_

#include "gs32_version.h"

#if IS_GS32F3xx(0x22)

#ifdef __cplusplus
extern "C"{
#endif

#include "inc/hw_types.h"
#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include <inc/hw_sysctl_dig_v22.h>

/*
 * DIG_SYSCTRL_LOCK   Lock Control Register   Offset 0x0000
 *
 *  Register Field      Field Range
 *  dig_sysctrl_lock    31 : 0      Lock Control Value.
 * 
 * Writing unlock value 0x5A5A5A5A unlocks, allowing write access to all registers protected by lock.
 * Writing any other value locks all registers protected by lock.
 */
__STATIC_INLINE void SysCtl_setDigSysctrlLock(void)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_SYSCTRL_LOCK) = 0x01;
}

__STATIC_INLINE void SysCtl_setDigSysctrlUnlock(void)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_SYSCTRL_LOCK) = 0x5A5A5A5A;
}

__STATIC_INLINE uint32_t SysCtl_getDigSysctrlLock(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_SYSCTRL_LOCK);
}


/*
 * CCM_MEM_CTRL     CCM Parameter Configuration Register   0x200
 *
 *  Register Field    Field Range
 *  ccm_mem_ctrl      31 : 0      RAM Configuration.
 *
 */
__STATIC_INLINE void SysCtl_setCcmMemCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_CCM_MEM_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getCcmMemCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_CCM_MEM_CTRL);
}



/*
 * CCM_MEM_PWR_CTRL   CCM Memory Low Power Configuration Register   Offset 0x0204
 *
 *  Register Field          Field Range
 *  ccm_mem_pwr_ctrl       31 : 0
 *
 */
__STATIC_INLINE void SysCtl_setCcmMemPwrCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_CCM_MEM_PWR_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getCcmMemPwrCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_CCM_MEM_PWR_CTRL);
}


/*
 * FLASH_MEM_CTRL     Flash Memory Parameter Configuration Register   0x208
 *
 *  Register Field          Field Range
 *  flash_mem_ctrl          31 : 0      Flash Memory Configuration.
 *
 */
__STATIC_INLINE void SysCtl_setFlashMemCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_FLASH_MEM_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getFlashMemCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_FLASH_MEM_CTRL);
}

/*
 * FLASH_MEM_PWR_CTRL     Flash Memory Low Power Configuration Register   0x20C
 *
 *  Register Field           Field Range
 *  flash_mem_pwr_ctrl       31 : 0      Low Power Configuration for Flash Memory.
 *
 */
__STATIC_INLINE void SysCtl_setFlashMemPwrCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_FLASH_MEM_PWR_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getFlashMemPwrCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_FLASH_MEM_PWR_CTRL);
}

/*
 * GS_MEM_CTRL     GS Memory RAM Parameter Configuration Register   Offset 0x0210
 *
 *  Register Field      Field Range
 *  gs_mem_ctrl         31 : 0      GS Memory RAM Configuration.
 *
 */
__STATIC_INLINE void SysCtl_setGsMemCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_GS_MEM_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getGsMemCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_GS_MEM_CTRL);
}


/*
 * GS_MEM_PWR_CTRL   GS Memory RAM Low Power Configuration Register   Offset 0x0214
 *
 *  Register Field          Field Range
 *  gs_mem_pwr_ctrl         31 : 0      Low Power Configuration for GS Memory RAM.
 *
 */
__STATIC_INLINE void SysCtl_setGsMemPwrCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_GS_MEM_PWR_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getGsMemPwrCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_GS_MEM_PWR_CTRL);
}

/*
 * MSG_RAM_CTRL     RAM Parameter Configuration Register   Offset 0x0218
 *
 *  Register Field      Field Range
 *  msg_ram_mem_ctrl    31 : 0      RAM Configuration for Message RAM.
 *
 */
__STATIC_INLINE void SysCtl_setMsgMemCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_MSG_MEM_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getMsgMemCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_MSG_MEM_CTRL);
}


/*
 * MSG_RAM_PWR_CTRL    RAM Low Power Configuration Register   Offset 0x021C
 *
 *  Register Field          Field Range
 *  msg_ram_mem_pwr_ctrl    31 : 0      Low Power Configuration for Message RAM.
 *
 */
__STATIC_INLINE void SysCtl_setMsgMemPwrCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_MSG_MEM_PWR_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getMsgMemPwrCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_MSG_MEM_PWR_CTRL);
}

/*
 * DSP_CORE_MEM_CTRL    RAM Parameter Configuration Register     0x220
 *
 *  Register Field Name      Field Range
 *  dsp_mem_ctrl             31:0
 *
 */
__STATIC_INLINE void SysCtl_setDspCoreMemCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DSP_CORE_MEM_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getDspCoreMemCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DSP_CORE_MEM_CTRL);
}

/*
 * DSP_CORE_MEM_PWR_CTRL    RAM Low Power Configuration Register     0x224
 *
 *  Register Field Name      Field Range
 *  dsp_mem_pwr_ctrl         31:0
 *
 */
__STATIC_INLINE void SysCtl_setDspCoreMemPwrCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DSP_CORE_MEM_PWR_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getDspCoreMemPwrCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DSP_CORE_MEM_PWR_CTRL);
}

/*
 * DSP_SYS_MEM_CTRL    RAM Parameter Configuration Register     0x228
 *
 *  Register Field Name      Field Range
 *  dsp_mem_ctrl             31:0
 *
 */
__STATIC_INLINE void SysCtl_setDspSysMemCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DSP_SYS_MEM_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getDspSysMemCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DSP_SYS_MEM_CTRL);
}

/*
 * DSP_SYS_MEM_PWR_CTRL    RAM Low Power Configuration Register     0x22C
 *
 *  Register Field Name      Field Range
 *  dsp_mem_pwr_ctrl         31:0
 *
 */
__STATIC_INLINE void SysCtl_setDspSysMemPwrCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DSP_SYS_MEM_PWR_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getDspSysMemPwrCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DSP_SYS_MEM_PWR_CTRL);
}

/*
 * CCM_ECC_ERR_RPT    CCM Status Reporting Register     Offset 0x0300
 * 
 *  Register Field Name          Field Range
 *  reserved                     31:2
 *  ccm_ram_rd_err               1
 *  ccm_ram_real_bad_rd_err      0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getCcmRamRdErr(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_CCM_ECC_ERR_RPT) >> 1) & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_clearCcmRamRdErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_CCM_ECC_ERR_RPT) = 0x01 << 1;
}

__STATIC_INLINE uint32_t SysCtl_getCcmRamRealBadRdErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_CCM_ECC_ERR_RPT) & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_clearCcmRamRealBadRdErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_CCM_ECC_ERR_RPT) = 0x01;
}


/*
 * SRAM0_PARITY_RPT    SRAM0 Parity Error Reporting     0x304
 *
 *  Register Field Name        Field Range
 *  reserved                   31:16
 *  sram0_parity_err           15:0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getSram0ParityErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_SRAM0_PARITY_RPT) & 0xFFFF;
}

__STATIC_INLINE uint32_t SysCtl_clearSram0ParityErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_SRAM0_PARITY_RPT) = 0xFFFF;
}


/*
 * SRAM1_PARITY_RPT    SRAM1 Parity Error Reporting     Offset 0x308
 *
 *  Register Field Name        Field Range
 *  reserved                   31:16
 *  sram1_parity_err           15:0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getSram1ParityErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_SRAM1_PARITY_RPT) & 0xFFFF;
}

__STATIC_INLINE uint32_t SysCtl_clearSram1ParityErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_SRAM1_PARITY_RPT) = 0xFFFF;
}


/*
 *  FLEX_PARITY_RPT   Flex Parity Error Reporting     0x30C
 *
 *  Register Field Name       Field Range
 *  reserved                  31:16
 *  flex_sram_parity_err      15:0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getFlexSramParityErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_FLEX_PARITY_RPT) & 0x1;
}

__STATIC_INLINE uint32_t SysCtl_clearFlexSramParityErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_FLEX_PARITY_RPT) = 0x1;
}


/*
 * GS_PARITY_RPT    GS RAM Parity Error Reporting     0x310
 * 
 *  Register Field Name        Field Range
 *  reserved                   31:29
 *  gs_sram_parity_err         0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getGsRamParityErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_GS_PARITY_RPT) & 0x1;
}

__STATIC_INLINE uint32_t SysCtl_clearGsRamParityErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_GS_PARITY_RPT) = 0x1;
}


/*
 *  MSG0_PARITY_RPT     DSP LM Parity Error Reporting     0x314
 * 
 *  Register Field Name         Field Range
 *  reserved                     31:4
 *  msg0_ram_parity_err         3:0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getMsg0ParityErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_MSG0_PARITY_RPT) & 0xF;
}

__STATIC_INLINE uint32_t SysCtl_clearMsg0ParityErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_MSG0_PARITY_RPT) = 0xF;
}


/*
 *  MSG1_PARITY_RPT     DSP LM Parity Error Reporting     0x318
 *
 *  Register Field Name         Field Range
 *  reserved                     31:4
 *  msg1_ram_parity_err         3:0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getMsg1ParityErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_MSG1_PARITY_RPT) & 0xF;
}

__STATIC_INLINE uint32_t SysCtl_clearMsg1ParityErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_MSG1_PARITY_RPT) = 0xF;
}


/*
 * FLASH_ICACHE_ECC_ERR_RPT   ECC Error Reporting     Offset 0x031C
 *
 *  Register Field Name              Field Range
 *  reserved                          31:2
 *  flash_icache_ecc_rd_err          1
 *  flash_icache_ecc_real_bad_err    0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getFlashIcacheRdErr(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_FLASH_ICACHE_ECC_ERR_RPT) >> 1) & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_clearFlashIcacheRdErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_FLASH_ICACHE_ECC_ERR_RPT) = 0x01 << 1;
}

__STATIC_INLINE uint32_t SysCtl_getFlashIcacheRealBadRdErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_FLASH_ICACHE_ECC_ERR_RPT) & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_clearFlashIcacheRealBadRdErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_FLASH_ICACHE_ECC_ERR_RPT) = 0x01;
}


/*
 * PVM_INT_RPT   PVM Reporting     Offset 0x0320
 *
 *  Register Field Name        Field Range
 *  reserved                   31:1
 *  pvm_rpt                    0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getPvmRpt(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_PVM_INT_RPT) & 0x1;
}

__STATIC_INLINE uint32_t SysCtl_clearPvmRpt(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_PVM_INT_RPT) = 0x1;
}


/*
 * SMP_MODE   Sampling Mode Configuration Register   Offset 0x400
 *
 *  Register Field Name       Field Range
 *  reserved                  31:1
 *  gs_ram_smp_mode           0
 *
 *  0: Functional mode
 *  1: Sampling mode,
 *  where the CPU can only perform read accesses
 *  (initiating write accesses are automatically masked and will not take effect),
 *  and write accesses are handled by the sampling mechanism.
 */
__STATIC_INLINE void SysCtl_setGsramSmpMode(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_GS_SRAM_SMP_MODE) = value & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_getGsramSmpMode(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_GS_SRAM_SMP_MODE) & 0x01;
}

/*
 * BOOT_PIN_LATCH_MSK   Boot Pin Latch Mask Whether Affected by System Reset     Offset: 0x500
 *
 *  Register Field Name       Field Range
 *  reserved                  31:1
 *  boot_pin_latch_msk        0
 *
 *  0: Boot pin is re-latched after system reset release.
 *  1: Boot pin is not re-latched after system reset release; it is only latched during power-up process.
 *
 */
__STATIC_INLINE void SysCtl_setBootPinLatchMask(uint32_t mask)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_BOOT_PIN_LATCH_MSK) = mask & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_getBootPinLatchMask(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_BOOT_PIN_LATCH_MSK) & 0x01;
}


/*
 * BOOT_PIN_LATCH_RPT   Boot Pin Latch Value     Offset: 0x504
 *
 *  Register Field Name         Field Range
 *  reserved                    31:2
 *  boot_pin_lock_data_rpt      1:0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getBootPinLatchRpt(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_BOOT_PIN_LATCH_RPT) & 0x03;
}

/*
 * DSP_CPU2_RESET_VECTOR   DSP Initial Vector Report   Offset: 0x050C
 *
 *  Register Field Name                  Field Range
 *  cfg_dsp_cpu2_reset_vector           31:0
 *
 */
__STATIC_INLINE void SysCtl_setCpu2ResetVector(uint32_t vector)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DSP_CPU2_RESET_VECTOR) = vector;
}

__STATIC_INLINE uint32_t SysCtl_getCpu2ResetVector(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DSP_CPU2_RESET_VECTOR);
}

__STATIC_INLINE void SysCtl_setCpu2DeadLoop(uint32_t addr)
{
    HWREG(addr) = (0x0000A001U);
}

/*
 * DSP_CPU1_RESET_VECTOR_RPT   CPU1 Initial Vector Report      Offset: 0x514
 *
 *  Register Field Name                  Field Range
 *  dsp_cpu1_reset_vector_rpt           31:0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getDspCpu1ResetVectorRpt(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DSP_CPU1_RESET_VECTOR_RPT);
}

/*
 * DSP_CPU2_RESET_VECTOR_RPT   CPU2 Initial Vector Report      Offset: 0x518
 *
 *  Register Field Name                  Field Range
 *  dsp_cpu2_reset_vector_rpt           31:0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getDspCpu2ResetVectorRpt(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DSP_CPU2_RESET_VECTOR_RPT);
}

/*
 * DSP_STAT_REG DSP Core Status Report Register      Offset: 0x608
 *
 *  Register Field Name                  Field Range
 *  reserved                             31:4
 *  dsp_cpu2_wfi_mode                   3
 *  dsp_cpu2_sleep_value                2
 *  dsp_cpu1_wfi_mode                   1
 *  dsp_cpu1_sleep_value                0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getDspCpu1SleepValue(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DSP_STAT_REG) & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_getCpu1WfiMode(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_DSP_STAT_REG) >> 1) & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_getDspCpu2SleepValue(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_DSP_STAT_REG) >> 2) & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_getCpu2WfiMode(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_DSP_STAT_REG) >> 3) & 0x01;
}

/*
 * DSP_HALT_RPT         DSP Halt Report Register      Offset: 0x610
 *
 *  Register Field Name                  Field Range
 *  reserved                             31:2
 *  dsp_cpu2_hart_halted                1
 *  dsp_cpu1_hart_halted                0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getCpu1HaltedSta(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DSP_HALT_RPT) & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_getCpu2HaltedSta(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_DSP_HALT_RPT) >> 1) & 0x01;
}

/*
 *  SLV_CORE_JTAG_CFG    Slave Core JTAG Configuration Register      Offset: 0x618
 *
 *  Register Field Name                  Field Range
 *  reserved                             31:2
 *  slv_core_jtag_2wire_reuse           1      Enables reuse of 2-wire mode for the slave core,
 *                                             0: Disabled, requires separate JTAG lines for slave core (GPIO MUX)
 *                                             1: Enabled, reuses the TDI/TDO interface of the master core's JTAG as the 2-wire debug interface for the slave core.
 *  slv_core_jtag_discrete_en           0      Enables independent debug for the slave core,
 *                                             0: Disabled, debug for slave core goes through the daisy chain
 *                                             1: Enabled, supports independent 2-wire, reuse 2-wire, or independent 4-wire debug for the slave core.
 * Note: For independent 2-wire or independent 4-wire debug, GPIO MUX also needs to be configured to route functionality to the slave core's JTAG.
 */
__STATIC_INLINE void SysCtl_setSlvCoreJtagDiscreteEn(uint32_t enable)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_SLV_CORE_JTAG_CFG) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_SLV_CORE_JTAG_CFG) & ~(1<<0)) | (enable & 0x01);
}

__STATIC_INLINE uint32_t SysCtl_getSlvCoreJtagDiscreteEn(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_SLV_CORE_JTAG_CFG) & 0x01;
}

__STATIC_INLINE void SysCtl_setSlvCoreJtag2WireReuse(uint32_t enable)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_SLV_CORE_JTAG_CFG) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_SLV_CORE_JTAG_CFG) & ~(1<<1)) | ((enable & 0x01) << 1);
}

__STATIC_INLINE uint32_t SysCtl_getSlvCoreJtag2WireReuse(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_SLV_CORE_JTAG_CFG) >> 1) & 0x01;
}


/*
 *  EFC_BIST_ENABLE         Embedded Flash Controller BIST Enable Register     Offset: 0x620
 *
 *  Register Field Name            Field Range
 *  reserved                       31:3
 *  bist_enable                    2      BIST Enable
 *  eflash_jtag_macro_sel          1      0: Macro 0;   1: Macro 1
 *  eflash_jtag_reuse_en           0      0: Use dedicated JTAG pins for FLASH
 *                                       1: Reuse chip JTAG pins (disconnects JTAG functionality)
 */
__STATIC_INLINE void SysCtl_setEflashJtagReuseEn(uint32_t enable)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_EFC_BIST_ENABLE) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_EFC_BIST_ENABLE) & ~(1<<0)) | (enable & 0x01);
}

__STATIC_INLINE uint32_t SysCtl_getEflashJtagReuseEn(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_EFC_BIST_ENABLE) & 0x01;
}

__STATIC_INLINE void SysCtl_setEflashJtagMacroSel(uint32_t enable)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_EFC_BIST_ENABLE) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_EFC_BIST_ENABLE) & ~(1<<1)) | ((enable & 0x01) << 1);
}

__STATIC_INLINE uint32_t SysCtl_getEflashJtagMacroSel(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_EFC_BIST_ENABLE) >> 1) & 0x01;
}

__STATIC_INLINE void SysCtl_setEfcBistEnable(uint32_t enable)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_EFC_BIST_ENABLE) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_EFC_BIST_ENABLE) & ~(1<<2)) | (enable & 0x01);
}

__STATIC_INLINE uint32_t SysCtl_getEfcBistEnable(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_EFC_BIST_ENABLE) >> 2) & 0x01;
}

/*
 *  BUS_AUSER_ID_CFG        BUS_USER_ID Configuration Register     Offset: 0x628
 *
 *  Register Field Name           Field Range
 *  reserved                      31:11
 *  cfg_cpu2ppi_hauser           10:8
 *  reserved                      7
 *  cfg_cpu1ppi_hauser           6:4
 *  reserved                      3:0
 */

typedef enum {
    DSP_CPU1_PPI_HAUSER = 4,
    DSP_CPU2_PPI_HAUSER = 8,
}BusHauserType_t;

__STATIC_INLINE void SysCtl_setBusAuserIdCfg(BusHauserType_t bus, uint32_t id)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_BUS_AUSER_ID_CFG) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_BUS_AUSER_ID_CFG) & ~(7<<bus)) | ((id & 0x07) << bus);
}

__STATIC_INLINE uint32_t SysCtl_getBusAuserIdCfg(BusHauserType_t bus)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_BUS_AUSER_ID_CFG) >> bus) & 0x07;
}

/*
 *  DSP_SYS_ERR_MASK        System Error Interrupt Selection Register     Offset: 0x650
 *
 *  Register Field Name           Field Range
 *  reserved                      31:16
 *  dsp_sys_err_mask              15:0
 */
__STATIC_INLINE void SysCtl_setDspSysErrMask(uint32_t mask)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DSP_SYS_ERR_MASK) = mask & 0xFFFF;
}

__STATIC_INLINE uint32_t SysCtl_getDspSysErrMask(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DSP_SYS_ERR_MASK) & 0xFFFF;
}


/*
 *  PWM_ECCERR_MASK            PWM ECC Error Mask Register        Offset: 0x658
 *
 *  Register Field Name            Field Range
 *  reserved                       31:16
 *  cfg_pwm_ecc_err_mask           15:0
 */
__STATIC_INLINE void SysCtl_setPwmEccErrMask(uint32_t mask)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_PWM_ECCERR_MASK) = mask & 0xFFFF;
}

__STATIC_INLINE uint32_t SysCtl_getPwmEccErrMask(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_PWM_ECCERR_MASK) & 0xFFFF;
}


/*
 *  PWM_SYSFAULT_MASK            PWM System Fault Mask Register      Offset: 0x65C
 *
 *  Register Field Name           Field Range
 *  reserved                      31:16
 *  cfg_pwm_sys_err_mask          15:0
 */
__STATIC_INLINE void SysCtl_setPwmSysErrMask(uint32_t mask)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_PWM_SYSFAULT_MASK) = mask & 0xFFFF;
}

__STATIC_INLINE uint32_t SysCtl_getPwmSysErrMask(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_PWM_SYSFAULT_MASK) & 0xFFFF;
}


/*
 *  ERRORTS_MASK                Error Timestamp Mask Register       Offset: 0x660
 *
 *  Register Field Name             Field Range
 *  reserved                        31:3
 *  cpu2_nmi_to_errorts_msk         2
 *  cpu1_nmi_to_errorts_msk         1
 */
__STATIC_INLINE void SysCtl_setCpu1NmiToErrorTsMsk(uint32_t mask)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_ERRORTS_MASK) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_ERRORTS_MASK) & ~(1<<1)) | ((mask & 0x01) << 1);
}

__STATIC_INLINE uint32_t SysCtl_getCpu1NmiToErrorTsMsk(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_ERRORTS_MASK) >> 1) & 0x01;
}

__STATIC_INLINE void SysCtl_setCpu2NmiToErrorTsMsk(uint32_t mask)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_ERRORTS_MASK) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_ERRORTS_MASK) & ~(1<<2)) | (mask & 0x01);
}

__STATIC_INLINE uint32_t SysCtl_getCpu2NmiToErrorTsMsk(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_ERRORTS_MASK) >> 2) & 0x01;
}


/*
 *  AXI_CLUSTER_QOS_CFG          AXI Cluster Quality of Service Configuration Register      Offset: 0x664
 *
 *  Register Field Name           Field Range
 *  reserved                      31:12
 *  dsp_cluster_mem_awqos         11:8
 *  reserved                      7:4
 *  dsp_cluster_mem_arqos         3:0
 */
__STATIC_INLINE void SysCtl_setDspClusterMemArqos(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_AXI_CLUSTER_QOS_CFG) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_AXI_CLUSTER_QOS_CFG) & ~(0x0F<<0)) | (value & 0x0F);
}

__STATIC_INLINE uint32_t SysCtl_getDspClusterMemArqos(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_AXI_CLUSTER_QOS_CFG) & 0x0F;
}

__STATIC_INLINE void SysCtl_setDspClusterMemAwqos(uint32_t mask)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_AXI_CLUSTER_QOS_CFG) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_AXI_CLUSTER_QOS_CFG) & ~(0x0F<<8)) | ((mask & 0x0F) << 8);
}

__STATIC_INLINE uint32_t SysCtl_getDspClusterMemAwqos(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_AXI_CLUSTER_QOS_CFG) >> 8) & 0x0F;
}


/*
 *  AXI_DMA0_QOS_CFG              AXI DMA0 Quality of Service Configuration Register     Offset: 0x668
 *
 *  Register Field Name           Field Range
 *  reserved                      31:12
 *  dsp_dma0_nic_awqos            11:8
 *  reserved                      7:4
 *  dsp_dma0_nic_arqos            3:0
 */
__STATIC_INLINE void SysCtl_setDspDspDma0Arqos(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_AXI_DMA0_QOS_CFG) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_AXI_DMA0_QOS_CFG) & ~(0x0F<<0)) | (value & 0x0F);
}

__STATIC_INLINE uint32_t SysCtl_getDspDspDma0Arqos(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_AXI_DMA0_QOS_CFG) & 0x0F;
}

__STATIC_INLINE void SysCtl_setDspDspDma0Awqos(uint32_t mask)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_AXI_DMA0_QOS_CFG) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_AXI_DMA0_QOS_CFG) & ~(0x0F<<8)) | ((mask & 0x0F) << 8);
}

__STATIC_INLINE uint32_t SysCtl_getDspDspDma0Awqos(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_AXI_DMA0_QOS_CFG) >> 8) & 0x0F;
}


/*
 *  AXI_DMA1_QOS_CFG              AXI DMA1 Quality of Service Configuration Register     Offset: 0x66C
 *
 *  Register Field Name           Field Range
 *  reserved                      31:12
 *  dsp_dma1_nic_awqos            11:8
 *  reserved                      7:4
 *  dsp_dma1_nic_arqos            3:0
 */
__STATIC_INLINE void SysCtl_setDspDspDma1Arqos(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_AXI_DMA1_QOS_CFG) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_AXI_DMA1_QOS_CFG) & ~(0x0F<<0)) | (value & 0x0F);
}

__STATIC_INLINE uint32_t SysCtl_getDspDspDma1Arqos(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_AXI_DMA1_QOS_CFG) & 0x0F;
}

__STATIC_INLINE void SysCtl_setDspDspDma1Awqos(uint32_t mask)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_AXI_DMA1_QOS_CFG) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_AXI_DMA1_QOS_CFG) & ~(0x0F<<8)) | ((mask & 0x0F) << 8);
}

__STATIC_INLINE uint32_t SysCtl_getDspDspDma1Awqos(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_AXI_DMA1_QOS_CFG) >> 8) & 0x0F;
}


/*
 *  AXI_SRAM_CACHEABLE            AXI SRAM Cacheable Configuration Register       Offset: 0x670
 *
 *  Register Field Name           Field Range
 *  reserved                      31:3
 *  cfg_flex_sram_cacheable       2
 *  cfg_axi_ram1_cacheable        1
 *  cfg_axi_ram0_cacheable        0
 */
__STATIC_INLINE void SysCtl_setAxiRam0Cacheable(uint32_t cacheable)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_AXI_SRAM_CACHEABLE) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_AXI_SRAM_CACHEABLE) & ~(0x01<<0)) | (cacheable & 0x01);
}

__STATIC_INLINE uint32_t SysCtl_getAxiRam0Cacheable(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_AXI_SRAM_CACHEABLE) & 0x01;
}

__STATIC_INLINE void SysCtl_setAxiRam1Cacheable(uint32_t cacheable)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_AXI_SRAM_CACHEABLE) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_AXI_SRAM_CACHEABLE) & ~(0x01<<1)) | ((cacheable & 0x01) << 1);
}

__STATIC_INLINE uint32_t SysCtl_getAxiRam1Cacheable(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_AXI_SRAM_CACHEABLE) >> 1) & 0x01;
}

__STATIC_INLINE void SysCtl_setFlexSramCacheable(uint32_t cacheable)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_AXI_SRAM_CACHEABLE) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_AXI_SRAM_CACHEABLE) & ~(0x01<<2)) | ((cacheable & 0x01) << 2);
}

__STATIC_INLINE uint32_t SysCtl_getFlexSramCacheable(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_AXI_SRAM_CACHEABLE) >> 2) & 0x01;
}


/*
 *  MEM_INI_REQ              Memory Initialization Request Configuration Register     Offset: 0x700
 *
 *  Register Field Name      Field Range
 *  reserved                 31:9
 *  ilm_mem_ini_req          7
 *  dlm_mem_ini_req          6
 *  gs_mem_ini_req           5
 *  msg1_mem_ini_req         4
 *  msg0_mem_ini_req         3
 *  ccm_mem_ini_req          2
 *  sram1_mem_ini_req        1
 *  sram0_mem_ini_req        0
 */
typedef enum {
    MEM_SRAM0       = 1<<0,
    MEM_SRAM1       = 1<<1,
    MEM_CCM         = 1<<2,
    MEM_MSG_RAM0    = 1<<3,
    MEM_MSG_RAM1    = 1<<4,
    MEM_GS          = 1<<5,
    MEM_DLM         = 1<<6,
    MEM_ILM         = 1<<7,
}MemIniType_t;
__STATIC_INLINE void SysCtl_setMemIniReq(MemIniType_t memTypes)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_MEM_INI_REQ) = memTypes;
}


/*
 *  MEM_INI_DONE_RPT          Memory Initialization Done Report Register     Offset: 0x704
 *
 *  Register Field Name      Field Range
 *  reserved                 31:9
 *  ilm_mem_ini_done_rpt      7
 *  dlm_mem_ini_done_rpt      6
 *  gs_mem_ini_done_rpt       5
 *  msg1_mem_ini_done_rpt     4
 *  msg0_mem_ini_done_rpt     3
 *  ccm_mem_ini_done_rpt      2
 *  sram1_mem_ini_done_rpt    1
 *  sram0_mem_ini_done_rpt    0
 */
__STATIC_INLINE uint32_t SysCtl_getMemIniDoneRpt(MemIniType_t memTypes)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_MEM_INI_DONE_RPT) & memTypes;
}

__STATIC_INLINE void SysCtl_clearMemIniDoneRpt(MemIniType_t memTypes)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_MEM_INI_DONE_RPT) = memTypes;
}


/*
 *  PMBUS_CTL_OEN            PMBus Control Output Enable Register     Offset: 0x708
 *
 *  Register Field Name      Field Range
 *  reserved                 31:1
 *  pmbus_ctl_oen            0
 */
__STATIC_INLINE void SysCtl_setPmbusCtlOen(uint32_t val)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_PMBUS_CTL_OEN) = val;
}

__STATIC_INLINE uint32_t SysCtl_getPmbusCtlOen(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_PMBUS_CTL_OEN) & 0x01;
}


/*
 *  SDFM_TYPE_CFG          Synchronous Data Flow Manager (SDFM) Type Configuration Register     Offset: 0x70C
 *
 *  Register Field Name    Field Range
 *  reserved               31:2
 *  sdfm_type_cfg          1:0
 */
__STATIC_INLINE void SysCtl_setSdfmTypeCfg(uint32_t type)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_SDFM_TYPE_CFG) = type;
}

__STATIC_INLINE uint32_t SysCtl_getSdfmTypeCfg(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_SDFM_TYPE_CFG) & 0x03;
}


/*
 *  TSENSOR_RPT            Temperature Sensor Report Register     Offset: 0x718
 *
 *  Register Field Name    Field Range
 *  reserved               31:12
 *  tsensor_data           11:0
 */
__STATIC_INLINE uint32_t SysCtl_getTsensorData(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_TSENSOR_RPT) & 0x0FF;
}


/*
 *  FSMC_CFG              Flexible Static Memory Controller (FSMC) Configuration Register     Offset: 0x71C
 *
 *  Register Field Name   Field Range
 *  reserved              31:8
 *  sysctrl_fsmc_reg      7:0
 */
__STATIC_INLINE void SysCtl_setFsmcCfg(uint32_t cfg)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_FSMC_CFG) = cfg & 0x0FF;
}

__STATIC_INLINE uint32_t SysCtl_getFsmcCfg(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_FSMC_CFG) & 0x0FF;
}


/*
 * DIG_SYSCTL_POR_SPARE0    Digital System Control Power-On Reset Spare Register 0    Offset: 0xF00
 * 
 *  Register Field Name        Field Range
 *  dig_sysctrl_por_spare0     31:0
 */
__STATIC_INLINE void SysCtl_setDigSysctlPorSpare0(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE0) = value;
}

__STATIC_INLINE uint32_t SysCtl_getDigSysctlPorSpare0(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE0);
}


/*
 * DIG_SYSCTL_POR_SPARE1    Digital System Control Power-On Reset Spare Register 1    Offset: 0xF04
 * 
 *  Register Field Name        Field Range
 *  dig_sysctrl_por_spare1     31:0
 */
__STATIC_INLINE void SysCtl_setDigSysctlPorSpare1(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE1) = value;
}

__STATIC_INLINE uint32_t SysCtl_getDigSysctlPorSpare1(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE1);
}

/**
 * @brief Checks the Power-On Reset (POR) flag status.
 *
 * @return bool Returns true if both bits 31 and 30 are set (POR flag is active), false otherwise.
 */
__STATIC_INLINE bool SysCtl_getPorFlag(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE1) & 0xC0000000) == 0xC0000000;
}

/**
 * @brief Clears the Power-On Reset (POR) flag.
 *
 * @param value Unused parameter (included for API consistency).
 */
__STATIC_INLINE void SysCtl_clearPorFlag(void)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE1) &= ~0xC0000000;
}

/*
 * DIG_SYSCTL_POR_SPARE2    Digital System Control Power-On Reset Spare Register 2    Offset: 0xF08
 * 
 *  Register Field Name        Field Range
 *  dig_sysctrl_por_spare2     31:0
 */
__STATIC_INLINE void SysCtl_setDigSysctlPorSpare2(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE2) = value;
}

__STATIC_INLINE uint32_t SysCtl_getDigSysctlPorSpare2(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE2);
}


/*
 * DIG_SYSCTL_SYS_SPARE0    Digital System Control System Spare Register 0    Offset: 0xF10
 * 
 *  Register Field Name        Field Range
 *  reserved                   31:1
 *  dig_sysctrl_sys_spare0     0
 */
__STATIC_INLINE void SysCtl_setDigSysctlSysSpare0(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_SYS_SPARE0) = value;
}

__STATIC_INLINE uint32_t SysCtl_getDigSysctlSysSpare0(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_SYS_SPARE0);
}


/*
 * DIG_SYSCTL_SYS_SPARE1    Digital System Control System Spare Register 1    Offset: 0xF14
 *
 *  Register Field Name        Field Range
 *  reserved                   31:1
 *  dig_sysctrl_sys_spare1     0
 */
__STATIC_INLINE void SysCtl_setDigSysctlSysSpare1(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_SYS_SPARE1) = value;
}

__STATIC_INLINE uint32_t SysCtl_getDigSysctlSysSpare1(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_SYS_SPARE1);
}

/*
 * DIG_SYSCTL_SYS_SPARE2    Digital System Control System Spare Register 2    Offset: 0xF18
 *
 *  Register Field Name        Field Range
 *  reserved                   31:1
 *  dig_sysctrl_sys_spare2     0
 */
__STATIC_INLINE void SysCtl_setDigSysctlSysSpare2(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_SYS_SPARE2) = value;
}

__STATIC_INLINE uint32_t SysCtl_getDigSysctlSysSpare2(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_SYS_SPARE2);
}


#ifdef __cplusplus
}
#endif

#endif

#endif /* DEVICE_DRIVERLIB_SYSCTL_DIG_H_ */
