/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    sysctl_dig_v30.h
*   @brief   digital system top level configuration
*            do NOT include this file directly, include "sysctl.h" instead
*
*/

#ifndef DEVICE_DRIVERLIB_SYSCTL_DIG_V30_H_
#define DEVICE_DRIVERLIB_SYSCTL_DIG_V30_H_

#include "gs32_version.h"

#if IS_GS32F00xx(0x30)

#ifdef __cplusplus
extern "C"{
#endif

#include "inc/hw_types.h"
#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include <inc/hw_sysctl_dig_v30.h>

/*
 * DIG_SYSCTRL_LOCK   Digital System Control Lock Control Register   Offset: 0x0000
 *
 *  Register Field Name        Field Range
 *  dig_sysctrl_lock           31:0
 * 
 * Writing the unlock value 0x5A5A5A5A unlocks the protected registers,
 * allowing write access to all registers protected by lock.
 * Writing any other value locks the protected registers.
 */
__STATIC_INLINE void SysCtl_setDigSysctrlLock(void)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_SYSCTRL_LOCK) = 0x01;
}

__STATIC_INLINE void SysCtl_setDigSysctrlUnlock(void)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_SYSCTRL_LOCK) = 0x5A5A5A5A;
}

__STATIC_INLINE uint32_t SysCtl_getDigSysctrlLock(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_SYSCTRL_LOCK);
}


/*
 * FLASH_MEM_CTRL    RAM parameter configuration register.    Offset: 0x200
 *
 * Register Field Name        Field Range
 * flash_mem_ctrl             31:0
 */
__STATIC_INLINE void SysCtl_setFlashMemCtrll(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_FLASH_MEM_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getFlashMemCtrll(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_FLASH_MEM_CTRL);
}



/*
 * FLASH_MEM_PWR_CTRL    RAM low-power configuration register.   Offset:  0x204
 *
 * Register Field Name        Field Range
 * flash_mem_pwr_ctrl    		31:0
 */
__STATIC_INLINE void SysCtl_setFlashMemPwrCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_FLASH_MEM_PWR_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getFlashMemPwrCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_FLASH_MEM_PWR_CTRL);
}


/*
 * GS_MEM_CTRL    RAM parameter configuration register.    Offset: 0x208
 *
 * Register Field Name        Field Range
 * gs_mem_ctrl    				31:0
 */
__STATIC_INLINE void SysCtl_setGsMemCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_GS_MEM_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getGsMemCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_GS_MEM_CTRL);
}

/*
 * GS_MEM_PWR_CTRL    RAM low-power configuration register.    Offset: 0x20C
 *
 * Register Field Name        Field Range
 * gs_mem_pwr_ctrl    				31:0
 */
__STATIC_INLINE void SysCtl_setGsMemPwrCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_GS_MEM_PWR_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getGsMemPwrCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_GS_MEM_PWR_CTRL);
}

/*
 * SWAP_MEM_CTRL    RAM parameter configuration register.     Offset: 0x210
 *
 * Register Field Name        Field Range
 * swap_mem_ctrl    			31:0
 */
__STATIC_INLINE void SysCtl_setSwapMemCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_SWAP_MEM_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getSwapMemCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_SWAP_MEM_CTRL);
}


/*
 * SWAP_MEM_PWR_CTRL    RAM low-power configuration register.    Offset: 0x214
 *
 * Register Field Name        Field Range
 * swap_mem_pwr_ctrl    		31:0
 */
__STATIC_INLINE void SysCtl_setSwapMemPwrCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_SWAP_MEM_PWR_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getSwapMemPwrCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_SWAP_MEM_PWR_CTRL);
}

/*
 * DSP_MEM_CTRL    RAM parameter configuration register.    Offset: 0x218
 *
 * Register Field Name        Field Range
 * dsp_mem_ctrl    				31:0
 */
__STATIC_INLINE void SysCtl_setDspMemCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DSP_MEM_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getDspMemCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DSP_MEM_CTRL);
}


/*
 * DSP_MEM_PWR_CTRL    RAM low-power configuration register.  Offset:  0x21C
 *
 * Register Field Name        Field Range
 * dsp_mem_pwr_ctrl    31:0
 */
__STATIC_INLINE void SysCtl_setDspMemPwrCtrl(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DSP_MEM_PWR_CTRL) = value;
}

__STATIC_INLINE uint32_t SysCtl_getDspMemPwrCtrl(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DSP_MEM_PWR_CTRL);
}


/*
 * GS_PARITY_RPT    GS parity error reporting.   Offset:  0x300
 *
 * Register Field Name        Field Range
 * reserved                  31:4
 * gs_ram_parity_err         3:0
 */

__STATIC_INLINE uint32_t SysCtl_getGsRamParityErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_GS_PARITY_RPT) & 0x0F;
}

__STATIC_INLINE void SysCtl_clearGsRamParityErr(void)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_GS_PARITY_RPT) = 0x0F;
}


/*
 * SWAP0_PARITY_RPT			SRAM0 parity error reporting.	 Offset:0x304
 *
 *  Register Field Name        Field Range
 *  reserved					31 :4
 *  swap0_ram_parity_err		3:0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getSwap0RamParityErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_SWAP0_PARITY_RPT) & 0x0F;
}

__STATIC_INLINE void SysCtl_clearSwap0RamParityErr(void)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_SWAP0_PARITY_RPT) = 0x0F;
}


/*
 * SWAP1_PARITY_RPT    SRAM1 odd parity error report     Offset: 0x308
 *
 *  Register Field Name        Field Range
 *  reserved                   31 : 4
 *  swap1_ram_parity_err       3 : 0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getSwap1RamParityErr(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_SWAP1_PARITY_RPT) & 0x0F;
}

__STATIC_INLINE void SysCtl_clearSwap1RamParityErr(void)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_SWAP1_PARITY_RPT) = 0x0F;
}


/*
 * PVM_INT_RPT    PVM report    Offset: 0x318
 *
 *  Register Field Name                Field Range
 *  reserved                           31 : 1
 *  pvm_rpt                            0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getPvmReport(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_PVM_INT_RPT) & 0x01;
}

__STATIC_INLINE void SysCtl_clearPvmReport(void)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_PVM_INT_RPT) = 0x01;
}


/*
 * GS_SRAM_SMP_MODE   GS RAM Sampling Mode Configuration Register.     Offset: 0x400
 *
 *  Register Field Name        Field Range
 *  reserved                  31 : 1
 *  gs_ram_smp_mode           0
 *  0: Functional mode
 *  1: Sampling mode, where CPU can only perform read accesses (write accesses initiated by CPU will be masked by default and will not take effect), write accesses are handled by sampling.
 *
 */
__STATIC_INLINE void SysCtl_setGsRamSmpMode(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_GS_SRAM_SMP_MODE) = value;
}

__STATIC_INLINE uint32_t SysCtl_getGsRamSmpMode(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_GS_SRAM_SMP_MODE) & 0x01;
}


/*
 * BOOT_PIN_LATCH_MSK	Boot pin latch affected by system reset		Offset: 0x500
 *
 *  Register Field Name        Field Range
 *  reserved	31 :1
 *  boot_pin_latch_msk	0
 *  0: System deassertion resets and relocks the boot pin.
 *  1: System deassertion does not relock the boot pin; it only locks during power-up."
 *
 *
 */
__STATIC_INLINE void SysCtl_setBootPinLatchMask(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_BOOT_PIN_LATCH_MSK) = value;
}

__STATIC_INLINE uint32_t SysCtl_getBootPinLatchMask(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_BOOT_PIN_LATCH_MSK) & 0x01;
}


/*
 * BOOT_PIN_LATCH_RPT    BOOT PIN latch value       Offset: 0x504
 *
 *  Register Field Name        Field Range
 *  reserved    				31 :2
 *  boot_pin_lock_data_rpt    	1:0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getBootPinLatchRpt(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_BOOT_PIN_LATCH_RPT) & 0x03;
}


/*
 * DSP_RESET_VECTOR_RPT    DSP initial vector report    Offset: 0x510
 *
 *  Register Field Name        Field Range
 *  dsp_reset_vector_rpt       31:0
 *
 */
__STATIC_INLINE uint32_t SysCtl_getDspResetVectorRpt(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DSP_RESET_VECTOR_RPT);
}


/*
 * DSP_STAT_REG    DSP core status report register    Offset: 0x608
 *
 *  Register Field Name        Field Range
 *  reserved                   31:2
 *  dsp_core_wfi_mode          1        DSP core status
 *  dsp_core_sleep_value       0        DSP core sleep status
 *
 */
__STATIC_INLINE uint32_t SysCtl_getDspCoreWfiMode(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_DSP_STAT_REG) >> 1) & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_getDspCoreSleepValue(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DSP_STAT_REG) & 0x01;
}


/*
 * DSP_HALT_RPT    DSP core halt report register    Offset: 0x610
 *
 *  Register Field Name    Field Range
 *  reserved               31:1
 *  dsp_hart_halted        0        DSP core halt report
 *
 *
 */
__STATIC_INLINE uint32_t SysCtl_getDspHartHalted(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DSP_HALT_RPT) & 0x1;
}


/*
 * DBG_FRC_OFF    Debug Software Control Switch    Offset: 0x614
 *
 *  Register Field Name    Field Range
 *  reserved               31:1
 *  debug_frc_off          0        Debug Close Configuration
 *
 *
 */
__STATIC_INLINE void SysCtl_setDebugFrcOff(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DBG_FRC_OFF) = value & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_getDebugFrcOff(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DBG_FRC_OFF) & 0x01;
}


/*
 * DSP_CORE_JTAG_CFG    SLV CORE JTAG configuration register.    Offset: 0x618
 *
 *  Register Field Name    Field Range
 * reserved                31 :2
 * dsp_jtag_2wire_reuse   1
 * 						slv_core_jtag_2wire_reuse: Enables the reuse of 2-wire mode for slave core JTAG.
 * 						0: Disable, meaning independent JTAG connection for slave core (GPIO MUX).
 * 						1: Enable, reusing the main core's JTAG TDI/TDO interface as the 2-wire debug interface for the slave core.
 * reserved                0
 */
__STATIC_INLINE void SysCtl_setDspJtag2WireReuse(uint32_t enable)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DSP_CORE_JTAG_CFG) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_DSP_CORE_JTAG_CFG) & ~(1 << 1)) | ((enable & 0x01) << 1);
}

__STATIC_INLINE uint32_t SysCtl_getDspJtag2WireReuse(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_DSP_CORE_JTAG_CFG) >> 1) & 0x01;
}


/*
 * EFLASH_CFG	EFLASH test configuration register.		Offset: 0x61C
 *
 *  Register Field Name    Field Range
 *  reserved				31 :3
 *  bist_enable				2		Bist enable
 *  reserved				1
 *  eflash_jtag_reuse_en	0		1: Reuse chip JTAG pins (function JTAG disconnected)
 *  								0: Use FLASH specific JTAG pins
 *
 */
__STATIC_INLINE void SysCtl_setBistEnable(uint32_t enable)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_EFLASH_CFG) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_EFLASH_CFG) & ~(1 << 2)) | ((enable & 0x01) << 2);
}

__STATIC_INLINE uint32_t SysCtl_getBistEnable(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_EFLASH_CFG) >> 2) & 0x01;
}

__STATIC_INLINE void SysCtl_setEflashJtagReuseEnable(uint32_t enable)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_EFLASH_CFG) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_EFLASH_CFG) & ~(1 << 0)) | ((enable & 0x01) << 0);
}

__STATIC_INLINE uint32_t SysCtl_getEflashJtagReuseEnable(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_EFLASH_CFG) & 0x01;
}


/*
 * DSP_SYS_ERR_MASK	System Error Interrupt Selection		Offset: 0x64C
 *
 *  Register Field Name       Field Range
 *  reserved	              31 :16
 *  dsp_sys_err_mask	      15 :0
 */
__STATIC_INLINE uint32_t SysCtl_getDspSysErrMask(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DSP_SYS_ERR_MASK) & 0xFFFF;
}

__STATIC_INLINE void SysCtl_setDspSysErrMask(uint32_t mask)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DSP_SYS_ERR_MASK) = mask & 0xFFFF;
}


/*
 * PWM_ECCERR_MASK	PWM ECC Error Mask		Offset: 0x650
 *
 *  Register Field Name       Field Range
 *  reserved					31 :16
 *  cfg_pwm_ecc_err_mask		15:0		PWM ECC Error Encapsulation Wave Condition Mask
 *
 */
__STATIC_INLINE void SysCtl_setCfgPwmEccErrMask(uint32_t mask)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_PWM_ECCERR_MASK) = mask & 0xFFFF;
}

__STATIC_INLINE uint32_t SysCtl_getCfgPwmEccErrMask(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_PWM_ECCERR_MASK) & 0xFFFF;
}


/*
 * PWM_SYSFAULT_MASK	PWM System Fault Mask		Offset: 0x654
 *
 *  Register Field Name    Field Range
 * reserved					31 :16
 * cfg_pwm_sys_err_mask		15:0		PWM System Error Mask Indicates
 *
 */
__STATIC_INLINE void SysCtl_setCfgPwmSysErrMask(uint32_t mask)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_PWM_SYSFAULT_MASK) = mask & 0xFFFF;
}

__STATIC_INLINE uint32_t SysCtl_getCfgPwmSysErrMask(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_PWM_SYSFAULT_MASK) & 0xFFFF;
}


/*
 * ILM_SWAP_EN	ILM 64K Space Swap Configuration		Offset: 0x658
 *
 *  Register Field Name       Field Range
 *  reserved					31 :1
 *  lm_swap_en						0	LM Space Swap Configuration
 *										0: Low space for LM usage space;
 *										1: High space for LM usage space
 *										Note: This configuration is only valid when ILM has two 64K spaces."
 *
 */
__STATIC_INLINE void SysCtl_setLmSwapEn(uint32_t enable)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_ILM_SWAP_EN) = enable & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_getLmSwapEn(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_ILM_SWAP_EN) & 0x01;
}


/*
 * SRAM_SWAP_EN	SRAM Space Swap Configuration	Offset: 0x65C
 *
 *  Register Field Name         Field Range
 *  reserved					31 :1
 *  swap_sram_sw_en					0	SRAM space swap configuration
 */
__STATIC_INLINE void SysCtl_setSwapSramSwEn(uint32_t enable)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_SRAM_SWAP_EN) = enable & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_getSwapSramSwEn(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_SRAM_SWAP_EN) & 0x01;
}


/*
 *  MEM_INI_REQ		MEM-INI-REQ Configuration		Offset: 0x700
 *
 *  Register Field Name      Field Range
 *  reserved					31 :4
 *  lm_mem_ini_req				3
 *  gs_mem_ini_req				2
 *  swap0_mem_ini_req			1
 *  swap1_mem_ini_req			0
 */
typedef enum {
    MEM_SWAP1       = 1<<0,
    MEM_SWAP0       = 1<<1,
    MEM_GS         	= 1<<2,
    MEM_LM    		= 1<<3,
}MemIniType_t;

__STATIC_INLINE void SysCtl_setMemIniReq(MemIniType_t memTypes)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_MEM_INI_REQ) |= memTypes;
}


/*
 *  MEM_INI_DONE_RPT    Initialization Completion Report    Offset: 0x704
 *
 *  Register Field Name      Field Range
 *  reserved    				31 : 5
 *  lm_mem_ini_done_rpt    		3
 *  gs_mem_ini_done_rpt    		2
 *  swap0_mem_ini_done_rpt 		1
 *  swap1_mem_ini_done_rpt 		0
 */
__STATIC_INLINE uint32_t SysCtl_getMemIniDoneRpt(MemIniType_t memTypes)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_MEM_INI_DONE_RPT) & memTypes;
}

__STATIC_INLINE void SysCtl_clearMemIniDoneRpt(MemIniType_t memTypes)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_MEM_INI_DONE_RPT) = memTypes;
}


/*
 * PMBUS_CTL_OEN		PMBus Control Output Enable		Offset: 0x708
 *
 *  Register Field Name     Field Range
 *  reserved				31 :1
 *  pmbus_ctl_oen			0
 */
__STATIC_INLINE uint32_t SysCtl_getPmbusCtlOen(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_PMBUS_CTL_OEN) & 0x01;
}

__STATIC_INLINE void SysCtl_setPmbusCtlOen(uint32_t enable)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_PMBUS_CTL_OEN) = enable & 0x01;
}


/*
 * SDFM_TYPE_CFG     SDFM Type Configuration Register     Offset: 0x70C
 *
 *  Register Field Name     Field Range
 *  reserved        		31 : 2
 *  sdfm_type_cfg       	 1 : 0
 */
__STATIC_INLINE uint32_t SysCtl_getSdfmTypeCfg(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_SDFM_TYPE_CFG) & 0x03;
}

__STATIC_INLINE void SysCtl_setSdfmTypeCfg(uint32_t config)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_SDFM_TYPE_CFG) = config & 0x03;
}


/*
 * SEC_EN_CFG           SEC_EN_CFG Register            Offset: 0x710
 *
 *  Register Field Name         Field Range
 *  reserved                    31:1
 *  sec_en_cfg                  0         SEC_EN configuration, setting sec_en to 1 after writing once.
 *
 */
__STATIC_INLINE uint32_t SysCtl_getSecEnCfg(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_SEC_EN_CFG) & 0x01;
}

__STATIC_INLINE void SysCtl_setSecEnCfg(uint32_t enable)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_SEC_EN_CFG) = enable & 0x01;
}


/*
 * SEC_EN_RPT    Secure Enable Reporting Register    Offset: 0x714
 *
 *  Register Field Name         Field Range
 *  reserved                    31:1
 *  sec_en_rpt                  0         Reporting the effective sec_en value.
 *
 *
 */
__STATIC_INLINE uint32_t SysCtl_getSecEnRpt(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_SEC_EN_RPT) & 0x01;
}


/*
 * AES_ENDIAN_CFG   AES Little Endian AHB Configuration   Offset:  0x718
 *
 *  Register Field Name              Field Range
 *  reserved                         31:1
 *  aes_little_endian_ahb            0
 *
 *
 */
__STATIC_INLINE void SysCtl_setAesLittleEndianAhb(uint32_t enable)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_AES_ENDIAN_CFG) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_AES_ENDIAN_CFG) & ~(1 << 0)) | (enable & 0x01);
}

__STATIC_INLINE uint32_t SysCtl_getAesLittleEndianAhb(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_AES_ENDIAN_CFG) & 0x01;
}


/*
 * FSMC_CFG    FSMC Configuration Register    Offset: 0x71C
 *
 *  Register Field Name            Field Range
 *  reserved        31:8
 *  sysctrl_fsmc_reg        7:0
 *
 */
__STATIC_INLINE void SysCtl_setSysctrlFsmcReg(uint32_t sysctrl_fsmc_reg)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_FSMC_CFG) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_FSMC_CFG) & ~(0xFF << 0)) | (sysctrl_fsmc_reg & 0xFF);
}

__STATIC_INLINE uint32_t SysCtl_getSysctrlFsmcReg(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_FSMC_CFG) & 0xFF;
}


/*
 * HRPWM_CLK_RATE    HRPWM Clock Rate Register    Offset: 0x720
 *
 *  Register Field Name      Field Range
 *  reserved        31:2
 *  hrpwm_clk_rate        1:0     Ratio of HRPWM clock to HR clock
 *                                  0   1:2;
 *                                  1   1:4;
 *                                  other   1:8
 *
 */
typedef enum HrpwmClkRate {
    HRPWM_CLK_RATE_1_2 = 0,
    HRPWM_CLK_RATE_1_4 = 1,
    HRPWM_CLK_RATE_1_8 = 2,
}HrpwmClkRate_t;

__STATIC_INLINE void SysCtl_setHrpwmClkRate(uint32_t hrpwm_clk_rate)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_HRPWM_CLK_RATE) =
            (HWREG(SYSCTL_BASE + SYSCTL_O_HRPWM_CLK_RATE) & ~(0x03 << 0)) | (hrpwm_clk_rate & 0x03);
}

__STATIC_INLINE uint32_t SysCtl_getHrpwmClkRate(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_HRPWM_CLK_RATE) & 0x03;
}


/*
 * DIG_SYSCTL_POR_FLAG0    Reserved Register    Offset: 0x900
 *
 *  Register Field Name      Field Range
 *  dig_sysctrl_por_flag0    31:0
 */
__STATIC_INLINE void SysCtl_setDigSysctrlPorFlag0(uint32_t dig_sysctrl_por_flag0)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_FLAG0) = dig_sysctrl_por_flag0;
}

__STATIC_INLINE uint32_t SysCtl_getDigSysctrlPorFlag0(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_FLAG0);
}


/*
 * DIG_SYSCTL_POR_FLAG1    Reserved Register    Offset: 0x904
 *
 *  Register Field Name      Field Range
 * dig_sysctrl_por_flag1    31:0
 */
__STATIC_INLINE void SysCtl_setDigSysctrlPorFlag1(uint32_t dig_sysctrl_por_flag1)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_FLAG1) = dig_sysctrl_por_flag1;
}

__STATIC_INLINE uint32_t SysCtl_getDigSysctrlPorFlag1(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_FLAG1);
}


/*
 * DIG_SYSCTL_SYS_FLAG0    Reserved Register    Offset: 0x908
 *
 *  Register Field Name      Field Range
 * dig_sysctrl_sys_flag0    31:0
 */
__STATIC_INLINE void SysCtl_setDigSysctrlSysFlag0(uint32_t dig_sysctrl_sys_flag0)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_SYS_FLAG0) = dig_sysctrl_sys_flag0;
}

__STATIC_INLINE uint32_t SysCtl_getDigSysctrlSysFlag0(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_SYS_FLAG0);
}

/*
 * DIG_SYSCTL_SYS_FLAG1    Reserve registers.    Offset: 0x90C
 *
 *  Register Field Name      Field Range
 * dig_sysctrl_sys_flag1    31:0
 */
__STATIC_INLINE void SysCtl_setDigSysctrlSysFlag1(uint32_t dig_sysctrl_sys_flag1)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_SYS_FLAG1) = dig_sysctrl_sys_flag1;
}

__STATIC_INLINE uint32_t SysCtl_getDigSysctrlSysFlag1(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_SYS_FLAG1);
}


/*
 * DIG_SYSCTL_POR_SPARE0    Digital System Control Power-On Reset Spare Register 0   Offset: 0xF00
 * 
 *  Register Field Name      Field Range
 *  dig_sysctrl_por_spare0   31:0
 *
 */
__STATIC_INLINE void SysCtl_setDigSysctlPorSpare0(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE0) = value;
}

__STATIC_INLINE uint32_t SysCtl_getDigSysctlPorSpare0(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE0);
}


/*
 * DIG_SYSCTL_POR_SPARE1    Digital System Control Power-On Reset Spare Register 1   Offset: 0xF04
 * 
 *  Register Field Name      Field Range
 *  dig_sysctrl_por_spare1   31:0
 *
 */
__STATIC_INLINE void SysCtl_setDigSysctlPorSpare1(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE1) = value;
}

__STATIC_INLINE uint32_t SysCtl_getDigSysctlPorSpare1(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE1);
}

/**
 * @brief Checks the Power-On Reset (POR) flag status.
 *
 * @return bool Returns true if both bits 31 and 30 are set (POR flag is active), false otherwise.
 */
__STATIC_INLINE bool SysCtl_getPorFlag(void)
{
    return (HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE1) & 0xC0000000) == 0xC0000000;
}

/**
 * @brief Clears the Power-On Reset (POR) flag.
 *
 * @param value Unused parameter (included for API consistency).
 */
__STATIC_INLINE void SysCtl_clearPorFlag(void)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE1) &= ~0xC0000000;
}


/*
 * DIG_SYSCTL_POR_SPARE2    Digital System Control Power-On Reset Spare Register 2   Offset: 0xF08
 * 
 *  Register Field Name      Field Range
 *  dig_sysctrl_por_spare2   31:0
 *
 */
__STATIC_INLINE void SysCtl_setDigSysctlPorSpare2(uint32_t value)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE2) = value;
}

__STATIC_INLINE uint32_t SysCtl_getDigSysctlPorSpare2(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE2);
}


/*
 * DIG_SYSCTL_SYS_SPARE0    Reserve registers.    Offset: 0xF10
 * 
 *  Register Field Name      Field Range
 *  dig_sysctrl_sys_spare0  31:0
 */
__STATIC_INLINE void SysCtl_setDigSysctlSysSpare0(uint32_t dig_sysctrl_sys_spare0)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_SYS_SPARE0) = dig_sysctrl_sys_spare0;
}

__STATIC_INLINE uint32_t SysCtl_getDigSysctlSysSpare0(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_SYS_SPARE0);
}


/*
 * DIG_SYSCTL_SYS_SPARE1    Reserve registers.    Offset: 0xF14
 *
 *  Register Field Name      Field Range
 *  dig_sysctrl_sys_spare1  31:0
 */
__STATIC_INLINE void SysCtl_setDigSysctlSysSpare1(uint32_t dig_sysctrl_sys_spare1)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_SYS_SPARE1) = dig_sysctrl_sys_spare1;
}

__STATIC_INLINE uint32_t SysCtl_getDigSysctlSysSpare1(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_SYS_SPARE1);
}


/*
 * DIG_SYSCTL_SYS_SPARE2    Reserve registers.    Offset: 0xF18
 *
 *  Register Field Name      Field Range
 *  dig_sysctrl_sys_spare2  31:0
 */
__STATIC_INLINE void SysCtl_setDigSysctlSysSpare2(uint32_t dig_sysctrl_sys_spare2)
{
    HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_SYS_SPARE2) = dig_sysctrl_sys_spare2;
}

__STATIC_INLINE uint32_t SysCtl_getDigSysctlSysSpare2(void)
{
    return HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_SYS_SPARE2);
}


#ifdef __cplusplus
}
#endif

#endif

#endif /* DEVICE_DRIVERLIB_SYSCTL_DIG_H_ */
