/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    sysctl_pinmux_v12.c
*   @brief   
*   @details
*
*/

/*
 * commit history
 * 20240625, Jihong, Fix 2.2 build error, GPIO analog mode is fixed.
 * 20240805, Zhaolei, exchange AIO234 AIO249 GPHAMSEL/GPHIE/GPHIS config for 0039/0049 100pin.
 * */

#include "gs32_version.h"

#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "inc/hw_sysctl_io_v12.h"
#include "sysctl/sysctl_io_v12.h"
#include "sysctl/sysctl_pinmux_v12.h"
#include "gpio.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#if IS_GS32F00xx(0x12)

#define AIO_CONFIG_REMAP_MODE    2      //0: noremap 1: before ECO1 2: after ECO1

#if AIO_CONFIG_REMAP_MODE == 1
__STATIC_FORCEINLINE uint32_t AIO_CONFIG_REMAP(uint32_t pin)
{
#if GS32_PART_NUM == 0x0049 && GS32_PIN_COUNT == 100
    if (pin == 234) {
        pin = 249;
    }
#elif GS32_PART_NUM == 0x0039 && GS32_PIN_COUNT == 100
    if(pin == 249) {
        pin = 234;
    }
#elif GS32_PIN_COUNT <= 80
    if (pin == 233) {
        pin = 235;
    } else if (pin == 236) {
        pin = 227;
    } else if (pin == 248) {
        pin = 238;
    }
#endif

    return pin;
}
#elif AIO_CONFIG_REMAP_MODE == 2
__STATIC_FORCEINLINE uint32_t AIO_CONFIG_REMAP(uint32_t pin)
{
#if GS32_PIN_COUNT <= 80
    if (pin == 236) {
        pin = 227;
    } else if (pin == 248) {
        pin = 238;
    }
#endif
    return pin;
}

#else
#define AIO_CONFIG_REMAP(pin)   pin
#endif  //AIO_CONFIG_REMAP_MODE
#else
#define AIO_CONFIG_REMAP(pin)   pin
#endif  //IS_GS32F00xx(0x12)

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

//*****************************************************************************
//
// GPIO_setDirectionMode
//
//*****************************************************************************
//GS32_DRIVER_GPIO_FUNC_T void
//SysCtl_PinMux_setDirectionMode(uint32_t pin, GPIO_Direction pinIO)
//{
//    volatile uint32_t *gpioBaseAddr;
//    uint32_t pinMask;
//
//    //
//    // Check the arguments.
//    //
//    ASSERT(GPIO_isPinValid(pin));
//
//    gpioBaseAddr = (uint32_t *)GPIOCTRL_BASE +
//                   ((pin / 32U) * GPIO_CTRL_REGS_STEP);
//    pinMask = (uint32_t)1U << (pin % 32U);
//
//    //
//    // Set the data direction
//    //
//    if(pinIO == GPIO_DIR_MODE_OUT)
//    {
//        //
//        // Output
//        //
//        gpioBaseAddr[GPIO_GPxDIR_INDEX] |= pinMask;
//    }
//    else
//    {
//        //
//        // Input
//        //
//        gpioBaseAddr[GPIO_GPxDIR_INDEX] &= ~pinMask;
//    }
//}

//*****************************************************************************
//
// GPIO_getDirectionMode
//
//*****************************************************************************
//GS32_DRIVER_GPIO_FUNC_T GPIO_Direction
//SysCtl_PinMux_getDirectionMode(uint32_t pin)
//{
//    volatile uint32_t *gpioBaseAddr;
//
//    //
//    // Check the arguments.
//    //
//    ASSERT(GPIO_isPinValid(pin));
//
//    gpioBaseAddr = (uint32_t *)GPIOCTRL_BASE +
//                   ((pin / 32U) * GPIO_CTRL_REGS_STEP);
//
//    return((GPIO_Direction)((uint32_t)((gpioBaseAddr[GPIO_GPxDIR_INDEX] >>
//                             (pin % 32U)) & 1U)));
//
//}

////*****************************************************************************
////
//// GPIO_setInterruptPin
////
////*****************************************************************************
//GS32_DRIVER_GPIO_FUNC_T void
//GPIO_setInterruptPin(uint32_t pin, GPIO_ExternalIntNum extIntNum)
//{
//    XBAR_InputNum input;
//
//    //
//    // Check the arguments.
//    //
//    ASSERT(GPIO_isPinValid(pin));
//
//    //
//    // Pick the X-BAR input that corresponds to the requested XINT.
//    //
//    switch(extIntNum)
//    {
//        case GPIO_INT_XINT1:
//            input = XBAR_INPUT4;
//            break;
//
//        case GPIO_INT_XINT2:
//            input = XBAR_INPUT5;
//            break;
//
//        case GPIO_INT_XINT3:
//            input = XBAR_INPUT6;
//            break;
//
//        case GPIO_INT_XINT4:
//            input = XBAR_INPUT13;
//            break;
//
//        case GPIO_INT_XINT5:
//            input = XBAR_INPUT14;
//            break;
//
//        default:
//            //
//            // Invalid interrupt. Shouldn't happen if enum value is used.
//            // XBAR_INPUT1 isn't tied to an XINT, so we'll use it to check for
//            // a bad value.
//            //
//            input = XBAR_INPUT1;
//            break;
//    }
//
//    if(input != XBAR_INPUT1)
//    {
//        XBAR_setInputPin(INPUTXBAR_BASE, input, (uint16_t)pin);
//    }
//}

//*****************************************************************************
//
// GPIO_setPadConfig
//
//*****************************************************************************
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setPadConfig(uint32_t pin, uint32_t pinType)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t pinMask;

    //
    // Check the arguments.
    //
    ASSERT(GPIO_isPinValid(pin));

    gpioBaseAddr = (uint32_t *)GPIOCTRL_BASE +
                   ((pin / 32U) * GPIO_CTRL_REGS_STEP);

    pinMask = (uint32_t)1U << (pin % 32U);

    //
    // Enable pull-up if necessary
    //
    if((pinType & GPIO_PIN_TYPE_PULLUP) != 0U)
    {
        gpioBaseAddr[GPIO_GPxPUD_INDEX] &= ~pinMask;
    }
    else
    {
        gpioBaseAddr[GPIO_GPxPUD_INDEX] |= pinMask;
    }


    //
    // Invert polarity if necessary
    //
    if((pinType & GPIO_PIN_TYPE_INVERT) != 0U)
    {
        gpioBaseAddr[GPIO_GPxINV_INDEX] |= pinMask;
    }
    else
    {
        gpioBaseAddr[GPIO_GPxINV_INDEX] &= ~pinMask;
    }

    //GS32
    gpioBaseAddr = (uint32_t *)GPIOCTRL_BASE +
    		        GPIO_O_GPAPDD/4 +
                   ((pin / 32U) * GPIO_GJADD_REGS_STEP);

    //
    // Enable pull-down if necessary
    //
    if((pinType & GPIO_PIN_TYPE_PULLDOWN) != 0U)
    {

        gpioBaseAddr[GPIO_GPxPDD_INDEX] &= ~pinMask;
    }
    else
    {

        gpioBaseAddr[GPIO_GPxPDD_INDEX] |= pinMask;
    }

}

//*****************************************************************************
//
// GPIO_getPadConfig
//
//*****************************************************************************
GS32_DRIVER_GPIO_FUNC_T uint32_t
GPIO_getPadConfig(uint32_t pin)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t pinMask;
    uint32_t pinTypeRes;

    //
    // Check the arguments.
    //
    ASSERT(GPIO_isPinValid(pin));

    gpioBaseAddr = (uint32_t *)GPIOCTRL_BASE +
                   ((pin / 32U) * GPIO_CTRL_REGS_STEP);
    pinMask = (uint32_t)1U << (pin % 32U);

    pinTypeRes = GPIO_PIN_TYPE_STD;

    //
    // Get pull-up value
    //
    if((gpioBaseAddr[GPIO_GPxPUD_INDEX] & pinMask) == 0U)
    {
        pinTypeRes |= GPIO_PIN_TYPE_PULLUP;
    }

    //
    // Get polarity value
    //
    if((gpioBaseAddr[GPIO_GPxINV_INDEX] & pinMask) != 0U)
    {
        pinTypeRes |= GPIO_PIN_TYPE_INVERT;
    }

    return(pinTypeRes);
}

//*****************************************************************************
//
// GPIO_setQualificationMode
//
//*****************************************************************************
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setQualificationMode(uint32_t pin, GPIO_QualificationMode qualification)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t qSelIndex;
    uint32_t shiftAmt;

    //
    // Check the arguments.
    //
    ASSERT(GPIO_isPinValid(pin));

    gpioBaseAddr = (uint32_t *)GPIOCTRL_BASE +
                   ((pin / 32U) * GPIO_CTRL_REGS_STEP);
    shiftAmt = (uint32_t)GPIO_GPAQSEL1_GPIO1_S * (pin % 16U);
    qSelIndex = GPIO_GPxQSEL_INDEX + ((pin % 32U) / 16U);

    //
    // Write the input qualification mode to the register.
    //

    gpioBaseAddr[qSelIndex] &= ~((uint32_t)GPIO_GPAQSEL1_GPIO0_M << shiftAmt);
    gpioBaseAddr[qSelIndex] |= (uint32_t)qualification << shiftAmt;
}

//*****************************************************************************
//
// GPIO_getQualificationMode
//
//*****************************************************************************
GS32_DRIVER_GPIO_FUNC_T GPIO_QualificationMode
GPIO_getQualificationMode(uint32_t pin)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t qSelIndex;
    uint32_t qualRes;
    uint32_t shiftAmt;

    //
    // Check the arguments.
    //
    ASSERT(GPIO_isPinValid(pin));

    gpioBaseAddr = (uint32_t *)GPIOCTRL_BASE +
                   ((pin / 32U) * GPIO_CTRL_REGS_STEP);
    shiftAmt = (uint32_t)GPIO_GPAQSEL1_GPIO1_S * (pin % 16U);
    qSelIndex = GPIO_GPxQSEL_INDEX + ((pin % 32U) / 16U);

    //
    // Read the qualification mode register and shift and mask to get the
    // value for the specified pin.
    //
    qualRes = (gpioBaseAddr[qSelIndex] >> shiftAmt) &
              (uint32_t)GPIO_GPAQSEL1_GPIO0_M;
    return((GPIO_QualificationMode)qualRes);
}

//*****************************************************************************
//
// GPIO_setQualificationPeriod
//
//*****************************************************************************
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setQualificationPeriod(uint32_t pin, uint32_t divider)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t pinMask, regVal, shiftAmt;

    //
    // Check the arguments.
    //
    ASSERT(GPIO_isPinValid(pin));
    ASSERT((divider >= 1U) && (divider <= 510U));

    shiftAmt = (pin % 32U) & ~((uint32_t)0x7U);
    pinMask = (uint32_t)0xFFU << shiftAmt;

    //
    // Divide divider by two to get the value that needs to go into the field.
    // Then shift it into the right place.
    //
    regVal = (divider / 2U) << shiftAmt;

    //
    // Write the divider parameter into the register.
    //
    gpioBaseAddr = (uint32_t *)GPIOCTRL_BASE +
                   ((pin / 32U) * GPIO_CTRL_REGS_STEP);

    gpioBaseAddr[GPIO_GPxCTRL_INDEX] &= ~pinMask;
    gpioBaseAddr[GPIO_GPxCTRL_INDEX] |= regVal;
}

//*****************************************************************************
//
// GPIO_setAnalogMode
//
//*****************************************************************************
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setAnalogMode(uint32_t pin, GPIO_AnalogMode mode)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t pinMask;

    //
    // Check the arguments.
    //
    ASSERT(((pin >= 224U) && (pin <= 254U) && (pin != 234U) && (pin != 235U) &&
           (pin != 243U) && (pin != 246U) && (pin != 250U)) ||
    		(pin == 13U) ||(pin == 20U) || (pin == 21U));

    pin = AIO_CONFIG_REMAP(pin);

    pinMask = (uint32_t)1U << (pin % 32U);
    gpioBaseAddr = (uint32_t *)GPIOCTRL_BASE +
                   ((pin / 32U) * GPIO_CTRL_REGS_STEP);

    //EALLOW;
#if IS_GS32F00xx(0x12)
    //
    // Set the analog mode selection.
    //
    if(mode == GPIO_ANALOG_ENABLED)
    {
        //
        // Enable analog mode
        //
        gpioBaseAddr[GPIO_GPxAMSEL_INDEX] |= pinMask;
        if((pin == 13U) ||(pin == 20U) || (pin == 21U))
        {
            //
            // Set AGPIOCTL
            //
            HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_AGPIOCTRLA +
                   ((pin / 32U) * 2U)) |= (pinMask);
        }
    }
    else
    {
        //
        // Disable analog mode
        //
        gpioBaseAddr[GPIO_GPxAMSEL_INDEX] &= ~pinMask;
        if((pin == 13U) ||(pin == 20U) || (pin == 21U))
        {
            //
            // Clear AGPIOCTL
            //
            HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_AGPIOCTRLA +
                   ((pin / 32U) * 2U)) &= ~(pinMask);
        }
    }
#elif IS_GS32F3xx(0x22)
    //
    // Set the analog mode selection.
    //
    if(mode == GPIO_ANALOG_ENABLED)
    {
        //
        // Enable analog mode
        //
        gpioBaseAddr[GPIO_GPxAMSEL_INDEX] |= pinMask;
    }
    else
    {
        //
        // Disable analog mode
        //
        gpioBaseAddr[GPIO_GPxAMSEL_INDEX] &= ~pinMask;
    }

#endif

    EDIS;
}

/**
 * \brief   GPIO MUX config.
 *
 * \param   pinConfig, high 16bit is pin number 0~61, low 16bit is pin Mux Select 0~15
 *          The following MACRO can be used as the input parameter, for example,
 *          GPIO_0_GPIO0,
 *          GPIO_0_EPWM1_A,
 *          GPIO_1_GPIO1,
 *          GPIO_1_I2C0_SCL,
 *          ......
 */
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setPinConfig(uint32_t pinConfig)
{
    uint32_t muxRegAddr;
    uint32_t pinMask, shiftAmt;

    muxRegAddr = (uint32_t)GPIOCTRL_BASE + (pinConfig >> 16);
    shiftAmt = ((pinConfig >> 8) & (uint32_t)0xFFU);
    pinMask = (uint32_t)0x3U << shiftAmt;

    //
    // Clear fields in MUX register first to avoid glitches
    //
    HWREG(muxRegAddr) &= ~pinMask;

    //
    // Write value into GMUX register
    //
    HWREG(muxRegAddr + GPIO_MUX_TO_GMUX) =
        (HWREG(muxRegAddr + GPIO_MUX_TO_GMUX) & ~pinMask) |
        (((pinConfig >> 2) & (uint32_t)0x3U) << shiftAmt);

    //
    // Write value into MUX register
    //
    HWREG(muxRegAddr) |= ((pinConfig & (uint32_t)0x3U) << shiftAmt);
}


//*****************************************************************************
//
// GPIO_setStrength : strength 0/1 (GPIO support PI2C) ; strength 0/1/2/3 (GPIO don't support PI2C)
//
//*****************************************************************************
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setStrength(uint32_t pin, GPIO_DriveStrength StrengthLevel)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t pinMask;
    uint32_t qSelIndex;

    //
    // Check the arguments.
    //
    ASSERT(GPIO_isPinValid(pin));

    gpioBaseAddr = (uint32_t *)GPIOCTRL_BASE + GPIO_O_GPAPDD/4 +
                           ((pin / 32U) * GPIO_GJADD_REGS_STEP);
    pinMask = (uint32_t)StrengthLevel << (2*(pin % 16U));

    qSelIndex = GPIO_GPxDS_INDEX + ((pin % 32U) / 16U);

    //
    // Enable driver strength
    //
    gpioBaseAddr[qSelIndex] &= ~(0x3 << (2*(pin % 16U)));
    gpioBaseAddr[qSelIndex] |= pinMask;



}

//*****************************************************************************
//
// GPIO_setSchmittTrigger
//
//*****************************************************************************
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setSchmittTrigger(uint32_t pin, bool pinType)
{
    uint32_t pinMask;

    pin = AIO_CONFIG_REMAP(pin);

    pinMask = (uint32_t)1U << (pin % 32U);
    //
    // Enable Schmitt Trigger
    //
    if(pin < 224)
    {
		if( pinType )
		{
			HWREG(GPIOCTRL_BASE + GPIO_O_GPAIS + (pin / 32U) * GPIO_GJADD_REGS_STEP * 4 ) |= pinMask;
		}
		else
		{
			HWREG(GPIOCTRL_BASE + GPIO_O_GPAIS + (pin / 32U) * GPIO_GJADD_REGS_STEP * 4 ) &= ~pinMask;
		}
    }
    else
    {
		if( pinType )
		{
			HWREG(GPIOCTRL_BASE + GPIO_O_GPHIS) |= pinMask;
		}
		else
		{
			HWREG(GPIOCTRL_BASE + GPIO_O_GPHIS) &= ~pinMask;
		}
    }
}

//*****************************************************************************
//
// GPIO_setOutPolInv
//
//*****************************************************************************
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setOutPolarityInv(uint32_t pin, bool pinType)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t pinMask;


    //
    // Check the arguments.
    //
    ASSERT(GPIO_isPinValid(pin));

    gpioBaseAddr = (uint32_t *)GPIOCTRL_BASE + GPIO_O_GPAPDD/4 +
                           ((pin / 32U) * GPIO_GJADD_REGS_STEP);
    pinMask = (uint32_t)1U << (pin % 32U);


    //
    // Enable Output Polarity Invert
    //
    if( pinType )
    {
        gpioBaseAddr[GPIO_GPxOUTINV_INDEX] |= pinMask;
    }
    else
    {
    	gpioBaseAddr[GPIO_GPxOUTINV_INDEX] &= ~pinMask;
    }


}

//*****************************************************************************
//
// GPIO_setLoop
//
//*****************************************************************************
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setLoopBack(uint32_t pin, bool pinType)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t pinMask;


    //
    // Check the arguments.
    //
    ASSERT(GPIO_isPinValid(pin));

    gpioBaseAddr = (uint32_t *)GPIOCTRL_BASE + GPIO_O_GPAPDD/4 +
                           ((pin / 32U) * GPIO_GJADD_REGS_STEP);
    pinMask = (uint32_t)1U << (pin % 32U);


    //
    // Enable Loop
    //
    if( pinType )
    {
        gpioBaseAddr[GPIO_GPxLOOPEN_INDEX] |= pinMask;
    }
    else
    {
    	gpioBaseAddr[GPIO_GPxLOOPEN_INDEX] &= ~pinMask;
    }

}

//*****************************************************************************
//
// GPIO_setI2CMode
//
//*****************************************************************************
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setI2CMode(uint32_t pin, bool pinType)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t pinMask;


    //
    // Check the arguments.
    //
    ASSERT(GPIO_isPinValid(pin));

    gpioBaseAddr = (uint32_t *)GPIOCTRL_BASE + GPIO_O_GPAPDD/4 +
                           ((pin / 32U) * GPIO_GJADD_REGS_STEP);
    pinMask = (uint32_t)1U << (pin % 32U);


    //
    // Enable PI2C Mode
    //
    if( pinType )
    {
        gpioBaseAddr[GPIO_GPxPI2C_MODE_INDEX] |= pinMask;
    }
    else
    {
    	gpioBaseAddr[GPIO_GPxPI2C_MODE_INDEX] &= ~pinMask;
    }

}

//*****************************************************************************
//
// GPIO_readPinOutReport
//
//*****************************************************************************
GS32_DRIVER_GPIO_FUNC_T uint32_t
GPIO_readPinOutReport(uint32_t pin)
{
	uint16_t RegValue;

    volatile uint32_t *gpioBaseAddr;



    gpioBaseAddr = (uint32_t *)GPIOCTRL_BASE + GPIO_O_GPAPDD/4 +
                            ((pin/32) * GPIO_GJADD_REGS_STEP);



	RegValue = gpioBaseAddr[GPIO_GPxPOUT_RPT_INDEX];

	return (((RegValue & (1 << (pin % 32))) > 0)? 1 : 0 );


}

//*****************************************************************************
//
// GPIO_readPortOutReport
//
//*****************************************************************************
GS32_DRIVER_GPIO_FUNC_T uint32_t
GPIO_readPortOutReport(GPIO_Port_32BIT port)
{
    volatile uint32_t *gpioBaseAddr;



    gpioBaseAddr = (uint32_t *)GPIOCTRL_BASE + GPIO_O_GPAPDD/4 +
                            (port* GPIO_GJADD_REGS_STEP);

    return gpioBaseAddr[GPIO_GPxPOUT_RPT_INDEX];


}

//*****************************************************************************
//
// GPIO_setSchmittTrigger_H
//
//*****************************************************************************
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setSchmittTrigger_H(uint32_t pin, bool pinType)
{


}

//*****************************************************************************
//
// GPIO_setInputEnableGPH
//
//*****************************************************************************
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setInputEnableGPH(uint32_t pin, bool pinType)
{

    uint32_t pinMask;

    pin = AIO_CONFIG_REMAP(pin);

    pinMask = (uint32_t)1U << (pin % 32U);

    if( pinType )
    {
    	HWREG(GPIOCTRL_BASE + GPIO_O_GPHIE) |= pinMask;
    }
    else
    {
    	HWREG(GPIOCTRL_BASE + GPIO_O_GPHIE) &= ~pinMask;
    }
}










#ifdef __cplusplus
}
#endif

#endif
