/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    sysctl_pinmux_v12.h
*   @brief   system control PinMux driver
*
*/
/*
*  commit history
*  20240324, Jason, add pin==133 for pin check for 2.0ES.
*  20241218,Hezhiyuan,Modify comments
*/

#ifndef DEVICE_DRIVERLIB_SYSCTL_PINMUX_V12_H_
#define DEVICE_DRIVERLIB_SYSCTL_PINMUX_V12_H_

#include "gs32_version.h"

#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include <stdbool.h>
#include <stdint.h>
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "debug.h"



/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/*
 * 
 * Useful defines used within the driver functions to access gpio registers.
 * Not intended for use by application code.
 *   
 * Divide by 4 is for DSP which has 32bits access
 * 
 */
#define GPIO_CTRL_REGS_STEP         ((GPIO_O_GPBCTRL - GPIO_O_GPACTRL) / 4U)
#define GPIO_EXT_REGS_STEP       	((GPIO_O_GPBPDD - GPIO_O_GPAPDD) / 4U)

#define GPIO_GPxCTRL_INDEX          (GPIO_O_GPACTRL / 4U)
#define GPIO_GPxQSEL_INDEX          (GPIO_O_GPAQSEL1 / 4U)
#define GPIO_GPxMUX_INDEX           (GPIO_O_GPAMUX1 / 4U)
//#define GPIO_GPxDIR_INDEX           (GPIO_O_GPADIR / 4U)
#define GPIO_GPxAMSEL_INDEX         (GPIO_O_GPAAMSEL / 4U)
#define GPIO_GPxPUD_INDEX           (GPIO_O_GPAPUD / 4U)
#define GPIO_GPxINV_INDEX           (GPIO_O_GPAINV / 4U)

#define GPIO_GPxGMUX_INDEX          (GPIO_O_GPAGMUX1 / 4U)

#define GPIO_MUX_TO_GMUX            (GPIO_O_GPAGMUX1 - GPIO_O_GPAMUX1)

#define MAX_GPIO_NUM                (68)

#define GPIO_GJADD_REGS_STEP         ((GPIO_O_GPBPDD - GPIO_O_GPAPDD) / 4U)
#define GPIO_GPxPDD_INDEX          ((GPIO_O_GPAPDD - GPIO_O_GPAPDD)/ 4U)
#define GPIO_GPxDS_INDEX          ((GPIO_O_GPADS1 - GPIO_O_GPAPDD)/ 4U)
#define GPIO_GPxIS_INDEX          ((GPIO_O_GPAIS - GPIO_O_GPAPDD)/ 4U)
#define GPIO_GPxOUTINV_INDEX          ((GPIO_O_GPAOUTINV - GPIO_O_GPAPDD)/ 4U)
#define GPIO_GPxLOOPEN_INDEX          ((GPIO_O_GPALOOPEN - GPIO_O_GPAPDD)/ 4U)
#define GPIO_GPxPI2C_MODE_INDEX          ((GPIO_O_GPAPI2C_MODE - GPIO_O_GPAPDD)/ 4U)
#define GPIO_GPxPOUT_RPT_INDEX          ((GPIO_O_GPA_POUT_RPT - GPIO_O_GPAPDD)/ 4U)
/**
 * IO_CFG register masks
 **/
#define IO_CFG_PULL_ENABLE                    0x00000001U
#define IO_CFG_PULL_DISABLE                   0x00000000U

#define IO_CFG_PULL_UP                        0x00000010U
#define IO_CFG_PULL_DOWN                      0x00000000U

#define IO_CFG_DS0_LV1                        0x00000100U
#define IO_CFG_DS0_LV0                        0x00000000U
#define IO_CFG_DS1_LV1                        0x00000200U
#define IO_CFG_DS1_LV0                        0x00000000U

#define IO_CFG_IS_ENABLE                      0x00001000U
#define IO_CFG_IS_DISABLE                     0x00000000U

#define IO_CFG_SYNC_HCLK                      0x00000000U
#define IO_CFG_SYNC_3SAMPLE                   0x00010000U
#define IO_CFG_SYNC_6SAMPLE                   0x00020000U
#define IO_CFG_SYNC_DISABLE                   0x00030000U

#define IO_CFG_INPUT_INVERT                   0x00100000U

#define IO_CFG_OUTPUT_INVERT                  0x00200000U

#define PINMUX_EPWM1_A_GP0                    1U
#define PINMUX_I2C0_SDA_GP0                   6U

/*
 *
 * Values that can be passed to GPIO_setPadConfig() as the pinType parameter
 * and returned by GPIO_getPadConfig().
 * 
 */
#define GPIO_PIN_TYPE_STD                     0x0000U //< Push-pull output or floating input
#define GPIO_PIN_TYPE_PULLUP                  0x0001U //< Pull-up enable for input
#define GPIO_PIN_TYPE_INVERT                  0x0002U //< Invert polarity on input
#define GPIO_PIN_TYPE_PULLDOWN                0x0008U //< Pull-down enable for input
#define GPIO_PIN_TYPE_OUTPUT_INVERT     	  0x0010U /* Inverted Push-Pull Output */

#define GPIO_PIN_TYPE_STRENGTH_0              0x0000U
#define GPIO_PIN_TYPE_STRENGTH_1              0x0200U
#define GPIO_PIN_TYPE_STRENGTH_2              0x0100U
#define GPIO_PIN_TYPE_STRENGTH_3              0x0300U
#define GPIO_PIN_TYPE_STRENGTH_MASK           0x0300U

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/**
* @brief PS config
* @details Set pull-up or Pull-down
*
* */
 typedef enum
{
    // Disable Pull-down
    IO_SET_PE_TYPE_DISABLE  = 0x0U,
    // Enable Pull-up
    IO_SET_PE_TYPE_ENABLE   = 0x1U
} IO_SET_PE_Type;

 /**
 * @brief PS config
 * @details Set pull-up or Pull-down
 *
 * */
  typedef enum
{
    // Disable Pull-down
    IO_SET_PS_TYPE_PULL_DOWN = 0x0U,
    // Enable Pull-up
    IO_SET_PS_TYPE_PULL_UP   = 0x1U
} IO_SET_PS_Type;
 
/**
* @brief PS config
* @details Set pull-up or Pull-down
*
* */
 typedef enum
{
    IO_SET_DS_TYPE_0 = 0x0U,
    IO_SET_DS_TYPE_1 = 0x1U,
    IO_SET_DS_TYPE_2 = 0x2U,
    IO_SET_DS_TYPE_3 = 0x3U,
} IO_SET_DS_Type;

/**
 * 
 * Values that can be passed to GPIO_setQualificationMode() as the
 * qualification parameter and returned by GPIO_getQualificationMode().
 * 
 */
typedef enum
{
    GPIO_QUAL_SYNC = 0x00U,                     //< Synchronization to SYSCLK
    GPIO_QUAL_3SAMPLE = 0x01U,                  //< Qualified with 3 samples
    GPIO_QUAL_6SAMPLE = 0x02U,                  //< Qualified with 6 samples
    GPIO_QUAL_ASYNC = 0x03U                     //< No synchronization
} GPIO_QualificationMode;

/**
 * 
 *  Values that can be passed to GPIO_setAnalogMode() as the mode parameter.
 * 
 */
typedef enum
{
    GPIO_ANALOG_DISABLED,       //< Pin is in digital mode
    GPIO_ANALOG_ENABLED         //< Pin is in analog mode
} GPIO_AnalogMode;

/**
 * 
 *  Values that can be passed to GPIO_setStrength() as the  mode parameter.
 * 
 */
typedef enum
{
	GPIO_PIN_DRIVE_STRENGTH_0,       //drive strength GPIO:2mA  PI2C GPIO:4mA
	GPIO_PIN_DRIVE_STRENGTH_1,		 //drive strength GPIO:4mA  PI2C GPIO:20mA
	GPIO_PIN_DRIVE_STRENGTH_2,		 //drive strength GPIO:8mA
	GPIO_PIN_DRIVE_STRENGTH_3        //drive strength GPIO:12mA
} GPIO_DriveStrength;

/**
 * 
 *  Values that can be passed to GPIO_setClkRef() as the  mode parameter.
 * 
 */
typedef enum
{
    GPIO_ClkRef_CRYSTAL 		= 0<<1 | 1<<0,      //HSE type crystal
    GPIO_ClkRef_OSCILLATOR 		= 1<<1 | 0<<0,   //HSE type oscillator
    GPIO_ClkRef_1MHz_TO_4MHz 	= 0<<2,        //1MHz < freq <= 4MHz
    GPIO_ClkRef_4MHz_TO_12MHz 	= 2<<2,       //4MHz < freq <= 12MHz
    GPIO_ClkRef_12MHz_TO_24MHz 	= 1<<2,      //12MHz < freq <= 24MHz
    GPIO_ClkRef_24MHz_TO_48MHz 	= 3<<2,      //24MHz < freq <= 48MHz
    GPIO_ClkRef_R_500K = 0<<4,              //500KOm feedback resistor
    GPIO_ClkRef_R_100K = 1<<4,              //100KOm feedback resistor
} GPIO_ClkRefMode;

/**
 * 
 *  Values that can be passed to GPIO_readPortData(), GPIO_setPortPins(),
 *  GPIO_clearPortPins(), and GPIO_togglePortPins() as the  port parameter.
 * 
 */
#if IS_GS32F00xx(0x12)
typedef enum
{
    GPIO_PORTA = 0,                    //< GPIO port A, 32 bits
    GPIO_PORTB = 1,                    //< GPIO port B, 32 bits
	GPIO_PORTC = 2,                    //< GPIO port C, 32 bits
//    GPIO_PORTD = 3,                    //< GPIO port D, 32 bits
//	GPIO_PORTE = 4,                    //< GPIO port E, 32 bits
//    GPIO_PORTF = 5,                    //< GPIO port F, 32 bits
//	GPIO_PORTG = 6,                    //< GPIO port G, 32 bits
	GPIO_PORTH = 7                     //< GPIO port H, 32 bits
} GPIO_Port_32BIT;
#elif IS_GS32F3xx(0x22)
typedef enum
{
    GPIO_PORTA = 0,                    //< GPIO port A, 32 bits
    GPIO_PORTB = 1,                    //< GPIO port B, 32 bits
	GPIO_PORTC = 2,                    //< GPIO port C, 32 bits
    GPIO_PORTD = 3,                    //< GPIO port D, 32 bits
	GPIO_PORTE = 4,                    //< GPIO port E, 32 bits
    GPIO_PORTF = 5,                    //< GPIO port F, 32 bits
	GPIO_PORTG = 6,                    //< GPIO port G, 32 bits
	GPIO_PORTH = 7                     //< GPIO port H, 32 bits
} GPIO_Port_32BIT;
#endif

//*****************************************************************************


/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */

/**
 * 
 * @brief   Checks a digital IO base address.
 * @details This function determines if a digital IO module base address is valid.
 * @param [in]   base    specifies the digital IO module base address.
 * @return  Returns true if the base address is valid and false otherwise.
 * 
 */
__STATIC_INLINE
bool GPIO_isPinValid(uint32_t pin)
{
    return(
        ((pin >= 0) && (pin <= MAX_GPIO_NUM))
		|| (pin == 133)
        );
}

/**
 * @brief   Configures the reference clock mode for the GPIO.
 *
 * @details This function sets the reference clock mode by writing to the `CLK_REF_PIN_CTRL` register.
 *          The `CLK_REF_PIN_CTRL` register controls the crystal or oscillator type, frequency range,
 *          and feedback resistor selection for the reference clock input.
 *
 * @param [in] mode  Specifies the reference clock configuration. It should be one of the values from
 *                   the `GPIO_ClkRefMode` enumeration.
 *
 * @note    The `GPIO_ClkRefMode` enumeration provides options to configure:
 *          - Crystal or oscillator selection.
 *          - Frequency range (1MHz to 48MHz).
 *          - Feedback resistor value (500K? or 100K?).
 *
 * @enum GPIO_ClkRefMode
 * | Mode                      | Description                                        |
 * |---------------------------|----------------------------------------------------|
 * | GPIO_ClkRef_CRYSTAL       | HSE type crystal.                                  |
 * | GPIO_ClkRef_OSCILLATOR    | HSE type oscillator.                               |
 * | GPIO_ClkRef_1MHz_TO_4MHz  | Frequency range: 1MHz < freq <= 4MHz.              |
 * | GPIO_ClkRef_4MHz_TO_12MHz | Frequency range: 4MHz < freq <= 12MHz.             |
 * | GPIO_ClkRef_12MHz_TO_24MHz| Frequency range: 12MHz < freq <= 24MHz.            |
 * | GPIO_ClkRef_24MHz_TO_48MHz| Frequency range: 24MHz < freq <= 48MHz.            |
 * | GPIO_ClkRef_R_500K        | 500K? feedback resistor.                           |
 * | GPIO_ClkRef_R_100K        | 100K? feedback resistor.                           |
 *
 * @return  None.
 *
 */
__STATIC_INLINE void
GPIO_setClkRef(GPIO_ClkRefMode mode)
{
    HWREG(GPIOCTRL_BASE + GPIO_O_HSE_CTRL) = mode;
}

/**
 * @brief   Retrieves the current reference clock configuration.
 *
 * @details This function reads the current configuration from the `CLK_REF_PIN_CTRL` register
 *          and returns it as a 32-bit value. The returned value reflects the active reference
 *          clock mode, including the crystal/oscillator type, frequency range, and feedback resistor.
 *
 * @return  Returns a 32-bit value representing the current reference clock configuration.
 *          The value corresponds to the bits set in the `CLK_REF_PIN_CTRL` register.
 *
 * @note    The returned value can be used to check the current reference clock settings
 *          and compare them with the desired settings.
 *
 */
__STATIC_INLINE uint32_t
GPIO_getClkRef(void)
{
    return HWREG(GPIOCTRL_BASE + GPIO_O_HSE_CTRL);
}

/**
 * @brief	Set the digital input function of AIO
 *
 * @param [in] pinType
 *
 * @return  None.
 *
 */
__STATIC_INLINE void
GPIO_setIE(bool pinType)
{
    if( pinType )
    {
    	HWREG(GPIOCTRL_BASE + GPIO_O_IO_IE_MODE) |= 0x01;
    }
    else
    {
    	HWREG(GPIOCTRL_BASE + GPIO_O_IO_IE_MODE) &= ~0x01;
    }
}

/**
 * @brief   Configures the number of OSCCLK clock cycles to qualify inputs during wake-up from standby mode.
 *
 * @details This function writes a value to the `LPMCR_QUALSTDBY` register to determine the number of OSCCLK clock cycles
 *          required to qualify the selected inputs when waking up the system from STANDBY mode. The valid range is
 *          from 0 (2 OSCCLK cycles) to 63 (65 OSCCLK cycles).
 *
 * @param   time  Number of OSCCLK cycles to qualify the wake-up inputs. Valid range:
 *                - `0x00` (2 OSCCLK cycles)
 *                - `0x01` (3 OSCCLK cycles)
 *                - ...
 *                - `0x3F` (65 OSCCLK cycles)
 *
 * @note    The value of `time` must not exceed `0x3F`. If an invalid value is passed, the behavior is undefined.
 *
 */
__STATIC_INLINE void
GPIO_setLPMCR_QUALSTDBY(uint32_t time)
{
    HWREG(GPIOCTRL_BASE + GPIO_O_HSE_CTRL) = time << 2;
}


/**
 *  
 *  @brief Sets the pad configuration for the specified pin.
 * 
 *  @param [in] pin is the identifying GPIO number of the pin.
 *  @param [in] pinType specifies the pin type.
 * 
 *  This function sets the pin type for the specified pin. The parameter
 *   pinType can be the following values:
 * 
 *  -  GPIO_PIN_TYPE_STD specifies a push-pull output or a floating input
 *  -  GPIO_PIN_TYPE_PULLUP specifies the pull-up is enabled for an input
 *  -  GPIO_PIN_TYPE_INVERT specifies inverted polarity on an input
 * 
 *   GPIO_PIN_TYPE_INVERT may be OR-ed with  GPIO_PIN_TYPE_STD or
 *   GPIO_PIN_TYPE_PULLUP.
 * 
 *  The pin is specified by its numerical value. For example, GPIO34 is
 *  specified by passing 34 as  pin.
 * 
 *  @return None.
 *  
 */
extern void
GPIO_setPadConfig(uint32_t pin, uint32_t pinType);

/**
 *  
 *  @brief Gets the pad configuration for a pin.
 * 
 *  @param [in] pin is the identifying GPIO number of the pin.
 * 
 *  This function returns the pin type for the specified pin.  The value
 *  returned corresponds to the values used in GPIO_setPadConfig().
 * 
 *  @return Returns a bit field of the values  GPIO_PIN_TYPE_STD,
 *   GPIO_PIN_TYPE_PULLUP, and  GPIO_PIN_TYPE_INVERT.
 *  
 */
extern uint32_t
GPIO_getPadConfig(uint32_t pin);

/**
 *  
 *  @brief Sets the qualification mode for the specified pin.
 * 
 *  @param [in] pin is the identifying GPIO number of the pin.
 *  @param [in] qualification specifies the qualification mode of the pin.
 * 
 *  This function sets the qualification mode for the specified pin. The
 *  parameter  qualification can be one of the following values:
 *  -  GPIO_QUAL_SYNC
 *  -  GPIO_QUAL_3SAMPLE
 *  -  GPIO_QUAL_6SAMPLE
 *  -  GPIO_QUAL_ASYNC
 * 
 *  To set the qualification sampling period, use
 *  GPIO_setQualificationPeriod().
 * 
 *  @return None.
 *  
 */
extern void
GPIO_setQualificationMode(uint32_t pin, GPIO_QualificationMode qualification);

/**
 *  
 *  @brief Gets the qualification type for the specified pin.
 * 
 *  @param [in] pin is the identifying GPIO number of the pin.
 * 
 *  @return Returns the qualification mode in the form of one of the values
 *   GPIO_QUAL_SYNC,  GPIO_QUAL_3SAMPLE,  GPIO_QUAL_6SAMPLE, or
 *   GPIO_QUAL_ASYNC.
 *  
 */
extern GPIO_QualificationMode
GPIO_getQualificationMode(uint32_t pin);

/**
 *  
 *  @brief Sets the qualification period for a set of pins
 * 
 *  @param [in] pin is the identifying GPIO number of the pin.
 *  @param [in] divider specifies the output drive strength.
 * 
 *  This function sets the qualification period for a set of 8 pins,
 *  specified by the  pin parameter. For instance, passing in 3 as the value
 *  of  pin will set the qualification period for GPIO0 through GPIO7, and a
 *  value of 98 will set the qualification period for GPIO96 through GPIO103.
 *  This is because the register field that configures the divider is shared.
 * 
 *  To think of this in terms of an equation, configuring  pin as n will
 *  configure GPIO (n & ~(7)) through GPIO ((n & ~(7)) + 7).
 * 
 *   divider is the value by which the frequency of SYSCLKOUT is divided. It
 *  can be 1 or an even value between 2 and 510 inclusive.
 * 
 *  @return None.
 *  
 */
extern void
GPIO_setQualificationPeriod(uint32_t pin, uint32_t divider);

/**
 *  
 *  @brief Sets the analog mode of the specified pin.
 * 
 *  @param [in] pin is the identifying GPIO number of the pin.
 *  @param [in] mode is the selected analog mode.
 * 
 *  This function configures the specified pin for either analog or digital
 *  mode. Not all GPIO pins have the ability to be switched to analog mode,
 *  so refer to the technical reference manual for details. This setting should
 *  be thought of as another level of muxing.
 * 
 *  The parameter  mode is an enumerated data type that can be one of the
 *  following values:
 * 
 *  - GPIO_ANALOG_DISABLED - Pin is in digital mode
 *  - GPIO_ANALOG_ENABLED - Pin is in analog mode
 * 
 *  The pin is specified by its numerical value. For example, GPIO34 is
 *  specified by passing 34 as  pin.
 * 
 *  @remarks
 *  - The pin parameter is applicable for both AIO and GPIO because
 *    the GPAxMSEL.GPIOy register configures for both
 * 
 *  @return None.
 *  
 */
extern void
GPIO_setAnalogMode(uint32_t pin, GPIO_AnalogMode mode);

/**
 *  
 *  @brief Configures the alternate function of a GPIO pin.
 * 
 *  @param [in] pinConfig is the pin configuration value, specified as only one
 *  of the GPIO_#_???? values.
 * 
 *  This function configures the pin mux that selects the peripheral function
 *  associated with a particular GPIO pin.  Only one peripheral function at a
 *  time can be associated with a GPIO pin, and each peripheral function should
 *  only be associated with a single GPIO pin at a time (despite the fact that
 *  many of them can be associated with more than one GPIO pin).
 * 
 *  The available mappings are supplied in <tt>pin_map.h</tt>.
 * 
 *  @return None.
 *  
 */
extern void
GPIO_setPinConfig(uint32_t pinConfig);
extern void
GPIO_setStrength(uint32_t pin, GPIO_DriveStrength StrengthLevel);
extern void
GPIO_setSchmittTrigger(uint32_t pin, bool pinType);
extern void
GPIO_setOutPolarityInv(uint32_t pin, bool pinType);
extern void
GPIO_setLoopBack(uint32_t pin, bool pinType);
extern void
GPIO_setI2CMode(uint32_t pin, bool pinType);
uint32_t
GPIO_readPinOutReport(uint32_t pin);
extern uint32_t
GPIO_readPortOutReport(GPIO_Port_32BIT port);
extern void
GPIO_setInputEnableGPH(uint32_t pin, bool pinType);



#ifdef __cplusplus
}
#endif

#endif

#endif /* DEVICE_DRIVERLIB_SYSCTL_PINMUX_H_ */
