/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    sysctl_subcrg_v30.h
*   @brief   do NOT include this file directly, include "sysctl.h" instead
*
*/
/*
*  commit history
*  20241218,Hezhiyuan,Modify comments
*/
#ifndef DEVICE_DRIVERLIB_SYSCTL_SUBCRG_V30_H_
#define DEVICE_DRIVERLIB_SYSCTL_SUBCRG_V30_H_

#include "gs32_version.h"

#if IS_GS32F00xx(0x30)

#ifdef __cplusplus
extern "C"{
#endif

#include "inc/hw_types.h"
#include "inc/hw_memmap.h"
#include "inc/hw_sysctl_subcrg.h"
#include "inc/hw_sysctl_ahb.h"

#define CRG_CFG_ADDR_SHIFT      25
#define CRG_CFG_ADDR_MASK       0x7F
#define CRG_CFG_MASK_SHIFT      5
#define CRG_CFG_MASK_MASK       0xFFFFF
#define CRG_CFG_OFF_SHIFT       0
#define CRG_CFG_OFF_MASK        0x1F

#define GS32F00xx_CRG_CFG(addr, mask, off)        ((addr) <<CRG_CFG_ADDR_SHIFT | (mask)<<CRG_CFG_MASK_SHIFT | (off))

/**
 *  
 *  The following are values that can be passed to SysCtl_enablePeripheral()
 *  and SysCtl_disablePeripheral() as the  peripheral parameter.
 *  
 */
typedef enum
{
    /*offset == 0x08*/
    SYSCTL_PERIPH_CLK_TIMER    = GS32F00xx_CRG_CFG(0x8, 0x01, 0),

    SYSCTL_PERIPH_CLK_SCI      = GS32F00xx_CRG_CFG(0x8, 0x07, 4),
    SYSCTL_PERIPH_CLK_SCIA     = GS32F00xx_CRG_CFG(0x8, 0x01, 4),
    SYSCTL_PERIPH_CLK_SCIB     = GS32F00xx_CRG_CFG(0x8, 0x01, 5),
    SYSCTL_PERIPH_CLK_SCIC     = GS32F00xx_CRG_CFG(0x8, 0x01, 6),

    SYSCTL_PERIPH_CLK_I2C      = GS32F00xx_CRG_CFG(0x8, 0x03, 8),
    SYSCTL_PERIPH_CLK_I2CA     = GS32F00xx_CRG_CFG(0x8, 0x01, 8),
    SYSCTL_PERIPH_CLK_I2CB     = GS32F00xx_CRG_CFG(0x8, 0x01, 9),

    SYSCTL_PERIPH_CLK_PMBUS    = GS32F00xx_CRG_CFG(0x8, 0x01, 10),

    SYSCTL_PERIPH_CLK_SPI      = GS32F00xx_CRG_CFG(0x8, 0x03, 12),
    SYSCTL_PERIPH_CLK_SPIA     = GS32F00xx_CRG_CFG(0x8, 0x01, 12),
    SYSCTL_PERIPH_CLK_SPIB     = GS32F00xx_CRG_CFG(0x8, 0x01, 13),

    SYSCTL_PERIPH_CLK_CAN      = GS32F00xx_CRG_CFG(0x8, 0x03, 16),
    SYSCTL_PERIPH_CLK_CANA     = GS32F00xx_CRG_CFG(0x8, 0x01, 16),
    SYSCTL_PERIPH_CLK_CANB     = GS32F00xx_CRG_CFG(0x8, 0x01, 17),

    SYSCTL_PERIPH_CLK_LIN      = GS32F00xx_CRG_CFG(0x8, 0x03, 20),
    SYSCTL_PERIPH_CLK_LINA     = GS32F00xx_CRG_CFG(0x8, 0x01, 20),
    SYSCTL_PERIPH_CLK_LINB     = GS32F00xx_CRG_CFG(0x8, 0x01, 21),

    SYSCTL_PERIPH_CLK_EXMC     = GS32F00xx_CRG_CFG(0x8, 0x01, 23),

    /*offset == 0x0C*/
    SYSCTL_PERIPH_CLK_CRC      = GS32F00xx_CRG_CFG(0xC, 0x01, 4),

    SYSCTL_PERIPH_CLK_EPG      = GS32F00xx_CRG_CFG(0xC, 0x01, 12),

    SYSCTL_PERIPH_CLK_ERAD     = GS32F00xx_CRG_CFG(0xC, 0x01, 16),

    SYSCTL_PERIPH_CLK_XBAR     = GS32F00xx_CRG_CFG(0xC, 0x01, 20),

    SYSCTL_PERIPH_CLK_AES      = GS32F00xx_CRG_CFG(0xC, 0x01, 22),

    /*offset == 0x10*/
    SYSCTL_PERIPH_CLK_CLB      = GS32F00xx_CRG_CFG(0x10, 0x07, 0),
    SYSCTL_PERIPH_CLK_CLB1     = GS32F00xx_CRG_CFG(0x10, 0x01, 0),
    SYSCTL_PERIPH_CLK_CLB2     = GS32F00xx_CRG_CFG(0x10, 0x01, 1),
    SYSCTL_PERIPH_CLK_CLBCOM   = GS32F00xx_CRG_CFG(0x10, 0x01, 2),

    SYSCTL_PERIPH_CLK_SDFM     = GS32F00xx_CRG_CFG(0x10, 0x07, 8),
    SYSCTL_PERIPH_CLK_SDFM1    = GS32F00xx_CRG_CFG(0x10, 0x01, 8),
    SYSCTL_PERIPH_CLK_SDFM2    = GS32F00xx_CRG_CFG(0x10, 0x01, 9),
    SYSCTL_PERIPH_CLK_SDFMCOM  = GS32F00xx_CRG_CFG(0x10, 0x01, 10),

    SYSCTL_PERIPH_CLK_EQEP     = GS32F00xx_CRG_CFG(0x10, 0x07, 12),
    SYSCTL_PERIPH_CLK_EQEP1    = GS32F00xx_CRG_CFG(0x10, 0x01, 12),
    SYSCTL_PERIPH_CLK_EQEP2    = GS32F00xx_CRG_CFG(0x10, 0x01, 13),
    SYSCTL_PERIPH_CLK_EQEP3    = GS32F00xx_CRG_CFG(0x10, 0x01, 14),

    SYSCTL_PERIPH_CLK_CLB_ENC     = GS32F00xx_CRG_CFG(0x10, 0x03, 16),
	SYSCTL_PERIPH_CLK_CLB_ENC1    = GS32F00xx_CRG_CFG(0x10, 0x01, 16),
	SYSCTL_PERIPH_CLK_CLB_ENC2    = GS32F00xx_CRG_CFG(0x10, 0x01, 17),

    /*offset == 0x14*/
    SYSCTL_PERIPH_CLK_EPWM     = GS32F00xx_CRG_CFG(0x14, 0x1FF, 0),
    SYSCTL_PERIPH_CLK_EPWM1    = GS32F00xx_CRG_CFG(0x14, 0x1, 0),
    SYSCTL_PERIPH_CLK_EPWM2    = GS32F00xx_CRG_CFG(0x14, 0x1, 1),
    SYSCTL_PERIPH_CLK_EPWM3    = GS32F00xx_CRG_CFG(0x14, 0x1, 2),
    SYSCTL_PERIPH_CLK_EPWM4    = GS32F00xx_CRG_CFG(0x14, 0x1, 3),
    SYSCTL_PERIPH_CLK_EPWM5    = GS32F00xx_CRG_CFG(0x14, 0x1, 4),
    SYSCTL_PERIPH_CLK_EPWM6    = GS32F00xx_CRG_CFG(0x14, 0x1, 5),
    SYSCTL_PERIPH_CLK_EPWM7    = GS32F00xx_CRG_CFG(0x14, 0x1, 6),
    SYSCTL_PERIPH_CLK_EPWM8    = GS32F00xx_CRG_CFG(0x14, 0x1, 7),
    SYSCTL_PERIPH_CLK_EPWMCOM  = GS32F00xx_CRG_CFG(0x14, 0x1, 8),

    /*offset == 0x18*/
    SYSCTL_PERIPH_CLK_ECAP     = GS32F00xx_CRG_CFG(0x18, 0x07, 0),
    SYSCTL_PERIPH_CLK_ECAP1    = GS32F00xx_CRG_CFG(0x18, 0x01, 0),
    SYSCTL_PERIPH_CLK_ECAP2    = GS32F00xx_CRG_CFG(0x18, 0x01, 1),
    SYSCTL_PERIPH_CLK_ECAP3    = GS32F00xx_CRG_CFG(0x18, 0x01, 2),

    /*offset == 0x1C*/
    SYSCTL_PERIPH_CLK_CMPSS    = GS32F00xx_CRG_CFG(0x1C, 0xF, 0),
    SYSCTL_PERIPH_CLK_CMPSS1   = GS32F00xx_CRG_CFG(0x1C, 0x1, 0),
    SYSCTL_PERIPH_CLK_CMPSS2   = GS32F00xx_CRG_CFG(0x1C, 0x1, 1),
    SYSCTL_PERIPH_CLK_CMPSS3   = GS32F00xx_CRG_CFG(0x1C, 0x1, 2),
    SYSCTL_PERIPH_CLK_CMPSS4   = GS32F00xx_CRG_CFG(0x1C, 0x1, 3),

    /*offset == 0x24*/
    SYSCTL_PERIPH_CLK_ADC      = GS32F00xx_CRG_CFG(0x24, 0x7, 0),
    SYSCTL_PERIPH_CLK_ADCA     = GS32F00xx_CRG_CFG(0x24, 0x1, 0),
    SYSCTL_PERIPH_CLK_ADCC     = GS32F00xx_CRG_CFG(0x24, 0x1, 1),
    SYSCTL_PERIPH_CLK_ADCCOM   = GS32F00xx_CRG_CFG(0x24, 0x1, 2),

    SYSCTL_PERIPH_CLK_TBCLKSYNC = 0xFFFFFFFF
} SysCtl_PeripheralPCLOCKCR;

/**
 *  
 *  The following are values that can be passed to SysCtl_resetPeripheral() as
 *  the  peripheral parameter.
 *  
 */
typedef enum
{
    /*offset == 0x100*/
    SYSCTL_PERIPH_RST_TIMER    = GS32F00xx_CRG_CFG(0x0, 0x01, 0),

    SYSCTL_PERIPH_RST_SCI      = GS32F00xx_CRG_CFG(0x0, 0x07, 4),
    SYSCTL_PERIPH_RST_SCIA     = GS32F00xx_CRG_CFG(0x0, 0x01, 4),
    SYSCTL_PERIPH_RST_SCIB     = GS32F00xx_CRG_CFG(0x0, 0x01, 5),
    SYSCTL_PERIPH_RST_SCIC     = GS32F00xx_CRG_CFG(0x0, 0x01, 6),

    SYSCTL_PERIPH_RST_I2C      = GS32F00xx_CRG_CFG(0x0, 0x03, 8),
    SYSCTL_PERIPH_RST_I2CA     = GS32F00xx_CRG_CFG(0x0, 0x01, 8),
    SYSCTL_PERIPH_RST_I2CB     = GS32F00xx_CRG_CFG(0x0, 0x01, 9),

    SYSCTL_PERIPH_RST_PMBUS    = GS32F00xx_CRG_CFG(0x0, 0x01, 10),

    SYSCTL_PERIPH_RST_SPI      = GS32F00xx_CRG_CFG(0x0, 0x03, 12),
    SYSCTL_PERIPH_RST_SPIA     = GS32F00xx_CRG_CFG(0x0, 0x01, 12),
    SYSCTL_PERIPH_RST_SPIB     = GS32F00xx_CRG_CFG(0x0, 0x01, 13),

    SYSCTL_PERIPH_RST_CAN      = GS32F00xx_CRG_CFG(0x0, 0x03, 16),
    SYSCTL_PERIPH_RST_CANA     = GS32F00xx_CRG_CFG(0x0, 0x01, 16),
    SYSCTL_PERIPH_RST_CANB     = GS32F00xx_CRG_CFG(0x0, 0x01, 17),

    SYSCTL_PERIPH_RST_LIN      = GS32F00xx_CRG_CFG(0x0, 0x03, 20),
    SYSCTL_PERIPH_RST_LINA     = GS32F00xx_CRG_CFG(0x0, 0x01, 20),
    SYSCTL_PERIPH_RST_LINB     = GS32F00xx_CRG_CFG(0x0, 0x01, 21),

    SYSCTL_PERIPH_RST_EXMC     = GS32F00xx_CRG_CFG(0x0, 0x01, 23),


    /*offset == 0x104*/

    SYSCTL_PERIPH_RST_CRC      = GS32F00xx_CRG_CFG(0x4, 0x01, 4),

    SYSCTL_PERIPH_RST_EPG      = GS32F00xx_CRG_CFG(0x4, 0x01, 12),

    SYSCTL_PERIPH_RST_ERAD     = GS32F00xx_CRG_CFG(0x4, 0x01, 16),

    SYSCTL_PERIPH_RST_XBAR     = GS32F00xx_CRG_CFG(0x4, 0x01, 20),

    SYSCTL_PERIPH_RST_AES      = GS32F00xx_CRG_CFG(0x4, 0x01, 22),

    /*offset == 0x108*/
    SYSCTL_PERIPH_RST_CLB      = GS32F00xx_CRG_CFG(0x8, 0x07, 0),
    SYSCTL_PERIPH_RST_CLB1     = GS32F00xx_CRG_CFG(0x8, 0x01, 0),
    SYSCTL_PERIPH_RST_CLB2     = GS32F00xx_CRG_CFG(0x8, 0x01, 1),
    SYSCTL_PERIPH_RST_CLBCOM   = GS32F00xx_CRG_CFG(0x8, 0x01, 2),

    SYSCTL_PERIPH_RST_SDFM     = GS32F00xx_CRG_CFG(0x8, 0x07, 8),
    SYSCTL_PERIPH_RST_SDFM1    = GS32F00xx_CRG_CFG(0x8, 0x01, 8),
    SYSCTL_PERIPH_RST_SDFM2    = GS32F00xx_CRG_CFG(0x8, 0x01, 9),
    SYSCTL_PERIPH_RST_SDFMCOM  = GS32F00xx_CRG_CFG(0x8, 0x01, 10),

    SYSCTL_PERIPH_RST_EQEP     = GS32F00xx_CRG_CFG(0x8, 0x07, 12),
    SYSCTL_PERIPH_RST_EQEP1    = GS32F00xx_CRG_CFG(0x8, 0x01, 12),
    SYSCTL_PERIPH_RST_EQEP2    = GS32F00xx_CRG_CFG(0x8, 0x01, 13),
    SYSCTL_PERIPH_RST_EQEP3    = GS32F00xx_CRG_CFG(0x8, 0x01, 14),

    SYSCTL_PERIPH_RST_CLB_ENC  = GS32F00xx_CRG_CFG(0x8, 0x03, 16),
	SYSCTL_PERIPH_RST_CLB_ENC1 = GS32F00xx_CRG_CFG(0x8, 0x01, 16),
	SYSCTL_PERIPH_RST_CLB_ENC2 = GS32F00xx_CRG_CFG(0x8, 0x01, 17),

    /*offset == 0x10C*/
    SYSCTL_PERIPH_RST_EPWM     = GS32F00xx_CRG_CFG(0xC, 0x1FF, 0),
    SYSCTL_PERIPH_RST_EPWM1    = GS32F00xx_CRG_CFG(0xC, 0x1, 0),
    SYSCTL_PERIPH_RST_EPWM2    = GS32F00xx_CRG_CFG(0xC, 0x1, 1),
    SYSCTL_PERIPH_RST_EPWM3    = GS32F00xx_CRG_CFG(0xC, 0x1, 2),
    SYSCTL_PERIPH_RST_EPWM4    = GS32F00xx_CRG_CFG(0xC, 0x1, 3),
    SYSCTL_PERIPH_RST_EPWM5    = GS32F00xx_CRG_CFG(0xC, 0x1, 4),
    SYSCTL_PERIPH_RST_EPWM6    = GS32F00xx_CRG_CFG(0xC, 0x1, 5),
    SYSCTL_PERIPH_RST_EPWM7    = GS32F00xx_CRG_CFG(0xC, 0x1, 6),
    SYSCTL_PERIPH_RST_EPWM8    = GS32F00xx_CRG_CFG(0xC, 0x1, 7),
    SYSCTL_PERIPH_RST_EPWMCOM  = GS32F00xx_CRG_CFG(0xC, 0x1, 8),

    /*offset == 0x110*/
    SYSCTL_PERIPH_RST_ECAP     = GS32F00xx_CRG_CFG(0x10, 0x07, 0),
    SYSCTL_PERIPH_RST_ECAP1    = GS32F00xx_CRG_CFG(0x10, 0x01, 0),
    SYSCTL_PERIPH_RST_ECAP2    = GS32F00xx_CRG_CFG(0x10, 0x01, 1),
    SYSCTL_PERIPH_RST_ECAP3    = GS32F00xx_CRG_CFG(0x10, 0x01, 2),

    /*offset == 0x114*/
    SYSCTL_PERIPH_RST_CMPSS    = GS32F00xx_CRG_CFG(0x14, 0xF, 0),
    SYSCTL_PERIPH_RST_CMPSS1   = GS32F00xx_CRG_CFG(0x14, 0x1, 0),
    SYSCTL_PERIPH_RST_CMPSS2   = GS32F00xx_CRG_CFG(0x14, 0x1, 1),
    SYSCTL_PERIPH_RST_CMPSS3   = GS32F00xx_CRG_CFG(0x14, 0x1, 2),
    SYSCTL_PERIPH_RST_CMPSS4   = GS32F00xx_CRG_CFG(0x14, 0x1, 3),

    /*offset == 0x11C*/
    SYSCTL_PERIPH_RST_ADC      = GS32F00xx_CRG_CFG(0x1C, 0x7, 0),
    SYSCTL_PERIPH_RST_ADCA     = GS32F00xx_CRG_CFG(0x1C, 0x1, 0),
    SYSCTL_PERIPH_RST_ADCC     = GS32F00xx_CRG_CFG(0x1C, 0x1, 1),
    SYSCTL_PERIPH_RST_ADCCOM   = GS32F00xx_CRG_CFG(0x1C, 0x1, 2),
} SysCtl_PeripheralSOFTPRES;


typedef enum {
    SYSCTL_PERIPH_ICG_BYPASS_ADC          = 1<<0,
    SYSCTL_PERIPH_ICG_BYPASS_APB          = 1<<3,
    SYSCTL_PERIPH_ICG_BYPASS_AHB          = 1<<4,
    SYSCTL_PERIPH_ICG_BYPASS_EPWM         = 1<<5,
    SYSCTL_PERIPH_ICG_BYPASS_ECAP         = 1<<6,
    SYSCTL_PERIPH_ICG_BYPASS_EQEP         = 1<<7,
    SYSCTL_PERIPH_ICG_BYPASS_CMPSS        = 1<<8,
    SYSCTL_PERIPH_ICG_BYPASS_CLB          = 1<<9,
    SYSCTL_PERIPH_ICG_BYPASS_XBAR         = 1<<10,
    SYSCTL_PERIPH_ICG_BYPASS_SDFM         = 1<<11,
    SYSCTL_PERIPH_ICG_BYPASS_ERAD         = 1<<13,
    SYSCTL_PERIPH_ICG_BYPASS_EPG          = 1<<14,
    SYSCTL_PERIPH_ICG_BYPASS_CRC          = 1<<16,
    SYSCTL_PERIPH_ICG_BYPASS_SEMA         = 1<<17,
    SYSCTL_PERIPH_ICG_BYPASS_DSP_DMAMUX   = 1<<19,
    SYSCTL_PERIPH_ICG_BYPASS_DSP_PARA     = 1<<22,
    SYSCTL_PERIPH_ICG_BYPASS_SCI_PARA     = 1<<24,
    SYSCTL_PERIPH_ICG_BYPASS_SPI_PARA     = 1<<26,
}SysCtl_PeripheralICGBYPASS;


/*
 * SUB_CRG_LOCK   Lock Control Register   Offset 0x0000
 *
 *  Register Field   Field Range
 *  sub_crg_lock     31 : 0
 *
 * Writing unlock value 0x5A5A5A5A unlocks all registers protected by lock.
 * Writing any other value locks all registers protected by lock.
 *
 */
__STATIC_INLINE void SysCtl_setSubCrgLock(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_SUB_CRG_LOCK) = 0x01;
}

__STATIC_INLINE void SysCtl_setSubCrgUnlock(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_SUB_CRG_LOCK) = 0x5A5A5A5A;
}

__STATIC_INLINE uint32_t SysCtl_getSubCrgLock(void)
{
    return HWREG(SUB_CRG_BASE + SUBCRG_CFG_SUB_CRG_LOCK);
}

/*
 * PARA_ICG_BYPASS   PARA_ICG_BYPASS Control Register 1   Offset 0x0030
 *
 *  Register Field   Field Range
 *  reserved         31:1
 *  para_icg_bypass  0
 *
 */
__STATIC_INLINE void SysCtl_setParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PARA_ICG_BYPASS) = bypass & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_getParaIcgBypass(void)
{
    return HWREG(SUB_CRG_BASE + SUBCRG_CFG_PARA_ICG_BYPASS) & 0x01;
}

/*
 * PERI_ICG_BYPASS   Peripheral Subsystem Module PARA_ICG_BYPASS Control Register 2   Offset 0x0034
 *
 *  Register Field              Field Range
 *  spi_para_icg_bypass              26
 *  reserved                         25
 *  uart_para_icg_bypass             24
 *  reserved                         23
 *  dsp_para_icg_bypass              22
 *  reserved                         21
 *  reserved                         20
 *  dsp_dmamux_para_icg_bypass       19
 *  reserved                         18
 *  sema_para_icg_bypass             17
 *  crc_para_icg_bypass              16
 *  reserved                         15
 *  epg_para_icg_bypass              14
 *  erad_para_icg_bypass             13
 *  reserved                         12
 *  sdfm_para_icg_bypass             11
 *  xbar_para_icg_bypass             10
 *  plu_para_icg_bypass              9
 *  cmpss_para_icg_bypass            8
 *  eqep_para_icg_bypass             7
 *  ecap_para_icg_bypass             6
 *  pwm_para_icg_bypass              5
 *  ahb1_comm_para_icg_bypass        4
 *  apb_comm_para_icg_bypass         3
 *  reserved                         2:1
 *  adc_para_icg_bypass              0
 *
 */

/* --- adc --- */
__STATIC_INLINE void SysCtl_setAdcParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<0)) | (bypass & 0x01);
}

__STATIC_INLINE uint32_t SysCtl_getAdcParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 0) & 0x01;
}

/* --- apb --- */
__STATIC_INLINE void SysCtl_setApbParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<3)) | ((bypass & 0x01) << 3);
}

__STATIC_INLINE uint32_t SysCtl_getApbParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 3) & 0x01;
}

/* --- ahb --- */
__STATIC_INLINE void SysCtl_setAhbParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<4)) | ((bypass & 0x01) << 4);
}

__STATIC_INLINE uint32_t SysCtl_getAhbParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 4) & 0x01;
}

/* --- epwm --- */
__STATIC_INLINE void SysCtl_setEpwmParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<5)) | ((bypass & 0x01) << 5);
}

__STATIC_INLINE uint32_t SysCtl_getEpwmParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 5) & 0x01;
}

/* --- ecap --- */
__STATIC_INLINE void SysCtl_setEcapParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<6)) | ((bypass & 0x01) << 6);
}

__STATIC_INLINE uint32_t SysCtl_getEcapParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 6) & 0x01;
}

/* --- eqep --- */
__STATIC_INLINE void SysCtl_setEqepParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<7)) | ((bypass & 0x01) << 7);
}

__STATIC_INLINE uint32_t SysCtl_getEqepParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 7) & 0x01;
}

/* --- cmpss --- */
__STATIC_INLINE void SysCtl_setCmpssParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<8)) | ((bypass & 0x01) << 8);
}

__STATIC_INLINE uint32_t SysCtl_getCmpssParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 8) & 0x01;
}

/* --- clb --- */
__STATIC_INLINE void SysCtl_setClbParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<9)) | ((bypass & 0x01) << 9);
}

__STATIC_INLINE uint32_t SysCtl_getClbParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 9) & 0x01;
}

/* --- xbar --- */
__STATIC_INLINE void SysCtl_setXbarParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<10)) | ((bypass & 0x01) << 10);
}

__STATIC_INLINE uint32_t SysCtl_getXbarParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 10) & 0x01;
}

/* --- sdfm --- */
__STATIC_INLINE void SysCtl_setSdfmParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<11)) | ((bypass & 0x01) << 11);
}

__STATIC_INLINE uint32_t SysCtl_getSdfmParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 11) & 0x01;
}

/* --- erad --- */
__STATIC_INLINE void SysCtl_setEradParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<13)) | ((bypass & 0x01) << 13);
}

__STATIC_INLINE uint32_t SysCtl_getEradParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 13) & 0x01;
}

/* --- epg --- */
__STATIC_INLINE void SysCtl_setEpgParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<14)) | ((bypass & 0x01) << 14);
}

__STATIC_INLINE uint32_t SysCtl_getEpgParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 14) & 0x01;
}

/* --- crc --- */
__STATIC_INLINE void SysCtl_setCrcParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<16)) | ((bypass & 0x01) << 16);
}

__STATIC_INLINE uint32_t SysCtl_getCrcParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 16) & 0x01;
}

/* --- sema --- */
__STATIC_INLINE void SysCtl_setSemaParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<17)) | ((bypass & 0x01) << 17);
}

__STATIC_INLINE uint32_t SysCtl_getSemaParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 17) & 0x01;
}


/* --- dsp dmamux --- */
__STATIC_INLINE void SysCtl_setDspDmamuxParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<19)) | ((bypass & 0x01) << 19);
}

__STATIC_INLINE uint32_t SysCtl_getDspDmamuxParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 19) & 0x01;
}

/* --- dsp para --- */
__STATIC_INLINE void SysCtl_setDspParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<22)) | ((bypass & 0x01) << 22);
}

__STATIC_INLINE uint32_t SysCtl_getDspParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 22) & 0x01;
}

/* --- SCI para --- */
__STATIC_INLINE void SysCtl_setSciParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<24)) | ((bypass & 0x01) << 24);
}

__STATIC_INLINE uint32_t SysCtl_getSciParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 24) & 0x01;
}

/* --- SPI para --- */
__STATIC_INLINE void SysCtl_setSpiParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<26)) | ((bypass & 0x01) << 26);
}

__STATIC_INLINE uint32_t SysCtl_getSpiParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 26) & 0x01;
}

/*
 * PERI_SRST1   SOFT_RSTN Control Register   Offset 0x0100
 *
 *  Register Field    Field Range
 *  reserved          31:24
 *  srst_exmc_n       23
 *  reserved          22
 *  srst_linb_n       21
 *  srst_lina_n       20
 *  reserved          19:18
 *  srst_canb_n       17
 *  srst_cana_n       16
 *  reserved          15:14
 *  srst_spib_n       13
 *  srst_spia_n       12
 *  reserved          11
 *  srst_pmbus_n      10
 *  srst_i2cb_n       9
 *  srst_i2ca_n       8
 *  reserved          7
 *  srst_uartc_n      6
 *  srst_uartb_n      5
 *  srst_uarta_n      4
 *  reserved          3:1
 *  srst_timer_n      0
 *
 */
/* --- timer ---*/
__STATIC_INLINE void SysCtl_resetCpuTimer(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) &= ~(1<<0);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) |= (1<<0);
}

/* --- sci --- */
__STATIC_INLINE void SysCtl_resetSci(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) &= ~(7<<4);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) |= (7<<4);
}

__STATIC_INLINE void SysCtl_resetScia(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) &= ~(1<<4);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) |= (1<<4);
}

__STATIC_INLINE void SysCtl_resetScib(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) &= ~(1<<5);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) |= (1<<5);
}

__STATIC_INLINE void SysCtl_resetScic(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) &= ~(1<<6);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) |= (1<<6);
}

/* --- i2c --- */
__STATIC_INLINE void SysCtl_resetI2c(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) &= ~(3<<8);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) |= (3<<8);
}

__STATIC_INLINE void SysCtl_resetI2ca(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) &= ~(1<<8);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) |= (1<<8);
}

__STATIC_INLINE void SysCtl_resetI2cb(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) &= ~(1<<9);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) |= (1<<9);
}

/* --- pmbus --- */
__STATIC_INLINE void SysCtl_resetPmbus(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) &= ~(1<<10);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) |= (1<<10);
}

/* --- spi --- */
__STATIC_INLINE void SysCtl_resetSpi(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) &= ~(3<<12);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) |= (3<<12);
}

__STATIC_INLINE void SysCtl_resetSpia(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) &= ~(1<<12);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) |= (1<<12);
}

__STATIC_INLINE void SysCtl_resetSpib(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) &= ~(1<<13);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) |= (1<<13);
}

/* --- can --- */
__STATIC_INLINE void SysCtl_resetCan(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) &= ~(3<<16);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) |= (3<<16);
}

__STATIC_INLINE void SysCtl_resetCana(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) &= ~(1<<16);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) |= (1<<16);
}

__STATIC_INLINE void SysCtl_resetCanb(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) &= ~(1<<17);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) |= (1<<17);
}

/* --- lin --- */
__STATIC_INLINE void SysCtl_resetLin(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) &= ~(3<<20);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) |= (3<<20);
}

__STATIC_INLINE void SysCtl_resetLina(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) &= ~(1<<20);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) |= (1<<20);
}

__STATIC_INLINE void SysCtl_resetLinb(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) &= ~(1<<21);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) |= (1<<21);
}

/* --- exmc --- */
__STATIC_INLINE void SysCtl_resetExmc(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) &= ~(1<<23);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST1) |= (1<<23);
}

/*
 * PERI_SRST2   SOFT_RSTN Control Register   Offset 0x0104
 *
 *  Register Field    Field Range
 *  reserved          31:23
 *  srst_aes_n        22
 *  reserved          21
 *  srst_xbar_n       20
 *  reserved          19:17
 *  srst_erad_n       16
 *  reserved          15:13
 *  srst_epg_n        12
 *  reserved          11:5
 *  srst_crc_n        4
 *  reserved          3:0
 *
 */

/* --- crc ---*/
__STATIC_INLINE void SysCtl_resetCrc(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST2) &= ~(1<<4);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST2) |= (1<<4);
}

/* --- epg ---*/
__STATIC_INLINE void SysCtl_resetEpg(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST2) &= ~(1<<12);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST2) |= (1<<12);
}

/* --- erad ---*/
__STATIC_INLINE void SysCtl_resetErad(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST2) &= ~(1<<16);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST2) |= (1<<16);
}

/* --- xbar ---*/
__STATIC_INLINE void SysCtl_resetXbar(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST2) &= ~(1<<20);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST2) |= (1<<20);
}

/* --- aes ---*/
__STATIC_INLINE void SysCtl_resetAes(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST2) &= ~(1<<22);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST2) |= (1<<22);
}

/*
 * PERI_SRST3   SOFT_RSTN Control Register   Offset 0x0108
 *
 *  Register Field    Field Range
 *  reserved          31:18
 *  srst_clb_enc_n    17:16
 *  reserved          15
 *  srst_qep_n        14:12
 *  reserved          11
 *  srst_sdfm_n       10:8
 *  reserved          7:5
 *  srst_plu_n        2:0
 *
 */
/* --- Clb ---*/
__STATIC_INLINE void SysCtl_resetClb(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) &= ~(0x07<<0);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) |= (0x07<<0);
}

__STATIC_INLINE void SysCtl_resetClb1(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) &= ~(0x1<<0);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) |= (0x1<<0);
}

__STATIC_INLINE void SysCtl_resetClb2(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) &= ~(0x1<<1);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) |= (0x1<<1);
}

__STATIC_INLINE void SysCtl_resetClbcom(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) &= ~(0x1<<2);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) |= (0x1<<2);
}

/* --- Sdfm ---*/
__STATIC_INLINE void SysCtl_resetSdfm(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) &= ~(0x7<<8);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) |= (0x7<<8);
}

__STATIC_INLINE void SysCtl_resetSdfm1(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) &= ~(0x1<<8);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) |= (0x1<<8);
}

__STATIC_INLINE void SysCtl_resetSdfm2(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) &= ~(0x1<<9);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) |= (0x1<<9);
}

__STATIC_INLINE void SysCtl_resetSdfmcom(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) &= ~(0x1<<10);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) |= (0x1<<10);
}

/* --- eqep ---*/
__STATIC_INLINE void SysCtl_resetEqep(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) &= ~(0x7<<12);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) |= (0x7<<12);
}

__STATIC_INLINE void SysCtl_resetEqep1(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) &= ~(0x1<<12);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) |= (0x1<<12);
}

__STATIC_INLINE void SysCtl_resetEqep2(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) &= ~(0x1<<13);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) |= (0x1<<13);
}

__STATIC_INLINE void SysCtl_resetEqep3(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) &= ~(0x1<<14);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) |= (0x1<<14);
}

/* --- clb_enc ---*/
__STATIC_INLINE void SysCtl_resetClbEnc(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) &= ~(0x3<<16);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) |= (0x3<<16);
}

__STATIC_INLINE void SysCtl_resetClbEnc1(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) &= ~(0x1<<16);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) |= (0x1<<16);
}

__STATIC_INLINE void SysCtl_resetClbEnc2(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) &= ~(0x1<<17);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST3) |= (0x1<<17);
}




/*
 * PERI_SRST4   SOFT_RSTN Control Register   Offset 0x010C
 *
 *  Register Field    Field Range
 *  reserved          31:13
 *  srst_pwm_n        8:0
 *
 */
/* --- epwm ---*/
__STATIC_INLINE void SysCtl_resetEpwm(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST4) &= ~(0x1FF<<0);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST4) |= (0x1FF<<0);
}

/* index from 1 to 8 */
__STATIC_INLINE void SysCtl_resetEpwmx(uint32_t index)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST4) &= ~(1<<(index-1));
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST4) |= (1<<(index-1));
}

__STATIC_INLINE void SysCtl_resetEpwmcom(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST4) &= ~(1<<8);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST4) |= (1<<8);
}

/*
 * PERI_SRST5   SOFT_RSTN Control Register   Offset 0x0110
 *
 *  Register Field    Field Range
 *  reserved          31:7
 *  srst_cap_n        2:0
 *
 */
/* --- ecap ---*/
__STATIC_INLINE void SysCtl_resetEcap(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST5) &= ~(0x7<<0);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST5) |= (0x7<<0);
}

/* index from 1 to 3 */
__STATIC_INLINE void SysCtl_resetEcapx(uint32_t index)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST5) &= ~(1<<(index-1));
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST5) |= (1<<(index-1));
}

/*
 * PERI_SRST6   SOFT_RSTN Control Register   Offset 0x0114
 *
 *  Register Field    Field Range
 *  reserved          31:7
 *  srst_cmp_n        3:0
 *
 */
/* --- cmpss ---*/
__STATIC_INLINE void SysCtl_resetCmpss(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST6) &= ~(0xF<<0);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST6) |= (0xF<<0);
}

/* index from 1 to 4 */
__STATIC_INLINE void SysCtl_resetCmpssx(uint32_t index)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST6) &= ~(1<<(index-1));
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST6) |= (1<<(index-1));
}

/*
 * PERI_SRST8   SOFT_RSTN Control Register   Offset 0x011C
 *
 *  Register Field    Field Range
 *  reserved          31:4
 *  srst_adc_n        2:0
 *
 */
/* --- adc ---*/
__STATIC_INLINE void SysCtl_resetAdc(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST8) &= ~(0x7<<0);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST8) |= (0x7<<0);
}

__STATIC_INLINE void SysCtl_resetAdca(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST8) &= ~(1<<0);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST8) |= (1<<0);
}

__STATIC_INLINE void SysCtl_resetAdcc(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST8) &= ~(1<<1);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST8) |= (1<<1);
}

__STATIC_INLINE void SysCtl_resetAdccom(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST8) &= ~(1<<2);
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_SRST8) |= (1<<2);
}

/**
 * @brief   Enables the clock to a peripheral.
 *
 * @param [in]   peripheral is the peripheral to enable, see type SysCtl_PeripheralPCLOCKCR.
 *
 *  Peripherals are enabled with this function.  At power-up, all peripherals
 *  are disabled; they must be enabled in order to operate or respond to
 *  register reads/writes.
 *
 *  @remarks 
 *  - Note that there should be at least 5 cycles delay between enabling the
 *    peripheral clock and accessing the peripheral registers. The delay should be
 *    added by the user if the peripheral is accessed immediately after this
 *    function call.
 * @return  None
 */
__STATIC_INLINE void SysCtl_enablePeripheral(SysCtl_PeripheralPCLOCKCR peripheral)
{
    if(peripheral != SYSCTL_PERIPH_CLK_TBCLKSYNC) {
        HWREG(SUB_CRG_BASE + ((peripheral>>CRG_CFG_ADDR_SHIFT) & CRG_CFG_ADDR_MASK)) |=
            ((peripheral >> CRG_CFG_MASK_SHIFT) & CRG_CFG_MASK_MASK) << (peripheral & CRG_CFG_OFF_MASK);
    } else {
        HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_TBCLKSYNC) |= 1<<18;
    }
}

/**
 * @brief   Disables a peripheral.
 *
 * @param [in]   peripheral is the peripheral to disable.
 *
 *  Peripherals are disabled with this function.  Once disabled, they will not
 *  operate or respond to register reads/writes. But the peripheral registers are not reset.
 *
 * @return  None
 */
__STATIC_INLINE void SysCtl_disablePeripheral(SysCtl_PeripheralPCLOCKCR peripheral)
{
    if(peripheral != SYSCTL_PERIPH_CLK_TBCLKSYNC) {
        HWREG(SUB_CRG_BASE + ((peripheral>>CRG_CFG_ADDR_SHIFT) & CRG_CFG_ADDR_MASK)) &=
            ~(((peripheral >> CRG_CFG_MASK_SHIFT) & CRG_CFG_MASK_MASK) << (peripheral & CRG_CFG_OFF_MASK));
    } else {
        HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_TBCLKSYNC) &= ~(1<<18);
    }
}


/**
 * @brief  peripheral icg bypass enable.
 *
 * @param [in]   peripheral is the peripheral to enable, see type SysCtl_PeripheralPCLOCKCR.
 *
 * @return  None
 */
__STATIC_INLINE void SysCtl_enablePeripheralIcgBypass(SysCtl_PeripheralICGBYPASS peripheral)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) |= peripheral;
}

/**
 * @brief   peripheral icg bypass disable
 *
 * @param [in]   peripheral is the peripheral to disable.
 *
 *
 * @return  None
 */
__STATIC_INLINE void SysCtl_disablePeripheralIcgBypass(SysCtl_PeripheralICGBYPASS peripheral)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) &= ~peripheral;
}


/**
 * 
 *  @brief Resets a peripheral
 * 
 *  @param [in] peripheral is the peripheral to reset.
 *  bit0~bit7 is the peripheral number, bit8~bit15 is the bit position in the register
 * 
 *  This function uses the Sysctl SRST_XXXX_N registers to reset a specified peripheral.
 *  Module registers will be returned to their reset states.
 *  Writing 0 put the peripheral into reset state.
 *  Writing 1 put the peripheral into release state.
 *  After reset and release, the peripheral registers returns to default values.
 * 
 *  @return None.
 * 
 */
__STATIC_INLINE void SysCtl_resetPeripheral(SysCtl_PeripheralSOFTPRES peripheral)
{
    HWREG(SUB_CRG_BASE + ((peripheral>>CRG_CFG_ADDR_SHIFT) & CRG_CFG_ADDR_MASK) + 0x100) &=
            ~(((peripheral >> CRG_CFG_MASK_SHIFT) & CRG_CFG_MASK_MASK) << (peripheral & CRG_CFG_OFF_MASK));

    HWREG(SUB_CRG_BASE + ((peripheral>>CRG_CFG_ADDR_SHIFT) & CRG_CFG_ADDR_MASK) + 0x100) |=
            (((peripheral >> CRG_CFG_MASK_SHIFT) & CRG_CFG_MASK_MASK) << (peripheral & CRG_CFG_OFF_MASK));
}

/*
 * This function is used to select the function of an analog pin.
 * There are two types of pins:
 *     type1 - analog input pin, function is selected by ASEL register in asysctl;
 *             ADC_A0~ADC_C14, CMP1, CMP3, CMP5, CMP7,
 *     type2 - AGPIO pin, function is selected by AGPIO_SEL register in sysctl_h;
 *             AGPIO13/20/21
 * The input parameter can be found in "asysctl.h" - Analog pin configuration macro definition
 * The input parameter is a 32bit integer combined of the following 3 parts:
 * bit20 - select analog pin or AGPIO pin;
 * bit16~19 is ASEL offset in asysctl or AGPIO_SEL offset in sysctl_h;
 * bit0~3 is ASEL value or AGPIO_SEL value
 * These registers base address is 0x40350200 - A0_SEL register for analog pin;
 * These registers base address is 0x40370540 - AGPIO13_SEL register for AGPIO pin;
 *
 */
__STATIC_INLINE void SysCtl_setAnalogPin(uint32_t analogPinCfg)
{
//    if((analogPinCfg & 0x01000000) == 0)
//        HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_A0_SEL + ((analogPinCfg>>16)*4)) = (analogPinCfg&0xFFFF);
//    else
//        HWREG(SYSCTL_H_BASE + SYSCTL_H_O_AGPIO13_SEL + ((analogPinCfg>>16)&0xFF)*4) = ((analogPinCfg & 0xFF) | (0x08));
}

#ifdef __cplusplus
}
#endif

#endif

#endif /* DEVICE_DRIVERLIB_SYSCTL_CRG_H_ */
