/*
 * uart_v30.h
 *
 *  Created on: 2025 May 6
 *      Author: hpec
 */

#ifndef DRIVERLIB_UART_V30_H_
#define DRIVERLIB_UART_V30_H_

#include "gs32_version.h"
#include "inc/hw_types.h"
#include "inc/hw_uart.h"
#include "debug.h"
#include "stdbool.h"

#if IS_GS32F00xx(0x30) || IS_GS32F3xx(0x23)

/**
 * @brief Interrupt Bitfield Mask
 */
#define SCI_INT_RXERR          0x01U
#define SCI_INT_RXRDY_BRKDT    0x02U
#define SCI_INT_TXRDY          0x04U
#define SCI_INT_TXFF           0x08U
#define SCI_INT_RXFF           0x10U
#define SCI_INT_FE             0x20U
#define SCI_INT_OE             0x40U
#define SCI_INT_PE             0x80U
#define SCI_INT_RXTIMEOUT      0x100U

/**
 * @brief Configure SCI data bit length
 */
#define SCI_CONFIG_WLEN_MASK   0x0007U
#define SCI_CONFIG_WLEN_8      0x0007U
#define SCI_CONFIG_WLEN_7      0x0006U
#define SCI_CONFIG_WLEN_6      0x0005U
#define SCI_CONFIG_WLEN_5      0x0004U

/**
 * @brief Configure stop bit length
 */
#define SCI_CONFIG_STOP_MASK   0x0080U
#define SCI_CONFIG_STOP_ONE    0x0000U
#define SCI_CONFIG_STOP_TWO    0x0080U

/**
 * @brief Configure parity check type
 */
#define SCI_CONFIG_PAR_MASK    0x0060U
typedef enum
{
    SCI_CONFIG_PAR_NONE = 0x0000U,
    SCI_CONFIG_PAR_EVEN = 0x0060U,
    SCI_CONFIG_PAR_ODD  = 0x0020U
} SCI_ParityType;

/**
 * @brief  The transmit FIFO generates an interrupt whenever the FIFO status
            bits (TXFFST4-0) are less than or equal to the Tx FIFO level configuration.
 */
typedef enum
{
    SCI_FIFO_TX0  = 0x0000U,
    SCI_FIFO_TX1  = 0x0001U,
    SCI_FIFO_TX2  = 0x0002U,
    SCI_FIFO_TX3  = 0x0003U,
    SCI_FIFO_TX4  = 0x0004U,
    SCI_FIFO_TX5  = 0x0005U,
    SCI_FIFO_TX6  = 0x0006U,
    SCI_FIFO_TX7  = 0x0007U,
    SCI_FIFO_TX8  = 0x0008U,
    SCI_FIFO_TX9  = 0x0009U,
    SCI_FIFO_TX10 = 0x000AU,
    SCI_FIFO_TX11 = 0x000BU,
    SCI_FIFO_TX12 = 0x000CU,
    SCI_FIFO_TX13 = 0x000DU,
    SCI_FIFO_TX14 = 0x000EU,
    SCI_FIFO_TX15 = 0x000FU,
    SCI_FIFO_TX16 = 0x0010U 
} SCI_TxFIFOLevel;

/**
 * @brief The receive FIFO generates an interrupt whenever the FIFO status 
            bits (RXFFST4-0) are greater than or equal to the FIFO level configuration.
 */
typedef enum
{
    SCI_FIFO_RX0  = 0x0000U, /* Receive interrupt empty */
    SCI_FIFO_RX1  = 0x0001U, /* Receive interrupt 1/16 full */
    SCI_FIFO_RX2  = 0x0002U, /* Receive interrupt 2/16 full */
    SCI_FIFO_RX3  = 0x0003U, /* Receive interrupt 3/16 full */
    SCI_FIFO_RX4  = 0x0004U, /* Receive interrupt 4/16 full */
    SCI_FIFO_RX5  = 0x0005U, /* Receive interrupt 5/16 full */
    SCI_FIFO_RX6  = 0x0006U, /* Receive interrupt 6/16 full */
    SCI_FIFO_RX7  = 0x0007U, /* Receive interrupt 7/16 full */
    SCI_FIFO_RX8  = 0x0008U, /* Receive interrupt 8/16 full */
    SCI_FIFO_RX9  = 0x0009U, /* Receive interrupt 9/16 full */
    SCI_FIFO_RX10 = 0x000AU, /* Receive interrupt 10/16 full */
    SCI_FIFO_RX11 = 0x000BU, /* Receive interrupt 11/16 full */
    SCI_FIFO_RX12 = 0x000CU, /* Receive interrupt 12/16 full */
    SCI_FIFO_RX13 = 0x000DU, /* Receive interrupt 13/16 full */
    SCI_FIFO_RX14 = 0x000EU, /* Receive interrupt 14/16 full */
    SCI_FIFO_RX15 = 0x000FU, /* Receive interrupt 15/16 full */
    SCI_FIFO_RX16 = 0x0010U  /* Receive interrupt full */
} SCI_RxFIFOLevel;

/**
 * @brief SCI receive status bitfield mask.
 */
#define SCI_RXSTATUS_WAKE       0x0002U
#define SCI_RXSTATUS_PARITY     0x0004U
#define SCI_RXSTATUS_OVERRUN    0x0008U
#define SCI_RXSTATUS_FRAMING    0x0010U
#define SCI_RXSTATUS_BREAK      0x0020U
#define SCI_RXSTATUS_READY      0x0040U
#define SCI_RXSTATUS_ERROR      0x0080U

#ifdef DEBUG
static inline bool
SCI_isBaseValid(uint32_t base)
{
    return(
           (base == SCIA_BASE) ||
           (base == SCIB_BASE)
          );
}
#endif

/**
 * @brief Sets the type of parity. parity specifies the type of parity to use.
 *        SCI_CONFIG_PAR_NONE/ SCI_CONFIG_PAR_EVEN/ SCI_CONFIG_PAR_ODD
 * @param [in] base is the base address of the SCI port.
 *
 * @return None
 */
static inline void
SCI_setParityMode(uint32_t base, SCI_ParityType parity)
{
    ASSERT(SCI_isBaseValid(base));
    ASSERT((parity == SCI_CONFIG_PAR_NONE) ||
    	   (parity == SCI_CONFIG_PAR_ODD) ||
    	   (parity == SCI_CONFIG_PAR_EVEN));

    /*Set the parity mode. */
    HWREGH(base + SCI_O_CCR) = ((HWREGH(base + SCI_O_CCR) &
                                 ~(SCI_CONFIG_PAR_MASK)) | (uint16_t)parity);
}

/**
 * @brief gets the type of parity.
 * @param [in] base is the base address of the SCI port.
 *
 * @return SCI_ParityType
 *  SCI_ParityType:
 *      SCI_CONFIG_PAR_NONE
 *      SCI_CONFIG_PAR_EVEN
 *      SCI_CONFIG_PAR_ODD
 */
static inline SCI_ParityType
SCI_getParityMode(uint32_t base)
{
    uint16_t parity;

    ASSERT(SCI_isBaseValid(base));

    /* Return the current parity setting. */
    parity = (HWREGH(base + SCI_O_CCR) & (SCI_CONFIG_PAR_MASK));

    return((SCI_ParityType)parity);
}

/**
 * @brief set multiprocess mode.
 * @param [in] base is the base address of the sci port.
 *
 * @return None.
 */
static inline void
SCI_setAddrMultiProcessorMode(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Enable the address-bit mode protocol */
    HWREGH(base + SCI_O_CCR) |= SCI_CCR_ADDRIDLE_MODE;
}

/**
 * @brief Sets the multiprocessor protocol to idle-line mode.
 * @param [in] base is the base address of the SCI port.
 *
 * @return None
 */
static inline void
SCI_setIdleMultiProcessorMode(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Disable the address-bit mode protocol */
    HWREGH(base + SCI_O_CCR) &= ~SCI_CCR_ADDRIDLE_MODE;
}

/**
 * @brief Locks Autobaud. This function performs an auto-baud detect for the SCI.
 * @param [in] base is the base address of the SCI port.
 *
 * @return None
 */
static inline void
SCI_lockAutobaud(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Prime the baud register */
    HWREGH(base + SCI_O_HBAUD) = 0x0U;
    HWREGH(base + SCI_O_LBAUD) = 0x1U;

    /* Prepare for autobaud detection.
       Set the CDC bit to enable autobaud detection and clear the ABD bit. */
    HWREGH(base + SCI_O_FFCT) |= SCI_FFCT_CDC;
    HWREGH(base + SCI_O_FFCT) |= SCI_FFCT_ABDCLR;
    HWREGH(base + SCI_O_FFCT) &= ~SCI_FFCT_ABDCLR;

    /* Wait until we correctly read an 'A' or 'a' and lock */
    while((HWREGH(base + SCI_O_FFCT) & SCI_FFCT_ABD) != SCI_FFCT_ABD)
    {
    }

    /* After autobaud lock, clear the ABD and CDC bits */
    HWREGH(base + SCI_O_FFCT) |= SCI_FFCT_ABDCLR;
    HWREGH(base + SCI_O_FFCT) &= ~SCI_FFCT_CDC;
}

/**
 * @brief Determines if the FIFO enhancement is enabled.
 * @param [in] base is the base address of the SCI port.
 *
 * @return
 * 		true if the FIFO enhancement is enabled
 * 		false if the FIFO enhancement is disabled.
 */
static inline bool
SCI_isFIFOEnabled(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

     /* Return true if the FIFO is enabled and false if it is disabled. */
    return(((HWREGH(base + SCI_O_FFTX) & SCI_FFTX_SCIFFENA) == SCI_FFTX_SCIFFENA) ? true : false);
}

/**
 * @brief Enables the transmit and receive FIFOs. This functions enables the transmit and receive FIFOs in the SCI.
 * @param [in] base is the base address of the SCI port.
 *
 * @return None
 */
static inline void
SCI_enableFIFO(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Enable the FIFO. */
    HWREGH(base + SCI_O_FFTX) |= SCI_FFTX_SCIRST;
    HWREGH(base + SCI_O_FFTX) |= SCI_FFTX_SCIFFENA | SCI_FFTX_TXFIFORESET;
    HWREGH(base + SCI_O_FFRX) |= SCI_FFRX_RXFIFORESET;
}

/**
 * @brief Disables the transmit and receive FIFOs. This functions disables the transmit and receive FIFOs in the SCI.
 * @param [in] base is the base address of the SCI port.
 * 
 * @return None.
 */
static inline void
SCI_disableFIFO(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Disable the FIFO. */
    HWREGH(base + SCI_O_FFTX) &= ~SCI_FFTX_SCIFFENA;
}

/**
 * @brief Resets the receive FIFO. This functions resets the receive FIFO of the SCI.
 * @param [in] base is the base address of the SCI port.
 *
 * @return None
 */
static inline void
SCI_resetRxFIFO(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Reset the specified FIFO. */
    HWREGH(base + SCI_O_FFRX) &= ~SCI_FFRX_RXFIFORESET;
    HWREGH(base + SCI_O_FFRX) |= SCI_FFRX_RXFIFORESET;
}

/**
 * @brief Resets the transmit FIFO.
 * @param [in] base is the base address of the SCI port.
 *
 * @return None.
 */
static inline void
SCI_resetTxFIFO(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Reset the specified FIFO. */
    HWREGH(base + SCI_O_FFTX) &= ~SCI_FFTX_TXFIFORESET;
    HWREGH(base + SCI_O_FFTX) |= SCI_FFTX_TXFIFORESET;
}

/**
 * @brief Resets the SCI Transmit and Receive Channels
 * @param [in] base is the base address of the SCI port.
 * 
 * @return None
 */
static inline void
SCI_resetChannels(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Reset the Tx and Rx Channels */
    HWREGH(base + SCI_O_FFTX) &= ~SCI_FFTX_SCIRST;
    HWREGH(base + SCI_O_FFTX) |= SCI_FFTX_SCIRST;
}


/**
 * @brief Sets the FIFO interrupt level at which interrupts are generated.
 * @param [in] base is the base address of the SCI port.
 * @param [in] txLevel is the transmit FIFO interrupt level.
 * @param [in] rxLevel is the receive FIFO interrupt level.
 * - txLevel:
 *    SCI_FIFO_TX0, SCI_FIFO_TX1, SCI_FIFO_TX2, ... , SCI_FIFO_TX15.
 * - rxLevel:
 *    SCI_FIFO_RX0, SCI_FIFO_RX1, SCI_FIFO_RX2, ... , SCI_FIFO_RX15.
 *
 * @return None.
 */
static inline void
SCI_setFIFOInterruptLevel(uint32_t base, SCI_TxFIFOLevel txLevel,
                          SCI_RxFIFOLevel rxLevel)
{
    ASSERT(SCI_isBaseValid(base));
    ASSERT(txLevel >= SCI_FIFO_TX0 && txLevel <= SCI_FIFO_TX16);
    ASSERT(rxLevel >= SCI_FIFO_RX0 && txLevel <= SCI_FIFO_RX16);

    if (SCI_isFIFOEnabled(base) == false)
    {
    	SCI_enableFIFO(base);
    }

    /* Set the FIFO interrupt levels. */
    HWREGH(base + SCI_O_FFTX) = (HWREGH(base + SCI_O_FFTX) &
                                 (~SCI_FFTX_TXFFIL_M)) | (uint16_t)txLevel;
    HWREGH(base + SCI_O_FFRX) = (HWREGH(base + SCI_O_FFRX) &
                                 (~SCI_FFRX_RXFFIL_M)) | (uint16_t)rxLevel;
}

/**
 * @brief Gets the FIFO interrupt level at which interrupts are generated.
 * @param [in] base is the base address of the SCI port.
 * @param [out] txLevel is a pointer to storage for the transmit FIFO interrupt level
 * @param [out] rxLevel is a pointer to storage for the receive FIFO interrupt level
 *  txLevel:
 *  SCI_FIFO_TX0, SCI_FIFO_TX1, SCI_FIFO_TX2, ... , SCI_FIFO_TX15.
 *  rxLevel:
 *  SCI_FIFO_RX0, SCI_FIFO_RX1, SCI_FIFO_RX2, ... , SCI_FIFO_RX15.
 *
 * @return None
 */
static inline void
SCI_getFIFOInterruptLevel(uint32_t base, SCI_TxFIFOLevel *txLevel,
                          SCI_RxFIFOLevel *rxLevel)
{
    ASSERT(SCI_isBaseValid(base));

    /* If it is not in FIFO mode, return 0. */
    if (SCI_isFIFOEnabled(base) == false)
    {
    	*txLevel = (SCI_TxFIFOLevel) SCI_FIFO_TX0;

    	*rxLevel = (SCI_RxFIFOLevel) SCI_FIFO_RX0;
    }

    /* Extract the transmit and receive FIFO levels. */
    *txLevel = (SCI_TxFIFOLevel)(HWREGH(base + SCI_O_FFTX) &
                                 SCI_FFTX_TXFFIL_M);
    *rxLevel = (SCI_RxFIFOLevel)(HWREGH(base + SCI_O_FFRX) &
                                 SCI_FFRX_RXFFIL_M);
}

/**
 * @brief Gets the current configuration of a SCI.
 * @param [in] base is the base address of the SCI port.
 * @param [in] lspclkHz is the rate of the clock supplied to the SCI module.
 * @param [out] baud is a pointer to storage for the baud rate.
 * @param [out] config is a pointer to storage for the data format.
 *
 * @return None
 */
static inline void
SCI_getConfig(uint32_t base, uint32_t lspclkHz, uint32_t *baud,
              uint32_t *config)
{
    uint8_t H_IntDiv = 0;
    uint8_t L_IntDiv = 0;
    uint8_t FracDiv = 0;
    uint32_t IntDiv = 0;
    uint32_t Divider = 0;
	ASSERT(SCI_isBaseValid(base));

	/* Obtain the high 8 bits of the integer part of the baud rate divide */
	H_IntDiv = HWREGH(base + SCI_O_HBAUD);

	/* Obtain the low 8 bits of the integer part of the baud rate divide */
	L_IntDiv = HWREGH(base + SCI_O_LBAUD);

	/* Get the fractional part of the baud rate divider */
	FracDiv = HWREGH(base + SCI_O_DLF);

	IntDiv = (H_IntDiv << 8U) | L_IntDiv;

	IntDiv = (IntDiv + 1) / 2;

	Divider = (IntDiv << 6 | FracDiv);

	*baud = (lspclkHz * 4U) / Divider;

    /* Get the parity, data length, and number of stop bits. */
    *config = (uint32_t)HWREGH(base + SCI_O_CCR) & (SCI_CONFIG_PAR_MASK |
                                                    SCI_CONFIG_STOP_MASK |
                                                    SCI_CONFIG_WLEN_MASK);
}

/**
 * @brief Enables transmitting and receiving.
 *        Enables SCI by taking SCI out of the software reset. Sets the TXENA, and
 *        RXENA bits which enables transmit and receive.
 * @param [in] base is the base address of the SCI port.
 *
 * @return None
 */
static inline void
SCI_enableModule(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Enable RX, TX, and the SCI.*/
    HWREGH(base + SCI_O_CTL1) |= (SCI_CTL1_TXENA | SCI_CTL1_RXENA |
                                  SCI_CTL1_SWRESET);
}

/**
 * @brief Disables transmitting and receiving.
 *        Clears the SCIEN, TXE, and RXE bits. The user should ensure that all the
 *        data has been sent before disable the module during transmission.
 * @param [in] base is the base address of the SCI port.
 *
 * @reutrn None
 */
static inline void
SCI_disableModule(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Disable the FIFO. */
    HWREGH(base + SCI_O_FFTX) &= ~(SCI_FFTX_SCIFFENA);

    /* Disable the SCI. */
    HWREGH(base + SCI_O_CTL1) &= ~(SCI_CTL1_TXENA | SCI_CTL1_RXENA);
}

/**
 * @brief Enables transmitting.
 * @param [in] base is the base address of the SCI port.
 * @detail Enables SCI by taking SCI out of the software reset. Sets the TXENA bit
 *  	   which enables transmit.
 *
 * @return None.
 */
static inline void
SCI_enableTxModule(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Enable TX and the SCI. */
    HWREGH(base + SCI_O_CTL1) |= (SCI_CTL1_TXENA | SCI_CTL1_SWRESET);
}

/**
 * @brief Disables transmitting. Clears the TXENA bit which disables transmit.
 * @param [in] base is the base address of the SCI port.
 *
 * @return None.
 */
static inline void
SCI_disableTxModule(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Disable TX. */
    HWREGH(base + SCI_O_CTL1) &= ~SCI_CTL1_TXENA;
}

/**
 * @brief Enables receiving. Sets the RXENA bit which enables receive.
 * @param [in] base is the base address of the SCI port.
 *
 * @return None.
 */
static inline void
SCI_enableRxModule(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Enable RX and the SCI. */
    HWREGH(base + SCI_O_CTL1) |= (SCI_CTL1_RXENA | SCI_CTL1_SWRESET);
}

/**
 * @brief Disables receiving. Clears the RXENA bit which disables receive.
 * @param [in] base is the base address of the SCI port.
 *
 * @return None
 */
static inline void
SCI_disableRxModule(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Disable RX. */
    HWREGH(base + SCI_O_CTL1) &= ~SCI_CTL1_RXENA;
}

/**
 * @brief Enables Sleep Mode by setting the SLEEP bit in SCICTL1 register.
 * @param [in] base is the base address of the SCI port.
 *
 * @return None
 */
static inline void
SCI_enableSleepMode(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Set sleep bit */
    HWREGH(base + SCI_O_CTL1) |= SCI_CTL1_SLEEP;
}

/**
 * @brief Disables Sleep Mode in SCI by clearing the SLEEP bit in SCICTL1 register.
 * @param [in] base is the base address of the SCI port.
 *
 * @return None
 */
static inline void
SCI_disableSleepMode(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Clear sleep bit */
    HWREGH(base + SCI_O_CTL1) &= ~SCI_CTL1_SLEEP;
}

/**
 * @brief Determines if there are any characters in the receive buffer when the
 * 		  FIFO enhancement is not enabled.
 * @param [in] base is the base address of the SCI port.
 *
 * @return bool
 * 		true if there is data in the receive buffer
 * 		false if there is no data in the receive buffer.
 */
static inline bool
SCI_isDataAvailableNonFIFO(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Return the availability of characters with FIFO disabled. */
    return(((HWREGH(base + SCI_O_RXST) & SCI_RXST_RXRDY) ==
            SCI_RXST_RXRDY) ? true : false);
}

/**
 * @brief Determines if there is any space in the transmit buffer when the FIFO
 * 		  enhancement is not enabled.
 * @param [in] base is the base address of the SCI port.
 *
 * @return bool
 * 		true if there is space available in the transmit buffer
 * 		false if there is no space available in the transmit buffer.
 */
static inline bool
SCI_isSpaceAvailableNonFIFO(uint32_t base)
{
	 /* Check the arguments. */
    ASSERT(SCI_isBaseValid(base));

    /* Return the availability of space. */
    return(((HWREGH(base + SCI_O_CTL2) & SCI_CTL2_TXRDY) == SCI_CTL2_TXRDY) ? true : false);
}

/**
 * @brief Get the transmit FIFO status This functions gets the current number of words in the transmit FIFO.
 * @param [in] base is the base address of the SCI port.
 *
 * @return the current number of words in the transmit FIFO specified.
 * 		   as one of the following:
 * 	 	   SCI_FIFO_TX0, SCI_FIFO_TX1, SCI_FIFO_TX2, SCI_FIFO_TX3,
 * 	 	   SCI_FIFO_TX4, ... , SCI_FIFO_TX16
 */
static inline SCI_TxFIFOLevel
SCI_getTxFIFOStatus(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    if (SCI_isFIFOEnabled(base) == false)
    {
    	return (SCI_TxFIFOLevel) SCI_FIFO_TX16;
    }

    /* Get the current FIFO status */
    return((SCI_TxFIFOLevel)((HWREGH(base + SCI_O_FFTX) & SCI_FFTX_TXFFST_M) >>
                             SCI_FFTX_TXFFST_S));
}


/**
 * @brief Get the receive FIFO status This functions gets the current number of words in the receive FIFO.
 * @param [in] base is the base address of the SCI port.
 *
 * @return Returns the current number of words in the receive FIFO specified
 * 		SCI_FIFO_RX0, SCI_FIFO_RX1, SCI_FIFO_RX2, SCI_FIFO_RX3, SCI_FIFO_RX4,
 * 		..., SCI_FIFO_RX16
 */
static inline SCI_RxFIFOLevel
SCI_getRxFIFOStatus(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    if (SCI_isFIFOEnabled(base) == false)
    {
    	return (SCI_RxFIFOLevel) SCI_FIFO_RX0;
    }

    /* Get the current FIFO status */
    return((SCI_RxFIFOLevel)((HWREGH(base + SCI_O_FFRX) & SCI_FFRX_RXFFST_M) >>
                             SCI_FFRX_RXFFST_S));
}

/**
 * @brief Determines whether the SCI transmitter is busy or not.
 * @param [in] base is the base address of the SCI port.
 *
 * @return bool
 * 		   true if the SCI is transmitting
 * 		   false if transmission are complete
 */
static inline bool
SCI_isTransmitterBusy(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Check if FIFO enhancement is enabled. */
    if(SCI_isFIFOEnabled(base))
    {
    	/* With FIFO enhancement, determine if the SCI is busy. */
        return(((HWREGH(base + SCI_O_FFTX) & SCI_FFTX_TXFFST_M) !=
                 0U) ? true : false);
    }
    else
    {
    	 /* Without FIFO enhancement, determine if the SCI is busy.
            Check if the transmit buffer and shift register empty. */
        return(((HWREGH(base + SCI_O_CTL2) & SCI_CTL2_TXEMPTY) ==
                SCI_CTL2_TXEMPTY) ? false : true);
    }
}

/**
 * @brief Waits to send a character from the specified port when the FIFO enhancement
 * 		  is enabled.
 * @param [in] base is the base address of the SCI port.
 * @param [in] data is the character to be transmitted.
 *
 * @return None.
 */
static inline void
SCI_writeCharBlockingFIFO(uint32_t base, uint16_t data)
{
    ASSERT(SCI_isBaseValid(base));

    /* Wait until space is available in the transmit FIFO. */
    while(SCI_getTxFIFOStatus(base) == SCI_FIFO_TX16)
    {
    }

    /* Send a char. */
    HWREGH(base + SCI_O_TXBUF) = data;
}

__STATIC_INLINE void
UART_writeCharBlockingFIFO(uint32_t base, uint8_t data)
{
    ASSERT(SCI_isBaseValid(base));

    /* Wait until space is available in the transmit FIFO. */
    while(SCI_getTxFIFOStatus(base) == SCI_FIFO_TX16)
    {
    }

    /* Send a char. */
    HWREGH(base + SCI_O_TXBUF) = data;
}

/**
 * @brief Waits to send a character from the specified port.
 * @param [in] base is the base address of the SCI port.
 * @param [in] data is the character to be transmitted.
 *
 * @return None
 */
static inline void
SCI_writeCharBlockingNonFIFO(uint32_t base, uint16_t data)
{
    ASSERT(SCI_isBaseValid(base));

    /* Wait until space is available in the transmit buffer. */
    while(!SCI_isSpaceAvailableNonFIFO(base))
    {
    }

    /* Send a char. */
    HWREGH(base + SCI_O_TXBUF) = data;
}



__STATIC_INLINE void
UART_writeCharBlockingNonFIFO(uint32_t base, uint8_t data)
{
    ASSERT(SCI_isBaseValid(base));

    /* Wait until space is available in the transmit buffer. */
    while(!SCI_isSpaceAvailableNonFIFO(base))
    {
    }

    /* Send a char. */
    HWREGH(base + SCI_O_TXBUF) = data;
}

/**
 * @brief Sends a character to the specified port.
 * @param [in] base is the base address of the SCI port.
 * @param [in] data is the character to be transmitted.
 * 
 * @return None.
 */
static inline void
SCI_writeCharNonBlocking(uint32_t base, uint16_t data)
{
    ASSERT(SCI_isBaseValid(base));

    /* Send a char. */
    HWREGH(base + SCI_O_TXBUF) = data;
}


__STATIC_INLINE void
UART_writeCharNonBlocking(uint32_t base, uint8_t data)
{
    ASSERT(SCI_isBaseValid(base));

    /* Send a char. */
    HWREGH(base + SCI_O_TXBUF) = data;
}

/**
 * @brief Gets current receiver status flags.
 * @param [in] base is the base address of the SCI port.
 *
 * @return Returns a bitwise OR combination of the receiver status flags
 * 			SCI_RXSTATUS_WAKE, SCI_RXSTATUS_PARITY, SCI_RXSTATUS_OVERRUN,
 * 			SCI_RXSTATUS_FRAMING, SCI_RXSTATUS_BREAK, SCI_RXSTATUS_READY,
 * 			SCI_RXSTATUS_ERROR.
 */
static inline uint16_t
SCI_getRxStatus(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Return the current value of the receive status register. */
    return(HWREGH(base + SCI_O_RXST));
}

/**
 * @brief Waits for a character from the specified port when the FIFO enhancement
 * 		  is enabled.
 * @param [in] base is the base address of the SCI port
 *
 * @retrurn Returns the character read from the specified port as \e uint16_t
 * 			or 0x0 in case of Error. The application must use
 * 			SCI_getRxStatus() API to check if some error occurred before
 * 			consuming the data
 */
static inline uint16_t
SCI_readCharBlockingFIFO(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Wait until a character is available in the receive FIFO. */
    while(SCI_getRxFIFOStatus(base) == SCI_FIFO_RX0)
    {
    	/* If there is any error return */
        if((SCI_getRxStatus(base) & SCI_RXSTATUS_ERROR) != 0U)
        {
            return(0U);
        }
    }

    /* Return the character from the receive buffer. */
    return((uint16_t)(HWREGH(base + SCI_O_RXBUF) & SCI_RXBUF_SAR_M));
}

/**
 * @brief Waits for a character from the specified port when the FIFO enhancement
 * 		  is not enabled.
 * @param [in] base is the base address of the SCI port.
 *
 * @return uint16_t returns the character read from the specified port as uint16_t.
 */
static inline uint16_t
SCI_readCharBlockingNonFIFO(uint32_t base)
{
	/* Check the arguments. */
    ASSERT(SCI_isBaseValid(base));

    /* Wait until a character is available in the receive FIFO. */
    while(!SCI_isDataAvailableNonFIFO(base))
    {
    }

    /* Return the character from the receive buffer. */
    return((uint16_t)(HWREGH(base + SCI_O_RXBUF) & SCI_RXBUF_SAR_M));
}

/**
 * @brief Receives a character from the specified port.
 * @param [in] base is the base address of the SCI port.
 *
 * @return uin16_t which is read from the receive buffer.
 */
static inline uint16_t
SCI_readCharNonBlocking(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Return the character from the receive buffer. */
    return((uint16_t)(HWREGH(base + SCI_O_RXBUF) & SCI_RXBUF_SAR_M));
}

/**
 * @brief erforms a software reset of the SCI and Clears all reported receiver
 * 	        status flags. It affects the
 *          operating flags of the SCI, but it neither affects the configuration bits
 *          nor restores the reset values.
 * @param [in] base is the base address of the SCI port.
 *
 * @return None.
 */
static inline void
SCI_performSoftwareReset(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* To clear all errors a sw reset of the module is required */
    HWREGH(base + SCI_O_CTL1) &= ~SCI_CTL1_SWRESET;
    HWREGH(base + SCI_O_CTL1) |= SCI_CTL1_SWRESET;
}

/**
 * @brief Enables Loop Back Test Mode
 * @param [in] base is the base address of the SCI port
 *
 * @return None.
 */
static inline void
SCI_enableLoopback(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Set the loop back mode. */
    HWREGH(base + SCI_O_CCR) |= SCI_CCR_LOOPBKENA;
}

/**
 * @brief Disables Loop Back Test Mode
 * @param [in] base is the base address of the SCI port.
 *
 * @return None
 */
static inline void
SCI_disableLoopback(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(SCI_isBaseValid(base));

    /* Clear the loop back mode. */
    HWREGH(base + SCI_O_CCR) &= ~SCI_CCR_LOOPBKENA;
}

/**
 * @brief Get the receive FIFO Overflow flag status
 * @param [in] base is the base address of the SCI port
 *
 * @return Returns true if overflow has occurred, else return false if
 * 		   an overflow hasn't occurred.
 */
static inline bool
SCI_getOverflowStatus(uint32_t base)
{
	/* Check the arguments. */
    ASSERT(SCI_isBaseValid(base));

    /* Return the current FIFO overflow status */
    return((HWREGH(base + SCI_O_FFRX) & SCI_FFRX_RXFFOVF) == SCI_FFRX_RXFFOVF);
}

/**
 * @brief Clear the receive FIFO Overflow flag status
 * @param [in] base is the base address of the SCI port
 *
 * @retrun None
 */
static inline void
SCI_clearOverflowStatus(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Clear the current FIFO overflow status */
    HWREGH(base + SCI_O_FFRX) |= SCI_FFRX_RXFFOVRCLR;
}

/**
 * @brief Sets SCI emulation mode.
 * @param [in] base is the base address of the SCI port.
 * @param [in] mode is the emulation mode.
 *
 * @return None
 */
static inline void
SCI_setEmulationMode(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Write the desired emulation mode to the register. */
    HWREGH(base + SCI_O_PRI) = SCI_PRI_FREESOFT_M;
}

/**
 * @brief Returns the Emulation Buffer Received Data
 * @param [in] base is the base address of the SCI port
 *
 * @return Rx emulation buffer data
 */
static inline uint16_t
SCI_readRxEmulationBuffer(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Return Emulation Buffer Received Data */
    return(HWREGH(base + SCI_O_RXEMU));
}

/**
 * @brief Configures the FIFO Transmit Delay
 *        This function sets the delay between every transfer from FIFO
 * 		  transmit buffer to transmit shift register. The delay is defined in
 * 		  number SCI serial clock cycles.
 * @param [in] base is the base address of the SCI port.
 * @param [in] delay Tx FIFO delay to be configured in cycles (0..0xFF)
 *
 * @return None
 */
static inline void
SCI_setTxFifoTransmitDelay(uint32_t base, uint16_t delay)
{
    ASSERT(SCI_isBaseValid(base));
    ASSERT(delay <= 0xFFU);

    /* Configure the FIFO Transmit Delay Bits */
    HWREGH(base + SCI_O_FFCT) = delay;
}

/**
 * @brief Set the Rx timeout value
 * @param [in] base is the base address of the SCI port.
 * @param [in] Configured timeout duration
 * @detail
 * 		When the RX has completed receiving one frame of data,
 * 		the counter begins to time. An RX_TIMEOUT interrupt is generated and reported
 * 		when the counter value equals the configured Timeout duration.
 * 		The unit is 1 baud clock.
 * @return None
 */

__STATIC_INLINE void
SCI_setRxTimeout(uint32_t base,uint32_t time)
{
    ASSERT(SCI_isBaseValid(base));

    /* Configure the SCI_O_RX_TIMEOUT_TH Bits */
    HWREG(base + SCI_O_RXTIMEOUTVAL) = time;
}

/**
 * @brief Sets the configuration of a SCI.
 * @param [in] base is the base address of the SCI port.
 * @param [in] lspclkHz is the rate of the clock supplied to the SCI module.
 * @param [in] baud is the desired baud rate.
 * @param [in] config is the data format for the port (number of data bits,
 * 	 		   number of stop bits, and parity).
 *
 * @return None.
 */
extern void
SCI_setConfig(uint32_t base, uint32_t lspclkHz, uint32_t baud,
              uint32_t config);


/**
 * @brief Waits to send an array of characters from the specified port.
 * 		  Sends the number of characters specified by length, starting at the
 * 		  address array, out of the transmit buffer for the specified port.
 * 		  If there is no space available in the transmit buffer, or the transmit
 * 		  FIFO if it is enabled, this function waits until there is space available
 * 		  and length number of characters are transmitted before returning.
 * 		  array is a pointer to uint16_t but only the least significant 8 bits
 * 		  are written to the SCI port.  SCI only transmits 8 bit characters.
 * @param [in] array is the address of the array of characters to be transmitted.
 * 			   It is pointer to the array of characters to be transmitted.
 * @param [in] length is the length of the array, or number of characters in the
 * 			   array to be transmitted.
 *
 * @return None.
 */
extern void
SCI_writeCharArray(uint32_t base, const uint16_t * const array,
                   uint16_t length);
extern void
UART_writeCharArray(uint32_t base, const uint8_t * const array,
                   uint32_t length);

/**
 * @brief Waits to receive an array of characters from the specified port.
 * 		  Receives an array of characters from the receive buffer for the specified
 * 		  port, and stores them as an array of characters starting at address
 * 		  array.  This function waits until the \e length number of characters are
 * 		  eceived before returning.
 * @param [in] array is the address of the array of characters to be received.
 * @param [in] length is the length of the array, or number of characters in the
 *
 * @return None
 *
 */
extern void
SCI_readCharArray(uint32_t base, uint16_t * const array, uint16_t length);

extern void
UART_readCharArray(uint32_t base, uint8_t * const array, uint32_t length);


/**
 * @brief Enables individual SCI interrupt sources.
 * 		  Enables the indicated SCI interrupt sources.  Only the sources that are
 * 		  enabled can be reflected to the processor interrupt; disabled sources have
 * 		  no effect on the processor.
 * 		  intFlags:
 * 			SCI_INT_RXERR      - RXERR Interrupt
 * 			SCI_INT_RXRDY_BRKDT - RXRDY/BRKDT Interrupt
 * 			SCI_INT_TXRDY      - TXRDY Interrupt
 * 			SCI_INT_TXFF       - TX FIFO Level Interrupt
 * 			SCI_INT_RXFF       - RX FIFO Level Interrupt
 * 			SCI_INT_FE         - Frame Error
 * 			SCI_INT_OE         - Overrun Error
 * 			SCI_INT_PE         - Parity Error
 * @param [in] base is the base address of the SCI port.
 * @param [in] intFlags is the bit mask of the interrupt sources to be enabled.
 *
 * @return None
 */
extern void
SCI_enableInterrupt(uint32_t base, uint32_t intFlags);

/**
 * @brief Disable individual SCI interrupt sources.
 * 		  Enables the indicated SCI interrupt sources.  Only the sources that are
 * 		  enabled can be reflected to the processor interrupt; disabled sources have
 * 		  no effect on the processor.
 * 		  intFlags:
 * 			SCI_INT_RXERR      - RXERR Interrupt
 * 			SCI_INT_RXRDY_BRKDT - RXRDY/BRKDT Interrupt
 * 			SCI_INT_TXRDY      - TXRDY Interrupt
 * 			SCI_INT_TXFF       - TX FIFO Level Interrupt
 * 			SCI_INT_RXFF       - RX FIFO Level Interrupt
 * 			SCI_INT_FE         - Frame Error
 * 			SCI_INT_OE         - Overrun Error
 * 			SCI_INT_PE         - Parity Error
 *
 * @param [in] base is the base address of the SCI port.
 * @param [in] intFlags is the bit mask of the interrupt sources to be disabled.
 *
 * @return None
 */
extern void
SCI_disableInterrupt(uint32_t base, uint32_t intFlags);

/**
 * @brief Gets the current interrupt status.
 * @param [in] base is the base address of the SCI port.
 *
 * @return intFlags :
 * 			SCI_INT_RXERR      - RXERR Interrupt
 * 			SCI_INT_RXRDY_BRKDT - RXRDY/BRKDT Interrupt
 * 			SCI_INT_TXRDY      - TXRDY Interrupt
 * 			SCI_INT_TXFF       - TX FIFO Level Interrupt
 * 			SCI_INT_RXFF       - RX FIFO Level Interrupt
 * 			SCI_INT_FE         - Frame Error
 * 			SCI_INT_OE         - Overrun Error
 * 			SCI_INT_PE         - Parity Error
 */
extern uint32_t
SCI_getInterruptStatus(uint32_t base);

/**
 * @brief Gets the current interrupt status.
 * @param [in] base is the base address of the SCI port.
 * @param [in] intFlags is the bit mask of the interrupt sources to be clear.
 *
 * @return intFlags :
 * 			SCI_INT_RXERR      - RXERR Interrupt
 * 			SCI_INT_RXRDY_BRKDT - RXRDY/BRKDT Interrupt
 * 			SCI_INT_TXRDY      - TXRDY Interrupt
 * 			SCI_INT_TXFF       - TX FIFO Level Interrupt
 * 			SCI_INT_RXFF       - RX FIFO Level Interrupt
 * 			SCI_INT_FE         - Frame Error
 * 			SCI_INT_OE         - Overrun Error
 * 			SCI_INT_PE         - Parity Error
 */
extern void
SCI_clearInterruptStatus(uint32_t base, uint32_t intFlags);

/**
 * @brief Sets SCI baud rate.
 * @param [in] base is the base address of the SCI port.
 * @param [in] lspclkHz is the rate of the clock supplied to the SCI module.
 * @param [in] baud is the desired baud rate.
 *
 * @return None.
 */
extern void
SCI_setBaud(uint32_t base, uint32_t lspclkHz, uint32_t baud);

/**
 * @brief Sets the SCI TXWAKE flag.
 *        This function sets the TXWAKE flag bit to indicate that the next frame
 * 		  is an address frame. TXWAKE bit controls selection of data-transmit feature
 * 		  based on which mode is selected from idle-line and address-bit.
 * @param [in] base is the base address of the SCI port.
 *
 * @return None.
 */
extern void
SCI_setWakeFlag(uint32_t base);

#endif /* IS_GS32F00xx(0x30) || IS_GS32F3xx(0x23) */
#endif /* DRIVERLIB_UART_V30_H_ */
