/*
*   Copyright (c) Gejian Semiconductors 2023
*   All rights reserved.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
*/


/**
*   @file    wd.h
*   @brief
*
*/
/**
*   commit history
*   20240318,XuShicheng, update the following APIs WD_SysCtl_enableDebugHold, WD_SysCtl_disableDebugHold,
*                        SysCtl_setWatchdogPrescaler/WD_SysCtl_enableOverflowReset/WD_SysCtl_disableOverflowReset
*
*   20240417, ZhaoLei, rewrite GS32F00xx
*   20240529, Jason, change API name from WD_SysCtl_enable... to be SysCtl_enableWatchdog...
*                    change API name from WatchdogOverflowReset to be WatchdogReset
*   2025/1/1  hezhiyuan, Distinguish wd and wwd by base, modify comments and APIs
*/

#ifndef DEVICE_DRIVERLIB_WD_H_
#define DEVICE_DRIVERLIB_WD_H_
#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include <stdbool.h>
#include <stdint.h>

#include "gs32_version.h"
#include "inc/hw_types.h"
#include "inc/hw_memmap.h"
#include "inc/hw_wd.h"
#include "inc/hw_dsp_rsv_cfg_v22.h"
#include "sysctl/sysctl_crg_v12.h"
#include "sysctl/sysctl_crg_v22.h"
#include "sysctl/sysctl_crg_v30.h"
#include "debug.h"
#include "device.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* The following are key for the register WD_LOCK   */
#define WD_LOCK_ENKEY                   0x1ACCE551U

/*  The following are key for the register SYSCTL_L_LOCK    */
#define WD_SYSCTL_L_LOCK_ENKEY          0x5A5A5A5AU

/*  The following are key for the register SYSCTL_H_LOCK    */
#define WD_SYSCTL_H_LOCK_ENKEY          0x5A5A5A5AU

/* The following are the maximum value for watchdog prescaler.      */
#define WD_SYSCTL_MAX_PRESCALER_VALUE   4096U

/* The following are the maximum value for watchdog divider.      */
#define WD_SYSCTL_MAX_DIVIDER_VALUE      64U



//Select based on the quantity of WD
#ifdef WD2_BASE
#define IS_WD_BASE(base) ((base == WD1_BASE) || (base == WD2_BASE))
#else
#define IS_WD_BASE(base) ((base == WD1_BASE))
#endif

//Select based on the quantity of WWD
#ifdef WWD2_BASE
#define IS_WWD_BASE(base) ((base == WWD1_BASE) || (base == WWD2_BASE))
#else
#define IS_WWD_BASE(base) ((base == WWD1_BASE))
#endif

extern uint32_t chipRevision;

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/*  The following are values that can be passed to Wd_setWatchdogMode() to switch
*  mode parameters.
*/
typedef enum
{
   /* 'first countdown interrupt second countdown reset' mode,  */
   WD_MODE_RESET,
   /* Watchdog can generate an interrupt signal; reset signal is disabled  */
   WD_MODE_INTERRUPT,
    /* 'first countdown reset' mode,  */
   WD_MODE_RESET_FIRST
} Wd_Mode;


/*  The following are values that indicates type of watchdog.
*/
typedef enum
{
   /* Watchdog without window value */
   WD_TYPE_NORMAL,
   /* Watchdog with window value */
   WD_TYPE_WINDOW
} Wd_Type;

/*
* The following are values that can be passed to
* SysCtl_setWatchdogPredivider() as the predivider parameter.
*/
typedef enum
{
   SYSCTL_WD_PREDIV_16   = 4,  //!< PREDIVCLK = INTOSC1 / 16
   SYSCTL_WD_PREDIV_32   = 5,  //!< PREDIVCLK = INTOSC1 / 32
   SYSCTL_WD_PREDIV_64   = 6,  //!< PREDIVCLK = INTOSC1 / 64
   SYSCTL_WD_PREDIV_128  = 7,  //!< PREDIVCLK = INTOSC1 / 128
   SYSCTL_WD_PREDIV_256  = 8,  //!< PREDIVCLK = INTOSC1 / 256
   SYSCTL_WD_PREDIV_512  = 9,  //!< PREDIVCLK = INTOSC1 / 512
   SYSCTL_WD_PREDIV_1024 = 10,  //!< PREDIVCLK = INTOSC1 / 1024
   SYSCTL_WD_PREDIV_2048 = 11,  //!< PREDIVCLK = INTOSC1 / 2048
   SYSCTL_WD_PREDIV_4096 = 12   //!< PREDIVCLK = INTOSC1 / 4096
} SysCtl_WDPredivider;

/*
*
* The following are values that can be passed to
* SysCtl_setWatchdogPrescaler() as the prescaler parameter.
*
*/
typedef enum
{
   SYSCTL_WD_PRESCALE_1  = 0,      //!< WDCLK = PREDIVCLK / 1
   SYSCTL_WD_PRESCALE_2  = 1,      //!< WDCLK = PREDIVCLK / 2
   SYSCTL_WD_PRESCALE_4  = 2,      //!< WDCLK = PREDIVCLK / 4
   SYSCTL_WD_PRESCALE_8  = 3,      //!< WDCLK = PREDIVCLK / 8
   SYSCTL_WD_PRESCALE_16 = 4,      //!< WDCLK = PREDIVCLK / 16
   SYSCTL_WD_PRESCALE_32 = 5,      //!< WDCLK = PREDIVCLK / 32
   SYSCTL_WD_PRESCALE_64 = 6       //!< WDCLK = PREDIVCLK / 64
} SysCtl_WDPrescaler;

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */

#ifdef DEBUG
/**
* @brief   This function determines if an Wd module base address is valid.
*
* @param   base   specifies the Wd module base address.
*          WD1_BASE    DSP core watchdog base address.
*
* @retval  Returns true if the base address is valid and false otherwise.
*/
__STATIC_INLINE bool WD_isBaseValid(uint32_t base)
{
    return (
        (base == WD1_BASE) || (base == WWD1_BASE)
#ifdef WD2_BASE
        || (base == WD2_BASE)
#endif
#ifdef WWD2_BASE
        || (base == WWD2_BASE)
#endif
    );
}
#endif

/**
* @brief   configure the initial value of Watchdog timer
*
* @param [in]   base   specifies the Wd module base address.
*               WD1_BASE    DSP core1 watchdog base address.
*               WD2_BASE    DSP core2 watchdog base address.(if available)
* @param   value Set the initial value for the countdown
* @retval  None
*/
__STATIC_INLINE void WD_initNormalWatchdog(uint32_t base,uint32_t value)
{
    if (IS_WD_BASE(base)) {
        HWREG(base+WD_O_LOAD) = value;
    } 
}

__STATIC_INLINE void SysCtl_initNormalWatchdog(uint32_t base,uint32_t value)
{
    if (IS_WD_BASE(base)) {
        HWREG(base+WD_O_LOAD) = value;
    } 
}

__STATIC_INLINE void SysCtl_initWatchdog(uint32_t base,uint32_t value)
{
    if (IS_WD_BASE(base)) {
        HWREG(base+WD_O_LOAD) = value;
    } 
}

/**
* @brief   This function initiate the window watchdog
*
* @param [in]   base   specifies the WWd module base address.
*               WWD1_BASE   DSP core1 window watchdog base address. 
*               WWD2_BASE   DSP core2 window watchdog base address.(if available)
* @param [in]   loadValue    Set the initial value for the countdown
* @param [in]   winValue     Window upper limit value(The lower limit value of the window is 0)
* @retval  None
*/
__STATIC_INLINE void WD_initWindowWatchdog(uint32_t base,uint32_t loadValue,uint32_t winValue)
{
    if (IS_WWD_BASE(base)) {
        HWREG(base+WWD_O_LOAD) = loadValue;
        HWREG(base+WWD_O_WINVALUE) = winValue;
    }
}

__STATIC_INLINE void SysCtl_initWindowWatchdog(uint32_t base,uint32_t loadValue,uint32_t winValue)
{
    if (IS_WWD_BASE(base)) {
        HWREG(base+WWD_O_LOAD) = loadValue;
        HWREG(base+WWD_O_WINVALUE) = winValue;
    }
}

/**
* @brief   This function enables the watchdog counter to be held during pauses or breakpoints.
*
* @param [in]   base   specifies the Wd module base address.
*               WD1_BASE    DSP core1 watchdog base address.
*               WD2_BASE    DSP core2 watchdog base address.(if available)
*               WWD1_BASE   DSP core1 window watchdog base address. 
*               WWD2_BASE   DSP core2 window watchdog base address.(if available)
*
* @retval  None
*/
__STATIC_INLINE void SysCtl_enableWatchdogDebugMode(uint32_t base)
{
    ASSERT(WD_isBaseValid(base));

#if (IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30))
    SysCtl_setDspWdgDebugHoldEn(1);
#elif IS_GS32F3xx(0x22)
    if((base == WD1_BASE) || (base == WWD1_BASE)){
        SysCtl_setDspCpu1WdgDebugHoldEn(1);
    }else if((base == WD2_BASE) || (base == WWD2_BASE)){
        SysCtl_setDspCpu2WdgDebugHoldEn(1);
    }
#endif
}


/**
* @brief   This function disables the watchdog counter to be held during pauses or breakpoints.
*
* @param [in]   base   specifies the Wd module base address.
*               WD1_BASE    DSP core1 watchdog base address.
*               WD2_BASE    DSP core2 watchdog base address.(if available)
*               WWD1_BASE   DSP core1 window watchdog base address. 
*               WWD2_BASE   DSP core2 window watchdog base address.(if available)
*
* @retval  None
*/
__STATIC_INLINE void SysCtl_disableWatchdogDebugMode(uint32_t base)
{
    ASSERT(WD_isBaseValid(base));

#if (IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30))
    SysCtl_setDspWdgDebugHoldEn(0);
#elif IS_GS32F3xx(0x22)
    if((base == WD1_BASE) || (base == WWD1_BASE)){
        SysCtl_setDspCpu1WdgDebugHoldEn(0);
    }else if((base == WD2_BASE) || (base == WWD2_BASE)){
        SysCtl_setDspCpu2WdgDebugHoldEn(0);
    }
#endif
}


/**
* @brief   "Configures whether the watchdog generates a reset or an interrupt signal"
*
* @param [in]   base   specifies the Wd module base address.
*               WD1_BASE    DSP core1 watchdog base address.
*               WD2_BASE    DSP core2 watchdog base address.(if available)
*               WWD1_BASE   DSP core1 window watchdog base address. 
*               WWD2_BASE   DSP core2 window watchdog base address.(if available)
* @param [in]   mode   When the mode parameter is WD_MODE_INTERRUPT,(Window watchdog disabled this function)
*                 the watchdog is enabled to generate watchdog interrupt signals,
*                 the interrupt signal is enabled, and the output reset signal is disabled.
*                 This will allow the watchdog module to wake up the device.
*
*                 When the mode parameter is WD_MODE_RESET,(Window watchdog disabled this function)
*                 the watchdog will enter the "first countdown interrupt second countdown reset" mode,
*                 enabling the output of reset and interrupt signals.
*                 This is the default configuration method for the watchdog.
*
*                 When the 'mode' parameter is set to WD_MODE_RESET_FIRST,
*                 the watchdog will enter the 'first countdown reset' mode,
*                 enabling the output reset signal and disabling the interrupt signal.
*
* @param [in]   type   (invalid)choose the watchdog type. The type parameter may take a
*               value of one of the following values:
*               -WD_TYPE_NORMAL
*               -WD_TYPE_WINDOW
*
* @retval  None
*/
__STATIC_INLINE void WD_setWatchdogMode(uint32_t base, Wd_Mode mode, Wd_Type type)
{
    ASSERT(WD_isBaseValid(base));
   
    if(IS_WD_BASE(base)){
        if (mode == WD_MODE_INTERRUPT){
            HWREG(base+WD_O_CONTROL) &= (~WD_CONTROL_RESEN);
        }
        else if (mode == WD_MODE_RESET){
            HWREG(base+WD_O_CONTROL) |= (WD_CONTROL_RESEN);
        }
        else if (mode == WD_MODE_RESET_FIRST){
            HWREG(base+WD_O_CONTROL) &= (~WD_CONTROL_RESEN);
            HWREG(base+WD_O_CONTROL) |= (WD_CONTROL_RESEN_FIRST);
        }
    }else if(IS_WWD_BASE(base)){
        if (mode == WD_MODE_RESET_FIRST){
            HWREG(base+WWD_O_CONTROL) &= (~WWD_CONTROL_RESEN);
            HWREG(base+WWD_O_CONTROL) |= (WWD_CONTROL_RESEN_FIRST);
        }
    //    else if (mode == WD_MODE_RESET){
    //         HWREG(base+WWD_O_CONTROL) |= (WWD_CONTROL_RESEN);
    //    }
    //    else if (mode == WD_MODE_INTERRUPT){
    //         HWREG(base+WWD_O_CONTROL) &= (~WWD_CONTROL_RESEN);
    //    }
   }
}


__STATIC_INLINE void SysCtl_setWatchdogMode(uint32_t base, Wd_Mode mode)
{
    ASSERT(WD_isBaseValid(base));
    
    if(IS_WD_BASE(base)){
        if (mode == WD_MODE_INTERRUPT){
            HWREG(base+WD_O_CONTROL) &= (~WD_CONTROL_RESEN);
        }
        else if (mode == WD_MODE_RESET){
            HWREG(base+WD_O_CONTROL) |= (WD_CONTROL_RESEN);
        }
        else if (mode == WD_MODE_RESET_FIRST){
            HWREG(base+WD_O_CONTROL) &= (~WD_CONTROL_RESEN);
            HWREG(base+WD_O_CONTROL) |= (WD_CONTROL_RESEN_FIRST);
        }
    }
}


__STATIC_INLINE void SysCtl_setWindowWatchdogMode(uint32_t base, Wd_Mode mode)
{
    ASSERT(WD_isBaseValid(base));
    
    if(IS_WWD_BASE(base)){
        if (mode == WD_MODE_RESET_FIRST){
            HWREG(base+WWD_O_CONTROL) &= (~WWD_CONTROL_RESEN);
            HWREG(base+WWD_O_CONTROL) |= (WWD_CONTROL_RESEN_FIRST);
        }
    }
}


/**
* @brief   "Gets the status of the watchdog interrupt signal"
*
* @param [in]   base   specifies the Wd module base address.
*               WD1_BASE    DSP core1 watchdog base address.
*               WD2_BASE    DSP core2 watchdog base address.(if available)
* @param [in]   type (invalid) choose the watchdog type. The type parameter may take a
*               value of one of the following values:
*               -WD_TYPE_NORMAL
*               -WD_TYPE_WINDOW
*
* @retval  true if the interrupt is active and false if it is not.
*/
__STATIC_INLINE boolean WD_isWatchdogInterruptActive(uint32_t base, Wd_Type type)
{
    ASSERT(WD_isBaseValid(base));

    if (IS_WD_BASE(base)) {
        return ((HWREG(base+WD_O_MIS) & WD_MIS_MINT) == 0x1U);
    } 
    // else if (IS_WWD_BASE(base)) {
    //     return ((HWREG(base+WWD_O_MIS) & WWD_MIS_MINT) == 0x1U);
    // }
    else {
    	ASSERT(false);
    	return(false);
    }
}

__STATIC_INLINE boolean SysCtl_isWatchdogInterruptActive(uint32_t base)
{
    ASSERT(WD_isBaseValid(base));

    if (IS_WD_BASE(base)) {
        return ((HWREG(base+WD_O_MIS) & WD_MIS_MINT) == 0x1U);
    }
    else {
    	ASSERT(false);
    	return(false);
    }
}

// __STATIC_INLINE boolean SysCtl_isWindowWatchdogInterruptActive(uint32_t base)
// {
//    ASSERT(WD_isBaseValid(base));
//    return (WD_isWatchdogInterruptActive(base,WD_TYPE_WINDOW));
// }


/**
* @brief   "This function disables the watchdog timer"
*
* @param [in]   base   specifies the Wd module base address.
*               WD1_BASE    DSP core1 watchdog base address.
*               WD2_BASE    DSP core2 watchdog base address.(if available)
*               WWD1_BASE   DSP core1 window watchdog base address. 
*               WWD2_BASE   DSP core2 window watchdog base address.(if available)
* @param [in]   type   (invalid)choose the watchdog type. The type parameter may take a
*               value of one of the following values:
*               -WD_TYPE_NORMAL
*               -WD_TYPE_WINDOW
*
* @retval  None
*/
__STATIC_INLINE void WD_disableWatchdog(uint32_t base,Wd_Type type)
{
    ASSERT(WD_isBaseValid(base));
    
    if (IS_WD_BASE(base)) {
        HWREG(base+WD_O_CONTROL) &= ~(WD_CONTROL_INTEN);
    } else if (IS_WWD_BASE(base)) {
        HWREG(base+WWD_O_CONTROL) &= ~(WWD_CONTROL_INTEN);
    }
}


__STATIC_INLINE void SysCtl_disableWatchdog(uint32_t base)
{
    ASSERT(WD_isBaseValid(base));

    if (IS_WD_BASE(base)) {
        HWREG(base+WD_O_CONTROL) &= ~(WD_CONTROL_INTEN);
    }
}

__STATIC_INLINE void SysCtl_disableWindowWatchdog(uint32_t base)
{
    ASSERT(WD_isBaseValid(base));
    
    if (IS_WWD_BASE(base)) {
        HWREG(base+WWD_O_CONTROL) &= ~(WWD_CONTROL_INTEN);
    }
}
/**
* @brief   "This function enables the watchdog timer."
*
* @param [in]   base   specifies the Wd module base address.
*               WD1_BASE    DSP core1 watchdog base address.
*               WD2_BASE    DSP core2 watchdog base address.(if available)
*               WWD1_BASE   DSP core1 window watchdog base address. 
*               WWD2_BASE   DSP core2 window watchdog base address.(if available)
* @param [in]   type   (invalid)choose the watchdog type. The type parameter may take a
*               value of one of the following values:
*               -WD_TYPE_NORMAL
*               -WD_TYPE_WINDOW
*
* @retval  None
*/
__STATIC_INLINE void WD_enableWatchdog(uint32_t base,Wd_Type type)
{
    ASSERT(WD_isBaseValid(base));
    
    if (IS_WD_BASE(base)) {
        HWREG(base+WD_O_CONTROL) |= (WD_CONTROL_INTEN);
    } else if (IS_WWD_BASE(base)) {
        HWREG(base+WWD_O_CONTROL) |= (WWD_CONTROL_INTEN);
    }
}

__STATIC_INLINE void SysCtl_enableWatchdog(uint32_t base)
{
    ASSERT(WD_isBaseValid(base));
    
    if (IS_WD_BASE(base)) {
        HWREG(base+WD_O_CONTROL) |= (WD_CONTROL_INTEN);
    }
}

__STATIC_INLINE void SysCtl_enableWindowWatchdog(uint32_t base)
{
    ASSERT(WD_isBaseValid(base));
    
    if (IS_WWD_BASE(base)) {
        HWREG(base+WWD_O_CONTROL) |= (WWD_CONTROL_INTEN);
    }
}


/**
* @brief   "This function can only be used after the watchdog is disabled, 
*           resetting the count value to the value in the WDOGLOAD register"
*
* @param [in]   base   specifies the Wd module base address.
*               WD1_BASE    DSP core1 watchdog base address.
*               WD2_BASE    DSP core2 watchdog base address.(if available)
*               WWD1_BASE   DSP core1 window watchdog base address. 
*               WWD2_BASE   DSP core2 window watchdog base address.(if available)
*
* @retval  None
*/
__STATIC_INLINE void WD_serviceWatchdog(uint32_t base)
{
    ASSERT(WD_isBaseValid(base));
    
    if (IS_WD_BASE(base)) {
#if IS_GS32F3xx(0x22)
        if (chipRevision >= 3) {
            uint32_t val;
            val = HWREG(DSP_RSV_CFG_BASE + CFG_DSP_WD_CLR);
            if(base == WD1_BASE){
                val ^= (0x1<<0);
            }else if(base == WD2_BASE){
                val ^= (0x1<<1);
            }
            SysCtl_delay(1);
            HWREG(DSP_RSV_CFG_BASE + CFG_DSP_WD_CLR)  = val;
        } else {
            HWREG(base+WD_O_INTCLR) = 0x07215AA1U;
            SysCtl_delay(1);
            HWREG(base+WD_O_INTCLR) = 0x0609CA7FU;
        }
#else
        HWREG(base+WD_O_INTCLR) = 0x07215AA1U;
        SysCtl_delay(1);
        HWREG(base+WD_O_INTCLR) = 0x0609CA7FU;
#endif

    } else if (IS_WWD_BASE(base)) {
        HWREG(base+WWD_O_INTCLR) = 0x07215AA1U;
        SysCtl_delay(1);
        HWREG(base+WWD_O_INTCLR) = 0x0609CA7FU;
    }
}


__STATIC_INLINE void SysCtl_serviceWatchdog(uint32_t base)
{
    ASSERT(WD_isBaseValid(base));
	if (IS_WD_BASE(base)) {
#if IS_GS32F3xx(0x22)
	    if (chipRevision >= 3) {
            uint32_t val;
            val = HWREG(DSP_RSV_CFG_BASE + CFG_DSP_WD_CLR);
            if(base == WD1_BASE){
                val ^= (0x1<<0);
            }else if(base == WD2_BASE){
                val ^= (0x1<<1);
            }
            SysCtl_delay(1);
            HWREG(DSP_RSV_CFG_BASE + CFG_DSP_WD_CLR)  = val;
	    } else {
	        HWREG(base+WD_O_INTCLR) = 0x07215AA1U;
	        SysCtl_delay(1);
	        HWREG(base+WD_O_INTCLR) = 0x0609CA7FU;
	    }
#else

	HWREG(base+WD_O_INTCLR) = 0x07215AA1U;
	SysCtl_delay(1);
	HWREG(base+WD_O_INTCLR) = 0x0609CA7FU;

#endif
	}
}

__STATIC_INLINE void SysCtl_serviceWindowWatchdog(uint32_t base)
{
   ASSERT(WD_isBaseValid(base));
   
   if (IS_WWD_BASE(base)) {
        HWREG(base+WWD_O_INTCLR) = 0x07215AA1U;
        SysCtl_delay(1);
        HWREG(base+WWD_O_INTCLR) = 0x0609CA7FU;
    }
}


/**
* @brief   Sets up watchdog clock (WDCLK) pre-divider.
*
* @param [in]   predivider   is the value that configures the pre-divider.
*
* @remarks
*  - This function sets up the watchdog clock (WDCLK) pre-divider. There are two
*  - dividers that scale INTOSC1 to WDCLK. The predivider parameter divides
*  - INTOSC1 down to PREDIVCLK and the prescaler (set by the
*  - SysCtl_setWatchdogPrescaler() function) divides PREDIVCLK down to WDCLK.
* @retval  None
*/
__STATIC_INLINE void
SysCtl_setWatchdogPredivider(uint32_t base,SysCtl_WDPredivider predivider)
{
    ASSERT(WD_isBaseValid(base));
#if (IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30))
   SysCtl_setDspWdgClkPreDiv((WDGPreDivVal_t)predivider);
#elif IS_GS32F3xx(0x22)
   if((base == WD1_BASE) || (base == WWD1_BASE)){
       SysCtl_setDspCpu1WdgClkPreDiv((WDGPreDivVal_t)predivider);
   }else if((base == WD2_BASE) || (base == WWD2_BASE)){
       SysCtl_setDspCpu2WdgClkPreDiv((WDGPreDivVal_t)predivider);
   }
#endif
}

/**
* @brief  Sets up watchdog clock (WDCLK) prescaler.
*
* @param [in]   prescaler   is the value that configures the watchdog clock relative
*  to the value from the pre-divider.
*
* @remarks
*  - This function sets up the watchdog clock (WDCLK) prescaler. There are two
*  - dividers that scale INTOSC1 to WDCLK. The predivider (set with the
*  - SysCtl_setWatchdogPredivider() function) divides INTOSC1 down to PREDIVCLK
*  - and the prescaler parameter divides PREDIVCLK down to WDCLK.
*
* @retval  None
*/
__STATIC_INLINE void
SysCtl_setWatchdogPrescaler(uint32_t base,SysCtl_WDPrescaler prescaler)
{
    ASSERT(WD_isBaseValid(base));
#if (IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30))
   SysCtl_setDspWdgClkScalerDiv((WDGScalerDivVal_t)prescaler);
#elif IS_GS32F3xx(0x22)
   if((base == WD1_BASE) || (base == WWD1_BASE)){
       SysCtl_setDspCpu1WdgClkScalerDiv((WDGScalerDivVal_t)prescaler);
   }else if((base == WD2_BASE) || (base == WWD2_BASE)){
       SysCtl_setDspCpu2WdgClkScalerDiv((WDGScalerDivVal_t)prescaler);
   }
#endif
}

/**
 * @brief   "This function causes the watchdog to immediately reset or interrupt, 
 *          and the specific effect is configured through WD_detWatchdogMod()"
 *
* @param [in]   base   specifies the Wd module base address.
*               WD1_BASE    DSP core1 watchdog base address.
*               WD2_BASE    DSP core2 watchdog base address.(if available)
*               WWD1_BASE   DSP core1 window watchdog base address. 
*               WWD2_BASE   DSP core2 window watchdog base address.(if available)
*
* @retval  None
*/
__STATIC_INLINE void SysCtl_triggerWatchdogSignal(uint32_t base)
{
    ASSERT(WD_isBaseValid(base));

    if (IS_WD_BASE(base)) {
        HWREG(base+WD_O_LOAD) = 0x00U;
    } else if (IS_WWD_BASE(base)) {
        HWREG(base+WWD_O_LOAD) = 0x00U;
    }
}

/**
* @brief   "Gets the current watchdog counter value"
*
* @param [in]   base   specifies the Wd module base address.
*               WD1_BASE    DSP core1 watchdog base address.
*               WD2_BASE    DSP core2 watchdog base address.(if available)
*               WWD1_BASE   DSP core1 window watchdog base address. 
*               WWD2_BASE   DSP core2 window watchdog base address.(if available)
* @param [in]   type (invalid) choose the watchdog type. The type parameter may take a
*               value of one of the following values:
*               -WD_TYPE_NORMAL
*               -WD_TYPE_WINDOW
*
* @retval  Returns the current value of the 32-bit watchdog counter. If this count
*          value overflows, watchdog outputs a interrupt or reset signal.
*/
__STATIC_INLINE uint32_t WD_getWatchdogCounterValue(uint32_t base,Wd_Type type)
{
    ASSERT(WD_isBaseValid(base));

    if (IS_WD_BASE(base)) {
        // Normal watchdog counter value.
        return (HWREG(base + WD_O_VALUE));
    }
    else if (IS_WWD_BASE(base)) {
        // Window watchdog counter value.
        return (HWREG(base + WWD_O_VALUE));
    }
    else {
        // Handle invalid base address to prevent compiler warning.
        ASSERT(false);  // Optional: Trigger an assertion in debug builds.
        return 0;       // Return a default value of 0 in release builds.
    }
}

__STATIC_INLINE uint32_t SysCtl_getWatchdogCounterValue(uint32_t base)
{
    ASSERT(WD_isBaseValid(base));
    return (HWREG(base+WD_O_VALUE));
}


__STATIC_INLINE uint32_t SysCtl_getWindowWatchdogCounterValue(uint32_t base)
{
    ASSERT(WD_isBaseValid(base));
    return (HWREG(base+WWD_O_VALUE));
}

/**
* @brief   "Set the window value only for the windowed watchdog."
*
* @param [in]   base   specifies the Wd module base address.
*               WWD1_BASE   DSP core1 window watchdog base address. 
*               WWD2_BASE   DSP core2 window watchdog base address.(if available)
* @param [in]   value   the window value for windowed watchdog.
*
* @retval  None
*/
__STATIC_INLINE void WD_setWatchdogWindowValue(uint32_t base,uint32_t value)
{
   ASSERT(WD_isBaseValid(base));
   HWREG(base+WWD_O_WINVALUE) = value;
}


__STATIC_INLINE void SysCtl_setWatchdogWindowValue(uint32_t base,uint32_t value)
{
   ASSERT(WD_isBaseValid(base));
   HWREG(base+WWD_O_WINVALUE) = value;
}

/**
* @brief   "This function cleans up interrupts when they're generated"
*
* @param [in]   base   specifies the Wd module base address.
*               WD1_BASE    DSP core1 watchdog base address.
*               WD2_BASE    DSP core2 watchdog base address.(if available)
*
* @retval  None
*/
__STATIC_INLINE void SysCtl_clearWatchdogInterrupt(uint32_t base)
{
    ASSERT(WD_isBaseValid(base));
	if (IS_WD_BASE(base)) {
#if IS_GS32F3xx(0x22)
	    if (chipRevision >= 3) {
            uint32_t val;
            val = HWREG(DSP_RSV_CFG_BASE + CFG_DSP_WD_CLR);
            if(base == WD1_BASE){
                val ^= (0x1<<0);
            }else if(base == WD2_BASE){
                val ^= (0x1<<1);
            }
            SysCtl_delay(1);
            HWREG(DSP_RSV_CFG_BASE + CFG_DSP_WD_CLR)  = val;
	    } else {
	        HWREG(base+WD_O_INTCLR) = 0x07215AA1U;
            SysCtl_delay(1);
            HWREG(base+WD_O_INTCLR) = 0x0609CA7FU;
	    }
#else

	HWREG(base+WD_O_INTCLR) = 0x07215AA1U;
	SysCtl_delay(1);
	HWREG(base+WD_O_INTCLR) = 0x0609CA7FU;

#endif
	}
    // else if (IS_WWD_BASE(base)) {
    //     HWREG(base+WWD_O_INTCLR) = 0x07215AA1U;
    //     HWREG(base+WWD_O_INTCLR) = 0x0609CA7FU;
    // }
}

#ifdef __cplusplus
}
#endif

#endif /* DEVICE_DRIVERLIB_WD_H_ */
