/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    device.c
*   @brief   device basic initialization
*   @details
*
*/
/*
 * commit history
 * 20240306, Jason Yang, Adjust the position of calling ASysCtl_initConfig in Device_init.
 *                       It should be called after peripherals are enabled, otherwise ADC configurations are not effective.
 *                       And this function ASysCtl_initConfig is applicable for both 1.1ES and 2.0ES.
 * 20240312,   Zhao lei, Modify peripheral instances control.
 * 20240316,   zeng jia, Modify flash driver.
 * 20240410, Jason Yang, add MACRO to enable "Device_setSysCtlInit" only for GS32F00xx.
 * 20240312,   Zhao lei, update "Device_disableAllPeripherals" "Device_enableAllPeripherals" and "Device_setSysCtlInit" for 1.2.
 * 20240430,   Zhao lei, add autotest apis.
 * 20240514,   Jason,    remove Sysctl_resetDma/QSPI for GS32F3xx
 * 20240515,   Zhao lei, add 3.0 vector remap table.
 * 20240521,   Zhao lei, update testFinished().
 * 20240521,   Zhao lei, init all irq trigger type to be ECLIC_POSTIVE_EDGE_TRIGGER
 * 20240531,   Jason, add empty function Device_initGPIO.
 * 20240617,   Zhao lei, flash_init is called by master only
 * 20240926,   Zhao lei, disable BOR before trim, enable BOR after trim, for 1.2; enable clb/sdfm clk by default
 * 20241008,   Zhao lei, disable all irq in device_int()
 * 20241120,   Zhao lei, remove BOR for 1.2 EC, remove VectorRemap
 * 20250113,   Zhao lei, clean code, move ECLIC irq type initialization to system_dsp.c
 */

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include <stdio.h>

#include "device.h"
#include "printf.h"
#include "inc/hw_types.h"
#include "sysctl.h"

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

void Device_disableAllPeripherals(void);

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

#ifdef __RUNNING_AUTOTEST__
/******************************************************************************
 * @brief    RUNNING_AUTOTEST is defined by GJTester
 ******************************************************************************/
#define WAIT_MAGIC                      0xAA55AA55
#define MAX_TEST_RESULT_INFO_LEN        256

typedef struct TestResult {
    int32_t exitCode;
    char * infoAddr;
    uint32_t infoLen;
}TestResult_t;

static void testInit(void)
{
    SysCtl_setDigSysctlSysSpare1(WAIT_MAGIC);
}

void testFinished(int32_t exitCode, char *fmt, ...)
{
    static TestResult_t testResult;
    static char testInfo[MAX_TEST_RESULT_INFO_LEN];

    va_list va;
    va_start(va, fmt);
    uint32_t infoLen = vsnprintf(testInfo, sizeof(testInfo), fmt, va);
    va_end(va);

    testResult.exitCode = exitCode;
    testResult.infoAddr = testInfo;
    testResult.infoLen = infoLen;

    SysCtl_setDigSysctlSysSpare1((uint32_t)&testResult);
}
#endif

/******************************************************************************
 * @brief    Disable all peripherals
 ******************************************************************************/
void Device_disableAllPeripherals(void)
{
#if IS_GS32F00xx(0x12)
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_FFT );
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_CRC );
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_SDFM );
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_LIN);
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_EPG);
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_ERAD);
#elif IS_GS32F3xx(0x22) || IS_GS32F00xx(0x30)
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_CRC );
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_SDFM );
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_LIN);
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_EPG);
#else
    #error "Device type must be defined."
#endif

    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_CAN);
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_ECAP);
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_CLB);
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_I2C);
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_EPWM);
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_EQEP );
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_SPI  );
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_TIMER );
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_SCI);
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_XBAR);
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_ADC);

#if !IS_GS32F00xx(0x30)   //no dac inside 3.0
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_DAC);
#endif

    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_CMPSS);
    //SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);
}

/******************************************************************************
 * @brief    Lock Flash control register
 ******************************************************************************/
void Flash_LockReg(uint32 flashCtlBase)
{
	uint32 u32FlashCtlValue = 0;
    //disable write protect
    HWREG(flashCtlBase + FLASH_O_UNLOCK1) = FLASH_UNLOCK_CODE1;
    HWREG(flashCtlBase + FLASH_O_UNLOCK2) = FLASH_UNLOCK_CODE2;
    HWREG(flashCtlBase + FLASH_O_UNLOCK1) = FLASH_UNLOCK_CODE3;

#if IS_GS32F00xx(0x30,0x12) || IS_GS32F3xx(0x22)
    //clear write_protect bit
	u32FlashCtlValue = HWREG(flashCtlBase + FLASH_O_WRITE_PROTECT);
	u32FlashCtlValue |= FLASH_CONTROL_WRITE_PROTECT;
	HWREG(flashCtlBase + FLASH_O_WRITE_PROTECT) = u32FlashCtlValue;
#else
    //clear write_protect bit
    u32FlashCtlValue = HWREG(flashCtlBase + FLASH_O_CONTROL);
    u32FlashCtlValue |= FLASH_CONTROL_WRITE_PROTECT;
    HWREG(flashCtlBase + FLASH_O_CONTROL) = u32FlashCtlValue;
#endif
}

/******************************************************************************
 * @brief    Lock Flash opt control register
 ******************************************************************************/
void Flash_OtpLockReg(uint32 flashCtlBase)
{
	uint32 u32FlashCtlValue = 0;
#if IS_GS32F00xx(0x12)
	u32FlashCtlValue |= 0x1;
	HWREG(flashCtlBase + OTP_LOCK) = u32FlashCtlValue;
#else
#endif
}

/******************************************************************************
 * @brief    Get flash lock status
 ******************************************************************************/
uint32 Flash_LockStatus(uint32 flashCtlBase)
{
	return HWREG(flashCtlBase + FLASH_O_WRITE_PROTECT) & FLASH_CONTROL_WRITE_PROTECT;
}

/******************************************************************************
 * @brief    Get flash opt lock status
 ******************************************************************************/
uint32 Flash_OptLockStatus(uint32 flashCtlBase)
{
#if IS_GS32F00xx(0x12)
	return HWREG(flashCtlBase + OPT_UNLOCK) & 0x1;
#endif
}

/******************************************************************************
 * @brief    UnLock flash control reg
 ******************************************************************************/
void Flash_UnLockReg(uint32 flashCtlBase)
{
	uint32 u32FlashCtlValue = 0;
    //disable write protect
    HWREG(flashCtlBase + FLASH_O_UNLOCK1) = FLASH_UNLOCK_CODE1;
    HWREG(flashCtlBase + FLASH_O_UNLOCK2) = FLASH_UNLOCK_CODE2;
    HWREG(flashCtlBase + FLASH_O_UNLOCK1) = FLASH_UNLOCK_CODE3;
#if IS_GS32F00xx(0x30,0x12) || IS_GS32F3xx(0x22)
    //clear write_protect bit
	u32FlashCtlValue = HWREG(flashCtlBase + FLASH_O_WRITE_PROTECT);
	u32FlashCtlValue &= ~FLASH_CONTROL_WRITE_PROTECT;
	HWREG(flashCtlBase + FLASH_O_WRITE_PROTECT) = u32FlashCtlValue;
#else
    //clear write_protect bit
    u32FlashCtlValue = HWREG(flashCtlBase + FLASH_O_CONTROL);
    u32FlashCtlValue &= ~FLASH_CONTROL_WRITE_PROTECT;
    HWREG(flashCtlBase + FLASH_O_CONTROL) = u32FlashCtlValue;
#endif
}

/******************************************************************************
 * @brief    Function to turn on all peripherals,
 *           enabling reads and writes to the peripherals' registers.
 * @Note     To reduce power, unused peripherals should be disabled.
 ******************************************************************************/
void Device_enableAllPeripherals(void)
{
#if IS_GS32F00xx(0x12, 0x30) || IS_GS32F3xx(0x22)

#if IS_GS32F00xx(0x12)
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_FFT );
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_ERAD);
#endif

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_PMBUS);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_LIN);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CRC);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EPG);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CLB);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_SDFM);    //6mA?

#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_GPIO);
#endif

#else
    #error Device GS32F00xx must be defined.
#endif

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TIMER);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_SCI);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_I2C);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_SPI );
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CAN);

#if !IS_GS32F00xx(0x30)   //no dac inside 3.0
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_DAC);
#endif

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_XBAR);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EQEP);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EPWM);      //~30mA
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_ECAP);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CMPSS);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_ADC);       //~10mA

#if IS_GS32F00xx(0x30)
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CLB_ENC);
#endif
}

/******************************************************************************
 * @brief   Function to reset all peripherals
 ******************************************************************************/
void Device_resetAllPeripherals(void)
{
    SysCtl_resetEpwm();
    SysCtl_resetEcap();
    SysCtl_resetEqep();
#if !IS_GS32F00xx(0x30)
    SysCtl_resetDac();
#endif
    SysCtl_resetAdc();
    SysCtl_resetCmpss();
    SysCtl_resetXbar();

    SysCtl_resetCpuTimer();
#if !IS_GS32F00xx(0x30)
    SysCtl_resetGpio();
#endif
    SysCtl_resetI2c();
    SysCtl_resetSci();

    SysCtl_resetCan();
    SysCtl_resetSpi();
    SysCtl_resetSdfm();
    SysCtl_resetClb();
    SysCtl_resetCrc();

    SysCtl_resetLin();

#if IS_GS32F00xx(0x30)
    SysCtl_resetClbEnc();
#endif
}

/******************************************************************************
 * @brief    Function to init flash
 * @details  Setup flash read wait cycles for specified sysclk frequency.
 ******************************************************************************/
void flash_init(uint32 sysClk)
{
	uint32 u32Index = 0;
	uint32 u32CtlBase = 0;
	uint32 u32FlashRegValue = 0;

	uint16_t waitstates = 0;

#if IS_GS32F00xx(0x12)
	uint32 u32EfcCtlBase[] = {FLASH_INTERFACE_EFC_BASE, FLASH_INTERFACE_EFC2_BASE};
	uint32 u32EfcCtlCount = 2;
#elif IS_GS32F00xx(0x30)
	uint32 u32EfcCtlBase[] = {FLASH_INTERFACE_EFC_BASE, FLASH_INTERFACE_EFC2_BASE};
	uint32 u32EfcCtlCount = 2;
#elif IS_GS32F3xx(0x21,0x22)
	uint32 u32EfcCtlBase[] = {FLASH_INTERFACE_EFC_BASE};
	uint32 u32EfcCtlCount = 1;
#else
#error "Use the GS32F00xx or GS32F3xx macro to define the specific chip model."
#endif

	if ( ( sysClk > 0 ) && ( sysClk <= 240000000U ) )
	{
		waitstates = 9;
	}
	else
	{
		waitstates = 10;
	}

	for ( u32Index=0; u32Index<u32EfcCtlCount; u32Index++)
	{
		u32CtlBase = u32EfcCtlBase[u32Index];

		Flash_UnLockReg(u32CtlBase);

	    //wake up sleep mode
	    Flash_wakeFromLPM(u32CtlBase);

	    /* Set waitstates according to frequency. */
	    Flash_setWaitstates(u32CtlBase, waitstates);

#if IS_GS32F00xx(0x30)
	    //close error_indicate,Prevents hardware errors
		u32FlashRegValue = HWREG(u32CtlBase + FLASH_O_STATIC_CFG);
		u32FlashRegValue &= ~FLASH_CONTROL_ERROR_INDICATE;
		HWREG(u32CtlBase + FLASH_O_STATIC_CFG) = u32FlashRegValue;
#else
	    //close error_indicate,Prevents hardware errors
		u32FlashRegValue = HWREG(u32CtlBase + FLASH_O_CONTROL);
		u32FlashRegValue &= ~FLASH_CONTROL_ERROR_INDICATE;
		HWREG(u32CtlBase + FLASH_O_CONTROL) = u32FlashRegValue;
#endif
		//clear all statues
		HWREG(u32CtlBase + FLASH_O_STATUS) = FLASH_STATUS_ALL;

		Flash_LockReg(u32CtlBase);

	    /*
		 * Force a pipeline flush to ensure that the write to the last register
		 * configured occurs before returning.
		 */
	    FLASH_DELAY_CONFIG;
	}
}

/******************************************************************************
 * @brief    Function to init flash wait cycles
 * @details
 ******************************************************************************/
void Flash_initConfig(void)
{
#ifdef FLASH_INTERFACE_EFC_BASE
#if IS_GS32F00xx(0x12)
    if ( 0 != Flash_OptLockStatus(FLASH_INTERFACE_EFC_BASE) )
        Flash_OtpLockReg(FLASH_INTERFACE_EFC_BASE);
#endif
    if ( 0 == Flash_LockStatus(FLASH_INTERFACE_EFC_BASE) )
        Flash_LockReg(FLASH_INTERFACE_EFC_BASE);
#endif
    flash_init(DEVICE_SYSCLK_FREQ);
}

/******************************************************************************
 * @brief    Function to init HRPWM
 * @details  Setup HRPWM DLL configure registers
 ******************************************************************************/
void HRPWM_initConfig(void)
{
#if IS_GS32F00xx(0x12)
    EPWM_setHrpwmDllCfg1(EPWMTOP_BASE, 0x07FFF800); //enable EPWM1A~EPWM8B

	if(!HWREGB(OPTION_BITS_BASE + 6)& 0x1){
		HWREGH(EPWM1_BASE + HRPWM_O_HRMSTEP) = 64;
	}
	else{
		uint8_t temp_cnt = 7;
		while(temp_cnt--){
		 if(HWREGB(OPTION_BITS_BASE + temp_cnt) == 0xFF){
			 break;}
		}
		if(temp_cnt){
			HWREGH(EPWM1_BASE + HRPWM_O_HRMSTEP) = 32;
		}
		else{
			HWREGH(EPWM1_BASE + HRPWM_O_HRMSTEP) = 64;
		}
	}
#elif IS_GS32F00xx(0x30) || IS_GS32F3xx(0x22)
    EPWM_setHrpwmDllCfg0(EPWMTOP_BASE, 0x00000001);
   //init EPWMXLINK3 register to make each channel AQCSFRC linked to itself
    EPWM_setupEPWMLinks(EPWM1_BASE, EPWM_LINK_WITH_EPWM_1, EPWM_LINK_AQCSFRC);
    EPWM_setupEPWMLinks(EPWM2_BASE, EPWM_LINK_WITH_EPWM_2, EPWM_LINK_AQCSFRC);
    EPWM_setupEPWMLinks(EPWM3_BASE, EPWM_LINK_WITH_EPWM_3, EPWM_LINK_AQCSFRC);
    EPWM_setupEPWMLinks(EPWM4_BASE, EPWM_LINK_WITH_EPWM_4, EPWM_LINK_AQCSFRC);
    EPWM_setupEPWMLinks(EPWM5_BASE, EPWM_LINK_WITH_EPWM_5, EPWM_LINK_AQCSFRC);
    EPWM_setupEPWMLinks(EPWM6_BASE, EPWM_LINK_WITH_EPWM_6, EPWM_LINK_AQCSFRC);
    EPWM_setupEPWMLinks(EPWM7_BASE, EPWM_LINK_WITH_EPWM_7, EPWM_LINK_AQCSFRC);
    EPWM_setupEPWMLinks(EPWM8_BASE, EPWM_LINK_WITH_EPWM_8, EPWM_LINK_AQCSFRC);
#if IS_GS32F3xx(0x22)
    EPWM_setupEPWMLinks(EPWM9_BASE, EPWM_LINK_WITH_EPWM_9, EPWM_LINK_AQCSFRC);
    EPWM_setupEPWMLinks(EPWM10_BASE, EPWM_LINK_WITH_EPWM_10, EPWM_LINK_AQCSFRC);
    EPWM_setupEPWMLinks(EPWM11_BASE, EPWM_LINK_WITH_EPWM_11, EPWM_LINK_AQCSFRC);
    EPWM_setupEPWMLinks(EPWM12_BASE, EPWM_LINK_WITH_EPWM_12, EPWM_LINK_AQCSFRC);
    EPWM_setupEPWMLinks(EPWM13_BASE, EPWM_LINK_WITH_EPWM_13, EPWM_LINK_AQCSFRC);
    EPWM_setupEPWMLinks(EPWM14_BASE, EPWM_LINK_WITH_EPWM_14, EPWM_LINK_AQCSFRC);
    EPWM_setupEPWMLinks(EPWM15_BASE, EPWM_LINK_WITH_EPWM_15, EPWM_LINK_AQCSFRC);
    EPWM_setupEPWMLinks(EPWM16_BASE, EPWM_LINK_WITH_EPWM_16, EPWM_LINK_AQCSFRC);
    EPWM_setupEPWMLinks(EPWM17_BASE, EPWM_LINK_WITH_EPWM_17, EPWM_LINK_AQCSFRC);
    EPWM_setupEPWMLinks(EPWM18_BASE, EPWM_LINK_WITH_EPWM_18, EPWM_LINK_AQCSFRC);

#endif


#ifdef DEVICE_AHBCLK_DIV
	#ifdef DEVICE_PLLCLK_FREQ
	uint16_t HRPWMCLK_Ratio  = (DEVICE_PLLCLK_FREQ/DEVICE_AHBCLK_FREQ);
	#else
	uint16_t HRPWMCLK_Ratio  = (DEVICE_SYSCLK_FREQ*2/DEVICE_AHBCLK_FREQ);
	#endif
#else
	#ifdef DEVICE_PLLCLK_FREQ
	uint16_t HRPWMCLK_Ratio  = (DEVICE_PLLCLK_FREQ/(DEVICE_SYSCLK_FREQ/2));
	#else
	uint16_t HRPWMCLK_Ratio  = (DEVICE_SYSCLK_FREQ*2/(DEVICE_SYSCLK_FREQ/2));
	#endif
#endif


#if IS_GS32F00xx(0x30)
    //HRPWM EN
    switch(HRPWMCLK_Ratio)
    {
    	case 2: HWREGH(0x400CF720) = 0;
    			break;
    	case 4:	HWREGH(0x400CF720) = 1;
    			break;
    	case 8: HWREGH(0x400CF720) = 2;
    			break;
    	default:
    			break;
    }

    //HRPWMUP-Down EN
    for(uint32_t i = 0,HRPWM_StmEnabled = HRPWM_UpDownDutyModeEnable_Num;
    	HRPWM_StmEnabled != 0 && i < sizeof(uint32_t)*8;
    	i++,HRPWM_StmEnabled >>= 1
		)
    {
    	if(HRPWM_StmEnabled&0x01)
    	{
    		HWREGH( EPWM1_BASE+0x800*i + EPWM_O_CMPSEL) |= 0x5;
    		HWREGH( EPWM1_BASE+0x800*i + EPWM_O_CMPSEL) |= 0x5;
    	}

    }


#endif

#endif

#if IS_GS32F00xx() && defined(ENABLE_HRPWM)
    //EPWM_setHrpwmDllCfg0(PREEPWM_BASE,0xFE1FCB83);
    EPWM_setHrpwmDllCfg0(PREEPWM_BASE,0x7E1FCB83);		//lcl_en = 0
    EPWM_setHrpwmDllCfg1(PREEPWM_BASE,0x603);
    EPWM_setEpwmtopHrpwmBypass(PREEPWM_BASE,0);
#endif
}

/******************************************************************************
 * @brief    Initialize XTAL, PLL, AHB/APB/CAN divider
 * @details
 ******************************************************************************/
void Clock_init(void)
{
    /* Initialize MCU clock: */
#ifdef DEVICE_AHBCLK_DIV
# ifdef DEVICE_PLLCLK_FREQ
    sysctl_clock_init(DEVICE_PLLCLK_FREQ, DEVICE_OSCSRC_FREQ, DEVICE_SYSCLK_DIV, DEVICE_AHBCLK_DIV, DEVICE_APBCLK_DIV);
# else
    sysctl_clock_init(DEVICE_SYSCLK_FREQ*2, DEVICE_OSCSRC_FREQ, 2, DEVICE_AHBCLK_DIV, DEVICE_APBCLK_DIV);
# endif
#else
# ifdef DEVICE_PLLCLK_FREQ
    sysctl_clock_init(DEVICE_PLLCLK_FREQ, DEVICE_OSCSRC_FREQ, DEVICE_SYSCLK_DIV, 2, DEVICE_APBCLK_DIV);
# else
    sysctl_clock_init(DEVICE_SYSCLK_FREQ*2, DEVICE_OSCSRC_FREQ, 2, 2, DEVICE_APBCLK_DIV);
# endif

#endif
}

/******************************************************************************
 * @brief    Initialize the device - clock, sysctl, flash, ECLIC, other peripherals
 * @details  Function to initialize the device. Primarily initializes system
 *           control to a known state by disabling the watchdog, setting up the
 *           SYSCLKOUT frequency, and enabling the clocks to the peripherals.
 ******************************************************************************/
void Device_init(void)
{
#ifdef __RUNNING_AUTOTEST__
    testInit();
#endif

    /* disable all peripherals' clocks */
    Device_disableAllPeripherals();

    /* reset all peripherals */
    Device_resetAllPeripherals();

#if IS_GS32F00xx(0x30)
    SysCtl_setObsSigDiv(1);
#if (GS32_PART_NUM==0x5000 || GS32_PART_NUM==0x4000)
    /* Enable IO function for MT5000 or MT4000 models 48,49,56,57 */
    SYSCTL_EnableIO_MT48_49_56_57();
#endif
#endif

    /* analog peripheral trim */
    ASysCtl_trim();

    //SysCtl_disableWatchdog();

#if !defined(STARTUP_SYSCLK_FREQ) || (defined(STARTUP_SYSCLK_FREQ) && BOOT_HARTID == 0 && (STARTUP_SYSCLK_FREQ < DEVICE_SYSCLK_FREQ))
    Flash_initConfig();
    Clock_init();           /* ~6000 cycles for internal OSC, ~35000 cycles for HSE crystal */
#else
    Clock_init();           /* ~6000 cycles for internal OSC, ~35000 cycles for HSE crystal */
    Flash_initConfig();
#endif

    /* enable all peripherals' clocks */
    Device_enableAllPeripherals();

    /* ADC specific initialization */
    ASysCtl_initConfig();

    /* HRPWM DLL specific initialization */
    HRPWM_initConfig();
}

/******************************************************************************
 * @brief     Function to initialize a SCI peripheral
 * @param[in] SCIBase, the SCI module base address.
 * @param[in] baud, the SCI baudrate.
 ******************************************************************************/
#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)
void UartPrint_init(uint32_t SCIBase, uint32_t baud)
{

#if !IS_GS32_PROTOTYPE()
	if (SCIBase == SCIA_BASE) {
#if IS_GS32F00xx(0x12)
        GPIO_setPadConfig(28, GPIO_PIN_TYPE_PULLUP);
        GPIO_setPadConfig(29, GPIO_PIN_TYPE_PULLUP);
#elif IS_GS32F3xx(0x22)
        GPIO_setPadConfig(42, GPIO_PIN_TYPE_PULLUP);
        GPIO_setPadConfig(43, GPIO_PIN_TYPE_PULLUP);
#endif
		GPIO_setPinConfig(SCIA_TX_PIN);
	    GPIO_setPinConfig(SCIA_RX_PIN);
	} else {
        GPIO_setPadConfig(14, GPIO_PIN_TYPE_PULLUP);
        GPIO_setPadConfig(15, GPIO_PIN_TYPE_PULLUP);

	    GPIO_setPinConfig(SCIB_TX_PIN);
	    GPIO_setPinConfig(SCIB_RX_PIN);
	}
#endif

    SCI_setBaud(SCIBase, DEVICE_APBCLK_FREQ, baud);
    SCI_setInitConfig(SCIBase, SCI_CONFIG_PAR_NONE, SCI_STOP_1BIT, SCI_CHAR_8BITS);

    SCI_enableFIFO(SCIBase);
    SCI_setFIFOInterruptLevel(SCIBase, SCI_FIFO_TX0, SCI_FIFO_RX1);
}

#endif

#if IS_GS32F00xx(0x30) || IS_GS32F3xx(0x23)
#if GS32_PART_NUM==0x035 || GS32_PART_NUM==0x027

void UartPrint_init(uint32_t SCIBase, uint32_t baud)
{
    GPIO_setPinConfig(GPIO_29_SCIA_TX); //GPIO_29_SCIA_TX
    GPIO_setPinConfig(GPIO_28_SCIA_RX); //GPIO_28_SCIA_RX

    GPIO_setPadConfig(29,GPIO_PIN_TYPE_PULLUP);
    GPIO_setPadConfig(28,GPIO_PIN_TYPE_PULLUP);

    GpioCtrlRegs.GPAPUD.bit.GPIO28 = 0x0;	//0:  Pull-Up Enable

    SCI_performSoftwareReset(SCIBase);
    SCI_setConfig(SCIBase, DEVICE_APBCLK_FREQ, baud,(SCI_CONFIG_WLEN_8|SCI_CONFIG_STOP_ONE|SCI_CONFIG_PAR_NONE));
    SCI_resetChannels(SCIBase);
    SCI_clearInterruptStatus(SCIBase, SCI_INT_TXFF | SCI_INT_RXFF);

//	SCI_enableFIFO(SCIBase);
//	SCI_setFIFOInterruptLevel(SCIBase, SCI_FIFO_TX0, SCI_FIFO_RX8);
    SCI_disableFIFO(SCIBase);
    SCI_enableModule(SCIBase);
    SCI_performSoftwareReset(SCIBase);

//	SCI_setFIFOInterruptLevel(SCIBase, SCI_FIFO_TX0, SCI_FIFO_RX8);
//	SCI_enableInterrupt(SCIBase, SCI_INT_TXFF | SCI_INT_RXFF);

}

#else

void UartPrint_init(uint32_t SCIBase, uint32_t baud)
{
    GPIO_setPinConfig(SCIA_TX_PIN);
    GPIO_setPinConfig(SCIA_RX_PIN);
#if GS32_PART_NUM==0x5000 || GS32_PART_NUM==0x4000
    GPIO_setPadConfig(62, GPIO_PIN_TYPE_PULLUP);
    GPIO_setPadConfig(63, GPIO_PIN_TYPE_PULLUP);
#else
    GPIO_setPadConfig(28, GPIO_PIN_TYPE_PULLUP);
    GPIO_setPadConfig(29, GPIO_PIN_TYPE_PULLUP);
#endif


    SCI_performSoftwareReset(SCIBase);
    SCI_setConfig(SCIBase, DEVICE_APBCLK_FREQ, baud,(SCI_CONFIG_WLEN_8|SCI_CONFIG_STOP_ONE|SCI_CONFIG_PAR_NONE));
    SCI_resetChannels(SCIBase);
    SCI_clearInterruptStatus(SCIBase, SCI_INT_TXFF | SCI_INT_RXFF);

    //	SCI_enableFIFO(SCIBase);
    //	SCI_setFIFOInterruptLevel(SCIBase, SCI_FIFO_TX0, SCI_FIFO_RX8);
    SCI_disableFIFO(SCIBase);
    SCI_enableModule(SCIBase);
    SCI_performSoftwareReset(SCIBase);

//	SCI_setFIFOInterruptLevel(SCIBase, SCI_FIFO_TX0, SCI_FIFO_RX8);
//	SCI_enableInterrupt(SCIBase, SCI_INT_TXFF | SCI_INT_RXFF);

}

#endif


#endif

/*
 * @brief   output one char to the specified UART TxFIFO.
 *          weak symbol used in printf.
 */
__WEAK int _write(int fd, char *pBuffer, int size)
{
    UART_writeCharArray(LOG_SCI_BASE, pBuffer, size);
    return size;
}

/*
 * @brief   output one char to the specified UART TxFIFO.
 *          weak symbol used in printf.
 */
__WEAK void _putchar(char ch)
{
    if (ch == '\n') {
        char r = '\r';
        UART_writeCharArray(LOG_SCI_BASE, &r, 1);
    }
    UART_writeCharArray(LOG_SCI_BASE, &ch, 1);
}

/*
 * @brief   output one char to the specified UART TxFIFO.
 *          weak symbol used in printf.
 */
__WEAK int __io_putchar(char ch)
{
    UART_writeCharBlockingFIFO(LOG_SCI_BASE, ch);
    return ch;
}

/*
 * @brief   output one char to the specified UART TxFIFO
 */
__WEAK int fputc(int ch, FILE *f)
{
    UART_writeCharBlockingFIFO(LOG_SCI_BASE, ch);
    return ch;
}

#ifdef  DEBUG
/**
  * \brief  Reports the name of the source file and the source line number
  *         where the assert_param error has occurred.
  * \param  file: pointer to the source file name
  * \param  line: assert_param error line source number
  * \retval None
  */
__WEAK void __error__(const char *file, uint32_t line)
{
  /* User can add his own implementation to report the file name and line number,
     ex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */
  printf("Assert error in file %s line %d\r\n", file, line);
  ESTOP0;
}
#endif /* DEBUG */


#ifdef __cplusplus
}
#endif


/*****END OF FILE****/
