/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/*
 * @File adc_ex05_soc_continuous.c
 *
 * @Tile ADC synchronous software triggering.
 *
 * This example sets up the ADC to convert continuously,
 * achieving maximum sampling rate.
 *
 * @External Connections:
 * - A0 pin should be connected to signal to convert
 *
 * @Watch Variables:
 * - adcAResults - A sequence of analog-to-digital conversion samples from pin A0.
 * 	 			   The time between samples is the minimum possible based on the
 * 	 			   ADC speed.
 */

#ifdef __cplusplus
extern "C"
{
#endif

/*
 * Include Files
 */
#include "adc_ex05_board.h"

/*
 * Macros & Typedefs
 */
/* buffer for storing conversion results(size must be multiple of 16) */
#define RESULTS_BUFFER_SIZE     256

/*
 * Global Constants
 */
/* buffer for storing conversion results */
uint16_t adcAResults[RESULTS_BUFFER_SIZE];
uint16_t resultsIndex;

/* Main */
void main(void)
{
	/* Initialize device clock and peripherals */
    Device_init();

    /* Board Initialization
     * - Configure the ADC and power it up
     * - Setup the ADC for continuous conversions on channel 0
     *
     */
    Board_init();

    /* Initialize results buffer */
    for(resultsIndex = 0; resultsIndex < RESULTS_BUFFER_SIZE; resultsIndex++)
    {
        adcAResults[resultsIndex] = 0;
    }
    resultsIndex = 0;

    /* Enable global Interrupts */
    EINT;

    /* Take conversions indefinitely in loop */
    do
    {
    	/* Enable ADC interrupts */
    	AdcaRegs.ADCINTSEL1N2.bit.INT1E = 1;
    	AdcaRegs.ADCINTSEL1N2.bit.INT2E = 1;
    	AdcaRegs.ADCINTSEL3N4.bit.INT3E = 1;
    	AdcaRegs.ADCINTSEL3N4.bit.INT4E = 1;

    	/* Clear all interrupts flags(INT1-4) */
        HWREGH(myADC0_BASE + ADC_O_INTFLGCLR) = 0x000F;

        /* Initialize results index */
        resultsIndex = 0;

        /* Software force start SOC0 to SOC7 */
        HWREGH(myADC0_BASE + ADC_O_SOCFRC1) = 0x00FF;

        /* Keep taking samples until the results buffer is full */
        while(resultsIndex < RESULTS_BUFFER_SIZE)
        {
        	/* Wait for first set of 8 conversions to complete */
            while(false == AdcaRegs.ADCINTFLG.bit.ADCINT3);

            /* Clear the interrupt flag */
            AdcaRegs.ADCINTFLGCLR.bit.ADCINT3 = 1;

            /*
             * Save results for first 8 conversions
             *
             * Note that during this time, the second 8 conversions have
             * already been triggered by EOC6->ADCIN1 and will be actively
             * converting while first 8 results are being saved
             */
            adcAResults[resultsIndex++] = AdcaResultRegs.ADCRESULT0;
            adcAResults[resultsIndex++] = AdcaResultRegs.ADCRESULT1;
            adcAResults[resultsIndex++] = AdcaResultRegs.ADCRESULT2;
            adcAResults[resultsIndex++] = AdcaResultRegs.ADCRESULT3;
            adcAResults[resultsIndex++] = AdcaResultRegs.ADCRESULT4;
            adcAResults[resultsIndex++] = AdcaResultRegs.ADCRESULT5;
            adcAResults[resultsIndex++] = AdcaResultRegs.ADCRESULT6;
            adcAResults[resultsIndex++] = AdcaResultRegs.ADCRESULT7;

            /* Wait for the second set of 8 conversions to complete */
            while(false == AdcaRegs.ADCINTFLG.bit.ADCINT4);

            /* Clear the interrupt flag */
            AdcaRegs.ADCINTFLGCLR.bit.ADCINT4 = 1;

            /*
             * Save results for first 8 conversions
             *
             * Note that during this time, the second 8 conversions have
             * already been triggered by EOC14->ADCIN2 and will be actively
             * converting while second 8 results are being saved
             */
            adcAResults[resultsIndex++] = AdcaResultRegs.ADCRESULT8;
            adcAResults[resultsIndex++] = AdcaResultRegs.ADCRESULT9;
            adcAResults[resultsIndex++] = AdcaResultRegs.ADCRESULT10;
            adcAResults[resultsIndex++] = AdcaResultRegs.ADCRESULT11;
            adcAResults[resultsIndex++] = AdcaResultRegs.ADCRESULT12;
            adcAResults[resultsIndex++] = AdcaResultRegs.ADCRESULT13;
            adcAResults[resultsIndex++] = AdcaResultRegs.ADCRESULT14;
            adcAResults[resultsIndex++] = AdcaResultRegs.ADCRESULT15;
        }

        /* Disable all ADCINT flags to stop sampling */
    	AdcaRegs.ADCINTSEL1N2.bit.INT1E = 0;
    	AdcaRegs.ADCINTSEL1N2.bit.INT2E = 0;
    	AdcaRegs.ADCINTSEL3N4.bit.INT3E = 0;
    	AdcaRegs.ADCINTSEL3N4.bit.INT4E = 0;

    }
    while(1);
}
#ifdef __cplusplus
}
#endif
