/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/*
 * @File adc_ex06_soc_continuous_dma.c
 *
 * @Tile ADC continuous conversions read by DMA.
 *
 * @External Connections:
 * - A0 & A1 & C3 pins should be connected to signals to convert
 *
 * @Watch Variables:
 * - ADCAResult0_AVERAGE - The 8th average of the voltage on pin A0
 * - ADCAResult1_AVERAGE - The 8th average of the voltage on pin A1
 * - myADC1DataBuffer - a digital representation of the voltage on pin C3
 *
 */

#ifdef __cplusplus
extern "C"{
#endif

/*
 * Include Files
 */
#include "adc_ex06_board.h"

/*
 * Macros & Typedefs
 */
#define RESULTS_BUFFER_SIZE     512
#define ADCA_RESULTS_BUFFER_SIZE	16
#define DMA_INT_CH1_IRQn        (INT_DMA1_CH1)
#define DMA_INT_CH2_IRQn        (INT_DMA1_CH2)

/*
 *  Local Variables
 */
uint16_t myADC0DataBuffer[ADCA_RESULTS_BUFFER_SIZE];
uint16_t myADC1DataBuffer[RESULTS_BUFFER_SIZE];

/*
 * Global Variables
 */
volatile uint32_t ADCAResult0_AVERAGE;
volatile uint32_t ADCAResult1_AVERAGE;
volatile uint16_t done2;
volatile uint16_t idx2;


/*
 * Local Function Prototypes
 */
void initializeDMA(void);
void configureDMAChannels(void);
void configureEPWM(uint32_t epwmBase);

/*
 * ISR Function Prototypes
 */
void DMA_CH1_IRQHandler();
void DMA_CH2_IRQHandler();

void main(void)
{
    uint16_t resultsIndex;

    /*
     * Initialize device clock and peripherals
     */
    Device_init();

    /*
     * Board Initializatrion
     */
    Board_init();

#if IS_GS32F3xx(0x22)
    Interrupt_register(INT_DMA1_CH1, DMA_CH1_IRQHandler);
    Interrupt_register(INT_DMA1_CH2, DMA_CH2_IRQHandler);
    Interrupt_enable(INT_DMA1_CH1);
    Interrupt_enable(INT_DMA1_CH2);

#elif IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30)
    Interrupt_register(INT_DMA1_CH1, DMA_CH1_IRQHandler);
    Interrupt_register(INT_DMA1_CH2, DMA_CH2_IRQHandler);
    Interrupt_enable(INT_DMA1_CH1);
    Interrupt_enable(INT_DMA1_CH2);
#endif

    idx2 = 1;

    /*
     * Stop the ePWM clock
     */
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);

    /*
     * Set up function for ePWM 2
     */
    configureEPWM(EPWM2_BASE);

    /*
     * Start the ePWM clock
     */
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);

    /*
     * Initialize the DMA & configure DMA channels 1 & 2
     */
    initializeDMA();
    configureDMAChannels();

    /*
     * Clearing all pending interrupt flags
     */
#if IS_GS32F3xx(0x22)
    XDMA_clearTriggerFlag(DMA1_CH1_BASE);
    XDMA_clearTriggerFlag(DMA1_CH2_BASE);
#elif IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30)
    DMA_clearTriggerFlag(DMA_CH1_BASE);
    DMA_clearTriggerFlag(DMA_CH2_BASE);
#endif
    ADC_clearInterruptStatus(myADC0_BASE,(ADC_INT_NUMBER1 | ADC_INT_NUMBER2));
    ADC_clearInterruptStatus(myADC1_BASE,(ADC_INT_NUMBER1 | ADC_INT_NUMBER2));
    EPWM_forceADCTriggerEventCountInit(EPWM2_BASE, EPWM_SOC_A);
    EPWM_clearADCTriggerFlag(EPWM2_BASE, EPWM_SOC_A);

    /*
     * Enable continuous operation by setting the last SOC to re-trigger the first.
     */
    ADC_setInterruptSOCTrigger(myADC0_BASE, ADC_SOC_NUMBER0,
                               ADC_INT_SOC_TRIGGER_ADCINT2);
    ADC_setInterruptSOCTrigger(myADC1_BASE, ADC_SOC_NUMBER0,
                               ADC_INT_SOC_TRIGGER_ADCINT2);

    /*
     * Enable global Interrupts and higher priority real-time debug events:
     */
    EINT;  // Enable Global interrupt INTM

    /*
     * Start DMA
     */
#if IS_GS32F3xx(0x22)
    XDMA_startChannel(DMA1_CH1_BASE);
    XDMA_startChannel(DMA1_CH2_BASE);
#elif IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30)
    DMA_startChannel(DMA_CH1_BASE);
    DMA_startChannel(DMA_CH2_BASE);
#endif
    /*
     * Finally, enable the SOCA trigger from ePWM.
     * This will kick off conversions at the next ePWM event.
     */
    EPWM_enableADCTrigger(EPWM2_BASE, EPWM_SOC_A);

    /*
     * Loop until the ISR signals the transfer is complete
     */
    while(done2 == 0)
    {
        __asm(" NOP");
    }
    while(1);

}
#if IS_GS32F3xx(0x22)
void DMA_CH1_IRQHandler()
{
	done2 = 1;
    XDMA_stopChannel(DMA1_CH1_BASE);////stop before clear,so will not DMA again
    XDMA_clearInterrupt(DMA1_CH1_BASE,XDMA_INT_BLOCK);////The next round will only start when the interrupt flag is cleared
    ADC_setInterruptSOCTrigger(myADC0_BASE, ADC_SOC_NUMBER0,ADC_INT_SOC_TRIGGER_NONE);
}
#elif IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30)

void DMA_CH1_IRQHandler()
{
	uint16_t i;

	ADCAResult0_AVERAGE = 0;

	ADCAResult1_AVERAGE = 0;

	for (i = 0; i < ADCA_RESULTS_BUFFER_SIZE; )
	{
		ADCAResult0_AVERAGE += myADC0DataBuffer[i];
		ADCAResult1_AVERAGE += myADC0DataBuffer[i+1];

		i = i + 2;
	}

	ADCAResult0_AVERAGE = ADCAResult0_AVERAGE / 8;

	ADCAResult1_AVERAGE = ADCAResult1_AVERAGE / 8;


    /* Clear DMA Block flag. */
    DMA_clearInterrupt(DMA_CH1_BASE,DMA_INT_BLOCK);
}

#endif


#if IS_GS32F3xx(0x22)
void DMA_CH2_IRQHandler()
{
	done2 = 1;
    XDMA_stopChannel(DMA1_CH2_BASE);//stop before clear,so will not DMA again
    XDMA_clearInterrupt(DMA1_CH2_BASE,XDMA_INT_BLOCK);//The next round will only start when the interrupt flag is cleared
    ADC_setInterruptSOCTrigger(myADC0_BASE, ADC_SOC_NUMBER0,ADC_INT_SOC_TRIGGER_NONE);
}
#elif IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30)
void DMA_CH2_IRQHandler()
{
    if(idx2 >= (RESULTS_BUFFER_SIZE/16 - 1))
    {
       done2 = 1;

       /* Stop channel before clear,so will not DMA again. */
       DMA_stopChannel(DMA_CH2_BASE);

       /* Clear DMA Block flag. */
       DMA_clearInterrupt(DMA_CH2_BASE,DMA_INT_BLOCK);

       /* Cancel interrupt triggers SOC */
       ADC_setInterruptSOCTrigger(myADC1_BASE, ADC_SOC_NUMBER0,ADC_INT_SOC_TRIGGER_NONE);
    }
    idx2++;
    DMA_clearInterrupt(DMA_CH2_BASE,DMA_INT_BLOCK);
    DMA_getInterruptStatus(DMA_CH2_BASE);
}

#endif

void configureEPWM(uint32_t epwmBase)
{
	/* Make the timer count up with a period of 40us */
    HWREGH(epwmBase + EPWM_O_TBCTL) = 0x0000U;
    EPWM_setTimeBasePeriod(epwmBase, 4000U);

    /* Set the A output on zero and reset on CMPA */
    EPWM_setActionQualifierAction(epwmBase, EPWM_AQ_OUTPUT_A,
                                            EPWM_AQ_OUTPUT_HIGH,
                                            EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);
    EPWM_setActionQualifierAction(epwmBase, EPWM_AQ_OUTPUT_A,
                                            EPWM_AQ_OUTPUT_LOW,
                                            EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);

    /* Set CMPA to 20us to get a 50% duty */
    EPWM_setCounterCompareValue(epwmBase, EPWM_COUNTER_COMPARE_A, 2000U);

    /* Start ADC when timer equals zero (note: don't enable yet) */
    EPWM_setADCTriggerSource(epwmBase, EPWM_SOC_A, EPWM_SOC_TBCTR_ZERO);
    EPWM_setADCTriggerEventPrescale(epwmBase, EPWM_SOC_A, 1U);

    /*
     * Enable initialization of the SOCA event counter. Since we are
     * disabling the ETSEL.SOCAEN bit, we need a way to reset the SOCACNT.
     * Hence, enable the counter initialize control.
     */
    EPWM_enableADCTriggerEventCountInit(epwmBase, EPWM_SOC_A);
}

void initializeDMA(void)
{
    /* Perform a hard reset on DMA */
#if IS_GS32F3xx(0x22)
    XDMA_initController(DMA1_BASE);
    XDMA_enableInterrupt(DMA1_BASE);
#elif IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30)
    DMA_initController(DMA_BASE);
    DMA_enableInterrupt(DMA_BASE);
#endif
}

/*
 * configureDMAChannels - Initialize DMA ch 1 to transfer ADCA results
 *                        and DMA ch 2 to transfer ADCB results
 */
void configureDMAChannels(void)
{
#if IS_GS32F3xx(0x22)
    XDMA_ConfigParams dma1Cfg = {0};
    dma1Cfg.enableInterrupt = 1;
    //enable DMA Trigger by DMA MUX
    dma1Cfg.dmaSrcReqId = DMAMUX_ReqId_Trig_ADCA_2int;
    dma1Cfg.srcAddr = (uint32_t)ADCARESULT_BASE;
    dma1Cfg.destAddr = (uint32_t)myADC0DataBuffer;
    dma1Cfg.blockTS = ADCA_RESULTS_BUFFER_SIZE / 2;
    dma1Cfg.ttfc    = XDMA_TT_FC_2_P2M_DMAC;
    dma1Cfg.srcBtl  = XDMA_BTL_1;
    dma1Cfg.reloadSrc = false;
    dma1Cfg.reloadDst = false;
    dma1Cfg.destBtl = XDMA_BTL_1;
    dma1Cfg.srcAddrDirect = XDMA_ADDR_NO_CHANGE;
    dma1Cfg.destAddrDirect = XDMA_ADDR_INCRE;
    dma1Cfg.srcTrWidthBytes = XDMA_TR_WIDTH_BYTE_4;
    dma1Cfg.destTrWidthBytes= XDMA_TR_WIDTH_BYTE_4;
    
    XDMA_configChannel(DMA1_CH1_BASE, &dma1Cfg);
    XDMA_clearInterrupt(DMA1_CH1_BASE, XDMA_INT_BLOCK);
    XDMA_unMaskInterrupt(DMA1_CH1_BASE, XDMA_INT_BLOCK);
    
    XDMA_ConfigParams dma2Cfg = {0};
    dma2Cfg.enableInterrupt = 1;
    dma2Cfg.dmaSrcReqId = DMAMUX_ReqId_Trig_ADCC_2int;
    dma2Cfg.srcAddr = (uint32_t)ADCCRESULT_BASE;
    dma2Cfg.destAddr = (uint32_t)myADC1DataBuffer;
    dma2Cfg.blockTS = 64;
    dma2Cfg.ttfc    = XDMA_TT_FC_2_P2M_DMAC;
    dma2Cfg.srcBtl  = XDMA_BTL_1;
    dma2Cfg.reloadSrc = false;
    dma2Cfg.reloadDst = false;
    dma2Cfg.destBtl = XDMA_BTL_1;
    dma2Cfg.srcAddrDirect = XDMA_ADDR_NO_CHANGE;
    dma2Cfg.destAddrDirect = XDMA_ADDR_INCRE;
    dma2Cfg.srcTrWidthBytes = XDMA_TR_WIDTH_BYTE_16;
    dma2Cfg.destTrWidthBytes= XDMA_TR_WIDTH_BYTE_16;

    XDMA_configChannel(DMA1_CH2_BASE, &dma2Cfg);
    XDMA_clearInterrupt(DMA1_CH2_BASE, XDMA_INT_BLOCK);
    XDMA_unMaskInterrupt(DMA1_CH2_BASE, XDMA_INT_BLOCK);
#elif IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30)
    DMA_ConfigParams dma1Cfg = {0};
    dma1Cfg.enableInterrupt = 1;
    /* enable DMA Trigger by DMA MUX */
    dma1Cfg.dmaSrcReqId = DMAMUX_ReqId_Trig_ADCA_2int;
    dma1Cfg.srcAddr = (uint32_t)ADCARESULT_BASE;
    dma1Cfg.destAddr = (uint32_t)myADC0DataBuffer;
    dma1Cfg.blockTS = ADCA_RESULTS_BUFFER_SIZE / 2;
    dma1Cfg.ttfc    = DMA_TT_FC_2_P2M_DMAC;
    dma1Cfg.srcBtl  = DMA_BTL_1;
    dma1Cfg.reloadSrc = false;
    dma1Cfg.reloadDst = true;
    dma1Cfg.destBtl = DMA_BTL_1;
    dma1Cfg.srcAddrDirect = DMA_ADDR_NO_CHANGE;
    dma1Cfg.destAddrDirect = DMA_ADDR_INCRE;
    dma1Cfg.srcTrWidthBytes = DMA_TR_WIDTH_BYTE_4;
    dma1Cfg.destTrWidthBytes= DMA_TR_WIDTH_BYTE_4;

    DMA_configChannel(DMA_CH1_BASE, &dma1Cfg);
    DMA_clearInterrupt(DMA_CH1_BASE, DMA_INT_BLOCK);
    DMA_unMaskInterrupt(DMA_CH1_BASE, DMA_INT_BLOCK);

    DMA_ConfigParams dma2Cfg = {0};
    dma2Cfg.enableInterrupt = 1;
    dma2Cfg.dmaSrcReqId = DMAMUX_ReqId_Trig_ADCC_2int;
    dma2Cfg.srcAddr = (uint32_t)ADCCRESULT_BASE;
    dma2Cfg.destAddr = (uint32_t)myADC1DataBuffer;
    dma2Cfg.blockTS = 16;
    dma2Cfg.ttfc    = DMA_TT_FC_2_P2M_DMAC;
    dma2Cfg.srcBtl  = DMA_BTL_1;
    dma2Cfg.reloadSrc = true;
    dma2Cfg.destBtl = DMA_BTL_1;
    dma2Cfg.srcAddrDirect = DMA_ADDR_INCRE;
    dma2Cfg.destAddrDirect = DMA_ADDR_INCRE;
    dma2Cfg.srcTrWidthBytes = DMA_TR_WIDTH_BYTE_2;
    dma2Cfg.destTrWidthBytes= DMA_TR_WIDTH_BYTE_2;

    DMA_configChannel(DMA_CH2_BASE, &dma2Cfg);
    DMA_clearInterrupt(DMA_CH2_BASE, DMA_INT_BLOCK);
    DMA_unMaskInterrupt(DMA_CH2_BASE, DMA_INT_BLOCK);
#endif

}


#ifdef __cplusplus
}
#endif

