/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef __cplusplus
extern "C"
{
#endif

/*
 * Include Files
 */
#include "adc_ex07_board.h"
#include "cpu.h"

/*
 * Macros & Typedefs
 */
#define EPWM_MODE_SELECT EPWM_MODE_SPWM_0

/*
 * @brief Board Configurations
 *
 * Initializes the rest of the modules.
 * Call this function in your application if you wish to do all module
 * initialization.
 * If you wish to not use some of the initializations, instead of the
 * Board_init use the individual Module_inits.
 */
void Board_init()
{
	myADC0_init();
	myADC1_init();
}
void myADC0_init()
{
	/*
	 * @brief ADC Initialization:
	 * 		  Write ADC configurations and power up the ADC.
	 */
	/* Select ADC Reference mode and Reference voltage */
	ADC_setVREF(myADC0_BASE, ADC_REFERENCE_INTERNAL, ADC_REFERENCE_3_3V);

	/* Set trim */
	AdcaRegs.ADCINLTRIM1 = ADC_TRIM_VALUE;

	/* Configures the analog-to-digital converter module prescaler. */
	AdcaRegs.ADCCTL2.bit.PRESCALE  = ADC_CLK_DIV_4_0;

	/* Sets the timing of the end-of-conversion pulse */
	AdcaRegs.ADCCTL1.bit.INTPULSEPOS = 1;

	/* Powers up the ADCA core. */
	AdcaRegs.ADCCTL1.bit.ADCPWDNZ = 1;

	/* Delay for 1ms to allow ADCA time to power up */
	DEVICE_DELAY_US(5000);

	/*
	 * @brief SOC Configuration:
	 * 		  Setup ADC EPWM channel and trigger settings.
	 */
	/* Disables SOC burst mode. */
	AdcaRegs.ADCBURSTCTL.bit.BURST_EN = 0;

	/* Sets the priority mode of the SOCs. */
	AdcaRegs.ADCSOCPRICTL.bit.SOCPRIORITY = 0;

	/*
	 * @brief Start of Conversion 0 Configuration.
	 *
	 * Configures a SOC in the ADC and its interrupt SOC trigger.
	 * 			SOC number		: 0
	 * 			Trigger			: ADC_TRIGGER_SW_ONLY
	 * 			Channel			: ADC_CH_ADCIN2
	 * 			Sample Window	: 50 SYSCLK cycles
	 * 			Interrupt Trigger: ADC_INT_SOC_TRIGGER_NONE
	 */
	AdcaRegs.ADCSOC0CTL.bit.TRIGSEL = 0;
	AdcaRegs.ADCSOC0CTL.bit.CHSEL = 2;
	AdcaRegs.ADCSOC0CTL.bit.ACQPS = 50U;
	AdcaRegs.ADCINTSOCSEL1.bit.SOC0 = 0;

	/*
	 * @brief Start of Conversion 1 Configuration.
	 *
	 * Configures a SOC in the ADC and its interrupt SOC trigger.
	 * 			SOC number		: 1
	 * 			Trigger			: ADC_TRIGGER_SW_ONLY
	 * 			Channel			: ADC_CH_ADCIN2
	 * 			Sample Window	: 50 SYSCLK cycles
	 * 			Interrupt Trigger: ADC_INT_SOC_TRIGGER_NONE
	 */
	AdcaRegs.ADCSOC1CTL.bit.TRIGSEL = 0;
	AdcaRegs.ADCSOC1CTL.bit.CHSEL = 2;
	AdcaRegs.ADCSOC1CTL.bit.ACQPS = 50U;
	AdcaRegs.ADCINTSOCSEL1.bit.SOC1 = 0;

	/*
	 * @brief ADC Interrupt Initialization:
	 *
	 * ADC Interrupt 1 Configuration
	 * 		Source	: ADC_INT_TRIGGER_EOC1
	 * 		Interrupt Source: enabled
	 * 		Continuous Mode	: disabled
	 */
	AdcaRegs.ADCINTSEL1N2.bit.INT1SEL = ADC_INT_TRIGGER_EOC1;
	AdcaRegs.ADCINTSEL1N2.bit.INT1E = 1;
	AdcaRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;
	AdcaRegs.ADCINTSEL1N2.bit.INT1CONT = 0;

	/*
	 * PPB Configuration: Configure high and low limits detection for ADCPPB
	 *
	 * Post Processing Block 1 Configuration
	 * 		Configures a post-processing block (PPB) in the ADC.
	 * 		PPB Number				: 1
	 * 		SOC/EOC number			: 1
	 * 		Calibration Offset		: 100
	 * 		Reference Offset		: 0
	 * 		Two's Complement		: Disabled
	 * 		Trip High Limit			: 0
	 * 		Trip Low Limit			: 0
	 * 		Clear PPB Event Flags	: Disabled
	 */
	AdcaRegs.ADCPPB1CONFIG.bit.CONFIG = 1;
	AdcaRegs.ADCEVTSEL.bit.PPB1TRIPHI = 0;
	AdcaRegs.ADCEVTSEL.bit.PPB1TRIPLO = 0;
	AdcaRegs.ADCEVTSEL.bit.PPB1ZERO = 0;
	AdcaRegs.ADCEVTINTSEL.bit.PPB1TRIPHI = 0;
	AdcaRegs.ADCEVTINTSEL.bit.PPB1TRIPLO = 0;
	AdcaRegs.ADCEVTINTSEL.bit.PPB1ZERO = 0;
	AdcaRegs.ADCPPB1OFFCAL.bit.OFFCAL = 100;
	AdcaRegs.ADCPPB1OFFREF = 0;
	AdcaRegs.ADCPPB1CONFIG.bit.TWOSCOMPEN = 0;
	AdcaRegs.ADCPPB1TRIPLO.bit.LIMITLO = 0;
	AdcaRegs.ADCPPB1TRIPHI = 0;
	AdcaRegs.ADCPPB1CONFIG.bit.CBCEN = 0;
}

void myADC1_init()
{
	/*
	 * @brief ADC Initialization:
	 * 		  Write ADC configurations and power up the ADC.
	 */
	/* Select ADC Reference mode and Reference voltage */
	ADC_setVREF(myADC1_BASE, ADC_REFERENCE_EXTERNAL, ADC_REFERENCE_3_3V);

	/* Set trim */
	AdccRegs.ADCINLTRIM1 = ADC_TRIM_VALUE;

	/* Configures the analog-to-digital converter module prescaler. */
	AdccRegs.ADCCTL2.bit.PRESCALE  = ADC_CLK_DIV_4_0;

	/* Sets the timing of the end-of-conversion pulse */
	AdccRegs.ADCCTL1.bit.INTPULSEPOS = 1;

	/* Powers up the ADCA core. */
	AdccRegs.ADCCTL1.bit.ADCPWDNZ = 1;

	/* Delay for 1ms to allow ADCA time to power up */
	DEVICE_DELAY_US(5000);

	/*
	 * @brief SOC Configuration:
	 * 		  Setup ADC EPWM channel and trigger settings.
	 */
	/* Disables SOC burst mode. */
	AdccRegs.ADCBURSTCTL.bit.BURST_EN = 0;

	/* Sets the priority mode of the SOCs. */
	AdccRegs.ADCSOCPRICTL.bit.SOCPRIORITY = 0;

	/*
	 * @brief Start of Conversion 0 Configuration.
	 *
	 * Configures a SOC in the ADC and its interrupt SOC trigger.
	 * 			SOC number		: 0
	 * 			Trigger			: ADC_TRIGGER_SW_ONLY
	 * 			Channel			: ADC_CH_ADCIN2
	 * 			Sample Window	: 50 SYSCLK cycles
	 * 			Interrupt Trigger: ADC_INT_SOC_TRIGGER_NONE
	 */
	AdccRegs.ADCSOC0CTL.bit.TRIGSEL = 0;
	AdccRegs.ADCSOC0CTL.bit.CHSEL = 2;
	AdccRegs.ADCSOC0CTL.bit.ACQPS = 50U;
	AdccRegs.ADCINTSOCSEL1.bit.SOC0 = 0;

	/*
	 * @brief Start of Conversion 1 Configuration.
	 *
	 * Configures a SOC in the ADC and its interrupt SOC trigger.
	 * 			SOC number		: 1
	 * 			Trigger			: ADC_TRIGGER_SW_ONLY
	 * 			Channel			: ADC_CH_ADCIN2
	 * 			Sample Window	: 50 SYSCLK cycles
	 * 			Interrupt Trigger: ADC_INT_SOC_TRIGGER_NONE
	 */
	AdccRegs.ADCSOC1CTL.bit.TRIGSEL = 0;
	AdccRegs.ADCSOC1CTL.bit.CHSEL = 2;
	AdccRegs.ADCSOC1CTL.bit.ACQPS = 50U;
	AdccRegs.ADCINTSOCSEL1.bit.SOC1 = 0;

	/*
	 * @brief ADC Interrupt Initialization:
	 *
	 * ADC Interrupt 1 Configuration
	 * 		Source	: ADC_INT_TRIGGER_EOC1
	 * 		Interrupt Source: enabled
	 * 		Continuous Mode	: disabled
	 */
	AdccRegs.ADCINTSEL1N2.bit.INT1SEL = ADC_INT_TRIGGER_EOC1;
	AdccRegs.ADCINTSEL1N2.bit.INT1E = 1;
	AdccRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;
	AdccRegs.ADCINTSEL1N2.bit.INT1CONT = 0;

	/*
	 * PPB Configuration: Configure high and low limits detection for ADCPPB
	 *
	 * Post Processing Block 1 Configuration
	 * 		Configures a post-processing block (PPB) in the ADC.
	 * 		PPB Number				: 1
	 * 		SOC/EOC number			: 1
	 * 		Calibration Offset		: -100
	 * 		Reference Offset		: 0
	 * 		Two's Complement		: Disabled
	 * 		Trip High Limit			: 0
	 * 		Trip Low Limit			: 0
	 * 		Clear PPB Event Flags	: Disabled
	 */
	AdccRegs.ADCPPB1CONFIG.bit.CONFIG = 1;
	AdccRegs.ADCEVTSEL.bit.PPB1TRIPHI = 0;
	AdccRegs.ADCEVTSEL.bit.PPB1TRIPLO = 0;
	AdccRegs.ADCEVTSEL.bit.PPB1ZERO = 0;
	AdccRegs.ADCEVTINTSEL.bit.PPB1TRIPHI = 0;
	AdccRegs.ADCEVTINTSEL.bit.PPB1TRIPLO = 0;
	AdccRegs.ADCEVTINTSEL.bit.PPB1ZERO = 0;
	AdccRegs.ADCPPB1OFFCAL.bit.OFFCAL = -100;
	AdccRegs.ADCPPB1OFFREF = 0;
	AdccRegs.ADCPPB1CONFIG.bit.TWOSCOMPEN = 0;
	AdccRegs.ADCPPB1TRIPLO.bit.LIMITLO = 0;
	AdccRegs.ADCPPB1TRIPHI = 0;
	AdccRegs.ADCPPB1CONFIG.bit.CBCEN = 0;
}

#ifdef __cplusplus

#endif
