/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/*
 * @File adc_ex08_ppb_limits.c
 *
 * @Title    ADC limits check using post-processing block
 *
 * This example sets up the ePWM to periodically trigger the ADC.
 * If the results are outside of the defined range,the post-processing block
 * will generate an interrupt.
 * The default limits are 1000LSBs and 3000LSBs.
 * With VREFHI set to 3.3V, the PPB will generate an interrupt
 * if the input voltage goes above about 2.4V or below about 0.8V.
 *
 * @External Connections:
 * - A0 should be connected to a signal to convert
 *
 * @Watch Variables
 * - myADC0Result0 - Digital representation of the voltage on pin A0.
 *
 */

/*
 * Include Files
 */
#include "adc_ex08_board.h"

/*
 * Global Variables
 */
uint32_t intStatus;
uint16_t myADC0Result0;

/*
 * Local Function Definitions
 */
void configureEPWM(uint32_t epwmBase);

/* Main */
void main(void)
{
	/* Initialize device clock and peripherals */
    Device_init();

    /*
     * Board Initialization
     * - Configure the ADC and power it up
     * - Setup the ADC for ePWM triggered conversions on channel 0
     * - Configure ADC post-processing limits
     *      SOC0 will generate an interrupt if conversion is above or below limits
     */
    Board_init();

    /* Configure the ePWM */
    configureEPWM(EPWM1_BASE);

    /* Enable global Interrupts */
    EINT;

    /* Start ePWM:Enable sync and clock to PWM */
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);

    /* Enable SOCA trigger */
    EPWM_enableADCTrigger(EPWM1_BASE, EPWM_SOC_A);

    /* Unfreeze epwm counter to count-up */
    EPWM_setTimeBaseCounterMode(EPWM1_BASE, EPWM_COUNTER_MODE_UP);

    /* Take conversions indefinitely in loop */
    do
    {
        /*
         * Wait while ePWM causes ADC conversions.
         * If the ADC conversions exceed the PPB limits, then an interrupt
         * will be generated. User can read ADC results either in idle loop
         * or ADCINT ISR as per application requirement.
         */
    }
    while(1);
}

/* configureEPWM - Setup SOC and compare values for EPWM */
void configureEPWM(uint32_t epwmBase)
{
	/* Disable SOCA trigger */
    EPWM_disableADCTrigger(epwmBase, EPWM_SOC_A);

    /* Trigger SOCA on CMPA up-count */
    EPWM_setADCTriggerSource(epwmBase, EPWM_SOC_A, EPWM_SOC_TBCTR_U_CMPA);

    /* Generate pulse on 1st event */
    EPWM_setADCTriggerEventPrescale(epwmBase, EPWM_SOC_A, 1U);

    /* Set compare A value to 2048 counts */
    EPWM_setCounterCompareValue(epwmBase, EPWM_COUNTER_COMPARE_A, 2048U);

    /* Set period to 4096 counts */
    EPWM_setTimeBasePeriod(epwmBase, 4096U);

    /* Freeze counter */
    EPWM_setTimeBaseCounterMode(epwmBase, EPWM_COUNTER_MODE_STOP_FREEZE);
}

/* adcAEvtISR - ISR for ADCA PPB */
void adcAEvtISR(void)
{
	/* Warning, you are outside of PPB limits */
    intStatus = AdcaRegs.ADCEVTSTAT.all;

    /* Read result */
    myADC0Result0 = AdcaResultRegs.ADCRESULT0;

    if((intStatus & ADC_EVT_TRIPHI) != 0U)
    {

    	/* Voltage exceeded high limit */
    	ESTOP0;

        /* Clear the trip flag and continue */
    	AdcaRegs.ADCEVTCLR.bit.PPB1TRIPHI = 1;
    }
    if((intStatus & ADC_EVT_TRIPLO) != 0U)
    {
        /* Voltage exceeded low limit */
    	ESTOP0;

        /* Clear the trip flag and continue */
    	AdcaRegs.ADCEVTCLR.bit.PPB1TRIPLO = 1;
    }
}

/* End of file */
