/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/*
 * @File adc_ex13_burst_mode_oversampling_bitfield.c
 *
 * @Title ADC burst mode oversampling
 *
 * This example is an ADC oversampling example implemented with software.
 * The ADC SOCs are configured in burst mode, triggered by the ePWM SOC
 * A event trigger.
 *
 * @External Connections:
 * - A2 should be connected to signals to convert.
 *
 * @Watch Variables
 * - lv_results - Array of digital values measured on pin A2
 *
 */

/*
 * Include Files
 */
#include "adc_ex12_board.h"

/*
 * Macros & Typedefs
 */
#define DO_TOGGLE 0
typedef enum
{
    Baseline = 0,           //!< No oversampling
    X2_Oversampling = 1,    //!< 2X oversampling
    X4_Oversampling = 2,    //!< 4X oversampling
    X8_Oversampling = 3,    //!< 8X oversampling
    X16_Oversampling = 4,   //!< 16X oversampling
} Oversampling_Amount;

/*
 * Global Variables
 */
uint16_t numBins = 8192;
volatile uint16_t nloops = 0;
volatile uint16_t lv_results[20]={[0 ... 19] = 0x5a};
Oversampling_Amount ovrsmplAmt = Baseline;

/*
 * ISR Function
 */
void INT_myADC0_1_ISR(void);

/* Main */
void main(void)
{
	/* Initialize device clock and peripherals */
    Device_init();

    /*
     * Set up ADCs:
     *
     * Burst mode configuration:
     * 1 burst of 16 conversions (SOC0 - SOC15) with equal priority
     *
     * Burst mode trigger is EPWM1 SOCA. Select the channels to be
     * converted and configure the S+H duration
     */
    Board_init();

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);

    /* Enable Global Interrupt (INTM) */
    EINT;

    while(1);
}

/* INT_myADC0_1_ISR - ADC A Interrupt Burst Mode ISR */
void INT_myADC0_1_ISR(void)
{
	/* Clear the interrupt flag */
	AdcaRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;

    /* Accumulate SOC results (store only 1 value per burst) */
    if(ovrsmplAmt == Baseline)
    {
    	/* 1X Oversampling */
        lv_results[nloops++] = AdcaResultRegs.ADCRESULT0;
    }
    else if(ovrsmplAmt == X2_Oversampling)
    {
    	/* 2X Oversampling */
    	lv_results[nloops++] =  AdcaResultRegs.ADCRESULT0 +  AdcaResultRegs.ADCRESULT1;
    }
    else if(ovrsmplAmt == X4_Oversampling)
    {
    	/* 4X Oversampling */
    	lv_results[nloops++] =  AdcaResultRegs.ADCRESULT0 +  AdcaResultRegs.ADCRESULT1 + AdcaResultRegs.ADCRESULT2 + AdcaResultRegs.ADCRESULT3;
    }
    else if(ovrsmplAmt == X8_Oversampling)
    {
    	/* 8X Oversampling */
    	lv_results[nloops++] =  AdcaResultRegs.ADCRESULT0 +  AdcaResultRegs.ADCRESULT1 + AdcaResultRegs.ADCRESULT2 + AdcaResultRegs.ADCRESULT3
    							+ AdcaResultRegs.ADCRESULT4 + AdcaResultRegs.ADCRESULT5 + AdcaResultRegs.ADCRESULT6 + AdcaResultRegs.ADCRESULT7;
    }
    else if(ovrsmplAmt == X16_Oversampling)
    {
    	/* 16X Oversampling */
    	lv_results[nloops++] =  AdcaResultRegs.ADCRESULT0 +  AdcaResultRegs.ADCRESULT1 + AdcaResultRegs.ADCRESULT2 + AdcaResultRegs.ADCRESULT3
    							+ AdcaResultRegs.ADCRESULT4 + AdcaResultRegs.ADCRESULT5 + AdcaResultRegs.ADCRESULT6 + AdcaResultRegs.ADCRESULT7
								+ AdcaResultRegs.ADCRESULT8 +  AdcaResultRegs.ADCRESULT9 + AdcaResultRegs.ADCRESULT10 + AdcaResultRegs.ADCRESULT11
    							+ AdcaResultRegs.ADCRESULT12 + AdcaResultRegs.ADCRESULT13 + AdcaResultRegs.ADCRESULT14 + AdcaResultRegs.ADCRESULT15;
    }

    /* Check if overflow has occurred */
    if(true == AdcaRegs.ADCINTOVF.bit.ADCINT1)
    {
    	AdcaRegs.ADCINTOVFCLR.bit.ADCINT1 = 1;
    	AdcaRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;
    }

    /* Check if all results are stored */
    if(nloops >= numBins)
    {
    	/* Disable ADC interrupt */
    	AdcaRegs.ADCINTSEL1N2.bit.INT1E = 0;
        ADC_disableInterrupt(myADC0_BASE, ADC_INT_NUMBER1);
    }

}

/* End of file */
