/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <string.h>

#include "device.h"
#include "can_bus_off.h"

CAN_RxMessage_t RxMsgBuf = {0};
volatile bool can_disable_flag = false;

void can_interrupt_irq_handler(void)
{
	uint32_t status;
	uint32_t err_status;
	uint32_t tx_err_count;
	uint32_t rx_err_count;
	uint8_t alc_value;

	status = CAN_getIntrStatus(CAN_BASE);

	if (status & CAN_RTIF_RIF_M) {
		CAN_receiveMsg(CAN_BASE, &RxMsgBuf);
	}

	if (status & CAN_ERRINT_BEIF_M) {
		err_status = CAN_getErrStatus(CAN_BASE);
		/* Get the number of error count. */
		CAN_getErrorCount(CAN_BASE, &rx_err_count, &tx_err_count);

		/* Bus off Interrupt. */
		if (err_status & CAN_STATUS_BUS_OFF) {
			/**
			 * User business logic code.
			 * Proactively clearing the bus off state is not advisable.
			 * The bus off state should be automatically
			 * detected and cleared by the CAN bus protocol.
			 */
#if (CAN_BUF_OFF_CLEAR)
			/* Clear bus off state. @note: Suggest using it only during debugging. */
			CAN_clearBusOff(CAN_BASE);
#else
			/* Reset CAN-Ctrl, Waiting for users to recover on their own. */
			CAN_setOpMode(CAN_BASE, CAN_OPERATION_MODE_SW_INIT);
			can_disable_flag = true;
#endif

		} else if (err_status & CAN_STATUS_EWARN){
			/**
			 * Bus error warning interrupt.
			 * User business logic code.
			 */
		} else {
			/**
			 * Bus error interrupt.
			 * User business logic code.
			 */
		}
		CAN_clearIntrStatus(CAN_BASE, CAN_ERRINT_BEIF_M);
	}

	if (status & CAN_ERRINT_ALIF_M) {
		/* Arbitration Lost Interrupt. */
		/* Obtain the bit position of arbitration failure. */
		CAN_getArbitrationLostCapture(CAN_BASE, &alc_value);

		CAN_clearIntrStatus(CAN_BASE, CAN_ERRINT_ALIF_M);
	}

	if (status & CAN_ERRINT_EPIF_M) {
		/* Error Passive Interrupt. */
		err_status = CAN_getErrStatus(CAN_BASE);

		/* check if the can bus entry error passive state. */
		if (err_status & CAN_STATUS_EPASS) {
			/* Get the number of error count. */
			CAN_getErrorCount(CAN_BASE, &rx_err_count, &tx_err_count);

			if (rx_err_count >= 128 && rx_err_count < 255) {
				/* Enter receive error passive. */
				/* User business logic code. */
			}

			if (tx_err_count >= 128 && tx_err_count < 255) {
				/* Enter transmit error passive. */
				/* User business logic code. */
			}
		}

		CAN_clearIntrStatus(CAN_BASE, CAN_ERRINT_EPIF_M);
	}

	/* Check if Rx buffer is empty.
	 * If the Rx buffer is not empty,
	 * do not clear the receive interrupt. */
	if (CAN_getRxBufState(CAN_BASE) == CAN_RXBUFFER_EMPTY)
		CAN_clearIntrStatus(CAN_BASE, CAN_RTIF_RIF_M);
}

void can_config(void)
{
	CAN_InitParams_t initParams = {0};
	CAN_FilterElement_t stdFiltelem = {0};

	initParams.fdMode = true;
	initParams.fdFrame = CAN_FD_ISO_11898_1;
	initParams.listenOnlyEnable = false;
	initParams.txbmode = CAN_TS_MODE_PRIORITY;
	initParams.timestamp = CAN_TIMESTAMP_EOF;

	/* accept all CAN ID frames. */
	stdFiltelem.acode = 0xc0;
	stdFiltelem.amsk = 0x1fffffff;
	stdFiltelem.filerFrameType = CAN_IDTYPE_BOTH_STD_EXT;

	/* Initial CAN module. */
	if (!CAN_initModule(CAN_BASE))
		while (1);

	/* Configuration CAN basic function. */
	if (!CAN_initConfig(CAN_BASE, &initParams))
		while (1);

	/* Set the arbitration segment bit rate based on sampling points. */
	if (!CAN_setBitRateSamplePoint(CAN_BASE, 40000000, 1U * 1000 * 1000, 0.80))
		while (1);

	/* Set the data segment bit rate based on sampling points */
	if (!CAN_setDataBitRateSamplePoint(CAN_BASE, 40000000, 4U * 1000 * 1000, 0.8))
		while (1);

	/* Set CAN ID acceptance filter 0. */
	if (!CAN_configMsgFilter(CAN_BASE, 0, &stdFiltelem))
		while (1);

	/* Enable CAN ID acceptance filter 0. */
	if (!CAN_configMsgFilterEnable(CAN_BASE, 0))
		while (1);

	/* Start running CAN module. */
	if (!CAN_startModule(CAN_BASE))
		while (1);

	/* Enable auto retransmission. */
	if (!CAN_setAutoRetransmission(CAN_BASE, true))
		while (1);

	/* Set bus error warning limit to 96. */
	if (!CAN_setErrWarnLevel(CAN_BASE, 0xb))
		while (1);
}

void can_ctrl_init(void)
{
	SysCtl_setCanClkSrcSel(CAN_CLK_TYPE_PLL);

	/* CAN clock source = 80MHz, PLL_CLK = 2* SYS_CLK */
	SysCtl_setCanClkDiv((DEVICE_SYSCLK_FREQ * 2 / 40000000));

	can_config();

	Interrupt_SetPriority(CAN_IRQ_NUM, 0, 0);
	Interrupt_register(CAN_IRQ_NUM, &can_interrupt_irq_handler);
	Interrupt_enable(CAN_IRQ_NUM);

	/* Enable Bus error, Arbitration Lost and Error Passive Interrupt. */
	CAN_enableInterrupt(CAN_BASE, CAN_RTIE_RIE_M | CAN_ERRINT_BEIE_M |
								CAN_ERRINT_ALIE_M | CAN_ERRINT_EPIE_M);
}
