/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
//#############################################################################
//
// FILE:   pto_qepdiv_Source.c
//
// Description:Example project for using PM pto_qepdiv Library.
//              Includes pto_qepdiv library and corresponding include
//              files. Initializes the encoders and performs delay compensation.
//              Runs pto_qepdiv command set.
//              Continuously Read position value in an infinite loop
//
//#############################################################################


//
// Included Files
//
#include "driverlib.h"
#include "device.h"
#include "stdint.h"
#include "pto_qepdiv.h"

#include "clb_config.h"
#include "clb.h"


//*****************************************************************************
//
//! Setup CLB tiles
//!
//! \param None
//!
//! Setup the CLB and other interconnect XBARs used in this application.
//! This is called during the CLB initialization. This function needs to be
//! called after every system reset.
//!
//! \return None
//
//*****************************************************************************
void
pto_qepdiv_setupPeriph(void)
{

    initTILE1(CLB1_BASE);    
    initTILE2(CLB2_BASE);

//    //
//    // Set the various counter loads and match values.
//    //
//    CLB_writeInterface(CLB1_BASE, CLB_ADDR_COUNTER_0_LOAD,
//                       TILE1_COUNTER_0_LOAD_VAL);
//    CLB_writeInterface(CLB1_BASE, CLB_ADDR_COUNTER_1_LOAD,
//                       TILE1_COUNTER_1_LOAD_VAL);
//    CLB_writeInterface(CLB1_BASE, CLB_ADDR_COUNTER_2_LOAD,
//                       TILE1_COUNTER_2_LOAD_VAL);
//    CLB_writeInterface(CLB1_BASE, CLB_ADDR_COUNTER_0_MATCH1,
//                       TILE1_COUNTER_0_MATCH1_VAL);
//    CLB_writeInterface(CLB1_BASE, CLB_ADDR_COUNTER_0_MATCH2,
//                       TILE1_COUNTER_0_MATCH2_VAL);
//    CLB_writeInterface(CLB1_BASE, CLB_ADDR_COUNTER_1_MATCH1,
//                       TILE1_COUNTER_1_MATCH1_VAL);
//    CLB_writeInterface(CLB1_BASE, CLB_ADDR_COUNTER_1_MATCH2,
//                       TILE1_COUNTER_1_MATCH2_VAL);
//    CLB_writeInterface(CLB1_BASE, CLB_ADDR_COUNTER_2_MATCH1,
//                       TILE1_COUNTER_2_MATCH1_VAL);
//    CLB_writeInterface(CLB1_BASE, CLB_ADDR_COUNTER_2_MATCH2,
//                       TILE1_COUNTER_2_MATCH2_VAL);
//
//    CLB_writeInterface(CLB2_BASE, CLB_ADDR_COUNTER_0_LOAD,
//                       TILE2_COUNTER_0_LOAD_VAL);
//    CLB_writeInterface(CLB2_BASE, CLB_ADDR_COUNTER_1_LOAD,
//                       TILE2_COUNTER_1_LOAD_VAL);
//    CLB_writeInterface(CLB2_BASE, CLB_ADDR_COUNTER_2_LOAD,
//                       TILE2_COUNTER_2_LOAD_VAL);
//    CLB_writeInterface(CLB2_BASE, CLB_ADDR_COUNTER_0_MATCH1,
//                       TILE2_COUNTER_0_MATCH1_VAL);
//    CLB_writeInterface(CLB2_BASE, CLB_ADDR_COUNTER_0_MATCH2,
//                       TILE2_COUNTER_0_MATCH2_VAL);
//    CLB_writeInterface(CLB2_BASE, CLB_ADDR_COUNTER_1_MATCH1,
//                       TILE2_COUNTER_1_MATCH1_VAL);
//    CLB_writeInterface(CLB2_BASE, CLB_ADDR_COUNTER_1_MATCH2,
//                       TILE2_COUNTER_1_MATCH2_VAL);
//    CLB_writeInterface(CLB2_BASE, CLB_ADDR_COUNTER_2_MATCH1,
//                       TILE2_COUNTER_2_MATCH1_VAL);
//    CLB_writeInterface(CLB2_BASE, CLB_ADDR_COUNTER_2_MATCH2,
//                       TILE2_COUNTER_2_MATCH2_VAL);

//    pto_qepdiv_initCLBXBAR();   // Initialize the CLB XBAR
}

//*****************************************************************************
//
//! Configure the divider
//!
//! \param divider is value of the divider
//! \param indexWidth is the number of cycles for which the index pulse output
//!        is kept on
//!
//! This function configures the divider. The divider value cannot be changed
//! dynamically. Users must reset the module using pto_qepdiv_reset before
//! reconfiguring the functionality.
//!
//! \return divider ptoFullPeriod is the return value, if the function is
//! executed successfully
//
//*****************************************************************************

uint16_t
pto_qepdiv_config(uint16_t divider, uint16_t indexWidth)
{
    CLB_writeInterface(CLB2_BASE, CLB_ADDR_COUNTER_0_MATCH2, divider * 4);
    CLB_writeInterface(CLB2_BASE, CLB_ADDR_COUNTER_0_MATCH1, divider * 2);
    CLB_writeInterface(CLB1_BASE, CLB_ADDR_COUNTER_2_MATCH1, indexWidth - 1);
    return(divider);
}

//*****************************************************************************
//
//! This function initiates the pulse generation.
//!
//! \param run parameters to be passed to start(1) or stop(0) the function
//!
//! This function initiates the pulse generation. This function must only be called after
//! pto_qepdiv_setupPeriph. Hence, the pto_qepdiv_startOperation function kick starts the pulse generation
//! that was set up earlier.
//!
//! \return None.
//
//*****************************************************************************
void
pto_qepdiv_startOperation(uint16_t run)
{
    EALLOW;
    CLB_enableCLB(CLB1_BASE);
    CLB_enableCLB(CLB2_BASE);


    CLB_setGPREG(CLB2_BASE, run);
    CLB_setGPREG(CLB1_BASE, run);
}

//*****************************************************************************
//
//! Reset the qepdiv parameters
//!
//! \param None
//!
//! This function resets the qepdiv parameters set by earlier configurations
//! and to begin a fresh setup. This function must be called in case the pulse
//! generation must be reset and started again at a later stage.
//!
//! \return None
//
//*****************************************************************************
void
pto_qepdiv_reset(void)
{
    pto_qepdiv_resetCLB();
    CLB_setGPREG(CLB1_BASE, 0);
    CLB_setGPREG(CLB2_BASE, 0);
}



//*****************************************************************************
//
//! Initialize the CLB XBAR-connect
//!
//! \param None
//!
//! This function configures Output-XBars to route CLB outputs to GPIO pins
//!
//! \return None
//
//*****************************************************************************
void
pto_qepdiv_initCLBXBAR(void) // see configuration for controlCARD
{
    // Setup of INPUTXBAR to AUXSIG has been moved to the
    // System demonstration project qepdiv_<device>.syscfg file.

    //
    // Configure CLB1 o/p 5 to OUTPUT XBAR to OUTPUT3
    //
#if defined(F2837x_F28004x)
    XBAR_setOutputMuxConfig(XBAR_OUTPUT3, XBAR_OUT_MUX03_CLB1_OUT5);
    XBAR_enableOutputMux(XBAR_OUTPUT3, XBAR_MUX03);
#elif defined(F28002x_F28003x)
	XBAR_setOutputMuxConfig(OUTPUTXBAR_BASE, XBAR_OUTPUT3, XBAR_OUT_MUX03_CLB1_OUT5);
	XBAR_enableOutputMux(OUTPUTXBAR_BASE, XBAR_OUTPUT3, XBAR_MUX03);
//#else
//    XBAR_setOutputMuxConfig(OUTPUTXBAR_BASE, XBAR_OUTPUT3, XBAR_OUT_MUX03_CLB1_OUT5);
//    XBAR_enableOutputMux(OUTPUTXBAR_BASE, XBAR_OUTPUT3, XBAR_MUX03);
#endif
    //
    // for testing
    //
    //XBAR_setOutputMuxConfig(XBAR_OUTPUT3, XBAR_OUT_MUX05_CLB2_OUT4);
    //XBAR_enableOutputMux(XBAR_OUTPUT3, XBAR_MUX05);
}

//*****************************************************************************
//
// Close the Doxygen group.
//! @}
//
//*****************************************************************************

//
// End of File
//
