/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    dma_dsp_ex01_ecap_pwm.c
*   @brief   
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "board_cfg.h"
#include "dma_dsp_ex01_ecap_pwm_board.h"
#include "log.h"
#include "load_img.h"
/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#define PWM_FREQUENCY       10000U // 10 Khz
#define PWM_DUTY            0.3f   // 30% duty
#define PWM_PRD_VAL         20000000U / PWM_FREQUENCY
#define PWM_CMP_VAL         (uint32_t)(PWM_DUTY * PWM_PRD_VAL)
#define ECAP_COUNT 16

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

vuint16_t cap3Count;
vuint16_t cap4Count;

#if (DMAC_VERSION == 0x30)
__attribute__((aligned(DCACHE_LINE_SIZE))) uint32_t ecapData[ECAP_COUNT*2] = {0};
#else
uint32_t ecapData[ECAP_COUNT*2] = {0};
#endif
/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/**
 * \brief   "// initEPWM - Configure ePWM" 
 *
 * \param   void   no parameters
 *
 * \retval  None
 */
void initEPWM(uint32_t base)
{
	SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);

    EPWM_setEmulationMode(base, EPWM_EMULATION_STOP_AFTER_FULL_CYCLE);

    //
    // Set-up TBCLK
    //
    EPWM_setTimeBasePeriod(base, PWM_FREQUENCY);
    EPWM_setPhaseShift(base, 0U);
    EPWM_setTimeBaseCounter(base, 0U);

    //
    // Set up counter mode
    //
    EPWM_setTimeBaseCounterMode(base, EPWM_COUNTER_MODE_UP);
    EPWM_disablePhaseShiftLoad(base);

    EPWM_setClockPrescaler(base,
                           EPWM_CLOCK_DIVIDER_1,
                           EPWM_HSCLOCK_DIVIDER_1);
    EPWM_setTimeBasePeriod(base, PWM_PRD_VAL);
    EPWM_setTimeBaseCounter(base,0U);
    EPWM_setTimeBaseCounterMode(base, EPWM_COUNTER_MODE_UP);
    EPWM_disablePhaseShiftLoad(base);
    EPWM_setPhaseShift(base, 0U);
    EPWM_setCounterCompareValue(base,EPWM_COUNTER_COMPARE_A,PWM_CMP_VAL);
    EPWM_setCounterCompareShadowLoadMode(base,EPWM_COUNTER_COMPARE_A,EPWM_COMP_LOAD_ON_CNTR_ZERO);
    EPWM_enableSyncOutPulseSource(base,EPWM_SYNC_OUT_PULSE_ON_CNTR_ZERO);
    EPWM_setActionQualifierAction(base,
                                  EPWM_AQ_OUTPUT_A,
                                  EPWM_AQ_OUTPUT_LOW,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);
    EPWM_setActionQualifierAction(base,
                                  EPWM_AQ_OUTPUT_A,
                                  EPWM_AQ_OUTPUT_HIGH,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);

    // Enable sync and clock to PWM
    //
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);
}

/**
 * \brief   "// initDMA - Configure DMA" 
 *
 * \param   void   no parameters
 *
 * \retval  None
 */
void initDMA()
{
    DMA_ConfigParams dmaCfg = {0};
        
    DMA_initController(DMA_BASE);

    /* Stop DMA channel first - why the DMA channel is still pending after reset? */
    DMA_stopChannel(DMA_CH0_BASE);
    dmaCfg.enableInterrupt = 1;
    dmaCfg.dmaSrcReqId = DMAMUX_ReqId_Trig_ecap1;
    dmaCfg.srcAddr = (myECAP_BASE + ECAP_O_CAP3); // The ecap 3-register address
    dmaCfg.destAddr = (uint32_t)ecapData;
    dmaCfg.blockTS = 2;  // Only 2 data were transmitted at a time, for each of ecap 3 and ecap 4
    dmaCfg.ttfc    = DMA_TT_FC_2_P2M_DMAC;
    dmaCfg.srcBtl  = DMA_BTL_8;
    dmaCfg.destBtl = DMA_BTL_8;
    dmaCfg.srcAddrDirect = DMA_ADDR_INCRE; //Need to transfer data from ecap 3 and ecap 4, so set to gro
    dmaCfg.destAddrDirect = DMA_ADDR_INCRE;
    dmaCfg.srcTrWidthBytes = DMA_TR_WIDTH_BYTE_4;
    dmaCfg.destTrWidthBytes= DMA_TR_WIDTH_BYTE_4;
    dmaCfg.reloadSrc = 1; //Set the source address auto reload to produce block interruption without tfr interrupt
    
    DMA_configChannel(DMA_CH0_BASE, &dmaCfg);

    /* Clear INT flag set in previous operation */
    /* Enable the IRQ so that we can poll the Status_Tfr flag */
    DMA_clearInterrupt(DMA_CH0_BASE, DMA_INT_TFR | DMA_INT_BLOCK | DMA_INT_ERR);
    DMA_unMaskInterrupt(DMA_CH0_BASE, DMA_INT_TFR | DMA_INT_BLOCK | DMA_INT_ERR);

    /* DMA channel enable */
    DMA_startChannel(DMA_CH0_BASE);
}

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

void main(void){

    uint8_t status = 0;
    uint8_t reload_cnt = 0;
    uint32_t tmp32;
#if (DMAC_VERSION == 0x30)
    unsigned long line_cnt = (unsigned long )ECAP_COUNT * 2U * (1U << DMA_TR_WIDTH_BYTE_4)/DCACHE_LINE_SIZE;
#endif 

    __disable_irq();
    
#if ((GS32F3xx & 0xFF00) || (GS32F00xx & 0xFF00))
    Device_init();
#endif

#if ((GS32F3xx & 0xFF) || (GS32F00xx & 0xFF))
    //
    // config arm clock different from RISCV
    //
    if (HWREG(CRG_CFG_BASE + 0x008) == 0) {
        HWREG(CRG_CFG_BASE + 0x008) = 1;        //Slow 2 normal
        HWREG(CRG_CFG_BASE + 0x010) = 0x0;      //arm clk div = 0
        HWREG(CRG_CFG_BASE + 0x01C) = 1;        //update
    }
#endif

    UartPrint_init(LOG_SCI_BASE, 115200);

    log_set_level(LOG_DEBUG);

    log_info("Hello DSP!\r\n");
    log_info("Core running @ %d MHz.\r\n", DEVICE_SYSCLK_FREQ / 1000000);
    
#if defined(LOAD_ARM)
    bringup_arm();
#endif
    __enable_irq();

    printf("hello: %s \r\n", __FILE__);

    // Board initialization
    // Configure GPIO 16 as eCAP input
    // Enable interrupts required for this example
    //
    Board_init();

    // Initialize counters
    cap3Count = 0U;
    cap4Count = 0U;

#if ((GS32F3xx & 0xFF) || (GS32F00xx & 0xFF))
    //
    // config for pwm output
    //
    HWREG(0x4003F000 + 0x120) = 0x10000;
    HWREG(0x40008000 + 0x31C) = 0x3;
#endif

    // Configure ePWM
    initEPWM(myEPWM_BASE);

    // Configure DMA
    initDMA();

    while((status & DMA_INT_TFR) == 0)
    {
        status = DMA_getInterruptStatus(DMA_CH0_BASE);
        if (status & DMA_INT_BLOCK)
        {
            reload_cnt++;
            if (reload_cnt == ECAP_COUNT - 1)
                DMA_disableReload(DMA_CH0_BASE);//Before transmitting the last data, close the reload, 
            DMA_clearInterrupt(DMA_CH0_BASE, DMA_INT_BLOCK); 
        }
    }
#if (DMAC_VERSION == 0x30)
    //
    // Very Important No.2, after DMA TFR DONE, need to flush Cache if use RAM when use cache feature.
    //
    if (!DMA_checkRamInAutoCache((uint32_t)ecapData, ECAP_COUNT * 2U * (1U << DMA_TR_WIDTH_BYTE_4)))
    {
        //
        //  Other RAM need Invalid the Cache by manual.
        //    
        MInvalDCacheLines((uint32_t)ecapData, line_cnt);
    }
#endif 

    // Loop forever. Suspend or place breakpoints to observe the buffers.
    while(1)
    {
        cap3Count = ECAP_getEventTimeStamp(myECAP_BASE, ECAP_EVENT_3);
        cap4Count = ECAP_getEventTimeStamp(myECAP_BASE, ECAP_EVENT_4);
    }
}

/**
 * \brief   "add brief description of this function" 
 *
 * \param   parameter1   description of parameter1
 * \param   parameter2   description of parameter2
 *
 * \retval  None
 */

#ifdef __cplusplus
}
#endif
