/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */



/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "epwm_ex09_board.h"

//
// Defines
//
#define BURST       2              // 2 words per transfer
#define TRANSFER    4              // 4 transfers (different configs)

//
// Globals
//


__attribute__((aligned(8))) uint16_t compareAConfigs[TRANSFER*BURST] =
{
//  			CMPAHR  ,   CMPA   ,
				1 << 8  ,  1001U   ,
				2 << 8  ,  2001U   ,
				3 << 8  ,  3001U   ,
				4 << 8  ,  4001U   ,
};

__attribute__((aligned(8))) uint16_t compareBConfigs[TRANSFER*BURST] =
{
//    		    CMPBHR  ,   CMPB ,
			    5 << 8  ,   1000U,
			    6 << 8  ,   2000U,
			    7 << 8  ,   3000U,
			    8 << 8  ,   4000U,
};




//// Place buffers in GSRAM
#pragma DATA_SECTION(compareAConfigs,     "ramgs0");
#pragma DATA_SECTION(compareBConfigs,     "ramgs0");
#pragma DATA_SECTION(phaseConfigs, 		  "ramgs0");
#pragma DATA_SECTION(PeriodConfigs, 	  "ramgs0");
//
// Function Prototypes
//

__interrupt void dmaCh5ISR(void);
__interrupt void dmaCh6ISR(void);

//*****************************************************************************
//
// Board Configurations
// Initializes the rest of the modules.
// Call this function in your application if you wish to do all module
// initialization.
// If you wish to not use some of the initializations, instead of the
// Board_init use the individual Module_inits
//
//*****************************************************************************
void Board_init()
{

    EALLOW;

    PinMux_init();
    EPWM_init();
    DMA_init();
    INTERRUPT_init();

    EDIS;


}

//*****************************************************************************
//
// PINMUX Configurations
//
//*****************************************************************************
void PinMux_init()
{


    //
    // EPWM1 -> myEPWM1 Pinmux
    //
    GPIO_setPinConfig(myEPWM1_EPWMA_PIN_CONFIG);
    GPIO_setPadConfig(myEPWM1_EPWMA_GPIO, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(myEPWM1_EPWMA_GPIO, GPIO_QUAL_SYNC);

    GPIO_setPinConfig(myEPWM1_EPWMB_PIN_CONFIG);
    GPIO_setPadConfig(myEPWM1_EPWMB_GPIO, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(myEPWM1_EPWMB_GPIO, GPIO_QUAL_SYNC);


}

//*****************************************************************************
//
// EPWM Configurations
//
//*****************************************************************************
void EPWM_init()
{
    //	ePWM1 Configurations
    //TB
    //	Set 64(64*1) frequency division
    EPWM_setClockPrescaler(myEPWM1_BASE, EPWM_CLOCK_DIVIDER_64, EPWM_HSCLOCK_DIVIDER_1);
    //	Set TBPRD = 2000
    EPWM_setTimeBasePeriod(myEPWM1_BASE, 20000);
    //	Set TBCTR = 0
    EPWM_setTimeBaseCounter(myEPWM1_BASE, 0);
    //	Counting mode: UP
    EPWM_setTimeBaseCounterMode(myEPWM1_BASE, EPWM_COUNTER_MODE_UP);
    //	Disable phase shift load
    EPWM_disablePhaseShiftLoad(myEPWM1_BASE);
    //	Set TBPHS = 0
    EPWM_setPhaseShift(myEPWM1_BASE, 0);


    //CC
    //	Set CMPA = 10000
    EPWM_setCounterCompareValue(myEPWM1_BASE, EPWM_COUNTER_COMPARE_A, 10000);
    //	Load the new CMPA when TBCTR = 0
    EPWM_setCounterCompareShadowLoadMode(myEPWM1_BASE, EPWM_COUNTER_COMPARE_A, EPWM_COMP_LOAD_ON_CNTR_ZERO);
    //	Set CMPB = 10000
    EPWM_setCounterCompareValue(myEPWM1_BASE, EPWM_COUNTER_COMPARE_B, 10000);
    //	Load the new CMPB when TBCTR = 0
    EPWM_setCounterCompareShadowLoadMode(myEPWM1_BASE, EPWM_COUNTER_COMPARE_B, EPWM_COMP_LOAD_ON_CNTR_ZERO);


    //AQ
    //	TBCTR = 0 -> ePWM1A:HIGH
    EPWM_setActionQualifierAction(myEPWM1_BASE,EPWM_AQ_OUTPUT_A,EPWM_AQ_OUTPUT_HIGH,EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);
    //	TBCTR = UP_CMPA -> ePWM1A:LOW
    EPWM_setActionQualifierAction(myEPWM1_BASE,EPWM_AQ_OUTPUT_A,EPWM_AQ_OUTPUT_LOW,EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
    //	TBCTR = 0 -> ePWM1B:HIGH
    EPWM_setActionQualifierAction(myEPWM1_BASE,EPWM_AQ_OUTPUT_B,EPWM_AQ_OUTPUT_HIGH,EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);
    //	TBCTR = UP_CMPB -> ePWM1B:LOW
    EPWM_setActionQualifierAction(myEPWM1_BASE,EPWM_AQ_OUTPUT_B,EPWM_AQ_OUTPUT_LOW,EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB);


    //Enable ADC SOC_A event
    EPWM_enableADCTrigger(myEPWM1_BASE, EPWM_SOC_A);
    //When TBCTR = 0, the ADC SOC_A signal is emitted
    EPWM_setADCTriggerSource(myEPWM1_BASE,EPWM_SOC_A,EPWM_SOC_TBCTR_ZERO);
    //Set the ePWM SOC event count to 1
    EPWM_setADCTriggerEventPrescale(myEPWM1_BASE,EPWM_SOC_A,1);
    //Clear SOC_A flag
    EPWM_clearADCTriggerFlag(myEPWM1_BASE,EPWM_SOC_A);

}



//*****************************************************************************
//
// DMA Configurations
//
//*****************************************************************************

#if ( IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30) )
void DMA_init()
{
    DMA_ConfigParams dmaCfg = {0};

    //
    // Initialize DMA
    //
    DMA_initController(myDMA_BASE);

    DMA_stopChannel(myDMA_CH5_BASE);

    DMA_stopChannel(myDMA_CH6_BASE);

    //
    // DMA CH5 Config
    //
    dmaCfg.enableInterrupt = 1;
    dmaCfg.dmaDstReqId = DMAMUX_ReqId_Trig_EPWM1xSOCA;
    dmaCfg.destAddr = (uint32_t)(myEPWM1_BASE + EPWM_O_CMPA);
    dmaCfg.srcAddr  = (uint32_t)(&compareAConfigs[0]);
    dmaCfg.blockTS  = TRANSFER;

    dmaCfg.ttfc     = DMA_TT_FC_1_M2P_DMAC;
    dmaCfg.srcBtl   = DMA_BTL_1;
    dmaCfg.destBtl  = DMA_BTL_1;

    dmaCfg.srcAddrDirect  = DMA_ADDR_INCRE;
    dmaCfg.destAddrDirect = DMA_ADDR_NO_CHANGE;

    dmaCfg.srcTrWidthBytes = DMA_TR_WIDTH_BYTE_4;
    dmaCfg.destTrWidthBytes= DMA_TR_WIDTH_BYTE_4;
    dmaCfg.reloadSrc = 1;
    dmaCfg.reloadDst = 0;

    DMA_configChannel(myDMA_CH5_BASE, &dmaCfg);
    DMA_clearInterrupt(myDMA_CH5_BASE, DMA_INT_BLOCK);
    DMA_unMaskInterrupt(myDMA_CH5_BASE, DMA_INT_BLOCK);


    //
    // DMA CH6 Config
    //
    dmaCfg.destAddr = (uint32_t)(myEPWM1_BASE + EPWM_O_CMPB);
    dmaCfg.srcAddr  = (uint32_t)(&compareBConfigs[0]);
    DMA_configChannel(myDMA_CH6_BASE, &dmaCfg);
    DMA_clearInterrupt(myDMA_CH6_BASE, DMA_INT_BLOCK);
    DMA_unMaskInterrupt(myDMA_CH6_BASE, DMA_INT_BLOCK);

}

#elif ( IS_GS32F3xx(0x22) )


void DMA_init()
{

    XDMA_ConfigParams XdmaCfg = {0};

    //
    // Initialize DMA
    //
    XDMA_initController(myDMA_BASE);

    XDMA_stopChannel(myDMA_CH5_BASE);

    XDMA_stopChannel(myDMA_CH6_BASE);

    //
    // DMA CH5 Config
    //
    XdmaCfg.enableInterrupt = 1;
    XdmaCfg.dmaDstReqId = DMAMUX_ReqId_Trig_EPWM1xSOCA;
    XdmaCfg.destAddr = (uint32_t)(myEPWM1_BASE + EPWM_O_CMPA);
    XdmaCfg.srcAddr = (uint32_t)(&compareAConfigs[0]);
    XdmaCfg.blockTS = TRANSFER;

    XdmaCfg.ttfc    = XDMA_TT_FC_1_M2P_DMAC;//DMA_TT_FC_1_M2P_DMAC;
    XdmaCfg.srcBtl  = XDMA_BTL_1;
    XdmaCfg.destBtl = XDMA_BTL_1;

    XdmaCfg.srcAddrDirect = XDMA_ADDR_INCRE;
    XdmaCfg.destAddrDirect = XDMA_ADDR_NO_CHANGE;

    XdmaCfg.srcTrWidthBytes = XDMA_TR_WIDTH_BYTE_4;
    XdmaCfg.destTrWidthBytes = XDMA_TR_WIDTH_BYTE_4;
    XdmaCfg.reloadSrc=1;
    XdmaCfg.reloadDst=0;

    XDMA_configChannel(myDMA_CH5_BASE, &XdmaCfg);
    XDMA_clearInterrupt(myDMA_CH5_BASE, XDMA_INT_BLOCK);
    XDMA_unMaskInterrupt(myDMA_CH5_BASE, XDMA_INT_BLOCK);


    //
    // DMA CH6 Config
    //
    XdmaCfg.destAddr = (uint32_t)(myEPWM1_BASE + EPWM_O_CMPB);
    XdmaCfg.srcAddr  = (uint32_t)(&compareBConfigs[0]);
    XDMA_configChannel(myDMA_CH6_BASE, &XdmaCfg);
    XDMA_clearInterrupt(myDMA_CH6_BASE, XDMA_INT_BLOCK);
    XDMA_unMaskInterrupt(myDMA_CH6_BASE, XDMA_INT_BLOCK);

}


#endif


void INTERRUPT_init()
{

    // Interrupt Settings for INT_myDMACH5
    // ISR need to be defined for the registered interrupts
    Interrupt_register(INT_myDMACH5, &dmaCh5ISR);
    Interrupt_enable(INT_myDMACH5);

    // Interrupt Settings for INT_myDMACH6
    // ISR need to be defined for the registered interrupts
    Interrupt_register(INT_myDMACH6, &dmaCh6ISR);
    Interrupt_enable(INT_myDMACH6);
}
