/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    Test_CpuTimer.c
*   @brief
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include <i2c_test06_dma_out.h>
#include "device.h"
#include "eeprom.h"



#define EEPROM_SLAVE_ADDRESS   0x50

#define EEPROM_PAGE_LEN   8
#define W_DATA_LEN   8 //all data length

#define W_ADDR       0x00 //Starting address for writing to EEPROM
#define EEPROM_DATA_ADDRESS_LEN   1

uint16_t wdata[W_DATA_LEN] = {0};
uint8_t rdata[W_DATA_LEN] = {0};
uint32_t g_i2c_base = I2CB_BASE;
uint32_t i2c_port_num = 0;


void i2c_err_irq_check(uint32_t irq_status)
{
    if (irq_status & I2C_INT_TX_ABRT)
    {

        I2C_clearInterruptTxAbrt(g_i2c_base);
    }

    if (irq_status & I2C_INT_TX_OVER)
    {
        I2C_clearInterruptTxOver(g_i2c_base);
    }

    if (irq_status & I2C_INT_RX_OVER)
    {
        I2C_clearInterruptRxOver(g_i2c_base);
    }

    if (irq_status & I2C_INT_RX_UNDER)
    {
        I2C_clearInterruptRxUnder(g_i2c_base);
    }
}

void I2CX_INTR_IRQHandler(void)
{
    uint32_t irq_status = I2C_getInterruptStatus(g_i2c_base);

    i2c_err_irq_check(irq_status);

    I2C_clearAllInterruptStatus(g_i2c_base);
}

void i2c_dma_tx(uint16_t *tx_buf, uint32_t tx_len)
{
	uint32_t i;
	uint32_t j;

#if(IS_GS32F3xx(0x22))

	XDMA_ConfigParams dmaCfg = {0};

	XDMA_initController(DMA1_BASE);

    /* Stop DMA channel first - why the DMA channel is still pending after reset? */
    XDMA_stopChannel(DMA1_CH1_BASE);

    dmaCfg.enableInterrupt = 1;
    dmaCfg.srcAddr = (uint32_t)(tx_buf);
    dmaCfg.destAddr = (I2CB_BASE + I2C_O_IC_DATA_CMD);
    dmaCfg.blockTS = tx_len;
    dmaCfg.ttfc    = XDMA_TT_FC_1_M2P_DMAC;
    dmaCfg.srcBtl  = XDMA_BTL_4;
    dmaCfg.destBtl = XDMA_BTL_4;
    dmaCfg.srcAddrDirect = XDMA_ADDR_INCRE;
    dmaCfg.destAddrDirect = XDMA_ADDR_NO_CHANGE;
    dmaCfg.srcTrWidthBytes = XDMA_TR_WIDTH_BYTE_2;
    dmaCfg.destTrWidthBytes= XDMA_TR_WIDTH_BYTE_2;
//    dmaCfg.reloadSrc = 1;
//    dmaCfg.reloadDst = 1;
    dmaCfg.dmaDstReqId = DMAMUX_ReqId_i2cb_tx;
    /* Clear INT flag set in previous operation */
    /* Enable the IRQ so that we can poll the Status_Tfr flag */
    XDMA_clearInterrupt(DMA1_CH1_BASE, XDMA_INT_TFR);
    XDMA_unMaskInterrupt(DMA1_CH1_BASE,XDMA_INT_TFR);
    XDMA_configChannel(DMA1_CH1_BASE, &dmaCfg);
    I2C_SendStatusCheck(I2CB_BASE);
    eeprom_DataAddrWrite(I2CB_BASE, W_ADDR);
    XDMA_startChannel(DMA1_CH1_BASE);

#else

    DMA_ConfigParams dmaCfg = {0};

    DMA_initController(DMA_BASE);

    /* Stop DMA channel first - why the DMA channel is still pending after reset? */
    DMA_stopChannel(DMA_CH1_BASE);

    dmaCfg.enableInterrupt = 1;
    dmaCfg.srcAddr = (uint32_t)(tx_buf);
    dmaCfg.destAddr = (I2CB_BASE + I2C_O_IC_DATA_CMD);
    dmaCfg.blockTS = tx_len;
    dmaCfg.ttfc    = DMA_TT_FC_1_M2P_DMAC;
    dmaCfg.srcBtl  = DMA_BTL_4;
    dmaCfg.destBtl = DMA_BTL_4;
    dmaCfg.srcAddrDirect = DMA_ADDR_INCRE;
    dmaCfg.destAddrDirect = DMA_ADDR_NO_CHANGE;
    dmaCfg.srcTrWidthBytes = DMA_TR_WIDTH_BYTE_2;
    dmaCfg.destTrWidthBytes= DMA_TR_WIDTH_BYTE_2;
//    dmaCfg.reloadSrc = 1;
//    dmaCfg.reloadDst = 1;
    dmaCfg.dmaDstReqId = DMAMUX_ReqId_i2cb_tx;
    /* Clear INT flag set in previous operation */
    /* Enable the IRQ so that we can poll the Status_Tfr flag */
    DMA_clearInterrupt(DMA_CH1_BASE, DMA_INT_TFR);
    DMA_unMaskInterrupt(DMA_CH1_BASE, DMA_INT_TFR);
    DMA_configChannel(DMA_CH1_BASE, &dmaCfg);
    I2C_SendStatusCheck(I2CB_BASE);
    eeprom_DataAddrWrite(I2CB_BASE, W_ADDR);
    DMA_startChannel(DMA_CH1_BASE);
    /* poll DMA Ch1 INT flag until Tx completed*/
    while((DMA_getInterruptStatus(DMA_CH1_BASE) & DMA_INT_TFR) == 0);
    DEVICE_DELAY_US(6000);

#endif



    eeprom_DataAddrWrite(I2CB_BASE, W_ADDR);
    eeprom_DataCmdRead(I2CB_BASE, rdata, W_DATA_LEN);

    for (j = 0; j < W_DATA_LEN; j++)
        {
            wdata[j] &= 0x00FFU;
        }

    //check data
    for (i = 0; i < W_DATA_LEN; i++)
    {
    	if (wdata[i] != rdata[i])
    	{
    		return;
    	}
    }
}

void i2c_eeprom_verify(uint32_t i2c_base)
{
#if(IS_GS32F3xx(0x22))
	uint32_t i2c_port_irqn = INT_I2CA;
#else
	uint32_t i2c_port_irqn = INT_I2C0;
#endif
    uint8_t i = 0;


    g_i2c_base = i2c_base;

    if (i2c_base == I2CA_BASE)
    {
#if(IS_GS32F3xx(0x22))
    	i2c_port_irqn = INT_I2CA;
#else
    	i2c_port_irqn = INT_I2C0;
#endif
    	i2c_port_num = 0;
    }
    else if (i2c_base == I2CB_BASE)
    {
#if(IS_GS32F3xx(0x22))
    	i2c_port_irqn = INT_I2CB;
#else
    	i2c_port_irqn = INT_I2C1;
#endif
    	i2c_port_num = 1;
    }
    else
    {
    	return;
    }


	for (i = 0; i < W_DATA_LEN; i++)
	{
		wdata[i]=i+1;

		if(i==W_DATA_LEN-1)
		{
			wdata[i]|=I2C_IC_DATA_CMD_STOP;
		}
	}

    //Configuring IIC Parameters
	I2C_disableModule(i2c_base);
	I2C_disableInterrupt(i2c_base, 0xFFFFFFF);
	I2C_setTargetAddress(i2c_base,EEPROM_SLAVE_ADDRESS);
	I2C_setConfig(i2c_base, I2C_MASTER_MODE | I2C_TX_EMPTY_CTRL | I2C_RESTART_EN);//
	I2C_initController(i2c_base, DEVICE_APBCLK_FREQ, 400000, I2C_DUTYCYCLE_50);
	I2C_enableModule(i2c_base);

    //enable error interrupt
    Interrupt_register(i2c_port_irqn, I2CX_INTR_IRQHandler);
    I2C_enableInterrupt(i2c_base, I2C_INT_RX_OVER | I2C_INT_RX_UNDER | I2C_INT_TX_ABRT | I2C_INT_TX_OVER);
    Interrupt_enable(i2c_port_irqn);
    I2C_enableDma(i2c_base);
    I2C_setDmaTdLr(i2c_base, 2);
//  I2C_setDmaRdLr(i2c_base, 3);

    eeprom_init(EEPROM_DATA_ADDRESS_LEN);
    i2c_dma_tx(wdata,W_DATA_LEN);
    //i2c_eeprom_operate(i2c_base);
}


 void i2c_test06_dma_out(void)
 {
	GPIO_setPinConfig(GPIO_2_I2CB_SDA);
	GPIO_setPadConfig(2, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);
	GPIO_setQualificationMode(2, GPIO_QUAL_ASYNC);

	GPIO_setPinConfig(GPIO_3_I2CB_SCL);
	GPIO_setPadConfig(3, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);
	GPIO_setQualificationMode(3, GPIO_QUAL_ASYNC);

	__enable_irq();

    i2c_eeprom_verify(I2CB_BASE);
 }

#ifdef __cplusplus
}
#endif

