/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "device.h"
#include "i2c_eeprom.h"

struct I2CMsg i2cMsgOut = {MSG_STATUS_SEND_WITHSTOP,
                           TARGET_ADDRESS,
                           NUM_BYTES,
                           EEPROM_HIGH_ADDR,
                           EEPROM_LOW_ADDR,
                           0x01,
                           0x23,
                           0x45,
                           0x67,
                           0x89,
                           0xAB,
                           0xCD,
                           0xEF};
struct I2CMsg i2cMsgIn  = {MSG_STATUS_SEND_NOSTOP,
                           TARGET_ADDRESS,
                           NUM_BYTES,
                           EEPROM_HIGH_ADDR,
                           EEPROM_LOW_ADDR};

struct I2CMsg *currentMsgPtr;

uint16_t passCount = 0;
uint16_t failCount = 0;

static void i2c_module_init(void)
{
    /* Must put I2C into reset before configuring it */
	I2C_disableModule(I2C_BASE);

	/* I2C configuration. Use a 400kHz I2CCLK with a 50% duty cycle. */
	I2C_initController(I2C_BASE, DEVICE_APBCLK_FREQ, 400000, I2C_DUTYCYCLE_50);
	I2C_setBitCount(I2C_BASE, I2C_BITCOUNT_8);
	I2C_setTargetAddress(I2C_BASE, TARGET_ADDRESS);
	I2C_setEmulationMode(I2C_BASE, I2C_EMULATION_FREE_RUN);
	I2C_disableLoopback(I2C_BASE);

	/* Enable stop condition and register-access-ready interrupts*/
	I2C_enableInterrupt(I2C_BASE, I2C_INT_STOP_CONDITION |
								   I2C_INT_REG_ACCESS_RDY);

	/* FIFO configuration */
	I2C_enableFIFO(I2C_BASE);
	I2C_clearInterruptStatus(I2C_BASE, I2C_INT_RXFF | I2C_INT_TXFF);
	/* Configuration complete. Enable the module. */
	I2C_enableModule(I2C_BASE);
}

static void i2c_gpio_init(void)
{

#if (GS32_PART_NUM != 0x35)
    GPIO_setPinConfig(I2C_SDA_PIN_MUX);
    GPIO_setPadConfig(I2C_SDA_PIN, GPIO_PIN_TYPE_PULLUP);
    GPIO_setQualificationMode(I2C_SDA_PIN, GPIO_QUAL_ASYNC);
    GPIO_setStrength(I2C_SDA_PIN,GPIO_DRV_20MA);

    GPIO_setPinConfig(I2C_SCL_PIN_MUX);
    GPIO_setPadConfig(I2C_SCL_PIN, GPIO_PIN_TYPE_PULLUP);
    GPIO_setQualificationMode(I2C_SCL_PIN, GPIO_QUAL_ASYNC);
    GPIO_setStrength(I2C_SCL_PIN,GPIO_DRV_20MA);

#else

    I2C_SDA_PIN_MUX;
    I2C_SCL_PIN_MUX;

#endif
}

__interrupt void i2cISR(void)
{
	I2C_InterruptSource intSource;
	uint16_t i;

	/* Read interrupt source */
	intSource = I2C_getInterruptSource(I2C_BASE);
	/* Interrupt source = stop condition detected */
	if(intSource == I2C_INTSRC_STOP_CONDITION)
	{
		/* If completed message was writing data, reset msg to inactive state */
		if(currentMsgPtr->msgStatus == MSG_STATUS_WRITE_BUSY)
		{
			currentMsgPtr->msgStatus = MSG_STATUS_INACTIVE;
		}
		else
		{
			/*
			 * If a message receives a NACK during the address setup portion of
			 * the EEPROM read, the code further below included in the register
			 * access ready interrupt source code will generate a stop
			 * condition. After the stop condition is received (here), set the
			 * message status to try again. User may want to limit the number
			 * of retries before generating an error.
			 */
			if(currentMsgPtr->msgStatus == MSG_STATUS_SEND_NOSTOP_BUSY)
			{
				currentMsgPtr->msgStatus = MSG_STATUS_SEND_NOSTOP;
			}
			/*
			 * If completed message was reading EEPROM data, reset message to
			 * inactive state and read data from FIFO.
			 */
			else if(currentMsgPtr->msgStatus == MSG_STATUS_READ_BUSY)
			{
				currentMsgPtr->msgStatus = MSG_STATUS_INACTIVE;
				for(i=0; i < NUM_BYTES; i++)
				{
					currentMsgPtr->msgBuffer[i] = I2C_getData(I2C_BASE);
				}

				/* Check received data */
				for(i=0; i < NUM_BYTES; i++)
				{

					if(i2cMsgIn.msgBuffer[i] == i2cMsgOut.msgBuffer[i])
					{
						passCount++;
					}
					else
					{
						failCount++;
					}
				}
			}
		}
		I2C_clearInterruptStatus(I2C_BASE,I2C_INTSRC_STOP_CONDITION);
	}
	/*
	 *
	 * Interrupt source = Register Access Ready
	 *
	 * This interrupt is used to determine when the EEPROM address setup
	 * portion of the read data communication is complete. Since no stop bit
	 * is commanded, this flag tells us when the message has been sent
	 * instead of the SCD flag.
	 */
	else if(intSource == I2C_INTSRC_REG_ACCESS_RDY)
	{
		/*
		 * If a NACK is received, clear the NACK bit and command a stop.
		 * Otherwise, move on to the read data portion of the communication.
		 */
		if((I2C_getStatus(I2C_BASE) & I2C_STS_NO_ACK) != 0)
		{

			I2C_sendStopCondition(I2C_BASE);
			I2C_clearStatus(I2C_BASE, I2C_STS_NO_ACK);
		}
		else if(currentMsgPtr->msgStatus == MSG_STATUS_SEND_NOSTOP_BUSY)
		{
			currentMsgPtr->msgStatus = MSG_STATUS_RESTART;
		}
		I2C_clearInterruptStatus(I2C_BASE,I2C_INTSRC_REG_ACCESS_RDY);
	}
	I2C_clearInterruptStatus(I2C_BASE,intSource);

}


void i2c_init(void)
{

	i2c_gpio_init();

	i2c_module_init();

	Interrupt_register(I2C_INT, &i2cISR);

	Interrupt_enable(I2C_INT);
}

uint16_t writeData(struct I2CMsg *msg)
{
    uint16_t i;

    /*
     * Wait until the STP bit is cleared from any previous controller
     * communication. Clearing of this bit by the module is delayed until after
     * the SCD bit is set. If this bit is not checked prior to initiating a new
     * message, the I2C could get confused.
     */
    if(I2C_getStopConditionStatus(I2C_BASE))
    {
        return(ERROR_STOP_NOT_READY);
    }

    /*
     * Setup target address
     */
    I2C_setTargetAddress(I2C_BASE, TARGET_ADDRESS);

    /*
     * Check if bus busy
     */
    if(I2C_isBusBusy(I2C_BASE))
    {
        return(ERROR_BUS_BUSY);
    }

    /*
     * Setup number of bytes to send msgBuffer and address
     */
    I2C_setDataCount(I2C_BASE, (msg->numBytes + EEPROM_ADDR_LENGTH));

    I2C_setConfig(I2C_BASE, I2C_CONTROLLER_SEND_MODE);
    I2C_sendStartCondition(I2C_BASE);

    /*
     * Setup data to send
     */
#if(EEPROM_ADDR_LENGTH == 2)
    I2C_putData(I2C_BASE, msg->memoryHighAddr);
#endif
    I2C_putData(I2C_BASE, msg->memoryLowAddr);

    for (i = 0; i < msg->numBytes; i++)
    {
        I2C_putData(I2C_BASE, msg->msgBuffer[i]);
    }

    I2C_sendStopCondition(I2C_BASE);

    return(SUCCESS);
}

uint16_t readData(struct I2CMsg *msg)
{
	/*
	 * Wait until the STP bit is cleared from any previous controller
	 * communication. Clearing of this bit by the module is delayed until after
	 * the SCD bit is set. If this bit is not checked prior to initiating a new
	 * message, the I2C could get confused.
	 */
    if(I2C_getStopConditionStatus(I2C_BASE))
    {
        return(ERROR_STOP_NOT_READY);
    }

    /* Setup target address */
    I2C_setTargetAddress(I2C_BASE, TARGET_ADDRESS);

    /*
     * If we are in the the address setup phase, send the address without a
     * stop condition.
     */
    if(msg->msgStatus == MSG_STATUS_SEND_NOSTOP)
    {
    	/* Check if bus busy */
        if(I2C_isBusBusy(I2C_BASE))
        {

            return(ERROR_BUS_BUSY);
        }

        I2C_setConfig(I2C_BASE, I2C_CONTROLLER_SEND_MODE);
		I2C_sendStartCondition(I2C_BASE);

        /* Send data to setup EEPROM address */
        I2C_setDataCount(I2C_BASE, EEPROM_ADDR_LENGTH);
#if(EEPROM_ADDR_LENGTH == 2)
        I2C_putData(I2C_BASE, msg->memoryHighAddr);
#endif
    	I2C_putData(I2C_BASE, msg->memoryLowAddr);

    }
    else if(msg->msgStatus == MSG_STATUS_RESTART)
    {
    	/*
    	 * Address setup phase has completed. Now setup how many bytes expected
    	 * and send restart as controller-receiver.
    	 */
        I2C_setDataCount(I2C_BASE, (msg->numBytes));
        I2C_setConfig(I2C_BASE, I2C_CONTROLLER_RECEIVE_MODE);
        I2C_sendStartCondition(I2C_BASE);
        I2C_sendStopCondition(I2C_BASE);
    }

    return(SUCCESS);
}

