/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifndef __I2C_EEPROM_H_
#define __I2C_EEPROM_H_

#ifdef __cplusplus
extern "C"{
#endif

#include "inc/hw_types.h"
#include "inc/hw_memmap.h"

#if (GS32_PART_NUM != 0x35)

#define I2C_BASE 	I2CA_BASE

#if(I2C_BASE == I2CA_BASE)

#define I2C_SDA_PIN		0U
#define I2C_SCL_PIN		1U

#define I2C_SDA_PIN_MUX	GPIO_0_I2CA_SDA
#define I2C_SCL_PIN_MUX	GPIO_1_I2CA_SCL

#define I2C_INT		INT_I2CA

#elif (I2C_BASE == I2CB_BASE)

#define I2C_SDA_PIN		2U
#define I2C_SCL_PIN		3U

#define I2C_SDA_PIN_MUX	GPIO_2_I2CB_SDA
#define I2C_SCL_PIN_MUX	GPIO_3_I2CB_SCL

#define I2C_INT		INT_I2CB

#endif

#else

#define I2C_BASE 	I2CB_BASE

#if(I2C_BASE == I2CA_BASE)

#define I2C_SDA_PIN_MUX		GpioCtrlRegs.GPAMUX2.bit.GPIO28 = (GPIO_28_I2CA_SDA & GPIO_MUX_MASK)
#define I2C_SCL_PIN_MUX		GpioCtrlRegs.GPAMUX2.bit.GPIO29 = (GPIO_29_I2CA_SCL & GPIO_MUX_MASK)

#define I2C_INT		INT_I2CA

#elif (I2C_BASE == I2CB_BASE)

#define I2C_SDA_PIN_MUX		GpioCtrlRegs.GPAMUX2.bit.GPIO30 = (GPIO_30_I2CB_SDA & GPIO_MUX_MASK)
#define I2C_SCL_PIN_MUX		GpioCtrlRegs.GPAMUX2.bit.GPIO31 = (GPIO_31_I2CB_SCL & GPIO_MUX_MASK)

#define I2C_INT		INT_I2CB

#endif

#endif

#define EEPROM_ADDR_LENGTH			1
#define TARGET_ADDRESS              0x50
#define EEPROM_HIGH_ADDR            0x00
#define EEPROM_LOW_ADDR             0x00
#define EEPROM_ADDR_LENGTH			1
#define NUM_BYTES                   8
/*
 * Max is currently 14 because of 2 address bytes and the 16-byte FIFO
 */
#define MAX_BUFFER_SIZE             14

//
// I2C message states for I2CMsg struct
//
#define MSG_STATUS_INACTIVE         0x0000 /* Message not in use, do not send */
#define MSG_STATUS_SEND_WITHSTOP    0x0010 /* Send message with stop bit */
#define MSG_STATUS_WRITE_BUSY       0x0011 /* Message sent, wait for stop */
#define MSG_STATUS_SEND_NOSTOP      0x0020 /* Send message without stop bit */
#define MSG_STATUS_SEND_NOSTOP_BUSY 0x0021 /* Message sent, wait for ARDY */
#define MSG_STATUS_RESTART          0x0022 /* Ready to become controller-receiver */
#define MSG_STATUS_READ_BUSY        0x0023 /* Wait for stop before reading data */

/*
 * Error messages for read and write functions
 */
#define ERROR_BUS_BUSY              0x1000
#define ERROR_STOP_NOT_READY        0x5555
#define SUCCESS                     0x0000

struct I2CMsg {
    uint16_t msgStatus;                  /* Word stating what state msg is in. */
                                         /* See MSG_STATUS_* defines above. */
    uint16_t targetAddr;                 /* Target address tied to the message. */
    uint16_t numBytes;                   /* Number of valid bytes in message. */
    uint16_t memoryHighAddr;             /* EEPROM address of data associated */
                                         /* with message (high byte). */
    uint16_t memoryLowAddr;              /* EEPROM address of data associated */
                                         /* with message (low byte). */
    uint16_t msgBuffer[MAX_BUFFER_SIZE]; /* Array holding message data. */
};

extern struct I2CMsg i2cMsgOut;
extern struct I2CMsg i2cMsgIn;
extern struct I2CMsg *currentMsgPtr;

void i2c_init(void);
uint16_t writeData(struct I2CMsg *msg);
uint16_t readData(struct I2CMsg *msg);

#ifdef __cplusplus
}
#endif

#endif
